/* C Example Program - NI Board Level Functions       */

/*
 This sample program is for reference only and can only be expected to
 function with a Fluke 8840 Digital Multimeter that has been properly
 configured in IBCONF.
*/

#define INCL_DOSFILEMGR
#define INCL_DOSDEVICES
#include <fcntl.h>
#include <stdio.h>
#include <os2.h>
#include "nicode.h"

#define DISABLE_ERRORPOPUPS 0
void gpiberr(HFILE brd0, char *msg);


main()
{
	HFILE	brd0;			   /* device handle	     */
   ULONG ActionTaken;
	APIRET	RetCode;		   /* API call return code   */
	int	cnt;			   /* bytes read	     */
	short	spr;		/* serial poll response   */
	char	rd[512]; 		   /* buffer for read	     */
	nidev	nid;		      /* device config structure*/
	short	mask ;	      /* wait mask 	     */
   ULONG BytesWritten;
   ULONG	ParmLength;
   ULONG DataLength;

/* Get the board handle, exit on error.
 */
   RetCode = DosOpen("GPIB0",  &brd0,  &ActionTaken,
                 0L,          /* file's new size */
                 FILE_READONLY,   /* file attribute is read only */
                 OPEN_ACTION_OPEN_IF_EXISTS,   /* open the file if it already exists */
                 OPEN_FLAGS_WRITE_THROUGH | OPEN_FLAGS_FAIL_ON_ERROR | 
           		  OPEN_FLAGS_NOINHERIT | OPEN_SHARE_DENYNONE | 
					  OPEN_ACCESS_READWRITE,
                 0);
	if (RetCode != 0) {
		printf("Unable to open board");
		exit(1);
	}

/* Send Interface Clear.
 */
   ParmLength = 0;
   DataLength = 0;
	RetCode = DosDevIOCtl(brd0,CATEGORY,SENDIFC,0,ParmLength, &ParmLength,
					          0, DataLength, &DataLength);
   if (RetCode != 0) {
     gpiberr(brd0,"SENDIFC Error");
   }

/*  Turn on the Remote Enable (REN) signal.
 */
   ParmLength = 0;
   DataLength = 0;
	RetCode = DosDevIOCtl(brd0,CATEGORY,SETREMOTE,0,ParmLength, &ParmLength,
					          0, DataLength, &DataLength);
   if (RetCode != 0) {
     gpiberr(brd0,"SETREMOTE Error");
   }

/* Inhibit front panel control with the Local Lockout (LLO) command (hex 11).
 * Place the device in remote mode by addressing it to listen. The listen
 * address of device three is its primary address, 3, plus hex 20.  This is an
 * ASCII "#".  Send the Device Clear (DCL) message (hex 14) to clear internal
 * device functions.  Finally, address the GPIB0 to talk by sending its talk
 * address, the pad 0 plus hex 40, or ASCII "@".  These commands can be found
 * in Appendix A of the Software Reference Manual.
 */
	cnt = 4;
   ParmLength = sizeof(cnt);
   DataLength = 4;
	RetCode = DosDevIOCtl(brd0,CATEGORY,CMD,&cnt,ParmLength, &ParmLength,
					          "\021#\024@", DataLength, &DataLength);
   if (RetCode != 0) {
     gpiberr(brd0,"CMD Error");
   }

/* Write the function, range, and trigger source instructions to the device.
 * This string is meaningful to the Fluke 8840 Digital Multimeter.
 */
   BytesWritten = 0;
	RetCode = DosWrite(brd0,"F3R7T3",6, &BytesWritten);
   if (RetCode != 0) {
     gpiberr(brd0,"DosWrite Error");
   }

/*  Send the GGET message (hex 8) to trigger a measurement reading.
 */
	cnt = 1;
   ParmLength = sizeof(cnt);
   DataLength = 1;
	RetCode = DosDevIOCtl(brd0,CATEGORY,CMD,&cnt,ParmLength, &ParmLength,
					          "\010", DataLength, &DataLength);
   if (RetCode != 0) {
     gpiberr(brd0,"CMD Error");
   }

/* Wait for the DVM to set SRQ (BSRQI) or for a timeout (BTIMO).
 * These status constants are defined in NICODE.H. If the current time
 * limit is too short, use BTempWrt to change it.
 */

  
   mask = BTIMO | BSRQI;

   ParmLength = sizeof(mask);
   DataLength = 0;
	RetCode = DosDevIOCtl(brd0, CATEGORY, BWAIT, &mask, ParmLength, &ParmLength,
					          0, DataLength, &DataLength);
   if (RetCode != 0)	 {
     gpiberr(brd0,"BWAIT Error");
   }

/*  If neither a timeout nor an error occurred, BWAIT must have returned
 *  on SRQ.  Next do a serial poll.  First unaddress bus devices by sending
 *  the untalk (UNT) command (ASCII "_", or hex 5F) and the unlisten (UNL)
 *  command (ASCII "?", or hex 3F).  Then send the Serial Poll Enable (SPE)
 *  command (hex 18) and the DVM's talk address (device three's pad, 3, plus
 *  hex 40, or ASCII "C") and the GPIB0 listen address (pad 0 plus hex 20,
 *  or ASCII space).  These commands can be found in Appendix A of the Software
 *  Reference Manual.
 */

	cnt = 5;
   ParmLength = sizeof(cnt);
   DataLength = 5;
	RetCode = DosDevIOCtl(brd0, CATEGORY, CMD, &cnt, ParmLength, &ParmLength,
					          "?_\030C ", DataLength, &DataLength);
   if (RetCode != 0) {
     gpiberr(brd0,"CMD Error");
   }

/* Now read the status byte.  If response is 0xC0, the device has
 * valid data to send; otherwise, it has a fault condition to report.
 */

   BytesWritten = 0;
	RetCode = DosRead(brd0, &spr, 1, &BytesWritten);
   if (RetCode != 0) {
     gpiberr(brd0,"DosRead Error");
   }
	if (!(spr & 0xC0))  {
      printf("Fluke has Fault condition.");
   }

 /* If more than one device were attached to the bus, it would be necessary to
    explicitly check the 0x40 bit of the device status word to be sure that
    another device had not been responsible for asserting SRQ.		      */

/* Complete the serial poll by sending the Serial Poll Disable (SPD) message
 * (hex 19). This command can be found in Appendix A of the Software
 * Reference Manual.
 */

	cnt = 1;
   ParmLength = sizeof(cnt);
   DataLength = 1;
	RetCode = DosDevIOCtl(brd0,CATEGORY,CMD,&cnt,ParmLength, &ParmLength,
					          "\031", DataLength, &DataLength);

   BytesWritten = 0;
	RetCode = DosRead(brd0, rd, 16, &BytesWritten);
   if (RetCode != 0)	 {
     gpiberr(brd0,"DosRead Error");
   }
   /* Call OFFLINE to disable the hardware and software. */
   ParmLength = 0;
   DataLength = 0;
	RetCode = DosDevIOCtl(brd0, CATEGORY, OFFLINE, 0, ParmLength, &ParmLength,
                         0, DataLength, &DataLength); 

  /* Return the handle to the system. */
	DosClose(brd0);
}

/*
 * ===========================================================================
 *                      Function GPIBERR
 *  This function will notify you that a API function failed by
 *  printing an error message.  The status variable status will also be
 *  printed in hexadecimal along with the mnemonic meaning of the bit position. 
 *  The status variable error will be printed in decimal along with the
 *  mnemonic meaning of the decimal value.  
 *
 *  The API function OFFLINE is called to disable the hardware and software.
 *
 *  The EXIT function will terminate this program.
 * ===========================================================================
 */

void gpiberr(HFILE brd0, char *msg)
{
   unsigned short status;
   unsigned short error;
   APIRET RetCode;
   ULONG ParmLength;
   ULONG DataLength; 

   printf("%s\n", msg);
   ParmLength = sizeof(status);
   DataLength = 0;
   RetCode = DosDevIOCtl(brd0, CATEGORY, STATUS, &status, ParmLength, &ParmLength,
                         0, DataLength, &DataLength);
   printf("status = 0x%x   <", status);
    if (status & BERR )  printf (" ERR");
    if (status & BTIMO)  printf (" TIMO");
    if (status & BEND )  printf (" END");
    if (status & BSRQI)  printf (" SRQI");
    if (status & BRQS )  printf (" RQS");
    if (status & BCMPL)  printf (" CMPL");
    if (status & BLOK )  printf (" LOK");
    if (status & BREM )  printf (" REM");
    if (status & BCIC )  printf (" CIC");
    if (status & BATN )  printf (" ATN");
    if (status & BTACS)  printf (" TACS");
    if (status & BLACS)  printf (" LACS");
    if (status & BDTAS)  printf (" DTAS");
    if (status & BDCAS)  printf (" DCAS");
    printf (" >\n");             

   /* The lower byte of the return code holds the GPIB specific error. */
    error = RetCode & 0x00FF ;
    if (error == EDVR) printf (" EDVR <DOS Error>\n");
    if (error == ECIC) printf (" ECIC <Not CIC>\n");
    if (error == ENOL) printf (" ENOL <No Listener>\n");
    if (error == EADR) printf (" EADR <Address error>\n");
    if (error == EARG) printf (" EARG <Invalid argument>\n");
    if (error == ESAC) printf (" ESAC <Not Sys Ctrlr>\n");
    if (error == EABO) printf (" EABO <Op. aborted>\n");
    if (error == ENEB) printf (" ENEB <No GPIB board>\n");
    if (error == EOIP) printf (" EOIP <Async I/O in prg>\n");
    if (error == ECAP) printf (" ECAP <No capability>\n");
    if (error == EFSO) printf (" EFSO <File sys. error>\n");
    if (error == EBUS) printf (" EBUS <Command error>\n");
    if (error == ESTB) printf (" ESTB <Status byte lost>\n");
    if (error == ESRQ) printf (" ESRQ <SRQ stuck on>\n");
    if (error == ETAB) printf (" ETAB <Table Overflow>\n");

   /* Call OFFLINE to disable the hardware and software. */
   ParmLength = 0;
   DataLength = 0;
	RetCode = DosDevIOCtl(brd0, CATEGORY, OFFLINE, 0, ParmLength, &ParmLength,
                         0, DataLength, &DataLength); 
  /* Return the handle to the system. */
   DosClose(brd0);
   exit (1);
}


