/****************************************************************************\
*                                                                            *
* Function: Brush dither support                                             *
*                                                                            *
* This algorithm for color dithering is patent pending and its use is
* restricted to Microsoft products and drivers for Microsoft products.
* Use in non-Microsoft products or in drivers for non-Microsoft product is
* prohibited without written permission from Microsoft.
*
* The patent application is the primary reference for the operation of the
* color dithering code.
*
* Note that in the comments and variable names, "vertex" means "vertex of
* either the inner (half intensity) or outer (full intensity) color cube."
* Vertices map to colors 0-6 and 248-255 of the Windows standard (required)
* 256-color palette, where vertices 0-6 and 248 are the vertices of the inner
* color cube, and 0 plus 249-255 are the vertices of the full color cube.
* Vertex 7 is 75% gray; this could be used in the dither, but that would break
* apps that depend on the exact Windows 3.1 dithering. This code is Window 3.1
* compatible.
*                                                                            *
* Author:                                                                    *
*                                                                            *
*       Microsoft                                                            *
*                                                                            *
* History:                                                                   *
*                                                                            *
*       05/24/93 Dave Schmenk - copied from Windows NT display driver.       *
*                                                                            *
******************************************************************************
*                                                                            *
*                     Copyright 1992 Microsoft Corporation                   *
*                                                                            *
\****************************************************************************/

#include "common.h"

/****************************************************************************\
* This function takes a value from 0 - 255 and uses it to create an          *
* 8x8 pile of bits in the form of a 1BPP bitmap.  It can also take an        *
* RGB value and make an 8x8 bitmap.  These can then be used as brushes       *
* to simulate color unavaible on the device.                                 *
*                                                                            *
* For monochrome the basic algorithm is equivalent to turning on bits        *
* in the 8x8 array according to the following order:                         *
*                                                                            *
*  00 32 08 40 02 34 10 42                                                   *
*  48 16 56 24 50 18 58 26                                                   *
*  12 44 04 36 14 46 06 38                                                   *
*  60 28 52 20 62 30 54 22                                                   *
*  03 35 11 43 01 33 09 41                                                   *
*  51 19 59 27 49 17 57 25                                                   *
*  15 47 07 39 13 45 05 37                                                   *
*  63 31 55 23 61 29 53 21                                                   *
*                                                                            *
* Reference: A Survey of Techniques for the Display of Continous             *
*            Tone Pictures on Bilevel Displays,;                             *
*            Jarvis, Judice, & Ninke;                                        *
*            COMPUTER GRAPHICS AND IMAGE PROCESSING 5, pp 13-40, (1976)      *
\****************************************************************************/

extern DWORD devSystemPalette[];
extern DWORD devPalette[];

//
// Tells which row to turn a pel on in when dithering for monochrome bitmaps.
//

static BYTE ajByte[] =
{
    0, 4, 0, 4, 2, 6, 2, 6,
    0, 4, 0, 4, 2, 6, 2, 6,
    1, 5, 1, 5, 3, 7, 3, 7,
    1, 5, 1, 5, 3, 7, 3, 7,
    0, 4, 0, 4, 2, 6, 2, 6,
    0, 4, 0, 4, 2, 6, 2, 6,
    1, 5, 1, 5, 3, 7, 3, 7,
    1, 5, 1, 5, 3, 7, 3, 7
};

//
// The array of monochrome bits used for dithering.
//

static BYTE ajBits[] =
{
    0x80, 0x08, 0x08, 0x80, 0x20, 0x02, 0x02, 0x20,
    0x20, 0x02, 0x02, 0x20, 0x80, 0x08, 0x08, 0x80,
    0x40, 0x04, 0x04, 0x40, 0x10, 0x01, 0x01, 0x10,
    0x10, 0x01, 0x01, 0x10, 0x40, 0x04, 0x04, 0x40,
    0x40, 0x04, 0x04, 0x40, 0x10, 0x01, 0x01, 0x10,
    0x10, 0x01, 0x01, 0x10, 0x40, 0x04, 0x04, 0x40,
    0x80, 0x08, 0x08, 0x80, 0x20, 0x02, 0x02, 0x20,
    0x20, 0x02, 0x02, 0x20, 0x80, 0x08, 0x08, 0x80
};

/******************************Public*Routine******************************\
* DrvDitherColor
*
* Dithers an RGB color to an 8X8 approximation using the reserved VGA colors
*
\**************************************************************************/

VOID PASCAL devDitherBrush
(
    LPBRUSH lpBrush,
    DWORD   dwColor
)
{
    USHORT usGrey, usRed, usGre, usBlu;

    DBGMSG("devDitherBrush\n\r");

    //
    // Calculate monochrome dither pattern first using color intensity.
    //

    lpBrush->brMonoBits[0] = 0;
    lpBrush->brMonoBits[1] = 0;
    lpBrush->brMonoBits[2] = 0;
    lpBrush->brMonoBits[3] = 0;
    lpBrush->brMonoBits[4] = 0;
    lpBrush->brMonoBits[5] = 0;
    lpBrush->brMonoBits[6] = 0;
    lpBrush->brMonoBits[7] = 0;

    usRed = (USHORT)(dwColor)       & 0xFF;
    usGre = (USHORT)(dwColor >> 8)  & 0xFF;
    usBlu = (USHORT)(dwColor >> 16) & 0xFF;

    // I = .30R + .59G + .11B
    // For convience the following ratios are used:
    //
    //  77/256 = 30.08%
    // 151/256 = 58.98%
    //  28/256 = 10.94%

    usGrey  = (((usRed * 77) + (usGre * 151) + (usBlu * 28)) >> 8) & 255;

    // Convert the RGBI from 0-255 to 0-64 notation.

    usGrey = (usGrey + 1) >> 2;

    while(usGrey)
    {
        usGrey--;
        lpBrush->brMonoBits[ajByte[usGrey]] |= ajBits[usGrey];
    }

    lpBrush->brClrFore = 0xFF000000L;
    lpBrush->brClrBack = 0xFFFFFFFFL;

    lpBrush->brFlags   |= BRUSH_SOLID;
    lpBrush->brClrSolid = SWIZZLE_RGB(dwColor);

    _asm
    {
        push    ds
        lds     si, lpBrush
        mov     ax, WORD PTR [si][8]            // BRUSH COLOR SOLID
        mov     dx, WORD PTR [si][9]
        mov     bl, dh
        mov     bh, al
        add     si, 36
        mov     cx, 8
FillPat:
        mov     [si], ax
        mov     [si + 2], bx
        mov     [si + 4], dx
        mov     [si + 6], ax
        mov     [si + 8], bx
        mov     [si + 10], dx
        mov     [si + 12], ax
        mov     [si + 14], bx
        mov     [si + 16], dx
        mov     [si + 18], ax
        mov     [si + 20], bx
        mov     [si + 22], dx
        add     si, 24
        dec     cx
        jnz     FillPat
        pop     ds
    }
}
