/*************************************************************************
*                                                 *
*  Microcode Downloader for the IRISVISION graphics boards           *
*      by Todd Nordland, Luther Kitahata, Allen Orcutt                   *
*                                                 *
*  $Revision: 1.1 $                                       *
*                                                 *
**************************************************************************/
//#include <conio.h>
//#include <windows.h>

#include <common.h>

#define WIN_DRIVER 1
#define AGR 1

#include "gfxboard.h"
#include "ge5_glob.h"
#include "mcout.h"
#include "iv_ucode.h"
#include "gecmds.h"

#include "newexe.h"
#include "msdosdll.h"
#include "externs.h"

#define FILE_START     0     /* used with _llseek() */
#define FILE_CURRENT     1     /* used with _llseek() */


/* forward references */
static void near gr1_save_masks(void);
static void near gr1_restore_masks(void);
int near get_ucode( int fp, uc_hdr_t far *ucp);
void near gr1_clr_gfxintr(void);

GRPsetup;

/*
 *  iv_mload - IRISVISION microcode downloading routine
 */
int _loadds
iv_mload(uc_hdr_t far *ucp)
{
    long addr;
    register long count;
    register unsigned long _huge *hlp;
    unsigned long _far *flp;
    long tmp;
    long dooms_day_counter;
    int fp;
    int rtn;

    /*
     * Access microcode resource
     */
    fp = access_resource(RT_UCODE, RN_IV_UCODE);
    if (!fp) {
#ifdef DEBUG
     OutputDebugString("iv_mload: access_resource error\r\n");
#endif /* DEBUG */
     return ABORTDOWNLD;
    }

    rtn = get_ucode(fp, ucp);
    _lclose(fp);

    if (rtn == E_UCREAD || rtn == E_UCMAGIC) {
#ifdef DEBUG
     OutputDebugString("iv_mload: microcode file error\r\n");
#endif /* DEBUG */
     return ABORTDOWNLD;
    }

#ifdef DEBUG
    OutputDebugString("iv_mload: downloading microcode\r\n");
#endif

    /* hold off interrupts */
    GFXIMASK_WR(0);

    /*
     *  Download microcode.
     */
    addr = 0;
    hlp = (unsigned long _huge *)GlobalLock(ucp->codehandle);
    for (count = 0; count < ucp->codelen; count += 2 * sizeof(long)) {
//     if (addr % HQM_PG_SIZE == 0) {
     if ((addr & (HQM_PG_SIZE-1)) == 0) {
         HQM_WR(addr);
     }
     HQMMSB_WR(0);
     URAM_WR(addr, *hlp++);
     HQMMSB_WR(1);
     URAM_WR(addr, *hlp++);
     addr++;
    }

    /* Do the Windows memory thing. */
    rtn = GlobalUnlock(ucp->codehandle);
    GlobalFree(ucp->codehandle);
    ucp->codehandle = 0;

    /*
     *  Download constants.
     */
    addr = DIVMODTBL;
    HQMMSB_WR(0);
    HQM_WR(addr);
    flp = (unsigned long _far *)GlobalLock(ucp->consthandle);
    for (count = 0; count < ucp->constlen; count += sizeof(long)) {
//     if (addr % HQM_PG_SIZE == 0) {
     if ((addr & (HQM_PG_SIZE-1)) == 0) {
         HQM_WR(addr);
     }
     DRAM_WR(addr, *flp++);
     addr++;
    }
    GlobalUnlock(ucp->consthandle);
    GlobalFree(ucp->consthandle);
    ucp->consthandle = 0;

    gr1_clr_gfxintr();  /* clear any existing graphics interrupt */

    /*
     *  Set HQ middle addr reg most sig. bit to 1 for the operations below.
     *  Then latch PC in HQ to 0 by reading address 0.
     */
    HQMMSB_WR(1);
    HQM_WR(0);
    URAM_RD(0, tmp);

#ifdef DEBUG
    OutputDebugString("iv_mload: unstalling HQ\r\n");
#endif
    HQCLRSTL();

    /*
     *  Poll GE interrupt bit.
     */
    dooms_day_counter = 0x100000;

    while ( !(GFXINTR_TEST(GFX_INT_GE)) ) {
     --dooms_day_counter;
     if (dooms_day_counter <= 0) {
#ifdef DEBUG
     OutputDebugString("iv_mload: dooms_day_counter timeout\r\n");
#endif /* DEBUG */
         return ABORTDOWNLD; /* actually not aborted but something's wrong */
     }
    }

    /* Clear GE interrupt. */
    gr1_clr_gfxintr();

    /*
     *  Take HQ out of stall mode.
     *  Then Set HQ middle addr reg most sig. bit to default value.
     */
#ifdef DEBUG
    OutputDebugString("iv_mload: microcode tests OK\r\n");
#endif
    HQMMSB_WR(1);
    HQCLRSTL();
    HQMMSB_WR(0);

    return 0;
}

/*
 *  gr1_save_masks -- mask off local io interrupts pertinent to gr1, save status
 */
static unsigned char ge_mask = 0, vr_mask = 0, fifo_mask = 0;

static void near
gr1_save_masks()
{
    ge_mask     = GFXIMASK & (unsigned char)GFX_GE_MASK;
    vr_mask     = GFXIMASK & (unsigned char)GFX_VR_MASK;
    fifo_mask     = GFXIMASK & (unsigned char)GFX_FIFO_MASK;

    MASK_GFXINTR( GFX_GE_MASK | GFX_VR_MASK | GFX_FIFO_MASK );
}


/*
 *  gr1_restore_masks -- restore saved local io interrupt masks.
 */
static void near
gr1_restore_masks()
{ 
    unsigned char tmask;

    tmask = (char)GFXIMASK;
    tmask &= ((~GFX_GE_MASK) & (~GFX_VR_MASK) & (~GFX_FIFO_MASK));

    tmask     |= ge_mask;
    tmask     |= vr_mask;
    tmask     |= fifo_mask;

    GFXIMASK_WR(tmask);
}


/*
 *  gr1_clr_gfxintr -- clear graphics interrupt
 */
void near
gr1_clr_gfxintr()
{
    GRPsetup;

    HQMMSB_WR(1);
    HQCLRINT();
    HQMMSB_WR(0);
    GFXINTR_CLR(GFX_INT_GE);  /* eddy needs this */
}
void far _pascal GetExePtr(unsigned int);

/* new access_resource 
   by passing FindResource DWORDS instead of LPSTR, we can
   use constants defined for UCODE RESOURCES.  ONLY requirement
   is that upper word of DWORD is 0, Which is the case since the
   constants are passed in as WORDS.
   

*/
int 
access_resource(unsigned short rtype, unsigned short rname)
{

   int ModuleHandle;
   int fp;
   int hRes;
   DWORD rnameD;
   DWORD rtypeD;

   rtypeD = (DWORD) rtype;
   rnameD = (DWORD) rname;
   _asm {
      push ds
      call GetExePtr
      mov ModuleHandle,ax
   }
   hRes = FindResource(ModuleHandle,(LPSTR) rnameD, (LPSTR) rtypeD);
   if (!hRes)  {
      OutputDebugString("Access_Resource Error: Find Resource Failed\r\n");
      return(FALSE);
   }
   fp = AccessResource(ModuleHandle,hRes);
   
   if (fp == -1) {
      OutputDebugString("Access_Resource Error: unable to open EXE file\r\n");
      return(FALSE);
   }
   return(fp);


}

/*****************************************************************************
 *   get_ucode-- get microcode in mc.out.h format, store in memory
 * 
 *****************************************************************************/
int near
get_ucode(int fp, uc_hdr_t far *ucp)
{
    mcout_t head;
    LPSTR bufp;
    HGLOBAL bufh;     /* handle to buffer allocated by Windows kernel */
    void _huge *huge_bufp;

    /* read header, check for correct magic number */
    if (_lread(fp, (LPSTR)&head, sizeof(head)) != sizeof(head))
     return(E_UCREAD);

    if (head.f_magic != GE5_MAGIC || head.f_version != DOS_VERSION)
        return(E_UCMAGIC);

    /* 
     *  For reading in constants we must make sure we don't need to get
     *  around the limit imposed by _lread() which doesn't accept a huge
     *  pointer to the destination buffer.
     */
    if (head.f_constlen > 0x7fff)
        return(E_UCMAGIC);
     
    if (_llseek(fp, -(long)sizeof(head), FILE_CURRENT) == -1)
     return(E_UCREAD);


    /*************************************************************************
     *  Transfer version and magic numbers to header structure.  Obtain
     *  buffer for descriptor, seek to desriptor, read it from file.
     */
    ucp->magic = head.f_magic;
    ucp->version = head.f_version;
    ucp->ucversion = head.f_ucversion;


    /*************************************************************************
     *  Transfer code length to header.  Obtain buffer for the code, seek
     *  to microcode and read it from file.
     */
    ucp->codelen = head.f_codelen;

    /* ask Windows to give us some memory for the actual microcode */
    bufh = GlobalAlloc(GMEM_MOVEABLE | GMEM_DISCARDABLE, head.f_codelen);
    if (bufh == NULL)
     goto ucread_err;
    else
     ucp->codehandle = bufh;

    /* lock the buffer down so we can get a pointer to it */
    huge_bufp = (void _huge *)GlobalLock(ucp->codehandle);
    if (huge_bufp == NULL) {
     GlobalFree(bufh);
     goto ucread_err;
    }

    if (_llseek(fp, head.f_codeoff, FILE_CURRENT) == -1) {
     goto buf2_err;
    }
    
    if ( _hread(fp, huge_bufp, head.f_codelen) != head.f_codelen ) {
     goto buf2_err;
    }

    if (_llseek(fp, -(head.f_codelen+head.f_codeoff), FILE_CURRENT) == -1) {
     goto buf2_err;
    }


    /*************************************************************************
     *  Transfer constant length to header then do microcode constants --
     *  obtain buffer, seek constants, read.
     */
    ucp->constlen = head.f_constlen;

    /* MORE! MORE! - now we need some memory for the ucode constants */
    bufh = GlobalAlloc(GMEM_MOVEABLE | GMEM_DISCARDABLE, head.f_constlen);
    if (bufh == NULL)
     goto buf2_err;
    else
     ucp->consthandle = bufh;

    /* lock the buffer down so we can get a pointer to it */
    bufp = GlobalLock(ucp->consthandle);
    if (bufp == NULL) {
     GlobalFree(bufh);
     goto buf2_err;
    }

    if (_llseek(fp, head.f_constoff, FILE_CURRENT) == -1) {
     goto buf3_err;
    }

    /*
     *  In the following _lread call we make the assumption that the
     *  length of the constants section of the microcode is actually less
     *  than 64K bytes and can therefore be represented by a unsigned short
     *  (i.e. a WORD in the Windows world).  If this isn't true then we'll
     *  have to do something like what was done above for downloading the
     *  actual "code" section of the microcode.
     */
    if ( _lread(fp, bufp, (WORD)head.f_constlen) != (WORD)head.f_constlen ) {
     goto buf3_err;
    }

    GlobalUnlock(ucp->consthandle);
    GlobalUnlock(ucp->codehandle);

    return( 0 );

/* ERROR exits */
buf3_err:
    GlobalUnlock(ucp->consthandle);
    GlobalFree(ucp->consthandle);
    ucp->consthandle = 0;
buf2_err:
    GlobalUnlock(ucp->codehandle);
    GlobalFree(ucp->codehandle);
    ucp->codehandle = 0;
ucread_err:
    return(E_UCREAD);
}







