/****************************************************************************\
*                                                                            *
* Function: Brush dither support                                             *
*                                                                            *
* This algorithm for color dithering is patent pending and its use is
* restricted to Microsoft products and drivers for Microsoft products.
* Use in non-Microsoft products or in drivers for non-Microsoft product is
* prohibited without written permission from Microsoft.
*
* The patent application is the primary reference for the operation of the
* color dithering code.
*
* Note that in the comments and variable names, "vertex" means "vertex of
* either the inner (half intensity) or outer (full intensity) color cube."
* Vertices map to colors 0-6 and 248-255 of the Windows standard (required)
* 256-color palette, where vertices 0-6 and 248 are the vertices of the inner
* color cube, and 0 plus 249-255 are the vertices of the full color cube.
* Vertex 7 is 75% gray; this could be used in the dither, but that would break
* apps that depend on the exact Windows 3.1 dithering. This code is Window 3.1
* compatible.
*                                                                            *
* Author:                                                                    *
*                                                                            *
*       Microsoft                                                            *
*                                                                            *
* History:                                                                   *
*                                                                            *
*       05/24/93 Dave Schmenk - copied from Windows NT display driver.       *
*                                                                            *
******************************************************************************
*                                                                            *
*                     Copyright 1992 Microsoft Corporation                   *
*                                                                            *
\****************************************************************************/

#include "common.h"

/****************************************************************************\
* This function takes a value from 0 - 255 and uses it to create an          *
* 8x8 pile of bits in the form of a 1BPP bitmap.  It can also take an        *
* RGB value and make an 8x8 bitmap.  These can then be used as brushes       *
* to simulate color unavaible on the device.                                 *
*                                                                            *
* For monochrome the basic algorithm is equivalent to turning on bits        *
* in the 8x8 array according to the following order:                         *
*                                                                            *
*  00 32 08 40 02 34 10 42                                                   *
*  48 16 56 24 50 18 58 26                                                   *
*  12 44 04 36 14 46 06 38                                                   *
*  60 28 52 20 62 30 54 22                                                   *
*  03 35 11 43 01 33 09 41                                                   *
*  51 19 59 27 49 17 57 25                                                   *
*  15 47 07 39 13 45 05 37                                                   *
*  63 31 55 23 61 29 53 21                                                   *
*                                                                            *
* Reference: A Survey of Techniques for the Display of Continous             *
*            Tone Pictures on Bilevel Displays,;                             *
*            Jarvis, Judice, & Ninke;                                        *
*            COMPUTER GRAPHICS AND IMAGE PROCESSING 5, pp 13-40, (1976)      *
\****************************************************************************/

#define SWAP_RB 0x00000004
#define SWAP_GB 0x00000002
#define SWAP_RG 0x00000001

#define SWAPTHEM(a,b) (usTemp = a, a = b, b = usTemp)

//
// PATTERNSIZE is the number of pixels in a dither pattern.
//

#define PATTERNSIZE 64

typedef struct _VERTEX_DATA
{
    USHORT usCount;  // # of pixels in this vertex
    USHORT usVertex; // vertex #
} VERTEX_DATA, FAR * LPVERTEX_DATA;

extern DWORD devSystemPalette[];
extern DWORD devPalette[];

//
// Tells which row to turn a pel on in when dithering for monochrome bitmaps.
//

static BYTE ajByte[] =
{
    0, 4, 0, 4, 2, 6, 2, 6,
    0, 4, 0, 4, 2, 6, 2, 6,
    1, 5, 1, 5, 3, 7, 3, 7,
    1, 5, 1, 5, 3, 7, 3, 7,
    0, 4, 0, 4, 2, 6, 2, 6,
    0, 4, 0, 4, 2, 6, 2, 6,
    1, 5, 1, 5, 3, 7, 3, 7,
    1, 5, 1, 5, 3, 7, 3, 7
};

//
// The array of monochrome bits used for dithering.
//

static BYTE ajBits[] =
{
    0x80, 0x08, 0x08, 0x80, 0x20, 0x02, 0x02, 0x20,
    0x20, 0x02, 0x02, 0x20, 0x80, 0x08, 0x08, 0x80,
    0x40, 0x04, 0x04, 0x40, 0x10, 0x01, 0x01, 0x10,
    0x10, 0x01, 0x01, 0x10, 0x40, 0x04, 0x04, 0x40,
    0x40, 0x04, 0x04, 0x40, 0x10, 0x01, 0x01, 0x10,
    0x10, 0x01, 0x01, 0x10, 0x40, 0x04, 0x04, 0x40,
    0x80, 0x08, 0x08, 0x80, 0x20, 0x02, 0x02, 0x20,
    0x20, 0x02, 0x02, 0x20, 0x80, 0x08, 0x08, 0x80
};

//
// Translates vertices back to the original subspace. Each row is a subspace,
// as encoded in usSymmetry, and each column is a vertex between 0 and 15.
//

BYTE jSwapSubSpace[8*16] =
{
    0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15,
    0, 2, 1, 3, 4, 6, 5, 7, 8, 10, 9, 11, 12, 14, 13, 15,
    0, 1, 4, 5, 2, 3, 6, 7, 8, 9, 12, 13, 10, 11, 14, 15,
    0, 4, 1, 5, 2, 6, 3, 7, 8, 12, 9, 13, 10, 14, 11, 15,
    0, 4, 2, 6, 1, 5, 3, 7, 8, 12, 10, 14, 9, 13, 11, 15,
    0, 2, 4, 6, 1, 3, 5, 7, 8, 10, 12, 14, 9, 11, 13, 15,
    0, 4, 1, 5, 2, 6, 3, 7, 8, 12, 9, 13, 10, 14, 11, 15,
    0, 1, 4, 5, 2, 3, 6, 7, 8, 9, 12, 13, 10, 11, 14, 15,
};

//
// Converts a nibble value in the range 0-15 to a word value containing the
// nibble value packed 8 times.
//

USHORT usNibbleToWordWithConvert[16] =
{
    0x0000,
    0x0101,
    0x0202,
    0x0303,
    0x0404,
    0x0505,
    0x0606,
    0xF8F8,
    0x0707,
    0xF9F9,
    0xFAFA,
    0xFBFB,
    0xFCFC,
    0xFDFD,
    0xFEFE,
    0xFFFF
};

//
// Specifies where in the dither pattern colors shousd be placed in order
// of increasing intensity.
//

USHORT ausDitherOrder[] =
{
  0, 36,  4, 32, 18, 54, 22, 50,
  2, 38,  6, 34, 16, 52, 20, 48,
  9, 45, 13, 41, 27, 63, 31, 59,
 11, 47, 15, 43, 25, 61, 29, 57,
  1, 37,  5, 33, 19, 55, 23, 51,
  3, 39,  7, 35, 17, 53, 21, 49,
  8, 44, 12, 40, 26, 62, 30, 58,
 10, 46, 14, 42, 24, 60, 28, 56,
};

//
// Array to convert to 256 color from 16 color. Maps from index that represents
// a 16-color vertex (color) to value that specifies the color index in the
// 256-color palette.
//

BYTE ajConvert[] =
{
    0,
    1,
    2,
    3,
    4,
    5,
    6,
    248,
    7,
    249,
    250,
    251,
    252,
    253,
    254,
    255
};

/******************************Public*Routine******************************\
* DrvDitherColor
*
* Dithers an RGB color to an 8X8 approximation using the reserved VGA colors
*
\**************************************************************************/

VOID PASCAL devDitherBrush
(
    LPBRUSH lpBrush,
    DWORD   dwColor
)
{
    LPUSHORT      lpColorBits;
    USHORT        usGrey, usRed, usGre, usBlu, usSymmetry;
    USHORT        usRedTemp, usGreenTemp, usBlueTemp, usTemp;
    VERTEX_DATA   vVertexData[4];
    LPVERTEX_DATA lpvVertexData;
    LPVERTEX_DATA lpvVertexDataEnd;
    LPUSHORT      lpusDitherOrder;
    USHORT        usNumPixels;
    BYTE          jColor;
    USHORT        usColor;
    LPVERTEX_DATA lpvMaxVertex;
    USHORT        usVertex0Temp, usVertex1Temp, usVertex2Temp, usVertex3Temp;
    LPBYTE        lpjDither;

    DBGMSG("devDitherBrush\n\r");

    //
    // Calculate monochrome dither pattern first using color intensity.
    //
    lpBrush->brMonoBits[0] = 0;
    lpBrush->brMonoBits[1] = 0;
    lpBrush->brMonoBits[2] = 0;
    lpBrush->brMonoBits[3] = 0;
    lpBrush->brMonoBits[4] = 0;
    lpBrush->brMonoBits[5] = 0;
    lpBrush->brMonoBits[6] = 0;
    lpBrush->brMonoBits[7] = 0;

    //
    // If this is a color index, get the palette RGB so we can do
    // something intelligent.
    //

    if (dwColor & 0xFF000000L)
    {
        usRed = (USHORT)(devPalette[(BYTE)dwColor])       & 0xFF;
        usGre = (USHORT)(devPalette[(BYTE)dwColor] >> 8)  & 0xFF;
        usBlu = (USHORT)(devPalette[(BYTE)dwColor] >> 16) & 0xFF;
    }
    else
    {
        usRed = (USHORT)(dwColor)       & 0xFF;
        usGre = (USHORT)(dwColor >> 8)  & 0xFF;
        usBlu = (USHORT)(dwColor >> 16) & 0xFF;
    }

    // I = .30R + .59G + .11B
    // For convience the following ratios are used:
    //
    //  77/256 = 30.08%
    // 151/256 = 58.98%
    //  28/256 = 10.94%

    usGrey  = (((usRed * 77) + (usGre * 151) + (usBlu * 28)) >> 8) & 255;

    // Convert the RGBI from 0-255 to 0-64 notation.

    usGrey = (usGrey + 1) >> 2;

    while(usGrey)
    {
        usGrey--;
        lpBrush->brMonoBits[ajByte[usGrey]] |= ajBits[usGrey];
    }

    lpBrush->brClrFore = 0xFF000000L;
    lpBrush->brClrBack = 0xFF0001FFL;

    lpColorBits = (LPUSHORT)lpBrush->brColorBits;
    lpjDither   = (LPBYTE)lpBrush->brColorBits;

    //
    // Is the color really a color index?
    //

    if (dwColor & 0xFF000000L)
    {
        //
        // Make this a solid brush.
        //

        lpBrush->brFlags   |= BRUSH_SOLID;
        usColor             = (USHORT)(dwColor & 0xFF);
        lpBrush->brClrSolid = usColor;
        usColor            |= usColor << 8;
        *lpColorBits        = usColor;
        *(lpColorBits+1)    = usColor;
        *(lpColorBits+2)    = usColor;
        *(lpColorBits+3)    = usColor;
        *(lpColorBits+4)    = usColor;
        *(lpColorBits+5)    = usColor;
        *(lpColorBits+6)    = usColor;
        *(lpColorBits+7)    = usColor;
        *(lpColorBits+8)    = usColor;
        *(lpColorBits+9)    = usColor;
        *(lpColorBits+10)   = usColor;
        *(lpColorBits+11)   = usColor;
        *(lpColorBits+12)   = usColor;
        *(lpColorBits+13)   = usColor;
        *(lpColorBits+14)   = usColor;
        *(lpColorBits+15)   = usColor;
        *(lpColorBits+16)   = usColor;
        *(lpColorBits+17)   = usColor;
        *(lpColorBits+18)   = usColor;
        *(lpColorBits+19)   = usColor;
        *(lpColorBits+20)   = usColor;
        *(lpColorBits+21)   = usColor;
        *(lpColorBits+22)   = usColor;
        *(lpColorBits+23)   = usColor;
        *(lpColorBits+24)   = usColor;
        *(lpColorBits+25)   = usColor;
        *(lpColorBits+26)   = usColor;
        *(lpColorBits+27)   = usColor;
        *(lpColorBits+28)   = usColor;
        *(lpColorBits+29)   = usColor;
        *(lpColorBits+30)   = usColor;
        *(lpColorBits+31)   = usColor;
        return;
    }

    //
    // Quick check for solid match.
    //

    for (usTemp = 0; usTemp < 20; usTemp++)
    {
        if (dwColor == (devSystemPalette[usTemp] & 0x00FFFFFFL))
        {
            //
            // Exact match found!
            //

            lpBrush->brFlags   |= BRUSH_SOLID;
            usColor             = (USHORT)(devSystemPalette[usTemp] >> 24);
            lpBrush->brClrSolid = usColor;
            usColor            |= usColor << 8;
            *lpColorBits        = usColor;
            *(lpColorBits+1)    = usColor;
            *(lpColorBits+2)    = usColor;
            *(lpColorBits+3)    = usColor;
            *(lpColorBits+4)    = usColor;
            *(lpColorBits+5)    = usColor;
            *(lpColorBits+6)    = usColor;
            *(lpColorBits+7)    = usColor;
            *(lpColorBits+8)    = usColor;
            *(lpColorBits+9)    = usColor;
            *(lpColorBits+10)   = usColor;
            *(lpColorBits+11)   = usColor;
            *(lpColorBits+12)   = usColor;
            *(lpColorBits+13)   = usColor;
            *(lpColorBits+14)   = usColor;
            *(lpColorBits+15)   = usColor;
            *(lpColorBits+16)   = usColor;
            *(lpColorBits+17)   = usColor;
            *(lpColorBits+18)   = usColor;
            *(lpColorBits+19)   = usColor;
            *(lpColorBits+20)   = usColor;
            *(lpColorBits+21)   = usColor;
            *(lpColorBits+22)   = usColor;
            *(lpColorBits+23)   = usColor;
            *(lpColorBits+24)   = usColor;
            *(lpColorBits+25)   = usColor;
            *(lpColorBits+26)   = usColor;
            *(lpColorBits+27)   = usColor;
            *(lpColorBits+28)   = usColor;
            *(lpColorBits+29)   = usColor;
            *(lpColorBits+30)   = usColor;
            *(lpColorBits+31)   = usColor;
            return;             
        }
    }

    // Split the color into red, green, and blue components
    usRedTemp   = (USHORT)(dwColor)       & 0xFF;
    usGreenTemp = (USHORT)(dwColor >> 8)  & 0xFF;
    usBlueTemp  = (USHORT)(dwColor >> 16) & 0xFF;

    // Sort the RGB so that the point is transformed into subspace 0, and
    // keep track of the swaps in usSymmetry so we can unravel it again
    // later.  We want r >= g >= b (subspace 0).
    usSymmetry = 0;
    if (usBlueTemp > usRedTemp)
    {
        SWAPTHEM(usBlueTemp,usRedTemp);
        usSymmetry = SWAP_RB;
    }

    if (usBlueTemp > usGreenTemp)
    {
        SWAPTHEM(usBlueTemp,usGreenTemp);
        usSymmetry |= SWAP_GB;
    }

    if (usGreenTemp > usRedTemp)
    {
        SWAPTHEM(usGreenTemp,usRedTemp);
        usSymmetry |= SWAP_RG;
    }

    usSymmetry <<= 4;   // for lookup purposes

    // Scale the values from 0-255 to 0-64. Note that the scaling is not
    // symmetric at the ends; this is done to match Windows 3.1 dithering
    usRed = (usRedTemp   + 1) >> 2;
    usGre = (usGreenTemp + 1) >> 2;
    usBlu = (usBlueTemp  + 1) >> 2;

    // Vertex count
    usTemp = 0;

    // Compute the subsubspace within subspace 0 in which the point lies,
    // then calcusate the # of pixels to dither in the colors that are the
    // four vertexes of the tetrahedron bounding the color we're emusating.
    // Only vertices with more than zero pixels are stored, and the
    // vertices are stored in order of increasing intensity, saving us the
    // need to sort them later
    if ((usRedTemp + usGreenTemp) > 256)
    {
        // Subsubspace 2 or 3
        if ((usRedTemp + usBlueTemp) > 256)
        {
            // Subsubspace 3
            // Calcusate the number of pixels per vertex, still in
            // subsubspace 3, then convert to original subspace. The pixel
            // counts and vertex numbers are matching pairs, stored in
            // ascending intensity order, skipping vertices with zero
            // pixels. The vertex intensity order for subsubspace 3 is:
            // 7, 9, 0x0B, 0x0F
            lpvVertexData = vVertexData;
            if ((usVertex0Temp = (64 - usRed) << 1) != 0)
            {
                lpvVertexData->usCount = usVertex0Temp;
                lpvVertexData++->usVertex = jSwapSubSpace[usSymmetry + 0x07];
                usTemp++;
            }
            usVertex2Temp = usGre - usBlu;
            usVertex3Temp = (usRed - 64) + usBlu;
            if ((usVertex1Temp = ((PATTERNSIZE - usVertex0Temp) -
                    usVertex2Temp) - usVertex3Temp) != 0)
            {
                lpvVertexData->usCount    = usVertex1Temp;
                lpvVertexData++->usVertex = jSwapSubSpace[usSymmetry + 0x09];
                usTemp++;
            }
            if (usVertex2Temp != 0)
            {
                lpvVertexData->usCount    = usVertex2Temp;
                lpvVertexData++->usVertex = jSwapSubSpace[usSymmetry + 0x0B];
                usTemp++;
            }
            if (usVertex3Temp != 0)
            {
                lpvVertexData->usCount    = usVertex3Temp;
                lpvVertexData++->usVertex = jSwapSubSpace[usSymmetry + 0x0F];
                usTemp++;
            }
        }
        else
        {
            // Subsubspace 2
            // Calcusate the number of pixels per vertex, still in
            // subsubspace 2, then convert to original subspace. The pixel
            // counts and vertex numbers are matching pairs, stored in
            // ascending intensity order, skipping vertices with zero
            // pixels. The vertex intensity order for subsubspace 2 is:
            // 3, 7, 9, 0x0B
            lpvVertexData  = vVertexData;
            usVertex1Temp = usBlu << 1;
            usVertex2Temp = usRed - usGre;
            usVertex3Temp = (usRed - 32) + (usGre - 32);
            if ((usVertex0Temp = ((PATTERNSIZE - usVertex1Temp) -
                        usVertex2Temp) - usVertex3Temp) != 0)
            {
                lpvVertexData->usCount    = usVertex0Temp;
                lpvVertexData++->usVertex = jSwapSubSpace[usSymmetry + 0x03];
                usTemp++;
            }
            if (usVertex1Temp != 0)
            {
                lpvVertexData->usCount    = usVertex1Temp;
                lpvVertexData++->usVertex = jSwapSubSpace[usSymmetry + 0x07];
                usTemp++;
            }
            if (usVertex2Temp != 0)
            {
                lpvVertexData->usCount    = usVertex2Temp;
                lpvVertexData++->usVertex = jSwapSubSpace[usSymmetry + 0x09];
                usTemp++;
            }
            if (usVertex3Temp != 0)
            {
                lpvVertexData->usCount    = usVertex3Temp;
                lpvVertexData++->usVertex = jSwapSubSpace[usSymmetry + 0x0B];
                usTemp++;
            }
        }
    }
    else
    {
        // Subsubspace 0 or 1
        if (usRedTemp > 128)
        {
            // Subsubspace 1
            // Calcusate the number of pixels per vertex, still in
            // subsubspace 1, then convert to original subspace. The pixel
            // counts and vertex numbers are matching pairs, stored in
            // ascending intensity order, skipping vertices with zero
            // pixels. The vertex intensity order for subsubspace 1 is:
            // 1, 3, 7, 9
            lpvVertexData = vVertexData;
            if ((usVertex0Temp = ((32 - usGre) + (32 - usRed)) << 1) != 0)
            {
                lpvVertexData->usCount    = usVertex0Temp;
                lpvVertexData++->usVertex = jSwapSubSpace[usSymmetry + 0x01];
                usTemp++;
            }
            usVertex2Temp = usBlu << 1;
            usVertex3Temp = (usRed - 32) << 1;
            if ((usVertex1Temp = ((PATTERNSIZE - usVertex0Temp) -
                    usVertex2Temp) - usVertex3Temp) != 0)
            {
                lpvVertexData->usCount    = usVertex1Temp;
                lpvVertexData++->usVertex = jSwapSubSpace[usSymmetry + 0x03];
                usTemp++;
            }
            if (usVertex2Temp != 0)
            {
                lpvVertexData->usCount    = usVertex2Temp;
                lpvVertexData++->usVertex = jSwapSubSpace[usSymmetry + 0x07];
                usTemp++;
            }
            if (usVertex3Temp != 0)
            {
                lpvVertexData->usCount    = usVertex3Temp;
                lpvVertexData++->usVertex = jSwapSubSpace[usSymmetry + 0x09];
                usTemp++;
            }
        }
        else
        {
            // Subsubspace 0
            // Calcusate the number of pixels per vertex, still in
            // subsubspace 0, then convert to original subspace. The pixel
            // counts and vertex numbers are matching pairs, stored in
            // ascending intensity order, skipping vertices with zero
            // pixels. The vertex intensity order for subsubspace 0 is:
            // 0, 1, 3, 7
            lpvVertexData = vVertexData;
            if ((usVertex0Temp = (32 - usRed) << 1) != 0)
            {
                lpvVertexData->usCount    = usVertex0Temp;
                lpvVertexData++->usVertex = jSwapSubSpace[usSymmetry + 0x00];
                usTemp++;
            }
            if ((usVertex1Temp = (usRed - usGre) << 1) != 0)
            {
                lpvVertexData->usCount    = usVertex1Temp;
                lpvVertexData++->usVertex = jSwapSubSpace[usSymmetry + 0x01];
                usTemp++;
            }
            usVertex3Temp = usBlu << 1;
            if ((usVertex2Temp = ((PATTERNSIZE - usVertex0Temp) -
                    usVertex1Temp) - usVertex3Temp) != 0)
            {
                lpvVertexData->usCount    = usVertex2Temp;
                lpvVertexData++->usVertex = jSwapSubSpace[usSymmetry + 0x03];
                usTemp++;
            }
            if (usVertex3Temp != 0)
            {
                lpvVertexData->usCount    = usVertex3Temp;
                lpvVertexData++->usVertex = jSwapSubSpace[usSymmetry + 0x07];
                usTemp++;
            }
        }
    }

    // Now that we have found the bounding vertices and the number of
    // pixels to dither for each vertex, we can create the dither pattern

    // Handle 1, 2, and 3 & 4 vertices per dither separately
    // usTemp = lpvVertexData - vVertexData;
                                                    // # of vertices with more
                                                    // than zero pixels
    if (usTemp > 2)
    {
        // There are 3 or 4 vertices in this dither

        if (usTemp == 3)
        {
            // There are 3 vertices in this dither

            // Find the vertex with the most pixels, and fill the whole
            // destination bitmap with that vertex's color, which is faster
            // than dithering it
            if (vVertexData[1].usCount >= vVertexData[2].usCount)
            {
                lpvMaxVertex = &vVertexData[1];
                usTemp      = vVertexData[1].usCount;
            }
            else
            {
                lpvMaxVertex = &vVertexData[2];
                usTemp      = vVertexData[2].usCount;
            }

        }
        else
        {
            // There are 4 vertices in this dither

            // Find the vertex with the most pixels, and fill the whole
            // destination bitmap with that vertex's color, which is faster
            // than dithering it
            if (vVertexData[2].usCount >= vVertexData[3].usCount)
            {
                lpvMaxVertex = &vVertexData[2];
                usTemp      = vVertexData[2].usCount;
            }
            else
            {
                lpvMaxVertex = &vVertexData[3];
                usTemp      = vVertexData[3].usCount;
            }
        }

        if (vVertexData[1].usCount > usTemp)
        {
            lpvMaxVertex = &vVertexData[1];
            usTemp      = vVertexData[1].usCount;
        }
        if (vVertexData[0].usCount > usTemp)
        {
            lpvMaxVertex = &vVertexData[0];
        }

        lpvVertexDataEnd = lpvVertexData;

        // Prepare a dword version of the most common vertex number (color)
        usColor  = usNibbleToWordWithConvert[lpvMaxVertex->usVertex];

        // Mark that the vertex we're about to do doesn't need to be done
        // later
        lpvMaxVertex->usVertex = 0xFF;

        // Block fill the dither pattern with the more common vertex
        *lpColorBits      = usColor;
        *(lpColorBits+1)  = usColor;
        *(lpColorBits+2)  = usColor;
        *(lpColorBits+3)  = usColor;
        *(lpColorBits+4)  = usColor;
        *(lpColorBits+5)  = usColor;
        *(lpColorBits+6)  = usColor;
        *(lpColorBits+7)  = usColor;
        *(lpColorBits+8)  = usColor;
        *(lpColorBits+9)  = usColor;
        *(lpColorBits+10) = usColor;
        *(lpColorBits+11) = usColor;
        *(lpColorBits+12) = usColor;
        *(lpColorBits+13) = usColor;
        *(lpColorBits+14) = usColor;
        *(lpColorBits+15) = usColor;
        *(lpColorBits+16) = usColor;
        *(lpColorBits+17) = usColor;
        *(lpColorBits+18) = usColor;
        *(lpColorBits+19) = usColor;
        *(lpColorBits+20) = usColor;
        *(lpColorBits+21) = usColor;
        *(lpColorBits+22) = usColor;
        *(lpColorBits+23) = usColor;
        *(lpColorBits+24) = usColor;
        *(lpColorBits+25) = usColor;
        *(lpColorBits+26) = usColor;
        *(lpColorBits+27) = usColor;
        *(lpColorBits+28) = usColor;
        *(lpColorBits+29) = usColor;
        *(lpColorBits+30) = usColor;
        *(lpColorBits+31) = usColor;

        // Now dither all the remaining vertices in order 0->2 or 0->3
        // (in order of increasing intensity)
        lpusDitherOrder = ausDitherOrder;
        lpvVertexData   = vVertexData;
        do
        {
            if (lpvVertexData->usVertex == 0xFF)
            {
                // This is the max vertex, which we already did, but we
                // have to account for it in the dither order
                lpusDitherOrder += lpvVertexData->usCount;
            }
            else
            {
                jColor      = ajConvert[lpvVertexData->usVertex];
                usNumPixels = lpvVertexData->usCount;
                switch (usNumPixels & 3)
                {
                    case 3:
                        lpjDither[*(lpusDitherOrder+2)] = jColor;
                    case 2:
                        lpjDither[*(lpusDitherOrder+1)] = jColor;
                    case 1:
                        lpjDither[*(lpusDitherOrder+0)] = jColor;
                        lpusDitherOrder += usNumPixels & 3;
                    case 0:
                        break;
                }
                if ((usNumPixels >>= 2) != 0)
                {
                    do
                    {
                        lpjDither[*lpusDitherOrder]     = jColor;
                        lpjDither[*(lpusDitherOrder+1)] = jColor;
                        lpjDither[*(lpusDitherOrder+2)] = jColor;
                        lpjDither[*(lpusDitherOrder+3)] = jColor;
                        lpusDitherOrder += 4;
                    }
                    while (--usNumPixels);
                }
            }
        }
        while (++lpvVertexData < lpvVertexDataEnd);

    }
    else if (usTemp == 2)
    {
        // There are exactly two vertices with more than zero pixels; fill
        // in the dither array as follows: block fill with vertex with more
        // points first, then dither in the other vertex
        if (vVertexData[0].usCount >= vVertexData[1].usCount)
        {
            // There are no more vertex 1 than vertex 0 pixels, so do
            // the block fill with vertex 0
            usColor  = usNibbleToWordWithConvert[vVertexData[0].usVertex];
            // Do the dither with vertex 1
            jColor = ajConvert[vVertexData[1].usVertex];
            usNumPixels = vVertexData[1].usCount;
            // Set where to start dithering with vertex 1 (vertex 0 is
            // lower intensity, so its pixels come first in the dither
            // order)
            lpusDitherOrder = ausDitherOrder + vVertexData[0].usCount;
        }
        else
        {
            // There are more vertex 1 pixels, so do the block fill
            // with vertex 1
            usColor  = usNibbleToWordWithConvert[vVertexData[1].usVertex];
            // Do the dither with vertex 0
            jColor = ajConvert[vVertexData[0].usVertex];
            usNumPixels = vVertexData[0].usCount;
            // Set where to start dithering with vertex 0 (vertex 0 is
            // lower intensity, so its pixels come first in the dither
            // order)
            lpusDitherOrder = ausDitherOrder;
        }

        // Block fill the dither pattern with the more common vertex
        *lpColorBits      = usColor;
        *(lpColorBits+1)  = usColor;
        *(lpColorBits+2)  = usColor;
        *(lpColorBits+3)  = usColor;
        *(lpColorBits+4)  = usColor;
        *(lpColorBits+5)  = usColor;
        *(lpColorBits+6)  = usColor;
        *(lpColorBits+7)  = usColor;
        *(lpColorBits+8)  = usColor;
        *(lpColorBits+9)  = usColor;
        *(lpColorBits+10) = usColor;
        *(lpColorBits+11) = usColor;
        *(lpColorBits+12) = usColor;
        *(lpColorBits+13) = usColor;
        *(lpColorBits+14) = usColor;
        *(lpColorBits+15) = usColor;
        *(lpColorBits+16) = usColor;
        *(lpColorBits+17) = usColor;
        *(lpColorBits+18) = usColor;
        *(lpColorBits+19) = usColor;
        *(lpColorBits+20) = usColor;
        *(lpColorBits+21) = usColor;
        *(lpColorBits+22) = usColor;
        *(lpColorBits+23) = usColor;
        *(lpColorBits+24) = usColor;
        *(lpColorBits+25) = usColor;
        *(lpColorBits+26) = usColor;
        *(lpColorBits+27) = usColor;
        *(lpColorBits+28) = usColor;
        *(lpColorBits+29) = usColor;
        *(lpColorBits+30) = usColor;
        *(lpColorBits+31) = usColor;

        // Dither in the less common vertex
        switch (usNumPixels & 3)
        {
            case 3:
                lpjDither[*(lpusDitherOrder+2)] = jColor;
            case 2:
                lpjDither[*(lpusDitherOrder+1)] = jColor;
            case 1:
                lpjDither[*(lpusDitherOrder+0)] = jColor;
                lpusDitherOrder += usNumPixels & 3;
            case 0:
                break;
        }
        if ((usNumPixels >>= 2) != 0)
        {
            do
            {
                lpjDither[*lpusDitherOrder]     = jColor;
                lpjDither[*(lpusDitherOrder+1)] = jColor;
                lpjDither[*(lpusDitherOrder+2)] = jColor;
                lpjDither[*(lpusDitherOrder+3)] = jColor;
                lpusDitherOrder += 4;
            }
            while (--usNumPixels);
        }

    }
    else
    {
        // There is only one vertex in this dither
        // No sorting or dithering is needed for just one color; we can
        // just generate the final DIB directly

        lpBrush->brFlags   |= BRUSH_SOLID;
        usColor             = usNibbleToWordWithConvert[vVertexData[0].usVertex];
        lpBrush->brClrSolid = usColor;
        *lpColorBits        = usColor;
        *(lpColorBits+1)    = usColor;
        *(lpColorBits+2)    = usColor;
        *(lpColorBits+3)    = usColor;
        *(lpColorBits+4)    = usColor;
        *(lpColorBits+5)    = usColor;
        *(lpColorBits+6)    = usColor;
        *(lpColorBits+7)    = usColor;
        *(lpColorBits+8)    = usColor;
        *(lpColorBits+9)    = usColor;
        *(lpColorBits+10)   = usColor;
        *(lpColorBits+11)   = usColor;
        *(lpColorBits+12)   = usColor;
        *(lpColorBits+13)   = usColor;
        *(lpColorBits+14)   = usColor;
        *(lpColorBits+15)   = usColor;
        *(lpColorBits+16)   = usColor;
        *(lpColorBits+17)   = usColor;
        *(lpColorBits+18)   = usColor;
        *(lpColorBits+19)   = usColor;
        *(lpColorBits+20)   = usColor;
        *(lpColorBits+21)   = usColor;
        *(lpColorBits+22)   = usColor;
        *(lpColorBits+23)   = usColor;
        *(lpColorBits+24)   = usColor;
        *(lpColorBits+25)   = usColor;
        *(lpColorBits+26)   = usColor;
        *(lpColorBits+27)   = usColor;
        *(lpColorBits+28)   = usColor;
        *(lpColorBits+29)   = usColor;
        *(lpColorBits+30)   = usColor;
        *(lpColorBits+31)   = usColor;
    }
}
