.386

;----------------------------------------------------------------------------;
;                                                                            ;
; Function: PatBLT 8BPP pattern to 8BPP destination                          ;
;                                                                            ;
;       8BPP pattern BLT fills a rectangular region of a bitmap with a       ;
;       repeating pattern using a binary ROP.  This routine fills the region ;
;       on a scanline basis using a procudure table to apply the individual  ;
;       ROP functions.  The scanline BLT procedures are optimized to fill    ;
;       the destination with 32 bit operations.  DWORD alignment is          ;
;       calculated for each scanline.  ROPs that can be easily synthesized   ;
;       with other ROPs by altering the source pattern are implemented to    ;
;       conserve code space.  Note that a whole row of the pattern is loaded ;
;       into ebx:esi.  No other memory references to the pattern are needed  ;
;       after loading the row to BLT the destination scanline.  The case of  ;
;       ROP == 0xC has been optimized slightly more than the others by       ;
;       unrolling the inner loop.                                            ;
;                                                                            ;
; Author:                                                                    ;
;                                                                            ;
;       Dave Schmenk                                                         ;
;                                                                            ;
; History:                                                                   ;
;                                                                            ;
;       05/07/93 Dave Schmenk - wrote it.                                    ;
;                                                                            ;
;----------------------------------------------------------------------------;
;                                                                            ;
;                     Copyright 1993 pellucid, inc.                          ;
;                                                                            ;
;----------------------------------------------------------------------------;

incDrawMode     EQU     1               ; Include DRAWMODE structure

                INCLUDE COMMON.INC
                INCLUDE BITBLT.INC

_TEXT           SEGMENT PUBLIC WORD USE16 'CODE'

pDst            EQU     [bp + 26]
pDstSegmnt      EQU     [bp + 28]
pDstOffset      EQU     [bp + 26]
dyDst           EQU     [bp + 24]
cDstScanSeg     EQU     [bp + 22]
cDstFillBytes   EQU     [bp + 20]
xDst            EQU     [bp + 18]
yDst            EQU     [bp + 16]
pPat            EQU     [bp + 12]
pPatSegmnt      EQU     [bp + 14]
pPatOffset      EQU     [bp + 12]
xSize           EQU     [bp + 10]
ySize           EQU     [bp + 8]
mode            EQU     [bp + 6]
rop             EQU     [bp + 4]

pDstScan        EQU     [bp - 2]
cDstByte        EQU     [bp - 4]
cScan           EQU     [bp - 6]
pfnScanBlt      EQU     [bp - 8]
xPat            EQU     [bp - 10]
clrSolid        EQU     [bp - 14]

mempBltPD88Tbl  DW      mempBltPD88r0, mempBltPD88r1, mempBltPD88r2, mempBltPD88r3
                DW      mempBltPD88r4, mempBltPD88r5, mempBltPD88r6, mempBltPD88r7
                DW      mempBltPD88r8, mempBltPD88r9, mempBltPD88rA, mempBltPD88rB
                DW      mempBltPD88rC, mempBltPD88rD, mempBltPD88rE, mempBltPD88rF

memBltPD_ToColor        PROC NEAR
        
        push    bp
        mov     bp, sp
        sub     sp, 16
        push    ds
        push    es
        push    si
        push    di
;
; Load pattern segment.
;
        mov     es, pPatSegmnt
;
; Save pointer to first scanline.
;
        mov     ax, yDst
        bmScanAddrInc cDstScanSeg, dyDst, cScan, pDstSegmnt, pDstOffset
        mov     ds, dx
        mov     di, ax
;
; Calculate destination constants.
;
        add     di, xDst                ; ds:[di] = address of first pixel
;
; Save calculated constants.
;
        mov     pDstScan, di
;
; Precalc horizontal pattern alignment.
;
        mov     si, pPatOffset
        mov     cx, xDst
        sub     cx, es:[si][BRUSH_X]
        and     cx, 7
        mov     xPat, cx
;
; Check for transparent HATCH patterns
;
;        cmp     WORD PTR mode, TRANSPARENT
;        je      HatchBlt
;
; Translate rop2 into function pointer.
;
        mov     bx, rop
        add     bx, bx
        mov     bx, mempBltPD88Tbl[bx]
        jmp     bx
memBltPD_ToColor        ENDP

;----------------------------------------------------------------------------;
;                                                                            ;
;                  Pattern BLT, 1BPP, rop = 0                                ;
;                                                                            ;
;----------------------------------------------------------------------------;

mempBltPD88r0      PROC NEAR
        xor     eax, eax
        jmp     mempBltSolid
mempBltPD88r0      ENDP

;----------------------------------------------------------------------------;
;                                                                            ;
;                  Pattern BLT, 8BPP, rop = 1                                ;
;                                                                            ;
;----------------------------------------------------------------------------;

mempBltPD88r1      PROC NEAR
;
; Preload bx for ScanLoop.
;
        mov     bx, yDst
;
; Main scanline blt loop.
;
ScanLoop:
;
; Get current scanline of brush into ebx:esi (8 pixels worth).
;
        mov     si, pPatOffset
        sub     bx, es:[si][BRUSH_Y]
        and     bx, 7
        shl     bx, 3
        add     si, bx
        mov     ebx, es:[si][BRUSH_COLOR]
        mov     edx, es:[si + 4][BRUSH_COLOR]
;
; Calculate nearest destination brush boundary
;
        mov     cx, xPat
        mov     ax, di
        and     ax, 3
        mov     si, ax
        sub     cx, ax                  ; ax = aligned pattern destintion address
        jz      AlignFirst
;
; Calculate which half of the brush is first to be drawn
;
        test    cl, 4
        jz      @F
        xchg    ebx, edx
        and     cx, 3
        jz      AlignFirst
;
; Align brush scanline in ebx:esi to nearest DWORD address
;
@@:     test    cl, 2
        jz      @F
        xchg    bx, dx
        ror     ebx, 16
        ror     edx, 16
@@:     test    cl, 1
        jz      AlignFirst
        xchg    bl, dl
        ror     ebx, 8
        ror     edx, 8
;
; Align first (partial) DWORD of brush to destination BYTE.
;
AlignFirst:
        mov     eax, ebx
        mov     cx, si
        shl     cl, 3
        ror     eax, cl
        mov     cx, xSize
PatAlign:
        or      al, [di]
        not     al
        mov     [di], al
        inc     di
        test    di, 3
        jz      @F
        shr     eax, 8
        dec     cx
        jnz     PatAlign
        jmp     NextScan
@@:     dec     cx
        jz      NextScan
        sub     cx, 4
        jbe     @F
PatLoop4:
        mov     eax, edx
        or      eax, [di]
        not     eax
        mov     [di], eax
        xchg    ebx, edx
        add     di, 4
        sub     cx, 4
        jae     PatLoop4
        mov     eax, edx
@@:     add     cx, 4
PatLoop1:
        or      al, [di]
        not     al
        mov     [di], al
        shr     eax, 8
        inc     di
        dec     cx
        jnz     PatLoop1
NextScan:
;
; Decrement scanline count.
;
        dec     WORD PTR ySize
        jz      Exit
;
; Increment current scanline pointer.
;
        mov     di, pDstScan
        bmScanInc di, ds, cDstScanSeg, WORD PTR dyDst, WORD PTR cScan, cDstFillBytes
        mov     pDstScan, di
;
; Increment current y coordinate for indexing into brush
; and get brush phase alignment.
;
        mov     bx, yDst
        inc     bx
        mov     yDst, bx
        jmp     ScanLoop
Exit:
        pop     di
        pop     si
        pop     es
        pop     ds
        leave
        ret     26
mempBltPD88r1      ENDP

;----------------------------------------------------------------------------;
;                                                                            ;
;                  Pattern BLT, 8BPP, rop = 2                                ;
;                                                                            ;
;----------------------------------------------------------------------------;

mempBltPD88r2      PROC NEAR
;
; Preload bx for ScanLoop.
;
        mov     bx, yDst
;
; Main scanline blt loop.
;
ScanLoop:
;
; Get current scanline of brush into ebx:esi (8 pixels worth).
;
        mov     si, pPatOffset
        sub     bx, es:[si][BRUSH_Y]
        and     bx, 7
        shl     bx, 3
        add     si, bx
        mov     ebx, es:[si][BRUSH_COLOR]
        mov     edx, es:[si + 4][BRUSH_COLOR]
;
; Calculate nearest destination brush boundary
;
        mov     cx, xPat
        mov     ax, di
        and     ax, 3
        mov     si, ax
        sub     cx, ax                  ; ax = aligned pattern destintion address
        jz      AlignFirst
;
; Calculate which half of the brush is first to be drawn
;
        test    cl, 4
        jz      @F
        xchg    ebx, edx
        and     cx, 3
        jz      AlignFirst
;
; Align brush scanline in ebx:esi to nearest DWORD address
;
@@:     test    cl, 2
        jz      @F
        xchg    bx, dx
        ror     ebx, 16
        ror     edx, 16
@@:     test    cl, 1
        jz      AlignFirst
        xchg    bl, dl
        ror     ebx, 8
        ror     edx, 8
;
; Align first (partial) DWORD of brush to destination BYTE.
;
AlignFirst:
        mov     eax, ebx
        mov     cx, si
        shl     cl, 3
        ror     eax, cl
        mov     cx, xSize
        not     ebx
        not     edx
        not     eax
PatAlign:
        and     [di], al
        inc     di
        test    di, 3
        jz      @F
        shr     eax, 8
        dec     cx
        jnz     PatAlign
        jmp     NextScan
@@:     dec     cx
        jz      NextScan
        sub     cx, 4
        jbe     @F
PatLoop4:
        mov     eax, edx
        and     [di], eax
        xchg    ebx, edx
        add     di, 4
        sub     cx, 4
        ja      PatLoop4
        mov     eax, edx
@@:     add     cx, 4
PatLoop1:
        and     [di], al
        shr     eax, 8
        inc     di
        dec     cx
        jnz     PatLoop1
NextScan:
;
; Decrement scanline count.
;
        dec     WORD PTR ySize
        jz      Exit
;
; Increment current scanline pointer.
;
        mov     di, pDstScan
        bmScanInc di, ds, cDstScanSeg, WORD PTR dyDst, WORD PTR cScan, cDstFillBytes
        mov     pDstScan, di
;
; Increment current y coordinate for indexing into brush
; and get brush phase alignment.
;
        mov     bx, yDst
        inc     bx
        mov     yDst, bx
        jmp     ScanLoop
Exit:
        pop     di
        pop     si
        pop     es
        pop     ds
        leave
        ret     26
mempBltPD88r2      ENDP

;----------------------------------------------------------------------------;
;                                                                            ;
;                  Pattern BLT, 8BPP, rop = 3                                ;
;                                                                            ;
;----------------------------------------------------------------------------;

mempBltPD88r3      PROC NEAR
        mov     si, pPatOffset
        test    WORD PTR es:[si][BRUSH_FLAGS], 4        ; 4 = Solid brush
        jz      @F
        mov     al, es:[si][BRUSH_CLR_SOLID]
        mov     ah, al
        mov     bx, ax
        shl     eax, 16
        mov     ax, bx
        not     eax
        jmp     mempBltSolid
;
; Preload bx for ScanLoop.
;
@@:     mov     bx, yDst
;
; Main scanline blt loop.
;
ScanLoop:
;
; Get current scanline of brush into ebx:esi (8 pixels worth).
;
        mov     si, pPatOffset
        sub     bx, es:[si][BRUSH_Y]
        and     bx, 7
        shl     bx, 3
        add     si, bx
        mov     ebx, es:[si][BRUSH_COLOR]
        mov     edx, es:[si + 4][BRUSH_COLOR]
;
; Calculate nearest destination brush boundary
;
        mov     cx, xPat
        mov     ax, di
        and     ax, 3
        mov     si, ax
        sub     cx, ax                  ; ax = aligned pattern destintion address
        jz      AlignFirst
;
; Calculate which half of the brush is first to be drawn
;
        test    cl, 4
        jz      @F
        xchg    ebx, edx
        and     cx, 3
        jz      AlignFirst
;
; Align brush scanline in ebx:esi to nearest DWORD address
;
@@:     test    cl, 2
        jz      @F
        xchg    bx, dx
        ror     ebx, 16
        ror     edx, 16
@@:     test    cl, 1
        jz      AlignFirst
        xchg    bl, dl
        ror     ebx, 8
        ror     edx, 8
;
; Align first (partial) DWORD of brush to destination BYTE.
;
AlignFirst:
        mov     eax, ebx
        mov     cx, si
        shl     cl, 3
        ror     eax, cl
        mov     cx, xSize
        not     eax
        not     ebx
        not     edx
PatAlign:
        mov     [di], al
        inc     di
        test    di, 3
        jz      @F
        shr     eax, 8
        dec     cx
        jnz     PatAlign
        jmp     NextScan
@@:     dec     cx
        jz      NextScan
        sub     cx, 4
        jbe     @F
PatLoop4:
        mov     eax, edx
        mov     [di], eax
        xchg    ebx, edx
        add     di, 4
        sub     cx, 4
        ja      PatLoop4
        mov     eax, edx
@@:     add     cx, 4
PatLoop1:
        and     [di], al
        shr     eax, 8
        inc     di
        dec     cx
        jnz     PatLoop1
NextScan:
;
; Decrement scanline count.
;
        dec     WORD PTR ySize
        jz      Exit
;
; Increment current scanline pointer.
;
        mov     di, pDstScan
        bmScanInc di, ds, cDstScanSeg, WORD PTR dyDst, WORD PTR cScan, cDstFillBytes
        mov     pDstScan, di
;
; Increment current y coordinate for indexing into brush
; and get brush phase alignment.
;
        mov     bx, yDst
        inc     bx
        mov     yDst, bx
        jmp     ScanLoop
Exit:
        pop     di
        pop     si
        pop     es
        pop     ds
        leave
        ret     26
mempBltPD88r3    ENDP

;----------------------------------------------------------------------------;
;                                                                            ;
;                  Pattern BLT, 8BPP, rop = 4                                ;
;                                                                            ;
;----------------------------------------------------------------------------;

mempBltPD88r4      PROC NEAR
;
; Preload bx for ScanLoop.
;
        mov     bx, yDst
;
; Main scanline blt loop.
;
ScanLoop:
;
; Get current scanline of brush into ebx:esi (8 pixels worth).
;
        mov     si, pPatOffset
        sub     bx, es:[si][BRUSH_Y]
        and     bx, 7
        shl     bx, 3
        add     si, bx
        mov     ebx, es:[si][BRUSH_COLOR]
        mov     edx, es:[si + 4][BRUSH_COLOR]
;
; Calculate nearest destination brush boundary
;
        mov     cx, xPat
        mov     ax, di
        and     ax, 3
        mov     si, ax
        sub     cx, ax                  ; ax = aligned pattern destintion address
        jz      AlignFirst
;
; Calculate which half of the brush is first to be drawn
;
        test    cl, 4
        jz      @F
        xchg    ebx, edx
        and     cx, 3
        jz      AlignFirst
;
; Align brush scanline in ebx:esi to nearest DWORD address
;
@@:     test    cl, 2
        jz      @F
        xchg    bx, dx
        ror     ebx, 16
        ror     edx, 16
@@:     test    cl, 1
        jz      AlignFirst
        xchg    bl, dl
        ror     ebx, 8
        ror     edx, 8
;
; Align first (partial) DWORD of brush to destination BYTE.
;
AlignFirst:
        mov     eax, ebx
        mov     cx, si
        shl     cl, 3
        ror     eax, cl
        mov     cx, xSize
        push    ebx
PatAlign:
        mov     bl, [di]
        not     bl
        and     al, bl
        mov     [di], al
        inc     di
        test    di, 3
        jz      @F
        shr     eax, 8
        dec     cx
        jnz     PatAlign
        pop     ebx
        jmp     NextScan
@@:     pop     ebx
        dec     cx
        jz      NextScan
        sub     cx, 4
        jbe     @F
PatLoop4:
        mov     eax, edx
        mov     esi, [di]
        not     esi
        and     eax, esi
        mov     [di], eax
        xchg    ebx, edx
        add     di, 4
        sub     cx, 4
        ja      PatLoop4
        mov     eax, edx
@@:     add     cx, 4
PatLoop1:
        mov     bl, [di]
        not     bl
        and     al, bl
        mov     [di], al
        shr     eax, 8
        inc     di
        dec     cx
        jnz     PatLoop1
NextScan:
;
; Decrement scanline count.
;
        dec     WORD PTR ySize
        jz      Exit
;
; Increment current scanline pointer.
;
        mov     di, pDstScan
        bmScanInc di, ds, cDstScanSeg, WORD PTR dyDst, WORD PTR cScan, cDstFillBytes
        mov     pDstScan, di
;
; Increment current y coordinate for indexing into brush
; and get brush phase alignment.
;
        mov     bx, yDst
        inc     bx
        mov     yDst, bx
        jmp     ScanLoop
Exit:
        pop     di
        pop     si
        pop     es
        pop     ds
        leave
        ret     26
mempBltPD88r4      ENDP

;----------------------------------------------------------------------------;
;                                                                            ;
;                  Pattern BLT, 8BPP, rop = 5                                ;
;                                                                            ;
;----------------------------------------------------------------------------;

mempBltPD88r5      PROC NEAR
;
; Main scanline blt loop.
;
ScanLoop:
        mov     cx, xSize
PatAlign:
        not     BYTE PTR [di]
        inc     di
        test    di, 3
        jz      @F
        dec     cx
        jnz     PatAlign
        jmp     NextScan
@@:     dec     cx
        jz      NextScan
        sub     cx, 4
        jbe     @F
PatLoop4:
        not     DWORD PTR [di]
        add     di, 4
        sub     cx, 4
        ja      PatLoop4
@@:     add     cx, 4
PatLoop1:
        not     BYTE PTR [di]
        inc     di
        dec     cx
        jnz     PatLoop1
NextScan:
;
; Decrement scanline count.
;
        dec     WORD PTR ySize
        jz      Exit
;
; Increment current scanline pointer.
;
        mov     di, pDstScan
        bmScanInc di, ds, cDstScanSeg, WORD PTR dyDst, WORD PTR cScan, cDstFillBytes
        mov     pDstScan, di
        jmp     ScanLoop
Exit:
        pop     di
        pop     si
        pop     es
        pop     ds
        leave
        ret     26
mempBltPD88r5      ENDP

;----------------------------------------------------------------------------;
;                                                                            ;
;                  Pattern BLT, 8BPP, rop = 6                                ;
;                                                                            ;
;----------------------------------------------------------------------------;

mempBltPD88r6      PROC NEAR
;
; Preload bx for ScanLoop.
;
        mov     bx, yDst
;
; Main scanline blt loop.
;
ScanLoop:
;
; Get current scanline of brush into ebx:esi (8 pixels worth).
;
        mov     si, pPatOffset
        sub     bx, es:[si][BRUSH_Y]
        and     bx, 7
        shl     bx, 3
        add     si, bx
        mov     ebx, es:[si][BRUSH_COLOR]
        mov     edx, es:[si + 4][BRUSH_COLOR]
;
; Calculate nearest destination brush boundary
;
        mov     cx, xPat
        mov     ax, di
        and     ax, 3
        mov     si, ax
        sub     cx, ax                  ; ax = aligned pattern destintion address
        jz      AlignFirst
;
; Calculate which half of the brush is first to be drawn
;
        test    cl, 4
        jz      @F
        xchg    ebx, edx
        and     cx, 3
        jz      AlignFirst
;
; Align brush scanline in ebx:esi to nearest DWORD address
;
@@:     test    cl, 2
        jz      @F
        xchg    bx, dx
        ror     ebx, 16
        ror     edx, 16
@@:     test    cl, 1
        jz      AlignFirst
        xchg    bl, dl
        ror     ebx, 8
        ror     edx, 8
;
; Align first (partial) DWORD of brush to destination BYTE.
;
AlignFirst:
        mov     eax, ebx
        mov     cx, si
        shl     cl, 3
        ror     eax, cl
        mov     cx, xSize
PatAlign:
        xor     [di], al
        inc     di
        test    di, 3
        jz      @F
        shr     eax, 8
        dec     cx
        jnz     PatAlign
        jmp     NextScan
@@:     dec     cx
        jz      NextScan
        sub     cx, 4
        jbe     @F
PatLoop4:
        mov     eax, edx
        xor     [di], eax
        xchg    ebx, edx
        add     di, 4
        sub     cx, 4
        ja      PatLoop4
        mov     eax, edx
@@:     add     cx, 4
PatLoop1:
        xor     [di], al
        shr     eax, 8
        inc     di
        dec     cx
        jnz     PatLoop1
NextScan:
;
; Decrement scanline count.
;
        dec     WORD PTR ySize
        jz      Exit
;
; Increment current scanline pointer.
;
        mov     di, pDstScan
        bmScanInc di, ds, cDstScanSeg, WORD PTR dyDst, WORD PTR cScan, cDstFillBytes
        mov     pDstScan, di
;
; Increment current y coordinate for indexing into brush
; and get brush phase alignment.
;
        mov     bx, yDst
        inc     bx
        mov     yDst, bx
        jmp     ScanLoop
Exit:
        pop     di
        pop     si
        pop     es
        pop     ds
        leave
        ret     26
mempBltPD88r6      ENDP

;----------------------------------------------------------------------------;
;                                                                            ;
;                  Pattern BLT, 8BPP, rop = 7                                ;
;                                                                            ;
;----------------------------------------------------------------------------;

mempBltPD88r7      PROC NEAR
;
; Preload bx for ScanLoop.
;
        mov     bx, yDst
;
; Main scanline blt loop.
;
ScanLoop:
;
; Get current scanline of brush into ebx:esi (8 pixels worth).
;
        mov     si, pPatOffset
        sub     bx, es:[si][BRUSH_Y]
        and     bx, 7
        shl     bx, 3
        add     si, bx
        mov     ebx, es:[si][BRUSH_COLOR]
        mov     edx, es:[si + 4][BRUSH_COLOR]
;
; Calculate nearest destination brush boundary
;
        mov     cx, xPat
        mov     ax, di
        and     ax, 3
        mov     si, ax
        sub     cx, ax                  ; ax = aligned pattern destintion address
        jz      AlignFirst
;
; Calculate which half of the brush is first to be drawn
;
        test    cl, 4
        jz      @F
        xchg    ebx, edx
        and     cx, 3
        jz      AlignFirst
;
; Align brush scanline in ebx:esi to nearest DWORD address
;
@@:     test    cl, 2
        jz      @F
        xchg    bx, dx
        ror     ebx, 16
        ror     edx, 16
@@:     test    cl, 1
        jz      AlignFirst
        xchg    bl, dl
        ror     ebx, 8
        ror     edx, 8
;
; Align first (partial) DWORD of brush to destination BYTE.
;
AlignFirst:
        mov     eax, ebx
        mov     cx, si
        shl     cl, 3
        ror     eax, cl
        mov     cx, xSize
PatAlign:
        and     al, [di]
        not     al
        mov     [di], al
        inc     di
        test    di, 3
        jz      @F
        shr     eax, 8
        dec     cx
        jnz     PatAlign
        jmp     NextScan
@@:     dec     cx
        jz      NextScan
        sub     cx, 4
        jbe     @F
PatLoop4:
        mov     eax, edx
        and     eax, [di]
        not     eax
        mov     [di], eax
        xchg    ebx, edx
        add     di, 4
        sub     cx, 4
        ja      PatLoop4
        mov     eax, edx
@@:     add     cx, 4
PatLoop1:
        and     al, [di]
        not     al
        mov     [di], al
        shr     eax, 8
        inc     di
        dec     cx
        jnz     PatLoop1
NextScan:
;
; Decrement scanline count.
;
        dec     WORD PTR ySize
        jz      Exit
;
; Increment current scanline pointer.
;
        mov     di, pDstScan
        bmScanInc di, ds, cDstScanSeg, WORD PTR dyDst, WORD PTR cScan, cDstFillBytes
        mov     pDstScan, di
;
; Increment current y coordinate for indexing into brush
; and get brush phase alignment.
;
        mov     bx, yDst
        inc     bx
        mov     yDst, bx
        jmp     ScanLoop
Exit:
        pop     di
        pop     si
        pop     es
        pop     ds
        leave
        ret     26
mempBltPD88r7      ENDP

;----------------------------------------------------------------------------;
;                                                                            ;
;                  Pattern BLT, 8BPP, rop = 8                                ;
;                                                                            ;
;----------------------------------------------------------------------------;

mempBltPD88r8      PROC NEAR
;
; Preload bx for ScanLoop.
;
        mov     bx, yDst
;
; Main scanline blt loop.
;
ScanLoop:
;
; Get current scanline of brush into ebx:esi (8 pixels worth).
;
        mov     si, pPatOffset
        sub     bx, es:[si][BRUSH_Y]
        and     bx, 7
        shl     bx, 3
        add     si, bx
        mov     ebx, es:[si][BRUSH_COLOR]
        mov     edx, es:[si + 4][BRUSH_COLOR]
;
; Calculate nearest destination brush boundary
;
        mov     cx, xPat
        mov     ax, di
        and     ax, 3
        mov     si, ax
        sub     cx, ax                  ; ax = aligned pattern destintion address
        jz      AlignFirst
;
; Calculate which half of the brush is first to be drawn
;
        test    cl, 4
        jz      @F
        xchg    ebx, edx
        and     cx, 3
        jz      AlignFirst
;
; Align brush scanline in ebx:esi to nearest DWORD address
;
@@:     test    cl, 2
        jz      @F
        xchg    bx, dx
        ror     ebx, 16
        ror     edx, 16
@@:     test    cl, 1
        jz      AlignFirst
        xchg    bl, dl
        ror     ebx, 8
        ror     edx, 8
;
; Align first (partial) DWORD of brush to destination BYTE.
;
AlignFirst:
        mov     eax, ebx
        mov     cx, si
        shl     cl, 3
        ror     eax, cl
        mov     cx, xSize
PatAlign:
        and     [di], al
        inc     di
        test    di, 3
        jz      @F
        shr     eax, 8
        dec     cx
        jnz     PatAlign
        jmp     NextScan
@@:     dec     cx
        jz      NextScan
        sub     cx, 4
        jbe     @F
PatLoop4:
        mov     eax, edx
        and     [di], eax
        xchg    ebx, edx
        add     di, 4
        sub     cx, 4
        ja      PatLoop4
        mov     eax, edx
@@:     add     cx, 4
PatLoop1:
        and     [di], al
        shr     eax, 8
        inc     di
        dec     cx
        jnz     PatLoop1
NextScan:
;
; Decrement scanline count.
;
        dec     WORD PTR ySize
        jz      Exit
;
; Increment current scanline pointer.
;
        mov     di, pDstScan
        bmScanInc di, ds, cDstScanSeg, WORD PTR dyDst, WORD PTR cScan, cDstFillBytes
        mov     pDstScan, di
;
; Increment current y coordinate for indexing into brush
; and get brush phase alignment.
;
        mov     bx, yDst
        inc     bx
        mov     yDst, bx
        jmp     ScanLoop
Exit:
        pop     di
        pop     si
        pop     es
        pop     ds
        leave
        ret     26
mempBltPD88r8      ENDP

;----------------------------------------------------------------------------;
;                                                                            ;
;                  Pattern BLT, 8BPP, rop = 9                                ;
;                                                                            ;
;----------------------------------------------------------------------------;

mempBltPD88r9      PROC NEAR
;
; Preload bx for ScanLoop.
;
        mov     bx, yDst
;
; Main scanline blt loop.
;
ScanLoop:
;
; Get current scanline of brush into ebx:esi (8 pixels worth).
;
        mov     si, pPatOffset
        sub     bx, es:[si][BRUSH_Y]
        and     bx, 7
        shl     bx, 3
        add     si, bx
        mov     ebx, es:[si][BRUSH_COLOR]
        mov     edx, es:[si + 4][BRUSH_COLOR]
;
; Calculate nearest destination brush boundary
;
        mov     cx, xPat
        mov     ax, di
        and     ax, 3
        mov     si, ax
        sub     cx, ax                  ; ax = aligned pattern destintion address
        jz      AlignFirst
;
; Calculate which half of the brush is first to be drawn
;
        test    cl, 4
        jz      @F
        xchg    ebx, edx
        and     cx, 3
        jz      AlignFirst
;
; Align brush scanline in ebx:esi to nearest DWORD address
;
@@:     test    cl, 2
        jz      @F
        xchg    bx, dx
        ror     ebx, 16
        ror     edx, 16
@@:     test    cl, 1
        jz      AlignFirst
        xchg    bl, dl
        ror     ebx, 8
        ror     edx, 8
;
; Align first (partial) DWORD of brush to destination BYTE.
;
AlignFirst:
        mov     eax, ebx
        mov     cx, si
        shl     cl, 3
        ror     eax, cl
        mov     cx, xSize
        not     ebx
        not     edx
        not     eax
PatAlign:
        xor     [di], al
        inc     di
        test    di, 3
        jz      @F
        shr     eax, 8
        dec     cx
        jnz     PatAlign
        jmp     NextScan
@@:     dec     cx
        jz      NextScan
        sub     cx, 4
        jbe     @F
PatLoop4:
        mov     eax, edx
        xor     [di], eax
        xchg    ebx, edx
        add     di, 4
        sub     cx, 4
        ja      PatLoop4
        mov     eax, edx
@@:     add     cx, 4
PatLoop1:
        xor     [di], al
        shr     eax, 8
        inc     di
        dec     cx
        jnz     PatLoop1
NextScan:
;
; Decrement scanline count.
;
        dec     WORD PTR ySize
        jz      Exit
;
; Increment current scanline pointer.
;
        mov     di, pDstScan
        bmScanInc di, ds, cDstScanSeg, WORD PTR dyDst, WORD PTR cScan, cDstFillBytes
        mov     pDstScan, di
;
; Increment current y coordinate for indexing into brush
; and get brush phase alignment.
;
        mov     bx, yDst
        inc     bx
        mov     yDst, bx
        jmp     ScanLoop
Exit:
        pop     di
        pop     si
        pop     es
        pop     ds
        leave
        ret     26
mempBltPD88r9      ENDP

;----------------------------------------------------------------------------;
;                                                                            ;
;                  Pattern BLT, 8BPP, rop = A                                ;
;                                                                            ;
;----------------------------------------------------------------------------;

mempBltPD88rA      PROC NEAR
        pop     di
        pop     si
        pop     es
        pop     ds
        leave
        ret     26
mempBltPD88rA      ENDP

;----------------------------------------------------------------------------;
;                                                                            ;
;                  Pattern BLT, 8BPP, rop = B                                ;
;                                                                            ;
;----------------------------------------------------------------------------;

mempBltPD88rB      PROC NEAR
;
; Preload bx for ScanLoop.
;
        mov     bx, yDst
;
; Main scanline blt loop.
;
ScanLoop:
;
; Get current scanline of brush into ebx:esi (8 pixels worth).
;
        mov     si, pPatOffset
        sub     bx, es:[si][BRUSH_Y]
        and     bx, 7
        shl     bx, 3
        add     si, bx
        mov     ebx, es:[si][BRUSH_COLOR]
        mov     edx, es:[si + 4][BRUSH_COLOR]
;
; Calculate nearest destination brush boundary
;
        mov     cx, xPat
        mov     ax, di
        and     ax, 3
        mov     si, ax
        sub     cx, ax                  ; ax = aligned pattern destintion address
        jz      AlignFirst
;
; Calculate which half of the brush is first to be drawn
;
        test    cl, 4
        jz      @F
        xchg    ebx, edx
        and     cx, 3
        jz      AlignFirst
;
; Align brush scanline in ebx:esi to nearest DWORD address
;
@@:     test    cl, 2
        jz      @F
        xchg    bx, dx
        ror     ebx, 16
        ror     edx, 16
@@:     test    cl, 1
        jz      AlignFirst
        xchg    bl, dl
        ror     ebx, 8
        ror     edx, 8
;
; Align first (partial) DWORD of brush to destination BYTE.
;
AlignFirst:
        mov     eax, ebx
        mov     cx, si
        shl     cl, 3
        ror     eax, cl
        mov     cx, xSize
        not     ebx
        not     edx
        not     eax
PatAlign:
        or      [di], al
        inc     di
        test    di, 3
        jz      @F
        shr     eax, 8
        dec     cx
        jnz     PatAlign
        jmp     NextScan
@@:     dec     cx
        jz      NextScan
        sub     cx, 4
        jbe     @F
PatLoop4:
        mov     eax, edx
        or      [di], eax
        xchg    ebx, edx
        add     di, 4
        sub     cx, 4
        ja      PatLoop4
        mov     eax, edx
@@:     add     cx, 4
PatLoop1:
        or      [di], al
        shr     eax, 8
        inc     di
        dec     cx
        jnz     PatLoop1
NextScan:
;
; Decrement scanline count.
;
        dec     WORD PTR ySize
        jz      Exit
;
; Increment current scanline pointer.
;
        mov     di, pDstScan
        bmScanInc di, ds, cDstScanSeg, WORD PTR dyDst, WORD PTR cScan, cDstFillBytes
        mov     pDstScan, di
;
; Increment current y coordinate for indexing into brush
; and get brush phase alignment.
;
        mov     bx, yDst
        inc     bx
        mov     yDst, bx
        jmp     ScanLoop
Exit:
        pop     di
        pop     si
        pop     es
        pop     ds
        leave
        ret     26
mempBltPD88rB      ENDP

;----------------------------------------------------------------------------;
;                                                                            ;
;                  Pattern BLT, 8BPP, rop = C                                ;
;                                                                            ;
;----------------------------------------------------------------------------;

mempBltPD88rC      PROC NEAR
        mov     si, pPatOffset
        test    WORD PTR es:[si][BRUSH_FLAGS], 4        ; 4 = Solid brush
        jz      @F
        mov     al, es:[si][BRUSH_CLR_SOLID]
        mov     ah, al
        mov     bx, ax
        shl     eax, 16
        mov     ax, bx
        jmp     mempBltSolid
;
; Preload bx for ScanLoop.
;
@@:     mov     bx, yDst
;
; Main scanline blt loop.
;
ScanLoop:
;
; Get current scanline of brush into ebx:esi (8 pixels worth).
;
        mov     si, pPatOffset
        sub     bx, es:[si][BRUSH_Y]
        and     bx, 7
        shl     bx, 3
        add     si, bx
        mov     ebx, es:[si][BRUSH_COLOR]
        mov     edx, es:[si + 4][BRUSH_COLOR]
;
; Calculate nearest destination brush boundary
;
        mov     cx, xPat
        mov     ax, di
        and     ax, 3
        mov     si, ax
        sub     cx, ax                  ; ax = aligned pattern destintion address
        jz      AlignFirst
;
; Calculate which half of the brush is first to be drawn
;
        test    cl, 4
        jz      @F
        xchg    ebx, edx
        and     cx, 3
        jz      AlignFirst
;
; Align brush scanline in ebx:esi to nearest DWORD address
;
@@:     test    cl, 2
        jz      @F
        xchg    bx, dx
        ror     ebx, 16
        ror     edx, 16
@@:     test    cl, 1
        jz      AlignFirst
        xchg    bl, dl
        ror     ebx, 8
        ror     edx, 8
;
; Align first (partial) DWORD of brush to destination BYTE.
;
AlignFirst:
        mov     eax, ebx
        mov     cx, si
        shl     cl, 3
        ror     eax, cl
        mov     cx, xSize
AlignPat:
        mov     [di], al
        dec     cx
        jz      NextScan
        inc     di
        test    di, 3
        jz      @F
        shr     eax, 8
        mov     [di], al
        dec     cx
        jz      NextScan
        inc     di
        test    di, 3
        jz      @F
        shr     eax, 8
        mov     [di], al
        dec     cx
        jz      NextScan
        inc     di
        test    di, 3
        jz      @F
        shr     eax, 8
        mov     [di], al
        dec     cx
        jz      NextScan
        inc     di
@@:     sub     cx, 32
        js      @F
PatLoop32:
        mov     [di], edx
        mov     [di + 4], ebx
        mov     [di + 8], edx
        mov     [di + 12], ebx
        mov     [di + 16], edx
        mov     [di + 20], ebx
        mov     [di + 24], edx
        mov     [di + 28], ebx
        jz      NextScan
        add     di, 32
        sub     cx, 32
        jae     PatLoop32
@@:     add     cx, 24                          ; 24 = 32 - 8
        js      @F
PatLoop8:
        mov     [di], edx
        mov     [di + 4], ebx
        jz      NextScan
        add     di, 8
        sub     cx, 8
        jae     PatLoop8
@@:     add     cx, 8
        mov     eax, edx
        cmp     cx, 4
        js      @F
        mov     [di], eax
        mov     eax, ebx
        add     di, 4
        sub     cx, 4
@@:     test    cx, 2
        jz      @F
        mov     [di], ax
        ror     eax, 16
        add     di, 2
@@:     test    cx, 1
        jz      NextScan
        mov     [di], al
NextScan:
;
; Decrement scanline count.
;
        dec     WORD PTR ySize
        jz      Exit
;
; Increment current scanline pointer.
;
        mov     di, pDstScan
        bmScanInc di, ds, cDstScanSeg, WORD PTR dyDst, WORD PTR cScan, cDstFillBytes
        mov     pDstScan, di
;
; Increment current y coordinate for indexing into brush
; and get brush phase alignment.
;
        mov     bx, yDst
        inc     bx
        mov     yDst, bx
        jmp     ScanLoop
Exit:
        pop     di
        pop     si
        pop     es
        pop     ds
        leave
        ret     26
mempBltPD88rC      ENDP

;----------------------------------------------------------------------------;
;                                                                            ;
;                  Pattern BLT, 8BPP, rop = D                                ;
;                                                                            ;
;----------------------------------------------------------------------------;

mempBltPD88rD      PROC NEAR
;
; Preload bx for ScanLoop.
;
        mov     bx, yDst
;
; Main scanline blt loop.
;
ScanLoop:
;
; Get current scanline of brush into ebx:esi (8 pixels worth).
;
        mov     si, pPatOffset
        sub     bx, es:[si][BRUSH_Y]
        and     bx, 7
        shl     bx, 3
        add     si, bx
        mov     ebx, es:[si][BRUSH_COLOR]
        mov     edx, es:[si + 4][BRUSH_COLOR]
;
; Calculate nearest destination brush boundary
;
        mov     cx, xPat
        mov     ax, di
        and     ax, 3
        mov     si, ax
        sub     cx, ax                  ; ax = aligned pattern destintion address
        jz      AlignFirst
;
; Calculate which half of the brush is first to be drawn
;
        test    cl, 4
        jz      @F
        xchg    ebx, edx
        and     cx, 3
        jz      AlignFirst
;
; Align brush scanline in ebx:esi to nearest DWORD address
;
@@:     test    cl, 2
        jz      @F
        xchg    bx, dx
        ror     ebx, 16
        ror     edx, 16
@@:     test    cl, 1
        jz      AlignFirst
        xchg    bl, dl
        ror     ebx, 8
        ror     edx, 8
;
; Align first (partial) DWORD of brush to destination BYTE.
;
AlignFirst:
        mov     eax, ebx
        mov     cx, si
        shl     cl, 3
        ror     eax, cl
        mov     cx, xSize
        push    ebx
PatAlign:
        mov     bl, [di]
        not     bl
        or      al, bl
        mov     [di], al
        inc     di
        test    di, 3
        jz      @F
        shr     eax, 8
        dec     cx
        jnz     PatAlign
        pop     ebx
        jmp     NextScan
@@:     pop     ebx
        dec     cx
        jz      NextScan
        sub     cx, 4
        jbe     @F
PatLoop4:
        mov     eax, edx
        mov     esi, [di]
        not     esi
        or      eax, esi
        mov     [di], eax
        xchg    ebx, edx
        add     di, 4
        sub     cx, 4
        ja      PatLoop4
        mov     eax, edx
@@:     add     cx, 4
PatLoop1:
        mov     bl, [di]
        not     bl
        or      al, bl
        mov     [di], al
        shr     eax, 8
        inc     di
        dec     cx
        jnz     PatLoop1
NextScan:
;
; Decrement scanline count.
;
        dec     WORD PTR ySize
        jz      Exit
;
; Increment current scanline pointer.
;
        mov     di, pDstScan
        bmScanInc di, ds, cDstScanSeg, WORD PTR dyDst, WORD PTR cScan, cDstFillBytes
        mov     pDstScan, di
;
; Increment current y coordinate for indexing into brush
; and get brush phase alignment.
;
        mov     bx, yDst
        inc     bx
        mov     yDst, bx
        jmp     ScanLoop
Exit:
        pop     di
        pop     si
        pop     es
        pop     ds
        leave
        ret     26
mempBltPD88rD      ENDP

;----------------------------------------------------------------------------;
;                                                                            ;
;                  Pattern BLT, 8BPP, rop = E                                ;
;                                                                            ;
;----------------------------------------------------------------------------;

mempBltPD88rE      PROC NEAR
;
; Preload bx for ScanLoop.
;
        mov     bx, yDst
;
; Main scanline blt loop.
;
ScanLoop:
;
; Get current scanline of brush into ebx:esi (8 pixels worth).
;
        mov     si, pPatOffset
        sub     bx, es:[si][BRUSH_Y]
        and     bx, 7
        shl     bx, 3
        add     si, bx
        mov     ebx, es:[si][BRUSH_COLOR]
        mov     edx, es:[si + 4][BRUSH_COLOR]
;
; Calculate nearest destination brush boundary
;
        mov     cx, xPat
        mov     ax, di
        and     ax, 3
        mov     si, ax
        sub     cx, ax                  ; ax = aligned pattern destintion address
        jz      AlignFirst
;
; Calculate which half of the brush is first to be drawn
;
        test    cl, 4
        jz      @F
        xchg    ebx, edx
        and     cx, 3
        jz      AlignFirst
;
; Align brush scanline in ebx:esi to nearest DWORD address
;
@@:     test    cl, 2
        jz      @F
        xchg    bx, dx
        ror     ebx, 16
        ror     edx, 16
@@:     test    cl, 1
        jz      AlignFirst
        xchg    bl, dl
        ror     ebx, 8
        ror     edx, 8
;
; Align first (partial) DWORD of brush to destination BYTE.
;
AlignFirst:
        mov     eax, ebx
        mov     cx, si
        shl     cl, 3
        ror     eax, cl
        mov     cx, xSize
PatAlign:
        or      [di], al
        inc     di
        test    di, 4
        jz      @F
        shr     eax, 8
        dec     cx
        jnz     PatAlign
        jmp     NextScan
@@:     dec     cx
        jz      NextScan
        sub     cx, 4
        jbe     @F
PatLoop4:
        mov     eax, edx
        or      [di], eax
        xchg    ebx, edx
        add     di, 4
        sub     cx, 4
        ja      PatLoop4
        mov     eax, edx
@@:     add     cx, 4
PatLoop1:
        or      [di], al
        shr     eax, 8
        inc     di
        dec     cx
        jnz     PatLoop1
NextScan:
;
; Decrement scanline count.
;
        dec     WORD PTR ySize
        jz      Exit
;
; Increment current scanline pointer.
;
        mov     di, pDstScan
        bmScanInc di, ds, cDstScanSeg, WORD PTR dyDst, WORD PTR cScan, cDstFillBytes
        mov     pDstScan, di
;
; Increment current y coordinate for indexing into brush
; and get brush phase alignment.
;
        mov     bx, yDst
        inc     bx
        mov     yDst, bx
        jmp     ScanLoop
Exit:
        pop     di
        pop     si
        pop     es
        pop     ds
        leave
        ret     26
mempBltPD88rE      ENDP

;----------------------------------------------------------------------------;
;                                                                            ;
;                  Pattern BLT, 8BPP, rop = F                                ;
;                                                                            ;
;----------------------------------------------------------------------------;

mempBltPD88rF      PROC NEAR
        xor     eax, eax
        not     eax
        jmp     mempBltSolid
mempBltPD88rF      ENDP

;----------------------------------------------------------------------------;
;                                                                            ;
;                  Solid BLT, 8BPP,   rop = C                                ;
;                                                                            ;
; This is a special case BLT called by ROPs 0, F, and C with a solid color.  ;
;                                                                            ;
;----------------------------------------------------------------------------;

mempBltSolid    PROC NEAR
        mov     clrSolid, eax
;
; Main scanline blt loop.
;
ScanLoop:
        mov     eax, clrSolid
        mov     cx, xSize
AlignPat:
        test    di, 3
        jz      @F
        mov     [di], al
        dec     cx
        jz      NextScan
        inc     di
        test    di, 3
        jz      @F
        mov     [di], al
        dec     cx
        jz      NextScan
        inc     di
        test    di, 3
        jz      @F
        mov     [di], al
        dec     cx
        jz      NextScan
        inc     di
        test    di, 3
        jz      @F
        mov     [di], al
        dec     cx
        jz      NextScan
        inc     di
@@:     sub     cx, 64
        js      @F
PatLoop64:
        mov     [di], eax
        mov     [di + 4], eax
        mov     [di + 8], eax
        mov     [di + 12], eax
        mov     [di + 16], eax
        mov     [di + 20], eax
        mov     [di + 24], eax
        mov     [di + 28], eax
        mov     [di + 32], eax
        mov     [di + 36], eax
        mov     [di + 40], eax
        mov     [di + 44], eax
        mov     [di + 48], eax
        mov     [di + 52], eax
        mov     [di + 56], eax
        mov     [di + 60], eax
        jz      NextScan
        add     di, 64
        sub     cx, 64
        jae     PatLoop64
@@:     add     cx, 56                          ; 56 = 64 - 8
        js      @F
PatLoop8:
        mov     [di], eax
        mov     [di + 4], eax
        jz      NextScan
        add     di, 8
        sub     cx, 8
        jae     PatLoop8
@@:     add     cx, 8
        cmp     cx, 4
        js      @F
        mov     [di], eax
        add     di, 4
        sub     cx, 4
@@:     test    cx, 2
        jz      @F
        mov     [di], ax
        add     di, 2
@@:     test    cx, 1
        jz      NextScan
        mov     [di], al
NextScan:
;
; Decrement scanline count.
;
        dec     WORD PTR ySize
        jz      Exit
;
; Increment current scanline pointer.
;
        mov     di, pDstScan
        bmScanInc di, ds, cDstScanSeg, WORD PTR dyDst, WORD PTR cScan, cDstFillBytes
        mov     pDstScan, di
        jmp     ScanLoop
Exit:
        pop     di
        pop     si
        pop     es
        pop     ds
        leave
        ret     26
mempBltSolid    ENDP


_TEXT           ENDS
                END
