/****************************************************************************\
*                                                                            *
* Function: BitBlt entrypoint                                                *
*                                                                            *
*       Direct control to either memory based BLT routines or device based   *
*       BLT routines.  Help device with BLTs that may be too difficut.       *
*                                                                            *
* Author:                                                                    *
*                                                                            *
*       Dave Schmenk                                                         *
*                                                                            *
* History:                                                                   *
*                                                                            *
*       05/11/93 Dave Schmenk - wrote it.                                    *
*                                                                            *
******************************************************************************
*                                                                            *
*                     Copyright 1992 pellucid, inc.                          *
*                                                                            *
\****************************************************************************/

#include "common.h"
#include "bitblt.h"

/****************************************************************************\
*                                                                            *
* ROP3 to ROP2 translation table                                             *
*                                                                            *
* Table to translate ternary rops to binary raster ops. Ternary              *
* rops that can't be translated to have bit 4 set to one. If the ROP3        *
* involves a source bitmap then bit 7 is set.  If the ROP3 involves a        *
* pattern brush,  then bit 6 is set to one.  ROPs that only involve the      *
* destination have bit 5 set to one.                                         *
*                                                                            *
*                                                                            *
* ROP4   #    -->     MIX             #                                      *
* ----  --            -------------- --                                      *
* 0     00            R2_BLACK       20                                      *
* DPon  05            R2_NOTMERGEPEN 41                                      *
* DPna  0A            R2_MASKNOTPEN  42                                      *
* Pn    0F            R2_NOTCOPYPEN  43                                      *
* DSon  11            R2_NOTMERGEPEN 81                                      *
* DSna  22            R2_MASKNOTPEN  82                                      *
* Sn    33            R2_NOTCOPYPEN  83                                      *
* SDna  44            R2_MASKPENNOT  84                                      *
* PDna  50            R2_MASKPENNOT  44                                      *
* Dn    55            R2_NOT         25                                      *
* DPx   5A            R2_XORPEN      46                                      *
* DPan  5F            R2_NOTMASKPEN  47                                      *
* DSx   66            R2_XORPEN      86                                      *
* DSan  77            R2_NOTMASKPEN  87                                      *
* DSa   88            R2_MASKPEN     88                                      *
* DSxn  99            R2_NOTXORPEN   89                                      *
* DPa   A0            R2_MASKPEN     48                                      *
* PDxn  A5            R2_NOTXORPEN   49                                      *
* D     AA            R2_NOP         2A                                      *
* DPno  AF            R2_MERGENOTPEN 4B                                      *
* DSno  BB            R2_MERGENOTPEN 8B                                      *
* S     CC            R2_COPYPEN     8C                                      *
* SDno  DD            R2_MERGEPENNOT 8D                                      *
* DSo   EE            R2_MERGEPEN    8E                                      *
* P     F0            R2_COPYPEN     4C                                      *
* PDno  F5            R2_MERGEPENNOT 4D                                      *
* DPo   FA            R2_MERGEPEN    4E                                      *
* 1     FF            R2_WHITE       3F                                      *
*                                                                            *
* History:                                                                   *
*                                                                            *
*  19/19/92 Dave Schmenk - Wrote it.                                         *
*                                                                            *
\****************************************************************************/

static BYTE xlateRop3Tbl[256] =
{
    0x00, 0x10, 0x10, 0x21, 0x10, 0x41, 0x10, 0x10,     // 0x00
    0x10, 0x10, 0x42, 0x10, 0x22, 0x10, 0x10, 0x43,     // 0x08
    0x10, 0x81, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,     // 0x10
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,     // 0x18
    0x10, 0x10, 0x82, 0x10, 0x10, 0x10, 0x10, 0x10,     // 0x20
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,     // 0x28
    0x24, 0x10, 0x10, 0x83, 0x10, 0x10, 0x10, 0x10,     // 0x30
    0x10, 0x10, 0x10, 0x10, 0x26, 0x10, 0x10, 0x27,     // 0x38
    0x10, 0x10, 0x10, 0x10, 0x84, 0x10, 0x10, 0x10,     // 0x40
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,     // 0x48
    0x44, 0x10, 0x10, 0x10, 0x10, 0x05, 0x10, 0x10,     // 0x50
    0x10, 0x10, 0x46, 0x10, 0x10, 0x10, 0x10, 0x47,     // 0x58
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x86, 0x10,     // 0x60
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,     // 0x68
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x87,     // 0x70
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,     // 0x78
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,     // 0x80
    0x88, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,     // 0x88
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,     // 0x90
    0x10, 0x89, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,     // 0x98
    0x48, 0x10, 0x10, 0x10, 0x10, 0x49, 0x10, 0x10,     // 0xA0
    0x10, 0x10, 0x0A, 0x10, 0x10, 0x10, 0x10, 0x4B,     // 0xA8
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,     // 0xB0
    0x10, 0x10, 0x10, 0x8B, 0x10, 0x10, 0x10, 0x10,     // 0xB8
    0x28, 0x10, 0x10, 0x29, 0x10, 0x10, 0x10, 0x10,     // 0xC0
    0x10, 0x10, 0x10, 0x10, 0x8C, 0x10, 0x10, 0x2B,     // 0xC8
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,     // 0xD0
    0x10, 0x10, 0x10, 0x10, 0x10, 0x8D, 0x10, 0x10,     // 0xD8
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,     // 0xE0
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x8E, 0x10,     // 0xE8
    0x4C, 0x10, 0x10, 0x2D, 0x10, 0x4D, 0x10, 0x10,     // 0xF0
    0x10, 0x10, 0x4E, 0x10, 0x2E, 0x10, 0x10, 0x0F      // 0xF8
};

BOOL WINAPI BitBlt
(
    LPBITMAP   lpDst,
    SHORT      wDstX,
    SHORT      wDstY,
    LPBITMAP   lpSrc,
    SHORT      wSrcX,
    SHORT      wSrcY,
    SHORT      wXext,
    SHORT      wYext,
    DWORD      Rop3,
    LPBRUSH    lpBrush,
    LPDRAWMODE lpDrawMode
)
{
    LPBITMAP lpTmpBitmap;
    DWORD    clrFore;
    DWORD    clrBack;
    WORD     bltCode;
    WORD     bltRop3;
    SHORT    mode;

    DBGMSG("BitBlt\n\r");

    //
    // Convert Rop3 into flags and possible Rop2 encoded into a WORD.
    //

    bltRop3 = (WORD)(Rop3 >> 16) & 0xFF;
    bltCode = xlateRop3Tbl[(BYTE)bltRop3];

    if (lpDst->bmType != 0)
    {
        //
        // Destination is screen.
        //

        if (!(bltCode & 0xF0))
        {
            //
            // Special destination BLT only.
            //

            if (bltCode == 0x0F)
            {
                clrFore = 0xFFFFFFFFL;
                bltCode = 0x0C;
            }
            else if (bltCode == 0x00)
            {
                clrFore = 0x0L;
                bltCode = 0x0C;
            }
            if (bltCode != 0x0A)
            {
                devBltPD_Solid(wDstX,
                               wDstY,
                               wXext,
                               wYext,
                               clrFore,
                               bltCode);
            }
        }
        else if (bltCode & BLT_PD)
        {
            //
            // BLT involves pattern and destination only.
            //

            if (lpBrush->brFlags & BRUSH_SOLID)
            {
                devBltPD_Solid(wDstX,
                               wDstY,
                               wXext,
                               wYext,
                               lpBrush->brClrSolid,
                               bltCode & 0x0F);
            }
            else if (lpBrush->brFlags & BRUSH_COLOR)
            {
                devBltPD_Color(wDstX,
                               wDstY,
                               lpBrush,
                               wXext,
                               wYext,
                               bltCode & 0x0F);
            }
            else if (lpBrush->brFlags != BRUSH_HOLLOW)
            {
                if (lpBrush->brFlags & BRUSH_MONO)
                {
                    //
                    // Fill in foreground and background colors from
                    // DRAWMODE structure for monochrome pattern brushes.
                    //

                    clrFore = lpDrawMode->TextColor;
                    clrBack = lpDrawMode->bkColor;
                    mode    = OPAQUE;
                }
                else
                {
                    //
                    // This is a HATCH brush, grab the background mode.
                    //
                    clrFore = lpBrush->brClrFore;
                    clrBack = lpBrush->brClrBack;
                    mode    = lpDrawMode->bkMode;
                }

                devBltPD_Mono(wDstX,
                              wDstY,
                              lpBrush,
                              wXext,
                              wYext,
                              clrFore,
                              clrBack,
                              mode,
                              bltCode & 0x0F);
            }
        }
        else
        {
            //
            // BLT involves source, possible pattern, and destination.
            //

            //
            // Clip to source dimensions.
            //

            if (wSrcX < 0)
            {
                wXext += wSrcX;
                wDstX -= wSrcX;
                wSrcX  = 0;

                //
                // Check if clipped away.
                //

                if ((wXext <= 0) || (wDstX < 0))
                    return(TRUE);
            }

            if ((wSrcX + wXext) > (SHORT)lpSrc->bmWidth)
            {
                wXext = (SHORT)lpSrc->bmWidth - wSrcX;

                //
                // Check if clipped away.
                //

                if (wXext <= 0)
                    return(TRUE);
            }

            if (wSrcY < 0)
            {
                wYext += wSrcY;
                wDstY -= wSrcY;
                wSrcY  = 0;

                //
                // Check if clipped away.
                //

                if ((wYext <= 0) || (wDstY < 0))
                    return(TRUE);
            }

            if ((wSrcY + wYext) > (SHORT)lpSrc->bmHeight)
            {
                wYext = (SHORT)lpSrc->bmHeight - wSrcY;

                //
                // Check if clipped away.
                //

                if (wYext <= 0)
                    return(TRUE);
            }

            if (bltCode & BLT_SD)
            {
                //
                // Only source and destination are required for this BLT.
                //

                if (lpSrc->bmType == 0)
                {
                    //
                    // Source is bitmap.
                    //

                    if (lpSrc->bmBitsPixel == 1)
                    {
                        //
                        // BLT monochrome bitmap to screen.
                        //

                        devBltSD_Mono(wDstX,
                                      wDstY,
                                      lpSrc->bmBits,
                                      lpSrc->bmWidthBytes,
                                      lpSrc->bmSegmentIndex ? lpSrc->bmScanSegment : 0x7FFF,
                                      lpSrc->bmFillBytes,
                                      wSrcX,
                                      wSrcY,
                                      wXext,
                                      wYext,
                                      lpDrawMode->TextColor,
                                      lpDrawMode->bkColor,
                                      lpDrawMode->bkMode,
                                      bltCode & 0x0F);
                    }
                    else
                    {
                        //
                        // BLT color bitmap to screen.
                        //

                        devBltSD_Color(wDstX,
                                       wDstY,
                                       lpSrc->bmBits,
                                       lpSrc->bmWidthBytes,
                                       lpSrc->bmSegmentIndex ? lpSrc->bmScanSegment : 0x7FFF,
                                       lpSrc->bmFillBytes,
                                       wSrcX,
                                       wSrcY,
                                       wXext,
                                       wYext,
                                       lpDrawMode->bkColor,
                                       lpDrawMode->bkMode,
                                       bltCode & 0x0F);
                    }
                }
                else
                {
                    //
                    // Source is device.  Use screen to screen BLT to move
                    // bits.
                    //

                    devBltSD_DevToDev(wDstX,
                                      wDstY,
                                      wSrcX,
                                      wSrcY,
                                      wXext,
                                      wYext,
                                      lpDrawMode->bkColor,
                                      lpDrawMode->bkMode,
                                      bltCode & 0x0F);
                }
            }
            else
            {
                //
                // If source, pattern, and destination needed for BLT, use
                // temporary memory bitmap to copy device bits into 
                // and call BitBlt again.  Copy final bits back to screen.
                //
#if 0
                if (lpSrc->bmType == 0)
                {
                    //
                    // Source is bitmap.
                    //

                    if (lpSrc->bmBitsPixel == 1)
                    {
                        //
                        // BLT monochrome pattern & bitmap to screen.
                        //

                        devBltPSD_Mono(wDstX,
                                       wDstY,
                                       lpSrc->bmBits,
                                       lpSrc->bmWidthBytes,
                                       lpSrc->bmSegmentIndex ? lpSrc->bmScanSegment : 0x7FFF,
                                       lpSrc->bmFillBytes,
                                       wSrcX,
                                       wSrcY,
                                       lpBrush,
                                       wXext,
                                       wYext,
                                       lpDrawMode->TextColor,
                                       lpDrawMode->bkColor,
                                       lpDrawMode->bkMode,
                                       bltRop3);
                    }
                    else
                    {
                        //
                        // BLT color pattern & bitmap to screen.
                        //

                        devBltPSD_Color(wDstX,
                                        wDstY,
                                        lpSrc->bmBits,
                                        lpSrc->bmWidthBytes,
                                        lpSrc->bmSegmentIndex ? lpSrc->bmScanSegment : 0x7FFF,
                                        lpSrc->bmFillBytes,
                                        wSrcX,
                                        wSrcY,
                                        lpBrush,
                                        wXext,
                                        wYext,
                                        lpDrawMode->bkColor,
                                        lpDrawMode->bkMode,
                                        bltRop3);
                    }
                }
                else
#endif
                {
                    //
                    // Hardware may not have been capable of this BLT.
                    // break it down into simpler terms.
                    //

                    if (bltCode & BLT_PS)
                    {
                        //
                        // Blt source first.
                        //

                        if (lpSrc->bmType == 0)
                        {
                            //
                            // Source is bitmap.
                            //

                            if (lpSrc->bmBitsPixel == 1)
                            {
                                //
                                // BLT monochrome bitmap to screen.
                                //

                                devBltSD_Mono(wDstX,
                                            wDstY,
                                            lpSrc->bmBits,
                                            lpSrc->bmWidthBytes,
                                            lpSrc->bmSegmentIndex ? lpSrc->bmScanSegment : 0x7FFF,
                                            lpSrc->bmFillBytes,
                                            wSrcX,
                                            wSrcY,
                                            wXext,
                                            wYext,
                                            lpDrawMode->TextColor,
                                            lpDrawMode->bkColor,
                                            lpDrawMode->bkMode,
                                            0x0C); // R2_COPYPEN
                            }
                            else
                            {
                                //
                                // BLT color bitmap to screen.
                                //

                                devBltSD_Color(wDstX,
                                            wDstY,
                                            lpSrc->bmBits,
                                            lpSrc->bmWidthBytes,
                                            lpSrc->bmSegmentIndex ? lpSrc->bmScanSegment : 0x7FFF,
                                            lpSrc->bmFillBytes,
                                            wSrcX,
                                            wSrcY,
                                            wXext,
                                            wYext,
                                            lpDrawMode->bkColor,
                                            lpDrawMode->bkMode,
                                            0x0C); // R2_COPYPEN
                            }
                        }
                        else
                        {
                            //
                            // Source is device.  Use screen to screen BLT to move
                            // bits.
                            //

                            devBltSD_DevToDev(wDstX,
                                              wDstY,
                                              wSrcX,
                                              wSrcY,
                                              wXext,
                                              wYext,
                                              lpDrawMode->bkColor,
                                              lpDrawMode->bkMode,
                                              0x0C); // R2_COPYPEN
                        }

                        //
                        // Blt pattern on top of source.
                        //

                        if (lpBrush->brFlags & BRUSH_SOLID)
                        {
                            devBltPD_Solid(wDstX,
                                           wDstY,
                                           wXext,
                                           wYext,
                                           lpBrush->brClrSolid,
                                           bltCode & 0x0F);
                        }
                        else if (lpBrush->brFlags & BRUSH_COLOR)
                        {
                            devBltPD_Color(wDstX,
                                           wDstY,
                                           lpBrush,
                                           wXext,
                                           wYext,
                                           bltCode & 0x0F);
                        }
                        else
                        {
                            if (lpBrush->brFlags & BRUSH_MONO)
                            {
                                //
                                // Fill in foreground and background colors from
                                // DRAWMODE structure for monochrome pattern brushes.
                                //

                                clrFore = lpDrawMode->TextColor;
                                clrBack = lpDrawMode->bkColor;
                                mode    = OPAQUE;
                            }
                            else
                            {
                                //
                                // This is a HATCH brush, grab the background mode.
                                //
                                clrFore = lpBrush->brClrFore;
                                clrBack = lpBrush->brClrBack;
                                mode    = lpDrawMode->bkMode;
                            }

                            devBltPD_Mono(wDstX,
                                          wDstY,
                                          lpBrush,
                                          wXext,
                                          wYext,
                                          clrFore,
                                          clrBack,
                                          mode,
                                          bltCode & 0x0F);
                        }
                    }
                    else
                    {
                        //
                        // Source is screen, use temporary bitmap.
                        //

                        lpTmpBitmap = CreateColorBitmap(wXext, wYext);

                        devBltSD_ToColor(lpTmpBitmap->bmBits,
                                         lpTmpBitmap->bmWidthBytes,
                                         lpTmpBitmap->bmSegmentIndex ? lpTmpBitmap->bmScanSegment : 0x7FFF,
                                         lpTmpBitmap->bmFillBytes,
                                         0,
                                         0,
                                         wDstX,
                                         wDstY,
                                         wXext,
                                         wYext);

                        BitBlt(lpTmpBitmap,
                            0,
                            0,
                            lpSrc,
                            wSrcX,
                            wSrcY,
                            wXext,
                            wYext,
                            Rop3,
                            lpBrush,
                            lpDrawMode);

                        devBltSD_Color(wDstX,
                                       wDstY,
                                       lpTmpBitmap->bmBits,
                                       lpTmpBitmap->bmWidthBytes,
                                       lpTmpBitmap->bmSegmentIndex ? lpTmpBitmap->bmScanSegment : 0x7FFF,
                                       lpTmpBitmap->bmFillBytes,
                                       0,
                                       0,
                                       wXext,
                                       wYext,
                                       0L,
                                       OPAQUE,
                                       0x0C);

                        DestroyBitmap(lpTmpBitmap);
                    }
                }
            }
        }
    }
    else
    {
        //
        // Destination is bitmap.
        //

        if (!(bltCode & 0xF0))
        {
            if (lpBrush == NULL)
            {
                //
                // A destination only ROP may not need a brush so one will
                // be provided that doesn't incur any additional conversion
                // overhead.
                //

                lpBrush = memGetTmpBrush();
                lpBrush->brFlags = BRUSH_HATCHED | BRUSH_SOLID;
            }

            bltCode |= BLT_PD;
        }

        if (bltCode & BLT_PD)
        {
            //
            // BLT involves pattern and destination only.
            //

            if (lpDst->bmBitsPixel == 1)
            {
                //
                // Destination is monochrome.
                //

                if (lpBrush->brFlags & BRUSH_COLOR)
                {
                    //
                    // Convert color bits into a monochrome brush.
                    //

                    if ((lpBrush->brClrFore != lpDrawMode->TextColor) ||
                        (lpBrush->brClrBack != lpDrawMode->bkColor))
                    {
                        lpBrush->brClrFore = lpDrawMode->TextColor;
                        lpBrush->brClrBack = lpDrawMode->bkColor;
                        memConvertColorBrush(lpBrush);
                    }
                }
                else if (lpBrush->brFlags & BRUSH_MONO)
                {
                    if ((lpBrush->brClrFore != lpDrawMode->TextColor) ||
                        (lpBrush->brClrBack != lpDrawMode->bkColor))
                    {
                        lpBrush->brClrFore = lpDrawMode->TextColor;
                        lpBrush->brClrBack = lpDrawMode->bkColor;
                        memConvertMonoBrush(lpBrush);
                    }
                }
                memBltPD_ToMono(lpDst->bmBits,
                                lpDst->bmWidthBytes,
                                lpDst->bmSegmentIndex ? lpDst->bmScanSegment : 0x7FFF,
                                lpDst->bmFillBytes,
                                wDstX,
                                wDstY,
                                lpBrush,
                                wXext,
                                wYext,
                                lpDrawMode->bkMode,
                                bltCode & 0x0F);
            }
            else
            {
                //
                // Destination is color.
                //

                if (lpBrush->brFlags & BRUSH_MONO)
                {
                    //
                    // Convert monochrome bits into a color brush.
                    //

                    if ((lpBrush->brClrFore != lpDrawMode->TextColor) ||
                        (lpBrush->brClrBack != lpDrawMode->bkColor))
                    {
                        lpBrush->brClrFore = lpDrawMode->TextColor;
                        lpBrush->brClrBack = lpDrawMode->bkColor;
                        memConvertMonoBrush(lpBrush);
                    }
                }
                memBltPD_ToColor(lpDst->bmBits,
                                 lpDst->bmWidthBytes,
                                 lpDst->bmSegmentIndex ? lpDst->bmScanSegment : 0x7FFF,
                                 lpDst->bmFillBytes,
                                 wDstX,
                                 wDstY,
                                 lpBrush,
                                 wXext,
                                 wYext,
                                 lpDrawMode->bkMode,
                                 bltCode & 0x0F);
            }
        }
        else
        {
            //
            // BLT involves source, possible pattern, and destination.
            //

            //
            // Clip to source dimensions.
            //

            if (wSrcX < 0)
            {
                wXext += wSrcX;
                wDstX -= wSrcX;
                wSrcX  = 0;

                //
                // Check if clipped away.
                //

                if ((wXext <= 0) || (wDstX < 0))
                    return(TRUE);
            }

            if ((wSrcX + wXext) > (SHORT)lpSrc->bmWidth)
            {
                wXext = (SHORT)lpSrc->bmWidth - wSrcX;

                //
                // Check if clipped away.
                //

                if (wXext <= 0)
                    return(TRUE);
            }

            if (wSrcY < 0)
            {
                wYext += wSrcY;
                wDstY -= wSrcY;
                wSrcY  = 0;

                //
                // Check if clipped away.
                //

                if ((wYext <= 0) || (wDstY < 0))
                    return(TRUE);
            }

            if ((wSrcY + wYext) > (SHORT)lpSrc->bmHeight)
            {
                wYext = (SHORT)lpSrc->bmHeight - wSrcY;

                //
                // Check if clipped away.
                //

                if (wYext <= 0)
                    return(TRUE);
            }

            if (bltCode & BLT_PS)
            {
                //
                // Handle this ROP3 with two passes. Munge bltCode for two passes.
                //

                bltCode = (bltCode & 0x0F) << 8 | BLT_PS | BLT_SD | 0x0C;
            }
            if (bltCode & BLT_SD)
            {
                //
                // Only source and destination are required for this BLT.
                //

                if (lpSrc->bmType == 0)
                {
                    //
                    // Source is bitmap.
                    //

                    if (lpDst->bmBitsPixel == 1)
                    {
                        //
                        // Destination is monochrome.
                        //

                        lpTmpBitmap = lpSrc;

                        if (lpSrc->bmBitsPixel != 1)
                        {
                            //
                            // BLT color bitmap to monochrome bitmap.  Convert
                            // color bitmap to monochrome bitmap.
                            //
                                                 
                            lpSrc = CreateMonoBitmap(wXext, wYext);

                            memConvertColorBitmap(lpSrc->bmBits,
                                                  lpSrc->bmWidthBytes,
                                                  lpSrc->bmSegmentIndex ? lpSrc->bmScanSegment : 0x7FFF,
                                                  lpSrc->bmFillBytes,
                                                  lpTmpBitmap->bmBits,
                                                  lpTmpBitmap->bmWidthBytes,
                                                  lpTmpBitmap->bmSegmentIndex ? lpTmpBitmap->bmScanSegment : 0x7FFF,
                                                  lpTmpBitmap->bmFillBytes,
                                                  wSrcX,
                                                  wSrcY,
                                                  wXext,
                                                  wYext,
                                                  lpDrawMode->bkColor);
                            wSrcX = 0;
                            wSrcY = 0;
                        }

                        //
                        // BLT monochrome bitmap to monochrome bitmap.
                        //

                        memBltSD_MonoToMono(lpDst->bmBits,
                                            lpDst->bmWidthBytes,
                                            lpDst->bmSegmentIndex ? lpDst->bmScanSegment : 0x7FFF,
                                            lpDst->bmFillBytes,
                                            wDstX,
                                            wDstY,
                                            lpSrc->bmBits,
                                            lpSrc->bmWidthBytes,
                                            lpSrc->bmSegmentIndex ? lpSrc->bmScanSegment : 0x7FFF,
                                            lpSrc->bmFillBytes,
                                            wSrcX,
                                            wSrcY,
                                            wXext,
                                            wYext,
                                            bltCode & 0x0F);
                        if (lpTmpBitmap != lpSrc)
                        {
                            //
                            // Destroy temporary monochrome bitmap.
                            //

                            DestroyBitmap(lpSrc);
                        }
                    }
                    else
                    {

                        //
                        // Destination is color.
                        //

                        if (lpSrc->bmBitsPixel == 1)
                        {
                            //
                            // BLT monochrome bitmap to color bitmap.
                            //

                            memBltSD_MonoToColor(lpDst->bmBits,
                                                 lpDst->bmWidthBytes,
                                                 lpDst->bmSegmentIndex ? lpDst->bmScanSegment : 0x7FFF,
                                                 lpDst->bmFillBytes,
                                                 wDstX,
                                                 wDstY,
                                                 lpSrc->bmBits,
                                                 lpSrc->bmWidthBytes,
                                                 lpSrc->bmSegmentIndex ? lpSrc->bmScanSegment : 0x7FFF,
                                                 lpSrc->bmFillBytes,
                                                 wSrcX,
                                                 wSrcY,
                                                 wXext,
                                                 wYext,
                                                 lpDrawMode->TextColor,
                                                 lpDrawMode->bkColor,
                                                 bltCode & 0x0F);
                        }
                        else
                        {
                            //
                            // BLT color bitmap to color bitmap.
                            //

                            memBltSD_ColorToColor(lpDst->bmBits,
                                                  lpDst->bmWidthBytes,
                                                  lpDst->bmSegmentIndex ? lpDst->bmScanSegment : 0x7FFF,
                                                  lpDst->bmFillBytes,
                                                  wDstX,
                                                  wDstY,
                                                  lpSrc->bmBits,
                                                  lpSrc->bmWidthBytes,
                                                  lpSrc->bmSegmentIndex ? lpSrc->bmScanSegment : 0x7FFF,
                                                  lpSrc->bmFillBytes,
                                                  wSrcX,
                                                  wSrcY,
                                                  wXext,
                                                  wYext,
                                                  bltCode & 0x0F);
                        }
                    }
                }
                else
                {
                    //
                    // Source is screen.
                    //

                    if ((lpDst->bmBitsPixel == 1) || (bltCode & 0x0F != 0x0C))
                    {
                        //
                        // This is a non srccopy BLT or the destination in monochrome.
                        // Time to punt.
                        //

                        lpTmpBitmap = CreateColorBitmap(wXext, wYext);

                        devBltSD_ToColor(lpTmpBitmap->bmBits,
                                         lpTmpBitmap->bmWidthBytes,
                                         lpTmpBitmap->bmSegmentIndex ? lpTmpBitmap->bmScanSegment : 0x7FFF,
                                         lpTmpBitmap->bmFillBytes,
                                         0,
                                         0,
                                         wSrcX,
                                         wSrcY,
                                         wXext,
                                         wYext);

                        BitBlt(lpDst,
                               wDstX,
                               wDstY,
                               lpTmpBitmap,
                               0,
                               0,
                               wXext,
                               wYext,
                               Rop3,
                               lpBrush,
                               lpDrawMode);

                        DestroyBitmap(lpTmpBitmap);
                    }
                    else
                    {
                        //
                        // A nice simple srccopy into a color bitmap.
                        //

                        devBltSD_ToColor(lpDst->bmBits,
                                         lpDst->bmWidthBytes,
                                         lpDst->bmSegmentIndex ? lpDst->bmScanSegment : 0x7FFF,
                                         lpDst->bmFillBytes,
                                         wDstX,
                                         wDstY,
                                         wSrcX,
                                         wSrcY,
                                         wXext,
                                         wYext);
                    }
                }
                if (bltCode & BLT_PS)
                {
                    //
                    // We are doing a two pass ROP so do pattern now.
                    //

                    if (lpDst->bmBitsPixel == 1)
                    {
                        //
                        // Destination is monochrome.
                        //

                        if (lpBrush->brFlags & BRUSH_COLOR)
                        {
                            //
                            // Convert color bits into a monochrome brush.
                            //

                            if ((lpBrush->brClrFore != lpDrawMode->TextColor) ||
                                (lpBrush->brClrBack != lpDrawMode->bkColor))
                            {
                                lpBrush->brClrFore = lpDrawMode->TextColor;
                                lpBrush->brClrBack = lpDrawMode->bkColor;
                                memConvertColorBrush(lpBrush);
                            }
                        }
                        else if (lpBrush->brFlags & BRUSH_MONO)
                        {
                            if ((lpBrush->brClrFore != lpDrawMode->TextColor) ||
                                (lpBrush->brClrBack != lpDrawMode->bkColor))
                            {
                                lpBrush->brClrFore = lpDrawMode->TextColor;
                                lpBrush->brClrBack = lpDrawMode->bkColor;
                                memConvertMonoBrush(lpBrush);
                            }
                        }
                        memBltPD_ToMono(lpDst->bmBits,
                                        lpDst->bmWidthBytes,
                                        lpDst->bmSegmentIndex ? lpDst->bmScanSegment : 0x7FFF,
                                        lpDst->bmFillBytes,
                                        wDstX,
                                        wDstY,
                                        lpBrush,
                                        wXext,
                                        wYext,
                                        lpDrawMode->bkMode,
                                        (bltCode >> 8) & 0x0F);
                    }
                    else
                    {
                        //
                        // Destination is color.
                        //

                        if (lpBrush->brFlags & BRUSH_MONO)
                        {
                            //
                            // Convert monochrome bits into a color brush.
                            //

                            if ((lpBrush->brClrFore != lpDrawMode->TextColor) ||
                                (lpBrush->brClrBack != lpDrawMode->bkColor))
                            {
                                lpBrush->brClrFore = lpDrawMode->TextColor;
                                lpBrush->brClrBack = lpDrawMode->bkColor;
                                memConvertMonoBrush(lpBrush);
                            }
                        }
                        memBltPD_ToColor(lpDst->bmBits,
                                        lpDst->bmWidthBytes,
                                        lpDst->bmSegmentIndex ? lpDst->bmScanSegment : 0x7FFF,
                                        lpDst->bmFillBytes,
                                        wDstX,
                                        wDstY,
                                        lpBrush,
                                        wXext,
                                        wYext,
                                        lpDrawMode->bkMode,
                                        (bltCode >> 8) & 0x0F);
                    }
                }
            }
            else
            {
                //
                // BLT involves source, pattern, and destination.
                //

                lpTmpBitmap = lpSrc;

                if (lpSrc->bmType != 0)
                {
                    //
                    // If source is screen, create a temporary bitmap
                    // and copy its bits into it.
                    //

                    lpSrc = CreateColorBitmap(wXext, wYext);

                    devBltSD_ToColor(lpSrc->bmBits,
                                     lpSrc->bmWidthBytes,
                                     lpSrc->bmSegmentIndex ? lpSrc->bmScanSegment : 0x7FFF,
                                     lpSrc->bmFillBytes,
                                     0,
                                     0,
                                     wSrcX,
                                     wSrcY,
                                     wXext,
                                     wYext);
                    wSrcX = 0;
                    wSrcY = 0;
                }

                if (lpDst->bmBitsPixel == 1)
                {
                    //
                    // Destination is monochrome.
                    //

                    if (lpBrush->brFlags & BRUSH_COLOR)
                    {
                        //
                        // Convert color bits into a monochrome brush.
                        //

                        if ((lpBrush->brClrFore != lpDrawMode->TextColor) ||
                            (lpBrush->brClrBack != lpDrawMode->bkColor))
                        {
                            lpBrush->brClrFore = lpDrawMode->TextColor;
                            lpBrush->brClrBack = lpDrawMode->bkColor;
                            memConvertColorBrush(lpBrush);
                        }
                    }
                    else if (lpBrush->brFlags & BRUSH_MONO)
                    {
                        if ((lpBrush->brClrFore != lpDrawMode->TextColor) ||
                            (lpBrush->brClrBack != lpDrawMode->bkColor))
                        {
                            lpBrush->brClrFore = lpDrawMode->TextColor;
                            lpBrush->brClrBack = lpDrawMode->bkColor;
                            memConvertMonoBrush(lpBrush);
                        }
                    }

                    if (lpSrc->bmBitsPixel != 1)
                    {
                        //
                        // BLT color bitmap to monochrome bitmap.  Convert
                        // color bitmap to monochrome bitmap.
                        //
                                                
                        lpSrc = CreateMonoBitmap(wXext, wYext);

                        memConvertColorBitmap(lpSrc->bmBits,
                                              lpSrc->bmWidthBytes,
                                              lpSrc->bmSegmentIndex ? lpSrc->bmScanSegment : 0x7FFF,
                                              lpSrc->bmFillBytes,
                                              lpTmpBitmap->bmBits,
                                              lpTmpBitmap->bmWidthBytes,
                                              lpTmpBitmap->bmSegmentIndex ? lpTmpBitmap->bmScanSegment : 0x7FFF,
                                              lpTmpBitmap->bmFillBytes,
                                              wSrcX,
                                              wSrcY,
                                              wXext,
                                              wYext,
                                              lpDrawMode->bkColor);
                        wSrcX = 0;
                        wSrcY = 0;
                    }

                    //
                    // BLT monochrome bitmap to monochrome bitmap.
                    //

                    memBltPSD_MonoToMono(lpDst->bmBits,
                                         lpDst->bmWidthBytes,
                                         lpDst->bmSegmentIndex ? lpDst->bmScanSegment : 0x7FFF,
                                         lpDst->bmFillBytes,
                                         wDstX,
                                         wDstY,
                                         lpSrc->bmBits,
                                         lpSrc->bmWidthBytes,
                                         lpSrc->bmSegmentIndex ? lpSrc->bmScanSegment : 0x7FFF,
                                         lpSrc->bmFillBytes,
                                         wSrcX,
                                         wSrcY,
                                         lpBrush,
                                         wXext,
                                         wYext,
                                         bltRop3);

                }
                else
                {
                    //
                    // Destination is color.
                    //

                    if (lpBrush->brFlags & BRUSH_MONO)
                    {
                        //
                        // Convert monochrome bits into a color brush.
                        //

                        if ((lpBrush->brClrFore != lpDrawMode->TextColor) ||
                            (lpBrush->brClrBack != lpDrawMode->bkColor))
                        {
                            lpBrush->brClrFore = lpDrawMode->TextColor;
                            lpBrush->brClrBack = lpDrawMode->bkColor;
                            memConvertMonoBrush(lpBrush);
                        }
                    }

                    if (lpSrc->bmBitsPixel == 1)
                    {
                        //
                        // BLT monochrome bitmap to color bitmap.
                        //

                        memBltPSD_MonoToColor(lpDst->bmBits,
                                              lpDst->bmWidthBytes,
                                              lpDst->bmSegmentIndex ? lpDst->bmScanSegment : 0x7FFF,
                                              lpDst->bmFillBytes,
                                              wDstX,
                                              wDstY,
                                              lpSrc->bmBits,
                                              lpSrc->bmWidthBytes,
                                              lpSrc->bmSegmentIndex ? lpSrc->bmScanSegment : 0x7FFF,
                                              lpSrc->bmFillBytes,
                                              wSrcX,
                                              wSrcY,
                                              lpBrush,
                                              wXext,
                                              wYext,
                                              lpDrawMode->TextColor,
                                              lpDrawMode->bkColor,
                                              bltRop3);

                    }
                    else
                    {
                        //
                        // BLT color bitmap to color bitmap.
                        //

                        memBltPSD_ColorToColor(lpDst->bmBits,
                                               lpDst->bmWidthBytes,
                                               lpDst->bmSegmentIndex ? lpDst->bmScanSegment : 0x7FFF,
                                               lpDst->bmFillBytes,
                                               wDstX,
                                               wDstY,
                                               lpSrc->bmBits,
                                               lpSrc->bmWidthBytes,
                                               lpSrc->bmSegmentIndex ? lpSrc->bmScanSegment : 0x7FFF,
                                               lpSrc->bmFillBytes,
                                               wSrcX,
                                               wSrcY,
                                               lpBrush,
                                               wXext,
                                               wYext,
                                               bltRop3);
                    }
                }

                //
                // Check if temporary bitmap needs to be released.
                //

                if (lpTmpBitmap != lpSrc)
                {
                    //
                    // Delete temporary bitmap.
                    //

                    DestroyBitmap(lpSrc);
                }
            }
        }
    }

    return(TRUE);
}
