/****************************************************************************\
*                                                                            *
* Function: Bitmap support                                                   *
*                                                                            *
*       Memory bitmap support routines called by common code to create       *
*       and destroy temporary bitmaps.                                       *
*                                                                            *
* Author:                                                                    *
*                                                                            *
*       Dave Schmenk                                                         *
*                                                                            *
* History:                                                                   *
*                                                                            *
*       05/24/93 Dave Schmenk - wrote it.                                    *
*                                                                            *
******************************************************************************
*                                                                            *
*                     Copyright 1993 pellucid, inc.                          *
*                                                                            *
\****************************************************************************/

#include "common.h"

extern _AHIncr;
extern BYTE memBitsPerPixel;
extern WORD memBytesPerPixel;

BOOL     bTmpBitmapInUse = FALSE;
HANDLE   hTmpBitmap;

VOID FAR PASCAL AllocTmpBitmaps
(
    VOID
)
{
    //
    // Allocate and lock 64K for temporary bitmap structure.
    //

    hTmpBitmap = GlobalAlloc(0, 0x10000L);
}

VOID FAR PASCAL FreeTmpBitmaps
(
    VOID
)
{
    //
    // Free temporary bitmap structure.
    //

    GlobalFree(hTmpBitmap);
}

LPBITMAP PASCAL CreateColorBitmap
(
    SHORT    wXext,
    SHORT    wYext
)
{
    HANDLE   hMem;
    LPBITMAP lpcbm;
    USHORT   SizeOfScan;
    ULONG    SizeOfBitmap;
    USHORT   FullSegCount;
    USHORT   PartialSegCount;

    DBGMSG("CreateColorBitmap\n\r");

    //
    // Calculate pertinent size information
    //

    SizeOfScan      = (wXext * memBytesPerPixel + 1) & 0xFFFE;
    SizeOfBitmap    = (ULONG)SizeOfScan * (ULONG)wYext + sizeof(BITMAP);

    if (SizeOfBitmap < 0x10000L)
    {
        //
        // Bitmap <= 64K.
        //

        if (bTmpBitmapInUse)
        {
            //
            // Temporary bitmap is in use, must allocate a new one.
            //

            hMem  = GlobalAlloc(0, SizeOfBitmap);
        }
        else
        {
            //
            // Grab handle to temporary bitmap.
            //

            hMem  = hTmpBitmap;
            bTmpBitmapInUse = TRUE;
        }
        lpcbm = (LPBITMAP)GlobalLock(hMem);
        lpcbm->futureUse5     = 0;
        lpcbm->bmScanSegment  = 0;
        lpcbm->bmSegmentIndex = 0;
        lpcbm->bmFillBytes    = 0;
        lpcbm->bmBits         = (LPBYTE)lpcbm + sizeof(BITMAP);
    }
    else
    {
        //
        // Bitmap > 64K.
        //

        if (bTmpBitmapInUse)
        {
            //
            // Temporary bitmap is in use, must allocate a new one.
            //

            hMem  = GlobalAlloc(0, sizeof(BITMAP));
        }
        else
        {
            //
            // Grab handle to temporary bitmap.
            //

            hMem  = hTmpBitmap;
            bTmpBitmapInUse = TRUE;
        }
        lpcbm = (LPBITMAP)GlobalLock(hMem);
        lpcbm->bmScanSegment = (USHORT)0xFFFF / SizeOfScan;
        lpcbm->bmSegmentIndex = (USHORT)&_AHIncr;
        lpcbm->bmFillBytes    = (USHORT)0xFFFF - (lpcbm->bmScanSegment * SizeOfScan);
        lpcbm->bmFillBytes++;

        //
        // Allocate the bitmap.
        //

        FullSegCount          = wYext / lpcbm->bmScanSegment;
        PartialSegCount       = wYext - (FullSegCount * lpcbm->bmScanSegment);
        lpcbm->futureUse5     = GlobalAlloc(0, ((ULONG)FullSegCount << 16) + PartialSegCount * SizeOfScan);
        lpcbm->bmBits         = (LPBYTE)GlobalLock(lpcbm->futureUse5);
    }

    lpcbm->futureUse4    = hMem;
    lpcbm->bmType        = 0;
    lpcbm->bmHeight      = wYext;
    lpcbm->bmWidth       = wXext;
    lpcbm->bmPlanes      = 1;
    lpcbm->bmBitsPixel   = memBitsPerPixel;
    lpcbm->bmWidthPlanes = 0;
    lpcbm->bmlpPDevice   = (LPBYTE)NULL;
    lpcbm->bmWidthBytes  = SizeOfScan;
    return(lpcbm);
}

LPBITMAP PASCAL CreateMonoBitmap
(
    SHORT    wXext,
    SHORT    wYext
)
{
    HANDLE   hMem;
    LPBITMAP lpcbm;
    USHORT   SizeOfScan;
    ULONG    SizeOfBitmap;
    USHORT   FullSegCount;
    USHORT   PartialSegCount;

    DBGMSG("memCreateMonoBitmap\n\r");

    //
    // Calculate pertinent size information
    //

    SizeOfScan      = ((wXext + 15) >> 3) & 0xFFFE;
    SizeOfBitmap    = (ULONG)SizeOfScan * (ULONG)wYext + sizeof(BITMAP);

    if (SizeOfBitmap < 0x10000L)
    {
        //
        // Bitmap <= 64K.
        //

        if (bTmpBitmapInUse)
        {
            //
            // Temporary bitmap is in use, must allocate a new one.
            //

            hMem  = GlobalAlloc(0, SizeOfBitmap);
        }
        else
        {
            //
            // Grab handle to temporary bitmap.
            //

            hMem  = hTmpBitmap;
            bTmpBitmapInUse = TRUE;
        }
        lpcbm = (LPBITMAP)GlobalLock(hMem);
        lpcbm->futureUse5     = 0;
        lpcbm->bmScanSegment  = 0;
        lpcbm->bmSegmentIndex = 0;
        lpcbm->bmFillBytes    = 0;
        lpcbm->bmBits         = (LPBYTE)lpcbm + sizeof(BITMAP);
    }
    else
    {
        //
        // Bitmap > 64K.
        //

        if (bTmpBitmapInUse)
        {
            //
            // Temporary bitmap is in use, must allocate a new one.
            //

            hMem  = GlobalAlloc(0, sizeof(BITMAP));
        }
        else
        {
            //
            // Grab handle to temporary bitmap.
            //

            hMem  = hTmpBitmap;
            bTmpBitmapInUse = TRUE;
        }
        lpcbm = (LPBITMAP)GlobalLock(hMem);
        lpcbm->bmScanSegment = (USHORT)0xFFFF / SizeOfScan;
        lpcbm->bmSegmentIndex = (USHORT)&_AHIncr;
        lpcbm->bmFillBytes    = (USHORT)0xFFFF - (lpcbm->bmScanSegment * SizeOfScan);
        lpcbm->bmFillBytes++;

        //
        // Allocate the bitmap.
        //

        FullSegCount          = wYext / lpcbm->bmScanSegment;
        PartialSegCount       = wYext - (FullSegCount * lpcbm->bmScanSegment);
        lpcbm->futureUse5     = GlobalAlloc(0, ((ULONG)FullSegCount << 16) + PartialSegCount * SizeOfScan);
        lpcbm->bmBits         = (LPBYTE)GlobalLock(lpcbm->futureUse5);
    }

    lpcbm->futureUse4    = hMem;
    lpcbm->bmType        = 0;
    lpcbm->bmHeight      = wYext;
    lpcbm->bmWidth       = wXext;
    lpcbm->bmPlanes      = 1;
    lpcbm->bmBitsPixel   = 1;
    lpcbm->bmWidthPlanes = 0;
    lpcbm->bmlpPDevice   = (LPBYTE)NULL;
    lpcbm->bmWidthBytes  = SizeOfScan;
    return(lpcbm);
}

VOID PASCAL DestroyBitmap
(
    LPBITMAP lpBitmap
)
{
    HANDLE hBitmapMem;

    DBGMSG("memDestroyBitmap\n\r");

    //
    // Discard bit buffer if allocated.
    //

    if (lpBitmap->futureUse5)
    {
        GlobalUnlock(lpBitmap->futureUse5);
        GlobalFree(lpBitmap->futureUse5);
    }

    //
    // Always unlock bitmap structure.
    //

    hBitmapMem = lpBitmap->futureUse4;
    GlobalUnlock(hBitmapMem);

    //
    // Free bitmap structure.
    //

    if (hBitmapMem != hTmpBitmap)
    {
        GlobalFree(hBitmapMem);
    }
    else
    {
        bTmpBitmapInUse = FALSE;
    }
}

VOID PASCAL SetBitmapPixel
(
    LPBITMAP lpDst,
    SHORT    x,
    SHORT    y,
    DWORD    clr,
    USHORT   rop2
)
{
    DWORD SrcBits;

    DBGMSG("SetPixel\n\r");

    //
    // Check for monochrome or color destination.
    //

    if (lpDst->bmBitsPixel == 1)
    {
        SrcBits = devMonoMatchLogicalColor(clr);

        //
        // Set the pixel.
        //

        memBltSD_MonoToMono(lpDst->bmBits,
                            lpDst->bmWidthBytes,
                            lpDst->bmSegmentIndex ? lpDst->bmScanSegment : 0x7FFF,
                            lpDst->bmFillBytes,
                            x,
                            y,
                            (LPBYTE)&SrcBits,
                            1,
                            0x7FFF,
                            0,
                            0,
                            0,
                            1,
                            1,
                            rop2);
    }
    else
    {
        //
        // Set pixel treating source color as a 1X1 color bitmap.
        //

        memBltSD_ColorToColor(lpDst->bmBits,
                              lpDst->bmWidthBytes,
                              lpDst->bmSegmentIndex ? lpDst->bmScanSegment : 0x7FFF,
                              lpDst->bmFillBytes,
                              x,
                              y,
                              (LPBYTE)&clr,
                              memBytesPerPixel,
                              0x7FFF,
                              0,
                              0,
                              0,
                              1,
                              1,
                              rop2);
    }
}

DWORD PASCAL GetBitmapPixel
(
    LPBITMAP lpSrc,
    SHORT    x,
    SHORT    y
)
{
    DWORD clr;

    DBGMSG("GetPixel\n\r");

    clr = 0;

    //
    // Check for monochrome or color destination.
    //

    if (lpSrc->bmBitsPixel == 1)
    {
        //
        // Get the pixel treating color as a 1X1 mono bitmap.
        //

        memBltSD_MonoToMono((LPBYTE)&clr,
                            1,
                            0x7FFF,
                            0,
                            0,
                            0,
                            lpSrc->bmBits,
                            lpSrc->bmWidthBytes,
                            lpSrc->bmSegmentIndex ? lpSrc->bmScanSegment : 0x7FFF,
                            lpSrc->bmFillBytes,
                            x,
                            y,
                            1,
                            1,
                            0x0C);

        //
        // Expand monochrome pixel into device color.
        //

        if (clr)
            clr = 0x00FFFFFFL;
    }
    else
    {
        //
        // Get pixel treating color as a 1X1 color bitmap.
        //

        memBltSD_ColorToColor((LPBYTE)&clr,
                              memBytesPerPixel,
                              0x7FFF,
                              0,
                              0,
                              0,
                              lpSrc->bmBits,
                              lpSrc->bmWidthBytes,
                              lpSrc->bmSegmentIndex ? lpSrc->bmScanSegment : 0x7FFF,
                              lpSrc->bmFillBytes,
                              x,
                              y,
                              1,
                              1,
                              0x0C);
    }

    return((DWORD)clr | 0xFF000000L);
}
