/****************************************************************************\
*                                                                            *
* Function: Output entrypoint                                                *
*                                                                            *
*       Direct control to corretn Output routine.  Only lines, rectangles    *
*       and scanlines are implelented.                                       *
*                                                                            *
* Author:                                                                    *
*                                                                            *
*       Dave Schmenk                                                         *
*                                                                            *
* History:                                                                   *
*                                                                            *
*       05/11/93 Dave Schmenk - wrote it.                                    *
*                                                                            *
******************************************************************************
*                                                                            *
*                     Copyright 1992 pellucid, inc.                          *
*                                                                            *
\****************************************************************************/

#include "common.h"
#include "output.h"

static SCANPROC  pfnScan = NULL;
static LINEPROC  pfnLineSeg;
static LPBITMAP  lpOutputDst;
static LPBRUSH   lpOutputBrush;
static DWORD     dwOutputColor;
static DWORD     dwOutputBkColor;
static USHORT    ropOutput;
static USHORT    modeOutput;
static SHORT     yTop;
static SHORT     yBottom;
static SHORT     yOutput;
static SHORT     PolyScan[MAX_POLY_SCAN][MAX_SEG_SCAN * 2];

//
// Output fill routines.
//

BOOL PASCAL BeginScanLine(LPBITMAP, LPBRUSH, LPPEN, LPDRAWMODE);
BOOL PASCAL CalcPolygonInterior(LPPOINT, LPRECT, SHORT);
VOID PASCAL DevicePolyLine(LPPOINT, WORD, LPRECT);
VOID PASCAL BitmapPolyLine(LPPOINT, WORD, LPRECT);
VOID PASCAL BitmapClippedLine(SHORT, SHORT, SHORT, SHORT, LPRECT);
VOID PASCAL BitmapLine(SHORT, SHORT, SHORT, SHORT);
VOID PASCAL MonoBitmapScans(SHORT, SHORT, SHORT);
VOID PASCAL ColorBitmapScans(SHORT, SHORT, SHORT);
VOID PASCAL SolidDeviceScans(SHORT, SHORT, SHORT);
VOID PASCAL MonoDeviceScans(SHORT, SHORT, SHORT);
VOID PASCAL ColorDeviceScans(SHORT, SHORT, SHORT);
VOID PASCAL MonoBitmapSegment(SHORT, SHORT, SHORT, SHORT);
VOID PASCAL ColorBitmapSegment(SHORT, SHORT, SHORT, SHORT);

SHORT WINAPI Output
(
    LPBITMAP   lpDst,
    WORD       wStyle,
    WORD       wCount,
    LPPOINT    lpPoints,
    LPPEN      lpPen,
    LPBRUSH    lpBrush,
    LPDRAWMODE lpDrawMode,
    LPRECT     lpClipRect
)
{
    SHORT  start;
    SHORT  stop;
    SHORT  length;
    SHORT  yScan;
    SHORT  yExt;

    DBGMSG("Output\n\r");

    switch (wStyle)
    {
        case OS_SCANLINES:
            if (pfnScan != NULL)
            {
                yOutput = lpPoints++->ycoord;
                while (--wCount)
                {
                    //
                    // Fill scanline segment if wide enough.
                    //

                    start  = lpPoints->xcoord;
                    length = lpPoints->ycoord - start;
                    if (length > 0)
                    {
                        (*pfnScan)(start, length, 1);
                    }
                    lpPoints++;
                }
            }
            else
            {
                //
                // We are being called with scanlines without beginscan first.
                //

                if (!BeginScanLine(lpDst, lpBrush, lpPen, lpDrawMode))
                    return(-1);
                yOutput = lpPoints++->ycoord;
                while (--wCount)
                {
                    //
                    // Fill scanline segment if wide enough.
                    //
                    start  = lpPoints->xcoord;
                    length = lpPoints->ycoord - start;
                    if (length > 0)
                        (*pfnScan)(start, length, 1);
                    lpPoints++;
                }

                //
                // Flag this as a scanline without beginscan.
                //

                pfnScan = NULL;
            }
            return(0);

        case OS_POLYGON:
            //
            // Fill polygon and draw border.
            //

            if (lpBrush && !(lpBrush->brFlags & BRUSH_HOLLOW))
            {
                //
                // Fill interior of polygon.
                //

                if (!CalcPolygonInterior(lpPoints, lpClipRect, wCount))
                    return(-1);

                if (!BeginScanLine(lpDst, lpBrush, lpPen, lpDrawMode))
                    return(-1);

                yOutput = yTop;
                yScan   = 0;
//                while (yOutput <= yBottom)
                while (yOutput < yBottom)
                {
                    //
                    // Look for scanline similarity.
                    //

                    start = PolyScan[yScan][SCAN_LEFT];
                    stop  = PolyScan[yScan][SCAN_RIGHT];
                    yExt = 1;
                    while ((yOutput + yExt <= yBottom)
                       && (start == PolyScan[yScan + yExt][SCAN_LEFT])
                       && (stop  == PolyScan[yScan + yExt][SCAN_RIGHT]))
                    {
                        yExt++;
                    }

                    //
                    // Fill scanline segment if wide enough.
                    //

                    if (start < stop)
                        (*pfnScan)(start, stop - start, yExt);

                    //
                    // Increment to next scanline.
                    //

                    yOutput += yExt;
                    yScan += yExt;
                }

                //
                // Flag polygon function being complete.
                //

                pfnScan = NULL;
            }

            //
            // Fall into polyline code to draw border.
            //

        case OS_POLYLINE:
            //
            // Draw polylines.
            //

            if ((lpPen == NULL) || (lpPen->pnFlags == LS_NOLINE))
            {
                //
                // No lines to draw.
                //

                return(1);
            }

            ropOutput     = lpDrawMode->Rop2 - 1;
            dwOutputColor = lpPen->pnColor;

            if (lpDst->bmType != 0)
            {
                //
                // Drawing to device.
                //

                DevicePolyLine(lpPoints, wCount, lpClipRect);
            }
            else
            {
                //
                // Drawing to a memory bitmap.
                //

                lpOutputDst = lpDst;
                BitmapPolyLine(lpPoints, wCount, lpClipRect);
            }
            return(1);

        case OS_ENDNSCAN:
            //
            // Flag scanline function being complete.
            //

            pfnScan = NULL;
            return(0);

        case OS_BEGINNSCAN:

            if (BeginScanLine(lpDst, lpBrush, lpPen, lpDrawMode))
                return(0);
            else
                return(-1);
    }

    //
    // Return error so function will be emulated.
    //

    return(-1);
}

BOOL PASCAL BeginScanLine
(
    LPBITMAP   lpDst,
    LPBRUSH    lpBrush,
    LPPEN      lpPen,
    LPDRAWMODE lpDrawMode
)
{
    //
    // Save ROP just for fun.
    //

    if (lpBrush == NULL)
    {
        //
        // Get ROP and set OPAQUE brush.
        //

        ropOutput  = lpDrawMode->Rop2 - 1;
        modeOutput = OPAQUE;

        if (lpDst->bmType != 0)
        {
            //
            // Device solid color.
            //

            dwOutputColor = lpPen->pnColor;
            pfnScan       = SolidDeviceScans;
        }
        else 
        {
            //
            // Memory bitmap solid color.
            //

            lpOutputDst               = lpDst;
            lpOutputBrush             = memGetTmpBrush();
            lpOutputBrush->brFlags    = BRUSH_SOLID;
            lpOutputBrush->brClrSolid = dwOutputColor;
            lpOutputBrush->brClrFore  = dwOutputColor;
            lpOutputBrush->brClrBack  = dwOutputColor;
            dwOutputColor             = lpPen->pnColor;
            dwOutputBkColor           = lpPen->pnColor;
            lpOutputBrush->brX        = 0;
            lpOutputBrush->brY        = 0;
            memConvertMonoBrush(lpOutputBrush);
            if (lpDst->bmBitsPixel == 1)
            {
                pfnScan = MonoBitmapScans;
            }
            else
            {
                pfnScan = ColorBitmapScans;
            }
        }
    }
    else
    {
        //
        // Hollow brushes don't do anything.
        //

        if (lpBrush->brFlags & BRUSH_HOLLOW)
            return(FALSE);

        //
        // Get ROP and assume OPAQUE brush.
        //

        ropOutput  = lpDrawMode->Rop2 - 1;
        modeOutput = OPAQUE;

        if (lpDst->bmType != 0)
        {
            //
            // Device brush fill.
            //

            if (lpBrush->brFlags & BRUSH_SOLID)
            {
                dwOutputColor = lpBrush->brClrSolid;
                pfnScan       = SolidDeviceScans;
            }
            else if (lpBrush->brFlags & BRUSH_COLOR)
            {
                lpOutputBrush = lpBrush;
                pfnScan       = ColorDeviceScans;
            }
            else
            {
                lpOutputBrush = lpBrush;
                if (lpBrush->brFlags & BRUSH_PATTERN)
                {
                    //
                    // Fill in foreground and background colors from
                    // DRAWMODE structure for monochrome pattern brushes.
                    //

                    dwOutputColor   = lpDrawMode->TextColor;
                    dwOutputBkColor = lpDrawMode->bkColor;
                    if ((lpBrush->brClrFore != dwOutputColor) ||
                        (lpBrush->brClrBack != dwOutputBkColor))
                    {
                        lpBrush->brClrFore = dwOutputColor;
                        lpBrush->brClrBack = dwOutputBkColor;
                        memConvertMonoBrush(lpBrush);
                    }
                }
                else
                {
                    //
                    // HATCH patterns usr their own colors.
                    //

                    dwOutputColor   = lpBrush->brClrFore;
                    dwOutputBkColor = lpBrush->brClrBack;
                    modeOutput      = lpDrawMode->bkMode;
                }
                pfnScan = MonoDeviceScans;
            }
        }
        else
        {
            //
            // Memory bitmap brush fill.
            //

            lpOutputDst     = lpDst;
            lpOutputBrush   = lpBrush;
            if (lpBrush->brFlags == BRUSH_HATCHED)
            {
                //
                // HATCH patterns usr their own colors.
                //

                dwOutputColor   = lpBrush->brClrFore;
                dwOutputBkColor = lpBrush->brClrBack;
                modeOutput      = lpDrawMode->bkMode;
            }
            else
            {
                //
                // Otherwise DRAWMODE has colors.
                //

                dwOutputColor   = lpDrawMode->TextColor;
                dwOutputBkColor = lpDrawMode->bkColor;
            }
            if (lpDst->bmBitsPixel == 1)
            {
                pfnScan         = MonoBitmapScans;
                if (lpBrush->brFlags & BRUSH_COLOR)
                {
                    //
                    // Convert color bits into a monochrome brush.
                    //

                    if ((lpBrush->brClrFore != lpDrawMode->TextColor) ||
                        (lpBrush->brClrBack != lpDrawMode->bkColor))
                    {
                        lpBrush->brClrFore = lpDrawMode->TextColor;
                        lpBrush->brClrBack = lpDrawMode->bkColor;
                        memConvertColorBrush(lpBrush);
                    }
                }
                else if (lpBrush->brFlags & BRUSH_MONO)
                {
                    if ((lpBrush->brClrFore != lpDrawMode->TextColor) ||
                        (lpBrush->brClrBack != lpDrawMode->bkColor))
                    {
                        lpBrush->brClrFore = lpDrawMode->TextColor;
                        lpBrush->brClrBack = lpDrawMode->bkColor;
                        memConvertMonoBrush(lpBrush);
                    }
                }
            }
            else
            {
                pfnScan = ColorBitmapScans;
                if (lpBrush->brFlags & BRUSH_MONO)
                {
                    //
                    // Convert monochrome bits into a color brush.
                    //

                    if ((lpBrush->brClrFore != dwOutputColor) ||
                        (lpBrush->brClrBack != dwOutputBkColor))
                    {
                        lpBrush->brClrFore = dwOutputColor;
                        lpBrush->brClrBack = dwOutputBkColor;
                        memConvertMonoBrush(lpBrush);
                    }
                }
            }
        }
    }

    return(TRUE);
}

BOOL PASCAL CalcPolygonInterior
(
    LPPOINT lpPoints,
    LPRECT  lpClipRect,
    SHORT   count
)
{
    SHORT  cScans;
    SHORT  x1;
    SHORT  x2;
    SHORT  y1;
    SHORT  y2;
    SHORT  dx;
    SHORT  dy;
    SHORT  x;
    SHORT  y;
    SHORT  minDir;
    SHORT  ErrA;
    SHORT  ErrB;
    SHORT  Err;

    //
    // Get y extents of polygon.
    //

    yBottom = yTop = lpPoints[0].ycoord;

    for (x = 1; x < count; x++)
    {
        if (yTop > lpPoints[x].ycoord)
            yTop = lpPoints[x].ycoord;
        if (yBottom < lpPoints[x].ycoord)
            yBottom = lpPoints[x].ycoord;
    }

    cScans = yBottom - yTop + 1;

    //
    // Check for a polygon that's just too big.
    //

    if (cScans > MAX_POLY_SCAN)
        return(FALSE);

    //
    // Check for convex polygon using cross product.
    //

    x1 = lpPoints[1].xcoord - lpPoints[0].xcoord;
    y1 = lpPoints[1].ycoord - lpPoints[0].ycoord;
    x2 = lpPoints[2].xcoord - lpPoints[0].xcoord;
    y2 = lpPoints[2].ycoord - lpPoints[0].ycoord;

    if ((x1 * y2 - x2 * y1) >= 0)
    {
        //
        // cross product is >= 0 so make sure the rest are too.
        //

        for (x = 3; x < count; x++)
        {
            x1 = lpPoints[x - 1].xcoord - lpPoints[x - 2].xcoord;
            y1 = lpPoints[x - 1].ycoord - lpPoints[x - 2].ycoord;
            x2 = lpPoints[x].xcoord     - lpPoints[x - 2].xcoord;
            y2 = lpPoints[x].ycoord     - lpPoints[x - 2].ycoord;
            if ((x1 * y2 - x2 * y1) < 0)
            {
                //
                // This polygon is not convex.
                //

                return(FALSE);
            }
        }
    }
    else
    {
        //
        // cross product is < 0 so make sure the rest are too.
        //

        for (x = 3; x < count; x++)
        {
            x1 = lpPoints[x - 1].xcoord - lpPoints[x - 2].xcoord;
            y1 = lpPoints[x - 1].ycoord - lpPoints[x - 2].ycoord;
            x2 = lpPoints[x].xcoord     - lpPoints[x - 2].xcoord;
            y2 = lpPoints[x].ycoord     - lpPoints[x - 2].ycoord;
            if ((x1 * y2 - x2 * y1) > 0)
            {
                //
                // This polygon is not convex.
                //

                return(FALSE);
            }
        }
    }

    //
    // Initialize scan array.
    //

    for (y = 0; y < cScans; y++)
    {
        PolyScan[y][SCAN_LEFT]  =  32767;
        PolyScan[y][SCAN_RIGHT] = -32768;
    }

    while (--count)
    {
        //
        // Scan convert all the edges.
        //

        x1 = lpPoints[0].xcoord;
        x2 = lpPoints[1].xcoord;
        y1 = lpPoints[0].ycoord - yTop;
        y2 = lpPoints[1].ycoord - yTop;

        dx = x1 - x2;
        dy = y1 - y2;

        if ((dx == 0) && (dy == 0))
        {
            lpPoints++;
            continue;
        }

        if (dx < 0) dx = -dx;
        if (dy < 0) dy = -dy;

        if (dx >= dy)
        {
            //
            // X major.
            //

            ErrB = 2 * (dy - dx);
            ErrA = 2 * dy;
            Err  = ErrA - dx;

            if (x1 > x2)
            {
                //
                // Swap endpoints.
                //

                x  = x1; x1 = x2; x2 = x;
                y  = y1; y1 = y2; y2 = y;

            }
            minDir = (y2 >= y1) ? 1 : -1;

            x = x1;
            y = y1;

            //
            // Add initial position to edge list.
            //

            if (x < PolyScan[y][SCAN_LEFT])
                PolyScan[y][SCAN_LEFT] = x;
            if (x > PolyScan[y][SCAN_RIGHT])
                PolyScan[y][SCAN_RIGHT] = x;

            while (x < x2)
            {
                if (Err <= 0)
                {
                    Err += ErrA;
                    ++x;
                }
                else
                {
                    //
                    // Add current position to edge list.
                    //
#if 0
                    if (x > PolyScan[y][SCAN_RIGHT])
                        PolyScan[y][SCAN_RIGHT] = x;
                    Err  += ErrB;
                    ++x;
                    y    += minDir;
                    if (x < PolyScan[y][SCAN_LEFT])
                        PolyScan[y][SCAN_LEFT] = x;
#else
                    ++x;
                    Err  += ErrB;
                    y    += minDir;
                    if (x < PolyScan[y][SCAN_LEFT])
                        PolyScan[y][SCAN_LEFT] = x;
                    if (x > PolyScan[y][SCAN_RIGHT])
                        PolyScan[y][SCAN_RIGHT] = x;
#endif
                }
            }

            //
            // Add final position to edge list.
            //

//            if (x > PolyScan[y][SCAN_RIGHT])
//                PolyScan[y][SCAN_RIGHT] = x;
        }
        else
        {
            //
            // Y major.
            //

            ErrB = 2 * (dx - dy);
            ErrA = 2 * dx;
            Err  = ErrA - dy;

            if (y1 > y2)
            {
                //
                // Swap endpoints.
                //

                x  = x1; x1 = x2; x2 = x;
                y  = y1; y1 = y2; y2 = y;
            }
            minDir = (x2 >= x1) ? 1 : -1;

            x = x1;
            y = y1;

            while (y <= y2)
            {
                //
                // Add position to edge list.
                //

                if (x < PolyScan[y][SCAN_LEFT])
                    PolyScan[y][SCAN_LEFT] = x;
                if (x > PolyScan[y][SCAN_RIGHT])
                    PolyScan[y][SCAN_RIGHT] = x;
                if (Err <= 0)
                {
                    Err += ErrA;
                    ++y;
                }
                else
                {
                    Err  += ErrB;
                    x    += minDir;
                    ++y;
                }
            }
        }
        lpPoints++;
    }

    //
    // Apply clipping rectangle if present.
    //

    if (lpClipRect)
    {
        for (y = yTop; y <= yBottom; y++)
        {
            if ((y >= lpClipRect->top) && (y < lpClipRect->bottom))
            {
                //
                // Clip to left and right edges.
                //

                if (PolyScan[y - yTop][SCAN_LEFT]  <  lpClipRect->left)
                    PolyScan[y - yTop][SCAN_LEFT]  =  lpClipRect->left;
                if (PolyScan[y - yTop][SCAN_RIGHT] >= lpClipRect->right)
                    PolyScan[y - yTop][SCAN_RIGHT] =  lpClipRect->right;
            }
            else
            {
                //
                // Outside of clip rect.
                //

                PolyScan[y - yTop][SCAN_LEFT]  =  32767;
                PolyScan[y - yTop][SCAN_RIGHT] = -32768;
            }
        }
    }

    return(TRUE);
}

VOID PASCAL BitmapClippedLine
(
    SHORT  x1,
    SHORT  y1,
    SHORT  x2,
    SHORT  y2,
    LPRECT lpClipRect
)
{
    SHORT  start;
    SHORT  length;
    SHORT  dx;
    SHORT  dy;
    SHORT  x;
    SHORT  y;
    SHORT  minDir;
    SHORT  ErrA;
    SHORT  ErrB;
    SHORT  Err;

    dx = x1 - x2;
    dy = y1 - y2;

    if ((dx == 0) && (dy == 0))
    {
        return;
    }

    if (dx < 0) dx = -dx;
    if (dy < 0) dy = -dy;

    if (dx >= dy)
    {
        //
        // X major.
        //

        ErrB = 2 * (dy - dx);
        ErrA = 2 * dy;
        Err  = ErrA - dx;

        if (x1 > x2)
        {
            //
            // Swap endpoints.
            //

            x  = x1; x1 = x2; x2 = x;
            y  = y1; y1 = y2; y2 = y;

            //
            // Since we are now starting at the endpoint,
            // we will account for not writing the last
            // pixel in a polyline.
            //

            modeOutput = LINE_SKIP_FIRST;
        }
        else
        {
            modeOutput = LINE_SKIP_LAST;
        }
        minDir = (y2 >= y1) ? 1 : -1;

        x = x1;
        y = y1;

        if (Err <= 0)
        {
            start = x++;
            if (modeOutput == LINE_SKIP_FIRST)
            {
                start = x;
            }
            Err += ErrA;
        }
        else
        {
            if (modeOutput != LINE_SKIP_FIRST)
            {
                if ((x >= lpClipRect->left)
                    && (x <  lpClipRect->right)
                    && (y >= lpClipRect->top)
                    && (y <  lpClipRect->bottom))
                    (*pfnLineSeg)(x, y, 1, 1);
            }
            Err  += ErrB;
            y    += minDir;
            start = ++x;
        }
        while (x < x2)
        {
            if (Err <= 0)
            {
                Err += ErrA;
                ++x;
            }
            else
            {
                //
                // Draw horizontal segment.
                //

                if ((y >= lpClipRect->top) && (y < lpClipRect->bottom))
                {
                    if ((x >= lpClipRect->left) && (start < lpClipRect->right))
                    {
                        if (start < lpClipRect->left)
                            start = lpClipRect->left;
                        if (x >= lpClipRect->right)
                            length = lpClipRect->right - start;
                        else 
                            length = x - start + 1;
                        if (length > 0)
                            (*pfnLineSeg)(start, y, length, 1);
                    }
                }
                Err  += ErrB;
                y    += minDir;
                start = ++x;
            }
        }

        //
        // Draw final segment.
        //

        if ((y >= lpClipRect->top) && (y < lpClipRect->bottom))
        {
            if ((x >= lpClipRect->left) && (start < lpClipRect->right))
            {
                if (start < lpClipRect->left)
                    start = lpClipRect->left;
                if (x >= lpClipRect->right)
                    length = lpClipRect->right - start;
                else if (modeOutput == LINE_SKIP_LAST)
                    length = x - start;
                else 
                    length = x - start + 1;
                if (length > 0)
                    (*pfnLineSeg)(start, y, length, 1);
            }
        }
    }
    else
    {
        //
        // Y major.
        //

        ErrB = 2 * (dx - dy);
        ErrA = 2 * dx;
        Err  = ErrA - dy;

        if (y1 > y2)
        {
            //
            // Swap endpoints.
            //

            x  = x1; x1 = x2; x2 = x;
            y  = y1; y1 = y2; y2 = y;

            //
            // Since we are now starting at the endpoint,
            // we will account for not writing the last
            // pixel in a polylyine.
            //

            modeOutput = LINE_SKIP_FIRST;
        }
        else
        {
            modeOutput = LINE_SKIP_LAST;
        }
        minDir = (x2 >= x1) ? 1 : -1;

        x = x1;
        y = y1;

        if (Err <= 0)
        {
            start = y++;
            if (modeOutput == LINE_SKIP_FIRST)
            {
                start = y;
            }
            Err += ErrA;
        }
        else
        {
            if (modeOutput != LINE_SKIP_FIRST)
            {
                if ((x >= lpClipRect->left)
                 && (x <  lpClipRect->right)
                 && (y >= lpClipRect->top)
                 && (y <  lpClipRect->bottom))
                 (*pfnLineSeg)(x, y, 1, 1);
            }
            Err += ErrB;
            x   += minDir;
            start = ++y;
        }
        while (y < y2)
        {
            if (Err <= 0)
            {
                Err += ErrA;
                ++y;
            }
            else
            {
                //
                // Draw vertical segment.
                //

                if ((x >= lpClipRect->left) && (x < lpClipRect->right))
                {
                    if ((y >= lpClipRect->top) && (start < lpClipRect->bottom))
                    {
                        if (start < lpClipRect->top)
                            start = lpClipRect->top;
                        if (y >= lpClipRect->bottom)
                            length = lpClipRect->bottom - start;
                        else 
                            length = y - start + 1;
                        if (length > 0)
                            (*pfnLineSeg)(x, start, 1, length);
                    }
                }
                Err  += ErrB;
                x    += minDir;
                start = ++y;
            }
        }

        //
        // Draw final segment.
        //

        if ((x >= lpClipRect->left) && (x < lpClipRect->right))
        {
            if ((y >= lpClipRect->top) && (start < lpClipRect->bottom))
            {
                if (start < lpClipRect->top)
                    start = lpClipRect->top;
                if (y >= lpClipRect->bottom)
                    length = lpClipRect->bottom - start;
                else if (modeOutput == LINE_SKIP_LAST)
                    length = y - start;
                else 
                    length = y - start + 1;
                if (length > 0)
                    (*pfnLineSeg)(x, start, 1, length);
            }
        }
    }
}

VOID PASCAL BitmapLine
(
    SHORT  x1,
    SHORT  y1,
    SHORT  x2,
    SHORT  y2
)
{
    SHORT  start;
    SHORT  length;
    SHORT  dx;
    SHORT  dy;
    SHORT  x;
    SHORT  y;
    SHORT  minDir;
    SHORT  ErrA;
    SHORT  ErrB;
    SHORT  Err;

    dx = x1 - x2;
    dy = y1 - y2;

    if ((dx == 0) && (dy == 0))
    {
        return;
    }

    if (dx < 0) dx = -dx;
    if (dy < 0) dy = -dy;

    if (dx >= dy)
    {
        //
        // X major.
        //

        ErrB = 2 * (dy - dx);
        ErrA = 2 * dy;
        Err  = ErrA - dx;

        if (x1 > x2)
        {
            //
            // Swap endpoints.
            //

            x  = x1; x1 = x2; x2 = x;
            y  = y1; y1 = y2; y2 = y;

            //
            // Since we are now starting at the endpoint,
            // we will account for not writing the last
            // pixel in a polyline.
            //

            modeOutput = LINE_SKIP_FIRST;
        }
        else
        {
            modeOutput = LINE_SKIP_LAST;
        }
        minDir = (y2 >= y1) ? 1 : -1;

        x = x1;
        y = y1;

        if (Err <= 0)
        {
            start = x++;
            if (modeOutput == LINE_SKIP_FIRST)
            {
                start = x;
            }
            Err += ErrA;
        }
        else
        {
            if (modeOutput != LINE_SKIP_FIRST)
            {
                (*pfnLineSeg)(x, y, 1, 1);
            }
            Err  += ErrB;
            y    += minDir;
            start = ++x;
        }
        while (x < x2)
        {
            if (Err <= 0)
            {
                Err += ErrA;
                ++x;
            }
            else
            {
                //
                // Draw horizontal segment.
                //

                length = x - start + 1;
                if (length > 0)
                    (*pfnLineSeg)(start, y, length, 1);
                Err  += ErrB;
                start = ++x;
                y    += minDir;
            }
        }

        //
        // Draw final segment.
        //

        if (modeOutput == LINE_SKIP_LAST)
            length = x - start;
        else 
            length = x - start + 1;
        if (length > 0)
            (*pfnLineSeg)(start, y, length, 1);
    }
    else
    {
        //
        // Y major.
        //

        ErrB = 2 * (dx - dy);
        ErrA = 2 * dx;
        Err  = ErrA - dy;

        if (y1 > y2)
        {
            //
            // Swap endpoints.
            //

            x  = x1; x1 = x2; x2 = x;
            y  = y1; y1 = y2; y2 = y;

            //
            // Since we are now starting at the endpoint,
            // we will account for not writing the last
            // pixel in a polylyine.
            //

            modeOutput = LINE_SKIP_FIRST;
        }
        else
        {
            modeOutput = LINE_SKIP_LAST;
        }
        minDir = (x2 >= x1) ? 1 : -1;

        x = x1;
        y = y1;

        if (Err <= 0)
        {
            start = y++;
            if (modeOutput == LINE_SKIP_FIRST)
            {
                start = y;
            }
            Err += ErrA;
        }
        else
        {
            if (modeOutput != LINE_SKIP_FIRST)
            {
                (*pfnLineSeg)(x, y, 1, 1);
            }
            Err += ErrB;
            x   += minDir;
            start = ++y;
        }
        while (y < y2)
        {
            if (Err <= 0)
            {
                Err += ErrA;
                ++y;
            }
            else
            {
                //
                // Draw vertical segment.
                //

                length = y - start + 1;
                if (length > 0)
                    (*pfnLineSeg)(x, start, 1, length);
                Err  += ErrB;
                x    += minDir;
                start = ++y;
            }
        }

        //
        // Draw final segment.
        //

        if (modeOutput == LINE_SKIP_LAST)
            length = y - start;
        else 
            length = y - start + 1;
        if (length > 0)
            (*pfnLineSeg)(x, start, 1, length);
    }
}

VOID PASCAL MonoBitmapScans
(
    SHORT xDst,
    SHORT xExt,
    SHORT yExt
)
{
    memBltPD_ToMono(lpOutputDst->bmBits,
                    lpOutputDst->bmWidthBytes,
                    lpOutputDst->bmSegmentIndex ? lpOutputDst->bmScanSegment : 0x7FFF,
                    lpOutputDst->bmFillBytes,
                    xDst,
                    yOutput,
                    lpOutputBrush,
                    xExt,
                    yExt,
                    modeOutput,
                    ropOutput);
}

VOID PASCAL ColorBitmapScans
(
    SHORT xDst,
    SHORT xExt,
    SHORT yExt
)
{
    memBltPD_ToColor(lpOutputDst->bmBits,
                     lpOutputDst->bmWidthBytes,
                     lpOutputDst->bmSegmentIndex ? lpOutputDst->bmScanSegment : 0x7FFF,
                     lpOutputDst->bmFillBytes,
                     xDst,
                     yOutput,
                     lpOutputBrush,
                     xExt,
                     yExt,
                     modeOutput,
                     ropOutput);
}

VOID PASCAL SolidDeviceScans
(
    SHORT xDst,
    SHORT xExt,
    SHORT yExt
)
{
    devBltPD_Solid(xDst,
                   yOutput,
                   xExt,
                   yExt,
                   dwOutputColor,
                   ropOutput);
}

VOID PASCAL MonoDeviceScans
(
    SHORT xDst,
    SHORT xExt,
    SHORT yExt
)
{
    devBltPD_Mono(xDst,
                  yOutput,
                  lpOutputBrush,
                  xExt,
                  yExt,
                  dwOutputColor,
                  dwOutputBkColor,
                  modeOutput,
                  ropOutput);
}

VOID PASCAL ColorDeviceScans
(
    SHORT xDst,
    SHORT xExt,
    SHORT yExt
)
{
    devBltPD_Color(xDst,
                   yOutput,
                   lpOutputBrush,
                   xExt,
                   yExt,
                   ropOutput);
}

VOID PASCAL MonoBitmapSegment
(
    SHORT x,
    SHORT y,
    SHORT cx,
    SHORT cy
)
{
    memBltPD_ToMono(lpOutputDst->bmBits,
                    lpOutputDst->bmWidthBytes,
                    lpOutputDst->bmSegmentIndex ? lpOutputDst->bmScanSegment : 0x7FFF,
                    lpOutputDst->bmFillBytes,
                    x,
                    y,
                    lpOutputBrush,
                    cx,
                    cy,
                    OPAQUE,
                    ropOutput);
}

VOID PASCAL ColorBitmapSegment
(
    SHORT x,
    SHORT y,
    SHORT cx,
    SHORT cy
)
{
    memBltPD_ToColor(lpOutputDst->bmBits,
                     lpOutputDst->bmWidthBytes,
                     lpOutputDst->bmSegmentIndex ? lpOutputDst->bmScanSegment : 0x7FFF,
                     lpOutputDst->bmFillBytes,
                     x,
                     y,
                     lpOutputBrush,
                     cx,
                     cy,
                     OPAQUE,
                     ropOutput);
}

VOID PASCAL DevicePolyLine
(
    LPPOINT lpPoints,
    WORD    wCount,
    LPRECT  lpClipRect
)
{
    SHORT x1;
    SHORT y1;
    SHORT x2;
    SHORT y2;
    SHORT dx;
    SHORT dy;
    SHORT start;
    SHORT length;

    while (--wCount)
    {
        x1 = lpPoints[0].xcoord;
        y1 = lpPoints[0].ycoord;
        x2 = lpPoints[1].xcoord;
        y2 = lpPoints[1].ycoord;

        dx = x1 - x2;
        dy = y1 - y2;

        if ((dx == 0) && (dy == 0))
        {
            lpPoints++;
            continue;
        }
        if (dy == 0)
        {
            //
            // Horizontal line.
            //

            if (lpClipRect)
            {
                //
                // Clip to rectangle.
                //

                if ((y1 >= lpClipRect->bottom) || (y1 < lpClipRect->top))
                {
                    lpPoints++;
                    continue;
                }
                if (x1 < x2)
                {
                    if ((x2 <= lpClipRect->left) || (x1 >= lpClipRect->right))
                    {
                        lpPoints++;
                        continue;
                    }
                    if (x1 < lpClipRect->left)
                        x1 = lpClipRect->left;
                    if (x2 > lpClipRect->right)
                        x2 = lpClipRect->right;
                    start  = x1;
                    length = x2 - x1;
                }
                else
                {
                    if ((x1 < lpClipRect->left) || (x2 >= lpClipRect->right))
                    {
                        lpPoints++;
                        continue;
                    }
                    if (x2 <  lpClipRect->left)
                        x2 =  lpClipRect->left - 1;
                    if (x1 >= lpClipRect->right)
                        x1 =  lpClipRect->right - 1;
                    start  = x2 + 1;
                    length = x1 - x2;
                }
                if (length == 0)
                {
                    lpPoints++;
                    continue;
                }
            }
            else
            {
                length = x1 - x2;
                if (length < 0)
                {
                    start  = x1;
                    length = -length;
                }
                else if (length > 0)
                {
                    start = x2 + 1;
                }
                else
                {
                    lpPoints++;
                    continue;
                }
            }

            //
            // Draw horizontal line real fast.
            //

            devBltPD_Solid(start,
                            y1,
                            length,
                            1,
                            dwOutputColor,
                            ropOutput);
        }
        else if (dx == 0)
        {
            //
            // Vertical line.
            //

            if (lpClipRect)
            {
                //
                // Clip to rectangle.
                //

                if ((x1 >= lpClipRect->right) || (x1 < lpClipRect->left))
                {
                    lpPoints++;
                    continue;
                }
                if (y1 < y2)
                {
                    if ((y2 <= lpClipRect->top) || (y1 >= lpClipRect->bottom))
                    {
                        lpPoints++;
                        continue;
                    }
                    if (y1 < lpClipRect->top)
                        y1 = lpClipRect->top;
                    if (y2 > lpClipRect->bottom)
                        y2 = lpClipRect->bottom;
                    start  = y1;
                    length = y2 - y1;
                }
                else
                {
                    if ((y1 < lpClipRect->top) || (y2 >= lpClipRect->bottom))
                    {
                        lpPoints++;
                        continue;
                    }
                    if (y2 <  lpClipRect->top)
                        y2 =  lpClipRect->top - 1;
                    if (y1 >= lpClipRect->bottom)
                        y1 =  lpClipRect->bottom - 1;
                    start  = y2 + 1;
                    length = y1 - y2;
                }
                if (length == 0)
                {
                    lpPoints++;
                    continue;
                }
            }
            else
            {
                length = y1 - y2;
                if (length < 0)
                {
                    start  = y1;
                    length = -length;
                }
                else if (length > 0)
                {
                    start = y2 + 1;
                }
                else
                {
                    lpPoints++;
                    continue;
                }
            }
            //
            // Draw vertical line real fast.
            //

            devBltPD_Solid(x1,
                            start,
                            1,
                            length,
                            dwOutputColor,
                            ropOutput);
        }
        else
        {
            if (lpClipRect)
            {
                //
                // Draw clipped line.
                //

                devClippedLine(x1, y1, x2, y2, lpClipRect, dwOutputColor, ropOutput);
            }
            else
            {
                //
                // Draw unclipped line.
                //

                devLine(x1, y1, x2, y2, dwOutputColor, ropOutput);
            }
        }

        //
        // Next polyline.
        //

        lpPoints++;
    }
}
VOID PASCAL BitmapPolyLine
(
    LPPOINT  lpPoints,
    WORD     wCount,
    LPRECT   lpClipRect
)
{
    //
    // Create a solid brush to use as a pen.
    //

    lpOutputBrush             = memGetTmpBrush();
    lpOutputBrush->brFlags    = BRUSH_SOLID;
    lpOutputBrush->brClrSolid = dwOutputColor;
    lpOutputBrush->brClrFore  = dwOutputColor;
    lpOutputBrush->brClrBack  = dwOutputColor;
    modeOutput                = OPAQUE;
    memConvertMonoBrush(lpOutputBrush);
    if (lpOutputDst->bmBitsPixel == 1)
    {
        pfnLineSeg = MonoBitmapSegment;
    }
    else
    {
        pfnLineSeg = ColorBitmapSegment;
    }

    if (lpClipRect)
    {
        while (--wCount)
        {
            BitmapClippedLine(lpPoints[0].xcoord,
                                    lpPoints[0].ycoord,
                                    lpPoints[1].xcoord,
                                    lpPoints[1].ycoord,
                                    lpClipRect);

            //
            // Next polyline.
            //

            lpPoints++;
        }
    }
    else
    {
        while (--wCount)
        {
            BitmapLine(lpPoints[0].xcoord,
                       lpPoints[0].ycoord,
                       lpPoints[1].xcoord,
                       lpPoints[1].ycoord);

            //
            // Next polyline.
            //

            lpPoints++;
        }
    }
}
