/****************************************************************************\
*                                                                            *
* Function: Text output entrypoints                                          *
*                                                                            *
*       Direct control to either memory based text routines or device based  *
*       text routines.  Enumeration functions are handled with memory        *
*       routines.                                                            *
*                                                                            *
* Author:                                                                    *
*                                                                            *
*       Dave Schmenk                                                         *
*                                                                            *
* History:                                                                   *
*                                                                            *
*       05/11/93 Dave Schmenk - wrote it.                                    *
*                                                                            *
******************************************************************************
*                                                                            *
*                     Copyright 1992 pellucid, inc.                          *
*                                                                            *
\****************************************************************************/

#include "common.h"
#include "bitblt.h"
#include "textout.h"

//
// Glyph information structure.
//

GLYPHINFO GlyphInfo[MAX_CHAR_COUNT + 1];

//
// Locals for iterative calls.
//

SHORT BreakErrExt;

//
// TextExtent calculates the extent of the character string and filles in
// the x coordinates of the characters in the vCharX array.
//

DWORD PASCAL CalcTextExtent(LPSTR, LPFONTINFO, LPSHORT, LPDRAWMODE, SHORT);
DWORD PASCAL GetGlyphInfo(LPSTR, LPFONTINFO, LPSHORT, LPDRAWMODE, SHORT, LPWORD);

DWORD WINAPI ExtTextOut
(
    LPBITMAP    lpDst,
    SHORT       wDstX,
    SHORT       wDstY,
    LPRECT      lpClipRect,
    LPSTR       lpString,
    SHORT       wCount,
    LPFONTINFO  lpFontInfo,
    LPDRAWMODE  lpDrawMode,
    LPTEXTXFORM lpTextXForm,
    LPSHORT     lpCharWidths,
    LPRECT      lpOpaqueRect,
    WORD        wOptions
)
{
    GLYPHPROC pfnDrawGlyphs;
    DWORD     TextExtent;
    RECT      rectClip;
    RECT      rectText;
    WORD      ropText;
    DWORD     TmpColor;
    WORD      TmpMode;
    LPBRUSH   lpTextBrush;
    LPBITMAP  lpTextBitmap;
    SHORT     ii;

    DBGMSG("ExtTextOut\n\r");

    if (wCount < 0)
    {
        return(CalcTextExtent(lpString, lpFontInfo, lpCharWidths, lpDrawMode, -wCount));
    }

    //
    // Do any rectangle clipping.
    //

    if (lpClipRect)
    {
        //
        // Copy ClipRect locally.
        //

        rectClip.left   = lpClipRect->left;
        rectClip.top    = lpClipRect->top;
        rectClip.right  = lpClipRect->right;
        rectClip.bottom = lpClipRect->bottom;
    }
    else
    {
        //
        // Clip rect is entire destination.
        //

        rectClip.left   = 0;
        rectClip.top    = 0;
        rectClip.right  = lpDst->bmWidth;
        rectClip.bottom = lpDst->bmHeight;
    }

    if (wOptions & (ETO_CLIPPED | ETO_OPAQUE))
    {
        //
        // Clip ClipRect with OpaqueRect.
        //

        ClipRect(&rectClip, lpOpaqueRect, &rectClip);
    }

    //
    // If clipped away, return.
    //

    if ((rectClip.left >= rectClip.right) || (rectClip.top >= rectClip.bottom))
        return(0L);

    if (lpDst->bmType == 0)
    {
        //
        // Memory bitmap is destination. Draw opaquing rectangle if required.
        // Get a temporary brush to use for filling.
        //

        lpTextBrush = memGetTmpBrush();
        lpTextBrush->brFlags = BRUSH_SOLID;

        if (wOptions & ETO_OPAQUE)
        {
            if (lpDst->bmBitsPixel == 1)
            {
                memBltPD_ToMono(lpDst->bmBits,
                                lpDst->bmWidthBytes,
                                lpDst->bmSegmentIndex ? lpDst->bmScanSegment : 0x7FFF,
                                lpDst->bmFillBytes,
                                rectClip.left,
                                rectClip.top,
                                lpTextBrush,
                                rectClip.right  - rectClip.left,
                                rectClip.bottom - rectClip.top,
                                OPAQUE,
                                (WORD)devMonoMatchLogicalColor(lpDrawMode->bkColor) & 0x0F);
            }
            else
            {
                //
                // Convert mono bits to solid color brush. Because
                // foreground and background are the same, the mono bits
                // don't have to be initialized to a solid color.
                //

                lpTextBrush->brClrSolid = lpDrawMode->bkColor;
                lpTextBrush->brClrFore  = lpDrawMode->bkColor;
                lpTextBrush->brClrBack  = lpDrawMode->bkColor;
                memConvertMonoBrush(lpTextBrush);

                //
                // Fill with opaque rect brush.
                //

                memBltPD_ToColor(lpDst->bmBits,
                                 lpDst->bmWidthBytes,
                                 lpDst->bmSegmentIndex ? lpDst->bmScanSegment : 0x7FFF,
                                 lpDst->bmFillBytes,
                                 rectClip.left,
                                 rectClip.top,
                                 lpTextBrush,
                                 rectClip.right  - rectClip.left,
                                 rectClip.bottom - rectClip.top,
                                 OPAQUE,
                                 0x0C); // R2_COPYPEN
            }
        }

        //
        // Bail out now if nothing left to do.
        //

        if (wCount == 0) return(0L);

        BreakErrExt     = lpDrawMode->BreakErr;
        TextExtent      = GetGlyphInfo(lpString, lpFontInfo, lpCharWidths, lpDrawMode, wCount, &wOptions);
        rectText.left   = wDstX;
        rectText.top    = wDstY;
        rectText.right  = wDstX + (SHORT)TextExtent;
        rectText.bottom = wDstY + (SHORT)(TextExtent >> 16);

        ClipRect(&rectClip, &rectText, &rectClip);

        //
        // If text clipped away, return.
        //

        if ((rectClip.left   >= rectText.right)  ||
            (rectClip.right  <= rectText.left)   ||
            (rectClip.top    >= rectText.bottom) ||
            (rectClip.bottom <= rectText.top))
        {
            return(0L);
        }

        //
        // Create temporary bitmap to draw string into and clear it.
        //

        lpTextBitmap = CreateMonoBitmap((SHORT)TextExtent + 1, (SHORT)(TextExtent >> 16) + 1);
        memBltPD_ToMono(lpTextBitmap->bmBits,
                        lpTextBitmap->bmWidthBytes,
                        lpTextBitmap->bmSegmentIndex ? lpTextBitmap->bmScanSegment : 0x7FFF,
                        lpTextBitmap->bmFillBytes,
                        0,
                        0,
                        lpTextBrush,
                        (SHORT)TextExtent,
                        (SHORT)(TextExtent >> 16),
                        OPAQUE,
                        0x00);

        //
        // Output string to temporary bitmap.
        //

        while (wCount > MAX_CHAR_COUNT)
        {
            //
            // Draw glyphs in batches 'cause theres just too many of them.
            //

             memDrawGlyphs(lpTextBitmap->bmBits,
                           lpTextBitmap->bmWidthBytes,
                           lpTextBitmap->bmSegmentIndex ? lpTextBitmap->bmScanSegment : 0x7FFF,
                           lpTextBitmap->bmFillBytes,
                           lpFontInfo,
                           GlyphInfo,
                           MAX_CHAR_COUNT);
            wCount   -= MAX_CHAR_COUNT;

            if (wCount)
            {
                lpString += MAX_CHAR_COUNT;
                if (lpCharWidths) lpCharWidths += MAX_CHAR_COUNT;
                wDstX = GlyphInfo[MAX_CHAR_COUNT].xPos;

                //
                // Get next batch of glyph infos.
                //

                GetGlyphInfo(lpString, lpFontInfo, lpCharWidths, lpDrawMode, wCount, &wOptions);

                //
                // Update horizontal positions for next glyph batch.
                //

                for (ii = 0; ii <= MAX_CHAR_COUNT; ii++)
                    GlyphInfo[ii].xPos += wDstX;
            }
        }
        if (wCount) memDrawGlyphs(lpTextBitmap->bmBits,
                                  lpTextBitmap->bmWidthBytes,
                                  lpTextBitmap->bmSegmentIndex ? lpTextBitmap->bmScanSegment : 0x7FFF,
                                  lpTextBitmap->bmFillBytes,
                                  lpFontInfo,
                                  GlyphInfo,
                                  wCount);

        //
        // Blt string bitmap to destination after clipping.
        //


        if (lpDst->bmBitsPixel == 1)
        {
            //
            // Blt mono glyphs using foreground color.
            //

            if (devMonoMatchLogicalColor(lpDrawMode->TextColor))
            {
                //
                // Drawing WHITE glyphs.
                //

                if (lpDrawMode->bkMode == OPAQUE)
                {
                    if (devMonoMatchLogicalColor(lpDrawMode->bkColor))
                    {
                        //
                        // With WHITE background.
                        //

                        ropText = 0x0F; // R2_WHITE
                    }
                    else
                    {
                        //
                        // With BLACK background.
                        //

                        ropText = 0x0C; // R2_COPYPEN
                    }
                }
                else
                {
                    //
                    // With TRANSPARENT background.
                    //

                    ropText = 0x0E; // R2_MERGEPEN
                }
            }
            else
            {
                //
                // Drawing BLACK glyphs.
                //

                if (lpDrawMode->bkMode == OPAQUE)
                {
                    if (devMonoMatchLogicalColor(lpDrawMode->bkColor))
                    {
                        //
                        // With WHITE background.
                        //

                        ropText = 0x03; // R2_NOTCOPYPEN
                    }
                    else
                    {
                        //
                        // With BLACK background.
                        //

                        ropText = 0x00; // R2_BLACK
                    }
                }
                else
                {
                    //
                    // With TRANSPARENT background.
                    //

                    ropText = 0x02; // R2_MASKNOTPEN
                }
            }

            memBltSD_MonoToMono(lpDst->bmBits,
                                lpDst->bmWidthBytes,
                                lpDst->bmSegmentIndex ? lpDst->bmScanSegment : 0x7FFF,
                                lpDst->bmFillBytes,
                                rectClip.left,
                                rectClip.top,
                                lpTextBitmap->bmBits,
                                lpTextBitmap->bmWidthBytes,
                                lpTextBitmap->bmSegmentIndex ? lpTextBitmap->bmScanSegment : 0x7FFF,
                                lpTextBitmap->bmFillBytes,
                                rectClip.left - rectText.left,
                                rectClip.top  - rectText.top,
                                rectClip.right  - rectClip.left,
                                rectClip.bottom - rectClip.top,
                                ropText);
        }
        else
        {
            //
            // Blt color glyphs.
            //

            if (lpDrawMode->bkMode == OPAQUE)
            {
                memBltSD_MonoToColor(lpDst->bmBits,
                                     lpDst->bmWidthBytes,
                                     lpDst->bmSegmentIndex ? lpDst->bmScanSegment : 0x7FFF,
                                     lpDst->bmFillBytes,
                                     rectClip.left,
                                     rectClip.top,
                                     lpTextBitmap->bmBits,
                                     lpTextBitmap->bmWidthBytes,
                                     lpTextBitmap->bmSegmentIndex ? lpTextBitmap->bmScanSegment : 0x7FFF,
                                     lpTextBitmap->bmFillBytes,
                                     rectClip.left - rectText.left,
                                     rectClip.top  - rectText.top,
                                     rectClip.right  - rectClip.left,
                                     rectClip.bottom - rectClip.top,
                                     lpDrawMode->bkColor,
                                     lpDrawMode->TextColor,
                                     0x0C); // R2_COPYPEN
            }
            else
            {
                //                    
                // First mask out foreground bits...
                //

                memBltSD_MonoToColor(lpDst->bmBits,
                                     lpDst->bmWidthBytes,
                                     lpDst->bmSegmentIndex ? lpDst->bmScanSegment : 0x7FFF,
                                     lpDst->bmFillBytes,
                                     rectClip.left,
                                     rectClip.top,
                                     lpTextBitmap->bmBits,
                                     lpTextBitmap->bmWidthBytes,
                                     lpTextBitmap->bmSegmentIndex ? lpTextBitmap->bmScanSegment : 0x7FFF,
                                     lpTextBitmap->bmFillBytes,
                                     rectClip.left - rectText.left,
                                     rectClip.top  - rectText.top,
                                     rectClip.right  - rectClip.left,
                                     rectClip.bottom - rectClip.top,
                                     0xFF000000L, // Hopefully device black
                                     0xFFFFFFFFL, // Hopefully device white
                                     0x02); // R2_MASKNOTPEN

                //
                // ... then OR in foreground color.
                //

                memBltSD_MonoToColor(lpDst->bmBits,
                                     lpDst->bmWidthBytes,
                                     lpDst->bmSegmentIndex ? lpDst->bmScanSegment : 0x7FFF,
                                     lpDst->bmFillBytes,
                                     rectClip.left,
                                     rectClip.top,
                                     lpTextBitmap->bmBits,
                                     lpTextBitmap->bmWidthBytes,
                                     lpTextBitmap->bmSegmentIndex ? lpTextBitmap->bmScanSegment : 0x7FFF,
                                     lpTextBitmap->bmFillBytes,
                                     rectClip.left - rectText.left,
                                     rectClip.top  - rectText.top,
                                     rectClip.right  - rectClip.left,
                                     rectClip.bottom - rectClip.top,
                                     0xFF000000L, // Hopefully device black
                                     lpDrawMode->TextColor,
                                     0x0E); // R2_MERGEPEN
            }
        }

        //
        // Destroy temporary text bitmap.
        //

        DestroyBitmap(lpTextBitmap);
    }
    else
    {
        //
        // Screen is destination.
        //

        if (wCount == 0)
        {
            //
            // If no text to draw check for opaque rect before leaving.
            //

            if (wOptions & ETO_OPAQUE)
            {
                devBltPD_Solid(rectClip.left,
                               rectClip.top,
                               rectClip.right  - rectClip.left,
                               rectClip.bottom - rectClip.top,
                               lpDrawMode->bkColor,
                               12); // R2_COPYPEN
            }
            return(0L);
        }

        //
        // Prepare device for drawing text.
        //

        BreakErrExt     = lpDrawMode->BreakErr;
        TextExtent      = GetGlyphInfo(lpString, lpFontInfo, lpCharWidths, lpDrawMode, wCount, &wOptions);
        rectText.left   = wDstX;
        rectText.top    = wDstY;
        rectText.right  = wDstX + (SHORT)TextExtent;
        rectText.bottom = wDstY + (SHORT)(TextExtent >> 16);

        //
        // If text clipped away, check for opaque rect and return.
        //

        if ((rectClip.left   >= rectText.right)  ||
            (rectClip.right  <= rectText.left)   ||
            (rectClip.top    >= rectText.bottom) ||
            (rectClip.bottom <= rectText.top))
        {
            if (wOptions & ETO_OPAQUE)
            {
                devBltPD_Solid(rectClip.left,
                               rectClip.top,
                               rectClip.right  - rectClip.left,
                               rectClip.bottom - rectClip.top,
                               lpDrawMode->bkColor,
                               12); // R2_COPYPEN
            }
            return(0L);
        }

        if (wOptions & ETO_OPAQUE)
        {
            //
            // Draw extents of opaque rectangle outside of that drawn by the
            // glyph routines.
            //

            TmpColor = lpDrawMode->bkColor;
            if (rectClip.top < rectText.top)
            {
                //
                // Top of opaque - text rect.
                //

                devBltPD_Solid(rectClip.left,
                               rectClip.top,
                               rectClip.right - rectClip.left,
                               rectText.top   - rectClip.top,
                               TmpColor,
                               12); // R2_COPYPEN
            }
            if (rectClip.bottom > rectText.bottom)
            {
                //
                // Bottom of opaque - text rect.
                //

                devBltPD_Solid(rectClip.left,
                               rectText.bottom,
                               rectClip.right  - rectClip.left,
                               rectClip.bottom - rectText.bottom,
                               TmpColor,
                               12); // R2_COPYPEN
            }
            if (rectClip.left < rectText.left)
            {
                //
                // Left of opaque - text rect.
                //

                devBltPD_Solid(rectClip.left,
                               rectClip.top,
                               rectText.left   - rectClip.left,
                               rectClip.bottom - rectClip.top,
                               TmpColor,
                               12); // R2_COPYPEN
            }
            if (rectClip.right > rectText.right)
            {
                //
                // Right of opaque - text rect.
                //

                devBltPD_Solid(rectText.right,
                               rectClip.top,
                               rectClip.right  - rectText.right,
                               rectClip.bottom - rectClip.top,
                               TmpColor,
                               12); // R2_COPYPEN
            }

            //
            // Force drawmode to opaque. Glyph routines may optimize by
            // filling background while drawing text.
            //

            TmpMode            = lpDrawMode->bkMode;
            lpDrawMode->bkMode = OPAQUE;
            pfnDrawGlyphs      = (GLYPHPROC)devTextFunc(&rectClip, &rectText, lpDrawMode, wOptions);
            lpDrawMode->bkMode = TmpMode;
        }
        else
        {
            pfnDrawGlyphs   = (GLYPHPROC)devTextFunc(&rectClip, &rectText, lpDrawMode, wOptions);
        }
        
        //
        // Output string to device.
        //

        while (wCount > MAX_CHAR_COUNT)
        {
            //
            // Output glyphs in batches if there are too many of them.
            //

            wDstX = (*pfnDrawGlyphs)(lpFontInfo, GlyphInfo, wDstX, wDstY, MAX_CHAR_COUNT);

            //
            // If we have drawn off the right edge, just exit.
            //

            if (wDstX > rectClip.right)
                return(0L);

            //
            // Prepare for next batch of glyphs.
            //

            lpString += MAX_CHAR_COUNT;
            if (lpCharWidths) lpCharWidths += MAX_CHAR_COUNT;
            wCount   -= MAX_CHAR_COUNT;
            GetGlyphInfo(lpString, lpFontInfo, lpCharWidths, lpDrawMode, wCount, &wOptions);
        }

        //
        // Draw last (only) batch of glyphs.
        //

        if (wCount) (*pfnDrawGlyphs)(lpFontInfo, GlyphInfo, wDstX, wDstY, wCount);
    }


    return(0L);
}

DWORD WINAPI StrBlt
(
    LPBITMAP    lpDst,
    WORD        wDstX,
    WORD        wDstY,
    LPRECT      lpClipRect,
    LPSTR       lpString,
    WORD        wCount,
    LPFONTINFO  lpFontInfo,
    LPDRAWMODE  lpDrawMode,
    LPTEXTXFORM lpTextXForm
)
{
    return(ExtTextOut(lpDst,
                      wDstX,
                      wDstY,
                      lpClipRect,
                      lpString,
                      wCount,
                      lpFontInfo,
                      lpDrawMode,
                      lpTextXForm,
                      (LPSHORT)NULL,
                      (LPRECT)NULL,
                      0));
}

