EXTERN          __AHIncr : ABS

bPreFetch       EQU     000000001b
bLastFetch      EQU     000000010b
bOneFetch       EQU     000000100b
bLeftToRight    EQU     001000000b
bRightToLeft    EQU     000000000b
bTopToBottom    EQU     010000000b
bBottomToTop    EQU     000000000b

;----------------------------------------------------------------------------;
;                                                                            ;
; The swzl macro swizzles bytes around in the register specifed.  This is    ;
; used in accessing 1BPP data and shifting it as 32 bits of big endian data. ;
;                                                                            ;
;----------------------------------------------------------------------------;

swzl            MACRO reg:REQ
IF "&reg" EQ "eax"
        xchg    al, ah
        ror     eax, 16
        xchg    al, ah
ENDIF
IF "&reg" EQ "ebx"
        xchg    bl, bh
        ror     ebx, 16
        xchg    bl, bh
ENDIF
IF "&reg" EQ "ecx"
        xchg    cl, ch
        ror     ecx, 16
        xchg    cl, ch
ENDIF
IF "&reg" EQ "edx"
        xchg    dl, dh
        ror     edx, 16
        xchg    dl, dh
ENDIF
IF "&reg" EQ "ax"
        xchg    al, ah
ENDIF
IF "&reg" EQ "bx"
        xchg    bl, bh
ENDIF
IF "&reg" EQ "cx"
        xchg    cl, ch
ENDIF
IF "&reg" EQ "dx"
        xchg    dl, dh
ENDIF
                ENDM

;----------------------------------------------------------------------------;
;                                                                            ;
; The rop3 macro scans the translated rop looking for terms to evaluate.     ;
; It is not written for speed but for functionality.  All two term rops are  ;
; special cased in other code and handled by static functions.               ;
;                                                                            ;
; Entry:  src     = SRC                                                      ;
;         pat     = PAT                                                      ;
;         es:[di] = *DST                                                     ;
;         xrop    = translated rop                                           ;
;                                                                            ;
; Exit:   al = result                                                        ;
;         ah = garbage                                                       ;
;         dl = 0                                                             ;
;                                                                            ;
;----------------------------------------------------------------------------;

rop3            MACRO src, pat, xrop
                LOCAL r3sp, r3dp, r3ds, r3p, r3s, r3d, r3n, r3exit
        mov     dl, xrop
        xor     al, al
;
; Evaluate Adsp
;
        add     dl, dl
        jnc     r3sp
        mov     al, es:[di]
        and     al, src
        and     al, pat
;
; Evaluate Asp
;
r3sp:   add     dl, dl
        jnc     r3dp
        mov     ah, src
        and     ah, pat
        xor     al, ah
;
; Evaluate Adp
;
r3dp:   add     dl, dl
        jnc     r3ds
        mov     ah, es:[di]
        and     ah, pat
        xor     al, ah
;
; Evaluate Ads
;
r3ds:   add     dl, dl
        jnc     r3p
        mov     ah, es:[di]
        and     ah, src
        xor     al, ah
;
; Evaluate Ap
;
r3p:    add     dl, dl
        jnc     r3s
        xor     al, pat
;
; Evaluate As
;
r3s:    add     dl, dl
        jnc     r3d
        xor     al, src
;
; Evaluate Ad
;
r3d:    add     dl, dl
        jnc     r3n
        xor     al, es:[di]
;
; Evaluate An
;
r3n:    add     dl, dl
        jnc     r3exit
        not     al
r3exit: 
                ENDM

;----------------------------------------------------------------------------;
;                                                                            ;
; The rop3_24 macro is identical to the rop3 macro except it operates on 24  ;
; bits of data instead of just eight,                                        ;
;                                                                            ;
; Entry:  src     = SRC                                                      ;
;         pat     = PAT                                                      ;
;         es:[di] = *DST                                                     ;
;         xrop    = translated rop                                           ;
;                                                                            ;
; Exit:   ax:dl   = result                                                   ;
;         cx      = garbage                                                  ;
;         dh      = 0                                                        ;
;         es:[di] = 0                                                        ;
;                                                                            ;
;----------------------------------------------------------------------------;

rop3_24         MACRO src, pat, xrop
                LOCAL r3sp, r3dp, r3ds, r3p, r3s, r3d, r3n, r3exit
        mov     dh, xrop
        xor     ax, ax
        xor     dl, dl
;
; Evaluate Adsp
;
        add     dh, dh
        jnc     r3sp
        mov     ax, es:[di]
        mov     dl, es:[di + 2]
        and     ax, src
        and     dl, src + 2
        and     ax, pat
        and     dl, pat + 2
;
; Evaluate Asp
;
r3sp:   add     dh, dh
        jnc     r3dp
        mov     cx, src
        and     cx, pat
        xor     ax, cx
        mov     cl, src + 2
        and     cl, pat + 2
        xor     dl, cl
;
; Evaluate Adp
;
r3dp:   add     dh, dh
        jnc     r3ds
        mov     cx, es:[di]
        and     cx, pat
        xor     ax, cx
        mov     cl, es:[di + 2] 
        and     cl, pat + 2
        xor     dl, cl
;
; Evaluate Ads
;
r3ds:   add     dh, dh
        jnc     r3p
        mov     cx, es:[di]
        and     cx, src
        xor     ax, cx
        mov     cl, es:[di + 2]
        and     cl, src + 2
        xor     dl, cl
;
; Evaluate Ap
;
r3p:    add     dh, dh
        jnc     r3s
        xor     ax, pat
        xor     dl, pat + 2
;
; Evaluate As
;
r3s:    add     dh, dh
        jnc     r3d
        xor     ax, src
        xor     dl, src + 2
;
; Evaluate Ad
;
r3d:    add     dh, dh
        jnc     r3n
        xor     ax, es:[di]
        xor     dl, es:[di + 2]
;
; Evaluate An
;
r3n:    add     dh, dh
        jnc     r3exit
        not     ax
        not     dl
r3exit: 
                ENDM

;----------------------------------------------------------------------------;
;                                                                            ;
; The bmScanAddrInc macro calculates the starting scanline address from the  ;
; y coordinate in AX.  The resulting address is in DX:AX.  The scanline      ;
; count in the resultant segment is saved in scanCount.  scanCount is        ;
; dependant on the y direction thus the two versions of this macro.          ;
;                                                                            ;
; Entry: ax        = Y coordinate                                            ;
;        scanSeg   = number of scanlines per segment                         ;
;        scanWidth = width of scanline in bytes                              ;
;        bmSeg     = beginning segment of bitmap                             ;
;        bmOff     = beginning offset of bitmap                              ;
;                                                                            ;
; Exit:  dx:ax     = scanline address of y coordinate                        ;
;        scanCount = scanline count in resultant segment                     ;
;                                                                            ;
;----------------------------------------------------------------------------;

bmScanAddrInc   MACRO scanSeg, scanWidth, scanCount, bmSeg, bmOff
                LOCAL sa1, sa2
        mov     dx, bmSeg
        inc     ax
sa1:
        cmp     ax, scanSeg
        jbe     sa2
        add     dx, __AHIncr
        sub     ax, scanSeg
        jmp     sa1
sa2:
        dec     ax
        mov     scanCount, ax
        mov     ax, scanSeg
        sub     ax, scanCount
        xchg    ax, scanCount
        imul    ax, scanWidth
        add     ax, bmOff
                ENDM

;----------------------------------------------------------------------------;
;                                                                            ;
; The bmScanAddrDec macro calculates the starting scanline address from the  ;
; y coordinate in AX.  The resulting address is in DX:AX.  The scanline      ;
; count in the resultant segment is saved in scanCount.  scanCount is        ;
; dependant on the y direction thus the two versions of this macro.          ;
;                                                                            ;
; Entry: ax        = Y coordinate                                            ;
;        scanSeg   = number of scanlines per segment                         ;
;        scanWidth = width of scanline in bytes                              ;
;        bmSeg     = beginning segment of bitmap                             ;
;        bmOff     = beginning offset of bitmap                              ;
;                                                                            ;
; Exit:  dx:ax     = scanline address of y coordinate                        ;
;        scanCount = scanline count in resultant segment                     ;
;                                                                            ;
;----------------------------------------------------------------------------;

bmScanAddrDec   MACRO scanSeg, scanWidth, scanCount, bmSeg, bmOff
                LOCAL sa1, sa2
        mov     dx, bmSeg
        inc     ax
sa1:
        cmp     ax, scanSeg
        jbe     sa2
        add     dx, __AHIncr
        sub     ax, scanSeg
        jmp     sa1
sa2:
        mov     scanCount, ax
        dec     ax
        imul    ax, scanWidth
        add     ax, bmOff
                ENDM

;----------------------------------------------------------------------------;
;                                                                            ;
; The bmScanInc macro increments the current scanline address.  The y        ;
; direction is positive (i.e. the next y coordinate = current y + 1).        ;
;                                                                            ;
; Entry: regOff    = current scanline offset register                        ;
;        regSeg    = current scanline segment register                       ;
;        scanSeg   = number of scanlines per segment                         ;
;        scanWidth = width of scanline in bytes                              ;
;        bmFill    = number of filler bytes at end of each segment           ;
;                                                                            ;
; Exit:  regOff    = next scanline offset                                    ;
;        regSeg    = next scanline segment                                   ;
;        scanCount = current scanline count in current segment               ;
;        ax        = garbage                                                 ;
;                                                                            ;
;----------------------------------------------------------------------------;

bmScanInc       MACRO regOff, regSeg, scanSeg, scanWidth, scanCount, bmFill
                LOCAL siexit
        add     regOff, scanWidth
        dec     scanCount
        jnz     siexit
        add     regOff, bmFill
        mov     ax, regSeg
        add     ax, __AHIncr
        mov     regSeg, ax
        mov     ax, scanSeg
        mov     scanCount, ax
siexit:
                ENDM

;----------------------------------------------------------------------------;
;                                                                            ;
; The bmScanDec macro decrements the current scanline address.  The y        ;
; direction is negative (i.e. the next y coordinate = current y - 1).        ;
;                                                                            ;
; Entry: regOff    = current scanline offset register                        ;
;        regSeg    = current scanline segment register                       ;
;        scanSeg   = number of scanlines per segment                         ;
;        scanWidth = width of scanline in bytes                              ;
;        bmFill    = number of filler bytes at end of each segment           ;
;                                                                            ;
; Exit:  regOff    = next scanline offset                                    ;
;        regSeg    = next scanline segment                                   ;
;        scanCount = current scanline count in current segment               ;
;        ax        = garbage                                                 ;
;                                                                            ;
;----------------------------------------------------------------------------;

bmScanDec       MACRO regOff, regSeg, scanSeg, scanWidth, scanCount, bmFill
                LOCAL sdexit
        sub     regOff, scanWidth
        dec     scanCount
        jnz     sdexit
        sub     regOff, bmFill
        mov     ax, regSeg
        sub     ax, __AHIncr
        mov     regSeg, ax
        mov     ax, scanSeg
        mov     scanCount, ax
sdexit:
                ENDM
