//
// Module Name: AVKAPI.H
//
// Description: The primary AVK application header file
//
// Status: Version 1 Release 2
//
//
// Copyright Intel Corp. 1990, 1993
//
// Copyright IBM Corp. 1990, 1993
// All Rights Reserved
//
// Description
// This file contains the API calls, the data definitions,
// notifications, error messages, and everything else interesting to apps,
// except for algorithm data, which is in AVKALG.H



// _OS2 is defined (e.g. in the makefile) if this is an OS2 compilation

#ifndef AVKAPI_DEFINED // avoid nesting this file
#define AVKAPI_DEFINED
//avk 1.2...
#define AVKAPI_VERSION 921023000L //actually, a date, a Vxx number , a digit

//AVKCOM.H defines many symbols that will commonly be used...
// to call AVK functions...
// to store, retrieve and interpret data about A/V motion/still files
// to call higher-level libraries that themselves call AVK.
// If you decide to study the AVKAPI.H file, it is recommended that
// you study AVKCOM.H and AVKALG.H as well.
// If you are searching for the actual definitions of symbols used in the
// AVK calls, you will find some of them defined in AVKCOM.H.
#include "AVKCOM.H" 			//common symbols...
// See AVKCOM.H for the following definitions...
// AVK_CALL definition
// PLANE definitions
// External Video keying types
// Audio impedance definitions
// Portable data types for AVK functions and data definitions
// Device Open Flags
// for video sync capture types returned by AvkDeviceVideoIn
// AVK_NTSC_ and AVK_PAL_ rate symbol definitions
// audchannel values -- how to record or playback audio

// audio algorithm definitions are in AVKCOM.H and AVKALG.H


#define AVK_MSG_MAX_NUM 		 128
#define AVK_MSG_BASE			 WM_USER		//AVK messages start here
#define AVK_USER				 (AVK_MSG_BASE + AVK_MSG_MAX_NUM)

#define AVK_NOCHANGE 0x80000000L

// AVK FRAME special defines...
#define AVK_NO_FRAME 0xFFFFFFFFL		//absence of a frame
#define AVK_NOTHING  (~0)				//absence of (de)compress, status info

// special pre-defined HANDLES...
#define  AVK_TIME_IMMEDIATE  0xFFFA
#define  AVK_VIEW_BLANK 	   0xFFF9	//blank view for any board

#define  AVK_CONN_DIGITIZER  0xFFF5 	//digitizer as connector source HANDLE
#define  AVK_SPECIAL_HANDLES 0xFFE0 	//reserve up to 16 special handles

//the two types of frame number for AvkStrmSetFrameNum
#define  AVK_SHOW_FRAME_NUM 	0		//the SHOWING frame number
#define  AVK_PROC_FRAME_NUM 	1		//the PROCESSED frame number (ready to show, etc)

//These symbols are defined for compatibility with release 1.0, but are
//no longer used. In effect, the new value of these two symbols is: 1
#define AVK_XRES_MULTIPLES 4			//image planar bitmaps xres is multiple of this
#define AVK_YRES_MULTIPLES 4			//image planar bitmaps yres is multiple of this

// status grabbing defines
#define AVK_STATUS_ALL  		((U16) (~0))  	//grab ALL possible status data
#define AVK_STATUS_DEFAULT  	((U16) (~0))  	//use with AvkGetStatus call

//MAXIMA for AVK... How many of each kind of object you may have...
#define AVK_MAX_SESSIONS 1
#define AVK_MAX_MATCH_FRAMES 4
#define AVK_MAX_BOARDS 1
#define AVK_MAX_IMAGES  30
#define AVK_MAX_IMGBUFS 8
#define AVK_ALLOWED_VIEWS 4 	//2 by spec, but allow more
#define AVK_MAX_VIEWS (AVK_ALLOWED_VIEWS+AVK_MAX_IMAGES)
#define AVK_MAX_GROUPS 4
#define AVK_MAX_BUFFERS 4
#define AVK_MAX_STREAMS_AUD 4	//audio play or capture, less for some audio rates

#ifdef _MULTISTREAM
#define AVK_MAX_STREAMS_VIDP 4  //video play streams
#define AVK_MAX_STREAMS 8		//total max in any case
#else
#define AVK_MAX_STREAMS_VIDP 1  //video play streams
#define AVK_MAX_STREAMS 4		//total max in any case
#endif

#define AVK_MAX_STREAMS_VIDC 1  //video capture streams
#define AVK_MAX_PHYSICAL_BUF_STRMS 32
#define AVK_MAX_DECODE_BITMAPS 8
#define AVK_MIN_DECODE_BITMAPS 4
#define AVK_MIN_ENCODE_BITMAPS 6
#define AVK_MAX_OBJCONS 32  			//this is: AVK_MAX_CONNECTORS

#define AVK_MAX_TIMES 8

//This XRES maximum defines the largest resolution allowed for either
//premonitoring or capture.
#define AVK_MAX_STREAM_XRES 306 		//maxima for streams (YUV9)
#define AVK_MAX_STREAM_YRES 288 		//288 for PAL (allow worst case always)
// The maximum xres for capture is actually smaller:
#define AVK_MAX_CAPSTREAM_XRES 256
// BUT the maximum xres for capture, for which the data can actually be
// PLAYED BACK is...
#define AVK_MAX_AVSTREAM_XRES (AVK_MAX_STREAM_XRES/2) //currently 153 for PAL

// group buffer directions....
#define AVK_BUF_PLAYBACK 1
#define AVK_BUF_CAPTURE 2



// group buffer sizes
#define AVK_GRP_BUF_MIN 65536L
#define AVK_GRP_BUF_MAX (512L*1024L)


//The two capture sources on the digitizer board...
#define AVK_DIGITIZER	AVK_CONN_DIGITIZER
#define AVK_DIGITIZER_1 AVK_CONN_DIGITIZER
// The following symbol is obsolete. Recompile with AVK_DIGITIZER_1 only!
//#define AVK_DIGITIZER_2 2

// capture connector option settings; the acceptable default value is zero.
#define AVK_PRE_MONITOR  1	//connector will only be used to pre-monitor.

//settings to turn monitoring off and on, in AvkDeviceAudioIn...
#define AVK_MONITOR_ON  1
#define AVK_MONITOR_OFF 0


////////////////// typedefs....


typedef I16  HAVK;  			// all handles

//To use the 32bit compiler, define the following symbol
//before including AVKAPI.H...
#ifdef _32BIT					//typedefs for IBM C SET 2 32-bit compiler...

typedef I16  * _Seg16 PHAVK;	// all handles, pointer to receive
#define AVKPTR * _Seg16         // A 'thunkable' ptr

#else							//alternate typedefs for MSC6 and IBMC (16bit)

typedef I16  far *PHAVK;		// all handles, pointer to receive
#define AVKPTR far *            // All AVK pointers are far pointers

#endif


//
//  This macro fills
//  reasonable default values for an rtv20args structure...
//  Use this macro as a tutorial... Be very careful to check its
// relevance to your specific situation, if you decide to invoke it.
//

/*
When capturing video data, attention must be paid to the xo, yo, xs, ys
values in the AVK_RTV_20_DEFAULT_ARGS algdata structure. The
xo and yo values should be multiples of 8; xs,ys should be multiples
of 16. The values that do not conform to these requirements
will be rounded down to the nearest multiple.
*/
#define AVK_RTV_20_DEFAULT_ARGS  {                                       \
        12,                                          /* arg count */     \
        AVK_RTV_2_0,                                 /* alg size */      \
        0,0,128,240,                                 /* xo,yo, xs, ys */ \
        3,                                           /* still period */  \
        0, 0,                                         /* Bytes,lines */  \
        AVK_RTV_20_PREFILTER | AVK_RTV_20_ASPECT_25, /* flags */         \
        2, 0                                         /* quant values */  \
}



///////////// enumerating typedefs...

//use these as hunger enables with AvkGrpBufWrite/Read...

#define   AVK_NO_CHANGE 		0
#define   AVK_ENABLE			1
#define   AVK_QUALIFIED_ENABLE  2		//for grpbuf: after incomplete write...
#define   AVK_DISABLE			3		//last item in list

// Connector effects...
#define  AVK_EFFECT_MIN 1
#define  AVK_EFFECT_TINT		AVK_EFFECT_MIN
#define  AVK_EFFECT_SATURATION  2
#define  AVK_EFFECT_BRIGHTNESS  3
#define  AVK_EFFECT_CONTRAST	4
#define  AVK_EFFECT_MAX AVK_EFFECT_CONTRAST



// bitmap format types
#define   AVK_YUV9  			AVK_BM_9		//see AVKCOM.H
#define   AVK_YUV16 			AVK_BM_16
#define   AVK_YUV24 			AVK_BM_24
#define   AVK_YUV12 			AVK_BM_12
//note, for compatibility with some existing applications,
//the old values 1,2,3,4 (respectively) for the above symbols will also be
//supported.
//The following symbol has been retained for compilation compatibility,
//but it has little semantic value and its use is now discouraged:
#define AVK_BITMAP_FORMAT_MAX ((U16) AVK_BM_9)


// output view-display types 
// ONLY AVK_VIEW_DISPLAY_DEFAULT is supported! These AVK_VIEW_DISPLAY
// symbols should not be used: _YC, _SYS, and _BOTH
#define   AVK_VIEW_DISPLAY_DEFAULT  	0
							//output to each of sys and Y-C if (in each case)
							//they can handle it.
#define   AVK_VIEW_DISPLAY_YC			1
							//view to Y-C connector; no change to XGA/VGA
							//unless analog keying AND view is VID/PAL
							//in which case switch xga/vga to blank view.
#define   AVK_VIEW_DISPLAY_SYS  		2
							//view to keyed monitor (normal case) AND
							// BLANK the Y-C connector output.
#define   AVK_VIEW_DISPLAY_BOTH 		3
							//output to both. not allowed if this view is keyed.

//The following flags can be Or'd into the OutputDev argument to
//force VGA and XGA displays to change between Keyed and unkeyed...
#define AVK_VIEW_DISPLAY_KEYED		0x4000
#define AVK_VIEW_DISPLAY_UNKEYED	0x2000


//use this to open your AVK session (begin or callback)
#define AVK_SESSION_DEFAULT (U32) (AVK_BUFFER_HUNGRY_O \
						| AVK_IDENTIFY_O \
						| AVK_BUFFER_STARVED_O \
						| AVK_GROUP_READY_O \
						| AVK_MATCH_FRAME_O \
						| AVK_GROUP_PAUSED_O \
						| AVK_GROUP_OUT_OF_SYNC_O \
						| AVK_PRESSING_DONE_O \
						| AVK_GROUP_FLUSHED_O \
						| AVK_NOTIFICATIONS_LOST_O \
						| AVK_CAPTURE_DATA_AVAILABLE_O \
						| AVK_UCODE_STALL_O \
						| AVK_CHECKSUM_FRAMES_O \
						| AVK_SCSI_EVENT_O \
						| AVK_CAUTION_O \
)


// MESSAGES SENT TO THE APPLICATION \\ option bits for each message...
#define  AVK_IDENTIFY			  (AVK_MSG_BASE +  0)
#define  AVK_IDENTIFY_O 			 (2L	 )
#define  AVK_BUFFER_HUNGRY  	  (AVK_MSG_BASE + 1)
#define  AVK_BUFFER_HUNGRY_O				  (2L<<1)  //AVK_SESSION bits...
#define  AVK_BUFFER_STARVED 	  (AVK_MSG_BASE + 2)
#define  AVK_BUFFER_STARVED_O				  (2L<<2)
#define  AVK_GROUP_READY		  (AVK_MSG_BASE + 3)
#define  AVK_GROUP_READY_O  				  (2L<<3)

#define  AVK_MATCH_FRAME		  (AVK_MSG_BASE + 5)
#define  AVK_MATCH_FRAME_O  				 (2L<< 5)
#define  AVK_UCODE_STALL		  (AVK_MSG_BASE + 6)
#define  AVK_UCODE_STALL_O  				 (2L<< 6)
#define  AVK_GROUP_PAUSED		  (AVK_MSG_BASE + 7)
#define  AVK_GROUP_PAUSED_O 				 (2L<< 7)
#define  AVK_GROUP_OUT_OF_SYNC    (AVK_MSG_BASE + 8)
#define  AVK_GROUP_OUT_OF_SYNC_O			 (2L<< 8)
#define  AVK_PRESSING_DONE  	  (AVK_MSG_BASE + 9) //(de)compress complete
#define  AVK_PRESSING_DONE_O				 (2L<< 9) //(de)compress complete
#define  AVK_GROUP_FLUSHED  	  (AVK_MSG_BASE + 10)
#define  AVK_GROUP_FLUSHED_O				 (2L<< 10)
#define  AVK_NOTIFICATIONS_LOST   (AVK_MSG_BASE + 11)
#define  AVK_NOTIFICATIONS_LOST_O			 (2L<< 11)
#define  AVK_GROUP_COMMITTED	  (AVK_MSG_BASE + 13)
#define  AVK_GROUP_COMMITTED_O  			 (2L<< 13)
#define  AVK_CAPTURE_DATA_AVAILABLE (AVK_MSG_BASE + 14)
#define  AVK_CAPTURE_DATA_AVAILABLE_O		(2L << 14)
#define  AVK_SCSI_EVENT  		  (AVK_MSG_BASE + 15)
#define  AVK_SCSI_EVENT_O					(2L << 15)
// The AVK_CAUTION notification can be sent to Windows programs,
// but not to OS/2 programs:
#define  AVK_CAUTION				  (AVK_MSG_BASE + 16)
#define  AVK_CAUTION_O				(2L << 16)

//The following option bits are reserved and shall not be defined in this file:
// (2L << 25) ... (2L << 28)

//the next option requests checksumming of write frame headers
//and rejection if checksum fails... THIS DOES NOT GENERATE A NOTIFICATION.
//NOTE: AVK does not commit to check every frame for proper checksumming,
//because AVK does not (in the long run) commit to parsing frame data at all
// for efficiency reasons. The following option tells AVK that whenever
//it happens to KNOW it has a frame header, it should checksum...
#define AVK_CHECKSUM_FRAMES_O				 (2L << 29)

//these option bits are reserved:
// (2L<<28)
// (2L<<27)

//these are not a default options...
//the next message (not standard) indicates
//that HAVK PARAM1 has been implicitly destroyed.
#define  AVK_OBJECT_SWAT		  (AVK_MSG_BASE + 12)
#define  AVK_OBJECT_SWAT_O  				 (2L<< 12)

#define AVK_MSG_HIGH (AVK_MSG_BASE + 15)
#define AVK_MSG_LOW  (AVK_MSG_BASE + 0)

// video display sync types
#define   AVK_VID_XGA			0x0100  //xga, no keying
#define   AVK_VID_XGA_KEYED 	0x8100  //xga plus keying
#define   AVK_VID_VGA			0x0010  //vga, no keying
#define   AVK_VID_VGA_KEYED 	0x8010  //vga plus keying
#define   AVK_VID_NTSC  		0x0001  //ntsc, no keying, dual screen
#define   AVK_VID_PAL			0x0002  // pal, no keying, dual screen


// ERROR CODES
//warnings...
#define AVK_ERR_WARNINGS 9 //0 to 9 are warnings. Above this, errors.

#define AVK_ERR_OK  	  0
#define AVK_WARN_NOHANDLE 1 			//handle table full, no more
#define AVK_WARN_HOSTMEM  2 			//can't allocate near or far host mem

										//the next symbol has two spellings
										//for compatibility with existing
										//programs...
#define AVK_WARN_TRUNCATION 4			//image read/write length truncated
#define AVK_ERR_WARN_TRUNCATION 4		//image read/write length truncated
										//OR, clip list truncated!



//errors...
#define AVK_ERR_HANDLE 10				//handle is invalid, or
										//not the correct type for
										//this operation
#define AVK_ERR_SESS   11				//not a valid session handle
#define AVK_ERR_DEVICE 12				//not a valid device handle
#define AVK_ERR_DEVNUM 13				// device number was not zero
#define AVK_ERR_VIEW   14				//not a valid view handle
#define AVK_ERR_TIME   15				//ditto time
#define AVK_ERR_GRP    16				//ditto group
#define AVK_ERR_GRPBUF 17				//ditto grpbuf
#define AVK_ERR_IMG    18				//ditto image
#define AVK_ERR_IMGBUF 19				//ditto imgbuf
#define AVK_ERR_STRM   20				//ditto stream
#define AVK_ERR_CONN   21				//ditto connector (objcon)
#define AVK_ERR_LIM_TOOMANY 22  		//cannot create any more of that object //ahs
#define AVK_ERR_LIM_IMMED	23  		//MUST use immediate time handle
#define AVK_ERR_LIM_TIME	24  		//MUST use REAL time handle
#define AVK_ERR_LIM_VIEW	25  		//unsuppt combo of view params
#define AVK_ERR_LIM_RES 	26  		//unsuppt bitmap size
#define AVK_ERR_BUSY_GRP	27  		//group playing or capturing
#define AVK_ERR_BUSY_VIEW	28  		//view is showing motion  can't delete it
#define AVK_ERR_BUSY_MON	29  		//not used yet //device is busy monitoring
#define AVK_ERR_BUSY_PRSG	30  		//(de)compression blocks this
#define AVK_ERR_UNSUP		31  		//this call not implemented yet.
#define AVK_ERR_WRONGAUD	32  		//can't mix audio algs
#define AVK_ERR_UNSUPAUD	33  		//no ucode avail for this audio param combo
#define AVK_ERR_CONDIG  	34  		//unsupp digitizer feature

#define AVK_ERR_NOTDIG  	36  		//digitizer required
#define AVK_ERR_VERS		37  		//unused
#define AVK_ERR_BMFMT		38  		//unknown or illegal (in this context) bitmap format.
#define AVK_ERR_VIDTYPE 	39  		//unknown vid type
#define AVK_ERR_ALG 		40  		//video algorithm not found
#define AVK_ERR_ALGARG  	41  		//invalid algorithm arguments.
#define AVK_ERR_ALGDATA 	42  		//invalid algorithm data
#define AVK_ERR_SRCBOX  	43  		//src  box coords in error
#define AVK_ERR_DSTBOX  	44  		//dest box coords in error
#define AVK_ERR_FLAG		45  		//unrecognzed flag setting
#define AVK_ERR_ENUM		46  		//unrecognzed enumerated const setting
#define AVK_ERR_BOOL		47  		//boolean arg  must be 0 or 1.
#define AVK_ERR_PRCNT		48  		//volume arg was over 200%
#define AVK_ERR_RANGE		49  		//value out of range
#define AVK_ERR_CONFIG  	50  		//file missing, or DviMonitorSync line is silly
#define AVK_ERR_OS  		51  		//function not available for this OS
#define AVK_ERR_BUFTYPE 	53  		//read operation on write bfr or vv
#define AVK_ERR_FAR 		54  		//probably not a good far pointer (or length>65535)
#define AVK_ERR_COMMIT  	56  		//group has unformatted streams or is incomplete
#define AVK_ERR_CONSRC  	57  		//connector mentions invalid src handle
#define AVK_ERR_CONDST  	58  		//connector mentions invalid dst handle

#define AVK_ERR_WRONG_CLIP_RANGE		59 //copy list coordinate error
#define AVK_ERR_NULL_POINTER_PASSED 	60 //call required a far pointer param
#define AVK_ERR_SYS_VIDEO_MODE  		61 //incorrect video mode or session(MSG/CALLB) type
#define AVK_ERR_INVALID_VALUE			62 //argument value unacceptable
#define AVK_ERR_DEVIOCTL				63 //ERROR COMMUNICATING WITH avd.sys  IN os2
#define AVK_ERR_INTERNAL				64 //A dll programming error has occurred.
#define AVK_ERR_COMMITTED				65 //too late to define/create  resources committed
#define AVK_ERR_COMMITTING  			66 //error while committing board resources.
#define AVK_ERR_UNCOMMITTING			67 //error while uncommitting board resources.
#define AVK_ERR_PAUSE_ALREADY_SCHEDULED 68 //can't sched pause when pause already scheduled
#define AVK_ERR_INVALID_PLAY_STATUS 	69 //can't do this op at this time
#define AVK_ERR_DEVDATA 				70 //failed to read or write device data
#define AVK_ERR_FUNCTION				71 //rejecting concurrent call to KMVL (NB: OS/2 accepts concurrent calls)
#define AVK_ERR_SYSCALL 				72 //native OS unexpectedly returned an error
#define AVK_ERR_RESOURCES				73 //call failed due to insufficient resources (any dll)
#define AVK_ERR_UCODE_MISSING			74 //missed ucode that's always required
#define AVK_ERR_DEVOPEN 				75 //Device Open failed
#define AVK_ERR_BOARD_IS_CLOSED 		76 //op allowed only if device is open
#define AVK_ERR_CHECKSUM				77 //checksum error in frame write data
#define AVK_ERR_BUFFER_TOO_SMALL		78 //file requires larger grpbuf, e.g. 3*lgest frm
#define AVK_ERR_DUPLICATE_CONNECTOR 	79 //illegal attempt to enable duplicate connector
#define AVK_ERR_UCODE_ERROR 			80 //ucode issues "panic" interrupt
#define AVK_ERR_UDATA_BAD				81 //ucode issues "bad data"  interrupt
#define AVK_ERR_AP_STALLED  			82 //audio micro processor not responding
#define AVK_ERR_PP_STALLED  			83 //pixel micro processor not responding
#define AVK_ERR_UU_STALLED  			84 //micro processors not responding
#define AVK_ERR_INDEX					85 //AVK structure[index] (indexing) INTERNAL error.
#define AVK_ERR_GETSTATUS				86 //must get status before making status calls
#define AVK_ERR_SUBSYSTEM				87 //AVK inadvertently failed to supply its own error code
										   //(should never happen, but might)
#define AVK_ERR_BITMAP_TOO_SMALL		88 //(de)compression data too large to fit

#define AVK_ERR_DIG_IN_PROGRESS 		89 //can't do this while digitizing

#define AVK_ERR_CANT_MIX_FRAME_RATES	90 //can't mix frame rates in a group buffer

#define AVK_ERR_MONITOR_WHILE_PLAY  	91 // cant monitor and playback
										   // together (audio only)

#define AVK_ERR_DEVMEM  				92 //can't allocate device mem

#define AVK_ERR_DIGISTARTUP 			93 //problem setting up digitizer

#define AVK_ERR_NODATA  				94 //no data available.

#define AVK_ERR_SEMA_STALLED			95 //gave up waiting for VRAM semaphore
#define AVK_ERR_LIM_SYNC				96 //current display sync disallows
										   //stream or connector parameter





/////////// structures....

typedef struct
{
		U32  AvkVram;			//in bytes, such as 2*1024L*1024L...
		U16 DisplayRevLevel; //note. these values are not available
								//from actual hardware...
		U16 DigitizerRevLevel;
		U16 AvkRevLevel;		//these are available...
		U16 AvkRevLevelTenths;
		U16 AvkRevLevelHundredths;
//the digitizer signals are not in this table becuase they take 1 to 2 secs
//to calculate. There will be a notification message instead.

//for these fields, the values are as in AVK.INI...
		U16 DviMonitorSync;
// 0:none	1:NTSC   2:PAL   16:VGA mode 18   256:XGA 1024x768

		U16 SysMonitorSync;
// 16: VGA mode 18  [640 x 480]  	   256: XGA mode [1024 x 768]

		U16 DviExtConn;
// 0: no keying 	1: analog keying (Y cable)  2: digital (windows only)
} DVICAPS;

typedef struct
{
	I16 x1, y1, x2, y2;
} BOX;

typedef BOX AVKPTR PBOX;

//These are the status data structures. Over time, they can change
//by the addition of additional data fields. If a program is not recompiled
//and requests less than the full structure, it will receive just
//what it asked for. In the structure definitions below, the presence
// of a cutoff line:  //------------------------------------
// shows the optional substructure lengths that are supported.


typedef struct  {		//volatile status data for a board..
		U32 TotalFreeVram;
		U32 DisplayVbiCount;
		U32 Unused;
		U32 AudVbiCount;
		U16 VidEventFlag; //reason for last interrupt by PP
		U16 AudEventFlag; //reason for last interrupt by AP
		U16 UcodeError;   //TRUE if very serious errors have occurred, e.g.:
						  //MVD_EVENT_... indicating serious error
		U16 DataPresent;  //non-zero if grab really happened for this item
		//-------------------------------------
		U32 LargestFreeVram;
} AVKSTATUSDEV;


typedef struct { //current system time or "relative" current system time
		I16 hour;		// 0-23
		I16 minute; 	// 0-59
		I16 second; 	// 0-59
		I16 hsecond;	// 0-99  (hundredths of seconds)
		U16 DataPresent; //non-zero if grab really happened for this item
} AVKSTATUSTIME;



typedef struct {		//volatile status data for a group...
		U16 playing;			//these are all boolean...
		U16 Recording;
		U16 Committed;
		U16 InTransition;		//TRUE if group hasn't reached desired
								//status yet (e.g. PLAYING, PAUSED)
		U16 DataPresent; //non-zero if grab really happened for this item
} AVKSTATUSGRP;


typedef struct {		//volatile status data for a buffer...
		U32 FreeBeginBufCt; 	// 0 or more free bytes at beginning of buffer
		U32 FreeEndBufCt;		// 0 or more free bytes at end of buffer
		U32 InUseCt;	// 0 or more bytes containing data
		U16 InError;	//TRUE if any A/V data errors have been detected
		U16 DataPresent; //non-zero if grab really happened for this item
} AVKSTATUSGRPBUF;


typedef struct {		//volatile status data for a stream...
		U32 ShowFrameNum; //current "on display" frame number, if ANY is current.
		U32 ProcFrameNum; //current processed frame number
		U16 StrmState; //This is StrmStat in the microcode, see below
						  //the above field sets the CATCHUP STRATEGY IN UCODE.
		U16 InSync; 	//TRUE if stream is regarded as currently in synch
						// (play direction only)
		I16 LoadedBitmaps; //The NUMBER of bitmaps with decompressed data,
						// if this is easy to determine.. (both cap and play)
		U16 DataPresent; //non-zero if grab really happened for this item
} AVKSTATUSSTRM;
// bits of signifigance in the StrmState value...
#define STRM_STAT_SLOW  0x10	//enable slowing down to frame rate
#define STRM_STAT_FAST  0x20	//enable speeding up  to frame rate


typedef struct {		//volatile status data for an image...
		U32 CompressedLen; //amount of compressed data, or zero.
						//very interesting in the case of capture.
						//it's undefined while "Pressing" is true.
		HAVK hImage;	//identify which image is being pressed, if any.
		U16 Pressing; //TRUE if active compress or decompress is occurring
		U16 DataPresent; //non-zero if grab really happened for this item
} AVKSTATUSIMGBUF;




////////// calls.....

//RULES FOR Data-typing the parameters in the calls:
// - anything that counts or numbers frame is U32
// - BOX coords are int
// - all other numeric data are I16.
// - all boolean, symbol, enumerated or flag data is U16
// - AVK_CALL definition, see avkcom.h

//BELOW: "undef" means call returns an error, it's not in release 1.

AVK_CALL AvkMatchFrame(HAVK hTime);
AVK_CALL AvkMatchFrameCancel(HAVK hTime);
AVK_CALL AvkDeviceVideoIn(HAVK hDev, U16 Digitizer);
AVK_CALL AvkDeviceAudioIn(HAVK hDev, U16 Audtype, U16 AudMonitor);

AVK_CALL AvkBeginMsg(HWND hWnd,  PHAVK phSess, U32 options);

#ifdef _32BIT					//see above...
AVK_CALL AvkBeginCallback(
	void ( * AVK_CALLBACK pcallback )(U32, HAVK, I16, U16, U32), 
	U32 id, PHAVK phSess, U32 options); 
#else
AVK_CALL AvkBeginCallback(
	void (AVK_CALLBACK *pcallback )(U32, HAVK, I16, U16, U32), 
	U32 id, PHAVK phSess, U32 options); 
#endif

// you should prototype your callback routine as follows....
//  void  AVK_CALLBACK myroutine(U32 id, HAVK hsession, I16 msg,
//  					U16 param1, U32 param2).


AVK_CALL AvkEnd(HAVK hSess);
AVK_CALL AvkGetDevCaps(I16 BoardNum, I16 Size, DVICAPS AVKPTR pCaps);
AVK_CALL AvkGetSubSysErrCode(HAVK hSess, U16 AVKPTR  pErrCode);
AVK_CALL AvkDeviceOpen(HAVK hSess, I16 DevNum, U16 Flags, PHAVK phDev);
AVK_CALL AvkDeviceClose(HAVK hDev);
AVK_CALL AvkTimeStrmFrameCreate(HAVK hstrm, U32 frame_no, PHAVK htime);
AVK_CALL AvkTimeDelete(HAVK htime);

AVK_CALL AvkGetStatus(HAVK hSess, HAVK hTime, U16 Aspects);
AVK_CALL AvkTimeStatus(HAVK hSess, I16 Len, AVKSTATUSTIME AVKPTR pStru);
AVK_CALL AvkDeviceStatus(HAVK hDev, I16 Len, AVKSTATUSDEV AVKPTR pStru);
AVK_CALL AvkGrpStatus(HAVK hGroup, I16 Len, AVKSTATUSGRP AVKPTR pStru);
AVK_CALL AvkGrpBufStatus(HAVK hGrpBuf, I16 Len, AVKSTATUSGRPBUF AVKPTR pStru);
AVK_CALL AvkStrmStatus(HAVK hStrm, I16 Len, AVKSTATUSSTRM AVKPTR pStru);
AVK_CALL AvkImgBufStatus(HAVK hImgBuf, I16 Len, AVKSTATUSIMGBUF AVKPTR pStru);

AVK_CALL AvkViewCreate(HAVK hDev, I16 xres, I16 yres, U16 BitmapFmt,
	U16 VidType, PHAVK phView);
AVK_CALL AvkViewDelete(HAVK hView);
AVK_CALL AvkViewDisplay(HAVK hDev, HAVK hView, HAVK hTime, U16 OutputDev);
AVK_CALL AvkViewCleanRect(HAVK hView, BOX AVKPTR  Rect);

AVK_CALL AvkGrpCreate(HAVK hDev, PHAVK phGrp);
AVK_CALL AvkGrpDelete(HAVK grp);
AVK_CALL AvkGrpStart(HAVK hgrp, HAVK htime);
AVK_CALL AvkGrpStep(HAVK hgrp, HAVK htime); //unsupported.

AVK_CALL AvkGrpPause(HAVK hGrp, HAVK hTime);
AVK_CALL AvkGrpPauseCancel(HAVK hGrp, HAVK hTime);
AVK_CALL AvkGrpFlush(HAVK hGrp);
AVK_CALL AvkGrpTempo(HAVK hGrp, I16 Percent, HAVK hTime);
AVK_CALL AvkGrpResync(HAVK hStrm);

AVK_CALL AvkGrpBufCreate(HAVK hgrp, U16 type, U32 bufsize, U32 hunger_gran,
	U16 strm_ct, PHAVK phGrpBuf);

AVK_CALL AvkGrpBufDelete(HAVK hbuf);
AVK_CALL AvkStrmSetFrameNum(HAVK hand,  U16 FrameType, U32 FrameNum);

AVK_CALL AvkGrpBufWrite(HAVK hbuf, U32 len,
	void AVKPTR pData, U32 AVKPTR pbytes_written, U16 HungerEnable);

AVK_CALL AvkGrpBufGetSpec(HAVK hBuf, I16 Len, U32 AVKPTR pInfoBuffer); //do not use.
AVK_CALL AvkGrpBufRead(HAVK hbuf, U32 Max,
	void AVKPTR pDestin, U32 AVKPTR pBytesRead, U16 HungerEnable);

AVK_CALL AvkVidStrmCreate(HAVK hbuf, I16 strm_index, PHAVK newh);

AVK_CALL AvkStrmDelete(HAVK hstrm);
AVK_CALL AvkVidStrmFormat(HAVK hstrm,
	I16 nr_of_bitmaps,
	I16 xres, I16 yres,
	U16 bitmap_format,
	U32 frame_rate, 	// microseconds per frame
	U16 decode_alg_name,
	void AVKPTR alg_args,
	I32 arg_len,
	I32 arg_area_size,
	void AVKPTR work_args,
	I32 work_len,
	I32 work_area_size);


AVK_CALL AvkStrmThreshold(HAVK hstrm, U32 frames);

AVK_CALL AvkAudStrmCreate(HAVK hbuf, I16 strm_index, PHAVK newh);
AVK_CALL AvkAudStrmFormat(HAVK hstrm,
	U32 frame_rate, 	// microseconds per frame
	U32 SamplesPerSecond,
	U16 decode_alg_name,
	U16 AudChannel, 	//for a good default value, use: AVK_AUD_MIX
	void AVKPTR  pAlgData,
	I32 AlgDataSize,
	I32 AlgDataSpace);

AVK_CALL AvkAudStrmVolume(HAVK hStrm, I16 Left, I16 Right);

AVK_CALL AvkImgCreate(HAVK hDev, I16 xres, I16 yres, U16 BitmapFormat, PHAVK phImage);
AVK_CALL AvkImgDelete(HAVK hImage);
AVK_CALL AvkImgWrite(HAVK hImage, U16 Planes, U32 Len,
		void AVKPTR pData, U32 Offset);
AVK_CALL AvkImgRead(HAVK hImage, U16 Planes, U32 Len, U32 AVKPTR  pActuallen,
		void AVKPTR pData, U32 Offset);
AVK_CALL AvkImgBufCreate(HAVK hDev, U32 BufSize, PHAVK phImgBuf);
AVK_CALL AvkImgBufDelete(HAVK hImgBuf);
AVK_CALL AvkImgBufWrite(HAVK hImgBuf, U32 Len, void AVKPTR pData,
		 U32 Offset);
AVK_CALL AvkImgBufRead(HAVK hImgBuf, U32 MaxLen, U32 AVKPTR pActuallen,
		void AVKPTR pData, U32 Offset);

AVK_CALL AvkImgCompress(HAVK hImgBuf, U16 CompressionFormat, HAVK hImage,
		void AVKPTR  pAlgData, I32 AlgDataSize, I32 AlgDataSpace,
		void AVKPTR  pWorkData, I32 WorkDataSize, I32 WorkDataSpace );
AVK_CALL AvkImgDecompress(HAVK hImgBuf, U16 CompressionFormat, HAVK hImage,
		void AVKPTR  pAlgData, I32 AlgDataSize, I32 AlgDataSpace,
		void AVKPTR  pWorkData, I32 WorkDataSize, I32 WorkDataSpace );

AVK_CALL AvkConnCreate( HAVK hSource, BOX AVKPTR pSrcBox, HAVK hDest,
				   BOX AVKPTR pDstBox, U16 options,PHAVK phConn);
AVK_CALL AvkConnDelete(HAVK hConn);
AVK_CALL AvkConnModSrcDst(HAVK hConn, BOX AVKPTR psrcbox,
		BOX AVKPTR pdstbox, HAVK hTime);
AVK_CALL AvkConnCopyList(HAVK hConn, I16 nBoxes, BOX AVKPTR pBoxArrayz,
		HAVK hTime);
AVK_CALL AvkConnEnable(HAVK hConn, HAVK hTime);
AVK_CALL AvkConnDisable(HAVK hConn, HAVK hTime);
AVK_CALL AvkConnHide(HAVK hConn, HAVK hTime);
AVK_CALL AvkConnModEffect(HAVK hConn, U16 WhichEffect, I16 Value, HAVK hTime);

#include "avkalg.h"  //include algorithm-related definitions...
#endif //ifndef AVKAPI_DEFINED

