#ifndef __vsAuth_h__
#define __vsAuth_h__
/*
**  This work is held in copyright as an unpublished work by Silicon
**  Graphics, Inc. All rights reserved.
*/


#include <vizserver/vs.h>
#include <vizserver/vsLoadable.h>
#include <vizserver/vsporting/vsAuth.h>

class vsAuth;


// Fields are name/value pairs where the value is always a string. The
// interpretation of the string is left to the vsAuth object.
class vsAuthField {
public:

    // Creating and Destroying
    vsAuthField(const char* name, const char* defValue=NULL,
	    vsBool aSecret=FALSE);

    ~vsAuthField();

    // Setting and Getting
    const char* getName() const { return name; }

    void setValue(const char* value);

    const char* getValue() const { return value; }

    vsBool isSecret() const { return fSecret; }

private:
    char* name;
    char* value;
    vsBool fSecret;
};


// Set of fields for the vsAuth object.
class vsAuthFieldSet {
public:
    // Creating and Destroying
    vsAuthFieldSet();
    vsAuthFieldSet(vsAuthField** adoptedFields, unsigned int count);
    virtual ~vsAuthFieldSet();

    // Getting and Setting
    unsigned int getFieldCount() const { return count; }
    vsAuthField* getField(unsigned int i) const;

    void appendField(vsAuthField* field);

private:
    vsAuthField** fields;
    unsigned int count, limit;
};


// Authenticated local user.
class vsAuthUser {
public:
    // Creating and Destroying
    vsAuthUser(const char* name, const char* pwd, long uid, long gid,
	    gid_t* gidSet, unsigned int gidSetCount, const char* shell,
	    const char* homeDir);

    virtual ~vsAuthUser();

    // Force the current process to become the user identified by this
    // vsAuthUser object. This action is irrevokable.
    // User Control
    virtual vsStatus become() const;

    // Return the name of the user
    // Getting and Setting
    const char* getName() const { return name; }

    // Return the encrypted password for this user
    const char* getEncPasswd() const { return pwd; }

    // Return the userid or group id of this user
    long getUID() const { return uid; }

    long getGID() const { return gid; }

    // Return the group membership set for this user
    const gid_t* getGIDSet(unsigned int& count) const
	{ count = gidSetCount; return gidSet; }


    // Return the default shell for this user
    const char* getShell() const { return shell; }


    // Return the home directory for this user
    const char* getHomeDir() const { return homeDir; }

protected:
    // Have the current process assume the identity of the user identified
    // by this vsAuthUser object.
    vsStatus assume() const;

    // Restore the identity of the current process to the real user.
    vsStatus restore() const;


private:
    char* name;
    char* pwd;
    long uid, gid;
    gid_t* gidSet;
    unsigned int gidSetCount;
    char* shell;
    char* homeDir;
};


// Data produced by the vsAuth (typically the authenticator).
class vsAuthRefData;
class vsAuthData {
public:
    // Creating and Destroying
    vsAuthData();
    vsAuthData(void* data, unsigned int length);
    vsAuthData(const vsAuthData& data);

    ~vsAuthData();


    // Initialization
    void init(void* data, unsigned int length);


    vsAuthData& operator=(const vsAuthData& data);

    // Getting
    const void* getData() const;

    unsigned int getLength() const;

private:
    vsAuthRefData* data;
};


// what is required
//	given a set of fields return an object that represents the
//      local user or an error.
//	vsAuth* create(const char* type, const char* host, const char* svc);
//	vsAuthFieldSet* getFields()
//	vsStatus genAuthRequest(vsAuthData& req);
//	vsStatus handleAuthRequest(const char* data, uint_t len,
//		vsAuthData*& reply);
//	vsStatus handleAuthReply(const char* data, uint_t len);
// What has to happen?
//	given a set of fields generate an authenticator
//	communicate that authenticator to the server
//	server verifies the authenticator is valid and returns an optional
// 	    reply (for mutual authentication)

// The vsAuthContext retains contextual information for a particular
// authenticated session between two principals.
class vsAuthContext {
public:
    // Destruction
    virtual ~vsAuthContext();

    // Getting
    // Returns an authenticator that can be presented to the other
    // principal for authenticating this principal.
    virtual vsStatus getAuthenticator(vsAuthData& data)=0;

    // Given the authenticator in data (of length length), authenticate
    // verifies the identity of the sender.
    virtual vsStatus authenticate(const char* data, unsigned int length)=0;

    // Returns the user (primarily for server consumption) that identifies
    // the client user on this server.
    virtual vsStatus getUser(const vsAuthUser*& user)=0;

    // User Creation
    vsAuthUser* createUser(uid_t uid);

    vsAuthUser* createUser(const char* userName);

protected:
    vsAuthContext();
    virtual vsAuthUser* createAuthUser(const char* name, const char* pwd,
	    long uid, long gid, gid_t* gidSet, unsigned int gidSetCount,
	    const char* shell, const char* homeDir);

private:
    vsAuthUser* createUser(struct passwd* pwd);
};


class vsAuth : public vsLoadable {

public:
    // Destruction
    virtual ~vsAuth();

    // Field Creation
    vsStatus createFields(vsAuthFieldSet*& fields) const;

    virtual vsStatus createContext(const char* hostname, const char* service,
	    const vsAuthFieldSet* fields, vsAuthContext*& context) const=0;

    virtual vsStatus createContext(const char* service, const void* data,
	    unsigned int length, vsAuthContext*& context) const=0;

    // Getting
    const char* getName() const;


protected:
    vsAuth();

    virtual vsStatus addFields(vsAuthFieldSet*& fields) const;

private:
    friend class vsAuthDB;
    const void* type;
    void initType(const void* type_) { type = type_; }
};

#define vsAuthImplement(_class, _desc) \
    vsLoadableImplement(vsAuth, _class, _desc)


extern vsStatus vsAuthLogOpen(const char* name);
extern void vsAuthLogClose();
extern void vsAuthLog(const char* user, const char* msg, ...);

#endif
