#if 0
    Copyright 1999-2000, Silicon Graphics, Inc.
    ALL RIGHTS RESERVED

    UNPUBLISHED -- Rights reserved under the copyright laws of the United
    States.   Use of a copyright notice is precautionary only and does not
    imply publication or disclosure.

    U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
    Use, duplication or disclosure by the Government is subject to restrictions
    as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
    in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
    in similar or successor clauses in the FAR, or the DOD or NASA FAR
    Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
    2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.

    THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
    INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
    DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
    PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
    GRAPHICS, INC.
#endif

#include <vizserver/vs.h>
#include <vizserver/vsCompressor.h>
#include <vizserver/vsFrame.h>
#include <vizserver/vsLock.h>
#include <vizserver/vsCompressorMsg.h>
#include <string.h>
#include <assert.h>
#include <stdio.h>
#include <map>

#define PARAMCAST (vsParamTypeCls*&)

class PersistantStorage {
public:
    PersistantStorage() : frameBuffer(NULL) { }
    ~PersistantStorage() { }

    virtual void create(const vsFieldInfo* inField) {
        int size = inField->getHeight() * inField->getWidth() *
                   vsFieldPixelBytes(inField->getFormat(), inField->getType());

        frameBuffer = new uchar_t [size];
        memset(frameBuffer, 0, size);
    }

    virtual void destroy(const vsFieldInfo*) {
        if (frameBuffer != NULL) {
            delete [] frameBuffer;
            frameBuffer = NULL;
        }
    }

    uchar_t* frameBuffer;
};

// Check for SGI o32 object compiles (see vsporting/vs.h)
#ifdef SGI_O32
typedef map<int, PersistantStorage*>            DrawableMap;
typedef map<int, PersistantStorage*>::iterator  DrawableMapIter;
#else
typedef std::map<int, PersistantStorage*>            DrawableMap;
typedef std::map<int, PersistantStorage*>::iterator  DrawableMapIter;
#endif

class NOPIFCompressor : public vsCompressor {
public:
    //All compressors need this declaration
    vsCompressorDeclare

    virtual size_t getMaxCompressedSize(const vsCmprParamArg& fieldInfo);
    virtual size_t compress(const vsCmprInputQueryArg& input,
                                  vsCmprOutputQueryArg& output);
    virtual void expand(const vsCmprInputQueryArg& input,
                              vsCmprOutputQueryArg& output);
    virtual void receiveMessage(const vsCmprParamArg& msg);	

private:
    DrawableMap drawableMap;	
};


// This is the Compressor Info block required of all compressors
static NOPIFCompressor::Info nopIFCInfo = {
    (char*)"NOP IF compressor",

    // Note! Because of our persistant storage mechinism, our compressor
    // is not thread safe from Vizserver's point of view, so let it know
    // that. This however does not prevent us from makeing our own threads
    // internal to the compression module.
    vsCompressorModuleTypes::ThreadUnsafe,

    // The following is the set of supported input pixel formats/types
    vsFieldPixelFormats::RGBAMask,
    vsFieldPixelTypes::UByteMask,

    // The following iss the set of supported output pixel formats/types
    vsFieldPixelFormats::RGBAMask,
    vsFieldPixelTypes::UByteMask,
};

// This macro generates the code needed for the dynamic loading
// mechanism of OpenGL Vizserver.
vsCompressorImplement(NOPIFCompressor, &nopIFCInfo, "NOP-IF")

size_t
NOPIFCompressor::getMaxCompressedSize(const vsCmprParamArg& input)
{
    if (input.isParamPresent(vsCmprParamTypes::FieldInfoParam)){
        vsFieldInfo* info = NULL;
        if (input.getParam(vsCmprParamTypes::FieldInfoParam,PARAMCAST info) == vsErrNone) {
            if (info->getParamName() == vsCmprParamTypes::FieldInfo) {
                int bpp = vsFieldPixelBytes(info->getFormat(),info->getType());
                return info->getWidth()*info->getHeight()*bpp;
            }
        }
    }

    assert(!"nopIFCompressor::getMaxCompressedSize - isParamPresent failed");
    return 0;
}

void 
NOPIFCompressor::receiveMessage(const vsCmprParamArg& msg) 
{
    vsFrameInfo* frameinfo = NULL;
    vsFieldInfo* in = NULL;
    vsStreamMessage* message = NULL;
    vsStatus sts = msg.getParam(vsCmprParamTypes::StreamMessageParam,
				PARAMCAST message);

    assert(message && (sts == vsErrNone));
    assert(message->getTypeName() == vsCmprQueryTypes::StreamMessage);

    sts = message->getParam(vsCmprParamTypes::FrameInfo,PARAMCAST frameinfo);
    assert(frameinfo);

    sts = frameinfo->getParam(vsFieldTypes::Color,PARAMCAST in);
    assert(in);

    switch(message->getMessageName()) {
        case vsStreamMessage::InitializeStream: {
            PersistantStorage* stor = new PersistantStorage;
            stor->create(in); 
            drawableMap[(int)in->getDrawableID()] = stor;
        } break;
        case vsStreamMessage::ShutdownStream: {
            DrawableMapIter iter = drawableMap.find((int)in->getDrawableID());
            if (iter != drawableMap.end()) {
                iter->second->destroy(in);
                delete iter->second;
                drawableMap.erase(iter);
            }
        } break;	
        case vsStreamMessage::ResizeStream: {
            DrawableMapIter iter = drawableMap.find((int)in->getDrawableID());
            if (iter != drawableMap.end()) {
                iter->second->destroy(in);
                iter->second->create(in);
            }
        } break;
    }
}

size_t
NOPIFCompressor::compress(const vsCmprInputQueryArg& input,
                          vsCmprOutputQueryArg& output)
{
    vsFrameInfo* frameInfo;
    vsFrameData* frameData;
    vsFrameData* outFrameData;
    vsFieldInfo* inFieldInfo;
    vsFieldData* inFieldData;
    vsFieldData* outFieldData;

    vsStatus stsFrInfo = input.getParam(vsCmprQueryTypes::FrameInfo,
                                        PARAMCAST frameInfo);

    vsStatus stsFrData = input.getParam(vsCmprQueryTypes::FrameData,
                                        PARAMCAST frameData);

    vsStatus stsOutFrData = output.getParam(vsCmprQueryTypes::FrameData,
                                        PARAMCAST outFrameData);

    if ((stsFrInfo != vsErrNone) ||
        (stsFrData != vsErrNone) ||
        (stsOutFrData != vsErrNone)) {
        fprintf(stderr, "nopCompressor - couldn't retrieve FrameInfo "
                 "and FrameData parameters.\n");
        return 0;
    }

    vsStatus stsInFInfo = frameInfo->getParam(vsFieldTypes::Color,
                                           PARAMCAST inFieldInfo);
    vsStatus stsInFData = frameData->getParam(vsFieldTypes::Color,
                                           PARAMCAST inFieldData);
    vsStatus stsOutFData = outFrameData->getParam(vsFieldTypes::Color,
                                           PARAMCAST outFieldData);

    if ((stsInFInfo != vsErrNone) ||
        (stsInFData != vsErrNone) ||
        (stsOutFData != vsErrNone)) {
        fprintf(stderr, "nopCompressor - couldn't retrieve FieldInfo "
                        "and FieldData parameters.\n");
        return 0;
    }

    int bpp = vsFieldPixelBytes(inFieldInfo->getFormat(),
                                inFieldInfo->getType());

    int size = inFieldInfo->getWidth() * inFieldInfo->getHeight() * bpp;

    void *in = NULL, *out = NULL;
    inFieldData->getData(in);
    outFieldData->getData(out);

    // get the persistant storage for this drawable
    DrawableMapIter iter = drawableMap.find((int)inFieldInfo->getDrawableID());

    // make sure the drawable exists in our list
    if (iter == drawableMap.end())
        return 0;

    uchar_t *pixMap = iter->second->frameBuffer;

    // create the difference frame to send
    for (int i = 0; i < size; i++) {
        if (pixMap[i] >= ((uchar_t*)in)[i])
            ((uchar_t*)out)[i] = pixMap[i] - ((uchar_t*)in)[i];
        else
            ((uchar_t*)out)[i] = pixMap[i] - ((uchar_t*)in)[i] +256;
    }

    // update the persistant storage with the frame we just sent
    memcpy(pixMap, in, size);

    return size;
}

void
NOPIFCompressor::expand(const vsCmprInputQueryArg& input,
                              vsCmprOutputQueryArg& output)
{
    vsFrameInfo* frameInfo;
    vsFrameData* frameData;
    vsFrameData* outFrameData;
    vsFieldInfo* inFieldInfo;
    vsFieldData* inFieldData;
    vsFieldData* outFieldData;

    vsStatus stsFrInfo = input.getParam(vsCmprQueryTypes::FrameInfo,
                                        PARAMCAST frameInfo);

    vsStatus stsFrData = input.getParam(vsCmprQueryTypes::FrameData,
                                        PARAMCAST frameData);

    vsStatus stsOutFrData = output.getParam(vsCmprQueryTypes::FrameData,
                                        PARAMCAST outFrameData);

    if ((stsFrInfo != vsErrNone) ||
        (stsFrData != vsErrNone) ||
        (stsOutFrData != vsErrNone)) {
        fprintf(stderr, "nopIFCompressor - couldn't retrieve FrameInfo "
                 "and FrameData parameters.\n");
        return;
    }

    vsStatus stsInFInfo = frameInfo->getParam(vsFieldTypes::Color,
                                           PARAMCAST inFieldInfo);
    vsStatus stsInFData = frameData->getParam(vsFieldTypes::Color,
                                           PARAMCAST inFieldData);
    vsStatus stsOutFData = outFrameData->getParam(vsFieldTypes::Color,
                                           PARAMCAST outFieldData);

    if ((stsInFInfo != vsErrNone) ||
        (stsInFData != vsErrNone) ||
        (stsOutFData != vsErrNone)) {
        fprintf(stderr, "nopIFCompressor - couldn't retrieve FieldInfo "
                 "and FieldData parameters.\n");
        return;
    }

    int bpp = vsFieldPixelBytes(inFieldInfo->getFormat(),
                                inFieldInfo->getType());

    int size = inFieldInfo->getWidth() * inFieldInfo->getHeight() * bpp;

    void *in = NULL, *out = NULL;
    inFieldData->getData(in);
    outFieldData->getData(out);

    // get the persistant storage for this drawable
    DrawableMapIter iter = drawableMap.find((int)inFieldInfo->getDrawableID());
    // make sure the drawable exists in our list
    if (iter == drawableMap.end())
        return;

    uchar_t *pixMap = iter->second->frameBuffer;

    // create the image with the difference frame
    for (int i = 0; i < size; i++) {
        if (pixMap[i] >= ((uchar_t*) in)[i])
            ((uchar_t*)out)[i] = pixMap[i] - ((uchar_t*) in)[i];
        else
            ((uchar_t*)out)[i] = pixMap[i] - ((uchar_t*) in)[i] + 256;
    }

    // update the the persistant storage with the "decompressed" image
    memcpy(pixMap,out,size);
}
