/*
 * uconv.h - Unicode Conversions
 *
 * Function:
 *     Declare structures and prototypes for Unicode Conversion functions.
 *
 * Copyright:
 *     Copyright (C) IBM Corp. 1995
 *
 * Notes:
 *     This is for the OS/2 PC prototype, and may not match the
 *     actual system header file.
 *
 */

#ifndef _ULS_UCONV
   #define _ULS_UCONV

   #include <callconv.h>
        #include <unidef.h>
        #include <ulserrno.h>
   #include <stddef.h>  /* size_t define */

   #ifdef __cplusplus
     extern "C" {
   #endif

   #if defined(__IBMCPP__) || defined(__IBMC__)
     #pragma pack(1)
   #else
     #pragma Align_members(1)
   #endif

   #ifndef _ULS_UCONVP
     #define _ULS_UCONVP
     typedef void * UconvObject;          /* uconv Type pointer            */
   #endif

   #ifndef _ULS_UNICHAR_DEFINED
     #define _ULS_UNICHAR_DEFINED
     typedef unsigned  short  UniChar ;
   #endif


   #ifndef ULS_ATTR
   #define ULS_ATTR
   /*
    * Substitution options
    */
   #define UCONV_OPTION_SUBSTITUTE_FROM_UNICODE  1
   #define UCONV_OPTION_SUBSTITUTE_TO_UNICODE    2
   #define UCONV_OPTION_SUBSTITUTE_BOTH          3

   /*
    * Conversion options
    */
   #define CVTTYPE_CDRA     0x00000002   /* Use CDRA control mapping */
   #define CVTTYPE_CTRL7F   0x00000001   /* Tread 0x7F as a control  */

   /*
    * Conversion mask.  This is a bit mask.  If the bit is on, the
    * corresponding character 0x00-0x1f is treated as a control.
    * Otherwise it is treated as a glyph.
    */
   #define DSPMASK_DATA    0xffffffff
   #define DSPMASK_DISPLAY 0x00000000
   #define DSPMASK_TAB     0x00000200
   #define DSPMASK_LF      0x00000400
   #define DSPMASK_CR      0x00002000
   #define DSPMASK_CRLF    0x00002400

   /*
    * Encoding schemes.  This is the primary item used to check if a
    * a codepage is valid for a particular purpose.  A length check
    * may also be made in some cases.
    */
   enum uconv_esid {                     /* Process Display  VIO    GPI   */
       ESID_sbcs_data        = 0x2100,   /*    x      x      x       x    */
       ESID_sbcs_pc          = 0x3100,   /*    x      x      x       x    */
       ESID_sbcs_ebcdic      = 0x1100,   /*           x      x       x    */
       ESID_sbcs_iso         = 0x4100,   /*    x      x      x       x    */
       ESID_sbcs_windows     = 0x4105,   /*    x      x      x       x    */
       ESID_sbcs_alt         = 0xF100,   /*           x      x       x    */
       ESID_dbcs_data        = 0x2200,   /*           x              x    */
       ESID_dbcs_pc          = 0x3200,   /*    x      x      x       x    */
       ESID_dbcs_ebcdic      = 0x1200,   /*                          x    */
       ESID_mbcs_data        = 0x2300,   /*           x      x       x    */
       ESID_mbcs_pc          = 0x3300,   /*           x              x    */
       ESID_mbcs_ebcdic      = 0x1301,   /*                               */
       ESID_ucs_2            = 0x7200,   /*                               */
       ESID_ugl              = 0x72FF,   /*                               */
       ESID_utf_8            = 0x7807,   /*           x      x       x    */
       ESID_upf_8            = 0x78FF    /*    x      x      x       x    */
   };

   #ifndef __conv_endian_t
       #define __conv_endian_t

       #define     ENDIAN_SYSTEM   0x0000
       #define     ENDIAN_BIG      0xfeff
       #define     ENDIAN_LITTLE   0xfffe

       typedef struct _conv_endian_rec {
           size_t  source : 16; /* Used by FromUcs                   */
           size_t  target : 16; /* Used by ToUcs                     */
       } conv_endian_t;
   #endif

   typedef struct _uconv_attribute_t {
       size_t         version : 32;  /* Q/S Version (must be zero)        */
       char           mb_min_len;    /* Q   Minimum char size             */
       char           mb_max_len;    /* Q   Maximum char size             */
       char           usc_min_len;   /* Q   UCS min size                  */
       char           usc_max_len;   /* Q   UCS max size                  */
       size_t         esid : 16;     /* Q   Encoding scheme ID            */
       size_t         options : 16;  /* Q/S Substitution options          */
       conv_endian_t  endian;        /* Q/S Source and target endian      */
       size_t         displaymask:32;/* Q/S Display/data mask             */
       size_t         converttype:32;/* Q/S Conversion type               */
       size_t         subchar_len:16;/* Q/S MBCS sub len      0=table     */
       size_t         subuni_len :16;/* Q/S Unicode sub len   0=table     */
       char           subchar[16];   /* Q/S MBCS sub characters           */
       UniChar        subuni[8];     /* Q/S Unicode sub characters        */
   } uconv_attribute_t;

   /*
    * User defined character range
    */
   typedef struct {             /* User Defined character range      */
       size_t   first : 16;     /* First codepoint                   */
       size_t   last  : 16;     /* Last codepoint                    */
   } udcrange_t;
   #endif  /* ULS_ATTR */
   typedef uls_error_t uconv_error_t;

   /*
    * UniCreateUconvObject: Create Unicode Conversion Object
    *
    * return code:
    *    0             - Conversion Object sucessfully initialized
    *    UCONV_EMFILE  - Maximum file descriptors are opened 
    *    UCONV_ENFILE  - Too many files currently open in the system
    *    UCONV_ENOMEM  - Insuffcient memory
    *    UCONV_EINVAL  - The conversion specified by code_set or the
    *                    modifier supplied is not recognized by the
    *                    implementation.
    *
    */
   uconv_error_t CALLCONV UniCreateUconvObject(
             UniChar     * code_set,   /* I  - Unicode name of uconv table */
             UconvObject * uobj  );    /* O  - Uconv object handle         */

   /*
    * UniQueryUconvObject: Query Unicode Coversion Object
    *
    * return code:
    *    0             - Operation successful
    *    UCONV_EBADF   - The conversion object specified is not
    *                    recognized by the implementation.
    */
   uconv_error_t CALLCONV UniQueryUconvObject(
             UconvObject         uobj, /* I  - Uconv object handle         */
             uconv_attribute_t * attr, /* O  - Uconv attributes            */
             size_t              size, /* I  - Size of attribute structure */
             char          first[256], /* O  - First byte of multibyte     */
             char          other[256], /* O  - Other byte of multibyte     */
             udcrange_t udcrange[32]); /* O  - User defined char range     */

   /*
    * UniSetUconvObject:  Set Unicode Conversion Object
    *
    * return code:
    *    0             - Operation successful
    *    UCONV_EBADF   - The conversion object specified is not
    *                    recognized by the implementation.
    *    UCONV_BADATTR - attribute supplied contains invalid attribute
    *                    for the conversion object
    */
   uconv_error_t CALLCONV UniSetUconvObject(
             UconvObject         uobj, /* I  - Uconv object handle         */
             uconv_attribute_t * attr, /* I  - Uconv attributes            */
             size_t              size);/* I  - Uconv attributes size       */

   /*
    * UniUconvToUcs:  Convert string to unicode
    *
    * return code:
    *    0             - Operation successful
    *    UCONV_EBADF   - The conversion object specified is not
    *                    recognized by the implementation.
    *    UCONV_E2BIG   - Input conversion stopped due to lack of space
    *                    in the output buffer
    *    UCONV_EINVAL  - Input conversion stopped due to incomplete
    *                    character or shift sequence at the end of the
    *                    input buffer.
    *    UCONV_EILSEQ  - Input conversion stopped due to an input byte
    *                    that does not belong to the input code set.
    */
   uconv_error_t CALLCONV UniUconvToUcs(
             UconvObject uobj,         /* I  - Uconv object handle         */
             void    * * inbuf,        /* IO - Input buffer                */
             size_t    * inbytes,      /* IO - Input buffer size (bytes)   */
             UniChar * * outbuf,       /* IO - Output buffer size          */
             size_t    * outchars,     /* IO - Output size (chars)         */
             size_t    * subst  );     /* IO - Substitution count          */

   /*
    * UniUconvFromUcs:  Convert string from unicode
    *    0             - Operation successful
    *    UCONV_EBADF   - The conversion object specified is not
    *                    recognized by the implementation.
    *    UCONV_E2BIG   - Input conversion stopped due to lack of space
    *                    in the output buffer
    *    UCONV_EINVAL  - Input conversion stopped due to incomplete
    *                    character or shift sequence at the end of the
    *                    input buffer.
    *    UCONV_EILSEQ  - Input conversion stopped due to an input byte
    *                    that does not belong to the input code set.
    */
   uconv_error_t CALLCONV UniUconvFromUcs(
             UconvObject uobj,         /* I  - Uconv object handle         */
             UniChar * * inbuf,        /* IO - Input buffer                */
             size_t    * inchars,      /* IO - Input buffer size (bytes)   */
             void    * * outbuf,       /* IO - Output buffer size          */
             size_t    * outbytes,     /* IO - Output size (chars)         */
             size_t    * subst   );    /* IO - Substitution count          */

   /*
    * UniFreeUconvObject: Close Unicode Conversion Object
    *    0             - Operation successful
    *    UCONV_EBADF   - The conversion object specified is not
    *                    recognized by the implementation.
    */
   uconv_error_t CALLCONV UniFreeUconvObject(
             UconvObject   uobj   );   /* I  - Uconv object handle         */

  #ifndef UCSID_DEF
    #define UCSID_DEF
    typedef unsigned int UCSID;
  #endif
  #ifndef CCSID_DEF
    #define CCSID_DEF
    typedef unsigned int CCSID;
  #endif
  #ifndef DOSCP_DEF
    #define DOSCP_DEF
    typedef unsigned int DosCP;
  #endif

  #define UNI_CCSID_TAG        (UniChar *)L"CCSID"     /*ccsid tag type     */
  #define UNI_DOSCP_TAG        (UniChar *)L"DosCP"     /*doscp tag type     */
  #define UNI_CHAR_TAG         (UniChar *)L"CharString"/*char * tag type    */
  #define UNI_UNICHAR_TAG      (UniChar *)L"UCSString" /*unichar * tag type */

  /*
   * UniUcsidToTag:   retieve codepage name or codepage id# from the
   *                  supplied ucsid
   *
   * return
   *     0             - Operation successful
   *     ULS_INVALID   - input parameter not valid
   *     ULS_NOMATCH   - data cannot be found
   */
  uconv_error_t CALLCONV UniUcsidToTag (
                               UCSID     ucsid,      /* input ucsid       */
                               UniChar  *tagType,    /* output tag type   */
                               void    **tagValue ); /* output            */

  /*
   * UniUcsidFromTag: retieve ucsid form codepage name or codepage id#
   *
   * return
   *     0             - Operation successful
   *     ULS_INVALID   - input parameter not valid
   *     ULS_NOMATCH   - data cannot be found
   */
  uconv_error_t CALLCONV UniUcsidFromTag (
                                 UCSID      *ucsid,     /* output ucsid    */
                                 UniChar    *tagType,   /* input tag type  */
                                 void       *tagValue );/* input           */

 /*
  * for compatibility with old return codes for ULS
  */
  #define UCONV_EOTHER           ULS_OTHER
  #define UCONV_EILSEQ           ULS_ILLEGALSEQUENCE
  #define UCONV_EINVAL           ULS_INVALID
  #define UCONV_E2BIG            ULS_BUFFERFULL
  #define UCONV_ENOMEM           ULS_NOMEMORY
  #define UCONV_EBADF            ULS_BADOBJECT
  #define UCONV_BADATTR          ULS_BADATTR
  #define UCONV_EMFILE           ULS_MAXFILESPERPROC
  #define UCONV_ENFILE           ULS_MAXFILES
  #define UCONV_NOTIMPLEMENTED   ULS_NOTIMPLEMENTED

  #define IBM_037         (UniChar *)L"IBM-37"
  #define IBM_273         (UniChar *)L"IBM-273"
  #define IBM_277         (UniChar *)L"IBM-277"
  #define IBM_278         (UniChar *)L"IBM-278"
  #define IBM_280         (UniChar *)L"IBM-280"
  #define IBM_281         (UniChar *)L"IBM-281"
  #define IBM_284         (UniChar *)L"IBM-284"
  #define IBM_285         (UniChar *)L"IBM-285"
  #define IBM_293         (UniChar *)L"IBM-293"
  #define IBM_297         (UniChar *)L"IBM-297"
  #define IBM_420         (UniChar *)L"IBM-420"
  #define IBM_424         (UniChar *)L"IBM-424"
  #define IBM_437         (UniChar *)L"IBM-437"
  #define IBM_500         (UniChar *)L"IBM-500"
  #define IBM_803         (UniChar *)L"IBM-803"
  #define IBM_819         (UniChar *)L"IBM-819"
  #define IBM_833         (UniChar *)L"IBM-833"
  #define IBM_836         (UniChar *)L"IBM-836"
  #define IBM_838         (UniChar *)L"IBM-838"
  #define IBM_839         (UniChar *)L"IBM-839"
  #define IBM_850         (UniChar *)L"IBM-850"
  #define IBM_852         (UniChar *)L"IBM-852"
  #define IBM_855         (UniChar *)L"IBM-855"
  #define IBM_856         (UniChar *)L"IBM-856"
  #define IBM_857         (UniChar *)L"IBM-857"
  #define IBM_860         (UniChar *)L"IBM-860"
  #define IBM_861         (UniChar *)L"IBM-861"
  #define IBM_862         (UniChar *)L"IBM-862"
  #define IBM_863         (UniChar *)L"IBM-863"
  #define IBM_864         (UniChar *)L"IBM-864"
  #define IBM_865         (UniChar *)L"IBM-865"
  #define IBM_869         (UniChar *)L"IBM-869"
  #define IBM_870         (UniChar *)L"IBM-870"
  #define IBM_871         (UniChar *)L"IBM-871"
  #define IBM_875         (UniChar *)L"IBM-875"
  #define IBM_910         (UniChar *)L"IBM-910"
  #define IBM_930         (UniChar *)L"IBM-930"
  #define IBM_932         (UniChar *)L"IBM-932"
  #define IBM_933         (UniChar *)L"IBM-933"
  #define IBM_934         (UniChar *)L"IBM-934"
  #define IBM_935         (UniChar *)L"IBM-935"
  #define IBM_936         (UniChar *)L"IBM-936"
  #define IBM_937         (UniChar *)L"IBM-937"
  #define IBM_938         (UniChar *)L"IBM-938"
  #define IBM_939         (UniChar *)L"IBM-939"
  #define IBM_942         (UniChar *)L"IBM-942"
  #define IBM_944         (UniChar *)L"IBM-944"
  #define IBM_946         (UniChar *)L"IBM-946"
  #define IBM_948         (UniChar *)L"IBM-948"
  #define IBM_949         (UniChar *)L"IBM-949"
  #define IBM_1025        (UniChar *)L"IBM-1025"
  #define IBM_1026        (UniChar *)L"IBM-1026"
  #define IBM_1027        (UniChar *)L"IBM-1027"
  #define IBM_1046        (UniChar *)L"IBM-1046"
  #define IBM_1047        (UniChar *)L"IBM-1047"
  #define UTF_8           (UniChar *)L"UTF-8"
  #define ISO8859_1       (UniChar *)L"IBM-819"
  #define ISO8859_2       (UniChar *)L"IBM-912"
  #define ISO8859_3       (UniChar *)L"IBM-913"
  #define ISO8859_4       (UniChar *)L"IBM-914"
  #define ISO8859_5       (UniChar *)L"IBM-915"
  #define ISO8859_6       (UniChar *)L"IBM-1089"
  #define ISO8859_7       (UniChar *)L"IBM-813"
  #define ISO8859_8       (UniChar *)L"IBM-916"
  #define ISO8859_9       (UniChar *)L"IBM-920"

  #if defined(__IBMCPP__) || defined(__IBMC__)
    #pragma pack()
  #else
    #pragma Align_members()
  #endif

  #ifdef __cplusplus
  }
  #endif

#endif
