/*	Copyright (c) 1990, 1991, 1992, 1993, 1994 Novell, Inc. All Rights Reserved.	*/
/*	Copyright (c) 1984, 1985, 1986, 1987, 1988, 1989, 1990 Novell, Inc. All Rights Reserved.	*/
/*	  All Rights Reserved  	*/

/*	THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF Novell Inc.	*/
/*	The copyright notice above does not evidence any   	*/
/*	actual or intended publication of such source code.	*/

#ifndef _FS_SFS_SFS_FSDIR_H	/* wrapper symbol for kernel use */
#define _FS_SFS_SFS_FSDIR_H	/* subject to change without notice */

#ident	"@(#)kern:fs/sfs/sfs_fsdir.h	1.9"
#ident	"$Header: $"

#if defined(__cplusplus)
extern "C" {
#endif
 
#ifdef _KERNEL_HEADERS

#include <util/types.h> /* REQUIRED */

#elif defined(_KERNEL) || defined(_KMEMUSER)

#include <sys/types.h> /* REQUIRED */

#endif /* _KERNEL_HEADERS */

/*
 * A directory consists of some number of blocks of DIRBLKSIZ
 * bytes, where DIRBLKSIZ is chosen such that it can be transferred
 * to disk in a single atomic operation (e.g. 512 bytes on most machines).
 *
 * Each DIRBLKSIZ byte block contains some number of directory entry
 * structures, which are of variable length.  Each directory entry has
 * a struct direct at the front of it, containing its inode number,
 * the length of the entry, and the length of the name contained in
 * the entry.  These are followed by the name padded to a 4 byte boundary
 * with null bytes.  All names are guaranteed null terminated.
 * The maximum length of a name in a directory is SFS_MAXNAMLEN.
 *
 * The macro SFS_DIRSIZ(dp) gives the amount of space required to represent
 * a directory entry.  Free space in a directory is represented by
 * entries which have dp->d_reclen > the amount of space required to represent
 * the directory.  All DIRBLKSIZ bytes in a directory block are claimed by the
 * directory entries.  This usually results in the last entry in a directory
 * having a large * dp->d_reclen.
 *
 * When entries are deleted from a directory, the space is
 * returned to the previous entry in the same directory
 * block by increasing its dp->d_reclen.  If the first entry of
 * a directory block is free, then its dp->d_ino is set to 0.
 * Entries other than the first in a directory do not normally have
 * dp->d_ino set to 0.
 */
#define DIRBLKSIZ	DEV_BSIZE
#define	SFS_MAXNAMLEN	255

struct	direct {
	u_long	d_ino;			/* inode number of entry */
	u_short	d_reclen;		/* length of this record */
	u_short	d_namlen;		/* length of string in d_name */
	char	d_name[SFS_MAXNAMLEN + 1]; /* name must be no longer than this */
};

/*
 * The DIRSIZ macro gives the minimum record length which will hold
 * the directory entry.  This requires the amount of space in struct direct
 * without the d_name field, plus enough space for the name with a terminating
 * null byte (dp->d_namlen+1), rounded up to a 4 byte boundary.
 */
#undef DIRSIZ
#define DIRSIZ(dp) \
    ((sizeof (struct direct) - (SFS_MAXNAMLEN+1)) + (((dp)->d_namlen+1 + 3) & ~3))

#if defined(_KERNEL) || defined(_KMEMUSER)
/*
 * Template for manipulating directories.
 * Should use struct direct's, but the name field
 * is SFS_MAXNAMLEN - 1, and this just won't do.
 */
struct dirtemplate {
	u_long	dot_ino;
	short	dot_reclen;
	short	dot_namlen;
	char	dot_name[4];		/* must be multiple of 4 */
	u_long	dotdot_ino;
	short	dotdot_reclen;
	short	dotdot_namlen;
	char	dotdot_name[4];		/* ditto */
};


/*
 * int
 * SFS_DIR_REMAINING(int off)
 *	Determine the length remaining in a directory block.
 *
 * Calling/Exit State:
 *	The remaining length is computed from position ``off'' in the
 *	directory block.
 *
 *	Returns the reamining length.
 */
#define SFS_DIR_REMAINING(off)	(DIRBLKSIZ - ((off) & (DIRBLKSIZ - 1)))

/*
 * boolean_t
 * SFS_DIR_MANGLED(int remaining, struct direct *ep)
 *	Determine if a directory entry is mangled.
 *
 * Calling/Exit State:
 *	``remaining'' gives the remaining length from the directory entry
 *	to the end of the directory block.  ``ep'' points to the entry.
 *
 *	Returns B_TRUE if the entry is mangled and B_FALSE otherwise.
 *
 * Remarks:
 *	We do suffient checking to ``bullet proof'' the kernel against
 *	trapping on an unmapped or unaligned address. However, we do not
 *	check for illegal name length or illegal characters in the name.
 *
 *	The checks verify:
 *		1) ep->d_reclen and ep->d_namelen are contained within
 *		   the directory block,
 *		2) ep is appropriately aligned,
 *		3) the record fits within the directory block, and
 *		4) the name fits within the record
 */
#define SFS_DIR_MANGLED(remaining, ep)	\
	(((remaining) < sizeof(struct direct) - (SFS_MAXNAMLEN - 1) || \
	 ((ep)->d_reclen & 0x3) != 0 || (int)(ep)->d_reclen > (remaining) || \
	 (int)(ep)->d_reclen < DIRSIZ(ep)))

#endif /* _KERNEL || _KMEMUSER */

#if defined(__cplusplus)
	}
#endif

#endif /* _FS_SFS_SFS_FSDIR_H */
