/*	Copyright (c) 1990, 1991, 1992, 1993, 1994 Novell, Inc. All Rights Reserved.	*/
/*	Copyright (c) 1984, 1985, 1986, 1987, 1988, 1989, 1990 Novell, Inc. All Rights Reserved.	*/
/*	  All Rights Reserved  	*/

/*	THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF Novell Inc.	*/
/*	The copyright notice above does not evidence any   	*/
/*	actual or intended publication of such source code.	*/

#ident	"@(#)kern-i386at:psm/compaq/pic.c	1.13"
#ident	"$Header: $"

/*
 * i8259 Programmable Interrupt Controllers (PIC)
 */

#include <svc/bootinfo.h>
#include <svc/eisa.h>
#include <svc/pic.h>
#include <svc/systm.h>
#include <util/cmn_err.h>
#include <util/engine.h>
#include <util/inline.h>
#include <util/ipl.h>
#include <util/plocal.h>
#include <util/sysmacros.h>
#include <util/types.h>
#include <util/debug.h>

#include <syspro.h>
#include <xl.h>

extern unsigned long level_intr_mask;

/* defined in conf.c, which is generated by config: */
extern void (*ivect[])();	/* interrupt routines */
extern uchar_t intpri[];	/* priority levels for interrupts */
extern int nintr;		/* number of interrupts */

/* defined in space.c file: */
extern ushort_t cmdport[];	/* command port addrs for pics */
extern ushort_t imrport[];	/* intr mask port addrs for pics */
extern uchar_t masterline[];	/* line this pic connected to */
extern uchar_t *curmaskp[];	/* current masks for pics */
extern uchar_t picbuffered;	/* true if pic buffered */
extern int npic;		/* number of pics configured */
extern struct irqtab irqtab[];/* per-IRQ info */
extern pl_t svcpri[];		/* interrupt service priority levels */
extern k_pl_t ipl;		/* per-processor ipl */
extern pl_t picipl;		/* ipl to which pic is currently programmed */

/*
 * iplmask[] contains the pic masks for each interrupt priority level.
 * It is effectively dimensioned iplmask[PLHI + 1][NPIC], and is initialized
 * from intpri[].
 *
 * Since code always runs at PLHI or below, we only need entries in iplmask
 * for PLBASE through PLHI, thus iplmask is dimensioned up to PLHI only.
 */
extern uchar_t *iplmaskp[];

void intnull(void);		/* null interrupt routine */


/*
 * The deferred interrupt stack and stack index.  The size of the deferred
 * interrupt stack is based on the following considerations:
 *
 *  (1)	At most one interrupt per service level can be deferred.  This
 *	is guaranteed because whenever an interrupt occurs, the system
 *	blocks all interrupts at the interrupt's service level, and the
 *	interrupts remain blocked until the service of the interrupt
 *	is completed.
 *
 *  (2)	Only interrupts whose interrupt priority level is at or below PLHI
 *	will be deferred.  This is guaranteed because the system always
 *	runs at PLHI or below, thus interrupts whose level is above PLHI
 *	will never be deferred.
 *
 *  (3)	The implication of (1) & (2) is that the maximum number of interrupts
 *	which will be deferred at any one time is equal to the number of
 *	priority levels from PL1 to PLHI, inclusive.  Since this implementation
 *	uses ordinal numbers to represent priority levels, we can say that the
 *	maximum number of deferred interrupts is PLHI - PL1 + 1, or just PLHI.
 *
 *  (4)	There is always a dummy value at the bottom of the deferred interrupt
 *	stack; see the initialization of picdeferred below.  Thus, the total
 *	size of picdeferred is PLHI + 1: PLHI for the maximum number of
 *	deferred interrupts, plus 1 for the dummy value.
 */
int picdeferred[MAXNUMCPU][PLHI + 1];	/* deferred interrupt stack */
int picdeferndx[MAXNUMCPU];		/* deferred interrupt stack index */
int *picdeferredp[MAXNUMCPU] = {	/* deferred interrupt stack pointers */
	(int *)&picdeferred[0]
};

volatile uint_t iv_cookie;
static int slaves;

void holdcpus(void *);
extern int picreload(void);

extern uchar_t sp_iomode;


/*
 * void
 * intrtab_init(int engnum)
 *	Initialize interrupt tables.
 *
 * Calling/Exit State:
 *	Must be called after the system is initialized and engines
 *	other than boot engine are brought online.
 */
void
intrtab_init(int engnum)
{
	/*
	 * Initialize picdeferredp for soft-spl.
	 */
	picdeferredp[engnum] = (int *)&picdeferred[engnum];
}


/*
 * void picinit(void)
 *	Initialize the Programmable Interrupt Controller (PIC) chip(s).
 *	The PIC interrupt masks are temporarily programmed to disallow
 *	all interrupts, and the logical level is set to PLBASE.  The
 *	PIC masks will be rewritten with their real values later, in
 *	picstart().
 *
 * Calling/Exit State:
 *	On entry, the IF flag is off, and ipl, picipl, and the PICs
 *	are uninitialized.
 *
 *	On exit, the IF flag is on, ipl and picipl are set to PLBASE,
 *	the PICs have been initialized, the PIC interrupt masks disallow
 *	all interrupts.
 */
void
picinit(void)
{
	int cmd, imr, pic, bit;
	struct irqtab *ip;
	int i;
	int engnum;


	engnum = myengnum;

	/*
	 * Identify lines on master to which slaves are connected.
	 */
	slaves = 0;
	for (pic = 1; pic < npic; pic++)        /* for each slave */
		slaves |= 1 << masterline[pic];

	/*
	 * Initialize the irqtab, which contains per-vector information.
	 */
	ip = irqtab;
	for (pic = 0; pic < npic; pic++) {	/* loop thru PICs */
		for (bit = 1; bit <= 0x80; bit <<= 1, ip++) {
			ip->irq_cmdport = cmdport[pic];
			ip->irq_flags = 0;
			if (pic != 0)
				ip->irq_flags |= IRQ_ONSLAVE;
			if (bit == PIC_IRQSPUR)
				ip->irq_flags |= IRQ_CHKSPUR;
		}
	}

	/*
	 * Initialize the PIC hardware, starting with the master PIC.
	 */

#ifdef AT380
	/*
	 * AT380 support: Set the vector registers to match the 8259
	 */
	outb(VRB0, PIC_VECTBASE);
	outb(VRB1, PIC_VECTBASE + 1);
	outb(VRB3, PIC_VECTBASE + 3);
	outb(VRB4, PIC_VECTBASE + 4);
	outb(VRB5, PIC_VECTBASE + 5);
	outb(VRB6, PIC_VECTBASE + 6);
	outb(VRB7, PIC_VECTBASE + 7);
#endif
	
	/* ICW1: Edge-triggered, Cascaded, need ICW4 */
	outb(cmdport[0], PIC_ICW1BASE|PIC_NEEDICW4);
	
	/* ICW2: start master vectors at PIC_VECTBASE */
	outb(imrport[0], PIC_VECTBASE);
	
	/* ICW3: define which lines are connected to slaves */
	outb(imrport[0], slaves);

#ifdef AT380
	/*
	 * AT380 support: Use auto-EOI mode.
	 */
	outb(imrport[0], PIC_AUTOEOI|PIC_86MODE);
	inb(cmdport[0]);	/* to avoid a bug in the 380 */
	inb(cmdport[0] + 2);	/* read back ICW2 to reset IRQ1.5 */
#else
	/* ICW4: buffered master (?), norm eoi, mcs 86 */
	outb(imrport[0],
	     picbuffered ? PIC_MASTERBUF|PIC_86MODE : PIC_86MODE);
#endif

	/* OCW1: Start the master with all interrupts off */
	outb(imrport[0], curmaskp[engnum][0] = 0xFF);
	
	/* OCW3: set master into "read isr mode" */
	outb(cmdport[0], PIC_READISR);
	
	/*
	 * Initialize slave PICs
	 */
#ifdef AT380
	/*
	 * AT380 support: Set the vector registers for bank C here. We
	 * know that bank C is the only slave.
	 */
	outb(VRC0, PIC_VECTBASE + 8);
	outb(VRC1, PIC_VECTBASE + 9);
	outb(VRC2, PIC_VECTBASE + 10);
	outb(VRC3, PIC_VECTBASE + 11);
	outb(VRC4, PIC_VECTBASE + 12);
	outb(VRC5, PIC_VECTBASE + 13);
	outb(VRC6, PIC_VECTBASE + 14);
	outb(VRC7, PIC_VECTBASE + 15);
#endif

	for (pic = 1; pic < npic; pic++) {
		cmd = cmdport[pic];
		imr = imrport[pic];

		/* ICW1: Edge-triggered, Cascaded, need ICW4 */
		outb(cmd, PIC_ICW1BASE|PIC_NEEDICW4);

		/* ICW2: set base of vectors */
		outb(imr, PIC_VECTBASE + pic * 8);

		/* ICW3: specify ID for this slave */
		outb(imr, masterline[pic]);

#ifdef AT380
		/*
		 * AT380 support: Use auto-EOI mode.
		 */
		outb(imr, PIC_AUTOEOI|PIC_86MODE);
		inb(cmd);	/* to avoid a bug in the 380 */
		inb(cmd + 2);	/* read back ICW2 to reset IRQ1.5 */
#else
		/* ICW4: buffered slave (?), norm eoi, mcs 86 */
		outb(imr,
		     picbuffered ? PIC_SLAVEBUF|PIC_86MODE : PIC_86MODE);
#endif

		/* OCW1: start the slave with all interrupts off */
		outb(imr, curmaskp[engnum][pic] = 0xFF);

		/* OCW3: set pic into "read isr mode" */
		outb(cmd, PIC_READISR);
	}

#ifdef AT380
	/*
	 * AT380 support: Initialize bank A to make it transparent from
	 * here on.  With auto-EOI mode, we never have to give it an EOI
	 * and it's transparent to the rest of the code.
	 */
	outb(ACMD_PORT, 0x1b);	/* ICW1 */
	outb(AIMR_PORT, 0x02);	/* ICW2 */
	outb(AIMR_PORT, 0x02);	/* ICW4 */
	outb(AIMR_PORT, 0xfb);	/* mask */
	outb(VRA0, 0x02);	/* shouldn't ever get these, so */
	outb(VRA1, 0x02);	/* have them come in as NMI's */
	outb(VRA1_5, 0x02);
	outb(VRA3, 0x02);
	outb(VRA4, 0x02);
	outb(VRA7, 0x02);
	inb(ACMD_PORT);		/* to avoid a bug in the 380 */
	inb(ACMD_PORT + 2);	/* read back ICW2 to reset IRQ1.5 */
#endif

	if (bootinfo.machflags & EISA_IO_BUS) {
		/*
		 * Set Level Mode for those IRQ's requiring it.
		 */
		for (i = 0; i < 16; ++i) {
			if (level_intr_mask & (1 << i))
				eisa_set_elt(i, LEVEL_TRIG);
		}
	}

	/*
	 * initialize ipl and picipl
	 */
	ipl = picipl = PLBASE;

	/*
	 * Initialize the deferred interrupt stack.  In order to simplify
	 * the testing of the deferred interrupt stack, the stack always
	 * contains a dummy interrupt number as the bottom-most element
	 * of the stack.  This dummy interrumpt number must be distinct
	 * from any valid interrupt (IRQ) number and also must have the
	 * property that its intpri entry is 0.  The value which is used
	 * is one more than the highest possible IRQ number; this is just
	 * npic * PIC_NIRQ.
	 *
	 * Set picdeferndx to 0 to indicate we're at the bottom of the
	 * stack, and set the bottom-most element to npic * PIC_NIRQ;
	 */
	picdeferndx[engnum] = 0;
	picdeferredp[engnum][picdeferndx[engnum]] = npic * PIC_NIRQ;

	/* Initially set all masks to disable all interrupts */
	for (i = (PLHI + 1) * npic; i-- != 0;)
		iplmaskp[engnum][i] = 0xFF;

	asm("sti");	/* ENABLE */
}


/*
 * void picstart(void)
 *      Enable normal interrupt masks and allow device interrupts.
 *
 * Calling/Exit State:
 *      Called from selfinit() during processor initialization.
 *
 * Remarks:
 *	Now we can enable interrupts.  This requires both an ENABLE
 *	to enable interrupts at the processor and an spl0 to set the
 *	priority level to PLBASE.
 */
void
picstart(void)
{
	int intno, bit, pic, level;


	/*
	 * Initialize PIC-related data structures:
	 *  (1)	Set up iplmask[][] from ivect[] and intpri[].  If an
	 *	interrupt number is configured, set its bit in the masks
	 *	at its priority level and higher.  Otherwise, mask it out
 	 *	for all priority levels, including PLBASE.  (Note that
	 *	interrupts above PLHI are never masked, since the
	 *	system never runs above PLHI, and iplmask only includes
	 *	entries up to and including PLHI, but not beyond it.)
	 *
	 *  (2)	Initialize the svcpri array.  For each hardware interrupt,
	 *	svcpri gives the priority level at which the interrupt service
	 *	routine should run.  The service priority level is the same
	 *	as the interrupt priority level, specified in intpri, unless
	 * 	the interrupt pl is above PLHI; in such cases, the service
	 *	pl is PLHI.  This is part of enforcing the constraint that no
	 *	code runs above PLHI.
	 *
	 * NOTE: master PIC must always be PIC zero.
	 *	 PIC base vector is always PIC_VECTBASE + picno * 8.
	 */

	bzero(iplmaskp[myengnum], (PLHI + 1) * npic);
	for (pic = 0; pic < npic; pic++) {	/* loop thru PICs */
		intno = pic * 8;
		for (bit = 1; bit <= 0x80; bit <<= 1, intno++) {
			if (pic == 0 && (bit & slaves)) {
				if (intno < nintr &&
				    ivect[intno] != intnull) {
					/*
					 *+ There is a conflict between the
					 *+ assignment of IRQ lines to slave
					 *+ PICs and the assignment of IRQ
					 *+ lines to devices.  The assignment
					 *+ of the IRQ lines to slave PICs is
					 *+ specified by the masterline array
					 *+ in the space.c file for the pic
					 *+ module, and the assignment of IRQ
					 *+ lines to devices is specified by
					 *+ the sdevice files.  Corrective
					 *+ action: check those files for
					 *+ conflicts, fix the conflicts,
					 *+ rebuild, and reboot.
					 */
					cmn_err(CE_WARN,
						"reserved interrupt vector"
						 " specified; ignored");
				}
			} else {
				if (intno >= nintr ||
				    ivect[intno] == intnull) {
					level = 0;
				} else if ((level = intpri[intno]) == 0 ||
						    level > PLMAX) {
					/*
					 *+ The intpri array contains an
					 *+ invalid pl value.  Corrective
					 *+ action: ensure that IPLs assigned
					 *+ in sdevice files are less than or
					 *+ equal to PLMAX, then rebuild the
					 *+ kernel and reboot.
					 */
					cmn_err(CE_WARN,
						"bad interrupt priority in "
						 " intpri[]: %d", level);
					level = 0;
				}

				/*
				 * If its a special interrupt (clock or inter-
				 * processor interrupt), then do not mask the
				 * interrupt on my engine. Similarly if we
				 * are on boot engine and the intr is already
				 * attached, then do not reset the level.
				 *
				 * On boot engine the inter-processor interrupt
				 * (IPI) will be initialized thru psm_intron, 
				 * but non-boot engines IPI is initialized
				 * here, becuase the intpri table would be
				 * already initialized.
				 *
				 * Note: Since the static drivers will call
				 * psm_intron to enable an interrupt, it is 
				 * no more necessary to check if an interrupt
				 * is bound to myengnum or if it is not bound
				 * to any engine and myengnum is equal to 
				 * the boot engine to explicitly enable those
				 * interrupts.
				 */
				if (!((intno == CLOCKINTR && sp_iomode & SYMINTR) ||
				      (myengnum == BOOTENG) ||
				      (intno == XINTR)))
					level = 0;

				for ( ; level <= PLHI; level++)
					iplmaskp[myengnum][level * npic + pic] |= bit;
			}

			svcpri[intno] = MIN(intpri[intno], PLHI);
		}
	}

#ifdef _MPSTATS
	/*
	 * Unblock/allow clock interrupts at PLHI to have a free-flowing
	 * ulbolt counter.
	 */
	if (myengnum == BOOTENG && sp_iomode & ASYMINTR)
		iplmaskp[myengnum][PLHI * npic] &= ~0x01;
#endif /* _MPSTATS */

	ipl = picipl = PLHI;	/* Force the spl0 to change PIC masks */

	spl0();
}


/*
 * void
 * nenableint(int iv, pl_t level, int engnum, int itype)
 *	Unmask the interrupt vector from the iplmask.
 *
 * Calling/Exit State:
 *	iv is the interrupt request no. that needs to be enabled.
 *	engnum is the engine on which the interrupt must be unmasked.
 *	level is the interrupt priority level of the iv.
 *	itype is the interrupt type.
 *
 *	mod_iv_lock is held on entry/exit.
 */
void
nenableint(int iv, pl_t level, int engnum, int itype)
{
	uchar_t	lv, mask;
	int	bit, pic;
	int	i;		/* interrupt priority level */
	struct emask iv_emask;
	extern void spxl_eisa_set_elt(int, int, int);


        if (nonline > 1)
                xcall_all(&iv_emask, B_FALSE, holdcpus,
                          (void *)iv_cookie);

	if (bootinfo.machflags & EISA_IO_BUS) {
		if (itype == 4) {
			if (myengnum == engnum || sp_iomode & ASYMINTR) {
				eisa_set_elt(iv, LEVEL_TRIG);
			} else if (sp_iomode & SYMINTR) {
				spxl_eisa_set_elt(engnum, iv, LEVEL_TRIG);
			}
		}
	}

	/* change PIC mask */

	/*
	 * Find the bit no corresponding to iv (irqno) that needs to
	 * be unmasked.
	 */
	pic = iv / PIC_NIRQ;
	bit = iv % PIC_NIRQ;

	mask = ~(1 << bit);

	i = PL0 * npic + pic;

	/*
	 * Unmask/Clear the interrupt's bit in iplmask[]
	 * for priorities lower than its own.
	 */
	for (lv = PLBASE; lv < (uchar_t)level; lv++) {
		iplmaskp[engnum][i] &= mask;
		i += npic;
	}

	mask = ~mask;

	/*
	 * Mask/Set the interrupt's bit in iplmask[]
	 * for priorities greater than, or equal to its own.
	 */
	for (; lv <= PLHI; lv++) {
		iplmaskp[engnum][i] |= mask;
		i += npic;
	}

	/*
	 * Reload pic masks for the local so that mask modifications 
	 * take effect immediately. For the non-local engines the
	 * return from xcall will reload the new picmask.
	 */
	if (myengnum == engnum)
		picreload();

	/* Done with pic mask change, release all the cpus */

	++iv_cookie;
}


/*
 * void ndisableint(int iv, pl_t level, int engnum, int itype)
 *	Mask the interrupt vector in the iplmask of the engine currently
 *	running on.
 *
 * Calling/Exit State:
 *	iv is the interrupt that needs to be disabled.
 *	engnum is the engine on which the interrupt must be masked.
 *	level is the interrupt priority level of the iv.
 *	itype is the interrupt type.
 *
 *	mod_iv_lock is held on entry/exit.
 */
/* ARGSUSED */
void ndisableint(int iv, pl_t level, int engnum, int itype)
{
	uchar_t	lv, mask;
	int	pic, bit;
	int	i;			/* interrupt priority level */
	struct emask iv_emask;


        if (nonline > 1)
                xcall_all(&iv_emask, B_FALSE, holdcpus,
                          (void *)iv_cookie);

	/* change PIC mask */

	/*
	 * find the bit no corressponding to irqno that needs to
	 * be unmasked.
	 */
	pic = iv / PIC_NIRQ;
	bit = iv % PIC_NIRQ;

        mask = 1 << bit;

        i = PL0 * npic + pic;

        /*
         * Mask/Set the interrupt's bit in iplmask[]
         * for priorities lower than its own.
         */
        for (lv = PLBASE; lv < (uchar_t)level; lv++) {
                iplmaskp[engnum][i] |= mask;
                i += npic;
        }

        /*
	 * Reload pic masks for the local so that mask modifications 
	 * take effect immediately. For the non-local engines the
	 * return from xcall will reload the new picmask.
         */
	if (myengnum == engnum)
		picreload();

        ++iv_cookie;
}


/*
 * void holdcpus(void *arg)
 *	Hold the cpu executing this routine until the value
 *	of the iv_cookie changed.
 *
 * Calling/Exit State:
 *	Called through xcall, the cpu is already in PLHI.
 */
void
holdcpus(void *arg)
{
	while (iv_cookie == (uint_t)arg)
		;
	picreload();
}


/*
 * void
 * intnull(void)
 *	Null interrupt routine.  Used to fill in empty slots in the
 *	interrupt handler table.
 *
 * Calling/Exit State:
 *	None.
 */
void
intnull(void)
{
}
