/*	Copyright (c) 1990, 1991, 1992, 1993, 1994 Novell, Inc. All Rights Reserved.	*/
/*	Copyright (c) 1984, 1985, 1986, 1987, 1988, 1989, 1990 Novell, Inc. All Rights Reserved.	*/
/*	  All Rights Reserved  	*/

/*	THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF Novell Inc.	*/
/*	The copyright notice above does not evidence any   	*/
/*	actual or intended publication of such source code.	*/

#ident	"@(#)kern-i386at:psm/pcmp/apic.c	1.20"
#ident	"$Header: $"

/* Support for Intel Advanced Programmable Interrupt Controller */

#include <svc/bootinfo.h>
#include <svc/eisa.h>
#include <svc/pic.h>
#include <svc/systm.h>
#include <util/cmn_err.h>
#include <util/inline.h>
#include <util/param.h>
#include <util/ipl.h>
#include <util/plocal.h>
#include <util/sysmacros.h>
#include <util/types.h>

#include <apic.h>
#include <pcmp.h>

#define PIC_LTIM        0x08

extern unsigned long level_intr_mask;

/* Programmable Interrupt Controllers */

/* defined in conf.c, which is generated by config: */
extern void (*ivect[])();	/* interrupt routines */
extern uchar_t intpri[];	/* priority levels for interrupts */
extern int nintr;		/* number of interrupts */

/* defined in space.c file: */
extern ushort_t cmdport[];	/* command port addrs for pics */
extern ushort_t imrport[];	/* intr mask port addrs for pics */
extern uchar_t masterpic[];	/* index of this pic's master */
extern uchar_t masterline[];	/* line this pic connected to */
extern uchar_t curmask[];	/* current masks for pics */
extern uchar_t picbuffered;	/* true if pic buffered */
extern int npic;		/* number of pics configured */
extern struct irqtab irqtab[];	/* per-IRQ info */
extern pl_t svcpri[];		/* interrupt service priority levels */
extern void clock();
extern void psm_aintr();
extern void psm_intr();
extern int intcpu[];
extern int intmp[];
extern k_pl_t ipl;		/* per-processor ipl */

/*
 * iplmask[] contains the pic masks for each interrupt priority level.
 * It is effectively dimensioned iplmask[PLHI + 1][NPIC], and is initialized
 * from intpri[].
 *
 * Since code always runs at PLHI or below, we only need entries in iplmask
 * for PLBASE through PLHI, thus iplmask is dimensioned up to PLHI only.
 */
extern uchar_t iplmask[];

void intnull(void);		/* null interrupt routine */

static int slaves;		/* bitmask of slave lines into master PIC */

volatile uint_t iv_cookie;
STATIC void holdcpus(void *);

/*
 * void picinit(void)
 *	Initialize the Programmable Interrupt Controller (PIC) chip(s).
 *	The PIC interrupt masks are temporarily programmed to disallow
 *	all interrupts, and the logical level is set to PLBASE.  The
 *	PIC masks will be rewritten with their real values later, in
 *	picstart().
 *
 * Calling/Exit State:
 *	On entry, the IF flag is off, and ipl, and the PICs
 *	are uninitialized.
 *
 *	On exit, the IF flag is on, ipl are set to PLBASE,
 *	the PICs have been initialized, the PIC interrupt masks disallow
 *	all interrupts.
 */

void
oldpicinit(void)
{
	int cmd, imr, pic, bit;
	struct irqtab *ip;
	int i;
	unsigned char   icw1_cmd;

	/*
	 * Identify lines on master to which slaves are connected.
	 */
	slaves = 0;
	for (pic = 1; pic < npic; pic++)        /* for each slave */
		slaves |= 1 << masterline[pic];

	/*
	 * Initialize the irqtab, which contains per-vector information.
	 */
	ip = irqtab;
	for (pic = 0; pic < npic; pic++) {	/* loop thru PICs */
		for (bit = 1; bit <= 0x80; bit <<= 1, ip++) {
			ip->irq_cmdport = cmdport[pic];
			ip->irq_flags = 0;
			if (pic != 0)
				ip->irq_flags |= IRQ_ONSLAVE;
			if (bit == PIC_IRQSPUR)
				ip->irq_flags |= IRQ_CHKSPUR;
		}
	}

	/*
	 * Initialize the PIC hardware, starting with the master PIC.
	 */

#ifdef AT380
	/*
	 * AT380 support: Set the vector registers to match the 8259
	 */
	outb(VRB0, PIC_VECTBASE);
	outb(VRB1, PIC_VECTBASE + 1);
	outb(VRB3, PIC_VECTBASE + 3);
	outb(VRB4, PIC_VECTBASE + 4);
	outb(VRB5, PIC_VECTBASE + 5);
	outb(VRB6, PIC_VECTBASE + 6);
	outb(VRB7, PIC_VECTBASE + 7);
#endif
	
        icw1_cmd = PIC_ICW1BASE|PIC_NEEDICW4;

        if (bootinfo.machflags & MC_BUS)
                icw1_cmd |= PIC_LTIM;

	/* ICW1: Edge-triggered, Cascaded, need ICW4 */
	outb(cmdport[0], icw1_cmd);
	
	/* ICW2: start master vectors at PIC_VECTBASE */
	outb(imrport[0], PIC_VECTBASE);
	
	/* ICW3: define which lines are connected to slaves */
	outb(imrport[0], slaves);

#ifdef AT380
	/*
	 * AT380 support: Use auto-EOI mode.
	 */
	outb(imrport[0], PIC_AUTOEOI|PIC_86MODE);
	inb(cmdport[0]);	/* to avoid a bug in the 380 */
	inb(cmdport[0] + 2);	/* read back ICW2 to reset IRQ1.5 */
#else
	/* ICW4: buffered master (?), norm eoi, mcs 86 */
	outb(imrport[0],
	     picbuffered ? PIC_MASTERBUF|PIC_86MODE : PIC_86MODE);
#endif

	/* OCW1: Start the master with all interrupts off */
	outb(imrport[0], curmask[0] = 0xFF);
	
	/* OCW3: set master into "read isr mode" */
	outb(cmdport[0], PIC_READISR);
	
	/*
	 * Initialize slave PICs
	 */
#ifdef AT380
	/*
	 * AT380 support: Set the vector registers for bank C here. We
	 * know that bank C is the only slave.
	 */
	outb(VRC0, PIC_VECTBASE + 8);
	outb(VRC1, PIC_VECTBASE + 9);
	outb(VRC2, PIC_VECTBASE + 10);
	outb(VRC3, PIC_VECTBASE + 11);
	outb(VRC4, PIC_VECTBASE + 12);
	outb(VRC5, PIC_VECTBASE + 13);
	outb(VRC6, PIC_VECTBASE + 14);
	outb(VRC7, PIC_VECTBASE + 15);
#endif

	for (pic = 1; pic < npic; pic++) {
		cmd = cmdport[pic];
		imr = imrport[pic];

		/* ICW1: Edge-triggered, Cascaded, need ICW4 */
		outb(cmd, icw1_cmd);

		/* ICW2: set base of vectors */
		outb(imr, PIC_VECTBASE + pic * 8);

		/* ICW3: specify ID for this slave */
		outb(imr, masterline[pic]);

#ifdef AT380
		/*
		 * AT380 support: Use auto-EOI mode.
		 */
		outb(imr, PIC_AUTOEOI|PIC_86MODE);
		inb(cmd);	/* to avoid a bug in the 380 */
		inb(cmd + 2);	/* read back ICW2 to reset IRQ1.5 */
#else
		/* ICW4: buffered slave (?), norm eoi, mcs 86 */
		outb(imr,
		     picbuffered ? PIC_SLAVEBUF|PIC_86MODE : PIC_86MODE);
#endif

		/* OCW1: start the slave with all interrupts off */
		outb(imr, curmask[pic] = 0xFF);

		/* OCW3: set pic into "read isr mode" */
		outb(cmd, PIC_READISR);
	}

#ifdef AT380
	/*
	 * AT380 support: Initialize bank A to make it transparent from
	 * here on.  With auto-EOI mode, we never have to give it an EOI
	 * and it's transparent to the rest of the code.
	 */
	outb(ACMD_PORT, 0x1b);	/* ICW1 */
	outb(AIMR_PORT, 0x02);	/* ICW2 */
	outb(AIMR_PORT, 0x02);	/* ICW4 */
	outb(AIMR_PORT, 0xfb);	/* mask */
	outb(VRA0, 0x02);	/* shouldn't ever get these, so */
	outb(VRA1, 0x02);	/* have them come in as NMI's */
	outb(VRA1_5, 0x02);
	outb(VRA3, 0x02);
	outb(VRA4, 0x02);
	outb(VRA7, 0x02);
	inb(ACMD_PORT);		/* to avoid a bug in the 380 */
	inb(ACMD_PORT + 2);	/* read back ICW2 to reset IRQ1.5 */
#endif

	if (bootinfo.machflags & EISA_IO_BUS) {
		/*
		 * Set Level Mode for those IRQ's requiring it.
		 */
		for (i = 0; i < 16; ++i) {
			if (level_intr_mask & (1 << i))
				eisa_set_elt(i, LEVEL_TRIG);
		}
	}

	/*
	 * initialize ipl
	 */
	ipl =  PLBASE;

	/* Initially set all masks to disable all interrupts */
	for (i = (PLHI + 1) * npic; i-- != 0;)
		iplmask[i] = 0xFF;

	asm("sti");	/* ENABLE */
}

int apic_mode = 0;                      /* PC+MP compliant if nonzero */
char pcmp_fib1 = 0, pcmp_fib2 = 0;      /* default system type, IMCR present */
int pcmp_paddr = 0;                     /* MP config table pointer      */
vaddr_t apic_table = 0;         /* virtual address of PC+MP config table
                                   OR internal default table */
extern char *apic_defaults[];
int clock_intr_vector;
static int internal_dma_chain = 0;  /* take dmachain intr from PIC as Extint */

struct _io_chg {
	int local_or_io;	/* 1 if io apic, 0 if local */
	int io_id;		/* id of i/o apic (if i/o apic) */
	int line;		/* line of apic */
	int val1;		/* first work of redirection */
	int val2;		/* second word of redirection */
} apic_io_chg;
int pic_to_apic[16] =  {0};

volatile vaddr_t	local_apic_addr;	/* virtual address for mapping
						   the local APIC */
volatile vaddr_t	io_apic_addr[MAXIOAPIC];/* virtual address for mapping
						   the IO APIC. */
volatile vaddr_t	taskpri_reg_addr;	/* virtual address for mapping
						   the task priority reg. */
volatile vaddr_t	eoi_reg_addr;		/* virtual address for mapping
						   the end of interrupt reg. */

int mpcentry_sizes[] = {
	sizeof(struct mpe_proc),
	sizeof(struct mpe_bus),
	sizeof(struct mpe_ioapic),
	sizeof(struct mpe_intr),
	sizeof(struct mpe_intr),
};

int apic_eng_id[MAXNUMCPU];
#define APIC_ID(eng)	apic_eng_id[eng]

/*
 *   --- apic_intpri is used by the common apic interrupt
 *       code to map a vector number/16 to a UNIX priority.  The apic_primask
 *       array is then used to appropriately set the task priority
 *       register of the local APIC.
 */
int apic_intpri[16] = {0,1,1,1, 1,1,1,2, 3,4,5,6, 8,8,8,8};

int apic_primask[] = {
	TPRI0, TPRI1, TPRI2, TPRI3,
	TPRI4, TPRI5, TPRI6, TPRI7,
	TPRIHI
};

/*
 *   --- apic_vecta stores the actual device driver handler address for
 *       a given interrupt.  The table is indexed by vector number, i.e.,
 *       0-255.  The table is used by the common apic interrupt code
 *       to call the driver interrupt routines...
 */
void (*apic_vecta[256])();

/*
 *   --- apic_vect_xln is used to store the 8259 related IRQ number which
 *       is passed to the driver interrupt handler as a parameter.  The
 *       actual INTEL vector number is used to index the table, i.e., 0-255.
 *       The table is used by the common apic interrupt code to obtain
 *       the IRQ number prior to calling the device driver handler.
 */
int apic_vect_xln[256];

/*
 *   --- This doubly dimensioned array (apic_frees) is used to assign a vector
 *       to a given IRQ number and priority.  A zero entry within
 *       each priority group, i.e., 16 available entries/group, indicates
 *       that the vector is available for usage.  When a vector
 *       is assigned the vector number is placed in the corresponding location
 *       of this array.  Some vectors may be pre-assigned for use as error
 *       vectors or reserved if required.
 */
#define NUMBER_OF_PRI 9 /* 0-8 */
#define VECTORS_PER_APIC_PRI 16

int apic_frees[NUMBER_OF_PRI][VECTORS_PER_APIC_PRI];

#ifndef CPUANY
#define CPUANY	(-1)
#endif

void
apic_stray(vec)
{
	cmn_err(CE_WARN, "weird interrupt at %x\n", vec);
}

void
apic_spur(vec)
{
	cmn_err(CE_WARN, "spurious interrupt at %x\n", vec);
}

apic_set_io_ids(ids,n)
int *ids;
{
	volatile long *ip;
	int i, c, id;

	for (i=0; i<n; i++) {
		id = ids[i] << 24;
		ip = (volatile long *)io_apic_addr[i];
		ip[AP_IO_REG] = AIR_ID;
		if (ip[AP_IO_DATA] != id)
			ip[AP_IO_DATA] = id;
	}
}

/*
 * void picstart(void)
 *	Enable normal interrupt masks and allow device interrupts.
 *
 * Calling/Exit State:
 *	Called from selfinit() during processor initialization.
 *
 * Remarks:
 *	Now we can enable interrupts.  This requires both an ENABLE
 *	to enable interrupts at the processor and an spl0 to set the
 *	priority level to PLBASE.
 */
void
apicstart(void)
{
	ipl =  PLHI;	/* Force the spl0 to change PIC masks */

	spl0();
}


#define CONN_IRQ	0
#define CONN_NMI	1
#define CONN_SMI	2
#define CONN_PIC	3

#define CONN_RET_LOC	0
#define CONN_RET_IO	1
#define CONN_RET_FAIL	-1

/*
 * find a connection for interrupt/nmi/pic
 * where (n-1)th byte of entry is equal to entval
 * return destination type (local apic OR i/o apic OR failure)
 * and pass back apic_id:apicline
 *  and optionally flags
 */
static
ap_find_connection(type,n,entval,destid_p,line_p,flags_p)
int *destid_p, *line_p, *flags_p;
{
	union mpcentry *ep, *ap_find_xentry();

	/* try local connection first */
	ep = ap_find_xentry(MP_ET_L_INTR,type,n,entval);
	if (ep) {
		*line_p = ep->i.dest_line;
		return CONN_RET_LOC;
	}

	/* try i/o apic connection */
	ep = ap_find_xentry(MP_ET_I_INTR,type,n,entval);
	if (ep) {
		*line_p = ep->i.dest_line;
		*destid_p = ep->i.dest_apicid;
		if (flags_p)
			*flags_p = ep->i.intr_flags;
		return CONN_RET_IO;
	}

	return CONN_RET_FAIL;
}

/*
 * void ndisableint(int iv, pl_t level, int engnum, int itype)
 *	Disable interrupts of specified interrupt number.
 *
 * Calling/Exit State:
 *	Called and exit with mod_iv_lock held at PLHI.
 */
void
ndisableint(int iv, pl_t level, int cpu, int itype)
{
        struct emask iv_emask;
        int     vector_number, group;
	int rv, io_id, line, flags;

        if (nonline > 1)
                xcall_all(&iv_emask, B_FALSE, holdcpus,
                          (void *)iv_cookie);

        for(vector_number=0; vector_number<256; vector_number++)
		if (apic_vect_xln[vector_number] == iv)
			break;
	if (vector_number >= 256)
        	cmn_err(CE_PANIC, "cannot find active apic slot for %d\n", iv);
	
	if (level >= 7)
		level = 8;
        for(group=0; group<16; group++)
		if (apic_frees[level][group] == vector_number)
			break;
	if (group >= VECTORS_PER_APIC_PRI)
		cmn_err(CE_PANIC, "cannot find active apic slot for %d\n", iv);

        apic_frees[level][group] = 0;
        apic_vecta[vector_number] = apic_stray;
        apic_vect_xln[vector_number] = 0;

	rv = ap_find_connection(CONN_IRQ,5,iv,&io_id,&line,&flags);

	apic_io_chg.line = line;
	apic_io_chg.local_or_io = rv;
	apic_io_chg.io_id = io_id;
	apic_io_chg.val1 = AV_MASK;
	apic_io_chg.val2 = 0;

#ifndef ALL_SEE_IO_APIC
	if (myengnum == 0)
#endif
	apic_io_change();
	++iv_cookie;
}

/*
 * void nenableint(int iv, pl_t level, int cpu, int intmp, int itype)
 *	Enable interrupts of specified interrupt number.
 *
 * Calling/Exit State:
 *	Called and exit with mod_iv_lock held at PLHI.
 */
void
nenableint(int iv, pl_t level, int cpu, int intmp, int itype)
{
        struct emask iv_emask;
        int group, vector_number, lvl, polarity;
	int rv,io_id,line,flags;

        if (nonline > 1)
                xcall_all(&iv_emask, B_FALSE, holdcpus,
                          (void *)iv_cookie);

	alloc_apic_vec(level, &vector_number);
	apic_vect_xln[vector_number] = iv;
	apic_vecta[vector_number] = ivect[iv];

	rv = ap_find_connection(CONN_IRQ,5,iv,&io_id,&line,&flags);

	polarity = 0;
        if (bootinfo.machflags & EISA_IO_BUS) {
                if (itype == 4) {
                        eisa_set_elt(iv, LEVEL_TRIG);
			lvl = AV_LEVEL;
		} else {
			lvl = AV_EDGE;
		}
        } else
		lvl = AV_EDGE;
	if (flags & MP_INTR_POVAL) {	/* table gives polarity */
		polarity = (flags & MP_INTR_POLOW) ?
			AV_POLOW : AV_POHIGH;
	}
	if (flags & MP_INTR_ELVAL) {	/* table gives edge/level */
		lvl = (flags & MP_INTR_ELLEVEL) ?
			AV_LEVEL : AV_EDGE;
	}

	apic_io_chg.line = line;
	apic_io_chg.local_or_io = rv;
	apic_io_chg.io_id = io_id;
        if (cpu == -1) {
		if (intmp == 1) {
                	apic_io_chg.val1 = vector_number | polarity |
					lvl | AV_LDEST | AV_LOPRI;
                	apic_io_chg.val2 = AV_TOALL;
		} else {
                	apic_io_chg.val1 = vector_number | polarity |
					lvl | AV_PDEST | AV_FIXED;
                	apic_io_chg.val2 = APIC_ID(0)<<24;
		}
        } else {
                apic_io_chg.val1 = vector_number | polarity |
					lvl | AV_PDEST | AV_FIXED;
                apic_io_chg.val2 = APIC_ID(cpu)<<24;
        }

#ifndef ALL_SEE_IO_APIC
	if (myengnum == 0)
#endif
	apic_io_change();
	++iv_cookie;
}

/*
 * Load change into I/O apic
 * Only cpu 0 can do this on some early designs
 */
apic_io_change()
{
	ap_set_redir(apic_io_chg.local_or_io,
		     apic_io_chg.io_id,
		     apic_io_chg.line,
		     apic_io_chg.val1,
		     apic_io_chg.val2);
#ifndef ALL_SEE_IO_APIC
	++iv_cookie;
#endif
}

/*
 * STATIC void holdcpus(void *arg)
 *	Hold the cpu executing this routine until the value
 *	of the iv_cookie changed.
 *
 * Calling/Exit State:
 *	Called through xcall, the cpu is already in PLHI.
 */
STATIC void
holdcpus(void *arg)
{
	while (iv_cookie == (uint_t)arg)
		;
#ifndef ALL_SEE_IO_APIC
	if (myengnum == 0) {
		apic_io_change();
	} else while (iv_cookie <= (uint_t)arg + 1)
		;
#endif
}


/*
 * void
 * intnull(void)
 *	Null interrupt routine.  Used to fill in empty slots in the
 *	interrupt handler table.
 *
 * Calling/Exit State:
 *	None.
 */
void
intnull(void)
{
}

/*
 * Assumptions for this version of picinit (for APIC):
 *
 * 1) There is only one i/o APIC
 * 2) The i/o APIC will be initialized by cpu 0 for everybody
 * 3) Interrupts go to a single processor or any processor
 * 4) priority is the same for all processors
 */

apic_susp()
{
	volatile long *lp;

	lp = (volatile long *)local_apic_addr;
	lp[AP_LDEST] = AV_IM_OFF;
}

apic_resu()
{
	volatile long *lp;

	lp = (volatile long *)local_apic_addr;
	lp[AP_LDEST] = APIC_LOGDEST(myengnum);
}

void
apic_table_look()
{
	vaddr_t ebda_addr, bmem_addr;
	int ebda_base, bmem_base;

	/* Search for PC+MP floating pointer 	*/

       /* PC+MP floating pointer signature "_MP_" search order:
	*	1. First 1k bytes of EBDA(Extended BIOS data Area)	
	*	   EBDA pointer is defined at 2-byte location (40:0Eh)
	*	   Standard EBDA segment located at 639K	
	*	   i.e., (40:0Eh) = C0h; (40:0Fh) = 9F;	
	*	2. Last 1k bytes of the base memory if EBDA undefined	
	*	   i.e., 639k-640k for system with 640k base memory
	*	3. ROM space 0F0000h-0FFFFFFh if nothing found in RAM
	*/

	ebda_addr = physmap(EBDA_PTR,2, KM_NOSLEEP);
	ebda_base = (*(unsigned short *)ebda_addr) << 4;
	if (ebda_base > 640*1024 || ebda_base < 510*1024)
		ebda_base = 0;		/* EBDA undefined */

	bmem_addr = physmap(BMEM_PTR,2, KM_NOSLEEP);
	bmem_base = (*(unsigned short *)bmem_addr) * 1024;
	if (bmem_base > 640*1024 || bmem_base < 510*1024)
		bmem_base = 0;		/* Base memory undefined */

	if (ebda_base) {
		if(!find_mp_fptr(ebda_base, ebda_base+1023))
			if(!find_mp_fptr(0xf0000, 0xfffff))
				cmn_err(CE_WARN, "MP floating pointer missing.\n");
	} else {
		if(!bmem_base || !find_mp_fptr(bmem_base, bmem_base+1023))
			if(!find_mp_fptr(0xf0000, 0xfffff))
				cmn_err(CE_WARN, "MP floating pointer missing!\n");
	}

	/* use a default table */
	if (apic_mode && pcmp_fib1)
		apic_table = (vaddr_t)apic_defaults[pcmp_fib1];

	if (apic_mode == 0) { /* no entry - use default */
		apic_mode = 1;
		pcmp_fib1 = MP_DEF_TYPE;
		pcmp_fib2 = MP_DEF_IMCR;
		apic_table = (vaddr_t)(apic_defaults[MP_DEF_TYPE]);
		cmn_err(CE_WARN, "Use default type %d.\n", pcmp_fib1);
	}
}

/*
 * This routine search PCMP floating pointer structure
 * from physical address b1 to b2
 */

find_mp_fptr(begin, end)
unsigned int	begin, end;
{
	vaddr_t v_begin;
	struct pcmp_fptr *fp, *endp;
		
	v_begin = physmap(begin, end-begin+1, KM_NOSLEEP);
	fp = (struct pcmp_fptr *)v_begin;
	endp = (struct pcmp_fptr *)(v_begin + (end-begin+1) - sizeof(*fp));

	for (fp; fp<=(struct pcmp_fptr *)endp; fp++) {
		if (fp->sig[0] == '_' && fp->sig[1] == 'M' &&
		    fp->sig[2] == 'P' && fp->sig[3] == '_') {
			char *cp, sum;
			int i;

			sum = 0;
			cp = (char *)fp;
			for (i=0; i<sizeof(*fp); i++)
				sum += *cp++;
			if (sum == 0) {
				apic_mode = 1;		/* PC+MP compliant */
				apic_table = physmap(fp->paddr, MMU_PAGESIZE, KM_NOSLEEP);
				pcmp_fib1 = fp->mp_feature_byte[0];
				pcmp_fib2 = fp->mp_feature_byte[1];
				return 1;
			}
		}
	}
	return 0;
}


void
apic_table_chk()
{
	int i,n;
	struct mpconfig *mp;
	char *cp, sum;

	if (apic_mode && pcmp_fib1)	/* using a default - assume its okay */
		return;

	/* get virtual address for it */
	mp =  (struct mpconfig *)apic_table;
	if (mp->hdr.sign[0] != 'P' || mp->hdr.sign[1] != 'C' ||
	    mp->hdr.sign[2] != 'M' || mp->hdr.sign[3] != 'P') {
		cmn_err(CE_WARN, "Bad MP config table signature\n");
		goto use_defaults;
	}

	/* check table length */
	n = mp->hdr.tbl_len;
	if (n < sizeof(struct mpchdr) || n > 0x1000) {
		cmn_err(CE_WARN, "MP config table length error\n");
		goto use_defaults;
	}

	/* calculate checksum */
	sum = 0;
	cp = (char *)mp;
	for (i=0; i<n; i++)
		sum += *cp++;
	if (sum != 0) {
		cmn_err(CE_WARN, "MP config table checksum %x error\n", sum);
		goto use_defaults;
	}
	return;

use_defaults:
	pcmp_fib1 = MP_DEF_TYPE;
	pcmp_fib2 = MP_DEF_IMCR;
	apic_table = (vaddr_t)(apic_defaults[MP_DEF_TYPE]);
	cmn_err(CE_WARN, "Bad MP config table! Use default type %d.\n", pcmp_fib1);
}


apic_n_io()
{
	struct mpconfig *mp;
	union mpcentry *ep;
	int ne, n, i, ty;
	int ids[MAXIOAPIC];

	mp = (struct mpconfig *)apic_table;
	ep = mp->entry;

	ne = mp->hdr.num_entry;
	n = 0;
	for (i=0; i<ne; i++) {
		ty = ep->bytes[0];
		if (ty < MP_ET_IOAPIC)
			goto next;
		if (ty != MP_ET_IOAPIC)
			break;
		if ((ep->a.ioapic_flags & 1) == 0)
			goto next;

		io_apic_addr[n] = physmap(ep->a.io_apic_adr,
					MMU_PAGESIZE, KM_NOSLEEP);
		ids[n] = ep->a.apic_id & 0xf;
		n++;
		if (n == MAXIOAPIC)
			break;
next:
		ep = (union mpcentry *)((char *)ep + mpcentry_sizes[ty]);
	}

	if (n == 0) {
		cmn_err(CE_WARN, "No I/O apics found - assume one\n");
		io_apic_addr[0] = physmap(APICIADR, MMU_PAGESIZE, KM_NOSLEEP);
		ids[0] = 15;
		n = 1;
	}
	apic_set_io_ids(ids,n);
	return n;
}

ap_table_ncpus()
{
	struct mpconfig *mp;
	union mpcentry *ep;
	int ne, n, i, ty, nonboot, id;

	mp = (struct mpconfig *)apic_table;
	ep = mp->entry;

	apic_eng_id[0] = ((volatile long *)local_apic_addr)[AP_ID] >> 24;
	ne = mp->hdr.num_entry;
	n = 0;
	nonboot = 0;
	for (i=0; i<ne; i++) {
		ty = ep->bytes[0];
		if (ty != MP_ET_PROC)
			break;
		if ((ep->p.cpu_flags & 1) == 0)
			goto next;
		id = ep->p.apic_id;
		if (id != apic_eng_id[0]) {
			apic_eng_id[nonboot+1] = id;
			nonboot++;
		}
		n++;
next:
		ep = (union mpcentry *)((char *)ep + mpcentry_sizes[ty]);
	}

	return n;
}

/*
 * find table entry with:
 *	entry type == byte0	(processor or i/o apic)
 *	apic id == byte1 (or byte1 == -1)
 *	must be enabled if ena_check
 * return index in *n_ptr
 */
static
union mpcentry *
ap_find_entry(byte0,byte1,ena_check,n_ptr)
int *n_ptr;
{
	struct mpconfig *mp;
	union mpcentry *ep;
	int ne, i, ty, n;

	mp = (struct mpconfig *)apic_table;
	ep = mp->entry;

	ne = mp->hdr.num_entry;
	n = -1;
	for (i=0; i<ne; i++) {
		ty = ep->bytes[0];
		if (ty > byte0)
			break;
		if (ty < byte0)
			goto next;
		/* check enable bit? */
		if (ena_check && (ep->bytes[3] & 1) == 0)
			goto next;
		n++;
		if (byte1 >= 0 && ep->bytes[1] != byte1)
			goto next;
		if (n_ptr)
			*n_ptr = n;
		return ep;
next:
		ep = (union mpcentry *)((char *)ep + mpcentry_sizes[ty]);
	}

	return 0;
}

/*
 * find table entry with:
 *	entry type == byte0
 *	sub-type == byte1
 *	byte[n] == entval
 */
static
union mpcentry *
ap_find_xentry(byte0,byte1,n,entval)
{
	struct mpconfig *mp;
	union mpcentry *ep;
	int ne, i, ty;

	mp = (struct mpconfig *)apic_table;
	ep = mp->entry;

	ne = mp->hdr.num_entry;
	for (i=0; i<ne; i++) {
		ty = ep->bytes[0];
		if (ty > byte0)
			break;
		if (ty < byte0)
			goto next;
		if (ep->bytes[1] != byte1 ||
		    ep->bytes[n] != entval)
			goto next;
		return ep;
next:
		ep = (union mpcentry *)((char *)ep + mpcentry_sizes[ty]);
	}

	return 0;
}
/*
 * set redirection entry for local or i/o apic (as per ioapic flag)
 * with apic_id specified (-1 for any)
 * at line specified
 *
 * v0 and v1 are the values to put in the redirection table entry
 */
static
ap_set_redir(ioapic,apic_id,line,v0,v1)
{
	volatile long *vp;
	union mpcentry *ep;
	int n;

	if (ioapic) {
		ep = ap_find_entry(MP_ET_IOAPIC,apic_id,1,&n);
		if (ep == 0)
			return;
		vp = (volatile long *)io_apic_addr[n];
		/* set second part (destination) first */
		vp[AP_IO_REG] = AIR_RDT2 + 2*line;
		vp[AP_IO_DATA] = v1;
		vp[AP_IO_REG] = AIR_RDT + 2*line;
		vp[AP_IO_DATA] = v0;
	} else  {
		vp = (volatile long *)local_apic_addr;
		if (line == 0)
			vp[AP_LVT_I0] = v0;
		else
			vp[AP_LVT_I1] = v0;
	}
}

void
apicinit()
{
	int nio;
	struct mpconfig *mp;
	volatile long *lp;

	/*
	 * the following may be skiped for engines other than 0.
	 * myengnum is always equal to 0 for boot engine.
	 */
	if (myengnum == 0) {
		apic_table_look();
		apic_table_chk();

		mp = (struct mpconfig *)apic_table;
		local_apic_addr = physmap(mp->hdr.loc_apic_adr,
				MMU_PAGESIZE, KM_NOSLEEP);
		taskpri_reg_addr = local_apic_addr + 0x80;
		eoi_reg_addr = local_apic_addr + 0xB0;
		nio = apic_n_io();
	}

	lp = (volatile long *)local_apic_addr;
	lp[AP_TASKPRI] = TPRIHI;	/* task pri */

	if (myengnum == 0) {
		oldpicinit();
		/* enable APIC symmetric I/O interrupt mode */
		if (pcmp_fib2 & MP_IMCRP) {
			outb(IMCR_ADDR, 0x70);
			outb(IMCR_DATA, 1);
		}
	} else {
		ipl = PLBASE;
	}

	lp[AP_LVT_I0] = AV_MASK;
	lp[AP_LVT_I1] = AV_MASK;
	lp[AP_LVT_TIMER] = AV_MASK;


	lp[AP_DESTFMT] = -1;
	lp[AP_LDEST] = APIC_LOGDEST(myengnum);

	lp[AP_SPUR] = 0x1cf;		/* enable local unit */

	if (myengnum != 0) {
		apic_xmsg(AV_RESET|AV_LEVEL|AV_DEASSERT|AV_XTOALL);
		return;
	}


	io_apic_init(nio);

	attach_drivers();
	apic_vecta[0xEE] = psm_aintr;
	attach_timer(clock);
	attach_nmi();
	if (internal_dma_chain)
		attach_chain(0);
}

io_apic_init(nio)
{
	volatile long *ip;
	int i, n;

	for (n=0; n<nio; n++) {
		ip = (volatile long *)io_apic_addr[n];
		/* mask unused slots */
		for (i=0; i<16; i++) {
			ip[AP_IO_REG] = AIR_RDT + 2*i;
			ip[AP_IO_DATA] = AV_MASK;
		}
	}
	for (i=0; i<256; i++)
		apic_vecta[i] = apic_stray;

	/*
	 * spurious interrupt will not generate EOI.
	 */
	apic_vecta[0xCF] = apic_spur;
}

alloc_apic_vec(pri, avp)
int *avp;
{
	int group, av;

	if (pri >= 7)
		pri = 8;
	for (group=0; group<16; group++)
		if (apic_frees[pri][group] == 0)
				goto found;
	cmn_err(CE_CONT, "cant allocate apic slot for %d\n", pri);
	for (;;) ;
found:
	av = APICINTS + (pri * 16) + group;
	apic_frees[pri][group] = av;

	*avp = av;
}

static
attach_drivers()
{
	unsigned char pri;
	void (*vec)();
	int irq, polarity, lvl, apic_vector_number;
	int io_id, line, rv, flags;

	for (irq=0; irq<16; irq++) {
		check_vec(irq,&pri,&vec);
		/*
		 * hpw notes:
		 * we don't need this here for up, but definitely for MP !!
	 	 */
		if (pri == 0)
			continue;
		if (vec == clock)	/* will be done in attach_timer */
			continue;

		alloc_apic_vec(pri, &apic_vector_number);
		apic_vect_xln[apic_vector_number] = irq;
		apic_vecta[apic_vector_number] = vec;

		rv = ap_find_connection(CONN_IRQ,5,irq,&io_id,&line,&flags);

		polarity = 0;
		if (bootinfo.machflags & EISA_IO_BUS) {
			if (level_intr_mask & (1<<irq))
				lvl = AV_LEVEL;
			else
				lvl = AV_EDGE;
		} else
			lvl = AV_EDGE;

		if (flags & MP_INTR_POVAL) {	/* table gives polarity */
			polarity = (flags & MP_INTR_POLOW) ?
				AV_POLOW : AV_POHIGH;
		}
		if (flags & MP_INTR_ELVAL) {	/* table gives edge/level */
			lvl = (flags & MP_INTR_ELLEVEL) ?
				AV_LEVEL : AV_EDGE;
		}

		if (rv < 0) {
			if (irq == 13)
				internal_dma_chain = 1;
			else
				cmn_err(CE_WARN, "irq line %d not connected\n",
					irq);
			continue;
		}

		if (intcpu[irq] == -1) { /* random */
		    if (intmp[irq] == 1) { /* multi-threaded */
			ap_set_redir(rv,io_id,line,
				apic_vector_number | lvl | polarity |
				AV_LDEST | AV_LOPRI,
				AV_TOALL);
		    } else {  /* all goes to boot engine */
			ap_set_redir(rv,io_id,line,
				apic_vector_number | lvl | polarity |
				AV_PDEST | AV_FIXED,
				APIC_ID(0)<<24);
		    }
		} else {  /* goes to intcpu[vec] */
			/*
			 * note that logical engine number in no
			 * way has a definite association with physical
			 * engine number.  it is up to the PSM to
			 * set up mapping between logical and physical
			 * engine number.
			 */
			ap_set_redir(rv,io_id,line,
				apic_vector_number | lvl | polarity |
				AV_PDEST | AV_FIXED,
				APIC_ID(intcpu[irq])<<24);
		}
	}
}

static
attach_timer(vec)
void (*vec)();
{
	volatile long *lp;
	int apic_vec;

	lp = (volatile long *)local_apic_addr;
	alloc_apic_vec(8,&apic_vec);
	apic_vecta[apic_vec] = vec;
	lp[AP_LVT_TIMER] = apic_vec | AV_MASK;
	clock_intr_vector = apic_vec;
}

static
attach_nmi()
{
	volatile long *lp;
	int rv, id, line;

	lp = (volatile long *)local_apic_addr;
	rv = ap_find_connection(CONN_NMI,4,0,&id,&line,0);
	if (rv >= 0)
		ap_set_redir(rv,id,line,
			AV_NMI|AV_LEVEL|AV_PDEST, APIC_ID(0));
}

static
attach_chain(eng)
{
	volatile long *lp;
	int apic_vec;
	int rv, io_id, line;
	void psm_intr();

	lp = (volatile long *)local_apic_addr;
	rv = ap_find_connection(CONN_PIC,4,0,&io_id,&line,0);
	if (rv >= 0)
		ap_set_redir(rv,io_id,line,
			AV_EXTINT|AV_EDGE|AV_PDEST, APIC_ID(eng));

	/* mask pic interrupts except  13-dma chaining */
	outb(imrport[0], 0xfb);
	outb(imrport[1], 0xdf);
	inb(imrport[0]);

	pic_to_apic[13] = apic_vec |
			(AV_FIXED|AV_PDEST|AV_EDGE|AV_XTOSELF);
}

apic_xmsg(x)
{
	volatile long *lp;
	pl_t pl;

	lp = (volatile long *)local_apic_addr;

		/* Disable interrupts before touching the 
		APIC's ICR register */
		asm ("pushfl");
		asm ("cli");

	while (lp[AP_ICMD] & AV_PENDING)
		;
	lp[AP_ICMD] = x;
	while (lp[AP_ICMD] & AV_PENDING)
		;
		/* Now restore interupt flag before exiting */
		asm ("popfl");
}

apic_xmsg2(x,eng)
{
	volatile long *lp;
	int x2;
	pl_t pl;

	x2 = APIC_ID(eng)<<24;

	lp = (volatile long *)local_apic_addr;

		/* Disable interrupts before touching the 
		APIC's ICR register */
		asm ("pushfl");
		asm ("cli");
	while (lp[AP_ICMD] & AV_PENDING)
		;
	lp[AP_ICMD2] = x2;
	while (lp[AP_ICMD] & AV_PENDING)
		;
	lp[AP_ICMD] = x;
	while (lp[AP_ICMD] & AV_PENDING)
		;
		/* Now restore interupt flag before exiting */
		asm ("popfl");
}

apic_selfintr(i)
{
	volatile long *lp;
	int x;

	if ((x=pic_to_apic[i]) == 0)
		return;

	lp = (volatile long *)local_apic_addr;

		/* Disable interrupts before touching the 
		APIC's ICR register */
		asm ("pushfl");
		asm ("cli");

	while (lp[AP_ICMD] & AV_PENDING)
		;
	lp[AP_ICMD] = x;
	while (lp[AP_ICMD] & AV_PENDING)
		;
		/* Now restore interupt flag before exiting */
		asm ("popfl");
}

apic_eoi13()
{
	outb(cmdport[1], PIC_NSEOI);
	outb(cmdport[0], PIC_NSEOI);
}


#define	_82489APIC_MASK	0xF0

apic_run_cpu(eng,addr)
{
	int id, i;
	union mpcentry *ep, *ap_find_entry();
	volatile long *lp;

	id = APIC_ID(eng);
	ep = ap_find_entry(MP_ET_PROC,id,1,0);
	if (ep == 0)
		return;

	lp = (volatile long *)local_apic_addr;
	if ((lp[AP_VERS] & _82489APIC_MASK) < 0x10) {
		/*
		 * INIT IPI for 82489DX.
		 */
		apic_xmsg2(AV_RESET|AV_LEVEL|AV_ASSERT, eng);
		apic_xmsg2(AV_RESET|AV_LEVEL|AV_DEASSERT, eng);
	} else {
		apic_xmsg2(AV_RESET|AV_LEVEL|AV_ASSERT, eng);
		apic_xmsg2(AV_RESET|AV_LEVEL|AV_DEASSERT, eng);

		apic_xmsg2(AV_STARTUP|AV_EDGE | (addr>>12), eng);
		apic_xmsg2(AV_STARTUP|AV_EDGE | (addr>>12), eng);
	}
}


check_vec(i, pp, vp)
unsigned char *pp;
void (**vp)();
{
	if (intpri[i] == 0) {
		*pp = 0;
		return;
	}
	*pp = intpri[i];
	*vp = ivect[i];
}

ipl0panic2()
{
	cmn_err(CE_PANIC,
  "ret_user: interrupt priority level not zero at return to user mode\n");
}

splpanic2(s)
{
	cmn_err(CE_PANIC,
		"splx(%d): logic error in misc.s\n", s);
}

#ifdef DEBUG
#ifdef SLOWSPL
spl7panic2()
{
	cmn_err(CE_PANIC,
		"spl7: logic error in misc.s\n");
}
#endif /* SLOWSPL */

splintpanic2()
{
	cmn_err(CE_PANIC,
		"splint: logic error in misc.s\n");
}

splxintpanic2()
{
	cmn_err(CE_PANIC,
		"splxint: logic error in misc.s\n");
}

setpicmaskspanic2()
{
	cmn_err(CE_PANIC,
		"setpicmasks: logic error in misc.s\n");
}
#endif /* DEBUG */

apic_llook()
{
	volatile long *p;
	long i, j, k, x;

	cmn_err(CE_CONT, "APIC registers\n");
	p = (volatile long *)local_apic_addr;
	for (i=0; i<=0x3f; i++) {
		if ((i&3) == 0)
			cmn_err(CE_CONT, "%x: ", i);
		x = p[i*4];
		cmn_err(CE_CONT, "%x ", x);
		if ((i&3) == 3)
			cmn_err(CE_CONT, "\n");
	}
}

apic_rlook(xcpuid)
{
	volatile long *p;
	long i, j, k, x;

	cmn_err(CE_CONT, "Remote APIC registers\n");
	p = (volatile long *)local_apic_addr;
	for (i=0; i<=0x3f; i++) {
		if ((i&3) == 0)
			cmn_err(CE_CONT, "%x: ", i);
		apic_xmsg2(i | 0x300, xcpuid);
		while ((p[4*0x30] & 0x30000) == 0x10000)
			;
		if ((p[4*0x30] & 0x30000) == 0x20000)
			x = p[0xc*4];
		else
			x = 0xabcdef01;
		cmn_err(CE_CONT, "%x ", x);
		if ((i&3) == 3)
			cmn_err(CE_CONT, "\n");
	}
}

apic_ilook(n)
{
	volatile long *p;
	long i, j, k, x;

	cmn_err(CE_CONT, "I/O APIC registers\n");
	p = (volatile long *)io_apic_addr[n];
	for (i=0; i<=0x2f; i++) {
		if ((i&3) == 0)
			cmn_err(CE_CONT, "%x: ", i);
		*p = i;
		x = p[4];
		cmn_err(CE_CONT, "%x ", x);
		if ((i&3) == 3)
			cmn_err(CE_CONT, "\n");
	}
}


static void
apic_reset_io()
{
	volatile long *ip;
	int i, n;

	for (n=0; n<MAXIOAPIC; n++) {
		ip = (volatile long *)io_apic_addr[n];
		if (ip != NULL) {
			/* mask unused slots */
			for (i=0; i<16; i++) {
				ip[AP_IO_REG] = AIR_RDT + 2*i;
				ip[AP_IO_DATA] = AV_MASK;
			}
		}
	}
}

#define	AV_XTOOTHERS	0x000C0000

apic_reset()
{
	volatile long *lp;
	int	tmp;

	apic_reset_io();

	lp = (volatile long *)local_apic_addr;
	lp[AP_LDEST] = AV_IM_OFF;
	lp[AP_LVT_I0] = AV_MASK;
	lp[AP_LVT_I1] = AV_MASK;

	/*
	 * disable APIC timer interrupts on processor
	 */
	lp[AP_LVT_TIMER] = AV_MASK;

	lp[AP_SPUR] = 0x0cf;            /* disable local unit */
	tmp = lp[AP_SPUR];              /* make sure writes don't get posted */
}
