set define '^' verify off
prompt ...wwv_flow_application_install
create or replace package wwv_flow_application_install as
--------------------------------------------------------------------------------
--
--  Copyright (c) Oracle Corporation 2010 - 2019. All Rights Reserved.
--
--    NAME
--      wwv_flow_application_install.sql
--
--    DESCRIPTION
--      Methods used to control APEX application installation behavior.  If not set,
--      then an APEX application will install normally.  However, if these global are set,
--      then these will take precedence over those specified in the application export file
--
--    RUNTIME DEPLOYMENT: YES
--
--    MODIFIED   (MM/DD/YYYY)
--      jkallman  01/28/2010 - Created
--      jkallman  08/16/2012 - Added publish_application
--      jkallman  08/19/2012 - Added p_new_translated_app_id to publish_application
--      cbcho     07/11/2014 - Added set_auto_install_sup_obj, get_auto_install_sup_obj (feature #1344)
--      pawolf    02/23/2015 - Added set_static_(app/theme/plugin)_file_prefix, get_static_(app/theme/plugin)_file_prefix (feature #1165)
--      cneumuel  01/26/2016 - Added set_workspace (feature #1756)
--      cneumuel  10/03/2016 - Added set_keep_sessions (feature #2067)
--      cneumuel  11/08/2016 - In get_keep_sessions, default to KEEP_SESSIONS_ON_UPGRADE (feature #2067)
--      cczarski  11/28/2017 - Added no_proxy_domains attribute (feature #2249)
--      cczarski  11/29/2017 - Added set|get_remote_server to change remote server base URLs during install (feature #2092, #2109)
--      cczarski  12/01/2017 - added ords_timezone to set|get_remote_server
--
--------------------------------------------------------------------------------

--##############################################################################
--#
--# CONFIGURATION
--#
--##############################################################################

--==============================================================================
-- Use this function to set the workspace ID for the application to be imported.
--
-- ARGUMENTS
-- * p_workspace_id: The workspace ID
--
-- EXAMPLE
--   Set workspace ID for workspace FRED_PROD.
--
--   declare
--       l_workspace_id number;
--   begin
--       select workspace_id into l_workspace_id
--         from apex_workspaces
--        where workspace = 'FRED_PROD';
--
--       apex_application_install.set_workspace_id (
--           p_workspace_id => l_workspace_id );
--   end;
--
-- SEE ALSO
--   set_workspace, get_workspace_id
--==============================================================================
procedure set_workspace_id(
    p_workspace_id in number );

--==============================================================================
-- Use this function to set the workspace ID for the application to be imported.
--
-- ARGUMENTS
-- * p_workspace: The workspace name
--
-- EXAMPLE
--   Set workspace ID for workspace FRED_PROD.
--
--   apex_application_install.set_workspace (
--       p_workspace => 'FRED_PROD' );
--
-- SEE ALSO
--   set_workspace, get_workspace_id
--==============================================================================
procedure set_workspace (
    p_workspace in varchar2 );

--==============================================================================
-- Use this function to get the workspace ID for the application to be imported.
--
-- ARGUMENTS
--
-- EXAMPLE
--   The following example returns the value of the workspace ID value in the APEX_APPLICATION_INSTALL package.
--
--   declare
--       l_workspace_id number;
--   begin
--       l_workspace_id := apex_application_install.get_workspace_id;
--   end;
--
-- SEE ALSO
--   set_workspace_id, set_workspace
--==============================================================================
function get_workspace_id
    return number;

--==============================================================================
-- Application ID
--==============================================================================
procedure set_application_id(
    p_application_id in number );

function get_application_id
    return number;

procedure generate_application_id;


--==============================================================================
-- Offset
--==============================================================================
procedure set_offset(
    p_offset in number );

procedure generate_offset;

function get_offset
    return number;


--==============================================================================
-- Schema
--==============================================================================
procedure set_schema(
    p_schema in varchar2 );

function get_schema
    return varchar2;


--==============================================================================
-- Name
--==============================================================================
procedure set_application_name(
    p_application_name in varchar2 );

function get_application_name
    return varchar2;


--==============================================================================
-- Alias
--==============================================================================
procedure set_application_alias(
    p_application_alias in varchar2 );

function get_application_alias
    return varchar2;


--==============================================================================
-- Image Prefix
--==============================================================================
procedure set_image_prefix(
    p_image_prefix in varchar2 );

function get_image_prefix
    return varchar2;


--==============================================================================
-- Proxy
--==============================================================================
procedure set_proxy(
    p_proxy            in varchar2,
    p_no_proxy_domains in varchar2 default null );

function get_proxy
    return varchar2;

function get_no_proxy_domains
    return varchar2;

--==============================================================================
-- Remote Servers
--==============================================================================
procedure set_remote_server(
    p_static_id        in varchar2,
    p_base_url         in varchar2,
    p_https_host       in varchar2 default null,
    p_ords_timezone    in varchar2 default null );

function get_remote_server_base_url(
    p_static_id        in varchar2 )
    return varchar2;

function get_remote_server_https_host(
    p_static_id        in varchar2 )
    return varchar2;

function get_remote_server_ords_tz(
    p_static_id        in varchar2 )
    return varchar2;

--==============================================================================
-- Auto Install of Supporting Objects
--==============================================================================
procedure set_auto_install_sup_obj(
    p_auto_install_sup_obj in boolean );

function get_auto_install_sup_obj
    return boolean;

--==============================================================================
-- Static Application File Prefix
--==============================================================================
procedure set_static_app_file_prefix(
    p_file_prefix in varchar2 );

function get_static_app_file_prefix
    return varchar2;

--==============================================================================
-- Static Theme File Prefix
--==============================================================================
procedure set_static_theme_file_prefix(
    p_theme_number in number,
    p_file_prefix  in varchar2 );

function get_static_theme_file_prefix(
    p_theme_number in number )
    return varchar2;

--==============================================================================
-- Static Plugin File Prefix
--==============================================================================
procedure set_static_plugin_file_prefix(
    p_plugin_type in varchar2,
    p_plugin_name in varchar2,
    p_file_prefix in varchar2 );

function get_static_plugin_file_prefix(
    p_plugin_type in varchar2,
    p_plugin_name in varchar2 )
    return varchar2;

--==============================================================================
-- Use this procedure to preserve sessions associated with the application on upgrades.
--
-- ARGUMENTS
-- * p_keep_sessions: TRUE if sessions should be preserved, FALSE if they should be deleted. The instance parameter
--                    KEEP_SESSIONS_ON_UPGRADE controls the default behavior. If it is N (the default), sessions will be
--                    deleted.
--
-- EXAMPLE
--   Install application 100 in workspace FRED_PROD and keep session state.
--
--     SQL> exec apex_application_install.set_workspace(p_workspace => 'FRED_PROD');
--     SQL> exec apex_application_install.set_keep_sessions(p_keep_sessions => true);
--     SQL> @f100.sql
--
-- SEE ALSO
--   get_keep_sessions, apex_instance_admin
--==============================================================================
procedure set_keep_sessions (
    p_keep_sessions in boolean );

--==============================================================================
-- Use this function to find out if sessions and session state will be preserved or deleted on upgrades.
--
-- EXAMPLE
--   Print whether sessions will be kept or deleted.
--
--     dbms_output.put_line (
--         case when apex_application_install.get_keep_sessions then 'sessions will be kept'
--         else 'sessions will be deleted'
--         end );
--
-- SEE ALSO
--   set_keep_sessions
--==============================================================================
function get_keep_sessions
    return boolean;

--==============================================================================
-- Use this procedure to override the active authentication scheme for the
-- applications that are about to be installed.
--
-- ARGUMENTS
-- * p_name: The name of the authentication scheme that should be activated. A
--           scheme with the given name must exist in the application. If null,
--           the active authentication scheme will remain unchanged.
--
-- EXAMPLE
--   Activate authentication scheme "SSO-Production" and install application
--   f100.sql, then reset the override for f101.sql to keep it's active scheme.
--
--     begin
--       apex_application_install.set_authentication_scheme (
--           p_name => 'SSO-Production' );
--     end;
--     /
--     @f100.sql
--     begin
--       apex_application_install.set_authentication_scheme (
--           p_name => null );
--     end;
--     /
--     @f101.sql
--
--  SEE ALSO
--    get_authentication_scheme
--==============================================================================
procedure set_authentication_scheme (
    p_name in varchar2 );

--==============================================================================
-- Use this function to retrieve the authentication scheme name that should
-- override the default.
--
-- ARGUMENTS
--
-- EXAMPLE
--   Print the authentication scheme override.
--
--     select apex_application_install.get_authentication_scheme from sys.dual;
--
--  SEE ALSO
--    set_authentication_scheme
--==============================================================================
function get_authentication_scheme
    return varchar2;

--==============================================================================
-- Use this procedure to override the build status for applications that are
-- about to be installed.
--
-- ARGUMENTS
-- * p_build_status: New build status to set application to.  Values include:
--                   - RUN_AND_BUILD - Developers and users can both run and
--                     develop the application.
--                   - RUN_ONLY - Users can only run the application.
--                     Developers cannot edit the application.
--
-- EXAMPLE
--   Set build status for app 100 to RUN_ONLY.
--
--     begin
--       apex_application_install.set_build_status (
--           p_build_status => 'RUN_ONLY' );
--     end;
--     /
--     @f100.sql
--==============================================================================
procedure set_build_status (
    p_build_status in varchar2 );

--==============================================================================
-- Use this function to retrieve the build status that should override the
-- default.
--
-- ARGUMENTS
--
-- EXAMPLE
--   Print the build status override.
--
--     select apex_application_install.get_build_status from sys.dual;
--
--  SEE ALSO
--    set_build_status
--==============================================================================
function get_build_status
    return varchar2;

--==============================================================================
-- Clear
--==============================================================================
procedure clear_all;

--##############################################################################
--#
--# IMPORT
--#
--##############################################################################

--==============================================================================
-- Kinds of install files.
--==============================================================================
subtype t_file_type is pls_integer range 1 .. 5;
c_file_type_workspace           constant t_file_type := 1;
c_file_type_app                 constant t_file_type := 2;
c_file_type_websheet            constant t_file_type := 3;
c_file_type_plugin              constant t_file_type := 4;
c_file_type_css                 constant t_file_type := 5;

--==============================================================================
-- Kinds of application usage.
--==============================================================================
subtype t_app_usage is pls_integer range 1..3;
c_app_usage_not_used            constant t_app_usage := 1;
c_app_usage_current_workspace   constant t_app_usage := 2;
c_app_usage_other_workspace     constant t_app_usage := 3;

--==============================================================================
-- Information in a source file that can be used to configure the installation.
--==============================================================================
type t_file_info is record (
    file_type                   t_file_type,
    workspace_id                number,
    version                     varchar2(10),
    app_id                      number,
    app_name                    varchar2(4000),
    app_alias                   varchar2(4000),
    app_owner                   varchar2(4000),
    build_status                varchar2(4000),
    has_install_script          boolean,
    app_id_usage                t_app_usage,
    app_alias_usage             t_app_usage );

--==============================================================================
-- Retrieve install information from a source file.
--
-- ARGUMENTS
-- * p_source: The source code, a table of (name, contents) with a single record
--             for normal APEX applications or multiple records for applications
--             that were split when exporting. Note that passing multiple
--             applications is not supported.
--
-- RETURNS
-- * Information about the application that can be used to configure
--   installation.
--
-- RAISES
-- * WWV_FLOW_IMP_PARSER.RUN_STMT_ERROR: The source contains invalid statements.
--
-- EXAMPLE
--   Fetch an application from a remote URL and print it's install
--   information.
--
--     declare
--         l_source apex_t_export_files;
--         l_info   apex_application_install.t_file_info;
--     begin
--         l_source := apex_t_export_files (
--                         apex_t_export_file (
--                             name     => 'f100.sql',
--                             contents => apex_web_service.make_rest_request (
--                                             p_url         => 'https://www.example.com/apps/f100.sql',
--                                             p_http_method => 'GET' )));
--         l_info   := apex_application_install.get_info (
--                         p_source      => l_source );
--         sys.dbms_output.put_line (apex_string.format (
--             p_message => q'!Type ................. %0
--                            !Workspace ............ %1
--                            !Version .............. %2
--                            !App ID ............... %3
--                            !App Name ............. %4
--                            !Alias ................ %5
--                            !Owner ................ %6
--                            !Build Status ......... %7
--                            !Has Install Script ... %8
--                            !App ID Usage ......... %9
--                            !App Alias Usage ...... %10!',
--             p0        => l_info.file_type,
--             p1        => l_info.workspace_id,
--             p2        => l_info.version,
--             p3        => l_info.app_id,
--             p4        => l_info.app_name,
--             p5        => l_info.app_alias,
--             p6        => l_info.app_owner,
--             p7        => l_info.build_status,
--             p8        => apex_debug.tochar(l_info.has_install_script),
--             p9        => l_info.app_id_usage,
--             p10       => l_info.app_alias_usage,
--             p_prefix  => '!' ));
--     end;
--
-- SEE ALSO
--   install, apex_export.get_application
--==============================================================================
function get_info (
    p_source                    in wwv_flow_t_export_files )
    return t_file_info;

--==============================================================================
-- Install an application. Use the apex_application_install.set% procedures to
-- configure installation parameters.
--
-- ARGUMENTS
-- * p_source:             The source code, a table of (name, contents) with a
--                         single record for normal APEX applications or
--                         multiple records for applications that were split
--                         when exporting. Note that passing multiple
--                         applications is not supported. If null (the default),
--                         import the source that was previously passed to
--                         get_info.
-- * p_overwrite_existing: If false (the default), raise an error instead of
--                         overwriting an existing application.
--
-- RAISES
-- * WWV_FLOW_IMP_PARSER.RUN_STMT_ERROR: The source contains invalid statements.
-- * SECURITY_GROUP_ID_INVALID: The current workspace conflicts with the install
--   workspace.
-- * WWV_FLOW_API.FLOW_ID_RESERVED_FOR_OTHER_WORKSPACE: The application ID is
--   used in another workspace.
-- * WWV_FLOW_API.FLOW_ID_RANGE_RESERVED: The application ID is reserved for
--   internal use.
-- * WWV_FLOW_API.FLOW_ID_OUT_OF_RANGE: The application ID used for installing
--   is not in a valid range.
-- * APPLICATION_ID_RESERVED: The application ID is in use in the current
--   workspace and p_overwrite_existing was set to false.
--
-- EXAMPLE
--   Fetch an application from a remote URL, then install it with a new ID and
--   new component ID offsets in workspace EXAMPLE.
--
--     declare
--         l_source apex_t_export_files;
--         l_info   apex_application_install.t_file_info;
--     begin
--         l_source := apex_t_export_files (
--                         apex_t_export_file (
--                             name     => 'f100.sql',
--                             contents => apex_web_service.make_rest_request (
--                                             p_url         => 'https://www.example.com/apps/f100.sql',
--                                             p_http_method => 'GET' )));
--
--         apex_util.set_workspace('EXAMPLE');
--         apex_application_install.generate_application_id;
--         apex_application_install.generate_offset;
--         apex_application_install.install (
--              p_source => l_source );
--     end;
--
-- SEE ALSO
--   get_info, apex_export.get_application
--==============================================================================
procedure install (
    p_source             in wwv_flow_t_export_files default null,
    p_overwrite_existing in boolean                 default false );

--==============================================================================
-- Remove an application. Use the apex_application_install.set% procedures to
-- configure installation parameters.
--
-- ARGUMENTS
-- * p_application_id:     The ID of the application to remove
--
-- RAISES
-- * WWV_FLOW_API.DELETE_APP_IN_DIFFERENT_WORKSPACE: The application is not
--   in this workspace.
-- * WWV_FLOW_API.FLOW_NOT_DELETED: The application was not deleted.
-- * WWV_FLOW.APP_NOT_FOUND_ERR: The application ID was not found.
--
-- EXAMPLE
--
--      begin
--          apex_application_install.set_workspace('EXAMPLE');
--          apex_application_install.set_keep_sessions(false);
--          apex_application_install.remove_application(100);
--      end;
--
--==============================================================================
procedure remove_application (
    p_application_id            in number );

--##############################################################################
--#
--# TRANSLATION
--#
--##############################################################################

--==============================================================================
-- Publish Application
--==============================================================================
procedure publish_application (
    p_application_id            in number,
    p_language_code             in varchar2,
    p_new_translated_app_id     in number default null );

end wwv_flow_application_install;
/
show errors
