set define '^' verify off
prompt ...wwv_flow_data_parser.sql
create or replace package wwv_flow_data_parser authid current_user as
--------------------------------------------------------------------------------
--
--  Copyright (c) Oracle Corporation 1999 - 2018. All Rights Reserved.
--
--    NAME
--      wwv_flow_data_parser.sql
--
--    DESCRIPTION
--      APEX File Parser: Implementation package. This package contains the implementation for the
--      file parser in APEX. The parser supports XML, JSON, CSV and XLSX files. The most important function
--      in this package is the PARSE function, which is implemented as a table function returning rows
--      of the WWV_FLOW_T_PARSER_ROW type. Thus the parser supports up to 300 columns.
--
--    PUBLIC SYNONYMS:
--        APEX_DATA_PARSER
--        WWV_FLOW_DATA_PARSER
--
--
--    MODIFIED   (MM/DD/YYYY)
--    cczarski    09/20/2017 - Created
-- 
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
-- public types
--------------------------------------------------------------------------------
subtype t_file_type        is pls_integer range 1..5;

type t_file_column is record(
    col_seq           pls_integer,
    name              varchar2(128),
    data_type         wwv_flow_exec_api.t_data_type,
    data_type_len     pls_integer,
    selector          varchar2(32767),
    decimal_char      varchar2(1),
    group_char        varchar2(1),
    format_mask       varchar2(128),
    clob_column       pls_integer,
    clob_col_all_rows boolean default false );

type t_file_columns is table of t_file_column index by pls_integer;

type t_file_profile is record(
    file_type          t_file_type,
    file_charset       varchar2(128),
    row_selector       varchar2(32767),
    is_single_row      boolean,
    first_row_headings boolean,
    xlsx_worksheet     varchar2(128),
    xml_namespaces     varchar2(4000),
    csv_delimiter      varchar2(4),
    csv_enclosed       varchar2(4),
    null_if            varchar2(20),
    parsed_rows        number,
    file_columns       t_file_columns );

--------------------------------------------------------------------------------
-- public constants
--------------------------------------------------------------------------------
c_file_type_xlsx              constant t_file_type      := 1;
c_file_type_csv               constant t_file_type      := 2;
c_file_type_xml               constant t_file_type      := 3;
c_file_type_json              constant t_file_type      := 4;
c_file_type_ics               constant t_file_type      := 5;

c_num_group_format            constant varchar2(57)     := '9G999G999G999GG999G999G999G999G999G999G999G999D9999999999';

c_value_cut_32k               constant varchar2(7)      := 'CUT_32K';
c_value_cut_4k                constant varchar2(6)      := 'CUT_4K';

--------------------------------------------------------------------------------
LF                            constant varchar2(1)      := wwv_flow.LF;
CR                            constant varchar2(1)      := wwv_flow.CR;
c_with_loop_killer            constant boolean          := false;


--==================================================================================================================
-- returns a file type, based on a file name extension.
--
-- PARAMETERS
--     p_file_name          file name to get the file type for
--
-- RETURNS
--     the file type as t_file_type
--
-- EXAMPLE
--
-- declare 
--     l_file_type wwv_flow_data_parser.t_file_type;
-- begin
--     l_file_type := wwv_flow_data_parser.get_file_type( 'test.xlsx' );
-- end;
--==================================================================================================================
function get_file_type(
    p_file_name in varchar2 ) return t_file_type;

--==================================================================================================================
-- returns worksheets of an XLSX workbook. This table returns an array of WWV_FLOW_T_PARSER_WORKSHEET rows. 
--
--    sheet_sequence          number,
--    sheet_display_name      varchar2(4000),
--    sheet_file_name         varchar2(4000),
--    sheet_path              varchar2(4000) )
--
-- PARAMETERS:
--     * p_content      XLSX worksheet as a BLOB
--
-- RETURNS
--     table with worksheet information
--
-- EXAMPLE:
--
-- select * from table(
--     wwv_flow_data_parser.get_xlsx_worksheets(
--         p_content => apex_web_service.make_rest_request_b('http://file-examples.com/wp-content/uploads/2017/02/file_example_XLSX_100.xlsx', 'GET' ) ) )
--
--
--  SHEET_SEQUENCE SHEET_DISPLAY_NAME   SHEET_FILE_NAME   SHEET_PATH              
--  -------------- -------------------- ----------------- ------------------------
--               1 Sheet1               sheet1.xml        worksheets/sheet1.xml   
--
--==================================================================================================================
function get_xlsx_worksheets(
    p_content   in blob ) return wwv_flow_t_parser_worksheets;

--==================================================================================================================
-- returns the current file profile in JSON format. A file profile is being generated when the parse() table
-- function runs and no file profile is being passed in. The file profile contains meta data about the parsed
-- files such as the CSV delimiter, the XLSX worksheet name and (most important) the columns found during parsing
-- and their data types.
--
-- The typical call sequence is as follows:
--
-- 1) Invoke PARSE 
--    Use this table function to parse the files and get rows and columns in order to display a data preview.
--    *While the function runs* it computes the file parser profile which can be used in subsequent calls 
--    in order to further process the data.
-- 2) Invoke GET_FILE_PROFILE
--    Retrieve file profile information in JSON format.
-- 3) further process the data
--
-- PARAMETERS
--    
-- RETURNS
--     file profile of the last PARSE() invocation in JSON format
--
-- EXAMPLE:
--
-- select line_number, col001,col002,col003,col004,col005,col006,col007,col008 
--   from table( 
--              wwv_flow_data_parser.parse(
--                  p_content         => apex_web_service.make_rest_request_b('http://file-examples.com/wp-content/uploads/2017/02/file_example_XLSX_100.xlsx', 'GET' ),
--                  p_file_name       => 'test.xlsx',
--                  p_xlsx_sheet_name => 'sheet1.xml') ) ;
--
--  LINE_NUMBER COL001   COL002       COL003       COL004   COL005          COL006   COL007       COL008 
--  ----------- -------- ------------ ------------ -------- --------------- -------- ------------ ------------- 
--            1 0        First Name   Last Name    Gender   Country         Age      Date         Id       
--            2 1        Dulce        Abril        Female   United States   32       15/10/2017   1562     
--            3 2        Mara         Hashimoto    Female   Great Britain   25       16/08/2016   1582     
--            4 3        Philip       Gent         Male     France          36       21/05/2015   2587     
--            5 4        Kathleen     Hanner       Female   United States   25       15/10/2017   3549     
--            6 5        Nereida      Magwood      Female   United States   58       16/08/2016   2468     
--            7 6        Gaston       Brumm        Male     United States   24       21/05/2015   2554     
--            8 7        Etta         Hurn         Female   Great Britain   56       15/10/2017   3598     
--            9 8        Earlean      Melgar       Female   United States   27       16/08/2016   2456     
--           10 9        Vincenza     Weiland      Female   United States   40       21/05/2015   6548     
--            : :        :            :            :        :               :        :            :
--
-- select wwv_flow_data_parser.get_file_profile from dual;
--
-- {
--    "file-type" : 1,
--    "csv-delimiter" : "",
--    "xslx-worksheet" : "sheet1.xml",
--    "headings-in-first-row" : true,
--    "file-encoding" : "AL32UTF8",
--    "single-row" : false,
--    "parsed-rows" : 2378,
--    "columns" : [
--       {
--          "format-mask" : "",
--          "name" : "C0",
--          "data-type" : 2,
--          "selector" : ""
--       },
--       {
--          "name" : "FIRST_NAME",
--          "data-type" : 1,
--          "selector" : "",
--          "format-mask" : ""
--       },
--       {
--          "selector" : "",
--          "data-type" : 1,
--          "name" : "LAST_NAME",
--          "format-mask" : ""
--       },
--       {
--          "format-mask" : "",
--          "data-type" : 1,
--          "name" : "GENDER",
--          "selector" : ""
--       },
--       {
--          "name" : "COUNTRY",
--          "data-type" : 1,
--          "selector" : "",
--          "format-mask" : ""
--       },
--       {
--          "data-type" : 2,
--          "name" : "AGE",
--          "selector" : "",
--          "format-mask" : ""
--       },
--       {
--          "format-mask" : "DD\"/\"MM\"/\"YYYY",
--          "selector" : "",
--          "data-type" : 3,
--          "name" : "DATE_"
--       },
--       {
--          "name" : "ID",
--          "data-type" : 2,
--          "selector" : "",
--          "format-mask" : ""
--       }
--    ],
--    "row-selector" : ""
-- }
--
--==================================================================================================================
function get_file_profile return clob;


--==================================================================================================================
-- converts a file profile to JSON format
--
-- PARAMETERS
--     p_file_profile     the data profile as an instance of t_file_profile
--
-- RETURNS
--     the file profile in JSON format.
--
-- EXAMPLE:
--    
-- declare
--     l_profile t_file_profile;
--     l_json    clob;
-- begin
--     -- build the file profile here ...
-- 
--     l_json := wwv_flow_data_parser.json_to_profile( l_profile );
--     
-- end;
-- 
--==================================================================================================================
function profile_to_json( p_file_profile in t_file_profile ) return clob;

--==================================================================================================================
-- converts a file profile in JSON format to an instance of the t_file_profile record
-- type.
--
-- PARAMETERS
--     p_json     the data profile in JSON format
--
-- RETURNS
--     the file profile in JSON format.
--
-- EXAMPLE:
--    
-- declare
--     l_profile t_file_profile;
-- begin
--     l_profile := wwv_flow_data_parser.json_to_profile( '{"file-type", "csv-delimiter" : "", ... }' );
--     
-- end;
--
--==================================================================================================================
function json_to_profile( p_json in clob ) return t_file_profile;

--==================================================================================================================
-- This is the main parser function. It allows to parse XML, XLSX, CSV or JSON files and returns a generic
-- table of the following structure:
--
-- LINE_NUMBER COL001 COL002 COL003 COL004 ... COL300
-- ----------- ------ ------ ------ ------     ------

-- All values will be returned in VARCHAR2 format. A returned table row can have a maximum of 300 columns. The
-- maximum length for a table column is 4000 bytes; there is no line length maximum.
-- File parsing happens on-the-fly as this function is being invoked. No data is written to a collection or
-- to a temporary table.
--
-- If the P_FILE_PROFILE parameter is *not* being passed, the
-- function will compute a file profile with column information during parse. If P_DETECT_DATA_TYPES is passed
-- as 'Y' (default), the function will also detect column data types during parse. The computed file profile
-- can be retrieved using GET_FILE_PROFILE after this function is finished.
--
-- 1) Invoke PARSE 
--    Use this table function to parse the files and get rows and columns in order to display a data preview.
--    *While the function runs* it computes the file parser profile which can be used in subsequent calls 
--    in order to further process the data.
-- 2) Invoke GET_FILE_PROFILE
--    Retrieve file profile information in JSON format.
-- 3) further process the data
--    a) use the WWV_FLOW_FILE_DATALOAD package to load into a new or existing table
--    b) generate a SQL query based on the data profile to perform custom processing
--
-- Notes:
--     -) JSON parsing is supported on 11.2 and 12.1.0.1 database versions. In this case, the function uses
--        WWV_FLOW_JSON and XMLTABLE functions. For performance reasons it's recommended to upgrade the
--        database to at least 12.2 - JSON parsing is faster by magnitudes on these versions.
--     -) XLSX parsing is done by using WWV_FLOW_ZIP to extract individual XML files from the XLSX archive; the 
--        actual XLSX parsing is then done by using the XMLTABLE SQL function.
--     
-- PARAMETERS:
--     P_CONTENT                      the file content to be parsed as a BLOB
--     P_FILE_NAME                    the name of the file; only used to derive the file type. Either P_FILE_NAME, P_FILE_TYPE
--                                    or P_FILE_PROFILE must be passed in.
--     P_FILE_TYPE                    the type of the file to be parsed. Use this to explicitly pass the file type in. Either 
--                                    P_FILE_NAME, P_FILE_TYPE or P_FILE_PROFILE must be passed in.
--     P_FILE_PROFILE                 File profile to be used for parsing. The file profile might have been computed in 
--                                    a previous PARSE() invocation. If passed in again, the function will skip some profile
--                                    detection logic and use the passed in profile - in order to improve performance.
--
--     P_DETECT_DATA_TYPES            Whether to detect data types (NUMBER, DATE, TIMESTAMP) during parsing. If set to 'Y',
--                                    the function will compute the file profile and also add data type information to it.
--                                    If set to 'N', no data types will be detected and all columns will be VARCHAR2.
--                                    Default: 'Y'
--     P_DECIMAL_CHAR                 use this decimal character when trying to detect NUMBER data types. If not specified,
--                                    the procedure will auto-detect the decimal character.
--
--     P_XLSX_SHEET_NAME              For XLSX workbooks. The name of the worksheet to parse. If omitted, the function will
--                                    use the first worksheet found.
--     P_ROW_SELECTOR                 For JSON and XML files. Pointer to the array / list of rows within the JSON or XML
--                                    file. If omitted, the function will ...
--                                    -) For XML files: Use "/*/*" (first tag under the root tag) as the row selector. 
--                                    -) For JSON files: Look for a JSON array and use the first array found.
--     P_CSV_ROW_DELIMITER            Override the default row delimiter for CSV parsing
--     P_CSV_COL_DELIMITER            Use a specific CSV column delimiter. If omitted, the function will detect the column
--                                    delimiter based on the first row contents.
--     P_CSV_ENCLOSED                 Override the default enclosure character for CSV parsing
--
--     P_SKIP_ROWS                    Skip the first N rows when parsing
--     P_ADD_HEADERS_ROW              for XML, JSON as ICS: Emit the column headers (tag, attr names) as the first row
--
--     P_NULLIF                       value to treat as NULL
--
--     P_FILE_CHARSET                 File encoding, if not UTF-8 (AL32UTF8)
--     P_MAX_ROWS                     Stop parsing after P_MAX_ROWS have been returned.
--     P_RETURN_ROWS                  Amount of rows to return. When P_RETURN_ROWS have been emitted, the function
--                                    will continue parsing (and refining the column profile) until P_MAX_ROWS has been
--                                    reached or until the ROWNUM < x clause of the SQL query kicks in and stops execution.
--
--     P_STORE_PROFILE_TO_COLLECTION  store the File profile which has been computed during parse into a collection. The
--                                    collection will be cleared, if it exists. Only be used for computed profiles.
--
-- RETURNS
--      rows of the wwv_flow_t_parser_row type
--     
--      LINE_NUMBER COL001 COL002 COL003 COL004 ... COL300
--      ----------- ------ ------ ------ ------     ------
--
-- EXAMPLE
--
-- select line_number, col001,col002,col003,col004,col005,col006,col007,col008 
--   from table( 
--              wwv_flow_data_parser.parse(
--                  p_content         => apex_web_service.make_rest_request_b('http://file-examples.com/wp-content/uploads/2017/02/file_example_XLSX_100.xlsx', 'GET' ),
--                  p_file_name       => 'test.xlsx',
--                  p_xlsx_sheet_name => 'sheet1.xml') ) ;
--
--  LINE_NUMBER COL001   COL002       COL003       COL004   COL005          COL006   COL007       COL008 
--  ----------- -------- ------------ ------------ -------- --------------- -------- ------------ ------------- 
--            1 0        First Name   Last Name    Gender   Country         Age      Date         Id       
--            2 1        Dulce        Abril        Female   United States   32       15/10/2017   1562     
--            3 2        Mara         Hashimoto    Female   Great Britain   25       16/08/2016   1582     
--            4 3        Philip       Gent         Male     France          36       21/05/2015   2587     
--            5 4        Kathleen     Hanner       Female   United States   25       15/10/2017   3549     
--            6 5        Nereida      Magwood      Female   United States   58       16/08/2016   2468     
--            7 6        Gaston       Brumm        Male     United States   24       21/05/2015   2554     
--            8 7        Etta         Hurn         Female   Great Britain   56       15/10/2017   3598     
--            9 8        Earlean      Melgar       Female   United States   27       16/08/2016   2456     
--           10 9        Vincenza     Weiland      Female   United States   40       21/05/2015   6548     
--            : :        :            :            :        :               :        :            :
--
-- select line_number, col001,col002,col003,col004,col005,col006,col007,col008 
--   from table( 
--              wwv_flow_data_parser.parse(
--                  p_content         => apex_web_service.make_rest_request_b('https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/all_hour.geojson', 'GET' ),
--                  p_file_name       => 'test.xlsx',
--                  p_xlsx_sheet_name => 'sheet1.xml') ) ;
--
-- LINE_NUMBER COL001    COL002   COL003                                COL004          COL005          
-- ----------- --------- ---------------------------------------------- --------------- --------------
--           1 Feature   1.5      41km E of Cape Yakataga, Alaska       1536513727239   1536514117117   
--           2 Feature   0.21     11km ENE of Aguanga, CA               1536513299520   1536513521231   
--           3 Feature   1.84     5km SSW of Pahala, Hawaii             1536513262940   1536513459610   
--           4 Feature   2.55     9km W of Volcano, Hawaii              1536513100890   1536513446680   
--           5 Feature   1.3      62km ESE of Cape Yakataga, Alaska     1536512917361   1536513322236   
--           6 Feature   1.79     7km SW of Tiptonville, Tennessee      1536512379690   1536512668010   
--           7 Feature   1.9      126km NNW of Arctic Village, Alaska   1536512346186   1536512846567   
--           8 Feature   1.4      105km NW of Arctic Village, Alaska    1536512140162   1536512846334
--
--
-- CLOB - SUPPORT
-- --------------
--
-- Starting with APEX 19.2, this package supports string values larger than 4000 bytes. 20 out of the 300 supported
-- columns can be handled as a CLOB. In this case, a second column (CLOB01 .. CLOB20) will be used to hold the CLOB
-- value. The level of CLOB support depends on the file type to be parsed.
--
-- a) CSV and XLSX:
--    - CLOB values are supported up to 32K
--    - CLOB columns can be detected during discovery
--    - When the data profile is discovered, values below 4000 bytes are normally returned as COLNNN. CLOB values
--      are returned in the CLOBNN column and the first 1000 characters are returned as COLNNN. If a data profile
--      is passed in and that has CLOB column defined, all values are returned in the CLOBNN column only.
-- b) XML
--    - CLOB values with more than 32K are supported 
--    - CLOB columns can be detected during discovery
--    - When the data profile is discovered, values below 4000 bytes are normally returned as COLNNN. CLOB values
--      are returned in the CLOBNN column and the first 1000 characters are returned as COLNNN. If a data profile
--      is passed in and that has CLOB column defined, all values are returned in the CLOBNN column only.
--
-- c) JSON
--    - CLOB values with more than 32K are supported 
--    - CLOB columns are *not* detected during discovery; CLOB support is only active, if a file profile 
--      containing CLOB column is passed in as the p_file_profile parameter
--    - Since JSON_TABLE does not support CLOBs on 12c databases, the parser uses XMLTYPE-based processing
--      if a file profile with CLOB columns is passed in. Processing will be significantly slower, then.
-- 
--==================================================================================================================
function parse(
    p_content                      in blob,
    p_file_name                    in varchar2     default null,
    p_file_type                    in t_file_type  default null,
    p_file_profile                 in clob         default null,
    --
    p_detect_data_types            in varchar2     default 'Y',
    p_decimal_char                 in varchar2     default null,
    --
    p_xlsx_sheet_name              in varchar2     default null,
    --
    p_row_selector                 in varchar2     default null,
    --
    p_csv_row_delimiter            in varchar2     default LF,
    p_csv_col_delimiter            in varchar2     default null,
    p_csv_enclosed                 in varchar2     default '"',
    --
    p_skip_rows                    in pls_integer  default 0,
    p_add_headers_row              in varchar2     default 'N',
    --
    p_nullif                       in varchar2     default null,
    --
    p_file_charset                 in varchar2     default 'AL32UTF8',
    p_max_rows                     in number       default null,
    p_return_rows                  in number       default null,
    --
    p_store_profile_to_collection  in varchar2     default null ) return wwv_flow_t_parser_table pipelined;

--==================================================================================================================
-- This is a convenience function to discover the column profile of a file. It just calls parse() and then returns
-- the generated file profile. This function is a shurtcut which can be used instead of first calling parse() and
-- then get_file_profile().
--     
-- PARAMETERS:
--     P_CONTENT             the file content to be parsed as a BLOB
--     P_FILE_NAME           the name of the file; used to derive the file type. 
--
--     P_DECIMAL_CHAR        use this decimal character when trying to detect NUMBER data types. If not specified,
--                           the procedure will auto-detect the decimal character.
-- 
--     P_XLSX_SHEET_NAME     For XLSX workbooks. The name of the worksheet to parse. If omitted, the function will
--                           use the first worksheet found.
--
--     P_ROW_SELECTOR        For JSON and XML files. Pointer to the array / list of rows within the JSON or XML
--                           file. If omitted, the function will ...
--                           -) For XML files: Use "/*/*" (first tag under the root tag) as the row selector. 
--                           -) For JSON files: Look for a JSON array and use the first array found.
--
--     P_CSV_ROW_DELIMITER   Override the default row delimiter for CSV parsing
--     P_CSV_COL_DELIMITER   Use a specific CSV column delimiter. If omitted, the function will detect the column
--                           delimiter based on the first row contents.
--     P_CSV_ENCLOSED        Override the default enclosure character for CSV parsing
--
--     P_FILE_CHARSET        File encoding, if not UTF-8 (AL32UTF8)
--     P_MAX_ROWS            Stop discovery after P_MAX_ROWS rows have been processed.
--
-- RETURNS
--      a CLOB containing the file profile in JSON format
--
-- EXAMPLE
--
-- select wwv_flow_data_parser.discover(
--            p_content => apex_web_service.make_rest_request_b('http://file-examples.com/wp-content/uploads/2017/02/file_example_XLSX_100.xlsx', 'GET' ),
--            p_file_name=>'large.xlsx' ) as profile_json
-- from dual;
--
--
-- PROFILE_JSON
-- -----------------------------------------------------------
-- {
--    "file-encoding" : "AL32UTF8",
--    "single-row" : false,
--    "file-type" : 1,
--    "parsed-rows" : 2189,
--    "columns" : [
--       {
--          "name" : "C0",
--          "format-mask" : "",
--          "selector" : "",
--          "data-type" : 2
--       },
--       {
--          "selector" : "",
--          "format-mask" : "",
--          "data-type" : 1,
--          "name" : "FIRST_NAME"
--       },
--       {
--          "name" : "LAST_NAME",
--          "format-mask" : "",
--          "selector" : "",
--          "data-type" : 1
--       },
--
--       :
--
--       {
--          "name" : "DATE_",
--          "format-mask" : "DD\"/\"MM\"/\"YYYY",
--          "data-type" : 3,
--          "selector" : ""
--       },
--       {
--          "format-mask" : "",
--          "selector" : "",
--          "data-type" : 2,
--          "name" : "ID"
--       }
--    ],
--    "row-selector" : "",
--    "headings-in-first-row" : true,
--    "xslx-worksheet" : "sheet1.xml",
--    "csv-delimiter" : ""
-- }
-- 
--==================================================================================================================
function discover(
    p_content             in blob,
    p_file_name           in varchar2,
    --
    p_decimal_char        in varchar2                                     default null,
    --
    p_xlsx_sheet_name     in varchar2                                     default null,
    --
    p_row_selector        in varchar2                                     default null,
    --
    p_csv_row_delimiter   in varchar2                                     default LF,
    p_csv_col_delimiter   in varchar2                                     default null,
    p_csv_enclosed        in varchar2                                     default '"',
    --
    p_skip_rows           in pls_integer                                  default 0,
    --
    p_file_charset        in varchar2                                     default 'AL32UTF8',
    p_max_rows            in number                                       default 200 ) return clob;

--==================================================================================================================
-- This is a convenience function to return the columns of a parser profile as a table in order to be consumed
-- by APEX components.
--
-- PARAMETERS
--     P_FILE_PROFILE        File profile to be used for parsing. The file profile might have been computed in 
--                           a previous PARSE() or DISCOVER() invocation. 
--
-- RETURNS
--     Profile column information as rows of WWV_FLOW_T_PARSER_COLUMNS
--
-- EXAMPLE
--     this example uses DISCOVER() to compute a file profile and then GET_COLUMS() to return the list
--     of columns among with their data types.
--
--     select *
--      from table(
--                  apex_data_parser.get_columns(
--                      apex_data_parser.discover(
--                          p_content => apex_web_service.make_rest_request_b('http://file-examples.com/wp-content/uploads/2017/02/file_example_XLSX_100.xlsx', 'GET' ),
--                          p_file_name=>'large.xlsx' )));
--
--     COLUMN_POSITION COLUMN_NAME   DATA_TYPE   FORMAT_MASK      
--     --------------- ------------- ----------- ------------------
--                   1 C0            NUMBER                       
--                   2 FIRST_NAME    VARCHAR2                     
--                   3 LAST_NAME     VARCHAR2                     
--                   4 GENDER        VARCHAR2                     
--                   5 COUNTRY       VARCHAR2                     
--                   6 AGE           NUMBER                       
--                   7 DATE_         DATE        DD"/"MM"/"YYYY   
--                   8 ID            NUMBER                       
-- 
--==================================================================================================================
function get_columns(
    p_profile             in clob ) return wwv_flow_t_parser_columns pipelined;

end wwv_flow_data_parser;
/
sho err
