set define '^' verify off
prompt ...wwv_flow_exec_web_src_utils.sql
create or replace package wwv_flow_exec_web_src_utils as
--------------------------------------------------------------------------------
--
--  Copyright (c) Oracle Corporation 1999 - 2018. All Rights Reserved.
--
--    NAME
--      wwv_flow_exec_web_src.sql
--
--    DESCRIPTION
--      Utility functions for Web Source Modules implementation
--
--
--    MODIFIED   (MM/DD/YYYY)
--    cczarski    11/29/2018 - created as part of Read/Write support for REST and REST Enabled SQL and Form support (feature #2374,#2376)
-- 
--------------------------------------------------------------------------------

c_content_type_json        constant varchar2(16) := 'application/json';


--======================================================================================================================
-- converts DATE, TIMESTAMP, TIMESTAMP_TZ values to a JSON date string in zulu timezone (YYYY-MM-DD"T"H24:MI:SS"Z")
-- Does time zone conversions as needed.
--
-- PARAMETERS:
--    * p_date                DATE value to be converted
--    * p_timestamp           TIMESTAMP value to be converted
--    * p_timestamp_ltz       TIMESTAMP WITH LOCAL TIME ZONE value to be converted
--    * p_timestamp_tz        TIMESTAMP WITH TIME ZONE value to be converted
--
--    * p_has_time_zone       if false, the passed in DATE or TIMESTAMP values are treated as they were in Zulu timezone
--                            if true, the passed in values will be converted to Zulu beforehand
--
-- RETURNS
--   The (possibly) converted value in JSON ISO 8601 syntax
--======================================================================================================================
function convert_to_json_zulu( 
    p_timestamp     in timestamp, 
    p_has_time_zone in boolean default true ) return varchar2;

function convert_to_json_zulu( 
    p_timestamp_tz  in timestamp with time zone, 
    p_has_time_zone in boolean default true ) return varchar2;

function convert_to_json_zulu( 
    p_timestamp_ltz in timestamp with local time zone, 
    p_has_time_zone in boolean default true ) return varchar2;

function convert_to_json_zulu( 
    p_date          in date, 
    p_has_time_zone in boolean default true ) return varchar2;


--======================================================================================================================
-- converts an interval day to second to ISO-8601 Duration
--
-- PARAMETERS 
--   * P_VALUE    INTERVAL DAY TO SECOND value to convert to ISO-8601
--
-- RETURNS:
--     ISO-8601 representation of the value passed in
--
--======================================================================================================================
function dsinterval_to_iso( 
    p_value in dsinterval_unconstrained ) return varchar2;

--======================================================================================================================
-- converts an interval year to month to ISO-8601 Duration
--
-- PARAMETERS 
--   * P_VALUE    INTERVAL YEAR TO MONTH value to convert to ISO-8601
--
-- RETURNS:
--     ISO-8601 representation of the value passed in
--
--======================================================================================================================
function yminterval_to_iso( 
    p_value in yminterval_unconstrained ) return varchar2;

--======================================================================================================================
-- return the value of a HTTP response header after WWV_FLOW_WEBSERVICES_API.MAKE_REST_REQUEST has been executed.
-- Returns NULL if the requested header does not exist.
-- 
-- PARAMETERS
--    * p_header_name        case-insenstive response header name
--
-- RETURNS
--    the response header value, NULL if it does not exist
--
--======================================================================================================================
function get_response_header_value( 
    p_header_name in varchar2 ) return varchar2;

--======================================================================================================================
-- adds a HTTP request header to wwv_flow_webservices_api.g_request_headers. Overwrites, if a header with this
-- name already exists.
--
-- PARAMETERS
--    * p_header_name        case-insenstive response header name
--    * p_header_value       value to set
--
--======================================================================================================================
procedure set_request_header( 
    p_header_name  in varchar2, 
    p_header_value in varchar2 );

--======================================================================================================================
-- returns true when an HTTP request header is already present in wwv_flow_webservices_api.g_request_headers
--
-- PARAMETERS
--    * p_header_name        case-insenstive response header name
--
--======================================================================================================================
function is_request_header_set( 
    p_header_name in varchar2 ) return boolean;

--======================================================================================================================
-- generates the response parsing SQL (including post-processing clauses) 
--
-- PARAMETERS:
--    * p_post_processing_type      post-processing type (SQL, WHERE, PLSQL_RETURNING_SQL, or NULL for no post-processing)
--    * p_post_processing_sql       post-processing SQL or PL/SQL code
--    * p_post_processing_where     post-processing WHERE clause
--    * p_post_processing_orderby   post-processing ORDER BY clause
--    * p_data_profile_id           ID of the data profile to use
--    * p_columns                   Return columns - pass emory arry to return all columns
--    * p_array_count               JSON document count to generate SQL for. Use this to parse multiple XML or JSON
--                                  responses in one call
--    * p_use_single_row_selector   use the "single row selector" defined in the data profile. The "normal" row selector
--                                  selects an array of rows, the "single row selector" selects a single row when no array
--                                  is present.
-- 
--    * p_parse_as_schema_override  parse PL/SQL function to return the SQL query as a specific schema
--
-- RETURNS:
--    The SQL Query as an instance of wwv_flow_global.vc_arr2
--======================================================================================================================
procedure generate_data_source_sql( 
    p_data_source_sql         out wwv_flow_global.vc_arr2,
    --
    p_post_processing_type     in wwv_flow_exec.t_source_post_processing,
    p_post_processing_sql      in varchar2,
    p_post_processing_where    in varchar2,
    p_post_processing_orderby  in varchar2,
    --
    p_columns                  in wwv_flow_exec_api.t_columns,
    p_data_profile_id          in number,
    p_use_single_row_selector  in boolean     default false,
    --
    p_array_count              in pls_integer default null,
    --
    p_parse_as_schema_override in varchar2    default null );

--======================================================================================================================
-- parse a given REST response which represents a single row. Typically used to parse the JSON response of
-- a POST or PUT REST operation to get the returning values.
-- 
-- PARAMETERS
--    * p_context             DML context object
--    * p_parse_context       context object to use for response parsing
--    * p_response_data       response data to be parsed
--    * p_operation           Web Source operation to get the data profile from
--
--======================================================================================================================
procedure parse_single_row_response(
    p_parse_context  in out nocopy wwv_flow_exec.t_context,
    p_context        in            wwv_flow_exec.t_context,
    --
    p_response_data  in            clob,
    --
    p_operation      in            wwv_flow_exec_web_src.t_web_source_operation );

--======================================================================================================================
-- builds a request body for a Web Source DML request. If a request body template is set, then #COLUMN# placeholders
-- will be replaced by the DML context column values. In this case, the request body can be any data format.
-- If no request body template is set, the function builds a JSON with the following structure:
-- {
--    "{column1-name}": "{column1-value}",
--    "{column2-name}": "{column2-value}",
--    :
-- }
-- 
-- PARAMETERS
--    * p_context               DML context object
--    * p_web_source_operation  Web Source operation to get the data profile from
--    * p_col_idx               Column Index of p_context.dml_column_values
--    * p_row_idx               Row Index of p_context.dml_column_values
--
-- RETURNS
--    a CLOB containing the request body
--
--======================================================================================================================
procedure build_dml_request_body(
    p_context               in            wwv_flow_exec.t_context,
    p_web_source_operation  in            wwv_flow_exec_web_src.t_web_source_operation,
    p_row_idx               in            pls_integer,
    --
    p_body                  in out nocopy clob );

--==============================================================================
-- convert a wwv_flow_exec_api.t_filters array to the external filter syntax 
-- for the given Web Source Module ID.
--
-- PARAMETERS
--    * p_web_src_module_id     ID of the Web Source Module
--    * p_filters               filters array
--    * p_order_bys             order by array
--
-- RETURNS
--    p_web_src_module_type     type of the web source module
--    p_external_filter         external filter syntax
--    p_external_order_by       external order_by syntax
--
--==============================================================================
procedure get_external_filters_orderbys(
    p_web_src_module_id     in number,
    p_filters               in wwv_flow_exec_api.t_filters,
    p_order_bys             in wwv_flow_exec_api.t_order_bys,
    --
    p_external_order_by    out varchar2,
    p_external_filter      out varchar2 );

end wwv_flow_exec_web_src_utils;
/
show err

set define '^'

