set define '^' verify off
prompt ...wwv_flow_lov
create or replace package wwv_flow_lov as
--------------------------------------------------------------------------------
--
--  Copyright (c) Oracle Corporation 2009 - 2019. All Rights Reserved.
--
--    NAME
--      wwv_flow_lov.sql
--
--    DESCRIPTION
--      This package is responsible for handling functionality related to List of Values.
--
--    MODIFIED   (MM/DD/YYYY)
--    cczarski    04/04/2019 - Created
--
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
-- Public type definitions
--------------------------------------------------------------------------------

subtype t_order_by is pls_integer range 1 .. 2;

/* move from wwv_flow_plugin_api.sql 
type t_lov_column is record (
    id                      wwv_flow_list_of_values_cols.id%type,
    column_name             wwv_flow_list_of_values_cols.query_column_name%type,
    heading                 wwv_flow_list_of_values_cols.heading%type,
    display_sequence        wwv_flow_list_of_values_cols.display_sequence%type,
    format_mask             wwv_flow_list_of_values_cols.format_mask%type,
    data_type               wwv_flow_exec_api.t_data_type,
    is_visible              boolean,
    is_searchable           boolean );

type t_lov_columns is table of t_lov_column index by pls_integer;
*/

type t_lov is record (
    lov_id                  number,
    lov_name                varchar2(255),
    -- 
    location                wwv_flow_exec_api.t_location,
    source_type             varchar2(23),       --todo constant
    lov_query               varchar2(32767),
    --
    query_meta_data         wwv_flow_exec.t_query_meta_data,
    --
    return_column           wwv_flow_global.t_dbms_id,
    display_column          wwv_flow_global.t_dbms_id,
    icon_column             wwv_flow_global.t_dbms_id,
    group_column            wwv_flow_global.t_dbms_id,
    group_sort_direction    varchar2(16),
    default_sort_column     wwv_flow_global.t_dbms_id,
    default_sort_direction  varchar2(16),
    oracle_text_column      wwv_flow_global.t_dbms_id,
    --
    use_template            boolean,
    display_values          varchar2(32767),
    return_values           varchar2(32767),
    delimiter               varchar2(32676),
    can_cache_data          boolean,
    lov_data                wwv_flow_t_temp_lov_data,
    order_by                t_order_by,
    --
    lov_columns             wwv_flow_plugin_api.t_lov_columns );

--------------------------------------------------------------------------------
-- Public constant definitions
--------------------------------------------------------------------------------
c_order_by_seq          constant t_order_by := 1; -- insert_order
c_order_by_display      constant t_order_by := 2; -- disp
c_start_of_group_column constant varchar2(22) := 'APEX$START_OF_GROUP';
c_end_of_group_column   constant varchar2(22) := 'APEX$END_OF_GROUP';

--------------------------------------------------------------------------------
-- Global variables
--------------------------------------------------------------------------------

--======================================================================================================================
-- returns query meta data for last LOV fetched with get_shared_lov.
--======================================================================================================================
function current_lov_query_meta_data return wwv_flow_exec.t_query_meta_data;

--======================================================================================================================
-- looks up a shared LOV ID by name.
--
-- PARAMETERS
--     p_lov_name         Name of the shared LOV
--     p_application_id   Application ID
--
-- RETURNS
--     the shared LOV ID
--======================================================================================================================
function get_shared_lov_id( 
    p_lov_name       in varchar2,
    p_case_sensitive in boolean default true,
    p_application_id in number  default null ) return number;

--======================================================================================================================
-- returns the SQL query returning LOV data with Display and return columns. Either a shared LOV ID or an 
-- inline LOV definition must be passed in. The function will return either the LOV SQL query or a query selecting
-- from a table function for static and remote LOVs.
--
-- PARAMETERS
--     p_inline_lov       Inline LOV definition to use
--     p_shared_lov_id    ID of a shared LOV
--
--     p_return_value     generate a SQL query selecting only a row for this return value
--     p_no_order_by      suppress additional ORDER BY clauses
--     p_use_template     whether to use a template for display values
-- 
-- RETURNS
--     a SQL query to be embedded into region SQL or to execute in order to get LOV values.
--======================================================================================================================
function get_lov_sql_query (
    p_inline_lov         in varchar2,
    p_shared_lov_id      in number,
    --
    p_target_location    in wwv_flow_exec_api.t_location default null,
    --
    p_return_value       in varchar2                     default null,
    p_no_order_by        in boolean                      default false,
    p_use_template       in boolean                      default false ) return varchar2;

------------------------------------------------------------------------------------------------------------------------
procedure get_lov_sql_query (
    p_inline_lov         in varchar2,
    p_shared_lov_id      in number,
    --
    p_target_location    in wwv_flow_exec_api.t_location default null,
    --
    p_return_value       in varchar2                     default null,
    p_no_order_by        in boolean                      default false,
    p_use_template       in boolean                      default false,
    --
    p_sql_query         out varchar2,
    p_display_column    out varchar2,
    p_return_column     out varchar2 );



--======================================================================================================================
-- fetches and caches a shared LOV. Also prepares the Query Meta Data globals with data source
-- meta data for the WWV_FLOW_EXEC package. A subsequent OPEN_QUERY_CONTEXT invocation with 
-- p_plugin_context => wwv_flow_lov.current_lov_query_meta_data will use the data source settings prepared with
-- this call. ID or Name can be used to identify the LOV
--
-- PARAMETERS
--     p_lov_id           ID of the Shared LOV
--     p_lov_name         Name of the Shared LOV
-- 
-- RETURNS
--     LOV meta data as an instance of the T_LOV type.
--======================================================================================================================
function get_shared_lov( 
    p_lov_id            in number,
    p_lov_name          in varchar2 default null,
    p_application_id    in number   default null ) 
    return t_lov;

--======================================================================================================================
-- returns static LOV data as instance of the wwv_flow_t_temp_lov_data table type. This function is used
-- by the SQL query generated by get_lov_sql_query for a static LOV.
--
-- PARAMETERS
--     p_lov_cache_idx          Cache Index of the LOV
-- 
-- RETURNS
--     static LOV data as instance of the wwv_flow_t_temp_lov_data table type        
--======================================================================================================================
function get_static_lov_data ( p_lov_cache_idx in pls_integer ) return wwv_flow_t_temp_lov_data;

--======================================================================================================================
-- gets the display value for a given LOV and a given return value.
--
-- PARAMETERS
--     p_inline_lov       Inline LOV definition to use
--     p_shared_lov_id    ID of a shared LOV
--
--     p_return_value     return value to look up the display value for.
--     
-- 
-- RETURNS
--     LOV Display value for the givem return value
--======================================================================================================================
function get_lov_display_value(
    p_inline_lov    in varchar2,
    p_shared_lov_id in number,
    --
    p_return_value  in varchar2 ) return varchar2 ;

--======================================================================================================================
-- builds an instance of the htmldb_item.lov_table type containing LOV data for usage within wwv_flow_item.
--
-- PARAMETERS
--     p_inline_lov       Inline LOV definition to use
--     p_shared_lov_id    ID of a shared LOV
-- 
-- RETURNS
--     htmldb_item.lov_table instance containing LOV data
--======================================================================================================================
function array_from_lov(
    p_inline_lov    in varchar2,
    p_shared_lov_id in number ) return htmldb_item.lov_table;

--======================================================================================================================
-- builds an instance of the htmldb_item.lov_table type containing LOV data for usage within wwv_flow_item.
--
-- PARAMETERS
--     p_shared_lov_name  Shared LOV Name
-- 
-- RETURNS
--     htmldb_item.lov_table instance containing LOV data
--======================================================================================================================
function array_from_lov(
    p_shared_lov_name in varchar2 ) return htmldb_item.lov_table;
--======================================================================================================================
-- Sets all item LOV attributes for a given item
--
-- PARAMETERS
--     p_item           The item interface
--     p_lov_name       Shared LOV Name
--======================================================================================================================
procedure set_item_lov_attributes(
    p_item              in out nocopy wwv_flow_plugin_api.t_item,
    p_inline_lov        in varchar2 default null,
    p_shared_lov_id     in number   default null,
    p_shared_lov_name   in varchar2,
    p_target_location   in wwv_flow_exec_api.t_location default null,
    p_use_template      in boolean default false );
--======================================================================================================================
-- Checks if a value is valid for a given List of Values
--
-- PARAMETERS
--
--======================================================================================================================
function is_valid_value (
    p_inline_lov        in varchar2     default null,
    p_shared_lov_id     in number,
    p_shared_lov_name   in varchar2     default null,
    p_value             in varchar2 ) 
    return boolean;
--==============================================================================
-- Returns a query context type for the current lov.
--
-- PARAMETERS
--
--==============================================================================
procedure open_query_context (
    p_context               in out nocopy wwv_flow_exec.t_context,
    p_first_row             in pls_integer                       default null,
    p_max_rows              in pls_integer                       default null,
    --
    p_item                  in wwv_flow_plugin_api.t_item,
    p_supports_grouping     in boolean                           default false,
    p_supports_icons        in boolean                           default false,
    p_supports_add_columns  in boolean                           default false,
    -- legacy
    p_search_text           in varchar2                          default null,
    p_filters               in wwv_flow_exec_api.t_filters       default wwv_flow_exec_api.c_empty_filters,
    p_order_bys             in wwv_flow_exec_api.t_order_bys     default wwv_flow_exec_api.c_empty_order_bys,
    --
    p_do_substitutions      in boolean                           default false );
--======================================================================================================================
-- todo: doc
--
-- PARAMETERS
--
--======================================================================================================================
procedure emit_lov_json (
    p_item                  in wwv_flow_plugin_api.t_item,         
    p_supports_grouping     in boolean  default false,
    p_supports_icons        in boolean  default false,
    p_escape                in boolean,
    p_do_substitutions      in boolean  default false );
--
--
end wwv_flow_lov;
/
show errors
