set define '^' verify off
prompt ...wwv_flow_public_key_util.sql
create or replace package wwv_flow_public_key_util as
--------------------------------------------------------------------------------
--
-- Copyright (c) Oracle Corporation 1999 - 2019. All Rights Reserved.
--
-- NAME
--   wwv_flow_public_key_util.sql
--
-- DESCRIPTION
--   Utilities for public key encryption, decrypting, signing and verification.
--
--   Implementation is in Java, but should be replaceable by ztpk_% kernel
--   functions.
--
-- GENERATE KEYS
--   To create compatible public and private keys in the correct format based on
--   an OpenSSH key, use the commands below:
--
--   1. Create 2048 bit private RSA key
--
--      $ openssl genpkey -algorithm RSA -pkeyopt rsa_keygen_bits:2048 -out key.pem
--
--   2. Extract public key
--
--      $ openssl rsa -in key.pem -pubout 2>/dev/null | grep -v -- --
--
--   3. Convert private key to PKCS8 format
--
--      $ cat key.pem | openssl pkcs8 -topk8 -nocrypt | grep -v -- --
--
-- GENERATE CERTIFICATE
--   To generate a certificate and private key with OpenSSL, use the commands
--   below:
--
--     $ openssl req -newkey rsa:2048 -nodes -x509 -days 365 -out cert.pem -keyout key.pem
--
-- RUNTIME DEPLOYMENT: YES
-- PUBLIC:             NO
--
-- MODIFIED   (MM/DD/YYYY)
--   cneumuel  05/20/2019 - Created
--
--------------------------------------------------------------------------------

--==============================================================================
-- Algorithm types for encryption/decription
--==============================================================================
subtype t_enc_algorithm is binary_integer range 0 .. 1;
c_rsa                   constant t_enc_algorithm default 0;
c_rsa_oaep_mgf1p        constant t_enc_algorithm default 1;

--==============================================================================
-- Algorithm types for signing and verification
--==============================================================================
subtype t_sig_algorithm is binary_integer range 0 .. 3;
c_sha1_rsa              constant t_sig_algorithm default 0;
c_sha256_rsa            constant t_sig_algorithm default 1;
c_sha384_rsa            constant t_sig_algorithm default 2;
c_sha512_rsa            constant t_sig_algorithm default 3;

--==============================================================================
-- Base64-encodes the input and removes all white space.
--==============================================================================
function b64_enc (
    p_src in raw )
    return varchar2;

--==============================================================================
-- Convert textual certificate/key to raw.
-- - Removes "---" lines (e.g. "-----BEGIN PRIVATE KEY-----")
-- - Removes all white space and line breaks
--==============================================================================
function b64_dec (
    p_src in varchar2 )
    return raw;

--==============================================================================
-- Given a x.509 certificate, compute the public key and algorithm. If
-- p_validate is true, validate the certificate.
--==============================================================================
procedure parse_certificate (
    p_src        in  varchar2,
    p_validate   in  boolean,
    p_algorithm  out t_sig_algorithm,
    p_public_key out varchar2 );

--==============================================================================
function encrypt (
    p_src           in raw,
    p_public_key    in varchar2,
    p_key_algorithm in t_enc_algorithm default c_rsa,
    p_enc_algorithm in t_enc_algorithm default c_rsa )
    return raw;

--==============================================================================
function decrypt (
    p_src           in raw,
    p_private_key   in varchar2,
    p_key_algorithm in t_enc_algorithm default c_rsa,
    p_enc_algorithm in t_enc_algorithm default c_rsa )
    return raw;

--==============================================================================
function sign (
    p_src           in raw,
    p_private_key   in varchar2,
    p_key_algorithm in t_enc_algorithm default c_rsa,
    p_sig_algorithm in t_sig_algorithm default c_sha256_rsa )
    return raw;

--==============================================================================
function verify (
    p_src           in raw,
    p_signature     in raw,
    p_public_key    in varchar2,
    p_key_algorithm in t_enc_algorithm default c_rsa,
    p_sig_algorithm in t_sig_algorithm default c_sha256_rsa )
    return boolean;

end wwv_flow_public_key_util;
/
show err
