set define off verify off
prompt ...wwv_flow_region_api
create or replace package wwv_flow_region_api as
--------------------------------------------------------------------------------
--
--  Copyright (c) Oracle Corporation 2012 - 2014. All Rights Reserved.
--
--    NAME
--      wwv_flow_region_api.sql
--
--    DESCRIPTION
--      This package is the public API for handling regions.
--
--    MODIFIED   (MM/DD/YYYY)
--      pawolf    04/12/2012 - Created
--      cneumuel  04/28/2014 - In purge_cache: added p_current_session_only (feature #1401)
--
--------------------------------------------------------------------------------

--==============================================================================
-- Returns TRUE if the current region is rendered read only and FALSE if not.
-- If the function is called from a context where no region is currently
-- processed it will return NULL. This function can for example be used in
-- conditions of a region or it's underlying items and buttons.
--
-- EXAMPLE
--   return apex_region.is_read_only;
--==============================================================================
function is_read_only return boolean;

--==============================================================================
-- Purge the region cache of the specified application, page and region.
--
-- PARAMETERS
-- * p_application_id       id of the application where the region caches should
--                          be purged. defaults to the current application.
-- * p_page_id              id of the page where the region caches should be
--                          purged. if no value is specified (which is the
--                          default) all regions of the application will be
--                          purged.
-- * p_region_id            id of a specific region on a page. if no value is
--                          specified all regions of the specified page will be
--                          purged.
-- * p_current_session_only specify true if you only want to purge entries that
--                          where saved for the current session. defaults to
--                          false.
--
-- EXAMPLE
--   Purge session specific region cache for the whole application.
--
--   begin
--       apex_region.purge_cache (
--           p_current_session_only => true );
--   end;
--==============================================================================
procedure purge_cache (
    p_application_id       in number default wwv_flow.g_flow_id,
    p_page_id              in number default null,
    p_region_id            in number default null,
    p_current_session_only in boolean default false );

--==============================================================================
-- returns an APEX_EXEC query context returning current region data. Only 
-- native regions are supported at this time.
--
-- PARAMETERS
-- * p_page_id               id of the page where the region is on
-- * p_region_id             id of a specific region to open the query context for
-- * p_component_id          region component ID to use. For Interactive Reports and Interactive Grids this is the saved 
--                           report ID within the current application page, for JET charts use the chart series ID. 
-- * p_view_mode             The view type available for the report. The values can be APEX_IR.C_VIEW_REPORT, 
--                           APEX_IR.C_VIEW_GROUPBY or APEX_IR.C_VIEW_PIVOT. If p_view is null, it gets the view 
--                           currently used by the report. If p_view passed which doesn't 
--                           exist for the current report, an error is raised.
--
-- * p_additional_filters    additional filters to apply to the context
--
-- * p_first_row             row index to start fetching at. Defaults to 1
-- * p_max_rows              maximum amount of rows to get. Default unlimited. 
-- * p_total_row_count       whether to retrieve the total row count. Defaults to false.
-- * p_total_row_count_limit Upper limit of rows to process the query on. This applies 
--                           to Interactive Report aggregations or ordering. 
--                           Default is no limit
--
-- EXAMPLE
--   Get the query context for a given saved interactive report on page 1 and print
--   the data out as JSON.
--
--   declare
--       l_context apex_exec.t_context;
--   begin
--       l_context := apex_region.open_query_context (
--                        p_page_id      => 1,
--                        p_component_id => 2505704029884282,
--                        p_report_id    => 880629800374638220);
--
--       apex_json.open_object;
--       apex_json.write_context( 'data', l_context );
--       apex_json.close_object;
--   end;
-- 
--==============================================================================
function open_query_context(
    p_page_id               in number,
    p_region_id             in number,
    p_component_id          in number   default null,
    p_view_mode             in varchar2 default null,
    --
    p_additional_filters    in wwv_flow_exec_api.t_filters default wwv_flow_exec_api.c_empty_filters,
    --
    p_first_row             in number   default null, 
    p_max_rows              in number   default null,
    p_total_row_count       in boolean  default false,
    p_total_row_count_limit in number   default null ) 
    return wwv_flow_exec_api.t_context;

end wwv_flow_region_api;
/
show errors
set define '^'
