/**
 * @license
 * Copyright (c) 2014, 2019, Oracle and/or its affiliates.
 * The Universal Permissive License (UPL), Version 1.0
 */
"use strict";
(function(){
ES6Promise['polyfill']();
// Copyright (c) 2011, 2013, Oracle and/or its affiliates.
// All rights reserved.

/* jslint browser: true*/

/**
 * @namespace
 * @name oj.Logger
 * @hideconstructor
 * @ojtsmodule
 * @since 1.0.0
 *
 * @classdesc
 * <h3>JET Logger</h3>
 *
 * <p>Logger object writes into the native browser console or a custom writer, if a custom writer is set as an option.
 * To use a custom writer, implement the following writer methods: log(), info(), warn(), error()
 *
 * <p>When any of the logging methods is called, it compares the requested log level with the value of a log level option
 * and logs the message if the log level is sufficient.
 *
 * <p>If the logging options are changed at a later point, the Logger will use the modified options for the subsequent log operations.
 *
 * <p>All the logging methods support string formatting, accept variable number of arguments and accept a function as a parameter.
 * When a callback function is specified as a parameter the function will be called if the log level is sufficient.
 *
 * <h3> Usage : </h3>
 * <pre class="prettyprint">
 * <code>
 * //optional calls, see defaults
 * oj.Logger.option("level",  oj.Logger.LEVEL_INFO);
 * oj.Logger.option("writer",  customWriter);  //an object that implements the following methods: log(), info(), warn(), error()
 *
 * // logging a message
 * oj.Logger.info("My log level is %d", oj.Logger.option("level"));  // string formatting
 * oj.Logger.warn("Beware of bugs", "in the above code");            // multiple parameters
 *
 * // using a callback function as a parameter
 * oj.Logger.info(function(){
 *    var foo = "This ";
 *    var bar = "is ";
 *    var zing = "a function";
 *    return foo + bar + zing;
 * });
 * </code></pre>
 *
 * @desc oj.Logger cannot be instantiated
 * @export
 */
var Logger = {};
/**
 * Log level none
 * @const
 * @export
 * @type {number}
 * @memberof oj.Logger
 * @alias LEVEL_NONE
 */
Logger.LEVEL_NONE = 0;
/**
 * Log level error
 * @const
 * @type {number}
 * @export
 * @memberof oj.Logger
 * @alias LEVEL_ERROR
 */
Logger.LEVEL_ERROR = 1;
/**
 * Log level warning
 * @const
 * @type {number}
 * @export
 * @memberof oj.Logger
 * @alias LEVEL_WARN
 */
Logger.LEVEL_WARN = 2;
/**
 * Log level info
 * @const
 * @type {number}
 * @export
 * @memberof oj.Logger
 * @alias LEVEL_INFO
 */
Logger.LEVEL_INFO = 3;
/**
 * Log level - general message
 * @const
 * @type {number}
 * @export
 * @memberof oj.Logger
 * @alias LEVEL_LOG
 */
Logger.LEVEL_LOG = 4;

/* private constants*/
Logger._METHOD_ERROR = 'error';
Logger._METHOD_WARN = 'warn';
Logger._METHOD_INFO = 'info';
Logger._METHOD_LOG = 'log';
Logger._defaultOptions = { level: Logger.LEVEL_ERROR, writer: null };
Logger._options = Logger._defaultOptions;


/* public members*/
/**
 * Writes an error message.
 * @param {any=} message A function that returns the message to be logged, a string containing zero or more substitution strings, or an object to be logged.
 *                      See examples in the overview section above.
 * @param {...any} optionalParams Objects with which to replace substitution strings within messages or simply additional objects to be logged.
 * @return {void}
 * @export
 * @memberof oj.Logger
 * @method error
 * @since 1.0.0
 * @ojsignature {target: "Type", for: "optionalParams", value: "any[]"}
 */
// eslint-disable-next-line no-unused-vars
Logger.error = function (message, optionalParams) {
  Logger._write(Logger.LEVEL_ERROR, Logger._METHOD_ERROR, arguments);
};

/**
 * Writes an informational  message.
 * @param {any=} message A function that returns the message to be logged, a string containing zero or more substitution strings, or an object to be logged.
 *                      See examples in the overview section above.
 * @param {...any} optionalParams Objects with which to replace substitution strings within messages or simply additional objects to be logged.
 * @return {void}
 * @export
 * @memberof oj.Logger
 * @method info
 * @since 1.0.0
 * @ojsignature {target: "Type", for: "optionalParams", value: "any[]"}
 */
// eslint-disable-next-line no-unused-vars
Logger.info = function (message, optionalParams) {
  Logger._write(Logger.LEVEL_INFO, Logger._METHOD_INFO, arguments);
};

/**
 * Writes a warning message.
 * @param {any=} message A function that returns the message to be logged, a string containing zero or more substitution strings, or an object to be logged.
 *                      See examples in the overview section above.
 * @param {...any} optionalParams Objects with which to replace substitution strings within messages or simply additional objects to be logged.
 * @export
 * @return {void}
 * @memberof oj.Logger
 * @method warn
 * @since 1.0.0
 * @ojsignature {target: "Type", for: "optionalParams", value: "any[]"}
 */
// eslint-disable-next-line no-unused-vars
Logger.warn = function (message, optionalParams) {
  Logger._write(Logger.LEVEL_WARN, Logger._METHOD_WARN, arguments);
};

/**
 * Writes a general message.
 * @param {any=} message A function that returns the message to be logged, a string containing zero or more substitution strings, or an object to be logged.
 *                      See examples in the overview section above.
 * @param {...any} optionalParams Objects with which to replace substitution strings within messages or simply additional objects to be logged.
 * @return {void}
 * @export
 * @memberof oj.Logger
 * @method log
 * @since 1.0.0
 * @ojsignature {target: "Type", for: "optionalParams", value: "any[]"}
 */
// eslint-disable-next-line no-unused-vars
Logger.log = function (message, optionalParams) {
  Logger._write(Logger.LEVEL_LOG, Logger._METHOD_LOG, arguments);
};

/**
 * Method for setting and getting logger option/options
 * <p>Sets/gets logger configuration - level and/or writer. Accepts variable number of arguments.
 * <p><h5>Defaults:</h5>
 * Default level: oj.Logger.LEVEL_ERROR<br/>
 * Default writer: null; writes to the console
 * <p><h5>Usages:</h5>
 * <i>oj.Logger.option(optionName)</i> gets the value associated the the specified optionName<br/>
 * <i>oj.Logger.option()</i> gets an object containing key/value pairs representing the logger options hash<br/>
 * <i>oj.Logger.option(optionName, value)</i> sets  the option value associated with optionName<br/>
 * <i>oj.Logger.option(options)</i> sets  one or more options for the logger
 *
 * @example <caption>Overriding default options</caption>
 * oj.Logger.option("level",  oj.Logger.LEVEL_INFO);
 * oj.Logger.option("writer",  customWriter);  //an object that implements the following methods: log(), info(), warn(), error()
 *
 * @param {Object|string} [key]
 * @param {any} [value]
 * @return {any}
 * @export
 * @memberof oj.Logger
 * @method option
 * @since 1.0.0
 * @ojsignature {target: "Type", for: "key", value: "'level'|'writer'|{level?: any, writer?: any}"}
 */
Logger.option = function (key, value) {
  // getters
  var ret = {};
  var i;
  var keys;

  if (arguments.length === 0) {
    keys = Object.keys(Logger._options);
    for (i = 0; i < keys.length; i++) {
      ret[keys[i]] = Logger._options[keys[i]];
    }
    return ret;
  }
  if (typeof key === 'string' && value === undefined) {
    return Logger._options[key] === undefined ? null : Logger._options[key];
  }

  // setters
  if (typeof key === 'string') {
    Logger._options[key] = value;
  } else { // case when all options are set in one call
    var options = key;
    keys = Object.keys(options);
    for (i = 0; i < keys.length; i++) {
      Logger.option(keys[i], options[keys[i]]);
    }
  }

  return undefined;
};

/* private members*/
/*
 * Helper method - calls a specified method on the available writer (console or custom)
 * if the logging level is sufficient
 */
Logger._write = function (level, method, args) {
  if (Logger.option('level') < level) {
    return;
  }

  var writer = Logger._getWriter();
  if (writer != null) {
    if (args.length === 1 && (args[0] instanceof Function)) {
      var msg = args[0]();
      // eslint-disable-next-line no-param-reassign
      args = [msg];
    }
    if (writer[method] && writer[method].apply) {
      writer[method].apply(writer, args);
    } else if (writer[method]) {
      writer[method] = Function.prototype.bind.call(writer[method], writer);
      Logger._write(level, method, args);
    }
  }
};

/*
 * Helper method - returns available writer (console or custom)
 */
Logger._getWriter = function () {
  var writer = null;
  if (Logger.option('writer')) {
    writer = Logger.option('writer');
  } else if (typeof window !== 'undefined' && window.console !== undefined) {
    writer = window.console;
  }
  return writer;
};

/**
 * Copyright (c) 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */

/* jslint browser: true*/
/* global self:true */

/**
 * Defines the oj namespace
 */

/**
 * @private
 */
var _scope = {};

//  - check if the window object is available
// Note that the 'typeof' check  is required
if (typeof window !== 'undefined') {
  _scope = window;
  // eslint-disable-next-line no-restricted-globals
} else if (typeof self !== 'undefined') {
  // eslint-disable-next-line no-restricted-globals
  _scope = self;
}

/**
 * @private
 */
var _oldVal = _scope.oj;

// eslint-disable-next-line no-unused-vars
var oj = {
  /**
   * @global
   * @member {string} version JET version numberr
   */
  version: '7.2.0',
  /**
   * @global
   * @member {string} revision JET source code revision number
   */
  revision: '2019-09-12_18-05-50',

  // This function is only meant to be used outside the library, so quoting the name
  // to avoid renaming is appropriate
  noConflict: function () {
    _scope.oj = _oldVal;
  }
};

_scope.oj = oj;

/*
** Copyright (c) 2008, 2011, Oracle and/or its affiliates. All rights reserved.
*/

/* global _scope:false */
/* jslint browser: true*/

/**
 * Assertion utilities.
 * The container is expected to have already initialized the oj.Assert Object before this
 * code is executed and initialized the oj.Assert.DEBUG flag/
 * @class
 * @export
 * @ignore
 */
oj.Assert = {};

/**
 * @private
 */
var _DEBUG = 'DEBUG';

/**
 * Forces DEBUG to be set to true
 * @export
 * @memberof oj.Assert
 */
oj.Assert.forceDebug = function () {
  oj.Assert[_DEBUG] = true;
};

/**
 * Forces DEBUG to be set to false
 * @export
 * @memberof oj.Assert
 */
oj.Assert.clearDebug = function () {
  oj.Assert[_DEBUG] = false;
};

/**
 * Determines whether oj.Assert is running in debug mode
 * @return {boolean} true for debug mode, false otherwise
 * @export
 * @memberof oj.Assert
 */
oj.Assert.isDebug = function () {
  return oj.Assert[_DEBUG] === true;
};


/**
 * Asserts that a condition is true.  If the condition does not
 * evaluate to true, an exception is thrown with the optional message
 * and reason
 * @param {boolean} condition condition to test
 * @param {string=} message message to display
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assert = function (
  condition,
  message
  ) {
  if (oj.Assert[_DEBUG] && !condition) {
    var myMessage = message || '';

    if (arguments.length > 2) {
      myMessage += '(';
      for (var i = 2; i < arguments.length; i += 1) {
        myMessage += arguments[i];
      }
      myMessage += ')';
    }
    oj.Assert.assertionFailed(myMessage, 1);
  }
};

/**
 * Convenience function for asserting when an abstact function is called
 * @export
 * @memberof oj.Assert
 */
oj.Assert.failedInAbstractFunction = function () {
  if (oj.Assert[_DEBUG]) {
    oj.Assert.assertionFailed('Abstract function called', 1);
  }
};


/**
 * Asserts that the the target object has the same prototype as the example
 * type
 * @param {Object} target description
 * @param {Function} theConstructor
 * @param {string=} reason
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertPrototype = function (
  target,
  theConstructor,
  reason
  ) {
  if (oj.Assert[_DEBUG]) {
    var thePrototype = theConstructor.prototype;

    if (target != null) {
      oj.Assert.assertType(theConstructor, 'function', null, 1, false);

      var isPrototypeOf = Object.prototype.isPrototypeOf;
      if (!isPrototypeOf.call(thePrototype, target)) {
        oj.Assert.assertionFailed("object '" + target + "' doesn't match prototype "
                                  + thePrototype,
                                  1,
                                  reason);
      }
    } else {
      oj.Assert.assertionFailed("null object doesn't match prototype " + thePrototype, 1, reason);
    }
  }
};

/**
 * Asserts that the the target object has the same prototype as the example
 * type or is null.
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertPrototypeOrNull = function (
  target,
  theConstructor,
  reason
  ) {
  if (oj.Assert[_DEBUG] && (target != null)) {
    oj.Assert.assertType(theConstructor, 'function', null, 1, false);
    var thePrototype = theConstructor.prototype;

    var isPrototypeOf = Object.prototype.isPrototypeOf;
    if (!isPrototypeOf.call(thePrototype, target)) {
      oj.Assert.assertionFailed("object '" + target + "' doesn't match prototype "
                                + thePrototype,
                                1,
                                reason);
    }
  }
};

/**
 * Asserts that the the target object has the same prototype as the example
 * types
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertPrototypes = function (
  target,
  instanceOne,
  instanceTwo,
  reason
  ) {
  if (oj.Assert[_DEBUG]) {
    var thePrototype = instanceOne.prototype;
    var thePrototypeTwo = instanceTwo.prototype;

    var isPrototypeOf = Object.prototype.isPrototypeOf;
    if (!(isPrototypeOf.call(thePrototype, target) ||
          isPrototypeOf.call(thePrototypeTwo, target))) {
      oj.Assert.assertionFailed("object '" + target + "' doesn't match prototype "
                                + thePrototype + ' or ' + thePrototypeTwo,
                                1,
                                reason);
    }
  }
};


/**
 * Asserts that the target is a DOM Node or Null
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertDomNodeOrNull = function (target, depth) {
  if (oj.Assert[_DEBUG] && target) {
    if (target.nodeType === undefined) {
      oj.Assert.assertionFailed(target + ' is not a DOM Node', depth + 1);
    }
  }
};

/**
 * Asserts that the target is a DOM Node
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertDomNode = function (target, depth) {
  if (oj.Assert[_DEBUG]) {
    if (!target || (target.nodeType === undefined)) {
      oj.Assert.assertionFailed(target + ' is not a DOM Node', depth + 1);
    }
  }
};

/**
 * Asserts that the target is a DOM Element and optionally has the specified
 * element name
 * @param {Object} target target object
 * @param {string=} nodeName name of the element
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertDomElement = function (target, nodeName) {
  if (oj.Assert[_DEBUG]) {
    oj.Assert.assertDomNode(target, 1);

    if (target.nodeType !== 1) {
      oj.Assert.assertionFailed(target + ' is not a DOM Element', 1);
    } else if (nodeName && (target.nodeName !== nodeName)) {
      oj.Assert.assertionFailed(target + ' is not a ' + nodeName + ' Element', 1);
    }
  }
};

/**
 * Asserts that the target is a DOM Element and optionally has the specified
 * element name
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertDomElementOrNull = function (target, nodeName) {
  if (oj.Assert[_DEBUG] && (target != null)) {
    oj.Assert.assertDomNode(target, 1);

    if (target.nodeType !== 1) {
      oj.Assert.assertionFailed(target + ' is not a DOM Element', 1);
    } else if (nodeName && (target.nodeName !== nodeName)) {
      oj.Assert.assertionFailed(target + ' is not a ' + nodeName + ' Element', 1);
    }
  }
};


/**
 * Asserts that the target object has the typeof specified
 *
 * @param {Object|null|string|undefined} target
 * @param {string} type typeof type that statisfies this condition
 * @param {string|undefined|null} prefix
 * @param {number} depth stack depth to skip when printing stack traces
 * @param {boolean} nullOK true if a null value satisfies this condition
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertType = function (target, type, prefix, depth, nullOK) {
  if (oj.Assert[_DEBUG]) {
    // either the target is null and null is OK, or the target better
    // be of the correct type
    var targetType = typeof target;
    if (!(((target == null) && nullOK) || (targetType === type))) {
      var message = target + ' is not of type ' + type;

      if (prefix) {
        message = prefix + message;
      }

      if (!depth) {
        // eslint-disable-next-line no-param-reassign
        depth = 0;
      }

      oj.Assert.assertionFailed(message, depth + 1);
    }
  }
};

/**
 * Asserts that the target is an Object
 * @param {Object} target description
 * @param {string=} prefix
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertObject = function (target, prefix) {
  if (oj.Assert[_DEBUG]) {
    oj.Assert.assertType(target, 'object', prefix, 1, false);
  }
};

/**
 * Asserts that the target is an Object or null
 * @param {Object} target description
 * @param {string=} prefix
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertObjectOrNull = function (target, prefix) {
  if (oj.Assert[_DEBUG]) {
    oj.Assert.assertType(target, 'object', prefix, 1, true);
  }
};

/**
 * Asserts that the target is a non-empty String
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertNonEmptyString = function (target, prefix) {
  if (oj.Assert[_DEBUG]) {
    oj.Assert.assertType(target, 'string', prefix, 1, false);
    oj.Assert.assert(target.length > 0, 'empty string');
  }
};

/**
 * Asserts that the target is a String
 * @param target target object
 * @param {string=} prefix prefix string
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertString = function (target, prefix) {
  if (oj.Assert[_DEBUG]) {
    oj.Assert.assertType(target, 'string', prefix, 1, false);
  }
};

/**
 * Asserts that the target is a String or null
 * @param {string|null|undefined|Object} target target object
 * @param {string=} prefix prefix string
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertStringOrNull = function (target, prefix) {
  if (oj.Assert[_DEBUG]) {
    oj.Assert.assertType(target, 'string', prefix, 1, true);
  }
};

/**
 * Asserts that the target is a Function
 * @param {Object} target target object
 * @param {string=} prefix prefix string
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertFunction = function (target, prefix) {
  if (oj.Assert[_DEBUG]) {
    oj.Assert.assertType(target, 'function', prefix, 1, false);
  }
};

/**
 * Asserts that the target is a Function or null
 * @param {Object} target target object
 * @param {string=} prefix prefix
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertFunctionOrNull = function (target, prefix) {
  if (oj.Assert[_DEBUG]) {
    oj.Assert.assertType(target, 'function', prefix, 1, true);
  }
};

/**
 * Asserts that the target is a boolean
 * @param {Object} target description
 * @param {string=} prefix
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertBoolean = function (target, prefix) {
  if (oj.Assert[_DEBUG]) {
    oj.Assert.assertType(target, 'boolean', prefix, 1, false);
  }
};

/**
 * Asserts that the target is a number
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertNumber = function (target, prefix) {
  if (oj.Assert[_DEBUG]) {
    oj.Assert.assertType(target, 'number', prefix, 1, false);
  }
};

/**
 * Asserts that the target is a number or Null
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertNumberOrNull = function (target, prefix) {
  if (oj.Assert[_DEBUG]) {
    oj.Assert.assertType(target, 'number', prefix, 1, true);
  }
};


/**
 * Asserts that the target object is an Array
 * @param {Object} target target object
 * @param {string=} message optional message
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertArray = function (
  target,
  message
  ) {
  if (oj.Assert[_DEBUG]) {
    if (!Array.isArray(target)) {
      if (message === undefined) {
        // eslint-disable-next-line no-param-reassign
        message = target + ' is not an array';
      }

      oj.Assert.assertionFailed(message, 1);
    }
  }
};

/**
 * Asserts that the target object is an Array or null
 * @param {Object} target target object
 * @param {string=} message optional message
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertArrayOrNull = function (
  target,
  message) {
  if (oj.Assert[_DEBUG] && (target != null)) {
    if (!Array.isArray(target)) {
      if (message === undefined) {
        // eslint-disable-next-line no-param-reassign
        message = target + ' is not an array';
      }

      oj.Assert.assertionFailed(message, 1);
    }
  }
};


/**
 * Asserts that the target object is not either a number, or convertible to a number
 * @param {Object} target target object
 * @param {string=} message optional message
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertNonNumeric = function (
  target,
  message
  ) {
  if (oj.Assert[_DEBUG]) {
    if (!isNaN(target)) {
      if (message === undefined) {
        // eslint-disable-next-line no-param-reassign
        message = target + ' is convertible to a number';
      }

      oj.Assert.assertionFailed(message, 1);
    }
  }
};

/**
 * Asserts that the target object is either a number, or convertible to a number
 * @param {Object} target target object
 * @param {string=} message optional message
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertNumeric = function (
  target,
  message
  ) {
  if (oj.Assert[_DEBUG]) {
    if (isNaN(target)) {
      if (message === undefined) {
        // eslint-disable-next-line no-param-reassign
        message = target + ' is not convertible to a number';
      }

      oj.Assert.assertionFailed(message, 1);
    }
  }
};

/**
 * Asserts that value String is in the Set
 * @param {Object} value value to check
 * @param {Object} set set to check
 * @param {string=} message optional message
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertInSet = function (
  value,
  set,
  message) {
  if ((value == null) || (set[value.toString()] === undefined)) {
    if (message === undefined) {
      var keyString = ' is not in set: {';

      var keys = Object.keys(set);
      for (var k = 0; k < keys.length; k++) {
        var key = keys[k];
        keyString += key;
        keyString += ',';
      }

      keyString += '}';

      // eslint-disable-next-line no-param-reassign
      message = value + keyString;
    }

    oj.Assert.assertionFailed(message, 1);
  }
};

/**
 * Base assertion failure support that supports specifying the stack skipping
 * level
 * @param {string} message Message to display
 * @param {number} skipLevel assertion level
 * @param {string=} reason reason to display
 * @export
 * @memberof oj.Assert
 */
oj.Assert.assertionFailed = function (
  message,
  skipLevel,
  reason) {
  if (!skipLevel) {
    // eslint-disable-next-line no-param-reassign
    skipLevel = 0;
  }

  var errorMessage = 'Assertion';

  if (reason) {
    errorMessage += ' (' + reason + ')';
  }

  errorMessage += ' failed: ';

  if (message !== undefined) {
    errorMessage += message;
  }

  var error = new Error(errorMessage);


  throw error;
};

/**
 * @private
 * @memberof oj.Assert
 */
var _assertSetting = _scope.__oj_Assert_DEBUG;

if (_assertSetting !== undefined) {
  oj.Assert[_DEBUG] = _assertSetting;
}

/*
** Copyright (c) 2004, 2012, Oracle and/or its affiliates. All rights reserved.
*/

/**
 * Utilities for working with collections
 * @export
 */
oj.CollectionUtils = {};


/**
 * Copies all of the properties of source into the target and returns the target
 *
 * @param {Object} target - target collection
 * @param {Object} source - source collection
 * @param {function(string)=} keyConverter a callback for converting the key
 * @param {boolean=} recurse - true if this method should recurse into plain Javascript object properties
 * @param {number=} maxRecursionDepth - the maximum depth of the recursion into plain Javascript object properties
 * @return target collection
 * @export
 * @memberof! oj.CollectionUtils
 */
oj.CollectionUtils.copyInto = function (
  target,
  source,
  keyConverter,
  recurse,
  maxRecursionDepth) {
  return oj.CollectionUtils._copyIntoImpl(target,
                                          source,
                                          keyConverter,
                                          recurse,
                                          maxRecursionDepth,
                                          0);
};


/**
 * Checks whether the object is a direct instance of Object
 * @param {Object} obj - object to test
 *
 * @return {boolean} true if the object is a direct instance of Object, false otherwise
 * @export
 * @memberof! oj.CollectionUtils
 */
oj.CollectionUtils.isPlainObject = function (obj) {
  if (obj !== null && typeof obj === 'object') {
    try {
      var hasOwnProperty = Object.prototype.hasOwnProperty;
      if (obj.constructor &&
          hasOwnProperty.call(obj.constructor.prototype, 'isPrototypeOf')) {
        return true;
      }
    } catch (e) {
      // Ignore errors
    }
  }

  return false;
};


/**
 * @private
 * @memberof! oj.CollectionUtils
 */
oj.CollectionUtils._copyIntoImpl = function (
  target,
  source,
  keyConverter,
  recurse,
  maxRecursionDepth,
  currentLevel) {
  var targetKey;

  if (maxRecursionDepth === undefined || maxRecursionDepth === null) {
    // eslint-disable-next-line no-param-reassign
    maxRecursionDepth = Number.MAX_VALUE;
  }

  if (target && source && (target !== source)) {
    var keys = Object.keys(source);
    for (var i = 0; i < keys.length; i++) {
      var k = keys[i];
      // allow the key mapping to be overridden
      if (keyConverter) {
        targetKey = keyConverter(k);
      } else {
        targetKey = k;
      }

      var sourceVal = source[k];

      var recursed = false;

      if (recurse && currentLevel < maxRecursionDepth) {
        var targetVal = target[targetKey];
        if (oj.CollectionUtils.isPlainObject(sourceVal) &&
           (targetVal == null || oj.CollectionUtils.isPlainObject(targetVal))) {
          recursed = true;
          // eslint-disable-next-line no-param-reassign
          target[targetKey] = targetVal || {};
          oj.CollectionUtils._copyIntoImpl(target[targetKey],
                                           sourceVal,
                                           keyConverter,
                                           true,
                                           maxRecursionDepth,
                                           currentLevel + 1);
        }
      }
      if (!recursed) {
        // eslint-disable-next-line no-param-reassign
        target[targetKey] = sourceVal;
      }
    }
  }

  return target;
};

/*
** Copyright (c) 2008, 2012, Oracle and/or its affiliates. All rights reserved.
**
**34567890123456789012345678901234567890123456789012345678901234567890123456789
*/

/* jslint browser: true*/
/* global define: false, Promise:false*/

/**
 * Base class of all OJET Objects.
 * <p>
 * To create a subclass of another oj.Object, use oj.Object.createSubclass.
 * The subclass can specify class-level initialization by implementing an
 * <code>InitClass()</code> method on its constructor.  <code>InitClass</code>
 * is guaranteed to be called only once per class.  Further, a class'
 * <code>InitClass</code> method is guranteed to be called only after its
 * superclass' class initialization has been called.  When <code>InitClass</code>
 * is called, <code>this</code> is the class' constructor.  This allows class
 * initialization implementations to be shared in some cases.
 * </p>
 */


/**
 * @constructor oj.Object
 * @ojtsignore
 * @since 1.0
 * @export
 */
oj.Object = function () {
  this.Init();
};

oj.Object.superclass = null;

/**
 * @private
 */
oj.Object._typeName = 'oj.Object';

// regular expressicloneon for stripping out the name of a function
/**
 * @private
 */
oj.Object._GET_FUNCTION_NAME_REGEXP = /function\s+([\w$][\w$\d]*)\s*\(/;
// oj.Object._TRIM_REGEXP = /(^\s*)|(\s*$)/g; this.replace(/(^\s*)|(\s*$)/g, "");

oj.Object.prototype = {};
oj.Object.prototype.constructor = oj.Object;


/**
 * Creates a subclass of a baseClass
 * @method createSubclass
 * @memberof oj.Object
 * @param {Object} extendingClass The class to extend from the base class
 * @param {Object} baseClass class to make the superclass of extendingClass
 * @param {string=} typeName to use for new class.  If not specified, the typeName will be extracted from the
 * baseClass's function if possible
 * @return {void}
 * @export
 */
oj.Object.createSubclass = function (
  extendingClass,
  baseClass,
  typeName) {  // optional name to name this class
  oj.Assert.assertFunction(extendingClass);
  oj.Assert.assertFunctionOrNull(baseClass);
  oj.Assert.assertStringOrNull(typeName);

  if (baseClass === undefined) {
    // assume oj.Object
    // eslint-disable-next-line no-param-reassign
    baseClass = oj.Object;
  }

  oj.Assert.assert(extendingClass !== baseClass, "Class can't extend itself");

  // use a temporary constructor to get our superclass as our prototype
  // without out having to initialize the superclass
  /**
   * @private
   * @constructor
   */
  var TempConstructor = oj.Object._tempSubclassConstructor;

  TempConstructor.prototype = baseClass.prototype;
  // eslint-disable-next-line no-param-reassign
  extendingClass.prototype = new TempConstructor();

  // eslint-disable-next-line no-param-reassign
  extendingClass.prototype.constructor = extendingClass;
  // eslint-disable-next-line no-param-reassign
  extendingClass.superclass = baseClass.prototype;

  if (typeName) {
    // eslint-disable-next-line no-param-reassign
    extendingClass._typeName = typeName;
  }
};

/**
 * Copies properties from the source object to the prototype of the target class
 * Only properties 'owned' by the source object will be copied, i.e. the properties
 * from the source object's prototype chain will not be included.
 * To copy properties from another class with methods defined on the prototype, pass
 * otherClass.prototype as the source.
 * @method copyPropertiesForClass
 * @memberof oj.Object
 * @param {Object} targetClass - the function whose prototype will be used a
 * copy target
 * @param {Object} source - object whose properties will be copied
 * @return {void}
 * @export
 */
oj.Object.copyPropertiesForClass = function (targetClass, source) {
  oj.Assert.assertFunction(targetClass);
  oj.Assert.assert(source != null, 'source object cannot be null');

  var props = Object.keys(source);
  for (var i = 0; i < props.length; i++) {
    var prop = props[i];
    // eslint-disable-next-line no-param-reassign
    targetClass.prototype[prop] = source[prop];
  }
};

/**
 * @private
 */
oj.Object._tempSubclassConstructor = function () {};


/**
 * Returns the class object for the instance
 * @method getClass
 * @memberof oj.Object
 * @instance
 * @param {Object=} otherInstance - if specified, the instance whose type
 * should be returned. Otherwise the type of this instance will be returned
 * @return {Object} the class object for the instance
 * @final
 * @export
 */
oj.Object.prototype.getClass = function (
  otherInstance) {
  if (otherInstance === undefined) {
    // eslint-disable-next-line no-param-reassign
    otherInstance = this;
  } else if (otherInstance === null) {
    return null;
  }
  return otherInstance.constructor;
};


/**
 * Returns a clone of this object.  The default implementation is a shallow
 * copy.  Subclassers can override this method to implement a deep copy.
 * @method clone
 * @memberof oj.Object
 * @instance
 * @return {Object} a clone of this object
 * @export
 */
oj.Object.prototype.clone = function () {
  var clone = new this.constructor();

  oj.CollectionUtils.copyInto(clone, this);

  return clone;
};

/**
 * @export
 * @method toString
 * @memberof oj.Object
 * @instance
 * @return {string}
 */
oj.Object.prototype.toString = function () {
  return this.toDebugString();
};

/**
 * @export
 * @method toDebugString
 * @memberof oj.Object
 * @instance
 * @return {string}
 */
oj.Object.prototype.toDebugString = function () {
  return this.getTypeName() + ' Object';
};


/**
 * Returns the type name for a class derived from oj.Object
 * @method getTypeName
 * @memberof oj.Object
 * @instance
 * @param {Object|null} clazz Class to get the name of
 * @return {string} name of the Class
 * @export
 */
oj.Object.getTypeName = function (clazz) {
  oj.Assert.assertFunction(clazz);

  var typeName = clazz._typeName;

  if (typeName == null) {
    var constructorText = clazz.toString();
    var matches = oj.Object._GET_FUNCTION_NAME_REGEXP.exec(constructorText);

    if (matches) {
      typeName = matches[1];
    } else {
      typeName = 'anonymous';
    }

    // cache the result on the function
    // eslint-disable-next-line no-param-reassign
    clazz._typeName = typeName;
  }

  return typeName;
};

/**
 * Returns the type name for this instance
 * @method getTypeName
 * @memberof oj.Object
 * @return {string} name of the Class
 * @final
 * @export
 */
oj.Object.prototype.getTypeName = function () {
  return oj.Object.getTypeName(this.constructor);
};

/**
 * Initializes the instance.  Subclasses of oj.Object must call
 * their superclass' Init
 * @export
 * @method Init
 * @return {void}
 * @memberof oj.Object
 * @instance
 */
oj.Object.prototype.Init = function () {
  if (oj.Assert.isDebug()) {
    oj.Assert.assert(this.getTypeName, 'Not an oj.Object');
  }

  // do any class initialization.  This code is duplicated from
  // oj.Object.ensureClassInitialization()

  var currClass = this.constructor;
  if (!currClass._initialized) {
    oj.Object._initClasses(currClass);
  }
};

/**
 * Ensures that a class is initialized.  Although class initialization occurs
 * by default the first time that an instance of a class is created, classes that
 * use static factory methods to create their instances may
 * still need to ensure that their class has been initialized when the factory
 * method is called.
 *
 * @method ensureClassInitialization
 * @memberof oj.Object
 * @param {Object} clazz The class to ensure initialization of
 * @return {void}
 * @export
 */
oj.Object.ensureClassInitialization = function (clazz) {
  oj.Assert.assertFunction(clazz);

  if (!clazz._initialized) {
    oj.Object._initClasses(clazz);
  }
};


/**
 * Indicates whether some other oj.Object is "equal to" this one.
 * Method is equivalent to java ".equals()" method.
 * @method equals
 * @memberof oj.Object
 * @instance
 * @param {Object} object - comparison target
 * @return {boolean} true if if the comparison target is equal to this object, false otherwise
 * @export
 */
oj.Object.prototype.equals = function (
  object) {
  return this === object;
};

/**
 * Binds the supplied callback function to an object
 * @method createCallback
 * @memberof oj.Object
 * @param {!Object} obj - object that will be available to the supplied callback
 * function as 'this'
 * @param {!Object} func - the original callback
 * @return {function()} a function that will be invoking the original callback with
 * 'this' object assigned to obj
 * @ojsignature {target: "Type", for: "returns", value: "()=>any"}
 * @export
 */
oj.Object.createCallback = function (obj, func) {
  oj.Assert.assertFunction(func);

  // All browsers supported by JET support bind() method
  return func.bind(obj);
};


/**
 * @private
 */
oj.Object._initClasses = function (currClass) {
  if (oj.Assert.isDebug()) {
    oj.Assert.assertFunction(currClass);
    oj.Assert.assert(!currClass._initialized);
  }

  // eslint-disable-next-line no-param-reassign
  currClass._initialized = true;

  var superclass = currClass.superclass;

  // initialize the superclass if necessary
  if (superclass) {
    var superclassConstructor = superclass.constructor;

    if (superclassConstructor && !superclassConstructor._initialized) {
      oj.Object._initClasses(superclassConstructor);
    }
  }


  // if the class has an initialization function, call it
  var InitClassFunc = currClass.InitClass;

  if (InitClassFunc) {
    InitClassFunc.call(currClass);
  }
};

/**
 * Compares 2 values using strict equality except for the case of
 * <ol>
 *   <li> Array [order matters]; will traverse through the arrays and compare oj.Object.compareValues(array[i], array2[i]) </li>
 *   <li> Instances that support valueOf [i.e. Boolean, String, Number, Date, and etc] will be compared by usage of that function </li>
 * </ol>
 * @param {any} obj1 The first value to compare.
 * @param {any} obj2 The second value to compare.
 * @return {boolean}
 * @public
 * @export
 * @method compareValues
 * @memberof oj.Object
 */
oj.Object.compareValues = function (obj1, obj2) {
  if (obj1 === obj2) {
    return true;
  }

  var obj1Type = typeof obj1;
  var obj2Type = typeof obj2;

  if (obj1Type !== obj2Type) {
    // of different type so consider them unequal
    return false;
  }

  // At this point means the types are equal

  // note that if the operand is an array or a null then typeof is an object
  // check if either is null and if so return false [i.e. case where one might be a null and another an object]
  // and one wishes to avoid the null pointer in the following checks. Note that null === null has been already tested
  if (obj1 === null || obj2 === null) {
    return false;
  }

  // now check for constructor since I think by here one has ruled out primitive values and if the constructors
  // aren't equal then return false
  if (obj1.constructor === obj2.constructor) {
    // these are special cases and will need to be modded on a need to have basis
    if (Array.isArray(obj1)) {
      return oj.Object._compareArrayValues(obj1, obj2);
    } else if (obj1.constructor === Object) {
      // for now invoke innerEquals and in the future if there are issues then resolve them
      return oj.Object.__innerEquals(obj1, obj2);
    } else if (obj1.valueOf && typeof obj1.valueOf === 'function') {
      // test cases for Boolean, String, Number, Date
      // Note if some future JavaScript constructors
      // do not impl it then it's their fault
      return obj1.valueOf() === obj2.valueOf();
    }
  }

  return false;
};

oj.Object._compareArrayValues = function (array1, array2) {
  if (array1.length !== array2.length) {
    return false;
  }

  for (var i = 0, j = array1.length; i < j; i++) {
    // recurse on each of the values, order does matter for our case since do not wish to search
    // for the value [expensive]
    if (!oj.Object.compareValues(array1[i], array2[i])) {
      return false;
    }
  }
  return true;
};

// Comparion of two Objects containing id and or index properties.
// Note: it returns false if one is an id and other is an index
// if ids are the same, index will be ignored if there is only one is provided
oj.Object._compareIdIndexObject = function (obj1, obj2) {
  if ((typeof obj1 === 'number' && typeof obj2 === 'number') ||
      (typeof obj1 === 'string' && typeof obj2 === 'string')) {
    return obj1 === obj2;
  }

  if (typeof obj1 === 'object' && typeof obj2 === 'object') {
    if (obj1.id && obj2.id) {
      if (obj1.id !== obj2.id) {
        return false;
      }

      if (obj1.index && obj2.index) {
        return obj1.index === obj2.index;
      }

      return true;
    } else if (obj1.index && obj2.index) {
      return obj1.index === obj2.index;
    }
  }

  return false;
};


// Comparion of two arrays containing ids, indexes, or objects where each object has id,
// index or both properties.
// order needn't be same but no duplicates
oj.Object._compareArrayIdIndexObject = function (array1, array2) {
  // null and [] are equals
  if (!array1) {
    return (!array2 || array2.length === 0);
  }

  if (!array2) {
    return (!array1 || array1.length === 0);
  }

  if (array1.length !== array2.length) {
    return false;
  }

  for (var i = 0; i < array1.length; i++) {
    var found = false;
    for (var j = 0; j < array2.length; j++) {
      if (oj.Object._compareIdIndexObject(array1[i], array2[j])) {
        found = true;
        break;
      }
    }
    if (!found) {
      return false;
    }
  }

  return true;
};


oj.Object.__innerEquals = function (obj1, obj2) {
  if (obj1 === obj2) {
    return true;
  }

  if (!(obj1 instanceof Object) || !(obj2 instanceof Object)) {
    return false;
  }

  if (obj1.constructor !== obj2.constructor) {
    return false;
  }

  var hasOwnProperty = Object.prototype.hasOwnProperty;
  var props1 = Object.keys(obj1);
  var prop;
  var i;

  for (i = 0; i < props1.length; i++) {
    prop = props1[i];

    if (hasOwnProperty.call(obj1, prop)) {
      if (!hasOwnProperty.call(obj2, prop)) {
        return false;
      }

      if (obj1[prop] !== obj2[prop]) {
        if (typeof (obj1[prop]) !== 'object') {
          return false;
        }

        if (!oj.Object.__innerEquals(obj1[prop], obj2[prop])) {
          return false;
        }
      }
    }
  }

  var props2 = Object.keys(obj2);
  for (i = 0; i < props2.length; i++) {
    prop = props2[i];

    if (hasOwnProperty.call(obj2, prop) && !hasOwnProperty.call(obj1, prop)) {
      return false;
    }
  }

  if (props1.length === 0 && props2.length === 0) {
    // we are dealing with objects that have no properties like Number or Date.
    return JSON.stringify(obj1) === JSON.stringify(obj2);
  }

  return true;
};

/**
 * @method isEmpty
 * @return {boolean}
 * @memberof oj.Object
 */
oj.Object.isEmpty = function (object) {
  var prop;
  // Test if an object is empty
  if (object === undefined || object === null) {
    return true;
  }

  for (prop in object) { // eslint-disable-line no-restricted-syntax
    if (object.hasOwnProperty(prop)) { // eslint-disable-line no-prototype-builtins
      return false;
    }
  }
  return true;
};

/**
 * @private
 * @return  {boolean} true if AMD Loader (such as Require.js) is present,
 *                    false otherwise
 */
oj.__isAmdLoaderPresent = function () {
  return (typeof define === 'function' && define.amd);
};

/**
 * Loads a file using require if AMD loader is present, otherwise returns null
 * If loading multiple files then use multiple calls to this and Promise.all
 * @private
 * @param {string} module sting of the module to load
 * @param {function} requireFunc what to use as the require function, if not specified require will be used
 * @returns {Promise|null} returns null if no AMD loader
 */
oj.__getRequirePromise = function (module, requireFunc) {
  if (oj.__isAmdLoaderPresent()) {
    return new Promise(
      function (resolve, reject) {
        requireFunc([module], resolve, reject);
      }
    );
  }
  return null;
};


/*
** Copyright (c) 2004, 2012, Oracle and/or its affiliates. All rights reserved.
*/

/**
 * String utilities.
 * @class oj.StringUtils
 * @export
 * @ignore
 */
oj.StringUtils = {};

oj.StringUtils._TRIM_ALL_RE = /^\s*|\s*$/g;

/**
 * Returns true if the value is null or if the trimmed value is of zero length.
 *
 * @param {Object|string|null} value
 * @returns {boolean} true if the string or Object (e.g., Array) is of zero length.
 * @export
 * @memberof oj.StringUtils
 */
oj.StringUtils.isEmpty = function (value) {
  if (value === null) {
    return true;
  }

  var trimValue = oj.StringUtils.trim(value);

  return (trimValue.length === 0);
};

/**
 * Returns true if the value is null, undefined or if the trimmed value is of zero length.
 *
 * @param {Object|string|null=} value
 * @returns {boolean} true if the string or Object (e.g., Array) is of zero length.
 * @export
 * @memberof oj.StringUtils
 */
oj.StringUtils.isEmptyOrUndefined = function (value) {
  if (value === undefined || oj.StringUtils.isEmpty(value)) {
    return true;
  }

  return false;
};

/**
 * Test if an object is a string (either a string constant or a string object)
 * @param {Object|string|null} obj object to test
 * @return {boolean} true if a string constant or string object
 * @export
 * @memberof oj.StringUtils
 */
oj.StringUtils.isString = function (obj) {
  return obj !== null && ((typeof obj === 'string') || obj instanceof String);
};

/**
 * Remove leading and trailing whitespace
 * @param {Object|string|null} data to trim
 * @return {Object|string|null}
 * @export
 * @memberof oj.StringUtils
 */
oj.StringUtils.trim = function (data) {
  if (oj.StringUtils.isString(data)) {
    return data.replace(oj.StringUtils._TRIM_ALL_RE, '');
  }

  return data;
};

/**
 * Port of the Java String.hashCode method.
 * http://erlycoder.com/49/javascript-hash-functions-to-convert-string-into-integer-hash-
 *
 * @param {string} str
 * @returns {number}
 * @public
 * @memberof oj.StringUtils
 */
oj.StringUtils.hashCode = function (str) {
  var hash = 0;
  if (str.length === 0) {
    return hash;
  }

  for (var i = 0; i < str.length; i++) {
    var c = str.charCodeAt(i);
    // eslint-disable-next-line no-bitwise
    hash = ((hash << 5) - hash) + c;
    // eslint-disable-next-line no-bitwise
    hash &= hash;
  }
  return hash;
};


// Polyfills for IE11
(function () {
  // String.startsWith requires for IE11
  if (!String.prototype.startsWith) {
    // eslint-disable-next-line no-extend-native
    String.prototype.startsWith = function (searchString, position) {
      // eslint-disable-next-line no-param-reassign
      position = position || 0;
      return this.substr(position, searchString.length) === searchString;
    };
  }

  // String.endsWith requires for IE11
  if (!String.prototype.endsWith) {
    // eslint-disable-next-line no-extend-native
    String.prototype.endsWith = function (searchString, position) {
      var subjectString = this.toString();
      if (typeof position !== 'number' || !isFinite(position) ||
          Math.floor(position) !== position || position > subjectString.length) {
        // eslint-disable-next-line no-param-reassign
        position = subjectString.length;
      }
      // eslint-disable-next-line no-param-reassign
      position -= searchString.length;
      var lastIndex = subjectString.lastIndexOf(searchString, position);
      return lastIndex !== -1 && lastIndex === position;
    };
  }
}());

/**
 * Copyright (c) 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */

/* jslint browser: true*/


/**
 * @ojtsignore
 * @class oj.AgentUtils
 * @classdesc Utilities for qualifying the user agent string.
 * @public
 * @ignore
 */
oj.AgentUtils = function () {};

/**
 * Identity of the target browser.
 * @enum {string}
 * @public
 * @memberof oj.AgentUtils
 */
oj.AgentUtils.BROWSER = {
  IE: 'ie',
  FIREFOX: 'firefox',
  SAFARI: 'safari',
  CHROME: 'chrome',
  EDGE: 'edge',
  UNKNOWN: 'unknown'
};
/**
 * Browser layout engine identity.
 * @enum {string}
 * @public
 * @memberof oj.AgentUtils
 */
oj.AgentUtils.ENGINE = {
  TRIDENT: 'trident',
  WEBKIT: 'webkit',
  GECKO: 'gecko',
  BLINK: 'blink',
  EDGE_HTML: 'edgehtml',
  UNKNOWN: 'unknown'
};
/**
 * Operating system identity.
 * @enum {string}
 * @public
 * @memberof oj.AgentUtils
 */
oj.AgentUtils.OS = {
  WINDOWS: 'Windows',
  SOLARIS: 'Solaris',
  MAC: 'Mac',
  UNKNOWN: 'Unknown',
  ANDROID: 'Android',
  IOS: 'IOS',
  WINDOWSPHONE: 'WindowsPhone',
  LINUX: 'Linux'
};
/**
 * Device type identity.
 * @enum {string}
 * @public
 * @memberof oj.AgentUtils
 */
oj.AgentUtils.DEVICETYPE = {
  PHONE: 'phone',
  TABLET: 'tablet',
  OTHERS: 'others'
};

/**
 * Parses the browser user agent string determining what browser and layout engine
 * is being used.
 *
 * @param {Object|null|string=} userAgent a specific agent string but defaults to navigator userAgent if not provided
 * @return {{os: oj.AgentUtils.OS, browser: oj.AgentUtils.BROWSER, browserVersion: number, deviceType: oj.AgentUtils.DEVICETYPE,
 *          engine: oj.AgentUtils.ENGINE, engineVersion: number, hashCode: number}}
 * @public
 * @memberof oj.AgentUtils
 */
oj.AgentUtils.getAgentInfo = function (userAgent) {
  if (oj.StringUtils.isEmptyOrUndefined(userAgent)) {
    // eslint-disable-next-line no-param-reassign
    userAgent = navigator.userAgent;
  }
  // eslint-disable-next-line no-param-reassign
  userAgent = userAgent.toLowerCase();
  /** @type {number} */
  var hashCode = oj.StringUtils.hashCode(userAgent);
  var currAgentInfo = oj.AgentUtils._currAgentInfo;
  if (currAgentInfo && currAgentInfo.hashCode === hashCode) {
    return {
      os: currAgentInfo.os,
      browser: currAgentInfo.browser,
      browserVersion: currAgentInfo.browserVersion,
      deviceType: currAgentInfo.deviceType,
      engine: currAgentInfo.engine,
      engineVersion: currAgentInfo.engineVersion,
      hashCode: currAgentInfo.hashCode
    };
  }
  /** @type {oj.AgentUtils.OS} */
  var os = oj.AgentUtils.OS.UNKNOWN;
  /** @type {oj.AgentUtils.BROWSER} */
  var browser = oj.AgentUtils.BROWSER.UNKNOWN;
  /** @type {number} */
  var browserVersion = 0;
  /** @type {oj.AgentUtils.DEVICETYPE} */
  var deviceType = oj.AgentUtils.DEVICETYPE.OTHERS;
  /** @type {oj.AgentUtils.ENGINE} */
  var engine = oj.AgentUtils.ENGINE.UNKNOWN;
  /** @type {number} */
  var engineVersion = 0;
  if (userAgent.indexOf('iphone') > -1 || userAgent.indexOf('ipad') > -1) {
    os = oj.AgentUtils.OS.IOS;
  } else if (userAgent.indexOf('mac') > -1) {
    os = oj.AgentUtils.OS.MAC;
  } else if (userAgent.indexOf('sunos') > -1) {
    os = oj.AgentUtils.OS.SOLARIS;
  } else if (userAgent.indexOf('android') > -1) {
    os = oj.AgentUtils.OS.ANDROID;
  } else if (userAgent.indexOf('linux') > -1) {
    os = oj.AgentUtils.OS.LINUX;
  } else if (userAgent.indexOf('windows phone') > -1) {
    os = oj.AgentUtils.OS.WINDOWSPHONE;
  } else if (userAgent.indexOf('win') > -1) {
    os = oj.AgentUtils.OS.WINDOWS;
  }

  if (os === oj.AgentUtils.OS.ANDROID) {
    // This works for Chrome, Firefox, and Edge on Android, even though only Chrome is officially supported.
    // This also works for Edge on Windows 10 Mobile, which announces itself as android-compatible user agent.
    deviceType = userAgent.indexOf('mobile') > -1 ?
      oj.AgentUtils.DEVICETYPE.PHONE : oj.AgentUtils.DEVICETYPE.TABLET;
  } else if (os === oj.AgentUtils.OS.IOS) {
    // This works for Safari, Chrome, Firefox, and Edge on iOS, even though only Safari is officially supported.
    deviceType = userAgent.indexOf('iphone') > -1 ?
      oj.AgentUtils.DEVICETYPE.PHONE : oj.AgentUtils.DEVICETYPE.TABLET;
  }

  if (userAgent.indexOf('msie') > -1) {
    browser = oj.AgentUtils.BROWSER.IE;
    browserVersion = oj.AgentUtils._parseFloatVersion(userAgent, /msie (\d+[.]\d+)/);
    if (userAgent.indexOf('trident')) {
      engine = oj.AgentUtils.ENGINE.TRIDENT;
      engineVersion = oj.AgentUtils._parseFloatVersion(userAgent, /trident\/(\d+[.]\d+)/);
    }
  } else if (userAgent.indexOf('trident') > -1) {
    browser = oj.AgentUtils.BROWSER.IE;
    browserVersion = oj.AgentUtils._parseFloatVersion(userAgent, /rv:(\d+[.]\d+)/);
    if (userAgent.indexOf('trident')) {
      engine = oj.AgentUtils.ENGINE.TRIDENT;
      engineVersion = oj.AgentUtils._parseFloatVersion(userAgent, /trident\/(\d+[.]\d+)/);
    }
  } else if (userAgent.indexOf('edge') > -1) {
    browser = oj.AgentUtils.BROWSER.EDGE;
    engineVersion = oj.AgentUtils._parseFloatVersion(userAgent, /edge\/(\d+[.]\d+)/);
    browserVersion = engineVersion;
    engine = oj.AgentUtils.ENGINE.EDGE_HTML;
  } else if (userAgent.indexOf('chrome') > -1) {
    browser = oj.AgentUtils.BROWSER.CHROME;
    browserVersion = oj.AgentUtils._parseFloatVersion(userAgent, /chrome\/(\d+[.]\d+)/);
    if (browserVersion >= 28) {
      engine = oj.AgentUtils.ENGINE.BLINK;
      engineVersion = browserVersion;
    } else {
      engine = oj.AgentUtils.ENGINE.WEBKIT;
      engineVersion = oj.AgentUtils._parseFloatVersion(userAgent, /applewebkit\/(\d+[.]\d+)/);
    }
  } else if (userAgent.indexOf('safari') > -1) {
    browser = oj.AgentUtils.BROWSER.SAFARI;
    browserVersion = oj.AgentUtils._parseFloatVersion(userAgent, /version\/(\d+[.]\d+)/);
    engine = oj.AgentUtils.ENGINE.WEBKIT;
    engineVersion = oj.AgentUtils._parseFloatVersion(userAgent, /applewebkit\/(\d+[.]\d+)/);
  } else if (userAgent.indexOf('firefox') > -1) {
    browser = oj.AgentUtils.BROWSER.FIREFOX;
    browserVersion = oj.AgentUtils._parseFloatVersion(userAgent, /rv:(\d+[.]\d+)/);
    engine = oj.AgentUtils.ENGINE.GECKO;
    engineVersion = oj.AgentUtils._parseFloatVersion(userAgent, /gecko\/(\d+)/);
  }

  currAgentInfo = {
    hashCode: hashCode,
    os: os,
    browser: browser,
    browserVersion: browserVersion,
    deviceType: deviceType,
    engine: engine,
    engineVersion: engineVersion
  };
  oj.AgentUtils._currAgentInfo = currAgentInfo;

  return {
    os: currAgentInfo.os,
    browser: currAgentInfo.browser,
    browserVersion: currAgentInfo.browserVersion,
    deviceType: currAgentInfo.deviceType,
    engine: currAgentInfo.engine,
    engineVersion: currAgentInfo.engineVersion,
    hashCode: currAgentInfo.hashCode
  };
};
/**
 * @param {string!} userAgent
 * @param {RegExp!} versionNumberPattern
 * @return {number}
 * @private
 * @memberof oj.AgentUtils
 */
oj.AgentUtils._parseFloatVersion = function (userAgent, versionNumberPattern) {
  var matches = userAgent.match(versionNumberPattern);
  if (matches) {
    var versionString = matches[1];
    if (versionString) {
      return parseFloat(versionString);
    }
  }

  return 0;
};


(function () {
  // Checks that a string either starts with 'array' or contains '|array'
  var _ARRAY_TYPE_EXP = /(^array)|(\|array)/;
  // Checks that a string either starts with 'object' or contains '|object'
  var _OBJ_TYPE_EXP = /(^object)|(\|object)/;

  var _ARRAY_VALUE_EXP = /\s*\[[^]*\]\s*/;
  var _OBJ_VALUE_EXP = /\s*\{[^]*\}\s*/;

  // Check for {{..}} and [[..]] at the beginning of strings to avoid matching
  // any usages mid string
  var _ATTR_EXP = /^(?:\{\{)([^]+)(?:\}\})$/;
  var _ATTR_EXP_RO = /^(?:\[\[)([^]+)(?:\]\])$/;

  /**
   * @ignore
   * @private
   */
  oj.__AttributeUtils = {};

  /**
   * @ignore
   * @return {{expr: (null|string), downstreamOnly: boolean}}
   * @private
   */
  oj.__AttributeUtils.getExpressionInfo = function (attrValue) {
    var info = {};
    if (attrValue) {
      var trimmedVal = attrValue.trim();
      var exp = _ATTR_EXP.exec(trimmedVal);
      exp = exp ? exp[1] : null;
      if (!exp) {
        info.downstreamOnly = true;
        exp = _ATTR_EXP_RO.exec(trimmedVal);
        exp = exp ? exp[1] : null;
      }
      info.expr = exp;
    }

    return info;
  };

  /**
   * @ignore
   * @param {string} attr attribute name
   * @return {string} property name
   * @private
   */
  oj.__AttributeUtils.attributeToPropertyName = function (attr) {
    return attr.toLowerCase().replace(/-(.)/g,
      function (match, group1) {
        return group1.toUpperCase();
      }
    );
  };

  /**
   * @ignore
   * @param {string} name property name
   * @return {string} attribute name
   * @private
   */
  oj.__AttributeUtils.propertyNameToAttribute = function (name) {
    return name.replace(/([A-Z])/g,
      function (match) {
        return '-' + match.toLowerCase();
      }
    );
  };

  /**
   * @ignore
   * @param {string} type event type (e.g. ojBeforeExpand)
   * @return {string} event listener property name (e.g. onOjBeforeExpand)
   * @private
   */
  oj.__AttributeUtils.eventTypeToEventListenerProperty = function (type) {
    return 'on' + type.substr(0, 1).toUpperCase() + type.substr(1);
  };

  /**
   * @ignore
   * @param {string} property event listener property name (e.g. onOjBeforeExpand)
   * @return {string|null} event type (e.g. ojBeforeExpand)
   * @private
   */
  oj.__AttributeUtils.eventListenerPropertyToEventType = function (property) {
    if (/^on[A-Z]/.test(property)) {
      return property.substr(2, 1).toLowerCase() + property.substr(3);
    }
    return null;
  };

  /**
   * @ignore
   * @param {string} name property name (e.g. expanded)
   * @return {string} change event type (e.g. expandedChanged)
   * @private
   */
  oj.__AttributeUtils.propertyNameToChangeEventType = function (name) {
    return name + 'Changed';
  };

  /**
   * @ignore
   * @param {string} trigger event trigger (e.g. beforeExpand)
   * @return {string} event type (e.g. ojBeforeExpand)
   * @private
   */
  oj.__AttributeUtils.eventTriggerToEventType = function (trigger) {
    return 'oj' + trigger.substr(0, 1).toUpperCase() + trigger.substr(1);
  };

  /**
   * Parses attribute values to the specified metadata type. Throws
   * an error if the value cannot be parsed to the metadata type
   * or if no type was provided.
   * @ignore
   * @param {Element} elem The element whose value we are parsing
   * @param {string} attr attribute
   * @param {string} value attribute value
   * @param {string} type property type
   * @return {any} coerced value
   * @private
   */
  oj.__AttributeUtils.coerceValue = function (elem, attr, value, type) {
    if (!type) {
      throw new Error('Unable to parse ' + attr + "='" + value + "' for " +
                      elem + ' with id ' + elem.id +
                      ' . This attribute only supports data bound values. Check the API doc for supported types');
    }

    // We only support primitive types and JSON objects for coerced properties.
    // Generally, we support parsing of a single type except for Object|string or Array|string cases
    // defined in metadata.
    var typeLwr = type.toLowerCase();
    // The below checks ignore the couble {{}} [[]] cases since expression checking occurs
    // before attribute value coercion
    // Tests to see if the value starts and ends with matched [...] ignoring whitespace
    var isValueArray = _ARRAY_VALUE_EXP.test(value);
    // Tests to see if the value starts and ends with matched {...} ignoring whitespace
    var isValueObj = _OBJ_VALUE_EXP.test(value);

    if ((_ARRAY_TYPE_EXP.test(typeLwr) && isValueArray) ||
        (_OBJ_TYPE_EXP.test(typeLwr) && isValueObj) ||
        (typeLwr === 'any' && (isValueArray || isValueObj))) {
      try {
        return JSON.parse(value);
      } catch (ex) {
        throw new Error('Unable to parse ' + attr + "='" + value + "' for " +
                        elem.tagName + ' with id ' + elem.id +
                        ' to a JSON Object. Check the value for correct JSON syntax, e.g. double quoted strings. ' + ex);
      }
    } else if (typeLwr === 'boolean') {
      return oj.__AttributeUtils.coerceBooleanValue(elem, attr, value, type);
    } else if (typeLwr === 'number') {
      if (!isNaN(value)) {
        return Number(value);
      }
    } else {
      var typeAr = type.split('|');
      // The any type will return a string if not matched as an object or array in first check
      if (typeAr.indexOf('string') !== -1 || typeLwr === 'any') {
        return value;
      }
    }

    throw new Error('Unable to parse ' + attr + "='" + value + "' for " +
                    elem + ' with id ' + elem.id + ' to a ' + type + '.');
  };

  /**
   * Parses boolean attribute values. Throws
   * an error if the value cannot be parsed.
   * @ignore
   * @param {Element} elem The element whose value we are parsing
   * @param {string} attr attribute
   * @param {string} value attribute value
   * @param {string} type property type
   * @return {boolean} coerced value
   * @private
   */
  oj.__AttributeUtils.coerceBooleanValue = function (elem, attr, value, type) {
    // Boolean attributes are considered true if the attribute is:
    // 1) Set to the empty string
    // 2) Present in the DOM without a value assignment
    // 3) Set to the 'true' string
    // 4) Set to the case-insensitive attribute name
    // Boolean values are considered false if set to the false string.
    // An error is thrown for all other values and the attribute value will not be set.
    if (value == null || value === 'true' || value === '' || value.toLowerCase() === attr) {
      return true;
    } else if (value === 'false') {
      return false;
    }

    throw new Error('Unable to parse ' + attr + "='" + value + "' for " +
                    elem + ' with id ' + elem.id + ' to a ' + type + '.');
  };
}());


(function () {
  if (typeof window === 'undefined') {
    return;
  }
  // polyfill for Element.closest()
  if (window.Element && !Element.prototype.closest) {
    Element.prototype.closest =
        function (s) {
          var matches = (this.document || this.ownerDocument).querySelectorAll(s);
          var i;
          var el = this;

          do { // eslint-disable-line no-cond-assign
            i = matches.length;
            while (--i >= 0 && matches.item(i) !== el) {} // eslint-disable-line no-plusplus, no-empty
          } while ((i < 0) && (el = el.parentElement));
          return el;
        };
  }
}());

/* global Set:false */

/**
 * Element utilities.
 * @class oj.ElementUtils
 * @ignore
 */
oj.ElementUtils = {
  /**
   * Custom element name check
   * @param {String} localName Element name
   * @return {boolean}
   * @ignore
   */
  isValidCustomElementName: function (localName) {
    var reservedTagList = new Set([
      'annotation-xml',
      'color-profile',
      'font-face',
      'font-face-src',
      'font-face-uri',
      'font-face-format',
      'font-face-name',
      'missing-glyph',
    ]);
    var reserved = reservedTagList.has(localName);
    var validForm = /^[a-z][.0-9_a-z]*-[-.0-9_a-z]*$/.test(localName);
    return !reserved && validForm && !localName.startsWith('oj-bind-', 0);
  }
};

/**
 * Copyright (c) 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */

/**
 * @preserve Copyright 2013 jQuery Foundation and other contributors
 * Released under the MIT license.
 * http://jquery.org/license
 */

/* jslint browser: true*/

/**
 * @export
 * @class oj.EventSource
 * @classdesc Object which supports subscribing to and firing events
 * @constructor
 * @since 1.1
 * @ojdeprecated {since: '5.0.0', description: 'Use DataProvider instead.'}
 */
oj.EventSource = function () {
  this.Init();
};

// Subclass from oj.Object
oj.Object.createSubclass(oj.EventSource, oj.Object, 'oj.EventSource');

/**
 * Initializes the instance.
 * @export
 */
oj.EventSource.prototype.Init = function () {
  this._eventHandlers = [];
  oj.EventSource.superclass.Init.call(this);
};

/**
 * Attach an event handler.
 * <p>Application can call this if it wants to be notified of an event.  It can call the <code class="prettyprint">off</code> method to detach the handler when it no longer wants to be notified.</p>
 * @param {string} eventType eventType
 * @param {function(Object)} eventHandler event handler function
 * @return {void}
 * @memberof oj.EventSource
 * @export
 */
oj.EventSource.prototype.on = function (eventType, eventHandler) {
  var foundEventHandler = false;

  for (var i = 0; i < this._eventHandlers.length; i++) {
    if (this._eventHandlers[i].eventType === eventType &&
        this._eventHandlers[i].eventHandlerFunc === eventHandler) {
      foundEventHandler = true;
      break;
    }
  }
  if (!foundEventHandler) {
    this._eventHandlers.push({ eventType: eventType, eventHandlerFunc: eventHandler });
  }
};

/**
 * Detach an event handler.
 * <p>Application can call this if it no longer wants to be notified of an event that it has attached an handler to using the <code class="prettyprint">on</code> method.</p>
 * @param {string} eventType eventType
 * @param {function(Object)} eventHandler event handler function
 * @return {void}
 * @memberof oj.EventSource
 * @export
 */
oj.EventSource.prototype.off = function (eventType, eventHandler) {
  for (var i = this._eventHandlers.length - 1; i >= 0; i--) {
    if (this._eventHandlers[i].eventType === eventType &&
        this._eventHandlers[i].eventHandlerFunc === eventHandler) {
      this._eventHandlers.splice(i, 1);
      break;
    }
  }
};

/**
 * Handle the event
 * @param {string} eventType  event type
 * @param {Object} event  event
 * @return {boolean} Returns false if event is cancelled
 * @memberof oj.EventSource
 * @export
 */
// eslint-disable-next-line no-unused-vars
oj.EventSource.prototype.handleEvent = function (eventType, event) {
  var returnValue;

  for (var i = 0; i < this._eventHandlers.length; i++) {
    var eventHandler = this._eventHandlers[i];
    if (eventHandler.eventType === eventType) {
      returnValue =
        eventHandler.eventHandlerFunc.apply(this, Array.prototype.slice.call(arguments).slice(1));

      if (returnValue === false) {
        // event cancelled
        return false;
      }
    }
  }

  return true;
};

/**
 * Key utilities.
 * @class oj.KeyUtils
 * @export
 * @ignore
 */
oj.KeyUtils = {};

/**
 * Determine whether the two keys specified are considered equal.
 *
 * @param {any} key1 first key to compare
 * @param {any} key2 second key to compare
 * @returns {boolean} true if the keys are considered the same, false otherwise.
 * @export
 * @memberof oj.KeyUtils
 */
oj.KeyUtils.equals = function (key1, key2) {
  // algorithm for key equality:
  // if the keys are of type primitive, then do === comparison
  // if the keys are object, then do deep comparison of properties
  // for now, this is the same as compareValues, but this allows us to diverge in the future
  // ex: generate hash with key and compare hash value instead
  return oj.Object.compareValues(key1, key2);
};

/* global Map:false */

/* The custom element (webcomponents) support requires the native CustomEvent
 * object.  This polyfill provides CustomEvent implementation for browsers that
 * don't support it yet.
 */
(function () {
  if (typeof window === 'undefined') {
    return;
  }

  // defaultPrevented is broken in IE.
  // https://connect.microsoft.com/IE/feedback/details/790389/event-defaultprevented-returns-false-after-preventdefault-was-called
  var workingDefaultPrevented = (function () {
    var e = document.createEvent('Event');
    e.initEvent('foo', true, true);
    e.preventDefault();
    return e.defaultPrevented;
  }());

  if (!workingDefaultPrevented) {
    var origPreventDefault = Event.prototype.preventDefault;
    Event.prototype.preventDefault = function () {
      if (!this.cancelable) {
        return;
      }

      origPreventDefault.call(this);

      Object.defineProperty(this, 'defaultPrevented', {
        get: function () {
          return true;
        },
        configurable: true
      });
    };
  }

  if (typeof window.CustomEvent === 'function') {
    return;
  }

  function CustomEvent(event, params) {
    // eslint-disable-next-line no-param-reassign
    params = params || { bubbles: false, cancelable: false, detail: undefined };

    var evt = document.createEvent('CustomEvent');

    evt.initCustomEvent(event, params.bubbles, params.cancelable, params.detail);
    return evt;
  }

  CustomEvent.prototype = Object.getPrototypeOf(new CustomEvent('bogusEvent'));

  window.CustomEvent = CustomEvent;
}());

/* This polyfill implements a proposed Microsoft standard [1] for effective yielding.
 * With the setImmediate global function, developers can yield control flow to the
 * user agent before running script.  The yield is similar to the setTimeout function
 * in that it is evaluated in the macrotask queue.  However, the setTimeout often has
 * a minimum delay and is also subject to long delays when the browser is placed in the
 * background.  The setImmediate function implemented by this polyfill invokes the
 * callback in the "next-tick" after the current macrotask queue has been exhausted.
 *
 * [1] https://dvcs.w3.org/hg/webperf/raw-file/tip/specs/setImmediate/Overview.html
 *
 * The strategy for the polyfill implementation uses the window.postMessage API for
 * creating a context for calling the target function evaulated as a macrotask. This
 * plugin will not work in a webworker where the window object doesn't exist.
 */
(function () {
  if (typeof window === 'undefined' || window.setImmediate || !window.postMessage) {
    return;
  }

  var _setImmediateMap;

  var _setImmediateCounter;
  function _nextId() {
    if (isNaN(_setImmediateCounter)) {
      _setImmediateCounter = 0;
    }

    _setImmediateCounter += 1;
    return _setImmediateCounter;
  }

  // postMessage "message" event listener for the setImmediate impl
  function _nextTickHandler(event) {
    var data = event.data;
    if (!data || data.message !== 'oj-setImmediate') {
      return;
    }

    var id = data.id;
    var entry = _setImmediateMap.get(id);
    clearImmediateImpl(id);

    if (entry) {
      var callback = entry.callback;
      var args = entry.args;
      callback.apply(window, args);
    }
  }

  function setImmediateImpl() {
    var callback = arguments[0];
    var slice = Array.prototype.slice;
    var args = slice.call(arguments, 1);

    if (typeof callback !== 'function') {
      // eslint-disable-next-line no-new-func
      callback = new Function(callback.toString());
    }

    var id = _nextId();

    if (!_setImmediateMap) {
      _setImmediateMap = new Map();
    }

    _setImmediateMap.set(id, { callback: callback, args: args });

    if (_setImmediateMap.size === 1) {
      window.addEventListener('message', _nextTickHandler);
    }

    window.postMessage({ id: id, message: 'oj-setImmediate' }, '*');
    return id;
  }

  function clearImmediateImpl(id) {
    if (!_setImmediateMap) {
      return;
    }

    _setImmediateMap.delete(id);

    if (_setImmediateMap.size < 1) {
      window.removeEventListener('message', _nextTickHandler);
      _setImmediateMap = null;
    }
  }

  window.setImmediate = setImmediateImpl;
  window.clearImmediate = clearImmediateImpl;
}());

(function () {
  if (typeof window === 'undefined' || window.__extends) {
    return;
  }

  var __extends = (this && this.__extends) || (function () {
    var extendStatics = Object.setPrototypeOf; // || // Now available everywhere including IE11
        // ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
        // function (d, b) { for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p]; };
    return function (d, b) {
      extendStatics(d, b);
      /**
       * @constructor
       */
      function __() {
        this.constructor = d;
      }
      // eslint-disable-next-line no-param-reassign
      d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
  }());

  window.__extends = __extends;
}());

(function () {
  if (typeof window === 'undefined') {
    return;
  }
  if (window.Symbol) {
    if (!window.Symbol.asyncIterator) {
      window.Symbol.asyncIterator = 'asyncIterator';
    }

    if (!window.Symbol.iterator) {
      window.Symbol.iterator = 'iterator';
    }
  } else {
    window.Symbol = {};
    window.Symbol.asyncIterator = 'asyncIterator';
    window.Symbol.iterator = 'iterator';
  }
}());

(function () {
  if (typeof window === 'undefined') {
    return;
  }

  if (new window.Set([0]).size === 0) {
    var NativeSet = window.Set;
    // eslint-disable-next-line no-inner-declarations
    function Set(iterable) {
      var set = new NativeSet();
      if (iterable) {
        iterable.forEach(set.add, set);
      }
      return set;
    }
    Set.prototype = NativeSet.prototype;
    // eslint-disable-next-line no-extend-native
    Set.prototype.constructor = Set;
    window.Set = Set;
  }
}());

/*
** Copyright (c) 2008, 2014, Oracle and/or its affiliates. All rights reserved.
**
**34567890123456789012345678901234567890123456789012345678901234567890123456789
*/

/* global _scope:false, Logger:false */

/**
 * @private
 */
var _checkpointManagerDelegate = _scope.__ojCheckpointManager;

/**
 * Global Checkpoint Manager Instance
 * @const
 * @export
 * @ignore
 */
oj.CHECKPOINT_MANAGER = {};

/**
 * Starts a checkpoint
 * @param {!string} name - the name of the checkpoint
 * @param {string=} description - optional description of the checkpoint
 * @export
 * @memberof! oj.CHECKPOINT_MANAGER
 */
oj.CHECKPOINT_MANAGER.startCheckpoint = function (name, description) {
  if (_checkpointManagerDelegate) {
    _checkpointManagerDelegate.startCheckpoint(name, description);
  }
};

/**
 * Ends a checkpoint
 * @param {!string} name - the name of the checkpoint
 * @export
 * @memberof! oj.CHECKPOINT_MANAGER
 */
oj.CHECKPOINT_MANAGER.endCheckpoint = function (name) {
  if (_checkpointManagerDelegate) {
    _checkpointManagerDelegate.endCheckpoint(name);
  }
};

/**
 * Retrieves a checkpoint record for a given name
 * @param {!string} name - the name of the checkpoint
 * @return {undefined|{start: number, end: number, duration: number, name: {string}, description: (string|undefined)}}
 * @export
 * @memberof! oj.CHECKPOINT_MANAGER
 */
oj.CHECKPOINT_MANAGER.getRecord = function (name) {
  return _checkpointManagerDelegate ? _checkpointManagerDelegate.getRecord(name) : undefined;
};

/**
 * Retrieves all checkpoint records matching a regular expression
 * @param {!RegExp} regexp - regular expression to match.
 * @return Array.{{start: number, end: number, duration: number, name: {string}, description: (string|undefined)}}
 * @export
 * @memberof! oj.CHECKPOINT_MANAGER
 */
oj.CHECKPOINT_MANAGER.matchRecords = function (regexp) {
  return _checkpointManagerDelegate ? _checkpointManagerDelegate.matchRecords(regexp) : [];
};

/**
 * Dumps matched records into oj.Logger
 * @param {!RegExp} regexp - regular expression for the records to dump.
 * @export
 * @memberof! oj.CHECKPOINT_MANAGER
 */
oj.CHECKPOINT_MANAGER.dump = function (regexp) {
  Logger.info(
    function () {
      var logMsg = 'Checkpoint Records:';
      var records = oj.CHECKPOINT_MANAGER.matchRecords(regexp);
      for (var i = 0; i < records.length; i++) {
        var record = records[i];
        logMsg = logMsg + '\n' + record.name;
        var desc = record.description;
        if (desc != null) {
          logMsg = logMsg + ' (' + desc + ')';
        }
        logMsg += ':\n';
        logMsg = logMsg + 'start: ' + record.start + '\tduration: ' + record.duration;
      }
      return logMsg;
    }
  );
};

/**
 * Copyright (c) 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */

/**
 * @preserve Copyright 2013 jQuery Foundation and other contributors
 * Released under the MIT license.
 * http://jquery.org/license
 */

/* global Promise:false, Map:false, Logger:false */

/**
 * <p>The purpose of the BusyContext API is to accommodate sequential dependencies of asynchronous
 * operations. A common use cases defining the API is for automation testing (qunit and webdriver).
 * Automation test developers can use this API to wait until components finish animation effects
 * or data fetch before trying to interact with components. The BusyContext is not limited to
 * test automation developers usages. It is also needed by page developers for waiting on run-time
 * operation readiness.</p>
 *
 * The Busy Context API will wait until busy states have resolved or a timeout period has elapsed.
 * There are several primary wait scenarios:
 * <ol>
 *   <li>Component creation and page bindings applied.</li>
 *   <li>Components that implement animation effects.</li>
 *   <li>Components that must fetch data from a REST endpoint.</li>
 *   <li>General wait conditions that are not specific to the Jet framework. The customer might
 *       choose to register some busy condition associated with application domain logic such
 *       as REST endpoints.</li>
 *   <li>Wait until the bootstrap of the page has completed - jet libraries loaded via requireJS.</li>
 * </ol>
 *
 * <p>The first step for waiting on a busy context is to determine what conditions are of interest
 *   to wait on. The granularity of a busy context can be scoped for the entirety of the page or
 *   for a specific DOM element. Busy contexts have hierarchical dependencies mirroring the
 *   document's DOM structure with the root being the page context. Depending on the particular
 *   scenario, developers might need to target one of the following busy context scopes:</p>
 * <ul>
 *   <li>Scoped for the Page - Automation test developers will more commonly choose the page busy
 *     context. This context represents the page as a whole. Automation developers commonly need
 *     to wait until the page is fully loaded before starting automation. More commonly, automation
 *     developers are interesting in testing the functionality of an application having multiple
 *     JET components versus just a single component.
 *
 *     <pre class="prettyprint">
 *     <code>
 *     var busyContext = oj.Context.getPageContext().getBusyContext();
 *     </code></pre>
 *
 *   </li>
 *   <li>Scoped for the nearest DOM Element - Application developers sometime need a mechanism to
 *     wait until a specific component operation has complete. For example, it might be desirable
 *     to wait until a component has been created and bindings applied before setting a property or
 *     calling a method on the component. Another scenario, waiting for a popup to finish
 *     open or close animation before initiating the next action in their application flow.
 *     For this problem space developers would need to obtain a busy context scoped for a DOM node.
 *     The "data-oj-context" marker attribute is used to define a busy context for a dom subtree.
 *
 *     <pre class="prettyprint">
 *     <code>
 *     <!-- subtree assigned a marker 'data-oj-context' attribute -->
 *     &lt;div id="mycontext" data-oj-context&gt;
 *        ...
 *        &lt;!-- JET content --&gt;
 *        ...
 *     &lt;/div&gt;
 *
 *     var node = document.querySelector("#mycontext");
 *     var busyContext = oj.Context.getContext(node).getBusyContext();
 *     busyContext.whenReady().then(function ()
 *     {
 *       var component = document.querySelector("#myInput");
 *       component.value = "foo";
 *       component.validate().then(function (isValid)
 *       {
 *         if (!isValid)
 *           component.value = "foobar";
 *       });
 *     });
 *     </code></pre>
 *
 *   </li>
 * </ul>
 *
 * The BusyContext API utilizes {@link Logger.LEVEL_LOG} to log detail busy state activity.
 * <pre class="prettyprint">
 * <code>
 *  Logger.option("level", Logger.LEVEL_LOG);
 * </code></pre>
 *
 * <b>This constructor should never be invoked by the application code directly.</b>
 * @hideconstructor
 * @param {Element=} hostNode DOM element associated with this busy context
 * @export
 * @constructor oj.BusyContext
 * @ojtsmodule
 * @ojtsnoexport
 * @ojtsexportastype Context
 * @since 2.1.0
 * @classdesc Framework service for querying the busy state of components on the page.
 */
oj.BusyContext = function (hostNode) {
  this.Init(hostNode);
};

oj.Object.createSubclass(oj.BusyContext, oj.Object, 'oj.BusyContext');


/**
 * see oj.BusyContext#setDefaultTimeout
 * @type {number}
 * @ignore
 * @private
 */
oj.BusyContext._defaultTimeout = Number.NaN;

/**
 * Sets a default for the optional <code>timeout</code> argument of the {@link oj.BusyContext#whenReady}
 * for all BusyContext instances. The default value will be implicitly used if a timeout argument is not
 * provided.
 *
 * @export
 * @see oj.BusyContext#whenReady
 * @since 3.1.0
 * @memberof oj.BusyContext
 * @method setDefaultTimeout
 * @param {number} timeout in milliseconds
 * @ojdeprecated {since: '6.0.0', description: 'Use oj.Context.setBusyContextDefaultTimeout instead.'}
 * @return {undefined}
 */
oj.BusyContext.setDefaultTimeout = function (timeout) {
  if (!isNaN(timeout)) {
    oj.BusyContext._defaultTimeout = timeout;
  }
};

/**
 * @param {Element=} hostNode DOM element associated with this busy context
 * @instance
 * @protected
 */
oj.BusyContext.prototype.Init = function (hostNode) {
  oj.BusyContext.superclass.Init.call(this);

  this._hostNode = hostNode;

  /**
   * Busy states cache.
   *
   * @type {?}
   * @ignore
   * @private
   */
  this._statesMap = new Map();

  /**
   * Coordinates resolution of the master when ready promise with one or more slave
   * when ready promises having a timeout period.
   *
   * @type {Object}
   * @ignore
   * @private
   */
  this._mediator = {
    /**
     * Returns a master primise that will resolve when all busy states have been resolved.
     *
     * @returns {Promise}
     * @ignore
     * @private
     */
    getMasterWhenReadyPromise: function () {
      if (!this._masterWhenReadyPromise) {
        this._masterWhenReadyPromise =
          new Promise(this._captureWhenReadyPromiseResolver.bind(this));
      }
      return this._masterWhenReadyPromise;
    },
    /**
     * Triggers resolution of the master promise and clears all timeouts associated with slave
     * when ready promises.
     *
     * @returns {void}
     * @ignore
     * @private
     */
    resolveMasterWhenReadyPromise: function () {
      if (this._masterWhenReadyPromiseResolver) {
        this._masterWhenReadyPromiseResolver(true);
      }
      this._masterWhenReadyPromise = null;
      this._masterWhenReadyPromiseResolver = null;
      this._masterWhenReadyPromiseRejecter = null;
    },
    /**
     * Triggers rejections of the master promise.
     * @param {Object=} error
     * @returns {void}
     * @ignore
     * @private
     */
    rejectMasterWhenReadyPromise: function (error) {
      if (this._masterWhenReadyPromiseRejecter) {
        this._masterWhenReadyPromiseRejecter(error);
      }
      this._masterWhenReadyPromise = null;
      this._masterWhenReadyPromiseRejecter = null;
      this._masterWhenReadyPromiseResolver = null;
    },
    /**
     * Returns a promise that will resolve when the master promise resolves or reject when
     * the slave timeout promise rejects.
     *
     * @param {Promise} master
     * @param {Function} generateErrorCallback
     * @param {number} timeout
     * @returns {Promise}
     * @ignore
     * @private
     */
    getSlaveTimeoutPromise: function (master, generateErrorCallback, timeout) {
      var timer;
      var slaveTimeoutPromise = new Promise(function (resolve, reject) {
        timer = window.setTimeout(function () {
          reject(generateErrorCallback());
        }, timeout);
      });
      this._slaveTimeoutPromiseTimers.push(timer);

      // When the master promise is resolved, all timers may be cleared
      return Promise.race([master, slaveTimeoutPromise])
                                  .then(this._clearAllSlaveTimeouts.bind(this));
    },

    /**
     * @private
     * @ignore
     * @return {Promise} resolves on the next-tick using setImmediate.
     */
    getNextTickPromise: function () {
      if (!this._nextTickPromise) {
        this._nextTickPromise = new Promise(function (resolve) {
          window.setImmediate(function () {
            this._nextTickPromise = null;
            resolve(true);
          }.bind(this));
        }.bind(this));
      }

      return this._nextTickPromise;
    },

    /**
     * Clears all window timeout timeers that are slave when ready promises.
     *
     * @returns {boolean}
     * @ignore
     * @private
     */
    _clearAllSlaveTimeouts: function () {
      var slaveTimeoutPromiseTimers = this._slaveTimeoutPromiseTimers;
      this._slaveTimeoutPromiseTimers = [];
      for (var i = 0; i < slaveTimeoutPromiseTimers.length; i++) {
        window.clearTimeout(slaveTimeoutPromiseTimers[i]);
      }

      return true;
    },
    /**
     * Promise executor function passed as the single master promise constructor.  Captures the
     * promise resolve callback function.  The resolve promise function will be called when all the
     * busy states have been removed.
     *
     * @param {Function} resolve
     * @param {Function} reject
     * @returns {void}
     * @ignore
     * @private
     */
    // eslint-disable-next-line no-unused-vars
    _captureWhenReadyPromiseResolver: function (resolve, reject) {
      this._masterWhenReadyPromiseResolver = resolve;
      this._masterWhenReadyPromiseRejecter = reject;
    },
    /**
     * Array of setTimeout timers that should be cancled when the busy state resolves.
     *
     * @type {Array.<number>}
     * @ignore
     * @private
     */
    _slaveTimeoutPromiseTimers: []
    /**
     * The master when ready promise that will resovle when all busy states resolve.
     *
     * @type {Promise|undefined}
     * @ignore
     * @private
     */
    // _masterWhenReadyPromise : undefined,
    /**
     * The resolve function of the masterWhenReadyPromise.
     *
     * @type {Function|undefined}
     * @ignore
     * @private
     */
    // _masterWhenReadyPromiseResolver : undefined,
        /**
     * The reject function of the masterWhenReadyPromise.
     *
     * @type {Function|undefined}
     * @ignore
     * @private
     */
    // _masterWhenReadyPromiseRejecter : undefined,
    /**
     * Promise evaluated next-tick.
     *
     * @type {Promise|undefined}
     * @ignore
     * @private
     */
    // _nextTickPromise : undefined
  };
};

/**
 * Logs the current registered busy states ordered acceding by the order they were added.
 * The cost of compiling the list is only made if the logger level is Logger.LEVEL_LOG.
 * @param {?} statesMap busy states
 * @returns {void}
 * @private
 */
oj.BusyContext._log = function (statesMap) {
  if (Logger.option('level') !== Logger.LEVEL_LOG) {
    return;
  }

  Logger.log('>> Busy states: %d', statesMap.size);

  var busyStates = oj.BusyContext._values(statesMap);
  if (busyStates.length > 0) {
    Logger.log(busyStates.join('\n'));
  }
};

/**
 * @param {?} statesMap busy states
 * @return {Array.<oj.BusyState>} Returns an array of busy states entries from the states map
 * @private
 */
oj.BusyContext._values = function (statesMap) {
  var busyStates = [];
  statesMap.forEach(function (value) {
    busyStates.push(value);
  });

  return busyStates;
};

/**
 * <p>Called by components or services performing a task that should be considered
 * in the overall busy state of the page. An example would be animation or fetching data.</p>
 *
 * Caveats:
 * <ul>
 *   <li>Busy context dependency relationships are determined at the point the first busy state
 *       is added.  If the DOM node is re-parented after a busy context was added, the context will
 *       maintain dependencies with any parent DOM contexts.</li>
 *   <li>The application logic creating busy states is responsible for ensuring these busy states
 *       are resolved. Busy states added internally by JET are automatically accounted for.
 *       The busy states added by the application logic must manage a reference to the resolve
 *       function associated with a busy state and it must be called to release the busy state.</li>
 * </ul>
 *
 * <pre class="prettyprint">
 * <code>// apply the marker attribute to the element
 * &lt;div id="context1" data-oj-context ... &gt;&lt;/&gt;
 * ...
 * ...
 * var context1 = document.querySelector("#context1");
 *
 * // obtain a busy context scoped for the target node
 * var busyContext1 = oj.Context.getContext(context1).getBusyContext();
 * // add a busy state to the target context
 * var options = {"description": "#context1 fetching data"};
 * var resolve = busyContext1.addBusyState(options);
 * ...
 * ...  // perform asynchronous operation that needs guarded by a busy state
 * ...
 * // resolve the busy state after the operation completes
 * resolve();
 * </code></pre>
 *
 * @since 2.1.0
 * @export
 * @memberof oj.BusyContext
 * @instance
 * @method addBusyState
 * @param {Object} options object that describes the busy state being registered.<br/>
 * @param {Object|function():string} options.description
 *         description: Option additional information of what is registering a busy state. Added to
 *                      logging and handling rejected status. Can be supplied as a Object or a
 *                      function.  If the type is an object the toString function needs to be
 *                      implemented.
 * @ojsignature [{target: "Type",
 *                value: "{
 *                         toString: ()=>string;
 *                         [propName: string]: any;
 *                       } | (() => string) | string",
 *                for: "options.description"}]
 * @returns {function():void} resolve function called by the registrant when the busy state completes.
 *                     The resultant function will throw an error if the busy state is no longer
 *                     registered.
 */
oj.BusyContext.prototype.addBusyState = function (options) {
  Logger.log("BusyContext.addBusyState: start scope='%s'", this._getDebugScope());

  var statesMap = this._statesMap;

  /** @type {oj.BusyState} */
  var busyState = new oj.BusyState(options[oj.BusyContext._DESCRIPTION]);

  Logger.log('>> ' + busyState);

  statesMap.set(busyState.id, busyState);

  this._addBusyStateToParent();

  Logger.log("BusyContext.addBusyState: end scope='%s'", this._getDebugScope());

  return this._removeBusyState.bind(this, busyState);
};

/**
 * Logs all active busy states to the {@link oj.Logger} at {Logger.LEVEL_INFO}.
 * <pre class="prettyprint">
 * <code>
 *  Logger.option("level", Logger.LEVEL_INFO);
 *  oj.Context.getPageContext().getBusyContext().dump("before popup open");
 * </code></pre>
 *
 * @export
 * @since 3.1.0
 * @memberof oj.BusyContext
 * @instance
 * @method dump
 * @param {string=} message optional text used to further denote a debugging point
 * @return {undefined}
 */
oj.BusyContext.prototype.dump = function (message) {
  Logger.info("BusyContext.dump: start scope='%s' %s", this._getDebugScope(),
                 message || '');

  var statesMap = this._statesMap;
  Logger.info('>> Busy states: %d', statesMap.size);

  var busyStates = oj.BusyContext._values(statesMap);
  if (busyStates.length > 0) {
    Logger.info(busyStates.join('\n'));
  }

  Logger.info("BusyContext.dump: start scope='%s' %s", this._getDebugScope(),
                 message || '');
};

/**
 * Returns an array of states representing the active busy states managed by the instance.
 *
 * @export
 * @since 3.1.0
 * @method getBusyStates
 * @memberof oj.BusyContext
 * @instance
 * @return {Array.<{id:string, description:string}>} active busy states managed by the context
 *         instance
 */
oj.BusyContext.prototype.getBusyStates = function () {
  var statesMap = this._statesMap;

   /** @type {?} */
  var busyStates = oj.BusyContext._values(statesMap);
  return busyStates;
};

/**
 * Forces all busy states per context instance to release.
 * Use with discretion - last course of action.
 *
 * @since 3.1.0
 * @method clear
 * @memberof oj.BusyContext
 * @instance
 * @export
 * @return {undefined}
 */
oj.BusyContext.prototype.clear = function () {
  Logger.log("BusyContext.clear: start scope='%s'", this._getDebugScope());

  var statesMap = this._statesMap;
  var busyStates = oj.BusyContext._values(statesMap);
  for (var i = 0; i < busyStates.length; i++) {
    /** @type {?} **/
    var busyState = busyStates[i];
    try {
      this._removeBusyState(busyState);
    } catch (e) {
      Logger.log('BusyContext.clear: %o', e);
    }

    Object.defineProperty(busyState, oj.BusyContext._OJ_RIP, { value: true, enumerable: false });
  }

  Logger.log("BusyContext.clear: end scope='%s'", this._getDebugScope());
};

/**
 * <p>Returns a Promise that will resolve when all registered busy states have completed or a maximum
 * timeout period has elapsed. The promise will be rejected if all the busy states are not resolved
 * within the timeout period. The busyness of the whenReady promsie is evaluated in the next-tick
 * of resolving a busy state.</p>
 *
 * "next-tick" is at the macrotask level. "whenReady" is waiting for the microtask queue to be exhausted,
 * yielding control flow to the user agent, before resolving busyness.
 *
 * @see oj.BusyContext#applicationBootstrapComplete
 * @since 2.1.0
 * @export
 * @memberof oj.BusyContext
 * @instance
 * @method whenReady
 * @param {number=} timeout "optional" maximum period in milliseconds the resultant promise
 *        will wait. Also see {@link oj.BusyContext.setDefaultTimeout}.
 * @returns {Promise.<boolean|Error>}
 */
oj.BusyContext.prototype.whenReady = function (timeout) {
  var debugScope = this._getDebugScope();

  Logger.log("BusyContext.whenReady: start, scope='%s', timeout=%d", debugScope, timeout);
  /** @type {?} */
  var statesMap = this._statesMap;

  var mediator = this._mediator;
  var nextTickPromise = mediator.getNextTickPromise();
  var bootstrapPromise = oj.BusyContext._BOOTSTRAP_MEDIATOR.whenReady();
  var promise = Promise.all([nextTickPromise, bootstrapPromise]).then(
    function () {
      Logger.log('BusyContext.whenReady: bootstrap mediator ready scope=%s', debugScope);

      try {
        // Since we are executing this code on 'next tick', it is safe to flush any JET throttled updates.
        // Doing so will allow us to take into account any busy states added in response to the pending updates
        oj.BusyContext._deliverThrottledUpdates();
      } catch (e) {
        Logger.error('Fatal exception delivering binding updates: %o', e);
        throw e;
      }

      if (statesMap.size === 0 && !this._waitingOnNextTickBusynessEval) {
        // no busy states, promise resolves immediately
        Logger.log('BusyContext.whenReady: resolved no busy states scope=%s', debugScope);
        return true;
      }

      Logger.log('BusyContext.whenReady: busy states returning master scope=%s', debugScope);
      return mediator.getMasterWhenReadyPromise();
    }.bind(this)
  );

  // if a timeout argument is not provided, check the default timeout
  if (isNaN(timeout) && !isNaN(oj.BusyContext._defaultTimeout)) {
    // eslint-disable-next-line no-param-reassign
    timeout = oj.BusyContext._defaultTimeout;
  }


  if (!isNaN(timeout)) {
    var handleTimeout = function () {
      var error;
      var expiredText = 'whenReady timeout of ' + timeout + 'ms expired ';

      oj.BusyContext._log(statesMap);
      var busyStates = oj.BusyContext._values(statesMap);

      if (!oj.BusyContext._BOOTSTRAP_MEDIATOR.isReady()) {
        error = new Error(expiredText + 'while the application is loading.' +
          ' Busy state enabled by setting the "window.oj_whenReady = true;" global variable.' +
          ' Application bootstrap busy state is released by calling' +
          ' "oj.Context.getPageContext().getBusyContext().applicationBootstrapComplete();".');
      } else {
        error = new Error(expiredText + 'with the following busy states: ' +
                            busyStates.join(', '));
      }

      error.busyStates = busyStates;

      Logger.log("BusyContext.whenReady: rejected scope='%s'\n%s", debugScope, error.message);
      return error;
    };
    promise = mediator.getSlaveTimeoutPromise(promise, handleTimeout, timeout);
  }


  Logger.log("BusyContext.whenReady: end scope='%s'", this._getDebugScope());
  return promise;
};

/**
 * <p>Describes the busyness of the context. The busyness is evaluated in the "next-tick" of a busy
 * state being resolved, meaning the number of busy states doesn't necessarily equate to readiness.
 * The readiness is in sync with the {@link oj.BusyContext#whenReady} resultant promise resolution.</p>
 *
 * @see oj.BusyContext#getBusyStates
 * @since 2.1.0
 * @export
 * @memberof oj.BusyContext
 * @instance
 * @method isReady
 * @returns {boolean} <code>true</code> if the context is not busy
 */
oj.BusyContext.prototype.isReady = function () {
  Logger.log("BusyContext.isReady: start scope='%s'", this._getDebugScope());
  var rtn = false;

  if (oj.BusyContext._BOOTSTRAP_MEDIATOR.isReady() && !this._waitingOnNextTickBusynessEval) {
    var statesMap = this._statesMap;

    rtn = statesMap.size === 0;
    oj.BusyContext._log(statesMap);
  }

  Logger.log("BusyContext.isReady: end scope='%s'", this._getDebugScope());
  return rtn;
};

/**
 * @private
 * @param {oj.BusyState} busyState added busy state
 * @returns {void}
 * @throws {Error} Busy state has already been resolved
 */
oj.BusyContext.prototype._removeBusyState = function (busyState) {
  var debugScope = this._getDebugScope();

  Logger.log("BusyContext._removeBusyState: start scope='%s'", debugScope);

  // The BusyState object is passed here instead of just the generated id to provide a more
  // descriptive message when the busy state is removed twice. The description (if provided) of
  // the busy state will be captured in the error message.

  var statesMap = this._statesMap;

  if (busyState[oj.BusyContext._OJ_RIP]) {
    Logger.log('Busy state has been forcefully resolved via clear:\n' + busyState);
    return;
  } else if (!statesMap.delete(busyState.id)) { // quoted to make the closure compiler happy
    throw new Error('Busy state has already been resolved:\n' + busyState);
  }

  Logger.log('BusyContext._removeBusyState: resolving busy state:\n' + busyState);
  if (statesMap.size === 0 && !this._waitingOnNextTickBusynessEval) {
    // no more busy states; evaluate busyness in the next tick
    this._waitingOnNextTickBusynessEval = true;
    window.setImmediate(this._evalBusyness.bind(this));
  }

  Logger.log("BusyContext._removeBusyState: end scope='%s'", debugScope);
};

/**
 * Evaluates the busyness of the context.
 * @private
 */
oj.BusyContext.prototype._evalBusyness = function () {
  var debugScope = this._getDebugScope();

  Logger.log("BusyContext._evalBusyness: begin scope='%s'", debugScope);

  try {
    // Since we are executing this code on 'next tick', it is safe to flush any JET throttled updates.
    // Doing so will allow us to take into account any busy states added in response to the pending updates
    oj.BusyContext._deliverThrottledUpdates();
  } catch (e) {
    Logger.error('Fatal exception delivering binding updates: %o', e);
    this._mediator.rejectMasterWhenReadyPromise(e);
    this._waitingOnNextTickBusynessEval = false;
    return;
  }

  var statesMap = this._statesMap;
  var mediator = this._mediator;

  // "appears" the Edge promise invokes the resolve callback immediately after
  // resolving versus waiting next micro tick.  Toggle the flag here so if
  // isReady() is called from the promise resolve callback, it returns true.
  this._waitingOnNextTickBusynessEval = false;
  if (statesMap.size === 0) {
    Logger.log('BusyContext._evalBusyness: resolving whenReady promises');

    mediator.resolveMasterWhenReadyPromise();
    this._resolveBusyStateForParent();
  } else {
    oj.BusyContext._log(statesMap);
  }

  Logger.log("BusyContext._evalBusyness: end scope='%s'", debugScope);
};

/**
 * <p>This function should be invoke by application domain logic to indicate all application
 * libraries are loaded and bootstrap processes complete.  The assumed strategy is that the
 * application will set a single global variable "oj_whenReady" from a inline script from the
 * document header section indicating the {@link oj.BusyContext#whenReady}
 * should {@link oj.BusyContext#addBusyState} until the application determines its bootstrap
 * sequence has completed.</p>
 *
 * Inline Script Example:
 * <pre class="prettyprint">
 * <code>
 * &lt;head&gt;
 *   &lt;script type=&quot;text/javascript&quot;&gt;
 *     // The "oj_whenReady" global variable enables a strategy that the busy context whenReady,
 *     // will implicitly add a busy state, until the application calls applicationBootstrapComplete
 *     // on the busy state context.
 *     window["oj_whenReady"] = true;
 *   &lt;/script&gt;
 * ...
 * ...
 * </code></pre>
 *
 * Requirejs callback Example:
 * <pre class="prettyprint">
 * <code>
 * require(['knockout', 'jquery', 'app', 'ojs/ojknockout', 'ojs/ojselectcombobox' ...],
 *   function(ko, $, app)
 *   {
 *     // release the application bootstrap busy state
 *     oj.Context.getPageContext().getBusyContext().applicationBootstrapComplete();
 *     ...
 *     ...
 *   });
 * </code></pre>
 *
 * @since 3.2.0
 * @export
 * @memberof oj.BusyContext
 * @instance
 * @method applicationBootstrapComplete
 * @returns {undefined}
 */
oj.BusyContext.prototype.applicationBootstrapComplete = function () {
  var debugScope = this._getDebugScope();
  Logger.log("BusyContext.applicationBootstrapComplete: begin scope='%s'", debugScope);

  oj.BusyContext._BOOTSTRAP_MEDIATOR.notifyComplete();

  Logger.log("BusyContext.applicationBootstrapComplete: end scope='%s'", debugScope);
};

/**
 * @ignore
 * @private
 * @return {oj.BusyContext} returns the nearest parent context
 */
oj.BusyContext.prototype._getParentBusyContext = function () {
  if (this._hostNode) {
    var parentContext = oj.Context.getContext(oj.Context.getParentElement(this._hostNode));
    if (parentContext) {
      return parentContext.getBusyContext();
    }
  }

  return null;
};

/**
 * Links a child context to its parent by registering a busy state with the parent
 * that will recursively register with its parent.
 *
 * @ignore
 * @private
 */
oj.BusyContext.prototype._addBusyStateToParent = function () {
  if (!this._parentNotified) {
    this._parentNotified = true;

    var parentContext = this._getParentBusyContext();
    if (parentContext) {
      var opts = {};
      opts[oj.BusyContext._DESCRIPTION] = this._getCompoundDescription.bind(this);
      this._parentResolveCallback = parentContext.addBusyState(opts);
    }
  }
};

/**
 * Resolves the busy state linking a child context with its parent.
 *
 * @ignore
 * @private
 */
oj.BusyContext.prototype._resolveBusyStateForParent = function () {
  this._parentNotified = false;
  if (this._parentResolveCallback) {
    this._parentResolveCallback();
    this._parentResolveCallback = null;
  }
};

/**
 * @private
 * @ignore
 * @return {string} description of all active busy states held by the context.
 */
oj.BusyContext.prototype._getCompoundDescription = function () {
  var busyStates = oj.BusyContext._values(this._statesMap);
  return ('[' + busyStates.join(', ') + ']');
};

/**
 * @private
 * @ignore
 * @return {string} context debug scope
 */
oj.BusyContext.prototype._getDebugScope = function () {
  function toSelector(node) {
    var selector = 'undefined';
    if (node) {
      if (node.id && node.id.length > 0) {
        selector = '#' + node.id;
      } else {
        selector = node.nodeName;
        if (node.hasAttribute('data-oj-context')) {
          selector += '[data-oj-context]';
        }

        var clazz = node.getAttribute('class');
        if (clazz) {
          selector += '.' + clazz.split(' ').join('.');
        }
      }
    }

    return selector;
  }

  if (!this._debugScope) {
    if (this._hostNode) {
      this._debugScope = toSelector(this._hostNode.parentElement) + ' > ' +
                         toSelector(this._hostNode);
    } else {
      this._debugScope = 'page';
    }
  }

  return this._debugScope;
};

/**
 * @since 3.1.0
 * @override
 * @memberof oj.BusyContext
 * @instance
 * @method toString
 * @returns {string} returns the value of the object as a string
 */
oj.BusyContext.prototype.toString = function () {
  var msg = 'Busy Context: [scope=';
  msg += this._getDebugScope();
  msg += ' states=' + this._getCompoundDescription() + ']';
  return msg;
};

/**
 * @ignore
 * @private
 */
oj.BusyContext._deliverThrottledUpdates = function () {
  // Dynamically check for the presence of ojs/ojknockout
  if (oj.ComponentBinding) {
    oj.ComponentBinding.deliverChanges();
  }
};

/**
 * @private
 * @ignore
 * @const
 * attribute name describing a busystate
 * @type {string}
 */
oj.BusyContext._DESCRIPTION = 'description';

/**
 * @ignore
 * @private
 * @constant
 * {@link oj.BusyState} property name indicating the instance is dead
 * @type {string}
 */
oj.BusyContext._OJ_RIP = '__ojRip';

/**
 * @ojtsignore
 * @private
 * @ignore
 */
oj.BusyContext._BOOTSTRAP_MEDIATOR = new /** @constructor */(function () {
  var _tracking;
  var _readyPromise;
  var _resolveCallback;

  if (typeof window !== 'undefined') {
    _tracking = window.oj_whenReady;
  }

  this.whenReady = function () {
    if (_readyPromise) {
      return _readyPromise;
    }

    if (!_tracking) {
      _readyPromise = Promise.resolve(true);
    } else {
      _readyPromise = new Promise(function (resolve) {
        _resolveCallback = resolve;
      });
    }
    return _readyPromise;
  };

  this.isReady = function () {
    return !_tracking;
  };

  this.notifyComplete = function () {
    if (_resolveCallback) {
      // resovle the promise in the next-tick.
      window.setImmediate(function () {
        _tracking = false;
        _resolveCallback(true);
        _resolveCallback = null;
      });
    } else {
      _tracking = false;
    }
  };
})();

/**
 * Copyright (c) 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */

/**
 * @preserve Copyright 2013 jQuery Foundation and other contributors
 * Released under the MIT license.
 * http://jquery.org/license
 */

/**
 * Internally used by the {@link oj.BusyContext} to track a components state
 * while it is performing a task such as animation or fetching data.
 *
 * @hideconstructor
 * @ignore
 * @protected
 * @constructor
 * @param {Function|Object|undefined} description of the component and cause
 *        of the busy state
 */
oj.BusyState = function (description) {
  /**
   * @ignore
   * @private
   * @type {?}
   */
  this._description = description;

  /**
   * @ignore
   * @private
   * @type {number}
   */
  this._addedWaitTs = oj.BusyState._getTs();

  /**
   * @ignore
   * @private
   * @type {string}
   */
  this._id = this._addedWaitTs.toString(36) + '_' + Math.random().toString(36); // @RandomNumberOk -
  // random number concatinated to the current timestamp is used for a unique id for a local Map
  // key. This random number is not used use as a cryptography key.
};

Object.defineProperties(oj.BusyState.prototype, {
  /**
   * Identifies the usage instance of a busy state.
   * @memberof oj.BusyState
   * @instance
   * @property {!string} id
   */
  id: {
    get: function () {
      return this._id;
    },
    enumerable: true
  },
  /**
   * Further definition of the busy state instance.
   * @memberof oj.BusyState
   * @instance
   * @property {?string} description
   */
  description:
  {
    get: function () {
      if (this._description) {
        if (this._description instanceof Function) {
          return this._description();
        }
        return this._description.toString();
      }
      return undefined;
    },
    enumerable: true
  }
});

/**
 * @override
 * @returns {string} returns the value of the object as a string
 */
oj.BusyState.prototype.toString = function () {
  var buff = 'Busy state: [description=';

  var description = this.description;

  if (description !== null) {
    buff += description;
  }

  var elapsed = oj.BusyState._getTs() - this._addedWaitTs;
  buff += ', elapsed=' + elapsed + ']';

  return buff;
};


/**
 * @private
 * @returns {number} current date represented by a number
 */
oj.BusyState._getTs = function () {
  // Safari V9.1.1 doesn't yet support performance.now
  return window.performance ? window.performance.now() : new Date().getTime();
};

/**
 * Copyright (c) 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */

/**
 * @preserve Copyright 2013 jQuery Foundation and other contributors
 * Released under the MIT license.
 * http://jquery.org/license
 */

/**
 * <b>The constructor should never be invoked by an application directly</b>. Use
 * {@link oj.Context.getPageContext} and {@link oj.Context.getContext} APIs to
 * retrieve an instance of the context.
 * @param {Element=} node DOM node where the context should be created
 * @export
 * @hideconstructor
 * @ojtsmodule
 * @constructor oj.Context
 * @since 2.1.0
 * @classdesc This is a general purpose context. Initially it only exposes the BusyContext
 * that keeps track of components that are currently animating or fetching data.
 * In the future this context might be expanded for other purposes.
 */
var Context = function (node) {
  this.Init(node);
};

oj.Object.createSubclass(Context, oj.Object, 'oj.Context');

/**
 * @method Init
 * @param {Element=} node DOM node where the context should be created
 * @instance
 * @memberof oj.Context
 * @instance
 * @protected
 */
Context.prototype.Init = function (node) {
  Context.superclass.Init.call(this);
  this._node = node;
};

/**
 * Returns the closest enclosing JET context for a node.
 * Any DOM element may be designated by the page author as a host of JET context.
 * The designation must be expressed in HTML markup by specifying the "data-oj-context"
 * attribute on the host element:

 * <pre class="prettyprint">
 * &lt;div data-oj-context>&lt;div>
 * </pre>
 *
 * <p>This method will walk up the element hierarchy starting with the source node to
 * find an element that has the data-oj-context attribute. If no such element is found,
 * the page context will be returned.</p>
 *
 * If the JET context is established on a particular element, the {@link oj.BusyContext}
 * associated with that context will be tracking busy states for that element and
 * its subtree
 *
 * @see oj.BusyContext for code examples
 * @method getContext
 * @memberof oj.Context
 * @param {Element} node DOM element whose enclosing context will be provided
 * @return {oj.Context} context object scoped per the target node
 * @since 2.2.0
 * @export
 */
Context.getContext = function (node) {
  while (node) {
    var context = node[Context._OJ_CONTEXT_INSTANCE];
    if (context) {
      return context;
    }
    if (node.hasAttribute(Context._OJ_CONTEXT_ATTRIBUTE)) {
      context = new Context(node);
      Object.defineProperty(node, Context._OJ_CONTEXT_INSTANCE,
                                                        { value: context });
      return context;
    }

    // eslint-disable-next-line no-param-reassign
    node = Context.getParentElement(node);
  }

  return Context.getPageContext();
};

/**
 * Static factory method that returns the page context.
 * @see oj.BusyContext for code examples
 * @export
 * @since 2.1.0
 * @method getPageContext
 * @return {oj.Context} context scoped for the page
 * @memberof oj.Context
 */
Context.getPageContext = function () {
  if (!Context._pageContext) { Context._pageContext = new Context(); }

  return Context._pageContext;
};

/**
 * @see oj.BusyContext for code examples
 * @since 2.1.0
 * @export
 * @method getBusyContext
 * @memberof oj.Context
 * @instance
 * @returns {oj.BusyContext} busy state context
 */
Context.prototype.getBusyContext = function () {
  if (!this._busyContext) { this._busyContext = new oj.BusyContext(this._node); }

  return this._busyContext;
};

/**
 * Sets a default for the optional <code>timeout</code> argument of the {@link oj.BusyContext#whenReady}
 * for all BusyContext instances. The default value will be implicitly used if a timeout argument is not
 * provided.
 *
 * @see oj.BusyContext#whenReady
 * @since 6.0.0
 * @memberof oj.Context
 * @method setBusyContextDefaultTimeout
 * @param {number} timeout in milliseconds
 */
Context.setBusyContextDefaultTimeout = function (timeout) {
  oj.BusyContext.setDefaultTimeout(timeout);
};

/**
 * @ignore
 * @private
 * @constant
 * Element marker attribute defining a context
 * @type {string}
 */
Context._OJ_CONTEXT_ATTRIBUTE = 'data-oj-context';

/**
 * @ignore
 * @private
 * @constant
 * Element property name for a context
 * @type {string}
 */
Context._OJ_CONTEXT_INSTANCE = '__ojContextInstance';

/**
 * @ignore
 * @private
 * @constant
 * attribute identifying an open popup
 * @type {string}
 */
Context._OJ_SURROGATE_ATTR = 'data-oj-surrogate-id';

/**
 * @ignore
 * @public
 * @param {Element} element target
 * @return {Element} the logical parent of an element accounting for open popups
 * @memberof oj.Context
 */
Context.getParentElement = function (element) {
  // @see oj.ZOrderUtils._SURROGATE_ATTR in "ojpopupcore/PopupService.js" for the details on how
  // this attribute is used by the popup service. The constant was re-declared to simplify module
  // dependencies.

  if (element && element.hasAttribute(Context._OJ_SURROGATE_ATTR)) {
    var surrogate = document.getElementById(element.getAttribute(Context._OJ_SURROGATE_ATTR));
    if (surrogate) { return surrogate.parentElement; }
  }

  return element.parentElement;
};

/*
** Copyright (c) 2008, 2013, Oracle and/or its affiliates. All rights reserved.
**
**34567890123456789012345678901234567890123456789012345678901234567890123456789
*/

/* jslint browser: true*/
/* global require:false, ojt:true, Promise:false */

/**
 * @namespace oj.Config
 * @hideconstructor
 * @classdesc Services for setting and retrieving configuration options
 * @since 1.0
 * @ojtsmodule
 * @export
 * @ojtsimport {module: "ojcspexpressionevaluator", type: "AMD", importName: "CspExpressionEvaluator"}
 */
var Config = {};

/**
 * Retrieves the type of device the application is running on.
 * @memberof oj.Config
 * @method getDeviceType
 * @return {"phone" | "tablet" | "others"} The device type
 * @export
 */
Config.getDeviceType = function () {
  return oj.AgentUtils.getAgentInfo().deviceType;
};

/**
 * Retrieves the current locale
 * @memberof oj.Config
 * @method getLocale
 * @return {string} current locale
 * @export
 */
Config.getLocale = function () {
  if (oj.__isAmdLoaderPresent()) {
    oj.Assert.assert(typeof ojt !== 'undefined', 'ojtranslations module must be defined');
    var rl = ojt._ojLocale_;

    // If Require.js internationalziation plugin resolved the locale to "root" (presumably because "lang" attribute was not
    // set, and neither navigator.language or navigator.userLanguage were not available), return "en"
    return (rl === 'root') ? 'en' : rl;
  }
  var loc = Config._locale;
  if (loc == null) {
    loc = document.documentElement.lang;
    if (!loc) {
      loc = navigator === undefined ? 'en' :
                            (navigator.language ||
                             navigator.userLanguage || 'en').toLowerCase();
    }
    loc = loc.toLowerCase();
    Config._locale = loc;
  }
  return loc;
};

/**
 * Changes the current locale
 * @method setLocale
 * @param {string} locale (language code and subtags separated by dash)
 * @param {function(): void} [callback] - for applications running with an AMD Loader (such as Require.js), this optional callback
 * will be invoked when the framework is done loading its translated resources and Locale Elements for the newly specified locale.
 * For applications running without an AMD loader, this optional callback will be invoked immediately
 * @return {undefined}
 * @export
 * @memberof oj.Config
 */
Config.setLocale = function (locale, callback) {
  if (oj.__isAmdLoaderPresent()) {
    var prefix = 'ojL10n!ojtranslations/nls/';
    var requestedBundles = [prefix + locale + '/ojtranslations'];

    var timezoneBundleCount = 0;

    // Request LocaleElements only if ojlocaledata module is loaded
    if (oj.LocaleData) {
      requestedBundles.push(prefix + locale + '/localeElements');

      if (oj.TimezoneData) {
        var tzBundles = oj.TimezoneData.__getBundleNames();
        timezoneBundleCount = tzBundles.length;
        tzBundles.forEach(
          function (bundle) {
            requestedBundles.push(prefix + locale + bundle);
          }
        );
      }
    }

    // eslint-disable-next-line global-require
    require(
      /* ojWebpackError: 'oj.Config.setLocale() is not supported when the ojs/ojcore module has been bundled by Webpack' */
      requestedBundles,
      function (translations, localeElements) {
        ojt = translations;

        if (localeElements) {
          oj.LocaleData.__updateBundle(localeElements);
        }

        for (var i = 0; i < timezoneBundleCount; i++) {
          var tzBundle = arguments[i + 2];
          oj.TimezoneData.__mergeIntoLocaleElements(tzBundle);
        }

        if (callback) {
          callback();
        }
      }
    );
  } else {
    Config._locale = locale;
    if (callback) {
      callback();
    }
  }
};

/**
 * Retrieves a URL for loading a component-specific resource.
 * The URL is resolved as follows:
 * 1. If the application has specified a base URL with setResourceBaseUrl(), the return values will be
 * a relative path appended to the base URL.
 * 2. Otherwise, if the application running with an AMD Loader (such as Require.js), the parent folder of a
 * module with ojs/ mapping will be used as a base URL.
 * 3. Otherwise, the original relative path will be returned.
 * @method getResourceUrl
 * @param {string} relativePath resource path
 * @return {string} resource URL
 * @see oj.Config.setResourceBaseUrl
 * @export
 * @memberof oj.Config
 */
Config.getResourceUrl = function (relativePath) {
  // Returning null and full URLs (containing protocol or a leading slash) as is
  var fullUrlExp = /^\/|:/;

  if (relativePath == null || fullUrlExp.test(relativePath)) {
    return relativePath;
  }

  var base = Config._resourceBaseUrl;

  if (base == null) {
    base = Config._getOjBaseUrl() || '';
  }

  var len = base.length;
  return base + (len === 0 || base.charAt(len - 1) === '/' ? '' : '/') + relativePath;
};

/**
 * Sets the base URL for retrieving component-specific resources
 * @method setResourceBaseUrl
 * @param {string} baseUrl base URL
 * @return {undefined}
 * @see oj.Config.getResourceUrl
 * @export
 * @memberof oj.Config
 */
Config.setResourceBaseUrl = function (baseUrl) {
  Config._resourceBaseUrl = baseUrl;
};

/**
 * Sets the automation mode.
 * @method setAutomationMode
 * @param {string} mode "enabled" for running in automation mode
 * @return {undefined}
 * @see oj.Config.getAutomationMode
 * @export
 * @memberof oj.Config
 */
Config.setAutomationMode = function (mode) {
  Config._automationMode = mode;
};

/**
 * Gets the automation mode.
 * @method getAutomationMode
 * @return {string} automation mode
 * @see oj.Config.setAutomationMode
 * @export
 * @memberof oj.Config
 */
Config.getAutomationMode = function () {
  return Config._automationMode;
};

/**
 * Return a string containing important version information about JET and the libraries
 * it has loaded
 * @method getVersionInfo
 * @return {string}
 * @export
 * @memberof oj.Config
 */
Config.getVersionInfo = function () {
  // JET information
  var info = 'Oracle JET Version: ' + oj.version + '\n';
  info += 'Oracle JET Revision: ' + oj.revision + '\n';

  var windowDefined = (typeof window !== 'undefined');

  // Browser information
  if (windowDefined && window.navigator) {
    info += 'Browser: ' + window.navigator.userAgent + '\n';
    info += 'Browser Platform: ' + window.navigator.platform + '\n';
  }

  // 3rd party libraries
  if ($) {
    if ($.fn) {
      info += 'jQuery Version: ' + $.fn.jquery + '\n';
    }
    if ($.ui && $.ui.version) {
      info += 'jQuery UI Version: ' + $.ui.version + '\n';
    }
  }
  if (oj.ComponentBinding) {
    info += 'Knockout Version: ' + oj.ComponentBinding.__getKnockoutVersion() + '\n';
  }

  // Local require doesn't have version #
  if (windowDefined && window.require) {
    info += 'Require Version: ' + window.require.version + '\n';
  }

  return info;
};

/**
 * Dump information to the browser's console containing important version information about JET and
 * the libraries it has loaded
 * @method logVersionInfo
 * @return {undefined}
 * @memberof oj.Config
 * @export
 */
Config.logVersionInfo = function () {
  // eslint-disable-next-line no-console
  console.log(Config.getVersionInfo());
};

/**
 * This method gets replaced by JET's Webpack plugin to return the value provided as baseResourceUrl in the
 * config for the plugin
 * @private
 * @ignore
 */
Config._getOjBaseUrl = function () {
  var base = null;
  if (oj.__isAmdLoaderPresent()) {
    // : use ojs/_foo_ instead of ojs/ojcore to handle the case when ojs.core ends up in a partition bundle
    // in a different location
    var modulePath = require.toUrl('ojs/_foo_');
    base = modulePath.replace(/[^/]*$/, '../');
  }
  return base;
};

/**
 * Retrives JET's template engine for dealing with inline templates (currently internal only)
 * @ignore
 * @memberof oj.Config
 * @private
 */
Config.__getTemplateEngine = function () {
  if (!Config._templateEnginePromise) {
    if (!oj.__isAmdLoaderPresent()) {
      throw new Error('JET Template engine cannot be loaded with an AMD loader');
    }
    Config._templateEnginePromise = new Promise(
      function (resolve, reject) {
        // eslint-disable-next-line global-require
        require(['./ojtemplateengine'], resolve, reject);
      }
    );
  }
  return Config._templateEnginePromise;
};

/**
 * Returns ojtranslation module. Called by oj.Translations, in this way we can make sure that the ojtranslation module
 * instance is shared between Config and Translations.
 * @ignore
 * @memberof oj.Config
 * @private
 */
Config.getConfigBundle = function () {
  return ojt;
};

/**
 * Returns expression evaluator
 * @return {undefined | Object}
 * @ignore
 * @export
 * @memberof oj.Config
 * @since 7.1.0
 */
Config.getExpressionEvaluator = function () {
  return Config._expressionEvaluator;
};

/**
 * Sets an optional CSP-compliant expression evaluator for the JET binding provider and JET ExpressionUtils.
 * This method can only be called once and must be called before applying
 * knockout bindings in the application for the first time.
 * @see <a href="oj.CspExpressionEvaluator.html">CspExpressionEvaluator</a>
 * @method setExpressionEvaluator
 * @param {Object} expressionEvaluator An instance of CspExpressionEvaluator class
 * @return {undefined}
 * @memberof oj.Config
 * @ojshortdesc Sets a CSP-compliant expression evaluator.
 * @export
 * @ojsignature {target:"Type", value: "oj.CspExpressionEvaluator", for: "expressionEvaluator"}
 * @since 7.1.0
 */
Config.setExpressionEvaluator = function (expressionEvaluator) {
  if (Config._expressionEvaluator) {
    throw new Error("JET Expression evaluator can't be set more than once.");
  }

  Config._expressionEvaluator = expressionEvaluator;
};


/*
** Copyright (c) 2015, Oracle and/or its affiliates. All rights reserved.
*/

/* jslint browser: true*/
/* global Logger:false */

/**
 * @namespace oj.ThemeUtils
 * @classdesc Services for getting information from the theme
 * @since 1.2.0
 * @ojtsmodule
 * @export
 */
var ThemeUtils = function () {};

/**
 * get the name of the current theme
 * @method getThemeName
 * @memberof oj.ThemeUtils
 * @export
 * @static
 * @memberof oj.ThemeUtils
 *
 * @return {string|null} the name of the theme
 */
ThemeUtils.getThemeName = function () {
  // get the map of theme info
  var themeMap = ThemeUtils.parseJSONFromFontFamily('oj-theme-json') || {};

  return themeMap.name;
};


/**
 * <p>Get the target platform of the current theme. </p>
 * <p>This API does not look at the user agent and therefore it
 *    tells you nothing about the current platform you are actually on.
 *    Instead it tells you the target platform the theme was written
 *    for so that programmatic behaviors that match the theme's UI can be written.
 *    This can be useful when writing a cross platform hybrid mobile app.  </p>
 *
 * <p>Example</p>
 * <pre class="prettyprint">
 * <code>
 * var themeTargetPlatform = oj.ThemeUtils.getThemeTargetPlatform();
 *
 * if (themeTargetPlatform == 'ios')
 *    // code for a behavior familiar in ios
 * else if (themeTargetPlatform == 'android')
 *    // code for a behavior familiar on android
 * else
 *    // code for the default behavior
 * </code></pre>
 * @export
 * @static
 * @method getThemeTargetPlatform
 * @memberof oj.ThemeUtils
 * @return {string|null} the target platform can be any string the theme
 * wants to send down, but common values are 'all', 'web', 'ios', 'android', 'windows'
 */
ThemeUtils.getThemeTargetPlatform = function () {
  // get the map of theme info
  var themeMap = ThemeUtils.parseJSONFromFontFamily('oj-theme-json') || {};

  return themeMap.targetPlatform;
};


/**
 * clear values cached in  [oj.ThemeUtils.parseJSONFromFontFamily]{@link oj.ThemeUtils.parseJSONFromFontFamily}
 * @export
 * @static
 * @method clearCache
 * @return {void}
 * @memberof oj.ThemeUtils
 */
ThemeUtils.clearCache = function () {
  this._cache = null;
};


/**
 *
 * <p>json can be sent down as the font family in classes
 * that look something like this
 * (on the sass side of things see the file
 * scss/utilities/_oj.utilities.json.scss
 * for information on jet mixins available to generate json):<p>
 *
 * <p>Example CSS</p>
 * <pre class="prettyprint">
 * <code>
 * .demo-map-json {
 *    font-family: '{"foo":"bar", "binky": 4}';
 * }
 *
 * .demo-list-json {
 *    font-family: '["foo","bar","binky"}';
 * }
 * </code></pre>
 * <p>Example Usage</p>
 * <pre class="prettyprint">
 * <code>
 * var mymap = oj.ThemeUtils.parseJSONFromFontFamily("demo-map-json");
 * var myarray = oj.ThemeUtils.parseJSONFromFontFamily("demo-list-json");
 * </code></pre>
 *
 * This function
 * <ul>
 *   <li>Gets the font family string by creating a dom element,
 *      applying the selector passed in, calling getcomputedstyle,
 *      and then reading the value for font-family.
 *   </li>
 *   <li>Parses the font family value by calling JSON.pars.</li>
 *   <li>Caches the parsed value because calling getComputedStyle is a perf hit.
 *       Subsequent requests for the same selector will return the cached value.
 *       Call [oj.ThemeUtils.clearCache]{@link oj.ThemeUtils.clearCache} if new css is loaded.</li>
 *   <li>Return the parsed value.</li>
 * </ul>
 *
 * <p>
 * If new css is loaded call oj.ThemeUtils.clearCache to clear the cache</p>
 *
 * @method parseJSONFromFontFamily
 * @memberof oj.ThemeUtils
 * @param {string} selector a class selector name, for example 'demo-map-json';
 * @return {any} the result of parsing the font family with JSON.parse.
 *      The returned value is cached, so if you modify the returned
 *      value it will be reflected in the cache.
 * @throws {SyntaxError} If JSON.parse throws a SyntaxError exception we will log an error and rethrow
 * @export
 * @static
 */
ThemeUtils.parseJSONFromFontFamily = function (selector) {
  // NOTE: I first tried code inspired by
  // https://css-tricks.com/making-sass-talk-to-javascript-with-json/
  // so I was using :before and content, for example
  //   .oj-button-option-defaults:before {
  //     content: '{"foo":"bar", "binky": 4}';
  //    }
  //
  //  however IE 11 has a bug where the computed style doesn't actually
  //  seem to be computed when it comes to :before,
  //  so if you set a class that affects :before after the page loads
  //  on IE getComputedStyle doesn't work.
  //  See the pen below, the yellow box has the class applied in js,
  //  computedstyle works on chrome, doesn't work on IE11 for
  //  class applied after page load.
  //     http://codepen.io/gabrielle/pen/OVOwev

  // if needed create the cache and initialize some things
  if (this._cache == null) {
    this._cache = {};

    // magic value that means null in the cache
    this._null_cache_value = {};

    // font family is inherited, so even if no selector/json
    // is sent down we will get a string like
    // 'HelveticaNeue',Helvetica,Arial,sans-serif' off of our generated element.
    // So save off the font family from the head
    // element to compare to what we read off our generated element.
    this._headfontstring = window.getComputedStyle(document.head).getPropertyValue('font-family');
  }

  // see if we already have a map for this component's option defaults
  var jsonval = this._cache[selector];

  // if there's something already cached return it
  if (jsonval === this._null_cache_value) {
    return null;
  } else if (jsonval != null) {
    return jsonval;
  }


  // there's nothing cached, so we need to create a dom element to apply the class to.
  // We're creating a meta element,
  // the hope is that the browser is smart enough to realize the
  // meta element isn't visible and therefore we avoid perf issues of calling
  // getcomputedstyle
  var elem = document.createElement('meta');
  elem.className = selector;
  document.head.appendChild(elem); // @HTMLUpdateOK
  var rawfontstring = window.getComputedStyle(elem).getPropertyValue('font-family');

  if (rawfontstring != null) {
    // if the raw font string is the same value as the saved header
    // font value then log a warning that no value was sent down.

    if (rawfontstring === this._headfontstring) {
      Logger.warn('parseJSONFromFontFamily: When the selector ', selector,
        ' is applied the font-family read off the dom element is ', rawfontstring,
        '. The parent dom elment has the same font-family value.',
        ' This is interpreted to mean that no value was sent down for selector ',
        selector, '. Null will be returned.');
    } else {
      // remove inconsistent quotes
      var fontstring = rawfontstring.replace(/^['"]+|\s+|\\|(;\s?})+|['"]$/g, '');
      // console.log("json fontstring for selector " + selector + ': ' + fontstring);

      if (fontstring) {
        try {
          jsonval = JSON.parse(fontstring);
        } catch (e) {
          // In Firefox you can turn off the page font
          // Options -> Content -> Fonts and Colors -> Advanced
          // Uncheck the "Allow pages to choose their own fonts, instead of my selections above"
          // In that case they stick something like 'serif,"'at the front of the font family,
          // so search for the first comma, add 2, and try parsing again.
          var commaindex = fontstring.indexOf(',');
          var reparseSuccess = false;

          if (commaindex > -1) {
            fontstring = fontstring.substring(commaindex + 2);

            try {
              jsonval = JSON.parse(fontstring);
              reparseSuccess = true;
            } catch (e2) {
              // Ignore Error
            }
          }

          if (reparseSuccess === false) {
            Logger.error('Error parsing json for selector ' + selector +
                            '.\nString being parsed is ' + fontstring + '. Error is:\n', e);

            // remove the meta tag
            document.head.removeChild(elem); // @HTMLUpdateOK
            throw e;
          }
        }
      }
    }
  }

  // remove the meta tag
  document.head.removeChild(elem); // @HTMLUpdateOK

  // cache the result
  if (jsonval == null) {
    this._cache[selector] = this._null_cache_value;
  } else {
    this._cache[selector] = jsonval;
  }

  // console.log(this._cache);

  return jsonval;
};


/* global Promise:false */

/**
 * Timing related utilities
 * @namespace
 * @name oj.TimerUtils
 * @since 4.1.0
 * @ojstatus preview
 * @ojtsignore
 */
var TimerUtils = {};

/**
 * A Timer encapsulates a Promise associated with a deferred function execution
 * and the ability to cancel the timer before timeout.
 * @interface Timer
 * @ojtsignore
 */
function Timer() {}
 /**
  * Get the Promise assocaited with this timer.  Promise callbacks will be
  * passed a single boolean value indicating if the timer's timeout expired
  * normally (without being canceled/cleared).  If the timer is left to expire
  * after its configured timeout has been exceeded, then it will pass
  * boolean(true) to the callbacks.  If the timer's {@link #clear} method is
  * called before its configured timeout has been reached, then the callbacks
  * will receive boolean(false).
  * @memberof Timer
  * @return {Promise.<boolean>} This timer's Promise
  */
Timer.prototype.getPromise = function () {};
 /**
  * Clears the timer and resolves the Promise.  If the normal timeout hasn't
  * yet expired, the value passed to the Promise callbacks will be boolean(false).
  * If the timeout has already expired, this function will do nothing, and all of
  * its Promise callbacks will receive boolean(true).
  * @return {void}
  * @memberof Timer
  */
Timer.prototype.clear = function () {};

/**
 * Get a Timer object with the given timeout in milliseconds.  The Promise
 * associated with the timer is resolved when the timeout window expires, or if
 * the clear() function is called.
 * This is useful for when code needs to be executed on timeout (setTimeout) and
 * must handle cleanup tasks such as clearing {@link BusyState} when the timer
 * expires or is canceled.
 *
 * @param  {number} timeout The timeout value in milliseconds to wait before the
 * promise is resolved.
 * @return {Timer}
 * A Timer object which encapsulates the Promise that will be
 * resolved once the timeout has been exceeded or cleared.
 * @export
 * @memberof oj.TimerUtils
 * @alias getTimer
 * @example <caption>Get a timer to execute code on normal timeout and
 * cancelation.  If the timeout occurs normally (not canceled), both
 * callbacks are executed and the value of the 'completed' parameter will be
 * true.</caption>
 * var timer = oj.TimerUtils.getTimer(1000);
 * timer.getPromise().then(function(completed) {
 *     if (completed) {
 *       // Delayed code
 *     }
 *   })
 * timer.getPromise().then(function() {
 *   // Code always to be run
 * })
 *
 * @example <caption>Get a timer to execute code on normal timeout and cancelation.
 * In this example, the timer is canceled before its timeout expires, and the
 * value of the 'completed' parameter will be false.</caption>
 * var timer = oj.TimerUtils.getTimer(1000);
 * timer.getPromise()
 *   .then(function(completed) {
 *     if (completed) {
 *       // Delayed code
 *     }
 *   })
 * timer.getPromise()
 *   .then(function() {
 *     // Code always to be run
 *   })
 * ...
 * timer.clear(); // timer cleared before timeout expires
 */
TimerUtils.getTimer = function (timeout) {
  return new TimerUtils._TimerImpl(timeout);
};

/**
 * @constructor
 * @implements {Timer}
 * @param  {number} timeout The timeout value in milliseconds.
 * @private
 */
TimerUtils._TimerImpl = function (timeout) {
  var _promise;
  var _resolve;
  var _timerId;

  this.getPromise = function () {
    return _promise;
  };
  this.clear = function () {
    window.clearTimeout(_timerId);
    _timerId = null;
    _timerDone(false);
  };

  /**
   * Called on normal and early timeout (cancelation)
   */
  function _timerDone(completed) {
    _timerId = null;
    _resolve(completed);
  }

  if (typeof window === 'undefined') {
    _promise = Promise.reject();
  } else {
    _promise = new Promise(function (resolve) {
      _resolve = resolve;
      _timerId = window.setTimeout(_timerDone.bind(null, true), timeout);
    });
  }
};

/*
** Copyright (c) 2014, Oracle and/or its affiliates. All rights reserved.
*/

/* jslint browser: true*/
/* global Logger:false */

/**
 * @namespace oj.ResponsiveUtils
 * @classdesc Utilities for working with the framework's responsive screen widths
 * and ranges. Often used in conjunction with {@link oj.ResponsiveKnockoutUtils}
 * to create knockout observables that can be used to drive responsive page behavior.
 * See the method doc below for specific examples.
 * @ojtsmodule
 * @since 1.1.0
 * @hideconstructor
 * @export
 */
var ResponsiveUtils = function () {};


/**
 * <p>In the jet sass files there are variables for
 * responsive screen widths, these look something like</p>
 *  <ul>
 *    <li>$screenSmallRange:  0, 767px;</li>
 *    <li>$screenMediumRange: 768px, 1023px;</li>
 *    <li>$screenLargeRange:  1024px, 1280px;</li>
 *    <li>$screenXlargeRange: 1281px, null;</li>
 *  </ul>
 *
 * <p>These constants are used to identify these ranges.</p>
 * @enum {string}
 * @memberof oj.ResponsiveUtils
 * @alias SCREEN_RANGE
 * @constant
 * @export
 */
ResponsiveUtils.SCREEN_RANGE = {
  /**
   * @expose
   * @constant
   */
  SM: 'sm',
  /**
   * @expose
   * @constant
   */
  MD: 'md',
  /**
   * @expose
   * @constant
   */
  LG: 'lg',
  /**
   * @expose
   * @constant
   */
  XL: 'xl',
  /**
   * @expose
   * @constant
   */
  XXL: 'xxl'

};


/**
 * <p>In the jet sass files there are variables for
 * responsive screen widths,
 * see {@link oj.ResponsiveUtils.SCREEN_RANGE} for details.
 * The jet sass files also has variables for
 * responsive queries like $responsiveQuerySmallUp,
 * $responsiveQuerySmallOnly, $responsiveQueryMediumUp, etc.</p>
 *
 * <p>These constants are used to identify these queries.</p>
 * @enum {string}
 * @memberof oj.ResponsiveUtils
 * @alias FRAMEWORK_QUERY_KEY
 * @constant
 * @export
 */
ResponsiveUtils.FRAMEWORK_QUERY_KEY = {
  /**
   * Matches screen width small and wider
   * @expose
   * @constant
   */
  SM_UP: 'sm-up',
  /**
   * matches screen width medium and wider
   * @expose
   * @constant
   */
  MD_UP: 'md-up',
  /**
   * matches screen width large and wider
   * @expose
   * @constant
   */
  LG_UP: 'lg-up',
  /**
   * matches screen width extra-large and wider
   * @expose
   * @constant
   */
  XL_UP: 'xl-up',
  /**
   * matches screen width extra-extra-large and wider
   * @expose
   * @constant
   */
  XXL_UP: 'xxl-up',

  /**
   * matches screen width small only
   * @expose
   * @constant
   */
  SM_ONLY: 'sm-only',
  /**
   * matches screen width medium only
   * @expose
   * @constant
   */
  MD_ONLY: 'md-only',
  /**
   * matches screen width large only
   * @expose
   * @constant
   */
  LG_ONLY: 'lg-only',
  /**
   * matches screen width extra-large only
   * @expose
   * @constant
   */
  XL_ONLY: 'xl-only',

  /**
   * matches screen width medium and narrower
   * @expose
   * @constant
   */
  MD_DOWN: 'md-down',
  /**
   * matches screen width large and narrower
   * @expose
   * @constant
   */
  LG_DOWN: 'lg-down',
  /**
   * matches screen width extra-large and narrower
   * @expose
   * @constant
   */
  XL_DOWN: 'xl-down',
  /**
   * matches high resolution screens
   * @expose
   * @constant
   */
  HIGH_RESOLUTION: 'high-resolution'

};

// used by the compare function
ResponsiveUtils._RANGE = {};
ResponsiveUtils._RANGE[ResponsiveUtils.SCREEN_RANGE.SM] = 0;
ResponsiveUtils._RANGE[ResponsiveUtils.SCREEN_RANGE.MD] = 1;
ResponsiveUtils._RANGE[ResponsiveUtils.SCREEN_RANGE.LG] = 2;
ResponsiveUtils._RANGE[ResponsiveUtils.SCREEN_RANGE.XL] = 3;
ResponsiveUtils._RANGE[ResponsiveUtils.SCREEN_RANGE.XXL] = 4;


/**
 * This idea/code is from zurb foundation, thanks zurb!
 *
 * In the jet sass files there are variables for
 * responsive screen sizes, these look something like

 *  <ul>
 *    <li>$screenSmallRange:  0, 767px;</li>
 *    <li>$screenMediumRange: 768px, 1023px;</li>
 *    <li>$screenLargeRange:  1024px, 1280px;</li>
 *    <li>$screenXlargeRange: 1281px, null;</li>
 *  </ul>
 *
 * <p>These variables in turn are used to generate responsive media queries in variables like
 * $responsiveQuerySmallUp, $responsiveQueryMediumUp, etc.</p>
 *
 * <p>we send down these media queries as the font family in classes
 * that look something like this:<p>
 *
 * <pre class="prettyprint">
 * <code>
 * .oj-mq-md {
 *    font-family: "/screen and (min-width: 768px)/";
 * }
 * </code></pre>
 *
 * <p>This function applies the class and then reads the font family off a dom
 * element to get the media query string</p>
 *
 * @param {string} selector a class selector name, for example 'oj-mq-md';
 * @return {string} the media query sent down for that class
 * @private
 */
ResponsiveUtils._getMediaQueryFromClass = function (selector) {
  var elem = /** @type {(Element | null)} */ (document.getElementsByClassName(selector).item(0));

  if (elem === null) {
    elem = document.createElement('meta');
    elem.className = selector;
    document.head.appendChild(elem); // @HTMLUpdateOK
  }

  var fontFamily = window.getComputedStyle(elem).getPropertyValue('font-family');

  return fontFamily.replace(/^[/\\'"]+|(;\s?})+|[/\\'"]+$/g, '');
};

/**
 * Get a framework (built in) media query string,
 * see {@link oj.ResponsiveUtils.FRAMEWORK_QUERY_KEY} for details on framework queries.
 * The media query string returned can be passed to
 * {@link oj.ResponsiveKnockoutUtils.createMediaQueryObservable} to create a knockout
 * observable, which in turn can be used to drive responsive page behavior.
 *
 * <p>Example:</p>
 * <pre class="prettyprint">
 * <code>
 *
 *     var lgQuery = oj.ResponsiveUtils.getFrameworkQuery(
 *                             oj.ResponsiveUtils.FRAMEWORK_QUERY_KEY.LG_UP);
 *
 *     self.large = oj.ResponsiveKnockoutUtils.createMediaQueryObservable(lgQuery);
 * </code></pre>
 *
 *
 * @method getFrameworkQuery
 * @memberof oj.ResponsiveUtils
 * @param {oj.ResponsiveUtils.FRAMEWORK_QUERY_KEY} frameworkQueryKey one of the FRAMEWORK_QUERY_KEY constants,
 *                       for example oj.ResponsiveUtils.FRAMEWORK_QUERY_KEY.MD_UP
 * @return {string | null} the media query to use for the framework query key passed in
 * @export
 * @static
 */
ResponsiveUtils.getFrameworkQuery = function (frameworkQueryKey) {
  var selector = 'oj-mq-' + frameworkQueryKey;
  var query = ResponsiveUtils._getMediaQueryFromClass(selector);

  if (query === 'null') {
    Logger.warn('Framework query not found. Please check that the value of the theming variable' +
                '$includeResponsiveMediaQueryClasses is set to true, if it' +
                'is set to false the media queries are not sent down to the browser.');
    return null;
  }
  return query;
};


/**
 * <p> Compare can be used in conjunction with
 * {@link oj.ResponsiveKnockoutUtils.createScreenRangeObservable}</p>
 *
 *
 * <p>Example:</p>
 * <pre class="prettyprint">
 * <code>
 *        // create an observable which returns the current screen range
 *        self.screenRange = oj.ResponsiveKnockoutUtils.createScreenRangeObservable();
 *
 *        self.label2 = ko.computed(function() {
 *          var range = self.screenRange();
 *
 *          if ( oj.ResponsiveUtils.compare(
 *                       range, oj.ResponsiveUtils.SCREEN_RANGE.MD) <= 0)
 *          {
 *            // code for when screen is in small or medium range
 *          }
 *          else if (range == oj.ResponsiveUtils.SCREEN_RANGE.XL)
 *          {
 *            // code for when screen is in XL range
 *          }
 *        });
 * </code></pre>
 *
 * @method compare
 * @memberof oj.ResponsiveUtils
 * @param {oj.ResponsiveUtils.SCREEN_RANGE} size1 one of the screen size constants,
 * for example oj.ResponsiveUtils.SCREEN_RANGE.MD
 * @param {oj.ResponsiveUtils.SCREEN_RANGE} size2 one of the screen size constants,
 * for example oj.ResponsiveUtils.SCREEN_RANGE.LG
 * @return {number} a negative integer if the first
 * argument is less than the second. Zero if the two are equal.
 * 1 or greater if the first argument is more than the second.
 *
 * @export
 * @static
 */
ResponsiveUtils.compare = function (size1, size2) {
  var range1 = ResponsiveUtils._RANGE[size1];
  var range2 = ResponsiveUtils._RANGE[size2];

  if (range1 == null) {
    throw new Error('size1 param ' + size1 +
                    ' illegal, please use one of the screen size constants like oj.ResponsiveUtils.SCREEN_RANGE.MD');
  }
  if (range2 == null) {
    throw new Error('size2 param ' + size2 +
                    ' illegal, please use one of the screen size constants like oj.ResponsiveUtils.SCREEN_RANGE.MD');
  }

  return range1 - range2;
};

/*
** Copyright (c) 2008, 2013, Oracle and/or its affiliates. All rights reserved.
**
**34567890123456789012345678901234567890123456789012345678901234567890123456789
*/

/* global Config:false */

/**
 * @namespace oj.Translations
 * @classdesc Services for Retrieving Translated Resources
 * @export
 * @since 1.0
 * @ojtsmodule
 * @hideconstructor
 */
var Translations = {};

/**
 * Sets the translation bundle used by JET
 * If an AMD loader (such as Require.js) is not present, this method should be called by the application to provide
 * translated strings for JET.
 * This method may also be used by an application that wants to completely replace the resource bundle that is automatically
 * fetched by an AMD loader.
 * @method setBundle
 * @memberof oj.Translations
 * @param {Object} bundle resource bundle that should be used by the framework
 * @return {void}
 * @export
 */
Translations.setBundle = function (bundle) {
  Translations._bundle = bundle;
};

/**
 * Retrives a translated resource for a given key
 * @method getResource
 * @memberof oj.Translations
 * @param {string} key resource key The dot character (.) within the key string
 * is interpreted as a separator for the name of a sub-section within the bundle.
 * For example, 'components.chart', would be read as the 'chart' section within
 * 'components'. Thus the key name for an individual section should never contain a dot.
 * @return {Object|string|null} resource associated with the key or null if none was found
 * @export
 */
Translations.getResource = function (key) {
  return Translations._getResourceString(key);
};

/**
 * Applies parameters to a format pattern
 * @method applyParameters
 * @memberof oj.Translations
 * @param {string} pattern pattern that may contain tokens like {0}, {1}, {name}. These tokens
 * will be used to define string keys for retrieving values from the parameters
 * object. Token strings should not contain comma (,)
 * or space characters, since they are reserved for future format type enhancements.
 * The reserved characters within a pattern are:
 * $ { } [ ]
 * These characters will not appear in the formatted output unless they are escaped
 * with a dollar character ('$').
 *
 * @param {Object|Array} parameters parameters to be inserted into the string. Both arrays and
 * Javascript objects with string keys are accepted.
 *
 * @return {string|null} formatted message or null if the pattern argument was null
 * @export
 */
Translations.applyParameters = function (pattern, parameters) {
  return (pattern == null) ? null : Translations._format(pattern, parameters);
};

/**
 * Retrieves a translated string after inserting optional parameters.
 * The method uses a key to retrieve a format pattern from the resource bundle.
 * Tokens like {0}, {1}, {name} within the pattern will be used to define placement
 * for the optional parameters.  Token strings should not contain comma (,)
 * or space characters, since they are reserved for future format type enhancements.
 * The reserved characters within a pattern are:
 * $ { } [ ]
 * These characters will not appear in the formatted output unless they are escaped
 * with a dollar character ('$').
 * @method getTranslatedString
 * @memberof oj.Translations
 * @param {string} key  translations resource key. The dot character (.) within the key string
 * is interpreted as a separator for the name of a sub-section within the bundle.
 * For example, 'components.chart', would be read as the 'chart' section within
 * 'components'. Thus the key name for an individual section should never contain a dot.
 *
 * @param {...(string|Object|Array)} var_args  - optional parameters to be inserted into the
 * translated pattern.
 *
 * If more than one var_args arguments are passed, they will be treated as an array
 * for replacing positional tokens like {0}, {1}, etc.
 * If a single argument is passed, it will be treated as a Javascript Object whose
 * keys will be matched to tokens within the pattern. Note that an Array is just
 * a special kind of such an Object.
 *
 * For backward compatibility, a var_args argument whose type is neither
 * Object or Array will be used to replace {0} in the pattern.
 *
 * @return formatted translated string
 * @ojsignature {target: "Type", for:"returns", value: "string"}
 * @export
 */
// eslint-disable-next-line camelcase, no-unused-vars
Translations.getTranslatedString = function (key, var_args) {
  var val = Translations._getResourceString(key);

  if (val == null) {
    return key;
  }

  var params = {};

  if (arguments.length > 2) {
    params = Array.prototype.slice.call(arguments, 1);
  } else if (arguments.length === 2) {
    params = arguments[1];
    if (typeof params !== 'object' && !(params instanceof Array)) {
      params = [params];
    }
  }

  return Translations.applyParameters(val, params);
};


/**
 * Provides a key-to-value map of the translated resources for a given component name
 * @method getComponentTranslations
 * @memberof oj.Translations
 * @param {string} componentName name of the component
 * @return {Object} a map of translated resources
 * @export
 */
Translations.getComponentTranslations = function (componentName) {
  var bundle = Translations._getBundle()[componentName];

  if (bundle == null) {
    return {};
  }

  // Assume that the set of keys remains constant regardless of the current locale
  var translations = {};
  var keys = Object.keys(bundle);
  for (var k = 0; k < keys.length; k++) {
    var key = keys[k];
    translations[key] = bundle[key];
  }

  return translations;
};

/**
 * Retrives a translated resource for a given key, accounting for nested keys
 * @param {string} key
 * @return {string|null} resource associated with the key or null if none was found
 * @private
 */
Translations._getResourceString = function (key) {
  // Account for dot separated nested keys
  var keys = key ? key.split('.') : [];
  var bundle = Translations._getBundle();
  oj.Assert.assertObject(bundle);

  // even though we start with a valid bundle it's possible that part or all of the key is invalid,
  // so check we have a valid bundle in the for loop
  // if we have a key like a.b.c
  for (var index = 0; index < keys.length && bundle; index++) {
    var subkey = keys[index];
    bundle = bundle[subkey];
  }

  return bundle || null;
};

Translations._format = function (formatString, parameters) {
  var formatLength = formatString.length;

  // Use the javascript StringBuffer technique.
  var buffer = [];

  var token = null;


  var escaped = false;
  var isToken = false;
  var isGroup = false;
  var isExcluded = false;

  var tokenTerminated; // this will be set to true when a comma or space is
                       // encountered in teh token
  var i;

  for (i = 0; i < formatLength; i++) {
    var ch = formatString.charAt(i);

    var accumulate = false;

    if (!escaped) {
      switch (ch) {
        case '$':
          escaped = true;
          break;

        case '{':
          if (!isExcluded) {
            if (!isToken) {
              tokenTerminated = false;
              token = [];
            }
            isToken = true;
          }
          break;

        case '}':
          if (isToken && token.length > 0) {
            var val = parameters[token.join('')];
            buffer.push((val === undefined) ? 'null' : val);
          }
          isToken = false;
          break;

        case '[':
          if (!isToken) {
            if (isGroup) {
              isExcluded = true;
            } else {
              isGroup = true;
            }
          }
          break;

        case ']':
          if (isExcluded) {
            isExcluded = false;
          } else {
            isGroup = false;
          }
          break;

        default:
          accumulate = true;
      }
    } else {
      accumulate = true;
      escaped = false;
    }

    if (accumulate) {
      if (isToken) {
        if (ch === ',' || ch === ' ') {
          tokenTerminated = true;
        } else if (!tokenTerminated) {
          token.push(ch);
        }
      } else if (!isExcluded) {
        buffer.push(ch);
      }
    }
  }

  // Use the javascript StringBuffer technique for toString()
  return buffer.join('');
};


Translations._getBundle = function () {
  var b = Translations._bundle;
  if (b) {
    return b;
  }

  if (oj.__isAmdLoaderPresent()) {
    var ojt = Config.getConfigBundle();
    oj.Assert.assert(ojt !== undefined, 'ojtranslations module must be defined');
    return ojt;
  }
  return {};
};

/**
 * @ojoverviewdoc ModuleLoadingOverview - [5]JET Module Loading
 * @classdesc
 * {@ojinclude "name":"moduleLoadingOverviewDoc"}
 */
/**
 * <h2 id="usage">Overview
 *   <a class="bookmarkable-link" title="Bookmarkable Link" href="#overview"></a>
 * </h2>
 * <p>
 *  JET classes and components are delivered via a set of <a href="https://github.com/amdjs/amdjs-api/wiki/AMD">asynchronous module definitions</a> (AMDs or more informally, modules).
 *  JET applications typically use <a href="https://requirejs.org/">RequireJS</a> to load the necessary modules and call API as required.  The values returned from JET modules come in one of three forms:
 * </p>
 * <h4>No return value</h4>
 *  <p>Some modules may not return any value at all.  The purpose of these modules is simply to load the associated JavaScript into memory,
 *     but the application typically does not directly interact with or instantiate this code. For example, modules that define
 *     JET Web Components typically would not have return values.
 *  </p>
 * <pre class="prettyprint"><code>
 * //Loading a JET component in your Typescript code
 *
 * //To typecheck the element APIs, import as below.
 * import {ojAccordion} from "ojs/ojaccordion";
 *
 * //For the transpiled javascript to load the element's module, import as below
 * import "ojs/ojaccordion";</code></pre>
 * <h4>One return value</h4>
 *  <p>Some modules directly return a single object or constructor function.  Applications would typically call functions on the returned
 *     object or instantiate new objects via the constructor function.  For example, the 'ojs/ojcontext' module
 *     has a single return value:
 *  </p>
 * <pre class="prettyprint"><code>
 * //Javascript example
 * define(['ojs/ojcontext'], function(Context) {
 *   var pageContext = Context.getPageContext();
 * })</code></pre>
 * <pre class="prettyprint"><code>
 * //TypeScript example
 * import Context = require('ojs/ojcontext');
 *   let pageContext = Context.getPageContext();
 * </code></pre>
 * <h4>Multiple return values</h4>
 *  <p>Some modules package several objects or constructor functions inside a single JavaScript object.  Applications would typically retrieve the relevant object or constructor function via a
 *     documented property on this object and then either call functions or instantiate new objects as appropriate.  For example, the 'ojs/ojvalidation-base' module has multiple return values.
 *  </p>
 * <pre class="prettyprint"><code>
 * //Javascript example
 * define(['ojs/ojvalidation-base'], function(validationBase) {
 *  var colorConverterFactory = validationBase.Validation.converterFactory('color');
 *  var colorConverter = new validationBase.ColorConverter({format: 'hsl'});
 * })</code></pre>
 * <pre class="prettyprint"><code>
 * //TypeScript example
 * import {Validation, LengthValidatorFactory} from "ojs/ojvalidation-base";
 *  let factory = Validation.validatorFactory('length') as LengthValidatorFactory;
 *  let validator = factory.createValidator({max: 10});
 *  myTextComponent.validators = [validator];
 *
 * //Another example
 * import * as Logger from "ojs/ojlogger";
 *  Logger.log("Please enter a valid input");
 * </code></pre>
 *
 * @ojfragment moduleLoadingOverviewDoc
 * @memberof ModuleLoadingOverview
 */

/* global Logger:false, Context:false, Config:false, ResponsiveUtils:false, ThemeUtils:false, TimerUtils:false, Translations:false */

oj.Logger = Logger;
oj.Context = Context;
oj.Config = Config;
oj.ResponsiveUtils = ResponsiveUtils;
oj.ThemeUtils = ThemeUtils;
oj.TimerUtils = TimerUtils;
oj.Translations = Translations;

/* global Set:false, Symbol:false */

/**
 * Contains all the core functionalities of KeySet.
 * @param {(Set|Array)=} keys A set of keys to initialize this KeySet with.
 * @ojstatus preview
 * @ignore
 * @ojtsignore
 * @export
 * @class KeySetImpl
 * @constructor
 * @since 5.1.0
 */
// eslint-disable-next-line no-unused-vars
var KeySetImpl = function (initialValues) {
  this.NOT_A_KEY = {};

  /**
   * Returns whether the specified key is contained in this set.
   * @private
   * @param {any} key the key to check whether it is contained in this set.
   * @return {boolean} true if the specified key is contained in this set, false otherwise.
   */
  this.has = function (key) {
    return (this.get(key) !== this.NOT_A_KEY);
  };

  /**
   * Finds the equavalent key of the specified key within this KeySet.
   * @private
   * @param {any} keyToFind the key to find
   * @return {any} the key in the key that is equivalent to keyToFind, or NO_KEY if nothing equivalent can be found.
   */
  this.get = function (keyToFind) {
    var iterator;
    var key;
    var found = this.NOT_A_KEY;
    var self = this;

    if (this._keys.has(keyToFind)) {
      return keyToFind;
    }

    // if it's a primitive, then we are done also
    if (!(keyToFind === Object(keyToFind))) {
      return this.NOT_A_KEY;
    }

    // using iterator if it's supported since we could break at any time
    if (typeof Symbol === 'function' && typeof Set.prototype[Symbol.iterator] === 'function') {
      iterator = this._keys[Symbol.iterator]();
      key = iterator.next();
      while (!key.done) {
        if (oj.KeyUtils.equals(key.value, keyToFind)) {
          return key.value;
        }
        key = iterator.next();
      }
    } else {
      // IE11 supports forEach
      this._keys.forEach(function (_key) {
        if (found === self.NOT_A_KEY && oj.KeyUtils.equals(_key, keyToFind)) {
          found = _key;
        }
      });
    }

    return found;
  };

  /**
   * Initialize the internal Set with a set of keys.
   * @private
   * @param {Set|Array|null|undefined} keys the initial keys to create the internal Set with.
   */
  this.InitializeWithKeys = function (keys) {
    this._keys = new Set(keys);
  };

  this.InitializeWithKeys(initialValues);
};

/* global KeySetImpl:false, Set:false */

/**
 * An immutable set of keys.
 * @class KeySet
 * @ojstatus preview
 * @classdesc The base class for ExpandedKeySet and ExpandAllKeySet.  Represents an immutable set of keys.
 * @constructor
 * @hideconstructor
 * @abstract
 * @since 4.1.0
 * @ojsignature {target: "Type", value: "abstract class KeySet<K>", genericParameters: [{"name": "K", "description": "Type of Key"}]}
 */
var KeySet = function () {};

// Subclass from oj.Object
oj.Object.createSubclass(KeySet, oj.Object, 'KeySet');

// make it available internally
oj.KeySet = KeySet;

/**
 * Sets the internal Set object.
 *
 * @param {Set} set the internal Set object to replace with.
 * @protected
 */
KeySet.prototype.SetInternal = function (set) {
  this._keys = set;
};

/**
 * Returns a new KeySet based on this set with the specified keys included.
 *
 * @param {Set|Array} keys a set of keys to add to this KeySet.
 * @return {KeySet} a new KeySet with the specified keys included.
 * @method
 * @name add
 * @memberof KeySet
 * @instance
 * @abstract
 * @ojsignature {target: "Type", value: "(keys: Set<K>|Array<K>): KeySet<K>"}
 */

/**
 * Returns a new KeySet that signals all keys are added to this set.
 *
 * @return {KeySet} a new KeySet that signals all keys are added to this set.
 * @method
 * @name addAll
 * @memberof KeySet
 * @instance
 * @abstract
 * @ojsignature {target: "Type", value: "(): KeySet<K>"}
 */

/**
 * Returns a new KeySet based on this set with the specified keys excluded.
 *
 * @param {Set|Array} keys a set of keys to remove from this KeySet.
 * @return {KeySet} a new KeySet with the specified keys excluded.
 * @method
 * @name delete
 * @memberof KeySet
 * @instance
 * @abstract
 * @ojsignature {target: "Type", value: "(keys: Set<K>|Array<K>): KeySet<K>"}
 */

/**
 * Returns whether this set should include all keys.
 *
 * @return {boolean} true if this set should include all keys, false otherwise.
 * @method
 * @name isAddAll
 * @memberof KeySet
 * @instance
 * @abstract
 */

/**
 * Returns whether the specified key is contained in this set.
 *
 * @param {any} key the key to check whether it is contained in this set.
 * @return {boolean} true if the specified key is contained in this set, false otherwise.
 * @method
 * @name has
 * @memberof KeySet
 * @instance
 * @abstract
 * @ojsignature {target: "Type", value: "K", for: "key"}
 */

/**
 * Returns a new KeySet containing no keys.
 *
 * @return {KeySet} a new KeySet with no keys.
 * @method
 * @name clear
 * @memberof KeySet
 * @instance
 * @abstract
 * @ojsignature {target: "Type", value: "(): KeySet<K>"}
 */

/**
 * Adds or deletes a set of keys from the internal Set object.
 * @param {boolean} isAdd true if add operation, false if delete operation
 * @param {Set|Array} keys keys to add or delete
 * @return {KeySet} returns current KeySet if add or delete is not performed, or a new KeySet with the
 *                     specified keys included (add) or excluded (delete).
 * @protected
 */
KeySet.prototype.AddOrDeleteInternal = function (isAdd, keys) {
  var newSet;
  var keySet;

  newSet = isAdd ? this._add(keys) : this._remove(keys);
  if (newSet == null) {
    return this;
  }

  keySet = /** @type {KeySet} */ (Object.create(Object.getPrototypeOf(this)));
  keySet.SetInternal(newSet);
  return keySet;
};

/**
 * Adds the specified keys to the internal Set object.
 * @param {Set|Array} keys a set of keys to add
 * @return {Set} a new Set based on this internal Set with the specified keys appended to the end, or null if nothing was added.
 * @private
 */
KeySet.prototype._add = function (keys) {
  var self = this;
  var newSet = null;

  keys.forEach(function (key) {
        // checks if it's already contained in the Set, can't use has() since it does a reference comparison
    if (key !== self.NOT_A_KEY && self.get(key) === self.NOT_A_KEY) {
      if (newSet == null) {
        newSet = self.Clone();
      }
      newSet.add(key);
    }
  });

  return newSet;
};

/**
 * Helper method to remove the specified keys from its set
 * @param {Set|Array} keys an interable set of keys to remove
 * @return {Set|null} a new Set based on this internal Set with the keys removed, or null if nothing is removed.
 * @private
 */
KeySet.prototype._remove = function (keys) {
  var self = this;
  var newSet = null;
  var keyToDelete;

  // first check if there's anything to remove
  if (this._keys.size === 0) {
    return null;
  }

  keys.forEach(function (key) {
    // see if we can find a equivalent key in this Set since delete does a reference comparison to find the item to delete
    keyToDelete = self.get(key);
    if (keyToDelete !== self.NOT_A_KEY) {
      if (newSet == null) {
        newSet = self.Clone();
      }
      newSet.delete(keyToDelete);
    }
  });

  return newSet;
};

/**
 * Returns the size of this Set.
 * @return {number} the size of this Set.
 * @protected
 */
KeySet.prototype.GetInternalSize = function () {
  return this._keys.size;
};

/**
 * Return a clone of the internal Set
 * @return {Set} the clone of the internal Set
 * @protected
 */
KeySet.prototype.Clone = function () {
  return new Set(this._keys);
};

KeySetImpl.call(KeySet.prototype);

/* global KeySet:false, ExpandAllKeySet:false */

/**
 * Create a new immutable KeySet containing the keys of the expanded items.
 * Use this KeySet when specifying individual keys to expand.
 *
 * @param {(Set|Array)=} keys A set of keys to initialize this KeySet with.
 * @ojstatus preview
 * @class ExpandedKeySet
 * @classdesc The ExpandedKeySet class contains a set of keys of the expanded items.  See
 * also the <a href="ObservableExpandedKeySet.html">observable</a> version of this class.
 * @extends {KeySet}
 * @constructor
 * @since 4.1.0
 * @ojdeprecated {since: '7.0.0', description: 'Use KeySetImpl instead.'}
 * @ojsignature [{target: "Type", value: "class ExpandedKeySet<K> extends KeySet<K>", genericParameters: [{"name": "K", "description": "Type of Key"}]},
 *               {target: "Type", value: "Set<K>|Array<K>", for:"keys"}]
 * @example <caption>Creates a new ExpandedKeySet with an initial set of keys to expand:</caption>
 * require(['ojs/ojkeyset'],
 *   function(keySet) {
 *     var expandedKeySet = new keySet.ExpandedKeySet(['group1', 'group3']);
 *   }
 * );
 */
var ExpandedKeySet = function (keys) {
  this.InitializeWithKeys(keys);
};

// Subclass from KeySet
oj.Object.createSubclass(ExpandedKeySet, KeySet, 'ExpandedKeySet');

// make it available internally
oj.ExpandedKeySet = ExpandedKeySet;

/**
 * Returns a new KeySet based on this set with the specified keys included.
 * If none of the keys specified are added then this KeySet is returned.
 *
 * @param {Set|Array} keys a set of keys to add to this KeySet.
 * @return {ExpandedKeySet} a new KeySet with the specified keys included.
 * @expose
 * @instance
 * @alias add
 * @memberof ExpandedKeySet
 * @ojsignature {target: "Type", value: "(keys: Set<K>|Array<K>): ExpandedKeySet<K>"}
 */
ExpandedKeySet.prototype.add = function (keys) {
  return /** @type {!ExpandedKeySet} */ (this.AddOrDeleteInternal(true, keys));
};

/**
 * Returns a new KeySet that signals all keys are added to this set.
 *
 * @return {ExpandAllKeySet} a new KeySet that signals all keys are added to this set.
 * @expose
 * @instance
 * @alias addAll
 * @memberof ExpandedKeySet
 * @ojsignature {target: "Type", value: "(): ExpandAllKeySet<K>"}
 */
ExpandedKeySet.prototype.addAll = function () {
  return new ExpandAllKeySet();
};

/**
 * Returns whether this set should include all keys.
 *
 * @return {boolean} true if this set includes all keys, false otherwise.
 * @expose
 * @instance
 * @alias isAddAll
 * @memberof ExpandedKeySet
 */
ExpandedKeySet.prototype.isAddAll = function () {
  return false;
};

/**
 * Returns a new KeySet based on this set with the specified keys excluded.
 * If none of the keys specified are deleted then this KeySet is returned.
 *
 * @param {Set|Array} keys a set of keys to remove from this KeySet.
 * @return {ExpandedKeySet} a new KeySet with the specified keys excluded.
 * @expose
 * @instance
 * @alias delete
 * @memberof ExpandedKeySet
 * @ojsignature {target: "Type", value: "(keys: Set<K>|Array<K>): ExpandedKeySet<K>"}
 */
ExpandedKeySet.prototype.delete = function (keys) {
  return /** @type {!ExpandedKeySet} */ (this.AddOrDeleteInternal(false, keys));
};

/**
 * Returns a new KeySet containing no keys.  If this KeySet already contains no keys then
 * the current KeySet is returned.
 *
 * @return {ExpandedKeySet} a new KeySet with no keys.
 * @expose
 * @instance
 * @alias clear
 * @memberof ExpandedKeySet
 * @ojsignature {target: "Type", value: "(): ExpandedKeySet<K>"}
 */
ExpandedKeySet.prototype.clear = function () {
  return this.GetInternalSize() === 0 ? this : new ExpandedKeySet();
};

/**
 * Returns whether the specified key is contained in this set.
 *
 * @param {any} key the key to check whether it is contained in this set.
 * @return {boolean} true if the specified key is contained in this set, false otherwise.
 * @expose
 * @instance
 * @alias has
 * @memberof ExpandedKeySet
 * @ojsignature {target: "Type", value: "K", for:"key"}
 */
ExpandedKeySet.prototype.has = function (key) {
  return (this.get(key) !== this.NOT_A_KEY);
};

/**
 * Returns the keys in this KeySet in the order they are added.
 *
 * @return {Set} the keys in this KeySet in the order they are added.
 * @expose
 * @instance
 * @alias values
 * @memberof ExpandedKeySet
 * @ojsignature {target: "Type", value: "Set<K>", for:"returns"}
 */
ExpandedKeySet.prototype.values = function () {
  return this.Clone();
};

/* global KeySet:false, ExpandedKeySet:false */

/**
 * Create a new immutable KeySet containing the keys of the collapsed items.
 * Use this KeySet when expanding all keys.
 *
 * @ojstatus preview
 * @class ExpandAllKeySet
 * @classdesc The ExpandAllKeySet class represents a set with all keys expanded.
 * @extends {KeySet}
 * @constructor
 * @since 4.1.0
 * @ojdeprecated {since: '7.0.0', description: 'Use AllKeySetImpl instead.'}
 * @ojsignature {target: "Type", value: "class ExpandAllKeySet<K> extends KeySet<K>",
 *               genericParameters: [{"name": "K", "description": "Type of Key"}]}
 * @example <caption>Creates a new ExpandAllKeySet to expand all keys</caption>
 * require(['ojs/ojkeyset'],
 *   function(keySet) {
 *     var expandAllKeySet = new keySet.ExpandAllKeySet();
 *   }
 * );
 */
var ExpandAllKeySet = function () {
  this.InitializeWithKeys(null);
};

// Subclass from KeySet
oj.Object.createSubclass(ExpandAllKeySet, KeySet, 'ExpandAllKeySet');

// make it available internally
oj.ExpandAllKeySet = ExpandAllKeySet;

/**
 * Returns a new KeySet with the specified keys excluded from a set of collapsed keys.
 * If the keys specified are already added then this KeySet is returned.
 *
 * @param {Set|Array} keys a set of keys to add to this KeySet.
 * @return {ExpandAllKeySet} a new KeySet with the specified keys included.
 * @expose
 * @instance
 * @alias add
 * @memberof! ExpandAllKeySet
 * @ojsignature {target: "Type", value: "(keys: Set<K>|Array<K>): ExpandAllKeySet<K>"}
 */
ExpandAllKeySet.prototype.add = function (keys) {
  // add keys on expand all = remove collapsed keys
  return /** @type {!ExpandAllKeySet} */ (this.AddOrDeleteInternal(false, keys));
};

/**
 * Returns a new KeySet that signals all keys are added to this set.  If this KeySet already
 * has all keys added, then this KeySet is returned.
 *
 * @return {ExpandAllKeySet} a new KeySet that signals all keys are added to this set.
 * @expose
 * @instance
 * @alias addAll
 * @memberof! ExpandAllKeySet
 * @ojsignature {target: "Type", value: "(): ExpandAllKeySet<K>"}
 */
ExpandAllKeySet.prototype.addAll = function () {
  return this.GetInternalSize() === 0 ? this : new ExpandAllKeySet();
};

/**
 * Returns whether this set should include all keys.
 *
 * @return {boolean} true if this set includes all keys, false otherwise.
 * @expose
 * @instance
 * @alias isAddAll
 * @memberof! ExpandAllKeySet
 */
ExpandAllKeySet.prototype.isAddAll = function () {
  return true;
};

/**
 * Returns a new KeySet based on this set with the specified keys included in a set of collapsed keys.
 * If the keys specified are already deleted then this KeySet is returned.
 *
 * @param {Set|Array} keys a set of keys to remove from this KeySet.
 * @return {ExpandAllKeySet} a new KeySet with the specified keys excluded.
 * @expose
 * @instance
 * @alias delete
 * @memberof! ExpandAllKeySet
 * @ojsignature {target: "Type", value: "(keys: Set<K>|Array<K>): ExpandAllKeySet<K>"}
 */
ExpandAllKeySet.prototype.delete = function (keys) {
  // remove keys on expand all = add collapsed keys
  return /** @type {!ExpandAllKeySet} */ (this.AddOrDeleteInternal(true, keys));
};

/**
 * Returns a new KeySet containing no keys.
 *
 * @return {ExpandedKeySet} a new KeySet with no keys.
 * @expose
 * @instance
 * @alias clear
 * @memberof! ExpandAllKeySet
 * @ojsignature {target: "Type", value: "(): ExpandedKeySet<K>"}
 */
ExpandAllKeySet.prototype.clear = function () {
  return new ExpandedKeySet();
};

/**
 * Returns whether the specified key is contained in this set.
 *
 * @param {any} key the key to check whether it is contained in this set.
 * @return {boolean} true if the specified key is contained in this set, false otherwise.
 * @expose
 * @instance
 * @alias has
 * @memberof! ExpandAllKeySet
 * @ojsignature {target: "Type", value: "K", for: "key"}
 */
ExpandAllKeySet.prototype.has = function (key) {
  return (this.get(key) === this.NOT_A_KEY);
};

/**
 * Returns a set of keys of the collapsed items.
 *
 * @return {Set} the keys of the collapsed items.
 * @expose
 * @instance
 * @alias deletedValues
 * @memberof! ExpandAllKeySet
 * @ojsignature {target: "Type", value: "Set<K>", for: "returns"}
 */
ExpandAllKeySet.prototype.deletedValues = function () {
  return this.Clone();
};

/* global KeySet:false, AllKeySetImpl:false */

/**
 * Create a new immutable KeySet containing the keys of items.
 * Use this KeySet when specifying individual keys to select or expand.
 *
 * @param {(Set|Array)=} keys A set of keys to initialize this KeySet with.
 * @ojstatus preview
 * @class KeySetImpl
 * @classdesc The KeySetImpl class contains a set of keys of items.
 * @extends {KeySet}
 * @constructor
 * @since 7.0.0
 * @ojsignature [{target: "Type", value: "class KeySetImpl<K> extends KeySet<K>", genericParameters: [{"name": "K", "description": "Type of Key"}]},
 *               {target: "Type", value: "Set<K>|Array<K>", for:"keys"}]
 * @example <caption>Creates a new KeySetImpl with an initial set of keys:</caption>
 * require(['ojs/ojkeyset'],
 *   function(keySet) {
 *     var KeySetImpl = new keySet.KeySetImpl(['item1', 'item3']);
 *   }
 * );
 */
var KeySetImpl = function (keys) {
  this.InitializeWithKeys(keys);
};

// Subclass from KeySet
oj.Object.createSubclass(KeySetImpl, KeySet, 'KeySetImpl');

// make it available internally
oj.KeySetImpl = KeySetImpl;

/**
 * Returns a new KeySet based on this set with the specified keys included.
 * If none of the keys specified are being added, then this KeySet is returned.
 *
 * @param {Set|Array} keys a set of keys to add to this KeySet.
 * @return {KeySetImpl} a new KeySet with the specified keys included.
 * @expose
 * @instance
 * @alias add
 * @memberof KeySetImpl
 * @ojsignature {target: "Type", value: "(keys: Set<K>|Array<K>): KeySetImpl<K>"}
 */
KeySetImpl.prototype.add = function (keys) {
  return /** @type {!KeySetImpl} */ (this.AddOrDeleteInternal(true, keys));
};

/**
 * Returns a new KeySet that represents a set with all keys.
 *
 * @return {AllKeySetImpl} a new KeySet that represents a set with all keys.
 * @expose
 * @instance
 * @alias addAll
 * @memberof KeySetImpl
 * @ojsignature {target: "Type", value: "(): AllKeySetImpl<K>"}
 */
KeySetImpl.prototype.addAll = function () {
  return new AllKeySetImpl();
};

/**
 * Determines whether this is a set that represents all keys.
 *
 * @return {boolean} true if this is a set that reprsents all keys, false otherwise.
 * @expose
 * @instance
 * @alias isAddAll
 * @memberof KeySetImpl
 */
KeySetImpl.prototype.isAddAll = function () {
  return false;
};

/**
 * Returns a new KeySet based on this set with the specified keys excluded.
 * If none of the keys specified are being deleted, then this KeySet is returned.
 *
 * @param {Set|Array} keys a set of keys to remove from this KeySet.
 * @return {KeySetImpl} a new KeySet with the specified keys excluded.
 * @expose
 * @instance
 * @alias delete
 * @memberof KeySetImpl
 * @ojsignature {target: "Type", value: "(keys: Set<K>|Array<K>): KeySetImpl<K>"}
 */
KeySetImpl.prototype.delete = function (keys) {
  return /** @type {!KeySetImpl} */ (this.AddOrDeleteInternal(false, keys));
};

/**
 * Returns a new KeySet containing no keys.  If this KeySet already contains no keys then
 * the current KeySet is returned.
 *
 * @return {KeySetImpl} a new KeySet with no keys.
 * @expose
 * @instance
 * @alias clear
 * @memberof KeySetImpl
 * @ojsignature {target: "Type", value: "(): KeySetImpl<K>"}
 */
KeySetImpl.prototype.clear = function () {
  return this.GetInternalSize() === 0 ? this : new KeySetImpl();
};

/**
 * Determines whether the specified key is in this set.
 *
 * @param {any} key the key to check whether it is in this set.
 * @return {boolean} true if the specified key is in the set, false otherwise.
 * @expose
 * @instance
 * @alias has
 * @memberof KeySetImpl
 * @ojsignature {target: "Type", value: "K", for:"key"}
 */
KeySetImpl.prototype.has = function (key) {
  return (this.get(key) !== this.NOT_A_KEY);
};

/**
 * Returns the keys in this KeySet in the order they are added.
 *
 * @return {Set} the keys in this KeySet in the order they are added.
 * @expose
 * @instance
 * @alias values
 * @memberof KeySetImpl
 * @ojsignature {target: "Type", value: "Set<K>", for:"returns"}
 */
KeySetImpl.prototype.values = function () {
  return this.Clone();
};

/* global KeySet:false, KeySetImpl:false */

/**
 * Create a new immutable KeySet that represents a set with all keys.
 * Use this KeySet when select or expand all keys.
 *
 * @ojstatus preview
 * @class AllKeySetImpl
 * @classdesc The AllKeySetImpl class represents a set with all keys.
 * @extends {KeySet}
 * @constructor
 * @since 7.0.0
 * @ojsignature {target: "Type", value: "class AllKeySetImpl<K> extends KeySet<K>", genericParameters: [{"name": "K", "description": "Type of Key"}]}
 * @example <caption>Creates a new AllKeySetImpl to select all keys</caption>
 * require(['ojs/ojkeyset'],
 *   function(keySet) {
 *     var AllKeySetImpl = new keySet.AllKeySetImpl();
 *   }
 * );
 */
var AllKeySetImpl = function () {
  this.InitializeWithKeys(null);
};

// Subclass from KeySet
oj.Object.createSubclass(AllKeySetImpl, KeySet, 'AllKeySetImpl');

// make it available internally
oj.AllKeySetImpl = AllKeySetImpl;

/**
 * Returns a new KeySet with the specified keys included in the set.  Specifically,
 * the specified keys will be deleted from the currently excluded keys.
 * If the keys specified are already added then this KeySet is returned.
 *
 * @param {Set|Array} keys a set of keys to add to this KeySet.
 * @return {AllKeySetImpl} a new KeySet with the specified keys included.
 * @expose
 * @instance
 * @alias add
 * @memberof! AllKeySetImpl
 * @ojsignature {target: "Type", value: "(keys: Set<K>|Array<K>): AllKeySetImpl<K>"}
 */
AllKeySetImpl.prototype.add = function (keys) {
  // add keys on all = remove deleted keys
  return /** @type {!AllKeySetImpl} */ (this.AddOrDeleteInternal(false, keys));
};

/**
 * Returns a new KeySet that represents a set with all keys.  If this KeySet already is
 * a set with all keys, then this would just return itself.
 *
 * @return {AllKeySetImpl} a new KeySet that represents a set with all keys.
 * @expose
 * @instance
 * @alias addAll
 * @memberof! AllKeySetImpl
 * @ojsignature {target: "Type", value: "(): AllKeySetImpl<K>"}
 */
AllKeySetImpl.prototype.addAll = function () {
  return this.GetInternalSize() === 0 ? this : new AllKeySetImpl();
};

/**
 * Determines whether this is a set that represents all keys.
 *
 * @return {boolean} true if this is a set that reprsents all keys, false otherwise.
 * @expose
 * @instance
 * @alias isAddAll
 * @memberof! AllKeySetImpl
 */
AllKeySetImpl.prototype.isAddAll = function () {
  return true;
};

/**
 * Returns a new KeySet based on this set with the specified keys deleted.  Specifically,
 * the returned KeySet represents all keys except for the keys deleted.
 * If the keys specified are already deleted then this KeySet is returned.
 *
 * @param {Set|Array} keys a set of keys to remove from this KeySet.
 * @return {AllKeySetImpl} a new KeySet with the specified keys excluded.
 * @expose
 * @instance
 * @alias delete
 * @memberof! AllKeySetImpl
 * @ojsignature {target: "Type", value: "(keys: Set<K>|Array<K>): AllKeySetImpl<K>"}
 */
AllKeySetImpl.prototype.delete = function (keys) {
  // remove keys on all = add to excluded keys
  return /** @type {!AllKeySetImpl} */ (this.AddOrDeleteInternal(true, keys));
};

/**
 * Returns a new KeySet containing no keys.
 *
 * @return {KeySetImpl} a new KeySet with no keys.
 * @expose
 * @instance
 * @alias clear
 * @memberof! AllKeySetImpl
 * @ojsignature {target: "Type", value: "(): KeySetImpl<K>"}
 */
AllKeySetImpl.prototype.clear = function () {
  return new KeySetImpl();
};

/**
 * Determines whether the specified key is in this set.
 *
 * @param {any} key the key to check whether it is in this set.
 * @return {boolean} true if the specified key is in this set, false otherwise.
 * @expose
 * @instance
 * @alias has
 * @memberof! AllKeySetImpl
 * @ojsignature {target: "Type", value: "K", for: "key"}
 */
AllKeySetImpl.prototype.has = function (key) {
  return (this.get(key) === this.NOT_A_KEY);
};

/**
 * Returns a set of keys of the items that are excluded from this set.
 *
 * @return {Set} the keys of the deleted items.
 * @expose
 * @instance
 * @alias deletedValues
 * @memberof! AllKeySetImpl
 * @ojsignature {target: "Type", value: "Set<K>", for: "returns"}
 */
AllKeySetImpl.prototype.deletedValues = function () {
  return this.Clone();
};

/**
 * Copyright (c) 2019, Oracle and/or its affiliates.
 * All rights reserved.
 */
/* global AllKeySetImpl:false, KeySetImpl:false */

/**
 * Contains a set of utility methods for working with KeySet.
 * @class
 * @ignore
 * @tsignore
 */
var KeySetUtils = {};

/**
 * Converts a KeySet into an array
 */
KeySetUtils.toArray = function (keyset) {
  var arr;

  var set = keyset.isAddAll() ? keyset.deletedValues() : keyset.values();
  if (Array.from) {
    arr = Array.from(set);
  } else {
    // IE11 does not support Array.from
    arr = [];
    set.forEach(function (value) {
      arr.push(value);
    });
  }
  arr.inverted = keyset.isAddAll();
  return arr;
};

/**
 * Converts an array into a KeySet
 */
KeySetUtils.toKeySet = function (arr) {
  var keyset = arr.inverted ? new AllKeySetImpl() : new KeySetImpl();
  return keyset.add(arr);
};

/**
 * Creates an attribute group handler that will generate stylistic attribute values such as colors or shapes based on data set categories.
 *
 * @param {Object.<string, *>=} [matchRules] A map of key value pairs for categories and the matching attribute value e.g. {"soda" : "square", "water" : "circle", "iced tea" : "triangleUp"}.
 *                            Attribute values listed in the matchRules object will be reserved only for the matching categories when getAttributeValue is called.
 * @ojsignature {target: "Type", value: "{[propName: string]: any}", for: "matchRules"}
 * @export
 * @constructor
 * @since 1.0
 * @name oj.AttributeGroupHandler
 */
var AttributeGroupHandler = function (matchRules) {
  this.Init(matchRules);
};

oj.Object.createSubclass(AttributeGroupHandler, oj.Object, 'AttributeGroupHandler');

AttributeGroupHandler.prototype.Init = function (matchRules) {
  this._assignments = {};
  this._valueIndex = 0;
  this._matchRules = {};
  if (matchRules != null) {
    var categories = Object.keys(matchRules);
    for (var i = 0; i < categories.length; i++) {
      var category = categories[i];
      this.addMatchRule(category, matchRules[category]);
    }
  }
  // Delay initializing value ramp by calling subclass getValueRamp impl until needed either for adding match rule or assigning values
};

/**
 * Returns the array of possible attribute values for this attribute group handler.
 * @returns {Array.<any>} The array of attribute values
 * @memberof oj.AttributeGroupHandler
 * @method getValueRamp
 * @instance
 * @export
 */
AttributeGroupHandler.prototype.getValueRamp = function () {
  return [];
};

/**
 * Assigns the given category an attribute value.  Will consistently return the same attribute value for equal categories.
 * @param {string} category The category to assign
 * @returns {any} The attribute value for the category
 * @memberof oj.AttributeGroupHandler
 * @method getValue
 * @instance
 * @export
 */
AttributeGroupHandler.prototype.getValue = function (category) {
  // When assigning value, first check match rules, then assign to next attribute group value.
  if (this._matchRules[category]) {
    return this._matchRules[category];
  } else if (!this._assignments[category]) {
    if (!this._values) {
      this._values = this.getValueRamp().slice();
    }

    this._assignments[category] = this._values[this._valueIndex];

    this._valueIndex += 1;
    if (this._valueIndex === this._values.length) {
      this._valueIndex = 0;
    }
  }
  return this._assignments[category];
};

/**
 * Returns the current list of assigned categories as an array of objects with "category" and "value" keys. Note that match rules are not
 * reflected in category assignments.
 * @return {Array.<Object.<string, *>>} The current list of category and value pairings
 * @ojsignature {target: "Type", value: "Array<{[propName: string]: any}>", for: "returns"}
 * @export
 * @memberof oj.AttributeGroupHandler
 * @method getCategoryAssignments
 * @instance
 */
AttributeGroupHandler.prototype.getCategoryAssignments = function () {
  var assignments = [];
  var categories = Object.keys(this._assignments);
  for (var i = 0; i < categories.length; i++) {
    var category = categories[i];
    assignments.push({ category: category, value: this._assignments[category] });
  }
  return assignments;
};

/**
 * Reserves an attribute value for the given category.  All match rules should be added before any category
 * assignments are done with the <a href="#getValue">getValue</a> API.
 * @param {string} category Used for checking inputs to getAttributeValue against when assigning an attribute value
 * @param {any} attributeValue The attribute value to assign for inputs matching the given category e.g. "square" or "circle"
 * @return {void}
 * @export
 * @memberof oj.AttributeGroupHandler
 * @method addMatchRule
 * @instance
 */
AttributeGroupHandler.prototype.addMatchRule = function (category, attributeValue) {
  this._matchRules[category] = attributeValue;
};

/**
 * Creates a color attribute group handler that will generate color attribute values.
 *
 * @param {Object.<string, string>=} [matchRules] A map of key value pairs for categories and the
 * matching attribute value e.g. {"soda" : "#336699", "water" : "#CC3300", "iced tea" : "#F7C808"}.
 * Attribute values listed in the matchRules object will be reserved only for the
 * matching categories when getAttributeValue is called.  Note that not all colors
 * in the default color ramp will meet minimum contrast requirements for text.
 * @ojsignature {target: "Type", value: "{[propName: string]: any}", for: "matchRules"}
 * @export
 * @constructor
 * @since 1.2
 * @extends oj.AttributeGroupHandler
 * @name oj.ColorAttributeGroupHandler
 */
var ColorAttributeGroupHandler = function (matchRules) {
  // Create the array of colors for this instance.
  this._attributeValues = [];

  if ($(document.body).hasClass('oj-hicontrast')) {
    // High Contrast: CSS colors get overridden to all white or all black. Use the default colors instead.
    this._attributeValues = ColorAttributeGroupHandler._DEFAULT_COLORS.slice();
  } else {
    // Process the colors from the skin if not done already.
    // To improve performance, append the divs for each style class first then process the colors for each div.
    var attrGpsDiv = ColorAttributeGroupHandler.__createAttrDiv();
    if (attrGpsDiv) {
      ColorAttributeGroupHandler.__processAttrDiv(attrGpsDiv);
      attrGpsDiv.remove();
    }

    // Clone and use the processed colors.
    if (ColorAttributeGroupHandler._colors.length > 0) {
      this._attributeValues = ColorAttributeGroupHandler._colors.slice();
    } else {
      this._attributeValues = ColorAttributeGroupHandler._DEFAULT_COLORS.slice();
    }
  }

  this.Init(matchRules);
};
// eslint-disable-next-line no-undef
oj.Object.createSubclass(ColorAttributeGroupHandler, AttributeGroupHandler,
                         'ColorAttributeGroupHandler');

/** @private */
ColorAttributeGroupHandler._DEFAULT_COLORS = [
  '#237bb1', '#68c182', '#fad55c', '#ed6647',
  '#8561c8', '#6ddbdb', '#ffb54d', '#e371b2',
  '#47bdef', '#a2bf39', '#a75dba', '#f7f37b'
];

/** @private */
ColorAttributeGroupHandler._STYLE_CLASSES = [
  'oj-dvt-category1', 'oj-dvt-category2', 'oj-dvt-category3',
  'oj-dvt-category4', 'oj-dvt-category5', 'oj-dvt-category6',
  'oj-dvt-category7', 'oj-dvt-category8',
  'oj-dvt-category9', 'oj-dvt-category10', 'oj-dvt-category11', 'oj-dvt-category12'
];

/** @private */
ColorAttributeGroupHandler._colors = null;

/**
 * Returns the array of possible color values for this attribute group handler.
 * @returns {Array.<string>} The array of color values
 * @export
 * @method getValueRamp
 * @memberof oj.ColorAttributeGroupHandler
 * @instance
 */
ColorAttributeGroupHandler.prototype.getValueRamp = function () {
  return this._attributeValues;
};

/**
 * Creates an element and appends a div for each style class
 * @returns {jQuery} The jQuery element containing divs for each style class
 * @ignore
 */
ColorAttributeGroupHandler.__createAttrDiv = function () {
  if (ColorAttributeGroupHandler._colors) {
    return null;
  }

  var attrGpsDiv = $(document.createElement('div'));
  attrGpsDiv.css('display', 'none;');
  attrGpsDiv.attr('id', 'attrGps');
  $(document.body).append(attrGpsDiv); // @HTMLUpdateOK
  for (var i = 0; i < ColorAttributeGroupHandler._STYLE_CLASSES.length; i++) {
    var childDiv = $(document.createElement('div'));
    childDiv.addClass(ColorAttributeGroupHandler._STYLE_CLASSES[i]);
    attrGpsDiv.append(childDiv); // @HTMLUpdateOK
  }
  return attrGpsDiv;
};

/**
 * Processes the colors for each div on the given element
 * @param {jQuery} attrGpsDiv The jQuery element containing divs for each style class
 * @return {void}
 * @ignore
 */
ColorAttributeGroupHandler.__processAttrDiv = function (attrGpsDiv) {
  ColorAttributeGroupHandler._colors = [];

  var childDivs = attrGpsDiv.children();
  for (var i = 0; i < childDivs.length; i++) {
    var childDiv = $(childDivs[i]);
    var color = childDiv.css('color');
    if (color) {
      ColorAttributeGroupHandler._colors.push(color);
    }
  }
};

/**
 * Creates a shape attribute group handler that will generate shape attribute values.
 *
 * @param {Object.<string, string>=} [matchRules] A map of key value pairs for categories and the matching attribute value e.g. {"soda" : "square", "water" : "circle", "iced tea" : "triangleUp"}.
 *                            Attribute values listed in the matchRules object will be reserved only for the matching categories when getAttributeValue is called.
 * @ojsignature {target: "Type", value: "{[propName: string]: any}", for: "matchRules"}
 * @export
 * @constructor
 * @since 1.0
 * @extends oj.AttributeGroupHandler
 * @name oj.ShapeAttributeGroupHandler
 */
var ShapeAttributeGroupHandler = function (matchRules) {
  this.Init(matchRules);
};

// eslint-disable-next-line no-undef
oj.Object.createSubclass(ShapeAttributeGroupHandler, AttributeGroupHandler, 'ShapeAttributeGroupHandler');

ShapeAttributeGroupHandler._attributeValues = ['square', 'circle', 'diamond', 'plus', 'triangleDown', 'triangleUp', 'human'];

/**
 * Returns the array of possible shape values for this attribute group handler.
 * @returns {Array.<string>} The array of shape values
 * @export
 * @memberof oj.ShapeAttributeGroupHandler
 * @method getValueRamp
 * @instance
 */
ShapeAttributeGroupHandler.prototype.getValueRamp = function () {
  return ShapeAttributeGroupHandler._attributeValues;
};

/* global AttributeGroupHandler:false, ColorAttributeGroupHandler:false, ShapeAttributeGroupHandler:false */
// Define a mapping variable that maps the return value of the module to the name used in the callback function of a require call.

var attributeGroupHandler = {};
attributeGroupHandler.AttributeGroupHandler = AttributeGroupHandler;
attributeGroupHandler.ColorAttributeGroupHandler = ColorAttributeGroupHandler;
attributeGroupHandler.ShapeAttributeGroupHandler = ShapeAttributeGroupHandler;

/**
 * Copyright (c) 2014, 2015 Oracle and/or its affiliates.
 * All rights reserved.
 */
/* jslint browser: true*/
/**
 * @constructor
 * @class oj.Events
 * @classdesc Supports event system for the common model ([oj.Collection]{@link oj.Collection} and
 * [oj.Model]{@link oj.Model})
 * @since 1.0.0
 * @ojtsignore
 */
oj.Events = oj.Events = // eslint-disable-line no-multi-assign
/** @lends oj.Events */
{
    /**
     * Add an event handler for an event type to the model or collection object.
     * @param {string|Object} eventType Types of event handlers to add (may be a single event type, a
     * space-delimited set of event types, or an object mapping events to callbacks).
     * @param {function(string, Object)} callback User's event handler callback function (called with the
     * eventType and model or collection object as parameters--the context will be the model or collection unless specified by context, below).
     * @param {Object=} context A context for the event
     * @return {undefined}
     * @ojsignature {target: "Type", for: "callback", value: "(eventType: string, data: object)=> void"}
     * @since 1.0.0
     * @memberof oj.Events
     */
  on: function (eventType, callback, context) {
    return this.OnInternal(eventType, callback, context, false, false);
  },

    /**
     * Remove an event handler for an event type from the model or collection object.
     * @param {(string|Object)=} eventType Types of event handlers to remove (may be a single event type, a
     * space-delimited set of event types, or a map of events to callbacks). If omitted, remove all event handlers.
     * @param {function(string, Object)=} callback If provided, remove handlers only for eventType events with the
     * given callback function.
     * @param {Object=} context If provided, remove handlers only for eventType events with the given callback
     * function and context object.
     * @return {undefined}
     * @ojsignature {target: "Type", for: "callback", value: "(eventType: string, data: object)=> void"}
     * @since 1.0.0
     * @memberof oj.Events
    */
  off: function (eventType, callback, context) {
    return this._offInternal(eventType, callback, context, false);
  },

    /**
     * Fire the given event type(s) for all registered handlers.
     * @param {string} eventType Types of event handlers to fire (may be a single event type or a space-delimited
     * set of event types).
     * @return {undefined}
     * @since 1.0.0
     * @memberof oj.Events
     */
  trigger: function (eventType) { // eslint-disable-line no-unused-vars
    var args = Array.prototype.slice.call(arguments);
                 // Inject a silent setting in there: if this is being called outside we want to fire all relevant
                 // events
    args.unshift(false);
    return oj.Events.TriggerInternal.apply(this, args);
  },

    /**
     * Add an event handler for an event type to the model or collection object, but only fire it once, then remove
     * it from the list of handlers.
     * @param {string} eventType Types of event handlers to add (may be a single event type or a space-delimited
     * set of event types).
     * @param {function(string, Object)} callback User's event handler callback function (called with the
     * eventType and model or collection object as parameters--the context will be the model or collection unless
     * specified by context, below).
     * @param {Object=} context A context for the event
     * @return {undefined}
     * @ojsignature {target: "Type", for: "callback", value: "(eventType: string, data: object)=> void"}
     * @since 1.0.0
     * @memberof oj.Events
     */
  once: function (eventType, callback, context) {
    return this._onceInternal(eventType, callback, context, false, null);
  },

    /**
     * Add an event handler for an event type to a second model or collection object ("otherObj"), but track it on
     * the called object.
     * @param {oj.Model|oj.Collection} otherObj Model or collection object on which to add this event handler.
     * @param {string} eventType Types of event handlers to add (may be a single event type or a space-delimited
     * set of event types).
     * @param {function(string, Object)} callback User's event handler callback function (called with the
     * eventType and model or collection object as parameters--the context will be the model or collection
     * unless specified by context, below).
     * @return {undefined}
     * @ojsignature {target: "Type", for: "callback", value: "(eventType: string, data: object)=> void"}
     * @since 1.0.0
     * @memberof oj.Events
     */
  listenTo: function (otherObj, eventType, callback) {
    var eventArray;
    var e;
    var event;
    var attr;
    var eventString;
    var listenerObj;
    var eventMap = {};
    var prop;

    if (eventType.constructor === String) {
      // Create a map out of it
      eventMap[eventType] = callback;
    } else {
      eventMap = eventType;
    }

    for (prop in eventMap) { // eslint-disable-line no-restricted-syntax
      if (eventMap.hasOwnProperty(prop)) { // eslint-disable-line no-prototype-builtins
        eventArray = oj.Events._getEvents(prop);
        for (e = 0; e < eventArray.length; e += 1) {
          event = eventArray[e].event;
          attr = eventArray[e].attribute;
          listenerObj = { event: event,
            attribute: attr,
            object: otherObj,
            callback: eventMap[prop] };
          eventString = attr ? event + ':' + attr : event;
          if (this._listeningTo === undefined) {
            this._listeningTo = [];
          }
          this._listeningTo.push(listenerObj);
          // fire
          otherObj.OnInternal(eventString, eventMap[prop], null, true, false);
        }
      }
    }
    return this;
  },

    /**
     * Add an event handler for an event type to a second model or collection object ("otherObj"), but track it on
     * the called object.  Only fire once.
     * @param {oj.Model|oj.Collection} otherObj Model or collection object on which to add this event handler.
     * @param {string} eventType Types of event handlers to add (may be a single event type or a space-delimited
     * set of event types).
     * @param {function(string, Object)} callback User's event handler callback function (called with the
     * eventType and model or collection object as parameters--the context will be the model or collection unless
     * specified by context, below).
     * @ojsignature {target: "Type", for: "callback", value: "(eventType: string, data: object)=> void"}
     * @return {undefined}
     * @since 1.0.0
     * @memberof oj.Events
     */
  listenToOnce: function (otherObj, eventType, callback) {
    var eventArray;
    var e;
    var event;
    var attr;
    var eventString;
    var listenerObj;
    var eventMap = {};
    var prop;

    if (eventType.constructor === String) {
      // Create a map out of it
      eventMap[eventType] = callback;
    } else {
      eventMap = (eventType);
    }

    for (prop in eventMap) { // eslint-disable-line no-restricted-syntax
      if (eventMap.hasOwnProperty(prop)) { // eslint-disable-line no-prototype-builtins
        eventArray = oj.Events._getEvents(prop);
        for (e = 0; e < eventArray.length; e += 1) {
          event = eventArray[e].event;
          attr = eventArray[e].attribute;
          listenerObj = { event: event,
            attribute: attr,
            object: otherObj,
            callback: eventMap[prop] };
          eventString = attr ? event + ':' + attr : event;
          if (this._listeningTo === undefined) {
            this._listeningTo = [];
          }
          this._listeningTo.push(listenerObj);
          // fire
          otherObj._onceInternal(eventString, eventMap[prop], null, true, this);
        }
      }
    }
    return this;
  },

    /**
     * Remove event handlers from a model or collection object. If the arguments are omitted, removes all event
     * handlers from the model or collection.
     * @param {oj.Model|oj.Collection=} otherObj If specified, remove event handlers that target otherObj from this model or
     * collection.
     * @param {string=} eventType If specified, remove the event handlers for the given event types from this
     * model or collection
     * @param {function(string, Object)=} callback If specified, remove event handlers that call the given user
     * callback function from this model or collection
     * @return {undefined}
     * @ojsignature {target: "Type", for: "callback", value: "(eventType: string, data: object)=> void"}
     * @since 1.0.0
     * @memberof oj.Events
     */
  stopListening: function (otherObj, eventType, callback) {
    var eventArray;
    var actualType;
    var eventMap = {};
    var e;
    var oneEvent;
    var oneAttr;
    var event;
    var objEqual;
    var eventEqual;
    var callbackEqual;
    var attrEqual;
    var i;
    var len;
    var cb;
    var prop;

    if (arguments == null || arguments.length <= 1) {
      len = this._listeningTo ? this._listeningTo.length : 0;
      // Remove everything
      for (i = 0; i < len; i++) {
        event = this._listeningTo[i];
        // If we have an "otherObj" argument, make sure that passes muster
        objEqual = otherObj ? otherObj === event.object : true;
        if (objEqual) {
          cb = event.object._offInternal;
          cb.apply(event.object, [event.event, event.callback, event.context, true]);
        }
      }
      this._listeningTo = [];
      return this;
    }

    actualType = eventType;
    // Account for missing otherObj
    if (otherObj && otherObj.constructor === String) {
      actualType = otherObj;
    }

    if (actualType.constructor === String) {
      // Create a map out of it
      eventMap[actualType] = callback;
    } else {
      eventMap = actualType;
    }

    for (prop in eventMap) { // eslint-disable-line no-restricted-syntax
      if (eventMap.hasOwnProperty(prop)) { // eslint-disable-line no-prototype-builtins
        eventArray = oj.Events._getEvents(prop);
        for (e = 0; e < eventArray.length; e += 1) {
          oneEvent = eventArray[e].event;
          oneAttr = eventArray[e].attribute;
          len = this._listeningTo ? this._listeningTo.length : 0;
          for (i = len - 1; i >= 0; i -= 1) {
            event = this._listeningTo[i];
            objEqual = otherObj ? otherObj === event.object : true;
            eventEqual = oneEvent ? oneEvent === event.event : true;
            callbackEqual = callback ? eventMap[prop] === event.callback : true;
            attrEqual = oneAttr ? oneAttr === event.attribute : true;
            if (objEqual && eventEqual && callbackEqual && attrEqual) {
              cb = this._listeningTo[i].object._offInternal;
              cb.apply(this._listeningTo[i].object, [this._listeningTo[i].event,
                this._listeningTo[i].callback,
                this._listeningTo[i].context, true]);
              this._listeningTo.splice(i, 1);
            }
          }
        }
      }
    }
    return this;
  }
};

// Aliases for backward compatibility
oj.Events.bind = oj.Events.on;
oj.Events.unbind = oj.Events.off;

/**
 * @export
 * Event types
 * @enum {string}
 * @memberof oj.Events
 */
oj.Events.EventType = {
        /** Triggered when a model is added to a collection<p>
         *  The event passes these arguments to the handler: <br>
         *  <ul>
         *  <b>model</b>: the model being added to the collection<br>
         *  <b>collection</b>: the collection to which the model has been added<br>
         *  <b>options</b>: any options passed in to the add call that triggered the event
         *  </ul>
         */
  ADD: 'add',
        /** Triggered by a collection during an add call once all models passed in have been added<p>
         * The event passes these arguments to the handler:<br>
         * <ul>
         * <b>collection</b>: the collection to which the models have been added<br>
         * <b>models</b>: the array of models that have been added <br>
         * <b>options</b>: any options passed in to the add call
         * </ul>
         */
  ALLADDED: 'alladded',
        /** Triggered when a model is removed from a collection<p>
         * The event passes these arguments to the handler: <br>
         * <ul>
         * <b>model</b>: the model being removed from the collection<br>
         * <b>collection</b>: the collection from which the model was removed<br>
         * <b>options</b>: <b>index</b>: the index of the model being removed
         * </ul>
         */
  REMOVE: 'remove',
        /** Triggered when a collection is reset (see oj.Collection.reset)<p>
         *  The event passes these arguments to the handler:<br>
         *  <ul>
         *  <b>collection</b>: the collection being reset<br>
         *  <b>options</b>: any options passed in to the reset call
         *  </ul>
         */
  RESET: 'reset',
        /** Triggered when a collection is refreshed (see oj.Collection.refresh)<p>
         *  The event passes these arguments to the handler: <br>
         *  <ul>
         *  <b>collection</b>: the collection being refreshed<br>
         *  <b>options</b>: any options passed in to the refresh call
         *  </ul>
         */
  REFRESH: 'refresh',
        /** Triggered when a collection is sorted.  If the second argument to the callback is set (options) and
         * 'add' is true, it means this sort event was triggered as a result of an add <p>
         *  The event passes these arguments to the handler:<br>
         *  <ul>
         *  <b>collection</b>: the collection being sorted<br>
         *  <b>options</b>: <b>add</b>: true if this sort event was triggered as the result of an add call,
         *  undefined or false if not
         *  </ul>
         */
  SORT: 'sort',
        /** Triggered when a model's attributes are changed.  This can be the result of a clear call on a model;
         * a property set call on a model; an unset call on a model; or the changing of properties due to the
         * merging of models (in an add, for example) <p>
         * The event passes these arguments to the handler:<br>
         * <ul>
         * <b>model</b>: the model on which the change occurred<br>
         * <b>value</b>: for property-specific change events, the new value of the property being changed<br>
         * <b>options</b>: any options passed in to the call that triggered the change event.  This is the second
         * argument passed for overall change events, and the third parameter (after value) for property-specific
         * change events.
         * </ul>
         */
  CHANGE: 'change',
        /** Triggered when a model is deleted from the data service (and thus from its Collection), due to a model
         * destroy call<p>
         * The event passes these arguments to the handler:<br>
         * <ul>
         * <b>model</b>: the model being deleted<br>
         * <b>collection</b>: the deleted model's collection, if any
         * </ul>
         */
  DESTROY: 'destroy',
        /** Triggered by a collection during a remove call once all models passed in have been removed and
         * destroyed<p>
         * The event passes these arguments to the handler:<br>
         * <ul>
         * <b>collection</b>: the collection from which the models have been removed<br>
         * <b>models</b>: the array of models that have been removed <br>
         * <b>options</b>: any options passed in to the remove call
         * </ul>
         */
  ALLREMOVED: 'allremoved',
        /** Triggered when a model or collection has sent a request to the data service <p>
         *  The event passes these arguments to the handler:<br>
         *  <ul>
         *  <b>collection or model</b>: the collection or model triggering the request<br>
         *  <b>xhr</b>: the xhr argument for the request<br>
         *  <b>options</b>: any options passed as part of the request
         *  </ul>
         */
  REQUEST: 'request',
        /** Triggered when a model or collection has been updated from the data service<p>
         *  The event passes these arguments to the handler:<br>
         *  <ul>
         *  <b>collection or model</b>: the collection or model that triggered the update<br>
         *  <b>response</b>: the response object from the data service<br>
         *  <b>options</b>: any options passed in to the call that triggered the update
         *  </ul>
         */
  SYNC: 'sync',
        /** Triggered when a model has failed to update on the data service<p>
         *  The event passes these arguments to the handler:<br>
         *  <b>collection or model</b>: the collection or model that made the call that resulted in the error<br>
         *  <b>xhr</b>: the xhr argument for the failing request, if any<br>
         *  <b>options</b>: any options passed in to the call that triggered the failing request, plus the status
         *  and error as textStatus and errorThrown
         *  </ul>
         */
  ERROR: 'error',
        /** Triggered on an error with data source interactions <p>
         * The event passes these arguments to the handler:<br>
         * <ul>
         * <b>model</b>: the model (or collection) on which the error operation happened <br>
         * <b>xhr</b>: the xhr involved, if relevant<br>
         * <b>options</b>: any options passed in to the call that triggered the invalid event
         * </ul>
         */
  INVALID: 'invalid',
        /** Triggered when all pending promises from Collection API calls have been resolved<p>
         * The event passes these arguments to the handler:<br>
         * <ul>
         * <b>collection</b>: the collection on which the promises have been resolved
         * </ul>
         */
  READY: 'ready',
        /** Triggered for any of the above events <p>
         * The event passes the name of the actual event and then any arguments normally passed to that event
         * following the name
         */
  ALL: 'all'
};


/**
 * @private
 * @param {Object} myClass
 * @param {Object=} source
 */
oj.Events.Mixin = function (myClass, source) {
  var methodName;
  var obj = source || this;
  for (methodName in obj) { // eslint-disable-line no-restricted-syntax
    if (typeof obj[methodName] === 'function') {
      myClass[methodName] = obj[methodName]; // eslint-disable-line no-param-reassign
    }
  }
    // Make sure actual vars are own copies
  myClass.eventHandlers = {}; // eslint-disable-line no-param-reassign
  myClass._listeningTo = []; // eslint-disable-line no-param-reassign
};

/**
 * @private
 */
oj.Events._onceInternal = function (eventType, callback, context, listenTo, otherObj) {
  var eventArray;
  var e;
  var event;
  var attr;
  var eventMap;
  var obj;
  var cxt = context;

  obj = this._getEventMap(eventType, callback, context);
  eventMap = obj.map;
  cxt = obj.context;

  var self = this;
  Object.keys(eventMap || {}).forEach(function (prop) {
    if (Object.prototype.hasOwnProperty.call(eventMap, prop)) {
      eventArray = self._getEvents(prop);

      for (e = 0; e < eventArray.length; e += 1) {
        event = eventArray[e].event;
        attr = eventArray[e].attribute;
        if (self.eventHandlers === undefined) {
          self.eventHandlers = [];
        }
        if (self.eventHandlers[event] === undefined) {
          self.eventHandlers[event] = [];
        }

        self.eventHandlers[event].push({ callback: eventMap[prop],
          context: cxt,
          attribute: attr,
          once: true,
          fired: false,
          listen: listenTo,
          otherObj: otherObj });
      }
    }
  });
  return this;
};

/**
 * @private
 */
oj.Events._shouldFire = function (handler) {
  if (handler.once) {
    if (!handler.fired) {
      handler.fired = true; // eslint-disable-line no-param-reassign
      return true;
    }
    return false;
  }
  return true;
};

/**
 * @private
 */
oj.Events._getContext = function (obj, handler) {
  return handler.context || handler.otherObj || obj;
};

/**
 * @protected
 */
oj.Events.TriggerInternal = function (silent, eventType) {
  var eventArray = this._getEvents(eventType);
  var e;
  var event;
  var attr;
  var eventsToFire;
  var handlers;
  var i;
  var args;
  var allHandlers;
  var callback;

  eventsToFire = [];
  for (e = 0; e < eventArray.length; e += 1) {
    event = eventArray[e].event;
    attr = eventArray[e].attribute;
    // Do specific event...
    eventsToFire.push({ event: event, attribute: attr });
  }
  for (e = 0; e < eventsToFire.length; e += 1) {
    allHandlers = this._getHandlers(this.eventHandlers, oj.Events.EventType.ALL);
    handlers = oj.Events._getHandlers(this.eventHandlers, eventsToFire[e].event, false);
    for (i = 0; i < (handlers ? handlers.length : 0); i += 1) {
      if (handlers[i].attribute === eventsToFire[e].attribute && handlers[i].callback) {
        args = Array.prototype.slice.call(arguments);
        if (handlers && handlers[i] && handlers[i].once) {
          // Remove it: only want to fire once--make sure we remove it from the original
          this._removeHandler(oj.Events._getHandlers(this.eventHandlers, eventsToFire[e].event,
            true), handlers[i]);
          // Now take it out of the other object's "listen to" list, if relevant
          if (handlers[i].otherObj) {
            // Clean up the "other object" if this was a triggered listenOnce
            handlers[i].otherObj.stopListening(this, eventType, handlers[i].callback);
          }
        }
        if (handlers && handlers[i] && this._shouldFire(handlers[i])) {
          callback = handlers[i].callback;
          // If this isn't a silent firing or this handler always wants to be called, make the call
          if (!silent || handlers[i].ignoreSilent) {
            callback.apply(oj.Events._getContext(this, handlers[i]), args.slice(2));
          }
        }
      }
    }
    // Handle all
    for (i = 0; i < (allHandlers ? allHandlers.length : 0); i += 1) {
      args = Array.prototype.slice.call(arguments);
      if (args.length > 0) {
        if (eventsToFire[e].attribute) {
          args[1] = eventsToFire[e].event + ':' + eventsToFire[e].attribute;
        } else {
          args[1] = eventsToFire[e].event;
        }
      }
      // All case--make sure to pass event name
      if (allHandlers && allHandlers[i] && allHandlers[i].callback &&
        this._shouldFire(allHandlers[i])) {
        callback = allHandlers[i].callback;
        // If this isn't a silent firing or this handler always wants to be called, make the call
        if (!silent || allHandlers[i].ignoreSilent) {
          callback.apply(oj.Events._getContext(this, allHandlers[i]), args.slice(1));
        }
      }
      if (allHandlers && allHandlers[i] && allHandlers[i].once) {
        // Remove it: only want to fire once
        this._removeHandler(this._getHandlers(this.eventHandlers, oj.Events.EventType.ALL, true),
                                       allHandlers[i]);
        // Now take it out of the other object's "listen to" list, if relevant
        if (allHandlers[i].otherObj) {
          // Clean up the "other object" if this was a triggered listenOnce
          allHandlers[i].otherObj.stopListening(this, oj.Events.EventType.ALL,
                                                                 allHandlers[i].callback);
        }
      }
    }
  }
  return this;
};

/**
 * @protected
 */
oj.Events.OnInternal = function (eventType, callback, context, listenTo, ignoreSilent) {
  var eventMap;
  var eventArray;
  var i;
  var event;
  var attr;
  var eventObj;
  var cxt = context;
  var prop;

  var obj = this._getEventMap(eventType, callback, context);
  eventMap = obj.map;
  cxt = obj.context;

  for (prop in eventMap) { // eslint-disable-line no-restricted-syntax
    if (eventMap.hasOwnProperty(prop)) { // eslint-disable-line no-prototype-builtins
      eventArray = this._getEvents(prop);

      for (i = 0; i < eventArray.length; i += 1) {
        event = eventArray[i].event;
        attr = eventArray[i].attribute;
        if (this.eventHandlers === undefined) {
          this.eventHandlers = [];
        }
        if (this.eventHandlers[event] === undefined) {
          this.eventHandlers[event] = [];
        }

        eventObj = { callback: eventMap[prop],
          context: cxt,
          attribute: attr,
          listen: listenTo,
          ignoreSilent: ignoreSilent };
        if (this._checkForHandler(this.eventHandlers[event], eventObj,
          oj.Events._handlersIdentical) === -1) {
          this.eventHandlers[event].push(eventObj);
        }
      }
    }
  }
  return this;
};

/**
 * @private
 */
oj.Events._offInternal = function (eventType, callback, context, listen) {
  var eventMap;
  var obj;
  var cxt = context;
  var prop;

  if (arguments == null || arguments.length === 0) {
         // Remove everything
    this.eventHandlers = {};
    return this;
  }

  if (eventType == null) {
    this._removeEvent(eventType, callback, context, listen);
    return this;
  }

  obj = this._getEventMap(eventType, callback, context);
  eventMap = obj.map;
  cxt = obj.context;

  for (prop in eventMap) { // eslint-disable-line no-restricted-syntax
    if (eventMap.hasOwnProperty(prop)) { // eslint-disable-line no-prototype-builtins
      this._removeEvent(prop, eventMap[prop], cxt, listen);
    }
  }
  return this;
};


/**
 * @private
 */
oj.Events._getEventMap = function (eventType, callback, context) {
  var eventMap = {};

  if (eventType.constructor === String) {
    // Create a map out of it
    eventMap[eventType] = callback;
  } else {
    eventMap = eventType;
    // If eventType is a map of events->callbacks, then the callback argument is now context
    return { map: eventMap, context: callback };
  }
  return { map: eventMap, context: context };
};

/**
 * @private
 */
oj.Events._removeEvent = function (eventType, callback, context, listen) {
  var eventArray = [];
  var e;
  var evt;
  var i;
  var attr;
  var handlers;
  var callbacks;
  var contexts;
  var attrs;
  var listenEq;

  if (eventType) {
    eventArray = oj.Events._getEvents(eventType);
  } else if (this.eventHandlers !== undefined) {
    // Walk entire eventHandlers property list
    var self = this;
    Object.keys(this.eventHandlers || {}).forEach(function (event) {
      if (Object.prototype.hasOwnProperty.call(self.eventHandlers, event)) {
        eventArray.push({ event: event });
      }
    });
  }

  for (e = 0; e < eventArray.length; e += 1) {
    evt = eventArray[e].event;
    attr = eventArray[e].attribute;
    if (this.eventHandlers !== undefined && this.eventHandlers[evt] instanceof Array) {
      handlers = this.eventHandlers[evt];
      for (i = handlers.length - 1; i >= 0; i -= 1) {
        callbacks = (callback === undefined || callback === null || handlers[i].callback ===
          callback);

        contexts = (context === undefined || context === null || handlers[i].context === context);
        attrs = (attr === undefined || attr === null || handlers[i].attribute === attr);
        listenEq = (listen === undefined || listen === null || handlers[i].listen === listen);
        if (callbacks && contexts && attrs && listenEq) {
          handlers.splice(i, 1);
        }
      }
      if (handlers.length === 0) {
        // Delete the entry
        delete this.eventHandlers[evt];
      }
    }
  }
};

/**
 * @private
 */
oj.Events._removeHandler = function (handlers, handler) {
  var i;
  var callbacks;
  var contexts;
  var attrs;
  var listenEq;
  var onceEq;

  if (handlers) {
    for (i = handlers.length - 1; i >= 0; i -= 1) {
      callbacks = (handler.callback === undefined || handler.callback === null ||
                         handlers[i].callback === handler.callback);

      contexts = (handler.context === undefined || handler.context === null ||
                        handlers[i].context === handler.context);
      attrs = (handler.attribute === undefined || handler.attribute === null ||
                     handlers[i].attribute === handler.attribute);
      listenEq = (handler.listen === undefined || handler.listen === null ||
                        handlers[i].listen === handler.listen);
      onceEq = (handler.once === undefined || handler.once === null ||
        handlers[i].once === handler.once);
      if (callbacks && contexts && attrs && listenEq && onceEq) {
        handlers.splice(i, 1);
      }
    }
  }
};

/**
 * @private
 */
oj.Events._getEvents = function (eventString) {
  var eventList = eventString ? eventString.split(' ') : [];
  var retList = [];
  var i;
  var eventWithAttr;
  var name;
  var attr;

  for (i = 0; i < eventList.length; i += 1) {
    eventWithAttr = eventList[i].split(':');
    name = eventWithAttr[0];
    attr = eventWithAttr.length > 1 ? eventWithAttr[1] : null;
    retList.push({ event: name, attribute: attr });
  }
  return retList;
};

/**
 * @private
 */
oj.Events._handlersIdentical = function (handler1, handler2) {
  return (handler1.callback === handler2.callback) && (handler1.attribute === handler2.attribute) &&
          (handler1.context === handler2.context) && (handler1.listen === handler2.listen) &&
          (handler1.once === handler2.once);
};

/**
 * @private
 */
oj.Events._listenersIdentical = function (listener1, listener2) {
  return (listener1.event === listener2.event) && (listener1.attribute === listener2.attribute) &&
          (listener1.context === listener2.context) && (listener1.object === listener2.object);
};

/**
 * @private
 */
oj.Events._checkForHandler = function (handlerList, handler, handlerTest) {
  var i;
  if (handlerList === undefined) {
    return -1;
  }

  for (i = 0; i < handlerList.length; i += 1) {
    if (handlerTest(handlerList[i], handler)) {
      return i;
    }
  }
  return -1;
};

/**
 * @private
 */
oj.Events._getHandlers = function (handlers, eventType, original) {
  if (handlers && handlers[eventType] instanceof Array) {
    if (original) {
      return handlers[eventType];
    }
    // Make a copy
    var handlerReturn = [];
    var i;
    for (i = 0; i < handlers[eventType].length; i++) {
      handlerReturn.push(handlers[eventType][i]);
    }
    return handlerReturn;
  }
  return null;
};

/**
 * Copyright (c) 2014, 2015 Oracle and/or its affiliates.
 * All rights reserved.
 */
/* jslint browser: true*/
/* global Promise:false, Logger:false */

/**
 * @export
 * @class oj.Collection
 * @classdesc Collection of Model objects
 *
 * @param {Array.<oj.Model>=} models Set of model objects to put into collection at construction
 *                 time.  If models contain actual
 *                 oj.Model objects, then any custom parse callback set on the collection must
 *                 be able to handle oj.Model objects as a possible argument
 * @param {Object=} options Passed through to the user's initialize routine, if any, upon
 *                  construction
 * @constructor
 * @since 1.0
 * @mixes oj.Events
 * @ojsignature {target: "Type", value: "class Collection"}
 */
oj.Collection = function (models, options) {
  if (oj.Collection._justExtending) {
    return;
  }

    // Initialize
  oj.Collection._init(this, models, options, null);
};


/**
 * Subclass from oj.Object
 * @private
 */
oj.Object.createSubclass(oj.Collection, oj.Object, 'oj.Collection');

/**
 * @desc Property specifying the [model]{@link oj.Model} class object used by the collection
 * @memberof oj.Collection
 * @type oj.Model
 * @export
 */
oj.Collection.prototype.model = null;


/**
 * @desc Function specifying how to construct the id for models in this collection.  Override to
 * change the construction of the id.
 * @memberof oj.Collection
 * @param {Object} attrs attributes of a model
 * @returns {null|string}
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.modelId = function (attrs) {
  var model = this.model;
  if (model && attrs) {
    return attrs[model.idAttribute || model.prototype.idAttribute || 'id'];
  }
  return null;
};

/**
 * @export
 * @desc Total number of models in the collection.  When the collection is virtual, not all of the
 * models may be locally available.
 * @memberof oj.Collection
 *
 * @type number
 * @since 1.0.0
 */
oj.Collection.prototype.length = undefined;

/**
 * @export
 * @desc Direct access to the collection's list of models objects<br/>
 * Note that this property should not be used directly when a collection is virtual, as
 * automatic fetches will not be triggered for undefined elements in the model.  Use at()
 * instead.
 * @memberof oj.Collection
 *
 * @type {Array.<oj.Model>}
 * @since 1.0.0
 */
oj.Collection.prototype.models = undefined;

/**
 * Tracking indices used
 * @private
 */
oj.Collection.prototype._modelIndices = [];

/**
 * @export
 * @desc The data service's server URL.
 * @memberof oj.Collection
 *
 * @type {null|string|function():string}
 * @since 1.0.0
 */
oj.Collection.prototype.url = null;

/**
 * @export
 * @desc Changes that have occured due to adds/removes since the last fetch.  This is a list of
 * indicies that have changed (location at which a model was added, deleted, or set).  They do
 * not shift with subsequent operations
 * @memberof oj.Collection
 * @type {Array.<number>}
 * @since 1.0.0
 */
oj.Collection.prototype.changes = [];


/**
 * A callback to allow users to customize the data service URLs.  The callback should accept
 * these parameters:<p>
 * <b>operation</b>: one of "create", "read", "update", "patch", or "delete", indicating the
 * type of operation for which to return the URL<p>
 * <b>collection</b>: the oj.Collection object requesting the URL<p>
 * <b>options</b>: any of the following properties:<br>
 * <ul>
 * <b>recordID</b>: id of the record involved, if relevant<br>
 * <b>fetchSize</b>: how many records to return.  If not set, return all.<br>
 * <b>startIndex</b>: Starting record number of the set to return.<br>
 * <b>startID</b>: Retrieve records starting with the record with the given unique ID.<br>
 * <b>since</b>: Retrieve records with timestamps after the given timestamp.<br>
 * <b>until</b>: Retrieve records with timestamps up to the given timestamp. Default is "until"<br>
 * <b>sort</b>:  field(s) by which to sort, if set<br>
 * <b>sortDir</b>: sort ascending or descending (asc/dsc)<br>
 * <b>query</b>: a set of attributes indicating filtering that should be done on the server.  See
 * [where]{@link oj.Collection#where} for complete documentation of query values<br>
 * <b>all</b>: true (along with 'query', above) indicates that this is a findWhere or where type
 * call that is expecting all models meeting the query condition to be returned<br>
 * </ul>
 * <p>
 * customURL callbacks should return either: null, in which case the default will be used; a
 * url string, which will be used with the standard HTTP method for the type of operation, or
 * an Object with any other attributes that should be passed to the ajax call.<br>
 * This object must at minimum include the URL, and other attributes as follows:<br>
 * <ul>
 * <b>url</b>: the custom URL string<br>
 * <b>type</b>: (optional) a string indicating the type of HTTP method to use (GET, POST,
 *              DELETE, etc.)<br>
 * <b>(other)</b>: (optional) any other attributes to pass in the ajax call<br>
 * </ul>
 * <p>
 * @type {function(string,oj.Collection,Object):(string|Object|null)|null}
 * @memberof oj.Collection
 * @ojsignature  {target: "Type", value: "function(string,oj.Collection,oj.Collection.CustomURLCallbackOptions):(string|Object|null)|null", for: "returns"}
 *
 * @export
 * @since 1.0.0
 */
oj.Collection.prototype.customURL = null;

/**
 * @typedef {Object} oj.Collection.CustomURLCallbackOptions
 * @property {string=} recordID id of the record involved, if relevant
 * @property {number=} fetchSize how many records to return.  If not set, return all
 * @property {number=} startIndex Starting record number of the set to return
 * @property {string=} startID Retrieve records starting with the record with the given unique ID
 * @property {string=} since Retrieve records with timestamps after the given timestamp
 * @property {string=} until Retrieve records with timestamps up to the given timestamp. Default is "until"
 * @property {string=} sort field(s) by which to sort, if set
 * @property {string=} sortDir sort ascending or descending (asc/dsc)
 * @property {Object=} query a set of attributes indicating filtering that should be done on the server.  See
 * [where]{@link oj.Collection#where} for complete documentation of query values
 * @property {boolean=} all true (along with 'query', above) indicates that this is a findWhere or where type
 * call that is expecting all models meeting the query condition to be returned
*/


/**
 * A callback function allowing users to extract their own paging/virtualization
 * return values from the server's response.<p>
 * It should accept these parameters:<p>
 * <b>response</b>: the Object data response coming back from the fetch call<p>
 * <p>
 * The callback should return either null, in which case the collection will look for the
 * default properties, or an object containing one or more of the following attribute/value
 * pairs (note that the Collection will look back to the response for default paging return
 * properties if not found in the returned object):<br>
 * <ul>
 * <b>totalResults</b>: the total number of records available on the server side, not just
 * in the current result.  By default the collection looks in the response for "totalResults"<br>
 * <b>limit</b>: the actual fetchSize from the server.  This may not be the client's fetchSize
 * or the number of records in the current result.  By default the collection looks in the
 * response for "limit".  This becomes the collection's "lastFetchSize" property<br>
 * <b>count</b>: the actual number of records returned by the server in the last result.  This
 * becomes the collection's "lastFetchCount".  By default the collection looks in the response
 * for "count".<br>
 * <b>offset</b>: the actual starting record number of the current result.  By default the
 * collection looks in the response for "offset"<br>
 * <b>hasMore</b>: boolean indicating whether or not there are more records available beyond
 * the current result.  By default the collection looks in the response for "hasMore"<br>
 * </ul>
 * <p>
 *
 * @type {(function(Object):(Object|null)|null)}
 * @memberof oj.Collection
 * @ojsignature {target: "Type",
 * value: "((response: object)=> oj.Collection.CustomPagingOptionsReturn|null)|null"}
 * @export
 * @since 1.0.0
 */
oj.Collection.prototype.customPagingOptions = null;

/**
 * @typedef {Object} oj.Collection.CustomPagingOptionsReturn
 * @property {number=} totalResults
 * @property {number=} limit
 * @property {number=} count
 * @property {number=} offset
 * @property {boolean=} hasMore
*/

/**
 * @export
 * @desc The server's fetch size.  This may not match [fetchSize]{@link oj.Collection#fetchSize}.
 * @memberof oj.Collection
 *
 * @type number
 * @since 1.0.0
 */
oj.Collection.prototype.lastFetchSize = undefined;

/**
 * @export
 * @desc Indicates whether or not there are more records available on the server, at indices
 * beyond the last fetch.
 * @memberof oj.Collection
 *
 * @type boolean
 * @since 1.0.0
 */
oj.Collection.prototype.hasMore = false;

/**
 * @export
 * @desc The total number of records available for this collection regardless of whether they
 * have been fetched or not.  For non-virtual collections this will equal the length.
 * @memberof oj.Collection
 *
 * @type number
 */
oj.Collection.prototype.totalResults = undefined;

/**
 *
 * @export
 * @desc The number of records actually fetched the last time the collection fetched from the
 * server.  This may or may not match [fetchSize]{@link oj.Collection#fetchSize} or
 * [lastFetchSize]{@link oj.Collection#lastFetchSize}
 * @memberof oj.Collection
 *
 * @type number
 */
oj.Collection.prototype.lastFetchCount = undefined;

/**
 * @export
 * @desc For virtual collections, the number of records to be kept in memory at any one
 * time.  The default of -1 indicates that no records are thrown out
 * @memberof oj.Collection
 *
 * @type number
 */
oj.Collection.prototype.modelLimit = -1;

/**
 * @export
 * @desc The actual starting index number at which models from the last server fetch
 * were inserted into the collection.
 * @memberof oj.Collection
 *
 * @type number
 */
oj.Collection.prototype.offset = undefined;

/**
 * @export
 * @desc The number of records to be fetched from the server in any one round trip.
 * The server's fetch size comes back as the "limit" property.  The default value
 * of -1 indicates that virtualization is not being used or is not available,
 * and all records will be fetched.<br>
 * The number of records actually fetched comes back as [count]{@link oj.Collection#count}<br>
 * @memberof oj.Collection
 *
 * @type number
 */
oj.Collection.prototype.fetchSize = -1;

/**
 * @export
 * @desc Sort direction for string-based sort comparators (model attribute names).  A value
 * of 1 indicates ascending sorts, -1 indicates descending.  The default is 1 (ascending).<br>
 * Users should call sort() after changing sort direction to ensure that models in the
 * collection are sorted correctly, or, for virtual collections, that there are no left
 * over models in an incorrect order.
 * @memberof oj.Collection
 *
 * @type number
 */
oj.Collection.prototype.sortDirection = 1;

/**
 * @export
 * @desc If set to a string, sort the collection using the given attribute of a model.<p>
 * If set to a function(Model), the function should return a string giving the model
 * attribute by which the sort should take place.<p>
 * If set to a function(Model1, Model2), then this function is called comparing Model1
 * and Model2 (see the JavaScript array.sort() for details)<p>
 * In the virtual case, comparator must be a string-based field comparator, which will
 * be passed to the server.<p>
 * Users should call sort() after making any changes to the comparator to ensure that
 * the models are correctly sorted, or that there are no leftover models sorted incorrectly
 * in the virtual case.
 * @memberof oj.Collection
 *
 * @type {null|string|function(oj.Model,oj.Model=):number}
 * @since 1.0.0
 */
oj.Collection.prototype.comparator = null;

/**
 * @export
 * @ojstatus preview
 * @memberof oj.Collection
 * @type {boolean}
 * @desc If true, do not insert the JET locale-based Accept-Language header.  If false,
 * let the Ajax system set the header.
 * @since 5.0.0
 */
oj.Collection.prototype.omitLanguageHeader = false;

/**
 * @private
 */
oj.Collection.prototype.Init = function () {
  oj.Collection.superclass.Init.call(this);
};

/**
 * Create a new, specific type of Collection object to represent a collection of records
 * from a JSON data set.
 * @param {Object=} properties Properties for the new Collection class.<p>
 * <b>parse</b>: a user callback function to allow parsing of the JSON collection after it's
 * returned from the data service.  If a collection is initialized with actual oj.Models or
 * collection.set is used with actual oj.Models, the parse callback must expect that the argument
 * passed to it may contain raw JSON data *or* oj.Model objects<br>
 * <b>model</b>: the specific type of [model]{@link oj.Model} object to use for each member of
 * the collection<br>
 * <b>url</b>: the URL string to use to get records from the data service<br>
 * <b>initialize</b>: a user callback function to be called when this collection is created.
 * Called in the context of the collection and passed: models, options.<br>
 * <b>comparator</b>: a user callback used on sort calls. May also be set to false to prevent
 * sorting.  See [comparator]{@link oj.Collection#comparator}<br>
 * <b>fetchSize</b>: the number of records to be fetched on each round trip to the server.
 * Overrides [fetchSize]{oj.Collection#fetchSize}. If not set, the collection will not consider
 * itself virtual<br>
 * <b>modelLimit</b>: the number of records to be held in memory at any one time, if
 * virtualization is in force.  The default is all records.  This uses an LRU algorithm to
 * determine which to roll off as more records are added.<br>
 * @param {Object=} classProperties optional properties that get attached to the constructor
 * of the extended Collection
 * @return {oj.Collection} new Collection object
 * @ojsignature [{target: "Type",
 *                value: "any",
 *                for: "returns"},
 * {target: "Type", value:"{parse?: (data: any)=> any, model?: oj.Model, url?: string,
 * initialize?: (models: Array<oj.Model>, options: object)=> void,
 * comparator?: null|string|((model1: oj.Model, model2?: oj.Model)=> number),
 * fetchSize?: number, modelLimit?: number, [propName: string]: any}", for: "properties"}]
 * @memberof oj.Collection
 * @this {oj.Collection}
 * @since 1.0.0
 * @export
 */
oj.Collection.extend = function (properties, classProperties) {
  var obj = null;
  oj.Collection._justExtending = true;
  obj = new oj.Collection();
  oj.Collection._justExtending = false;

    // Add regular properties from this "parent"
    // oj.Events.Mixin(obj, this.prototype);
  $.extend(obj, this.prototype);

  var Collection;
  if (properties && properties.constructor && Object.prototype.hasOwnProperty.call(properties, 'constructor')) {
    Collection = properties.constructor;
  } else {
    Collection = function (models, options) {
      oj.Collection._init(this, models, options, properties);
    };
  }

  if (classProperties) {
    Object.keys(classProperties).forEach(function (prop) {
      if (Object.prototype.hasOwnProperty.call(classProperties, prop)) {
        Collection[prop] = classProperties[prop];
      }
    });
  }

  if (properties) {
    Object.keys(properties).forEach(function (prop) {
      if (Object.prototype.hasOwnProperty.call(properties, prop)) {
        obj[prop] = properties[prop];
      }
    });
  }

  $.extend(Collection, this);
  Collection.prototype = obj;

    // Allow extending resulting obj
  Collection.extend = oj.Collection.extend;

  Collection.prototype.constructor = Collection;

  return Collection;
};


/**
 * @private
 */
oj.Collection._init = function (collection, models, options, properties) {
  var i;
  var optionlist;
  var modelList;
  var prop;

  collection.Init();

    // Augment with Event
  oj.Events.Mixin(collection);

    // First, copy all properties passed in
  if (properties) {
    for (prop in properties) { // eslint-disable-line no-restricted-syntax
      if (Object.prototype.hasOwnProperty.call(properties, prop)) {
        collection[prop] = properties[prop];  // eslint-disable-line no-param-reassign
      }
    }
  }

    // Check options
  var opt = options || {};
  optionlist = ['comparator', 'customPagingOptions', 'customURL',
    oj.Collection._FETCH_SIZE_PROP, 'model', 'modelLimit', 'sortDirection', 'url'];
  for (i = 0; i < optionlist.length; i++) {
    if (Object.prototype.hasOwnProperty.call(opt, optionlist[i]) &&
        opt[optionlist[i]] !== undefined) {
      collection[optionlist[i]] = opt[optionlist[i]]; // eslint-disable-line no-param-reassign
    }
  }
  if (collection._getFetchSize(null) === undefined) {
    collection.setFetchSize(-1);
  }
  if (collection.modelLimit === undefined) {
    collection.setModelLimit(-1);
  }
  collection.hasMore = false; // eslint-disable-line no-param-reassign
  collection.lruCount = 0; // eslint-disable-line no-param-reassign

  collection._setModels([], true);
  var localModels = models;
  if (opt.parse) {
    localModels = collection.parse(models);
  }
  if (localModels != null) {
    opt.noparse = true;
    modelList = (localModels instanceof Array) ? localModels : [localModels];
    collection._addInternal(modelList, opt, true, false);
  }
  collection._setLength();
  if (!localModels) {
        // Make sure totalResults is uninitialized at first, though, for non virtual case--0 could be the length of the first fetch
    collection.totalResults = undefined; // eslint-disable-line no-param-reassign
  }

  if (properties && properties.initialize) {
    properties.initialize.call(collection, localModels, opt);
  }
};


// Placeholder for event mixins
oj.Collection.prototype.on = function (event, callback) {}; // eslint-disable-line no-unused-vars
oj.Collection.prototype.OnInternal = function (event, callback, context, listenTo, ignoreSilent) {}; // eslint-disable-line no-unused-vars
oj.Collection.prototype.TriggerInternal = function (silent, event, arg1, arg2, options) {}; // eslint-disable-line no-unused-vars

/**
 * Fire request event
 * @private
 */
oj.Collection.prototype._fireRequest = function (collection, xhr, options, silent) {
  this.TriggerInternal(silent, oj.Events.EventType.REQUEST, collection, xhr, options);
};

/**
 * @private
 */
oj.Collection.prototype._resetChanges = function () {
  this.changes = [];
};

/**
 * @private
 */
oj.Collection.prototype._setChangeAt = function (start, count) {
  for (var at = start; at < start + count; at++) {
    if (this.changes.indexOf(at) === -1) {
      this.changes.push(at);
            // this.changes.sort(function(a,b) { return a-b;});
    }
  }
};


/**
 * @private
 */
oj.Collection.prototype._setModels = function (models, clearing) {
  this.models = models;
  if (clearing) {
    this._modelIndices = [];
    this._resetChanges();
  } else {
    for (var i = 0; i < models.length; i++) {
      if (models[i]) {
        this._modelIndices.push(i);
      }
    }
  }
};

/**
 * @private
 */
oj.Collection.prototype._getModels = function () {
  return this.models;
};

/**
 * @private
 */
oj.Collection.prototype._getModelsLength = function () {
  return this._getModels().length;
};

/**
 * Designed to check if index exceeds the length of the models.  If we're in a virtual and "no totalResults" case,
 * we're never over the upper limit
 * @private
 */
oj.Collection.prototype._overUpperLimit = function (index) {
  if (index < this._getModelsLength()) {
    return false;
  }
  if (this.IsVirtual()) {
    if (!this._hasTotalResults() || this._getModelsLength() === 0) {
      return false;
    }
  }
  return true;
};

/**
 * @private
 */
oj.Collection.prototype._hasTotalResults = function () {
  return oj.Collection._defined(this.totalResults);
};

/**
 * @private
 */
oj.Collection._defined = function (value) {
  return value != null;
};

/**
 * @private
 */
oj.Collection.prototype._pushModel = function (model) {
  this._getModels().push(model);
  this._modelIndices.push(this._getModelsLength() - 1);
  this._setChangeAt(this._getModelsLength() - 1, 1);
};

/**
 * @private
 */
oj.Collection.prototype._pushModels = function (model) {
    // Model is being added to the end, it should be made the head
  this._makeModelHead(model);
  this._pushModel(model);
  this.lruCount = this.lruCount + 1;
  model.SetIndex(this._getModelsLength() - 1);
};

/**
 * @private
 */
oj.Collection.prototype._reduceLRU = function (removed) {
  if (removed) {
    for (var i = 0; i < removed.length; i++) {
      if (removed[i]) {
        this.lruCount = this.lruCount - 1;
      }
    }
  }
};

/**
 * @param {number} start
 * @param {number} count
 * @param {Object=} model
 * @private
 */
oj.Collection.prototype._spliceModels = function (start, count, model) {
    // Clean up prev/next links for models being removed
  for (var i = start; i < start + count; i++) {
    this._removePrevNext(this._getModel(i));
  }
  if (model === undefined) {
    this._reduceLRU(this._getModels().splice(start, count));
    this._spliceModelIndices(start, (start + count) - 1);
  } else {
    this._reduceLRU(this._getModels().splice(start, count, model));
    this._spliceModelIndices(start, (start + count) - 1);
    this._insertModelIndex(start);
    this._makeModelHead(model);
  }
  this._setChangeAt(start, count);
  if (this.lruCount < 0) {
    this.lruCount = 0;
  }
  this._realignModelIndices(start);
};

/**
 * @private
 */
oj.Collection.prototype._getModel = function (index) {
  return this._getModels()[index];
};

/**
 * Realign all the indices of the models (after sort for example)
 * @private
 */
oj.Collection.prototype._realignModelIndices = function (start) {
  var index;
  for (var i = 0; i < this._modelIndices.length; i++) {
    index = this._modelIndices[i];
    if (index >= start && this._getModel(index)) {
      this._getModel(index).SetIndex(index);
    }
  }
};

/**
 * Update next/prev pointers as though the given model were being removed
 * @private
 */
oj.Collection.prototype._removePrevNext = function (model) {
  if (!model) {
    return;
  }

  var oldPrev = model.GetPrevious();
  var oldNext = model.GetNext();
    // Link the two surrounding previous/next elements to each other, because this one is being replaced and moved
    // to the head
  if (oldPrev) {
    oldPrev.SetNext(oldNext);
  } else {
        // This element used to be the head
    this.head = oldNext;
  }
  if (oldNext) {
    oldNext.SetPrevious(oldPrev);
  } else {
        // This element used to be the tail
    this.tail = oldPrev;
  }
};

/**
 * @private
 */
oj.Collection.prototype._makeModelHead = function (model) {
    // Make this new model the most recently used: the head
  model.SetNext(this.head);
  if (this.head) {
    this.head.SetPrevious(model);
  } else {
        // No head: list is empty-->make tail the same element
    this.tail = model;
  }
  model.SetPrevious(null);
  this.head = model;
};

/**
 * Mark the model index tracker as having a used slot
 * @private
 */
oj.Collection.prototype._setModelIndex = function (index) {
  if (this._modelIndices.indexOf(index) === -1) {
    this._modelIndices.push(index);
  }
};


/**
 * Insert the given model at the given index
 * @private
 */
oj.Collection.prototype._insertModelIndex = function (start) {
    // Up all the indices of models with index greater than start
  for (var i = 0; i < this._modelIndices.length; i++) {
    if (this._modelIndices[i] >= start) {
      this._modelIndices[i] = this._modelIndices[i] + 1;
    }
  }
    // Now add the new one
  this._modelIndices.push(start);
};

/**
 * Splice out the given model index
 * @private
 */
oj.Collection.prototype._spliceModelIndices = function (start, end) {
  var localEnd = (end === undefined) ? start : end;
  this._clearModelIndices(start, localEnd);

  var count = (localEnd - start) + 1;
    // Reduce the indexes of any models above the endpoint by the number of models removed
  for (var i = 0; i < this._modelIndices.length; i++) {
    if (this._modelIndices[i] > localEnd) {
      this._modelIndices[i] -= count;
    }
  }
};

/**
 * Clear the given model index
 * @private
 */
oj.Collection.prototype._clearModelIndices = function (start, end) {
  var localEnd = (end === undefined) ? start : end;
    // Knock out any of the deleted model's indexes from the list
  for (var i = start; i <= localEnd; i++) {
    var idx = this._modelIndices.indexOf(start);
    if (idx > -1) {
      this._modelIndices.splice(idx, 1);
    }
  }
};

/**
 * @private
 */
oj.Collection.prototype._setModel = function (index, model) {
  var oldModel = this._getModel(index);
  this._removePrevNext(oldModel);
  if (!oldModel) {
        // Newly "inserted" model
    this.lruCount = this.lruCount + 1;
  }
  this._setChangeAt(index, 1);
  this._getModels()[index] = model;
  if (model) {
    this._setModelIndex(index);
    model.SetIndex(index);
    this._makeModelHead(model);
  }
};

/**
 * Clean off n models from tail (oldest) of prev/next list
 * @private
 */
oj.Collection.prototype._clearOutModels = function (n) {
  var current = this.tail;
  var index;
  var model;
  var i = 0;

  this.tail = null;
  while (current && i < n) {
        // Erase this model from collection, iff it hasn't changed
    index = current.GetIndex();
    model = this._getModel(index);
    if (!(model && model.hasChanged())) {
      this.lruCount = this.lruCount - 1;
      if (index > -1) {
                // this._getModels()[index] = undefined;
        this._setModel(index, undefined);
        this._clearModelIndices(index, index);
      }

            // Clear its pointers
      current.SetNext(null);
      current = current.SetPrevious(null);
      i += 1;
    } else {
            // Lock down the tail to this one we're not deleting
      if (!this.tail) {
        this.tail = current;
      }
      current = current.GetPrevious();
    }
  }
    // Make sure we set tail if not already set
  if (!this.tail) {
    this.tail = current;
  }
  if (this.lruCount < 0) {
    this.lruCount = 0;
  }
  if (this.lruCount === 0) {
    this.head = null;
    this.tail = null;
  }
};


/**
 * Reset the LRU list
 * @private
 */
oj.Collection.prototype._resetLRU = function () {
  this.lruCount = 0;
  this.head = null;
  this.tail = null;
};

/**
 * Make sure we have room in the LRU
 * @private
 */
oj.Collection.prototype._manageLRU = function (incoming) {
  if (this.IsVirtual()) {
    var limit = this._getModelLimit();
    if (limit > -1) {
      if (this.lruCount + incoming > limit) {
                // Must flush the amount over the limit
        this._clearOutModels((this.lruCount + incoming) - limit);
      }
    }
  }
};

/**
 * @export
 * @desc Return a copy of the Collection
 * @memberof oj.Collection
 *
 * @return {oj.Collection} copy of the Collection
 */
oj.Collection.prototype.clone = function () {
  return this._cloneInternal(true);
};

/**
 * @private
 */
oj.Collection.prototype._cloneInternal = function (withProperties) {
  var c = new this.constructor();
  var i;

    // Only copy locally if virtual
  var model;
  if (this.IsVirtual()) {
    c = this._copyFetchProperties(c);
    c._resetModelsToFullLength(this.totalResults);
  }

  c = this._copyProperties(c);
  if (withProperties) {
        // Try to copy models only if told to--we may only need the shell of the collection with properties
        // Make a copy of the model indices (it will be modified by this process)
    var indices = [];
    for (i = 0; i < this._modelIndices.length; i++) {
      indices.push(this._modelIndices[i]);
    }
        // Sort them in reverse order so we eliminate the higher indexes first so as not to disrupt the position of
        // the earlier ones
    indices.sort(function (a, b) {
      return a - b;
    });

    var index;
    for (i = 0; i < indices.length; i++) {
      index = indices[i];
      model = this._atInternal(index, null, true, false);
      if (model) {
        c._addInternal(model.clone(), { at: index }, true, false);
      }
    }
  }
  return c;
};

/**
 * Copy critical properties in clone
 * @private
 */
oj.Collection.prototype._copyProperties = function (collection) {
  var props = ['comparator', 'model', 'modelId'];
  var prop;
  var i;

  for (i = 0; i < props.length; i++) {
    prop = props[i];
    collection[prop] = this[prop]; // eslint-disable-line no-param-reassign
  }
  return collection;
};

/**
 * Copy critical fetch properties in clone
 * @private
 */
oj.Collection.prototype._copyFetchProperties = function (collection) {
  var props = ['totalResults', 'hasMore', oj.Collection._FETCH_SIZE_PROP];
  var prop;
  var i;

  for (i = 0; i < props.length; i++) {
    prop = props[i];
    collection[prop] = this[prop]; // eslint-disable-line no-param-reassign
  }
  return collection;
};

/**
 * @private
 */
oj.Collection.prototype._getLength = function () {
  return this.length;
};

/**
 * @private
 */
oj.Collection.prototype._setLength = function () {
  var modelsLen = this._getModelsLength();
  this.length = modelsLen;
  if (!this.IsVirtual()) {
    this.totalResults = modelsLen;
  }
};

/**
 * Create a model instance using the model property if set
 * @param {Object} collection
 * @param {Object=} attrs
 * @param {Object=} options
 * @returns {null|Object}
 * @private
 */
oj.Collection._createModel = function (collection, attrs, options) {
  if (collection.model) {
    return $.isFunction(collection.model) ? new collection.model(attrs, options) : // eslint-disable-line new-cap
                                                   new collection.model.constructor(attrs, options); // eslint-disable-line new-cap
  }
  return null;
};

/**
 * @private
 */
oj.Collection.prototype._newModel = function (m, parse, options, ignoreDefaults) {
  var newModel = null;
  var validationValue;
  var opt = options || {};

//    opt.noparse = !parse;
  opt.ignoreDefaults = ignoreDefaults;

  if (m instanceof oj.Model) {
    newModel = m;
  } else if (this.model) {
            // model is defined
    newModel = oj.Collection._createModel(this, m, opt);
  } else {
            // Set this collection on the model
    opt.collection = this;
    newModel = new oj.Model(m, opt);
  }
    // Validate
  if (opt.validate && newModel.validate) {
    validationValue = newModel.validate(newModel.attributes);
    if (validationValue) {
      opt.validationError = validationValue;
      this.TriggerInternal(false, oj.Events.EventType.INVALID, newModel, validationValue, opt);
      return null;
    }
  }
  return newModel;
};

/**
 * Add a model or models to the end of the collection.<br>
 * Events:<p>
 * <ul>
 * <b>add</b>: fired for each model added, passing the collection, model added, and options<br>
 * <b>alladded</b>: fired after all models have been added, passing the collection, array of models added, and
 * options<br>
 * </ul>
 * <p>
 * Note that for virtual collections, if a new model is added after being saved up to the server, no add event
 * will be fired as the
 * collection will already "see" the model as existing.  Note that a warning will be logged if this add is not
 * a force, not merging, and duplicate IDs are found.
 * @param {oj.Model|Object|Array.<Object>|Array.<oj.Model>} m Model object (or array of models) to add. These can be already-created instance of
 * the oj.Model object, or sets of attribute/values, which will be wrapped by add() using the collection's model.
 * @param {Object=} options <b>silent</b>: if set, do not fire events<br>
 *                          <b>at</b>: splice the new model into the collection at the value given (at:index)<br>
 *                          <b>merge</b>: if set, and if the given model already exists in the collection (matched
 *                          by [id]{@link oj.Model#id}), then merge the attribute/value sets, firing change
 *                          events<br>
 *                          <b>sort</b>: if set, do not re-sort the collection even if the comparator is set.<br>
 *                          <b>force</b>: if set to true, do an add to the collection no matter whether the item is
 *                          found or not<br>
 *                          <b>deferred</b>: if true, return a promise as though this collection were virtual
 *                          whether it is or not<br>
 *
 * @returns {Promise.<Array>|Array.<oj.Model>} The model or models added to the collection (or found/merged if appropriate).  If
 * deferred or virtual, return the model or models added in a promise when the set has completed
 * @memberof oj.Collection
 * @ojsignature [{target: "Type", value:"{silent?: boolean, at?: number, merge?: boolean, sort?: boolean, force?: boolean, deferred?: boolean,
 *               [propName: string]: any}", for: "options"},
 *               {target: "Type", value: "Promise<Array<oj.Model>>|Array<oj.Model>", for: "returns"}]
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.add = function (m, options) {
  this._manageLRU(1);
  var opt = options || {};
  return this._handlePromise(this._addInternal(m, options, false, opt.deferred));
};

/**
 * fillIn: true indicates that we're just trying to use add() after a fetch to
 * insert into a preallocated list of models, not truly do an add/merge from the API
 * @private
 */
oj.Collection.prototype._addInternal = function (m, options, fillIn, deferred) {
    // Get options
  var opt = options || {};
  var modelArray = [];
  var at = opt.at;
  var silent = opt.silent;
  var force = opt.force;
  var i;
  var index;
  var cid;
  var merge = opt.merge || false;
  var sort = opt.sort;
  var needSort = true;
  var added = false;
  var addedModels = [];
  var modelReturnList = [];

  if (at !== undefined && at < 0) {
        // Normalize it using the length-- another BackboneJS test case
    at += this._getLength() + 1;
  }

  if (m instanceof Array) {
    modelArray = m;
  } else {
    modelArray.push(m);
  }

  function addToCollection(collection, newModel) {
    if (at === undefined) {
      collection._pushModels(newModel);
      index = collection._getModelsLength() - 1;
      collection._getModel(index).SetCid();
    } else {
      index = at;
      if (collection.IsVirtual() && fillIn) {
                // Array has been preallocated in this case
        collection._setModel(index, newModel);
      } else {
        collection._spliceModels(index, 0, newModel);
      }
      collection._getModel(index).SetCid();
            // Increment at so that later models will be added right after their predecessors, if an array is
            // passed in
      at += 1;
    }
    if (newModel.GetCollection() === undefined) {
      newModel.SetCollection(collection);
    }
    collection._setLength();
    collection._listenToModel(newModel);
    added = true;
  }

  function resortAndFireEvents(collection, existingModel, modelFoundInCollection,
    newModel, resortDeferred) {
    // Now resort if required (don't resort if either told not to, or if 'at' option is set)
    // and if there's more than one model
    var resortOpt = opt || {};
    if (fillIn) {
      resortOpt.fillIn = true;
    }
    var comparator = resortOpt.comparator || collection._hasComparator();
        // If we're filling in to a blank, also check that there's no comparator
    var fillInSort = !fillIn || (fillIn && comparator);
    if (fillInSort && needSort && existingModel === undefined && !sort &&
        at === undefined && collection._getLength() > 1) {
      if (index > -1) {
        cid = collection._getModel(index).cid;
      }
      var sortOpt = {};
      oj.CollectionUtils.copyInto(sortOpt, resortOpt);
      sortOpt.add = true;
      collection.sort(sortOpt);
            // Reset index--can't get it back if virtual--set to -1
      if (index > -1) {
        if (collection.IsVirtual()) {
          index = -1;
        } else {
          index = collection.indexOf(collection.getByCid(cid), resortDeferred);
        }
      }
    }

    if (added) {
      // Pass index property in resortOpt, if at is specified
      if (resortOpt.at) {
        resortOpt.index = index;
      }
      if (newModel) {
        newModel.TriggerInternal(silent, oj.Events.EventType.ADD, newModel, collection, resortOpt);
        addedModels.push(newModel);
      } else {
        modelFoundInCollection.TriggerInternal(silent, oj.Events.EventType.ADD,
          modelFoundInCollection, collection, resortOpt);
        addedModels.push(modelFoundInCollection);
      }
    }
  }

  function mergeAttrs(collection, modelToTryAndMerge, modelFoundInCollection, newModel,
    mergeDeferred) {
    var existingModel;
    var modelAdded = null;

    if (!force && merge && modelFoundInCollection) {
            // Try to merge the attributes--we're merging and the model (by id) was already in the collection
      needSort = modelFoundInCollection.Merge(modelToTryAndMerge, collection.comparator, silent);
      modelAdded = modelFoundInCollection;
    } else {
            // Make sure model is not already in there
      if (!force) {
        if (fillIn) {
                  // Only bother if we have a real id set--comparing cids on a fill in is useless
          existingModel = !newModel.isNew() ? collection._getLocal(newModel) : undefined;
        } else {
          existingModel = collection._getLocal(newModel);
        }
        if (existingModel && fillIn && at !== existingModel.index) {
                    // We're filling in a virtual collection: we should *not* be finding the new model already in
                    // the collection if we're not merging and not forcing: this indicates duplicate ids
                    // throw new Error("Duplicate IDs fetched or added without merging");
          Logger.warn('Duplicate ID fetched or added without merging, the id = ' +
                                   existingModel.GetId());
        }
      }

      if (existingModel === undefined) {
        addToCollection(collection, newModel);
        modelAdded = newModel;
      } else {
        modelAdded = existingModel;
      }
    }

    resortAndFireEvents(collection, existingModel, modelFoundInCollection, newModel, mergeDeferred);

    return modelAdded;
  }

  function doAdd(collection, model, addDeferred) {
    added = false;
    var newModel = collection._newModel(model, true, opt, false);
    var modelToTryAndMerge = null;
    var modelFoundInCollection = null;
    if (newModel != null) {
      index = -1;
            // Make sure ID is up to date
      newModel.SetupId();

            // Use original model array not cloned model if merging--otherwise we won't find the model in the
            // collection
      modelToTryAndMerge = model instanceof oj.Model ? model : newModel;
      if (addDeferred) {
        if (force) {
          return new Promise(function (resolve) {
            var mergedModel = mergeAttrs(collection, modelToTryAndMerge, undefined, newModel,
              addDeferred);
            modelReturnList.push(mergedModel);
            resolve(mergedModel);
          });
        }
        return collection._getInternal(modelToTryAndMerge, { silent: true },
          addDeferred, true).then(
            function (modInfo) {
              modelFoundInCollection = modInfo.m;
              var mod = mergeAttrs(collection,
                   modelToTryAndMerge, modelFoundInCollection,
                   newModel, addDeferred);
              modelReturnList.push(mod);
            });
      }
      if (!force && merge) {
                // Grab the actual model we want to merge from the collection, if the caller has indicated that
                // we aren't forcing an add and we want to merge
        modelFoundInCollection = fillIn ? collection._getLocal(modelToTryAndMerge) :
                                                  collection.get(modelToTryAndMerge);
      }
      var modelAdded = mergeAttrs(collection, modelToTryAndMerge, modelFoundInCollection, newModel,
                                  addDeferred);
      if (modelAdded) {
        modelReturnList.push(modelAdded);
      }
    } else {
            // Add boolean falses for invalid models
      modelReturnList.push(false);
    }
    return Promise.resolve();
  }

  function _parse(collection, array) {
        // Must stop parsing if coming in from reset or constructor
    if (collection.parse && opt.parse && !opt.noparse) {
      return collection.parse(array);
    }
    return array;
  }

  if (!fillIn && (this.IsVirtual() || deferred)) {
    var self = this;
    return new Promise(function (allResolve, allReject) {
      var doTask = function (loc) {
        return new Promise(function (resolve, reject) {
          doAdd(self, modelArray[loc], true).then(function () {
            resolve(loc + 1);
          }, reject);
        });
      };

      var currentStep = Promise.resolve(0);

      modelArray = _parse(self, modelArray);
      for (i = 0; i < modelArray.length; i++) {
        currentStep = currentStep.then(doTask);
      }
      currentStep.then(function () {
        if (addedModels.length > 0) {
          self.TriggerInternal(opt.silent, oj.Events.EventType.ALLADDED, self, addedModels,
                                         opt);
        }
        allResolve(oj.Collection._returnModels(modelReturnList));
      }, allReject);
    });
  }

  modelArray = _parse(this, modelArray);
  for (i = 0; i < modelArray.length; i++) {
    doAdd(this, modelArray[i], false);
  }
  if (addedModels.length > 0) {
    this.TriggerInternal(opt.silent, oj.Events.EventType.ALLADDED, this, addedModels, opt);
  }
  return oj.Collection._returnModels(modelReturnList);
};

/**
 * @private
 */
oj.Collection._returnModels = function (modelReturnList) {
  if (modelReturnList.length === 1) {
    return modelReturnList[0];
  }
  return modelReturnList;
};

/**
 * @private
 */
oj.Collection.prototype._hasComparator = function () {
  return oj.Collection._defined(this.comparator);
};

/**
 * Sort the models in the collection.  For virtual collections, any locally stored models are cleaned out in
 * preparation for new fetches sending server-sorted models down to the client.  No fetch is automatically
 * performed.<p>
 * For non-virtual collections, the models are sorted based on the comparator property.<p>
 * See [comparator}{@link oj.Collection#comparator}<p>
 * For virtual collections, all sorting must be handled by the server.  If the string (attribute) comparator and
 * sortDirection are set, then this information
 * will be passed back via URL parameters, or passed to the customURL callback for application-construction of the
 * appropriate URL.  Function-based custom comparators are ignored in virtual collections.
 * Events:<p>
 * Fires a sort event, passing the collection and options<p>
 * @param {Object=} options <b>silent</b>: if true, do not fire the sort event<br>
 * <b>startIndex</b>: if provided, and if collection is virtual, do a fetch of startIndex+fetchSize immediately
 * after the sort and return a promise.  See [comparator}{@link oj.Collection#setRangeLocal}<p>
 * @return {Promise|null} if virtual and if startIndex is provided in options, a promise Object that resolves upon
 * completion.  The promise will be passed an Object containing start and count properties that represent
 * the *actual* starting position (start), count (count), and array (models) of the Models fetched, which may be
 * fewer than what was requested.  The promise will be rejected on an error and will pass the ajax status,
 * xhr object, error, and collection, if relevant.
 *
 * @memberof oj.Collection
 * @ojsignature  [{target: "Type", value:"{silent?: boolean, startIndex?: number, [propName: string]: any}", for: "options"},
 *                {target: "Type", value: "Promise<oj.Collection.SetRangeLocalPromise>|null", for: "returns"}]
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.sort = function (options) {
  var opt = options || {};
  var silent = opt.silent;
  var comparator = this.comparator;
  var self;
  var eventOpts;

    // Check for comparator
  if (!this._hasComparator()) {
    return null;
  }

    // This is a no-op in case of virtualization: we should just clear things out so that
    // any elements will be refetched
  if (this.IsVirtual()) {
    var totalResults = this.totalResults;
    if (this._hasTotalResults()) {
            // Make sure to set up the array if the length changes (i.e., from 0 to totalResults--need to
            // preallocate)
      this._setModels(new Array(totalResults), true);
    } else {
            // No totalresults
      this._setModels([], true);
      this._resetLRU();
      this._setLength();
    }
    eventOpts = opt.add ? { add: true } : null;
    this.TriggerInternal(silent, oj.Events.EventType.SORT, this, eventOpts, null);
    var startIndex = opt.startIndex;
    if (startIndex !== undefined && startIndex !== null) {
      return this.setRangeLocal(startIndex, this._getFetchSize(opt));
    }
    return null;
  }

  self = this;
  this._getModels().sort(function (a, b) {
    return oj.Collection.SortFunc(a, b, comparator, self, self);
  });
  this._realignModelIndices(0);
    // Indicate this sort is due to an add
  eventOpts = opt.add ? { add: true } : null;
  this.TriggerInternal(silent, oj.Events.EventType.SORT, this, eventOpts, null);
  return null;
};

/**
 * @private
 */
oj.Collection._getKey = function (val, attr) {
  if (val instanceof oj.Model) {
    return val.get(attr);
  }
  return oj.Model.GetPropValue(val, attr);
};

/**
 * @private
 */
oj.Collection.SortFunc = function (a, b, comparator, collection, self) {
  var keyA;
  var keyB;
  var i;
  var retVal;

  if ($.isFunction(comparator)) {
        // How many args?
    if (comparator.length === 1) {
            // "sortBy" comparator option
      keyA = comparator.call(self, a);
      keyB = comparator.call(self, b);
      var attrs1 = oj.StringUtils.isString(keyA) ? keyA.split(',') : [keyA];
      var attrs2 = oj.StringUtils.isString(keyB) ? keyB.split(',') : [keyB];
      for (i = 0; i < attrs1.length; i++) {
        retVal = oj.Collection._compareKeys(attrs1[i], attrs2[i], collection.sortDirection);
        if (retVal !== 0) {
          return retVal;
        }
      }
    }
        // "sort" comparator option
    return comparator.call(self, a, b);
  }
    // String option
  if (oj.StringUtils.isString(comparator)) {
    var attrs = comparator.split(',');

    for (i = 0; i < attrs.length; i++) {
      keyA = oj.Collection._getKey(a, attrs[i]);
      keyB = oj.Collection._getKey(b, attrs[i]);
      retVal = oj.Collection._compareKeys(keyA, keyB, collection.sortDirection);
      if (retVal !== 0) {
        return retVal;
      }
    }
  }
  return 0;
};

/**
 * Return the index at which the given model would be inserted, using the collection comparator See
 * [sort}{@link oj.Collection#sort}.  Not supported for virtual collections.
 *
 * @param {string|function(oj.Model,oj.Model=):Object=} comparator optional comparator to override the default
 * @returns {number} index at which model would be inserted.  -1 if no comparator
 * @throws {Error} when called on a virtual collection
 * @memberof oj.Collection
 * @export
 * @since 1.0.0
 */
oj.Collection.prototype.sortedIndex = function (model, comparator) {
  var comp = comparator || this.comparator;
  var self;
  var test;

    // Check for comparator
  if (!comp) {
    return -1;
  }

  this._throwErrIfVirtual('sortedIndex');

  self = this;
  test = function (a, b) {
    var keyA;
    var keyB;

    if ($.isFunction(comp)) {
                // How many args?
      if (comp.length === 1) {
                    // "sortBy" comparator option
        keyA = comp.call(self, a);
        keyB = comp.call(self, b);
        var attrs1 = oj.StringUtils.isString(keyA) ? keyA.split(',') : [keyA];
        var attrs2 = oj.StringUtils.isString(keyB) ? keyB.split(',') : [keyB];
        var retVal;
        var i;
        for (i = 0; i < attrs1.length; i++) {
          retVal = oj.Collection._compareKeys(attrs1[i], attrs2[i], self.sortDirection);
          if (retVal !== 0) {
            return retVal;
          }
        }
      }
                // "sort" comparator option
      return comp.call(self, a, b);
    }
            // String option
    if (oj.StringUtils.isString(comp)) {
      keyA = a.get(comp);
      keyB = b.get(comp);
      return oj.Collection._compareKeys(keyA, keyB, self.sortDirection);
    }
    return 0;
  };
  return oj.Collection._find(this._getModels(), model, test);
};

/**
 * Binary search and return the index at which model would be inserted into sorted modelArray
 * @private
 */
oj.Collection._find = function (modelArray, model, comparator) {
  function search(min, max) {
    var cid;
    var id;
    var mid;

    if (min > max) {
      return -1;
    }

    cid = model.GetCid();
    id = model.GetId();
    if (modelArray[min].Match(id, cid)) {
      return min;
    }
    if (modelArray[max].Match(id, cid)) {
      return max;
    }

    mid = Math.floor((max + min) / 2);
    if (comparator(modelArray[mid], model) === -1) {
      return search(min + 1, mid);
    }
    if (comparator(modelArray[mid], model) === 1) {
      return search(mid, max - 1);
    }
    return mid;
  }

  return search(0, modelArray.length - 1);
};

/**
 * @private
 */
oj.Collection._compareKeys = function (keyA, keyB, sortDirection) {
  if (sortDirection === -1) {
    if (keyA < keyB) {
      return 1;
    }
    if (keyB < keyA) {
      return -1;
    }
  } else {
    if (keyA > keyB) {
      return 1;
    }
    if (keyB > keyA) {
      return -1;
    }
  }
  return 0;
};


/**
 * Add the given model to the front of the collection<br>
 * For events that may be fired, see [add]{@link oj.Collection#add}.<br>
 * @param {oj.Model|Object} m model (or set of attribute/value pairs) to add to the beginning of the collection
 * @param {Object=} options See [add]{@link oj.Collection#add}
 * @returns {Promise.<Array>|Array.<oj.Model>} The model or models added to the collection.  If
 * deferred or virtual, return the model or models added in a promise when the set has completed
 * @memberof oj.Collection
 * @ojsignature [{target: "Type", value:"{silent?: boolean, at?: number, merge?: boolean, sort?: boolean, force?: boolean, deferred?: boolean,
 *               [propName: string]: any}", for: "options"},
 *               {target: "Type", value: "Promise<Array<oj.Model>>|Array<oj.Model>", for: "returns"}]
 * @export
 * @since 1.0.0
 */
oj.Collection.prototype.unshift = function (m, options) {
    // Like an add but set 'at' to zero if not specified
  var opt = {};
  oj.CollectionUtils.copyInto(opt, options || {});
  if (!opt.at) {
    opt.at = 0;
  }
  this._manageLRU(1);
  return this._handlePromise(this._addInternal(m, opt, false, opt.deferred));
};

/**
 * @private
 */
oj.Collection.prototype._handlePromise = function (result) {
  if ($.isFunction(result.then)) {
    return this._addPromise(function () {
      return result;
    });
  }
  return result;
};

/**
 * Remove the first model from the collection and return it.<br>
 * For events that may be fired, see [remove]{@link oj.Collection#remove}.<br>
 * @param {Object=} options same as remove
 * @property {boolean=} silent if set, do not fire events
 * @property {boolean=} deferred if true, return a promise as though this collection were virtual whether it is or not
 * @return {oj.Model|Promise.<oj.Model>|null} model that was removed.  If this is a virtual collection, this will return a promise
 *                  which will resolve passing the model value that was removed
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.shift = function (options) {
  var opt = options || {};
  var deferred = this._getDeferred(opt);
  var retVal;
  if (this.IsVirtual() || deferred) {
    var self = this;
    return this._atInternal(0, opt, false, true).then(function (model) {
      retVal = self._removeInternal(model, 0, opt);
      self.TriggerInternal(opt.silent, oj.Events.EventType.ALLREMOVED, self, [retVal], opt);

      return retVal;
    });
  }
  retVal = this._removeInternal(this.at(0), 0, opt);
  this.TriggerInternal(opt.silent, oj.Events.EventType.ALLREMOVED, this, [retVal], opt);
  return retVal;
};

/**
 * @export
 * @desc Return an array of models found in the Collection, excepting the last n.
 * @memberof oj.Collection
 * @param {number=} n number of models to leave off the returned array; defaults to 1
 * @return {Array.<oj.Model>} array of models from 0 to the length of the collection - n - 1
 * @throws {Error} when called on a virtual collection
 * @since 1.0.0
 */
oj.Collection.prototype.initial = function (n) {
  var index = (n === undefined) ? 1 : n;

  this._throwErrIfVirtual('initial');

  var array = [];
  var i;
  for (i = 0; i < this._getLength() - index; i += 1) {
    array.push(this.at(i));
  }
  return array;
};

/**
 * @private
 */
oj.Collection.prototype._getDeferred = function (options) {
  var opt = options || {};
  return opt.deferred;
};

/**
 * Return the last model in the collection.  If n is passed in, then the last n models are returned as an array
 * Note that if the collection is virtual, and totalResults is not returned by the server, the results returned
 * by last can be difficult to predict.  They depend on the fetch sizes, last known offset of a fetch, etc.
 * If code is using a server that does not return totalResults the use of last is not recommended.<br>
 * For events that may be fired if the collection is virtual, see [fetch]{@link oj.Collection#fetch}.<br>
 * @param {number=} n number of models to return.  Defaults to 1
 * @param {Object=} options
 * @property {boolean=} deferred if true, return a promise as though this collection were virtual whether it is or not
 * @return {Promise.<oj.Model>|Array.<oj.Model>|null} array of n models from the end of the Collection.  If this is a virtual collection,
 *                             this will return a promise which will resolve passing the array or single model
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.last = function (n, options) {
  var deferred = this._getDeferred(options);
  var index = (n === undefined) ? 1 : n;
  var len;

  if (index === 1) {
    len = this._getModelsLength();
    if (len === 0) {
      len = index;
    }
    if (len > 0) {
      return this._atInternal(len - 1, options, false, deferred);
    }
    return null;
  }

  var array = [];
  var i;
  len = this._getLength();
  if (deferred || this.IsVirtual()) {
        // Loop using deferred
    var start = len - n;

        // Handle edge or no totalResults cases
    if (start < 0) {
      start = 0;
    }
    if (len === 0) {
            // No totalresults, probably
      len = n;
    }

    var self = this;
    return this._addPromise(function () {
      return self.IterativeAt(start, len);
    });
  }

  for (i = len - n; i < len; i += 1) {
    array.push(this.at(i));
  }
  return array;
};

/**
 * Loop calling at() in a deferred way and return a promise to be resolved when all the elements are sequentially
 * fetched
 * @protected
 */
oj.Collection.prototype.IterativeAt = function (start, end) {
  var array = [];
  var i;
  var self = this;
  return new Promise(function (allResolve, allReject) {
    var doTask = function (index) {
            // Make sure we're not asking beyond what we know the server can deliver, if virtual
      if (self.IsVirtual() && self._hasTotalResults() && index >= self.totalResults) {
        return Promise.resolve(index + 1);
      }
      return new Promise(function (resolve, reject) {
        self._deferredAt(index, null).then(function (model) {
          array.push(model);
          resolve(index + 1);
        }, reject);
      });
    };

    var currentStep = Promise.resolve(start);
    for (i = start; i < end; i++) {
      currentStep = currentStep.then(doTask);
    }
    currentStep.then(function () {
      allResolve(array);
    }, allReject);
  });
};

/**
 * @private
 */
oj.Collection.prototype._getDefaultFetchSize = function (n) {
  if (n === undefined || n === null) {
    return this[oj.Collection._FETCH_SIZE_PROP];
  }

  return n;
};

/**
 * @private
 */
oj.Collection.prototype._calculateNextStart = function () {
  var lastFetch = this.lastFetchCount;
  if (lastFetch === undefined || lastFetch === null) {
    lastFetch = this[oj.Collection._FETCH_SIZE_PROP];
  }
  if (this.offset === undefined || this.offset === null) {
        // Assume zero offset (0+lastFetch)
    return lastFetch;
  }
  return this.offset + lastFetch;
};

/**
 * Fetch the next set of models from the server.<br>
 * For events that may be fired if the collection is virtual, see [fetch]{@link oj.Collection#fetch}.<br>
 *
 * @param {number} n number of models to fetch.  If undefined or null, the collection will attempt to use the
 * overall [fetchSize]{@link oj.Collection#fetchSize} property value
 * @param {Object=} options Options to control next<p>
 *                  <b>success</b>: a user callback called when the fetch has completed successfully. This makes
 *                  the fetch an asynchronous process. The callback is called passing the Collection object, raw
 *                  response, and the options argument.<br>
 *                  <b>error</b>: a user callback function called if the fetch fails. The callback is called
 *                  passing the collection object, xhr, and options arguments.<br>
 * @return {Object|null} xhr ajax object, by default.  null if nothing to fetch (the success callback will still be
 * called).  Note if [sync]{@link oj.Collection#sync} has been replaced, this would be the value returned by the
 * custom implementation.
 * @memberof oj.Collection
 * @since 1.0.0
 * @ojsignature {target: "Type", value:"{success?: (collection: oj.Collection, response: any, options: object)=> void,
 *                                                  error?: (collection: oj.Collection, xhr: any, options: object)=> void, [propName: string]: any}", for: "options"}
 * @export
 */
oj.Collection.prototype.next = function (n, options) {
  var opt = options || {};
  opt[oj.Collection._FETCH_SIZE_PROP] = this._getDefaultFetchSize(n);

  var start = this._calculateNextStart();
  var length = this._getLength();
  if (length === 0 && opt[oj.Collection._FETCH_SIZE_PROP] > 0) {
        // If we have a fetch size and we have no length let next() do a fetchSize fetch starting at zero to
        // kick things off
    start = 0;
  } else if (start >= length) {
        // No op -- still call success because the items are already fetched.
    var self = this;
    if (opt.success) {
      opt.success.call(oj.Model.GetContext(opt, self), self, null, opt);
    }

    return null;
  }
  opt.startIndex = start;
  return this.fetch(opt);
};

/**
 * @private
 */
oj.Collection.prototype._calculatePrevStart = function (n) {
  if (this.offset === undefined || this.offset === null) {
        // Assume zero: we can't back up beyond that so if the offset wasn't set there's nothing to do
    return 0;
  }
  return this.offset - n;
};

/**
 * Fetch the previous set of models from the server.<br>
 * For events that may be fired if the collection is virtual, see [fetch]{@link oj.Collection#fetch}.<br>
 * @param {number} n number of models to fetch.  If undefined or null, the collection will attempt to use the
 * overall fetchSize property value
 * @param {Object=} options Options to control previous<p>
 *                  <b>success</b>: a user callback called when the fetch has completed successfully. This makes
 *                  the fetch an asynchronous process. The callback is called passing the Collection object, raw
 *                  response, and the options argument.<p>
 *                  <b>error</b>: a user callback function called if the fetch fails. The callback is called
 *                  passing the collection object, xhr, and options arguments.<p>
 * @return {Object} xhr ajax object, by default. null if there is nothing earlier to fetch (no fetch has happened
 * or the last fetch started at 0).  The success callback will still be called.  Note if
 * [sync]{@link oj.Collection#sync} has been replaced, this would be the value returned by the custom
 * implementation.
 * @memberof oj.Collection
 * @since 1.0.0
 * @ojsignature {target: "Type", value:"{success?: (collection: oj.Collection, response: any, options: object)=> void,
 *                                                  error?: (collection: oj.Collection, xhr: any, options: object)=> void, [propName: string]: any}", for: "options"}
 * @export
 */
oj.Collection.prototype.previous = function (n, options) {
  var opt = options || {};
  if (this.offset === 0) {
        // No op -- still call success (if we've fetched before--lastFetchCount is other than zero) because the
        // items are already fetched.
    var self = this;
    if (opt.success && this.lastFetchCount) {
      opt.success.call(oj.Model.GetContext(opt, self), self, null, opt);
    }

    return null;
  }
  opt[oj.Collection._FETCH_SIZE_PROP] = this._getDefaultFetchSize(n);
  var start = this._calculatePrevStart(opt[oj.Collection._FETCH_SIZE_PROP]);
  if (start < 0) {
        // Only fetch from 0 to the last fetch's starting point...
    opt[oj.Collection._FETCH_SIZE_PROP] = this.offset;
    start = 0;
  }
  opt.startIndex = start;
  return this.fetch(opt);
};


/**
 * Set or change the number of models held at any one time
 *
 * @param {number} n maximum number of models to keep at a time
 * @return {undefined}
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.setModelLimit = function (n) {
  this.modelLimit = n;
    // Clean out down to the new limit, if necessary
  this._manageLRU(0);
};

/**
 * @private
 */
oj.Collection.prototype._getModelLimit = function () {
  return this.modelLimit;
};

/**
 * Set or change the number of models to fetch with each server request
 *
 * @param {number} n number of models to fetch with each request
 * @return {undefined}
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.setFetchSize = function (n) {
  this[oj.Collection._FETCH_SIZE_PROP] = n;
};


/**
 * Return the array of models found in the Collection starting with index n.<br>
 * For events that may be fired if the collection is virtual, see [fetch]{@link oj.Collection#fetch}.<br>
 * @param {number=} n index at which to start the returned array of models.  Defaults to 1.
 * @param {Object=} options
 * @property {boolean=} deferred if true, return a promise as though this collection were virtual whether it is or not
 * @return {Array.<oj.Model>|Promise} array of models from the collection.  If this is a virtual collection, or
 *                  if deferred is passed as true, return a promise which resolves passing the array of models.
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.rest = function (n, options) {
  var deferred = this._getDeferred(options);
  var index = (n === undefined) ? 1 : n;

  var array = [];
  var i;
    // TODO
  if (this.IsVirtual() || deferred) {
    var self = this;
    return this._addPromise(function () {
      return self.IterativeAt(index, self._getLength());
    });
  }

  for (i = index; i < this._getLength(); i += 1) {
    array.push(this.at(i));
  }
  return array;
};

/**
 * Remove a model from the collection, if found.<br>
 * Events:<p>
 * <ul>
 * <b>remove</b>: fired for each model removed, passing the model removed, collection, and options<br>
 * <b>allremoved</b>: fired after all models have been removed, passing the collection, array of models removed,
 * and options<br>
 * </ul>
 * <p>
 * @param {oj.Model|Array.<oj.Model>} m model object or array of models to remove.
 * @param {Object=} options
 * @property {boolean=} silent if set, do not fire events
 * @return {Array.<oj.Model>|oj.Model} an array of models or single model removed
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.remove = function (m, options) {
  var opt = options || {};
  var modArray = [];
  var mod;

  if (m instanceof Array) {
    modArray = m;
  } else {
    modArray.push(m);
  }

  var modsRemoved = [];
  for (mod = modArray.length - 1; mod >= 0; mod -= 1) {
        // Done to keep array in order matching one passed in--we do removal in reverse
    modsRemoved.unshift(this._removeInternal(modArray[mod], -1, opt));
  }
  this.TriggerInternal(opt.silent, oj.Events.EventType.ALLREMOVED, this, modArray, opt);
  return oj.Collection._returnModels(modsRemoved);
};

/**
 * @private
 */
oj.Collection.prototype._removeInternal = function (model, index, options) {
  var optCopy = options || {};
  var modInfo = index === -1 ? this._getInternal(model) : oj.Collection._getModinfo(index, model);
  var silent = optCopy.silent;

  var n = modInfo.index;
  if (n > -1) {
    var mod = modInfo.m;
        // only unset the collection setting if it's mine
    if (mod !== undefined && mod.GetCollection() === this) {
      mod.SetCollection(null);
    }
    this._spliceModels(n, 1);
    this._setLength();
        // if (!silent) {
    var opt = {};
    oj.CollectionUtils.copyInto(opt, optCopy);
    opt.index = n;
    if (mod !== undefined) {
      mod.TriggerInternal(silent, oj.Events.EventType.REMOVE, mod, this, opt);
    }
      //  }
        // Unlisten after event fired
    this._unlistenToModel(mod);
  }
  return modInfo.m;
};


/**
 * @private
 */
oj.Collection.prototype._unlistenToModel = function (m) {
  if (m !== undefined) {
    m.off(null, null, this);
  }
};

/**
 * @private
 */
oj.Collection.prototype._listenToModel = function (m) {
  m.OnInternal(oj.Events.EventType.ALL, this._modelEvent, this, false, true);
};

/**
 * Handle model destroyed events via the all listener
 * @private
 */
oj.Collection.prototype._modelEvent = function (event, model, collection, options) {
  if (event === oj.Events.EventType.DESTROY) {
    this.remove(model);
  }

    // Don't process general events if we're not the target
  if (collection !== undefined && collection instanceof oj.Collection && collection !== this) {
    return;
  }

    // Throw up to the collection
  var silent = options && options.silent;
  this.TriggerInternal(silent, event, model, collection, options);
};

/**
 * Clear all data from the collection and refetch (if non-virtual).  If virtual, clear all data.
 * If fetch() cannot be called (e.g., there is no URL) then the caller must reload the collection wtih data.
 * The existing data is still cleared.
 * Events (if silent is not set):<p>
 * <ul>
 * <b>refresh</b>: fired passing the collection and options<br>
 * For more events that may be fired if the collection is not virtual, see [fetch]{@link oj.Collection#fetch}.
 * </ul>
 * <p>
 * @param {Object=} options user options<p>
 *                          <b>silent</b>: if set, do not fire a refresh event<p>
 * <b>startIndex</b>: if provided, and if collection is virtual, do a fetch of startIndex+fetchSize immediately
 * after the refresh and return a promise.  See [comparator}{@link oj.Collection#setRangeLocal}<p>
 * @return {Promise.<undefined|Object>} promise object resolved when complete (in case there is a fetch for non-virtual mode).  If
 * startIndex is provided as an option, the returned promise resolution is the same as setRangeLocal.
 * @ojsignature  {target: "Type", value: "Promise<oj.Collection.SetRangeLocalPromise|undefined>", for: "returns"}
 * @memberof oj.Collection
 * @since 1.0.0
 * @ojsignature {target: "Type", value:"{silent?: boolean, startIndex?: number, [propName: string]: any}", for: "options"}
 * @export
 */
oj.Collection.prototype.refresh = function (options) {
  var optCopy = options || {};

  var self = this;
  var silent;
  return this._addPromise(function () {
    return new Promise(function (resolve, reject) {
      if (!self.IsVirtual()) {
        silent = optCopy.silent !== undefined && optCopy.silent;
        try {
                    // Do a reset, with silent
          self.reset(null, { silent: true });
                    // Local: do a fetch to fill back up
                    // In case options are passed to refresh-->fetch
          var opt = {};
          Object.keys(optCopy).forEach(function (prop) {
            if (Object.prototype.hasOwnProperty.call(optCopy, prop)) {
              opt[prop] = optCopy[prop];
            }
          });
          opt.success = function (collection, response, successOpts) {
            self.TriggerInternal(silent, oj.Events.EventType.REFRESH, self, successOpts, null);
            resolve({ collection: collection, response: response, options: successOpts });
          };
          opt.error = function (xhr, status, error) {
            reject(oj.Collection._createRejectionError(xhr, status, error, self, optCopy, false));
          };
          self._fetchInternal(opt, -1, false);
          return;
        } catch (e) {
                    // This is OK if it's a URLError: just fire the event: local collection without custom sync
          if (e instanceof oj.URLError) {
                        // if it's a completely local collection, it's a no-op other than the event
            self.TriggerInternal(silent, oj.Events.EventType.REFRESH, self, optCopy, null);
            resolve({ collection: self, options: optCopy });
            return;
          }
          throw e;
        }
      }
            // Virtual
      var startIndex = optCopy.startIndex;

      self._setModels([], true);
      self._resetLRU();

            // Clear totalresults.
      self.totalResults = undefined;
      self._setLength();

      silent = optCopy.silent !== undefined && optCopy.silent;
      self.TriggerInternal(silent, oj.Events.EventType.REFRESH, self, optCopy, null);
      if (startIndex === undefined || startIndex === null) {
        startIndex = 0;
      }
      if (startIndex !== undefined && startIndex !== null) {
        // Do a set range local
        self._setRangeLocalInternal(startIndex, self._getFetchSize(optCopy)).then(
        function (actual) {
          resolve(actual);
        }, function (err) {
          reject(err);
        });
      } else {
        resolve(undefined);
      }
    });
  });
};

/**
 * Remove and replace the collection's entire list of models with a new set of models, if provided. Otherwise,
 * empty the collection.  totalResults is reset when no new data is passed in, set to the new length in the
 * non-virtual case if data is passed in, and left as was in the virtual case if data is passed in.<br>
 * Events (if silent is not set):<p>
 * <ul>
 * <b>reset</b>: fired passing the collection and options.  The new length of the collection should be in effect
 * when the event is fired<br>
 * For events that may be fired if data is passed in, see [add]{@link oj.Collection#add}.
 * </ul>
 * <p>
 * @param {Object=} data Array of model objects or attribute/value pair objects with which to replace the
 * collection's data.
 * @param {Object=} options user options, passed to event, unless silent<br>
 *                          <b>silent</b>: if set, do not fire events<p>
 * @returns {oj.Model|Array.<oj.Model>} The model or models added to the collection, if passed in
 * @memberof oj.Collection
 * @since 1.0.0
 * @ojsignature {target: "Type", value:"{silent?: boolean, [propName: string]: any}", for: "options"}
 * @export
 */
oj.Collection.prototype.reset = function (data, options) {
  var opts = {};
  oj.CollectionUtils.copyInto(opts, options || {});

  opts.previousModels = this._getModels();

  var index;
  var model;
  for (var i = 0; i < this._modelIndices.length; i++) {
    index = this._modelIndices[i];
    model = this._getModel(index);
    if (model) {
      this._unlistenToModel(model);
      model.SetCollection(null);
    }
  }
  this._setModels([], true);
  this._resetLRU();

  var silent = opts.silent !== undefined && opts.silent;
  if (!data) {
    this._setLength();
        // Clear totalresults
    this.totalResults = undefined;
    this.TriggerInternal(silent, oj.Events.EventType.RESET, this, opts, null);
    return null;
  }

  var retObj = null;
  // Parse collection
  var newData = data;
  if (opts.parse) {
    newData = this.parse(data);
  }

  this._manageLRU((newData instanceof Array) ? newData.length : 1);
  opts.noparse = true;
  retObj = this._addInternal(newData, opts, true, false);
  this._setLength();
  this.TriggerInternal(silent, oj.Events.EventType.RESET, this, opts, null);

  return this._handlePromise(retObj);
};

/**
 * Return the model object found at the given index of the collection, or a promise object that will pass the model
 * as an argument when it resolves.<p>
 * For events that may be fired if the collection is virtual, see [fetch]{@link oj.Collection#fetch}.
 * @param {number} index Index for which to return the model object.
 * @param {Object=} options <b>fetchSize</b>: fetch size to use if the call needs to fetch more records from the
 * server, if virtualized.  Overrides the overall fetchSize setting <br>
 *                  <b>deferred</b>: if true, return a deferred/promise object as described below.  If not
 *                  specified, the type of return value will be determined by whether or not the collection is
 *                  virtual<br>
 *
 * @return {oj.Model|Promise.<oj.Model>|null} Model [model]{@link oj.Model} located at index. If index is out of range, returns null.
 * If this is a virtual collection, or if deferred is specified and true, at will return a Promise object which
 * will resolve passing the model at the given index (or null if out of range)
 * @memberof oj.Collection
 * @since 1.0.0
 * @ojsignature {target: "Type", value:"{fetchSize?: number, deferred?: boolean, [propName: string]: any}", for: "options"}
 * @export
 */
oj.Collection.prototype.at = function (index, options) {
  var deferred = this._getDeferred(options);
  return this._atInternal(index, options, false, deferred);
};

/**
 * Local indicates that only what's stored locally should be returned (if true)--no fetching
 * @private
 */
oj.Collection.prototype._atInternal = function (index, options, local, deferred) {
  var n = index;
  if (n < 0) {
        // Normalize it using the length-- another BackboneJS test case
    n += this._getLength();
  }

  if (n < 0 || this._overUpperLimit(n)) {
    if (!local && (this.IsVirtual() || deferred)) {
      return this._addPromise(function () {
        return Promise.resolve(null);
      });
    }
    return null;
  }
  var self = this;
  if (!local && (this.IsVirtual() || deferred)) {
    return this._addPromise(function () {
      return self._deferredAt(n, options);
    });
  }
  return this._getModel(n);
};

/**
 * Return a promise, the resolution of which indicates that all promise-returning API in process have completed.
 *
 * @returns {Promise} a promise that when resolved, indicates that all unresolved promise-returning API calls
 * made at the moment of the whenReady have completed
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.whenReady = function () {
  if (this._promises) {
        // If we have an active chain, return it
    return this._promises;
  }
    // Otherwise return an immediately resolved promise that means nothing...
  return Promise.resolve();
};

/**
 * Add a task to a chained list of Promises
 * @private
 */
oj.Collection.prototype._addPromise = function (promiseTask) {
  var self = this;
    // Undefined, so set it up initially
  if (this._promises === undefined) {
    this._promiseCount = 0;
    this._promises = Promise.resolve();
  }
    // Track the number we have left to resolve
  this._promiseCount = this._promiseCount + 1;
    // Chain this new promise callback task to the end of the list
  this._promises = this._promises.then(promiseTask.bind(self)).then(
        function (arg) {
            // Resolved successfully--decrement our count and clean up if we have none left to resolve
          self._promiseCount -= 1;
          if (self._promiseCount === 0) {
            self._promises = undefined;
                // Fire the ready event
            self.TriggerInternal(false, oj.Events.EventType.READY, self, null, null);
          }
            // Resolve the true promise with the value we're given
          return arg;
        },
        function (error) {
            // Rejected--decrement our count and clean up if we have none left to resolve
          self._promiseCount -= 1;
          if (self._promiseCount === 0) {
            self._promises = undefined;
          }
            // Reject the promise
          return Promise.reject(error);
        });

    // Return the chain with the new promise at the end
  return this._promises;
};

/**
 * Add an xhr to a list of active xhrs
 * @private
 */
oj.Collection.prototype._addxhr = function (xhr) {
  if (xhr && xhr.abort) {
    if (this._xhrs === undefined) {
      this._xhrs = [];
    }
        // Listen to this xhr to know when to remove it
    var self = this;
    this._xhrs.push(xhr);
    xhr.done(function () {
            // Find the xhr
      var loc = self._xhrs ? self._xhrs.indexOf(xhr) : -1;
      if (loc > -1) {
                // Remove it from the list
        self._xhrs.splice(loc, 1);
      }
    });
  }
};

/**
 * Cancel all xhr requests known to this collection as still pending.  Return a promise that is resolved when all of the requests abort, and all
 * the promises tied to those requests resolve with an error
 *
 * @returns {Promise.<null>} a promise that when resolved, indicates that all the open xhr objects generated by this collection have aborted
 * @memberof oj.Collection
 * @since 2.1.0
 * @export
 */
oj.Collection.prototype.abort = function () {
  // Abort all pending XHR requests
  var self = this;
  function createPromise(index, resolve) {
    self._xhrs[index].then(function (data, status) {
      if (status === 'abort') {
                    // Remove from list
        self._xhrs.splice(index, 1);
                    // If this is the last one, resolve the promise we returned
        if (self._xhrs.length === 0) {
          self.whenReady().then(function () {
            resolve(null);
          }, function () {
            resolve(null);
          });
        }
      }
    }, function () {
      // Remove from list
      self._xhrs.splice(index, 1);
      // If this is the last one, resolve the promise we returned
      if (self._xhrs.length === 0) {
        self.whenReady().then(function () {
          resolve(null);
        }, function () {
          resolve(null);
        });
      }
    });
  }

  if (this._xhrs && this._xhrs.length > 0) {
    return new Promise(function (resolve) {
      // Count down so we can remove them
      for (var i = self._xhrs.length - 1; i >= 0; i--) {
        createPromise(i, resolve);
        self._xhrs[i].abort();
      }
    });
  }
  return Promise.resolve();
};

/**
 * @private
 */
oj.Collection.prototype._deferredAt = function (index, options) {
  var self = this;
    // If it's virtual, we need to see if this item has been fetched or not: if not, we need to fetch it + fetchSize
  var model = self._getModel(index);
  if (model === undefined) {
    return new Promise(function (resolve, reject) {
            // Go fetch
      var opts = {};
      oj.CollectionUtils.copyInto(opts, options || {});
      opts.context = self;
      opts.startIndex = index;
      opts.error = function (xhr, status, error) {
                // Handle potential errors
        reject(oj.Collection._createRejectionError(xhr, status, error, self, options, false));
      };

      opts.success = function () {
        resolve(self._getModel(index));
      };

      self._fetchInternal(opts, -1, false);
    });
  }
  return new Promise(function (resolve) {
    resolve(model);
  });
};

/**
 * Return the first model object from the collection whose client ID is the given model cid
 * @param {string} clientId Client ID (see Model cid) for which to return the model object, if found.
 * @return {oj.Model|null} First model object in the collection where model.cid = clientId. If none are found,
 * returns null.
 *
 * @throws {Error} when called on a virtual Collection if the item isn't found in memory
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.getByCid = function (clientId) {
  var models = this._getModels();
  var index;
  var foundModel = null;
  for (var i = 0; i < this._modelIndices.length; i++) {
    index = this._modelIndices[i];
    if (models[index] && clientId === models[index].cid) {
      foundModel = models[index];
      break;
    }
  }
  if (foundModel) {
    return foundModel;
  }
  if (this.IsVirtual()) {
    throw new Error('Not found locally and not supported by server for virtual collections');
  }
  return null;
};


/**
 * Return the first model object from the collection whose model id value is the given id or cid, or the id or
 * cid from a passed in model
 * Note this method will not function as expected if the id or cid is not set<br>
 * For events that may be fired if the collection is virtual, see [fetch]{@link oj.Collection#fetch}.<br>
 *
 * @param {Object|string} id ID, cid, or Model (see Model id or cid) for which to return the model object, if found.
 * @param {Object=} options
 * @property {number=} fetchSize fetch size to use if the call needs to fetch more records from the server,
 *                  if virtualized.  Overrides the overall fetchSize setting<p>
 * @property {boolean=} deferred if true, return a promise as though this collection were virtual whether it
 *                  is or not
 * @return {oj.Model|null|Promise.<oj.Model>} First model object in the collection where model.id = id or model.cid = id. If
 *                  none are found, returns null.  If deferred or virtual, return a promise passing the model
 *                  when done
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.get = function (id, options) {
  var deferred = this._getDeferred(options);
  var internalGet = this._getInternal(id, options, deferred);
  if (internalGet) {
        // Is this a deferred object?
    if ($.isFunction(internalGet.then)) {
      return this._addPromise(function () {
        return new Promise(function (resolve, reject) {
          internalGet.then(function (modInfo) {
            resolve(modInfo.m);
          },
                    function (err) {
                      reject(err);
                    });
        });
      });
    }
    if (this.IsVirtual()) {
      return this._addPromise(function () {
        return new Promise(function (resolve) {
          resolve(internalGet.m);
        });
      });
    }
    if (Object.prototype.hasOwnProperty.call(internalGet, 'm')) {
      return internalGet.m;
    }
  }
  return null;
};

/**
 * @private
 */
oj.Collection.prototype._getLocal = function (id) {
  var internalGet = this._getLocalInternal(id);
  if (internalGet) {
    return internalGet.m;
  }
  return null;
};

/**
 * @private
 */
oj.Collection.prototype._getLocalInternal = function (id) {
  var cid = id;
  var localId = id;
  if (id instanceof oj.Model) {
        // Get the cid
    cid = id.GetCid();
        // Get the id
    localId = id.GetId();
  } else if (oj.Collection._defined(id) && id.id !== undefined) {
    localId = id.id;
  }
  var foundObj = null;
  var len = this._modelIndices.length;
  var model;
  var models = this._getModels();
  var index;
  for (var i = 0; i < len; i++) {
    index = this._modelIndices[i];
    model = models[index];
    if (model !== undefined && model.Match(localId, cid)) {
      foundObj = oj.Collection._getModinfo(index, model);
      break;
    }
  }
  if (foundObj) {
    return foundObj;
  }
  return oj.Collection._getModinfo(-1, undefined);
};

/**
 * @private
 * @param {Object|null|string} id
 * @param {Object=} options
 * @param {boolean=} deferred
 * @param {boolean=} fillIn
 * @returns {Object}
 */
oj.Collection.prototype._getInternal = function (id, options, deferred, fillIn) {
  var cid = id;
  var localId = id;
  var fill = fillIn === undefined ? false : fillIn;
  if (id instanceof oj.Model) {
    // Get the cid
    cid = id.GetCid();
    // Get the id
    localId = id.GetId();
  } else if (oj.Collection._defined(id) && id.id !== undefined) {
    localId = id.id;
  }

  var foundObj = null;
  var models = this._getModels();
  var index;
  for (var i = 0; i < this._modelIndices.length; i++) {
    index = this._modelIndices[i];
    if (models[index] && models[index].Match(localId, cid)) {
      var retObj = oj.Collection._getModinfo(index, models[index]);
      foundObj = retObj;
      break;
    }
  }

  if (foundObj) {
    if (deferred) {
      return new Promise(function (resolve) {
        resolve(foundObj);
      });
    }
    return foundObj;
  }
    // If virtual, might be undefined because it needs to be fetched
  if (this.IsVirtual()) {
        // Try to fetch using start ID.  cid not supported
    if (localId === undefined && cid !== undefined) {
      return new Promise(function (resolve) {
        resolve(oj.Collection._getModinfo(-1, undefined));
      });
    }
    var self = this;
    return new Promise(function (resolve, reject) {
      var resp = function (response) {
        if (response != null) {
          var ind = self._getOffset();
          // Check that the model at index is the right one
          var model = self._getModel(ind);
          if (model !== undefined && model.Match(localId, cid)) {
            resolve(oj.Collection._getModinfo(ind, model));
          } else {
            resolve(oj.Collection._getModinfo(-1, undefined));
          }
        } else {
          resolve(oj.Collection._getModinfo(-1, undefined));
        }
      };

            // Go fetch
      var opts = {};
      oj.CollectionUtils.copyInto(opts, options || {});
      opts.context = self;
      opts.startID = localId;
      opts.error = function (xhr, status, error) {
                // Handle potential errors
        reject(oj.Collection._createRejectionError(xhr, status, error, self, options, false));
      };
      opts.success = function (collection, response) {
        resp(response);
      };

      self._fetchInternal(opts, -1, fill);
    });
  }

  var undefinedModInfo = oj.Collection._getModinfo(-1, undefined);
  if (deferred) {
    return new Promise(function (resolve) {
      resolve(undefinedModInfo);
    });
  }
  return undefinedModInfo;
};

/**
 * @private
 */
oj.Collection._getModinfo = function (index, model) {
  return { index: index, m: model };
};

/**
 * @private
 */
oj.Collection.prototype._parseImpl = function (response) {
  // Try to interpret ADFbc like controls where a collection is hanging off a property
  if (response instanceof Array) {
    return response;
  }

  if (!response) {
    return response;
  }

  // See if any of the properties contain arrays
  var prop;
  for (prop in response) { // eslint-disable-line no-restricted-syntax
    if (Object.prototype.hasOwnProperty.call(response, prop)) {
      if (response[prop] instanceof Array) {
        return response[prop];
      }
    }
  }
  return response;
};

/**
 * Optional callback to parse responses from the server.  It is called with the server's response and should return a response (possibly modified) for processing
 * @ojstatus preview
 * @type {function(Object):Object}
 * @since 1.0.0
 * @memberof oj.Collection
 * @export
 */
oj.Collection.prototype.parse = oj.Collection.prototype._parseImpl;

/**
 * Determine if actual means we are complete
 * @private
 */
oj.Collection.prototype._checkActual = function (start, count, actual) {
    // Are we at the end with what actually came back?  Then this request should satisfy the setLocalRange
  if (this._hasTotalResults() && (actual.start + actual.count >= this.totalResults)) {
    return true;
  }
  return (actual.start === start && actual.count === count);
};

/**
 *
 * Tell the collection to try and ensure that the given range is available locally.  Note that if the collection
 * is virtual, setRangeLocal may make repeated fetch calls to the server to satisfy the request, stopping only
 * when the range has been made local or the end of the records on the server is reached.<br>
 * For events that may be fired if the collection is virtual, see [fetch]{@link oj.Collection#fetch}.<br>
 *
 * @param {number} start starting index to make local
 * @param {number} count number of elements to make local
 * @param {Object=} options Options to control whether events are fired by this call (silent: true)
 * @return {Promise} a promise Object that resolves upon completion.  The promise will be passed an Object
 * containing start and count properties that represent the *actual* starting position (start), count (count),
 * and array (models) of the Models fetched, which may be fewer than what was requested.  The promise will be
 * rejected on an error and will pass the ajax status, xhr object, error, and collection, if relevant.
 * @memberof oj.Collection
 * @ojsignature  [{target: "Type", value:"{silent?: boolean}", for: "options"}, {target: "Type", value: "Promise<oj.Collection.SetRangeLocalPromise>", for: "returns"}]
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.setRangeLocal = function (start, count, options) {
  var self = this;
  return this._addPromise(function () {
    return self._setRangeLocalInternal(start, count, options);
  });
};

/**
 * @typedef {Object} oj.Collection.SetRangeLocalPromise
 * @property {number} start starting index of fetched models
 * @property {number} count number of models fetched
 * @property {Array.<oj.Model>} models array of models fetched
 * @ojsignature  [{target: "Type", value: "Model[]", for: "models"}]
 */

/**
 * @private
 */
oj.Collection.prototype._setRangeLocalInternal = function (start, count, options) {
  if (this.IsVirtual()) {
        // make sure we reconcile the length to what we think the totalresults are--if there have been any non
        // fetched changes in length we don't want to be placing things wrong
    this._resetModelsToFullLength(this.totalResults);
  }
  var actual = this._getLocalRange(start, count);
  var self = this;
  if (this._checkActual(start, count, actual)) {
    return new Promise(function (resolve) {
      resolve(actual);
    });
  }

    // Manage the LRU - set model limit at least as high as the count we're trying to fetch
  var modelLimit = this._getModelLimit();
  if (modelLimit > -1 && modelLimit < count) {
    this.modelLimit = count;
  }
  return new Promise(function (resolve, reject) {
    self._setRangeLocalFetch(start, count, -1, { start: start, count: count }, resolve,
                             reject, true, options);
  });
};

/**
 * @private
 */
oj.Collection.prototype._setRangeLocalFetch = function (start, count, placement, original,
  resolve, reject, fill, options) {
  var self = this;
  var localStart = start;
  var resp = function () {
    var actual = self._getLocalRange(original.start, original.count);
    if (fill && self._hasTotalResults() && actual.count < original.count) {
      // The range wasn't fulfilled: try again
      var newPlacement = actual.start + actual.count;
      // Try the next block...don't repeat the request
      var newStart = localStart + (self.lastFetchCount ? self.lastFetchCount : count);
      if (newStart < self.totalResults) {
        self._setRangeLocalFetch(newStart, count, newPlacement, original, resolve, reject,
          fill, options);
      } else {
        // Can't go any further
        resolve(actual);
      }
    } else {
      resolve(actual);
    }
  };

    // Go fetch
  var limit = localStart + count;
    // Get the greater of the limit-localStart or fetchSize
  if (this[oj.Collection._FETCH_SIZE_PROP] && this[oj.Collection._FETCH_SIZE_PROP] > count) {
    limit = this[oj.Collection._FETCH_SIZE_PROP] + localStart;
  }

    // Now, to optimize, move localStart up to the first undefined model in the sequence
  var opts = null;
  if (this.IsVirtual()) {
    var newStart = this._getFirstMissingModel(localStart, limit);
    if (newStart > localStart) {
      localStart = newStart;
      // Recheck the limit
      limit = localStart + count;
      // Get the greater of the limit-localStart or fetchSize
      if (this[oj.Collection._FETCH_SIZE_PROP] && this[oj.Collection._FETCH_SIZE_PROP] > count) {
        limit = this[oj.Collection._FETCH_SIZE_PROP] + localStart;
      }
    }
    opts = { context: this, startIndex: localStart, fetchSize: limit - localStart };
  } else {
    opts = { context: this };
  }
  opts.error = function (xhr, status, error) {
        // Handle potential errors
    reject(oj.Collection._createRejectionError(xhr, status, error, self, null, false));
  };
  opts.success = function () {
    resp();
  };
  if (options && options.silent) {
    opts.silent = options.silent;
  }

  try {
    this._fetchInternal(opts, placement, placement > -1);
  } catch (e) {
        // This is OK if it's a URLError: local collection with no means of fetching: just resolve
    if (e instanceof oj.URLError) {
      var actual = self._getLocalRange(localStart, count);
      resolve(actual);
    }
  }
};

/**
 * @private
 */
oj.Collection._createRejectionError = function (xhr, status, error, collection, options,
  fireError) {
  var silent = false;
  if (options && options.silent) {
    silent = options.silent;
  }
    // To avoid duplication in many cases...
  if (fireError) {
    oj.Model._triggerError(collection, silent, options, status, error, xhr);
  }
  var err = new Error(status);
  err.xhr = xhr;
  err.error = error;
  err.collection = collection;
  err.status = status;
  return err;
};

/**
 * @private
 */
oj.Collection.prototype._getMaxLength = function (start, count) {
  var len = this._getModelsLength();
  if (len === 0) {
        // This is an exception: could be uninitialized
    return start + count;
  }
  return start + count > len ? len : start + count;
};

/**
 * Determine if every element of the given range is filled in locally
 *
 * @param {number} start starting index to check
 * @param {number} count number of elements to check
 * @return {boolean} true if all elements are local, false otherwise
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.isRangeLocal = function (start, count) {
  var localRange = this._getLocalRange(start, count);
  if (this._getModelsLength() === 0) {
    // If we have no models length, then range cannot be thought of as local unless the count
    // is zero--edge case
    return count === 0;
  }
  return start === localRange.start && (count === localRange.count ||
         start + count > this._getModelsLength());
};

/**
 * @private
 */
oj.Collection.prototype._getModelArray = function (start, count) {
  var retArr = [];
  var models = this._getModels();
  var end = start + count;
  for (var i = start; i < end; i++) {
    retArr.push(models[i]);
  }
  return retArr;
};

/**
 * For a given range of models, return the actual range which are local within that set.
 * @private
 */
oj.Collection.prototype._getLocalRange = function (start, count) {
    // Not virtual, local if there are any models
  if (!this.IsVirtual()) {
    if (this._getModelsLength() > 0) {
      if (start + count > this._getModelsLength()) {
                // Over the boundary
        var c = this._getModelsLength() - start;
        return { start: start, count: c, models: this._getModelArray(start, c) };
      }

      return { start: start, count: count, models: this._getModelArray(start, count) };
    }
    return { start: start, count: 0, models: [] };
  }
  var limit = this._getMaxLength(start, count);
    // Adjust for no totalResults
  if (!this._hasTotalResults() && limit < start + count) {
        // We don't know if it's local or not
    return { start: start,
      count: (limit - start),
      models: this._getModelArray(start, limit - start) };
  }
  if (limit === 0) {
        // There nothing here
    return { start: start, count: 0, models: [] };
  }
  var firstMissingModel = this._getFirstMissingModel(start, limit);
  if (firstMissingModel > -1) {
    return { start: start,
      count: (firstMissingModel - start),
      models: this._getModelArray(start, firstMissingModel - start) };
  }
    // Make sure start doesn't overrun the end!
  var localCount = count;
  if (start > limit) {
    localCount = 0;
  } else if (start + localCount > limit) {
    // Make sure that start+count doesn't overrun the end
    localCount = limit - start;
  }
  return { start: start, count: localCount, models: this._getModelArray(start, localCount) };
};

/**
 * Return the first model between start and limit that's undefined
 * @private
 */
oj.Collection.prototype._getFirstMissingModel = function (start, limit) {
  for (var i = start; i < limit; i++) {
    if (this._getModel(i) === undefined) {
      return i;
    }
  }
  return -1;
};

/**
 * Loads the Collection object from the data service URL. Performs a data "read."<p>
 * Events:<p>
 * <ul>
 * <b>request</b>: fired when the request to fetch is going to the server, passing the collection, xhr object, and
 * options<br>
 * <b>sync</b>: fired when the collection is fetched from the data service, passing the collection and the raw
 * response<br>
 * <b>error</b>: fired if there is an error during the fetch, passing the collection, xhr object, options<br>
 * </ul>
 * <p>
 * @param {Object=} options Options to control fetch<p>
 *                  <b>success</b>: a user callback called when the fetch has completed successfully. This makes
 *                  the fetch an asynchronous process. The callback is called passing the Collection object, raw
 *                  response, and the fetch options argument.<br>
 *                  <b>error</b>: a user callback function called if the fetch fails. The callback is called
 *                  passing the collection object, xhr, and options arguments.<br>
 *                  <b>add</b>: if set, new records retrieved from the data service will be added to those models
 *                  already in the collection. If not set, the records retrieved will be passed to the reset()
 *                  method, effectively replacing the previous contents of the collection with the new data.
 *                  Not supported for virtual/paging cases.<br>
 *                  <b>set</b>: if true, fetch will try to use the set function to try and merge the fetched models
 *                  with those already in the collection, on the client.  The default behavior is to reset the
 *                  collection before adding back in the fetched models.  This default is the reverse of
 *                  Backbone's current default.<br>
 *                  <b>startIndex</b>: numeric index with which to start fetching Models from the server.  The
 *                  page setting controls the number of Models to be fetched.  startID takes precedence over
 *                  startIndex if both are specified.  If both are specified and startID isn't supported then
 *                  startIndex will be used instead.<br>
 *                  <b>startID</b>: unique ID of the Model to start fetching from the server.  The page setting
 *                  controls the number of Models to be fetched.  Note if this is not supported by the server
 *                  then startID will be ignored.<br>
 *                  <b>since</b>: fetch records having a timestamp since the given UTC time<br>
 *                  <b>until</b>: fetch records having a timestamp up to the given UTC time<br>
 *                  <b>fetchSize</b>: use specified page size instead of collection's setting<br>
 * @return {Object} xhr ajax object, by default.  If [sync]{@link oj.Collection#sync} has been replaced, this
 * would be the value returned by the custom implementation.
 * @memberof oj.Collection
 * @since 1.0.0
 * @ojsignature {target: "Type", value:"{success?: (collection: oj.Collection, response: any, options: object)=> void,
 *                                                  error?: (collection: oj.Collection, xhr: any, options: object)=> void,
 *                                                  add?: boolean, set?: boolean, startIndex?: number, startID?: any, since?: any,
 *                                                  until?: any, fetchSize?: number, [propName: string]: any}", for: "options"}
 * @export
 */
oj.Collection.prototype.fetch = function (options) {
  var xhr = this._fetchInternal(options, -1, false);
  this._addPromise(function () {
    return Promise.resolve(xhr);
  });
  return xhr;
};

/**
 * fillIn is used to indicate that this fetch is just the result of a get() or part of an add(), etc., when virtual
 * @private
 */
oj.Collection.prototype._fetchInternal = function (options, placement, fillIn) {
  function doReset(collection, opt, fill, totalResults) {
    if (!collection.IsVirtual()) {
      // If we're not doing a "fetch add", delete all the current models
      if (!opt.add && !opt.useset) {
        // Reset with internal model
        collection.reset(null, { silent: true });
      }
    } else if (!fill) {
      // If we're not infilling based on an at, get, etc., delete all the current local models
      collection._resetModelsToFullLength(totalResults);
    }
  }

  var opt = options || {};
  var success = opt.success;
  var self;
  var errFunc = opt.error;

  if (opt.set) {
    opt.useset = !!opt.set;
  }

  // Set up the parsing option
  if (opt.parse === undefined) {
    opt.parse = true;
  }
  self = this;

  opt.error = function (xhr, status, error) {
    oj.Model._triggerError(self, false, options, status, error, xhr);
    if (errFunc) {
      errFunc.call(oj.Model.GetContext(options, self), xhr, status, error);
    }
  };

  opt.success = function (response) {
    // Run the entire returned dataset through the collection's parser (either default no-op or user specified)
    var data;
    try {
      data = self.parse(response, options);
    } catch (e) {
      oj.Collection._reportError(self, e, opt.error, options);
      return;
    }

    // Pull any virtualization properties out of the response
    var resetTotalResults;
    if (!self._setPagingReturnValues(response, options, data, fillIn)) {
      // totalResults was not calculated: tell the reset
      resetTotalResults = self.totalResults;
    }

    var dataList = null;

    var manageLRU = false;
    var locPlace = placement;
    if (!opt.add && !self.model) {
      // We're not doing a "fetch add" and we don't have a model set on the collection
      if (!fillIn) {
        if (self.IsVirtual()) {
          // Virtual case only
          // Clean out the collection
          doReset(self, opt, fillIn, resetTotalResults);

          // Check for passed in offset
          if (placement === -1) {
            manageLRU = true;
            locPlace = self._getOffset();
          }
          // Put the new data in
          dataList = self._putDataIntoCollection(data, locPlace, manageLRU);
        } else if (opt.useset) {
          // New backbone option to not reset
          self._setInternal(data, false, opt, false);
        } else {
          // Replace the data in the collection with a new set (non virtual)
          self.reset(data, { silent: true });
        }
      }
    } else {
      // We have a model and/or we're doing a "fetch add"
      // Clean out the old models if we're not "adding" or infilling for virtual
      doReset(self, opt, fillIn, resetTotalResults);

      // Parse each returned model (if appropriate), and put it into the collection, either from the
      // zeroth offset if non-virtual or using the appropriate offset if virtual
      try {
        // Check for passed in offset
        if (placement === -1) {
          manageLRU = true;
          locPlace = self._getOffset();
        }
        dataList = self._fillInCollectionWithParsedData(data, locPlace, manageLRU, opt);
      } catch (e) {
        oj.Collection._reportError(self, e, opt.error, options);
        return;
      }
    }
    if (self.IsVirtual()) {
      // Take in the number of records actually fetched
      if (dataList) {
        self.lastFetchCount = dataList.length;
      }
    }

    // Fire the sync event
    var silent = !!opt.silent;
    self.TriggerInternal(silent, oj.Events.EventType.SYNC, self, response, opt);
    // Call the caller's success callback, if specified
    if (success) {
      success.call(oj.Model.GetContext(options, self), self, response, opt);
    }
  };
  // Make the actual fetch call using ajax, etc.
  return this._fetchCall(opt);
};

/**
 * Puts server data into an empty virtual collection using a "silent add"
 * @private
 */
oj.Collection.prototype._putDataIntoCollection = function (data, placement, manageLRU) {
  var dataList;

  if (data) {
    dataList = (data instanceof Array) ? data : [data];

    var addOpt = {};
    // Only manage the LRU if we're not trying to achieve a range
    if (manageLRU) {
      this._manageLRU(dataList.length);
    }
    var insertPos = placement;
    var virtual = this.IsVirtual();
    var prevItem = null;
    for (var i = 0; i < dataList.length; i += 1) {
      if (virtual) {
        addOpt = { at: insertPos };
        prevItem = this._atInternal(insertPos, null, true, false);
      }
      // Don't fire add events
      addOpt.silent = true;
      this._addInternal(dataList[i], addOpt, true, false);
      // If virtual, make sure the item was really added where we thought--in other words, what's there now
      // shouldn't match what was there otherwise could be duplicate id and don't increment counter
      if (this._atInternal(insertPos, null, true, false) !== prevItem) {
        insertPos += 1;
      }
    }
  }
  return dataList;
};

/**
 * Parse each model returned, if appropriate, and push them into the (empty) collection with appropriate offsets
 * if virtual
 * @private
 */
oj.Collection.prototype._fillInCollectionWithParsedData = function (data, placement, manageLRU,
  options) {
  var opt = options || {};
  var parse = opt.parse;
  var modelInstance = oj.Collection._createModel(this);
  var dataList = null;

  if (data) {
    dataList = (data instanceof Array) ? data : [data];

    var addOpt = {};
    var parsedModel;

    // Only manage the LRU if we're not trying to achieve a range
    if (manageLRU) {
      this._manageLRU(dataList.length);
    }
    var virtual = this.IsVirtual();
    var i;
    if (opt.useset && !virtual) {
      // New backbone option
      for (i = 0; i < dataList.length; i += 1) {
        if (modelInstance && parse) {
          parsedModel = modelInstance.parse(dataList[i]);
        } else {
          parsedModel = dataList[i];
        }

        dataList[i] = parsedModel;
      }
      this._setInternal(dataList, false, opt, false);
    } else {
      var prevItem = null;
      var insertPos = placement;
      for (i = 0; i < dataList.length; i += 1) {
        if (modelInstance && parse) {
          parsedModel = modelInstance.parse(dataList[i]);
        } else {
          parsedModel = dataList[i];
        }

        if (virtual) {
          addOpt = { at: insertPos };
          prevItem = this._atInternal(insertPos, addOpt, true, false);
        }
                // Don't fire add events
        addOpt.silent = true;
        this._addInternal(parsedModel, addOpt, true, false);
                // If virtual, make sure the item was really added where we thought--in other words, what's there
                // now shouldn't match what was there otherwise could be duplicate id and don't increment counter
        if (this._atInternal(insertPos, null, true, false) !== prevItem) {
          insertPos += 1;
        }
      }
    }
  }
  return dataList;
};

/**
 * @private
 */
oj.Collection._reportError = function (collection, e, errorFunc, options) {
  Logger.error(e.toString());
  if (errorFunc) {
    errorFunc.call(oj.Model.GetContext(options, collection), collection, e, options);
  }
};

/**
 * Used in virtualization to conduct server-based searches: returns list of fetched models via a promise but does
 * not add them to the collection model list
 * @private
 */
oj.Collection.prototype._fetchOnly = function (options) {
  var opt = options || {};
  var success = opt.success;
  var parsedModel;
  var self;

  if (opt.parse === undefined) {
    opt.parse = true;
  }
  self = this;

  opt.success = function (response) {
    var i;
    var modelInstance;
    var dataList = null;
    var fetchedModels = [];

    var data;
    try {
      data = self.parse(response, options);
    } catch (e) {
      oj.Collection._reportError(self, e, opt.error, options);
      return;
    }

    if (!opt.add && !self.model) {
      dataList = (data instanceof Array) ? data : [data];
    } else {
      modelInstance = oj.Collection._createModel(self);

      if (data) {
        dataList = (data instanceof Array) ? data : [data];

        for (i = 0; i < dataList.length; i += 1) {
          if (modelInstance && opt.parse) {
            try {
              parsedModel = modelInstance.parse(dataList[i]);
            } catch (e) {
              oj.Collection._reportError(self, e, opt.error, options);
              return;
            }
          } else {
            parsedModel = dataList[i];
          }
          fetchedModels.push(self._newModel(parsedModel));
        }
      }
    }

    self.TriggerInternal(false, oj.Events.EventType.SYNC, self, response, opt);
    if (success) {
      success.call(oj.Model.GetContext(options, self), self, fetchedModels, opt);
    }
  };
  return this._fetchCall(opt);
};

/**
 * @private
 */
oj.Collection.prototype._fetchCall = function (opt) {
  try {
    return oj.Model._internalSync('read', this, opt);
  } catch (e) {
         // Some kind of error: trigger an error event
    oj.Model._triggerError(this, false, opt, null, e, null);
    throw e;
  }
};

/**
 * @private
 */
oj.Collection.prototype._resetModelsToFullLength = function (totalResults) {
  // var totalResults = this.totalResults;
  if (totalResults !== undefined && this._getModelsLength() !== totalResults) {
        // Make sure to set up the array if the length changes (i.e., from 0 to totalResults--need to preallocate)
    this._setModels(new Array(totalResults), true);
    this._resetLRU();
    this._setLength();
    return true;
  }
  return false;
};

/**
 * @private
 */
oj.Collection.prototype._getFetchSize = function (options) {
  var opt = options || {};
  return opt[oj.Collection._FETCH_SIZE_PROP] || this[oj.Collection._FETCH_SIZE_PROP];
};

/**
 * Are we doing virtualization/paging?
 * @protected
 */
oj.Collection.prototype.IsVirtual = function () {
  return this._getFetchSize(null) > -1;
};

/**
 * @private
 */
oj.Collection.prototype._getReturnProperty = function (customObj, response, property,
  optionValue, defaultValue) {
  var value = parseInt(oj.Collection._getProp(customObj, response, property), 10);
  if (value === undefined || value === null || isNaN(value)) {
        // use fetchsize
    return optionValue || defaultValue;
  }
  return value;
};

/**
 * @private
 */
oj.Collection.prototype._cleanTotalResults = function (totalResults) {
    // In case server (incorrectly) passes back a -1, treat it as undefined
  if (totalResults === -1) {
    return undefined;
  }
  return totalResults;
};

/**
 * Parse out some of the paging return values we might expect in a virtual response
 * Return true if totalResults was calculated on a rest call that has none
 * @private
 */
oj.Collection.prototype._setPagingReturnValues = function (response, options, data, fillIn) {
  var customObj = {};
  // See if there's a custom call out
  if (this.customPagingOptions) {
    customObj = this.customPagingOptions.call(this, response);
    if (!customObj) {
      customObj = {};
    }
  }
  // What limit was actually used to generate this response?
  var opt = options || {};
  this.lastFetchSize = this._getReturnProperty(customObj, response, 'limit', opt.fetchSize,
                                                    this.fetchSize);

  // What offset was actually used to generate this response?
  this.offset = this._getReturnProperty(customObj, response, 'offset', opt.startIndex, 0);

  // How many records actually came back?
  this.lastFetchCount = this._getReturnProperty(customObj, response, 'count', this.lastFetchCount,
                                                     this.lastFetchCount);

  // What is the total number of records possible for this collection?
  this.totalResults = this._cleanTotalResults(this._getReturnProperty(customObj, response, 'totalResults',
                                                   this.totalResults, this.totalResults));

  // Is there more?
  this.hasMore = this._getHasMore(oj.Collection._getProp(customObj, response, 'hasMore'),
                                       this.offset, this.lastFetchSize, this.totalResults);

  // Adjust total results to account for the case where the server tells us there's no more data, and
  // totalResults wasn't set by the server...but don't do it for simple gets/adds
  var retVal = false;
  if (!fillIn) {
        // We want to know if the server *actually* returned values for these things, not if they defaulted above
    var totalResultsReturned = this._cleanTotalResults(parseInt(oj.Collection._getProp(customObj,
      response, 'totalResults'), 10));
    var lastFetchCountReturned = parseInt(oj.Collection._getProp(customObj, response, 'count'),
      10);
    this.totalResults = this._adjustTotalResults(totalResultsReturned, this.hasMore,
      this.offset, lastFetchCountReturned, data && Array.isArray(data) ? data.length : 0);
    retVal = (totalResultsReturned === undefined || isNaN(totalResultsReturned) ||
              totalResultsReturned === null);
  }

  // Was fetchSize set?  If not, set it to limit
  if (!this.IsVirtual() && this.totalResults && this.totalResults !== this.lastFetchCount &&
        this.lastFetchSize) {
    this.setFetchSize(this.lastFetchSize);
  }
  return retVal;
};

/**
 * @private
 */
oj.Collection.prototype._adjustTotalResults = function (totalResultsReturned, hasMore, offset,
  lastFetchCount, dataLength) {
    // Fix for : if hasMore is false, and totalResults wasn't set by the server, we should set it to
    // the last thing fetched here.  There is no more.
  if (!hasMore) {
        // No more data, per server
    if (isNaN(totalResultsReturned)) {
            // TotalResults was not returned by the server here
            // If lastFetchCount was set, use that as the final total.  Otherwise, use the length of data passed
            // in...all + offset
      var count = !isNaN(lastFetchCount) ? lastFetchCount : dataLength;
      return count + offset;
    }
  }
  return this.totalResults;
};

/**
 * @private
 */
oj.Collection.prototype._getHasMore = function (hasMore, offset, lastFetchSize, totalResults) {
  if (oj.Collection._defined(hasMore)) {
    return hasMore;
  }
    // Special case: return true if totalResults not defined
  if (totalResults === undefined || totalResults === null) {
    return true;
  }
    // Not there: figure it out.  It's true unless we're walking off the end
  return !((offset + lastFetchSize > totalResults));
};

/**
 * @private
 */
oj.Collection._getProp = function (custom, response, prop) {
  if (Object.prototype.hasOwnProperty.call(custom, prop)) {
    return custom[prop];
  }
  return response ? response[prop] : undefined;
};

/**
 * @private
 */
oj.Collection.prototype._getOffset = function () {
  return (oj.Collection._defined(this.offset) ? this.offset : 0);
};

/**
 * Creates a new model, saves it to the data service, and adds it on to the collection.<p>
 * Events:<p>
 * <ul>
 * <b>add</b>: fired when the model is added, passing the collection, model added, and options, if not silent<br>
 * <b>alladded</b>: fired after all models have been added, passing the collection, array of models added, and
 * options<br>
 * <b>request:</b>: fired when the request to save is going to the server<br>
 * <b>sync:</b>: fired when the model is saved to the data service, passing the model and the raw response<br>
 * </ul>
 * @param {Object=} attributes Set of attribute/value pairs with which to initialize the new model object, or a new
 * Model object
 * @param {Object=} options Options to control save (see [save]{@link oj.Model#save}), or add (see
 * [add]{@link oj.Collection#add}).  Plus:<p>
 *                          <b>deferred</b>: if true, return a promise as though this collection were virtual
 *                          whether it is or not<br>
 * @return {oj.Model|boolean|Promise.<oj.Model>} new model or false if validation failed.  If virtual, returns a promise that
 * resolves with the new model
 * @memberof oj.Collection
 * @since 1.0.0
 * @ojsignature {target: "Type", value:"{silent?: boolean, at?: number, merge?: boolean, sort?: boolean, force?: boolean, deferred?: boolean,
 *               [propName: string]: any}", for: "options"}
 * @export
 */
oj.Collection.prototype.create = function (attributes, options) {
  var self = this;
  var opt = options || {};
  var deferred = this._getDeferred(opt);

  function doSave(collection, newModel, validate, localOpt) {
    newModel.save(attributes instanceof oj.Model ? null : attributes, localOpt);

    return newModel;
  }

  function doAdd(newModel, addOpts) {
    if (opt.wait) {
            // Don't do adding now
      if (self.IsVirtual() || deferred) {
        return self._addPromise(function () {
          return Promise.resolve(undefined);
        });
      }
      return null;
    }
    return self.add(newModel, addOpts);
  }

    // Save the user's context and callback, if any
  var newModel = this._newModel(attributes, true, opt, false);
  var callback = opt.success;
  var context = opt.context;
  var validate = opt.validate;
  opt.context = this;
  opt.success = function (model, resp, successOpts) {
    // Make sure we pass xhr
    if (successOpts.xhr) {
      opt.xhr = successOpts.xhr;
    }
    if (opt.wait) {
      // Trigger delayed add events
      self.add(newModel, opt);
    }
    if (callback) {
      callback.call(context != null ? context : self, model, resp, opt);
    }
  };

  // Did validation pass?
  if (newModel == null) {
    return false;
  }

  // Force a save in case user has set value of idAttribute on the new Model
  opt.forceNew = newModel.GetId() != null;

  var addOpts = oj.Model._copyOptions(opt);

  newModel.SetCollection(this);
  if (deferred || this.IsVirtual()) {
    return new Promise(function (resolve) {
      addOpts.merge = true;
      addOpts.deferred = true;
      doAdd(newModel, addOpts)
                    .then(function () {
                      opt.success = function (model, resp, successOpts) {
                                    // Make sure we pass xhr
                        if (successOpts.xhr) {
                          opt.xhr = successOpts.xhr;
                        }
                        if (opt.wait) {
                                        // Trigger delayed add event
                                        // Force the add if virtual because we know it was successfully saved: we
                                        // don't want to go back to the server to check
                          if (self.IsVirtual()) {
                            addOpts.force = true;
                          }
                          self.add(newModel, addOpts).then(function () {
                            if (callback) {
                              callback.call(context != null ? context : self, model, resp, opt);
                            }
                            resolve(model);
                          });
                        } else {
                          if (callback) {
                            callback.call(context != null ? context : self, model, resp, opt);
                          }
                          resolve(model);
                        }
                      };
                      var model = doSave(self, newModel, validate, opt);
                                // make sure that success is called first if successful...promise resolved
                                // second
                      if (!model) {
                                    // Failed: make sure we resolve the promise.  Otherwise promise will
                                    // be resolved by success call, above
                        resolve(model);
                      }
                    });
    });
  }

  addOpts.merge = true;
  doAdd(newModel, addOpts);
  var model = doSave(this, newModel, validate, opt);
  return model;
};

/**
 * Return a list of all the values of attr found in the collection
 *
 * @param {string} attr attribute to return
 * @return {Array.<Object>} array of values of attr
 *
 * @throws {Error} when called on a virtual collection
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.pluck = function (attr) {
  var arr = [];
  var i;

  this._throwErrIfVirtual('pluck');

  for (i = 0; i < this._getLength(); i += 1) {
    arr.push(this.at(i).get(attr));
  }
  return arr;
};

/**
 * Return an array of models that contain the given attribute/value pairs.  Note that this function, along with
 * findWhere, expects server-resolved filtering to return *all* models that meet the criteria, even in virtual
 * cases.  The fetchSize will be set to the value of totalResults for this call to indicate that all should
 * be returned.<br>
 * Events: for events, if virtual, see [fetch]{@link oj.Collection#fetch}
 *
 * @param {Object|Array.<Object>} attrs attribute/value pairs to find.  The attribute/value pairs are ANDed together.  If
 *                             attrs is an array of attribute/value pairs, then these are ORed together
 *                             If the value is an object (or an array of objects, in which case the single
 *                             attribute must meet all of the value/comparator conditions), then if it has both
 *                             'value' and 'comparator' parameters these will be interpreted as
 *                             expressions needing custom commparisons.  The comparator value may either be a
 *                             string or a comparator callback function.  Strings are only valid where the
 *                             filtering is sent back to the data service (virtual collections).  In the case of
 *                             a comparator function, the function always takes the signature function(model,
 *                             attr, value), and for non-virtual collections, is called for each
 *                             Model in the collection with the associated attribute and value.  The function
 *                             should return true if the model meets the attribute/value condition, and false if
 *                             not.  For cases where the filtering is to be done on the server, the function will
 *                             be called once per attr/value pair with a null model, and the function should
 *                             return the string to pass as the comparison in the expression for the filtering
 *                             parameter in the URL sent back to the server.  Note that the array of value object
 *                             case is really only meaningful for server-evaluated filters where a complex
 *                             construction on a single attribute might be needed (e.g., x>v1 && x <=v2)
 *                             For example:<p>
 *                             {Dept:53,Name:'Smith'}<br>
 *                             will return an array of models that have a Dept=53 and a Name=Smith, or, for
 *                             server-filtered collections, a ?q=Dept=53+Name=Smith parameter will be sent with
 *                             the URL.<p>
 *                             [{Dept:53},{Dept:90}]<br>
 *                             will return all models that have a Dept of 53 or 90.  Or, ?q=Dept=53,Dept=90 will
 *                             be sent to the server.<p>
 *                             {Dept:{value:53,comparator:function(model, attr, value) { return model.get(attr) !==
 *                             value;}}}<br>
 *                             will return all models that do not have a Dept of 53.<p>
 *                             {Dept:{value:53,comparator:'<>'}}<br>
 *                             For server-evaluated filters, a parameter ?q=Dept<>53 will be sent with the URL.
 *                             This form is an error on locally-evaluated colleection filters<p>
 *                             {Dept:{value:53,comparator:function(model, attr, value) { return "<>";}}}<br>
 *                             expresses the same thing for server-evaluated filters<p>
 *                             {Dept:[{value:53,comparator:'<'},{value:90,comparator:'<'}]}<br>
 *                             For server-evaluated filters, a parameter ?q=Dept>53+Dept<93 will be sent to the
 *                             server<p>
 * @param {Object=} options
 * @property {boolean=} deferred if true, return a promise as though this collection were virtual
 * whether it is or not<p>
 *
 * @return {Array.<oj.Model>|Promise.<Array>} array of models.  If virtual or deferred, a promise that resolves with the returned
 * array from the server
 * @ojsignature  {target: "Type", value: "Promise<Array<oj.Model>>|Array<oj.Model>", for: "returns"}
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.where = function (attrs, options) {
  return this._handlePromise(this._whereInternal(attrs, options));
};

/**
 * @private
 */
oj.Collection.prototype._whereInternal = function (attrs, options) {
  var opt = options || {};
  var deferred = this._getDeferred(opt);
  var self = this;
  if (this.IsVirtual()) {
    return new Promise(function (resolve, reject) {
      var success = function (collection, fetchedModels) {
        resolve(fetchedModels);
      };
            // Send the attributes for a server-based filter; also indicate that we need *all* the attributes.
            // In the standard REST URL construction this is accomplished by leaving off fetchSize/start indices,
            // etc.
      var opts = { query: attrs,
        all: true,
        success: success,
        error: function (xhr, status, error) {
          reject(oj.Collection._createRejectionError(xhr, status, error, self, opt, true));
        } };
      self._fetchOnly(opts);
    });
  }

  var arr = [];
  var i;
  var m;
  for (i = 0; i < this._getLength(); i += 1) {
    m = this.at(i);
    if (m.Contains(attrs)) {
      arr.push(m);
    }
  }
  if (deferred) {
    return new Promise(function (resolve) {
      resolve(arr);
    });
  }
  return arr;
};

/**
 * Return a collection of models that contain the given attribute/value pairs.
 * Note that this returns a non-virtual collection with all the models returned by the server,
 * even if the original collection is virtual.  Virtual collections doing filtering on the server should return
 * all models that meet
 * the critera.  See [where]{@link oj.Collection#where}
 * See [where]{@link oj.Collection#where} for complete documentation of events and parameters
 * @param {Object|Array.<Object>} attrs attribute/value pairs to find.  The attribute/value pairs are ANDed together.  If
 * @param {Object=} options
 * @property {boolean=} deferred if true, return a promise as though this collection were virtual
 * whether it is or not<p>
 *
 * @return {oj.Collection|Promise.<oj.Collection>} A collection containing models with given attribute/value pairs.  If virtual or
 * deferred, a promise that resolves with the collection returned by the server
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.whereToCollection = function (attrs, options) {
  var opt = options || {};
  var deferred = this._getDeferred(opt);
  var self = this;
  if (this.IsVirtual() || deferred) {
    return self._addPromise(function () {
      return new Promise(function (resolve, reject) {
        return self._whereInternal(attrs, opt).then(function (models) {
          var collection = self._makeNewCollection(models);
          resolve(collection);
        }, function (err) {
          reject(err);
        });
      });
    });
  }

  var models = this._whereInternal(attrs, opt);
  var newCollection = this._makeNewCollection(models);
  newCollection[oj.Collection._FETCH_SIZE_PROP] = -1;
  newCollection._setLength();
  return newCollection;
};

/**
 * @private
 */
oj.Collection.prototype._makeNewCollection = function (models) {
  var collection = this._cloneInternal(false);
  collection._setModels(models, false);
  collection._resetLRU();
  collection._setLength();
  return collection;
};

/**
 * @private
 */
oj.Collection.prototype._throwErrIfVirtual = function (func) {
  if (this.IsVirtual()) {
    throw new Error(func + ' not valid on a virtual Collection');
  }
};

/**
 * Return an array whose entries are determined by the results of calling the passed iterator function.  The
 * iterator will be called for each model in the collection
 *
 * @param {function(oj.Model):Object} iterator function to determine the mapped value for each model
 * @param {Object=} context context with which to make the calls on iterator
 * @returns {Array.<Object>} array of values determined by the return value of calls made to iterator for each model
 *
 * @throws {Error} when called on a virtual Collection
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.map = function (iterator, context) {
  var retArr = [];
  var value;

  this._throwErrIfVirtual('map');

  this._getModels().forEach(function (model) {
    value = iterator.call(context || this, model);
    retArr.push(value);
  });
  return retArr;
};

/**
 * @desc Iterates over the models in the collection and calls the given iterator function
 *
 * @param {function(oj.Model)} iterator function to call for each model
 * @param {Object=} context context with which to make the calls on iterator
 * @return {undefined}
 *
 * @throws {Error} when called on a virtual collection
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.each = function (iterator, context) {
  this._throwErrIfVirtual('each');

  this._getModels().forEach(iterator, context);
};

/**
 * Return the length of the collection
 * @returns {number} length of the collection
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.size = function () {
  return this._getLength();
};

/**
 * Return the models sorted determined by the iterator function (or property, if a string value).  If a function,
 * the function should return the attribute by which to sort.
 *
 * @param {string|function(oj.Model):string} iterator method called or property used to get the attribute to sort by.
 * @param {Object=} context context with which to make the calls on iterator
 * @returns {Array.<oj.Model>} models sorted using iterator
 * @throws {Error} when called on a virtual collection
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.sortBy = function (iterator, context) {
  var retArr = [];
  var self;

  this._throwErrIfVirtual('sortBy');

  this._getModels().forEach(function (model) {
    retArr.push(model);
  });
  self = this;

  retArr.sort(function (a, b) {
    var keyA;
    var keyB;

    if ($.isFunction(iterator)) {
                // "sortBy" comparator option
      keyA = iterator.call(context || self, a);
      keyB = iterator.call(context || self, b);
      return oj.Collection._compareKeys(keyA, keyB, self.sortDirection);
    }
            // String option
    keyA = a.get(iterator);
    keyB = b.get(iterator);
    return oj.Collection._compareKeys(keyA, keyB, self.sortDirection);
  });

  return retArr;
};

/**
 * @desc Return the collection with models grouped into sets determined by the iterator function (or property, if
 * a string value)
 *
 * @param {string|function(oj.Model):Object} iterator method called or property (if a string) used to get the group key
 * @param {Object=} context context with which to make the calls on iterator
 * @returns {Object} models grouped into sets
 *
 * @throws {Error} when called on a virtual collection
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.groupBy = function (iterator, context) {
  var retObj = {};
  var groupVal;

  this._throwErrIfVirtual('groupBy');

  this._getModels().forEach(function (model) {
    if ($.isFunction(iterator)) {
      groupVal = iterator.call(context || this, model);
    } else {
      groupVal = model.get(iterator);
    }
    if (retObj[groupVal] === undefined) {
      retObj[groupVal] = [];
    }
    retObj[groupVal].push(model);
  }, this);
  return retObj;
};

/**
 * @desc Return an object with models as values for their properties determined by the iterator function or
 * property string
 *
 * @param {string|function(oj.Model)} iterator method called or property (if a string) used to get the index attribute
 * @param {Object=} context context with which to make the calls on iterator
 * @returns {Object} models listed as property values where the properties are the values returned by iterator or
 * the attribute value given by the iterator string
 *
 * @throws {Error} when called on a virtual collection
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.indexBy = function (iterator, context) {
  var retObj = {};
  var index;

  this._throwErrIfVirtual('indexBy');

  this._getModels().forEach(function (model) {
    if ($.isFunction(iterator)) {
      index = iterator.call(context || this, model);
    } else {
      index = model.get(iterator);
    }
    retObj[index] = model;
  }, this);
  return retObj;
};

/**
 * Return the "minimum" model in the collection, as determined by calls to iterator.  The return value of
 * iterator (called with a model passed in) will be compared against the current minimum
 *
 * @param {function(oj.Model):Object} iterator function to determine a model's value for checking for the minimum
 * @param {Object=} context context with which to make the calls on iterator
 * @returns {oj.Model} "Minimum" model in the collection
 *
 * @throws {Error} when called on a virtual Collection
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.min = function (iterator, context) {
  var minModel = {};
  var minModelValue;
  var currValue;

  this._throwErrIfVirtual('min');

  if (this._getModelsLength() === 0) {
    return null;
  }
    // Get vals started
  minModel = this._getModel(0);
  minModelValue = iterator.call(context || this, this._getModel(0));

  this._getModels().forEach(function (model, i) {
    if (i >= 1) {
      currValue = iterator.call(context || this, model);
      if (currValue < minModelValue) {
        minModel = model;
        minModelValue = currValue;
      }
    }
  }, this);
  return minModel;
};

/**
 * Return the "maximum" model in the collection, as determined by calls to iterator.  The return value of
 * iterator (called with a model passed in) will be compared against the current maximum
 *
 * @param {function(oj.Model):Object} iterator function to determine a model's value for checking for the maximum
 * @param {Object=} context context with which to make the calls on iterator
 * @returns {oj.Model} "Maximum" model in the collection
 *
 * @throws {Error} when called on a virtual collection
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.max = function (iterator, context) {
  var maxModel = {};
  var maxModelValue;
  var currValue;

  this._throwErrIfVirtual('max');
  if (this._getModelsLength() === 0) {
    return null;
  }
  // Get vals started
  maxModel = this._getModel(0);
  maxModelValue = iterator.call(context, this._getModel(0));

  this._getModels().forEach(function (model, i) {
    if (i >= 1) {
      currValue = iterator.call(context || this, model);
      if (currValue > maxModelValue) {
        maxModel = model;
        maxModelValue = currValue;
      }
    }
  }, this);
  return maxModel;
};

/**
 * @export
 * @desc Return an array of models that cause passed-in iterator to return true
 * @memberof oj.Collection
 *
 * @param {function(oj.Model)} iterator function to determine if a model should be included or not.  Should return
 * true or false
 * @param {Object=} context context with which to make the calls on iterator
 * @returns {Array.<oj.Model>} array of models that caused iterator to return true
 *
 * @throws {Error} when called on a virtual Collection
 * @since 1.0.0
 */
oj.Collection.prototype.filter = function (iterator, context) {
  var retArr = [];

  this._throwErrIfVirtual('filter');

  this._getModels().forEach(function (model) {
    if (iterator.call(context || this, model)) {
      retArr.push(model);
    }
  });
  return retArr;
};

/**
 * Return an array of models minus those passed in as arguments
 * @param {...oj.Model} var_args models models to remove from the returned array
 * @returns {Array.<oj.Model>} array of models from the collection minus those passed in to models
 *
 * @throws {Error} when called on a virtual Collection
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.without = function (var_args) { // eslint-disable-line
  var retArr = [];
  var j;
  var id;
  var cid;
  var add;

  this._throwErrIfVirtual('without');

  var model;
    // Test each model in the collection
  for (var i = 0; i < this._getModels().length; i++) {
    add = true;
    model = this._getModel(i);
    for (j = 0; j < arguments.length; j += 1) {
            // Get the cid
      cid = arguments[j].GetCid();
            // Get the id
      id = arguments[j].GetId();
      if (model.Match(id, cid)) {
                // If it's found, don't return it--we're "subtracting" those from the return value, which starts
                // as all models in the collection
        add = false;
        break;
      }
    }
    if (add) {
      retArr.push(model);
    }
  }
  return retArr;
};

/**
 * Return an array of models in the collection but not passed in the array arguments
 * @param {...Array.<oj.Model>} var_args models arrays of models to check against the collection
 * @returns {Array.<oj.Model>} array of models from the collection not passed in as arguments
 * @ojsignature {target: "Type", for: "var_args", value: "oj.Model[][]"}
 *
 * @throws {Error} when called on a virtual Collection
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.difference = function (var_args) { // eslint-disable-line
  var retArr = [];
  var j;
  var k;
  var id;
  var cid;
  var add;

  this._throwErrIfVirtual('difference');

  var model;
  for (var i = 0; i < this._getModels().length; i++) {
    add = true;
    model = this._getModel(i);
    for (j = 0; j < arguments.length; j += 1) {
            // Each argument is assumed to be an array of oj.Models
      for (k = 0; k < arguments[j].length; k++) {
                // Get the cid
        cid = arguments[j][k].GetCid();
                // Get the id
        id = arguments[j][k].GetId();
        if (model.Match(id, cid)) {
          add = false;
          break;
        }
      }
      if (!add) {
                // We found this model somewhere in the arguments--we're not going to add it so get out
        break;
      }
    }
    if (add) {
      retArr.push(model);
    }
  }
  return retArr;
};

/**
 * Determine if the collection has any models
 *
 * @returns {boolean} true if collection is empty
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.isEmpty = function () {
  return this._getLength() === 0;
};

/**
 * @export
 * @desc Return true if any of the models in the collection pass the test made by calling the iterator function
 * parameter
 * @memberof oj.Collection
 *
 * @param {function(Object)} iterator function called with each model to determine if it "passes".  The function
 * should return true or false.
 * @param {Object=} context context with which to make the calls on iterator
 * @returns {boolean} true if any of the models cause the iterator function to return true
 *
 * @throws {Error} when called on a virtual collection
 * @since 1.0.0
 */
oj.Collection.prototype.any = function (iterator, context) {
  this._throwErrIfVirtual('any');

  var model;
  for (var i = 0; i < this._getModelsLength(); i += 1) {
    model = this._getModel(i);
    if (iterator.call(context || this, model)) {
      return true;
    }
  }
  return false;
};

/**
 * @export
 * @desc A version of where that only returns the first element found<br>
 * Events: for events, if virtual, see [fetch]{@link oj.Collection#fetch}<br>
 * @memberof oj.Collection
 * @param {Object|Array.<Object>} attrs attribute/value pairs to find.
 * See [where]{@link oj.Collection#where} for more details and examples.
 * @param {Object=} options
 * @property {boolean=} deferred if true, return a promise as though this collection were virtual
 * whether it is or not<p>
 *
 * @returns {oj.Model|Promise.<oj.Model>} first model found with the attribute/value pairs.  If virtual or deferred, a promise
 * that resolves with the returned model from the server
 * @since 1.0.0
 * @ojsignature {target: "Type", value:"{deferred?: boolean, [propName: string]: any}", for: "options"}
 */
oj.Collection.prototype.findWhere = function (attrs, options) {
  var deferred = this._getDeferred(options);
  var self = this;
  if (this.IsVirtual() || deferred) {
    return this._addPromise(function () {
      return new Promise(function (resolve) {
        self._whereInternal(attrs, options).then(function (modelList) {
          if (modelList && modelList.length > 0) {
            resolve(modelList[0]);
          }
          resolve(null);
        });
      });
    });
  }

  var arr = this._whereInternal(attrs, options);
  if (arr.length > 0) {
    return arr[0];
  }
  return null;
};

/**
 * Return a shallow copy of the models from start to end (if specified), in an array<br>
 * For events that may be fired if the collection is virtual, see [fetch]{@link oj.Collection#fetch}.<br>
 *
 * @param {number} start model to start the return array with
 * @param {number=} end model to end the return array with, if specified (not inclusive).  If not, returns to the
 * end of the collection
 * @param {Object=} options
 * @property {boolean=} deferred if true, return a promise as though this collection were virtual
 * whether it is or not
 * @return {Array.<oj.Model>|Promise} array of model objects from start to end, or a promise that resolves specifying the
 * returned array when done
 * @memberof oj.Collection
 * @ojsignature  {target: "Type", value: "Promise<Array<oj.Model>>|Array<oj.Model>", for: "returns"}
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.slice = function (start, end, options) {
  var deferred = this._getDeferred(options);
  var ret = [];
  var i;
  var localEnd = end;
  if (localEnd === undefined) {
    if (this.IsVirtual() && !this._hasTotalResults()) {
      // We can't set the end: throw an error
      throw new Error('End must be set for virtual collections with no totalResults');
    }
    localEnd = this._getModelsLength();
  }

  if (deferred || this.IsVirtual()) {
    var self = this;
    return this._addPromise(function () {
      // Loop using deferred
      return self.IterativeAt(start, localEnd);
    });
  }

  for (i = start; i < localEnd; i += 1) {
    ret.push(this._getModel(i));
  }
  return ret;
};

/**
 * Update the collection with a model or models.  Depending on the options, new models will be added, existing
 * models will be merged, and unspecified models will be removed.
 * The model cid is used to determine whether a given model exists or not.<br>
 * May fire events as specified by [add]{@link oj.Collection#add} or [remove]{@link oj.Collection#remove},
 * depending on the options.
 *
 * @param {Object} models an array of or single model with which to update the collection.  If models contains
 * actual oj.Model objects, then any custom 'parse' function set on the collection needs to take that into
 * account and be prepared to handle an array of oj.Model.
 * @param {Object=} options <b>add</b>:false stops the addition of new models<br>
 *                          <b>remove</b>: false stops the removal of missing models<br>
 *                          <b>merge</b>: false prevents the merging of existing models<br>
 *                          <b>silent</b>: true prevents notifications on adds, removes, etc.<br>
 *                          <b>deferred</b>: if true, return a promise as though this collection were virtual
 *                          whether it is or not
 *
 * @returns {Promise|null} if deferred or virtual, return a promise that resolves when the set has completed
 * @memberof oj.Collection
 * @since 1.0.0
 * @ojsignature {target: "Type", value:"{add?: boolean, remove?: boolean, merge?: boolean, silent?: boolean, deferred?: boolean,
 *               [propName: string]: any}", for: "options"}
 * @export
 */
oj.Collection.prototype.set = function (models, options) {
  var deferred = this._getDeferred(options);
  return this._setInternal(models, true, options, deferred || this.IsVirtual());
};

oj.Collection._removeAfterSet = function (collection, models, remove, foundModels, options) {
    // Now remove models that weren't found
    // get an array of all models

    // Can't avoid looping over everything because we *have* to clean up even unfetched models, in order to fire
    // events, etc.
  if (remove) {
    for (var i = models.length - 1; i >= 0; i -= 1) {
      if (foundModels.indexOf(i) === -1) {
        collection._removeInternal(models[i], i, options);
      }
    }
  }
};

/**
 * Swap two models, and indicate if anything was actually swapped
 * @private
 */
oj.Collection.prototype._swapModels = function (oldIndex, newIndex, remove, add) {
  if (this._hasComparator() || !remove || !add) {
    return { index: oldIndex, swapped: false };
  }
    // Make sure in range
  var len = this._getModelsLength();
  if (oldIndex >= len || newIndex >= len) {
    return { index: oldIndex, swapped: false };
  }
    // Swap
  var oldModel = this._getModel(oldIndex);
  var newModel = this._getModel(newIndex);
    // this._getModels()[oldIndex] = newModel;
  this._setModel(oldIndex, newModel);
  newModel.SetIndex(oldIndex);
    // this._getModels()[newIndex] = oldModel;
  this._setModel(newIndex, oldModel);
  oldModel.SetIndex(newIndex);

  return { index: newIndex, swapped: (newIndex !== oldIndex) };
};

/**
 * @private
 */
oj.Collection.prototype._setInternal = function (models, parse, opt, deferred) {
    // Determine if any of the options are set
  var options = opt || {};
  var add = options.add === undefined ? true : options.add;
  var remove = options.remove === undefined ? true : options.remove;
  var merge = options.merge === undefined ? true : options.merge;
  var foundModels = [];
  var currModel = null;
  var i;
  var modelList;

  var mods = parse ? this.parse(models) : models;

  modelList = Array.isArray(mods) ? mods : [mods];

  if (deferred) {
    var self = this;
    return this._addPromise(function () {
      return self._deferredSet(modelList, self._getModels(), options, remove, add, merge, parse);
    });
  }

    // Go through the passed in models and determine what to do
  var swapped = false;
  for (i = 0; i < modelList.length; i += 1) {
    currModel = this._updateModel(this._newModel(modelList[i], parse, options, true), add,
      merge, deferred);
    if (currModel !== -1) {
            // And swap it into the position as passed in, if no comparator and we're removing, so that the slots
            // are exact
      var obj = this._swapModels(currModel, i, remove, add);
      var newLoc = obj.index;
      if (obj.swapped) {
        swapped = true;
      }
            // Save its new location as found
      if (foundModels.indexOf(newLoc) === -1) {
        foundModels.push(newLoc);
      }
    }
  }
  if (swapped) {
    var eventOpts = options.add ? { add: true } : null;
    this.TriggerInternal(options.silent, oj.Events.EventType.SORT, this, eventOpts, null);
  }

  oj.Collection._removeAfterSet(this, this._getModels(), remove, foundModels, options);
  return null;
};

/**
 * Handle the updates/removes on virtual collections
 * @private
 */
oj.Collection.prototype._deferredSet = function (modelList, modelsCopy, options, remove, add,
  merge, parse) {
  var foundModels = [];
  var i;

    // Go through the passed in models and determine what to do
  var self = this;
  return new Promise(function (allResolve, allReject) {
    var doTask = function (index) {
      return new Promise(function (resolve, reject) {
        self._updateModel(self._newModel(modelList[index], parse, options, true), add, merge,
                                                             true).then(function (currModel) {
                                                               if (currModel !== -1) {
                                                                 foundModels.push(currModel);
                                                               }
                                                               resolve(index + 1);
                                                             }, reject);
      });
    };

    var currentStep = Promise.resolve(0);

    for (i = 0; i < modelList.length; i += 1) {
      currentStep = currentStep.then(doTask);
    }
    currentStep.then(function () {
      oj.Collection._removeAfterSet(self, modelsCopy, remove, foundModels, options);
      allResolve(undefined);
    }, allReject);
  });
};


/**
 * Return the index of the given model after updating it, if it was found.  Otherwise it is added and a -1 is
 * returned
 * @private
 */
oj.Collection.prototype._updateModel = function (model, add, merge, deferred) {
  function update(collection, found, def) {
    var index = found ? found.index : -1;
    var foundModel = found ? found.m : null;

    var opt = {};
    if (foundModel) {
      if (merge) {
        // Do merge if not overridden
        opt = { merge: merge };
        if (def) {
          return new Promise(function (resolve) {
            collection._addInternal(model, opt, false, true).then(function () {
              resolve(index);
            });
          });
        }
        collection.add(model, opt);
      }
    } else if (add) {
      if (def) {
        return new Promise(function (resolve) {
          collection._addInternal(model, opt, false, true).then(function () {
            resolve(collection._getLength() - 1);
          });
        });
      }
      collection.add(model);
      index = collection._getLength() - 1;
    }
    return index;
  }

  // Check to see if this model is in the collection
  if (deferred || this.IsVirtual()) {
    var self = this;

    return new Promise(function (resolve) {
      self._getInternal(model, { silent: true }, deferred).then(function (found) {
        update(self, found, true).then(function (index) {
          resolve(index);
        });
      });
    });
  }

  var found = this._getInternal(model);
  return update(this, found, false);
};

/**
 * Return a copy of the Collection's list of current attribute/value pairs.
 *
 * @return {Array.<Object>} an array containing all the Collection's current sets of attribute/value pairs.
 * @throws {Error} when called on a virtual collection
 *
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.toJSON = function () {
  var retArr = [];

  this._throwErrIfVirtual('toJSON');

  this._getModels().forEach(function (model) {
    retArr.push(model.toJSON());
  });
  return retArr;
};

/**
 * Return the first model object in the collection, or an array of the first n model objects from the collection.<br>
 * For events that may be fired if the collection is virtual, see [fetch]{@link oj.Collection#fetch}.<br>
 * @param {number=} n Number of model objects to include in the array, starting with the first.
 * @param {Object=} options
 * @property {boolean=} deferred if true, return a promise as though this collection were virtual whether it is or not
 * @return {Array.<oj.Model>|null|Promise} An array of n model objects found in the collection, starting with the first. If n
 * is not included, returns all of the collection's models as an array.  If deferred or virtual, returns a promise
 * that resolves with the array or model
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.first = function (n, options) {
  var deferred = this._getDeferred(options);
  var elementCount = this._getLength();
  var retArray = [];
  var i;
  var self = this;

  var num = n;
  if (num) {
    elementCount = num;
  } else {
    num = 1;
  }

  var virtual = this.IsVirtual() || deferred;

  if (num === 1) {
    if (virtual) {
      return this._addPromise(function () {
        return self._deferredAt(0, null);
      });
    }

    if (this._getModelsLength() > 0) {
      return this._getModel(0);
    }
    return null;
  }

  if (elementCount > this._getModelsLength()) {
    if (this.IsVirtual() && !this._hasTotalResults()) {
      // Virtual, no total results: don't restrict elementCount
    } else {
      elementCount = this._getModelsLength();
    }
  }

  if (virtual) {
    return this._addPromise(function () {
      return self.IterativeAt(0, elementCount);
    });
  }

  for (i = 0; i < elementCount; i += 1) {
    retArray.push(this._getModel(i));
  }
  return retArray;
};

/**
 * Return the array index location of the given model object.<br>
 * For events that may be fired if the collection is virtual, see [fetch]{@link oj.Collection#fetch}.<br>
 * @param {oj.Model|string} model Model object (or Model id) to locate
 * @param {Object=} options
 * @property {boolean=} deferred if true, return a promise as though this collection were virtual
 * whether it is or not
 * @return {number|Promise.<number>} The index of the given model object, or a promise that will resolve with the index
 * when complete.  If the object is not found, returns -1.
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.indexOf = function (model, options) {
  var location;
  var deferred = this._getDeferred(options);

  if (this.IsVirtual() || deferred) {
    var self = this;
    return this._addPromise(function () {
      return self._getInternal(model, null, true).then(function (loc) {
        return loc.index;
      });
    });
  }
  location = this._getInternal(model);

  return location.index;
};

/**
 * Determine if the given model object is present in the collection.<br>
 * For events that may be fired if the collection is virtual, see [fetch]{@link oj.Collection#fetch}.<br>
 * @param {Object} model Model object (or Model id) to locate
 * @param {Object=} options
 * @property {boolean=} deferred if true, return a promise as though this collection were virtual
 * whether it is or not
 * @return {boolean|Promise.<boolean>} true if the model is contained in the collection, false if not. If deferred, a
 * promise that will resolve with true or false when complete.
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.contains = function (model, options) {
  var location;
  var deferred = this._getDeferred(options);

  if (this.IsVirtual() || deferred) {
    var self = this;
    return this._addPromise(function () {
      return self._getInternal(model, null, true).then(function (loc) {
        return loc.index > -1;
      });
    });
  }
  location = this._getInternal(model);

  return location.index > -1;
};

/**
 * An alias for [contains]{@link oj.Collection#contains}
 * @param {Object} model Model object (or Model id) to locate
 * @param {Object=} options
 * @property {boolean=} deferred if true, return a promise as though this collection were virtual
 * whether it is or not
 * @return {boolean|Promise.<boolean>} true if the model is contained in the collection, false if not. If deferred, a
 * promise that will resolve with true or false when complete.
 * @kind function
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.include = oj.Collection.prototype.contains;

// Only look on models already fetched
oj.Collection.prototype._localIndexOf = function (model) {
  var location = this._getLocalInternal(model);

  return location !== undefined ? location.index : -1;
};

/**
 * Remove the last model from the collection and return it<br>
 * For events that may be fired if the collection is virtual, see [remove]{@link oj.Collection#remove}.<br>
 * @param {Object=} options Options for the method:<p>
 * <b>silent</b>: if set, do not fire a remove event <br>
 * <b>deferred</b>: if true, return a promise as though this collection were virtual whether it is or not<br>
 * @return {oj.Model|Promise.<oj.Model>} the model that was removed, or a promise that will resolve with the model that was
 * removed when complete
 * @memberof oj.Collection
 * @since 1.0.0
 * @ojsignature {target: "Type", value:"{silent?: boolean, deferred?: boolean, [propName: string]: any}", for: "options"}
 * @export
 */
oj.Collection.prototype.pop = function (options) {
  var deferred = this._getDeferred(options);
  if (this.IsVirtual() || deferred) {
    var self = this;
    return this._atInternal(this._getLength() - 1, options, false, true).then(function (model) {
      self.remove(model, options);
      return model;
    });
  }

  var m = /** @type {oj.Model} */ (this.at(this._getLength() - 1));
  this.remove(m, options);
  return m;
};

/**
 * Add the given model to the end of the Collection<br>
 * For events that may be fired if the collection is virtual, see [add]{@link oj.Collection#add}.<br>
 * @param {oj.Model|Object} m model to add to the end of the Collection (or a set of attribute value pairs)
 * @param {Object=} options same options as [add]{@link oj.Collection#add}
 * @return {Promise.<Array>|undefined} if deferred or virtual, a promise that will be resolved when the function is done.  Otherwise
 * undefined
 * @memberof oj.Collection
 * @ojsignature [{target: "Type", value:"{silent?: boolean, at?: number, merge?: boolean, sort?: boolean,
 *                                        force?: boolean, deferred?: boolean, [propName: string]: any}", for: "options"},
 *               {target: "Type", value: "Promise<Array<oj.Model>>|undefined", for: "returns"}]
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.push = function (m, options) {
  var deferred = this._getDeferred(options);
  this._manageLRU(1);
  return this._handlePromise(this._addInternal(m, options, false, deferred));
};

/**
 * Returns the index of the last location of the given model.  Not supported in virtual cases.
 * @param {oj.Model} model Model object to locate
 * @param {number=} fromIndex optionally start search at the given index
 * @return {number} The last index of the given model object.  If the object is not found, returns -1.
 * @throws {Error} when called on a virtual collection
 * @memberof oj.Collection
 * @since 1.0.0
 * @export
 */
oj.Collection.prototype.lastIndexOf = function (model, fromIndex) {
  var i;
  var locIndex = fromIndex;

  this._throwErrIfVirtual('lastIndexOf');

  if (locIndex === undefined) {
    locIndex = 0;
  }

  for (i = this._getLength() - 1; i >= locIndex; i -= 1) {
    if (oj.Object.__innerEquals(model, this.at(i))) {
      return i;
    }
  }
  return -1;
};

/**
 * @private
 */
oj.Collection.prototype._getSortAttrs = function (sortStr) {
  if (sortStr === undefined) {
    return [];
  }
  return sortStr.split(',');
};

/**
 * Return a URL query string based on an array of or a single attr/value pair set
 * @private
 */
oj.Collection._getQueryString = function (q) {
  function expression(left, right, compare) {
    return left + compare + right;
  }

  function processQuery(query, input) {
    var exp;
    var str = input;
    Object.keys(query || {}).forEach(function (prop) {
      if (Object.prototype.hasOwnProperty.call(query, prop)) {
        var val = Array.isArray(query[prop]) ? query[prop] : [query[prop]];
        for (var j = 0; j < val.length; j++) {
          if (oj.Model.IsComplexValue(val[j])) {
            var value = val[j].value;
            var compare = null;
            var comparator = val[j].comparator;
            if ($.isFunction(comparator)) {
              compare = comparator(null, prop, value);
            } else {
              compare = comparator;
            }
            exp = expression(prop, value, compare);
          } else {
            exp = expression(prop, query[prop], '=');
          }
          str += exp + '+';
        }
      }
    });
    // Remove trailing '+'
    str = str.substring(0, str.length - 1) + ',';
    return str;
  }

  var queries = Array.isArray(q) ? q : [q];
  var str = '';
  var i;
  for (i = 0; i < queries.length; i++) {
    str = processQuery(queries[i], str);
  }
    // Remove trailing ','
  if (str.substring(str.length - 1) === ',') {
    return str.substring(0, str.length - 1);
  }
  return str;
};

/**
 * @protected
 */
oj.Collection.prototype.ModifyOptionsForCustomURL = function (options) {
  var opt = {};
  Object.keys(options || {}).forEach(function (prop) {
    if (Object.prototype.hasOwnProperty.call(options, prop)) {
      opt[prop] = options[prop];
    }
  });
  var comparator = this.comparator;
  if (comparator && oj.StringUtils.isString(comparator)) {
    var attrs = this._getSortAttrs(comparator);
    for (var i = 0; i < attrs.length; i++) {
      if (i === 0) {
        opt.sort = attrs[i];
      } else {
        opt.sort += ',' + attrs[i];
      }
    }
    opt.sortDir = this._getSortDirStr();
  }
    // Put fetchSize on if appropriate, and not already set
  if (this.IsVirtual()) {
    opt[oj.Collection._FETCH_SIZE_PROP] = this._getFetchSize(opt);
  }
  return opt;
};

/**
 * Determine if this collection is URL-based
 * @private
 */
oj.Collection.prototype.IsUrlBased = function (options) {
  var customURL = this.customURL;
  if ($.isFunction(customURL)) {
    return true;
  }
  var url = this.GetCollectionFetchUrl(options);
  return oj.Collection._defined(url);
};

/**
 * Build a URL with parameters for the collection fetch
 * @protected
 */
oj.Collection.prototype.GetCollectionFetchUrl = function (opt) {
  var url = oj.Model.GetPropValue(this, 'url');

    // Adorn it with options, if any
  if (this.IsVirtual()) {
    var options = opt || {};
    var all = options.all;

        // Put in page size
    var limit = null;
    if (all) {
      var totalResults = this.totalResults;
      limit = totalResults || this._getFetchSize(options);
    } else {
      limit = this._getFetchSize(options);
    }
    if (url && url.indexOf('?') > -1) {
            // Already have a param coming in
      url += '&';
    } else {
      url += '?';
    }
    url += 'limit=' + limit;

    if (!all) {
      if (oj.Collection._defined(options.startIndex)) {
        url += '&offset=' + options.startIndex;
      }
      if (options.startID) {
        url += '&fromID=' + options.startID;
      }
      if (options.since) {
        url += '&since=' + options.since;
      }
      if (options.until) {
        url += '&until=' + options.until;
      }
    }
        // Query
    if (options.query) {
      var queryString = oj.Collection._getQueryString(options.query);
      if (queryString && queryString.length > 0) {
        url += '&q=' + queryString;
      }
    }

        // Add sorting
    var comparator = this.comparator;
    if (comparator && oj.StringUtils.isString(comparator)) {
      var attrs = this._getSortAttrs(comparator);
      var sortDirStr = this._getSortDirStr();
      var i;
      for (i = 0; i < attrs.length; i++) {
        if (i === 0) {
          url += '&orderBy=' + attrs[i] + ':' + sortDirStr;
        } else {
          url += ',' + attrs[i] + ':' + sortDirStr;
        }
      }
    }
        // Always ask for totalresults
    url += '&totalResults=true';
  }
  return url;
};

/**
 * @private
 */
oj.Collection.prototype._getSortDirStr = function () {
  if (this.sortDirection === -1) {
    return 'desc';
  }
  return 'asc';
};

/**
 * @export
 * Called to perfrom server interactions, such as reading the collection.  Designed to be overridden by users
 *
 * @param {string} method "read"
 * @param {oj.Collection} collection the collection to be read (fetched)
 * @param {Object=} options to control sync<br>
 * <b>success</b>: called if sync succeeds.  Called with the data being fetched<br>
 * <b>error</b>: called if sync fails.  Called with xhr, status, and error info, per jQuery Ajax (all if
 * available)<br>
 * @return {Object} xhr response from ajax by default
 * @memberof oj.Collection
 * @since 1.0.0
 * @ojsignature {target: "Type", value:"{success?: (response?: any)=> void,
 *                                                  error?: (xhr: any, status: any, error: any)=> void, [propName: string]: any}", for: "options"}
 * @alias oj.Collection.prototype.sync
 */
oj.Collection.prototype.sync = function (method, collection, options) {
  return oj.sync(method, collection, options);
};

/**
 * Constants
 * @private
 */
oj.Collection._FETCH_SIZE_PROP = 'fetchSize';

/**
 * Copyright (c) 2014, 2015 Oracle and/or its affiliates.
 * All rights reserved.
 */

/* jslint browser: true*/

/**
 * @export
 * @class oj.Model
 * @classdesc Object representing name/value pairs for a data service record
 *
 * @param {Object=} attributes Initial set of attribute/value pairs with which to seed this Model object
 * @param {Object=} options
 *                  collection: collection for this model
 * @constructor
 * @since 1.0
 * @mixes oj.Events
 * @ojsignature {target: "Type", value: "class Model"}
 */
oj.Model = function (attributes, options) {
  oj.Model._init(this, attributes, options, null);
};


/**
 * Subclass from oj.Object
 * @private
 */
oj.Object.createSubclass(oj.Model, oj.Object, 'oj.Model');

/**
 * @private
 */
oj.Model.prototype.Init = function () {
  oj.Model.superclass.Init.call(this);
};

/**
 *
 * @export
 * @desc Attribute/value pairs in the model.
 * @memberof oj.Model
 *
 * @type Object
 * @since 1.0.0
 */
oj.Model.prototype.attributes = {};

/**
 * @export
 * @desc The set of attribute/value pairs that serve as default values when new Model objects are created.
 * @memberof oj.Model
 *
 * @type Object
 * @since 1.0.0
 */
oj.Model.prototype.defaults = {};

/**
 * @export
 * @desc The model's unique ID.  This can be set by the application or retrieved from the data service. This ID
 * will be appended to the URL for single-record data operations (update, delete).
 * @memberof oj.Model
 *
 * @type {string|null}
 * @since 1.0.0
 */
oj.Model.prototype.id = null;

/**
 * @desc The name of the model property to be used as the unique ID. See [id]{@link oj.Model#id}. This defaults to
 * a value of "id".
 * @memberof oj.Model
 *
 * @type {string|null}
 * @since 1.0.0
 * @export
 */
oj.Model.prototype.idAttribute = 'id';

/**
 * @export
 * @desc The base url on the data service used to perform CRUD operations on models.  If not defined, the model
 * will look to its collection.  One or the other must be defined before CRUD operations can succeed.
 * @memberof oj.Model
 *
 * @type {string|null}
 * @since 1.0.0
 */
oj.Model.prototype.urlRoot = null;

/**
 * @export
 * @desc A callback to allow users to completely customize the data service URLs
 * The callback should accept these parameters:<p>
 * <b>operation (string)</b>: one of "create", "read", "update", "patch", or "delete", indicating the type of
 * operation for which to return the URL<br>
 * <b>model (Object)</b>: the oj.Model object requesting the URL<br>
 * <b>options (Object)</b>: one or more of the following properties:<br>
 * <ul>
 * <b>recordID</b>: id of the record involved, if relevant<br>
 * </ul>
 *
 * customURL callbacks should return either: null, in which case the default will be used; a url string, which
 * will be used with the standard HTTP method for the type of operation, or an Object with with any other
 * attributes that should be passed to the ajax call.<br>
 * This object must at minimum include the URL, and other attributes as follows:<br>
 * <ul>
 * <b>url</b>: giving the custom URL string<br>
 * <b>type</b>: (optional) a string indicating the type of HTTP method to use (GET, POST, DELETE, etc.)<br>
 * <b>(other)</b>: (optional) any other ajax attributes to pass in the ajax call
 * </ul>
 * <p>
 *
 * @memberof oj.Model
 * @type {function(string,oj.Model,Object):(string|Object|null)|null}
 * @ojsignature  {target: "Type", value: "function(string,oj.Model,oj.Model.CustomURLCallbackOptions):(string|Object|null)|null", for: "returns"}
 * @since 1.0.0
 */
oj.Model.prototype.customURL = null;

/**
 * @typedef {Object} oj.Model.CustomURLCallbackOptions
 * @property {string=} recordID id of the record involved, if relevant
*/


/**
 * @export
 * @desc A callback to allow optional custom validation of model attributes during save, set, etc.
 * The callback should accept these parameters:<p>
 * <b>attributes (Object)</b>: the attributes to validation<br>
 * <b>options (Object)</b>: the options passed in to the model call making the validation check<br>
 *
 * The validate callback should return nothing if the attributes are valid, or an error string or object if the validation fails<br>
 * <p>
 *
 * @memberof oj.Model
 * @type {function(Object,Object):(string|Object|null)|null}
 * @since 2.3.0
 */
oj.Model.prototype.validate = null;

/**
 * @export
 * @memberof oj.Model
 * @type {string|Object|null}
 * @desc The last value returned from a validate callback
 * @since 2.3.0
 */
oj.Model.prototype.validationError = null;

/**
 * @export
 * @memberof oj.Model
 * @ojstatus preview
 * @type {boolean}
 * @desc If true, do not insert the JET locale-based Accept-Language header.  If false, let the Ajax system set the header.
 * @since 5.0.0
 */
oj.Model.prototype.omitLanguageHeader = false;

/**
 * @private
 */
oj.Model._idCount = 0;

/**
 * @private
 */
oj.Model._init = function (model, attributes, opt, properties) {
  var parse;
  var attrCopy;
  var prop;

  if (oj.Model._justExtending) {
    return;
  }

  model.Init(); // eslint-disable-line no-param-reassign

  // Augment with Event
  oj.Events.Mixin(model);

  model._clearChanged(); // eslint-disable-line no-param-reassign
  model.previousAttrs = {}; // eslint-disable-line no-param-reassign
  model.nestedSet = false; // eslint-disable-line no-param-reassign
  model.index = -1; // eslint-disable-line no-param-reassign

  var options = opt || {};

  // Deep copy actual data if found
  model.attributes = {}; // eslint-disable-line no-param-reassign
  if (model.defaults && !options.ignoreDefaults) {
    model.attributes = oj.Model._cloneAttributes($.isFunction(model.defaults) ? // eslint-disable-line no-param-reassign
      model.defaults() : model.defaults, null);
  }

  // First, copy all properties passed in
  for (prop in properties) { // eslint-disable-line no-restricted-syntax
    if (Object.prototype.hasOwnProperty.call(properties, prop)) {
      model[prop] = properties[prop]; // eslint-disable-line no-param-reassign
    }
  }

  if (attributes) {
    parse = options.parse;
    if ($.isFunction(parse)) {
      model.parse = parse; // eslint-disable-line no-param-reassign
    }

    attrCopy = oj.Model._cloneAttributes(attributes, model.attributes);
    model.attributes = attrCopy; // eslint-disable-line no-param-reassign

    attrCopy = parse ? model.parse(attrCopy) : attrCopy;
    if (attrCopy == null || attrCopy === undefined) {
      // Reset it
      model.attributes = {}; // eslint-disable-line no-param-reassign
    } else {
      // Move them in
      for (prop in attrCopy) { // eslint-disable-line no-restricted-syntax
        if (Object.prototype.hasOwnProperty.call(attrCopy, prop)) {
          model._setProp(prop, attrCopy[prop], false, false, options, true);
        }
      }
    }
  }

  model.SetCid();

    // Grab collection option, if there
  model.SetCollection(options.collection);

  if (options.customURL) {
    model.customURL = options.customURL; // eslint-disable-line no-param-reassign
  }

    // If URL is set, use that
  if (options.url) {
    model.url = options.url; // eslint-disable-line no-param-reassign
  }

  if (options.urlRoot) {
    model.urlRoot = options.urlRoot; // eslint-disable-line no-param-reassign
  }

  if (model.initialize) {
    model.initialize(attributes, options);
  }

  model.SetupId();
};


/**
 * Create a new, specific type of model object to represent single records from a JSON data set.
 * @param {Object=} properties Properties for the new Model class.<br>
 *                  <b>defaults</b>: an Object containing starting attribute/value pairs for some or all of the
 *                  model's potential attributes<br>
 *                  <b>parse</b>: a user callback function to allow parsing of JSON record objects as they are
 *                  returned from the data service<br>
 *                  <b>parseSave</b>: a user callback function to allow conversion of models back into a format
 *                  appropriate for the data service on save calls<br>
 *                  <b>urlRoot</b>: the URL to use to get records from the data service in the abscence of a
 *                  collection (when an id is appended)<br>
 *                  <b>initialize</b>: a user callback function to be called when this model is created<br>
 *                  <b>validate</b>: a user callback function that will be called before a save to the data
 *                  service occurs. The callback is passed the current set of attributes and save options.
 *                  <br>
 * @param {Object=} classProperties properties that attach to the whole class
 * @return {oj.Model} new Model object
 * @export
 * @ojsignature [{target: "Type",
 *                value: "any",
 *                for: "returns"},
 *               {target: "Type", value:"{parse?: (data: any)=> any, parseSave?: (data: any)=> any, urlRoot?: string,
 *                                        initialize?: (models: Array<oj.Model>, options: object)=> void,
 *                                        validate?: null|object|string|((attributes: object, options?: oj.Model)=> number), [propName: string]: any}", for: "properties"}]
 * @memberof oj.Model
 * @this {oj.Model}
 * @since 1.0.0
 */
oj.Model.extend = function (properties, classProperties) {
  oj.Model._justExtending = true;
  var obj;

  obj = new oj.Model();
  oj.Model._justExtending = false;

    // Add regular properties from this "parent"
    // oj.Events.Mixin(obj, this.prototype);
  $.extend(obj, this.prototype);

    // Grab properties
  var props = properties || {};
  Object.keys(props).forEach(function (prop) {
    if (Object.prototype.hasOwnProperty.call(props, prop)) {
      obj[prop] = props[prop];
    }
  });

  var Model;

  if (props && props.constructor && Object.prototype.hasOwnProperty.call(props,
    'constructor')) {
    Model = props.constructor;
  } else {
    Model = function (attributes, options) {
      oj.Model._init(this, attributes, options, props);
    };
  }

  $.extend(Model, this);
  Model.prototype = obj;

    // Allow extending resulting obj
  Model.extend = oj.Model.extend;

  Model.prototype.constructor = Model;

    // Add class properties from this
  oj.Events.Mixin(Model, this);

  if (classProperties) {
    Object.keys(classProperties).forEach(function (prop) {
      if (Object.prototype.hasOwnProperty.call(classProperties, prop)) {
        Model[prop] = classProperties[prop];
      }
    });
  }


  return Model;
};

/**
 * Placeholder for event mixins
 * @private
 */
oj.Model.prototype.TriggerInternal = function (silent, event, arg1, arg2, options) {}; // eslint-disable-line no-unused-vars

/**
 * @protected
 */
oj.Model.prototype.SetCid = function () {
    // Create cid property if necessary
  if (!this.GetCid()) {
    this.cid = 'id' + oj.Model._idCount;
    oj.Model._idCount += 1;
  }
};

/**
 * @protected
 */
oj.Model.prototype.GetCid = function () {
  return this.cid;
};

/**
 * Index within collection
 * @protected
 */
oj.Model.prototype.SetIndex = function (index) {
  this.index = index;
};

/**
 * @protected
 */
oj.Model.prototype.GetIndex = function () {
  return this.index;
};

/**
 * LRU functions
 * @protected
 */
oj.Model.prototype.SetNext = function (model) {
  var retVal = this.nextModel;
  this.nextModel = model;
  return retVal;
};

/**
 * @protected
 */
oj.Model.prototype.GetNext = function () {
  return this.nextModel;
};

/**
 * @protected
 */
oj.Model.prototype.SetPrevious = function (model) {
  var retVal = this.previousModel;
  this.previousModel = model;
  return retVal;
};

/**
 * @protected
 */
oj.Model.prototype.GetPrevious = function () {
  return this.previousModel;
};

/**
 * Merge the given model's attributes with this model's attributes
 * @protected
 */
oj.Model.prototype.Merge = function (model, comparator, silent) {
  var needSort = false;
  var isStringComparator = oj.StringUtils.isString(comparator);
  var valueChange;
  var changes = false;

  var self = this;
  Object.keys(model.attributes || {}).forEach(function (prop) {
    if (Object.prototype.hasOwnProperty.call(model.attributes, prop)) {
      valueChange = (self.attributes[prop] !== model.attributes[prop]);
      if (isStringComparator) {
                // We have a string comparator--does it match this property?  If we hit a property that doesn't
                // match, we need sort
        if (prop === comparator) {
                    // The property matches the comparator property: are we changing the value?
          if (valueChange) {
            needSort = true;
          }
        }
      } else if (valueChange) {
        needSort = true;
      }
      if (valueChange) {
        changes = true;
        self.attributes[prop] = model.attributes[prop];
        self._addChange(prop, model.attributes[prop]);
        self._fireAttrChange(prop, self.attributes[prop], null, silent);
      }
    }
  });
  this.SetupId();
    // Only fire master change if there were any changes
  if (changes) {
    this._fireChange(null, silent);
  }
  return needSort;
};

/**
 * @private
 */
oj.Model._hasProperties = function (object) {
  if (object && object instanceof Object) {
    var prop;
    for (prop in object) { // eslint-disable-line no-restricted-syntax
      if (Object.prototype.hasOwnProperty.call(object, prop)) {
        return true;
      }
    }
  }
  return false;
};

/**
 * @protected
 */
oj.Model.prototype.SetCollection = function (coll) {
  if (coll == null) {
    delete this.collection;
    return;
  }
  this.collection = coll;
    // This can depend on the collection
  this.SetupId();
};

/**
 * @protected
 */
oj.Model.prototype.GetCollection = function () {
  return this.collection;
};

/**
 * @private
 */
oj.Model.prototype._fireAttrChange = function (prop, value, options, silent) {
  if (prop != null) {
    this.TriggerInternal(silent, oj.Events.EventType.CHANGE + ':' + prop, this, value, options);
  }
};

/**
 * @private
 */
oj.Model.prototype._fireChange = function (options, silent) {
  this.TriggerInternal(silent, oj.Events.EventType.CHANGE, this, options, null);
};

/**
 * @protected
 */
oj.Model.prototype.SetupId = function () {
    // Replicate id attribute at top level
  var id = null;
    // Ask for collection's function if available
  if (this.collection && this.collection.modelId) {
    var modFunc = this.collection.modelId;
    id = $.isFunction(modFunc) ? modFunc.call(this.collection, this.attributes) : modFunc;
  }
  if (!id) {
    var idAttr = this._getIdAttr();
    id = this.attributes != null ? this.attributes[idAttr] : null;
  }
    // Supposedly this should always be model.id...who knew?
  this.id = id;
};

/**
 * @private
 */
oj.Model.prototype._setPropInternal = function (prop, value, copyRegardless) {
  var equality = oj.Object.__innerEquals(this.attributes[prop], value);
  if (copyRegardless || !equality) {
    this.attributes[prop] = value;
    this.SetupId();
        // Return value management here seems bizarre due to backbone tests: do the direct set if copyRegardless,
        // but only return if the inner equals was different
    return !equality;
  }
  return false;
};

/**
 * @private
 */
oj.Model.prototype._clearChanged = function () {
  this.changed = {};
};

/**
 * @private
 */
oj.Model.prototype._addChange = function (property, value) {
  this.changed[property] = value;
};

/**
 * @ignore
 * @private
 * @param {Object|string} prop
 * @param {Object|null|undefined} value
 * @param {boolean} copyRegardless
 * @param {boolean} propertyBag
 * @param {Object} options
 * @param {boolean} init
 * @returns {boolean}
 */
oj.Model.prototype._setProp = function (prop, value, copyRegardless, propertyBag, options, init) {
  if (prop == null) {
    return true;
  }

  var attrs = {};
  var isNested = this.nestedSet;
  var opts;

  if (!propertyBag) {
    attrs[prop] = value;
  } else {
    // We've passed in a whole property bag at once: validate all together
    Object.keys(prop).forEach(function (p) {
      if (Object.prototype.hasOwnProperty.call(prop, p)) {
        attrs[p] = prop[p];
      }
    });
  }
  opts = options || {};

  if (!this._checkValid(attrs, { validate: opts.validate }, false)) {
    return false;
  }

  if (!isNested) {
    this._clearChanged();
    this.changes = [];
  }

    // Store old value
  if (!this.nestedSet && !init) {
    this.previousAttrs = oj.Model._cloneAttributes(this.attributes, null);
  }

  this.nestedSet = true;
  var self = this;
  Object.keys(attrs).forEach(function (p) {
    if (Object.prototype.hasOwnProperty.call(attrs, p)) {
      if (self._setPropInternal(p, attrs[p], copyRegardless)) {
        // Trigger changes
        self._addChange(p, attrs[p]);
        self.changes.push(p);
      } else {
        delete attrs[p];
      }
    }
  });
    // Fire events: don't fire if silent
  var silent = opts.silent;
  Object.keys(attrs).forEach(function (p) {
    if (Object.prototype.hasOwnProperty.call(attrs, p)) {
      if (!silent && (self.changes.length > 0 || (isNested && self.changes.indexOf(p) === -1))) {
        self.pendingChanges = true;
        self.pendingOpts = opts;
      }
      self._fireAttrChange(p, attrs[p], opts, silent);
    }
  });

  if (isNested) {
    return true;
  }
  if (!silent && !isNested) {
    while (this.pendingChanges) {
      this.pendingChanges = false;
      this._fireChange(this.pendingOpts, silent);
    }
  }

  this.nestedSet = false;
  return true;
};


/**
 * Clears all attributes from the model<br>
 * Events:<p>
 * <ul>
 * <b>change:attr</b>: fired for each attribute cleared, passing the model, name of the changed property, and
 * options<br>
 * <b>change:all</b>: fired after all attributes have been cleaered, passing the model and options<br>
 * </ul>
 * <p>
 *
 * @param {Object=} options
 * @property {boolean=} silent if true, do not fire events
 * @property {boolean=} validate if true, validate the unsetting of all properties
 * @return {oj.Model|boolean} the model, or false if validation on clear fails
 * @memberof oj.Model
 * @since 1.0.0
 * @export
 */
oj.Model.prototype.clear = function (options) {
  // Use unset to silently clear, to track changes to attributes
  var unsetOpt = { silent: true };
  var silent;
  var p;
  var opt = options || {};
  silent = opt.silent;
  unsetOpt.validate = opt.validate;
  this._clearChanged();

  var self = this;
  for (p in self.attributes) { // eslint-disable-line no-restricted-syntax
    if (Object.prototype.hasOwnProperty.call(self.attributes, p)) {
      if (!this._unsetInternal(p, unsetOpt, true)) {
        return false;
      }
      this.TriggerInternal(silent, oj.Events.EventType.CHANGE + ':' + p, this, undefined, null);
    }
  }
  this.attributes = {};
  this.SetupId();

  this._fireAttrChange(null, null, null, silent);
  this._fireChange(null, silent);
  return this;
};

/**
 * @private
 */
oj.Model._cloneAttributes = function (oldData, nd) {
  var newData = nd || {};

  // Handle not overwriting defaults with undefined
  // IE11 issue with Object.keys
  var newDataKeys = typeof newData === 'object' ? Object.keys(newData) : [];

  var canUseJson = true;

  var prop;
  if (newDataKeys.length > 0) {
    for (prop in newData) { // eslint-disable-line no-restricted-syntax
      if (Object.prototype.hasOwnProperty.call(newData, prop) &&
          Object.prototype.hasOwnProperty.call(oldData, prop)) {
        // They both have this: now is oldData undefined?
        if (oldData[prop] === undefined) {
          // Remove it so it doesn't get copied/overwritten
          delete oldData[prop]; // eslint-disable-line no-param-reassign
        }
      }
    }
    oj.CollectionUtils.copyInto(newData, oldData, undefined, true, 10000);
    return newData;
  }
  var type;
  for (prop in oldData) { // eslint-disable-line no-restricted-syntax, guard-for-in
    type = $.type(oldData[prop]);
    if (type === 'function' || type === 'undefined' || type === 'date' || type === 'array' || type === 'object') {
      canUseJson = false;
      break;
    }
  }

  if (canUseJson) {
    newData = JSON.parse(JSON.stringify(oldData));
  } else if (typeof oldData === 'object') {
    // IE11 issue with Object.keys
    oj.CollectionUtils.copyInto(newData, oldData, undefined, true, 10000);
  }
  return newData;
};

/**
 * Return a copy of the model with identical attributes and settings
 * @return {oj.Model} copy of the model
 * @memberof oj.Model
 * @since 1.0.0
 * @export
 */
oj.Model.prototype.clone = function () {
  var c = new this.constructor();
  var prop;

  for (prop in this) { // eslint-disable-line no-restricted-syntax
        // Shallow copy all but data
    if (Object.prototype.hasOwnProperty.call(this, prop) && this[prop] !== this.attributes) {
      c[prop] = this[prop];
    }
  }
    // Deep copy data
  c.attributes = oj.Model._cloneAttributes(this.attributes, null);

    // Remove the cid--this should be unique
  delete c.cid;
    // Set a new cid
  c.SetCid();

  c.SetupId();

  return c;
};

/**
 * Does this model match the given id or cid?
 * @protected
 */
oj.Model.prototype.Match = function (id, cid) {
  var modId = this.GetId();
  var modCid;
  if (modId !== undefined && modId == id) { // eslint-disable-line eqeqeq
    return true;
  }
  modCid = this.cid;
  if (modCid !== undefined && modCid == cid) { // eslint-disable-line eqeqeq
    return true;
  }
  return false;
};

/**
 * Set the value(s) of one or more attributes of the model, and fire events.
 * Events:<p>
 * <ul>
 * <b>change:attr</b>: fired for each attribute set, passing the model, name of the changed property, and options<br>
 * <b>change:all</b>: fired after all attributes have been set, passing the model and options<br>
 * </ul>
 * <p>
 * @param {string|Object} property Property attribute name to set, or an Object containing attribute/value pairs
 * @param {Object=} value Value for property if property is not an Object containing attribute/value pairs
 * @param {Object=} options
 * @property {boolean=} silent prevent events from firing
 * @property {boolean=} unset delete all the properties passed in rather than setting them<br>
 * @returns {oj.Model|boolean} the model itself, false if validation failed on set
 * @memberof oj.Model
 * @since 1.0.0
 * @export
 */
oj.Model.prototype.set = function (property, value, options) {
  var opts = options || {};
  var prop;
  var valid = true;

  if (arguments && arguments.length > 0) {
    // Check if first arg is not a string (property name)
    if (!oj.StringUtils.isString(property)) {
      // Options, if present, must be second argument...value, because a string/value
      // pair wasn't what was passed in
      opts = value || {};
      // For set, pass entire thing to setProp
      if (opts.unset) {
        for (prop in property) { // eslint-disable-line no-restricted-syntax
          if (Object.prototype.hasOwnProperty.call(property, prop)) {
            this._unsetInternal(prop, null, false);
          }
        }
      } else if (!this._setProp(property, null, true, true, opts, false)) {
        valid = false;
      }
    } else if (opts.unset) {
      // Not a property bag?  We assume it's a property/value argument
      this._unsetInternal(property, null, false);
    } else if (!this._setProp(property, value, false, false, opts, false)) {
      valid = false;
    }
  }
  if (valid) {
    return this;
  }
  return false;
};

/**
 * Deletes the given property from the model.<br>
 * Events:<p>
 * <ul>
 * <b>change:attr</b>: fired for each attribute unset: passing the model, name of the changed property, and
 * options<br>
 * <b>change:all</b>: fired after all attributes have been unset: passing the model and options<br>
 * @param {string} property Property to remove from model
 * @param {Object=} options
 * @property {boolean=} silent do not fire change events if true
 * @returns {boolean} false if validation of the unset fails
 * @memberof oj.Model
 * @since 1.0.0
 * @export
 */
oj.Model.prototype.unset = function (property, options) {
  return this._unsetInternal(property, options, false);
};


/**
 * @private
 */
oj.Model.prototype._unsetInternal = function (property, opts, clear) {
  var options = opts || {};
  var silent = options.silent;
  var attrs = {};

  if (this.has(property)) {
    if (!this._checkValid(attrs, options, false)) {
      return false;
    }
    if (!clear) {
      this._clearChanged();
    }

        // attrs[property] = undefined;
    delete this.attributes[property];
    this._addChange(property, undefined);
        // if (!silent) {
    this._fireAttrChange(property, null, null, silent);
    this._fireChange(null, silent);
        // }
  }
  this.SetupId();
  return true;
};

/**
 * Returns the value of a property from the model.
 * @param {string} property Property to get from model
 * @return {any} value of property
 * @memberof oj.Model
 * @since 1.0.0
 * @export
 */
oj.Model.prototype.get = function (property) {
  return this.attributes[property];
};

/**
 * Determines if the Model has a certain property set, vs. undefined.
 * @param {string} property Property to check for
 * @return {boolean} true if the model contains the given property, false if undefined.
 * @memberof oj.Model
 * @export
 */
oj.Model.prototype.has = function (property) {
  return oj.Collection._defined(this.attributes[property]);
};

/**
 * Loads the Model object from the data service URL. Performs a data "read."<br>
 * Events:<p>
 * <ul>
 * <b>request</b>: fired when the request to fetch is going to the server, passing the model, xhr object, and
 * options<br>
 * <b>sync</b>: fired when the model is fetched from the data service, passing the model and the raw response<br>
 * <b>error</b>: fired if there is an error during the fetch, passing the model, xhr ajax object, and options<br>
 * </ul>
 * <p>
 * @param {Object=} options Options to control fetch<p>
 * <b>success</b>: a user callback called when the fetch has completed successfully. This makes the fetch an
 * asynchronous process. The callback is called passing the Model object, raw response, and the fetch options
 * argument.<p>
 * <b>error</b>: a user callback function called if the fetch fails. The callback is called passing the model
 * object, xhr, and options arguments.<p>
 * @return {Object} xhr ajax object, by default.  If [sync]{@link oj.Model#sync} has been replaced, this would be
 * the value returned by the custom implementation.
 * @memberof oj.Model
 * @since 1.0.0
 * @ojsignature {target: "Type", value:"{success?: (model: oj.Model, response: any, options: object)=> void,
 *                                                  error?: (model: oj.Model, xhr: any, options: object)=> void,
 *                                                  [propName: string]: any}", for: "options"}
 * @export
 */
oj.Model.prototype.fetch = function (options) {
  var tempOpts = options || {};
  var success = tempOpts.success;
  var userErr = tempOpts.error;
  var self = this;
  var opts;

  opts = oj.Model._copyOptions(tempOpts);
  opts.error = function (xhr, status, err) {
    // Trigger an error event
    oj.Model._triggerError(self, false, tempOpts, status, err, xhr);

    if (userErr) {
      userErr.apply(self, arguments);
    }
  };

  opts.success = function (response) {
    // Make sure we pass xhr
    if (opts.xhr) {
      tempOpts.xhr = opts.xhr;
    }
    oj.Model._fireSyncEvent(self, response, opts, false);

    if ($.isFunction(self.parse)) {
      self.set(self.parse(response), opts);
    }
    if (success) {
      success.call(oj.Model.GetContext(opts, self), self, response, tempOpts);
    }
  };
  return oj.Model._internalSync('read', this, opts);
};


/**
 * @private
 */
oj.Model.prototype._parseImpl = function (rawData) {
  return rawData;
};

/**
 * Optional callback to parse responses from the server.  It is called with the server's response with a model's data and should return a response (possibly modified) for processing
 * @type {function(Object):Object}
 * @ojstatus preview
 * @memberof oj.Model
 * @since 1.0.0
 * @export
 */
oj.Model.prototype.parse = oj.Model.prototype._parseImpl;

/**
 * Return the URL used to access this model in the data source
 *
 * @returns {string|null} url to access this model in the data source
 * @memberof oj.Model
 * @since 1.0.0
 * @export
 */
oj.Model.prototype.url = function () {
  var urlRoot = this._getUrlRoot();
  var id = this.GetId();
  var coll;
  var collUrl;
  var slash;

  if (urlRoot) {
    return id ? urlRoot + '/' + encodeURIComponent(id) : urlRoot;
  }

  coll = this.collection;
  if (coll) {
    collUrl = oj.Model.GetPropValue(coll, 'url');
    if (id && collUrl) {
      slash = oj.Model._getLastChar(collUrl) === '/' ? '' : '/';
      return collUrl + slash + encodeURIComponent(this.GetId());
    }
    return collUrl;
  }

  throw new oj.URLError();
};


/**
 * Return all of the model's attributes as an array
 *
 * @returns {Array.<Object>} array of all the model's attributes
 * @memberof oj.Model
 * @since 1.0.0
 * @export
 */
oj.Model.prototype.keys = function () {
  var retArray = [];
  var self = this;

  Object.keys(self.attributes || {}).forEach(function (prop) {
    if (Object.prototype.hasOwnProperty.call(self.attributes, prop)) {
      retArray.push(prop);
    }
  });
  return retArray;
};


/**
 * Return all of the model's attributes values as an array
 *
 * @returns {Array.<Object>} array of all the model's attributes values
 * @memberof oj.Model
 * @since 1.0.0
 * @export
 */
oj.Model.prototype.values = function () {
  var retArray = [];
  var self = this;

  Object.keys(self.attributes || {}).forEach(function (prop) {
    if (Object.prototype.hasOwnProperty.call(self.attributes, prop)) {
      retArray.push(self.get(prop));
    }
  });
  return retArray;
};

/**
 * Return an array of attributes/value pairs found in the model
 *
 * @returns {Array.<Object>} returns the model's attribute/value pairs as an array
 * @memberof oj.Model
 * @since 1.0.0
 * @export
 */
oj.Model.prototype.pairs = function () {
  var retObj = [];
  var item;
  var self = this;

  Object.keys(self.attributes || {}).forEach(function (prop) {
    if (Object.prototype.hasOwnProperty.call(self.attributes, prop)) {
      item = [];
      item.push(prop);
      item.push(self.get(prop));
      retObj.push(item);
    }
  });
  return retObj;
};

/**
 * Return attribute/value pairs for the model minus those attributes listed in keys
 *
 * @param {Array.<Object>|Object} keys keys to exclude from the returned attribute/value pairs
 *
 * @returns {Object} array of the model's attribute/value pairs except those listed in keys
 * @memberof oj.Model
 * @since 1.0.0
 * @export
 */
oj.Model.prototype.omit = function (keys) {
  var keyArr = [];
  var i;
  var retObj = {};

  if (keys instanceof Array) {
    keyArr = keys;
  } else {
    for (i = 0; i < arguments.length; i++) {
      keyArr.push(arguments[i]);
    }
  }
  var self = this;

  Object.keys(self.attributes || {}).forEach(function (prop) {
    if (Object.prototype.hasOwnProperty.call(self.attributes, prop)) {
      if (keyArr.indexOf(prop) === -1) {
        retObj[prop] = self.get(prop);
      }
    }
  });
  return retObj;
};

/**
 * Return attribute/value pairs for the model for the keys
 *
 * @param {Array.<Object>|Object} keys keys for which to return attribute/value pairs
 *
 * @returns {Array.<Object>} array of the model's attribute/value pairs filtered by keys
 * @memberof oj.Model
 * @since 1.0.0
 * @export
 */
oj.Model.prototype.pick = function (keys) {
  var keyArr = [];
  var i;
  var retObj = {};

  if (keys instanceof Array) {
    keyArr = keys;
  } else {
    for (i = 0; i < arguments.length; i++) {
      keyArr.push(arguments[i]);
    }
  }
  for (i = 0; i < keyArr.length; i++) {
    if (Object.prototype.hasOwnProperty.call(this.attributes, keyArr[i])) {
      retObj[keyArr[i]] = this.get(keyArr[i]);
    }
  }
  return retObj;
};

/**
 * Return an array of value/attribute pairs found in the model
 *
 * @returns {Object} returns the model's value/attribute pairs as an array
 * @memberof oj.Model
 * @since 1.0.0
 * @export
 */
oj.Model.prototype.invert = function () {
  var retObj = {};
  var val;
  var self = this;

  Object.keys(self.attributes || {}).forEach(function (prop) {
    if (Object.prototype.hasOwnProperty.call(self.attributes, prop)) {
      val = self.get(prop);
      retObj[val] = prop;
    }
  });
  return retObj;
};

/**
 * @private
 */
oj.Model._getLastChar = function (str) {
  return str.charAt(str.length - 1);
};

/**
 * @private
 */
oj.Model.prototype._saveUrl = function () {
  var urlRoot = this._getUrlRoot();
  if (urlRoot) {
    return urlRoot;
  }

  if (this.GetCollection()) {
    return this.GetCollection().url;
  }

  return null;
};

/**
 * @private
 */
oj.Model.prototype._getUrlRoot = function () {
  return oj.Model.GetPropValue(this, 'urlRoot');
};

/**
 * @private
 */
oj.Model.prototype._parseSaveImpl = function (modelData) {
  return modelData;
};

/**
 * Callback function when writing a model to the server
 * @type {function(Object):Object}
 * @since 1.0.0
 * @memberof oj.Model
 * @ojstatus preview
 * @export
 */
oj.Model.prototype.parseSave = oj.Model.prototype._parseSaveImpl;

/**
 * Check to see if the model is valid by running the validate callback, if it is set
 *
 * @returns {boolean} true if validate passes or if no validate callback
 * @memberof oj.Model
 * @since 1.0.0
 * @export
 */
oj.Model.prototype.isValid = function () {
  var options = {};
  options.validate = this.validate;
  return this._checkValid(this.attributes, options, false);
};

/**
 * @private
 */
oj.Model._isValidateSet = function (opt, save) {
  var options = opt || {};
  if (options.validate !== undefined && options.validate !== null) {
    return options.validate;
  }
    // The "default" is different for save vs. set
  return save;
};

/**
 * @private
 */
oj.Model.prototype._checkValid = function (attributes, opt, save) {
  var options = opt || {};
  var validate = this.validate;
  if (validate && oj.Model._isValidateSet(options, save)) {
        // If we have a validate override and it returns something, don't save
    this.validationError = validate.call(this, attributes, options);
    if (this.validationError) {
      this.TriggerInternal(false, oj.Events.EventType.INVALID, this, this.validationError, options);
      return false;
    }
  }
  return true;
};

/**
 * @private
 */
oj.Model._processArgs = function (args) {
  var ignoreLastArg = false;
  var options = {};
  var attributes = {};
  var i;

  if (args) {
    if (args.length > 0) {
            // Check if the last argument is not the first argument
      if (args.length > 1) {
        if (args[args.length - 1] && oj.Model._hasProperties(args[args.length - 1])) {
                    // Last arg is options: ignore later
          ignoreLastArg = true;
          options = args[args.length - 1] || {};
        }
      }
      if (args[0] == null) {
        return { attributes: null, options: options };
      }

      // Check if first arg is property bag
      if (oj.Model._hasProperties(args[0]) || oj.Object.isEmpty(args[0])) {
        Object.keys(args[0]).forEach(function (prop) {
          if (Object.prototype.hasOwnProperty.call(args[0], prop)) {
            attributes[prop] = args[0][prop];
          }
        });
      } else {
        // Not a property bag?  We assume arguments are a series of attr/values
        for (i = 0; i < args.length; i += 2) {
          // Process the arg as long as its: defined, and isn't the last argument where we're supposed to
          // ignore the last argument due to it being 'options'
          if (args[i] !== undefined || i < args.length - 1 || (!ignoreLastArg && i ===
            args.length - 1)) {
            attributes[args[i]] = args[i + 1];
          }
        }
      }
    }
  }
  return { attributes: attributes, options: options };
};

/**
 * @private
 */
oj.Model._copyOptions = function (opt) {
  var optReturn = {};
  var options = opt || {};

  Object.keys(options).forEach(function (prop) {
    if (Object.prototype.hasOwnProperty.call(options, prop)) {
      optReturn[prop] = options[prop];
    }
  });
  return optReturn;
};

/**
 * @private
 */
oj.Model._triggerError = function (self, silent, opt, status, err, xhr) {
  var options = opt || {};
  options.textStatus = status;
  options.errorThrown = err;
  self.TriggerInternal(silent, oj.Events.EventType.ERROR, self, xhr, options);
};


/**
 * Saves the current Model object to the data service. Performs a data "update."<br>
 * Events:<p>
 * <ul>
 * <b>change:attr</b>: fired for each attribute changed, if passed in as part of the save: passing the model, name
 * of the changed property, and options<br>
 * <b>change:all</b>: fired after all attributes have been set, if passed in as part of the save: passing the
 * model and options<br>
 * <b>request</b>: fired when the request to save is going to the server, passing the model, xhr object, and
 * options<br>
 * <b>sync</b>: fired when the model is saved to the data service, passing the model and the raw response<br>
 * <b>error</b>: fired if there is an error during the save, passing the model, xhr ajax object, and options<br>
 * </ul>
 * <p>
 * @param {Object=} attributes One or more attribute name/value pairs to set on the model before the save.
 * @param {Object=} options Options to control save<br>
 * <b>success</b>: a user callback called when the save has completed successfully. This makes the save an
 * asynchronous process. The callback is called passing the Model object, response from the AJAX call, and the
 * fetch options argument.<p>
 * <b>error</b>: a user callback function called if the save fails. <p>
 * <b>contentType</b>: in case the user's REST service requires a different POST content type than the default,
 * 'application/json'<p>
 * <b>validate</b>: should the validation routine be called if available<p>
 * <b>wait</b>: if true, wait for the server call before setting the attributes on the model<p>
 * <b>patch</b>: should only changed attributes be sent via a PATCH?<p>
 * <b>attrs</b>: pass a set of attributes to completely control the set of attributes that are saved to the
 * server (generally used with patch)
 * @return {Object|boolean} returns false if validation failed, or the xhr object
 * @memberof oj.Model
 * @since 1.0.0
 * @ojsignature {target: "Type", value:"{success?: (model: oj.Model, response: any, options: object)=> void,
 *                                                  error?: (model: oj.Model, xhr: any, options: object)=> void,
 *                                                  contentType?: string, valdiate?: boolean, wait?: boolean, patch?: boolean, attrs?: object,
 *                                                  [propName: string]: any}", for: "options"}
 * @export
 */
oj.Model.prototype.save = function (attributes, options) {
  var forceNew;
  var success;
  var callback;
  var self;
  var userErr;
  var patch;
  var argResults = oj.Model._processArgs(arguments);
  var opts;
  var oldAttrs;
  var attrArgs;
  attrArgs = attributes === undefined ? undefined : argResults.attributes;

  var tempOpts = options || {};
  opts = oj.Model._copyOptions(argResults.options);

  var validAttrs = $.extend(true, {}, this.attributes, attrArgs);
  if (!this._checkValid(validAttrs, opts, true)) {
    return false;
  }

  if (!opts.wait) {
    this.set(attrArgs);
  }

  forceNew = opts.forceNew === undefined ? false : opts.forceNew;
  self = this;
  userErr = opts.error;
  patch = opts.patch;

  opts.error = function (xhr, status, err) {
                            // Trigger an error event
    oj.Model._triggerError(self, false, tempOpts, status, err, xhr);

    if (userErr) {
      userErr.apply(self, arguments);
    }
  };

  opts.saveAttrs = opts.wait ? this._attrUnion(attrArgs) : this.attributes;

    // Must temporarily at least set attrs for toJSON()
  oldAttrs = this.attributes;
    // Swap in what's to be saved and call toJSON()
  this.attributes = opts.saveAttrs;
  opts.saveAttrs = this.toJSON();
  this.attributes = oldAttrs;

  if (!forceNew && !this.isNew()) {
    success = opts.success;
    opts.success =
            function (resp) {
              var attrs;

                // Make sure we pass xhr
              if (opts.xhr) {
                tempOpts.xhr = opts.xhr;
              }

              if (resp && !oj.Object.isEmpty(resp)) {
                if ($.isFunction(self.parse)) {
                  attrs = self.parse(resp);
                } else {
                  attrs = resp;
                }

                self.attributes = $.extend(true, self.attributes, attrs);

                // Merge attrs from response/parse into arg attrs if different--server takes priority in case
                // of 'wait'
                if (opts.wait) {
                  Object.keys(attrArgs || {}).forEach(function (prop) {
                    if (Object.prototype.hasOwnProperty.call(attrs, prop)) {
                      // Prioritize the one in attrs
                      attrArgs[prop] = attrs[prop];
                    }
                  });
                }
                self.SetupId();
              }
              oj.Model._fireSyncEvent(self, resp, opts, false);

              if (opts.wait) {
                self.set(attrArgs);
              }

              if (success) {
                success.call(oj.Model.GetContext(opts, self), self, resp, tempOpts);
              }
              self._clearChanged();
            };
        // If caller passes in attrs themselves, just use those
    if (!opts.attrs) {
      if (attrArgs === undefined) {
        opts.attrs = undefined;
      } else {
        opts.attrs = patch ? attrArgs : opts.saveAttrs;
      }
    }
    return oj.Model._internalSync(patch ? 'patch' : 'update', this, opts);
  }

  callback = oj.Model._getSuccess(opts);
  opts.success = function (resp) {
    var attrs;
       // Make sure we pass xhr
    if (opts.xhr) {
      tempOpts.xhr = opts.xhr;
    }
    if (resp && !oj.Object.isEmpty(resp)) {
      if ($.isFunction(self.parse)) {
        attrs = self.parse(resp);
      } else {
        attrs = resp;
      }
      if (!self._checkValid(attrs, opts, true)) {
        return;
      }

      self.attributes = $.extend(true, self.attributes, attrs);

      // Merge attrs from response/parse into arg attrs if different--server takes priority in case of 'wait'
      if (opts.wait) {
        Object.keys(attrArgs || {}).forEach(function (prop) {
          if (Object.prototype.hasOwnProperty.call(attrs, prop)) {
            // Prioritize the one in attrs
            attrArgs[prop] = attrs[prop];
          }
        });
      }
      self.SetupId();
    }
    oj.Model._fireSyncEvent(self, resp, opts, false);
    if (opts.wait) {
      self.set(attrArgs);
    }

    if (callback) {
      callback.call(oj.Model.GetContext(opts, self), self, resp, tempOpts);
    }
    self._clearChanged();
  };

    // If caller passed in attrs, just use those
  if (!opts.attrs) {
    opts.attrs = opts.saveAttrs;
  }

    // Turn on parse flag
  opts.parse = true;

    // Bizarre case tested by backboneJS--if this is a new model, but we're patching, make sure we only save the
    // explicit attrs if passed in by user
  if (patch) {
    opts.saveAttrs = opts.attrs;
  }
  return oj.Model._internalSync('create', this, opts);
};

/**
 * @private
 */
oj.Model.prototype._attrUnion = function (attrs) {
  var attrReturn = {};
  var self = this;

  Object.keys(self.attributes || {}).forEach(function (prop) {
    if (Object.prototype.hasOwnProperty.call(self.attributes, prop)) {
      attrReturn[prop] = self.attributes[prop];
    }
  });
  Object.keys(attrs || {}).forEach(function (prop) {
    if (Object.prototype.hasOwnProperty.call(attrs, prop)) {
      attrReturn[prop] = attrs[prop];
    }
  });
  return attrReturn;
};

/**
 * @protected
 */
oj.Model.GetPropValue = function (obj, property) {
  if (obj) {
    if ($.isFunction(obj[property])) {
      return obj[property]();
    }

    return obj[property];
  }
  return $.isFunction(property) ? property() : property;
};

/**
 * @protected
 */
oj.Model.IsComplexValue = function (val) {
  return val && Object.prototype.hasOwnProperty.call(val, 'value') && Object.prototype.hasOwnProperty.call(val, 'comparator');
};

/**
 * Does this model contain all of the given attribute/value pairs?
 * @private
 */
oj.Model.prototype._hasAttrs = function (attrs) {
  var prop;

  for (prop in attrs) { // eslint-disable-line no-restricted-syntax
    if (Object.prototype.hasOwnProperty.call(attrs, prop)) {
      if (!Object.prototype.hasOwnProperty.call(this.attributes, prop)) {
        return false;
      }

      var val = Array.isArray(attrs[prop]) ? attrs[prop] : [attrs[prop]];
      for (var i = 0; i < val.length; i++) {
        if (oj.Model.IsComplexValue(val[i])) {
          var comparator = val[i].comparator;
          var value = val[i].value;
          if (oj.StringUtils.isString(comparator)) {
            throw new Error('String comparator invalid for local where/findWhere');
          }
          if (!comparator(this, prop, value)) {
            return false;
          }
        } else if (attrs[prop] !== this.attributes[prop]) {
          // Array case meaningless here.  Model can't be == value1 and value2
          return false;
        }
      }
    }
  }
  return true;
};

/**
 * Return a function that determines if this model contains all of the property/values in attrs
 *
 * @param {Object} attrs property/value pairs
 * @returns {function(oj.Model):boolean} function taking an oj.Model that returns true if all of attrs are contained within it
 * @memberof oj.Model
 * @since 1.1.0
 * @export
 */
oj.Model.prototype.matches = function (attrs) {
  return (function (model) {
    for (var prop in attrs) { // eslint-disable-line no-restricted-syntax
      if (model.get(prop) !== attrs[prop]) {
        return false;
      }
    }
    return true;
  }(this));
};

/**
 * See if this model contains any of the given attribute/value pairs
 * @protected
 */
oj.Model.prototype.Contains = function (attrs) {
  var attrList = (attrs.constructor === Array) ? attrs : [attrs];
  var i;

  for (i = 0; i < attrList.length; i++) {
    if (this._hasAttrs(attrList[i])) {
      return true;
    }
  }
  return false;
};

/**
 * @private
 */
oj.Model._getSuccess = function (options) {
  return options != null && options.success ? options.success : null;
};

/**
 * @protected
 */
oj.Model.GetContext = function (options, model) {
  if (options !== undefined && options.context !== undefined) {
    return options.context;
  }
  return model;
};

/**
 * Determines if this model object has been assigned an id value yet. This indicates whether or not the model's
 * data has been saved to or fetched from the data service at any point.
 * @returns {boolean} true if the Model object has not had its id set yet, false if not.
 * @memberof oj.Model
 * @since 1.0.0
 * @export
 */
oj.Model.prototype.isNew = function () {
  return this.GetId() === undefined;
};

/**
 * @private
 */
oj.Model.prototype._getIdAttr = function () {
  return this.idAttribute || 'id';
};

/**
 * @protected
 */
oj.Model.prototype.GetId = function () {
  return this.id;
};

/**
 * Return the set of attributes and values that have changed since the last set.  Note that a Model.fetch() uses
 * set to store the returned attribute data. If attribute/value pairs are passed in, check those to see if they're
 * different than the model.
 * Return false if there were no changes
 * @param {Object=} attributes One or more attribute/value pairs to check against the model for changes
 * @return {Object|boolean} the set of all attribute value pairs that have changed since last set, if no
 * attributes passed in; the set of all attribute value pairs that are different than those listed in the
 * attributes parameter, if present.  False if no changes
 * @memberof oj.Model
 * @since 1.0.0
 * @export
 */
oj.Model.prototype.changedAttributes = function (attributes) {
  if (attributes) {
    var internalChanges = {};
    var self = this;
    Object.keys(attributes).forEach(function (prop) {
      if (Object.prototype.hasOwnProperty.call(attributes, prop)) {
        if (!oj.Object.__innerEquals(attributes[prop], self.attributes[prop])) {
          internalChanges[prop] = attributes[prop];
        }
      }
    });
    return oj.Object.isEmpty(internalChanges) ? false : internalChanges;
  }
  return oj.Object.isEmpty(this.changed) ? false : this.changed;
};

/**
 * Return true if the Model object has had any changes made to its values, or if any changes have been made to
 * the optional attribute passed in.
 * @param {string=} attribute attribute to check for changes
 * @returns {boolean} true if the Model object has had any changes since retrieval or last update at all (if no
 * attributes parameter); true if the Model object has had changes to the passed-in attribute
 * since retrieval or last update (if attribute parameter present).
 * @memberof oj.Model
 * @since 1.0.0
 * @export
 */
oj.Model.prototype.hasChanged = function (attribute) {
  if (attribute !== undefined) {
    return oj.Model._hasProperties(this.changed) &&
      Object.prototype.hasOwnProperty.call(this.changed, attribute);
  }
  return oj.Model._hasProperties(this.changed);
};


/**
 * Delete the record represented by this model object from the data service.  If the server responds with virtual
 * mode response properties (such as totalResults), these will be picked up at the end of the delete by the
 * model's collection.  Note that by default, the delete isn't sent with an HTTP data type so the object
 * returning these properties needs to be a string version of the JSON object.<br>
 * Events:<p>
 * <ul>
 * <b>destroy</b>: fired when the model is destroyed, either before or after the server call, depending on the
 * setting of wait.  The model and model's collection are passed in<br>
 * <b>request</b>: fired right as the call is made to request the server delete the model.  Passes the model,
 * xhr ajax object, and options.<br>
 * <b>sync</b>: fired after the server succeeds in destroying the model.  Passes the model, the raw response data,
 * and options.
 * </ul>
 * @param {Object=} options Options for the destroy operation:<br>
 * <b>wait</b>: if true, wait until the server is done to fire the destroy event.  Otherewise, fire immediately
 * and regardless of success or failure<br>
 * <b>success</b>: callback function called if the server call succeeds, passing the model, response data, and
 * options<br>
 * <b>error</b>: callback function on failure of the server call, firing an error event and passing the model,
 * xhr, status, and error values.<br>
 * @return {boolean}
 * @memberof oj.Model
 * @since 1.0.0
 * @ojsignature {target: "Type", value:"{success?: (model: oj.Model, response: any, options: object)=> void,
 *                                                  error?: (model: oj.Model, xhr: any, options: object)=> void,
 *                                                  wait?: boolean, [propName: string]: any}", for: "options"}
 * @export
 */
oj.Model.prototype.destroy = function (options) {
  var tempOpt = options || {};
  var isWait = tempOpt.wait;
  var callback;
  var userErr = tempOpt.error;
  var self = this;
  var xhr;
  var opts;

  opts = oj.Model._copyOptions(tempOpt);
  callback = oj.Model._getSuccess(opts);
    // Grab the collection off the model in case we need to update
  var collection = this.GetCollection();

  opts.success = function (data) {
    // Make sure we pass xhr
    if (opts.xhr) {
      tempOpt.xhr = opts.xhr;
    }

    // Give an opportunity to update any collection paging properties, like totalResults due to this destroy
    if (collection) {
      // Make sure to parse the data if necessary
      var props = oj.StringUtils.isString(data) && !oj.StringUtils.isEmpty(data) ?
        JSON.parse(data) : data;

      collection._setPagingReturnValues(props, null, data, true);
    }
    if (isWait) {
      self._fireDestroy(false);
    }
    oj.Model._fireSyncEvent(self, data, opts, false);

    if (callback) {
      callback.call(oj.Model.GetContext(opts, self), self, data, tempOpt);
    }
  };
  opts.error = function (xhrParam) {
    // Trigger an error event
    self.TriggerInternal(false, oj.Events.EventType.ERROR, self, xhrParam, tempOpt);

    if (userErr) {
      userErr.apply(self, arguments);
    }
  };

  if (!this.isNew()) {
    xhr = oj.Model._internalSync('delete', this, opts);
    if (!isWait) {
      this._fireDestroy(false);
    }
    return xhr;
  }
  if (!isWait) {
    this._fireDestroy(false);
  }
  if (callback) {
    callback.call(oj.Model.GetContext(opts, self), self, null, tempOpt);
  }
  return false;
};

/**
 * Fire request event
 * @private
 */
oj.Model.prototype._fireRequest = function (model, xhr, options, silent) {
  this.TriggerInternal(silent, oj.Events.EventType.REQUEST, model, xhr, options);
};

/**
 * Fire destroy event to all listeners
 * @private
 */
oj.Model.prototype._fireDestroy = function (silent) {
  this.TriggerInternal(silent, oj.Events.EventType.DESTROY, this, this.collection, null);
};

/**
 * Fire sync event to all listeners
 * @private
 */
oj.Model._fireSyncEvent = function (model, resp, options, silent) {
  model.TriggerInternal(silent, oj.Events.EventType.SYNC, model, resp, options);
};

/**
 * Return a copy of Model's current attribute/value pairs
 * @return {Object} a copy of the Model's current set of attribute/value pairs.
 * @memberof oj.Model
 * @since 1.0.0
 * @export
 */
oj.Model.prototype.toJSON = function () {
  var retObj = {};
  var self = this;

  Object.keys(self.attributes || {}).forEach(function (prop) {
    if (Object.prototype.hasOwnProperty.call(self.attributes, prop)) {
      if (Array.isArray(self.attributes[prop])) {
        retObj[prop] = self.attributes[prop].slice(0);
      } else {
        retObj[prop] = self.attributes[prop];
      }
    }
  });
  return retObj;
};

/**
 * Return the previous value of the given attribute, if any.
 *
 * @param {string} attr
 * @returns {Object} previous value of attr, if any.  If the attribute has not changed, returns undefined
 * @memberof oj.Model
 * @since 1.0.0
 * @export
 */
oj.Model.prototype.previous = function (attr) {
  return this.previousAttrs[attr];
};

/**
 * Return a copy of the model's previously set attributes
 *
 * @returns {Object} a copy of the model's previous attributes
 * @memberof oj.Model
 * @since 1.0.0
 * @export
 */
oj.Model.prototype.previousAttributes = function () {
  return this.previousAttrs;
};

/**
 * @export
 * Performs communications with the server.  Can be overridden/replaced by clients
 *
 * @param {string} method "create", "read", "update", or "delete"
 * @param {oj.Model} model Model to be read/saved/deleted/created
 * @param {Object=} options to control sync:<br>
 * <b>success</b>: called if sync succeeds:<br>
 * <ul>
 * For create, called with the response (attribute/value pairs); status (Ajax by default, but user-defined);
 * xhr object (Ajax--if available)<br>
 * For read, called with the response (attribute/value pairs being fetched)<br>
 * For update, same as create<br>
 * For delete, called with the oj.Model deleted, the data response (ignored), and the options passed to the
 * destroy call<br>
 * </ul>
 * <b>error</b>: called if sync fails.  Called with xhr, status, and error info, per jQuery Ajax (all if
 * available)<p>
 *
 * @return {Object} xhr object
 * @memberof oj.Model
 * @instance
 * @since 1.0.0
 * @alias sync
 */
oj.Model.prototype.sync = function (method, model, options) {
  return oj.sync(method, model, options);
};

/**
 * Internal processing before sync-- we want this stuff to happen even if user replaces sync
 * @private
 */
oj.Model._internalSync = function (method, model, opt) {
  var options = opt || {};
  // If Model/Collection has OAuth object, then create Authorization header (see oj.RestImpl.addOptions)
  if (model.oauth) {
    options.oauthHeader = model.oauth.getHeader();
  }

  // Make sure to transfer the data type if it's set on the calling object
  if (!options.dataType && model.dataType) {
    options.dataType = model.dataType;
  }
  if (!options.jsonpCallback && model.jsonpCallback) {
    options.jsonpCallback = model.jsonpCallback;
  }

  // Do parsing if necessary and tuck it on options
  if (method === 'create' || method === 'patch' || method === 'update') {
    options.parsedData = model.parseSave.call(model, method === 'patch' ? options.attrs :
                                                                               options.saveAttrs);
  }
  var recordId = null;
  if (model instanceof oj.Model) {
    recordId = model.GetId();
  }
  var newOpt = {};
  Object.keys(options || {}).forEach(function (prop) {
    newOpt[prop] = options[prop];
  });
  var urlOpt = oj.Model.SetCustomURLOptions(recordId, model, options);
  Object.keys(urlOpt || {}).forEach(function (prop) {
    newOpt[prop] = urlOpt[prop];
  });
    // Make sure we send back xhr in options-- can come from return value or passed back through options
  options.xhr = model.sync(method, model, newOpt);
  if (newOpt.xhr) {
    options.xhr = newOpt.xhr;
  }
  return options.xhr;
};

/**
 * Get all custom URL options
 * @protected
 */
oj.Model.SetCustomURLOptions = function (recordID, context, opt) {
  var options = context instanceof oj.Collection ? context.ModifyOptionsForCustomURL(opt) : {};
  if (recordID) {
    options.recordID = recordID;
  }
  return options;
};

/**
 * @export
 * @desc Master server access method for all models and collections.  Replace oj.sync with a new implementation
 * to customize all oj.Model and oj.Collection server interactions
 *
 * @param {string} method "create", "read", "update", "patch", or "delete"
 * @param {oj.Model|oj.Collection} model Model (or Collection to be read) to be read/saved/deleted/created
 * @param {Object=} options to control sync<br>
 * success: called if sync succeeds<br>
 * error: called if sync fails<br>
 * others are passed to jQuery<br>
 * Options that would normally be passed to a customURL callback are also included<br>
 *
 *
 * @return {Object} xhr object
 * @memberof oj
 * @global
 * @since 1.0.0
 * @alias oj.sync
 */
oj.sync = function (method, model, options) {
  var tempOpt = options || {};
  function _fireAndReturn(xhr) {
    model._fireRequest(model, xhr, tempOpt, tempOpt.silent);
    return xhr;
  }

  var restService;
  var success = tempOpt.success;
  var error = tempOpt.error;
  var url;

  if (method.valueOf() === 'create') {
    url = model._saveUrl();
    url = url || oj.Model.GetPropValue(model, 'url');
    restService = new oj.RestImpl(url, model);
    return _fireAndReturn(restService.addRecord(tempOpt.parsedData, error, tempOpt, model));
  }

  if (method.valueOf() === 'read') {
    if (model instanceof oj.Model) {
      url = tempOpt.url ? tempOpt.url : oj.Model.GetPropValue(model, 'url');
      restService = new oj.RestImpl(url, model);
      return _fireAndReturn(restService.getRecord(success, error, model.GetId(), tempOpt,
                                  oj.Model.GetContext(tempOpt, model)));
    }
    // Collection fetch
    url = model.GetCollectionFetchUrl(tempOpt);
    restService = new oj.RestImpl(url, model);
    return _fireAndReturn(restService.getRecords(success, error, tempOpt, model));
  }

  restService = new oj.RestImpl(oj.Model.GetPropValue(model, 'url'), model);
  var recordId = null;
  if (model instanceof oj.Model) {
    recordId = model.GetId();
  }
  if (method.valueOf() === 'update') {
    return _fireAndReturn(restService.updateRecord(success, recordId, tempOpt.parsedData,
      error, tempOpt, model, false));
  }
  if (method.valueOf() === 'patch') {
    return _fireAndReturn(restService.updateRecord(success, recordId, tempOpt.parsedData,
      error, tempOpt, model, true));
  }
  if (method.valueOf() === 'delete') {
    return _fireAndReturn(restService.deleteRecord(recordId, error, tempOpt, model));
  }
  return null;
};


/**
 * @private
 */
oj.Model._urlError = function (ajaxOptions) {
  if (!ajaxOptions.url) {
    throw new Error('The url property or function must be specified');
  }
};

/**
 * @export
 * @desc Master Ajax entry point for all oj.Model and oj.Collection server interactions, when they are using the
 * default sync implementations.  oj.ajax passes through to jQuery ajax by default.
 * See {@link http://api.jquery.com/jquery.ajax/} for expected parameters and return value.
 * @param {Object=} settings optional ajax settings
 *
 * @return {Object} xhr object
 * @memberof oj
 * @global
 * @since 1.0.0
 * @alias oj.ajax
 */
oj.ajax = function (settings) { // eslint-disable-line no-unused-vars
  if (arguments && arguments.length > 0) {
    oj.Model._urlError(arguments[0]);
  }
  return $.ajax.apply(oj, arguments);
};

/**
 * Copyright (c) 2014, 2015 Oracle and/or its affiliates.
 * All rights reserved.
 */

/* jslint browser: true*/
/* global jQuery:false*/

/**
 * @export
 * @class oj.OAuth
 * @classdesc Member of Model objects. Object representing name/value pairs for a data service record
 *
 * @param {Object} attributes Initial set of attribute/value pairs with which to seed this OAuth object
 * @param {string} header Actual name for the Authorization header (default 'Authorization')
 * @example <caption>Initialize OAuth with client credentials</caption>
 * var myOAuth = new oj.OAuth('X-Authorization', {...Client Credentials ...});
 *
 * @example <caption>Initialize OAuth with access_token</caption>
 * var myOAuth = new oj.OAuth('X-Authorization', {...Access Token...});
 *
 * @example <caption>Initialize empty OAuth and set access_token</caption>
 * var myOAuth = new oj.OAuth();
 * myOAuth.setAccessTokenResponse({...Access Token...});
 *
 * @constructor
 * @since 1.0.0
*/
oj.OAuth = function (header, attributes) {
  oj.OAuth._init(this, attributes || {}, header || 'Authorization');
};

// Subclass from oj.Object
oj.Object.createSubclass(oj.OAuth, oj.Object, 'oj.OAuth');
oj.OAuth.prototype.Init = function () {
  oj.OAuth.superclass.Init.call(this);
};

/**
 * Calculates Authorization header based on client credentials or access_token
 * @return {Object} OAuth 2.0 Authorization header
 * @example <caption>Get Authorization header</caption>
 * myOAuth.getHeader();
 *
 * @memberof oj.OAuth
 * @export
 */
oj.OAuth.prototype.getHeader = function () {
  var headers = {};
  if (!this.accessTokenResponse.access_token) {
    this.clientCredentialGrant();
  }
  headers[this.accessTokenRequest.auth_header] = 'Bearer ' + this.accessTokenResponse.access_token;
  return headers;
};

/**
 * Check is OAuth initialized (not null access_token).
 * @return {boolean} true/false
 * @example <caption>Check if OAuth initialized</caption>
 * if(myOAuth.isInitialized()) console.log('Initialized');
 *
 * @memberof oj.OAuth
 * @export
 */
oj.OAuth.prototype.isInitialized = function () {
  if (this.accessTokenResponse && this.accessTokenResponse.access_token) {
    return true;
  }
  return false;
};

/**
 * Request for access_token(bearer token) using Client Credential Authorization Grant.
 * Initialize response part of the OAuth object (access_token, e.t.c.)
 * @return {undefined}
 * @example <caption>Set/Re-set response part of the OAuth object using Client Credentials</caption>
 * myOAuth.clientCredentialGrant();
 *
 * @memberof oj.OAuth
 * @export
 */
oj.OAuth.prototype.clientCredentialGrant = function () {
  var headers = {};
  var self = this;
  headers[self.accessTokenRequest.auth_header] = 'Basic ' +
    oj.OAuth._base64_encode(self.accessTokenRequest.client_id + ':' +
    self.accessTokenRequest.client_secret);

  $.ajax({
    type: 'POST',
    async: false,
    url: this.accessTokenRequest.bearer_url,
    data: 'grant_type=client_credentials',
    headers: headers,
    success: function (data) {
      self.accessTokenResponse = oj.OAuth._initAccessToken(self.accessTokenResponse, data);
    },
    error: function (jqXHR) {
      throw new Error(jqXHR.responseText);
    }
  });
};

/**
 * Set response part of the OAuth object (access_token, e.t.c.)
 * @param {Object} data current response
 * @return {undefined}
 * @example <caption>'Initialize' response part of the OAuth object with access_token</caption>
 * myOAuth.setAccessTokenResponse({...Access Token...});
 *
 * @memberof oj.OAuth
 * @export
 */
oj.OAuth.prototype.setAccessTokenResponse = function (data) {
  this.accessTokenResopnse = oj.OAuth._initAccessToken(this.accessTokenResponse, data);
};

/**
 * Get response part of the OAuth object (access_token, e.t.c.)
 * @return {Object} cached response
 * @memberof oj.OAuth
 * @export
 */
oj.OAuth.prototype.getAccessTokenResponse = function () {
  return this.accessTokenResponse;
};

/**
 * Clean response part of the OAuth object (access_token, e.t.c.)
 * Null and remove all data from response part of the OAuth object
 * @return {undefined}
 * @memberof oj.OAuth
 * @export
 */
oj.OAuth.prototype.cleanAccessTokenResponse = function () {
  oj.OAuth._cleanAccessToken(this.accessTokenResponse);
};

/**
 * Set request part of the OAuth object (client credentials, uri endpoint)
 * @param {Object} data current client credentials and uri
 * @return {undefined}
 * @example <caption>'Initialize' request part of the OAuth object with client credentials and calculate
 * access_token</caption>
 * myOAuth.setAccessTokenRequest({...Client Credentials ...});
 * myOAuth.clientCredentialGrant();
 *
 * @memberof oj.OAuth
 * @export
 */
oj.OAuth.prototype.setAccessTokenRequest = function (data) {
  this.accessTokenRequest = oj.OAuth._initAccessToken(this.accessTokenRequest, data);
};

/**
 * Get request part of the OAuth object (client credentials, uri endpoint)
 * @return {Object} cached request
 * @memberof oj.OAuth
 * @export
 */
oj.OAuth.prototype.getAccessTokenRequest = function () {
  return this.accessTokenRequest;
};

/**
 * Clean request part of the OAuth object (client credentials, uri endpoint)
 * Null and remove all data from request part of the OAuth object
 * @return {undefined}
 * @memberof oj.OAuth
 * @export
 */
oj.OAuth.prototype.cleanAccessTokenRequest = function () {
  oj.OAuth._cleanAccessToken(this.accessTokenRequest);
};

/**
 * @private
 * @param {Object} oauth
 * @param {Object} attributes
 * @param {string|null} header
 */
oj.OAuth._init = function (oauth, attributes, header) {
  var oa = oauth;
  oa.Init();
  oa.accessTokenRequest = {};
  oa.accessTokenResponse = {};

  if (attributes.access_token) { // access_token has higher preference
    oa.accessTokenResponse = oj.OAuth._initAccessToken(oa.accessTokenResponse, attributes);
  } else if (attributes.client_id && attributes.client_secret && attributes.bearer_url) {
    // Client Credential Grant
    oa.accessTokenResponse = oj.OAuth._initAccessToken(oa.accessTokenRequest, attributes);
  }
  oa.accessTokenRequest.auth_header = header;
};

/**
 * @private
 * @param {Object} oauthObj - Request/Response object to deal with
 * @param {Object} data - object to populate
 */
oj.OAuth._initAccessToken = function (oauthObj, data) {
  var dat = data || {};
  var obj = oauthObj || {};
  Object.keys(dat).forEach(function (prop) {
    if (Object.prototype.hasOwnProperty.call(dat, prop)) {
      obj[prop] = dat[prop];
    }
  });
  return obj;
};

/**
 * @private
 * @param {Object} oauthObj - Request/Response object to deal with
 */
oj.OAuth._cleanAccessToken = function (oauthObj) {
  var obj = oauthObj || {};

  Object.keys(obj).forEach(function (key) {
    if (Object.prototype.hasOwnProperty.call(obj, key)) {
      if (key !== 'auth_header') {
        obj[key] = null;
        delete obj[key];
      }
    }
  });
};

/**
 * @private
 * @param {string} a The data to calculate the base64 representation from
 * @return {string} The base64 representation
 */
oj.OAuth._base64_encode = function (a) {
  var d;
  var e;
  var f;
  var b;
  var g = 0;
  var h = 0;
  var i = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=';
  var c = [];

  do {
    d = a.charCodeAt(g);
    g += 1;
    e = a.charCodeAt(g);
    g += 1;
    f = a.charCodeAt(g);
    g += 1;
    b = (d << 16) | (e << 8) | f; // eslint-disable-line no-bitwise
    d = (b >> 18) & 63; // eslint-disable-line no-bitwise
    e = (b >> 12) & 63; // eslint-disable-line no-bitwise
    f = (b >> 6) & 63; // eslint-disable-line no-bitwise
    b &= 63; // eslint-disable-line no-bitwise
    c[h] = i.charAt(d) + i.charAt(e) + i.charAt(f) + i.charAt(b);
    h += 1;
  } while (g < a.length);
  c = c.join('');
  d = a.length % 3;
  return (d ? c.slice(0, d - 3) : c) + '==='.slice(d || 3);
};

/**
 * Copyright (c) 2014, 2015 Oracle and/or its affiliates.
 * All rights reserved.
 */

/* jslint browser: true*/
/* global jQuery:false, Config:false*/
/**
 * @private
 * @constructor
 */
oj.RestImpl = function (rootURL, model) {
  this.rootURL = rootURL;
  this.model = model;
  this.customURL = model.customURL;
  $.support.cors = true;
};

oj.RestImpl._HEADER_PROP = 'headers';

// Add the properties in options to starter, if not already there
oj.RestImpl.addOptions = function (starter, options, customOptions) {
  var initial = $.extend(true, starter, customOptions);
  if (options) {
    Object.keys(options || {}).forEach(function (prop) {
      if (Object.prototype.hasOwnProperty.call(options, prop) && prop !== 'oauthHeader') {
        if (!Object.prototype.hasOwnProperty.call(initial, prop)) {
          initial[prop] = options[prop];
        }
        if (prop === oj.RestImpl._HEADER_PROP) {
                  // Deep merge
          initial[prop] = $.extend(true, initial[prop], options[prop]);
        }
      }
    });
  }

  if (options.oauthHeader) {
        // if there are no any headers then create a new one.
    if (!initial[oj.RestImpl._HEADER_PROP]) initial[oj.RestImpl._HEADER_PROP] = {};
    Object.keys(options.oauthHeader || {}).forEach(function (prop) {
      if (Object.prototype.hasOwnProperty.call(options.oauthHeader, prop)) {
        if (!Object.prototype.hasOwnProperty.call(initial[oj.RestImpl._HEADER_PROP], prop)) {
          initial[oj.RestImpl._HEADER_PROP][prop] = options.oauthHeader[prop];
        }
      }
    });
  }

  return initial;
};

oj.RestImpl.prototype.getRecords = function (callback, errFunc, options, context) {
  var opt = options || {};
  var isJsonp = opt.dataType === 'jsonp';
  var urlInfo = this._getURL('read', this.rootURL, this.customURL, null, context, opt);
  var ajaxOptions = {
    crossDomain: opt.crossDomain || !isJsonp,
    dataType: this._getDataType(opt),
    jsonpCallback: opt.jsonpCallback,
    context: context !== null ? context : this,
    success: callback,
    error: errFunc
  };
  ajaxOptions = this._addHeaderProp(ajaxOptions);
  ajaxOptions = oj.RestImpl.addOptions(ajaxOptions, opt, urlInfo);
  opt.xhr = this.ajax(ajaxOptions, context);
  return opt.xhr;
};

oj.RestImpl.prototype._addHeaderProp = function (options) {
  var opt = options || {};
  if (!(this.model && this.model.omitLanguageHeader)) {
    opt[oj.RestImpl._HEADER_PROP] = { 'Accept-Language': this.getLocale() };
  }
  return opt;
};

oj.RestImpl.prototype.getRecord = function (success, error, recordID, options, context) {
  var opt = options || {};
  var isJsonp = opt.dataType === 'jsonp';
  var urlInfo = this._getURL('read', this.rootURL, this.customURL, recordID, context, opt);
  var ajaxOptions = {
    crossDomain: opt.crossDomain || !isJsonp,
    dataType: this._getDataType(opt),
    jsonpCallback: opt.jsonpCallback,
    context: context !== null ? context : this,
    success: success,
    error: error
  };
  ajaxOptions = this._addHeaderProp(ajaxOptions);
  ajaxOptions = oj.RestImpl.addOptions(ajaxOptions, opt, urlInfo);
  opt.xhr = this.ajax(ajaxOptions, context);
  return opt.xhr;
};


oj.RestImpl.prototype.updateRecord = function (callback, recordID, record, error,
  options, context, patch) {
  var opt = options || {};
  var isJsonp = opt.dataType === 'jsonp';
  var urlInfo = this._getURL(patch ? 'patch' : 'update', this.rootURL, this.customURL, recordID, context, opt);
  var emulateHTTP = oj.RestImpl._emulateHTTP(opt);
  var ajaxOptions = {
    crossDomain: opt.crossDomain || !isJsonp,
    contentType: this._getContentType(opt),
    dataType: this._getDataType(opt),
    jsonpCallback: opt.jsonpCallback,
    data: this._getData(JSON.stringify(record), opt, urlInfo),
    emulateHTTP: emulateHTTP,
    emulateJSON: oj.RestImpl._emulateJSON(opt),
    success: callback,
    error: error,
    context: context !== null ? context : this
  };
  ajaxOptions = this._addHeaderProp(ajaxOptions);
  ajaxOptions = oj.RestImpl.addOptions(ajaxOptions, opt, urlInfo);
  ajaxOptions = oj.RestImpl._beforeSendMod(emulateHTTP, ajaxOptions);
  opt.xhr = this.ajax(ajaxOptions, context);
  return opt.xhr;
};

oj.RestImpl._beforeSendMod = function (emulateHTTP, options) {
  var opt = options || {};
  if (emulateHTTP) {
        // Do a before send xhr mod for this case
    var beforeSend = opt.beforeSend;
    opt.beforeSend = function (xhr) {
      xhr.setRequestHeader('X-HTTP-Method-Override', opt._method);
      if (beforeSend) {
        return beforeSend.apply(this, arguments);
      }
      return null;
    };
  }
  return opt;
};

oj.RestImpl.prototype._getData = function (data, options, urlInfo) {
  if (oj.RestImpl._emulateJSON(options)) {
        // Push record and _method into an object
    var retObj = { _method: urlInfo._method ? urlInfo._method : urlInfo.type };
    if (data) {
      retObj.model = data;
    }
    return retObj;
  }
  return data;
};

oj.RestImpl.prototype._getHTTPMethod = function (operation, options) {
  if (options.type) {
    return { method: options.type };
  }
  var method = null;
  if (operation === 'create') {
    method = 'POST';
  }
  if (operation === 'delete') {
    method = 'DELETE';
  }
  if (operation === 'patch') {
    method = 'PATCH';
  }
  if (operation === 'update') {
    method = 'PUT';
  }
  if (oj.RestImpl._emulateHTTP(options)) {
        // Convert method to POST, put original method under data._method
    var retObj = {};
    retObj.method = 'POST';
    retObj._method = method;
    return retObj;
  }
  if (method === null) {
    method = 'GET';
  }
  return { method: method };
};

oj.RestImpl._emulateHTTP = function (options) {
  return options.emulateHTTP || oj.emulateHTTP;
};

oj.RestImpl._emulateJSON = function (options) {
  return options.emulateJSON || oj.emulateJSON;
};

oj.RestImpl.prototype._getURL = function (operation, rootURL, customURL, recordID,
  context, options) {
  var httpMethod = this._getHTTPMethod(operation, options);
  if ($.isFunction(customURL)) {
    var result = customURL.call(this, operation, context,
                                   oj.Model.SetCustomURLOptions(recordID, context, options));
    if (oj.StringUtils.isString(result)) {
      var ret = { url: result, type: httpMethod.method };
      if (httpMethod._method) {
        ret._method = httpMethod._method;
      }
      return ret;
    } else if (result) {
      result.url = Object.prototype.hasOwnProperty.call(result, 'url') ? result.url : rootURL;
      if (!Object.prototype.hasOwnProperty.call(result, 'type')) {
        result.type = httpMethod.method;
      }
      if (!Object.prototype.hasOwnProperty.call(result, 'data')) {
        if (httpMethod._method) {
          result._method = httpMethod._method;
        }
      }
      return result;
    }
  }
  var retObj = { url: oj.Model.GetPropValue(null, rootURL), type: httpMethod.method };
  if (httpMethod._method) {
    retObj._method = httpMethod._method;
  }
  return retObj;
};

oj.RestImpl.prototype.deleteRecord = function (recordID, error, options, context) {
  var opt = options || {};
  var isJsonp = opt.dataType === 'jsonp';
  var urlInfo = this._getURL('delete', this.rootURL, this.customURL, recordID, context, opt);

  var emulateHTTP = oj.RestImpl._emulateHTTP(opt);
  var emulateJSON = oj.RestImpl._emulateJSON(opt);
  var ajaxOptions = {
    crossDomain: opt.crossDomain || !isJsonp,
    success: opt.success,
    error: error,
    context: context !== null ? context : this,
    emulateHTTP: emulateHTTP,
    emulateJSON: emulateJSON
  };
  var data = this._getData(null, opt, urlInfo);
  if (data) {
    ajaxOptions.data = data;
  }
  ajaxOptions = oj.RestImpl.addOptions(ajaxOptions, opt, urlInfo);
  ajaxOptions = oj.RestImpl._beforeSendMod(emulateHTTP, ajaxOptions);
  opt.xhr = this.ajax(ajaxOptions, context);
  return opt.xhr;
};

oj.RestImpl.prototype.addRecord = function (record, error, options, context) {
  var opt = options || {};
  var recordStr = JSON.stringify(record);
  var isJsonp = opt.dataType === 'jsonp';
  var urlInfo = this._getURL('create', this.rootURL, this.customURL, null, context, opt);

  var emulateHTTP = oj.RestImpl._emulateHTTP(opt);
  var ajaxOptions = {
    crossDomain: opt.crossDomain || !isJsonp,
    contentType: opt.contentType || 'application/json',
    dataType: this._getDataType(opt),
    jsonpCallback: opt.jsonpCallback,
    data: this._getData(recordStr, opt, urlInfo),
    success: opt.success,
    error: error,
    emulateHTTP: emulateHTTP,
    emulateJSON: oj.RestImpl._emulateJSON(opt),
    context: context !== null ? context : this
  };
  ajaxOptions = this._addHeaderProp(ajaxOptions);
  ajaxOptions = oj.RestImpl.addOptions(ajaxOptions, opt, urlInfo);
  opt.xhr = this.ajax(ajaxOptions, context);

  return opt.xhr;
};

oj.RestImpl.prototype._getDataType = function (options) {
  if (oj.RestImpl._emulateJSON(options) && !oj.RestImpl._emulateHTTP(options)) {
    return 'application/x-www-form-urlencoded';
  }
  return options.dataType || 'json';
};

oj.RestImpl.prototype._getContentType = function (options) {
  if (oj.RestImpl._emulateJSON(options) && !oj.RestImpl._emulateHTTP(options)) {
    return 'application/x-www-form-urlencoded';
  }

  return options.contentType || 'application/json';
};

oj.RestImpl.prototype.getLocale = function () {
  return Config.getLocale();
};

oj.RestImpl.prototype.ajax = function (settings, collection) {
  if (settings.url === null || settings.url === undefined) {
    throw new oj.URLError();
  }

  var xhr = oj.ajax(settings);
  if (collection._addxhr) {
    collection._addxhr(xhr);
  }
  return xhr;
};


/**
 * Copyright (c) 2014, 2015 Oracle and/or its affiliates.
 * All rights reserved.
 */

/* jslint browser: true*/

/**
 * @constructor
 * @class oj.URLError
 * @classdesc Constructs a URLError, thrown when API calls are made that require a URL but no URL is
 * defined.
 * @since 1.0.0
 * @export
 */
oj.URLError = function () {
  this.name = 'URLError';
  this.message = 'No URL defined';
};
oj.URLError.prototype = new Error();
oj.URLError.constructor = oj.URLError;

// Define a mapping variable that maps the return value of the module to the name used in the callback function of a require call.

var Model = {};
Model.Collection = oj.Collection;
Model.Events = oj.Events;
Model.Model = oj.Model;
Model.OAuth = oj.OAuth;
Model.URLError = oj.URLError;

/**
 * Copyright (c) 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */

/**
 * Constructs a message object.
 * <p><strong>NOTE:</strong>  Instead of using the constructor, please use an Object
 * that duck-types oj.Message - has summary, detail, and severity properties.
 * Creating an oj.Message Object provides no additional value than
 * the duck-typed Object.
 * </p>
 * @param {string} summary - Localized summary message text
 * @param {string} detail - Localized detail message text
 * @param {(number|string)=} severity - An optional severity for this message. Use constants
 * oj.Message.SEVERITY_LEVEL for number types and oj.Message.SEVERITY_TYPE for string types. Default
 * is SEVERITY_ERROR if no severity is specified
 * @constructor
 * @ojtsmodule
 * @export
 * @since 0.6.0
 * @example <caption>Set application messages using the
 * <code class="prettyprint">messages-custom</code> attribute. This example creates messages
 * the recommended way, by creating an Object that duck-types oj.Message.</caption>
 * --- HTML ---
 * &lt;oj-input-text messages-custom="{{appMessages}}");>&lt;/oj-input-text>
 *
 * --- Javascript ---
 * // for messagesCustom property
 * self.appMessages = ko.observable();
 * var msgs = [];
 * msgs.push({summary: "Error Summary", detail: "Error Detail",
 *  severity: oj.Message.SEVERITY_TYPE['CONFIRMATION']});
 * self.appMessages(msgs);
 * @ojsignature [{target: "Type",
 *                value: "oj.Message.SEVERITY_TYPE| oj.Message.SEVERITY_LEVEL",
 *                for: "severity"}]
 */
oj.Message = function (summary, detail, severity) {
  this.Init(summary, detail, severity);
};

/**
 * Indicates the type of severity that the message represents.
 * @enum {string}
 * @export
 */
oj.Message.SEVERITY_TYPE = {
  /**
   * Indicates a confirmation that an operation or task was completed. This is the lowest severity
   * level.
   */
  CONFIRMATION: 'confirmation',

  /**
   * Indicates information or operation messages. This has a lower severity level than warning.
   */
  INFO: 'info',

  /**
   * Indicates an application condition or situation that might require users' attention. This has a
   * lower severity than error.
   */
  WARNING: 'warning',

  /**
   * Used when data inaccuracies occur when completing a field and that needs fixing before user can
   * continue. This has a lower severity level than fatal.
   * fatal.
   */
  ERROR: 'error',

  /**
   * Used when a critical application error or an unknown failure occurs. This is the highest
   * severity level.
   * @const
   * @export
   */
  FATAL: 'fatal'
};


/**
 * Message severity level
 * @enum {number}
 * @export
 */
oj.Message.SEVERITY_LEVEL = {
  FATAL: 5,
  ERROR: 4,
  WARNING: 3,
  INFO: 2,
  CONFIRMATION: 1
};

// Subclass from oj.Object
oj.Object.createSubclass(oj.Message, oj.Object, 'oj.Message');

/**
 * Localized summary text.
 *
 * @member
 * @name summary
 * @memberof oj.Message
 * @instance
 * @type {string}
 * @default ""
 */

/**
 * Localized detail text.
 *
 * @member
 * @name detail
 * @memberof oj.Message
 * @instance
 * @type {string}
 * @default ""
 */

/**
 * Severity type of message. See oj.Message.SEVERITY_TYPE for string types and oj.Message.SEVERITY_LEVEL for number types.
 *
 * @member
 * @name severity
 * @memberof oj.Message
 * @instance
 * @type {string|number}
 * @ojsignature [{target: "Type",
 *                value: "oj.Message.SEVERITY_TYPE| oj.Message.SEVERITY_LEVEL"}]
 * @default oj.Message.SEVERITY_TYPE.ERROR
 */

/**
 * Initializes Message instance with the set options
 * @param {string} summary a localized summary message text
 * @param {string} detail a localized detail message text
 * @param {number|string=} severity - An optional severity for this message.  Use constants
 * oj.Message.SEVERITY_LEVEL for number types and oj.Message.SEVERITY_TYPE for string types.
 *
 * @export
 * @ignore
 */
oj.Message.prototype.Init = function (summary, detail, severity) {
  oj.Message.superclass.Init.call(this);
  this.summary = summary;
  this.detail = detail;
  this.severity = severity || oj.Message.SEVERITY_TYPE.ERROR; // defaults to ERROR
};


/**
 * A convenience method that returns the severity level when given either a severity level of type
 * number or a severity type of string.
 * If severity level is not provided or is not valid this returns a severity error.
 * @param {(string|number)=} severity
 * @return {number}
 * @memberof oj.Message
 * @ojsignature [{target: "Type",
 *                value: "oj.Message.SEVERITY_LEVEL",
 *                for: "returns"},
 *               {target: "Type",
 *                value: "oj.Message.SEVERITY_TYPE|oj.Message.SEVERITY_LEVEL",
 *                for: "severity"}]
 * @public
 * @export
 */
oj.Message.getSeverityLevel = function (severity) {
  var _severity = severity;
  if (_severity) {
    if (typeof severity === 'string') {
      var index = oj.Message._LEVEL_TO_TYPE.indexOf(_severity, 1);
      if (index === -1) {
        _severity = oj.Message.SEVERITY_LEVEL.ERROR;
      } else {
        _severity = index;
      }
    } else if (typeof _severity === 'number' &&
               (_severity < oj.Message.SEVERITY_LEVEL.CONFIRMATION ||
                _severity > oj.Message.SEVERITY_LEVEL.FATAL)) {
      _severity = oj.Message.SEVERITY_LEVEL.ERROR;
    }
  }

  return _severity || oj.Message.SEVERITY_LEVEL.ERROR;
};

/**
 * A convenience method that returns the severity type when given either a severity level of type
 * number or a severity type of string.
 * If severity level is not provided or is not valid this return a severity error.
 * @param {(string|number)=} level
 * @return {string}
 * @ojsignature [{target: "Type",
 *                value: "oj.Message.SEVERITY_TYPE",
 *                for: "returns"},
 *               {target: "Type",
 *                value: "oj.Message.SEVERITY_TYPE|oj.Message.SEVERITY_LEVEL",
 *                for: "level"}]
 * @memberof oj.Message
 * @public
 * @export
 */
oj.Message.getSeverityType = function (level) {
  var _level = level;
  if (_level) {
    if (typeof _level === 'string') {
      var index = oj.Message._LEVEL_TO_TYPE.indexOf(_level, 1);
      if (index === -1) {
        // when given an unrecognized type return "error"
        _level = oj.Message.SEVERITY_TYPE.ERROR;
      }
    } else if (typeof _level === 'number') {
      if (_level < oj.Message.SEVERITY_LEVEL.CONFIRMATION ||
          _level > oj.Message.SEVERITY_LEVEL.FATAL) {
        _level = oj.Message.SEVERITY_TYPE.ERROR;
      } else {
        _level = oj.Message._LEVEL_TO_TYPE[level];
      }
    }
  }
  return _level || oj.Message.SEVERITY_TYPE.ERROR;
};

/**
 * Returns the max severity level in a array of message objects.
 *
 * @param {Array.<oj.Message>=} messages an array of message instances or duck typed messages
 * @returns {number} -1 if none can be determined; otherwise a severity level as defined by
 * oj.Message.SEVERITY_LEVEL.
 * @export
 * @memberof oj.Message
 * @ojsignature [{target: "Type",
 *                value: "oj.Message.SEVERITY_LEVEL| -1",
 *                for: "returns"}]
 * @public
 */
oj.Message.getMaxSeverity = function (messages) {
  var maxLevel = -1;

  if (messages && messages.length > 0) {
    $.each(messages, function (i, message) {
      var currLevel = oj.Message.getSeverityLevel(message.severity);
      maxLevel = maxLevel < currLevel ? currLevel : maxLevel;
    });
  }

  return maxLevel;
};

/**
 * Returns false if messages are of severity error or greater.
 *
 * @param {Array.<oj.Message>} messages an array of message instances or duck-typed messages
 * @returns {boolean} true if none of the messages are of severity error or greater. false otherwise
 * @export
 * @memberof oj.Message
 * @public
 */
oj.Message.isValid = function (messages) {
  var maxSeverity = oj.Message.getMaxSeverity(messages);
  if (maxSeverity >= oj.Message.SEVERITY_LEVEL.ERROR) {
    return false;
  }

  return true;
};

/**
 * @private
 * @type Array
 */
oj.Message._LEVEL_TO_TYPE = [
  'none', // this can never be set
  oj.Message.SEVERITY_TYPE.CONFIRMATION,
  oj.Message.SEVERITY_TYPE.INFO,
  oj.Message.SEVERITY_TYPE.WARNING,
  oj.Message.SEVERITY_TYPE.ERROR,
  oj.Message.SEVERITY_TYPE.FATAL
];

// mapping a variable to the name used in the require callback function for this module
// it is used in a no-require environment
// eslint-disable-next-line no-unused-vars
var Message = oj.Message;

/**
 * Copyright (c) 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */

/**
 * Extends oj.Message to represent a component specific message, this defines options that control
 * how the message will display.
 *
 * @param {string} summary - Localized summary message text
 * @param {string} detail - Localized detail message text
 * @param {number|string} severity - An optional severity for this message. Use constants
 * oj.Message.SEVERITY_LEVEL for number types and oj.Message.SEVERITY_TYPE for string types. Default
 * is SEVERITY_ERROR if no severity is specified
 * @param {Object} options - an Object literal that defines the following properties
 * @property {string} display - whether the message needs to be displayed immediately or not.
 * Accepted values are 'immediate', 'deferred'. The default is 'immediate'.
 * @property {string} context - the context the component was in when the validation messages was
 * added.<p>
 *
 * NOTE: messages added to the component directly by applications are unknown context and always
 * shown {display: 'immediate', context: ''}. </p>
 * @private
 * @constructor
 * @since 0.7.0
 */
oj.ComponentMessage = function (summary, detail, severity, options) {
  this.Init(summary, detail, severity, options);
};

// Subclass from oj.Message
oj.Object.createSubclass(oj.ComponentMessage, oj.Message, 'oj.ComponentMessage');

/**
 * Determines whether the message is displayed immediately or not. Deferred messages are not shown
 * to the user right away but are deferred until component explicitly does. See
 * {@link oj.editableValue#showMessages}.
 * @private
 * @const
 * @type {Object}
 */
oj.ComponentMessage.DISPLAY = { SHOWN: 'shown', HIDDEN: 'hidden' };

/**
 * The default display options to use when none set.
 *
 * @type {Object}
 * @private
 */
oj.ComponentMessage._DEFAULT_OPTIONS = {
  display: oj.ComponentMessage.DISPLAY.SHOWN,
  context: ''
};

/**
 * Initializes the strategy based on the display options that specify the messaging artifacts that
 * will be displayed by this strategy.
 *
 * @param {string} summary - Localized summary message text
 * @param {string} detail - Localized detail message text
 * @param {number|string} severity - An optional severity for this message. Use constants
 * oj.Message.SEVERITY_LEVEL for number types and oj.Message.SEVERITY_TYPE for string types. Default
 * is SEVERITY_ERROR if no severity is specified
 * @param {Object} options - an Object literal that defines the following properties
 * @memberof! oj.ComponentMessage
 * @instance
 * @protected
 * @ignore
 */
oj.ComponentMessage.prototype.Init = function (summary, detail, severity, options) {
  oj.ComponentMessage.superclass.Init.call(this, summary, detail, severity);

  this._options = $.extend({}, oj.ComponentMessage._DEFAULT_OPTIONS, options);
};

/**
 * Clones this and returns the new instance.
 *
 * @memberof! oj.ComponentMessage
 * @instance
 * @protected
 * @ignore
 * @returns {Object}
 */
oj.ComponentMessage.prototype.clone = function () {
  return new oj.ComponentMessage(this.summary, this.detail, this.severity, this._options);
};

/**
 * Whether a message can display on the UI.
 *
 * @memberof! oj.ComponentMessage
 * @returns {boolean} true if messages can be displayed; false if marked as deferred by component.
 * @instance
 * @protected
 * @ignore
 */
oj.ComponentMessage.prototype.canDisplay = function () {
  return !(this._options && this._options.display ?
           this._options.display === oj.ComponentMessage.DISPLAY.HIDDEN : false);
};

/**
 *
 * Called by framework when the message needs to be shown
 *
 * @returns {boolean} true if shown; false if message display was already shown
 *
 * @memberof! oj.ComponentMessage
 * @instance
 * @private
 */
oj.ComponentMessage.prototype._forceDisplayToShown = function () {
  if (this._options && oj.ComponentMessage.DISPLAY.HIDDEN === this._options.display) {
    this._options.display = oj.ComponentMessage.DISPLAY.SHOWN;
    return true;
  }

  return false;
};

/**
 * Called by framework to determine if message was added by component versus app.
 *
 * @returns {boolean} true if messages was added by component. Usually the context is set.
 *
 * @memberof! oj.ComponentMessage
 * @instance
 * @private
 */
oj.ComponentMessage.prototype._isMessageAddedByComponent = function () {
  if (this._options && this._options.context) {
    return true;
  }

  return false;
};

/* global Promise:false, Set:false, Logger:false, Context:false */

/**
 * Custom element bridge prototype.
 *
 * @class
 * @ignore
 */
oj.BaseCustomElementBridge = {};

/**
 * Prototype for subclasses
 * @ignore
 */
oj.BaseCustomElementBridge.proto =
{
  getClass: function (descriptor) {
    var proto = Object.create(HTMLElement.prototype);
    this.InitializePrototype(proto);

    var metadata = this.GetMetadata(descriptor);
    // Enumerate metadata to define the prototype properties, methods, and events
    oj.BaseCustomElementBridge._enumerateMetadataForKey(proto, metadata, 'properties', this.DefinePropertyCallback.bind(this));
    oj.BaseCustomElementBridge._enumerateMetadataForKey(proto, metadata, 'methods', this.DefineMethodCallback.bind(this));

    // Add additional element methods not defined in metadata, e.g. getNodeBySubId/getSubIdByNode or get/setProperty
    this.AddComponentMethods(proto);

    proto.setProperties = function (props) {
      oj.BaseCustomElementBridge.getInstance(this)._setProperties(this, props);
    };

    // The set/unset methods are used for TypeScript only so we should define these as non enumerated properties
    Object.defineProperty(proto, 'set', { value: function (prop, value) { this.setProperty(prop, value); } });
    Object.defineProperty(proto, 'unset', { value: function (prop) { this.setProperty(prop, undefined); } });

    // Add lifecycle listeners
    proto.attributeChangedCallback = this._attributeChangedCallback;
    proto.connectedCallback = this._connectedCallback;
    proto.disconnectedCallback = this._detachedCallback;

    var constructorFunc = function () {
      var reflect = window.Reflect;
      var ret;
      if (typeof reflect !== 'undefined') {
        ret = reflect.construct(HTMLElement, [], this.constructor);
      } else {
        ret = HTMLElement.call(this);
      }
      return ret;
    };

    var bridge = this;
    Object.defineProperty(constructorFunc, 'observedAttributes',
      {
        get: function () {
          return bridge.GetAttributes(metadata);
        }
      });

    Object.defineProperty(proto, 'constructor', { value: constructorFunc, writable: true, configurable: true });
    constructorFunc.prototype = proto;
    Object.setPrototypeOf(constructorFunc, HTMLElement);

    return constructorFunc;
  },

  playbackEarlyPropertySets: function (element) {
    this._bCanSetProperty = true;
    this.PlaybackEarlyPropertySets(element);
  },

  resolveBindingProvider: function (provider) {
    if (this._bpResolve) {
      this._bpResolve(provider);
    } else {
      this._bpInst = provider;
    }
  },

  resolveDelayedReadyPromise: function () {
    this.GetDelayedReadyPromise().resolvePromise();
  },

  whenCreated: function () {
    return this._whenCreatedPromise;
  },

  // eslint-disable-next-line no-unused-vars
  AddComponentMethods: function (proto) {},

  // eslint-disable-next-line no-unused-vars
  CreateComponent: function (element) {},

  // eslint-disable-next-line no-unused-vars
  DefineMethodCallback: function (proto, method, methodMeta) {},

  // eslint-disable-next-line no-unused-vars
  DefinePropertyCallback: function (proto, property, propertyMeta) {},

  /**
   * Returns a promise that will be resolved when the component has been initialized.
   * @return {Promise}
   */
  GetDelayedReadyPromise: function () {
    if (!this._delayedReady) {
      this._delayedReady = new oj.BaseCustomElementBridge.__DelayedPromise();
    }
    return this._delayedReady;
  },

  GetAttributes: function (metadata) {
    return oj.BaseCustomElementBridge.getAttributes(metadata.properties);
  },

  GetMetadata: function (descriptor) {
    return descriptor[oj.BaseCustomElementBridge.DESC_KEY_META];
  },

  /**
   * Returns the aliased component property for a given custom element property,
   * e.g. readOnly for oj-switch's readonly custom element property.
   * Will return the original property if there is no aliasing.
   * @param {string} property The property to check
   */
  GetAliasForProperty: function (property) {
    return property;
  },

  GetDefaultValue: function (propertyMeta) {
    return oj.BaseCustomElementBridge._consolidateDefaults(propertyMeta);
  },

  GetTrackChildrenOption: function () {
    return this.METADATA.extension && this.METADATA.extension._TRACK_CHILDREN ?
      this.METADATA.extension._TRACK_CHILDREN : 'none';
  },

  // eslint-disable-next-line no-unused-vars
  HandleAttributeChanged: function (element, attr, oldValue, newValue) {},

  // eslint-disable-next-line no-unused-vars
  HandleBindingsApplied: function (element, bindingContext) {},

  // eslint-disable-next-line no-unused-vars
  HandleDetached: function (element) {},

  // eslint-disable-next-line no-unused-vars
  HandleReattached: function (element) {},

  // eslint-disable-next-line no-unused-vars
  InitializeElement: function (element) {},

  // eslint-disable-next-line no-unused-vars
  InitializePrototype: function (proto) {},

  BatchedPropertySet: function (elem, props) {
    var keys = Object.keys(props);

    for (var i = 0; i < keys.length; i++) {
      var key = keys[i];
      elem.setProperty(key, props[key]);
    }
  },

  GetEventListenerProperty: function (property) {
    var event = oj.__AttributeUtils.eventListenerPropertyToEventType(property);
    // Get event listener
    var eventListener = this._eventListeners[event];
    if (eventListener) {
      return eventListener.getListener();
    }
    return undefined;
  },

  GetProperty: function (element, prop, props) {
    var event = oj.__AttributeUtils.eventListenerPropertyToEventType(prop);
    var meta = oj.BaseCustomElementBridge
        .__GetPropertyMetadata(prop, oj.BaseCustomElementBridge.getProperties(this, element));

    // For event listener and non component properties, retrieve the value directly stored on the element.
    // For top level properties, this will delegate to our 'set' methods so we can handle default values.
    // props is the properties object we pass the definitional element or composite ViewModel
    if (event || !meta || prop.indexOf('.') === -1) {
      return props[prop];
    }
    return oj.BaseCustomElementBridge.__GetProperty(props, prop);
  },

  InitializeBridge: function (element) {
    // Initialize property storage and other variables needed for property sets.
    // Since early property sets can occur before the element's connected callback
    // is triggered we can't rely on performing this logic there. The cases where
    // the connected callback isn't called before a property set can occur if the
    // custom element is programatically created and sets are done before adding
    // the element to the DOM or if the element is stamped by knockout and its expressions
    // processed disconnected as in the case for oj-bind-for-each.
    var descriptor = oj.BaseCustomElementBridge.__GetDescriptor(element.tagName);
    this.METADATA = this.GetMetadata(descriptor);
    this._eventListeners = {};
  },

  PlaybackEarlyPropertySets: function (element) {
    if (this._earlySets) {
      while (this._earlySets.length) {
        var setObj = this._earlySets.shift();
        element.setProperty(setObj.property, setObj.value);
      }
    }
  },

  /**
   * Returns true if this property set was handled as an early property set.
   * An early property set is any set that occurs before the component is created.
   * These sets do not trigger [property]Changed events.
   * @param {string} prop
   * @param {any} value
   * @return {boolean}
   */
  SaveEarlyPropertySet: function (prop, value) {
    // Do not save sets that occur during oj.BaseCustomElementBridge.__InitProperties
    // or expression evaluation. We can process these as normal. Playback occurs before the
    // binding provider promise is resolved leading to component creation.
    if (this.__INITIALIZING_PROPS || this._bCanSetProperty) {
      return false;
    }

    if (!this._earlySets) {
      this._earlySets = [];
    }

    this._earlySets.push({ property: prop, value: value });
    return true;
  },

  SetEventListenerProperty: function (element, property, value) {
    var event = oj.__AttributeUtils.eventListenerPropertyToEventType(property);
    // Get event listener
    var eventListener = this._eventListeners[event];
    if (!eventListener) {
      // Create the wrapper
      eventListener = this._createEventListenerWrapper();
      this._eventListeners[event] = eventListener;
      element.addEventListener(event, eventListener);
    }
    if (value == null || value instanceof Function) {
      eventListener.setListener(value);
    } else {
      oj.BaseCustomElementBridge.__ThrowTypeError(element, property, value, 'function');
    }
  },

  SetProperty: function (element, prop, value, props, bOuter) {
    // Check value against any defined enums
    var event = oj.__AttributeUtils.eventListenerPropertyToEventType(prop);
    var meta = oj.BaseCustomElementBridge
        .__GetPropertyMetadata(prop, oj.BaseCustomElementBridge.getProperties(this, element));
    if (event || !meta) {
      // eslint-disable-next-line no-param-reassign
      element[prop] = value;
    } else {
      // props is the properties object we pass the definitional element or composite ViewModel
      var previousValue = element.getProperty(prop);
      var propPath = prop.split('.');
      var topProp = propPath[0];
      // If the top level property is an object, make a copy otherwise the old/new values
      // will be the same.
      var topPropPrevValue = props[topProp];
      if (oj.CollectionUtils.isPlainObject(topPropPrevValue)) {
        topPropPrevValue = oj.CollectionUtils.copyInto({}, topPropPrevValue, undefined, true);
      }

      if (!oj.BaseCustomElementBridge.__CompareOptionValues(prop, meta, value, previousValue)) {
        var isSubprop = prop.indexOf('.') !== -1;
        if (isSubprop) {
          // Set a flag for the case a subproperty results in a set of the top level property
          // which was not instantiated to an empty object to avoid firing two events.
          this._SKIP_PROP_CHANGE_EVENT = true;
        }

        if (bOuter) {
          // This ultimately triggers our element defined property setter
          this.ValidateAndSetProperty(this.GetAliasForProperty.bind(this),
                                      props, prop, value, element);
        } else {
          // Skip validation for inner sets so we don't throw an error when updating readOnly writeable properties
          oj.BaseCustomElementBridge.__SetProperty(this.GetAliasForProperty.bind(this),
                                                   props, prop, value);
        }

        this._SKIP_PROP_CHANGE_EVENT = false;
        // Property change events for top level properties will be triggered by ValidateAndSetProperty so avoid firing twice
        if (isSubprop) {
          var subprop = {};
          subprop.path = prop;
          subprop.value = value;
          subprop.previousValue = previousValue;
          // Pass the top level property value/previousValues
          var updatedFrom = bOuter ? 'external' : 'internal';
          oj.BaseCustomElementBridge.__FirePropertyChangeEvent(element, topProp, props[topProp],
                                                               topPropPrevValue, updatedFrom,
                                                               subprop);
        }
        return { property: topProp, propertySet: true, isSubproperty: isSubprop };
      }
    }
    // We return true if a component property is updated with a different value and false
    // for other cases like on[Event] property updates
    return { property: null, propertySet: false, isSubproperty: false };
  },

  ValidateAndSetProperty: function (propNameFun, componentProps, property, value, element) {
    var _value = this.ValidatePropertySet(element, property, value);
    oj.BaseCustomElementBridge.__SetProperty(propNameFun, componentProps, property, _value);
  },

  ValidatePropertySet: function (element, property, value) {
    var propsMeta = oj.BaseCustomElementBridge.getProperties(this, element);
    var propMeta = oj.BaseCustomElementBridge.__GetPropertyMetadata(property, propsMeta);
    var propAr = property.split('.');

    if (!propMeta) {
      Logger.warn(oj.BaseCustomElementBridge.getElementInfo(element) + ": Ignoring property set for undefined property '" + property + "'.");
      return undefined;
    }

    // Check readOnly property for top level property
    if (propsMeta[propAr[0]].readOnly) {
      this.throwError(element, "Read-only property '" + property + "' cannot be set.");
    }

    oj.BaseCustomElementBridge.checkEnumValues(element, property, value, propMeta);

    // TODO support checking for null values once we generate metadata from jsDoc and have accurate info
    // about component support for undefined/null
    if (value != null) {
      return oj.BaseCustomElementBridge.checkType(element, property, value, propMeta);
    }

    return value;
  },

  _attributeChangedCallback: function (attr, oldValue, newValue) {
    var bridge = oj.BaseCustomElementBridge.getInstance(this);

    if (bridge._bCreateCalled) {
      var prop = oj.__AttributeUtils.attributeToPropertyName(attr);
      var propMeta = oj.BaseCustomElementBridge
          .__GetPropertyMetadata(prop, oj.BaseCustomElementBridge.getProperties(bridge, this));

      oj.BaseCustomElementBridge.__CheckOverlappingAttribute(this, attr);

      // removeAttribute calls return null as the newValue which we want to treat as
      // a property unset and convert to undefined. We allow property null sets as an
      // actual property override.
      if (newValue === null) {
        // eslint-disable-next-line no-param-reassign
        newValue = undefined;
      }

      var params = {
        detail: { attribute: attr, value: newValue, previousValue: oldValue }
      };
      this.dispatchEvent(new CustomEvent('attribute-changed', params));

      var expression = oj.__AttributeUtils.getExpressionInfo(newValue).expr;
      if (!expression) {
        if (propMeta) {
          this.setProperty(prop, oj.BaseCustomElementBridge.__ParseAttrValue(this, attr,
                                                                             prop, newValue,
                                                                             propMeta));
        }

        // This allows subclasses to handle special cases like global transfer
        // attributes for JET components
        bridge.HandleAttributeChanged(this, attr, oldValue, newValue);
      }
    }
  },

  _connected: function (element) {
    this._bConnected = true;
    if (!this._bCreateCalled) { // initial attach
      this._bCreateCalled = true;

      this._registerBusyState(element);

      this._monitorReadyPromise(element);

      this.InitializeElement(element);

      var self = this;

      var createComponentCallback = function () {
        try {
          self.CreateComponent(element);
        } catch (ex) {
          // If an error occurs during component creation, resolve the busy context and throw an error.
          self.throwError(element, 'Error while rendering component. ' + ex);
        }
      };

      var preCreatePromise = this._getBindingProvider(element);
      var trackOption = oj.BaseCustomElementBridge.getTrackChildrenOption(element);
      if (trackOption !== 'none') {
        // this will return a promise that will get automatically chained to the binding provider promise
        preCreatePromise = preCreatePromise.then(function () {
          return self._whenChildrenCreated(element);
        });
      }

      this._whenCreatedPromise = preCreatePromise.then(createComponentCallback);
    } else if (!this._complete) {
      // If the component had been previously disconnected, and the 'ready'
      // promise is still not resolved, we need to re-register the busy state
      this._registerBusyState(element);
    } else {
      this.HandleReattached(element);
    }
  },

  _connectedCallback: function () {
    var bridge = oj.BaseCustomElementBridge.getInstance(this);
    bridge._connected(this);
  },

  _detachedCallback: function () {
    var bridge = oj.BaseCustomElementBridge.getInstance(this);
    bridge._bConnected = false;
    if (!bridge._complete) {
      bridge._resolveBusyState(this);
    } else {
      bridge.HandleDetached(this);
    }
  },

  // This wrapper does not provide any additional functionality to event listener functions
  // It exists solely to preserve event listener order
  _createEventListenerWrapper: function () {
    var eventListener;
    var domListener = function (event) {
      if (eventListener) {
        eventListener(event);
      }
    };
    domListener.setListener = function (listener) {
      eventListener = listener;
    };
    domListener.getListener = function () {
      return eventListener;
    };
    return domListener;
  },

  _monitorReadyPromise: function (element) {
    var self = this;

    var completeHandler = function () {
      // If the component is disconnected, the busy state
      // must be already resolved
      if (self._bConnected) {
        self._resolveBusyState(element);
      }
      self._complete = true;
    };

    this.GetDelayedReadyPromise().getPromise().then(
      function () {
        // Add marker class to unhide components
        element.classList.add('oj-complete');
        completeHandler();
      },
      function () {
        // Add marker class to mark that there was an error duing upgrade so consumers like
        // VBCS can apply their own styling to incorrectly setup custom elements.
        element.classList.add('oj-incomplete');
        completeHandler();
      }
    );
  },

  _registerBusyState: function (element) {
    var busyContext = Context.getContext(element).getBusyContext();
    this._initCompleteCallback = busyContext.addBusyState({ description: oj.BaseCustomElementBridge.getElementInfo(element) + ' is being upgraded.' });
  },

  _resolveBusyState: function (element) {
    var callback = this._initCompleteCallback;
    if (!callback) {
      this.throwError(element, 'Unexpected call to _resolveBusyState().');
    }

    this._initCompleteCallback = null;
    callback();
  },

  _setProperties: function (elem, props) {
    var mutationKeys = []; // keys for the 'dot mutation' properties
    var regularProps = {}; // the rest of the properties
    var hasRegularProps = false;

    var keys = Object.keys(props);
    for (var i = 0; i < keys.length; i++) {
      var key = keys[i];
      if (key.indexOf('.') >= 0) {
        mutationKeys.push(key);
      } else {
        regularProps[key] = props[key];
        hasRegularProps = true;
      }
    }

    // Regular property updates may be batched
    if (hasRegularProps) {
      this.BatchedPropertySet(elem, regularProps);
    }

    // 'Dot notation' properties can only be set individually for now
    for (var p = 0; p < mutationKeys.length; p++) {
      var mkey = mutationKeys[p];
      elem.setProperty(mkey, props[mkey]);
    }
  },

  throwError: function (elem, msg) {
    this.GetDelayedReadyPromise().rejectPromise();
    throw new Error(oj.BaseCustomElementBridge.getElementInfo(elem) + ': ' + msg);
  },

  _setBpResolver: function (resolve) {
    this._bpResolve = resolve;
  },

  _getBindingProvider: function (element) {
    var name = this._getBindingProviderName(element);

    if (name === oj.BaseCustomElementBridge._NO_BINDING_PROVIDER) {
      this.playbackEarlyPropertySets(element);

      return Promise.resolve(null);
    } else if (name === 'knockout') {
      if (this._bpInst) {
        return Promise.resolve(this._bpInst);
      }

      return new Promise(this._setBpResolver.bind(this));
    }

    this.throwError(element, "Unknown binding provider '" + name + "'.");
    return undefined; // not reachable but eslint is too stupid to figure that out.
  },

  _getBindingProviderName: function (element) {
    var cachedProp = '_ojBndgPrv';

    var name = element[cachedProp];
    if (name) {
      return name;
    }

    name = element.getAttribute('data-oj-binding-provider') ||
                oj.BaseCustomElementBridge._getCompositeBindingProviderName(element);

    if (!name) {
      var parent = element.parentElement;
      if (parent == null) {
        if (element === document.documentElement) {
          name = 'knockout'; // the default
        } else {
          this.throwError(element, 'Cannot determine binding provider for a disconnected subtree.');
        }
      } else {
        name = this._getBindingProviderName(parent);
      }
    }
    // cache provider name as a non-enumerable property
    Object.defineProperty(element, cachedProp, { value: name });

    return name;
  },

  // used by _whenChildrenCreated() call - called for trackOption "immediate" or "nearestCustomElement"
  _getChildrenToTrack: function (element, trackOption, trackedElements) {
    var children = element.childNodes;
    for (var i = 0; i < children.length; i++) {
      var child = children[i];
      if (oj.ElementUtils.isValidCustomElementName(child.localName)) {
        trackedElements.push(child);
      } else if (trackOption === 'nearestCustomElement') {
        this._getChildrenToTrack(child, trackOption, trackedElements);
      }
    }
    return trackedElements;
  },

  // tracks upgrade and creation of relevant children for the element
  _whenChildrenCreated: function (element) {
    var _UPGRADE_MESSAGE_INTERVAL = 20000;
    var trackOption = oj.BaseCustomElementBridge.getTrackChildrenOption(element);
    var busyContext = Context.getContext(element).getBusyContext();
    var trackedElements = this._getChildrenToTrack(element, trackOption, []);

    // map tracked elements to promises
    var promises = trackedElements.map(function (trackedElement) {
      // register busy state for the element
      var resolveElementDefinedBusyState = busyContext.addBusyState({ description: 'Waiting for element ' + trackedElement.localName + ' to be defined.' });

      // setup a timer to log 'waiting' message for an element
      var timer = setInterval(function () {
        Logger.warn('Warning: waiting for element ' + trackedElement.localName + ' to be defined.');
      }, _UPGRADE_MESSAGE_INTERVAL);

      // return a promise that will be resolved, when element is defined and created
      return customElements
              .whenDefined(trackedElement.localName)
              .then(function () {
                resolveElementDefinedBusyState();
                clearInterval(timer);
                if (oj.BaseCustomElementBridge.getRegistered(trackedElement.tagName)) {
                  return oj.BaseCustomElementBridge.getInstance(trackedElement).whenCreated();
                }
                return null;
              })
              .catch(function (error) {
                resolveElementDefinedBusyState();
                clearInterval(timer);
                throw new Error('Error defining element ' + trackedElement.localName + ' : ' + error);
              });
    });
    return Promise.all(promises);
  }

};

/** ***********************/
/* PUBLIC STATIC METHODS */
/** ***********************/

/**
 * Returns the attributes including the dot notation versions of all complex properties
 * not including readOnly properties.
 * @param {Object} props The properties object
 * @return {Array}
 * @ignore
 */
oj.BaseCustomElementBridge.getAttributes = function (props) {
  var attrs = [];
  oj.BaseCustomElementBridge._getAttributesFromProperties('', props, attrs);
  return attrs;
};

/**
 * Returns track children option for the element - 'none', 'immediate' or 'nearestCustomElement'
 * @param {Element} element Custom element
 * @return {String}
 * @ignore
 */
oj.BaseCustomElementBridge.getTrackChildrenOption = function (element) {
  var bridge = oj.BaseCustomElementBridge.getInstance(element);
  return bridge ? bridge.GetTrackChildrenOption() : 'none';
};

/**
 * Helper method for Returns the attributes including the dot notation versions of all complex attributes
 * stored on a bridge instance
 * @param {string} propName The property to evaluate
 * @param {Object} props The properties object
 * @param {Array} attrs The attribute array to add to
 * @ignore
 */
oj.BaseCustomElementBridge._getAttributesFromProperties = function (propName, props, attrs) {
  if (props) {
    var propKeys = Object.keys(props);
    for (var i = 0; i < propKeys.length; i++) {
      var prop = propKeys[i];
      var propMeta = props[prop];
      if (!propMeta.readOnly) {
        var concatName = propName + prop;
        attrs.push(oj.__AttributeUtils.propertyNameToAttribute(concatName));
        if (propMeta.properties) {
          oj.BaseCustomElementBridge._getAttributesFromProperties(concatName + '.', propMeta.properties, attrs);
        }
      }
    }
  }
};

/**
 * Returns a string including the element tag name and id for use in error messages and logging.
 * @param {Element} element The element to get the information for.
 * @return {string}
 * @ignore
 */
oj.BaseCustomElementBridge.getElementInfo = function (element) {
  if (element) {
    return element.tagName.toLowerCase() + " with id '" + element.id + "'";
  }
  return '';
};

/**
 * Returns the bridge instance for an element.
 * @ignore
 */
oj.BaseCustomElementBridge.getInstance = function (element) {
  var instance = element[oj.BaseCustomElementBridge._INSTANCE_KEY];
  if (!instance) {
    var info = oj.BaseCustomElementBridge._registry[element.tagName.toLowerCase()];
    if (!info) {
      Logger.error(oj.BaseCustomElementBridge.getElementInfo(element) + ' Attempt to interact with the custom element before it has been registered.');
    }

    instance = Object.create(info.bridgeProto);
    instance.InitializeBridge(element);
    Object.defineProperty(element, oj.BaseCustomElementBridge._INSTANCE_KEY, { value: instance });
  }

  return instance;
};

/**
 * Returns the properties stored on a bridge instance
 * @param  {Object} bridge The bridge instance
 * @param  {Object} element The element instance
 * @return {Object}
 * @ignore
 */
// eslint-disable-next-line no-unused-vars
oj.BaseCustomElementBridge.getProperties = function (bridge, element) {
  return bridge.METADATA.properties;
};

/**
 * Returns an object if JET component tag has been registered, null otherwise.
 * @param  {string}  tagName The tag name to look up
 * @return {Object|null} True if the component module has been loaded and registered
 * @ignore
 */
oj.BaseCustomElementBridge.getRegistered = function (tagName) {
  if (tagName) {
    var info = oj.BaseCustomElementBridge._registry[tagName.toLowerCase()];
    if (info) {
      return { composite: info.composite };
    }
  }

  return null;
};

/**
 * Returns the slot map of slot name to slotted child elements for a given custom element.
 * If the given element has no children, this method returns an empty object.
 * Note that the default slot name is mapped to the empty string.
 * @param  {Element} element The custom element
 * @return {Object} A map of the child elements for a given custom element.
 * @ignore
 */
oj.BaseCustomElementBridge.getSlotMap = function (element) {
  var slotMap = {};
  var childNodeList = element.childNodes;
  for (var i = 0; i < childNodeList.length; i++) {
    var child = childNodeList[i];
    // Only assign Text and Element nodes to a slot
    if (oj.BaseCustomElementBridge.isSlotable(child)) {
      var slot = oj.BaseCustomElementBridge.getSlotAssignment(child);
      if (!slotMap[slot]) {
        slotMap[slot] = [];
      }
      slotMap[slot].push(child);
    }
  }
  return slotMap;
};

/**
 * Returns the slot that the node should get assigned to.
 * Note that the default slot name is mapped to the empty string.
 * @param  {Node} node The custom element
 * @return {string} The slot name of the element
 * @ignore
 */
oj.BaseCustomElementBridge.getSlotAssignment = function (node) {
  // Text nodes and elements with no slot attribute map to the default slot.
  // __oj_slots is the slot attribute saved from an oj-bind-slot or oj-bind-template-slot element
  // Remember that the slot name can be the empty string so we should do a null check instead of just using || directly
  var slot = node.__oj_slots != null ? node.__oj_slots : (node.getAttribute && node.getAttribute('slot'));
  if (!slot) {
    slot = '';
  }
  return slot;
};

/**
 * Returns true if an element is slot assignable.
 * @param {Element} node The element to check
 * @return {boolean}
 * @ignore
 */
oj.BaseCustomElementBridge.isSlotable = function (node) {
  // Ignore text nodes that only contain whitespace
  return node.nodeType === 1 || (node.nodeType === 3 && node.nodeValue.trim());
};

/** ***************************/
/* NON PUBLIC STATIC METHODS */
/** ***************************/

/**
 * @ignore
 */
oj.BaseCustomElementBridge._NO_BINDING_PROVIDER = 'none';

/**
 * @ignore
 */
oj.BaseCustomElementBridge._enumerateMetadataForKey = function (proto, metadata, key, callback) {
  if (!metadata || !metadata[key]) {
    return;
  }

  var values = metadata[key];
  var names = Object.keys(values);
  names.forEach(
    function (name) {
      callback(proto, name, values[name]);
    }
  );
};


/**
 * Returns a binding provder name if the element is managed by a JET composite
 * @ignore
 */
oj.BaseCustomElementBridge._getCompositeBindingProviderName = function (element) {
  // for upstream dependency we will still rely components being registered on the oj namespace.
  var name = oj.Composite ? oj.Composite.getBindingProviderName(element.parentElement) : null;
  return name;
};


/**
 * Verify metadata for required properties
 * @ignore
 */
oj.BaseCustomElementBridge._verifyMetadata = function (tagName, metadata) {
  if (metadata) {
    // Verify that declared properties don't override any global HTML element properties
    var properties = metadata.properties;
    if (properties) {
      // We are not currently checking for redefined aria-*, data-*, or event handler attributes, e.g. onclick.
      var globals = oj.BaseCustomElementBridge._GLOBAL_PROPERTIES;
      for (var i = 0; i < globals.length; i++) {
        if (properties[globals[i]]) {
          Logger.error("Error registering composite %s. Redefined global HTML element attribute '%s' in metadata.", tagName, globals[i]);
        }
      }
    }
  }
};

/**
 * Checks to see whether a value is valid for an element property's enum and throws an error if not.
 * @param  {Element}  element The custom element
 * @param  {string}  property The property to check
 * @param  {string}  value The property value
 * @param  {Object}  metadata The property metadata
 * @ignore
 */
oj.BaseCustomElementBridge.checkEnumValues = function (element, property, value, metadata) {
  // Only check enum values for string types
  if (typeof value === 'string' && metadata) {
    var enums = metadata.enumValues;
    if (enums && enums.indexOf(value) === -1) {
      var bridge = oj.BaseCustomElementBridge.getInstance(element);
      bridge.throwError(element, "Invalid value '" + value + "' found for property '" + property +
        "'. Expected one of the following '" + enums.toString() + "'.");
    }
  }
};


/**
 * Checks to see whether a value is valid for an element property and throws an error if not.
 * @param  {Element}  element The custom element
 * @param  {string}  property The property to check
 * @param  {string}  value The property value
 * @param  {Object}  metadata The property metadata
 * @ignore
 */
oj.BaseCustomElementBridge.checkType = function (element, property, value, metadata) {
  // We currently support checking of single typed properties of type: string, number,
  // boolean, Array, Object OR properties w/ two possible types where
  // the value can either be of type string|Array, string|Object, string|function, Array|Promise.
  // Any other types are currently skipped, but can be validated by the component in a future ER.
  var type = metadata.type;
  if (type) {
    type = type.toLowerCase();

    var typeAr = type.split('|');
    var typeOf = typeof value;
    if (typeAr.length === 1) {
      if ((type.substring(0, 5) === 'array' && !Array.isArray(value)) ||
          (type.substring(0, 6) === 'object' && typeOf !== 'object') ||
          (type === 'number' && !(typeof value === 'number' && isFinite(value))) || // Number.isFinite isn't availabe on IE11
          (type === 'string' && typeOf !== 'string')) {
        oj.BaseCustomElementBridge.__ThrowTypeError(element, property, value, type);
      }

      // Treat boolean property sets like the DOM does where any value that passes
      // 'if (boolVal)' results in a true prop set
      if (type === 'boolean') {
        // eslint-disable-next-line no-param-reassign
        value = !!value;
      }
    } else if (typeAr.length === 2) {
      var strIdx = typeAr.indexOf('string');
      var promiseIdx = typeAr.indexOf('promise');
      var otherType;
      if (strIdx !== -1 && typeOf !== 'string') {
        otherType = strIdx === 0 ? typeAr[1] : typeAr[0];
        if ((otherType === 'function' && typeOf !== 'function') ||
            (otherType.substring(0, 5) === 'array' && !Array.isArray(value)) ||
            (otherType.substring(0, 6) === 'object' && typeOf !== 'object')) {
          oj.BaseCustomElementBridge.__ThrowTypeError(element, property, value, type);
        }
      } else if (promiseIdx !== -1 && !(value instanceof Promise)) {
        otherType = promiseIdx === 0 ? typeAr[1] : typeAr[0];
        if ((otherType.substring(0, 5) === 'array' && !Array.isArray(value))) {
          oj.BaseCustomElementBridge.__ThrowTypeError(element, property, value, type);
        }
      }
    }
  }
  return value;
};

/**
 * Compares two values, returning true if they are equal. Does a deeper check for writeback values
 * because we can't prevent knockout from triggering a second property set with the same values
 * when writing back, but we do want to prevent the addtional update and property changed event.
 * @ignore
 */
oj.BaseCustomElementBridge.__CompareOptionValues = function (property, metadata, value1, value2) {
  if (metadata.writeback) {
    return oj.Object.compareValues(value1, value2);
  }
  return value1 === value2;
};

/**
 * @ignore
 */
oj.BaseCustomElementBridge.__ThrowTypeError = function (element, property, value, type) {
  var bridge = oj.BaseCustomElementBridge.getInstance(element);
  bridge.throwError(element, "Invalid type '" + (typeof value) + "' found for property '" +
    property + "'. Expected value of type '" + type + "'.");
};

/**
 * @ignore
 */
oj.BaseCustomElementBridge.__GetDescriptor = function (tagName) {
  return oj.BaseCustomElementBridge._registry[tagName.toLowerCase()].descriptor;
};

/**
 * @ignore
 */
oj.BaseCustomElementBridge.__GetCache = function (tagName) {
  if (tagName) {
    return oj.BaseCustomElementBridge._registry[tagName.toLowerCase()].cache;
  }
  return null;
};

/**
 * Checks to see if there are any overlapping attribute for the given element and attribute
 * @ignore
 */
oj.BaseCustomElementBridge.__CheckOverlappingAttribute = function (element, attr) {
  var attrPath = attr.split('.');
  if (attrPath.length > 1) {
    attrPath.pop();
    while (attrPath.length) {
      var attrSubPath = attrPath.join('.');
      if (element.hasAttribute(attrSubPath)) {
        var bridge = oj.BaseCustomElementBridge.getInstance(element);
        bridge.throwError(element, "Cannot set overlapping attributes '" + attr + "' and '" + attrSubPath + "'.");
      }
      attrPath.pop();
    }
  }
};

/**
  * Returns the metadata for the property, walking down the metadata hierarchy
  * for subproperties.
  * @param {string} prop The property including dot notation if applicable
  * @param {Object} metadata The component metadata
  * @return {Object|null} The metadata for the property or subproperty or
  *                       null if not a component property, e.g. a global attribute
  * @ignore
  */
oj.BaseCustomElementBridge.__GetPropertyMetadata = function (prop, metadata) {
  var meta = metadata;
  var propAr = prop.split('.');
  for (var i = 0; i < propAr.length; i++) {
    meta = meta[propAr[i]];
    if (!meta) {
      break;
    }

    if (propAr.length > 1 && i < propAr.length - 1) {
      meta = meta.properties;
      if (!meta) {
        break;
      }
    }
  }
  return meta;
};

/**
 * @ignore
 */
oj.BaseCustomElementBridge.__InitProperties = function (element, componentProps) {
  var bridge = oj.BaseCustomElementBridge.getInstance(element);
  bridge.__INITIALIZING_PROPS = true;
  var metaProps = oj.BaseCustomElementBridge.getProperties(bridge, element);
  if (metaProps) {
    var attrs = element.attributes; // attrs is a NodeList
    for (var i = 0; i < attrs.length; i++) {
      var attr = attrs[i];
      var property = oj.__AttributeUtils.attributeToPropertyName(attr.nodeName);

      // See if attribute is a component property
      var meta = oj.BaseCustomElementBridge.__GetPropertyMetadata(property, metaProps);
      if (meta && !meta.readOnly) {
        // If complex property, check if there are any overlapping attributes
        oj.BaseCustomElementBridge.__CheckOverlappingAttribute(element, attr.nodeName);

        var info = oj.__AttributeUtils.getExpressionInfo(attr.value);
        if (!info.expr) {
          var value = oj.BaseCustomElementBridge.__ParseAttrValue(element,
                                                                  attr.nodeName,
                                                                  property,
                                                                  attr.value,
                                                                  meta);
          bridge.ValidateAndSetProperty(bridge.GetAliasForProperty.bind(bridge),
                                        componentProps, property, value, element);
        }
      }
    }
  }
  bridge.__INITIALIZING_PROPS = false;
};

/**
 * @param {function} propNameFun A function that returns the actual property name to use, e.g. an alias
 * @param {Object} componentProps The object to set the new property value on which is the
 *                                element for outer property sets and the property bag for inner sets.
 * @param {string} property The property name
 * @param {Object} value The value to set for the property
 * @ignore
 */
oj.BaseCustomElementBridge.__SetProperty = function (propNameFun, componentProps, property, value) {
  var propsObj = componentProps;
  var propPath = property.split('.');
  var branchedProps;
  // Set subproperty, initializing parent objects along the way unless the top level
  // property is not defined since setting it to an empty object will trigger a property changed
  // event. Instead, branch and set at the end. We only have listeners on top level properties
  // so setting a subproperty will not trigger a property changed event along the way.
  var topProp = propNameFun(propPath[0]);
  if (propPath.length > 1 && !componentProps[topProp]) {
    branchedProps = {};
    propsObj = branchedProps;
  }

  // Walk to the correct location
  for (var i = 0; i < propPath.length; i++) {
    var subprop = propNameFun(propPath[i]);
    if (i === propPath.length - 1) {
      propsObj[subprop] = value;
    } else if (!propsObj[subprop]) {
      propsObj[subprop] = {};
    }
    propsObj = propsObj[subprop];
  }

  // Update the original component properties if we branched
  if (branchedProps) {
    // eslint-disable-next-line no-param-reassign
    componentProps[topProp] = branchedProps[topProp];
  }
};

/**
 * @ignore
 */
oj.BaseCustomElementBridge.__GetProperty = function (componentProps, property) {
  var propsObj = componentProps;
  var propPath = property.split('.');
  for (var i = 0; i < propPath.length; i++) {
    var subprop = propPath[i];
    // If no metadata is passed in, assume that the value has already been evaluated
    if (i === propPath.length - 1) {
      return propsObj[subprop];
    } else if (!propsObj[subprop]) {
      return undefined;
    }
    propsObj = propsObj[subprop];
  }
  return undefined;
};

/**
 * Returns the coerced attribute value using a custom parse function or the framework default.
 * @ignore
 */
oj.BaseCustomElementBridge.__ParseAttrValue = function (elem, attr, prop, val, metadata) {
  if (val == null) {
    return val;
  }

  var type = metadata.type;
  var bridge = oj.BaseCustomElementBridge.getInstance(elem);
  function _coerceVal(value) {
    var coercedValue;
    try {
      coercedValue = oj.__AttributeUtils.coerceValue(elem, attr, value, type);
    } catch (ex) {
      bridge.throwError(elem, ex);
    }
    return coercedValue;
  }

  var parseFunction = oj.BaseCustomElementBridge.__GetDescriptor(elem.tagName).parseFunction;
  if (parseFunction) {
    return parseFunction(val, prop, metadata,
      function (value) {
        return _coerceVal(value);
      }
    );
  }
  return _coerceVal(val);
};

/**
 * @ignore
 */
oj.BaseCustomElementBridge.__ProcessEventListeners = function (_metadata) {
  var metadata = oj.CollectionUtils.copyInto({}, _metadata, undefined, true, 1);
  metadata.properties = metadata.properties || {};
  oj.BaseCustomElementBridge._enumerateMetadataForKey(null, metadata, 'properties',
    function (proto, property) {
      var eventName = oj.__AttributeUtils.propertyNameToChangeEventType(property);
      var eventListenerProperty = oj.__AttributeUtils.eventTypeToEventListenerProperty(eventName);
      metadata.properties[eventListenerProperty] = { _derived: true, _eventListener: true };
    }
  );
  oj.BaseCustomElementBridge._enumerateMetadataForKey(null, metadata, 'events',
    function (proto, event) {
      var eventListenerProperty = oj.__AttributeUtils.eventTypeToEventListenerProperty(event);
      metadata.properties[eventListenerProperty] = { _derived: true, _eventListener: true };
    }
  );
  return metadata;
};

/**
 * @ignore
 * @param {string} tagName
 * @param {Object} descriptor
 * @param {Object} bridgeProto
 * @param {boolean=} isComposite
 */
oj.BaseCustomElementBridge.__Register = function (tagName, descriptor, bridgeProto, isComposite) {
  var name = tagName.toLowerCase();
  if (!oj.BaseCustomElementBridge._registry[name]) {
    if (!descriptor) {
      Logger.error('Cannot register ' + tagName + '. Missing a descriptor.');
    }

    oj.BaseCustomElementBridge
      ._verifyMetadata(tagName, descriptor[oj.BaseCustomElementBridge.DESC_KEY_META]);
    oj.BaseCustomElementBridge._registry[name] =
      { descriptor: descriptor, bridgeProto: bridgeProto, composite: isComposite, cache: {} };
    return true;
  }
  return false;
};

/**
 * @ignore
 * @param {Element} element
 * @param {string} name
 * @param {Object} value
 * @param {Object} previousValue
 * @param {string} updatedFrom
 * @param {Object=} subprop
 */
oj.BaseCustomElementBridge.__FirePropertyChangeEvent =
  function (element, name, value, previousValue, updatedFrom, subprop) {
    var bridge = oj.BaseCustomElementBridge.getInstance(element);
    // There are cases where a subproperty set can trigger a top level property set
    // if the top level property was not instantiated to an empty object. We don't want
    // to fire two events for that case. The BaseCustomElementBridge has logic to fire
    // the subproperty change event there.
    if (!bridge._SKIP_PROP_CHANGE_EVENT) {
      var detail = {};
      if (subprop) {
        detail.subproperty = subprop;
      }
      detail.value = value;
      detail.previousValue = previousValue;
      detail.updatedFrom = updatedFrom;

      // Check if the subclass needs to do anything before we fire the property change event,
      // e.g. composites that need to call the propertyChanged ViewModel callback.
      if (bridge.beforePropertyChangedEvent) {
        bridge.beforePropertyChangedEvent(element, name, detail);
      }
      // The bridge sets the ready to fire flag after the component has been instantiated.
      // We shouldn't fire property changed events before then unless the update comes from internally
      // for cases like readOnly property updates.
      if (updatedFrom !== 'external' || bridge.__READY_TO_FIRE) {
        element.dispatchEvent(new CustomEvent(name + 'Changed', { detail: detail }));
      }
    }
  };

/**
 * @ignore
 */
oj.BaseCustomElementBridge.__DefineDynamicObjectProperty =
  function (obj, property, getter, setter) {
    Object.defineProperty(obj, property, {
      enumerable: true,
      get: getter,
      set: setter
    });
  };

var _UNIQUE_INCR = 0;
var _UNIQUE = '_ojcustomelem';

/**
 * @ignore
 */
oj.BaseCustomElementBridge.__GetUnique = function () {
  var ret = _UNIQUE + _UNIQUE_INCR;
  _UNIQUE_INCR += 1;
  return ret;
};

/**
 * Default values can be specified at the top level or at leaf subproperties.
 * This utility walks complex property subproperties to generate default value
 * @property {object} metadata
 * @ignore
 */
oj.BaseCustomElementBridge._consolidateDefaults = function (metadata) {
  var defaultValue = metadata.value;
  if (defaultValue !== undefined) {
    // Make a copy if the default value is an Object or Array to prevent modification
    // of the metadata copy and store in the propertyTracker so we have a copy
    // to modify in place for the object case
    if (Array.isArray(defaultValue)) {
      return defaultValue.slice();
    } else if (defaultValue !== null && typeof defaultValue === 'object') {
      return oj.CollectionUtils.copyInto({}, defaultValue, undefined, true);
    }
    return defaultValue;
  }
  // If top level metadata isn't specified, check subproperties.
  // Note that we are not handling cases where both top level and subproperty
  // default values are provided, leaving that to auditing and build tools to check.
  var subMeta = metadata.properties;
  if (subMeta) {
    var complexDefault = {};
    var keys = Object.keys(subMeta);
    for (var i = 0; i < keys.length; i++) {
      var subpropDefault = oj.BaseCustomElementBridge._consolidateDefaults(subMeta[keys[i]]);
      if (subpropDefault !== undefined) {
        complexDefault[keys[i]] = subpropDefault;
      }
    }
    return (Object.keys(complexDefault).length > 0) ? complexDefault : undefined;
  }
  return undefined;
};

/**
 * @ignore
 */
oj.BaseCustomElementBridge._registry = {};

/** @ignore */
oj.BaseCustomElementBridge.DESC_KEY_CSS = 'css';
/** @ignore */
oj.BaseCustomElementBridge.DESC_KEY_META = 'metadata';
/** @ignore */
oj.BaseCustomElementBridge.DESC_KEY_PARSE_FUN = 'parseFunction';
/** @ignore */
oj.BaseCustomElementBridge.DESC_KEY_VIEW = 'view';
/** @ignore */
oj.BaseCustomElementBridge.DESC_KEY_VIEW_MODEL = 'viewModel';
/** @ignore */
oj.BaseCustomElementBridge._GLOBAL_PROPERTIES = ['accesskey', 'autocapitalize', 'class',
  'contenteditable', 'contextmenu', 'dir', 'draggable', 'dropzone', 'hidden', 'id', 'is',
  'itemid', 'itemprop', 'itemref', 'itemscope', 'itemtype', 'lang', 'slot', 'spellcheck',
  'style', 'tabindex', 'title', 'translate'];

/**
 * @ignore
 */
oj.BaseCustomElementBridge._INSTANCE_KEY = '_ojBridge';

/**
 * Class used to track the create state.
 * @constructor
 * @protected
 * @ignore
 */
oj.BaseCustomElementBridge.__DelayedPromise = function () {
  var _promise;
  var _resolve;
  var _reject;

  /**
   * Returns the create Promise, creating one as needed.
   * @return {Promise}
   * @ignore
   */
  this.getPromise = function () {
    if (!_promise) {
      _promise = new Promise(function (resolve, reject) {
        _resolve = resolve;
        _reject = reject;
      });
    }
    return _promise;
  };

  /**
   * Rejects the create Promise if one exists.
   * @ignore
   */
  this.rejectPromise = function (reason) {
    if (_reject) {
      _reject(reason);
    }
  };

  /**
   * Resolves the create Promise if one exists.
   * @ignore
   */
  this.resolvePromise = function (value) {
    if (_resolve) {
      _resolve(value);
    }
  };
};

/**
 * @ojoverviewdoc CustomElementOverview - [2]JET Web Components
 * @classdesc
 * {@ojinclude "name":"customElementOverviewDoc"}
 */

/**
 * <h2 id="ce-overview-section" class="subsection-title">
 *   Overview<a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-overview-section"></a>
 * </h2>
 * <p>
 *   JET components and <a href="CompositeOverview.html">custom components</a>, collectively referred to as <b>JET Web Components</b>,
 *   are implemented as <a href="https://developer.mozilla.org/en-US/docs/Web/Web_Components/Custom_Elements">custom HTML elements</a>
 *   and extend the HTMLElement interface. This means that JET custom elements automatically inherit
 *   <a href="https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes">global attributes</a>
 *   and programmatic access to these components is similar to interacting with native HTML elements.
 *   All JET components live in the "oj" namespace and have HTML element names starting with "oj-". We will use
 *   the term "JET component" to refer to both native JET custom elements and custom elements implemented using the
 *   <a href="oj.Composite.html">Composite</a> component APIs after this point.
 * </p>
 * <h2 id="ce-overview-upgrade-section" class="subsection-title">
 *   Upgrading a Custom Element<a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-overview-upgrade-section"></a>
 * </h2>
 * <p>
 *   The upgrade process will begin for current JET custom elements in the DOM when the component module is loaded,
 *   registering a class constructor with its tag name using the
 *   <a href="https://developer.mozilla.org/en-US/docs/Web/API/CustomElementRegistry/define">CustomElementRegistry</a>
 *   <code>define()</code> API. Existing elements matching the registered tag name will be updated to inherit the new
 *   class definition and all of the component properties and methods will be available on the custom element after
 *   this process completes. Additionally, JET components will resolve any data bindings during the upgrade process.
 *   The application is responsible for calling their binding provider to apply bindings or for adding a
 *   <code>data-oj-binding-provider="none"</code> attribute in their page to indicate that no data bindings exist.
 *   Note that the JET custom element upgrade process will not complete until data bindings
 *   are resolved or no binding provider is indicated using the <code>data-oj-binding-provider</code> attribute.
 *   Also, due to JET components' data binding support, all JET component upgrades will occur asynchronously regardless
 *   of whether a binding provider is used.
 *   Please see the <a href="#ce-databind-section">data binding</a> section for more details on binding providers and data binding.
 *   The application should not interact with the JET custom element except to  programmatically set properties until the
 *   custom element upgrade is complete. The recommended way to wait on the asynchronous upgrade process is to use an
 *   element-scoped or page-level <a href="oj.BusyContext.html">BusyContext</a>.
 * </p>
 * <h2 id="ce-overview-usage-section" class="subsection-title">
 *   Using a JET Custom Element<a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-overview-usage-section"></a>
 * </h2>
 * <p>
 *   Custom elements can be used declaratively in HTML by using the component tag name and attributes. They are not self closing
 *   elements and applications should include a closing tag. To interact with them  programmatically, DOM APIs can be used to
 *   retrieve the element and then access properties and methods directly on the element instance. JET custom elements can also fire
 *   <code>CustomEvents</code> for which the application can attach event listeners both declaratively and  programmatically.
 *   The rest of this document discusses these features in more detail.
 * </p>
 *
 * <h2 id="ce-attributes-section" class="subsection-title">
 *   Attributes<a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-attributes-section"></a>
 * </h2>
 * <p>
 *   Attribute values set as string literals will be parsed and coerced to the property type. JET currently
 *   only supports the following string literal type coercions: boolean, number, string, Object and Array, where
 *   Object and Array types must use JSON notation with double quoted strings. A special "any" type is also supported
 *   and is described <a href="#ce-attrs-any-section">below</a>. All other types should to be set using
 *   <a href="#ce-databind-section">expression syntax</a> in the DOM or using the element's property setters or <code>setProperty</code>
 *   and <code>setProperties</code> methods programmatically. Unless updates are done via the DOM element.setAttribute(),
 *   the DOM's attribute value will not reflect changes like those done via the property setters or the setProperty and
 *   setProperties methods. Attribute removals are treated as unsetting of a property where the component default value will be used if one exists.
 * </p>
 * <p>
 *   As described <a href="#ce-databind-section">below</a>, JET uses [[...]] and {{...}} syntax to represent data bound expressions.
 *   JET does not currently provide any escaping syntax for "[[" or "{{" appearing at the beginning of the attribute
 *   value. You will need to add a space character to avoid having the string literal value interpreted as a binding
 *   expression (e.g. &lt;oj-some-element some-attribute='[ ["arrayValue1", "arrayValue2", ... ] ]'>&lt;/oj-some-element>).
 * </p>
 * <h2 id="ce-attrs-boolean-section" class="subsection-title">
 *   Boolean Attributes<a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-attrs-boolean-section"></a>
 * </h2>
 * <p>
 *   JET components treat boolean attributes differently than HTML5. Since a common application use case
 *   is to toggle a data bound boolean attribute, JET will coerce the string literal "false" to the boolean
 *   false for boolean attributes. The absence of a boolean attribute in the DOM will also be interpreted as false.
 *   JET will coerce the following string literal values to the boolean true and throw an Error for all other
 *   invalid values.
 *   <ul>
 *    <li>No value assignment (e.g. &lt;oj-some-element boolean-attribute>&lt;/oj-some-element>)</li>
 *    <li>Empty string (e.g. &lt;oj-some-element boolean-attribute="">&lt;/oj-some-element>)</li>
 *    <li>The "true" string literal (e.g. &lt;oj-some-element boolean-attribute="true">&lt;/oj-some-element>)</li>
 *    <li>The case-insensitive attribute name (e.g. &lt;oj-some-element boolean-attribute="boolean-attribute">&lt;/oj-some-element>)</li>
 *   </ul>
 * </p>
 * <h2 id="ce-attrs-object-section" class="subsection-title">
 *   Object-Typed Attributes<a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-attrs-object-section"></a>
 * </h2>
 * <p>
 *   Attributes that support Object type can be declaratively set using dot notation.
 *   Note that applications should not set overlapping attributes as these will cause an error to be thrown.
 *   <pre class="prettyprint">
 *   <code>
 * &lt;!-- person is an Object typed attribute with a firstName subproperty -->
 * &lt;oj-some-element person.first-name="{{name}}">&lt;/oj-some-element>
 *
 * &lt;!-- overlapping attributes will throw an error -->
 * &lt;oj-some-element person="{{personInfo}}" person.first-name="{{name}}">&lt;/oj-some-element>
 *   </code>
 *   </pre>
 *   If applications need to programmatically set subproperties, they can call the JET components's <code>setProperty</code>
 *   method with dot notation using the camelCased property syntax (e.g. element.setProperty("person.firstName", "Sally")).
 * </p>
 * <h2 id="ce-attrs-any-section" class="subsection-title">
 *   Any-Typed Attributes<a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-attrs-any-section"></a>
 * </h2>
 * <p>
 *   Attributes that support any type are documented with type {any} in the API doc and will be coerced as
 *   Objects, Arrays, or strings when set in HTML as a string literal. Numeric types are not supported due
 *   to the fact that we cannot determine whether value="2" on a property supporting any type should be
 *   coerced to a string or a number. The application should use data binding for all other value types and ensure
 *   that when linking any-typed attributes across multiple components, that the resolved types will match, e.g.
 *   do not data bind an <oj-select-one> <code>value</code> attribute to a numeric value and use a string literal
 *   number for its child <oj-option> <code>value</code> attributes since those would evaluate to strings.
 * </p>
 *
 * <h2 id="ce-databind-section" class="subsection-title">
 *   Data Binding<a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-databind-section"></a>
 * </h2>
 * <p>
 *   Applications can use the JET data binding syntax in order to use expressions or a non coercible attribute
 *   type (e.g. a type other than boolean, number, string, Object or Array) declaratively in HTML.
 *   This syntax can be used on both JET custom elements and native HTML elements.
 *   The application is responsible for applying bindings using a supported binding provider which then notifies
 *   JET framework code that the bindings have been resolved and to finish the custom element upgrade process.
 * </p>
 * <h2 id="ce-databind-bindingprovider-section" class="subsection-title">
 *   Binding Providers<a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-databind-bindingprovider-section"></a>
 * </h2>
 * <p>
 *   The binding provider is responsible for setting and updating attribute
 *   expressions and any custom elements within its managed subtree will not finish upgrading until it
 *   applies bindings on that subtree. By default, there is a single binding provider for a page,
 *   but subtree specific binding providers can be added by using the <code>data-oj-binding-provider</code>
 *   attribute with values of "none" and "knockout". The default binding provider is knockout, but if a
 *   page or DOM subtree does not use any expression syntax or knockout, the application can set
 *   <code>data-oj-binding-provider="none"</code> on that element so its dependent JET custom elements
 *   do not need to wait for bindings to be applied to finish upgrading. Note that regardless of whether a
 *   binding provider is used, the custom element upgrade process will be asynchronous. <b>When using the
 *   knockout binding provider, applications should require the ojknockout module.</b>
 * </p>
 * <h2 id="ce-databind-syntax-section" class="subsection-title">
 *   Data Binding Syntax for JET Components<a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-databind-syntax-section"></a>
 * </h2>
 * <p>
 *   Data binding syntax can be used directly on component attributes. See the specific component API doc
 *   for the complete list of component attributes. Global HTML attributes inherited from HTMLElement can also be
 *   data bound, but require special syntax described below. JET detects data bound attributes by looking for values
 *   wrapped with {{...}} or [[...]]. Please note that there should be no spaces between the braces
 *   when using the data bind syntax (e.g. some-attribute="[ [...] ]"). The {{...}} wrapped expression indicates that the
 *   application is allowing the component to update the expression which can be a knockout observable. Attributes
 *   bound using [[...]] will not be updated or "written back" to by the component. Unless the component attribute documents
 *   that it supports "writeback", we recommend that the [[...]] syntax be used, e.g. selection-mode="[[currentSelectionMode]]".
 * </p>
 * <h2 id="ce-databind-writeback-section" class="subsection-title">
 *   Writeback<a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-databind-writeback-section"></a>
 * </h2>
 * <p>
 *   Certain properties such as "value" on editable components support updating the associated expression
 *   automatically whenever their value changes. This usually occurs after user interaction such as with selection
 *   or typing into an input field. This expression update functionality is also known as "writeback".
 *   Applications can control expression writeback by using the {{...}} syntax for two-way writable
 *   binding expressions or [[...]] for one-way only expressions. The one-way expressions should
 *   be used when the application needs expressions strictly for "downstream-only" purposes, e.g. only for
 *   updating a component property. Note that if a writeback attribute is bound using the "downstream-only"
 *   syntax, the application and component states can become out of sync. This is different from the
 *   read-only properties, which are "upstream-only", e.g. they are used only to monitor component state.
 *   Thus an expression associated with a read-only property should always use the {{}} syntax.
 *   Most component properties do not writeback and those that do will indicate it in their API doc.
 * </p>
 * <pre class="prettyprint">
 *   <code>
 *   &lt;oj-some-element value="[[currentValue]]" selection={{currentSelection}}>&lt;/oj-some-element>
 *   </code>
 * </pre>
 * <h2 id="ce-databind-global-section" class="subsection-title">
 *   Data Binding Syntax for Native HTML Elements and Global Attributes
 *   <a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-databind-global-section"></a>
 * </h2>
 * <p>
 *   JET's data binding syntax can also be used on native HTML elements and
 *   <a href="https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes">global attributes</a>
 *   to create one-way bindings by prefixing the attribute
 *   name with ":" (e.g. :id="[[idVar]]"). The attribute binding syntax results in the attribute being set in the DOM
 *   with the evaluated expression. Since global HTML attributes are always string typed, expressions using the ":"
 *   prefixing should resolve to strings with the exception of the style and class attributes which support additional types
 *   and are described in more detail below. In the case of component attributes, applications are recommended to bind
 *   the attribute names directly and avoid the use of the ":" prefix.
 * </p>
 * <h2 id="ce-databind-class-section" class="subsection-title">
 *   :Class Attribute<a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-databind-class-section"></a>
 * </h2>
 * <p>
 *   The class attribute binding supports a space delimited string of classes, an Array of classes, or an Object whose keys are
 *   individual style classes and whose values are booleans to determine whether those style classes should be present in the DOM
 *   (e.g. :class="[[{errorClass: hasErrors}]]"). Note that the Array and string types will override existing values in the class
 *   attribute when updates occur, whereas the Object type will only add and remove the classes specified. Since JET custom elements
 *   add their own classes, we recommend using the Object type when using the class attribute binding on JET custom elements.
 * </p>
 * <h2 id="ce-databind-style-section" class="subsection-title">
 *   :Style Attribute<a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-databind-style-section"></a>
 * </h2>
 * <p>
 *   When using the style attribute binding with an Object type, the style Object names should be the JavaScript
 *   names for that style (e.g. "fontWeight" instead of "font-weight" style='{"fontWeight": "..."}').
 *   Since the style attribute supports Object types, it also supports dot notation for setting style subproperties
 *   directly (e.g. :style.font-weight="[[...]]").
 * </p>
 *
 * <h2 id="ce-properties-section" class="subsection-title">
 *   Properties<a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-properties-section"></a>
 * </h2>
 * <p>
 *   In addition to properties inherited from the HTMLElement prototype, attributes listed
 *   in the component API doc will also be exposed as properties on the JET custom element.
 *   See the <a href="#ce-proptoattr-section">property to attribute mapping</a>
 *   section below to see the syntax difference between setting attributes and properties.
 *   These properties can be set at any time, but can only be retrieved once the HTML element
 *   is fully upgraded. Early property sets before the component has been upgraded
 *   will not result in [property]Changed events and will be passed to the component as part of its initial state.
 * </p>
 * <h2 id="ce-properties-readonlywriteback-section" class="subsection-title">
 *   Read-only and Writeback Properties
 *   <a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-properties-readonlywriteback-section"></a>
 * </h2>
 * <p>
 *   Some properties are specially marked as read-only or supporting writeback in the component API doc.
 *   Read-only properties can only be read and not set by the application and generally support writeback.
 *   Writeback properties support automatic updates if they are bound using two way data binding syntax to an
 *   expression, e.g. value="{{valueObservable}}".
 *   Applications can bind an expression to a read-only attribute in HTML by using the {{..}} to ensure
 *   that updates will be reflected in the observable, but should not use this syntax to try and push a
 *   value to the read-only attribute which will result in an error state.
 *   Similarly, property sets using the setProperty, setProperties, or the element property setters should also
 *   be avoided for a read-only property.
 * </p>
 * <h2 id="ce-properties-subproperties-section" class="subsection-title">Subproperties
 *   <a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-properties-subproperties-section"></a>
 * </h2>
 * <p>
 *   Some JET components support complex properties where the top level property is of type Object and it
 *   contains additional subproperties. If the application needs to set a single subproperty
 *   instead of the entire complex property, the <code>setProperty</code> method should be used
 *   instead to ensure that [property]Changed events will be fired with the subproperty changes.
 *   Note that directly updating the subproperty via dot notation (e.g. element.topProp.subProp = newValue)
 *   will not result in a [property]Changed event being fired.
 * </p>
 * <h2 id="ce-properties-unset-section" class="subsection-title">Unsetting of a Property
 *   <a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-properties-unset-section"></a>
 * </h2>
 * <p>
 *   The undefined value is treated as unsetting of a property when passed to the
 *   property setter and will result in the component using the default value if one exists. Unsetting of
 *   subproperties using the element's <code>setProperty</code> is not supported. Subproperties can only
 *   only be unset when the top level property is unset.
 *   Property sets will not result in DOM attribute updates and after the custom
 *   element is upgraded, the application should use the custom element properties, not attributes to check
 *   the current value.
 * </p>
 * <h2 id="ce-properties-changed-section" class="subsection-title">[property]Changed Events
 *   <a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-properties-changed-section"></a>
 * </h2>
 * <p>
 *   When a property or attribute value changes, a [property]Changed <code>CustomEvent</code>
 *   will be fired with the following properties in the event's detail property.
 *   <table class="props">
 *     <thead>
 *       <tr>
 *         <th>Name</th>
 *         <th>Type</th>
 *         <th>Description</th>
 *       </tr>
 *     </thead>
 *     <tbody>
 *       <tr>
 *      <td>value</td>
 *      <td>any</td>
 *      <td>The current value of the property that changed.</td>
 *    </tr>
 *    <tr>
 *      <td>previousValue</td>
 *      <td>any</td>
 *      <td>The previous value of the property that changed.</td>
 *    </tr>
 *    <tr>
 *      <td>updatedFrom</td>
 *      <td>string</td>
 *      <td>
 *        Where the property was updated from. Supported values are:
 *        <ul>
 *          <li>external - By the application, using either the element's property setter, setAttribute, or external data binding.</li>
 *          <li>internal - By the component, e.g. after user interaction with a text field or selection.</li>
 *        </ul>
 *      </td>
 *    </tr>
 *    <tr>
 *      <td>subproperty</td>
 *      <td>Object</td>
 *      <td>An object holding information about the subproperty that changed.
 *        <table class="props">
 *          <thead>
 *            <tr>
 *              <th>Name</th>
 *              <th>Type</th>
 *              <th>Description</th>
 *            </tr>
 *          </thead>
 *        <tbody>
 *          <tr>
 *            <td>path</td>
 *            <td>string</td>
 *            <td>
 *              The subproperty path that changed, starting from the top level
 *              property with subproperties delimited by ".".
 *            </td>
 *          </tr>
 *          <tr>
 *            <td>value</td>
 *            <td>any</td>
 *            <td>The current value of the subproperty that changed.</td>
 *          </tr>
 *          <tr>
 *            <td>previousValue</td>
 *            <td>any</td>
 *            <td>The previous value of the subproperty that changed.</td>
 *            </tr>
 *            </tbody>
 *          </table>
 *        </td>
 *      </tr>
 *    </tbody>
 *   </table>
 * </p>
 * <p>
 *   Please note that in order for components to be notified of a property change for Array properties, the
 *   value should be data bound and updated using an expression, setting the property to an updated copy
 *   by calling slice(), or by refreshing the component after an in place Array mutation.
 * </p>
 * <p>
 *   The application can listen to these [property]Changed events by adding a listener either declaratively:
 *   <pre class="prettyprint">
 *   <code>
 * &lt;oj-some-element value="{{currentValue}}" on-value-changed="{{valueChangedListener}}">&lt;/oj-some-element>
 *   </code>
 *   </pre>
 *   or programmatically using addEventListener :
 *   <pre class="prettyprint">
 *   <code>
 * someElement.addEventListener("valueChanged", function(event) {...});
 *   </code>
 *   </pre>
 * </p>
 * <h2 id="ce-proptoattr-section" class="subsection-title">Property-to-Attribute Mapping
 *   <a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-proptoattr-section"></a>
 * </h2>
 * <p>
 *   The following rules apply when mapping property to attribute names:
 *   <ul>
 *     <li>Attribute names are case insensitive. CamelCased properties are mapped to
 *   kebab-cased attribute names by inserting a dash before the uppercase letter and converting that letter to lower case
 *   (e.g. a "chartType" property will be mapped to a "chart-type" attribute).</li>
 *     <li> The reverse occurs when mapping a property name from an attribute name.</li>
 *   </ul>
 * </p>
 *
 * <h2 id="ce-methods-section" class="subsection-title">
 *   Methods<a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-methods-section"></a>
 * </h2>
 * <p>
 *   Methods can be accessed on the JET component after the element is fully upgraded. See
 *   the component API doc for specifics.
 * </p>
 *
 * <h2 id="ce-events-section" class="subsection-title">
 *   Events and Listeners<a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-events-section"></a>
 * </h2>
 * <p>
 *   By default, JET components will fire [property]Changed (e.g. valueChanged) <code>CustomEvents</code>
 *   whenever a property is updated. These events, unlike other component events, are non bubbling.
 *   See the <a href="#ce-properties-section">properties section</a> above
 *   for details on the event payload.
 * </p>
 * <p>
 *   In addition to supporting event listeners via the standard <code>addEventListener</code>
 *   mechanism, both JET custom elements and native HTML elements support declarative specification of event
 *   listeners via <code>on-[event-name]</code> attributes (e.g. <code>on-click</code>,
 *   <code>on-value-changed</code> or <code>on-oj-expand</code>). The attributes ultimately delegate to the standard
 *   <code>addEventListener</code> mechanism and only support data bound expressions
 *   that evaluate to functions; arbitrary JavaScript will not be accepted.
 *   Sample usages can be seen in the attribute API doc.
 * </p>
 * <p>
 *   In addition to the event parameter, event listeners specified via <code>on-[event-name]</code>
 *   attributes will receive two additional parameters when they are invoked: <code>data</code> and <code>bindingContext</code>.
 *   The <code>bindingContext</code> parameter provides the listener with the entire data binding context that
 *   was applied to the element while the data parameter provides convenient access to relevant data.
 *   When in an iteration context (e.g. inside an <code>oj-bind-for-each</code>), the <code>data</code> parameter
 *   is equal to <code>bindingContext["$current"]</code>; otherwise, it is equal to <code>bindingContext["$data"]</code>.
 *   These event listeners should be written with signatures of <code>function(event, data, bindingContext)</code>.
 *   Event listeners that are specified via addEventListener will not receive these additional parameters;
 *   they will be invoked with the single event parameter only.
 * </p>
 * <p>
 *   Please note that there is a
 *   current limitation where event listeners specified using this syntax can only be
 *   set during component initialization. Subsequent setAttribute calls for the
 *   event listener attributes will be ignored.
 * </p>
 *
 * <h2 id="ce-slots-section" class="subsection-title">
 *   Slots<a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-slots-section"></a>
 * </h2>
 * <p>
 *  Some JET components allow application provided child content. This child content will be moved by the JET component to
 *  a designated "slot" and is referred to as slot content. Slot content can have one of two characteristics:
 *  <ul>
 *    <li>
 *      Named slot content - Any direct child element with a slot attribute will be moved into that named slot, e.g.  &lt;span slot='startIcon'>... &lt;/span>.
 *      All supported named slots are described in the API Doc. Child elements with unsupported named slots will be removed from the DOM.
 *    </li>
 *    <li>
 *      Default slot content - Any direct child element lacking a slot attribute will be moved to the default slot, also known as a regular child.
 *    </li>
 *  </ul>
 *  Bindings are applied to slot content in the application's context with the exception of template slots which are described
 *  below. Slot content are moved to their designated component slots after bindings are applied. <b>Please note that only text and
 *  element nodes can be assigned to a slot. Comment nodes are not eligible, so oj-bind-* elements which resolve to comment nodes
 *  after bindings are applied should be wrapped in a span or other element node for slotting.</b>
 * </p>
 * <h3 id="ce-slots-template-section" class="subsection-title">
 *   Template Slots<a class="bookmarkable-link" title="Bookmarkable Link" href="#ce-slots-template-section"></a>
 * </h3>
 * <p>
 *  Some components support template slots which allow the application to pass a template element with a DOM fragment that
 *  will be stamped out by the component. Bindings are not applied to template slot content until they are stamped out by the
 *  component. All template slot children will have access to the following variables:
 *  <ul>
 *    <li>$current - Default variable that contains component exposed subproperties as documented in the component's API doc.</li>
 *    <li>component-level template alias - Set by the application if the component has provided a component-level alias attribute
 *        as part of its API. Provides a template alias available to all template slot binding contexts and has the same
 *        subproperties as the $current variable.</li>
 *    <li>template-level alias - Set by the application on the template element via the 'data-oj-as' attribute. Provides an alias
 *        for a specific template instance and has the same subproperties as the $current variable.</li>
 *  </ul>
 *  Note that $current is always availble on the binding context regardless of whether any application provided aliases are set.
 * </p>
 *
 * @ojfragment customElementOverviewDoc - General description doc fragment that shows up in every component's page via a link.
 * @memberof CustomElementOverview
 */

/**
 * @private
 * @export
 */
var VComponent = /** @class */ (function () {
    function VComponent(_tag, props, content) {
        if (props === void 0) { props = {}; }
        this._tag = _tag;
        this.props = props;
        this.content = content;
        // TODO: remove ref variable once we support internal property set API
        this.patching = false;
    }
    /**
     * Called on element sets, patch, and if the root properties
     * change after a mount call.
     * @param props {Object} The current component properties
     * @ignore
     */
    VComponent.prototype.updateUI = function (props) {
        var oldVnode = this._vnode;
        this.props = this.GetRootProps(props, this.props);
        this._vnode = PetitDom.h(this._tag, this.props, this.render());
        // Set the node to the root element that was passed in if not already
        // set. PetitDom.patch will set _node, but other code paths will not.
        if (!this._vnode._node) {
            this._vnode._node = this.ref;
        }
        this._vnode._isRoot = oldVnode._isRoot;
        // We use this (hacky) patching flag to avoid recursive calls
        // to updateUI from the component's attributeChangedCallback.
        this.patching = true;
        PetitDom.patch(this._vnode, oldVnode);
        this.patching = false;
    };
    /**
     * Called by the VirtualElementBridge during the browser's connected
     * callback to instantiate our virtual component and generate the child
     * DOM node.
     * @ignore
     */
    VComponent.prototype.mountContent = function (props, content, rootElem) {
        this.ref = rootElem;
        return this._mount(props, content, rootElem);
    };
    // Called by petit-dom when the constructor is passed to the h function instead of
    // a custom element tag name. This will be called instead of mountContent.
    VComponent.prototype.mount = function (props, content) {
        this.ref = this._mount(props, content);
        return this.ref;
    };
    // Called by petit-dom when the constructor is passed to the h function instead of
    // a custom element tag name.
    VComponent.prototype.patch = function (node, props, oldProps, content, oldContent) {
        // updateUI is also called by VirtualElementBridge. We don't need to pass in oldProps
        // since our current this.props would be the oldProps and we do not support slot content
        // changes after initial render so we can ignore changes in content.
        this.updateUI(props);
    };
    // Called by petit-dom when the constructor is passed to the h function instead of
    // a custom element tag name.
    VComponent.prototype.unmount = function (node) {
        return PetitDom.unmount(this._vnode);
    };
    VComponent.prototype.slot = function (slotName, defaultContent) {
        // This method should only be called in subclass' render()
        // function meaning that mount() would have been called
        // and the slot map should have already been populated.
        if (this._slotMap) {
            return this._slotMap[slotName] || defaultContent;
        }
        else {
            throw new Error("Cannot access slot map before mount has been called");
        }
    };
    // TODO use the actual metadata type once we define it for Composites
    VComponent.register = function (tagName, metadata, constr) {
        oj.VirtualElementBridge.register(tagName, metadata, constr);
    };
    // Given the set of root properties that were provided to the virtual
    // component, returns a (possibly) augmented set of properties that
    // should be applied back to the component's root element.
    //
    // This would be used, for example, in cases where the component wants
    // to apply a style class to itself during rendering.
    //
    // Implementations should *not* modify the props object, but rather should
    // create a new object to hold the augmented set of properties, or return
    // props unmodified if no changes are needed.
    // eslint-disable-next-line no-unused-vars
    VComponent.prototype.GetRootProps = function (props, oldProps) {
        return props;
    };
    /**
     * Returns the slot map of slot name to slotted child elements for a given set
     * of child nodes. If the given element has no children, this method returns an
     * empty object. Note that the default slot name is mapped to the empty string.
     * @param {Array} childNodes An array of virtual nodes
     * @return {Object} A map of the child elements for a given custom element.
     * @ignore
     */
    VComponent.getSlotMap = function (childNodes) {
        var slotMap = {};
        childNodes.forEach(function (vnode) {
            var slot = vnode.props.slot || '';
            if (!slotMap[slot]) {
                slotMap[slot] = [];
            }
            // Generate the slotMap with the virtual node or the real DOM node.
            // petit-dom knows how to deal with both.
            slotMap[slot].push(vnode._refNode || vnode);
        });
        return slotMap;
    };
    /**
     * Renders the child DOM for this component and returns DOM node to mount which can
     * either be the child content if a root custom element node was passed in or the root
     * custom element node itself.
     * @param {Object} props The properties for this component
     * @param {Array|Object} content The slot content for this component
     * @param {HTMLElement} rootElem An optional live DOM node representing the custom
     *                               element root for this component
     * @return {HTMElement}
     * @private
     */
    VComponent.prototype._mount = function (props, content, rootElem) {
        if (rootElem === void 0) { rootElem = null; }
        // Before rendering content, generate the slot map given any slot content.
        // Note that we do not need to create a storage node for virtual nodes
        if (!this._slotMap && content) {
            this._slotMap = VComponent.getSlotMap(Array.isArray(content) ? content : [content]);
        }
        // Merge application and component provided properties
        this.props = this.GetRootProps(props, this.props);
        // Generate the virtual nodes representing the child DOM for this custom element
        var childContent = this.render();
        // Generate the virtual node for this custom element
        this._vnode = PetitDom.h(this._tag, this.props, childContent);
        // Our mounting logic changes depending on whether we are being instantiated by
        // the browser's connected callback or by our VComponent constructor in virtual DOM
        var mountNode;
        if (rootElem) {
            // Set the _node on the vnode because we won't be calling mount which would set it
            this._vnode._node = rootElem;
            // Set a flag that this vnode is a root custom element node
            // so we don't call setAttribute on it and override data bound
            // attributes. Internally generated DOM won't have data binding
            // so it's safe to call setAttribute on those elements.
            this._vnode._isRoot = true;
            // Since we're given the root custom element we don't need to render a DOM node
            // for it. However we do still need to generate a virtual node so we still call the
            // h function with the tag name. We only need to mount the child content.
            if (childContent) {
                mountNode = PetitDom.mount(childContent);
                // Always set null binding provider to child content since
                // virtually rendered content does not support knockout
                mountNode.setAttribute('data-oj-binding-provider', 'none');
            }
        }
        else {
            mountNode = PetitDom.mount(this._vnode);
            // Stash a reference of the virtual component onto the DOM node
            // so we don't regenerate the class during the connected callback
            Object.defineProperty(mountNode, '_vcomp', { value: this, enumerable: false });
        }
        return mountNode;
    };
    return VComponent;
}());

/* global vdom:false */

/**
 * @class
 * @ignore
 */
oj.VirtualElementBridge = {};

/**
 * Prototype for the JET component definitional bridge instance
 */
oj.VirtualElementBridge.proto = Object.create(oj.BaseCustomElementBridge.proto);

oj.CollectionUtils.copyInto(oj.VirtualElementBridge.proto, {
  AddComponentMethods: function (proto) {
    // eslint-disable-next-line no-param-reassign
    proto.setProperty = function (prop, value) {
      var bridge = oj.BaseCustomElementBridge.getInstance(this);
      if (!bridge.SaveEarlyPropertySet(prop, value)) {
        bridge.SetProperty(this, prop, value, this, true);
      }
    };
    // eslint-disable-next-line no-param-reassign
    proto.getProperty = function (prop) {
      // 'this' is the property object we pass to the definitional element contructor to track internal property changes
      var bridge = oj.BaseCustomElementBridge.getInstance(this);
      return bridge.GetProperty(this, prop, this);
    };
  },

  CreateComponent: function (element) {
    if (!element._vcomp) {
      if (oj.Components) {
        oj.Components.unmarkPendingSubtreeHidden(element);
      }
      // We expose a similar set of properties as composites except that props is
      // not a Promise and we don't expose any slot information.
      // At the moment some definitional elements have mutation observers so they don't need
      // to rely on refresh being called to be alerted of new children so any cached slotMap
      // can become out of sync. We should add this once we build in support to auto detect
      // added/removed children to custom elements.
      // this._CONTEXT = {
      //   element: element,
      //   props: this._PROPS_PROXY,
      //   unique: oj.BaseCustomElementBridge.__GetUnique(),
      // };
      // this._CONTEXT.uniqueId = element.id ? element.id : this._CONTEXT.unique;
      var descriptor = oj.BaseCustomElementBridge.__GetDescriptor(element.tagName);
      Object.defineProperty(element, '_vcomp',
        { value: new descriptor._CONSTRUCTOR(this._PROPS), enumerable: false });
      this._mountRoot(element, element._vcomp, this._PROPS);
    }

    // Set flag when we can fire property change events
    this.__READY_TO_FIRE = true;

    // Resolve the component busy state
    this.resolveDelayedReadyPromise();
  },

  // eslint-disable-next-line no-unused-vars
  DefineMethodCallback: function (proto, method, methodMeta) {
    // eslint-disable-next-line no-param-reassign
    proto[method] = function () {
      // The VComponent is asynchronously instantiated by CreateComponent so we
      // need to check that this has happened before we call any methods defined on it.
      // Custom elements are upgraded synchronously meaning the method will be available
      // on the HTMLElement, but we tell applications to wait on the component busy context
      // before accessing properties and methods due to the asynch CreateComponent call.
      if (!this._vcomp) {
        var bridge = oj.BaseCustomElementBridge.getInstance(this);
        bridge.throwError(this, 'Cannot access methods before element is upgraded.');
      }
      return this._vcomp[method].apply(this._vcomp, arguments);
    };
  },

  DefinePropertyCallback: function (proto, property, propertyMeta) {
    function set(value, bOuterSet) {
      // Properties can be set before the component is created. These early
      // sets are actually saved until after component creation and played back.
      if (!this._BRIDGE.SaveEarlyPropertySet(property, value)) {
        var previousValue = this._BRIDGE._PROPS[property];
        if (!oj.BaseCustomElementBridge.__CompareOptionValues(property, propertyMeta,
                                                              value, previousValue)) {
          // Skip validation for inner sets so we don't throw an error when updating readOnly writeable properties
          if (bOuterSet) {
            // eslint-disable-next-line no-param-reassign
            value = this._BRIDGE.ValidatePropertySet(this._ELEMENT, property, value);
          }
          if (propertyMeta._eventListener) {
            this._BRIDGE.SetEventListenerProperty(this._ELEMENT, property, value);
            this._BRIDGE._PROPS[property] = value;
          } else {
            this._BRIDGE._PROPS[property] = value;
            oj.BaseCustomElementBridge.__FirePropertyChangeEvent(
              this._ELEMENT, property, value, previousValue, bOuterSet ? 'external' : 'internal'
            );
          }

          // this will get called before connected callback so short circuit updateUI for that case
          if (this._ELEMENT._vcomp) {
            this._BRIDGE._updateUI(this._ELEMENT);
          }
        }
      }
    }

    function innerSet(value) {
      set.bind(this)(value, false);
    }

    // Called on the custom element
    function outerSet(value) {
      var bridge = oj.BaseCustomElementBridge.getInstance(this);
      set.bind(bridge._PROPS_PROXY)(value, true);
    }

    function get() {
      var value = this._BRIDGE._PROPS[property];
      // If the attribute has not been set, return the default value
      if (value === undefined) {
        value = this._BRIDGE.GetDefaultValue(propertyMeta);
        this._BRIDGE._PROPS[property] = value;
      }
      return value;
    }

    function innerGet() {
      return get.bind(this)();
    }

    // Called on the custom element
    function outerGet() {
      var bridge = oj.BaseCustomElementBridge.getInstance(this);
      return get.bind(bridge._PROPS_PROXY)();
    }

    // Don't add event listener properties for inner props
    if (!propertyMeta._derived) {
      oj.BaseCustomElementBridge.__DefineDynamicObjectProperty(proto._propsProto, property,
                                                               innerGet, innerSet);
    }
    oj.BaseCustomElementBridge.__DefineDynamicObjectProperty(proto, property, outerGet, outerSet);
  },

  // eslint-disable-next-line no-unused-vars
  HandleReattached: function (element) {
    // TODO call VComponent hook to allow component to reattach event listeners?
  },

  // eslint-disable-next-line no-unused-vars
  HandleDeattached: function (element) {
    // TODO call VComponent hook to allow component to detach event listeners?
  },

  InitializeElement: function (element) {
    if (!element._vcomp) {
      if (oj.Components) {
        oj.Components.markPendingSubtreeHidden(element);
      }
      oj.BaseCustomElementBridge.__InitProperties(element, element);
    }
  },

  InitializePrototype: function (proto) {
    // Invoke callback on the superclass
    oj.BaseCustomElementBridge.proto.InitializePrototype.call(this, proto);

    Object.defineProperty(proto, '_propsProto', { value: {} });
  },

  InitializeBridge: function (element) {
    // Invoke callback on the superclass
    oj.BaseCustomElementBridge.proto.InitializeBridge.call(this, element);

    this._EXTENSION = this.METADATA.extension || {};

    // For tracking all properties
    this._PROPS = {};

    // Has getters/setters and calls to set properties on this._PROPS
    if (element._propsProto) {
      this._PROPS_PROXY = Object.create(element._propsProto);
      this._PROPS_PROXY._BRIDGE = this;
      this._PROPS_PROXY._ELEMENT = element;
    }
  },

  PlaybackEarlyPropertySets: function (element) {
    if (!element._vcomp) {
      oj.BaseCustomElementBridge.proto.PlaybackEarlyPropertySets.call(this, element);
    }
  },

  _mountRoot: function (element, vcomp, props) {
    var vprops = _copyProps(props);
    var content = oj.VirtualElementBridge._processSlotContent(element);
    var childNode = vcomp.mountContent(vprops, content, element);
    if (childNode) {
      element.appendChild(childNode);
    }

    // The virtual component may have produced new root properties during
    // mounting (eg. maybe have introduced new root classes that need to
    // be applied).  If so, we manually patch these back to our custom
    // element.
    var newProps = vcomp.props;
    if (newProps !== vprops) {
      this._updateUI(element, newProps);
    }
  },

  _updateUI: function (element, props) {
    if (!element._vcomp.patching) {
      // Copying props on every attr change may end up being expensive
      // (eg. if multiple attributes are set on the same component
      // instance).  Eventually we'll perform the updateUI
      // asynchronously, which will allow us to avoid repeated copying.
      var vprops = _copyProps(props || this._PROPS);
      element._vcomp.updateUI(vprops);
    }
  }
});

/**
 * @export
 */
oj.VirtualElementBridge.register = function (tagName, metadata, constr) {
  if (!constr) {
    oj.Logger.error('Required constructor needed to register custom element: ' + tagName);
  }
  var descriptor = {};
  descriptor[oj.BaseCustomElementBridge.DESC_KEY_META] = metadata;
  if (!descriptor.extension) {
    descriptor.extension = {};
  }
  descriptor._CONSTRUCTOR = constr;

  if (oj.BaseCustomElementBridge.__Register(tagName, descriptor, oj.VirtualElementBridge.proto)) {
    customElements.define(tagName.toLowerCase(),
                          oj.VirtualElementBridge.proto.getClass(descriptor));
  }
};

/**
 * Creates a storage node for a custom element, moves all slot content to
 * the storage node and returns an Array of virtual nodes representing the
 * slot content or null if the custom element has no slot content.
 * @param {Element} element The custom element
 * @return {Array|null}
 * @private
 */
oj.VirtualElementBridge._processSlotContent = function (element) {
  if (element.childNodes) {
    // Needed to replicate what shadow DOM does since we don't have a
    // shadow root to hide slot content that do not map to a component
    // defined slot.
    if (!element._nodeStorage) {
      // eslint-disable-next-line no-param-reassign
      element._nodeStorage = document.createElement('div');
      // eslint-disable-next-line no-param-reassign
      element._nodeStorage.style.display = 'none';
      element.appendChild(element._nodeStorage);
    }
    // Array of virtual nodes we will pass to the VComponent mountContent method
    var content = [];
    var assignableNodes = [];
    for (var i = 0; i < element.childNodes.length - 1; i++) {
      var node = element.childNodes[i];
      if (oj.BaseCustomElementBridge.isSlotable(node)) {
        // Create a lightweight virtual node that contains a reference
        // back to the original slot content and slot value
        content.push({
          _refNode: node,
          props: { slot: oj.BaseCustomElementBridge.getSlotAssignment(node) }
        });
        assignableNodes.push(node);
      }
    }
    assignableNodes.forEach(function (assignableNode) {
      element._nodeStorage.appendChild(assignableNode); // @HTMLUpdateOK
    });
    // Notifies JET components inside nodeStorage that they have been hidden
    // For upstream or indirect dependency we will still rely components being registered on the oj namespace.
    if (oj.Components) {
      oj.Components.subtreeHidden(element._nodeStorage);
    }
    return content;
  }
  return null;
};

// We need to make a copy of this._props any time we hand off
// props to the vcomp, as we mutate our own copy and vcomp
// should not be exposed to these changes (until we hand off
// a new copy)
function _copyProps(props) {
  var propsCopy = {};
  // eslint-disable-next-line no-restricted-syntax
  for (var prop in props) {
    // Because of the way we create and copy the props prototype we
    // can't use getOwnPropertyNames to retrieve the prop getters
    if (prop !== 'getProperty' && prop !== 'setProperty' && prop !== '_ELEMENT' && prop !== '_BRIDGE') {
      propsCopy[prop] = props[prop];
    }
  }
  return propsCopy;
}

/*!
  petit-dom - v0.2.2
  https://github.com/yelouafi/petit-dom
  Copyright (C) 2017 Yassine Elouafi;
  Licensed under the MIT license

  Modification notice: The code is obtained from https://github.com/yelouafi/petit-dom
  and modified by Oracle JET team to be included into Oracle JET project.
*/

/* eslint-disable */
var PetitDom = (function () {

  var EMPTYO = {};
  var EMPTYAR = [];
  var isArray = Array.isArray;
  var isVNode = function isVNode(c) {
    return c && (c._vnode != null || c._text != null);
  };
  var isComponent = function isComponent(c) {
    return c && c.mount && c.patch && c.unmount;
  };

  function h(type, props, contArg) {
    var content,
        args,
        i,
        isSVG = false;
    var len = arguments.length - 2;

    if (typeof type !== "string") {
      if (len === 1) {
        content = contArg;
      } else if (len > 1) {
        args = Array(len);
        for (i = 0; i < len; i++) {
          args[i] = arguments[i + 2];
        }
        content = args;
      }
    } else {
      isSVG = type === "svg";
      if (len === 1) {
        if (isArray(contArg)) {
          content = maybeFlatten(contArg, isSVG);
        } else if (isVNode(contArg)) {
          contArg.isSVG = isSVG;
          content = [contArg];
        } else if (contArg instanceof Node) {
          content = [contArg];
        } else {
          content = [{ _text: contArg == null ? "" : contArg }];
        }
      } else if (len > 1) {
        args = Array(len);
        for (i = 0; i < len; i++) {
          args[i] = arguments[i + 2];
        }
        content = maybeFlatten(args, isSVG);
      } else {
        content = EMPTYAR;
      }
    }

    // sort the props object into categories so we can set as:
    // 1) attributes 2) properties 3) event listeners
    return {
      _vnode: true,
      isSVG: isSVG,
      type: type,
      key: props && props.key || null,
      props: props || EMPTYO,
      propSets: sortProps(props) || EMPTYO,
      content: content
    };
  }

  function maybeFlatten(arr, isSVG) {
    for (var i = 0; i < arr.length; i++) {
      var ch = arr[i];
      if (isArray(ch)) {
        return flattenChildren(arr, i, arr.slice(0, i), isSVG);
      } else if (ch instanceof Node) {
        arr[i] = ch;
      } else if (!isVNode(ch)) {
        arr[i] = { _text: ch == null ? "" : ch };
      } else if (isSVG && !ch.isSVG) {
        ch.isSVG = true;
      }
    }
    return arr;
  }

  function flattenChildren(children, start, arr, isSVG) {
    for (var i = start; i < children.length; i++) {
      var ch = children[i];
      if (isArray(ch)) {
        flattenChildren(ch, 0, arr, isSVG);
      } else if (ch instanceof Node) {
        arr.push(ch);
      } else if (isVNode(ch)) {
        if (isSVG && !ch.isSVG) {
          ch.isSVG = true;
        }
        arr.push(ch);
      } else if (ch == null || typeof ch === 'string') {
        arr.push({ _text: ch == null ? "" : ch });
      } else {
        arr.push(ch);
      }
    }
    return arr;
  }

  var SVG_NS = "http://www.w3.org/2000/svg";
  /**
    TODO: activate full namespaced attributes (not supported in JSX)
    const XML_NS = "http://www.w3.org/XML/1998/namespace"
  **/
  var XLINK_NS = "http://www.w3.org/1999/xlink";
  var NS_ATTRS = {
    show: XLINK_NS,
    actuate: XLINK_NS,
    href: XLINK_NS
  };

  function defShouldUpdate(p1, p2, c1, c2) {
    if (c1 !== c2) return true;
    for (var key in p1) {
      if (p1[key] !== p2[key]) return true;
    }
    return false;
  }

  function mount(c) {
    var node;
    if (c._text != null) {
      node = document.createTextNode(c._text);
    } else if (c._vnode === true) {
      var type = c.type,
          props = c.props,
          propSets = c.propSets,
          content = c.content,
          isSVG = c.isSVG;

      if (typeof type === "string") {
        // TODO : {is} for custom elements
        if (!isSVG) {
          node = document.createElement(type);
        } else {
          node = document.createElementNS(SVG_NS, type);
        }
        patchDOM(node, propSets, null);
        if (!isArray(content)) {
          var isDomNode  = content instanceof Node;
          var childNode = isDomNode  ? content : mount(content);
          node.appendChild(childNode);
          if (isDomNode  && oj.Components) {
            oj.Components.subtreeShown(childNode);
          }
        } else {
          appendChildren(node, content);
        }
      } else if (isComponent(type)) {
        node = type.mount(props, content);
      } else if (typeof type === "function") {
        if (isComponent(type.prototype)) {
          var instance = new type(props, content);
          node = instance.mount(props, content);
          c._data = instance;
        } else {
          var vnode = type(props, content);
          node = mount(vnode);
          c._data = vnode;
        }
      }
    }
    if (node == null) {
      throw new Error("Unkown node type!");
    }
    c._node = node;
    return node;
  }

  function appendChildren(parent, children) {
    var start = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : 0;
    var end = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : children.length - 1;
    var beforeNode = arguments[4];

    while (start <= end) {
      var ch = children[start++];
      // Check to see if child is a vdom or a live DOM node
      var isDomNode  = ch instanceof Node;
      var content = isDomNode  ? ch : mount(ch);
      parent.insertBefore(content, beforeNode);
      // Notifies JET components in node that they have been shown
      // For upstream or indirect dependency we will still rely components being registered on the oj namespace.
      if (isDomNode  && oj.Components) {
        oj.Components.subtreeShown(content);
      }
    }
  }

  function removeChildren(parent, children) {
    var start = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : 0;
    var end = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : children.length - 1;

    var cleared = void 0;
    if (parent.childNodes.length === end - start + 1) {
      parent.textContent = "";
      cleared = true;
    }
    while (start <= end) {
      var ch = children[start++];
      if (!cleared) parent.removeChild(ch._node);
      unmount(ch);
    }
  }

  function unmount(ch) {
    if (isArray(ch)) {
      for (var i = 0; i < ch.length; i++) {
        unmount(ch[i]);
      }
    } else if (ch._vnode === true) {
      if (isComponent(ch.type)) {
        ch.type.unmount(ch._node);
      } else if (typeof ch.type === "function" && isComponent(ch.type.prototype)) {
        ch._data.unmount(ch._node);
      } else if (ch.content != null) {
        unmount(ch.content);
      }
    }
  }

  function propertyNameToAttribute(name) {
    return name.replace(/([A-Z])/g,
      function (match) {
        return '-' + match.toLowerCase();
      }
    );
  }

  function sortProps(props) {
    // store attribute, property, and event listener
    // sets separately
    // special-case style handling
    var sets = {
      attrs: {},
      props: {},
      listeners: {},
      style: {}
    };
    for (var key in props) {
      var value = props[key];
      if (key === 'style') {
        // To be compliant with CSP's unsafe-inline restrictions, we always want to set
        // style properties individually vs. setting the entire style attribute
        sets.style = value;
      } else {
        // Note: innerHTML is not currently supported
        if (typeof value === 'string') {
          // TODO Should we make this more efficient?
          var attrName = propertyNameToAttribute(key);
          sets.attrs[attrName] = value;
        } else {
          // Event listeners will be added with addEventListener
          var eventType = eventListenerPropertyToEventType(key);
          if (eventType) {
            sets.listeners[key] = {type: eventType, listener: value};
          } else {
            sets.props[key] = value;
          }
        }
      }
    }
    return sets;
  }

  function patchDOM(el, sets, oldSets) {
    // We only set global attributes, everything else is set as property
    // or is an event listener.  Style is special-cased
    patchStyle(el, sets.style, oldSets ? oldSets.style : null);
    patchAttrs(el, sets.attrs, oldSets ? oldSets.attrs : null);
    patchListeners(el, sets.listeners, oldSets ? oldSets.listeners : null);
    patchProps(el, sets.props, oldSets ? oldSets.props : null);
  }

  function patchStyle(el, style, oldStyle) {
    if (style !== oldStyle) {
      // Unset style properties by setting them to empty string
      // null is supposed to work as well, but doesn't in IE
      _patchProperties(el.style, style || {}, oldStyle || {}, '');
    }
  }

  function patchAttrs(el, attrs, oldAttrs) {
    for (var key in attrs) {
      var oldv = oldAttrs != null ? oldAttrs[key] : null;
      var newv = attrs[key];
      if (oldv !== newv) {
        if (newv === true) {
          el.setAttribute(key, "");
        } else if (newv === false) {
          el.removeAttribute(key);
        } else {
          if (newv != null) {
            var ns = NS_ATTRS[key];
            if (ns !== undefined) {
              el.setAttributeNS(ns, key, newv);
            } else {
              el.setAttribute(key, newv);
            }
          } else {
            el.removeAttribute(key);
          }
        }
      }
    }
    for (var key in oldAttrs) {
      if (!(key in attrs)) {
        el.removeAttribute(key);
      }
    }
  }

  function patchProps(el, props, oldProps) {
    // Unset component properties by setting them to undefined
    _patchProperties(el, props || {}, oldProps || {}, undefined);
  }

  function _patchProperties(propertyHolder, props, oldProps, unsetValue) {
    for (var key in props) {
      var oldv = oldProps[key];
      var newv = props[key];
      if (oldv !== newv) {
        propertyHolder[key] = newv;
      }
    }
    for (var key in oldProps) {
      if (!(key in props)) {
        propertyHolder[key] = unsetValue;
      }
    }
  }

  /**
   * Calls addEventListener to set listeners on a given element.
   * The listeners and oldListeners contains a map of keys, e.g. 'onClick' to an object that contains
   * the event type and the listener, e.g. { 'onClick: { 'type': 'click', 'listener': clickCallback } }
   * @param {HTMLElement} el
   * @param {Object} listeners
   * @param {Object} oldListeners
   */
  function patchListeners(el, listeners, oldListeners) {
    for (var key in listeners) {
      var oldv = oldListeners != null ? (oldListeners[key] || {}) : {};
      var newv = listeners[key];
      if (oldv.listener !== newv.listener) {
        var eventType = newv.type;
        if (oldv.listener) {
          el.removeEventListener(eventType, oldv.listener);
        }
        el.addEventListener(eventType, newv.listener);
      }
    }
    for (var key in oldListeners) {
      if (!(key in listeners)) {
        var oldv = oldListeners[key];
        el.removeEventListener(oldv.type, oldv.listener);
      }
    }
  }

  function eventListenerPropertyToEventType(property) {
    if (/^on[A-Z]/.test(property)) {
      return property.substr(2, 1).toLowerCase() + property.substr(3);
    }
    return null;
  }

  function patch(newch, oldch, parent) {
    var childNode = oldch._node;

    if (oldch === newch) {
      return childNode;
    }

    var t1, t2;
    if ((t1 = oldch._text) != null && (t2 = newch._text) != null) {
      if (t1 !== t2) {
        childNode.nodeValue = t2;
      }
    } else if (oldch.type === newch.type && oldch.isSVG === newch.isSVG) {
      var type = oldch.type;

      if (isComponent(type)) {
        type.patch(childNode, newch.props, oldch.props, newch.content, oldch.content);
      } else if (typeof type === "function") {
        if (isComponent(type.prototype)) {
          var instance = oldch._data;
          instance.patch(childNode, newch.props, oldch.props, newch.content, oldch.content);
          newch._data = instance;
        } else {
          var shouldUpdateFn = type.shouldUpdate || defShouldUpdate;
          if (shouldUpdateFn(newch.props, oldch.props, newch.content, oldch.content)) {
            var vnode = type(newch.props, newch.content);
            childNode = patch(vnode, oldch._data, parent);
            newch._data = vnode;
          } else {
            newch._data = oldch._data;
          }
        }
      } else if (typeof type === "string") {
        // Skip patching the root node for a custom element
        // since all attributes will be set by application and
        // we don't want to override data bound attributes.
        if (!oldch._isRoot) {
          patchDOM(childNode, newch.propSets, oldch.propSets);
        }
        patchContent(childNode, newch.content, oldch.content);
      } else if (oldch instanceof Node && newch instanceof Node) {
        if (oldch !== newch && parent) {
          parent.replaceChild(newch, oldch);
        }
      } else {
        throw new Error("Unknown node type! " + type);
      }
    } else {
      childNode = mount(newch);
      if (parent) {
        parent.replaceChild(childNode, oldch._node);
      }
      unmount(oldch);
    }

    newch._node = childNode;
    return childNode;
  }

  function patchContent(parent, content, oldContent) {
    if (!isArray(content) && !isArray(oldContent)) {
      if (content !== oldContent) {
        patch(content, oldContent, parent);
      }
    } else if (isArray(content) && isArray(oldContent)) {
      diffChildren(parent, content, oldContent);
    } else {
      removeChildren(parent, oldContent, 0, oldContent.length - 1);
      appendChildren(parent, content);
    }
  }

  function canPatch(v1, v2) {
    return v1.key === v2.key;
  }

  function diffChildren(parent, children, oldChildren) {
    var newStart = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : 0;
    var newEnd = arguments.length > 4 && arguments[4] !== undefined ? arguments[4] : children.length - 1;
    var oldStart = arguments.length > 5 && arguments[5] !== undefined ? arguments[5] : 0;
    var oldEnd = arguments.length > 6 && arguments[6] !== undefined ? arguments[6] : oldChildren.length - 1;

    if (children === oldChildren) return;
    var oldCh;

    /**
      Before applying the diff algorithm we try some preprocessing optimizations
      to reduce the cost
      See https://neil.fraser.name/writing/diff/ for the full details.
        In the following : indel = INsertion/DELetion
    **/

    // common prefix/suffix

    var k = diffCommonPrefix(children, oldChildren, newStart, newEnd, oldStart, oldEnd, canPatch, parent);
    newStart += k;
    oldStart += k;

    k = diffCommonSufffix(children, oldChildren, newStart, newEnd, oldStart, oldEnd, canPatch, parent);
    newEnd -= k;
    oldEnd -= k;

    if (newStart > newEnd && oldStart > oldEnd) {
      return;
    }

    // simple indel: one of the 2 sequences is empty after common prefix/suffix removal

    // old sequence is empty -> insertion
    if (newStart <= newEnd && oldStart > oldEnd) {
      oldCh = oldChildren[oldStart];
      appendChildren(parent, children, newStart, newEnd, oldCh && oldCh._node);
      return;
    }

    // new sequence is empty -> deletion
    if (oldStart <= oldEnd && newStart > newEnd) {
      removeChildren(parent, oldChildren, oldStart, oldEnd);
      return;
    }

    // 2 simple indels: the shortest sequence is a subsequence of the longest
    var oldRem = oldEnd - oldStart + 1;
    var newRem = newEnd - newStart + 1;
    k = -1;
    if (oldRem < newRem) {
      k = indexOf(children, oldChildren, newStart, newEnd, oldStart, oldEnd, canPatch);
      if (k >= 0) {
        oldCh = oldChildren[oldStart];
        appendChildren(parent, children, newStart, k - 1, oldCh._node);
        var upperLimit = k + oldRem;
        newStart = k;
        while (newStart < upperLimit) {
          patch(children[newStart++], oldChildren[oldStart++]);
        }
        oldCh = oldChildren[oldEnd];
        appendChildren(parent, children, newStart, newEnd, oldCh && oldCh._node.nextSibling);
        return;
      }
    } else if (oldRem > newRem) {
      k = indexOf(oldChildren, children, oldStart, oldEnd, newStart, newEnd, canPatch);
      if (k >= 0) {
        removeChildren(parent, oldChildren, oldStart, k - 1);
        upperLimit = k + newRem;
        oldStart = k;
        while (oldStart < upperLimit) {
          patch(children[newStart++], oldChildren[oldStart++]);
        }
        removeChildren(parent, oldChildren, oldStart, oldEnd);
        return;
      }
    }

    // fast case: difference between the 2 sequences is only one item
    if (oldStart === oldEnd) {
      var node = oldChildren[oldStart]._node;
      appendChildren(parent, children, newStart, newEnd, node);
      parent.removeChild(node);
      unmount(node);
      return;
    }
    if (newStart === newEnd) {
      parent.insertBefore(mount(children[newStart]), oldChildren[oldStart]._node);
      removeChildren(parent, oldChildren, oldStart, oldEnd);
      return;
    }

    /*
      last preopt
      if we can find a subsequence that's at least half the longest sequence the it's guaranteed to
      be the longest common subsequence. This allows us to find the lcs using a simple O(N) algorithm
    */
    var hm;
    /*var oldShorter = oldRem < newRem;
    if (oldShorter) {
      hm = diffHalfMatch(
        children,
        oldChildren,
        newStart,
        newEnd,
        oldStart,
        oldEnd,
        canPatch
      );
    } else {
      hm = diffHalfMatch(
        oldChildren,
        children,
        oldStart,
        oldEnd,
        newStart,
        newEnd,
        canPatch
      );
    }
    if (hm) {
      var newStartHm = oldShorter ? hm.start1 : hm.start2;
      var newEndHm = newStartHm + hm.length - 1;
      var oldStartHm = oldShorter ? hm.start2 : hm.start1;
      var oldEndHm = oldStartHm + hm.length - 1;
      for (var i = newStartHm, j = oldStartHm; i <= newEndHm; i++, j++) {
        patch(children[i], oldChildren[j], parent);
      }
      diffChildren(
        parent,
        children,
        oldChildren,
        newStart,
        newStartHm - 1,
        oldStart,
        oldStartHm - 1
      );
      diffChildren(
        parent,
        children,
        oldChildren,
        newEndHm + 1,
        newEnd,
        oldEndHm + 1,
        oldEnd
      );
      return;
    }*/

    /*
      Run the diff algorithm
      First try the O(ND) algorithm. If O(ND) cost is high (Too match diffs between the 2 seqs)
      then fallback to Map lookup based algorithm
    */
    if (!hm) {
      var failed = diffOND(parent, children, oldChildren, newStart, newEnd, oldStart, oldEnd);
      if (failed) diffWithMap(parent, children, oldChildren, newStart, newEnd, oldStart, oldEnd);
    }
  }

  function diffCommonPrefix(s1, s2, start1, end1, start2, end2, eq, parent) {
    var k = 0,
        c1,
        c2;
    while (start1 <= end1 && start2 <= end2 && eq(c1 = s1[start1], c2 = s2[start2])) {
      if (parent) patch(c1, c2, parent);
      start1++;
      start2++;
      k++;
    }
    return k;
  }

  function diffCommonSufffix(s1, s2, start1, end1, start2, end2, eq, parent) {
    var k = 0,
        c1,
        c2;
    while (start1 <= end1 && start2 <= end2 && eq(c1 = s1[end1], c2 = s2[end2])) {
      if (parent) patch(c1, c2, parent);
      end1--;
      end2--;
      k++;
    }
    return k;
  }
  /*
  function diffHalfMatch(s1, s2, start1, end1, start2, end2, eq) {
    var len1 = end1 - start1 + 1;
    var len2 = end2 - start2 + 1;

    if (len1 < 2 || len2 < 1) {
      return null;
    }

    var hm1 = halfMatchInt(start1 + Math.ceil(len1 / 4));
    var hm2 = halfMatchInt(start1 + Math.ceil(len1 / 2));
    return !hm1 && !hm2
      ? null
      : !hm1 ? hm2 : !hm2 ? hm1 : hm1.length > hm2.length ? hm1 : hm2;

    function halfMatchInt(seedStart) {
      var seedEnd = seedStart + Math.floor(len1 / 4);
      var j = start2 - 1;
      var bestCS = { length: 0 };
      while (
        j < end2 &&
        (j = indexOf(s2, s1, j + 1, end2, seedStart, seedEnd, eq)) !== -1
      ) {
        var prefixLen = diffCommonPrefix(s1, s2, seedStart, end1, j, end2, eq);
        var suffixLen = diffCommonSufffix(
          s1,
          s2,
          start1,
          seedStart - 1,
          start2,
          j - 1,
          eq
        );
        if (bestCS.length < prefixLen + suffixLen) {
          bestCS.start1 = seedStart - suffixLen;
          bestCS.start2 = j - suffixLen;
          bestCS.length = prefixLen + suffixLen;
        }
      }
      return bestCS.length >= len1 / 2 ? bestCS : null;
    }
  }
  */
  var PATCH = 2;
  var INSERTION = 4;
  var DELETION = 8;

  /**
    Find the shortest edit script between the old and new sequences
    This is equivalent to finding the shortest path in the edit graph of the 2 sequences
    see "An O(ND) Difference Algorithm and Its Variations" at
    http://citeseerx.ist.psu.edu/viewdoc/download?doi=10.1.1.4.6927&rep=rep1&type=pdf
  **/
  function diffOND(parent, children, oldChildren) {
    var newStart = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : 0;
    var newEnd = arguments.length > 4 && arguments[4] !== undefined ? arguments[4] : children.length - 1;
    var oldStart = arguments.length > 5 && arguments[5] !== undefined ? arguments[5] : 0;
    var oldEnd = arguments.length > 6 && arguments[6] !== undefined ? arguments[6] : oldChildren.length - 1;

    var rows = newEnd - newStart + 1;
    var cols = oldEnd - oldStart + 1;
    var dmax = rows + cols;

    var v = [];
    var d, k, r, c, pv, cv, pd;
    outer: for (d = 0; d <= dmax; d++) {
      if (d > 50) return true;
      pd = d - 1;
      pv = d ? v[d - 1] : [0, 0];
      cv = v[d] = [];
      for (k = -d; k <= d; k += 2) {
        if (k === -d || k !== d && pv[pd + k - 1] < pv[pd + k + 1]) {
          c = pv[pd + k + 1];
        } else {
          c = pv[pd + k - 1] + 1;
        }
        r = c - k;
        while (c < cols && r < rows && canPatch(oldChildren[oldStart + c], children[newStart + r])) {
          c++;
          r++;
        }
        if (c === cols && r === rows) {
          break outer;
        }
        cv[d + k] = c;
      }
    }

    var diff = Array(d / 2 + dmax / 2);
    var deleteMap = {};
    var oldCh;
    var diffIdx = diff.length - 1;
    for (d = v.length - 1; d >= 0; d--) {
      while (c > 0 && r > 0 && canPatch(oldChildren[oldStart + c - 1], children[newStart + r - 1])) {
        // diagonal edge = equality
        diff[diffIdx--] = PATCH;
        c--;
        r--;
      }
      if (!d) break;
      pd = d - 1;
      pv = d ? v[d - 1] : [0, 0];
      k = c - r;
      if (k === -d || k !== d && pv[pd + k - 1] < pv[pd + k + 1]) {
        // vertical edge = insertion
        r--;
        diff[diffIdx--] = INSERTION;
      } else {
        // horizontal edge = deletion
        c--;
        diff[diffIdx--] = DELETION;
        oldCh = oldChildren[oldStart + c];
        if (oldCh.key != null) {
          deleteMap[oldCh.key] = oldStart + c;
        }
      }
    }

    applyDiff(parent, diff, children, oldChildren, newStart, oldStart, deleteMap);
  }

  function applyDiff(parent, diff, children, oldChildren, newStart, oldStart, deleteMap) {
    var ch,
        oldCh,
        node,
        oldMatchIdx,
        moveMap = {};
    for (var i = 0, chIdx = newStart, oldChIdx = oldStart; i < diff.length; i++) {
      var op = diff[i];
      if (op === PATCH) {
        patch(children[chIdx++], oldChildren[oldChIdx++], parent);
      } else if (op === INSERTION) {
        ch = children[chIdx++];
        oldMatchIdx = null;
        if (ch.key != null) {
          oldMatchIdx = deleteMap[ch.key];
        }
        if (oldMatchIdx != null) {
          node = patch(ch, oldChildren[oldMatchIdx]);
          moveMap[ch.key] = oldMatchIdx;
        } else {
          node = mount(ch);
        }
        parent.insertBefore(node, oldChIdx < oldChildren.length ? oldChildren[oldChIdx]._node : null);
      } else if (op === DELETION) {
        oldChIdx++;
      }
    }

    for (i = 0, oldChIdx = oldStart; i < diff.length; i++) {
      var _op = diff[i];
      if (_op === PATCH) {
        oldChIdx++;
      } else if (_op === DELETION) {
        oldCh = oldChildren[oldChIdx++];
        if (oldCh.key == null || moveMap[oldCh.key] == null) {
          parent.removeChild(oldCh._node);
          unmount(oldCh);
        }
      }
    }
  }

  /**
    A simplified implementation of Hunt-Szymanski algorithm
    see "A Fast Algorithm for Computing Longest Common Subsequences"
    http://citeseerx.ist.psu.edu/viewdoc/download?doi=10.1.1.608.1614&rep=rep1&type=pdf
    This implementation supposes keys are unique so we only use
    simple object maps to build the match list
  **/
  function diffWithMap(parent, children, oldChildren, newStart, newEnd, oldStart, oldEnd) {
    var keymap = {},
        unkeyed = [],
        idxUnkeyed = 0,
        ch,
        oldCh,
        k,
        idxInOld,
        key;

    var newLen = newEnd - newStart + 1;
    var oldLen = oldEnd - oldStart + 1;
    var minLen = Math.min(newLen, oldLen);
    var tresh = Array(minLen + 1);
    tresh[0] = -1;

    for (var i = 1; i < tresh.length; i++) {
      tresh[i] = oldEnd + 1;
    }
    var link = Array(minLen);

    for (i = oldStart; i <= oldEnd; i++) {
      oldCh = oldChildren[i];
      key = oldCh.key;
      if (key != null) {
        keymap[key] = i;
      } else {
        unkeyed.push(i);
      }
    }

    for (i = newStart; i <= newEnd; i++) {
      ch = children[i];
      idxInOld = ch.key == null ? unkeyed[idxUnkeyed++] : keymap[ch.key];
      if (idxInOld != null) {
        k = findK(tresh, idxInOld);
        if (k >= 0) {
          tresh[k] = idxInOld;
          link[k] = { newi: i, oldi: idxInOld, prev: link[k - 1] };
        }
      }
    }

    k = tresh.length - 1;
    while (tresh[k] > oldEnd) {
      k--;
    }var ptr = link[k];
    var diff = Array(oldLen + newLen - k);
    var curNewi = newEnd,
        curOldi = oldEnd;
    var d = diff.length - 1;
    while (ptr) {
      var _ptr = ptr,
          newi = _ptr.newi,
          oldi = _ptr.oldi;

      while (curNewi > newi) {
        diff[d--] = INSERTION;
        curNewi--;
      }
      while (curOldi > oldi) {
        diff[d--] = DELETION;
        curOldi--;
      }
      diff[d--] = PATCH;
      curNewi--;
      curOldi--;
      ptr = ptr.prev;
    }
    while (curNewi >= newStart) {
      diff[d--] = INSERTION;
      curNewi--;
    }
    while (curOldi >= oldStart) {
      diff[d--] = DELETION;
      curOldi--;
    }
    applyDiff(parent, diff, children, oldChildren, newStart, oldStart, keymap);
  }

  function findK(ktr, j) {
    var lo = 1;
    var hi = ktr.length - 1;
    while (lo <= hi) {
      var mid = Math.ceil((lo + hi) / 2);
      if (j < ktr[mid]) hi = mid - 1;else lo = mid + 1;
    }
    return lo;
  }

  function indexOf(a, suba, aStart, aEnd, subaStart, subaEnd, eq) {
    var j = subaStart,
        k = -1;
    var subaLen = subaEnd - subaStart + 1;
    while (aStart <= aEnd && aEnd - aStart + 1 >= subaLen) {
      if (eq(a[aStart], suba[j])) {
        if (k < 0) k = aStart;
        j++;
        if (j > subaEnd) return k;
      } else {
        k = -1;
        j = subaStart;
      }
      aStart++;
    }
    return -1;
  }

  return {
    h: h,
    mount: mount,
    patch: patch,
    unmount: unmount,
    diffChildren: diffChildren
  }
}());

/*
** Copyright (c) 2008, 2013, Oracle and/or its affiliates. All rights reserved.
**
**34567890123456789012345678901234567890123456789012345678901234567890123456789
*/

/**
 * @namespace oj.Components
 * @classdesc JET Component services
 * @since 1.0
 * @export
 * @ojtsmodule
 * @hideconstructor
 */
oj.Components = {};
// this is the variable name that the AMD module will return in the require callback (used in a no-require environment)
// eslint-disable-next-line no-unused-vars
var Components = oj.Components;

/**
 * @private
 */
oj.Components._OJ_CONTAINER_ATTR = 'data-oj-container';

/**
 * @private
 */
var _OJ_WIDGET_NAMES_DATA = 'oj-component-names';

/**
 * Marks the element which is a jQueryUI component
 * @private
 */
var _OJ_COMPONENT_NODE_CLASS = 'oj-component-initnode';

/**
 * Marks an element as being hidden.
 *
 * @private
 */
var _OJ_SUBTREE_HIDDEN_CLASS = 'oj-subtree-hidden';

/**
 * Marks an element as a container that will control hidden of its children
 * once it finishes initializing
 *
 * @private
 */
var _OJ_PENDING_SUBTREE_HIDDEN_CLASS = 'oj-pending-subtree-hidden';

/**
 * Sets default options values for JET components.
 * @param {!Object} options - property values that will be merged into the values
 * that were previously set using this method. The options object is expected to have the format demonstrated
 * by the following example:
 * <pre>
 * {
 *   'default': // properties for all JET components
 *   {
 *     'option1': 'somevalue'
 *   },
 *   'editableValue': // properties for editableValue components
 *   {
 *     'option1': 'somevalue1',
 *     'option2': oj.Components.createDynamicPropertyGetter(function(context){
 *                                 return context['containers'].indexOf('ojTable') >= 0 ? 'tableValue' : 'normalValue'})
 *   },
 *   'ojText': // properties for instances of ojText
 *   {
 *     'option1': 'somevalue2'
 *   }
 * }
 * </pre>
 * To specify a dynamic getter for the property, pass your callback to oj.Components.createDynamicPropertyGetter(). Note
 * that dynamic getters nested within a complex property value are not supported
 * @see oj.Components.createDynamicPropertyGetter
 * @return {void}
 * @export
 * @ojtsignore
 */
oj.Components.setDefaultOptions = function (options) {
  var props = oj.Components._defaultProperties || {};

  var keys = Object.keys(options);

  keys.forEach(function (key) {
    var value = options[key];
    if (!oj.CollectionUtils.isPlainObject(value)) {
      throw new Error('Invalid default options');
    }
    props[key] = _accumulateValues(props[key] || {}, value, false);
  }
  );

  oj.Components._defaultProperties = props;
};

/**
 * Retrieves default option values for JET components. This method should only be used internally by JET.
 * @deprecated since version 2.2
 * @ignore
 * @return {Object} default option values
 * @see oj.Components.setDefaultOptions
 * @export
 */
oj.Components.getDefaultOptions = function () {
  return (oj.Components._defaultProperties || {});
};


/**
 * Creates a dynamic getter that can be used as a property value in oj.Components.setDefaultOptions()
 * @param {!Function} callback - dynamic property callback. The callback will receive a context object as a parameter.
 * The following properties are currently supported on the context object:
 * <ul>
 * <li>containers - an array of component names of the current component's containers that require special behavior from
 * their children</li>
 * <li>element - component's host DOM element</li>
 * </ul>
 * The callback should return the computed property value
 *
 * @return {Object} - dynamic property getter
 * @see oj.Components.setDefaultOptions
 * @export
 * @ojtsignore
 */
oj.Components.createDynamicPropertyGetter = function (callback) {
  return new __ojDynamicGetter(callback);
};

/**
 * This method should only be used for JQueryUI components and will return null if used
 * with a custom element. Retrieves widget constructor associated with the HTML element
 * or null if none is found. The returned constructor is already bound to the associated
 * JQuery element, so it can be invoked as a function directly. For example:
 * <pre>
 * widgetConstructor("option", "label", "custom"); // sets label option
 * </pre>
 * If widgetName is not specified, and if more than one widget is associated with the element,
 * the method will a return the widget that was created first.
 * @param {?(Element|Node)} element - HTML element
 * @param {string=} widgetName - optional widget name
 * @return {Function|null} widget constructor
 * @export
 * @ojtsignore
 */
oj.Components.getWidgetConstructor = function (element, widgetName) {
  if (element && !oj.BaseCustomElementBridge.getRegistered(element.tagName)) {
    return oj.Components.__GetWidgetConstructor(element, widgetName);
  }
  return null;
};

/**
 * Internal version for components to call which won't return null for
 * custom elements. See public method for jsDoc.
 * @param {?(Element|Node)} element - HTML element
 * @param {string=} widgetName - optional widget name
 * @return {Function|null} widget constructor
 * @ignore
 */
oj.Components.__GetWidgetConstructor = function (element, widgetName) {
  var jelem = $(element);

  var data = jelem.data(_OJ_WIDGET_NAMES_DATA);
  if (data) {
    if (widgetName == null) {
      // eslint-disable-next-line no-param-reassign
      widgetName = data[0];
    } else if (data.indexOf(widgetName) < 0) {
      // eslint-disable-next-line no-param-reassign
      widgetName = undefined;
    }

    if (widgetName != null) {
      var func = jelem[widgetName];
      if ((typeof func) === 'function') {
        return func.bind(jelem);
      }
    }
  }
  return null;
};

/**
 * Notifies JET framework that a subtree possibly containing JET components has been inserted
 * into the document programmatically.
 *
 * Note that there is no need to call this method when the new DOM is being inserted by the template engine
 * in Knockout.js
 * @param {!Element} node - the root of the subtree
 * @see oj.Components.subtreeDetached
 * @return {void}
 * @export
 */
oj.Components.subtreeAttached = function (node) {
  oj.DomUtils.fixResizeListeners(node);
  _applyToComponents(node,
    function (instance) {
      instance.__handleSubtreeAttached();
    }
  );
};

/**
 * Notifies JET framework that a subtree possibly containing JET components has been removed
 * from the document programmatically.
 *
 * Note that calling this method is not needs after calling JQuery's .remove() because all JET components would have been
 * already destroyed in that case. Similarly, there is no need to call this method after the subtree has been removed by
 * Knockout.js
 * @param {!Element} node - the root of the subtree
 * @see oj.Components.subtreeAttached
 * @return {void}
 * @export
 */
oj.Components.subtreeDetached = function (node) {
  _applyToComponents(node,
    function (instance) {
      instance.__handleSubtreeDetached();
    }
  );
};

/**
 * Notifies JET framework that a subtree possibly containing JET components is no longer hidden with display:none style
 * This method should be called by the application if the 'display' style is being changed from 'hidden' programmatically,
 * such as when JQuery's .show() method is called.
 * For cases where subtree is shown on initial render, this method should be called with the options parameter set to
 * {'initialRender':true}, that will result in _NotifyInitShown() calls to the subtree components.
 * All oj-defer elements in the entire subtree will be activated. Note that subtreeShown currently notifies the entire
 * subtree as well. This generally means that only non-nested oj-defer elements make sense in a subtree.
 *
 * @param {!Element} node - the root of the subtree
 * @param {Object=} options Options to control subtreeShown
 * @param {boolean} options.initialRender The index at which to start fetching records.
 * @see oj.Components.subtreeHidden
 * @return {void}
 * @export
 */
oj.Components.subtreeShown = function (node, options) {
  var _options = options || {};
  var isInitialRender = _options.initialRender;
  if (!isInitialRender) {
    oj.DomUtils.fixResizeListeners(node);
  }

  var _node = $(node)[0]; // Strip possible jQuery wrapper

  unmarkSubtreeHidden(_node);

  _applyHideShowToComponents(_node, function (instance) {
    if (isInitialRender) {
      instance._NotifyInitShown();
    } else {
      instance._NotifyShown();
    }
  }, true);
};

/**
 * Notifies JET framework that a subtree possibly containing JET components has been hidden  with display:none style
 * This method should be called by the application after the subtree has been hidden programmatically, such as
 * when JQuery's .hide() method is called.
 *
 * @param {!Element} node - the root of the subtree
 * @see oj.Components.subtreeShown
 * @return {void}
 * @export
 */
oj.Components.subtreeHidden = function (node) {
  var _node = $(node)[0]; // Strip possible jQuery wrapper

  _applyHideShowToComponents(_node, function (instance) {
    instance._NotifyHidden();
  }, false);

  markSubtreeHidden(_node);
};

/**
 * Add a marker class indicating that this subtree is hidden.
 *
 * @ignore
 */
function markSubtreeHidden(element) {
  element.classList.add(_OJ_SUBTREE_HIDDEN_CLASS);
}

/**
 * Remove the marker class indicating that this subtree is hidden.
 *
 * @ignore
 */
function unmarkSubtreeHidden(element) {
  element.classList.remove(_OJ_SUBTREE_HIDDEN_CLASS);
}

/**
 * Called by CCAs and certain custom elements when they are first connected
 * to indicate that this component is initializing and will control
 * whether its child subtrees are hidden.
 *
 * @ignore
 */
oj.Components.markPendingSubtreeHidden = function (element) {
  element.classList.add(_OJ_PENDING_SUBTREE_HIDDEN_CLASS);
};


/**
 * Called by CCAs and certain custom elements right before they are first rendered.
 * This component will control whether its child subtrees are hidden.
 *
 * @ignore
 */
oj.Components.unmarkPendingSubtreeHidden = function (element) {
  element.classList.remove(_OJ_PENDING_SUBTREE_HIDDEN_CLASS);
};

/**
 * Determines if a component identified by the <code>widgetName</code> has been
 * bound and initialized on a given <code>jelement</code>.
 *
 * @param {jQuery} jelement to which the component is bound
 * @param {string} widgetName constructor name of the target component.
 * @return {boolean} <code>true</code> if the component identified by the widgetName
 *  has be bound and initialized to the target element.
 * @ojtsignore
 */
oj.Components.isComponentInitialized = function (jelement, widgetName) {
  /** @type {?} */
  var widgets = jelement.data(_OJ_WIDGET_NAMES_DATA);
  if ($.isArray(widgets) && widgets.indexOf(widgetName) > -1 &&
      jelement.is('.' + _OJ_COMPONENT_NODE_CLASS)) {
    return true;
  }
  return false;
};

/**
 * @ignore
 */
oj.Components.__getDefaultOptions = function (hierarchyNames) {
  var defaults = {};
  var allProperties = oj.Components.getDefaultOptions();

  for (var i = hierarchyNames.length - 1; i >= 0; i--) {
    var name = hierarchyNames[i];
    var props = allProperties[name];
    if (props !== undefined) {
      defaults = _accumulateValues(defaults, props, true);
    }
  }

  return defaults;
};

/**
 * Retrieves the JET component element that
 * the node is in.
 * @param {?(Element|Node)} node - DOM node
 * @return {?(Element|Node)} componentElement - JET component element
 * A component element is the DOM element on which the JET component is
 * initialized.
 * @export
 * @ojtsignore
*/
oj.Components.getComponentElementByNode = function (node) {
  // Temporarily exposing this private flag in order to allow
  // MonkeyTalk to access JET components that are part of the
  // composite's implementation. We are not exposing this flag
  // as a public API as a) accessing composite implementation
  // components is bad and b) the need for this method will soon
  // go away. Once our new automation API (recording adapters) are
  // in place, even MonkeyTalk will no longer need this flag. Adding
  // this as a stop-gap measure to allow MonkeyTalk to carry on in the
  // meantime. Callers other than MonkeyTalk must avoid specifying this flag.
  var mtAccessCompositeInternals = !!(arguments.length > 1 && arguments[1]);
  return _getComponentElementByNode(node, mtAccessCompositeInternals);
};

/**
 * Private method implementing the functionality of
 * getComponentElementByNode. This was done because Closure
 * throws an error when the private, undocumented flag (mtAccessCompositeInternals)
 * is passed to recursive calls of getComponentElementByNode.
 * @private
*/
function _getComponentElementByNode(node, mtAccessCompositeInternals) {
  if (node == null) {
    return null;
  }
  // node can be a Node or Element but we call some Element only APIs
  // so we need to do an additional isElement check first
  var isElement = node.nodeType === 1;
  // for upstream or indirect dependency we will still rely components being registered on the oj namespace.
  var containingComposite = (oj.Composite && !mtAccessCompositeInternals ?
                             oj.Composite.getContainingComposite(node) : null);
  if (containingComposite) { // node is in or is a composite, return composite
    return containingComposite;
  } else if (isElement && node.hasAttribute('data-oj-internal')) { // node is an internal component
    if (node.parentNode instanceof Element &&
        node.parentNode.hasAttribute('data-oj-surrogate-id')) { // internal component is a popup
      // eslint-disable-next-line no-param-reassign
      node = document.querySelector('[data-oj-popup-' + node.id + '-parent]'); // retrieves popups parent element
      return _getComponentElementByNode(node, mtAccessCompositeInternals);
    }
    return _getComponentElementByNode(node.parentNode, mtAccessCompositeInternals);
  } else if (_isComponentElement(node)) { // node is a component element
    return node;
  } else if (isElement && node.classList.contains('oj-component')) { // node is component wrapper
    // eslint-disable-next-line no-param-reassign
    node = node.querySelector('.oj-component-initnode:not([data-oj-internal])') || node;
    if (_isJQueryUI(node)) {
      return node;
    }
  } else if (isElement && node.hasAttribute('data-oj-containerid')) { // node is non-internal component popup e.g listbox
    // eslint-disable-next-line no-param-reassign
    node = document.getElementById(node.getAttribute('data-oj-containerid'));
    return _getComponentElementByNode(node, mtAccessCompositeInternals);
  }
  return _getComponentElementByNode(node.parentNode, mtAccessCompositeInternals);
}

/**
 * Retrieves the subId of the node as
 * as part of a locator object i.e. at least
 * {subId: subIdOfNode}
 * @param {?Element} componentElement - JET component element
 * @param {?Element} node - DOM node
 * @return {any} locator - object with at least a subId
 * or null if the node does not have a subId
 * @export
 * @ojtsignore
*/
oj.Components.getSubIdByNode = function (componentElement, node) {
  return oj.Components.callComponentMethod(componentElement, 'getSubIdByNode', node);
};

/**
 * Returns the component DOM node indicated
 * by the locator parameter.
 * @param {?Element} componentElement - JET component element
 * @param {Object} locator - Object containing, at minimum,
 * a subId property, whose value is a string that identifies
 * a particular DOM node in this component.
 * @return {any} node - The DOM node located by
 * the locator, or null if none is found
 * @export
 * @ojtsignore
*/
oj.Components.getNodeBySubId = function (componentElement, locator) {
  return oj.Components.callComponentMethod(componentElement, 'getNodeBySubId', locator);
};

/**
 * Retrieves the specified option of
 * the specified JET component element
 * @param {?Element} componentElement - JET component element
 * @param {string} option - option to retrieve
 * @return {any} value of option
 * @export
 * @ojtsignore
*/
oj.Components.getComponentOption = function (componentElement, option) {
  if (!_isComponentElement(componentElement)) {
    throw new Error('node is not a component element');
  } else if (_isCompositeOrCustom(componentElement)) {
    if (componentElement.getProperty) {
      return componentElement.getProperty.call(componentElement, option);
    }
  } else {
    return oj.Components.__GetWidgetConstructor(componentElement)('option', option);
  }
  return undefined;
};

/**
 * Sets the specified option of the specified
 * JET component element to the specified value
 * @param {?Element} componentElement - JET component element
 * @param {string} option - option to set
 * @param {any} value - value to set option to
 * @return {void}
 * @export
 * @ojtsignore
*/
oj.Components.setComponentOption = function (componentElement, option, value) {
  if (!_isComponentElement(componentElement)) {
    throw new Error('node is not a component element');
  } else if (_isCompositeOrCustom(componentElement)) {
    if (componentElement.setProperty) {
      componentElement.setProperty.call(componentElement, option, value);
    }
  } else {
    oj.Components.__GetWidgetConstructor(componentElement)('option', option, value);
  }
};

/**
 * Calls the specified JET component element's method
 * with the given arguments
 * @param {?Element} componentElement - JET component element
 * @param {string} method - name of JET component element method to call
 * @param {...*} methodArguments - list of arguments to pass to method call
 * @return {any}
 * @export
 * @ojtsignore
*/
// eslint-disable-next-line no-unused-vars
oj.Components.callComponentMethod = function (componentElement, method, methodArguments) {
  if (!_isComponentElement(componentElement)) {
    throw new Error('node is not a component element');
  } else if (_isCompositeOrCustom(componentElement)) {
    if (componentElement[method]) {
      return componentElement[method].apply(componentElement, [].slice.call(arguments, 2));
    }
  } else {
    return oj.Components.__GetWidgetConstructor(componentElement)
      .apply($(componentElement), [].slice.call(arguments, 1));
  }
  return undefined;
};

/**
 * @private
 */
function _applyToComponents(subtreeRoot, jqCallback) {
  var processFunc = function () {
    var jelem = $(this);
    var names = jelem.data(_OJ_WIDGET_NAMES_DATA);
    if (names != null) {
      for (var i = 0; i < names.length; i++) {
        var instance = jelem.data('oj-' + names[i]);
        if (instance != null) {
          jqCallback(instance);
        }
      }
    }
  };

  var locator = $(subtreeRoot);

  // Include the root node itself, and not just children ()
  if (locator.hasClass(_OJ_COMPONENT_NODE_CLASS)) {
    processFunc.call(subtreeRoot);
  }

  locator.find('.' + _OJ_COMPONENT_NODE_CLASS).each(processFunc);
}

/**
 * @private
 */
function _applyHideShowToComponents(subtreeRoot, jqCallback, activateDefer) {
  // Detect hidden without forcing a layout.
  function isHidden(_node) {
    var node = _node;
    while (node) {
      if (node.nodeType === Node.DOCUMENT_NODE) {
        return false; // Walked up to document.  Not hidden
      }
      if (node.classList.contains(_OJ_SUBTREE_HIDDEN_CLASS)) {
        return true;
      }
      node = node.parentNode;
    }
    return true; // Didn't find document, so it must be detached and therefore hidden.
  }

  /**
   * Both node lists must be in document order.
   * Return new array containing nodes in 'allNodes' that are not in 'hiddenNodes'
   * @private
   */
  function filterHidden(allNodes, hiddenNodes) {
    var shownNodes = [];
    var j = 0;
    for (var i = 0; i < hiddenNodes.length; i++) {
      var hidden = hiddenNodes[i];
      while (j < allNodes.length && allNodes[j] !== hidden) {
        shownNodes.push(allNodes[j]);
        j += 1;
      }
      j += 1;
    }
    while (j < allNodes.length) {
      shownNodes.push(allNodes[j]);
      j += 1;
    }
    return shownNodes;
  }

  function processFunc(node) {
    if (jqCallback && node.classList.contains(_OJ_COMPONENT_NODE_CLASS)) {
      var jelem = $(node);
      var names = jelem.data(_OJ_WIDGET_NAMES_DATA);
      if (names != null) {
        for (var i = 0; i < names.length; i++) {
          var instance = jelem.data('oj-' + names[i]);
          if (instance != null) {
            jqCallback(instance);
          }
        }
      }
    }

    if (activateDefer && node.tagName.toLowerCase() === 'oj-defer') {
      node._activate();
    }
  }

  if (!isHidden(subtreeRoot)) {
    processFunc(subtreeRoot);

    // Create selectors for jquery components and oj-defer as needed.
    var selectors = ['.' + _OJ_COMPONENT_NODE_CLASS];

    if (activateDefer) {
      selectors.push('oj-defer');
    }

    var hiddenSelectors = [];
    selectors.forEach(function (s) {
      hiddenSelectors.push('.' + _OJ_SUBTREE_HIDDEN_CLASS + ' ' + s);
      hiddenSelectors.push('.' + _OJ_PENDING_SUBTREE_HIDDEN_CLASS + ' ' + s);
    });

    // Create assemble a selector that gets all matches and the subset that are hidden
    var selector = selectors.join(',');
    var hiddenSelector = hiddenSelectors.join(',');

    // Fetch all matching elements and those that are hidden.
    // Use the second list to filter out hidden elements.
    var allNodes = subtreeRoot.querySelectorAll(selector);
    var hiddenNodes = subtreeRoot.querySelectorAll(hiddenSelector);
    var shownNodes = filterHidden(allNodes, hiddenNodes);

    for (var i = 0; i < shownNodes.length; i++) {
      processFunc(shownNodes[i]);
    }
  }
}

/**
 * @constructor
 * @param {!Function} callback
 * @private
 */
function __ojDynamicGetter(callback) {
  this.getCallback = function () {
    return callback;
  };
}

/**
 * @ignore
 */
function _accumulateValues(target, source, valueInArray) {
  var keys = Object.keys(source);

  keys.forEach(function (key) {
    var holder = target[key] || [];
    var sourceVal = source[key];
    if (valueInArray) {
      holder = holder.concat(sourceVal);
    } else {
      holder.push(sourceVal);
    }
    // eslint-disable-next-line no-param-reassign
    target[key] = holder;
  }
  );
  return target;
}

/**
 * @ignore
 */
function _isCompositeOrCustom(node) {
  return oj.BaseCustomElementBridge.getRegistered(node.tagName);
}

/**
 * @ignore
 */
function _isJQueryUI(node) {
  return !!oj.Components.__GetWidgetConstructor(node);
}


/**
 * @ignore
 */
function _isComponentElement(node) {
  return (_isCompositeOrCustom(node) || _isJQueryUI(node));
}

/**
 * Copyright (c) 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */

/* global Promise:false, _OJ_COMPONENT_NODE_CLASS:false, _OJ_WIDGET_NAMES_DATA:false,
          __ojDynamicGetter:false, Translations:false, Logger: false*/
/* jslint browser: true*/

/**
 * @private
 */
var _OJ_TRANSLATIONS_OPTION = 'translations';

/**
 * @private
 */
var _OJ_TRANSLATIONS_PREFIX = _OJ_TRANSLATIONS_OPTION + '.';

/**
 * @private
 */
var _OJ_COMPONENT_EVENT_OVERRIDES = {
  isDefaultPrevented: function () {
    return false;
  },
  preventDefault: function () {
    this.isDefaultPrevented = _returnTrue;
  },
  stopPropagation: function () {
    this.isPropagationStopped = _returnTrue;
  },
  stopImmediatePropagation: function () {
    this.isImmediatePropagationStopped = _returnTrue;
  }
};

(function () { // BaseComponent wrapper function, to keep "private static members" private
  /**
   * @private
   */
  var _BASE_COMPONENT = 'baseComponent';
  var _STATE_CONNECTED = 0;
  var _STATE_DISCONNECTED = 1;

  // -----------------------------------------------------------------------------
  // "private static members" shared by all components
  // -----------------------------------------------------------------------------

  var _lastActiveElement;

  /**
   * @ojcomponent oj.baseComponent
   * @abstract
   * @since 0.6.0
   */
  $.widget('oj.' + _BASE_COMPONENT, {
    options: {
      /**
       * <p>There is no restriction on the order in which the JET Menu and the referencing component are initialized.  However, when specifying
       * the Menu via the HTML attribute, the referenced DOM element must be in the document at the time that the referencing component is
       * initialized.
       *
       * @ojfragment contextMenuInitOrderDoc - Decomped to fragment so Tabs, Tree, and MasonryLayout can override the fragment to convey their init order restrictions.
       * @memberof oj.baseComponent
       */
      /**
       * <p>To help determine whether it's appropriate to cancel the launch or customize the menu, the <code class="prettyprint">beforeOpen</code>
       * listener can use component API's to determine which table cell, chart item, etc., is the target of the context menu. See the JSDoc and
       * demos of the individual components for details.  Keep in mind that any such logic must work whether the context menu was launched via right-click,
       * <kbd>Shift-F10</kbd>, <kbd>Press & Hold</kbd>, or component-specific touch gesture.
       *
       * @ojfragment contextMenuTargetDoc - Decomped to fragment so components can override the fragment to convey their specific API's for this.
       * @memberof oj.baseComponent
       */
      /**
       * <p>Identifies the [JET Menu]{@link oj.ojMenu} that the component should launch as a context menu on right-click, <kbd>Shift-F10</kbd>, <kbd>Press & Hold</kbd>,
       * or component-specific gesture. If specified, the browser's native context menu will be replaced by the specified JET Menu.
       *
       * <p>The value can be an HTML element, JQ selector, JQ object, NodeList, or array of elements.  In all cases, the first indicated element is used.
       *
       * <p>To specify a JET context menu on a DOM element that is not a JET component, see the <code class="prettyprint">ojContextMenu</code> binding.
       *
       * <p>To make the page semantically accurate from the outset, applications are encouraged to specify the context menu via the standard
       * HTML5 syntax shown in the below example.  When the component is initialized, the context menu thus specified will be set on the component.
       *
       * {@ojinclude "name":"contextMenuInitOrderDoc"}
       *
       * <p>After create time, the <code class="prettyprint">contextMenu</code> option should be set via this API, not by setting the DOM attribute.
       *
       * <p>The application can register a listener for the Menu's [beforeOpen]{@link oj.ojMenu#event:beforeOpen} event.  The listener can cancel the
       * launch via <code class="prettyprint">event.preventDefault()</code>, or it can customize the menu contents by editing the menu DOM directly,
       * and then calling [refresh()]{@link oj.ojMenu#refresh} on the Menu.
       *
       * {@ojinclude "name":"contextMenuTargetDoc"}
       *
       * @ojfragment contextMenuDoc - Decomped to fragment so subclasses can extend the verbiage as needed, by ojinclude'ing this fragment and then adding their own verbiage.
       * @memberof oj.baseComponent
       */
      /**
       * {@ojinclude "name":"contextMenuDoc"}
       *
       * @ignore
       * @expose
       * @memberof oj.baseComponent
       * @instance
       * @type {Element|Array.<Element>|string|jQuery|NodeList}
       * @default <code class="prettyprint">null</code>
       *
       * @example <caption>Initialize a JET component with a context menu:</caption>
       * // via recommended HTML5 syntax:
       * &lt;div id="myComponent" contextmenu="myMenu" data-bind="ojComponent: { ... }>
       *
       * // via JET initializer (less preferred) :
       * // Foo is the component, e.g., InputText, InputNumber, Select, etc.
       * $( ".selector" ).ojFoo({ "contextMenu": "#myMenu" });
       *
       * @example <caption>Get or set the <code class="prettyprint">contextMenu</code> option, after initialization:</caption>
       * // getter
       * // Foo is the component, e.g., InputText, InputNumber, Select, etc.
       * var menu = $( ".selector" ).ojFoo( "option", "contextMenu" );
       *
       * // setter
       * // Foo is the component, e.g., InputText, InputNumber, Select, etc.
       * $( ".selector" ).ojFoo( "option", "contextMenu", ".my-marker-class" );
       *
       * @example <caption>Set a JET context menu on an ordinary HTML element:</caption>
       * &lt;a href="#" id="myAnchor" contextmenu="myMenu" data-bind="ojContextMenu: {}">Some text</a>
       */
      contextMenu: null,

      /**
       * <p>Attributes specified here will be set on the component's root DOM element at creation time.
       * This is particularly useful for components like Dialog that wrap themselves in a new root element
       * at creation time.
       *
       * <p>The supported attributes are <code class="prettyprint">id</code>, which overwrites any existing value,
       * and <code class="prettyprint">class</code> and <code class="prettyprint">style</code>, which are appended
       * to the current class and style, if any.
       *
       * <p>Setting this option after component creation has no effect.  At that time, the root element already
       * exists, and can be accessed directly via the <code class="prettyprint">widget</code> method, per the second example below.
       *
       * @example <caption>Initialize a JET component, specifying a set of attributes to be set
       * on the component's root DOM element:</caption>
       * // Foo is the component, e.g., Menu, Button, InputText, InputNumber, Select, etc.
       * $( ".selector" ).ojFoo({ "rootAttributes": {
       *   "id": "myId",
       *   "style": "max-width:100%; color:blue;",
       *   "class": "my-class"
       * }});
       *
       * @example <caption>After initialization, <code class="prettyprint">rootAttributes</code> should not be used.  It is
       * not needed at that time, as attributes of the root DOM element can simply be set directly, using
       * <code class="prettyprint">widget</code>:</caption>
       * // Foo is the component, e.g., Menu, Button, InputText, InputNumber, Select, etc.
       * $( ".selector" ).ojFoo( "widget" ).css( "height", "100px" );
       * $( ".selector" ).ojFoo( "widget" ).addClass( "my-class" );
       *
       * @ignore
       * @expose
       * @memberof oj.baseComponent
       * @instance
       * @type {?Object}
       * @default <code class="prettyprint">null</code>
       */
      rootAttributes: null,

      /**
       * <p>A collection of translated resources from the translation bundle, or <code class="prettyprint">null</code> if this
       * component has no resources.  Resources may be accessed and overridden individually or collectively, as seen in the examples.
       *
       * <p> If the component does not contain any translatable resource, the default value of this attribute will be
       * <code class="prettyprint">null</code>. If not, an object containing all resources relevant to the component.
       *
       * <p>If this component has translations, their documentation immediately follows this doc entry.
       *
       * @member
       * @name translations
       * @ojshortdesc A collection of translated resources from the translation bundle, or null if this component has no resources.
       * @memberof oj.baseComponent
       * @instance
       * @ojtranslatable
       * @type {object|null}
       *
       *
       * @example <caption>Initialize the component, overriding some translated resources and leaving the others intact:</caption>
       * &lt;!-- Using dot notation -->
       * &lt;oj-some-element translations.some-key='some value' translations.some-other-key='some other value'>&lt;/oj-some-element>
       *
       * &lt;!-- Using JSON notation -->
       * &lt;oj-some-element translations='{"someKey":"some value", "someOtherKey":"some other value"}'>&lt;/oj-some-element>
       *
       * @example <caption>Get or set the <code class="prettyprint">translations</code> property after initialization:</caption>
       * // Get one
       * var value = myComponent.translations.someKey;
       *
       * // Set one, leaving the others intact. Always use the setProperty API for
       * // subproperties rather than setting a subproperty directly.
       * myComponent.setProperty('translations.someKey', 'some value');
       *
       * // Get all
       * var values = myComponent.translations;
       *
       * // Set all.  Must list every resource key, as those not listed are lost.
       * myComponent.translations = {
       *     someKey: 'some value',
       *     someOtherKey: 'some other value'
       * };
       *
       */
      // translations property is initialized programmatically, so this top-level API doc lives in this virtual comment.
      // Translations for all components are listed and JSDoc'ed in rt\src\main\resources\nls\root\ojtranslations.js.
      // That JSDoc appears in the same generated doc page as this top-level doc.


      // Events
      /**
       * Fired whenever a supported component option changes, whether due to user interaction or programmatic
       * intervention.  If the new value is the same as the previous value, no event will be fired.  The event
       * listener will receive two parameters described below:
       *
       * @property {Event} event <code class="prettyprint">jQuery</code> event object
       * @property {Object} ui event payload
       * @property {string} ui.option the name of the option that changed.
       * @property {Object} ui.previousValue - an Object holding the previous value of the option.
       * When previousValue is not a primitive type, i.e., is an Object, it may hold the same value as
       * the value property.
       * @property {Object} ui.value - an Object holding the current value of the option.
       * @property {?Object} ui.subproperty - an Object holding information about the subproperty that changed.
       * @property {string} ui.subproperty.path - the subproperty path that changed.
       * @property {Object} ui.subproperty.previousValue - an Object holding the previous value of the subproperty.
       * @property {Object} ui.subproperty.value - an Object holding the current value of the subproperty.
       * @property {Object} ui.optionMetadata information about the option that changed
       * @property {string} ui.optionMetadata.writeback <code class="prettyprint">"shouldWrite"</code> or
       *  <code class="prettyprint">"shouldNotWrite"</code>. For use by the JET writeback mechanism;
       *  'shouldWrite' indicates that the value should be written to the observable.
       *
       * @example <caption>Initialize component with the <code class="prettyprint">optionChange</code> callback</caption>
       * // Foo is Button, InputText, etc.
       * $(".selector").ojFoo({
       *   'optionChange': function (event, ui) {}
       * });
       * @example <caption>Bind an event listener to the ojoptionchange event</caption>
       * $(".selector").on({
       *   'ojoptionchange': function (event, ui) {
       *       // verify that the component firing the event is a component of interest
       *       if ($(event.target).is(".mySelector")) {
       *           window.console.log("option that changed is: " + ui['option']);
       *       }
       *   };
       * });
       *
       * @ignore
       * @memberof oj.baseComponent
       * @expose
       * @event
       * @instance
       */
      optionChange: undefined,

      /**
       * <p>Triggered before the component is destroyed. This event cannot be canceled; the
       * component will always be destroyed regardless.
       *
       * @example <caption>Initialize component with the <code class="prettyprint">destroy</code> callback</caption>
       * // Foo is Button, InputText, etc.
       * $(".selector").ojFoo({
       *   'destroy': function (event, data) {}
       * });
       * @example <caption>Bind an event listener to the destroy event</caption>
       * $(".selector").on({
       *   'ojdestroy': function (event, data) {
       *       // verify that the component firing the event is a component of interest
       *       if ($(event.target).is(".mySelector")) {
       *           window.console.log("The DOM node id for the destroyed component is : %s", event.target.id);
       *       }
       *   };
       * });
       *
       * @ignore
       * @memberof oj.baseComponent
       * @expose
       * @event
       * @instance
       */
      destroy: undefined
    },

    // TODO: flesh out JSDoc verbiage, re: call after dom changes underneath component...
    /**
     * Refreshes the component.
     * @return {void}
     * @expose
     * @memberof oj.baseComponent
     * @instance
     */
    refresh: function () {
      this._propertyContext = null;
      // if application sets the context menu after initialization, it must refresh the component
      this._SetupContextMenu();
    },

    /**
     * <p>Overridden to save off component's default options and the options passed into the constructor (to be passed into
     * the _InitOptions() call).
     *
     * <p>This method is final. Components should instead override one or more of the overridable create-time methods
     * listed in <a href="#_ComponentCreate">_ComponentCreate</a>.
     *
     * @memberof oj.baseComponent
     * @instance
     * @private
     * @final
     */
    _createWidget: function (options, element) {
      // Save wrapper element
      if (options) {
        this.OuterWrapper = options._wrapper;
      }

      // There is no need to clone these objects since they are not modified by the _createWidget() in the base class
      this._originalDefaults = this.options || {};
      this._constructorOptions = options || {};

      this._super(options, element);
      this._AfterCreateEvent();
    },

    /**
     * <p>Reads the <code class="prettyprint">rootAttributes</code> option, and sets the root attributes on the
     * component's root DOM element.  See <a href="#rootAttributes">rootAttributes</a> for the set of supported
     * attributes and how they are handled.
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     * @throws if unsupported attributes are supplied.
     */
    _SetRootAttributes: function () {
      var value = this.options.rootAttributes;
      if (value) {
        var widget = this.widget();
        if (widget == null) {
          return;
        }

        var classValue = value.class;

        if (classValue) {
          widget.addClass(classValue);
        }

        var styleValue = value.style;

        if (styleValue) {
          var currStyle = widget.attr('style');

          if (currStyle) {
            widget.attr('style', currStyle + ';' + styleValue);
          } else {
            widget.attr('style', styleValue);
          }
        }

        // make shallow copy, remove class and style from the copy, and set all
        // remaining attrs on the element.  Currently id is the only remaining attr
        // that we support.
        value = $.extend({}, value);
        delete value.class;
        delete value.style;

        widget.attr(value);

        delete value.id; // remove the remaining supported value
        var unsupportedAttrs = Object.keys(value);
        if (unsupportedAttrs.length) {
          throw new Error('Unsupported values passed to rootAttributes option: ' +
                          unsupportedAttrs.toString());
        }
      }
    },

    /**
     * <p>This method is final in JET. Components should instead override one or more of the overridable create-time methods
     * listed in <a href="#_ComponentCreate">_ComponentCreate</a>.
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     * @final
     */
    _create: function () {
      this._SaveAttributes(this.element);
      this._InitOptions(this._originalDefaults, this._constructorOptions);

      delete this._originalDefaults;
      delete this._constructorOptions;

      this._ComponentCreate();
      this._AfterCreate();

      // allow subcomponent to setup needed resources
      // after the component is created.
      this._SetupResources();

      // Marker class for all JET components on the init node (as opposed to the outer node)
      // This marker class is used to:
      // 1) find all JET components within a subtree
      // 2) to prevent FOUC:  init nodes NOT yet having this class are hidden.
      this.element.addClass(_OJ_COMPONENT_NODE_CLASS);
    },

    /**
     * <p>This method is not used in JET. Components should instead override <a href="#_InitOptions">_InitOptions</a>.
     *
     * @method
     * @name oj.baseComponent#_getCreateOptions
     * @memberof oj.baseComponent
     * @instance
     * @protected
     * @final
     */

    /**
     * <p>This method is called before <a href="#_ComponentCreate">_ComponentCreate</a>, at which point
     * the component has not yet been rendered.  Component options should be initialized in this method,
     * so that their final values are in place when <a href="#_ComponentCreate">_ComponentCreate</a> is called.
     *
     * <p>This includes getting option values from the DOM, where applicable, and coercing option
     * values (however derived) to their appropriate data type if needed.
     *
     * <p>No work other than setting options should be done in this method.  In particular, nothing should be
     * set on the DOM until <a href="#_ComponentCreate">_ComponentCreate</a>, e.g. setting the <code class="prettyprint">disabled</code>
     * DOM attribute from the <code class="prettyprint">disabled</code> option.
     *
     * <p>A given option (like <code class="prettyprint">disabled</code>) appears in the <code class="prettyprint">constructorOptions</code>
     * param iff the app set it in the constructor:
     *
     * <ul>
     *   <li>If it appears in <code class="prettyprint">constructorOptions</code>, it should win over what's in the DOM
     *     (e.g. <code class="prettyprint">disabled</code> DOM attribute).  If for some reason you need to tweak the value
     *     that the app set, then enable writeback when doing so:
     *     <code class="prettyprint">this.option('foo', bar, {'_context': {writeback: true, internalSet: true}})</code>.</li>
     *   <li>If it doesn't appear in <code class="prettyprint">constructorOptions</code>, then that option definitely is not bound,
     *     so writeback is not needed.  So if you need to set the option (e.g. from a DOM attribute), use
     *     <code class="prettyprint">this.option('foo', bar, {'_context': {internalSet: true}})</code>.</li>
     * </ul>
     *
     * <p>Overrides of this method should call <code class="prettyprint">this._super</code> first.
     *
     * @param {!Object} originalDefaults - original default options defined on the component and its ancestors
     * @param {?Object} constructorOptions - options passed into the widget constructor
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    _InitOptions: function (originalDefaults, constructorOptions) {
      this._setupDefaultOptions(originalDefaults, constructorOptions);
      this._initContextMenuOption(constructorOptions);
    },

    /**
     * <p>All component create-time initialization lives in this method, except the logic that specifically
     * needs to live in <a href="#_InitOptions">_InitOptions</a>, <a href="#_AfterCreate">_AfterCreate</a>,
     * or <a href="#_AfterCreateEvent">_AfterCreateEvent</a>,
     * per the documentation for those methods.  All DOM creation must happen here, since the intent of
     * <a href="#_AfterCreate">_AfterCreate</a>, which is called next, is to contain superclass logic that must
     * run after that DOM is created.
     *
     * <p>Overrides of this method should call <code class="prettyprint">this._super</code> first.
     *
     * <p>Summary of create-time methods that components can override, in the order that they are called:
     *
     * <ol>
     *   <li><a href="#_InitOptions">_InitOptions</a></li>
     *   <li><a href="#_ComponentCreate">_ComponentCreate</a> (this method)</li>
     *   <li><a href="#_AfterCreate">_AfterCreate</a></li>
     *   <li>(The <code class="prettyprint">create</code> event is fired here.)</li>
     *   <li><a href="#_AfterCreateEvent">_AfterCreateEvent</a></li>
     * </ol>
     *
     * <p>For all of these methods, the contract is that overrides must call <code class="prettyprint">this._super</code> <i>first</i>, so e.g., the
     * <code class="prettyprint">_ComponentCreate</code> entry means <code class="prettyprint">baseComponent._ComponentCreate</code>,
     * then <code class="prettyprint">_ComponentCreate</code> in any intermediate subclasses, then
     * <code class="prettyprint">_ComponentCreate</code> in the leaf subclass.
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    _ComponentCreate: function () {
      // Store widget name, so that oj.Components.__GetWidgetConstructor() can get widget from the element
      _storeWidgetName(this.element, this.widgetName);

      // namespace facilitates removing activeable and hoverable handlers handlers separately
      this.activeableEventNamespace = this.eventNamespace + 'activeable';
      this.hoverableEventNamespace = this.eventNamespace + 'hoverable';
    },

    /**
     * <p>This method is called after <a href="#_ComponentCreate">_ComponentCreate</a>, but before the
     * <code class="prettyprint">create</code> event is fired.  The JET base component does
     * tasks here that must happen after the component (subclass) has created itself in its override of
     * <a href="#_ComponentCreate">_ComponentCreate</a>.  Notably, the base component handles the
     * <a href="#rootAttributes">rootAttributes</a> and <a href="#contextMenu">contextMenu</a> options here,
     * since those options operate on the component root node, which for some components is created in their override
     * of <a href="#_ComponentCreate">_ComponentCreate</a>.
     *
     * <p>Subclasses should override this method only if they have tasks that must happen after a superclass's
     * implementation of this method, e.g. tasks that must happen after the context menu is set on the component.
     *
     * <p>Overrides of this method should call <code class="prettyprint">this._super</code> first.
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    _AfterCreate: function () {
      this._SetRootAttributes(); // do first, since has no dependencies, but other stuff might care about these attrs

      // namespace facilitates removing contextMenu handlers separately, if app clears the "contextMenu" option
      this.contextMenuEventNamespace = this.eventNamespace + 'contextMenu';
      // same for activeable and hoverable handlers
      this.activeableEventNamespace = this.eventNamespace + 'activeable';
      this.hoverableEventNamespace = this.eventNamespace + 'hoverable';
    },

    /**
     * <p>This method is called after the <code class="prettyprint">create</code> event is fired.
     * Components usually should not override this method, as it is rarely correct to wait until after the
     * <code class="prettyprint">create</code> event to perform a create-time task.
     *
     * <p>An example of a correct usage of this method is [Dialog's auto-open behavior]{@link oj.ojDialog#initialVisibility},
     * which needs to happen after the <code class="prettyprint">create</code> event.
     *
     * <p>Only <i>behaviors</i> (like Dialog auto-open behavior) should occur in this method.  Component <i>initialization</i>
     * must occur earlier, before the <code class="prettyprint">create</code> event is fired, so that
     * <code class="prettyprint">create</code> listeners see a fully inited component.
     *
     * <p>Overrides of this method should call <code class="prettyprint">this._super</code> first.
     *
     * <p>Do not confuse this method with the <a href="#_AfterCreate">_AfterCreate</a> method, which is more commonly used.
     *
     * @method
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    _AfterCreateEvent: $.noop,

    /**
     * <p>JET components should almost never implement this JQUI method.  Please consult an architect if you believe you have an exception.  Reasons:
     * <ul>
     *   <li>This method is called at create time, after the <code class="prettyprint">create</code> event is fired.  It is rare
     *       for that to be the appropriate time to perform a create-time task.  For those rare cases, we have the
     *       <a href="#_AfterCreateEvent">_AfterCreateEvent</a> method, which is preferred over this method since it is called only
     *       at that time, not also at re-init time (see next).</li>
     *   <li>This method is also called at "re-init" time, i.e. when the initializer is called after the component has already been created.
     *       JET has not yet identified any desired semantics for re-initing a component.</li>
     * </ul>
     *
     * @method
     * @name oj.baseComponent#_init
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */

    //  - remove JQUI memory leaks and CSS cruft introduced in 1.12 and 1.12.1
    _setOptionClasses: function () {},
    _setOptionDisabled: function () {},
    _classes: function () {
      return '';
    },
    _removeClass: function () {
      return this;
    },
    _addClass: function () {
      return this;
    },
    _toggleClass: function () {
      return this;
    },

    /**
     * <p>Saves the element's attributes. This is called during _create.
     * <a href="#_RestoreAttributes">_RestoreAttributes</a> will restore all these attributes
     * and is called during _destroy.
     * </p>
     * <p> This base class default implementation does nothing.
     * </p>
     * <p>We also have <a href="#_SaveAllAttributes">_SaveAllAttributes</a> and
     * <a href="#_RestoreAllAttributes">_RestoreAllAttributes</a> methods
     *  that save and restore <i>all</i> the attributes on an element.
     *  Component subclasses can opt into these _SaveAllAttributes/_RestoreAllAttributes
     *  implementations by overriding _SaveAttributes and _RestoreAttributes to call
     *  _SaveAllAttributes/_RestoreAllAttributes. If the subclass wants a different implementation
     *  (like save only the 'class' attribute), it can provide the implementation itself in
     *  _SaveAttributes/_RestoreAttributes.
     *
     *
     * @param {Object} element - jQuery selection to save attributes for
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    // eslint-disable-next-line no-unused-vars
    _SaveAttributes: function (element) {
      // default implementation does nothing.
    },
    /**
     * <p>Saves all the element's attributes within an internal variable.
     * <a href="#_RestoreAllAttributes">_RestoreAllAttributes</a> will restore the attributes
     * from this internal variable.</p>
     * <p>
     * This method is final in JET.
     * Subclasses can override _RestoreAttributes and call _RestoreAllAttributes.
     * </p>
     *
     * <p>The JSON variable will be held as:
     *
     * <pre class="prettyprint">
     * <code>[
     *   {
     *   "element" : element[i],
     *   "attributes" :
     *     {
     *       attributes[m]["name"] : {"attr": attributes[m]["value"]
     *     }
     *   }
     * ]
     * </code></pre>
     *
     * @param {Object} element - jQuery selection to save attributes for
     * @memberof oj.baseComponent
     * @instance
     * @protected
     * @final
     */
    _SaveAllAttributes: function (element) {
      var self = this;
      this._savedAttributes = [];

      $.each(element, function (index, ele) {
        // need to be able to save for multiple elements
        var saveAttributes = {};
        var save = { element: ele, attributes: saveAttributes };
        var attributes = ele.attributes;

        self._savedAttributes.push(save);

        $.each(attributes, function (index2, attr) {
          // for proper access certain so called attributes should be accessed as properties
          // [i.e. required, disabled] so fetch them initially
          var attrName = attr.name;

          saveAttributes[attrName] = { attr: attr.value };
        });
      });
    },

    /**
     * <p>Gets the saved attributes for the provided element.
     *
     * <p>If you don't override <a href="#_SaveAttributes">_SaveAttributes</a> and
     * <a href="#_RestoreAttributes">_RestoreAttributes</a>, then this will return null.
     * <p>If you override _SaveAttributes to call <a href="#_SaveAllAttributes">_SaveAllAttributes</a>,
     * then this will return all the attributes.
     * If you override _SaveAttributes/_RestoreAttributes to do your own thing, then you may also have
     * to override _GetSavedAttributes to return whatever you saved if you need access to the saved
     * attributes.
     *
     * @param {Object} element - jQuery selection, should be a single entry
     * @return {Object|null} savedAttributes - attributes that were saved for this element
     * in <a href="#_SaveAttributes">_SaveAttributes</a>, or null if none were saved.
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    _GetSavedAttributes: function (element) {
      var savedAttributes = this._savedAttributes;

      // The component may not have saved any attributes. If so, return.
      if (savedAttributes === undefined) {
        return null;
      }

      var domElement = element[0];

      for (var i = 0, j = savedAttributes.length; i < j; i++) {
        var curr = savedAttributes[i];

        if (curr.element === domElement) {
          return curr.attributes;
        }
      }

      return {};
    },

    /**
     * <p>Restore the attributes saved in <a href="#_SaveAttributes">_SaveAttributes</a>.</p>
     * <p>
     * _SaveAttributes is called during _create. And _RestoreAttributes is called during _destroy.
     * </p>
     * <p> This base class default implementation does nothing.
     * </p>
     * <p>We also have <a href="#_SaveAllAttributes">_SaveAllAttributes</a> and
     * <a href="#_RestoreAllAttributes">_RestoreAllAttributes</a> methods
     *  that save and restore <i>all</i> the attributes on an element.
     *  Component subclasses can opt into these _SaveAllAttributes/_RestoreAllAttributes
     *  implementations by overriding _SaveAttributes and _RestoreAttributes to call
     *  _SaveAllAttributes/_RestoreAllAttributes. If the subclass wants a different implementation
     *  (like save only the 'class' attribute), it can provide the implementation itself in
     *  _SaveAttributes/_GetSavedAttributes/_RestoreAttributes.
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    _RestoreAttributes: function () {
      // default implementation does nothing.
    },
    /**
     * <p>Restores <i>all</i> the element's attributes which were saved in
     * <a href="#_SaveAllAttributes">_SaveAllAttributes</a>.
     * This method is final in JET.</p>
     * <p>
     * If a subclass wants to save/restore all attributes on create/destroy, then the
     * subclass can override <a href="#_SaveAttributes">_SaveAttributes</a>
     *  and call  <a href="#_SaveAllAttributes">_SaveAllAttributes</a> and also
     *  override <a href="#_RestoreAttributes">_RestoreAttributes</a>
     *  and call <a href="#_RestoreAllAttributes">_RestoreAllAttributes</a>.
     *
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     * @final
     */
    _RestoreAllAttributes: function () {
      $.each(this._savedAttributes, function (index, savedAttr) {
        var element = $(savedAttr.element);
        var attributes = savedAttr.attributes;

        // sanity check
        if (element.length === 1) {
          var currAttr = savedAttr.element.attributes;
          var removeAttr = [];
          var i;
          var j;

          // request is to remove any attributes that didn't exist previously
          // need to store the attributes in an array and remove them afterwards as otherwise there are side affects
          for (i = 0, j = currAttr.length; i < j; i++) {
            if (!(currAttr[i].name in attributes)) {
              removeAttr.push(currAttr[i].name);
            }
          }

          for (i = 0, j = removeAttr.length; i < j; i++) {
            element.removeAttr(removeAttr[i]);
          }

          var attributeKeys = Object.keys(attributes);
          for (i = 0; i < attributeKeys.length; i++) {
            var attribute = attributeKeys[i];
            element.attr(attribute, attributes[attribute].attr);
          }
        }
      });
    },

    /**
     * <p>Determines the name of the translation bundle section for this component.
     *
     * @return {string} the name of this component's translations section
     * @memberof oj.baseComponent
     * @protected
     */
    _GetTranslationsSectionName: function () {
      return this.widgetFullName;
    },


    /**
     * Compares 2 option values for equality and returns true if they are equal; false otherwise.
     * This method is called before _setOptions()/_internalSetOptions() to prevent an extra call
     * with the same values when observables are written back. Components should override this
     * method for options with non primitive writeback values like Arrays or Objects and ensure
     * their metadata has writeback properties correctly indicated.
     *
     * @param {String} option - the name of the option
     * @param {Object} value1 first value
     * @param {Object} value2 another value
     * @return {boolean}
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    _CompareOptionValues: function (option, value1, value2) {
      // We process the metadata for custom elements for writeback properties and save them on the component.
      // For jQuery syntax, components are expected to override this method to check writeback values since
      // there's not always a straightforward mapping of custom element to jQuery widget name.
      if (this._IsCustomElement() && this._getWritebackOption(option)) {
        return oj.Object.compareValues(value1, value2);
      }

      return value1 === value2;
    },


    /**
     * <p>Retrieves a translated string after inserting optional parameters.
     *
     * @param {string} key the translations resource key
     * The key is used to retrieve a format pattern from the component options, or if none
     * is found - from the translated resource bundle.
     * Tokens like {0}, {1}, {name} within the pattern will be used to define placement
     * for the optional parameters.  Token strings should not contain comma (,)
     * or space characters, since they are reserved for future format type enhancements.
     * The reserved characters within a pattern are:
     * $ { } [ ]
     * These characters will not appear in the formatted output unless they are escaped
     * with a dollar character ('$').
     *
     * @param {...string|Object|Array} var_args  - optional parameters to be inserted into the
     * translated pattern.
     *
     * If more than one var_args arguments are passed, they will be treated as an array
     * for replacing positional tokens like {0}, {1}, etc.
     * If a single argument is passed, it will be treated as a Javascript Object whose
     * keys will be matched to tokens within the pattern. Note that an Array is just
     * a special kind of such an Object.
     *
     * For backward compatibility, a var_args argument whose type is neither
     * Object or Array will be used to replace {0} in the pattern.
     *
     * @return formatted translated string or the key argument if the resource for the
     * key was not found
     *
     * @memberof oj.baseComponent
     * @instance
     * @private
     */
    // TODO: non-public methods need to start with "_".  Pinged architect, who thinks this
    // method should become protected post-V1, which would imply a capital _GetTranslatedString
    // eslint-disable-next-line no-unused-vars, camelcase
    getTranslatedString: function (key, var_args) {
      var params = {};

      if (arguments.length > 2) {
        params = Array.prototype.slice.call(arguments, 1);
      } else if (arguments.length === 2) {
        params = arguments[1];
        if (typeof params !== 'object' && !(params instanceof Array)) {
          params = [params];
        }
      }
      var pattern = this.option(_OJ_TRANSLATIONS_PREFIX + key);
      // pattern could be undefined
      return (pattern == null) ? key :
        Translations.applyParameters(pattern.toString(), params);
    },

    // Subclasses should doc their sub-id's in the Sub-ID's section, via the ojsubid tag, not by overriding
    // and extending this method doc, which should remain general purpose.
    /**
     * <p>Returns the DOM node indicated by the <code class="prettyprint">locator</code> parameter.
     *
     * <p>If the <code class="prettyprint">locator</code> or its <code class="prettyprint">subId</code> is
     * <code class="prettyprint">null</code>, then this method returns this element.
     *
     * <p>If a non-null <code class="prettyprint">subId</code> is provided but no corresponding node
     * can be located, then this method returns <code class="prettyprint">null</code>.
     *
     * <p>This method is intended for use in test automation only, and should not be used in a production environment.
     *
     * @expose
     * @memberof oj.baseComponent
     * @instance
     * @ignore
     *
     * @param {Object} locator An Object containing, at minimum, a <code class="prettyprint">subId</code>
     * property, whose value is a string that identifies a particular DOM node in this element.
     *
     * <p>If this component has any subIds, then they are documented in the
     * <a href="#subids-section">Sub-ID's</a> section of this document.
     *
     * <p>Some components may support additional fields of the
     * <code class="prettyprint">locator</code> Object, to further specify the desired node.
     *
     * @returns {Element|null} The DOM node located by the
     * <code class="prettyprint">locator</code>, or <code class="prettyprint">null</code> if none is found.
     *
     * @example <caption>Get the node for a certain subId:</caption>
     * var node = myComponent.getNodeBySubId({'subId': 'oj-some-sub-id'});
     */
    getNodeBySubId: function (locator) {
      if (locator == null || locator.subId == null) {
        return this.element ? this.element[0] : null;
      }

      // Non-null locators have to be handled by the component subclasses
      return null;
    },

    /**
     * <p>Returns the subId string for the given DOM node in this element.  For details, see
     * <a href="#getNodeBySubId">getNodeBySubId</a> and the <a href="#subids-section">Sub-ID's</a>
     * section of this document.
     *
     * <p>This method is intended for use in test automation only, and should not be used in a production environment.
     *
     * @ojfragment getSubIdByNodeDesc
     * @memberof oj.baseComponent
     */
    /**
     * DOM node in this element
     *
     * @ojfragment getSubIdByNodeNodeParam
     * @memberof oj.baseComponent
     */
    /**
     * The subId for the DOM node, or <code class="prettyprint">null</code> if none is found.
     *
     * @ojfragment getSubIdByNodeReturn
     * @memberof oj.baseComponent
     */
    /**
     * Get the subId for a certain DOM node:
     *
     * @ojfragment getSubIdByNodeCaption
     * @memberof oj.baseComponent
     */
    /**
     * var locator = myComponent.getSubIdByNode(nodeInsideElement);
     *
     * @ojfragment getSubIdByNodeExample
     * @memberof oj.baseComponent
     */

    // While a subclass could technically extend the verbiage by adding its own verbiage after the ojinclude,
    // please doc sub-id's in the subid's section, not by extending this method doc.
    /**
     * {@ojinclude "name":"getSubIdByNodeDesc"}
     *
     * @expose
     * @memberof oj.baseComponent
     * @instance
     * @ignore
     *
     * @param {!Element} node {@ojinclude "name":"getSubIdByNodeNodeParam"}
     * @returns {Object|null} {@ojinclude "name":"getSubIdByNodeReturn"}
     *
     * @example <caption>{@ojinclude "name":"getSubIdByNodeCaption"}</caption>
     * {@ojinclude "name":"getSubIdByNodeExample"}
     */
    // eslint-disable-next-line no-unused-vars
    getSubIdByNode: function (node) {
      return null;
    },

    // Overridden to set oj-hover and oj-focus classes
    // TODO: Move JSDoc from subclasses to here.  Don't include above internal comment.  Make at-final.
    destroy: function () {
      if (this._IsCustomElement()) {
        throw new Error('destroy cannot be called on a custom element');
      }

      // Fire 'destroy' event
      this._trigger('destroy');

      // Since jQuery event listeners get removed before the destroy() method whne jQuery.clean() cleans up the subtree,
      // we need to fire a custom DOM event as well. This will allow component binding to execute destroy callbacks for
      // custom bindings and managed attributes.

      oj.DomUtils.dispatchEvent(this.element[0], new CustomEvent('_ojDestroy'));

      // allow subcomponent to release resources they hold.
      this._ReleaseResources();

      this._super();

      // remove hover and active listeners
      //    this.widget().off(this.eventNamespace);

      // clean up states
      this.element.removeClass(_OJ_COMPONENT_NODE_CLASS);
      this.widget().removeClass('oj-disabled');

      // pass init node (this.element), not root node if different (this.widget()), since all elements in
      // the root node subtree but not the init node subtree should have been removed by the call to _super.
      this._removeStateClasses(this.element);

      _removeWidgetName(this.element, this.widgetName);

      this._RestoreAttributes();

      // TODO: move this to _RestoreAttributes?
      if (this._initialCmDomAttr) {
        this.element.attr('contextmenu', this._initialCmDomAttr);
      } else {
        this.element.removeAttr('contextmenu');
      }

      this._propertyContext = null;
    },

    /*
     * Internal notes:
     * Overridden to pass extra flags to _setOption
     * param {...Object} var_args - key (or map), value, flags
     */
    /**
     * <p>This method has several overloads, which get and set component options and their fields.  The functionality is unchanged from
     * that provided by JQUI.  See the examples for details on each overload.
     *
     * @ignore
     * @expose
     * @memberof oj.baseComponent
     * @instance
     * @final
     *
     * @param {string|Object=} optionName the option name (string, first two overloads), or the map (Object, last overload).
     *        Omitted in the third overload.
     * @param {Object=} value a value to set for the option.  Second overload only.
     * @return {Object|undefined} The getter overloads return the retrieved value(s).  When called via the public jQuery syntax, the setter overloads
     *         return the object on which they were called, to facilitate method chaining.
     *
     * @example <caption>First overload: get one option:
     * <p>This overload accepts a (possibly dot-separated) <code class="prettyprint">optionName</code> param as a string, and returns
     * the current value of that option.</caption>
     * var isDisabled = $( ".selector" ).ojFoo( "option", "disabled" ); // Foo is Button, Menu, etc.
     *
     * // For object-valued options, dot notation can be used to get the value of a field or nested field.
     * var startIcon = $( ".selector" ).ojButton( "option", "icons.start" ); // icons is object with "start" field
     *
     * @example <caption>Second overload: set one option:
     * <p>This overload accepts two params: a (possibly dot-separated) <code class="prettyprint">optionName</code> string, and a new value to
     * which that option will be set.</caption>
     * $( ".selector" ).ojFoo( "option", "disabled", true ); // Foo is Button, Menu, etc.
     *
     * // For object-valued options, dot notation can be used to set the value
     * // of a field or nested field, without altering the rest of the object.
     * $( ".selector" ).ojButton( "option", "icons.start", myStartIcon ); // icons is object with "start" field
     *
     * @example <caption>Third overload: get all options:
     * <p>This overload accepts no params, and returns a map of key/value pairs representing all the component
     * options and their values.</caption>
     * var options = $( ".selector" ).ojFoo( "option" ); // Foo is Button, Menu, etc.
     *
     * @example <caption>Fourth overload: set one or more options:
     * <p>This overload accepts a single map of option-value pairs to set on the component.  Unlike the first two
     * overloads, dot notation cannot be used.</caption>
     * $( ".selector" ).ojFoo( "option", { disabled: true, bar: 42 } ); // Foo is Button, Menu, etc.
     */
    option: function (optionName, value) { // actually varArgs per comment above the JSDoc, but GCC warns unless matches the @param that we wish to doc
      if (arguments.length === 0) {
        // don't return a reference to the internal hash
        return $.widget.extend({}, this.options);
      }

      var key = arguments[0];

      var options = key;
      var subkey = null;

      var flags = {};
      var i;

      if (typeof key === 'string') {
        // handle nested keys, e.g., "foo.bar" => { foo: { bar: ___ } }
        options = {};
        var parts = key.split('.');
        key = parts.shift();
        if (parts.length) {
          subkey = parts.join('.');

          var curOption;
          try {
            // Inform dynamic getters that the subkey is being set
            if (arguments.length > 1) {
              this._settingNestedKey = subkey;
            }

            curOption = $.widget.extend({}, this.options[key]);
            options[key] = curOption;
          } finally {
            this._settingNestedKey = null;
          }

          for (i = 0; i < parts.length - 1; i++) {
            curOption[parts[i]] = curOption[parts[i]] || {};
            curOption = curOption[parts[i]];
          }

          key = parts.pop();
          if (arguments.length === 1) {
            return curOption[key] === undefined ? null : curOption[key];
          }

          curOption[key] = value;
        } else {
          if (arguments.length === 1) {
            return this.options[key] === undefined ? null : this.options[key];
          }
          options[key] = value;
        }

        flags = arguments[2] || flags;
      } else {
        flags = arguments[1] || flags;
      }

      // Store subkey on the flags to let _setOption() know that dot notation was used
      if (subkey != null) {
        var subprop = {
          path: optionName,
          value: value
        };
        flags = $.widget.extend({}, flags, { subkey: subkey, subproperty: subprop });
      }

      var context = flags ? flags._context : null;
      var internalSet = context ? context.internalSet : false;

      // This method can be called twice with the same value for writeback properties
      // so we need to go through the options object and only pass through the changed values
      var newOptions = {};
      var optionKeys = Object.keys(options);
      for (i = 0; i < optionKeys.length; i++) {
        var option = optionKeys[i];
        var newValue = options[option];
        var oldValue = this.options[option];
        // The changed flag is set when components have updated an object or array value in place
        var changed = flags && flags.changed;
        if (changed || !this._CompareOptionValues(option, oldValue, newValue)) {
          newOptions[option] = newValue;
        } else if (this._IsCustomElement()) {
          Logger.info(oj.BaseCustomElementBridge.getElementInfo(this.element[0]) + ": Ignoring property set for property '" +
            option + "' with same value.");
        }
      }

      if (Object.keys(newOptions).length > 0) {
        // Avoid _setOption() calls for internal sets, since component's _setOption()
        // and setOptions() overrides do not expect to be called in that case
        if (internalSet) {
          this._internalSetOptions(newOptions, flags);
        } else {
          this._setOptions(newOptions, flags);
        }
      }

      return this;
    },

    /**
     * option() calls this rather than _setOption() if the caller was internal.
     *
     * @memberof oj.baseComponent
     * @instance
     * @private
     */
    _internalSetOptions: function (options, flags) {
      var keys = Object.keys(options);
      for (var i = 0; i < keys.length; i++) {
        var key = keys[i];
        var value = options[key];
        var oldValue = this.options[key];
        this.options[key] = value;
        this._optionChanged(key, value, oldValue, flags);
      }
    },

    /**
     * <p>Overridden to pass extra flags to _setOption.
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     * @ignore
     */
    _setOptions: function (options, flags) {
      var keys = Object.keys(options);
      for (var i = 0; i < keys.length; i++) {
        var key = keys[i];
        var value = options[key];
        this._setOption(key, value, flags);
      }

      return this;
    },

    /**
     * Overridden to set oj-hover and oj-focus classes.
     * Components should not call this method directly, but instead call option().
     * @memberof oj.baseComponent
     * @instance
     * @protected
     * @ignore
     */
    _setOption: function (key, value, flags) {
      var originalValue = this.options[key];

      if (key === 'disabled') {
        // The JQUI superclass method has hard-coded style classes in the 'if key === "disabled"' block, so unfortunately
        // we must copy that logic here with updated style classes, and NOT call _super() for the disabled case.
        // TBD: keep this logic updated if superclass method changes.
        this.options[key] = value;

        // TBD: widget() is not always the thing that should have aria-disabled on it.  E.g. for the checkbox/radio flavors of ojButton,
        // widget() returns the root node, but aria-disabled belongs on the <input>.  We fixed this JQUI bug in ojButton by having ojButton
        // override this method to remove it from the root node and add it to the input.  Would be better for each component to know which
        // element to apply that to, e.g. an overridable method returning that element, or copying "hoverable" paradigm if appropriate.
        // In the cases where this.element is different than widget(), this.element is more likely to be the right thing, so maybe change
        // default to that.
        // Update: this issue is getting even more awkward now that we have "effectively disabled".  Probably need to refactor this code!
        this.widget()
          .toggleClass('oj-disabled', !!value)
          .attr('aria-disabled', value);

        if (value) {
          this._removeStateClasses(this.widget());
        }
      } else {
        try {
          var subkey = (flags == null) ? null : flags.subkey;
          if (subkey != null) {
            this._settingNestedKey = subkey;
          }

          this._super(key, value);
        } finally {
          this._settingNestedKey = null;
        }

        // if contextMenu option wasn't set before, we'll need to start detect gesture
        if (key === 'contextMenu') {
          this._SetupContextMenu();
        }
      }

      this._optionChanged(key, value, originalValue, flags);

      return this;
    },

    /**
     * @memberof oj.baseComponent
     * @instance
     * @private
     */
    _optionChanged: function (key, value, originalValue, flags) {
      // Assume that all values are different from originalValues as we do equality
      // checking before calling this method or trust components that have set the
      // 'changed' flag to indicate that they updated in place for an Object or Array
      // since we won't see any difference for those cases.
      // var changed = false;
      var context = null;

      var writeback = false;
      var readOnly = false;
      var originalEvent = null;
      var updatedFrom = 'external';

      var optionMetadata = null;

      var extraData;

      if (flags) {
        context = flags._context;

        if (context) {
          // Skip firing an option changed event for certain cases like
          // custom element default value sets where the bridge interprets an
          // application undefined value set to the property default defined in the
          // metadata.
          if (context.skipEvent) {
            return;
          }

          originalEvent = context.originalEvent;
          writeback = (context.writeback === undefined) ? originalEvent != null :
            context.writeback;
          readOnly = context.readOnly;
          optionMetadata = context.optionMetadata;
          extraData = context.extraData;
          if (context.internalSet) {
            updatedFrom = 'internal';
          }
        }
      }

      optionMetadata = optionMetadata || {};
      optionMetadata.writeback = writeback ? 'shouldWrite' : 'shouldNotWrite';

      if (readOnly) {
        optionMetadata.readOnly = true;
      }

      var optionChangeData = {
        option: key,
        previousValue: originalValue,
        value: value,
        optionMetadata: optionMetadata,
        updatedFrom: updatedFrom
      };

      var subkey = (flags == null) ? null : flags.subkey;
      // Walk previousValue object and find the subproperty previousValue
      if (subkey) {
        var subprops = subkey.split('.');
        var originalSubpropValue = originalValue;
        subprops.forEach(function (subprop) {
          if (!originalSubpropValue) {
            return;
          }
          originalSubpropValue = originalSubpropValue[subprop];
        });
        var subproperty = flags.subproperty;
        subproperty.previousValue = originalSubpropValue;
        optionChangeData.subproperty = subproperty;
      }

      if (extraData != null) {
        optionChangeData = $.extend({}, extraData, optionChangeData);
      }

      this._trigger('optionChange', originalEvent, optionChangeData);
    },

    /**
     * <p>Sets up needed resources for this component, for example, add
     * listeners. This is called during _create.
     * <a href="#_ReleaseResources">_ReleaseResources</a> will release resources
     * help by this component, and is called during destroy.
     * </p>
     *  Component subclasses can opt in by overriding _SetupResources and
     *   _ReleaseResources.
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    _SetupResources: function () {
      this._SetupContextMenu();
    },

    /**
     * <p>Release resources held by this component, for example, remove
     * listeners. This is called during destroy.
     * <a href="#_SetupResources">_SetupResources</a> will set up resources
     * needed by this component, and is called during _create.
     * </p>
     *  Component subclasses can opt in by overriding _SetupResources and
     *   _ReleaseResources.
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    _ReleaseResources: function () {
      this._ReleaseContextMenu();
    },

    /**
     * <p>Overridden to change the way the component events are treating original events:
     *
     * 1) preventDefault(), stopPropagation() and stopImmediatePropagation() no longer invoke
     *    the corresponding methods on the .originalEvent
     * 2) Properties of the .originalEvent are no longer copied to the new event being
     *    triggered
     *
     * @param {string} type - component event type
     * @param {?Object} event - original event
     * @param {Object=} data - event data
     * @return {boolean} true if the default action has not been prevented, false otherwise
     *
     * @private
     */
    _trigger: function (type, event, data) {
      return this._trigger2(type, event, data).proceed;
    },

    /**
     * <p>Same as _trigger(), but returns an object containing both the "prevented" status and the event.
     *
     * <p>This is useful for event chaining, so that the returned event (e.g. Menu's select event) can be
     * passed as the originalEvent of a subsequent event caused by the first (e.g. Menu's close event).
     *
     * @param {string} type - component event type
     * @param {?Object} event - original event
     * @param {Object=} data - event data
     * @return {!{proceed: boolean, event: ($.Event|CustomEvent)}}
     *     proceed is true if the default action has not been prevented, false otherwise
     *     event is the new event that was triggered
     *
     * @private
     */
    _trigger2: function (type, event, data) {
      var eventData = data || {};

      if (this._IsCustomElement()) {
        return this._triggerCustomEvent(type, event, eventData);
      }

      var callback = this.options[type];

      var jqEvent = $.Event(event, _OJ_COMPONENT_EVENT_OVERRIDES);
      jqEvent.type = (this.widgetEventPrefix + type).toLowerCase();

      // the original event may come from any element
      // so we need to reset the target on the new event
      jqEvent.target = this.element[0];

      this.element.trigger(jqEvent, eventData);

      return {
        proceed: !(($.isFunction(callback) &&
                    callback.apply(this.element[0], [jqEvent].concat(eventData)) === false) ||
                   jqEvent.isDefaultPrevented()),
        event: jqEvent
      };
    },

    /**
     * <p>Fires a CustomEvent instead of a jQuery event for when this component is created as a custom element.
     *
     * @param {string} type - component event type
     * @param {?Object} event - original event
     * @param {Object} data - event data
     * @return {!{proceed: boolean, event: CustomEvent}}
     *     proceed is true if the default action has not been prevented, false otherwise
     *     event is the new event that was triggered
     *
     * @private
     */
    _triggerCustomEvent: function (type, event, data) {
      var eventName;
      var detail = {};
      var bubbles;
      var cancelable;
      var rootElement = this._getRootElement();

      if (type === 'optionChange') {
        var property = oj.CustomElementBridge.getPropertyForAlias(rootElement, data.option);
        if (!oj.CustomElementBridge.isKnownProperty(rootElement, property)) {
          return { proceed: true, event: null };
        }
        eventName = oj.__AttributeUtils.propertyNameToChangeEventType(property);

        // Copy over component specific optionChange event properties, promoting those exposed
        // in the optionMetadata to the top level alongside value/previousValue/subproperty.
        var keys = Object.keys(data);
        for (var i = 0; i < keys.length; i++) {
          var key = keys[i];
          if (key !== 'option') {
            if (key === 'optionMetadata') {
              // Do not expose optionMetadata for custom element property change events
              // Instead, promote any component specific metadata to the top level.
              var metaKeys = Object.keys(data[key]);
              for (var j = 0; j < metaKeys.length; j++) {
                var metaKey = metaKeys[j];
                if (metaKey !== 'writeback' && metaKey !== 'component') {
                  detail[metaKey] = data[key][metaKey];
                }
              }
            } else {
              detail[key] = data[key];
            }
          }
        }
      } else {
        eventName = oj.__AttributeUtils.eventTriggerToEventType(type);
        if (!oj.CustomElementBridge.isKnownEvent(rootElement, eventName)) {
          return { proceed: true, event: null };
        }
        bubbles = true;
        cancelable = true;
        detail = this._resolveJQueryObjects(data);
      }
      if (event) {
        detail.originalEvent = event instanceof $.Event ? event.originalEvent : event;
      }
      var params = { detail: detail };
      if (bubbles) {
        params.bubbles = true;
      }
      if (cancelable) {
        params.cancelable = true;
      }

      var customEvent = new CustomEvent(eventName, params);
      rootElement.dispatchEvent(customEvent);
      return { proceed: !customEvent.defaultPrevented, event: customEvent };
    },

    /**
     * <p>Creates a shallow copy of the passed in object where any top-level JQuery objects have been resolved to their underlying objects.
     * @param {Object} data the original object
     * @return {Object} the resolved object copy
     *
     * @private
     */
    _resolveJQueryObjects: function (data) {
      var resolved = oj.CollectionUtils.copyInto({}, data);
      var keys = Object.keys(resolved);
      for (var k = 0; k < keys.length; k++) {
        var key = keys[k];
        var val = resolved[key];
        if (val && val instanceof $) {
          if (val.length === 0) {
            resolved[key] = null;
          } else if (val.length === 1) {
            resolved[key] = val[0];
          } else {
            resolved = val.toArray();
          }
        }
      }
      return resolved;
    },


    /**
     * <p>Sets contextMenu option from DOM if option not set.
     *
     * <p>Do not override.  To be called only from _InitOptions().
     *
     * @memberof oj.baseComponent
     * @instance
     * @private
     * @final
     */
    _initContextMenuOption: function (constructorOptions) {
      var contextMenu = this.element.attr('contextmenu');

      this._initialCmDomAttr = contextMenu; // TODO: remove this after the _RestoreAttributes() call in destroy() is uncommented

      if (contextMenu && !('contextMenu' in constructorOptions)) { // if app set DOM attr but not option, then set the option from the DOM
        this.option('contextMenu',
                    document.getElementById(contextMenu),
                    { _context: { internalSet: true } }); // writeback not needed since "not in constructorOptions" means "not bound"
      }
    },

    /**
     * Handler
     * @param {Element} contextMenu root element of context menu
     * @param {Event} event the dom event
     * @param {string} eventType the type of event
     * @private
     */
    _handleContextMenuGesture: function (contextMenu, event, eventType) {
      // For components like Button where "effectively disabled" --> "not focusable", keyboard CM launch is impossible, so
      // allowing right-click access would be an a11y issue.  If there's ever a need to enable this for focusable effectively
      // disabled components, we can always replace the _IsEffectivelyDisabled() call with a new protected method whose
      // baseComponent impl returns _IsEffectivelyDisabled().
      if (this._IsEffectivelyDisabled()) {
        return;
      }

      // contextMenu should reference the latest one from the scope
      var menu;
      if (contextMenu.tagName === 'OJ-MENU') {
        menu = contextMenu;
      } else {
        var constructor = oj.Components.__GetWidgetConstructor(contextMenu, 'ojMenu');
        menu = constructor && constructor('instance');
        if (!menu) {
          throw new Error('Invalid JET Menu.'); // keeping old behavior
        }
      }

      this._NotifyContextMenuGesture(menu, event, eventType);

      // todo: modify NotifyContextMenuGesture contract so we don't need to check visible
      if ($(contextMenu).is(':visible')) {
        event.preventDefault(); // don't show native context menu
      }
    },

    /**
     * <p>Call this method from _SetupResources().  It sets up listeners needed to detect context menu gestures.
     *
     * <p>We don't look for the menu until context menu gesture has been detected on the first launch,
     * so that the menu needn't be inited before this component.
     *
     * <p>If needed, override <code class="prettyprint">_NotifyContextMenuGesture()</code>, not this private method.
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    _SetupContextMenu: function () {
      var contextMenu = this._GetContextMenu();
      if (!contextMenu) {
        contextMenu = this._GetDefaultContextMenu();
      }

      if (contextMenu && this._contextMenuGestureInit === undefined) {
        this._contextMenuGestureInit = contextMenu;

        var self = this;
        oj.GestureUtils.startDetectContextMenuGesture(
          this.widget()[0], function (event, eventType) {
            self._handleContextMenuGesture(contextMenu, event, eventType);
          });
      }
    },

    /**
     * <p>This method removes contextMenu functionality from the component and specified menu.
     *
     * @memberof oj.baseComponent
     * @instance
     * @private
     */
    _ReleaseContextMenu: function () {
      this._contextMenuGestureInit = undefined;

      // access menu elem directly, rather than using the widget() of the Menu component, so listener is cleared even if component no longer exists.
      // $(contextMenuOption).off( this.contextMenuEventNamespace );
      oj.GestureUtils.stopDetectContextMenuGesture(this.widget()[0]);
    },

    /**
     * Helper method to retrieve the context menu element
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    _GetContextMenu: function () {
      if (this._IsCustomElement()) {
        var slotMap = oj.BaseCustomElementBridge.getSlotMap(this._getRootElement());
        var slot = slotMap.contextMenu;
        if (slot && slot.length > 0) {
          return slot[0];
        }
      } else if (this.options.contextMenu) {
        return $(this.options.contextMenu).first()[0];
      }

      return null;
    },

    /**
     * <p>When the <a href="#contextMenu">contextMenu</a> option is set, this method is called when the user invokes the context menu via
     * the default gestures: right-click, <kbd>Press & Hold</kbd>, and <kbd>Shift-F10</kbd>.  Components should not call this method directly.
     *
     * <p>The default implementation simply calls <a href="#_OpenContextMenu">this._OpenContextMenu(event, eventType)</a>.
     * Overrides of this method should call that same method, perhaps with additional params, not [menu.open()]{@link oj.ojMenu#open}.
     *
     * <p>This method may be overridden by components needing to do things like the following:
     *
     * <ul>
     * <li>Customize the [launcher]{@link oj.ojMenu#openOptions.launcher} or [position]{@link oj.ojMenu#openOptions.position} passed to
     * <a href="#_OpenContextMenu">_OpenContextMenu()</a>.  See that method for guidance on these customizations.</li>
     *
     * <li>Customize the menu contents.  E.g. some components need to enable/disable built-in commands like <kbd>Cut</kbd> and <kbd>Paste</kbd>,
     * based on state at launch time.</li>
     *
     * <li>Bail out in some cases.  E.g. components with UX approval to use <kbd>PressHoldRelease</kbd> rather than <kbd>Press & Hold</kbd> can override this method
     * to say <code class="prettyprint">if (eventType !== "touch") this._OpenContextMenu(event, eventType);</code>.  When those components
     * detect the alternate context menu gesture (e.g. <kbd>PressHoldRelease</kbd>), that separate listener should call <a href="#_OpenContextMenu">this._OpenContextMenu()</a>,
     * not this method (<code class="prettyprint">_NotifyContextMenuGesture()</code>), and not [menu.open()]{@link oj.ojMenu#open}.  </li>
     * </ul>
     *
     * <p>Components needing to do per-launch setup like the above tasks should do so in an override of this method, <i>not</i> in
     * a [beforeOpen]{@link oj.ojMenu#event:beforeOpen} listener or an <a href="#_OpenContextMenu">_OpenContextMenu()</a> override.
     * This is discussed more fully <a href="#_OpenContextMenu">here</a>.
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     *
     * @param {!Object} menu The JET Menu to open as a context menu.  Always non-<code class="prettyprint">null</code>.
     * @param {!Event} event What triggered the menu launch.  Always non-<code class="prettyprint">null</code>.
     * @param {string} eventType "mouse", "touch", or "keyboard".  Never <code class="prettyprint">null</code>.
     */
    _NotifyContextMenuGesture: function (menu, event, eventType) {
      this._OpenContextMenu(event, eventType);
    },

    /**
     * <p>The only correct way for a component to open its context menu is by calling this method, not by calling [Menu.open()]{@link oj.ojMenu#open} or
     * <a href="#_NotifyContextMenuGesture">_NotifyContextMenuGesture()</a>.  This method should be called in two cases:
     *
     * <ul>
     * <li>This method is called by <a href="#_NotifyContextMenuGesture">_NotifyContextMenuGesture()</a> and its overrides.  That method is
     * called when the baseComponent detects the default context menu gestures: right-click, <kbd>Press & Hold</kbd>, and <kbd>Shift-F10</kbd>.</li>
     *
     * <li>Components with UX-approved support for alternate context menu gestures like <kbd>PressHoldRelease</kbd> should call this method directly
     * when those gestures are detected.</li>
     * </ul>
     *
     * <p>Components needing to customize how the context menu is launched, or do any per-launch setup, should do so in the caller of this method,
     * (which is one of the two callers listed above), often by customizing the params passed to this method
     * (<code class="prettyprint">_OpenContextMenu</code>) per the guidance below.  This setup should <i>not</i> be done in the following ways:
     *
     * <ul>
     * <li>Components should not perform setup in a [beforeOpen]{@link oj.ojMenu#event:beforeOpen} listener, as this can cause a race
     * condition where behavior depends on who got their listener registered first: the component or the app.  The only correct component use
     * of a <code class="prettyprint">beforeOpen</code> listener is when there's a need to detect whether <i>something else</i> launched the menu.</li>
     *
     * <li>Components should not override this method (<code class="prettyprint">_OpenContextMenu</code>), as this method is final.  Instead, customize
     * the params that are passed to it.</li>
     * </ul>
     *
     * <p><b>Guidance on setting OpenOptions fields:</b>
     *
     * <p><b>Launcher:</b>
     *
     * <p>Depending on individual component needs, any focusable element within the component can be the appropriate
     * [launcher]{@link oj.ojMenu#openOptions.launcher} for this launch.
     *
     * <p>Browser focus returns to the launcher on menu dismissal, so the launcher must at least be focusable.  Typically a tabbable (not just
     * focusable) element is safer, since it just focuses something the user could have focused on their own.
     *
     * <p>By default (i.e. if <code class="prettyprint">openOptions</code> is not passed, or if it lacks a <code class="prettyprint">launcher</code>
     * field), the component init node is used as the launcher for this launch.  If that is not focusable or is suboptimal for a given
     * component, that component should pass something else.  E.g. components with a "roving tabstop" (like Toolbar) should typically choose the
     * current tabstop as their launcher.
     *
     * <p>The [:focusable]{@link http://api.jqueryui.com/focusable-selector/} and [:tabbable]{@link http://api.jqueryui.com/tabbable-selector/} selectors
     * may come in handy for choosing a launcher, e.g. something like <code class="prettyprint">this.widget().find(".my-class:tabbable").first()</code>.
     *
     * <p><b>Position:</b>
     *
     * <p>By default, this method applies [positioning]{@link oj.ojMenu#openOptions.position} that differs from Menu's default in the following ways:
     * (The specific settings are subject to change.)
     *
     * <ul>
     * <li>For mouse and touch events, the menu is positioned relative to the event, not the launcher.</li>
     *
     * <li>For touch events, <code class="prettyprint">"my"</code> is set to <code class="prettyprint">"start>40 center"</code>,
     * to avoid having the context menu obscured by the user's finger.</li>
     * </ul>
     *
     * <p>Usually, if <code class="prettyprint">position</code> needs to be customized at all, the only thing that needs changing is its
     * <code class="prettyprint">"of"</code> field, and only for keyboard launches (since mouse/touch launches should almost certainly keep
     * the default <code class="prettyprint">"event"</code> positioning).  This situation arises anytime the element relative to which the menu
     * should be positioned for keyboard launches is different than the <code class="prettyprint">launcher</code> element (the element to which
     * focus should be returned upon dismissal).  For this case, <code class="prettyprint">{ "position": {"of": eventType==="keyboard" ? someElement : "event"} }</code>
     * can be passed as the <code class="prettyprint">openOptions</code> param.
     *
     * <p>Be careful not to clobber useful defaults by specifying too much.  E.g. if you only want to customize <code class="prettyprint">"of"</code>,
     * don't pass other fields like <code class="prettyprint">"my"</code>, since your value will be used for all modalities (mouse, touch, keyboard),
     * replacing the modality-specific defaults that are usually correct.  Likewise, don't forget the
     * <code class="prettyprint">eventType==="keyboard"</code> check if you only want to customize <code class="prettyprint">"of"</code> for keyboard launches.
     *
     * <p><b>InitialFocus:</b>
     *
     * <p>This method forces [initialFocus]{@link oj.ojMenu#openOptions.initialFocus} to <code class="prettyprint">"menu"</code> for this
     * launch, so the caller needn't specify it.
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     * @final
     *
     * @param {!Event} event What triggered the context menu launch.  Must be non-<code class="prettyprint">null</code>.
     * @param {string} eventType "mouse", "touch", or "keyboard".  Must be non-<code class="prettyprint">null</code>.  Passed explicitly since caller
     *        knows what it's listening for, and since events like <code class="prettyprint">contextmenu</code> and
     *        <code class="prettyprint">click</code> can be generated by various input modalities, making it potentially error-prone for
     *        this method to determine how they were generated.
     * @param {Object=} openOptions Options to merge with this method's defaults, which are discussed above.  The result will be passed to
     *        [Menu.open()]{@link oj.ojMenu#open}.  May be <code class="prettyprint">null</code> or omitted.  See also the
     *        <code class="prettyprint">shallow</code> param.
     * @param {Object=} submenuOpenOptions Options to be passed through to [Menu.open()]{@link oj.ojMenu#open}.  May be <code class="prettyprint">null</code>
     *        or omitted.
     * @param {boolean=} shallow Whether to perform a deep or shallow merge of <code class="prettyprint">openOptions</code> with this method's default
     *        value.  The default and most commonly correct / useful value is <code class="prettyprint">false</code>.
     *
     *        <ul>
     *        <li>If <code class="prettyprint">true</code>, a shallow merge is performed, meaning that the caller's <code class="prettyprint">position</code>
     *        object, if passed, will completely replace this method's default <code class="prettyprint">position</code> object.</li>
     *
     *        <li>If <code class="prettyprint">false</code> or omitted, a deep merge is performed.  For example, if the caller wishes to tweak
     *        <code class="prettyprint">position.of</code> while keeping this method's defaults for <code class="prettyprint">position.my</code>,
     *        <code class="prettyprint">position.at</code>, etc., it can pass <code class="prettyprint">{"of": anOfValue}</code> as the
     *        <code class="prettyprint">position</code> value.</li>
     *        </ul>
     *
     *        <p>The <code class="prettyprint">shallow</code> param is n/a for <code class="prettyprint">submenuOpenOptions</code>, since this method doesn't
     *        apply any defaults to that.  (It's a direct pass-through.)
     */
    _OpenContextMenu: function (event, eventType, openOptions, submenuOpenOptions, shallow) {
      var contextMenuNode = this._GetContextMenu();
      if (!contextMenuNode) {
        // the context menu node could have been currently opened with another launcher
        if (this._contextMenuGestureInit && $(this._contextMenuGestureInit).is(':visible')) {
          contextMenuNode = this._contextMenuGestureInit;
        } else {
          contextMenuNode = this._GetDefaultContextMenu();
        }
      }

      if (contextMenuNode) {
        // Note: our touch positioning is similar to that of the iOS touch callout (bubble with "Open in New Tab", etc.), which is offset from the pressHold location as follows:
        // - to the right, vertically centered.  (by default)
        // - to the left, vertically centered.  (if fits better)
        // - above or below, horizontally centered.  (if fits better)
        // An offset like 40 prevents it from opening right under your finger, and is similar to iOS's offset.  It also prevents the issue (on iOS7 at least)
        // where touchend after the pressHold can dismiss the CM b/c the menu gets the touchend.

        var position = {
          mouse: {
            my: 'start top',
            at: 'start bottom',
            of: event,
            collision: 'flipfit'
          },
          touch: {
            my: 'start>40 center',
            at: 'start bottom',
            of: event,
            collision: 'flipfit'
          },
          keyboard: {
            my: 'start top',
            at: 'start bottom',
            of: 'launcher',
            collision: 'flipfit'
          }
        };

        var defaults = { launcher: this.element, position: position[eventType] }; // used for fields caller omitted
        var forcedOptions = { initialFocus: 'menu' };

        var mergedOpenOptions = (shallow) ?
          $.extend(defaults, openOptions, forcedOptions) :
          $.extend(true, defaults, openOptions, forcedOptions);

        contextMenuNode.__openingContextMenu = true; // Hack.  See todo on this ivar in Menu.open().
        if (contextMenuNode.tagName === 'OJ-MENU') {
          contextMenuNode.open(event, mergedOpenOptions, submenuOpenOptions);
        } else {
          var constructor = oj.Components.__GetWidgetConstructor(contextMenuNode, 'ojMenu');
          var menu = constructor && constructor('instance');
          menu.open(event, mergedOpenOptions, submenuOpenOptions);
          // Open is immediate for jquery UI menus.
          // Win FF will fire a contextmenu event on shift+F10 long after the keypress was prevented.
          // jquery ui needs immediate focus to the menu on open. The contextmenu event is fired on
          // the menu versus the launcher. This logic prevents the context menu event within a 50ms
          // window after the menu is open.
          var eatEventHandler = function (e) { e.preventDefault(); };
          contextMenuNode.addEventListener('contextmenu', eatEventHandler);
          window.setTimeout(function () { contextMenuNode.removeEventListener('contextmenu', eatEventHandler); }, 50);
        }
        contextMenuNode.__openingContextMenu = false;
      }
    },

    /**
     * Retrieve the default context menu.
     * @return {Element|null} the root element for the default context menu, or null if there is no default context menu.  The default is null.
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    _GetDefaultContextMenu: function () {
      return null;
    },

    /**
     * <p>Removes the <code class="prettyprint">oj-hover</code>, <code class="prettyprint">oj-focus</code>,
     * <code class="prettyprint">oj-focus-highlight</code>, and
     * <code class="prettyprint">oj-active</code> classes from the specified element and its subtree.
     *
     * @memberof oj.baseComponent
     * @instance
     * @private
     *
     * @param {!jQuery} element The element from whose subtree the 3 classes should be removed
     */
    _removeStateClasses: function (element) {
      element.removeClass('oj-hover oj-focus oj-focus-highlight oj-active');
      element.find('.oj-hover').removeClass('oj-hover');
      element.find('.oj-focus').removeClass('oj-focus');
      element.find('.oj-focus-highlight').removeClass('oj-focus-highlight');
      element.find('.oj-active').removeClass('oj-active');
    },


    /**
     * @private
     * @return {boolean} true if there is no touch detected within the last 500 ms
     */
    _isRealMouseEvent: function () {
      return !oj.DomUtils.recentTouchEnd();
    },

    /**
     * Add mouse listners to toggle oj-hover class
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected

     * @param {(!Object|!jQuery)} options This param can either be the element
     * (convenience syntax for callers needing to
     *   specify only the element(s) that would
     *   otherwise have been passed as <code class="prettyprint">options.element</code>)
     *   or an object supporting the following fields:
     * @param {jQuery} options.element The element(s) to receive the <code class="prettyprint">oj-hover</code> class on hover
     *   Required if <code class="prettyprint">afterToggle</code> is specified.
     * @param {?function(string)} options.afterToggle Optional callback function called each time the hover classes have been toggled,
     *   after the toggle.  The string "mouseenter" or "mouseleave" is passed, indicating whether the classes were added or removed.
     *   Components with consistency requirements, such as "<code class="prettyprint">oj-default</code> must be applied iff no state classes
     *   such as <code class="prettyprint">oj-hover</code> are applied," can enforce those rules in this callback.
     * @see #_RemoveHoverable
     */
    _AddHoverable: function (options) {
      var element;

      if ($.isPlainObject(options)) {
        element = options.element;
      } else {
        element = options;
        // eslint-disable-next-line no-param-reassign
        options = {};
      }

      var afterToggle = options.afterToggle || $.noop;
      var markerClass = 'oj-hover';

      element.on('mouseenter' + this.hoverableEventNamespace,
                 this._hoverStartHandler.bind(this, afterToggle))
        .on('mouseleave' + this.hoverableEventNamespace,
            this._hoverAndActiveEndHandler.bind(this, markerClass, afterToggle));
    },

    /**
     * Remove mouse listners that were registered in _AddHoverable
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     * @param {!jQuery} element The same element passed to _AddHoverable
     * @see #_AddHoverable
     */
    _RemoveHoverable: function (element) {
      if (element) {
        element.off(this.hoverableEventNamespace);
      }
    },

    /**
     * Add touch and mouse listeners to toggle oj-active class
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected

     * @param {(!Object|!jQuery)} options This parameter can either be the element
     * (convenience syntax for callers needing to
     *   specify only the element(s) that would
     *   otherwise have been passed as <code class="prettyprint">options.element</code>)
     *   or an object supporting the following fields:
     * @param {jQuery} options.element The element(s) to receive the
     * <code class="prettyprint">oj-active</code> class on active
     *   Required if <code class="prettyprint">afterToggle</code> is specified.
     * @param {?function(string)} options.afterToggle Optional callback function called each time
     *   the active classes have been toggled, after the toggle.  The event.type string is passed
     *   and indicates whether the classes were added or removed. The active classes are added on
     *   "touchstart" or "mousedown" or "mouseenter" and the active classes are removed on
     *   "touchend" or "touchcancel" or "mouseup" or "mouseleave".
     *   Components with consistency requirements, such as
     *   "<code class="prettyprint">oj-default</code> must be applied iff no state classes
     *   such as <code class="prettyprint">oj-active</code> are applied,"
     *   can enforce those rules in this callback.
     * @see #_RemoveActiveable
     */
    _AddActiveable: function (options) {
      var element;

      if ($.isPlainObject(options)) {
        element = options.element;
      } else {
        element = options;
        // eslint-disable-next-line no-param-reassign
        options = {};
      }

      var afterToggle = options.afterToggle || $.noop;
      var markerClass = 'oj-active';

      if (oj.DomUtils.isTouchSupported()) {
        element.on('touchstart' + this.activeableEventNamespace,
                   this._activeStartHandler.bind(this, afterToggle))
          .on('touchend' + this.activeableEventNamespace + ' ' +
              'touchcancel' + this.activeableEventNamespace,
              this._hoverAndActiveEndHandler.bind(this, markerClass, afterToggle));
      }

      element.on('mousedown' + this.activeableEventNamespace,
                 this._activeStartHandler.bind(this, afterToggle))
        .on('mouseup' + this.activeableEventNamespace,
            this._hoverAndActiveEndHandler.bind(this, markerClass, afterToggle))
        // mouseenter/mouseleave is for the case where you mousedown, then move mouse
        // out of element, then move mouse back. We want oj-active to disappear when you move
        // outside and reappear when you move back.
        .on('mouseenter' + this.activeableEventNamespace,
            this._activeStartHandler.bind(this, afterToggle))
        .on('mouseleave' + this.activeableEventNamespace,
            this._hoverAndActiveEndHandler.bind(this, markerClass, afterToggle));
    },

    /**
     * Remove touch and mouse listeners that were registered in _AddActiveable
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     * @param {!jQuery} element The same element passed to _AddActiveable
     * @see #_AddActiveable
     */
    _RemoveActiveable: function (element) {
      if (element) {
        element.off(this.activeableEventNamespace);
        _lastActiveElement = null;
      }
    },

    /**
     * Add oj-active style class and call the afterToggleFunction.
     * Set _lastActiveElement to event.currentTarget on mousedown. Clear it on mouseup.
     * _lastActiveElement is used to remove and add back oj-active on mouseleave and mouseenter
     * if the mouse stays down.
     * @private
     */
    _activeStartHandler: function (afterToggleFunction, event) {
      var elem = $(event.currentTarget);

      // do nothing on mouseenter if _lastActiveElement ivar is not the currentTarget or child of
      // currentTarget. Checking for the child is needed in case there are two nested
      // dom nodes that have set_AddActiveable. Since events bubble, the _lastActiveElement will
      // be the outermost dom that got the mousedown event.
      if (event.type === 'mouseenter' && !this._isTargetInActiveElement(event.currentTarget)) {
        return;
      }

      // do this for either touchstart or real mouse events, but not mouse compatibility event
      if (!elem.hasClass('oj-disabled') &&
          (event.type === 'touchstart' || this._isRealMouseEvent(event))) {
        elem.addClass('oj-active');
        afterToggleFunction(event.type);

        // If we get mousedown on the element, we want oj-active to be removed on mouseleave
        // and added back on mouseenter if the mouse stays down.
        if (event.type === 'mousedown') {
          _lastActiveElement = event.currentTarget;

          this.document.one('mouseup', function () {
            _lastActiveElement = null;
          });
        }
      }
    },

    /**
     * @private
     */
    _hoverStartHandler: function (afterToggleFunction, event) {
      // do this for real mouseenter, but not mouse compatibility event
      var elem = $(event.currentTarget);
      if (!elem.hasClass('oj-disabled') && this._isRealMouseEvent(event)) {
        elem.addClass('oj-hover');
        afterToggleFunction(event.type);
      }
    },

    /**
     * Remove markerClass and call the afterToggleFunction.
     * @private
     */
    _hoverAndActiveEndHandler: function (markerClass, afterToggleFunction, event) {
      // for oj-active we don't care about mouseleave unless it was triggered when _lastActiveElement
      // was set on the currentTarget. (see _activeStartHandler). If that's not the case, return.
      if (markerClass === 'oj-active' && event.type === 'mouseleave' &&
          !this._isTargetInActiveElement(event.currentTarget)) {
        return;
      }
      $(event.currentTarget).removeClass(markerClass);
      afterToggleFunction(event.type);
    },
    /**
     * Returns true if the event target is _lastActiveElement or a child of _lastActiveElement.
     * Checking for the child is needed in case there are two nested dom nodes that have set
     * _AddActiveable. For example, inputDateTime -> the trigger root container and the trigger icon.
     * We are only keeping track of the _lastActiveElement, which means the ancestor element will
     * be stored in _lastActiveElement because the mouseleave event bubbles.
     * @private
     */
    _isTargetInActiveElement: function (currentTarget) {
      return (_lastActiveElement === currentTarget ||
              (_lastActiveElement != null && $.contains(_lastActiveElement, currentTarget)));
    },

    // We no longer use _hoverable, but should still override it to ensure the JQUI impl is not called.
    _hoverable: function () {},

    // The internal JSDoc of the DomUtils version of this API refers to this doc, so if changes are made here, that doc
    // must be updated as needed.
    /**
     * <p>Sets JET's "focus" CSS classes when the element is focused and removes them when focus is lost.
     *
     * <p>The <code class="prettyprint">oj-focus</code> class is set on all focuses.
     *
     * <p>Some components additionally have an <code class="prettyprint">oj-focus-highlight</code> class, which applies a focus
     * indicator that is appropriate on a subset of the occasions that <code class="prettyprint">oj-focus</code> is appropriate.
     * Those components should pass <code class="prettyprint">true</code> for the <code class="prettyprint">applyHighlight</code>
     * param, in which case the <code class="prettyprint">oj-focus-highlight</code> class is set if appropriate given the
     * current focus highlight policy.
     *
     * <h5>Focus highlight policy</h5>
     *
     * <p>The focus highlight policy supports the 3 values listed below.  By default, it is retrieved from the
     * <code class="prettyprint">$focusHighlightPolicy</code> SASS variable, shared by many components and patterns.  Components
     * with different needs, including those exposing a component-specific SASS variable or other API for this, should see the
     * <code class="prettyprint">getFocusHighlightPolicy</code> parameter below.
     *
     * Valid focus highlight policies:
     *
     * <table class="generic-table">
     *   <thead>
     *     <tr>
     *       <th>Policy</th>
     *       <th>Description</th>
     *     </tr>
     *   </thead>
     *   <tbody>
     *     <tr>
     *       <td>"nonPointer"</td>
     *       <td>Indicates that the component should apply the <code class="prettyprint">oj-focus-highlight</code>
     *           class only for focuses not resulting from pointer (touch or mouse) interaction.  (In the built-in themes, the
     *           SASS variable defaults to this value.)</td>
     *     </tr>
     *     <tr>
     *       <td>"all"</td>
     *       <td>Indicates that the component should apply the class for all focuses.</td>
     *     </tr>
     *     <tr>
     *       <td>"none"</td>
     *       <td>Indicates that the component should never apply the class, because the application has taken responsibility
     *           for applying the class when needed for accessibility.</td>
     *     </tr>
     *   </tbody>
     * </table>
     *
     * <h5>Toggling the classes</h5>
     *
     * <p>Components that toggle these focus classes outside of this API must maintain the invariant that
     * <code class="prettyprint">oj-focus-highlight</code> is applied to a given element in a (not necessarily strict) subset
     * of cases that <code class="prettyprint">oj-focus</code> is applied to that element.
     *
     * <p>Typically the specified element should be within the component subtree, in which case the classes will
     * automatically be removed from the element when the component is destroyed, when its <code class="prettyprint">disabled</code>
     * option is set to true, and when <code class="prettyprint">_NotifyDetached()</code> is called.
     *
     * <p>As a minor exception, for components that wrap themselves in a new root node at create time, if the specified
     * element is within the root node's subtree but not within the init node's subtree, then at destroy time only, the
     * classes will not be removed, since <code class="prettyprint">destroy()</code> is expected to remove such nodes.
     *
     * <p>If the element is NOT in the component subtree, then the caller is responsible for removing the classes at the
     * times listed above.
     *
     * <h5>Listeners</h5>
     *
     * <p>If <code class="prettyprint">setupHandlers</code> is not passed, or if <code class="prettyprint">setupHandlers</code>
     * is passed and uses <code class="prettyprint">_on</code> to register its listeners as seen in the example, then
     * the listeners are not invoked when the component is disabled, and the listeners are automatically cleaned up when the
     * component is destroyed.  Otherwise, the caller is responsible for ensuring that the disabled state is handled correctly,
     * and removing the listeners at destroy time.
     *
     * <h5>Related API's</h5>
     *
     * <p>Non-component internal callers should see oj.DomUtils.makeFocusable().  Per its JSDoc (unpublished; see the source), it
     * has a couple of additional usage considerations.
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     *
     * @param {(!Object|!jQuery)} options This param can either be the element (convenience syntax for callers needing to
     *   specify only the element(s) that would otherwise have been passed as <code class="prettyprint">options.element</code>)
     *   or an object supporting the following fields:
     * @param {jQuery} options.element The element(s) to receive the <code class="prettyprint">oj-focus</code> classes on focus.
     *   Required if <code class="prettyprint">setupHandlers</code> not passed; ignored otherwise.
     * @param {boolean} options.applyHighlight <code class="prettyprint">true</code> if the <code class="prettyprint">oj-focus-highlight</code>
     *   class should be applied when appropriate.  <code class="prettyprint">false</code> or omitted if that class should never be applied.
     * @param {?function(string)} options.afterToggle Optional callback function called each time the focus classes have been toggled,
     *   after the toggle.  The
     *   string "focusin" or "focusout" is passed, indicating whether the classes were added or removed.  Components
     *   with consistency requirements, such as "<code class="prettyprint">oj-default</code> must be applied iff no state classes such
     *   as <code class="prettyprint">oj-focus</code> are applied," can enforce those rules in this callback.
     * @param {?function()} options.getFocusHighlightPolicy Optional if <code class="prettyprint">applyHighlight</code> is
     *   <code class="prettyprint">true</code>; ignored otherwise.  Components with a component-specific focus policy
     *   mechanism should pass a function that always returns one of the three valid values listed above, keeping in mind
     *   that this method can be called on every focus.  See the example.
     * @param {?function()} options.recentPointer Relevant iff <code class="prettyprint">applyHighlight</code> is
     *   <code class="prettyprint">true</code> and the focus highlight policy is <code class="prettyprint">"nonPointer"</code>;
     *   ignored otherwise.  Recent pointer activity is considered to have occurred if (a) a mouse button or finger has
     *   recently been down or up, or (b) this optional callback function returns true.  Components wishing to additionally take into
     *   account (say) recent pointer <i>movements</i> can supply a function returning true if those movements have been detected,
     *   keeping in mind that this method can be called on every focus.  See the example.
     * @param {?function(function(!jQuery),function(!jQuery))} options.setupHandlers Can be omitted by components whose focus
     *   classes need to be added and removed on focusin and focusout, respectively.  Components needing to add/remove those
     *   classes in response to other events should specify this parameter, which is called once, immediately.  See the examples.
     *
     * @example <caption>Opt into the highlight behavior, and specify a function to be called every time the classes are toggled:</caption>
     * var self = this;
     * this._focusable({
     *     'element': this.element,
     *     'applyHighlight': true,
     *     'afterToggle' : function() {
     *         self._toggleDefaultClasses();
     *     }
     * });
     *
     * @example <caption>Arrange for mouse movement to be considered <u>in addition to</u> mouse/finger up/down.
     *   Also supply a component-specific focusHighlightPolicy:</caption>
     * var self = this;
     * this._focusable({
     *     'element': someElement,
     *     'applyHighlight': true,
     *     'recentPointer' : function() {
     *         // A timestamp-based approach avoids the risk of getting stuck in an inaccessible
     *         // state if (say) mouseenter is not followed by mouseleave for some reason.
     *         var millisSincePointerMove = Date.now() - _myPointerMoveTimestamp;
     *         var isRecent = millisSincePointerMove < myThreshold;
     *         return isRecent;
     *     },
     *     'getFocusHighlightPolicy' : function() {
     *         // Return the value of a component-specific SASS $variable, component option, or other
     *         // component-specific mechanism, either "all", "none", or "nonPointer".  SASS variables
     *         // should be pulled into JS once statically on load, not per-instance or per-focus.
     *     }
     * });
     *
     * @example <caption>Add/remove the focus classes in response to events other than focusin/focusout:</caption>
     * var self = this;
     * this._focusable({
     *     'applyHighlight': myBooleanValue,
     *     'setupHandlers': function( focusInHandler, focusOutHandler) {
     *         self._on( self.element, {
     *             // This example uses focus/blur listeners, which don't bubble, rather than the
     *             // default focusin/focusout (which bubble).  This is useful when one focusable
     *             // element is a descendant of another.
     *             focus: function( event ) {
     *                 focusInHandler($( event.currentTarget ));
     *             },
     *             blur: function( event ) {
     *                 focusOutHandler($( event.currentTarget ));
     *             }
     *         });
     *     }
     * });
     *
     * @example <caption>Alternate usage of <code class="prettyprint">setupHandlers</code>, which simply stashes the
     *   handlers so they can be called from the component's existing handlers:</caption>
     * var self = this;
     * this._focusable({
     *     'applyHighlight': myBooleanValue,
     *     'setupHandlers': function( focusInHandler, focusOutHandler) {
     *         self._focusInHandler = focusInHandler;
     *         self._focusOutHandler = focusOutHandler;
     *     }
     * });
     */
    _focusable: function (options) {
      if (!$.isPlainObject(options)) {
        // eslint-disable-next-line no-param-reassign
        options = { element: options };
      }

      // eslint-disable-next-line no-param-reassign
      options.component = this;
      oj.DomUtils.makeFocusable(options);
    },

    /**
     * Remove all listener references that were attached to the element.
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    _UnregisterChildNode: function (element) {
      if (element) {
        $(element).off(this.eventNamespace);
        var bindings = this.bindings;
        if (bindings) {
          this.bindings = $(bindings.not(element));
        }
      }
    },

    /**
     * <p>Determines whether the component is LTR or RTL.
     *
     * <p>Component responsibilities:
     *
     * <ul>
     * <li>All components must determine directionality exclusively by calling this protected superclass method.
     *     (So that any future updates to the logic can be made in this one place.)</li>
     * <li>Components that need to know the directionality must call this method at create-time
     *     and from <code class="prettyprint">refresh()</code>, and cache the value.
     * <li>Components should not call this at other times, and should instead use the cached value.  (This avoids constant DOM
     *     queries, and avoids any future issues with component reparenting (i.e. popups) if support for directional islands is added.)</li>
     * </ul>
     *
     * <p>App responsibilities:
     *
     * <ul>
     * <li>The app specifies directionality by setting the HTML <code class="prettyprint">"dir"</code> attribute on the
     *     <code class="prettyprint">&lt;html></code> node.  When omitted, the default is <code class="prettyprint">"ltr"</code>.
     *     (Per-component directionality / directional islands are not currently supported due to inadequate CSS support.)</li>
     * <li>As with any DOM change, the app must <code class="prettyprint">refresh()</code> the component if the directionality changes dynamically.
     *   (This provides a hook for component housekeeping, and allows caching.)</li>
     * </ul>
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     * @return {string} the reading direction, either <code class="prettyprint">"ltr"</code> or <code class="prettyprint">"rtl"</code>
     * @default <code class="prettyprint">"ltr"</code>
     */
    _GetReadingDirection: function () {
      return oj.DomUtils.getReadingDirection();
    },

    /**
     * <p>Notifies the component that its subtree has been connected to the document programmatically after the component has
     * been created.
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    _NotifyAttached: function () {
      this._propertyContext = null;
    },

    /**
     * <p>Notifies the component that its subtree has been removed from the document programmatically after the component has
     * been created.
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    _NotifyDetached: function () {
      this._propertyContext = null;
      this._removeStateClasses(this.widget());
    },

    /**
     * <p>Notifies the component that its subtree is initially visible after the component has
     * been created.
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    _NotifyInitShown: function () {

    },

    /**
     * <p>Notifies the component that its subtree has been made visible programmatically after the component has
     * been created.
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    _NotifyShown: function () {

    },

    /**
     * <p>Notifies the component that its subtree has been made hidden programmatically after the component has
     * been created.
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    _NotifyHidden: function () {

    },

    /**
     * <p>Determines whether this component is effectively disabled, i.e. it has its 'disabled' attribute set to true
     * or it has been disabled by its ancestor component.
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     * @return {boolean} true if the component has been effectively disabled, false otherwise
     */
    _IsEffectivelyDisabled: function () {
      return !!((this.options.disabled || this._ancestorDisabled));
    },

    /**
     * <p>Sets the ancestor-provided disabled state on this component.
     *
     * @memberof oj.baseComponent
     * @instance
     * @private
     * @param {boolean} disabled - true if this component is being disabled by its ancestor component, false otherwise
     */
    __setAncestorComponentDisabled: function (disabled) {
      this._ancestorDisabled = disabled;
    },


    /**
     * @memberof oj.baseComponent
     * @instance
     * @private
     */
    _getTranslationSectionLoader: function () {
      var sectionNames = [];

      var self = this;

      var index = 0;

      this._traverseWidgetHierarchy(function (proto) {
        // retrive translation section name for the widget and all of its ancestors

        // Since _GetTranslationsSectionName() is a protected method, we can only call it on the widget instance.
        // For superclases, we will assume that their section names can only be their full widget name

        var name = (index === 0) ? self._GetTranslationsSectionName() : proto.widgetFullName;
        index += 1;

        var section = Translations.getComponentTranslations(name);

        if (section != null && !$.isEmptyObject(section)) {
          sectionNames.push(name);
        }
      });

      var count = sectionNames.length;

      if (count > 0) {
        return function () {
          // Optimize for the most common case where superclasses do not define translations
          if (count === 1) {
            return Translations.getComponentTranslations(sectionNames[0]);
          }

          var trs = {};

          for (var i = count - 1; i >= 0; i--) {
            $.widget.extend(trs, Translations.getComponentTranslations(sectionNames[i]));
          }

          return trs;
        };
      }
      return null;
    },

    /**
     * @memberof oj.baseComponent
     * @instance
     * @private
     */
    _getDynamicPropertyContext: function () {
      if (!this._propertyContext) {
        var c = {};
        this._propertyContext = c;
        var element = this.element[0];
        c.containers = _getSpecialContainerNames(element);
        c.element = element;
        c.isCustomElement = this._IsCustomElement();
      }
      return this._propertyContext;
    },

    /**
     * @memberof oj.baseComponent
     * @instance
     * @private
     */
    _setupDefaultOptions: function (originalDefaults, constructorOptions) {
      var options = this.options;

      // Load component translations
      var translationLoader = this._getTranslationSectionLoader();

      var currVal = constructorOptions[_OJ_TRANSLATIONS_OPTION];

      if (translationLoader != null && (currVal === undefined || $.isPlainObject(currVal))) {
        _defineDynamicProperty(this, undefined, constructorOptions[_OJ_TRANSLATIONS_OPTION],
                               options, _OJ_TRANSLATIONS_OPTION, translationLoader);
      }


      // Load options specified with oj.Components.setDefaultOptions()
      this._loadGlobalDefaultOptions(originalDefaults, constructorOptions);
    },

    /**
     * @memberof oj.baseComponent
     * @instance
     * @private
     */
    _loadGlobalDefaultOptions: function (originalDefaults, constructorOptions) {
      var options = this.options;

      var widgetHierNames = [];

      // walk up the widget hierarchy
      this._traverseWidgetHierarchy(
        function (proto) {
          widgetHierNames.push(proto.widgetName);
        }
      );

      widgetHierNames.push('default');


      // get properties applicable to this component
      var defaults = oj.Components.__getDefaultOptions(widgetHierNames);

      if ($.isEmptyObject(defaults)) {
        return;
      }

      var self = this;

      var contextCallback = function () {
        return self._getDynamicPropertyContext();
      };

      var props = Object.keys(defaults);
      for (var i = 0; i < props.length; i++) {
        var prop = props[i];
        var val = constructorOptions[prop];

        if (val === undefined || $.isPlainObject(val)) {
          var defaultValueList = defaults[prop];
          if (defaultValueList) {
            var callback = _getCompoundDynamicGetter(defaultValueList);
            if (callback) {
              _defineDynamicProperty(this, originalDefaults[prop], val, options,
                                     prop, callback, contextCallback);
            } else {
              var list = [originalDefaults[prop]].concat(defaultValueList);
              list.push(val);
              options[prop] = _mergeOptionLayers(list);
            }
          }
        }
      }
    },

    /**
     * @memberof oj.baseComponent
     * @instance
     * @private
     */
    _traverseWidgetHierarchy: function (callback) {
      var proto = this.constructor.prototype;
      while (proto != null && proto.namespace === 'oj') {
        callback(proto);
        proto = Object.getPrototypeOf(proto);
      }
    },

    /**
     * @memberof oj.baseComponent
     * @instance
     * @private
     */
    _getRootElement: function () {
      return this.OuterWrapper || this.element[0];
    },

    /**
     * Determines whether the component is being rendered as a custom element.
     * @return {boolean} True if the component is being rendered as a custom element
     *
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    _IsCustomElement: function () {
      return oj.BaseCustomElementBridge.getRegistered(this._getRootElement().tagName) != null;
    },

    /**
     * Prepares a custom renderer context object for either the JQuery or custom element syntax,
     * removing and exposing keys as needed.
     * @param {Object} context The renderer context object.
     * @return {Object} The cleaned up renderer context.
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    _FixRendererContext: function (context) {
      if (this._IsCustomElement()) {
        // Do a shallow copy to avoid setter/getters from being lost
        var contextCopy = oj.CollectionUtils.copyInto({}, context);
        // remove component or widget constructor references and expose element reference instead
        delete contextCopy.component;
        contextCopy.componentElement = this._getRootElement();
        return contextCopy;
      }
      return context;
    },

    /**
     * Returns a wrapper function for custom elements that converts an object
     * returned by a custom renderer into an old format supported by widgets
     * @param {Function} origRenderer Renderer function called to create custom content
     * @return {Function} A wrapper function that will used to convert result into toolkit format
     * @protected
     * @memberof oj.baseComponent
     */
    _WrapCustomElementRenderer: function (origRenderer) {
      if (this._IsCustomElement() && typeof origRenderer === 'function') {
        var customRenderer = function (context) {
          var obj = origRenderer(context);
          return obj && obj.insert ? obj.insert : null;
        };
        return customRenderer;
      }
      return origRenderer;
    },

    /**
     * Stores a map of writeback options that we reference during option comparison.
     * Package private method called from the CustomElementBridge.
     * @param  {Object} options The writeback options map
     * @memberof oj.baseComponent
     * @instance
     * @private
     */
    __saveWritebackOptions: function (options) {
      this._writebackOptions = options;
    },

    /**
     * Returns true if an option should be written back.
     * @param  {string} option The option to lookup
     * @memberof oj.baseComponent
     * @instance
     * @private
     */
    _getWritebackOption: function (option) {
      if (this._writebackOptions && this._writebackOptions[option]) {
        return true;
      }
      return false;
    },

    /**
     * Called by oj.Components.subtreeAttached and will only call _NotifyAttached
     * for non custom elements. Custom elements are notified when they are
     * attached from the DOM so oj.Components.subtreeAttached is unnecessary.
     * @memberof oj.baseComponent
     * @instance
     * @private
     */
    __handleSubtreeAttached: function () {
      if (!this._IsCustomElement()) {
        this._NotifyAttached();
      }
    },

    /**
     * Called by oj.Components.subtreeAttached and will only call _NotifyDetached
     * for non custom elements. Custom elements are notified when they are
     * detached from the DOM so oj.Components.subtreeDetached is unnecessary.
     * @memberof oj.baseComponent
     * @instance
     * @private
     */
    __handleSubtreeDetached: function () {
      if (!this._IsCustomElement()) {
        this._NotifyDetached();
      }
    },

    /**
     * Whether special handling is needed for connected and disconnected ops
     * @return {boolean} returns true if the component wants to suppress disconnect and connect operations that happened
     *                   in quick succession, since they can be very expensive.  Returns false otherwise, which is the default.
     * @memberof oj.baseComponent
     * @protected
     */
    _VerifyConnectedForSetup: function () {
      return false;
    },

    /**
     * Called by the CustomElementBridge when the custom element is attached
     * to the DOM.
     * @memberof oj.baseComponent
     * @instance
     * @private
     */
    __handleConnected: function () {
      this._NotifyAttached();
      if (!this.__delayConnectDisconnect(_STATE_CONNECTED)) {
        this._SetupResources();
      }
    },

    /**
     * Called by the CustomElementBridge when the custom element is detached
     * from the DOM.
     * @memberof oj.baseComponent
     * @instance
     * @private
     */
    __handleDisconnected: function () {
      // note that when it is delayed, then NotifyDetached would be called before ReleaseResources
      // this is fine for all the components that will use delayed disconnect, will need to re-visit if that is not the case.
      if (!this.__delayConnectDisconnect(_STATE_DISCONNECTED)) {
        this._ReleaseResources();
      }
      this._NotifyDetached();
    },

    /**
     * Delay the call to SetupResources and ReleaseResources as part of connected and disconnected.
     * See _verifyConnectedForSetup method for details.
     * @return {boolean} true if SetupResources/ReleaseResources has been delayed, false otherwise.
     * @memberof oj.baseComponent
     * @instance
     * @private
     */
    __delayConnectDisconnect: function (state) {
      if (!this._VerifyConnectedForSetup()) {
        return false;
      }

      if (this.connectedState === undefined) {
        var self = this;
        Promise.resolve().then(function () {
          if (self.connectedState === state) {
            if (state === _STATE_CONNECTED) {
              self._SetupResources();
            } else {
              self._ReleaseResources();
            }
          }
          self.connectedState = undefined;
        });
      }
      this.connectedState = state;

      return true;
    },

   /**
    * Method called by the CustomElementBridge to notify the component of changes to
    * any watched attributes registered in its metadata extension._WATCHED_ATTRS property.
    * @param {string} attr The name of the watched attribute
    * @param {string} oldValue The old attribute value
    * @param {string} newValue The new attribute value
    * @memberof oj.baseComponent
    * @instance
    * @private
    */
    __handleWatchedAttribute: function (attr, oldValue, newValue) {
      this._WatchedAttributeChanged(attr, oldValue, newValue);
    },

   /**
    * Method for components to override in order to handle changes to watched attributes.
    * @param {string} attr The name of the watched attribute
    * @param {string} oldValue The old attribute value
    * @param {string} newValue The new attribute value
    * @memberof oj.baseComponent
    * @instance
    * @protected
    */
    // eslint-disable-next-line no-unused-vars
    _WatchedAttributeChanged: function (attr, oldValue, newValue) {},

    /**
     * Method called by the CustomElementBridge to get the element to call focus on for this custom element
     * which can be the root custom element or an HTML element like an input or select.
     * @return {Element}
     * @memberof oj.baseComponent
     * @instance
     * @private
     */
    __getFocusElement: function () {
      return this.GetFocusElement();
    },

    /**
     * Returns the current focusable element for this component which can be the root custom element
     * or an HTML element like an input or select.
     * @return {Element}
     * @memberof oj.baseComponent
     * @instance
     * @protected
     */
    GetFocusElement: function () {
      return this.element[0];
    },

    /**
     * <p>The following CSS classes can be applied by the page author as needed.
     *
     * @ojfragment ojStylingDocIntro - For use in the Styling section of the JSDoc, above the table of CSS classes .
     * @memberof oj.baseComponent
     */

    /**
     * Class
     *
     * @ojfragment ojStylingDocClassHeader - For use in the "Class" <th> in the Styling section of the JSDoc.
     * @memberof oj.baseComponent
     */

    /**
     * Can be applied to
     *
     * @ojfragment ojStylingDocApplyHeader - For use in the "Can be applied to" <th> in the Styling section of the JSDoc.
     * @memberof oj.baseComponent
     */

    /**
     * Description
     *
     * @ojfragment ojStylingDocDescriptionHeader - For use in the "Description" <th> in the Styling section of the JSDoc.
     * @memberof oj.baseComponent
     */

    /**
     * Example
     *
     * @ojfragment ojStylingDocExampleHeader - For use in the "Example" <th> in the Styling section of the JSDoc.
     * @memberof oj.baseComponent
     */

    /**
     * Under normal circumstances this class is applied automatically. It is documented here for the rare cases that an app
     * developer needs per-instance control.
     *
     * <p>The <code class="prettyprint">oj-focus-highlight</code> class applies focus styling that may not be desirable when
     * the focus results from pointer interaction (touch or mouse), but which is needed for accessibility when the focus
     * occurs by a non-pointer mechanism, for example keyboard or initial page load.
     *
     * <p>The application-level behavior for this component is controlled in the theme by the
     * <code class="prettyprint">$focusHighlightPolicy</code> SASS variable; however, note that this same variable controls
     * the focus highlight policy of many components and patterns. The values for the variable are:
     *
     * <ul>
     *   <li><code class="prettyprint">nonPointer</code>: <code class="prettyprint">oj-focus-highlight</code> is applied only
     *       when focus is not the result of pointer interaction. Most themes default to this value.</li>
     *   <li><code class="prettyprint">all</code>: <code class="prettyprint">oj-focus-highlight</code> is applied regardless
     *       of the focus mechanism.</li>
     *   <li><code class="prettyprint">none</code>: <code class="prettyprint">oj-focus-highlight</code> is never applied. This
     *       behavior is not accessible, and is intended for use when the application wishes to use its own event listener to
     *       precisely control when the class is applied (see below). The application must ensure the accessibility of the result.</li>
     * </ul>
     *
     * <p>To change the behavior on a per-instance basis, the application can set the SASS variable as desired and then use
     * event listeners to toggle this class as needed.
     *
     * @ojfragment ojFocusHighlightDoc - For use in the Styling table of components using the oj-focus-highlight class with the
     *    $focusHighlightPolicy var.  Components using that class with a component-specific mechanism instead of that $var will need
     *    different verbiage, which could be decomped to another baseComponent fragment if shareable by multiple components.
     * @memberof oj.baseComponent
     */
  });

  // Remove base component from the jQuery prototype, so it could not be created
  // directly by page authors

  delete $.fn[_BASE_COMPONENT];
}()); // end of BaseComponent wrapper function


// -----------------------------------------------------------------------------
// End of baseComponent, start of other content
// -----------------------------------------------------------------------------

/**
 * <p>This method is our version of $.widget, i.e. the static initializer of a component such as ojButton.
 * It calls that method, plus does any other static init we need.
 *
 * TODO:
 * - Consider moving this method into its own file.
 * - For base param, make the type oj.baseComponent rather than Object, but need to declare that as a type first.  Review how that's done.
 *
 * @private
 * @param {string} name typically of the form "oj.ojMenu"
 * @param {Object} base NOT optional (unlike JQUI)
 * @param {Object} prototype
 * @param {boolean=} isHidden - if true, indicates that the component name should not
 * be available on jQuery prototype
 */
oj.__registerWidget = function (name, base, prototype, isHidden) {
  $.widget(name, base, prototype);

  if (isHidden) {
    var globalName = name.split('.')[1];
    delete $.fn[globalName];
  }

  // create single-OJ pseudo-selector for component, e.g. ":oj-menu", in addition to the ":oj-ojMenu" that $.widget() creates.
  // for private components it will begin with an underscore, e.g.,  ":_oj-radio"
  if (name.substring(0, 5) === 'oj.oj' || name.substring(0, 6) === 'oj._oj') {
    var nameArray = name.split('.'); // ["oj", "ojMenu"], ["oj", "_ojRadio"]
    var namespace = nameArray[0];    // "oj"
    var simpleName = nameArray[1];  // "ojMenu", "_ojRadio"
    var fullName = namespace + '-' + simpleName; // "oj-ojMenu", "oj-_ojRadio"
    var isPrivate = simpleName.substring(0, 1) === '_';
    // if private, make the single-oj pseudo-selector start with an underscore, like this -> "_oj-radio"
    var modifiedFullName; // "oj-Menu", "_oj-Radio".  Lowercased below.
    if (isPrivate) {
      modifiedFullName = '_' + namespace + '-' + simpleName.substring(3);
    } else {
      modifiedFullName = namespace + '-' + simpleName.substring(2);
    }

    // Capitalization doesn't seem to matter with JQ pseudos, e.g. for the existing double-oj pseudo, both $(":oj-ojMenu") and $(":oj-ojmenu") work.
    // So, follow JQUI's pattern of using toLowerCase here, which will lowercase not only the "M' in "Menu", but also any camelcased chars after that.
    $.expr.pseudos[modifiedFullName.toLowerCase()] = function (elem) {
      return !!$.data(elem, fullName);
    };
  }
};


/**
 * @param {Object} self
 * @param {Object|undefined} originalDefaultValue
 * @param {?Object} constructorValue
 * @param {!Object} options
 * @param {string} prop
 * @param {Function} getter
 * @param {Function=} contextCallback
 * @private
 */
function _defineDynamicProperty(
  self, originalDefaultValue, constructorValue, options, prop, getter, contextCallback
) {
  var override = constructorValue;
  var replaced = false;
  var overriddenSubkeys = {};

  // eslint-disable-next-line no-param-reassign
  delete options[prop];

  Object.defineProperty(options, prop, {
    get: function () {
      // Once the option is replaced, we no longer merge in defaults
      if (replaced) {
        return override;
      }

      if (self._settingNestedKey != null) {
        // The getter is getting called from the option() method that will be mutating the current
        // object. We need to return only the override portion in this case to avoid the defaults being
        // reapplied as an override

        return override;
      }

      var defaultVal = getter(contextCallback ? contextCallback() : prop);

      return _mergeOptionLayers([originalDefaultValue, defaultVal, override], overriddenSubkeys);
    },
    set: function (value) {
      override = value;

      if (self._settingNestedKey != null) {
        overriddenSubkeys[self._settingNestedKey] = true;
      } else { // The entire option has been replaced
        replaced = true;
      }
    },
    enumerable: true
  }
                       );
}

/**
 * @ignore
 */
function _getCompoundDynamicGetter(values) {
  if (values.length === 1) {
    var val = values[0];
    return (val instanceof __ojDynamicGetter) ? val.getCallback() : null;
  }

  var hasGetters = false;
  for (var i = 0; i < values.length && !hasGetters; i++) {
    var value = values[i];
    if (value != null && value instanceof __ojDynamicGetter) {
      hasGetters = true;
    }
  }

  if (hasGetters) {
    var getter = function (context) {
      var resolvedVals = [];
      values.forEach(function (_value) {
        if (_value != null && _value instanceof __ojDynamicGetter) {
          resolvedVals.push(_value.getCallback()(context));
        } else {
          resolvedVals.push(_value);
        }
      });

      return _mergeOptionLayers(resolvedVals);
    };
    return getter;
  }

  return null;
}

/**
 * @private
 */
function _getSpecialContainerNames(element) {
  var elem = element;
  var containers = [];
  while (elem) {
    var ga = elem.getAttribute;
    var name = ga ? ga.call(elem, oj.Components._OJ_CONTAINER_ATTR) : null;
    if (name != null) {
      containers.push(name);
    }
    elem = elem.parentNode;
  }

  return containers;
}

/**
 * @private
 */
function _storeWidgetName(element, widgetName) {
  var data = element.data(_OJ_WIDGET_NAMES_DATA);
  if (!data) {
    data = [];
    element.data(_OJ_WIDGET_NAMES_DATA, data);
  }
  if (data.indexOf(widgetName) < 0) {
    data.push(widgetName);
  }
}

/**
 * @private
 */
function _removeWidgetName(element, widgetName) {
  var data = element.data(_OJ_WIDGET_NAMES_DATA);
  if (data) {
    var index = data.indexOf(widgetName);
    if (index >= 0) {
      data.splice(index, 1);
      if (data.length === 0) {
        element.removeData(_OJ_WIDGET_NAMES_DATA);
      }
    }
  }
}

/**
 * @private
 * @param {Array} values - values to merge
 * @param {Object=} overriddenSubkeys subkeys where the merging should not occur, i.e.
 * the value from corresponsing subkey on the last element of values array should win
 */
function _mergeOptionLayers(values, overriddenSubkeys) {
  var result;
  for (var i = 0; i < values.length; i++) {
    var value = values[i];
    if (value !== undefined) {
      if ($.isPlainObject(value)) {
        var input = $.isPlainObject(result) ? [result, value] : [value];
        // The last object (overrides) is always fully merged in
        result = _mergeObjectsWithExclusions({}, input,
                                             (i === values.length - 1) ? null : overriddenSubkeys,
                                             null);
      } else {
        result = value;
      }
    }
  }
  return result;
}

/**
 * @private
 */
function _mergeObjectsWithExclusions(target, input, ignoreSubkeys, basePath) {
  var inputLength = input.length;

  for (var inputIndex = 0; inputIndex < inputLength; inputIndex++) {
    var source = input[inputIndex];
    var keys = Object.keys(source);
    for (var i = 0; i < keys.length; i++) {
      var key = keys[i];
      var path;
      if (ignoreSubkeys == null) {
        path = null;
      } else if (basePath == null) {
        path = key;
      } else {
        path = basePath + '.' + key;
      }
      // Ignore all sources when the current path is registered in ignoreSubkeys
      if (ignoreSubkeys == null || !ignoreSubkeys[path]) {
        var value = source[key];
        if (value !== undefined) {
          if ($.isPlainObject(value)) {
            var params = $.isPlainObject(target[key]) ? [target[key], value] : [value];
            // eslint-disable-next-line no-param-reassign
            target[key] = _mergeObjectsWithExclusions({}, params, ignoreSubkeys, path);
          } else {
            // eslint-disable-next-line no-param-reassign
            target[key] = value;
          }
        }
      }
    }
  }
  return target;
}

/**
 * @private
 */
function _returnTrue() {
  return true;
}

/**
 * Returns an object with context for the given child DOM node. This will always contain the subid for the node,
 * defined as the 'subId' property on the context object. Additional component specific information may also be included.
 *
 * For more details on returned objects, see <a href="#contextobjects-section">context objects</a>.
 *
 * @ojfragment nodeContextDoc
 * @memberof oj.baseComponent
 */

/**
 * The child DOM node
 *
 * @ojfragment nodeContextParam
 * @memberof oj.baseComponent
 */

/**
 * The context for the DOM node, or <code class="prettyprint">null</code> when none is found.
 *
 * @ojfragment nodeContextReturn
 * @memberof oj.baseComponent
 */

/**
 * // Returns {'subId': 'oj-some-sub-id', 'componentSpecificProperty': someValue, ...}
 * var context = myComponent.getContextByNode(nodeInsideElement);
 *
 * @ojfragment nodeContextExample
 * @memberof oj.baseComponent
 */

/**
 * <p>The contextMenu slot is set on the <code class="prettyprint">oj-menu</code> within this element.
 * This is used to designate the JET Menu that this component should launch as a context menu on right-click, Shift-F10, Press & Hold, or component-specific gesture.
 * If specified, the browser's native context menu will be replaced by the JET Menu specified in this slot.
 * <p>
 * The application can register a listener for the Menu's ojBeforeOpen event. The listener can cancel the launch via event.preventDefault(),
 * or it can customize the menu contents by editing the menu DOM directly, and then calling refresh() on the Menu.
 * <p>
 * To help determine whether it's appropriate to cancel the launch or customize the menu, the ojBeforeOpen listener can use component API's to determine which
 * table cell, chart item, etc., is the target of the context menu. See the JSDoc of the individual components for details.
 * <p>
 * Keep in mind that any such logic must work whether the context menu was launched via right-click, Shift-F10, Press & Hold, or component-specific touch gesture.
 *
 * @ojslot contextMenu
 * @memberof oj.baseComponent
 *
 * @ojshortdesc The contextMenu slot is set on the oj-menu instance within this element.  It designates the JET Menu to launch as a context menu.
 * @ojmaxitems 1
 *
 * @example <caption>Initialize the component with a context menu:</caption>
 * &lt;oj-some-element>
 *     &lt;-- use the contextMenu slot to designate this as the context menu for this component -->
 *     &lt;oj-menu slot="contextMenu" style="display:none" aria-label="Some element's context menu">
 * ...
 *     &lt;/oj-menu>
 * &lt;/oj-some-element>
 */

/**
 * Sets a property or a subproperty (of a complex property) and notifies the component
 * of the change, triggering a [property]Changed event.
 * The value should be of the same type as the type of the attribute mentioned in this API document.
 *
 * @function setProperty
 * @since 4.0.0
 * @param {string} property - The property name to set. Supports dot notation for subproperty access.
 * @param {any} value - The new value to set the property to.
 * @return {void}
 *
 * @expose
 * @memberof oj.baseComponent
 * @ojshortdesc Sets a property or a single subproperty for complex properties and notifies the component of the change, triggering a corresponding event.
 * @instance
 *
 * @ojtsexample <caption>Set a single subproperty of a complex property:</caption>
 * myComponent.setProperty('complexProperty.subProperty1.subProperty2', "someValue");
 */
/**
 * Retrieves the value of a property or a subproperty.
 * The return type will be the same as the type of the property as specified in this API document.
 * If the method is invoked with an incorrect property/subproperty name, it returns undefined.
 * @function getProperty
 * @since 4.0.0
 * @param {string} property - The property name to get. Supports dot notation for subproperty access.
 * @return {any}
 *
 * @expose
 * @memberof oj.baseComponent
 * @ojshortdesc Retrieves the value of a property or a subproperty.
 * @instance
 *
 * @ojtsexample <caption>Get a single subproperty of a complex property:</caption>
 * let subpropValue = myComponent.getProperty('complexProperty.subProperty1.subProperty2');
 */
/**
 * Performs a batch set of properties.
 * The type of value for each property being set must match the type of the property as specified in this
 * API document.
 * @function setProperties
 * @since 4.0.0
 * @param {Object} properties - An object containing the property and value pairs to set.
 * @return {void}
 *
 * @expose
 * @memberof oj.baseComponent
 * @ojshortdesc Performs a batch set of properties.
 * @instance
 *
 * @ojtsexample <caption>Set a batch of properties:</caption>
 * myComponent.setProperties({"prop1": "value1", "prop2.subprop": "value2", "prop3": "value3"});
 */

/**
 * Copyright (c) 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */
/* global Message:false */

/**
 * Component Messaging Utilities.
 * @param {Object} component instance
 * @protected
 * @constructor
 * @since 0.6.0
 * @ignore
 */
oj.ComponentMessaging = function (component) {
  this.Init(component);
};

// Subclass from oj.Object
oj.Object.createSubclass(oj.ComponentMessaging, oj.Object, 'oj.ComponentMessaging');

/**
 * Default display types supported for component messaging.
 * @memberof! oj.ComponentMessaging
 * @const
 * @protected
 * @ignore
 */
oj.ComponentMessaging._DISPLAY_TYPE = {
  NONE: 'none',
  NOTEWINDOW: 'notewindow',
  PLACEHOLDER: 'placeholder',
  INLINE: 'inline'
};

/**
 * Tracks the messaging strategy callback function by type, used to instantiate messaging strategies.
 * Implementations register by type.
 * @memberof! oj.ComponentMessaging
 * @const
 * @protected
 * @ignore
 */
oj.ComponentMessaging._DISPLAY_TYPE_TO_CALLBACK = {};

/**
 * Stores the constructor function callback object used to constuct a strategy object for the
 * specified type.
 *
 * @param {string} type
 * @param {Function} strategyConstructorCallback a constructor callback that can be used to constuct
 * a strategy object for the specified type
 * @ignore
 */
oj.ComponentMessaging.registerMessagingStrategy = function (type, strategyConstructorCallback) {
  if (type && typeof strategyConstructorCallback === 'function') {
    oj.ComponentMessaging._DISPLAY_TYPE_TO_CALLBACK[type] = strategyConstructorCallback;
  }
};

/**
 * Initializes the strategy based on the display options that specify the messaging artifacts that
 * will be displayed by this strategy.
 *
 * @param {Object} component widget instance
 * @memberof! oj.ComponentMessaging
 * @instance
 * @protected
 */
oj.ComponentMessaging.prototype.Init = function (component) {
  oj.ComponentMessaging.superclass.Init.call(this);

  this._component = component;
  this._activated = false;

  // sets up this._strategies, which is a list of what we will display in each type of
  // displayOption. E.g., 'inline': 'messages', 'notewindow': converterHint, validatorHint, etc.
  // See EditableValues displayOption attribute jsdoc for more details.
  this._initializeMessagingStrategies();
};

/**
 * Utility function that activates messaging on the component using the strategy provided.
 * @param {Object} launcher element(s) to which messaging applies
 * @param {Object} content
 * @private
 */
oj.ComponentMessaging.prototype.activate = function (launcher, contentElement, content) {
  var that = this;
  oj.Assert.assertObject(content);
  this._launcher = launcher;
  this._contentElement = contentElement;

  this._messagingContent = oj.CollectionUtils.copyInto(this._messagingContent || {}, content);

  // if already active, reinitialize strategies based on new messagingDisplay preferences.
  if (!this._isActive()) {
    // for each 'messaging strategy' (e.g., inline == InlineMessagingStrategy,
    // notewindow == PopupMessagingStrategy, etc), call .activate which initializes
    // the strategy.
    $.each(this._strategies, function (i, strategy) {
      strategy.activate(that);
    });
    this._activated = true;
  } else {
    this._reactivate();
  }
};

/**
 * Utility function that updates messaging on the component for the content provided, using the
 * strategies.
 *
 * @param {Object} content
 * @private
 */
// TODO: component messaging could take the component instance
oj.ComponentMessaging.prototype.update = function (content) {
  oj.Assert.assertObject(content);
  oj.Assert.assertBoolean(this._activated);

  this._messagingContent = oj.CollectionUtils.copyInto(this._messagingContent || {}, content);

  if (this._activated) {
    $.each(this._strategies, function (i, strategy) {
      if (strategy.shouldUpdate(content)) {
        strategy.update();
      }
    });
  }
};

/**
 * Utility function that deactivates messaging on the component using the strategy provided.
 * @private
 */
oj.ComponentMessaging.prototype.deactivate = function () {
  $.each(this._strategies, function (i, strategy) {
    strategy.deactivate();
  });

  this._activated = false;
  this._component = null;
  this._launcher = null;
  this._contentElement = null;
  this._strategies = {};
};

/**
 * Utility function that closes anything that needs to be closed when oj.Components.subtreeHidden
 * is called. e.g, popup.
 * @private
 */
oj.ComponentMessaging.prototype.close = function () {
  if (this._activated) {
    $.each(this._strategies, function (i, strategy) {
      strategy.close();
    });
  }
};

/**
 * Creates a messaging strategy for the specified type, initializing it with the options provided.
 * @param {string|number} type defined by oj.ComponentMessaging._DISPLAY_TYPE. For example,
 * a displayType of 'notewindow' creates a PopupComponentMessaging strategy. See
 * registerMessagingStrategy where we register the type and the callback to call for a given type.
 * We currently have PopupMessagingStrategy, DefaultMessagingStrategy, PlaceholderMessagingStrategy,
 * and InlineComponentStrategy.
 * e.g., In PopupComponentMessaging.js:
 *  oj.ComponentMessaging.registerMessagingStrategy(oj.ComponentMessaging._DISPLAY_TYPE.NOTEWINDOW,
 *                              oj.PopupMessagingStrategy
 * @param {Array.<string>|undefined} artifactsForType (e.g., 'messages', 'title', 'validatorHints')
 *
 * @private
 * @instance
 * @memberOf !oj.ComponentMessaging
 */
oj.ComponentMessaging.prototype._createMessagingStrategy = function (type, artifactsForType) {
  var Callback = oj.ComponentMessaging._DISPLAY_TYPE_TO_CALLBACK[type] ||
      oj.ComponentMessaging._DISPLAY_TYPE_TO_CALLBACK[oj.ComponentMessaging._DISPLAY_TYPE.NONE];

  // dynamically instantiate the strategy objects.
  return new Callback(artifactsForType);
};

/**
 * Returns the component instance or null
 *
 * @return {Object|null}
 * @private
 * @instance
 * @memberOf !oj.ComponentMessaging
 */
oj.ComponentMessaging.prototype._getComponent = function () {
  return this._component || null;
};

/**
 * Returns the launcher jquery element. This is the element on the component to which messaging
 * applies.
 *
 * @return {Object|null} null if messaging is not activated.
 * @private
 * @instance
 * @memberOf !oj.ComponentMessaging
 */
oj.ComponentMessaging.prototype._getLauncher = function () {
  return this._launcher || null;
};


/**
 * Returns the jquery element on the component to which aria-invalid
 * applies. This is either the launcher itself or the inputs
 * within the launcher dom.  JAWS only reads aria-invalid when it is on the input/textara/select.
 * <p>
 * In the case of radioset/checkboxset, for example, where the launcher
 * is the root dom element and the inputs are with it, we return the inputs.
 *
 * </p>
 *
 * @return {Object|null} null if launcher is null
 * @private
 * @instance
 * @memberOf !oj.ComponentMessaging
 */
oj.ComponentMessaging.prototype._getContentElement = function () {
  return this._contentElement || null;
};

/**
 * Returns the last saved messagingContent object.
 *
 * @return {Object}
 * @private
 * @instance
 * @memberOf !oj.ComponentMessaging
 */
oj.ComponentMessaging.prototype._getMessagingContent = function () {
  return this._messagingContent || {};
};

/**
 * Whether the component messaging is activated.
 * @return {boolean}
 * @private
 */
oj.ComponentMessaging.prototype._isActive = function () {
  return this._activated;
};

/**
 * Returns a key/value array: displayTypes -> array of artifacts using that displayType.
 * where artifacts is 'messages', 'converterHint', 'validatorHint', 'title';
 * e.g.,
 * artifactsByDisplayType[oj.ComponentMessaging._DISPLAY_TYPE.NOTEWINDOW] = ['messages', 'converterHints']
 * artifactsByDisplayType[oj.ComponentMessaging._DISPLAY_TYPE.NONE] = ['validatorHints']
 * The types of messaging content for which displayOptions can be configured include
 * messages, converterHint, validatorHint and title.
 * The displayOptions for each type is specified either as an array of strings or a string.
 * When an array is specified the first display option takes precedence over the second and so on,
 * so we will only have ONE display type per artifact.
 */
oj.ComponentMessaging.prototype._getResolvedMessagingDisplayOptions = function () {
  var artifactsByDisplayType = {};
  var artifactDisplayTypeResolved = false;
  var compPH = this._component.options.placeholder;
  var messagingPreferences = this._component.options.displayOptions || {};
  var $messagingPreferences = {};
  var self = this;

  // first resolve primary display options for each artifact.
  // E.g. at the end of this loop you should have something like this
  // {messages: 'notewindow', converterHint: 'placeholder', validatorHint: 'notewindow', title: 'none'}
  var keys = Object.keys(messagingPreferences);
  for (var i = 0; i < keys.length; i++) {
    var key = keys[i];
    var displayTypes = messagingPreferences[key];
    // loop over array of displayTypes preferred for artifact.
    // artifacts are 'messages', 'converterHint', 'validatorHint', 'title'
    artifactDisplayTypeResolved = false;
    var artifact = key + '';
      // we take either array or string values for displayOptions.
    if (Array.isArray(displayTypes)) {
      for (var j = 0; j < displayTypes.length; j++) {
        var displayType = displayTypes[j];
        if (!artifactDisplayTypeResolved) {
          artifactDisplayTypeResolved =
            self._resolveDisplayTypeForArtifact(artifact, displayType, compPH,
                                                $messagingPreferences);
        }
      }
    } else if (typeof displayTypes === 'string') {
      if (!artifactDisplayTypeResolved) {
        artifactDisplayTypeResolved =
          self._resolveDisplayTypeForArtifact(artifact, displayTypes, compPH,
                                              $messagingPreferences);
      }
    }

    // if we couldn't resolve then use "none". E.g., validationHint: ['none']
    if (!artifactDisplayTypeResolved) {
      $messagingPreferences[artifact] = oj.ComponentMessaging._DISPLAY_TYPE.NONE;
    }
  }

  // collate by displayType -> artifact. but first reset
  $.each(oj.ComponentMessaging._DISPLAY_TYPE, function (type, name) {
    artifactsByDisplayType[name] = [];
  });

  $.each($messagingPreferences, function (_artifact, _displayType) {
    // an artifact eventually resolves to one displayType.
    artifactsByDisplayType[_displayType].push(_artifact);
  });

  return artifactsByDisplayType;
};

oj.ComponentMessaging.prototype._resolveDisplayTypeForArtifact = function (
  artifact,
  displayType,
  compPH,
  $messagingPreferences) {
  var artifactDisplayTypeResolved = false;
  switch (displayType) {
    // placeholder display is special in that it's only supported on 'converterHint'.
    case oj.ComponentMessaging._DISPLAY_TYPE.PLACEHOLDER :

      if (artifact === 'converterHint') {
        // if placeholder is the first preference for converterHint, it's used under certain
        // conditions
        // if options.placeholder is not set then use 'converterHint' as the default
        // 'placeholder'
        // alternately if (options.placeholder), i.e., a custom placeholder is set, then
        // ignore the placeholder displayType and use the next display type as the default
        // for the artifact. We may have a fallback displayType in which case we use it,
        // otherwise we use 'none'. E.g.,
        // {'converterHint': ['placeholder', 'notewindow']} // use notewindow
        // {'converterHint': ['placeholder']}               // use none

        if (!artifactDisplayTypeResolved) {
          if (!compPH) {
            // eslint-disable-next-line no-param-reassign
            $messagingPreferences[artifact] = displayType;
            artifactDisplayTypeResolved = true;
          }
        }
      } else {
        // displayType 'placeholder' is not supported on other artifacts
        // ignore if present
        // TODO: In the future we may want to support configuring validatorHint ot title as
        // placeholder as well.
      }

      break;
    // inline display is special in that it's only supported on 'messages'.
    case oj.ComponentMessaging._DISPLAY_TYPE.INLINE :

      if (artifact === 'messages') {
        if (!artifactDisplayTypeResolved) {
          // eslint-disable-next-line no-param-reassign
          $messagingPreferences[artifact] = displayType;
          artifactDisplayTypeResolved = true;
        }
      } else {
        // displayType 'inline' is not supported on other artifacts
        // ignore if present
      }

      break;

    default:
      if (!artifactDisplayTypeResolved) {
        // eslint-disable-next-line no-param-reassign
        $messagingPreferences[artifact] = displayType;
        artifactDisplayTypeResolved = true;
      }
      break;
  }

  return artifactDisplayTypeResolved;
};

/**
 * Creates messaging strategies for the component based on the displayOptions.
 * @private
 */
oj.ComponentMessaging.prototype._initializeMessagingStrategies = function () {
  var artifactsByDisplayType = this._getResolvedMessagingDisplayOptions();
  var displayInNoteWindow = artifactsByDisplayType[oj.ComponentMessaging._DISPLAY_TYPE.NOTEWINDOW];
  var displayNone = artifactsByDisplayType[oj.ComponentMessaging._DISPLAY_TYPE.NONE];
  var displayInPlaceholder =
      artifactsByDisplayType[oj.ComponentMessaging._DISPLAY_TYPE.PLACEHOLDER];
  var displayInline = artifactsByDisplayType[oj.ComponentMessaging._DISPLAY_TYPE.INLINE];
  var messagingStrategies = {};

  if (displayInNoteWindow.length > 0) {
    // displayInNoteWindow is an array of the artifacts that want to be displayed in the note window
    // e.g., 'messages', 'converterHints', etc.
    messagingStrategies[oj.ComponentMessaging._DISPLAY_TYPE.NOTEWINDOW] =
      this._createMessagingStrategy(oj.ComponentMessaging._DISPLAY_TYPE.NOTEWINDOW,
                                    displayInNoteWindow);
  }

  if (displayInPlaceholder.length > 0) {
    // displayInPlaceholder is an array of the artifacts that want to be displayed in placeholder
    // e.g., 'converterHints'
    messagingStrategies[oj.ComponentMessaging._DISPLAY_TYPE.PLACEHOLDER] =
      this._createMessagingStrategy(oj.ComponentMessaging._DISPLAY_TYPE.PLACEHOLDER,
                                    displayInPlaceholder);
  }

  if (displayInline.length > 0) {
    // displayInPlaceholder is an array of the artifacts that want to be displayed in placeholder
    // e.g., 'converterHints'
    messagingStrategies[oj.ComponentMessaging._DISPLAY_TYPE.INLINE] =
      this._createMessagingStrategy(oj.ComponentMessaging._DISPLAY_TYPE.INLINE, displayInline);
  }

  messagingStrategies[oj.ComponentMessaging._DISPLAY_TYPE.NONE] =
    this._createMessagingStrategy(oj.ComponentMessaging._DISPLAY_TYPE.NONE, displayNone);

  this._strategies = messagingStrategies;
};

/**
 * Reinitializes component messaging with new messagingDisplayOptions.
 *
 * @private
 */
oj.ComponentMessaging.prototype._reactivate = function () {
  var artifactsByDisplayType = this._getResolvedMessagingDisplayOptions();
  var strategy;
  var cm = this;

  // for every displayType being requested either create the messaging strategy for the type or
  // reuse existing strategy if it has already been created.
  $.each(artifactsByDisplayType, function (type, artifactsForType) {
    // eslint-disable-next-line no-param-reassign
    type += ''; // coerce to avoid GCC warning
    strategy = cm._strategies[type];
    if (artifactsForType && artifactsForType.length > 0) {
      if (!strategy) {
        // create a strategy if one doesn't exist for the type
        strategy = cm._createMessagingStrategy(type, artifactsForType);
        cm._strategies[type] = strategy;
        strategy.activate(cm);
      } else if (strategy) {
        // update strategy with the latest displayOptions if already present. we don;t
        // want to remove it once activated.
        strategy.reactivate(artifactsForType);
      }
    } else if (strategy && oj.ComponentMessaging._DISPLAY_TYPE.NONE !== type) {
      // if we have no artifacts to show for a type, then remove the strategy.
      // only if its other than the DefaultMessagingStrategy as it's always needed to theme
      // component.
      strategy.deactivate();
      delete cm._strategies[type];
    }
  });
};

/**
 * A base messaging strategy class that is initialized with a set of displayOptions. This object
 * also provides helper methods for its subclasses.
 *
 * @param {Array.<string>} displayOptions an array of messaging artifacts displayed.
 *
 * @constructor
 * @class oj.MessagingStrategy
 * @private
 */
oj.MessagingStrategy = function (displayOptions) {
  this.Init(displayOptions);
};

// Subclass from oj.Object
oj.Object.createSubclass(oj.MessagingStrategy, oj.Object, 'oj.MessagingStrategy');

/**
 * Initializes the strategy based on the display options that specify the messaging artifacts that
 * will be displayed by this strategy.
 *
 * @param {Array.<string>} displayOptions an array of messaging artifacts displayed.
 * @private
 */
oj.MessagingStrategy.prototype.Init = function (displayOptions) {
  oj.Assert.assertArray(displayOptions);
  oj.MessagingStrategy.superclass.Init.call(this);

  this._displayOptions = displayOptions;
};

oj.MessagingStrategy.prototype.activate = function (cm) {
  this._componentMessaging = cm;
};

/**
 * Cleans up messaging artifacts that were created on the component instance. E.g., destroys any
 * widgets it created, removes styles added etc.
 *
 * @private
 */
oj.MessagingStrategy.prototype.deactivate = function () {
};

/**
 * Utility function that closes anything that needs to be closed when oj.Components.subtreeHidden
 * is called. e.g, popup.
 *
 * @private
 */
oj.MessagingStrategy.prototype.close = function () {
};

/**
 * Reinitializes with the new display options and updates component messaging using the new content.
 *
 * @param {Array.<string>} newDisplayOptions
 * @private
 */
oj.MessagingStrategy.prototype.reactivate = function (newDisplayOptions) {
  this.Init(newDisplayOptions);
};

/**
 * Returns true always. Subclasses can override to ignore updates.
 *
 * @param {Object=} content the messaging content that is being updated
 * @return {boolean}
 *
 * @private
 */
// eslint-disable-next-line no-unused-vars
oj.MessagingStrategy.prototype.shouldUpdate = function (content) {
  return true;
};


/**
 * Updates component with instance using the content provided.
 *
 * @private
 */
oj.MessagingStrategy.prototype.update = function () {
};

// P R O T E C T E D  M E T H O D S
/**
 * Gets the launcher element for which the messaging is applied.
 * @return {Object} the jquery element of the form element.
 * @private
 */
oj.MessagingStrategy.prototype.GetLauncher = function () {
  return this._componentMessaging._getLauncher();
};

/**
 * @return {Object} the jquery element of the form element.
 * @private
 */
oj.MessagingStrategy.prototype.GetContentElement = function () {
  return this._componentMessaging._getContentElement();
};

/**
 * Gets the component (widget).
 * @return {Object} the jet component instance
 * @private
 */
oj.MessagingStrategy.prototype.GetComponent = function () {
  return this._componentMessaging._getComponent();
};

/**
 * Returns an array of messages.
 *
 * @return {Array} of messages each an instance of oj.Message
 * @private
 */
oj.MessagingStrategy.prototype.GetMessages = function () {
  return this.GetValidityState().getMessages();
};

oj.MessagingStrategy.prototype.GetMaxSeverity = function () {
  return this.GetValidityState().getMaxSeverity();
};

/**
 * Gets the converter hint.
 *
 * @return {Array} an array of hints, each a string.
 * @private
 */
oj.MessagingStrategy.prototype.GetConverterHint = function () {
  var hints = [];
  var mc = this._getMessagingContent();
  var converterHint = mc && mc.converterHint;
  if (converterHint) {
    hints.push(converterHint);
  }

  return hints;
};

oj.MessagingStrategy.prototype.GetValidatorHints = function () {
  var hints = [];
  var mc = this._getMessagingContent();
  var vHints = (mc && mc.validatorHint) || [];

  $.each(vHints, function (index, hint) {
    hints.push(hint);
  });

  return hints;
};


/**
 * Gets the short description.
 * @return {string} title or ""
 * @private
 */
oj.MessagingStrategy.prototype.GetTitle = function () {
  var mc = this._getMessagingContent();
  return (mc && mc.title) || '';
};

/**
 * Gets the validityState, an instance of oj.ComponentValidity or null.
 * @private
 */
oj.MessagingStrategy.prototype.GetValidityState = function () {
  var mc = this._getMessagingContent();
  return (mc && mc.validityState) || null;
};

/**
 * Whether the strategy is displaying messages or not.
 * @return {boolean} true if strategy has messages to display
 * @private
 */
oj.MessagingStrategy.prototype.HasMessages = function () {
  var messages = this.GetMessages();
  return !!((messages && messages.length > 0));
};

oj.MessagingStrategy.prototype.ShowMessages = function () {
  return this._displayOptions.indexOf('messages') !== -1;
};

oj.MessagingStrategy.prototype.ShowConverterHint = function () {
  return this._displayOptions.indexOf('converterHint') !== -1;
};

oj.MessagingStrategy.prototype.ShowValidatorHint = function () {
  return this._displayOptions.indexOf('validatorHint') !== -1;
};

oj.MessagingStrategy.prototype.ShowTitle = function () {
  return this._displayOptions.indexOf('title') !== -1 ||
         this._displayOptions.indexOf('helpInstruction') !== -1;
};

/**
 * Returns true if we have invalid messages; false otherwise.
 *
 * @return {boolean}
 * @private
 */
oj.MessagingStrategy.prototype.IsInvalid = function () {
  return this.GetValidityState().isInvalid();
};


/**
 * Gets the messagingContent stored in ComponentMessaging instance
 * @return {Object}
 * @private
 */
oj.MessagingStrategy.prototype._getMessagingContent = function () {
  if (this._componentMessaging) {
    return this._componentMessaging._getMessagingContent();
  }

  return {};
};


/**
 * A messaging strategy that updates the component theming and accessibility attributes.
 *
 * @param {Array.<string>} displayOptions .
 * @constructor
 * @extends {oj.MessagingStrategy}
 * @private
 */
oj.DefaultMessagingStrategy = function (displayOptions) {
  this.Init(displayOptions);
};

oj.ComponentMessaging.registerMessagingStrategy(oj.ComponentMessaging._DISPLAY_TYPE.NONE,
                                                oj.DefaultMessagingStrategy);

// TODO: Need to retrieve style selectors from a Style Manager
oj.DefaultMessagingStrategy._SELECTOR_STATE_INVALID = 'oj-invalid';
oj.DefaultMessagingStrategy._SELECTOR_STATE_WARNING = 'oj-warning';

oj.Object.createSubclass(oj.DefaultMessagingStrategy, oj.MessagingStrategy,
                         'oj.DefaultMessagingStrategy');

/**
 * Updates component theming, a11y attributes using the latest component state and its messaging
 * content.
 *
 * @private
 */
oj.DefaultMessagingStrategy.prototype.update = function () {
  oj.DefaultMessagingStrategy.superclass.update.call(this);

  var launcher = this.GetLauncher();
  var maxSeverity = this.GetMaxSeverity();
  var removeClasses = [];
  var addClasses = [];
  var invalid = false;
  var component = this.GetComponent();
  var jqRoot = component.widget();

  if (!launcher) {
    return;
  }

  // apply element error styling if invalid
  if (this.IsInvalid()) {
    // enable tooltip; set invalid class and aria invalid
    // TODO: oj classes should be set on the root DOM
    removeClasses.push(oj.DefaultMessagingStrategy._SELECTOR_STATE_WARNING);
    addClasses.push(oj.DefaultMessagingStrategy._SELECTOR_STATE_INVALID);
    invalid = true;
  } else if (this.HasMessages() && maxSeverity === Message.SEVERITY_LEVEL.WARNING) {
    // TODO: add warning or other severity state
    removeClasses.push(oj.DefaultMessagingStrategy._SELECTOR_STATE_INVALID);
    addClasses.push(oj.DefaultMessagingStrategy._SELECTOR_STATE_WARNING);
  } else {
    // for all other messages we remove selectors
    removeClasses.push(oj.DefaultMessagingStrategy._SELECTOR_STATE_INVALID);
    removeClasses.push(oj.DefaultMessagingStrategy._SELECTOR_STATE_WARNING);
  }

  jqRoot.removeClass(removeClasses.join(' '))
          .addClass(addClasses.join(' ')); // classes added to root
  // aria-invalid needs to be on an input/textarea
  this.GetContentElement().attr({ 'aria-invalid': invalid });
};

/**
 * Cleans up messaging artifacts that were created on the component instance. E.g., destroys any
 * widgets it created, removes styles added etc.
 *
 * @private
 */
oj.DefaultMessagingStrategy.prototype.deactivate = function () {
  var jqRoot = this.GetComponent().widget();

  jqRoot.removeClass(oj.DefaultMessagingStrategy._SELECTOR_STATE_INVALID)
                  .removeClass(oj.DefaultMessagingStrategy._SELECTOR_STATE_WARNING);
  this.GetContentElement().removeAttr('aria-invalid');
  oj.DefaultMessagingStrategy.superclass.deactivate.call(this);
};


/**
 * A messaging strategy that uses html5 placeholder (for now) to set/remove placeholder content.
 *
 * @param {Array.<string>} displayOptions an array of messaging artifacts displayed in the placeholder.
 * @constructor
 * @extends {oj.MessagingStrategy}
 * @private
 */
oj.PlaceholderMessagingStrategy = function (displayOptions) {
  this.Init(displayOptions);
};

oj.ComponentMessaging.registerMessagingStrategy(oj.ComponentMessaging._DISPLAY_TYPE.PLACEHOLDER,
                                                oj.PlaceholderMessagingStrategy);

// Subclass from oj.MessagingStrategy
oj.Object.createSubclass(oj.PlaceholderMessagingStrategy, oj.MessagingStrategy,
                         'oj.PlaceholderMessagingStrategy');

/**
 * Initializer
 *
 * @param {Array.<string>} displayOptions an array of messaging artifacts displayed in the notewindow.
 * @private
 */
oj.PlaceholderMessagingStrategy.prototype.Init = function (displayOptions) {
  oj.PlaceholderMessagingStrategy.superclass.Init.call(this, displayOptions);
};

/**
 * Sets up a placeholder for the component instance using the converter hint.
 *
 * @param {Object} cm a reference to an instance of oj.ComponentMessaging that provides access to
 * the latest messaging content.
 *
 * @private
 */
oj.PlaceholderMessagingStrategy.prototype.activate = function (cm) {
  oj.PlaceholderMessagingStrategy.superclass.activate.call(this, cm);
  this._refreshPlaceholder();
};

oj.PlaceholderMessagingStrategy.prototype.reactivate = function (newDisplayOptions) {
  oj.PlaceholderMessagingStrategy.superclass.reactivate.call(this, newDisplayOptions);
  this._refreshPlaceholder();
};

/**
 * Returns true if the content being updated includes converterHint prop. This method is an
 * optimization because the update() method is called too often and any time any content changes.
 * The only time PlaceholderMessagingStrategy#update needs to execute is when the converter hint
 * changes.
 *
 * @param {Object=} content the messaging content that is being updated
 * @return {boolean}
 *
 * @private
 */
oj.PlaceholderMessagingStrategy.prototype.shouldUpdate = function (content) {
  return !!(content && content.converterHint !== undefined);
};

oj.PlaceholderMessagingStrategy.prototype.update = function () {
  oj.PlaceholderMessagingStrategy.superclass.update.call(this);
  this._refreshPlaceholder();
};

// a default placeholder is set on the component, and that is typically the converter hint
oj.PlaceholderMessagingStrategy.prototype._refreshPlaceholder = function () {
  var launcher = this.GetLauncher();

  if (this.ShowPlaceholderContent() && launcher) {
    var hints = this.GetConverterHint();
    var content = hints.length ? hints[0] : '';
    var context = {};
    context.internalMessagingSet = true; // to indicate to component that placeholder is being
    // set from messaging module
    this.GetComponent().option({ placeholder: content }, { _context: context });
  }
};

oj.PlaceholderMessagingStrategy.prototype.ShowPlaceholderContent = function () {
  // we have a placeholder to set/show if we have converterHint set.
  return this.ShowConverterHint();
};

/**
 * The ComponentValidity object represent a component's current validity state. The instance
 * provides specific methods to retrieve info such as <p>
 *  - whether the component is valid <p>
 *  - the messages currently tracked on the component.<p>
 *  - the max severity level of the messages, e.g., fatal, error etc. See oj.Message for details
 *
 * @param {boolean} valid
 * @param {Array} messages
 * @constructor
 * @private
 */
oj.ComponentValidity = function (valid, messages) {
  // TODO: provide methods that allow model implementations to instruct the elements to showMessages,
  // especially the ones marked for 'lazy' notification.
  this.Init(valid, messages);
};

/**
 * whether there are invalid messages among the list of messages.
 *
 * @param {Array} messages list of messages
 * @returns {boolean} true if we have invalid messages; false otherwise
 */
oj.ComponentValidity.isInvalid = function (messages) {
  var maxLevel = Message.getMaxSeverity(messages);
  if (maxLevel >= Message.SEVERITY_LEVEL.ERROR) {
    return true;
  }

  return false;
};

// Subclass from oj.Object
oj.Object.createSubclass(oj.ComponentValidity, oj.Object, 'oj.ComponentValidity');

/**
 * The jquery element whose validity this object describes
 * @param {boolean} valid
 * @param {Array} messages instances of oj.Message
 */
oj.ComponentValidity.prototype.Init = function (valid, messages) {
  oj.ComponentValidity.superclass.Init.call(this);
  this._initialize(valid, messages);
};

/**
 * Returns a boolean true if valid; false if element not valid
 * @returns {boolean}
 * @private
 */
oj.ComponentValidity.prototype.isInvalid = function () {
  return this._invalid;
};

/**
 * Returns an Array or messages that we are marked for immediate display or an empty array.
 * @private
 * @returns {Array}
 */
oj.ComponentValidity.prototype.getMessages = function () {
  return this._messages;
};

/**
 * Returns the max severity level.
 * @return {number}
 * @private
 */
oj.ComponentValidity.prototype.getMaxSeverity = function () {
  return this._maxSeverity;
};

/**
 * Updates the validity state for the component.
 *
 * @param {boolean} valid
 * @param {Array} messages instances of oj.Message
 * @private
 */
oj.ComponentValidity.prototype.update = function (valid, messages) {
  this._initialize(valid, messages);
};

oj.ComponentValidity.prototype._initialize = function (valid, messages) {
  this._compValid = valid;
  this._compMessages = messages;

  this._messages = this._getImmediateMessages(); // messages currently showing
  this._maxSeverity = Message.getMaxSeverity(this._messages); // max severity of messages currently showing
  this._invalid = oj.ComponentValidity.isInvalid(this._messages);
};

/**
 * Returns an array of messages that are marked for immediate display.
 *
 * @return {Array} of messages each an instance of oj.Message
 * @private
 */
oj.ComponentValidity.prototype._getImmediateMessages = function () {
  var messages = this._compMessages || [];
  var immediateMsgs = [];
  for (var index = 0; index < messages.length; index++) {
    var msg = messages[index];
    // gather component messages marked for immediate display
    if (!(msg instanceof oj.ComponentMessage) || msg.canDisplay()) {
      immediateMsgs.push(msg);
    }
  }

  return immediateMsgs;
};

/**
 * JET component custom element bridge.
 *
 * This bridge ensures that JET components with child JET custom elements
 * can access child properties before the child busy state resolves.
 * This bridge does not guarantee that all properties for the child
 * will be available to the application before its busy states resolves,
 * e.g data bound attribute values.
 *
 * Applications should still wait on the element or page level
 * busy context before accessing properties or methods.
 *
 * @class
 * @ignore
 */
oj.CustomElementBridge = {};

/* global Logger:false */

/**
 * Prototype for the JET component custom element bridge instance
 */
oj.CustomElementBridge.proto = Object.create(oj.BaseCustomElementBridge.proto);

oj.CollectionUtils.copyInto(oj.CustomElementBridge.proto, {
  AddComponentMethods: function (proto) {
    // Add subproperty getter/setter
    // eslint-disable-next-line no-param-reassign
    proto.setProperty = function (prop, value) {
      var bridge = oj.BaseCustomElementBridge.getInstance(this);
      if (!bridge.SaveEarlyPropertySet(prop, value)) {
        if (!bridge._setEventProperty(this, prop, value) &&
            !bridge._validateAndSetCopyProperty(this, prop, value, null)) {
          // If not an event or copy property, check to see if it's a component specific property
          var meta = oj.BaseCustomElementBridge.__GetPropertyMetadata(
            prop, oj.BaseCustomElementBridge.getProperties(bridge, this)
          );
          // For non component specific properties, just set directly on the element instead.
          if (!meta) {
            this[prop] = value;
          } else {
            oj.CustomElementBridge._getPropertyAccessor(this, prop)(value);
          }
        }
      }
    };
    // eslint-disable-next-line no-param-reassign
    proto.getProperty = function (prop) {
      var bridge = oj.BaseCustomElementBridge.getInstance(this);
      var meta = oj.BaseCustomElementBridge.__GetPropertyMetadata(
        prop, oj.BaseCustomElementBridge.getProperties(bridge, this)
      );
      var event = oj.__AttributeUtils.eventListenerPropertyToEventType(prop);

      // For event listeners and non component specific properties, return the property from the element.
      // Otherwise, return the widget property and let the widget handle dot notation for subproperties.
      if (event || !meta) {
        return this[prop];
      }

      var ext = meta ? meta.extension : null;

      if (ext && ext._COPY_TO_INNER_ELEM) {
        return bridge._getCopyProperty(this, prop, meta);
      }
      return oj.CustomElementBridge._getPropertyAccessor(this, prop)();
    };
    // Override HTMLELement's focus/blur methods so we can call focus/blur on an inner element if needed.
    // eslint-disable-next-line no-param-reassign
    proto.focus = function () {
      var bridge = oj.BaseCustomElementBridge.getInstance(this);
      // If focus is called before the component has been created there
      // will be no saved widget instance yet so call the HTMLElement
      // focus instead.
      if (bridge._WIDGET_INSTANCE) {
        var focusElem = bridge._WIDGET_INSTANCE.__getFocusElement();
        if (focusElem) {
          if (focusElem !== this) {
            focusElem.focus();
          } else {
            HTMLElement.prototype.focus.call(this);
          }
        }
      } else {
        HTMLElement.prototype.focus.call(this);
      }
    };
    // eslint-disable-next-line no-param-reassign
    proto.blur = function () {
      var bridge = oj.BaseCustomElementBridge.getInstance(this);
      if (bridge._WIDGET_INSTANCE) {
        var focusElem = bridge._WIDGET_INSTANCE.__getFocusElement();
        if (focusElem) {
          if (focusElem !== this) {
            focusElem.blur();
          } else {
            HTMLElement.prototype.blur.call(this);
          }
        }
      } else {
        HTMLElement.prototype.focus.call(this);
      }
    };
  },

  BatchedPropertySet: function (elem, props) {
    var keys = Object.keys(props);
    var processedMap = {};
    var i;

    for (i = 0; i < keys.length; i++) {
      var property = keys[i];
      var value = props[property];

      // exclude event proprties and transfer attributes from batch updates
      if (!this._setEventProperty(elem, property, value) &&
          !this._validateAndSetCopyProperty(elem, property, value, null)) {
        value = this.ValidatePropertySet(elem, property, value);

        property = this.GetAliasForProperty(property);
        processedMap[property] = value;
      }
    }
    // Skip batched property sets if widget constructor isn't available meaning
    // the widget wasn't instantiated due to an error on creation or destroyed.
    var widgetConstructor = oj.Components.__GetWidgetConstructor(this._WIDGET_ELEM);
    if (widgetConstructor) {
      widgetConstructor('option', processedMap);
    } else {
      for (i = 0; i < keys.length; i++) {
        var key = keys[i];
        elem.setProperty(key, props[key]);
      }
    }
  },

  CreateComponent: function (element) {
    var innerDomFun = this._INNER_DOM_FUNCTION;
    this._WIDGET_ELEM = oj.CustomElementBridge._getWidgetElement(
      element, innerDomFun ? innerDomFun(element) : this._EXTENSION._INNER_ELEM
    );

    // Transfer global attributes and copy tagged properties to child element if one exists
    if (this._WIDGET_ELEM !== element) {
      var transferAttrs = this._EXTENSION._GLOBAL_TRANSFER_ATTRS || [];
      for (var i = 0; i < transferAttrs.length; i++) {
        var attr = transferAttrs[i];
        if (element.hasAttribute(attr)) {
          this._WIDGET_ELEM.setAttribute(attr, element.getAttribute(attr));
          // Remove attribute from custom element after transfering value to inner element
          // Set a flag so we know that we're removing the attribute, not app so
          // that on attribute changed we don't remove it again
          this._removingTransfer = true;
          element.removeAttribute(attr);
        }
      }

      this._copyProperties();
    }

    oj.Components.unmarkPendingSubtreeHidden(element);

    // Initialize jQuery object with options and pass element as wrapper if needed
    var locator = $(this._WIDGET_ELEM);
    var widgetConstructor = $(this._WIDGET_ELEM)[this._EXTENSION._WIDGET_NAME].bind(locator);
    widgetConstructor(this._PROPS);
    this._WIDGET = widgetConstructor;
    this._WIDGET_INSTANCE = widgetConstructor('instance');

    if (this._WRITEBACK_PROPS) {
      this._WIDGET_INSTANCE.__saveWritebackOptions(this._WRITEBACK_PROPS);
    }

    // After parsing the DOM attribute values and initializing properties, remove the disabled
    // property if it exists due to 
    if (element.hasAttribute('disabled') && !this._disabledProcessed) {
      oj.CustomElementBridge._removeDisabledAttribute(element);
    }

    // Setup blur/focus listeners on inner element so we can trigger on the root custom element for 
    var getFocusEventPropagator = function (type) {
      return function () {
        // Ensure that the target is the custom element, not the inner element, so create
        // a new event and dispatch on the custom element.
        var focusEvent = document.createEvent('UIEvent');
        focusEvent.initEvent(type, false, false);
        element.dispatchEvent(focusEvent);
      };
    };
    var focusElem = this._WIDGET_INSTANCE.__getFocusElement();
    if (focusElem && focusElem !== element) {
      focusElem.addEventListener('focus', getFocusEventPropagator('focus'));
      focusElem.addEventListener('blur', getFocusEventPropagator('blur'));
    }

    // Set flag when we can fire property change events
    this.__READY_TO_FIRE = true;

    // Resolve the component busy state
    this.resolveDelayedReadyPromise();
  },

  DefineMethodCallback: function (proto, method, methodMeta) {
    // eslint-disable-next-line no-param-reassign
    proto[method] = function () {
      var bridge = oj.BaseCustomElementBridge.getInstance(this);
      var methodName = methodMeta.internalName || method;
      // Pass in null as thisArg to apply since the widget constructor is prebound to the jQuery element
      return bridge._WIDGET.apply(null, [methodName].concat([].slice.call(arguments)));
    };
  },

  DefinePropertyCallback: function (proto, property, propertyMeta) {
    var ext = propertyMeta.extension;
    Object.defineProperty(proto, property, {
      enumerable: true,
      get: function () {
        var bridge = oj.BaseCustomElementBridge.getInstance(this);
        if (propertyMeta._eventListener) {
          return bridge.GetEventListenerProperty(property);
        } else if (ext && ext._COPY_TO_INNER_ELEM) {
          return bridge._getCopyProperty(this, property, propertyMeta);
        }

        return oj.CustomElementBridge._getPropertyAccessor(this, property)();
      },
      set: function (value) {
        var bridge = oj.BaseCustomElementBridge.getInstance(this);
        // Properties can be set before the component is created. These early
        // sets are actually saved until after component creation and played back.
        if (!bridge.SaveEarlyPropertySet(property, value)) {
          if (propertyMeta._eventListener) {
            bridge.SetEventListenerProperty(this, property, value);
          } else if (!bridge._validateAndSetCopyProperty(this, property, value, propertyMeta)) {
            // For widget based components, see if there is a default value assigned in the
            // metadata if application tries to unset the property. For composites and
            // definitional elements, this is handled in the getter since the bridge handles
            // sets/gets, but for widget based components the bridge only calls through to the
            // widget code which initializes the component to the default value (which should match
            // the metadata default)
            var flags = {};
            if (value === undefined) {
              flags = { _context: { skipEvent: true } };
              // eslint-disable-next-line no-param-reassign
              value = bridge.GetDefaultValue(propertyMeta);
              // Usually the widget logic fires the property changed events, but in this case
              // the app has set undefined, but we're setting the default value on the widget so
              // we'll handle firing the property changed from the bridge code for this case
              // and skip the event in the widget code.
              if (bridge.__READY_TO_FIRE) {
                var previousValue = this[property];
                oj.BaseCustomElementBridge.__FirePropertyChangeEvent(
                  this, property, undefined, previousValue, 'external'
                );
              }
            }
            oj.CustomElementBridge._getPropertyAccessor(this, property, flags)(value);
          }
        }
      }
    });
  },

  GetAttributes: function (metadata) {
    var attrs = oj.BaseCustomElementBridge.getAttributes(metadata.properties);
    if (metadata.extension._GLOBAL_TRANSFER_ATTRS) {
      attrs = attrs.concat(metadata.extension._GLOBAL_TRANSFER_ATTRS);
    }
    // Private array based API to allow widget based components to specify any
    // additional attributes they want to get notified about, e.g. data-oj-input-id.
    // These attributes will get passed through to the widget via the
    // __handleWatchedAttribute method.
    if (metadata.extension._WATCHED_ATTRS) {
      attrs = attrs.concat(metadata.extension._WATCHED_ATTRS);
    }
    return attrs;
  },

  GetAliasForProperty: function (property) {
    // Aliasing only supported for top level properties
    var alias = this._EXTENSION._ALIASED_PROPS;
    if (alias && alias[property]) {
      return alias[property];
    }
    return property;
  },

  InitializeElement: function (element) {
    // Invoke callback on the superclass
    oj.BaseCustomElementBridge.proto.InitializeElement.call(this, element);

    if (this._EXTENSION._CONTROLS_SUBTREE_HIDDEN) {
      oj.Components.markPendingSubtreeHidden(element);
    }

    oj.BaseCustomElementBridge.__InitProperties(element, this._PROPS);
  },

  HandleAttributeChanged: function (element, attr, oldValue, newValue) {
    var transferAttrs = this._EXTENSION._GLOBAL_TRANSFER_ATTRS;
    var bTransfer = transferAttrs && transferAttrs.indexOf(attr) !== -1;
    var watchedAttrs = this._EXTENSION._WATCHED_ATTRS;
    var bWatchedAttr = watchedAttrs && watchedAttrs.indexOf(attr) !== -1;
    if (bTransfer && this._WIDGET_ELEM) {
      if (!this._removingTransfer) {
        // When we transfer the attribute the app will not be able to remove the
        // attribute from the DOM, we will recommend binding the value if the value
        // needs to be toggled.
        this._WIDGET_ELEM.setAttribute(attr, newValue);
        // Remove attribute from custom element after transfering value to inner element
        // Set a flag so we know that we're removing the attribute, not app so
        // that on attribute changed we don't remove it again
        this._removingTransfer = true;
        element.removeAttribute(attr);
      } else if (this._removingTransfer) {
        this._removingTransfer = false;
      }
    } else if (bWatchedAttr && oldValue !== newValue && this._WIDGET_INSTANCE) {
      // Check to see if this is attribute is being watched by the component
      // in which case we will just pass this through as is without converting
      // the attribute to a property name. Components are responsible for retrieving
      // attribute values on component initialization. This method only handles changes
      // after the fact.
      this._WIDGET_INSTANCE.__handleWatchedAttribute(attr, oldValue, newValue);
    }
  },

  HandleDetached: function (element) {
    // Invoke callback on the superclass
    oj.BaseCustomElementBridge.proto.HandleDetached.call(this, element);

    // Only call __handleDisconnected if the component hasn't previously
    // been destroyed which we can check by seeing if the widget constructor is null
    if (oj.Components.__GetWidgetConstructor(this._WIDGET_ELEM) && this._WIDGET_INSTANCE) {
      this._WIDGET_INSTANCE.__handleDisconnected();
    }
  },

  HandleReattached: function (element) {
    // Invoke callback on the superclass
    oj.BaseCustomElementBridge.proto.HandleReattached.call(this, element);

    if (this._WIDGET_INSTANCE) {
      this._WIDGET_INSTANCE.__handleConnected();
    }
  },

  InitializeBridge: function (element) {
    // Invoke callback on the superclass
    oj.BaseCustomElementBridge.proto.InitializeBridge.call(this, element);

    var descriptor = oj.BaseCustomElementBridge.__GetDescriptor(element.tagName);
    this._INNER_DOM_FUNCTION = descriptor.innerDomFunction;

    this._EXTENSION = this.METADATA.extension || {};

    this._PROPS = (this._EXTENSION._INNER_ELEM || this._INNER_DOM_FUNCTION) ?
      { _wrapper: element } : {};
    this._setupPropertyAccumulator(element, this._PROPS);

    // Checks metadata for copy and writeback properties
    this._processProperties(element);
  },

  _attributeChangedCallback: function (attr, oldValue, newValue) {
    var bridge = oj.BaseCustomElementBridge.getInstance(this);

    // Due to  where IE11 disables child inputs for a parent with the disabled attribute,
    // we will remove the disabled attribute after we save the value and will ignore all disabled
    // attribute sets after component initialization when the application can just as easily use the property
    // setter instead. Expressions will be handled in the CustomElementBinding.
    if (attr === 'disabled' && bridge._disabledProcessed) {
      // Always remove the disabled attribute even after component initialization and log warning.
      // A null value indicates that the value was removed already.
      if (newValue != null) {
        Logger.warn("Ignoring 'disabled' attribute change after component initialization. Use element property setter instead.");
        oj.CustomElementBridge._removeDisabledAttribute(this);
      }
      return;
    }

    // Invoke callback on the superclass
    oj.BaseCustomElementBridge.proto._attributeChangedCallback.call(this, attr, oldValue, newValue);
  },

  _copyProperties: function () {
    // Copies properties from the bridge _PROPS before the widget is instantiated
    // removing copied props from the object
    if (this._COPY_ATTRS) {
      for (var i = 0; i < this._COPY_ATTRS.length; i++) {
        var attr = this._COPY_ATTRS[i];
        var propName = oj.__AttributeUtils.attributeToPropertyName(attr);
        if (Object.prototype.hasOwnProperty.call(this._PROPS, propName)) {
          var value = this._PROPS[propName];
          this._setCopyProperty(attr, value);
          // Delete the attribute we just copied from the options that we
          // instantiate the widget with
          delete this._PROPS[propName];
        }
      }
    }
  },

  _getCopyProperty: function (elem, prop, propMeta) {
    var attrName = oj.__AttributeUtils.propertyNameToAttribute(prop);
    var ext = propMeta.extension;
    if (ext._ATTRIBUTE_ONLY) {
      if (this._WIDGET_ELEM.hasAttribute(attrName)) {
        var value = this._WIDGET_ELEM.getAttribute(attrName);
        var coercedValue;
        try {
          coercedValue = oj.__AttributeUtils.coerceValue(elem, attrName, value, propMeta.type);
        } catch (ex) {
          this.throwError(elem, ex);
        }
        return coercedValue;
      }
      return null;
    }
    return this._WIDGET_ELEM[prop];
  },

  _processProperties: function (elem) {
    var props = oj.BaseCustomElementBridge.getProperties(this, elem);
    if (props) {
      var propKeys = Object.keys(props);
      for (var i = 0; i < propKeys.length; i++) {
        var propName = propKeys[i];
        var propMeta = props[propName];
        // Store writeback properties on the bridge and set on widget when we instantiate it later
        if (propMeta.writeback) {
          if (!this._WRITEBACK_PROPS) {
            this._WRITEBACK_PROPS = {};
          }
          this._WRITEBACK_PROPS[propName] = true;
        }
        // Store properties to copy to inner element for easy lookup
        var ext = propMeta.extension;
        if (ext && ext._COPY_TO_INNER_ELEM) {
          if (!this._COPY_ATTRS) {
            this._COPY_ATTRS = [];
          }
          this._COPY_ATTRS.push(propName);
        }
      }
    }
  },

  _setCopyProperty: function (attribute, value) {
    if (value == null || value === false) {
      this._WIDGET_ELEM.removeAttribute(attribute);
    } else if (value === true) {
      this._WIDGET_ELEM.setAttribute(attribute, '');
    } else {
      this._WIDGET_ELEM.setAttribute(attribute, value);
    }
  },

  _setupPropertyAccumulator: function (element, widgetOptions) {
    // Add an element function that will track property values until expressions are all evaluated.
    // This object will be replaced with the actual widget constructor.
    this._WIDGET = function (method, prop, value) {
      // Allow property access before widget is created for element binding and dynamic element creation
      if (method === 'option') {
        oj.BaseCustomElementBridge.__SetProperty(
          this.GetAliasForProperty.bind(this), widgetOptions, prop, value
        );
        return widgetOptions[prop];
      }

      // throw is eslint hack to fix consistent-return
      throw this.throwError(element, 'Cannot access methods before element is upgraded.');
    };
  },

  _validateAndSetCopyProperty: function (elem, prop, value, propMeta) {
    // propMeta is could be null so we should retrieve it if not passed in

    var attrName = oj.__AttributeUtils.propertyNameToAttribute(prop);
    var isCopy = this._COPY_ATTRS && this._COPY_ATTRS.indexOf(attrName) !== -1;
    // If widget hasn't been instantiated skip setting until CreateComponent
    if (isCopy) {
      // We need to validate the value so that we don't copy an invalid value.
      // eslint-disable-next-line no-param-reassign
      value = this.ValidatePropertySet(elem, prop, value);

      if (this._WIDGET_ELEM) {
        if (!propMeta) {
          // eslint-disable-next-line no-param-reassign
          propMeta = oj.BaseCustomElementBridge.__GetPropertyMetadata(
            prop, oj.BaseCustomElementBridge.getProperties(this, elem)
          );
        }

        var previousValue = this._getCopyProperty(elem, prop, propMeta);
        this._setCopyProperty(attrName, value);
        // Fire a property change event for the copy properties since we don't actually pass
        // these to the widget. The widget will never update these properties themselves so
        // all updates are external.
        oj.BaseCustomElementBridge.__FirePropertyChangeEvent(
          elem, prop, this._getCopyProperty(elem, prop, propMeta), previousValue, 'external'
        );
      } else {
        // Save the value until inner widget is created and we can copy them over
        this._PROPS[attrName] = value;
      }
    }
    return isCopy;
  },

  _setEventProperty: function (elem, prop, value) {
    var isEvent = false;
    var event = oj.__AttributeUtils.eventListenerPropertyToEventType(prop);
    if (event) {
      // eslint-disable-next-line no-param-reassign
      elem[prop] = value;
      isEvent = true;
    }
    return isEvent;
  },


});

/** ***********************/
/* PUBLIC STATIC METHODS */
/** ***********************/

/**
 * Returns the metadata object for the given component.
 * @param  {string} tagName        The component tag name
 * @return {Object}                The component metadata object
 * @ignore
 */
oj.CustomElementBridge.getMetadata = function (tagName) {
  return oj.CustomElementBridge._METADATA_MAP[tagName.toLowerCase()];
};

/**
 * Checks whether the specified event type was declared in the metadata for this custom element
 * @param {Element} element the custom element
 * @param {string} type the event type (e.g. "beforeExpand")
 * @return {boolean} true if the event type was declared in the metadata, false otherwise
 * @ignore
 */
oj.CustomElementBridge.isKnownEvent = function (element, type) {
  var bridge = oj.BaseCustomElementBridge.getInstance(element);
  return (bridge.METADATA.events && bridge.METADATA.events[type]) != null;
};

/**
 * Checks whether the specified property was declared in the metadata for this custom element
 * @param {Element} element the custom element
 * @param {string} prop the property name (e.g. "selection")
 * @return {boolean} true if the property was declared in the metadata, false otherwise
 * @ignore
 */
oj.CustomElementBridge.isKnownProperty = function (element, prop) {
  var bridge = oj.BaseCustomElementBridge.getInstance(element);
  return (bridge.METADATA.properties && bridge.METADATA.properties[prop]) != null;
};

/**
 * Returns the custom element property for a given aliased component property which can be used
 * for converting an internal optionChange event, e.g. returning readonly for oj-switch's readOnly
 * property so we can fire a readonly-changed event instead of readOnly-changed.
 * Will return the original property if there is no aliasing.
 * @param {Element} element The custom element
 * @param {string} property The component property
 * @return {string}
 * @ignore
 */
oj.CustomElementBridge.getPropertyForAlias = function (element, property) {
  // Aliasing only supported for top level properties
  var bridge = oj.BaseCustomElementBridge.getInstance(element);
  var alias = bridge._EXTENSION._COMPONENT_TO_ELEMENT_ALIASES;
  if (alias && alias[property]) {
    return alias[property];
  }
  return property;
};

/**
 * Registers a component as a custom element.
 * @param {string} tagName The component tag name (all lower case), which should contain a dash '-' and not be a reserved tag name.
 * @param {Object} descriptor The registration descriptor. The descriptor will contain keys for metadata and other component overrides.
 * @param {Object} descriptor.metadata The JSON object containing info like the widget name, whether component has an inner element, an outer wrapper, and the component metadata.
 * @param {function(string, string, Object, function(string))} descriptor.parseFunction The function that will be called to parse attribute values.
 * Note that this function is only called for non bound attributes. The parseFunction will take the following parameters:
 * <ul>
 *  <li>{string} value: The value to parse.</li>
 *  <li>{string} name: The name of the attribute.</li>
 *  <li>{Object} meta: The metadata object for the property which can include its type, default value,
 *      and any extensions that the composite has provided on top of the required metadata.</li>
 *  <li>{function(string)} defaultParseFunction: The default parse function for the given attribute
 *      type which is used when a custom parse function isn't provided and takes as its parameters
 *      the value to parse.</li>
 * </ul>
 * @param {Element} descriptor.innerDomFunction The function that will be called to return the tag name of the inner DOM element, e.g. 'button' or 'a'
 * The innerDomFunction will take the following parameters:
 * <ul>
 *  <li>{Element} element: The component custom element.</li>
 * </ul>
 * @ignore
 */
oj.CustomElementBridge.register = function (tagName, descriptor) {
  var meta = descriptor[oj.BaseCustomElementBridge.DESC_KEY_META];
  meta = oj.BaseCustomElementBridge.__ProcessEventListeners(meta);
  // eslint-disable-next-line no-param-reassign
  descriptor[oj.BaseCustomElementBridge.DESC_KEY_META] = meta;
  oj.CustomElementBridge._METADATA_MAP[tagName.toLowerCase()] = meta;

  var ext = meta.extension;
  // Use the simple definitional element prototype if no real widget is associated with this custom element
  var proto = ext &&
      ext._WIDGET_NAME ? oj.CustomElementBridge.proto : oj.DefinitionalElementBridge.proto;

  // Create component to element property alias mapping for easy optionChange lookup and stash it in the extension object
  var aliasMap = ext._ALIASED_PROPS;
  if (aliasMap) {
    ext._COMPONENT_TO_ELEMENT_ALIASES = {};
    var aliases = Object.keys(aliasMap);
    aliases.forEach(function (alias) {
      ext._COMPONENT_TO_ELEMENT_ALIASES[aliasMap[alias]] = alias;
    });
  }

  if (oj.BaseCustomElementBridge.__Register(tagName, descriptor, proto)) {
    customElements.define(tagName.toLowerCase(), proto.getClass(descriptor));
  }
};

/** ***************************/
/* NON PUBLIC STATIC METHODS */
/** ***************************/

/**
 * Returns a property accessor for setting/getting options
 * @private
 */
oj.CustomElementBridge._getPropertyAccessor = function (element, property, flags) {
  function optionAccessor(value) {
    var bridge = oj.BaseCustomElementBridge.getInstance(element);
    // option set case
    if (arguments.length === 1) {
      // eslint-disable-next-line no-param-reassign
      value = bridge.ValidatePropertySet(element, property, value);
      // eslint-disable-next-line no-param-reassign
      property = bridge.GetAliasForProperty(property);
      bridge._WIDGET('option', property, value, flags);
      return undefined;
    }
    // option get case
    // eslint-disable-next-line no-param-reassign
    property = bridge.GetAliasForProperty(property);
    return bridge._WIDGET('option', property);
  }

  return optionAccessor.bind(element);
};

/**
 * Returns the element that the widget constructor will be instantiated on which can be the custom element or a child element.
 * @private
 */
oj.CustomElementBridge._getWidgetElement = function (element, innerTagName) {
  // If component widget is bound to an inner child element like <ul> for <oj-list-view>,
  // create one only if the application does not provide it.
  var widgetElem = element;
  if (innerTagName) {
    var firstChild = element.firstElementChild;
    if (firstChild && firstChild.tagName.toLowerCase() === innerTagName) {
      widgetElem = firstChild;
    } else {
      widgetElem = document.createElement(innerTagName);
      // Make a copy of the custom element children before appending the inner element
      var children = [];
      var nodeList = element.childNodes;
      for (var i = 0; i < nodeList.length; i++) {
        children.push(nodeList[i]);
      }

      element.appendChild(widgetElem); // @HtmlUpdateOk
      // If we create the inner child element, check to see if there are any children
      // to move like for <oj-button> which can have a child elements that should be moved to
      // the newly created inner <button> element.
      while (children.length) {
        var child = children.shift();
        // Only move default slot children to inner child element. Default slot children are those
        // that do not explictly set a slot attribute (or have one passed from a composite) or have slot=''.
        // The component will be responsible for moving all named slot children.
        // For example, it does not make sense for <oj-list-view> to move contextMenu slot to its inner <ul> element.
        if (!oj.BaseCustomElementBridge.getSlotAssignment(child)) {
          widgetElem.appendChild(child);
        }
      }
    }
    // add data-oj-internal attribute for automation tests
    widgetElem.setAttribute('data-oj-internal', '');
  }
  return widgetElem;
};

/**
 * Removes the disabled attribute from an element and marks the bridge as having
 * processed the value to prevent evaluation of additional attribute sets.
 * @param  {Element} element The custom element
 * @private
 */
oj.CustomElementBridge._removeDisabledAttribute = function (element) {
  var bridge = oj.BaseCustomElementBridge.getInstance(element);
  bridge._disabledProcessed = true;
  element.removeAttribute('disabled');
};

/**
 * Map of registered custom element names
 * @private
 */
oj.CustomElementBridge._METADATA_MAP = {};

/**
 * Copyright (c) 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */
var DataProviderFeatureChecker = /** @class */ (function () {
    function DataProviderFeatureChecker() {
    }
    DataProviderFeatureChecker.isDataProvider = function (dataprovider) {
        if (dataprovider['fetchFirst']) {
            return true;
        }
        return false;
    };
    DataProviderFeatureChecker.isTreeDataProvider = function (dataprovider) {
        if (dataprovider['getChildDataProvider']) {
            return true;
        }
        return false;
    };
    return DataProviderFeatureChecker;
}());
oj.DataProviderFeatureChecker = DataProviderFeatureChecker;

/**
 * Copyright (c) 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */

/**
 * @preserve Copyright 2013 jQuery Foundation and other contributors
 * Released under the MIT license.
 * http://jquery.org/license
 */

/* jslint browser: true,devel:true*/
/**
 * Utilities for getting DataProvider features.
 * @ignore
 */

/**
 * End of jsdoc
 */

/* global Logger:false */

/**
 * A bridge for a custom element that renders using a constructor
 * function. Note that when a constructor function is provided, the new instance isn't
 * created until the CreateComponent method so property changes that occur before the
 * component instance is created will no-op.
 *
 * Components that provide a constructor function should implement the following methods:
 * createDOM - Called when the component is instantiated
 * updateDOM - Called after createDOM and when the component needs to do a full render on
 * refresh and property changes if the component is not handling them separately.
 * handlePropertyChanged - (optional) Called when properties change and should return true if
 * the component has handled the property change and does not need to do a full render. If
 * false is returned, updateDOM will be called to do a full render.
 *
 * When the constructor function is called, the bridge will pass a context object
 * with the following keys:
 * element - The custom element
 * props - A proxy for the element properties with setter/getter and setProperty APIs allowing the
 *         component to control writeback.
 * unique - A unique ID that the component can append to the custom element ID to generate unique IDs
 *
 * Note that components supporting the constructor function approach may eventually
 * be refactored into composites once composites support non template rendering.
 *
 * This bridge ensures that JET components with child JET custom elements
 * can access child properties before the child busy state resolves.
 * This bridge does not guarantee that all properties for the child
 * will be available to the application before its busy states resolves,
 * e.g data bound attribute values.
 *
 * Applications should still wait on the element or page level
 * busy context before accessing properties or methods.
 *
 * @class
 * @ignore
 */
oj.DefinitionalElementBridge = {};

/**
 * Prototype for the JET component definitional bridge instance
 */
oj.DefinitionalElementBridge.proto = Object.create(oj.BaseCustomElementBridge.proto);

oj.CollectionUtils.copyInto(oj.DefinitionalElementBridge.proto, {
  beforePropertyChangedEvent: function (element, property, detail) {
    // Call the renderer function so the definitional element can refresh its UI
    var changedProp = property;
    var value = detail.value;
    if (detail.subproperty) {
      changedProp = detail.subproperty.path;
      value = detail.subproperty.value;
    }
    this._partialRender(element, changedProp, value);
  },

  AddComponentMethods: function (proto) {
    // Add refresh and subproperty getter/setter methods for all definitional elements
    // eslint-disable-next-line no-param-reassign
    proto.refresh = function () {
      var bridge = oj.BaseCustomElementBridge.getInstance(this);
      bridge._fullRender(this);
    };
    // eslint-disable-next-line no-param-reassign
    proto.setProperty = function (prop, value) {
      var bridge = oj.BaseCustomElementBridge.getInstance(this);
      if (!bridge.SaveEarlyPropertySet(prop, value)) {
        bridge.SetProperty(this, prop, value, this, true);
      }
    };
    // eslint-disable-next-line no-param-reassign
    proto.getProperty = function (prop) {
      // 'this' is the property object we pass to the definitional element contructor to track internal property changes
      var bridge = oj.BaseCustomElementBridge.getInstance(this);
      return bridge.GetProperty(this, prop, this);
    };
    // eslint-disable-next-line no-param-reassign
    proto._propsProto.setProperty = function (prop, value) {
      // 'this' is the property object we pass to the definitional element contructor to track internal property changes
      this._BRIDGE.SetProperty(this._ELEMENT, prop, value, this, false);
    };
    // eslint-disable-next-line no-param-reassign
    proto._propsProto.getProperty = function (prop) {
      return this._BRIDGE.GetProperty(this, prop, this);
    };
  },

  CreateComponent: function (element) {
    oj.Components.unmarkPendingSubtreeHidden(element);

    if (!this._INSTANCE && this._EXTENSION._CONSTRUCTOR) {
      // We expose a similar set of properties as composites except that props is
      // not a Promise and we don't expose any slot information.
      // At the moment some definitional elements have mutation observers so they don't need
      // to rely on refresh being called to be alerted of new children so any cached slotMap
      // can become out of sync. We should add this once we build in support to auto detect
      // added/removed children to custom elements.
      this._CONTEXT = {
        element: element,
        props: this._PROPS_PROXY,
        unique: oj.BaseCustomElementBridge.__GetUnique(),
      };
      this._CONTEXT.uniqueId = element.id ? element.id : this._CONTEXT.unique;
      this._INSTANCE = new this._EXTENSION._CONSTRUCTOR(this._CONTEXT);
      // Let the component initialize any additional DOM and then do a full render
      if (this._INSTANCE.createDOM) {
        this._INSTANCE.createDOM();
      }
      if (this._INSTANCE.updateDOM) {
        this._INSTANCE.updateDOM();
      }
    }

    // Set flag when we can fire property change events
    this.__READY_TO_FIRE = true;

    // Resolve the component busy state
    this.resolveDelayedReadyPromise();
  },

  DefineMethodCallback: function (proto, method, methodMeta) {
    // eslint-disable-next-line no-param-reassign
    proto[method] = function () {
      var bridge = oj.BaseCustomElementBridge.getInstance(this);
      if (bridge._INSTANCE) {
        var methodName = methodMeta.internalName || method;
        return bridge._INSTANCE[methodName].apply(bridge._INSTANCE, arguments);
      }
      return undefined;
    };
  },

  DefinePropertyCallback: function (proto, property, propertyMeta) {
    function set(value, bOuterSet) {
      // Properties can be set before the component is created. These early
      // sets are actually saved until after component creation and played back.
      if (!this._BRIDGE.SaveEarlyPropertySet(property, value)) {
        var previousValue = this._BRIDGE._PROPS[property];
        if (!oj.BaseCustomElementBridge.__CompareOptionValues(property, propertyMeta,
                                                              value, previousValue)) {
          // Skip validation for inner sets so we don't throw an error when updating readOnly writeable properties
          if (bOuterSet) {
            // eslint-disable-next-line no-param-reassign
            value = this._BRIDGE.ValidatePropertySet(this._ELEMENT, property, value);
          }
          if (propertyMeta._eventListener) {
            this._BRIDGE.SetEventListenerProperty(this._ELEMENT, property, value);
            this._BRIDGE._PROPS[property] = value;
          } else {
            this._BRIDGE._PROPS[property] = value;
            oj.BaseCustomElementBridge.__FirePropertyChangeEvent(
              this._ELEMENT, property, value, previousValue, bOuterSet ? 'external' : 'internal'
            );
          }
        } else {
          Logger.info(oj.BaseCustomElementBridge.getElementInfo(this._ELEMENT) + ": Ignoring property set for property '" +
            property + "' with same value.");
        }
      }
    }

    function innerSet(value) {
      set.bind(this)(value, false);
    }

    // Called on the custom element
    function outerSet(value) {
      var bridge = oj.BaseCustomElementBridge.getInstance(this);
      set.bind(bridge._PROPS_PROXY)(value, true);
    }

    function get() {
      var value = this._BRIDGE._PROPS[property];
      // If the attribute has not been set, return the default value
      if (value === undefined) {
        value = this._BRIDGE.GetDefaultValue(propertyMeta);
        this._BRIDGE._PROPS[property] = value;
      }
      return value;
    }

    function innerGet() {
      return get.bind(this)();
    }

    // Called on the custom element
    function outerGet() {
      var bridge = oj.BaseCustomElementBridge.getInstance(this);
      return get.bind(bridge._PROPS_PROXY)();
    }

    // Don't add event listener properties for inner props
    if (!propertyMeta._derived) {
      oj.BaseCustomElementBridge.__DefineDynamicObjectProperty(proto._propsProto, property,
                                                               innerGet, innerSet);
    }
    oj.BaseCustomElementBridge.__DefineDynamicObjectProperty(proto, property, outerGet, outerSet);
  },

  InitializeElement: function (element) {
    // Invoke callback on the superclass
    oj.BaseCustomElementBridge.proto.InitializeElement.call(this, element);

    if (this._EXTENSION._CONTROLS_SUBTREE_HIDDEN) {
      oj.Components.markPendingSubtreeHidden(element);
    }

    oj.BaseCustomElementBridge.__InitProperties(element, element);
  },

  InitializePrototype: function (proto) {
    // Invoke callback on the superclass
    oj.BaseCustomElementBridge.proto.InitializePrototype.call(this, proto);

    Object.defineProperty(proto, '_propsProto', { value: {} });
  },

  InitializeBridge: function (element) {
    // Invoke callback on the superclass
    oj.BaseCustomElementBridge.proto.InitializeBridge.call(this, element);

    this._EXTENSION = this.METADATA.extension || {};

    // For tracking outer property sets
    this._PROPS = {};

    // / For tracking inner property sets
    if (element._propsProto) {
      this._PROPS_PROXY = Object.create(element._propsProto);
      this._PROPS_PROXY._BRIDGE = this;
      this._PROPS_PROXY._ELEMENT = element;
    }
  },

  // eslint-disable-next-line no-unused-vars
  _fullRender: function (element) {
    if (this._INSTANCE && this._INSTANCE.updateDOM) {
      this._INSTANCE.updateDOM();
    }
  },

  _partialRender: function (element, property, value) {
    if (this._INSTANCE) {
      // For partial renders, check to see if the component is handling the property change
      // or if it should do a full render
      var handlePropChangedFun = this._INSTANCE.handlePropertyChanged;
      var fullRender = !handlePropChangedFun || !handlePropChangedFun(property, value);
      if (fullRender && this._INSTANCE.updateDOM) {
        this._INSTANCE.updateDOM();
      }
    }
  }

});

/* jslint browser: true*/
/*
** Copyright (c) 2004, 2012, Oracle and/or its affiliates. All rights reserved.
*/

/**
 * DOM utilities.
 * @ignore
 */
oj.DomUtils = {};

oj.DomUtils._HTML_START_TAG = '\x3chtml\x3e';
oj.DomUtils._HTML_END_TAG = '\x3c/html\x3e';
oj.DomUtils._LEGAL_ELEMENTS = {
  SPAN: 1,
  B: 1,
  I: 1,
  EM: 1,
  BR: 1,
  HR: 1,
  LI: 1,
  OL: 1,
  UL: 1,
  P: 1,
  TT: 1,
  BIG: 1,
  SMALL: 1,
  PRE: 1
};
oj.DomUtils._LEGAL_ATTRIBUTES = { class: 1, style: 1 };

/**
 * Returns true if the value is null or if the trimmed value is of zero length.
 *
 * @param {string|null} content
 * @return {boolean} true if the string is wrapped in <html> tag.
 */
oj.DomUtils.isHTMLContent = function (content) {
  if (content.indexOf(oj.DomUtils._HTML_START_TAG) === 0 &&
      content.lastIndexOf(oj.DomUtils._HTML_END_TAG) === content.length - 7) {
    return true;
  }

  return false;
};

oj.DomUtils.cleanHtml = function (value) {
  var offSpan = $(document.createElement('span')).get(0);
  offSpan.innerHTML = value;  // @HTMLUpdateOK safe manipulation
  if (value && value.indexOf('\x3c') >= 0) {
    oj.DomUtils._cleanElementHtml(offSpan);
  }
  return offSpan;
};

oj.DomUtils._cleanElementHtml = function (node) {
  var children = node.childNodes;

  for (var count = children.length - 1; count >= 0; count--) {
    var child = children.item(count);
    if (child && child.nodeType === 1) {
      if (oj.DomUtils._LEGAL_ELEMENTS[child.nodeName]) {
        var attrs = child.attributes;
        for (var i = attrs.length - 1; i >= 0; i--) {
          var attr = attrs[i];
          // jquery - the .attr() method returns undefined for attributes that have not been set.
          var childHasAttr = $(child).attr(attr.name) !== undefined;
          if (childHasAttr) {
            if (!oj.DomUtils._LEGAL_ATTRIBUTES[attr.name]) {
              child.removeAttribute(attr.nodeName);
            }
          }
        }
        oj.DomUtils._cleanElementHtml(child);
      } else if (child) {
        node.removeChild(child);
      }
    }
  }
};

/**
 * Checks to see if the "ancestorNode" is a ancestor of "node".
 *
 * @param {!Element} ancestorNode dom subtree to check to see if the target node exists
 * @param {!Element} node target node to check to see if it exists within a subtree rooted at the ancestorNode
 * @return {boolean} <code>true</code> if the "ancestorNode" is a ancestor of "node".
 */
oj.DomUtils.isAncestor = function (ancestorNode, node) {
  // These can cause problems in IE11: sometimes the node is just an "empty" object
  // oj.Assert.assertDomElement(ancestorNode);
  // oj.Assert.assertDomElement(node);

  var parentNode = node.parentNode;

  while (parentNode) {
    if (parentNode === ancestorNode) {
      return true;
    }

    parentNode = parentNode.parentNode;
  }

  return false;
};

/**
 * Checks to see if the "ancestorNode" is a ancestor of "node" or if they are the same.
 *
 * @param {!Element} ancestorNode dom subtree to check to see if the target node exists
 * @param {!Element} node target node to check to see if it exists within a subtree rooted at the ancestorNode
 * @return {boolean} <code>true</code> if the "ancestorNode" is a ancestor of "node" or if they are the same
 */
oj.DomUtils.isAncestorOrSelf = function (ancestorNode, node) {
  // These can cause problems in IE11: sometimes the node is just an "empty" object
  // oj.Assert.assertDomElement(ancestorNode);
  // oj.Assert.assertDomElement(node);

  return (node === ancestorNode) ?
          true :
          oj.DomUtils.isAncestor(ancestorNode, node);
};


/**
 * Adds a resize listener for a block or inline-block element
 * @param {!Element} elem - node where the listener should be added
 * @param {!Function} listener - listener to be added. The listener will receive
 * two parameters: 1) the new width in pixels; 2) the new height in pixels
 * @param {number=} collapseEventTimeout - timeout in milliseconds for collapsing
 * multiple resize events into one
 * @export
 */
oj.DomUtils.addResizeListener = function (elem, listener, collapseEventTimeout) {
  var jelem = $(elem);
  var tracker = jelem.data(oj.DomUtils._RESIZE_TRACKER_KEY);
  if (tracker == null) {
    tracker = new oj.DomUtils._ResizeTracker(elem);
    jelem.data(oj.DomUtils._RESIZE_TRACKER_KEY, tracker);
    tracker.start();
  }
  tracker.addListener(listener, collapseEventTimeout);
};

/**
 * Removes a resize listener
 * @param {!Element} elem - node whose listener should be removed
 * @param {!Function} listener - listener to be removed
 * @export
 */
oj.DomUtils.removeResizeListener = function (elem, listener) {
  var jelem = $(elem);
  var tracker = jelem.data(oj.DomUtils._RESIZE_TRACKER_KEY);
  if (tracker != null) {
    tracker.removeListener(listener);
    if (tracker.isEmpty()) {
      tracker.stop();
      jelem.removeData(oj.DomUtils._RESIZE_TRACKER_KEY);
    }
  }
};


/**
 * Fixes resize listeners after a subtree has been connected to the DOM or after
 * its display:none stayle has been removed
 * @param {!Element} subtreeRoot - subtree root
 */
oj.DomUtils.fixResizeListeners = function (subtreeRoot) {
  $(subtreeRoot).find('.oj-helper-detect-expansion').parent().each(
    function (index, div) {
      var tracker = $(div).data(oj.DomUtils._RESIZE_TRACKER_KEY);
      if (tracker != null) {
        tracker.init(true);
      }
    }
  );
};

/**
 * Determines whether a special 'meta' key was pressed when the event was fired.
 * For Mac OS, the 'meta' key is mapped to the 'Command' key, for all other platforms it is mapped
 * to the 'Control' key.
 * Note that this method will only work for the events that support .ctrlKey and .metaKey fields.
 * @param {!Object} evt - the event
 * @return true if the meta key is pressed, false otherwise
 */
oj.DomUtils.isMetaKeyPressed = function (evt) {
  var agentInfo = oj.AgentUtils.getAgentInfo();
  return (oj.AgentUtils.OS.MAC === agentInfo.os ? evt.metaKey : evt.ctrlKey);
};

/**
 * Dispatches an event on the element
 * @param {!Element} element
 * @param {!Event} evt event object
 */
oj.DomUtils.dispatchEvent = function (element, evt) {
  // Workaround for Mozilla issue #329509 - dispatchEvent() throws an error if
  // the element is disabled and disconnected
  // Also, IE simply ignores the .dispatchEvent() call for disabled elements
  var dis = 'disabled';
  var oldDisabled = element[dis];
  try {
    // eslint-disable-next-line no-param-reassign
    element[dis] = false;
    element.dispatchEvent(evt);
  } finally {
    // eslint-disable-next-line no-param-reassign
    element[dis] = oldDisabled;
  }
};

/**
 * @private
 */
oj.DomUtils._invokeAfterPaint = (window.requestAnimationFrame ||
                                 window.mozRequestAnimationFrame ||
                                 window.webkitRequestAnimationFrame ||
                                 function (fn) {
                                   return window.setTimeout(fn, 0);
                                 }
                                ).bind(window);

/**
 * @private
 */
oj.DomUtils._cancelInvokeAfterPaint = (window.cancelAnimationFrame ||
                                       window.mozCancelAnimationFrame ||
                                       window.webkitCancelAnimationFrame ||
                                       function (id) {
                                         return window.clearTimeout(id);
                                       }
                                      ).bind(window);

/**
 * Utility class for tracking resize events for a given element and  sispatching them
 * to listeners
 * @constructor
 * @ignore
 * @private
 */
oj.DomUtils._ResizeTracker = function (div) {
  var _listeners = $.Callbacks();
  var _collapsingManagers = [];
  var _collapsingListeners = [];

  var _RETRY_MAX_COUNT = 2;
  var _retrySetScroll = 0;
  var _invokeId = null;
  var _oldWidth = null;
  var _oldHeight = null;
  var _detectExpansion = null;
  var _detectContraction = null;
  var _resizeListener = null;
  var _scrollListener = null;

  this.addListener = function (listener, collapseEventTimeout) {
    if (collapseEventTimeout === undefined || collapseEventTimeout === 0) {
      _listeners.add(listener);
    } else {
      _collapsingManagers.push(
              new oj.DomUtils._collapsingListenerManager(listener, collapseEventTimeout));
      _collapsingListeners.push(listener);
    }
  };

  this.removeListener = function (listener) {
    var index = _collapsingListeners.indexOf(listener);
    if (index >= 0) {
      _collapsingListeners.splice(index, 1);
      var removed = _collapsingManagers.splice(index, 1);
      removed[0].stop();
    } else {
      _listeners.remove(listener);
    }
  };

  this.isEmpty = function () {
    return !_listeners.has() && _collapsingListeners.length === 0;
  };

  this.start = function () {
    _scrollListener = _handleScroll.bind(this);

    // : Use native onresize support on teh DIV in IE9/10 and  since no scroll events are fired on the
    // contraction/expansion DIVs in IE9
    if (div.attachEvent) {
      _resizeListener = _handleResize.bind(this);
      div.attachEvent('onresize', _resizeListener);
    } else {
      var firstChild = div.childNodes[0];

      // This child DIV will track expansion events. It is meant to be 1px taller and wider than the DIV
      // whose resize events we are tracking. After we set its scrollTop and scrollLeft to 1, any increate in size
      // will fire a scroll event
      _detectExpansion = document.createElement('div');
      _detectExpansion.className = 'oj-helper-detect-expansion';
      var expansionChild = document.createElement('div');
      _detectExpansion.appendChild(expansionChild); // @HTMLUpdateOK; expansionChild constructed by the code above
      if (firstChild != null) {
        div.insertBefore(_detectExpansion, firstChild);// @HTMLUpdateOK; _detectExpansion constructed by the code above
      } else {
        div.appendChild(_detectExpansion);// @HTMLUpdateOK; _detectExpansion constructed by the code above
      }

      _detectExpansion.addEventListener('scroll', _scrollListener, false);

      // This child DIV will track contraction events. Its height and width are set to 200%. After we set its scrollTop and
      // scrollLeft to the current height and width of its parent, any decrease in size will fire a scroll event
      _detectContraction = document.createElement('div');
      _detectContraction.className = 'oj-helper-detect-contraction';

      var contractionChild = document.createElement('div');
      contractionChild.style.width = '200%';
      contractionChild.style.height = '200%';
      _detectContraction.appendChild(contractionChild); // @HTMLUpdateOK; contractionChild constructed by the code above
      div.insertBefore(_detectContraction, _detectExpansion); // @HTMLUpdateOK; _detectContraction constructed by the code above

      _detectContraction.addEventListener('scroll', _scrollListener, false);

      this.init(false);
    }
  };

  this.stop = function () {
    if (_invokeId != null) {
      oj.DomUtils._cancelInvokeAfterPaint(_invokeId);
      _invokeId = null;
    }
    if (_detectExpansion != null) {
      _detectExpansion.removeEventListener('scroll', _scrollListener);
      _detectContraction.removeEventListener('scroll', _scrollListener);
      // Check before removing to prevent CustomElement polyfill from throwing
      // a NotFoundError when removeChild is called with an element not in the DOM
      if (_detectExpansion.parentNode) {
        div.removeChild(_detectExpansion);
      }
      if (_detectContraction.parentNode) {
        div.removeChild(_detectContraction);
      }
    } else {
      // assume IE9/10
      div.detachEvent('onresize', _resizeListener);
    }
  };

  this.init = function (isFixup) {
    var adjusted = _checkSize(isFixup);
    if (isFixup && !adjusted && _detectExpansion.offsetParent != null) {
      _adjust(_oldWidth, _oldHeight);
    }
  };


  function _checkSize(fireEvent) {
    var adjusted = false;
    if (_detectExpansion.offsetParent != null) {
      var newWidth = _detectExpansion.offsetWidth;
      var newHeight = _detectExpansion.offsetHeight;

      if (_oldWidth !== newWidth || _oldHeight !== newHeight) {
        _retrySetScroll = _RETRY_MAX_COUNT;
        _adjust(newWidth, newHeight);
        adjusted = true;

        if (fireEvent) {
          _notifyListeners(true);
        }
      }
    }

    return adjusted;
  }

  function _notifyListeners(useAfterPaint) {
    var newWidth = div.offsetWidth;
    var newHeight = div.offsetHeight;
    if (_listeners.has()) {
      if (!useAfterPaint) {
        _listeners.fire(newWidth, newHeight);
      } else {
        if (_invokeId !== null) {
          oj.DomUtils._cancelInvokeAfterPaint(_invokeId);
        }

        _invokeId = oj.DomUtils._invokeAfterPaint(
        function () {
          _invokeId = null;
          _listeners.fire(newWidth, newHeight);
        }
      );
      }
    }

    for (var i = 0; i < _collapsingManagers.length; i++) {
      _collapsingManagers[i].getCallback()(newWidth, newHeight);
    }
  }

  function _handleScroll(evt) {
    evt.stopPropagation();
    if (!_checkSize(true)) {
      // Workaround for the WebKit issue where scrollLeft gets reset to 0 without the DIV being expanded
      // We will retry to the set the scrollTop only twice to avoid infinite loops
      if (_retrySetScroll > 0 && _detectExpansion.offsetParent != null &&
            (_detectExpansion.scrollLeft === 0 || _detectExpansion.scrollTop === 0)) {
        _retrySetScroll -= 1;
        _adjust(_oldWidth, _oldHeight);
      }
    }
  }

  function _handleResize() {
    _notifyListeners(false);
  }

  function _adjust(width, height) {
    _oldWidth = width;
    _oldHeight = height;

    var expansionChildStyle = _detectExpansion.firstChild.style;

    var delta = 1;

    // The following loop is a workaround for the WebKit issue with zoom < 100% -
    // the scrollTop/Left gets reset to 0 because it gets computed to a value less than 1px.
    // We will try up to the delta of 5 to support scaling down to 20% of the original size
    do {
      expansionChildStyle.width = width + delta + 'px';
      expansionChildStyle.height = height + delta + 'px';
      _detectExpansion.scrollLeft = delta;
      _detectExpansion.scrollTop = delta;
      delta += 1;
    } while ((_detectExpansion.scrollTop === 0 || _detectExpansion.scrollLeft === 0) &&
             delta <= 5);


    _detectContraction.scrollLeft = width;
    _detectContraction.scrollTop = height;
  }
};

oj.DomUtils._RESIZE_TRACKER_KEY = '_ojResizeTracker';


/**
 * Returns true if the name is a valid identifier
 *
 * @param {string} name
 * @return {boolean} true if the name is a valid identifier
 */
oj.DomUtils.isValidIdentifier = function (name) {
  return /^[A-Za-z][0-9A-Z_a-z-]*$/.test(name);
};


/**
 * @constructor
 * @ignore
 */

oj.DomUtils._collapsingListenerManager = function (originalCallback, timeout) {
  var _lastArgs = null;
  var _timerId = null;

  var _timerCallback = function () {
    originalCallback.apply(null, _lastArgs);
    _timerId = null;
  };

  var _callback = function () {
    _lastArgs = Array.prototype.slice.call(arguments);
    if (_timerId == null) {
      _timerId = window.setTimeout(_timerCallback, timeout);
    }
  };

  this.getCallback = function () {
    return _callback;
  };

  this.stop = function () {
    if (_timerId != null) {
      window.clearTimeout(_timerId);
      _timerId = null;
    }
  };
};

/**
 * @return {boolean} true if touch is supported
 */
oj.DomUtils.isTouchSupported = function () {
  return ('ontouchstart' in window) // C, FF, Safari, Edge
    || (navigator.msMaxTouchPoints > 0) // IE10
    || (navigator.maxTouchPoints > 0);  // IE11
};

/**
 * @ignore
 */
oj.DomUtils.setInKoCleanExternal = function (node) {
  oj.DomUtils._koCleanNode = node;
};

/**
 * Delegates to JQuery's unwrap() if the component's node is not currently
 * being removed by Knockout
 * @param {Object} locator
 * @param {Object=} replaceLocator - locator to be replaced. I fthis parameter is ommitted,
 * the parent node will be replaced
 * @ignore
 */
oj.DomUtils.unwrap = function (locator, replaceLocator) {
  var koCleanNode = oj.DomUtils._koCleanNode;


  if (koCleanNode) {
    if (locator.get(0) === koCleanNode) { // skip unwrap
      return;
    }
  }

  if (replaceLocator) {
    replaceLocator.replaceWith(locator); // @HtmlUpdateOk
  } else {
    locator.unwrap();
  }
};

/**
 * Determines if the mouse event target is on browser chrome - i.e. "scrollbar".
 * If the event is not a mouse event with a clientX and clientY, the resultant will
 * be false.
 *
 * @param {Event} event native dom event
 * @returns {boolean} <code>true</code> if the target of the mouse event is browser
 *          chrome such as scrollbars.
 * @public
 */
oj.DomUtils.isChromeEvent = function (event) {
  /**
   * @param {Event} event
   * @return {boolean}
   */
  function _isChromeEventGecko(_event) {
    // assume that if we can't access the original target of the event, then it's because
    // the target was implemented in XUL and is part of the chrome;
    try {
      return !_event.originalTarget.localName;
    } catch (e) {
      return true;
    }
  }

  /**
   * @param {Event} event
   * @return {boolean}
   */
  function _isChromeEventIE(_event) {
    /*
      //IE has a specific API for this but doesn't seem to want to work in automation.
      //The webkit method works in IE too.  Using that over componentFromPoint but leaving
      //the code for future reference.
      //
      var target = event.target;
      var chromePart = target.componentFromPoint(event.clientX, event.clientY);
      if (oj.StringUtils.isEmpty(chromePart))
        return false;
      else
        return true;
    */
    return _isChromeEventWebkit(_event);
  }

  /**
   * @param {Event} event
   * @return {boolean}
   */
  function _isChromeEventWebkit(_event) {
    var domTarget = _event.target;
    var target = $(domTarget);

    var pos = domTarget.getBoundingClientRect();
    var sbw = oj.DomUtils.getScrollBarWidth();
    var isLTR = oj.DomUtils.getReadingDirection() === 'ltr';
    if (isLTR && ((domTarget.nodeName === 'HTML' || target.css('overflow-x') !== 'visible') &&
                  _event.clientX > (pos.right - sbw))) {
      return true;
    } else if (!isLTR && domTarget.nodeName === 'HTML' && _event.clientX > (pos.left - sbw)) {
      // ltr scrollbar is always on the right
      return true;
    } else if (!isLTR && target.css('overflow-x') !== 'visible' &&
             _event.clientX < (pos.left + sbw)) {
      // RTL scrollbar on the document is still on the right
      return true;
    } else if ((domTarget.nodeName === 'HTML' || target.css('overflow-y') !== 'visible') &&
             _event.clientY > (pos.bottom - sbw)) {
      // RTL scrollbar not on the document is on the left
      return true;
    } // below the scrollbar
    return false;
  }

  // verify event is a mouse event
  if (!('clientX' in event) || !('clientY' in event)) {
    return false;
  }

  var agentInfo = oj.AgentUtils.getAgentInfo();

  if (oj.AgentUtils.OS.ANDROID === agentInfo.os ||
      oj.AgentUtils.OS.IOS === agentInfo.os) {
    return false;
  }

  if (oj.AgentUtils.ENGINE.GECKO === agentInfo.engine) {
    return _isChromeEventGecko(event);
  } else if (oj.AgentUtils.ENGINE.WEBKIT === agentInfo.engine ||
             oj.AgentUtils.ENGINE.BLINK === agentInfo.engine) {
    return _isChromeEventWebkit(event);
  }
  if (oj.AgentUtils.BROWSER.IE === agentInfo.browser) {
    return _isChromeEventIE(event);
  }
  return false;
};

/**
 * @returns {number} width of the browser scrollbar
 */
oj.DomUtils.getScrollBarWidth = function () {
  var scrollBarWidth = oj.DomUtils._scrollBarWidth;
  if ($.isNumeric(scrollBarWidth)) {
    return scrollBarWidth;
  }

  /** @type {jQuery} **/
  var scrollBarMeasure = $('<div />');
  $(document.body).append(scrollBarMeasure); // @HTMLUpdateOK; scrollBarMeasure constructed by the code above
  scrollBarMeasure.width(50).height(50)
    .css({
      overflow: 'scroll',
      visibility: 'hidden',
      position: 'absolute'
    });

  /** @type {jQuery} **/
  var scrollBarMeasureContent = $('<div />');
  scrollBarMeasureContent.height(1);
  scrollBarMeasure.append(scrollBarMeasureContent);  // @HTMLUpdateOK; scrollBarMeasureContent constructed by the code above

  var insideWidth = scrollBarMeasureContent.width();
  var outsideWitdh = scrollBarMeasure.width();
  scrollBarMeasure.remove();

  scrollBarWidth = outsideWitdh - insideWidth;
  oj.DomUtils._scrollBarWidth = scrollBarWidth;
  return scrollBarWidth;
};

/**
 * @returns {string!} "rtl" or "ltr"
 */
oj.DomUtils.getReadingDirection = function () {
  var dir = document.documentElement.getAttribute('dir');
  if (dir) {
    dir = dir.toLowerCase();
  }
  return (dir === 'rtl') ? 'rtl' : 'ltr';
};

/**
 * Retrieve the bidi independent position of the horizontal scroll position that
 * is consistent across all browsers.
 * @param {Element} elem the element to retrieve the scrollLeft from
 * @return {number} the element's scrollLeft
 */
oj.DomUtils.getScrollLeft = function (elem) {
  if (oj.DomUtils.getReadingDirection() === 'rtl') {
    var browser = oj.AgentUtils.getAgentInfo().browser;
    if (browser === oj.AgentUtils.BROWSER.FIREFOX ||
        browser === oj.AgentUtils.BROWSER.IE ||
        browser === oj.AgentUtils.BROWSER.EDGE) {
      return Math.abs(elem.scrollLeft);
    }

      // webkit
    return Math.max(0, elem.scrollWidth - elem.clientWidth - elem.scrollLeft);
  }

  return elem.scrollLeft;
};

/**
 * Sets the bidi independent position of the horizontal scroll position that
 * is consistent across all browsers.
 * @param {Element} elem the element to set the scrollLeft on
 * @param {number} scrollLeft the element's new scrollLeft
 */
oj.DomUtils.setScrollLeft = function (elem, scrollLeft) {
  if (oj.DomUtils.getReadingDirection() === 'rtl') {
    var browser = oj.AgentUtils.getAgentInfo().browser;
    if (browser === oj.AgentUtils.BROWSER.FIREFOX) {
      // see mozilla , even though it's marked as fixed, they basically
      // did not change anything.  It still expects a negative value for RTL
      // eslint-disable-next-line no-param-reassign
      elem.scrollLeft = -scrollLeft;
    } else if (browser === oj.AgentUtils.BROWSER.IE ||
               browser === oj.AgentUtils.BROWSER.EDGE) {
      // eslint-disable-next-line no-param-reassign
      elem.scrollLeft = scrollLeft;
    } else {
      // webkit
      // eslint-disable-next-line no-param-reassign
      elem.scrollLeft = Math.max(0, elem.scrollWidth - elem.clientWidth - scrollLeft);
    }
  } else {
    // eslint-disable-next-line no-param-reassign
    elem.scrollLeft = scrollLeft;
  }
};

/**
 * Converts a CSS length attribute into a integer value.
 * Conversion errors or non-number will result in a zero
 * resultant.
 *
 * @param {?} cssLength style attribute
 * @return {number} value as integer
 */
oj.DomUtils.getCSSLengthAsInt = function (cssLength) {
  if (!isNaN(cssLength)) {
    return parseInt(cssLength, 10);
  }

  if (cssLength && cssLength.length > 0 && cssLength !== 'auto') {
    var intLength = parseInt(cssLength, 10);

    if (isNaN(intLength)) {
      intLength = 0;
    }

    return intLength;
  }

  return 0;
};

/**
 * Converts a CSS attribute into a float value.
 * Conversion errors or non-number will result in a zero
 * resultant.
 *
 * @param {?} cssLength style attribute
 * @return {number} value as integer
 */
oj.DomUtils.getCSSLengthAsFloat = function (cssLength) {
  if (!isNaN(cssLength)) {
    return parseFloat(cssLength);
  }

  if (cssLength && cssLength.length > 0) {
    var floatLength = parseFloat(cssLength);

    if (isNaN(floatLength)) {
      floatLength = 0;
    }

    return floatLength;
  }

  return 0;
};

/**
 * Key used to store the logical parent of the popup element
 * as a jQuery data property. The logical parent refers the launcher of a popup.
 * @const
 * @private
 * @type {string}
 */
oj.DomUtils._LOGICAL_PARENT_DATA = 'oj-logical-parent';

/**
 * This method returns the launcher of a popup when it's open.
 * Returns undefined otherwise.
 *
 * @param {jQuery} element jquery element
 * @returns {any}
 * @see #setLogicalParent
 */
oj.DomUtils.getLogicalParent = function (element) {
  if (element) {
    return element.data(oj.DomUtils._LOGICAL_PARENT_DATA);
  }

  return undefined;
};

/**
 * Set the logical parent as a jQuery data property
 *
 * @param {jQuery} element jquery element
 * @param {jQuery | null} parent jquery element
 * @see #getLogicalParent
 */
oj.DomUtils.setLogicalParent = function (element, parent) {
  if (!element) {
    return;
  }

  if (parent === null) {
    element.removeData(oj.DomUtils._LOGICAL_PARENT_DATA);
  } else {
    element.data(oj.DomUtils._LOGICAL_PARENT_DATA, parent);
  }
};

/**
 * Checks to see if the "ancestorNode" is a logical ancestor of "node"
 *
 * @param {!Element} ancestorNode dom subtree to check to see if the target node exists
 * @param {!Element} node target node to check to see if it exists within a subtree rooted at the ancestorNode
 * @return {boolean} <code>true</code> if the "ancestorNode" is a logical ancestor of "node" or if they are the same
 */
oj.DomUtils.isLogicalAncestorOrSelf = function (ancestorNode, node) {
  oj.Assert.assertDomElement(ancestorNode);
  oj.Assert.assertDomElement(node);

  var parentNode = node;
  while (parentNode) {
    if (parentNode === ancestorNode) {
      return true;
    }

    var logicalParent = oj.DomUtils.getLogicalParent($(parentNode));
    if (logicalParent) {
      parentNode = logicalParent[0];
    } else {
      parentNode = parentNode.parentNode;
    }
  }

  return false;
};


/**
 * Checks whether the href represents a safe URL
 * @param {!string} href - HREF to test
 * @param {Array=} whitelist - optional list of the allowed protocols. Protocol name has to use lowercase letters and
 * be followed by a ':'. If the parameter is ommitted, ['http:', 'https:'] will be used
 * @throws {Exception} an error if the HREF represents an invalid URL
 * @ignore
 */
oj.DomUtils.validateURL = function (href, whitelist) {
  var allowed = whitelist || ['http:', 'https:'];

  var link = document.createElement('a');
  link.href = href;

  var protocol = link.protocol;
  if (protocol != null) {
    protocol = protocol.toLowerCase();
  }
  // if it isn't on the allowed list and it isn't '', throw an error.
  // IE11 returns '' for hrefs like 'abc', other browsers return 'https'
  // and we want to allow hrefs like 'abc' since those are relative urls.
  if ((allowed.indexOf(protocol) < 0) && (protocol !== '')) {
    throw new Error(protocol + ' is not a valid URL protocol');
  }
};

/**
 * Cancels native context menu events for hybrid mobile applications.
 * @private
 */
oj.DomUtils._supressNativeContextMenu = function () {
  if ($(document.body).hasClass('oj-hybrid')) {
    document.body.addEventListener('contextmenu',
      function (event) {
        event.preventDefault();
      }, true);
  }
};
oj.DomUtils._supressNativeContextMenu();

// standard duration of a pressHold gesture.  Point of reference: default
// JQ Mobile threshold to be a press-and-hold is 750ms.
oj.DomUtils.PRESS_HOLD_THRESHOLD = 750;


// ------------------------------------------------------------------------------------------------
// Recent touch end
// ------------------------------------------------------------------------------------------------

/**
 * Returns true if a touchend or touchcancel has been detected anywhere in the document in the last 500 ms.
 * Note: This function adds event listeners only once per document load.
 *
 * @return {boolean} boolean indicating whether a touch has recently been detected
 */
oj.DomUtils.recentTouchEnd = (function () {
  // This function is immediately executed and returns the recentTouchEnd function
  // and therefore only execute once per document load.

  var touchTimestamp = 0;
  var TOUCH_THRESHOLD = 500;

  function _touchEndHandler() {
    touchTimestamp = Date.now();
  }

  // --- Document listeners ---
  document.addEventListener('touchend', _touchEndHandler, true);
  document.addEventListener('touchcancel', _touchEndHandler, true);

  // --- The function assigned to oj.DomUtils.recentTouchEnd ---

  return function () {
    // must be at least 300 for the "300ms" delay
    return (Date.now() - touchTimestamp) < TOUCH_THRESHOLD;
  };
}());

/**
 * Returns true if a touchstart has been detected anywhere in the document in the last 800 ms.
 * Note: This function adds event listeners only once per document load.
 *
 * @return {boolean} boolean indicating whether a touch has recently been detected
 */
oj.DomUtils.recentTouchStart = (function () {
  // This function is immediately executed and returns the recentTouchStart function
  // and therefore only execute once per document load.

  var touchTimestamp = 0;
  // 800 because this is used to ignore mouseenter and focusin on 'press', and a 'press'
  // is usually detected after 750ms.
  var TOUCH_THRESHOLD = oj.DomUtils.PRESS_HOLD_THRESHOLD + 50;

  function _touchStartHandler() {
    touchTimestamp = Date.now();
  }

  // --- Document listeners ---
  document.addEventListener('touchstart', _touchStartHandler, true);

  // --- The function assigned to oj.DomUtils.recentTouchStart ---

  return function () {
    // must be at least TOUCH_THRESHOLD for the  delay
    return (Date.now() - touchTimestamp) < TOUCH_THRESHOLD;
  };
}());

// ------------------------------------------------------------------------------------------------
// Recent pointer
// ------------------------------------------------------------------------------------------------

/**
 * Returns true if a touchstart, touchend, mousedown, or mouseup has been detected anywhere in the
 * document in the last n ms, where n is calibrated across a variety of platforms to make this API
 * a maximally reliable indicator of whether the code now running was likely "caused by" the
 * specified touch and mouse interaction, vs. some other thing (e.g. mousemove, keyboard, or page
 * load).  E.g. the makeFocusable() / _focusable() mechanism uses this API to vary the focus theming
 * depending on whether the element was focused via keyboard or pointer.
 *
 * @return {boolean} boolean indicating whether a mouse button or finger has recently been down or up
 */
oj.DomUtils.recentPointer = (function () {
  // The comments in this function are tailored to the makeFocusable() usage.

  // - Let "pointer down" mean mousedown or touchstart, and "pointer up" likewise.  (Not MS pointer events.)
  // - Event order can be 1) mousedown>focus>mouseup (like push buttons) or 2) mousedown>mouseup>focus (like toggle buttons).
  // - For 2, semantics for "focus caused by pointer" must be "if pointer interaction in last n ms," rather than "if pointer is currently down".
  // - Those "last n ms" semantics are preferred for 1 as well, rather than relying on pointer up to cancel a state set by pointer down,
  //   since if the pointer up is never received, we'd get stuck in an inaccessible state.
  // - So both pointer down and pointer up set a timestamp, and recentPointer() returns true if Date.now() is within n ms of that timestamp,
  //   where n is higher for touchstart per below.

  // Timestamp of last mousedown/up or touchstart/end. Initial value of 0 (1/1/1970) guarantees that if element is focused before any
  // mouse/touch interaction, then recentPointer() is false, so focus ring appears as desired.
  var pointerTimestamp = 0;

  var pointerTimestampIsTouchStart; // whether the latest timestamp is for touchstart vs. touchend/mouse

  // On Edge (Surface Win10), the lag from the up event to resulting programmatic focus is routinely ~350ms, even when the 300ms "tap delay" has
  // been prevented and confirmed to be absent.  (In Chrome on same device the same lag is ~10 ms.)  So use 600ms to be safe.  Even on Chrome,
  // the lag from the down/up event to natively induced focus can routinely be well into the 1xx ms range. Can exceed 600 if needed. There is no
  // need for a tight bound; if there was pointer interaction in the last second or so, it's perfectly reasonable to suppress the focus ring.
  var POINTER_THRESHOLD_CUSHION = 600;

  // If the number of millis since the last pointer down or up is < this threshold, then recentPointer() considers it recent and returns true.
  // See also TOUCHSTART_THRESHOLD.
  var POINTER_THRESHOLD = POINTER_THRESHOLD_CUSHION;

  // For touchstart only, use 750+600ms so that focus set by a 750ms pressHold gesture (e.g. context menu) is recognized as touch-related.  Same
  // 600ms padding as for POINTER_THRESHOLD.  A high threshold is OK, as it is used only for actual pressHolds (and the unusual case where the
  // pointer up is never received), since for normal clicks and taps, the pointerUp replaces the "1350ms after touchstart" policy with a "600ms
  // after pointerUp" policy. On Edge and desktop FF (desktop version runs on hybrid devices like Surface), which lack touchstart, context menus
  // are launched by the contextmenu event, which happen after the pointer up in both browsers, so the fact that we're using the higher
  // threshold only for touchstart should not be a problem there.
  var TOUCHSTART_THRESHOLD = oj.DomUtils.PRESS_HOLD_THRESHOLD + POINTER_THRESHOLD_CUSHION;


  // --- Document listeners ---

  // Use capture phase to make sure we hear the events before someone cancels them
  document.addEventListener('mousedown', function () {
    // If the mousedown immediately follows a touchstart, i.e. if it seems to be the compatibility mousedown
    // corresponding to the touchstart, then we want to consider it a "recent pointer activity" until the end time
    // that is max(touchstartTime + TOUCHSTART_THRESHOLD, now + POINTER_THRESHOLD), where now is mousedownTime in this
    // case.  (I.e. it would defeat the purpose if the inevitable mousedown replaced the longer touchstart threshold with
    // a shorter one.)  We don't do this in the touchend/mouseup listeners, as those obviously happen after the pressHold
    // is over, in which case the following analysis applies:
    // - If the pressHold was < PRESS_HOLD_THRESHOLD ms,
    // - then the higher TOUCHSTART_THRESHOLD is not needed or relevant, since anything focused on pressHold
    //   (like a context menu) never happened,
    // - else the touchend/mouseup happened > PRESS_HOLD_THRESHOLD ms after the touchstart, so in the max() above,
    //   the 2nd quantity is always bigger (later).
    var now = Date.now();
    if ((!pointerTimestampIsTouchStart) ||
        (now > pointerTimestamp + oj.DomUtils.PRESS_HOLD_THRESHOLD)) {
      pointerTimestamp = now;
      pointerTimestampIsTouchStart = false;
    }
  }, true);

  document.addEventListener('touchstart', function () {
    pointerTimestamp = Date.now();
    pointerTimestampIsTouchStart = true;
  }, true);

  document.addEventListener('mouseup', function () {
    pointerTimestamp = Date.now();
    pointerTimestampIsTouchStart = false;
  }, true);

  document.addEventListener('touchend', function () {
    pointerTimestamp = Date.now();
    pointerTimestampIsTouchStart = false;
  }, true);


  // --- The function assigned to oj.DomUtils.recentPointer ---

  return function () {
    var millisSincePointer = Date.now() - pointerTimestamp;
    var threshold = pointerTimestampIsTouchStart ? TOUCHSTART_THRESHOLD : POINTER_THRESHOLD;
    var isRecent = millisSincePointer < threshold;
    return isRecent;
  };
}());


// ------------------------------------------------------------------------------------------------
// Utility for suppressing focus ring for mouse/touch interaction, but not KB or other interaction:
// ------------------------------------------------------------------------------------------------

/**
 * This API works like baseComponent's _focusable() API (see its detailed JSDoc), with the
 * similarities and differences listed below.  This API is intended for non-component callers;
 * components should typically call the baseComponent API via this._focusable().
 *
 * Comparison to baseComponent._focusable() :
 *
 * - This function's "options" param must be an object.  Only baseComponent._focusable()
 *   supports the backward-compatibility syntax where the options param can be the element.
 * - Same usage of oj-focus, oj-focus-highlight, and $focusHighlightPolicy.
 * - Same required invariant that oj-focus-highlight must not be set if oj-focus is not set.
 * - Same parameters with same semantics, plus the additional "component" and "remove" params
 *   discussed below.
 * - New options.component param, which takes a JET component instance.  (When a component is
 *   involved, typically that component should call this._focusable() rather than calling this
 *   version of the method directly.)
 *
 * If options.component is specified, then the following things work like the baseComponent
 * version of this API:
 *
 * - If the specified element is in the component subtree,
 *   then the classes will automatically be removed when the component is
 *   destroyed/disabled/detached, as detailed in the baseComponent JSDoc,
 *   else the caller has the same responsibility to remove the classes at those times.
 * - Same rules as to whether listeners are automatically cleaned up, or suppressed when the
 *   component is disabled, vs. being the caller's responsibility to handle those things.
 *
 * If options.component is NOT specified (for non-component callers), then those things are
 * the caller's responsibility.  Specifically:
 *
 * - Class removal can be done directly, as needed.
 * - To remove the listeners, see the following.
 *
 * Listener removal:
 *
 * - If options.component was specified, see above.
 * - Else if options.setupHandlers was specified, then only the caller knows what listeners were
 *   registered and how, so it is the caller's responsibility to remove them directly when needed.
 * - The remaining case is that options.component and options.setupHandlers were not specified.
 *   To remove from element e both the 2 classes and all listeners applied to e by all previous
 *   invocations of makeFocusable() where these options were not specified,
 *   call makeFocusable( {'element': e, 'remove': true} ).
 */
// If this is named focusable(), Closure Compiler generates a warning, and fails to rename the function in minified code,
// which suggests that focusable (not just _focusable) is apparently externed somewhere (although not in
// 3rdparty\jquery\externs\jquery-1.8.js, main\javascript\externs.js, or build\tools\closure\compiler.jar\externs.zip\),
// perhaps for JQUI's :focusable selector.  So name it makeFocusable().
oj.DomUtils.makeFocusable = (function () {
  var nextId = 0; // used for unique namespace, for "remove" functionality

  // This private var is shared by all callers that use makeFocusable() and don't supply their own focus highlight policy.
  // If the oj-focus-config SASS object ever acquires a 2nd field, should continue to call pJFFF() only once, statically.
  var FOCUS_HIGHLIGHT_POLICY = (oj.ThemeUtils.parseJSONFromFontFamily('oj-focus-config') || {})
      .focusHighlightPolicy;

  /**
   * @param {function()} focusPolicyCallback Optional getter passed to makeFocusable() by callers wishing to get use a caller-
   *   specific focus policy mechanism instead of the built-in mechanism.
   * @param {function()} recentPointerCallback Optional function passed to makeFocusable() by callers wishing to use a caller-
   *   specific mechanism in addition to the built-in mechanism.
   * @return {boolean} boolean indicating whether it is appropriate to apply the <code class="prettyprint">oj-focus-highlight</code>
   *   CSS class for a focus happening at the time of this method call.
   */
  var shouldApplyFocusHighlight = function (focusPolicyCallback, recentPointerCallback) {
    var focusHighlightPolicy = focusPolicyCallback ? focusPolicyCallback() : FOCUS_HIGHLIGHT_POLICY;
    switch (focusHighlightPolicy) {
      case 'all':
        return true;
      case 'none':
        return false;
      default: // "nonPointer" or no value provided (e.g. SASS var missing)
        return !(oj.DomUtils.recentPointer() ||
                 (recentPointerCallback && recentPointerCallback()));
    }
  };

  // the function assigned to oj.DomUtils.makeFocusable
  var makeFocusable = function (options) {
    var element = options.element;

    var dataKey = 'ojFocusable';
    var namespacePrefix = '.' + dataKey;
    var namespaceSeparator = ' ' + namespacePrefix;

    if (options.remove) {
      element.removeClass('oj-focus oj-focus-highlight');

      // id's of listeners needing removal
      var ids = element.data(dataKey);
      if (ids == null) {
        return;
      }

      // map ids to namespaces.  "2" -> ".ojFocusable2".  "2,7" -> ".ojFocusable2 .ojFocusable7"
      var namespaces = namespacePrefix + ('' + ids).split(',').join(namespaceSeparator);
      element.off(namespaces) // remove the listeners
             .removeData(dataKey); // clear list of listener id's needing removal
      return;
    }

    var afterToggle = options.afterToggle || $.noop;

    function applyOnlyFocus(_element) {
      _element.addClass('oj-focus');
      afterToggle('focusin');
    }

    function applyBothClasses(_element) {
      _element.addClass('oj-focus');
      if (shouldApplyFocusHighlight(options.getFocusHighlightPolicy, options.recentPointer)) {
        _element.addClass('oj-focus-highlight');
      }
      afterToggle('focusin');
    }

    var addClasses = options.applyHighlight ? applyBothClasses : applyOnlyFocus;

    function removeClasses(_element) {
      _element.removeClass('oj-focus oj-focus-highlight');
      afterToggle('focusout');
    }

    var setupHandlers = options.setupHandlers || function (focusInHandler, focusOutHandler) {
      var component = options.component;
      var focusInListener = function (event) {
        focusInHandler($(event.currentTarget));
      };
      var focusOutListener = function (event) {
        focusOutHandler($(event.currentTarget));
      };

      if (component) {
        component._on(element, {
          focusin: focusInListener,
          focusout: focusOutListener
        });
      } else {
        // neither options.component nor options.setupHandlers were passed, so we must provide a
        // way for the caller to remove the listeners.  That's done via the "remove" param, which
        // uses the namespaces that we stash via data().
        var id = nextId;
        nextId += 1;

        // list of id's of existing listeners needing removal
        var _ids = element.data(dataKey);

        // append id to that list, or start new list if first one
        element.data(dataKey, _ids == null ? id : _ids + ',' + id);

        // add listeners namespaced by that id
        var handlers = {};
        var namespace = namespacePrefix + id;
        handlers['focusin' + namespace] = focusInListener;
        handlers['focusout' + namespace] = focusOutListener;
        element.on(handlers);
      }
    };

    setupHandlers(addClasses, removeClasses);
  };

  return makeFocusable;
}());

/* jslint browser: true*/
/*
** Copyright (c) 2015, Oracle and/or its affiliates. All rights reserved.
*/

/**
 * Focus utilities.
 * @ignore
 */
oj.FocusUtils = {};

oj.FocusUtils._TABBABLE = ':tabbable,iframe';

// These functions inspired by AdfFocusUtils

/**
 * Tests whether the specified element contains the keyboard focus.
 * @param {!Element} element Element for which to check if it contains focus.
 * @returns {boolean} True if the element contains focus, false otherwise.
 */
oj.FocusUtils.containsFocus = function (element) {
  var activeElem = document.activeElement;
  // FIX : if either elem is undefined, just return false
  if (!element || !activeElem) {
    return false;
  }

  return oj.DomUtils.isAncestorOrSelf(element, activeElem);
};

/**
 * Sets focus to the specified element.
 * @param {!Element} element Element to focus.
 */
oj.FocusUtils.focusElement = function (element) {
  element.focus();
};

/**
 * Sets focus to the first tabbable element inside the given element, which
 * may be the given element itself.
 * @param {!Element} element Element to start searching for a tabbable element in.
 * @returns {Element} The DOM element that was focused, if any.
 */
oj.FocusUtils.focusFirstTabStop = function (element) {
  var focusElement = oj.FocusUtils.getFirstTabStop(element);

  if (focusElement) {
    oj.FocusUtils.focusElement(focusElement);
  }

  return focusElement;
};

/**
 * Return true if the activeElement is the first tabbable. Used to ensure that tabbing cycles through dialogs/popups.
 * @param {!Element} element Element containing tabbable elements.
 * @returns {boolean} <code>true</code> if the active element is the first tabbable.
 */
oj.FocusUtils.isFirstActiveElement = function (element) {
  var jqElem = $(element);
  var jqFocusables = jqElem.find(oj.FocusUtils._TABBABLE);

  if (jqFocusables == null || jqFocusables.length === 0) return false;
  var first = jqFocusables[0];

  if (document.activeElement === first) return true;

  //
  // Return true if the activeElement is in the "first tabble set".
  // Check to see if the first tabbable and the active element are members
  // of the same radio set.
  // If this is the case, then return true.
  //
  if (first.name === document.activeElement.name && first.type === 'radio' && document.activeElement.type === 'radio') {
    return true;
  }
  return false;
};

/**
 * Return true if the activeElement is the last tabbable. Used to ensure that tabbing cycles through dialogs/popups.
 * @param {!Element} element Element containing tabbable elements.
 * @returns {boolean} <code>true</code> if the active element is the last tabbable.
 */
oj.FocusUtils.isLastActiveElement = function (element) {
  var jqElem = $(element);
  var jqFocusables = jqElem.find(oj.FocusUtils._TABBABLE);

  if (jqFocusables == null || jqFocusables.length === 0) return false;
  var last = jqFocusables[jqFocusables.length - 1];

  if (document.activeElement === last) return true;

  //
  // Return true if the activeElement is in the "first tabble set".
  // Check to see if the last tabbable and the active element are members
  // of the same radio set.
  // If this is the case, then return true.
  //
  if (last.name === document.activeElement.name && last.type === 'radio' && document.activeElement.type === 'radio') {
    return true;
  }
  return false;
};

/**
 * Get the first tabbable element inside the given element, which may be the
 * given element itself.
 * @param {!Element} element Element to start searching for a tabbable element in.
 * @returns {Element} The first tabbable element inside the given element.
 */
oj.FocusUtils.getFirstTabStop = function (element) {
  var jqElem = $(element);
  if (jqElem.is(oj.FocusUtils._TABBABLE)) {
    return element;
  }
  var jqFocusables = jqElem.find(oj.FocusUtils._TABBABLE);
  if (jqFocusables && jqFocusables.length > 0) {
    //
    // Handle set-based content (radiosets).
    // Return the first selected radioset item.
    // Note that there are two cases
    //   Common case - a single radioset
    //   Other case - multiple radiosets
    // In both cases we return the first selected radioset item.
    //
    if (jqFocusables[0].classList.contains('oj-radio')) {
      var selectedItem = jqFocusables.filter('.oj-selected.oj-radio');
      if (selectedItem && selectedItem.length) {
        return selectedItem[0];
      }
      return jqFocusables[0];
    }
    return jqFocusables[0];
  }
  return null;
};

/**
 * Get the last tabbable element inside the given element, which may be the
 * given element itself.
 * @param {!Element} element Element to start searching for a tabbable element in.
 * @returns {Element} The last tabbable element inside the given element.
 */
oj.FocusUtils.getLastTabStop = function (element) {
  var jqElem = $(element);
  var jqFocusables = jqElem.find(oj.FocusUtils._TABBABLE);

  if (jqFocusables && jqFocusables.length > 0) {
    //
    // Handle set-based content (radiosets).
    // Return the last selected radioset item.
    //
    if (jqFocusables[jqFocusables.length - 1].classList.contains('oj-radio')) {
      var selectedItem = jqFocusables.filter('.oj-selected.oj-radio');
      if (selectedItem && selectedItem.length) {
        return selectedItem[selectedItem.length - 1];
      }
      return jqFocusables[jqFocusables.length - 1];
    }
    return jqFocusables[jqFocusables.length - 1];
  }
  return null;
};

/**
 * Extends the jquery ":focusable" pseudo selector check for a Safari browser specific
 * exception - an anchor element not having a tabindex attribute.
 *
 * @param {Element} element target dom element to test if it will take focus
 * @returns {boolean} <code>true</code> if the target element is focusable
 */
oj.FocusUtils.isFocusable = function (element) {
  if ($(element).is(':focusable')) {
    // An anchor element in safari will not take focus unless it has a tabindex.
    // http://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#Clicking_and_focus
    if (element.nodeName === 'A' && !element.hasAttribute('tabindex') &&
        oj.AgentUtils.getAgentInfo().browser === oj.AgentUtils.BROWSER.SAFARI) {
      return false;
    }
    return true;
  }

  return false;
};

/* jslint browser: true*/
/*
** Copyright (c) 2017, Oracle and/or its affiliates. All rights reserved.
*/

/**
 * Gesture utilities provided internally for JET components, currently only context menu gesture are available.
 * Moved from ojcomponentcore and made into static methods.
 * @ignore
 */
oj.GestureUtils = {};

/**
 * Event namespace used by context menu internal event registration.
 * Previously we got the namespace from the widget.
 */
oj.GestureUtils._EVENT_NAMESPACE = '.contextMenu';

/**
 * Utility method to tear down any artifacts created by GestureUtils.startDetectContextMenuGesture
 * @param {Element} rootNode the root element of the component
 */
oj.GestureUtils.stopDetectContextMenuGesture = function (rootNode) {
  if (rootNode._clickListener) {
    $(rootNode).off(oj.GestureUtils._EVENT_NAMESPACE)
      .removeClass('oj-menu-context-menu-launcher')[0]
      .removeEventListener('click', rootNode._clickListener, true);

    // the other 2 contextMenu timeouts don't need to be cleared here
    clearTimeout(rootNode._contextMenuPressHoldTimer);

    // eslint-disable-next-line no-param-reassign
    delete rootNode._clickListener;
    // eslint-disable-next-line no-param-reassign
    delete rootNode._contextMenuPressHoldTimer;
  }
};

/**
 * Utility method to setup context menu gesture detection on a component
 * @param {Element} rootNode the root node of the component
 * @param {function(Event, string)} callback callback to invoke on the component when context menu gesture is detected
 */
oj.GestureUtils.startDetectContextMenuGesture = function (rootNode, callback) {
  // Note: Whether or not we use Hammer to detect press-hold, this code would need to do the following things seen below:
  //
  // (1) Prevent the compatibility mousedown event from triggering Menu's clickAway logic.
  // (2) Prevent press-hold from also generating a click (unless Hammer does this automatically; I'm guessing it doesn't).
  // (3) Ensure we don't respond to *both* press-hold and contextmenu events on Android.
  //
  // So the only thing that Hammer would replace is:
  //
  // (4) Detecting the press-hold.
  //
  // Not currently using Hammer for (4), since:
  //
  // - This code predates Hammer, and was already stable after extensive iteration / fine-tuning.
  // - We use the same listeners for parts of 1-4. If moved 4 off to Hammer (separate listener), just need to ensure that
  //   we don't introduce any race conditions, etc.  (May be easy or hard, just need to look.)
  // - Hammer only wants to have one instance per DOM node, else they fight to control some things like touch-action. So
  //   a prereq for having this baseComponent logic put Hammer on components is to work out a protocol for super- and sub-
  //   classes to share the same instance and not step on each other.  Not insurmountable; just need to have the conversation.
  //   Tracked by ER 21357133, which links to detailed wiki.

  var pressHoldThreshold = oj.DomUtils.PRESS_HOLD_THRESHOLD; // launch CM at 750ms per UX spec
  var isPressHold = false; // to prevent pressHold from generating a click
  var contextMenuPressHoldTimer;

  var touchInProgress = false;

  // 5px is Hammer default.  (Didn't check whether they apply that separately to x and y like us, or to the hypotenuse,
  // but it's within a couple px either way -- think 3-4-5 triangle.)
  var maxAllowedMovement = 5;
  var touchPageX;
  var touchPageY;

  var doubleOpenTimer; // to prevent double open.  see usage below.
  var doubleOpenThreshold = 300; // made up this number.  TBD: Tweak as needed to make all platforms happy.
  var doubleOpenType = null; // "touchstart" or "contextmenu"

  var namespace = oj.GestureUtils._EVENT_NAMESPACE;

  var contextMenuPressHoldJustEnded = false;

  function launch(event, eventType, pressHold) {
    // ensure that pressHold doesn't result in a click.  Set this before the bailouts below.
    isPressHold = pressHold;

    // In Mobile Safari only, mousedown fires *after* the touchend, which causes at least 2 problems:
    // 1) CM launches after 750ms (good), then disappears when lift finger (bad), because touchend -->
    // mousedown, which calls Menu's "clikAway" mousedown listener, which dismisses Menu.
    // 2) The isPressHold logic needs to reset the isPressHold ivar on any event that can start a click,
    // including mousedown.  This problem causes the mousedown listener to incorrectly clear the ivar
    // after a pressHold, which broke the whole mechanism.
    // SOLUTION FOR 1-2:  On each launch (at 750ms), set a one-time touchend listener that will set a
    // var and clear it 50ms later.  While the var is set, both mousedown listeners can disregard the
    // mousedown.  Make the var a static var in Menu, since Menu's listener is static, and since this
    // launcher component can get/set it via an (effectively static) menu method.
    // NON-SOLUTIONS:  Cancelling touchstart or touchend, via pD() and sP(), doesn't cancel iPad's mousedown.
    // Cancelling mousedown from here doesn't work even if capture phase, since ojMenu's listener is capture phase.
    // TIMING: The following block should be before the doubleOpen bailout.
    if (isPressHold) {
      $(rootNode).one('touchend' + namespace, function () {
        var touchendMousedownThreshold = 50; // 50ms.  Make as small as possible to prevent unwanted side effects.
        contextMenuPressHoldJustEnded = true;
        setTimeout(function () {
          contextMenuPressHoldJustEnded = false;
        }, touchendMousedownThreshold);
      });
    }

    // On platforms like Android Chrome where long presses already fire the contextmenu event, the pressHold
    // logic causes the menu to open twice, once for the pressHold, once for the contextmenu.  There's no
    // guarantee which will happen first, but as long as they happen within doubleOpenThreshold ms
    // of each other, this logic should prevent the double open.
    // Note: Another option is a platform-specific solution where we only use pressHold for platforms that need
    // it (that don't already fire a contextmenu event for pressHold), but architectural preference is to avoid
    // platform-specific solutions if possible.
    if ((doubleOpenType === 'touchstart' && event.type === 'contextmenu')
        || (doubleOpenType === 'contextmenu' && event.type === 'touchstart')
        || (doubleOpenType === 'keydown' && event.type === 'contextmenu')) {
      // FF 60.2.2esr (32-bit) Win fires a rogue contextmenu event following the prevented keydown. What's odd is
      // preventing the keydown for shift+F10 prevents keypress but still files the contextmenu event.
      // Seems like "fallout" (behavior not yet correct) from bug https://bugzilla.mozilla.org/show_bug.cgi?id=1382199
      // For this case, prevent the native context menu within double open timeout window
      if (doubleOpenType === 'keydown' && event.type === 'contextmenu') event.preventDefault();
      doubleOpenType = null;
      clearTimeout(doubleOpenTimer);
      return;
    }

    // If a nested element or component already showed a JET context menu for this event, don't replace it with ours.
    // Hack: must check defaultPrevented on the nested event too, because for touchstart events on iOS7 at least, when
    // the outer component reaches this point, event is a different JQ wrapper event than the one on which the inner
    // component previously called preventDefault, although they both wrap the same native originalEvent.  The new wrapper
    // never had its isDefaultPrevented field set to the returnTrue method, so must check the nested originalEvent.
    // This never seems to happen with right-click and Shift-F10 events.  Has nothing to do with the setTimeout: the events
    // received by the rootNode.on("touchstart"...) code are different (firstWrapper==secondWrapper returns false).
    // TODO: link to JQ bug once filed.
    if (event.isDefaultPrevented() ||
        (event.originalEvent && event.originalEvent.defaultPrevented)) {
      return;
    }

    callback(event, eventType);

    // if _NotifyContextMenuGesture() (or subclass override of it) actually opened the CM, and if that launch wasn't
    // cancelled by a beforeOpen listener...
    if (event.isDefaultPrevented()) {
      // see double-open comments above
      if (event.type === 'touchstart' || event.type === 'contextmenu' || event.type === 'keydown') {
        doubleOpenType = event.type;
        doubleOpenTimer = setTimeout(function () {
          doubleOpenType = null;
        }, doubleOpenThreshold);
      }
    }
  }

  // At least some of the time, the pressHold gesture also fires a click event same as a short tap.  Prevent that here.
  var _clickListener = function (event) {
    if (isPressHold) {
      // For Mobile Safari capture phase at least, returning false doesn't work; must use pD() and sP() explicitly.
      event.preventDefault();
      event.stopPropagation();
      isPressHold = false;
    }
  };

  // eslint-disable-next-line no-param-reassign
  rootNode._clickListener = _clickListener;

  // Use capture phase to make sure we cancel it before any regular bubble listeners hear it.
  rootNode.addEventListener('click', _clickListener, true);

  $(rootNode)
    .on('touchstart' + namespace + ' ' +
        'mousedown' + namespace, function (event) {
          // for mousedown-after-touchend Mobile Safari issue explained above where __contextMenuPressHoldJustEnded is set.
          if (event.type === 'mousedown' && contextMenuPressHoldJustEnded) {
            return undefined;
          }

          // reset isPressHold flag for all events that can start a click.
          isPressHold = false;

          // start a pressHold timer on touchstart.  If not cancelled before 750ms by touchend/etc., will launch the CM.
          // isolate the context menu long tap to a single touch point.
          if (event.type === 'touchstart' && event.originalEvent.touches.length === 1) {
            // note starting position so touchmove handler can tell if touch moved too much
            var firstTouch = event.originalEvent.touches[0];
            touchPageX = firstTouch.pageX;
            touchPageY = firstTouch.pageY;

            touchInProgress = true;
            contextMenuPressHoldTimer =
              setTimeout(launch.bind(undefined, event, 'touch', true), pressHoldThreshold);
            // eslint-disable-next-line no-param-reassign
            rootNode._contextMenuPressHoldTimer = contextMenuPressHoldTimer;
          }

          return true;
        })

  // if the touch moves too much, it's not a pressHold
    .on('touchmove' + namespace, function (event) {
      var firstTouch = event.originalEvent.touches[0];
      if (Math.abs(touchPageX - firstTouch.pageX) > maxAllowedMovement
          || Math.abs(touchPageY - firstTouch.pageY) > maxAllowedMovement) {
        touchInProgress = false;
        clearTimeout(contextMenuPressHoldTimer);
      }
      return true;
    })

  // if the touch ends before the 750ms is up, it's not a long enough pressHold to show the CM
    .on('touchend' + namespace + ' ' +
        'touchcancel' + namespace, function () {
          touchInProgress = false;
          clearTimeout(contextMenuPressHoldTimer);
          return true;
        })
    .on('keydown' + namespace + ' ' +
        'contextmenu' + namespace, function (event) {
          if (event.type === 'contextmenu' // right-click.  pressHold for Android but not iOS
              || (event.which === 121 && event.shiftKey)) { // Shift-F10
            var eventType;
            if (touchInProgress) {
              eventType = 'touch';
            } else if (event.type === 'keydown') {
              eventType = 'keyboard';
            } else {
              eventType = 'mouse';
            }

            launch(event, eventType, false);
          }

          return true;
        });

  // Does 2 things:
  // 1) Prevents native context menu / callout from appearing in Mobile Safari.  E.g. for links, native CM has "Open in New Tab".
  // 2) In Mobile Safari and Android Chrome, prevents pressHold from selecting the text and showing the selection handles and (in Safari) the Copy/Define callout.
  // In UX discussion, we decided to prevent both of these things for all JET components for now.  If problems, can always, say, add protected method allowing
  // subclass to opt out (e.g. if they need 1 and/or 2 to work).
  // Per discussion with architects, do #2 only for touch devices, so that text selection isn't prevented on desktop.  Since #1
  // is a no-op for non-touch, we can accomplish this by omitting the entire style class, which does 1 and 2, for non-touch.
  // Per comments in scss file, the suppression of 1 and 2 has issues in old versions of Mobile Safari.
  if (oj.DomUtils.isTouchSupported()) {
    $(rootNode).addClass('oj-menu-context-menu-launcher');
  }
};

/* jslint browser: true*/
/**
 * in some OS/browser combinations you can attempt to detect high contrast mode
 * in javascript, go to the url below and look for "High Contrast"
 * https://www.w3.org/TR/2016/WD-wai-aria-practices-1.1-20160317/
 *
 * This function uses a variation of the code in the "High Contrast" section of
 * the site above to try and detect high contrast mode
 * by script, but it by no means definitively tells you whether or not you
 * are actually in high contrast mode. As discussed at the url above you
 * may need to have a user preference setting for high contrast.
 *
 * If the script is able to detect high contrast mode it sets the class
 * "oj-hicontrast" on the body tag. When "oj-high-contrast" is present
 * JET provides alternate informational images that are specially designed
 * for high contrast users.
 * @private
 */
function _ojHighContrast() {
  // using a data uri, I googled for shortest uri to get this one since
  // I don't care about the actual image, but I do want a legit image
  // otherwise I see an error in chrome and I don't want users to be
  // confused by seeing any error.
  var div = document.createElement('div');
  div.style.border = '1px solid';
  div.style.borderColor = 'red green';
  div.style.position = 'absolute';
  div.style.top = '-999px';
  div.style.backgroundImage = 'url(data:image/gif;base64,R0lGODlhAQABAIAAAP///wAAACwAAAAAAQABAAACAkQBADs=)';
  var body = document.body;
  body.appendChild(div);     // @HTMLUpdateOK safe manipulation
  var computedStyles = window.getComputedStyle(div);

  var bki = computedStyles.backgroundImage;

  if (computedStyles.borderTopColor === computedStyles.borderRightColor ||
      (bki != null && (bki === 'none' || bki === 'url (invalid-url:)'))) {
    body.classList.add('oj-hicontrast');
  }

  body.removeChild(div);
}

$(document).ready(function () {
  _ojHighContrast();
});

// Copyright (c) 2013, Oracle and/or its affiliates.
// All rights reserved.

/* jslint browser: true*/

/**
 * @export
 * @class
 * @since 1.0
 * @classdesc Common test support in JavaScript
 * @ojtsignore
 */
oj.Test = {};


/**
 * A global application flag that can be set by a test to indicate that all page startup processing is done
 * and an external automated test can begin
 * @export
 * @type {boolean}
 */
oj.Test.ready = false;

/**
 * @export
 * Return the node found given the locator
 * @param {Object|string} locator A locator which is either a JSON string (to be parsed using $.parseJSON), or an Object with the following properties:
 *                                             element: the component's selector, determined by the test author when laying out the page
 *                                             subId: the string, documented by the component, that the component expects in getNodeBySubId to locate a particular subcomponent
 *  @returns {any} the subcomponent located by the subId string passed in locator, if found.
 */
oj.Test.domNodeForLocator = function (locator) {
  var locObj = locator;
  if (oj.StringUtils.isString(locator)) {
    var locStr = /** @type {string} */ (locator);
    try {
      locObj = JSON.parse(locStr);
    } catch (e) {
      return null;
    }
  }
  if (locObj && locObj.element) {
    var element = $(locObj.element);
    if (element && element.length > 0) {
      delete locObj.element;
      var id = /** @type {Object} */ (locObj);
      return oj.Components.getNodeBySubId(element[0], id);
    }
  }
  return null;
};

/**
 * @return {number} total number of open popups
 * @export
 * @since 1.1.0
 */
oj.Test.getOpenPopupCount = function () {
  return oj.ZOrderUtils.getOpenPopupCount();
};

/**
 * Returns a jQuery set of popup root elements that are open and actively
 * managed by the popup framework.
 *
 * @return {!jQuery}
 * @export
 * @since 1.1.0
 */
oj.Test.findOpenPopups = function () {
  return oj.ZOrderUtils.findOpenPopups();
};

/**
 * Utility used for testing. Compares two jQuery singleton wappered elements
 * determining which element has the greatest stacking context.
 *
 * @export
 * @param {jQuery} el1 first element to compare
 * @param {jQuery} el2 second element to compare
 * @return {number} 0 if elements have the same stacking context;
 *                  1 if the first element has a greater stacking context;
 *                 -1 when the second element has a greater stacking context;
 * @since 1.1.0
 */
oj.Test.compareStackingContexts = function (el1, el2) {
  return oj.ZOrderUtils.compareStackingContexts(el1, el2);
};

// override jQuery's cleanData method to bypass cleanup of custom elements and composites
$.cleanData = (function (orig) {
  return function (elems) {
    var nonCustomElements = [];
    for (var i = 0; i < elems.length; i++) {
      var elem = elems[i];
      if (elem == null) {
        break;
      }

      // Skip cleaning any elements that are custom elements or are created by a custom element
      var bSkip = false;
      var constr = oj.Components.__GetWidgetConstructor(elem);
      if (constr) {
        bSkip = constr('instance')._IsCustomElement();
        if (!bSkip) {
          var parent = oj.Components.getComponentElementByNode(elem);
          bSkip = parent && oj.BaseCustomElementBridge.getRegistered(parent.tagName);
        }
      }
      if (!bSkip) {
        nonCustomElements.push(elem);
      }
    }
    if (nonCustomElements.length > 0) {
      orig(nonCustomElements);
    }
  };
}($.cleanData));

/**
 * Copyright (c) 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */

/**
 * @class oj.LabelledByUtils
 * @classdesc JET Labelled Component Utils
 * @export
 * @since 7.1.0
 * @hideconstructor
 * @ignore
 *
 */
var LabelledByUtils = {};

// S T A T I C    V A R S


/**
* String used in the label element's id for custom &lt;oj-label>
* @const
* @ignore
* @type {string}
*/
LabelledByUtils.CUSTOM_LABEL_ELEMENT_ID = '|label';

/**
 * For custom element only. Only called from the 'set' components, like oj-radioset.
 * When labelledBy changes, we need to update the aria-labelledby attribute.
 * Note: If labelledBy changes from a value to null, we should still remove the oldValue from
 * aria-labelledby.
 * Note: This way isn't perfect since it assumes the internal label's id is oj-label id
 * + '|label'. Used by oj-radioset, oj-checkboxset, oj-color*, oj-buttonset
 * @param {Element} rootElement the root element, like oj-radioset. It must have an id.
 * @param {string|null} originalValue the old value of the labelledBy option
 * @param {string|null} value the new value of the labelledBy option.
 * @param {jQuery} $elems jquery Object containing the node(s) to add/remove aria-labelledby to.
 * @private
 * @ignore
 */
LabelledByUtils._updateLabelledBy = function (rootElement, originalValue, value, $elems) {
  var suffix = LabelledByUtils.CUSTOM_LABEL_ELEMENT_ID;
  var byId;
  var tokens;
  var originalTokens;
  var i;

  if (!this._IsCustomElement()) {
    return;
  }

  // originalValue is the 'old' value of labelledBy and value is the 'new' value of
  // labelledBy. The most likely use case if originalValue is null. Check for that first.
  if (!originalValue && value) {
    // value can be a space-separated list of ids, so we need to split it and add the suffix
    // to each one and put it back into a space-separated list.
    // same thing as above, but for 'value'.
    tokens = value.split(/\s+/);
    for (i = 0; i < tokens.length; i++) {
      byId = tokens[i];
      // this adds one id at a time to aria-labelledBy attribute. It
      // makes sure there are not duplicates.
      LabelledByUtils._addAriaLabelledBy($elems, byId + suffix);
      // this sets data-oj-set-id on oj-label which in turn sets
      // described-by on oj-radioset or other 'set' form component.
      LabelledByUtils._addSetIdOnLabel(byId, rootElement.id);
    }
  } else if (originalValue && !value) {
    // if  original value has a value and value doesn't, remove all
    // value can be a space-separated list of ids, so we need to split it and add the suffix
    // to each one and put it back into a space-separated list.
    // same thing as above, but for 'value'.
    tokens = originalValue.split(/\s+/);
    for (i = 0; i < tokens.length; i++) {
      byId = tokens[i];
      // this adds/removes one id at a time to aria-labelledBy attribute. It
      // makes sure there are not duplicates.
      LabelledByUtils._removeAriaLabelledBy($elems, byId + suffix);
      LabelledByUtils._removeDescribedByWithPrefix(rootElement, byId + '|');
    }
  } else if (originalValue && value) {
    // if both original value and value have values, then we should figure out which are the
    // same and ignore them, and remove the ones from original value that are unique and
    // add the ones for value that are unique.
    tokens = value.split(/\s+/);
    originalTokens = originalValue.split(/\s+/);
    for (i = 0; i < originalTokens.length; i++) {
      byId = originalTokens[i];
      if (value.indexOf(byId) === -1) {
        // not in both, so remove it (add the suffix)
        LabelledByUtils._removeAriaLabelledBy($elems, byId + suffix);
        LabelledByUtils._removeDescribedByWithPrefix(rootElement, byId + '|');
      }
    }
    for (i = 0; i < tokens.length; i++) {
      byId = tokens[i];
      if (originalValue.indexOf(byId) === -1) {
        // not in both, so add it (add the suffix)
        LabelledByUtils._addAriaLabelledBy($elems, byId + suffix);
        LabelledByUtils._addSetIdOnLabel(byId, rootElement.id);
      }
    }
  }
};

/** .
 * Add the id to the widget's aria-labelledby attribute.
 * @param {jQuery} $elems the jquery element(s) that represents the node on which aria-labelledby is
 * @param {string} id id to add to aria-labelledby
 * @private
 * @ignore
 */
LabelledByUtils._addAriaLabelledBy = function ($elems, id) {
  var index;

  $elems.each(function () {
    var labelledBy = this.getAttribute('aria-labelledby');
    var tokens;

    tokens = labelledBy ? labelledBy.split(/\s+/) : [];
    // Get index that id is in the tokens, if at all.
    index = tokens.indexOf(id);
    // add id if it isn't already there
    if (index === -1) {
      tokens.push(id);
    }
    labelledBy = tokens.join(' ').trim();
    if (labelledBy == null) {
      this.removeAttribute('aria-labelledBy');
    } else {
      this.setAttribute('aria-labelledBy', labelledBy);
    }
  });
};

/** .
 * Add 'data-oj-set-id' on oj-label, which in turn will
 * set described-by back on the Form component.
 * @param {string} ojLabelId the oj-label element's id.
 * @param {string} formComponentId the id of the form component
 * @private
 * @ignore
 */
LabelledByUtils._addSetIdOnLabel = function (ojLabelId, formComponentId) {
  var ojLabel = document.getElementById(ojLabelId);
  if (ojLabel && !ojLabel.getAttribute('data-oj-set-id')) {
    ojLabel.setAttribute('data-oj-set-id', formComponentId);
  }
};

/**
 * Remove the id from the widget's aria-labelledby attribute.
 * @param {jQuery} $elems the jquery element(s) that represents the node on which aria-labelledby is
 * @param {string} id id to remove from aria-labelledby
 * @private
 * @ignore
 */
LabelledByUtils._removeAriaLabelledBy = function ($elems, id) {
  var labelledBy;

  $elems.each(function () {
    var index;
    var tokens;

    // get aria-labelledby that is on the element(s)
    labelledBy = this.getAttribute('aria-labelledby');
    // split into tokens
    tokens = labelledBy ? labelledBy.split(/\s+/) : [];
    // Get index that id is in the tokens, if at all.
    index = tokens.indexOf(id);
    // remove that from the tokens array
    if (index !== -1) {
      tokens.splice(index, 1);
    }
    // join the tokens back together and trim whitespace
    labelledBy = tokens.join(' ').trim();
    if (labelledBy) {
      this.setAttribute('aria-labelledby', labelledBy);
    } else {
      this.removeAttribute('aria-labelledby');
    }
  });
};

/**
 * Remove the id that starts with the prefix from the element's described-by attribute.
 * @param {Element} element the element, like oj-radioset
 * @param {string} prefix prefix of the described-by value to remove.
 * @private
 * @ignore
 */
LabelledByUtils._removeDescribedByWithPrefix = function (element, prefix) {
  var describedBy;
  var tokens;

  describedBy = element.getAttribute('described-by');
  // split into tokens
  tokens = describedBy ? describedBy.split(/\s+/) : [];
  tokens = tokens.filter(function (item) {
    return item.indexOf(prefix) === -1;
  });
  // join the tokens back together and trim whitespace
  describedBy = tokens.join(' ').trim();
  if (describedBy) {
    element.setAttribute('described-by', describedBy);
  } else {
    element.removeAttribute('described-by');
  }
};

/** For custom element only.
 * When describedBy changes, we need to update the aria-described attribute.
 * @param {string|null} originalValue the old value of the labelledBy option
 * @param {string|null} value the new value of the labelledBy option.
 * @private
 * @ignore
 */
LabelledByUtils._updateDescribedBy = function (originalValue, value) {
  var byId;
  var tokens;
  var originalTokens;
  var i;

  if (!this._IsCustomElement()) {
    return;
  }

  // originalValue is the 'old' value of describedBy and value is the 'new' value of
  // describedBy. The most likely use case if originalValue is null. Check for that first.
  if (!originalValue && value) {
    // value can be a space-separated list of ids, so we need to split it
    // to each one and put it back into a space-separated list.
    // same thing as above, but for 'value'.
    tokens = value.split(/\s+/);
    for (i = 0; i < tokens.length; i++) {
      byId = tokens[i];
      // this adds one id at a time to aria-labelledBy attribute. It
      // makes sure there are not duplicates.
      this._addAriaDescribedBy(byId);
    }
  } else if (originalValue && !value) {
    // if  original value has a value and value doesn't, remove all
    // value can be a space-separated list of ids, so we need to split it
    // to each one and put it back into a space-separated list.
    // same thing as above, but for 'value'.
    tokens = originalValue.split(/\s+/);
    for (i = 0; i < tokens.length; i++) {
      byId = tokens[i];
      // this adds/removes one id at a time to aria-labelledBy attribute. It
      // makes sure there are not duplicates.
      this._removeAriaDescribedBy(byId);
    }
  } else if (originalValue && value) {
    // if both original value and value have values, then we should figure out which are the
    // same and ignore them, and remove the ones from original value that are unique and
    // add the ones for value that are unique.
    tokens = value.split(/\s+/);
    originalTokens = originalValue.split(/\s+/);
    for (i = 0; i < originalTokens.length; i++) {
      byId = originalTokens[i];
      if (value.indexOf(byId) === -1) {
        // not in both, so remove it
        this._removeAriaDescribedBy(byId);
      }
    }
    for (i = 0; i < tokens.length; i++) {
      byId = tokens[i];
      if (originalValue.indexOf(byId) === -1) {
        // not in both, so add it
        this._addAriaDescribedBy(byId);
      }
    }
  }
};

/**
 * Add the aria-describedby on the content element(s) if it isn't already there.
 *
 * @param {string} id the id for aria-describedby
 * @private
 * @ignore
 *
 */
LabelledByUtils._addAriaDescribedBy = function (id) {
  var contentElements = this._GetContentElement();
  var index;

  contentElements.each(function () {
    var describedby = this.getAttribute('aria-describedby');
    var tokens;

    tokens = describedby ? describedby.split(/\s+/) : [];
        // Get index that id is in the tokens, if at all.
    index = tokens.indexOf(id);
    // add id if it isn't already there
    if (index === -1) {
      tokens.push(id);
    }
    describedby = tokens.join(' ').trim();

    if (describedby == null) {
      this.removeAttribute('aria-describedby');
    } else {
      this.setAttribute('aria-describedby', describedby);
    }
  });
};

/**
 * Remove the aria-describedby from the content element(s).
 *
 * @param {string} id the id for aria-describedby
 * @private
 * @ignore
 *
 */
LabelledByUtils._removeAriaDescribedBy = function (id) {
  var contentElements = this._GetContentElement();

  contentElements.each(function () {
    var describedby;
    var index;
    var tokens;

    // get aria-describedby that is on the content element(s)
    describedby = this.getAttribute('aria-describedby');
    // split into tokens
    tokens = describedby ? describedby.split(/\s+/) : [];
    // Get index that id is in the tokens, if at all.
    index = tokens.indexOf(id);
    // remove that from the tokens array
    if (index !== -1) {
      tokens.splice(index, 1);
    }
    // join the tokens back together and trim whitespace
    describedby = tokens.join(' ').trim();
    if (describedby) {
      this.setAttribute('aria-describedby', describedby);
    } else {
      this.removeAttribute('aria-describedby');
    }
  });
};


/**
 * Copyright (c) 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */

/* global Promise:false, __ValidationBase:false, Logger:false, Context:false, ThemeUtils:false, */

/**
 * @class oj.EditableValueUtils
 * @classdesc JET Editable Component Utils
 * @export
 * @since 0.6.0
 * @hideconstructor
 * @ignore
 *
 */
oj.EditableValueUtils = {};

// S T A T I C    V A R S
/**
 * The various contexts under which validation can be run by component.
 * @ignore
 */
oj.EditableValueUtils.validationContext = {
  COMPONENT_CREATE: 1,
  CONVERTER_OPTION_CHANGE: 2,
  DISABLED_OPTION_CHANGE: 3,
  READONLY_OPTION_CHANGE: 4,
  REFRESH_METHOD: 5,
  REQUIRED_OPTION_CHANGE: 6,
  RESET_METHOD: 7,
  USER_ACTION: 8,
  VALIDATE_METHOD: 9,
  VALIDATORS_OPTION_CHANGE: 10,
  VALUE_OPTION_CHANGE: 11
};

/**
 * Default validation options used by validate method.
 * @ignore
 */
oj.EditableValueUtils.validateMethodOptions = { doValueChangeCheck: false,
  validationContext: oj.EditableValueUtils.validationContext.VALIDATE_METHOD };

/**
 * Default validation options used when converter option changes
 * @ignore
 */
oj.EditableValueUtils.converterOptionOptions = { doValueChangeCheck: false,
  doNotClearMessages: true,
  validationContext: oj.EditableValueUtils.validationContext.CONVERTER_OPTION_CHANGE };

/**
 * Default validation options used when disabled option changes
 * @ignore
 */
oj.EditableValueUtils.disabledOptionOptions = { doValueChangeCheck: false,
  doNotClearMessages: true,
  validationContext: oj.EditableValueUtils.validationContext.DISABLED_OPTION_CHANGE };

/**
 * Default validation options used when required option changes
 * @ignore
 */
oj.EditableValueUtils.requiredOptionOptions = { doValueChangeCheck: false,
  doNotClearMessages: true,
  validationContext: oj.EditableValueUtils.validationContext.REQUIRED_OPTION_CHANGE };

/**
 * Default validation options used when readOnly option changes
 * @ignore
 */
oj.EditableValueUtils.readOnlyOptionOptions = { doValueChangeCheck: false,
  doNotClearMessages: true,
  validationContext: oj.EditableValueUtils.validationContext.READONLY_OPTION_CHANGE };

/**
 * Default validation options used when refresh method is called.
 * @ignore
 */
oj.EditableValueUtils.refreshMethodOptions = { doValueChangeCheck: false,
  doNotClearMessages: true,
  validationContext: oj.EditableValueUtils.validationContext.REFRESH_METHOD };
/**
 * Default validation options used when validators option changes
 * @ignore
 *  */
oj.EditableValueUtils.validatorsOptionOptions = { doValueChangeCheck: false,
  doNotClearMessages: true,
  validationContext: oj.EditableValueUtils.validationContext.VALIDATORS_OPTION_CHANGE };

/**
* String used in the id on the span that surrounds the help icon.
* @const
* @private
* @ignore
* @type {string}
*/
var _REQUIRED_ICON_ID = '_requiredIcon';

/**
* Enum for validate() return values
* @const
* @ignore
*/
oj.EditableValueUtils.VALIDATE_VALUES = {
  VALID: 'valid',
  INVALID: 'invalid'
};

/**
 * This method is called during _InitOptions() to initialize a component option value from DOM. This
 * uusally is the case when the option value is undefined,
 * i.e., this.options.optionName === undefined.
 * <br/>
 * Returns the attribute value for the given attribute on the element appropriately converted, or
 * the default, if the attribute isn't set on the element.<br/>
 *
 * @param {Object} element the element the component is initialized with.
 * @param {string} attribute the name of the element's attribtue. Example, value, disabled etc.
 *
 * @returns {Object} a JSON object containing the following properties - <br/>
 * <ul>
 * <li><b>fromDom</b> - whether the option value was initialized from DOM. When true the option's
 * value is written back (to observable).</li>
 * <li><b>value</b> - the option value. the attribute value or the default if the attribute isn't
 * set on the element.</li>
 * </ul>
 *
 * @private
 */
oj.EditableValueUtils.getAttributeValue = function (element, attribute) {
  var result;
  var returnVal = {};

  if (element && attribute) {
    var elem = element[0];
    switch (attribute) {
      case 'disabled' :
        result = elem.hasAttribute('disabled') ?
          !!elem.disabled : undefined;
        break;

      case 'pattern':
        result = elem.pattern || undefined;
        break;

      case 'placeholder':
        result = elem.placeholder || undefined;
        break;

      case 'readonly':
        result = elem.hasAttribute('readonly') ?
          !!elem.readOnly : undefined;
        break;

      case 'required':
        // If attribute is present
        //   - if the required property is undefined then return true since the attribute is set.
        //   - Otherwise set to !!propVal
        if (elem.hasAttribute('required')) {
          var propVal = elem.required;
          if (propVal !== undefined) {
            result = !!propVal;
          } else {
            result = true; // any attribute value indicates true, even required='false'
          }
        } else {
          result = undefined;
        }
        break;

      case 'title':
        result = elem.hasAttribute('title') ?
          elem.title : undefined;
        break;

      case 'value':
        // element attribute may not be set, in which case default to null
        result = element.val() || undefined;
        break;

      case 'min':
      case 'max':
      default:
        // same logic for min + max as in default
        result = elem.getAttribute(attribute) || undefined;
        break;
    }
  }

  if (result !== undefined) {
    returnVal.fromDom = true;
    returnVal.value = result;
  } else {
    returnVal.fromDom = false;
    // returnVal.value = defaultValue;
  }
  return returnVal;
};

/**
 * NOTE: This is unnecessary to call for custom elements.
 * Called from component._InitOptions() with an array of options
 * that the component might need to initialize from DOM (e.g.,
 * disabled, required, title, etc). This function loops through each of these and if the
 * constructorOptions[option] is undefined, it tries to get the option from DOM.
 * The constructorOptions hold the options that the page author sets on the component, usually via
 * knockout bindings, and that takes precedence over DOM.
 * e.g., <input id="id" type="text" required data-bind="ojComponent: {component: 'ojInputText',
                                  value: value}"/>
 * value is a constructorOption and required is a DOM option in this example. If you have:
 * e.g., <input id="id" type="text" required data-bind="ojComponent: {component: 'ojInputText',
                                  value: value, required: false}"/>
 * required is both a constructorOption of false and a DOM of true. The constructorOption takes
 * precedence.
 * <p>
 * IMPORTANT: Do not call this method after component has been created, since option values are
 * mutated directly after that point.</p>
 *
 * The 'final' value an option uses/initializes from, can come from these places (in order of least
 * to most likely) - <br/>
 * <ol>
 * <li>component default - this is the widget default </li><br/>
 * <li>app default - this is what a page author defines for the value in the page/app</li> <br/>
 * <li>dom value - if your option also has a dom attribute, this is the value set on element for
 * component. </li> <br/>
 * <li>constructor value - this is the value page author sets on the component binding </li><br/>
 * </ol>
 *
 * At the time _InitOptions is called, (1), (2) and (4) are merged, but this may not be the value a
 * component wants for the option, especially when (4) is undefined. For example, if these values
 * were set for a component - <br/>
 * (1) - 'foo'<br/>
 * (2) - 'bar'<br/>
 * (3) - 'lucy'<br/>
 * (4) - undefined<br/>
 * <p>
 * at the time _InitOptions is called, this.options.option is set to 'bar'. But because DOM value
 * wins over app default or component default, the component needs to check if the constructor value was
 * undefined and if so, set option to the dom value which is 'lucy' in this example. This is what
 * this function does.<br/>
 * This method always defaults the value to be - this.options.option -
 * because we think if neither (3) nor (4) is set, then the value from (2) should win. <br/>
 * </p>
 *
 * @param {Object} props Array holding Object-literal that a component provides
 * with the following properties that helps determine the final value for one or more options.
 *
 * @property {string} props.attribute - name of DOM attribute
 * @property {string|undefined} props.option - name of the option if different from attribute name.
 *
 * @property {Function|boolean|undefined} props.coerceDomValue - if the DOM value is set and
 * coercing the dom value is needed, then either set to boolean true, which uses the default
 * coercion rules for common attributes (a), or provide a custom callback (b). <p>
 * E.g., 'value' option for input number, input date etc. have special rules for coercing the value,
 *  so thse provide a custom callback. For common attributes like required and disabled, set the
 *  value to true so the default oj.EditableValueUtils#coerceDomValueForOption method gets used.
 *
 * @property {boolean|undefined} props.validateOption - if set to true, then it calls
 * oj.EditableValueUtils.validateValueForOption method to validate the option.
 *
 * @param {Object} constructorOptions the options set on the component instance, often using
 * component binding. (this is the value page author sets on the component binding)
 * @param {Object} comp component instance.
 * @param {Function=} postprocessCallback - optional callback that will receive a map of initialized
 * options for post-processing
 * @ignore
 * @public
 */
oj.EditableValueUtils.initializeOptionsFromDom = function (
  props, constructorOptions, comp, postprocessCallback
) {
  var initializedOptions = {};

  // Loop through props to initialize option
  for (var i = 0; i < props.length; i++) {
    var finalValue;
    var result;
    var prop = props[i];
    var attribute = prop.attribute;
    var option = prop.option || attribute;
    var coerceDomValue = prop.coerceDomValue;
    var validateOption = prop.validateOption;
    var element = comp.element;
    var previousValue = comp.options[option];

    /* The precedence for the value that an option uses is as follows from lowest to highest -
     *
     * (1) component default - this is the widget default, already merged in to comp.options
     * (2) app default - this is what a page author defines for the value in the page / app,
     * already merged in to comp.options
     * (3) dom value - if your option also has a dom attribute, this is the value set on element.
     * (4) constructor value - this is the value page author sets on the component binding, already
     * merged in to comp.options.
     *
     * When (4) is undefined then attempt to default from (3).
     */

    // Step 1: use DOM value
    if (constructorOptions[option] === undefined) {
      previousValue = comp.options[option];
      result = oj.EditableValueUtils.getAttributeValue(element, attribute);

      // if we are using domValue then coerce the dom value before writing to options and trigegr
      // option change so the value is written back (to ko)
      if (result.fromDom) {
        finalValue = result.value;

        // only required needs coercing so not bad
        if (coerceDomValue) {
          if (typeof coerceDomValue === 'boolean') {
            finalValue = oj.EditableValueUtils.coerceDomValueForOption(option, finalValue);
          } else if (typeof coerceDomValue === 'function') {
            finalValue = coerceDomValue.call(comp, finalValue);
          }
        }
        initializedOptions[option] = finalValue;
      }
    }

    var valueToValidate =
        (option in initializedOptions) ? initializedOptions[option] : previousValue;

    // Step 2: validate the option value if needed
    if (validateOption) {
      if (typeof validateOption === 'boolean') {
        oj.EditableValueUtils.validateValueForOption(option, valueToValidate);
      }
    }
  }

  if (postprocessCallback != null) {
    postprocessCallback(initializedOptions);
  }

  comp.option(initializedOptions, { _context: { writeback: true, internalSet: true } });
};

/**
 * Validates value set for the option and throws error if invalid.
 *
 * @param {string} option name of the option. Validates options common to all edtiableValue
 * holders.
 * @param {string|Object|boolean|number|undefined} value of the option that is validated
 *
 * @throws {Error} if option value is invalid
 * @public
 * @ignore
 */
oj.EditableValueUtils.validateValueForOption = function (option, value) {
  var error = false;

  switch (option) {
    case 'required' :
      if (value !== null && typeof value !== 'boolean') {
        error = true;
      }
      break;

    case 'readOnly':
    case 'disabled' :
      if (value !== null && typeof value !== 'boolean') {
        error = true;
      }
      break;
    default:
      break;
  }

  if (error) {
    throw new Error("Option '" + option + "' has invalid value set: " + value);
  }
};


/**
 * Coerces the dom value being used for the option, and throws error if invalid.
 *
 * @param {string} option name of the option.
 * @param {string|Object|boolean|number|null} domValue dom value that is being coerced to the
 * option value
 * @throws {Error} if domValue cannot be coerced appropriately
 * @public
 * @ignore
 */
oj.EditableValueUtils.coerceDomValueForOption = function (option, domValue) {
  var coerced = domValue;
  switch (option) {
    case 'required' :
      coerced = !!domValue;
      break;
    default:
      break;
  }

  return coerced;
};


/**
 * set pickerAttributes on a popup picker
 *
 * @param {jQuery} picker popup picker
 * @param {Object} pickerAttributes supported attributes are class and style, which are appended to the picker class and style, if any.
 *
 * @ignore
 */
oj.EditableValueUtils.setPickerAttributes = function (picker, pickerAttributes) {
  //  - let the popup picker accept the custom css class name from the component
  if (picker && pickerAttributes) {
    var classValue = pickerAttributes.class;
    if (classValue) {
      var classes = classValue.split(' ');
      // IE11 doesn't support destructured parameters so we need to iterate across the list
      // of classes
      for (var i = 0, len = classes.length; i < len; ++i) {
        picker[0].classList.add(classes[i]);
      }
    }

    var styleValue = pickerAttributes.style;
    if (styleValue) {
      var pickerElem = picker[0];
      var currStyle = pickerElem.getAttribute('style');
      if (currStyle) {
        pickerElem.setAttribute('style', currStyle + ';' + styleValue);
      } else {
        pickerElem.setAttribute('style', styleValue);
      }
    }
  }
};

/**
 * Helper to see if a special property was set to indicate we definitely have no label.
 * This is a performance enhancement for the corner case where input components are rendered in a
 * ojTable. Input components rendered in an ojTable have no label so we don't need to waste time
 * looking for labels.
 * @param {jQuery} widget The component widget.
 * @return {boolean}
 * @ignore
 * @private
 */
oj.EditableValueUtils.hasNoLabelFlag = function (widget) {
  return widget[0].hasAttribute('data-oj-no-labelledby');
};

/**
 * Given the labelledBy (e.g., this.options.labelledBy), use this to
 * get the oj-label's label element's id when there is a for/id relationship
 * between the oj-label and the form component, but the form component wants to
 * write aria-labelledby on a div instead of using the for/id relationship in dom.
 * Some components need this information to
 * use as their aria-labelledby on their dom element(s) that takes focus. An example
 * is oj-switch and oj-slider which put display:none on its input and uses aria-labelledby
 * on its thumb.
 * This is the preferred way rather than using a 'for' attribute search to find the oj-label.
 * @param labelledBy
 * @param defaultLabelId
 * @return {string|null} return the string to use as the aria-labelledby on the form component's
 * focusable element. If oj-label doesn't exist, this will return null.
 * @ignore
 * @private
 */
oj.EditableValueUtils._getOjLabelAriaLabelledBy = function (labelledBy, defaultLabelId) {
  var ariaLabelledBy;
  var ojlabels = oj.EditableValueUtils._getCustomOjLabelElements(labelledBy);
  if (ojlabels) {
    ariaLabelledBy = '';
    for (var j = 0; j < ojlabels.length; j++) {
      var ojlabel = ojlabels[j];
      var oneLabelElementId = ojlabel.getAttribute('label-id');
      if (!oneLabelElementId) {
        var labelElement = ojlabel.querySelector('label');
        if (labelElement) {
          oneLabelElementId = labelElement.getAttribute('id');
        } else {
          // this is the case where the form component has
          // labelled-by pointing to oj-label that hasn't been
          // upgraded yet and doesn't have label-id on it.
          // this isn't the way the form component and its label
          // should be linked, but it is possible.
          ojlabel.setAttribute('label-id', defaultLabelId);
          oneLabelElementId = defaultLabelId;
        }
      }
      ariaLabelledBy += oneLabelElementId;
      if (j + 1 < ojlabels.length) {
        ariaLabelledBy += ' ';
      }
    }
  }
  return ariaLabelledBy;
};

/**
 * @ignore
 * @private
 */
oj.EditableValueUtils._getCustomOjLabelElements = function (labelledBy) {
  var labelElements = [];

  if (labelledBy) {
    // split into individual ids
    var split = labelledBy.split(/\s+/);
    for (var i = 0; i < split.length; i++) {
      var labelId = split[i];
      var labelElement = document.getElementById(labelId);
      // don't push any null elements. it's possible labelled-by element can't be found.
      if (labelElement) {
        labelElements.push(labelElement);
      } else {
        Logger.info('Cannot find oj-label with id ' + labelElement);
      }
    }
  }
  return labelElements;
};
/**
 * Called during component initialization. Sets the sub-id on the input and
 * set data-oj-input-id attribute on each oj-label
 * that is pointing to the form component with the labelled-by attribute.
 * @ignore
 * @private
 */
oj.EditableValueUtils._setInputId = function (contentElement, inputId, labelledBy) {
  if (inputId) {
    oj.EditableValueUtils.setSubIdForCustomLabelFor(contentElement, inputId);
    // most likely it is one label per component, but it is possible to have more than one
    // label per component.
    var ojlabels = oj.EditableValueUtils._getCustomOjLabelElements(labelledBy);
    if (ojlabels) {
      var id = contentElement.id;
      for (var i = 0; i < ojlabels.length; i++) {
        var ojlabel = ojlabels[i];
        ojlabel.setAttribute('data-oj-input-id', id);
      }
    }
  }
};
/**
 * Called when labelledBy option is changed on the form components with inputs, like
 * oj-input-text. Sets the data-oj-input-id attribute on each oj-label
 * that is pointing to the form component with the labelled-by attribute and also
 * updates the required validator's translation label text, if any.
 * @ignore
 * @private
 */
oj.EditableValueUtils._labelledByChangedForInputComp = function (labelledBy, id) {
  var ojlabels = oj.EditableValueUtils._getCustomOjLabelElements(labelledBy);
  if (ojlabels) {
    for (var i = 0; i < ojlabels.length; i++) {
      var ojlabel = ojlabels[i];
      ojlabel.setAttribute('data-oj-input-id', id);
      // update the required translation text
      if (this._IsRequired() && this.options.translations.required) {
        this._implicitReqValidator = null;
        this._getImplicitRequiredValidator();
      }
    }
  }
};

/**
 * Validates the component's display value using the converter and all validators registered on
 * the component and updates the <code class="prettyprint">value</code> option by performing the
 * following steps.
 *
 * <p>
 * <ol>
 * <li>All messages are cleared, including custom messages added by the app. </li>
 * <li>If no converter is present then processing continues to next step. If a converter is
 * present, the UI value is first converted (i.e., parsed). If there is a parse error then
 * messages are shown.</li>
 * <li>If there are no validators setup for the component the <code class="prettyprint">value</code>
 * option is updated using the display value. Otherwise all
 * validators are run in sequence using the parsed value from the previous step. The implicit
 * required validator is run first if the component is marked required. When a validation error is
 * encountered it is remembered and the next validator in the sequence is run. </li>
 * <li>At the end of validation if there are errors, the messages are shown.
 * If there were no errors, then the
 * <code class="prettyprint">value</code> option is updated.</li>
 * </ol>
 *
 * @example <caption>Validate component using its current value.</caption>
 * myComp.validate();
 *
 * @example <caption>Validate component and use the Promise's resolved state.</caption>
 * myComp.validate().then(
 *  function(result) {
 *    if(result === "valid")
 *    {
 *      submitForm();
 *    }
 *  });
 *
 *
 * @return {Promise} Promise resolves to "valid" if there were no converter parse errors and
 * the component passed all validations.
 * The Promise resolves to "invalid" if there were converter parse errors or
 * if there were validation errors.
 *
 * @ignore
 * @private
 */
oj.EditableValueUtils.validate = function () {
  var returnValue;

  // clear all messages; run full validation on display value
  // _SetValue returns boolean or Promise that resolves to a Boolean.
  returnValue = this._SetValue(
    this._GetDisplayValue(), null, this._VALIDATE_METHOD_OPTIONS);

  if (this._IsCustomElement()) {
    if (!(returnValue instanceof Promise)) {
      returnValue = Promise.resolve(returnValue ? 'valid' : 'invalid');
    } else {
      // convert true to 'valid' and false to 'invalid'
      return returnValue.then(function (booleanSetValueReturn) {
        return Promise.resolve(booleanSetValueReturn ? 'valid' : 'invalid');
      });
    }
  }

  return returnValue;
};

/**
 * Called from _AfterCreate for IsCustomElement form components so that they will
 * be assocatied with their oj-label element correctly.
 * Set the sub-id on the element so if they have a oj-label
 * pointing to it with the 'for' attrbiute, JAWS will read the label.
 * @param {Element} element
 * @param {string} widgetId
 * @ignore
 * @private
 */
oj.EditableValueUtils.setSubIdForCustomLabelFor = function (element, widgetId) {
  element.setAttribute('id', widgetId + '|input');
};

/**
 * Refresh everything that needs refreshing when the required option is toggled:
 * refreshes theming, aria-required, label.
 *
 * Used by EditableValue components that support the required option.
 * The component links to this function like this:
 *   _refreshRequired : oj.EditableValueUtils._refreshRequired,
 * @param {Object=} value the current value of the required option
 * @private
 * @ignore
 */
oj.EditableValueUtils._refreshRequired = function (value) {
  var id;
  var contentNode;
  var ariaValue;
  var ariaRequiredUnsupported = this._AriaRequiredUnsupported();

  this._refreshTheming('required', value);
  // refresh aria-required
  // Most inputs/selects need aria-required on the input element (aka 'content')
  // But it is not legal to have aria-required on radio/checkboxes.
  if (!ariaRequiredUnsupported) {
    contentNode = this._GetContentElement();

    ariaValue = value; // (value == "required") ? true : false;
    if (ariaValue && contentNode) {
      contentNode[0].setAttribute('aria-required', ariaValue);
    } else {
      contentNode[0].removeAttribute('aria-required');
    }
  }

  if (!this._IsCustomElement()) {
    if (!this.$label) {
      this._createOjLabel();
    }

    // need to keep the label's required in sync with the input's required
    if (this.$label) {
      this.$label.ojLabel('option', 'showRequired', value);
      // in most cases aria-required is supported and that is what we do to get JAWS to say
      // "Required" on focus of the input. But in the case of a 'set' of items where one is required,
      // say radioset/checkboxset, what do we do? aria-required doesn't make sense (nor is it valid
      // as it fails validation in some accessibility validators) on each input, when really it is
      // one in the set that is required, not each one. This is what we are doing from v1 on: we
      // put aria-describedby to point to the required icon text.
      if (ariaRequiredUnsupported) {
        // if aria-labelledby is set,
        // add/remove aria-describedby to the inputs pointing to
        // the label+"_requiredIcon".
        id = this._getAriaLabelledById(this.element);
        if (id) {
          if (value) {
            this._addAriaDescribedBy(id + _REQUIRED_ICON_ID);
          } else {
            this._removeAriaDescribedBy(id + _REQUIRED_ICON_ID);
          }
        }
      }
    }
  }
};

/**
 * Performs post processing after required option is set by taking the following steps.
 *
 * - if component is invalid and has messgesShown -> required: false/true -> clear component errors;
 * run full validation with UI value (we don't know if the UI error is from a required validator
 * or something else);<br/>
 * &nbsp;&nbsp;- if there are validation errors, then value not pushed to model; messagesShown is
 * updated<br/>
 * &nbsp;&nbsp;- if no errors result from the validation, push value to model; author needs to
 * listen to optionChange(value) to clear custom errors.<br/>
 *
 * - if component is invalid and has messagesHidden -> required: false -> clear component
 * errors; no deferred validation is run.<br/>
 * - if component has no error -> required: true -> run deferred validation (we don't want to flag
 * errors unnecessarily)<br/>
 * - messagesCustom is never cleared<br/>
 *
 * @param {string} option
 *
 * @private
 * @ignore
 */
// eslint-disable-next-line no-unused-vars
oj.EditableValueUtils._AfterSetOptionRequired = function (option) {
  // refresh hints, theming and aria to reflect new state
  this._refreshRequired(this._IsRequired());
  this._runMixedValidationAfterSetOption(oj.EditableValueUtils.requiredOptionOptions);
};

/**
 * When validators option changes, take the following steps.
 *
 * - Clear the cached normalized list of all validator instances. push new hints to messaging.<br/>
 * - if component is valid -> validators changes -> no change<br/>
 * - if component is invalid has messagesShown -> validators changes -> clear all component
 * messages and re-run full validation on displayValue. if there are no errors push value to
 * model;<br/>
 * - if component is invalid has messagesHidden -> validators changes -> do nothing; doesn't change
 * the required-ness of component <br/>
 * - messagesCustom is not cleared.<br/>
 *
 * NOTE: The behavior applies to any option that creates implicit validators - min, max, pattern,
 * etc. Components can call this method when these options change.
 *
 * @returns {undefined}
 * @private
 * @ignore
 */
oj.EditableValueUtils._AfterSetOptionValidators = function () {
  var displayValue;
  // resets all validators and pushes new hints to messaging
  this._ResetAllValidators();

  if (this._hasInvalidMessagesShowing()) {
    this._clearComponentMessages();
    displayValue = this._GetDisplayValue();
    // runs full validation on the display value. May be async
    this._SetValue(displayValue, null, oj.EditableValueUtils.validatorsOptionOptions);
  }
};

/**
 * When async-validators property changes, take the following steps.
 *
 * - Clear the cached normalized list of all async and sync validator instances.
 *  push new hints to messaging.<br/>
 * - if component is valid -> validators changes -> no change<br/>
 * - if component is invalid has messagesShown -> validators changes -> clear all component
 * messages and re-run full validation on displayValue. if there are no errors push value to
 * model;<br/>
 * - if component is invalid has messagesHidden -> validators changes -> do nothing; doesn't change
 * the required-ness of component <br/>
 * - messagesCustom is not cleared.<br/>
 *
 *
 * @returns {undefined}
 * @private
 * @ignore
 */
oj.EditableValueUtils._AfterSetOptionAsyncValidators = function () {
  // resets validators and pushes new hints to messaging
  this._AfterSetOptionValidators();
};

/**
 * Performs post processing after converter option changes by taking the following steps.
 *
 * - always push new converter hint to messaging <br/>
 * - if component has no errors -> refresh UI value<br/>
 * - if component is invalid has messagesShown -> clear all component errors and run full
 * validation using display value. <br/>
 * &nbsp;&nbsp;- if there are validation errors, value is not pushed to model; messagesShown is
 * updated.<br/>
 * &nbsp;&nbsp;- if no errors result from the validation, push value to model; author needs to
 * listen to optionChange(value) to clear custom errors.<br/>
 * - if component is invalid has messagesHidden -> refresh UI value. no need to run deferred
 * validations. <br/>
 * - messagesCustom is never cleared<br/>
 *
 * @param {String} option
 *
 * @returns {undefined}
 * @private
 * @ignore
 */
// called when 'converter' option changed, usually from option/setOption calls
oj.EditableValueUtils._AfterSetOptionConverter = function () {
  // clear the cached converter instance and push new hint to messaging
  this._converter = null;
  this._converterChangedCounter += 1;

  var converter = this._GetConverter();
  if (converter instanceof Promise) {
    var self = this;
    this._setBusyStateAsyncConverterLoading();
    var converterCounter = this._converterChangedCounter;
    this._loadingConverter(converter).then(function () {
      if (converterCounter === self._converterChangedCounter) {
        self._ResetConverter();
      }
      self._clearBusyStateAsyncConverterLoading();
    });
  } else {
    this._ResetConverter();
  }
};

/**
 * Performs post processing after we have the loaded converter
 * during component initialization.
 *
 * @returns {undefined}
 * @private
 * @ignore
 */
oj.EditableValueUtils._AfterCreateConverterCached = function () {
  // we do this here for a couple reasons
  // 1. because here we have the final value; an empty placeholder
  // shows up if data changed after first binding. 
  // 2. we do not want the placeholder displayed while the loading
  // indication is showing.
  if (this._HasPlaceholderSet()) {
    // update element placeholder
    this._SetPlaceholder(this.options.placeholder);
    this._customPlaceholderSet = true;
  }
  // can't show validator hints or converter hints until we have the converter
  // because some validators have the converter as an option.
  this._initComponentMessaging(this._MESSAGING_CONTENT_UPDATE_TYPE.ALL);
  // need a converter to format the value
  this._Refresh('value', this.options.value, false);
  // trigger messagesShownChanged for messagesShown if it's non-empty.
  // this.options['messagesShown'] would have been
  // updated in _ComponentCreate if messagesCustom was non-empty. Because we are setting
  // the 'changed' flag to true, the messagesShownChanged event will be fired, and that's what we want.
  if (this.options.messagesShown.length > 0) {
    this._setMessagesOption('messagesShown', this.options.messagesShown, null, true);
  }
};
/**
 * Called when converter option changes and we have the new converter.
 *
 * @private
 * @ignore
 */
oj.EditableValueUtils._ResetConverter = function () {
  var displayValue;

  this._getComponentMessaging().update(
    this._getMessagingContent(this._MESSAGING_CONTENT_UPDATE_TYPE.CONVERTER_HINT));

  if (this._hasInvalidMessagesShowing()) {
    this._clearComponentMessages();
    displayValue = this._GetDisplayValue();
    // runs full validation on the display value. May be async
    this._SetValue(displayValue, null, oj.EditableValueUtils.converterOptionOptions);
  } else {
    // refresh UI display value when there was no need to run full validation
    this._Refresh('converter', this.options.converter, true);
  }
};

/**
 * Returns an array of all validators built by merging the validators option set on the component
 * and the implicit validators setup by the component. <br/>
 * This does not include the implicit required validator. Components can override to add to this
 * array of validators.
 *
 * @return {Array} of validators
 *
 * @private
 * @ignore
 */
oj.EditableValueUtils._GetNormalizedValidatorsFromOption = function () {
  var i;
  var isValidatorInstance = true;
  var normalizedValidators = [];
  var validator;
  var validatorsOption;
  var vOptions;
  var vType;
  var vTypeStr;


  validatorsOption = this.options.validators;

  if (validatorsOption) {
    // Normalize validators
    for (i = 0; i < validatorsOption.length; i++) {
      validator = validatorsOption[i];
      if (typeof validator === 'object') {
        // check if we have an actual validator instance that implements the validate() method
        // oj.Validation.__doImplementsCheck(validator, oj.Validator);
        if (!(validator.validate && typeof validator.validate === 'function')) {
          isValidatorInstance = false;
        }

        if (!isValidatorInstance) {
          // we maybe dealing with an object literal
          // {'type': 'numberRange', 'options': { 'min': 100, 'max': 1000,
          //                                    'hint': {'min': 'some hint about min'}}}
          vTypeStr = validator.type;
          if (vTypeStr && typeof vTypeStr === 'string') {
            vType = __ValidationBase.Validation.validatorFactory(vTypeStr);
            if (vType) {
              vOptions = oj.CollectionUtils.copyInto({}, validator.options) || {};
              // we push converter into the options if not provided explicitly. This is to allow
              // validators to format values shown in the hint and messages
              vOptions.converter = vOptions.converter || this._GetConverter();
              vOptions.label = vOptions.label || this._getLabelText();
              validator = vType.createValidator(vOptions);
            } else {
              Logger.error('Unable to locate a validatorFactory for the requested type: ' + vTypeStr);
            }
          }
        }

        normalizedValidators.push(validator);
      } else {
        Logger.error('Unable to parse the validator provided:' + validator);
      }
    }
  }
  return normalizedValidators;
};
/**
 * Returns an array of all async validators built by the async-validators attribute
 * set on the component. In this release, Objects that have validate method (
 * and also they could have a hint property) are considered AsyncValidators and
 * AsyncValidator Objects. In future releases we will allow Objects with types, like
 * {'type': 'numberRange',
 * 'options': { 'min': 100, 'max': 1000, 'hint': {'min': 'some hint about min'}}}
 *
 * @return {Array} of async validators
 *
 * @private
 * @ignore
 */
oj.EditableValueUtils._GetNormalizedAsyncValidatorsFromOption = function () {
  var i;
  var normalizedValidators = [];
  var validator;
  var validatorsOption;

  validatorsOption = this.options.asyncValidators;

  // Normalize validators
  for (i = 0; i < validatorsOption.length; i++) {
    validator = validatorsOption[i];
    if (typeof validator === 'object') {
      // check if we have an actual asyncvalidator object that implements the validate() method
      if (validator.validate && typeof validator.validate === 'function') {
        normalizedValidators.push(validator);
      }
    } else {
      Logger.error('Unable to parse the validator provided:' + validator);
    }
  }

  return normalizedValidators;
};
/**
 * Returns the normalized converter instance.
 * This could return a Promise during component initialization or when changing the
 * component's converter property.
 *
 * @return {Object|null|Promise<Object>|Promise<null>} a converter instance or null
 * or a Promise to a converter instance or null.
 *
 * @private
 * @ignore
 */
oj.EditableValueUtils._GetConverter = function () {
  var converterOption;
  var converterInstanceReturn;
  var self = this;
  var converterPromise;

  // this._converter holds the instance
  if (!this._converter) {
    converterOption = this.options.converter;
    if (converterOption instanceof Promise) {
      converterPromise = converterOption;
    } else {
      converterInstanceReturn =
      __ValidationBase.IntlConverterUtils.getConverterInstance(converterOption);
      if (converterInstanceReturn instanceof Promise) {
        converterPromise = converterInstanceReturn;
      }
    }

    if (converterPromise) {
      return converterPromise.then(function (ci) {
        self._converter = ci;
        return self._converter || null;
      });
    }
    this._converter = converterInstanceReturn;
  }

  return this._converter || null;
};

/**
 * Set busy state for component for async validators for the displayValue.
 * We want to clear busy state for the same displayValue, not for a different displayValue.
 * I suppose if they type in 111, then 222, then 111, we may clear for second 111 before first,
 * but that seems incredibly unlikely.
 * @param {string} displayValue the displayValue busystate we want to set
 *
 * @private
 * @ignore
 */
oj.EditableValueUtils._SetBusyState = function (displayValue) {
  if (this._resolveBusyStateAsyncMap === undefined) {
    // eslint-disable-next-line no-undef
    this._resolveBusyStateAsyncMap = new Map();
  }

  var resolveBusyStateAsync = this._resolveBusyStateAsyncMap.get(displayValue);

  // Set a page-level busy state if not already set for this displayValue
  if (!resolveBusyStateAsync) {
    var domElem = this.element[0];
    var busyContext = Context.getContext(domElem).getBusyContext();
    var description = 'The page is waiting for async validators for displayValue ' + displayValue;

    if (domElem && domElem.id) {
      description += ' for "' + domElem.id + '" ';
    }
    description += 'to finish.';

    resolveBusyStateAsync = busyContext.addBusyState({ description: description });
    this._resolveBusyStateAsyncMap.set(displayValue, resolveBusyStateAsync);
  }
};

/**
 * Clear busy state for async validators for the displayValue.
 * @param {string} displayValue the displayValue busystate we want to clear
 * @private
 * @ignore
 */
oj.EditableValueUtils._ClearBusyState = function (displayValue) {
  var resolveBusyStateAsync;
  if (this._resolveBusyStateAsyncMap !== undefined) {
    resolveBusyStateAsync = this._resolveBusyStateAsyncMap.get(displayValue);
    if (resolveBusyStateAsync) {
      resolveBusyStateAsync();
      this._resolveBusyStateAsyncMap.delete(displayValue);
    }
  }
};

/**
 * Set busy state for component for async validators hint.
 * We want to clear busy state for the same hint not for a different hint.
 * I use a counter here.
 * @param {number} counter the counter for the busystate we want to set
 *
 * @private
 * @ignore
 */
oj.EditableValueUtils._SetBusyStateAsyncValidatorHint = function (counter) {
  if (this._resolveBusyStateAsyncValidatorHintMap === undefined) {
    // eslint-disable-next-line no-undef
    this._resolveBusyStateAsyncValidatorHintMap = new Map();
  }

  var resolveBusyStateAsyncHint = this._resolveBusyStateAsyncValidatorHintMap.get(counter);

  // Set a page-level busy state if not already set for this counter
  if (!resolveBusyStateAsyncHint) {
    var domElem = this.element[0];
    var busyContext = Context.getContext(domElem).getBusyContext();
    var description = 'The page is waiting for async validator hint for counter ' + counter;

    if (domElem && domElem.id) {
      description += ' for "' + domElem.id + '" ';
    }
    description += 'to finish.';

    resolveBusyStateAsyncHint = busyContext.addBusyState({ description: description });
    this._resolveBusyStateAsyncValidatorHintMap.set(counter, resolveBusyStateAsyncHint);
  }
};

/**
 * Clear busy state for async validators hint for the counter
 * @param {number} counter the counter for the busystate we want to clear
 * @private
 * @ignore
 */
oj.EditableValueUtils._ClearBusyStateAsyncValidatorHint = function (counter) {
  var resolveBusyStateAsyncHint;
  if (this._resolveBusyStateAsyncValidatorHintMap !== undefined) {
    resolveBusyStateAsyncHint = this._resolveBusyStateAsyncValidatorHintMap.get(counter);
    if (resolveBusyStateAsyncHint) {
      resolveBusyStateAsyncHint();
      this._resolveBusyStateAsyncValidatorHintMap.delete(counter);
    }
  }
};

/**
 * Set busy state for component for async converter loading.
 *
 * @private
 * @ignore
 */
oj.EditableValueUtils._SetBusyStateAsyncConverterLoading = function () {
  // Set a page-level busy state if not already set for async converter loading
  if (!this._resolveBusyStateAsyncConverterLoading) {
    var domElem = this.element[0];
    var busyContext = Context.getContext(domElem).getBusyContext();
    var description = 'The page is waiting for async converter loading ';

    if (domElem && domElem.id) {
      description += 'for "' + domElem.id + '" ';
    }
    description += 'to finish.';

    this._resolveBusyStateAsyncConverterLoading =
      busyContext.addBusyState({ description: description });
  }
};

/**
 * Clear busy state for async converter loading
 * @private
 * @ignore
 */
oj.EditableValueUtils._ClearBusyStateAsyncConverterLoading = function () {
  if (this._resolveBusyStateAsyncConverterLoading !== undefined) {
    this._resolveBusyStateAsyncConverterLoading();
    delete this._resolveBusyStateAsyncConverterLoading;
  }
};

/**
 * Retrieve the delay before showing status
 * @return {number} the delay in ms
 * @private
 * @ignore
 */
oj.EditableValueUtils._getShowLoadingDelay = function () {
  if (this._defaultOptions == null) {
    this._defaultOptions = ThemeUtils.parseJSONFromFontFamily('oj-form-control-option-defaults');
  }
  var delay = parseInt(this._defaultOptions.showIndicatorDelay, 10);

  return isNaN(delay) ? 0 : delay;
};

/**
 * Set the type of the input element based on virtualKeyboard option.
 *
 * @param {Array.<string>} allowedTypes an array of allowed types
 *
 * @protected
 * @ignore
 */
oj.EditableValueUtils._SetInputType = function (allowedTypes) {
  // Default input type is text
  var inputType = 'text';
  var agentInfo = oj.AgentUtils.getAgentInfo();

  // Only change the type on mobile browsers
  if (agentInfo.os === oj.AgentUtils.OS.ANDROID ||
      agentInfo.os === oj.AgentUtils.OS.IOS ||
      agentInfo.os === oj.AgentUtils.OS.WINDOWSPHONE) {
    // Get input type from component's virtualKeyboard option
    if (allowedTypes.indexOf(this.options.virtualKeyboard) >= 0) {
      inputType = this.options.virtualKeyboard;
    } else {
      // Get input type from converter's virtualKeyboardHint option
      var converter = this._GetConverter();
      if (converter && converter.resolvedOptions) {
        var resOptions = converter.resolvedOptions();
        if (allowedTypes.indexOf(resOptions.virtualKeyboardHint) >= 0) {
          inputType = resOptions.virtualKeyboardHint;
        }
      }
    }
  }

  if (inputType == null) {
    this.element[0].removeAttribute('type');
  } else {
    this.element[0].setAttribute('type', inputType);
  }
};

/**
 * Copyright (c) 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */
/* global Promise:false, Components:false, Message:false, __ValidationBase:false, Logger:false, Translations:false, LabelledByUtils:false */
/**
 * The various validation modes
 * @ignore
 */

var _sValidationMode = {
  FULL: 1,
  VALIDATORS_ONLY: 2,
  REQUIRED_VALIDATOR_ONLY: 3
};

/**
* String used in the id on the span that surrounds the help icon.
* @const
* @private
* @type {string}
*/
var _HELP_ICON_ID = '_helpIcon';

/**
* valid state constants
* @const
* @private
* @type {string}
*/
var _VALID = 'valid';

/**
* valid state constants
* @const
* @private
* @type {string}
*/
var _INVALID_HIDDEN = 'invalidHidden';

/**
* valid state constants
* @const
* @private
* @type {string}
*/
var _INVALID_SHOWN = 'invalidShown';

/**
* valid state constants
* @const
* @private
* @type {string}
*/
var _PENDING = 'pending';


// E D I T A B L E V A L U E    A B S T R A C T   W I D G E T
/**
 * @ojcomponent oj.editableValue
 * @augments oj.baseComponent
 * @ojsignature [{
 *                target: "Type",
 *                value: "abstract class editableValue<V, SP extends editableValueSettableProperties<V, SV, RV>, SV= V, RV= V> extends baseComponent<SP>"
 *               },
 *               {
 *                target: "Type",
 *                value: "editableValueSettableProperties<V, SV=V, RV=V> extends baseComponentSettableProperties",
 *                for: "SettableProperties"
 *               }
 *              ]
 * @ojtsimport {module: "ojmessaging", type:"AMD", importName: "Message"}
 * @abstract
 * @since 0.6.0
 * @ojshortdesc Abstract EditableValue element
 * @ojrole input
 * @hideconstructor
 *
 * @classdesc
 * Abstract base class for all editable components that are value holders and that require
 * validation and messaging capabilities. <br/>
 *
 * {@ojinclude "name":"validationAndMessagingDoc"}
 *  * <p>
 * Note: The <code class="prettyprint">required</code>, <code class="prettyprint">validators</code>,
 * <code class="prettyprint">converter</code> properties and the <code class="prettyprint">validate</code>
 * method are not on all EditableValue components so they are not on the EditableValue class.
 * See the EditableValue subclasses for which ones have which of these properties. For example,
 * oj-switch, oj-slider, oj-color-palette, and oj-color-spectrum do not have the
 * <code class="prettyprint">validate</code> method nor do they have the
 * <code class="prettyprint">required</code>, <code class="prettyprint">validators</code>,
 * <code class="prettyprint">converter</code> properties since the components
 * wouldn't do anything with these properties anyway. A user can't type into these components and there
 * is no visual representation for 'nothing is set' on these components. Whereas InputBase, inputNumber,
 * inputSearch and combobox do have these properties since a user can type into the field (so you may
 * need to convert it and validate it) and also blank it out (so you may need to mark it required and
 * run the required validator).
 * </p>

 * <p>
 * <h3 id="declarative-binding-section">
 * Declarative Binding
 * <a class="bookmarkable-link" title="Bookmarkable Link" href="#declarative-binding-section"></a>
 * </h3>
 * When the component's <code class="prettyprint">value</code> property is bound to a Knockout
 * observable and when the value changes, whether the observable is updated or not, and whether a
 * 'writeback' to the observable happens or not, depends on the action that caused the value to
 * change.
 * <ul>
 * <li>when the value changes as a result of user interaction </li>
 * <li>when the value changes because normal validation was run as a result of these properties
 * being changed by the app - <code class="prettyprint">converter</code>, <code class="prettyprint">disabled</code>,
 * <code class="prettyprint">required</code>, <code class="prettyprint">validators</code>, then the
 * value is written to the observable. See the specific property docs for details.</li>
 * <li>when the value changes because normal validation was run as a result of these methods being
 * called by the app -
 * <code class="prettyprint">refresh</code>, <code class="prettyprint">validate</code>,
 * then the value is written to the observable. See the specific method docs for details.</li>
 * <li>when the value changes due to programmatic intervention by app then the value is not written
 * back to observable. This is based on the assumption that the app has mutated the observable
 * already. In this case updating the component's <code class="prettyprint">value</code> property
 * alone will not propagate the change automatically to the observable. Updating the observable is
 * recommended as this will propagate the change automatically to the component.
 * </li>
 * </ul>
 * </p>
 *
 * @example <caption>Initialize component</caption>
 * &lt;oj-input-text id="foo" value="abc"/&gt;
 * @example <caption>Initialize component value using two way data binding</caption>
 * &lt;oj-input-text id="foo" value="{{salary}}"/&gt;
 * &lt;script&gt;
 * &nbsp;&nbsp;var salary = ko.observable('abc');
 * &lt;/script&gt;
 */
oj.__registerWidget('oj.editableValue', $.oj.baseComponent,
  {
    widgetEventPrefix: 'oj',

    options:
    {
      /**
       * It is used to establish a relationship between this component and another element.
       * Typically this is not used by the application developer, but by the oj-label custom element's
       * code. One use case is where the oj-label custom element code writes described-by
       * on its form component for accessibility reasons.
       * To facilitate correct screen reader behavior, the described-by attribute is
       * copied to the aria-describedby attribute on the component's dom element.
       * @example <caption>Initialize component with the <code class="prettyprint">described-by</code> attribute specified:</caption>
       * &lt;oj-some-element described-by="someId">&lt;/oj-some-element>
       *
       * @example <caption>Get or set the <code class="prettyprint">describedBy</code> property after initialization:</caption>
       * // getter
       * var descById = myComp.describedBy;
       *
       * // setter
       * myComp.describedBy = "someId";
       *
       * @ojshortdesc Specifies a relationship between this component and another element.
       * @expose
       * @type {?string}
       * @public
       * @instance
       * @memberof oj.editableValue
       * @since 4.0.0
       */
      describedBy: null,
      /**
       * Whether the component is disabled. The default is false.
       *
       * <p>
       * When the <code class="prettyprint">disabled</code> property changes due to programmatic
       * intervention, the component may clear messages and run validation in some cases. </br>
       * <ul>
       * <li>when a required component is initialized as disabled
       * <code class="prettyprint">value="{{currentValue}}" required disabled</code>,
       * deferred validation is skipped.</li>
       * <li>when a disabled component is enabled,
       *  <ul>
       *  <li>if component is invalid and showing messages then all component messages are cleared,
       *  and full validation run using the display value.
       *   <ul>
       *    <li>if there are validation errors, they are shown.</li>
       *    <li>if no errors result from the validation, the <code class="prettyprint">value</code>
       *    property is updated. Page authors can listen to the <code class="prettyprint">valueChanged</code>
       *    event to clear custom errors.</li>
       *   </ul>
       *  </li>
       *
       *  <li>if component is valid and has no errors then deferred validation is run.
       *    <ul>
       *    <li>if there is a deferred validation error, then the valid property is updated. </li>
       *    </ul>
       *  </li>
       *  <li>if component is invalid and deferred errors then component messages are cleared and
       *  deferred validation re-run.
       *    <ul>
       *    <li>if there is a deferred validation error, then the valid property is updated.</li>
       *    </ul>
       *  </li>
       *  </ul>
       * </li>
       * <li>when enabled component is disabled then no validation is run and the component appears
       * disabled.</li>
       * </ul>
       * </p>
       *
       * @example <caption>Initialize component with <code class="prettyprint">disabled</code> attribute:</caption>
       * &lt;oj-some-element disabled>&lt;/oj-some-element>
       *
       * @example <caption>Get or set the <code class="prettyprint">disabled</code> property after initialization:</caption>
       * // getter
       * var disabled = myComp.disabled;
       *
       * // setter
       * myComp.disabled = false;
       *
       * @ojshortdesc Specifies whether the component is disabled. The default is false.
       * @expose
       * @type {boolean}
       * @default false
       * @public
       * @instance
       * @memberof oj.editableValue
       * @since 0.7.0
       */
      disabled: false,
      /**
       * Display options for auxilliary content that determines where it should be displayed
       * in relation to the component.
       *
       * <p>
       * The types of messaging content for which display options can be configured include
       * <code class="prettyprint">converterHint</code>, <code class="prettyprint">helpInstruction</code>,
       * <code class="prettyprint">messages</code>, and <code class="prettyprint">validatorHint</code>.<br/>
       * The display options for each type is specified either as an array of strings or a string. When
       * an array is specified the first display option takes precedence over the second and so on.
       * </p>
       * <p>
       * When display-options changes due to programmatic intervention, the component updates its
       * display to reflect the updated choices. For example, if 'help.instruction' property goes from
       * 'notewindow' to 'none' then it no longer shows in the notewindow.
       * </p>
       * <p>
       * A side note: help.instruction and message detail text can include formatted HTML text, whereas hints and
       * message summary text cannot. If you use formatted text, it should be accessible
       * and make sense to the user if formatting wasn't there.
       * To format the help.instruction, you could do this:
       * <pre class="prettyprint"><code>&lt;html>Enter &lt;b>at least&lt;/b> 6 characters&lt;/html></code></pre>
       * </p>
       *
       * @example <caption>Override default values for <code class="prettyprint">display-options</code>
       * for one component:</caption>
       * // In this example, the display-options are changed from the defaults.
       * // The 'converterHint' is none, the 'validatorHint' is none and the 'helpInstruction' is none,
       * // so only the 'messages' will display in its default state.
       * //
       * &lt;oj-some-element display-options='{"converterHint": "none",
       *                                     "validatorHint": "none",
       *                                     "helpInstruction": "none"}'>&lt;/oj-some-element>
       *
       * @example <caption>Get or set the <code class="prettyprint">displayOptions</code> property after initialization:</caption>
       * // Get one subproperty
       * var hint = myComp.displayOptions.converterHint;
       *
       * // Set one, leaving the others intact. Use the setProperty API for
       * // subproperties so that a property change event is fired.
       * myComp.setProperty("displayOptions.converterHint", "none");
       *
       * // get all
       * var options = myComp.displayOptions;
       *
       * // set all.  Must list every resource key, as those not listed are lost.
       * myComp.displayOptions = {converterHint: "none", validatorHint: "none", helpInstruction: "none"};
       *
       * @ojshortdesc Display options for the form field's messages, converter and validator hints, and help instruction text.
       * @expose
       * @access public
       * @instance
       * @memberof oj.editableValue
       * @type {Object}
       * @since 0.7
       */
      displayOptions: {
          /**
           * Display options for auxilliary converter hint text that determines where it should be displayed
           * in relation to the component.  When there is already a placeholder set on the component,
           * the converter hint falls back to display type of 'notewindow'.
           *
           * @expose
           * @name displayOptions.converterHint
           * @ojshortdesc Display options for auxilliary converter hint text that determines where it should be displayed in relation to the component.
           * @memberof! oj.editableValue
           * @instance
           * @type {Array<string> | string}
           * @default ['placeholder','notewindow']
           * @since 0.7
           * @ojsignature {target: "Type", value: "Array<'placeholder'|'notewindow'|'none'>|'placeholder'|'notewindow'|'none'", jsdocOverride: true}
           */
          /**
           * Display options for auxilliary help instruction text that determines where it should be displayed
           * in relation to the component.
           *
           * @expose
           * @name displayOptions.helpInstruction
           * @memberof! oj.editableValue
           * @instance
           * @type {Array<string> | string}
           * @default ['notewindow']
           * @since 0.7
           * @ojsignature {target: "Type", value: "Array<'notewindow'|'none'>|'notewindow'|'none'", jsdocOverride: true}
           */
          /**
           * Display options for auxilliary message text that determines where it should be displayed
           * in relation to the component.
           *
           * @expose
           * @name displayOptions.messages
           * @memberof! oj.editableValue
           * @instance
           * @type {Array<string> | string}
           * @default ['inline']
           * @since 0.7
           * @ojsignature {target: "Type", value: "Array<'inline'|'notewindow'|'none'>|'inline'|'notewindow'|'none'", jsdocOverride: true}
           */
          /**
           * Display options for auxilliary validator hint text that determines where it should be displayed
           * in relation to the component.
           *
           * @expose
           * @name displayOptions.validatorHint
           * @memberof! oj.editableValue
           * @instance
           * @type {Array<string> | string}
           * @default ['notewindow']
           * @since 0.7
           * @ojsignature {target: "Type", value: "Array<'notewindow'|'none'>|'notewindow'|'none'",  jsdocOverride: true}
           */
      },
      /**
       * Form component help information.
       * @expose
       * @memberof oj.editableValue
       * @instance
       * @public
       * @type {Object}
       * @since 0.7.0
       */
      help: undefined,
      /**
       * <p>help definition text.  See the top-level <code class="prettyprint">help</code> option for details.
       *
       * @expose
       * @alias help.definition
       * @memberof! oj.editableValue
       * @instance
       * @type {?string}
       * @ignore
       * @default null
       *
       * @example <caption>Get or set the <code class="prettyprint">help.definition</code> sub-option, after initialization:</caption>
       * // getter
       * var definitionText = myInputComp.help.definition;
       *
       * // setter:
       * myInputComp.help.definition = "Enter your name";
       */
      /**
       * <p>help source url.  See the top-level <code class="prettyprint">help</code> option for details.
       *
       * @expose
       * @alias help.source
       * @memberof! oj.editableValue
       * @instance
       * @ignore
       * @type {?string}
       * @default null
       *
       * @example <caption>Get or set the <code class="prettyprint">help.source</code> sub-option, after initialization:</caption>
       * // getter
       * var helpSource = myInputComp.help.source;
       *
       * // setter:
       * myInputComp.help.source = "www.abc.com";
       */
      /**
       * Represents hints for oj-form-layout element to render help information on the label of the editable component.
       * <p>This is used only if the editable component is added as a direct child to an oj-form-layout element, and the labelHint property is also specified.</p>
       *
       * <p>
       * The helpHints object contains a definition property and a source property.
       * </p>
       * <ul>
       * <li><code class="prettyprint">definition</code> - hint for help definition text.</li>
       * <li><code class="prettyprint">source</code> - hint for help source URL.</li>
       * </ul>
       *
       * @example <caption>Initialize the component with help hints:</caption>
       * &lt;!-- Using dot notation -->
       * &lt;oj-some-element help-hints.definition='some value' help-hints.source='some-url'>&lt;/oj-some-element>
       *
       * &lt;!-- Using JSON notation -->
       * &lt;oj-some-element help-hints='{"definition":"some value", "source":"some-url"}'>&lt;/oj-some-element>
       *
       * @example <caption>Get or set the <code class="prettyprint">helpHints</code> property after
       * initialization:</caption>
       *
       * // Get one
       * var value = myComponent.helpHints.definition;
       *
       * // Set one, leaving the others intact. Always use the setProperty API for
       * // subproperties rather than setting a subproperty directly.
       * myComponent.setProperty('helpHints.definition', 'some new value');
       *
       * // Get all
       * var values = myComponent.helpHints;
       *
       * // Set all.  Must list every subproperty, as those not listed are lost.
       * myComponent.helpHints = {
       *     definition: 'some new value',
       *     source: 'some-new-url'
       * };
       *
       * @ojshortdesc Represents hints for an oj-form-layout element to render help information on the label of the editable component.
       * @expose
       * @access public
       * @memberof oj.editableValue
       * @ojtranslatable
       * @instance
       * @type {Object}
       * @since 4.1.0
       */
      helpHints:
      {
        /**
         * Hint for help definition text associated with the label.
         * <p>It is what shows up when the user hovers over the help icon, or tabs into the help icon, or press and holds the help icon on a mobile device. No formatted text is available for help definition attribute.</p>
         *
         * <p>See the <a href="#helpHints">help-hints</a> attribute for usage examples.</p>
         *
         * @ojshortdesc Hint for help definition text associated with the label.
         * @expose
         * @alias helpHints.definition
         * @memberof! oj.editableValue
         * @instance
         * @type {string}
         * @ojsignature {target:"Type", value: "?"}
         * @default ""
         * @since 4.1.0
         */
        definition: '',
        /**
         * Hint for help source URL associated with the label.
         * <p>If present, a help icon will render next to the label. For security reasons we only support urls with protocol http: or https:. If the url doesn't comply we ignore it and throw an error.
         * Pass in an encoded URL since we do not encode the URL.</p>
         *
         * <p>See the <a href="#helpHints">help-hints</a> attribute for usage examples.</p>
         *
         * @ojshortdesc Hint for help source URL associated with the label.
         * @expose
         * @alias helpHints.source
         * @memberof! oj.editableValue
         * @instance
         * @type {string}
         * @ojsignature {target:"Type", value: "?"}
         * @default ""
         * @since 4.1.0
         */
        source: ''
      },
      /**
       * Represents a hint for oj-form-layout element to render a label on the editable component.
       * <p>This is used only if the editable component is added as a direct child to an oj-form-layout element.</p>
       *
       * <p>
       * When labelHint is present it gives a hint to the oj-form-layout element to create an oj-label element for the editable component.
       * When the <code class="prettyprint">labelHint</code> property changes oj-form-layout element refreshes to
       * display the updated label information.
       * </p>
       *
       * @example <caption>Add the component as a direct child of oj-form-layout. Initialize the component with the <code class="prettyprint">label-hint</code> attribute specified.</caption>
       * &lt;oj-form-layout id= 'someId'>
       * &lt;/oj-some-element label-hint='input label'>&lt;/oj-some-element>
       * &lt;/oj-form-layout>
       *
       * @example <caption>Get or set the <code class="prettyprint">labelHint</code> property after
       * initialization:</caption>
       *
       * // getter
       * var value = myComponent.labelHint;
       *
       * // setter
       * myComponent.labelHint = 'some new value'
       *
       * @ojshortdesc Represents a hint for oj-form-layout element to render a label on the editable component.
       * @expose
       * @access public
       * @instance
       * @alias labelHint
       * @ojtranslatable
       * @default ""
       * @memberof oj.editableValue
       * @type {string}
       * @since 4.1.0
       */
      labelHint: '',
      /**
       * List of messages an app would add to the component when it has business/custom validation
       * errors that it wants the component to show. This allows the app to perform further validation
       * before sending data to the server. When this option is set the message shows to the
       * user right away. To clear the custom message, set <code class="prettyprint">messagesCustom</code>
       * back to an empty array.<br/>
       * <p>Each message in the array is an object that duck types oj.Message.
       * See {@link oj.Message} for details.
       * </p>
       * <p>
       * See the <a href="#validation-section">Validation and Messages</a> section
       * for details on when the component clears <code class="prettyprint">messagesCustom</code>;
       * for example, when full validation is run.
       * </p>
       *
       *
       * @example <caption>Get or set the <code class="prettyprint">messagesCustom</code> property after initialization:</caption>
       * // getter
       * var customMsgs = myComp.messagesCustom;
       *
       * // setter
       * myComp.messagesCustom = [{summary:"hello", detail:"detail", severity:oj.Message.SEVERITY_LEVEL.INFO}];
       *
       * @example <caption>Set messagesCustom when there are cross-validation errors:</caption>
       * --- HTML ---
       * &lt;oj-some-element messages-custom='{{messagesCustom}}'>&lt;/oj-some-element>
       *
       * --- ViewModel code ---
       * self.messagesCustom = ko.observableArray([]);
       *
       * // the app's function that gets called when the user presses the submit button
       * if (!myValidateCrossValidationFields())
       * {
       *   // the app adds a custom messages to the component and it is displayed right away
       *   var msgs = [];
       *   msgs.push({'summary':'Cross field error', 'detail':'Field 1 needs to be less than Field 2'});
       *   self.messagesCustom(msgs);
       * }
       * else
       * {
       *   // submit data to the server
       * }
       *
       * @ojshortdesc A list of messages added by an application to the component. See the Help documentation for more information.
       * @expose
       * @access public
       * @instance
       * @memberof oj.editableValue
       * @default []
       * @type {Array.<Object>}
       * @ojsignature {target: "Type", value: "Array<oj.Message>"}
       * @since 0.7.0
       * @ojwriteback
       */
      messagesCustom: [],

      /**
       * List of messages currently hidden on component, these are added by component when it runs
       * deferred validation. Each message in the array is an object that duck types oj.Message.
       * See {@link oj.Message} for
       * details. <br/>
       *
       * <p>
       * This is a read-only option so page authors cannot set or change it directly.
       * </p>
       *
       * <p>
       * These messages are not shown to the end-user by default, but page author
       * can show hidden messages using the {@link showMessages} method.
       * </p>
       *
       * @example <caption>Get <code class="prettyprint">messagesShown</code> for the component:</caption>
       * // Foo is InputText, InputNumber, Select, etc.
       * var messages = myInputComp.messagesShown;
       *
       * @expose
       * @access public
       * @instance
       * @memberof oj.editableValue
       * @default []
       * @type {Array.<Object>|undefined}
       * @since 0.7.0
       * @see #showMessages
       * @readonly
       * @ignore
       * @ojwriteback
       */
      messagesHidden: undefined,

      /**
       * List of messages currently shown on component, these include messages generated both by the
       * component and ones provided by app using <code class="prettyprint">messagesCustom</code>.
       * Each message in the array is an object that duck types oj.Message.
       * See {@link oj.Message} for details. <br/>
       *
       * <p>
       * This is a read-only option so page authors cannot set or change it directly.
       * </p>
       *
       * <p>
       * Messages retrieved using the <code class="prettyprint">messagesShown</code> option are by
       * default shown inline, but this can be controlled using the 'messages' property of
       * the <code class="prettyprint">displayOptions</code> option.
       * </p>
       *
       * @example <caption>Get <code class="prettyprint">messagesShown</code> for the component:</caption>
       * // Foo is InputText, InputNumber, Select, etc.
       * var messages = myInputComp.messagesShown;
       *
       * @expose
       * @access public
       * @instance
       * @memberof oj.editableValue
       * @default []
       * @type {Array.<Object>|undefined}
       * @since 0.7.0
       * @readonly
       * @ignore
       * @ojwriteback
       */
      messagesShown: undefined,


      /**
       * Represents advisory information for the component, such as would be appropriate for a tooltip.
       *
       * <p>
       * When help.instruction is present it is by default displayed in the notewindow, or as determined by the
       * 'helpInstruction' property set on the <code class="prettyprint">displayOptions</code> attribute.
       * When the <code class="prettyprint">help.instruction</code> property changes the component refreshes to
       * display the updated information.
       * </p>
       *
       * <p>
       * JET takes the help instruction text and creates a notewindow with the text. The notewindow pops up
       * when the field takes focus and closes when the field loses focus.
       * </p>
       * <p>
       * How is help.instruction better than the html 'title' attribute?
       * The html 'title' attribute only shows up as a tooltip on mouse over, not on keyboard and not in a mobile
       * device. So the html 'title' would only be for text that is not important enough to show all users, or
       * for text that you show the users in another way as well, like in the label.
       * Also you cannot theme the native browser's title window like you can the JET
       * notewindow, so low vision users may have a hard time seeing the 'title' window.
       * For these reasons, the JET EditableValue components do not use the HTML's 'title'
       * attribute and instead use the help.instruction attribute.
       * </p>
       *
       * <p>
       * To include formatted text in the help.instruction, format the string using html tags.
       * For example the
       * help.instruction might look like:
       * <pre class="prettyprint"><code>&lt;oj-some-element help.instruction="&lt;html>Enter &lt;b>at least&lt;/b> 6 characters&lt;/html>">&lt;/oj-some-element></code></pre>
       * If you use formatted text, it should be accessible
       * and make sense to the user if formatting wasn't there.
       *
       * @example <caption>Initialize the component with the <code class="prettyprint">help.instruction</code> sub-attribute:</caption>
       * &lt;oj-some-element help.instruction="some tooltip">&lt;/oj-some-element>
       *
       * @example <caption>Get or set the <code class="prettyprint">help.instruction</code> property after initialization:</caption>
       * // Get one subproperty
       * var instr = myComp.help.instruction;
       *
       * // Set one subproperty, leaving the others intact. Use the setProperty API for
       * // subproperties so that a property change event is fired.
       * myComponent.setProperty('help.instruction', 'some new value');
       *
       * // Get all
       * var values = myComponent.help;
       *
       * // Set all.  Must list every resource key, as those not listed are lost.
       * myComponent.help = {
       *   instruction: 'some new value'
       * };
       *
       * @ojshortdesc Represents advisory information for the component, such as would be appropriate for a tooltip.
       * @expose
       * @access public
       * @instance
       * @alias help.instruction
       * @ojtranslatable
       * @default ""
       * @memberof! oj.editableValue
       * @type {string=}
       * @since 4.0.0
       */
      title: '',

      /**
       * <p>
       * The current valid state of the component. It is evaluated on initial render.
       * It is re-evaluated
       * <ul>
       *   <li>after each validator (validators or async-validators) is run (full or deferred)</li>
       *   <li>when messagesCustom is updated,
       *   since messagesCustom can be added by the app developer any time.</li>
       *   <li>when showMessages() is called. Since showMessages() moves the
       *   hidden messages into messages shown,
       *   if the valid state was "invalidHidden" then it would become "invalidShown".</li>
       *   <li>when the required property has changed. If a component is empty and has required
       *   set, the valid state may be "invalidHidden" (if no invalid messages are being shown as well).
       *   If required property is removed, the valid state would change to "valid".</li>
       * </ul>
       * </p>
       * <p>
       *  Note: New valid states may be added to the list of valid values in future releases.
       *  Any new values will start with "invalid"
       *  if it is an invalid state, "pending" if it is pending state,
       *  and "valid" if it is a valid state.
       * </p>
       * @example <caption>Get <code class="prettyprint">valid</code> attribute, after initialization:</caption>
       * // Getter:
       * var valid = myComp.valid;
       * @example <caption>Set the <code class="prettyprint">on-valid-changed</code>
       *  listener so you can do work in the ViewModel based on the
       *  <code class="prettyprint">valid</code> property:</caption>
       * &lt;oj-some-element id='username' on-valid-changed='[[validChangedListener]]'>
       * &lt;/oj-some-element>
       * &lt;oj-some-element id='password' on-valid-changed='[[validChangedListener]]'>
       * &lt;/oj-some-element>
       * &lt;oj-button disabled='[[componentDisabled]]' on-click='[[submit]]'>Submit&lt;/oj-button>
       * -- ViewModel --
       * self.validChangedListener = function (event) {
       *   var enableButton;
       *   var usernameValidState;
       *   var passwordValidState;
       *
       *   // update the button's disabled state.
       *   usernameValidState = document.getElementById("username").valid;
       *   passwordValidState = document.getElementById("password").valid;
       *
       *   // this updates the Submit button's disabled property's observable based
       *   // on the valid state of two components, username and password.
       *   // It is up to the application how it wants to handle the “pending�? state
       *   // but we think that in general buttons don’t need to be
       *   // enabled / disabled based on the "pending" state.
       *   enableButton =
       *    (usernameValidState !== "invalidShown") &&
       *    (passwordValidState !== "invalidShown");
       *   self.componentDisabled(!enableButton);;
       * };
       *
       * @ojshortdesc The validity state of the component
       * @expose
       * @access public
       * @instance
       * @type {string}
       * @ojvalue {string} "valid" The component is valid
       * @ojvalue {string} "pending" The component is waiting for the validation state to be determined.
       * The "pending" state is set at the start of the convert/validate process.
       * @ojvalue {string} "invalidHidden" The component has invalid messages hidden
       *    and no invalid messages showing. An invalid message is one with severity "error" or higher.
       * @ojvalue {string} "invalidShown" The component has invalid messages showing.
       *  An invalid message is one with severity "error" or higher.
       * @ojwriteback
       * @readonly
       * @memberof oj.editableValue
       * @since 4.2.0
       * @ojstatus preview
       */
      valid: undefined,

      /**
       * The value of the component.
       *
       * <p>
       * When <code class="prettyprint">value</code> property changes due to programmatic
       * intervention, the component always clears all messages
       * including <code class="prettyprint">messagesCustom</code>, runs deferred validation, and
       * always refreshes UI display value.</br>
       *
       * <h4>Running Validation</h4>
       * <ul>
       * <li>component always runs deferred validation; the
       * <code class="prettyprint">valid</code> property is updated with the result.</li>
       * </ul>
       * </p>
       *
       * @example <caption>Initialize the component with the <code class="prettyprint">value</code> attribute specified:</caption>
       * &lt;oj-some-element value='10'>&lt;/oj-some-element>
       * @example <caption>Get or set <code class="prettyprint">value</code> attribute, after initialization:</caption>
       * // Getter: returns '10'
       * var val = myComp.value;
       * // Setter: sets '20'
       * myComp.value = '20';
       *
       * @ojshortdesc The value of the component.
       * @expose
       * @access public
       * @instance
       * @default null
       * @ojwriteback
       * @ojeventgroup common
       * @memberof oj.editableValue
       * @since 0.6.0
       * @type {any}
       * @ojsignature {
       *                 target: "Accessor",
       *                 value: {
       *                          GetterType: "V|null",
       *                          SetterType: "SV|null"}
       *              }
       */
      value: undefined,

      // Events
      /**
       * Triggered when a default animation is about to start on an element owned by the component.
       *
       * <p>The default animation can be cancelled by calling <code class="prettyprint">event.preventDefault</code>, followed by
       * a call to <code class="prettyprint">event.detail.endCallback</code>.  <code class="prettyprint">event.detail.endCallback</code> should be
       * called immediately after <code class="prettyprint">event.preventDefault</code> if the application merely wants to cancel animation,
       * or it should be called when the custom animation ends if the application is invoking another animation function.  Failure to
       * call <code class="prettyprint">event.detail.endCallback</code> may prevent the component from working properly.</p>
       * <p>For more information on customizing animations, see the documentation of
       * <a href="oj.AnimationUtils.html">oj.AnimationUtils</a>.</p>
       *
       * <caption>The default animations are controlled via the theme (SCSS) :</caption>
       * <pre class="prettyprint"><code>
       * // default animations for "notewindow" display option
       * $popupTailOpenAnimation: (effect: "zoomIn", transformOrigin: "#myPosition") !default;
       * $popupTailCloseAnimation: (effect: "none") !default;
       *
       * // default animations for "inline" display option
       * $messageComponentInlineOpenAnimation: (effect: "expand", startMaxHeight: "#oldHeight") !default;
       * $messageComponentInlineCloseAnimation: (effect: "collapse", endMaxHeight: "#newHeight") !default;
       * </code></pre>
       *
       * @ojshortdesc Triggered when a default animation is about to start, such as when the component is being opened/closed or a child item is being added/removed.
       *
       * @expose
       * @event
       * @memberof oj.editableValue
       * @since 4.0.0
       * @ojbubbles
       * @ojcancelable
       * @instance
       * @property {string} action The action that triggers the animation. Supported values are:
       *                    <ul>
       *                      <li>"inline-open" - when an inline message container opens or increases in size</li>
       *                      <li>"inline-close" - when an inline message container closes or decreases in size</li>
       *                      <li>"notewindow-open" - when a note window opens</li>
       *                      <li>"notewindow-close" - when a note window closes</li>
       *                    </ul>
       * @property {Element} element The element being animated.
       * @property {function():void} endCallback If the event listener calls event.preventDefault to
       *            cancel the default animation, it must call the endCallback function when it
       *            finishes its own animation handling and any custom animation has ended.
       *
       * @example <caption>Define an event listener for the
       *          <code class="prettyprint">ojAnimateStart</code> event to override the default
       *          "inline-open" animation:</caption>
       * var listener = function( event )
       *   {
       *     // Change the "inline-open" animation for inline message
       *     if (event.detail.action == "inline-open") {
       *       // Cancel default animation
       *       event.preventDefault();
       *       // Invoke new animation and call endCallback when the animation ends
       *       oj.AnimationUtils.fadeIn(event.detail.element).then(event.detail.endCallback);
       *     }
       *   };
       *
       * @example <caption>Define an event listener for the
       *          <code class="prettyprint">ojAnimateStart</code> event to cancel the default
       *          "notewindow-close" animation:</caption>
       * var listener = function( event )
       *   {
       *     // Change the "notewindow-close" animation for note window
       *     if (ui.action == "notewindow-close") {
       *       // Cancel default animation
       *       event.preventDefault();
       *       // Call endCallback immediately to indicate no more animation
       *       event.detail.endCallback();
       *     }
       *   };
       */
      animateStart: null,
      /**
       * Triggered when a default animation has ended.
       *
       * @expose
       * @event
       * @ojbubbles
       * @ojcancelable
       * @memberof oj.editableValue
       * @since 4.0.0
       * @instance
       * @property {string} action The action that triggers the animation. Supported values are:
       *                    <ul>
       *                      <li>"inline-open" - when an inline message container opens or increases in size</li>
       *                      <li>"inline-close" - when an inline message container closes or decreases in size</li>
       *                      <li>"notewindow-open" - when a note window opens</li>
       *                      <li>"notewindow-close" - when a note window closes</li>
       *                    </ul>
       * @property {Element} element The element being animated.
       * @example <caption>Define an event listener for the
       *          <code class="prettyprint">ojAnimateEnd</code> event to add any processing after the end of
       *          "inline-open" animation:</caption>
       * var listener = function( event )
       * {
       *   // Check if this is the end of "inline-open" animation for inline message
       *   if (event.detail.action == "inline-open") {
       *     // Add any processing here
       *   }
       * };
       */
      animateEnd: null
    },

    // P U B L I C    M E T H O D S

    // @inheritdoc
    getNodeBySubId: function (locator) {
      var node;
      var subId;

      node = this._super(locator);

      // this subId is only for non-custom element components so skip if custom element
      if (!node && !this._IsCustomElement()) {
        subId = locator.subId;

        if (subId === 'oj-label-help-icon') {
          var label = this._GetLabelElement();

          if (label) {
            node = label.parent().find('.oj-label-help-icon');
          }
        }
      }
      // Non-null locators have to be handled by the component subclasses
      return node || nul