/**
 * @license
 * Copyright (c) 2014, 2019, Oracle and/or its affiliates.
 * The Universal Permissive License (UPL), Version 1.0
 */
define(['ojs/ojcore', 'jquery', 'ojs/ojtranslation', 'ojs/ojmessaging', 'ojs/ojlocaledata', 'ojs/ojlogger'], function(oj, $, Translations, Message, LocaleData, Logger)
{
  "use strict";
/**
 * Copyright (c) 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */

/*
 DESCRIPTION
 oj.OraI18nUtils provides helper functions for converter objects.

 PRIVATE CLASSES
 <list of private classes defined - with one-line descriptions>

 NOTES
 <other useful comments, qualifications, etc.>

 */


oj.OraI18nUtils = {};
// supported numbering systems
oj.OraI18nUtils.numeringSystems = {
  latn: '\u0030\u0031\u0032\u0033\u0034\u0035\u0036\u0037\u0038\u0039',
  arab: '\u0660\u0661\u0662\u0663\u0664\u0665\u0666\u0667\u0668\u0669',
  thai: '\u0e50\u0e51\u0e52\u0e53\u0e54\u0e55\u0e56\u0e57\u0e58\u0e59'
};

oj.OraI18nUtils.regexTrim = /^\s+|\s+$|\u200f|\u200e/g;
oj.OraI18nUtils.regexTrimNumber = /\s+|\u200f|\u200e/g;
oj.OraI18nUtils.regexTrimRightZeros = /0+$/g;
oj.OraI18nUtils.zeros = ['0', '00', '000'];
// ISO 8601 string accepted values:
// -date only: YYYY or YYYY-MM or YYYY-MM-dd
// -time only without timezone: Thh:mm or Thh:mm:ss or Thh:mm:ss.SSS
// -time only with timezone: any of the time values above followed by any of the following:
// Z or +/-hh:mm or +/-hhmm or +/-hh
// -date time: any of the date values followed by any of the time values
oj.OraI18nUtils._ISO_DATE_REGEXP =
  /^[+-]?\d{4}(?:-\d{2}(?:-\d{2})?)?(?:T\d{2}:\d{2}(?::\d{2}(?:\.\d{1,3})?)?(Z|[+-]\d{2}(?::?\d{2})?)?)?$|^T\d{2}:\d{2}(?::\d{2}(?:\.\d{1,3})?)?(Z|[+-]\d{2}(?::?\d{2})?)?$/;
/**
 * Returns the timezone offset between UTC and the local time in Etc/GMT[+-]hh:mm syntax.
 * The offset is positive if the local timezone is behind UTC and negative if
 * it is ahead. The offset range is between Etc/GMT-14 and Etc/GMT+12 (UTC-12 and UTC+14)
 * Examples:
 * 1- The local time is UTC-7 (Pacific Daylight Time):
 * oj.OraI18nUtils. getLocalTimeZoneOffset() will return the string "Etc/GMT+07:00"
 * 2- The local time is UTC+1 (Central European Standard Time):
 * oj.OraI18nUtils. getLocalTimeZoneOffset() will return the string "Etc/GMT-01:00"
 * @returns {string}
 */
oj.OraI18nUtils.getLocalTimeZoneOffset = function () {
  var d = new Date();
  var offset = d.getTimezoneOffset();
  return oj.OraI18nUtils.getTimeStringFromOffset('Etc/GMT', offset, false, false);
};


/*
 * Will return timezone if it exists.
 */
// This private method is not called, commenting out for now to remove it from code coverage calculations
// oj.OraI18nUtils._getTimeZone = function (isoString)
// {
//  if (!isoString || typeof isoString !== "string")
//  {
//    return null;
//  }
//  var match = oj.OraI18nUtils._ISO_DATE_REGEXP.exec(isoString);
//  //make sure it is iso string
//  if (match === null)
//  {
//    oj.OraI18nUtils._throwInvalidISOString(isoString);
//  }
//  if (match[1] !== undefined)
//    return match[1];
//  return null;
// };

/*
 * Will return local isoString provided a date.
 */
oj.OraI18nUtils.dateToLocalIso = function (_date) {
  var date = _date;
  if (typeof date === 'number') {
    date = new Date(date);
  }
  var isoStr = (oj.OraI18nUtils.padZeros(date.getFullYear(), 4) + '-' +
                oj.OraI18nUtils.padZeros((date.getMonth() + 1), 2) + '-' +
                oj.OraI18nUtils.padZeros(date.getDate(), 2) + 'T' +
                oj.OraI18nUtils.padZeros((date.getHours()), 2) + ':' +
                oj.OraI18nUtils.padZeros((date.getMinutes()), 2) + ':' +
                oj.OraI18nUtils.padZeros((date.getSeconds()), 2));
  if (date.getMilliseconds() > 0) {
    isoStr +=
      '.' + oj.OraI18nUtils.trimRightZeros(oj.OraI18nUtils.padZeros(date.getMilliseconds(), 3));
  }
  return isoStr;
};

oj.OraI18nUtils.partsToIsoString = function (parts) {
  var isoStr = (oj.OraI18nUtils.padZeros(parts[0], 4) + '-' +
                oj.OraI18nUtils.padZeros(parts[1], 2) + '-' +
                oj.OraI18nUtils.padZeros(parts[2], 2) + 'T' +
                oj.OraI18nUtils.padZeros(parts[3], 2) + ':' +
                oj.OraI18nUtils.padZeros(parts[4], 2) + ':' +
                oj.OraI18nUtils.padZeros(parts[5], 2));
  if (parts[6] > 0) {
    isoStr += '.' + oj.OraI18nUtils.trimRightZeros(oj.OraI18nUtils.padZeros(parts[6], 3));
  }
  return isoStr;
};

oj.OraI18nUtils.isoToLocalDate = function (isoString) {
  if (!isoString || typeof isoString !== 'string') {
    return null;
  }
  return this._isoToLocalDateIgnoreTimezone(isoString);
};

oj.OraI18nUtils._isoToLocalDateIgnoreTimezone = function (isoString) {
  var datetime = oj.OraI18nUtils._IsoStrParts(isoString);
  return new Date(datetime[0], datetime[1] - 1, datetime[2], datetime[3],
                  datetime[4], datetime[5], datetime[6]);
};

oj.OraI18nUtils._IsoStrParts = function (isoString) {
  var tst = oj.OraI18nUtils._ISO_DATE_REGEXP.test(isoString);
  if (tst === false) {
    oj.OraI18nUtils._throwInvalidISOStringSyntax(isoString);
  }
  var splitted = isoString.split('T');
  var tIndex = isoString.indexOf('T');
  var today = new Date();
  var i;
  var isBC = false;
  var datetime = [today.getFullYear(), today.getMonth() + 1, today.getDate(), 0, 0, 0, 0];

  if (splitted[0] !== '') {
    // contains date portion
    // test if date is BC, year is negative
    if (oj.OraI18nUtils.startsWith(splitted[0], '-')) {
      splitted[0] = splitted[0].substr(1);
      isBC = true;
    }
    var dateSplitted = splitted[0].split('-');
    for (i = 0; i < dateSplitted.length; i++) {
      var val = parseInt(dateSplitted[i], 10);
      // validate month
      if (i === 1) {
        if (val < 1 || val > 12) {
          oj.OraI18nUtils._throwInvalidISOStringRange(isoString, 'month', val, 1, 12);
        }
      }
      // validate day
      if (i === 2) {
        var nbDays = oj.OraI18nUtils._getDaysInMonth(datetime[0], datetime[1] - 1);
        if (val < 1 || val > nbDays) {
          oj.OraI18nUtils._throwInvalidISOStringRange(isoString, 'day', val, 1, nbDays);
        }
      }
      datetime[i] = val;
    }
    if (isBC) {
      datetime[0] = -datetime[0];
    }
  }

  if (tIndex !== -1) {
    var milliSecSplitted = splitted[1].split('.'); // contain millseconds
    var timeSplitted = milliSecSplitted[0].split(':'); // contain hours, minutes, seconds

    for (i = 0; i < timeSplitted.length; i++) {
      var tVal = parseInt(timeSplitted[i], 10);
      // validate hour
      if (i === 0) {
        if (tVal < 0 || tVal > 24) {
          oj.OraI18nUtils._throwInvalidISOStringRange(isoString, 'hour', tVal, 0, 24);
        }
      }
      // validate minute
      if (i === 1) {
        if (tVal < 0 || tVal > 59) {
          oj.OraI18nUtils._throwInvalidISOStringRange(isoString, 'minute', tVal, 0, 59);
        }
      }
      // validate second
      if (i === 2) {
        if (tVal < 0 || tVal > 59) {
          oj.OraI18nUtils._throwInvalidISOStringRange(isoString, 'second', tVal, 0, 59);
        }
      }
      datetime[3 + i] = tVal;
    }

    if (milliSecSplitted.length === 2 && milliSecSplitted[1]) {
      datetime[6] = parseInt(oj.OraI18nUtils.zeroPad(milliSecSplitted[1], 3, false), 10);
    }
  }
  return datetime;
};

oj.OraI18nUtils.getISOStrFormatInfo = function (isoStr) {
  var res = {
    format: null,
    dateTime: null,
    timeZone: '',
    isoStrParts: null
  };
  var exe = oj.OraI18nUtils._ISO_DATE_REGEXP.exec(isoStr);

  if (exe === null) {
    oj.OraI18nUtils._throwInvalidISOStringSyntax(isoStr);
  }
  if (exe[1] === undefined && exe[2] === undefined) {
    res.format = 'local';
    res.dateTime = isoStr;
    res.isoStrParts = oj.OraI18nUtils._IsoStrParts(res.dateTime);
    return res;
  }
  res.timeZone = (exe[1] !== undefined) ? exe[1] : exe[2];
  if (res.timeZone === 'Z') {
    res.format = 'zulu';
  } else {
    res.format = 'offset';
  }
  var isoStrLen = isoStr.length;
  var timeZoneLen = res.timeZone.length;
  res.dateTime = isoStr.substring(0, isoStrLen - timeZoneLen);
  res.isoStrParts = oj.OraI18nUtils._IsoStrParts(res.dateTime);
  return res;
};

// This private method is not called, commenting out for now to remove it from code coverage calculations
// oj.OraI18nUtils._throwTimeZoneNotSupported = function () {
//  var msg, error, errorInfo;
//  msg = "time zone is not supported";
//  error = new Error(msg);
//  errorInfo = {
//    'errorCode': 'timeZoneNotSupported'
//  };
//  error['errorInfo'] = errorInfo;
//  throw error;
// };

oj.OraI18nUtils._isLeapYear = function (y) {
  if (y % 400 === 0) {
    return true;
  } else if (y % 100 === 0) {
    return false;
  } else if (y % 4 === 0) {
    return true;
  }
  return false;
};

// Get days in month depending on month and leap year
oj.OraI18nUtils._getDaysInMonth = function (y, m) {
  switch (m) {
    case 0 :
    case 2 :
    case 4 :
    case 6 :
    case 7 :
    case 9 :
    case 11 :
      return 31;
    case 1:
      if (oj.OraI18nUtils._isLeapYear(y)) {
        return 29;
      }
      return 28;
    default:
      return 30;
  }
};
oj.OraI18nUtils._throwInvalidISOStringRange = function (isoStr, name, displayValue,
  displayLow, displayHigh) {
  var msg = 'The string ' + isoStr + ' is not a valid ISO 8601 string: ' + displayValue +
          ' is out of range.  Enter a value between ' + displayLow +
          ' and ' + displayHigh + ' for ' + name;
  var rangeError = new RangeError(msg);
  var errorInfo = {
    errorCode: 'isoStringOutOfRange',
    parameterMap: {
      isoString: isoStr,
      value: displayValue,
      minValue: displayLow,
      maxValue: displayHigh,
      propertyName: name
    }
  };
  rangeError.errorInfo = errorInfo;
  throw rangeError;
};

oj.OraI18nUtils._throwInvalidISOStringSyntax = function (str) {
  var msg = 'The string ' + str + ' is not a valid ISO 8601 string syntax.';
  var error = new Error(msg);
  var errorInfo = {
    errorCode: 'invalidISOString',
    parameterMap: {
      isoStr: str
    }
  };
  error.errorInfo = errorInfo;
  throw error;
};

oj.OraI18nUtils.trim = function (value) {
  return (value + '').replace(oj.OraI18nUtils.regexTrim, '');
};

oj.OraI18nUtils.trimRightZeros = function (value) {
  return (value + '').replace(oj.OraI18nUtils.regexTrimRightZeros, '');
};


oj.OraI18nUtils.trimNumber = function (value) {
  var s = (value + '').replace(oj.OraI18nUtils.regexTrimNumber, '');
  return s;
};

oj.OraI18nUtils.startsWith = function (value, pattern) {
  return value.indexOf(pattern) === 0;
};

oj.OraI18nUtils.toUpper = function (value) {
  // "he-IL" has non-breaking space in weekday names.
  return value.split('\u00A0').join(' ').toUpperCase();
};

oj.OraI18nUtils.padZeros = function (num, c) {
  var s = num + '';
  var isNegative = false;
  if (num < 0) {
    s = s.substr(1);
    isNegative = true;
  }
  if (c > 1 && s.length < c) {
    s = (oj.OraI18nUtils.zeros[c - 2] + s);
    s = s.substr(s.length - c, c);
  }
  if (isNegative) {
    s = '-' + s;
  }
  return s;
};

oj.OraI18nUtils.zeroPad = function (str, count, left) {
  var result = '' + str;

  for (var l = result.length; l < count; l += 1) {
    result = (left ? ('0' + result) : (result + '0'));
  }
  return result;
};

oj.OraI18nUtils.getTimeStringFromOffset = function (prefix, offset, reverseSign, alwaysMinutes) {
  var isNegative = reverseSign ? offset >= 0 : offset < 0;
  var absOffset = Math.abs(offset);
  var hours = Math.floor(absOffset / 60);
  var minutes = absOffset % 60;
  var sign = isNegative ? '-' : '+';

  if (alwaysMinutes) {
    hours = oj.OraI18nUtils.zeroPad(hours, 2, true);
  }
  var str = prefix + sign + hours;
  if (minutes > 0 || alwaysMinutes) {
    str += ':' + oj.OraI18nUtils.zeroPad(minutes, 2, true);
  }
  return str;
};

// get the numbering system key from the locale's unicode extension.
// Verify that the locale data has a numbers entry for it, if not return latn as default.
oj.OraI18nUtils.getNumberingSystemKey = function (localeElements, locale) {
  if (locale === undefined) {
    return 'latn';
  }
  var numberingSystemKey = oj.OraI18nUtils.getNumberingExtension(locale);
  var symbols = 'symbols-numberSystem-' + numberingSystemKey;
  if (localeElements.numbers[symbols] === undefined) {
    numberingSystemKey = 'latn';
  }
  return numberingSystemKey;
};

// return the language part
oj.OraI18nUtils.getBCP47Lang = function (tag) {
  var arr = tag.split('-');
  return arr[0];
};

// return the region part. tag is lang or lang-region or lang-script or
// lang-script-region
oj.OraI18nUtils.getBCP47Region = function (tag) {
  var arr = tag.split('-');
  if (arr.length === 3) {
    return arr[2];
  }
  if (arr.length === 2) {
    if (arr[1].length === 2) {
      return arr[1];
    }
  }
  return '001';
};


// get the unicode numbering system extension.
oj.OraI18nUtils.getNumberingExtension = function (_locale) {
  var locale = _locale || 'en-US';
  var idx = locale.indexOf('-u-nu-');
  var numbering = 'latn';
  if (idx !== -1) {
    numbering = locale.substr(idx + 6, 4);
  }
  return numbering;
};

oj.OraI18nUtils.haveSamePropertiesLength = function (obj) {
  return Object.keys(obj).length;
};

// cldr locale data start with "main" node.
// return the subnode under main.
oj.OraI18nUtils.getLocaleElementsMainNode = function (bundle) {
  var mainNode = bundle.main;
  var keys = Object.keys(mainNode);
  return mainNode[keys[0]];
};

// get the locale which is a subnode of "main".
oj.OraI18nUtils.getLocaleElementsMainNodeKey = function (bundle) {
  var mainNode = bundle.main;
  var keys = Object.keys(mainNode);
  return keys[0];
};

oj.OraI18nUtils._toBoolean = function (value) {
  if (typeof value === 'string') {
    var s = value.toLowerCase().trim();
    switch (s) {
      case 'true':
      case '1':
        return true;
      case 'false':
      case '0':
        return false;
      default:
        return value;
    }
  }
  return value;
};
// Return a function getOption.
// The getOption function extracts the value of the property named
// property from the provided options object, converts it to the required type,
// checks whether it is one of a List of allowed values, and fills in a
// fallback value if necessary.
oj.OraI18nUtils.getGetOption = function (options, getOptionCaller) {
  if (options === undefined) {
    throw new Error('Internal ' + getOptionCaller +
        ' error. Default options missing.');
  }

  var getOption = function getOption(property, type, values, defaultValue) {
    if (options[property] !== undefined) {
      var value = options[property];
      switch (type) {
        case 'boolean':
          value = oj.OraI18nUtils._toBoolean(value);
          break;
        case 'string':
          value = String(value);
          break;
        case 'number':
          value = Number(value);
          break;
        default:
          throw new Error('Internal error. Wrong value type.');
      }
      if (values !== undefined && values.indexOf(value) === -1) {
        var expectedValues = [];
        for (var i = 0; i < values.length; i++) {
          expectedValues.push(values[i]);
        }
        var msg = "The value '" + options[property] +
            "' is out of range for '" + getOptionCaller +
            "' options property '" + property + "'. Valid values: " +
            expectedValues;
        var rangeError = new RangeError(msg);
        var errorInfo = {
          errorCode: 'optionOutOfRange',
          parameterMap: {
            propertyName: property,
            propertyValue: options[property],
            propertyValueValid: expectedValues,
            caller: getOptionCaller
          }
        };
        rangeError.errorInfo = errorInfo;
        throw rangeError;
      }

      return value;
    }
    return defaultValue;
  };

  return getOption;
};

/**
 * matches a string to a reference string and returns the start and end indexes
 * of the match in the referensed string. The locale and options arguments let
 * applications specify the language whose sort order should be used and customize
 * the behavior of the function.
 *
 * @param {string} str the reference string
 * @param {string} pat The string against which the reference string is compared
 * @param {string} locale a BCP 47 language tag
 * @param {Object=} options Optional. An object with the following property:
 * sensitivity:
 *   Which differences in the strings should lead to non-zero result values. Possible values are:
 *   "base": Only strings that differ in base letters compare as unequal. Examples: a ? b,  a = A.
 *   "accent": Only strings that differ in base letters or accents and other diacritic marks compare as unequal. Examples: a ? b, , a = A.
 *   "case": Only strings that differ in base letters or case compare as unequal. Examples: a ? b, a ? A.
 *   "variant": Strings that differ in base letters, accents and other diacritic marks, or case compare as unequal.
 *   The default is base.
 * @return {Array|null} an array containing the start and end indexes of the match or null if there is no match.
 */
oj.OraI18nUtils.matchString = function (str, pat, locale, options) {
  if (options === undefined) {
    // eslint-disable-next-line no-param-reassign
    options = { sensitivity: 'base', usage: 'sort' };
  }
  var getOption = oj.OraI18nUtils.getGetOption(options, 'oj.OraI18nUtils.matchString');
  // eslint-disable-next-line no-param-reassign
  options.usage = getOption('usage', 'string', ['sort', 'search'], 'sort');
  // eslint-disable-next-line no-param-reassign
  options.sensitivity = getOption('sensitivity', 'string',
                                  ['base', 'accent', 'case', 'variant'], 'base');
  var len = str.length;
  var patLen = pat.length - 1;
  for (var i = 0; i < len; i++) {
    for (var j = 0; j < 3; j++) {
      var len2 = len - i;
      len2 = Math.min(len2, patLen + j);
      var str2 = str.substr(i, len2);
      var res = str2.localeCompare(pat, locale, options);
      if (res === 0) {
        var end = i + (len2 - 1);
        var ret = [i, end];
        return ret;
      }
    }
  }
  return null;
};

var _DEFAULT_TIME_PORTION = 'T00:00:00.000';
var _DATE_TIME_KEYS = {
  fullYear: { pos: 0, pad: 4 },
  month: { pos: 1, pad: 2 },
  date: { pos: 2, pad: 2 },
  hours: { pos: 3, pad: 2 },
  minutes: { pos: 4, pad: 2 },
  seconds: { pos: 5, pad: 2 },
  milliseconds: { pos: 6, pad: 3 },
  timeZone: { pos: 7 }
};

/**
 * Parses the isoString and returns a JavaScript Date object
 *
 * @param {string} isoString isoString to parse and to return Date of
 * @return {Date} the parsed JavaScript Date Object
 */
oj.OraI18nUtils.isoToDate = function (isoString) {
  // note new Date w/ isoString in IE fails so need to use parsing from momentjs support
  return new Date(this._normalizeIsoString(isoString));
};

/**
 * Will return an updated toIsoString using the timePortion from the fromIsoString or from the default
 * oj.OraI18nUtils.DEFAULT_TIME_PORTION
 *
 * @private
 * @param {string} fromIsoString isoString that may not be a complete isoString
 * @param {string} toIsoString isoString that may not be a complete isoString
 * @returns {string} modified toIsoString with original date portion and the time portion from the fromIsoString
 * @since 1.1
 */
oj.OraI18nUtils._copyTimeOver = function (fromIsoString, toIsoString) {
  if (!fromIsoString || !toIsoString) {
    throw new Error('Provided invalid arguments');
  }

  // need to only normalize toIsoString, since copying only time from fromIsoString
  var normalizedToIsoString = this._normalizeIsoString(toIsoString);

  var fromTimeIndex = fromIsoString.indexOf('T');
  var toTimeIndex = normalizedToIsoString.indexOf('T');
  var toDatePortion = normalizedToIsoString.substring(0, toTimeIndex);
  var fromTimePortion = (fromTimeIndex !== -1 ?
                         fromIsoString.substring(fromTimeIndex) :
                         _DEFAULT_TIME_PORTION);

  return toDatePortion + fromTimePortion;
};

/**
 * Clears the time portion of the isoString
 *
 * @private
 * @param {string} isoString isoString that may not be a complete isoString
 * @returns {string} an updated isoString
 * @since 1.1
 */
oj.OraI18nUtils._clearTime = function (isoString) {
  return this._dateTime(isoString, { hours: 0, minutes: 0, seconds: 0, milliseconds: 0 });
};

/**
 * Will accept an isoString and perform a get operation or a set operation depending on whether param is an Array
 * or a JSON
 *
 * The keys for the get and set operation are defined in _DATE_TIME_KEYS.
 *
 * Note the handling of month starting with 0 in Date object and being 1 based in isoString will be handled by the function
 * with the usage of doParseValue. Meaning when you doParseValue and you are getting the value it will automatically
 * decrement the value and when you are setting the param it will check if the value is of number and if so will
 * increment it.
 *
 * @private
 * @param {string} isoString isoString that may not be a complete isoString
 * @param {Array|Object} actionParam if an Array will be a get operation, if a JSON will be a set operation
 * @param {boolean=} doParseValue whether one should parseInt the value during the get request
 * @returns {Object|string} an Object when a get operation and a string when a set operation
 * @since 1.1
 */
oj.OraI18nUtils._dateTime = function (isoString, actionParam, doParseValue) {
  if (!isoString || !actionParam) {
    throw new Error('Invalid argument invocation');
  }

  var pos;
  var value;
  var retVal = null;
  var dateTimeKeys = _DATE_TIME_KEYS;
  var oraUtilsPadZero = this.padZeros;
  var isoStringNormalized = this._normalizeIsoString(isoString); // note intentionally normalizing
  var captured = (/(\d{4})-(\d{2})-(\d{2})T(\d{2}):(\d{2}):?(\d{2})?\.?(\d{3})?(.*)?/
                  .exec(isoStringNormalized));

  if (!captured) {
    throw new Error('Unable to capture anything');
  }

  captured = captured.slice(1);

  if (Array.isArray(actionParam)) {
    retVal = {};

    // means an array so perform a get operation
    for (var i = 0, len = actionParam.length; i < len; i++) {
      var key = actionParam[i];

      if (key in dateTimeKeys) {
        pos = dateTimeKeys[key].pos;
        value = captured[pos];

        if (doParseValue && key === 'timeZone') {
          throw new Error('Dude you tried to ask timezone to be parsed');
        }

        if (doParseValue) {
          var parsed = parseInt(value, 10);
          retVal[key] = pos === 1 ? parsed - 1 : parsed; // since month is 0 based, though about having a callback but month only special
        } else {
          retVal[key] = value;
        }
      }
    }
  } else if ($.isPlainObject(actionParam)) {
    var objKeys = Object.keys(actionParam);
    for (var k = 0; k < objKeys.length; k++) {
      var objKey = objKeys[k];
      var dtKey = dateTimeKeys[objKey];
      pos = dtKey.pos;
      value = actionParam[objKey];

      // special case for month again, 0 based so check if number and if so increment it
      if (pos === 1 && typeof value === 'number') {
        value += 1;
      }
      captured[pos] = dtKey.pad ? oraUtilsPadZero(value, dtKey.pad) : value;
    }
    // "2015-02-02T21:12:30.255Z"
    retVal = captured[0] + '-' + captured[1] + '-' + captured[2] + 'T' + captured[3] + ':' + captured[4] + ':' + captured[5] +
        (captured.length > 6 && captured[6] ? ('.' + captured[6] + (captured.length === 8 && captured[7] ? captured[7] : '')) : '');
  }

  return retVal;
};

/**
 * So the problem is Jet uses incomplete isoString which causes issues in different browsers.
 *
 * For instance for a new Date().toISOString() => 2015-02-02T18:00:37.007Z
 * ojInputDate stores 2015-02-02
 * ojInputTime stores T18:00:37.007Z
 *
 * yet constructing new Date(val) on above causes different results or errors in different browsers, so
 * this function is to normalize them. Note it is assumed that the point is creating the Date object from the
 * normalized isoString. Meaning if both contain only the time portion today's date will appended to it.
 *
 * Here are the use cases
 *
 * @private
 * @param {string} isoString isoString that may not be a complete isoString
 * @returns {string} a normalized isoString
 * @since 1.1
 */
oj.OraI18nUtils._normalizeIsoString = function (isoString) {
  if (!isoString) {
    throw new Error('Provided invalid arguments');
  }

  var checkTime = function (timeValue) {
    var splitted = timeValue.split(':');
    if (splitted.length > 1) {
      return timeValue;
    }
    // need at least hour + minute for proper parsing on browser except IE
    return timeValue + ':00';
  };

  var todayIsoString = new Date().toISOString();
  var todayDatePortion = todayIsoString.substring(0, todayIsoString.indexOf('T'));
  var timeIndex = isoString.indexOf('T');
  var datePortion = timeIndex === -1 ? isoString : isoString.substring(0, timeIndex);
  var timePortion = (timeIndex !== -1 ?
                     checkTime(isoString.substring(timeIndex)) :
                     _DEFAULT_TIME_PORTION);

  datePortion = datePortion || todayDatePortion;

  return datePortion + timePortion;
};

/**
 * formats a parameterized string
 *
 * @param {string} str string to be formatted. EX: "{0} bytes"
 * @param {Array} params array of parameters to be substituted in the string
 * @returns {string} The formatted string
 * @since 4.0.0
 */
oj.OraI18nUtils.formatString = function (str, params) {
  var len = params.length;
  var result = str;

  for (var i = 0; i < len; i++) {
    var token = '{' + i + '}';
    result = result.replace(token, params[i]);
  }
  return result;
};

/**
 * Returns a person's initials
 *
 * @param {string=} firstName first name
 * @param {string=} lastName last name or surname
 * @returns {string|undefined} uppercase concatenation of first letter of first name and first letter
 * of last name.
 * There are the following special cases:
 * - If the name is Arabic characters, it returns empty string.
 * - If the name is Hindi characters, it returns the first letter of the first name.
 * - If the name is Thai characters, it returns the first letter of the first name.
 * - If the name is Korean characters, it returns the first name.
 * - If the name is Japanese or Chinese characters, it returns the last name.
 *
 * @since 4.0.0
 */
oj.OraI18nUtils.getInitials = function (firstName, lastName) {
  // We assume the names are valid. We test the first character only.
  var c = 0;
  var c1 = 0;
  var u;
  if (firstName !== undefined && firstName.length > 0) {
    c = firstName.charCodeAt(0);
  }

  // Arabic characters. Return empty string
  if (c >= 0x0600 && c <= 0x06FF) {
    return '';
  }

  // Hindi characters. Return first letter of the first name
  if (c >= 0x0900 && c <= 0x097F) {
    return firstName.charAt(0);
  }

  // Thai characters. Return first letter of the first name
  if (c >= 0x0E00 && c <= 0x0E7F) {
    return firstName.charAt(0);
  }

  // Korean characters. Return first name
  if ((c >= 0x1100 && c <= 0x11FF) || (c >= 0x3130 && c <= 0x318F) ||
      (c >= 0xA960 && c <= 0xA97F) || (c >= 0xAC00 && c <= 0xD7FF)) {
    return firstName;
  }

  // Japanese and Chinese characters. Return last name
  if (lastName !== undefined && lastName.length > 0) {
    c = lastName.charCodeAt(0);
  }
  if ((c >= 0x2E80 && c <= 0x2FDF) || (c >= 0x3000 && c <= 0x312F) ||
      (c >= 0x3190 && c <= 0x31FF) || (c >= 0x3300 && c <= 0x4DBF) ||
      (c >= 0x4E00 && c <= 0x9FFF) || (c >= 0xF900 && c <= 0xFAFF)) {
    return lastName;
  }

  // Handle surrogate characters for Japanese and Chinese characters.
  if (c >= 0xD800 && c <= 0xDBFF) {
    if (lastName.length < 2) {
      return '';
    }
    c1 = lastName.charCodeAt(1);
    // c1 must be in DC00-DFFF range
    if (c1 < 0xDC00 || c1 > 0xDFFF) {
      return '';
    }
    // Convert high and low surrogates into unicode scalar.
    u = ((c - 0xD800) * 0x400) + (c1 - 0xDC00) + 0x10000;
    // test the blocks
    if ((u >= 0x1B000 && u <= 0x1B0FF) || (u >= 0x1F200 && u <= 0x1F2FF) ||
        (u >= 0x20000 && u <= 0x2A6DF) || (u >= 0x2A700 && u <= 0x2B73F) ||
        (u >= 0x2B740 && u <= 0x2B81F) || (u >= 0x2B820 && u <= 0x2CEAF) ||
        (u >= 0x2F800 && u <= 0x2FA1F)) {
      return lastName;
    }
  }

  // return default
  c = '';
  c1 = '';
  if (firstName !== undefined && firstName.length > 0) {
    c = firstName.charAt(0).toUpperCase();
  }
  if (lastName !== undefined && lastName.length > 0) {
    c1 = lastName.charAt(0).toUpperCase();
  }
  return c + c1;
};


/**
 * Copyright (c) 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */

/**
 * Used to register and retrieve converterFactories and validationFactories.
 * There is no need to call the constructor.
 * @see oj.Validation.converterFactory
 * @see oj.Validation.validatorFactory
 * @see oj.ConverterFactory
 * @see oj.ValidatorFactory
 * @name oj.Validation
 * @namespace
 * @hideconstructor
 * @export
 * @since 0.6
 *
 */
oj.Validation = {};

/**
 * Internal properties to hold all factory provider callbacks or instances by name
 * @private
 */
oj.Validation._converterFactories = {}; oj.Validation._validatorFactories = {};

/**
 * Internal properties to hold the default factory instances.
 * @private
 */
oj.Validation._defaultConverterFactories = {}; oj.Validation._defaultValidatorFactories = {};

/**
 * Internal property that identifies the type that is the contract for conveters and validators.
 * @private
 */
oj.Validation._CONTRACTS = {
  converter: { name: 'oj.ConverterFactory', type: oj.ConverterFactory },
  validator: { name: 'oj.ValidatorFactory', type: oj.ValidatorFactory }
};

/**
 * Method to register and retrieve converter factory instances by name.
 * When passed only the name, an existing factory (registered for the name) is returned. Callers can
 * expect to get back the default 'number', 'datetime', or 'color' converters.
 * When passed two arguments, a new factory for the name is registered. If the name already exists
 * the new instance replaces the old one.
 *
 * @param {string} type a case insensitive name of the converter factory. e.g., 'number' (or
 * oj.ConverterFactory.CONVERTER_TYPE_NUMBER), 'datetime' (or oj.ConverterFactory.CONVERTER_TYPE_DATETIME),
 * or 'color' (or oj.ConverterFactory.CONVERTER_TYPE_COLOR).
 * @param {Object=} instance the instance of the factory that implements the contract for
 * oj.ConverterFactory.
 *
 * @export
 * @ojsignature {target: "Type",
 *                value: "<CF extends oj.ConverterFactory<any>>(type:'number'|'color'|'datetime'|string, instance?:CF): CF|null"}
 * @see oj.ConverterFactory
 * @example <caption>Initialize a JET datetime converter instance using options
 * and use it in a knockout-binding and bind it to the
 * converter attribute on the JET form element.</caption>
 * ----- Javascript ViewModel----
 * self.converter = ko.observable(
    oj.Validation.converterFactory(oj.ConverterFactory.CONVERTER_TYPE_DATETIME).
      createConverter({formatType: 'date', dateFormat: 'full'}));
 * ----- HTML -----
 * &lt;oj-input-date value="{{endDate}}" converter="[[converter]]">&lt/oj-input-date>
 * @example <caption>Create your own converter, register it, and use it when displaying information
 * on the page to the user.</caption>
 * ----- Javascript -----
 * RelativeDateTimeConverterFactory = (function () {
 *  function _createRelativeDateTimeConverter(options)
 *  {
 *    // this is a custom converter that formats the datetime into "Today" or
 *    // "Tomorrow", or "This Week"/"Next Week", etc. See the Converter API or
 *    // the Custom Converter demos for how to create a custom converter.
 *    return new RelativeDateTimeConverter(options);
 *  }
 *  return {
 *    'createConverter': function (options) {
 *      return _createRelativeDateTimeConverter(options);
 *    }
 *  };
 *  }());
 * oj.Validation.converterFactory("relativeDate", RelativeDateTimeConverterFactory);
 *  // Use our custom relativeDate converter.
 *  // In this demo, we want to see in the Schedule For column the words
 *  // Today or Tomorrow so we set the relativeField option's value
 *  // to 'day'. If we want to see This Week, we'd set it to 'week', etc.
 *  var rdConverter =  oj.Validation.converterFactory("relativeDate")
 *  .createConverter({relativeField: 'day', year: "numeric", month: "numeric", day: "numeric"});
 *  ...
 *  // Our custom converter's format function returns an object with 'value' and 'title'.
 *  // We put the 'value' in innerHTML so the user can read it. E.g., Today or Tomorrow.
 *  // And we put the actual date in the title. The user can read it when they hover over
 *  // the word Today or Tomorrow.
 *  content = rdConverter.format(context.row.ScheduleFor);
 *  span.setAttribute('title', content.title);
 *  span.innerHTML = content.value;
 * ...
 * @example <caption>On your JET form component that has a 'converter' attribute,
 * set the 'converter' attribute using "type" of a registered converter. This
 * could be a JET Converter or a oj.Converter
 * (or Object that duck-types it) that you wrote. (it must implement parse and format
 * since our JET form components call format and parse.</caption>
 * &lt;oj-input-date value="{{date}}"
 * converter= '{"type":"datetime",
 * "options": {"year": "numeric", "month": "long", "day": "numeric"}}'> &lt;/oj-input-date>
 */
oj.Validation.converterFactory = function (type, instance) {
  var retValue;
  if (type && !instance) {
    // getter
    retValue = oj.Validation._getFactory(type, oj.Validation._converterFactories);
  } else if (type && instance) {
    // setter
    retValue = oj.Validation._registerFactory(type,
                                              instance,
                                              oj.Validation._converterFactories,
                                              oj.Validation._CONTRACTS.converter);
  }

  return retValue;
};

/**
 * Method to register and retrieve validator factory instances by name.
 * When passed only the name, an existing factory (registered for the name) is returned.
 * When passed two arguments, a new factory for the name is registered. If the name already exists
 * the new instance replaces the old one.
 *
 * @param {string} type a case insensitive name of the validator factory.
 * @param {Object=} instance the instance of the factory that implements the contract for
 * oj.ValidatorFactory.
 * @example <caption>Initialize a JET validator instance using options
 * and use it in a knockout-binding and bind it to the
 * validators attribute on the JET form element.</caption>
 * ----- Javascript ViewModel----
 * var options = {min: 5, max: 10};
 * var validator =
 * oj.Validation.validatorFactory(
 * oj.ValidatorFactory.VALIDATOR_TYPE_LENGTH).createValidator(options);
 * self.validator = ko.observable([validator]);
 * ----- HTML -----
 * &lt;oj-input-text validators="[[validator]]">&lt/oj-input-text>
 * @example <caption>An easier usage is to return an Object since the validators option
 * takes an array where each item is either an instance that duck types oj.Validator,
 * or is an Object literal containing type and options.</caption>
 * ----- Javascript ViewModel -----
 * self.validators = ko.computed(function() {
 * return [{type: 'numberRange', options: { min: 10000.05, max: 25000.95,
 * hint: {inRange: 'Enter a value between {min} and {max}.'}}}];
 * });
 * ----- HTML -----
 * &lt;oj-input-text id="numberRange1" value="{{numberValue1}}" validators="[[validators]]">
 * &lt;/oj-input-text>
 * @example <caption>Get a registered validator with your options, and calls its validate method
 * on your own value.</caption>
 * var reqValOptions = {
 *   'hint': reqTrans['hint'] || null,
 *   'label': this._getLabelText(),
 *   'messageSummary': reqTrans['messageSummary'] || null,
 *   'messageDetail': reqTrans['messageDetail'] || null
 * };
 * var vf = oj.Validation.validatorFactory(oj.ValidatorFactory.VALIDATOR_TYPE_REQUIRED);
 * var requiredValidator = vf.createValidator(reqValOptions);
 * var isValid = validator.validate(value);
 * @example <caption>Create and register your own factory</caption>
 * MyOwnNumberRangeValidatorFactory = (function () {
 *   function _createNumberRangeValidator(options) {
 *     return new MyOwnNumberRangeValidator(options);
 *   }
 *
 *   return {
 *     'createValidator': function(options) {
 *       return _createNumberRangeValidator(options);
 *   }
 *   };
 * }());
 * oj.Validation.validatorFactory(oj.ValidatorFactory.VALIDATOR_TYPE_NUMBERRANGE, MyOwnNumberRangeValidatorFactory);
 * @export
 * @since 0.6
 * @see oj.ValidatorFactory
 * @ojsignature {target: "Type",
 *   value: "<VF extends oj.ValidatorFactory<any>>(type:'required'|'regexp'|'numberRange'|'length'|'dateTimeRange'|'dateRestriction'|string, instance?:VF): VF|null"}
 */
oj.Validation.validatorFactory = function (type, instance) {
  var retValue;
  if (type && !instance) {
    // getter
    retValue = oj.Validation._getFactory(type, oj.Validation._validatorFactories);
  } else if (type && instance) {
    // setter
    retValue = oj.Validation._registerFactory(type,
                                              instance,
                                              oj.Validation._validatorFactories,
                                              oj.Validation._CONTRACTS.validator);
  }

  return retValue;
};

/**
 * Returns the default converter factory instances for the supported types as defined by the
 * oj.ConverterFactory. Instead of calling this method, it should be
 * sufficient to use oj.Validation.converterFactory.
 *
 * @param {string} type The default converter factory for the type. Supported types are 'number' and
 * 'datetime'
 * @return {Object|null} an instance of oj.ConverterFactory or null if an unknown type is requested.
 *
 * @export
 * @ojtsignore
 * @ojsignature {target:"Type",
 *    value: "<V>(type: 'number'|'color'|'datetime'): oj.ConverterFactory<V>|null"}
 * @see oj.ConverterFactory
 * @see oj.Validation.converterFactory
 *
 */
oj.Validation.getDefaultConverterFactory = function (type) {
  return oj.Validation._getFactory(type, oj.Validation._defaultConverterFactories);
};

/**
 * Returns the default validator factory instance for the requested types as defined by the
 * oj.ValidatorFactory.  Instead of calling this method, it should be
 * sufficient to use oj.Validation.validatorFactory.
 *
 * @param {string} type The default converter factory for the type. Supported types are 'number' and
 * 'datetime'
 * @return {Object|null} an instance of oj.ConverterFactory or null if an unknown type is requested.
 *
 * @export
 * @ojtsignore
 * @ojsignature {target:"Type",
 *    value: "<V>(type: 'required'|'regexp'|'numberRange'|'length'|'dateTimeRange'|'dateRestriction'): oj.ValidatorFactory<V>|null"}
 * @see oj.ValidatorFactory
 * @see oj.Validation.validatorFactory
 */
oj.Validation.getDefaultValidatorFactory = function (type) {
  return oj.Validation._getFactory(type, oj.Validation._defaultValidatorFactories);
};

// PACKAGE PRIVATE METHODS
/**
 * Called only by internal jet converter factory implementations.
 *
 * @param {string} name
 * @param {Object} instance
 * @private
 */
oj.Validation.__registerDefaultConverterFactory = function (name, instance) {
  // save to both factories
  var contractDef = oj.Validation._CONTRACTS.converter;
  oj.Validation._registerFactory(name, instance,
                                 oj.Validation._defaultConverterFactories,
                                 contractDef);
  oj.Validation._registerFactory(name, instance,
                                 oj.Validation._converterFactories,
                                 contractDef);
};

/**
 * Called only by internal jet validator factory implementations.
 *
 * @param {string} name of the validator factory
 * @param {Object} instance of the validator factory that creates instances of the validator
 * @private
 */
oj.Validation.__registerDefaultValidatorFactory = function (name, instance) {
  // save to both factories
  var contractDef = oj.Validation._CONTRACTS.validator;
  oj.Validation._registerFactory(name, instance,
                                 oj.Validation._defaultValidatorFactories,
                                 contractDef);
  oj.Validation._registerFactory(name, instance,
                                 oj.Validation._validatorFactories,
                                 contractDef);
};


/**
 * Checks that the instance implements the interface type. If it doesn't it throws an error.
 * @param {Object} instance
 * @param {Object} type
 * @param {string} typeName
 * @throws {Error} if instance does not implement the methods defined on type.
 * @private
 */
oj.Validation._doImplementsCheck = function (instance, type, typeName) {
  if (type) {
    // Check that instance duck types providerType
    if (!oj.Validation._quacksLike(instance, type)) {
      throw new Error('Factory instance does not implement the methods expected by the factory of type ' + typeName);
    }
  }
};

/**
 * Retrieves the converter factory by name from the provided factories.
 * @private
 */
oj.Validation._getFactory = function (name, factories) {
  oj.Assert.assertString(name);
  var cachedInstance = null;

  var lowerName = name.toLowerCase();

  // getter called to retrieve the factory instance
  var providerProps = factories[lowerName] || {};
  cachedInstance = providerProps.instance || null;
  // TODO: log a warning that name is null
  return cachedInstance;
};

/**
 * Tests whether an object 'quacks like a duck'. Returns `true` if the thingie has all of the
 * methods of the second, parameter 'duck'. Returns `false` otherwise.
 *
 * @param {Object} thingie the object to test.
 * @param {Object} duck The archetypal object, or 'duck', that the test is against.
 * @private
*/
oj.Validation._quacksLike = function (thingie, duck) {
  var valid = true;

  oj.Assert.assertObject(thingie);
  oj.Assert.assertObject(duck);

  var properties = Object.keys(duck);
  for (var i = 0; i < properties.length; i++) {
    var property = properties[i];
    // Ensure that thingie defines the same functions as duck. We don't care about other properties
    if (typeof duck[property] === 'function' &&
        (!thingie[property] || typeof thingie[property] !== 'function')) {
      valid = false;
      break;
    }
  }

  return valid;
};

/**
 * Registers the factory instance by the name, storing it into the factories object, after ensuring
 * that the instance duck types the specified contract.
 *
 * @private
 */
oj.Validation._registerFactory = function (name, instance, factories, contractDef) {
  oj.Assert.assertString(name);
  oj.Assert.assertObject(instance);

  // set new provider factory function clearing out the previously stored instance
  var props = {};
  props.instance = instance;
  oj.Validation._doImplementsCheck(instance, contractDef.type, contractDef.name);

  // Save to default and public factories
  // eslint-disable-next-line no-param-reassign
  factories[name.toLowerCase()] = props;
};

/**
 * Contract for a ConverterFactory that provides a factory method to create a converter instance for
 * the requested type. Factories handle the details of object creation.
 * It allows the consumers of the factory to create specific converters
 * without knowing the internals of the converter creation.
 * <p>
 * JET provides three factory implementations for number and datetime
 * and color types that implement this contract.
 * </p>
 * <p>
 * Customers can register custom converter factories for the supported
 * types or create and register factories for new types altogether.
 * </p>
 * @example <caption>Get the ConverterFactory for 'datetime', and create
 * a JET dateTime converter with specific options.
 * See {@link oj.DateTimeConverterFactory} for what to set for options
 * for the DateTime Converter. </caption>
 * var dateTimeCvtr = oj.Validation.converterFactory(oj.ConverterFactory.CONVERTER_TYPE_DATETIME);
 * var dateOptions = {day: 'numeric', month: 'numeric'};
 * var dayMonthConverter = dateTimeCvtr.createConverter(dateOptions);
 * @example <caption>Create a custom ConverterFactory with a new type and register that ConverterFactory.</caption>
 * // Define a new ConverterFactory for relative datetimes, like Today, Yesterday, Tomorrow
 * RelativeDateTimeConverterFactory = (function () {
 *   function _createRelativeDateTimeConverter(options)
 *   {
 *     // this is a custom converter
 *     // See the Custom Converter JET demo for details on how to create
 *     // your own Converter. Or see the Converter jsdoc.
 *     return new RelativeDateTimeConverter(options);
 *   }
 *   return {
 *     'createConverter': function (options) {
 *       return _createRelativeDateTimeConverter(options);
 *     }
 *   };
 * }());
 *
 * // Register the custom factory with the new "relativeDate" type
 * oj.Validation.converterFactory("relativeDate", RelativeDateTimeConverterFactory);
 *
 * // Get the custom factory using the new type.
 * var rdConverter =  oj.Validation.converterFactory("relativeDate")
 * .createConverter({relativeField: 'day', year: "numeric", month: "numeric", day: "numeric"});
 * @name oj.ConverterFactory
 * @abstract
 * @class
 * @export
 * @ojtsnoexport
 * @ojtsexportastype
 * @since 0.6
 * @ojsignature {target: "Type", value: "class ConverterFactory<V>", genericParameters: [{"name": "V", "description": "Type of value to be converted"}]}
 * @see oj.Validation
 * @see oj.NumberConverterFactory
 * @see oj.DateTimeConverterFactory
 * @see oj.ColorConverterFactory
 */
oj.ConverterFactory =
{
  /**
   * Default type for a factory used to create number converters. This type is passed to the
   * [Validation.converterFactory]{@link oj.Validation#converterFactory} method to retrieve the
   * number converter factory of type {@link oj.NumberConverterFactory}.
   * @example <caption>Create a JET number converter with options</caption>
   * var convFactory = oj.Validation.converterFactory(oj.ConverterFactory.CONVERTER_TYPE_NUMBER);
   * var converter  =  convFactory.createConverter({
   *  style: 'currency',
   *  currency: 'USD',
   *  currencyDisplay: 'symbol',
   *  pattern: '¤ ##,##0.00'});
   * @expose
   * @const
   * @member
   * @type {string}
   */
  CONVERTER_TYPE_NUMBER: 'number',

  /**
   * Default type for a factory used to create datetime converters. This type is passed to the
   * [Validation.converterFactory]{@link oj.Validation#converterFactory} method to retrieve the
   * datetime converter factory of type {@link oj.DateTimeConverterFactory}.
   * @example <caption>Create a JET dateTime converter with options</caption>
   * var dateTimeCvtr = oj.Validation.converterFactory(oj.ConverterFactory.CONVERTER_TYPE_DATETIME);
   * var dateOptions = {day: 'numeric', month: 'numeric'};
   * var dayMonthConverter = dateTimeCvtr.createConverter(dateOptions);
   *
   * @expose
   * @const
   * @member
   * @type {string}
   */

  CONVERTER_TYPE_DATETIME: 'datetime',

  /**
   * Default type for a factory used to create color converters. This type is passed to the
   * [Validation.converterFactory]{@link oj.Validation#converterFactory} method to retrieve the
   * color converter factory of type {@link oj.ColorConverterFactory}.
   * @example <caption>Create a JET color converter with options</caption>
   * this._convFactory = oj.Validation.converterFactory(oj.ConverterFactory.CONVERTER_TYPE_COLOR);
   * this._convHex  =  this._convFactory.createConverter({"format": "hex"})
   * @expose
   * @const
   * @member
   * @type {string}
   */
  CONVERTER_TYPE_COLOR: 'color',

  /**
   * Creates an immutable converter instance of the type the factory implements.
   *
   * @param {(Object|null)} options an object literal containing properties required by the converter
   * for its initialization. The properties provided in the options is implementation specific.
   *
   * @return {oj.Converter} a converter instance.
   * @ojsignature { target: "Type",
   *                value: "oj.Converter<V>",
   *                for: "returns"}
   * @instance
   * @throws {TypeError} if an unrecognized type was provided
   * @expose
   *
   * @example <caption>Create a JET dateTime converter with options</caption>
   * var dateTimeCvtr = oj.Validation.converterFactory(oj.ConverterFactory.CONVERTER_TYPE_DATETIME);
   * var dateOptions = {day: 'numeric', month: 'numeric'};
   * var dayMonthConverter = dateTimeCvtr.createConverter(dateOptions);
   * @example <caption>Create your own ConverterFactory and Converter, register the Converter on your
   * ConverterFactory, and use it when displaying relative date information
   * on the page to the user.</caption>
   * ----- Javascript -----
   * // Define new ConverterFactory
   * RelativeDateTimeConverterFactory = (function () {
   *   function _createRelativeDateTimeConverter(options)
   *   {
   *     // this is a custom converter See the Converter API or
   *     // Custom Converter JET demo for details on
   *     // how to create a custom converter.
   *     return new RelativeDateTimeConverter(options);
   *   }
   *   return {
   *     'createConverter': function (options) {
   *       return _createRelativeDateTimeConverter(options);
   *     }
   *   };
   * }());
   * // Register the custom factory with the new type
   * oj.Validation.converterFactory("relativeDate", RelativeDateTimeConverterFactory);
   * // Get the custom factory using the new type.
   * var rdConverter =  oj.Validation.converterFactory("relativeDate")
   * .createConverter({relativeField: 'day', year: "numeric", month: "numeric", day: "numeric"});
   *  ...
   *  // Our custom converter's format function returns an object with 'value' and 'title'.
   *  // We put the 'value' in innerHTML so the user can read it. E.g., Today or Tomorrow.
   *  // And we put the actual date in the title. The user can read it when they hover over
   *  // the word Today or Tomorrow.
   *  content = rdConverter.format(context.row.ScheduleFor);
   *  span.setAttribute('title', content.title);
   *  span.innerHTML = content.value;
   * ...
   */
  // eslint-disable-next-line no-unused-vars
  createConverter: function (options) {}
};

/**
 * Contract for a ValidatorFactory that provides a factory method to create a validator instance for
 * the requested type. JET provides several factory implementations that implement this contract -
 * for example dateRestriction, dateTimeRange, numberRange, length, required, regexp.
 * <p> Customers can
 * register custom validator factories for the supported types or create and register factories for
 * new types altogether.
 * </p>
 * @example <caption>Create a JET regexp validator</caption>
 * var validatorFactory =
 *   oj.Validation.validatorFactory(oj.ValidatorFactory.VALIDATOR_TYPE_REGEXP);
 *
 * var options =
 *   {pattern: '[a-zA-Z0-9]{3,}',
 *   hint: 'enter at least 3 letters or numbers.',
 *   messageDetail: 'You must enter at least 3 letters or numbers.'}
 *
 * var validator = validatorFactory.createValidator(options);
 * @example <caption>Create and register your own ValidatorFactory</caption>
 * MyOwnNumberRangeValidatorFactory = (function () {
 *   function _createNumberRangeValidator(options) {
 *     // See Validator api or Custom Validator demos for how to create your own Validator
 *     return new MyOwnNumberRangeValidator(options);
 *   }
 *
 *   return {
 *     'createValidator': function(options) {
 *       return _createNumberRangeValidator(options);
 *   }
 *   };
 * }());
 *
 * oj.Validation.validatorFactory(oj.ValidatorFactory.VALIDATOR_TYPE_NUMBERRANGE, MyOwnNumberRangeValidatorFactory);
 * @name oj.ValidatorFactory
 * @abstract
 * @class
 * @ojsignature {target: "Type", value: "class ValidatorFactory<V>", genericParameters: [{"name": "V", "description": "Type of value to be validated"}]}
 * @export
 * @ojtsnoexport
 * @ojtsexportastype
 * @since 0.6
 * @see oj.Validation
 * @see oj.DateRestrictionValidatorFactory
 * @see oj.DateTimeRangeValidatorFactory
 * @see oj.LengthValidatorFactory
 * @see oj.NumberRangeValidatorFactory
 * @see oj.RegExpValidatorFactory
 * @see oj.RequiredValidatorFactory
 */
oj.ValidatorFactory =
{
  /**
   * Default type for a factory used to create required validators. This type is passed to the
   * [Validation.validatorFactory]{@link oj.Validation#validatorFactory} method to retrieve the
   * required validator factory of type {@link oj.RequiredValidatorFactory}.
   * @example <caption>Create a JET required validator</caption>
   * var rvf = oj.Validation.validatorFactory(oj.ValidatorFactory.VALIDATOR_TYPE_REQUIRED);
   * var options = {'hint' : 'a value is required for this field'};
   * var requiredValidator = rvf.createValidator(options);
   * @expose
   * @const
   * @member
   * @type {string}
   */
  VALIDATOR_TYPE_REQUIRED: 'required',

  /**
   * Default type for a factory used to create regExp validators. This type is passed to the
   * [Validation.validatorFactory]{@link oj.Validation#validatorFactory} method to retrieve the
   * regExp validator factory of type {@link oj.RegExpValidatorFactory}.
   * @example <caption>Create a JET regexp validator</caption>
   * var validatorFactory =
   * oj.Validation.validatorFactory(oj.ValidatorFactory.VALIDATOR_TYPE_REGEXP);
   * var options =
   * {pattern: '[a-zA-Z0-9]{3,}',
   * hint: 'enter at least 3 letters or numbers.',
   * messageDetail: 'You must enter at least 3 letters or numbers.'}
   * var validator = validatorFactory.createValidator(options);
   * @expose
   * @const
   * @member
   * @type {string}
   */
  VALIDATOR_TYPE_REGEXP: 'regexp',

  /**
   * Default type for a factory used to create numberRange validators. This type is passed to the
   * [Validation.validatorFactory]{@link oj.Validation#validatorFactory} method to retrieve the
   * numberRange validator factory of type {@link oj.NumberRangeValidatorFactory}.
   * @example <caption>Create a JET numberrange validator</caption>
   * var validatorFactory =
   * oj.Validation.validatorFactory(oj.ValidatorFactory.VALIDATOR_TYPE_NUMBERRANGE);
   * var options = {min: 10000.05, max: 25000.95,
   * hint: {inRange: 'Enter a value between {min} and {max}.'}};
   * var validator = validatorFactory.createValidator(options);
   * @expose
   * @const
   * @member
   * @type {string}
   */
  VALIDATOR_TYPE_NUMBERRANGE: 'numberRange',

  /**
   * Default type for a factory used to create length validators. This type is passed to the
   * [Validation.validatorFactory]{@link oj.Validation#validatorFactory} method to retrieve the
   * length validator factory of type {@link oj.LengthValidatorFactory}.
   * @example <caption>Create a JET length validator</caption>
   * var validatorFactory =
   * oj.Validation.validatorFactory(oj.ValidatorFactory.VALIDATOR_TYPE_LENGTH);
   * var options = {min: 5, max: 10};
   * var validator = validatorFactory.createValidator(options);
   * @expose
   * @const
   * @member
   * @type {string}
   */
  VALIDATOR_TYPE_LENGTH: 'length',

  /**
   * Default type for a factory used to create required validators. This type is passed to the
   * [Validation.validatorFactory]{@link oj.Validation#validatorFactory} method to retrieve the
   * dateTimeRange validator factory of type {@link oj.DateTimeRangeValidatorFactory}.
   * @example <caption>Create a JET datetime validator</caption>
   * var validator =
   * oj.Validation.validatorFactory(oj.ValidatorFactory.VALIDATOR_TYPE_DATETIMERANGE)
   * .createValidator({{max: oj.IntlConverterUtils.dateToLocalIso(new Date()),
   * min: oj.IntlConverterUtils.dateToLocalIso(new Date(2000, 00, 01)),
   * hint: {'inRange': 'Enter a date that falls in the current millennium.'}});
   * @expose
   * @const
   * @member
   * @type {string}
   */
  VALIDATOR_TYPE_DATETIMERANGE: 'dateTimeRange',

  /**
   * Default type for a factory used to create date restriction validators. This type is passed to
   * the [Validation.validatorFactory]{@link oj.Validation#validatorFactory} method to retrieve the
   * dateRestriction validator factory of type {@link oj.DateRestrictionValidatorFactory}.
   * @example <caption>Create a JET dateRestriction validator</caption>
   * var validatorFactory =
   * oj.Validation.validatorFactory(oj.ValidatorFactory.VALIDATOR_TYPE_DATERESTRICTION);
   * var options = {dayFormatter : self.aprilFoolsFormatter, // your own formatter code
   * message : {messageDayMetaData : 'You can\'t fool me! Try a different date.'}};
   * var validator = validatorFactory.createValidator(options);
   *
   * @expose
   * @const
   * @member
   * @type {string}
   */
  VALIDATOR_TYPE_DATERESTRICTION: 'dateRestriction',

  /**
   * Creates an immutable validator instance of the type the factory implements.
   * For the specific options parameters, see the Validator jsdoc for the validator you
   * are creating, e.g., oj.RequiredValidator if you are creating a required validator.
   *
   * @example <caption>create an instance of the required validator using the factory</caption>
   * var rvf = oj.Validation.validatorFactory(oj.ValidatorFactory.VALIDATOR_TYPE_REQUIRED);
   * var options = {'hint' : 'a value is required for this field'};
   * var requiredValidator = rvf.createValidator(options);
   * @param {(Object|null)} options an object literal containing properties required by the validator
   * for its initialization. The properties provided in the options is implementation specific.
   *
   * @return {Object} a validator instance.
   * @throws {TypeError} if an unrecognized type was provided
   * @ojsignature { target: "Type",
   *                value: "oj.Validator<V>",
   *                for: "returns"}
   * @instance
   * @expose
   * @see oj.DateRestrictionValidator
   * @see oj.DateTimeRangeValidator
   * @see oj.LengthValidator
   * @see oj.NumberRangeValidator
   * @see oj.RegExpValidator
   * @see oj.RequiredValidator
   */
  // eslint-disable-next-line no-unused-vars
  createValidator: function (options) {}

  /**
   * A ducktype for accessing validator created and/ registered by referring to {@link oj.Validation.validatorFactory}
   * @typedef {object} oj.Validation.RegisteredValidator
   * @property {string} type the name of the factory registered validator
   * @property {Object=} options options to the validator
   */

   /**
   * A ducktype for accessing converter created and/ registered by referring to {@link oj.Validation.converterFactory}
   * @typedef {object} oj.Validation.RegisteredConverter
   * @property {string} type the name of the factory registered converter
   * @property {Object=} options options to the converter
   */
};

/**
 * Copyright (c) 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */
/* global Message:false */
/**
 * Converter Contract
 */

/**
 * Constructs an immutable instance of Converter.
 *
 * @param {Object=} options an object literal used to provide an optional information to
 * initialize the converter.<p>
 * @export
 * @ojsignature {target: "Type", value: "class Converter<V>",
 *               genericParameters: [{"name": "V", "description": "Type of value to be converted. Parse will
 *                 convert string to this type and format will format this type to string"}]}
 * @abstract
 * @ojtsnoexport
 * @ojtsexportastype
 * @class
 * @since 0.6
 */
oj.Converter = function (options) {
  this.Init(options);
};

// Subclass from oj.Object
oj.Object.createSubclass(oj.Converter, oj.Object, 'oj.Converter');


/**
 * Initializes converter instance with the set options
 * @param {Object=} options an object literal used to provide an optional information to
 * initialize the converter.<p>
 * @export
 * @ignore
 */
oj.Converter.prototype.Init = function (options) {
  oj.Converter.superclass.Init.call(this);
  // should we make options truly immutable? non-configurable, non-enumerable, non-writable
  // Object.defineProperty(oj.Converter.prototype, "_options", {value: options});
  this._options = options;
};


/**
 * Returns a hint that describes the converter format expected.
 * @method getHint
 * @return {string|null} a hint describing the format the value is expected to be in.
 * @memberof oj.Converter
 * @ojsignature {target: "Type", value: "?(): string|null"}
 * @instance
 */

/**
 * Returns the options called with converter initialization.
 * @return {Object} an object of options.
 * @ojsignature {target: "Type", value: "?(): object"}
 * @export
 */
oj.Converter.prototype.getOptions = function () {
  return (this._options || {});
};


/**
 * Parses a String value using the options provided.
 * @method parse
 * @param {string} value to parse
 * @return {any} the parsed value.
 * @ojsignature [
 *                {target: "Type", value: "V|null", for: "returns"}
 *              ]
 * @throws {Error} if parsing fails
 * @memberof oj.Converter
 * @instance
 */

/**
 * Formats the value using the options provided.
 *
 * @param {any} value the value to be formatted for display
 * @return {(string|null)} the localized and formatted value suitable for display
 * @throws {Error} if formatting fails.
 * @method format
 * @ojsignature {target: "Type", value: "V", for: "value"}
 * @memberof oj.Converter
 * @instance
 */

/**
 * Returns an object literal with locale and formatting options computed during initialization of
 * the object. If options was not provided at the time of initialization, the properties will be
 * derived from the locale defaults.
 * @return {Object} an object of resolved options.
 * @ojsignature {target: "Type", value: "?(): object"}
 * @export
 */
oj.Converter.prototype.resolvedOptions = function () {
  var resolved = {};
  // returns a clone of this._options
  $.extend(resolved, this._options);

  return resolved;
};


// oj.ConverterError

/**
 * Constructs a ConverterError instance from a summary and detail
 *
 * @param {string} summary a localized String that provides a summary of the error
 * @param {string} detail a localized String that provides a detail of the error
 * @constructor
 * @ojtsnoexport
 * @ojtsexportastype
 * @since 0.6
 * @export
 */
oj.ConverterError = function (summary, detail) {
  var message = {
    summary: summary,
    detail: detail,
    severity: Message.SEVERITY_LEVEL.ERROR
  };
  this.Init(message);
};

oj.ConverterError.prototype = new Error();

/**
 * Initializes the instance.
 * @param {Object} message an Object that duck-types oj.Message which is an
 * Object with summary, detail, and severity properties.
 * @export
 * @ignore
 */
oj.ConverterError.prototype.Init = function (message) {
  var detail = message.detail;
  var summary = message.summary;
  this._message = message;

  // so browser can get to e.name and e.message
  this.name = 'Converter Error';
  this.message = detail || summary;
};

/**
 * Returns an Object that duck-types oj.Message which is an
 * Object with summary, detail, and severity properties.
 *
 * @return {Object} an Object that duck-types oj.Message which is an
 * Object with summary, detail, and severity properties.
 * @ojsignature {target: "Type", value: "oj.Message", for: "returns"}
 * @export
 */
oj.ConverterError.prototype.getMessage = function () {
  return this._message;
};

/**
 * Copyright (c) 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */
/* global Message:false */
/**
 * Validator Contract
 */

/**
 *  @example <caption>Create a Validator and implement its methods.</caption>
 *
 *  //Validator to ensure that the selected time is a multiple of 15 minute interval.
 *  //This converter takes in a converter option which will be used to format the hint/error
 *  //message shown to the user. It is mandatory to pass the converter option.
 *  var TimeIncrementValidator = function (options) {
 *    if(options && options.converter){
 *      this._converter = options.converter;
 *    }
 *  };
 *  //Need to be a subclass of oj.Validator
 *  oj.Object.createSubclass(TimeIncrementValidator, oj.Validator, "TimeIncrementValidator");
 *
 *  // Validates if the passed in value is a multiple of 15 minute interval.
 *  // Throws an error if the validation fails.
 *  TimeIncrementValidator.prototype.validate = function (value)
 *  {
 *    if (value)
 *    {
 *      var currentTime = oj.IntlConverterUtils.isoToLocalDate(value);
 *      var previousValidValue, nextValidValue, sampleMinutes;
 *      var minutes = currentTime.getMinutes();
 *      //Check if the minute is in increment of 15 by taking a modulo
 *      if ((minutes % 15) !== 0)
 *      {
 *        sampleMinute = Math.floor((minutes / 15))*15;
 *        currentTime.setMinutes(sampleMinute);
 *        previousValidValue = oj.IntlConverterUtils.dateToLocalIso(currentTime);
 *        sampleMinute = sampleMinute+ 15;
 *        if(sampleMinute >= 60){
 *          sampleMinute = 0;
 *          currentTime.setTime(currentTime.getTime() + (60*60*1000));
 *        }
 *        currentTime.setMinutes(sampleMinute);
 *        nextValidValue = oj.IntlConverterUtils.dateToLocalIso(currentTime);
 *        throw new oj.ValidatorError("Only multiples of 15 minute intervals are allowed.", "For example, " + this._converter.format(previousValidValue) +
 *                                     " or "+ this._converter.format(nextValidValue));
 *      }
 *    }
 *  };
 *
 *  //Generates a hint message with 4 different samples of valid values.
 *  TimeIncrementValidator.prototype.getHint = function ()
 *  {
 *    var currentTime = new Date();
 *    var hintMessage = "Only 15 minute intervals accepted, for example";
 *    //generate 4 sample values
 *    for (var i = 0; i < 4; i++) {
 *      currentTime.setMinutes(i * 15);
 *      hintMessage = hintMessage + ", " + this._converter.format(oj.IntlConverterUtils.dateToLocalIso(currentTime));
 *    }
 *    return hintMessage;
 *  };
 * @example <caption>Create a Validator and implement its methods. Bind it to the
 * JET form component which will call this 'validate' to validate the user's input.</caption>
 *  // Validator that ensures endDate is never less than start date
 *  self.endDateValidator = {
 *    'validate' : function(value)
 *    {
 *      var value =  oj.IntlConverterUtils.isoToLocalDate(value);
 *      var start = oj.IntlConverterUtils.isoToLocalDate(self.startDate());
 *      if (value && !(value.getFullYear() > start.getFullYear() ||
 *      {
 *        throw new oj.ValidatorError('End Date cannot be less than Start Date');
 *      }
 *    },
 *    'getHint' : function() { return "End Date needs to be greater than Start Date");}
 *  };
 *  -- HTML --
 *  &lt;oj-input-date id="nextday" value="{{endDate}}"
 *  validators="{{[weekendDateValidator, endDateValidator]}}">&lt;/oj-input-date>
 * @abstract
 * @class
 * @param {Object=} options An object which contains the options for the validator
 * @ojsignature {target: "Type", value: "class Validator<V>", genericParameters: [{"name": "V", "description": "Type of value to be validated"}]}
 * @ojtsnoexport
 * @ojtsexportastype
 * @export
 * @since 0.6
 *
 */
oj.Validator = function (options) {
  this.Init(options);
};

// Subclass from oj.Object
oj.Object.createSubclass(oj.Validator, oj.Object, 'oj.Validator');

/**
 * Initializes validator instance with the set options
 * @export
 * @ignore
 */
// eslint-disable-next-line no-unused-vars
oj.Validator.prototype.Init = function (options) {
  oj.Validator.superclass.Init.call(this);
};


/**
 * Validates the value.
 * The function typically returns if the validation passes and throws an error if it fails.
 *
 * @example <caption>Create a Validator and implement the validate method. Bind it to the
 * JET form component which will call this 'validate' to validate the user's input.</caption>
 *  // Validator that ensures endDate is never less than start date
 *  self.endDateValidator = {
 *    'validate' : function(value)
 *    {
 *      var value =  oj.IntlConverterUtils.isoToLocalDate(value);
 *      var start = oj.IntlConverterUtils.isoToLocalDate(self.startDate());
 *      if (value && !(value.getFullYear() > start.getFullYear() ||
 *      {
 *        throw new oj.ValidatorError('End Date cannot be less than Start Date');
 *      }
 *    },
 *    'getHint' : function() { return "End Date needs to be greater than Start Date");}
 *  };
 *  -- HTML --
 *  &lt;oj-input-date id="nextday" value="{{endDate}}"
 *  validators="{{[weekendDateValidator, endDateValidator]}}">&lt;/oj-input-date>
 * @param {any} value to be validated
 * @return {void}
 * @ojdeprecated {since: '6.2.0', target: "returnType", value: ["any"], description: 'This currently returns any type
 * if successful, like boolean, original value, or nothing.
 * In v8.0 it will return nothing if successful.'}
 * @throws {Error} if validation fails
 * @method validate
 * @ojsignature [{target: "Type",
 *                value: "V",
 *                for: "value"},
 *                {target: "Type",
 *                value: "void",
 *                for: "returns"}]
 * @memberof oj.Validator
 * @instance
 */


/**
 * Returns a hint that describes the validator rule.
 * @example <caption>Create a Validator and implement the getHint method. Bind it to the
 * JET form component which will show the hint.</caption>
 *  // Validator that ensures endDate is never less than start date
 *  self.endDateValidator = {
 *    'validate' : function(value)
 *    {
 *      ...
 *    },
 *    'getHint' : function() { return "End Date needs to be greater than Start Date");}
 *  };
 *  -- HTML --
 *  &lt;oj-input-date id="nextday" value="{{endDate}}"
 *  validators="{{[weekendDateValidator, endDateValidator]}}">&lt;/oj-input-date>
 * @returns {string|null} a hint string or null
 * @method getHint
 * @memberof oj.Validator
 * @ojsignature {target: "Type", value: "?(): string|null"}
 * @instance
 */

// ValidatorError

/**
 * Constructs a ValidatorError instance from a summary and detail
 *
 * @param {string} summary a localized String that provides a summary of the error
 * @param {string} detail a localized String that provides a detail of the error
 * @example <caption>throw new oj.ValidationError from custom validator's validate method</caption>
 *  // A custom validator whose validate method ensures that the value is not 'junk'.
 *  self.noJunkValidator = {
 *  'validate' : function (value)
 *  {
 *    value = value + "";
 *    if (value && value.toLowerCase()  === "junk")
 *    {
 *      throw new oj.ValidatorError("Invalid value", "You cannot enter a value that is 'junk'!!");
 *    }
 *    return;
 *    }
 *  };
 * @constructor
 * @ojtsnoexport
 * @ojtsexportastype
 * @ojtsimport {module: "ojmessaging", type: "AMD", importName: "Message"}
 * @since 0.6
 * @export
 */
oj.ValidatorError = function (summary, detail) {
  var message = {
    summary: summary,
    detail: detail,
    severity: Message.SEVERITY_LEVEL.ERROR };
  this.Init(message);
};

oj.ValidatorError.prototype = new Error();

/**
 * Initializes the instance.
 * @param {Object} message an Object that duck-types oj.Message which is an
 * Object with summary, detail, and severity keys. The severity is
 * oj.Message.SEVERITY_LEVEL['ERROR'].
 * @export
 * @ignore
 */
oj.ValidatorError.prototype.Init = function (message) {
  var detail = message.detail;
  var summary = message.summary;
  this._message = message;

  // so browser can get to e.name and e.message
  this.name = 'Validator Error';
  this.message = detail || summary;
};

/**
 * Returns an Object that duck-types oj.Message which is an
 * Object with summary, detail, and severity properties. The severity is
 * oj.Message.SEVERITY_LEVEL['ERROR']
 *
 * @example <caption>get the oj.ValidationError that was thrown and get the message</caption>
 * var ojmessage = valError.getMessage();
 * var severity = ojmessage['severity'] || oj.Message.SEVERITY_LEVEL['ERROR'];
 * var summary = ojmessage['summary'];
 * var detail = ojmessage['detail'];
 * @returns {Object} an Object that duck-types oj.Message which is an
 * Object with summary, detail, and severity properties.
 * The severity is
 * oj.Message.SEVERITY_LEVEL['ERROR']
 * @ojsignature {target: "Type", value: "oj.Message", for: "returns"}
 * @export
 */
oj.ValidatorError.prototype.getMessage = function () {
  return this._message;
};

/**
 * Copyright (c) 2008, 2013, Oracle and/or its affiliates.
 * All rights reserved.
 */
/* global Translations:false */
/**
 * Constructs a RegExpValidator that ensures the value matches the provided pattern.
 * In most use cases you will use the ValidatorFactory to get an instance of a
 * RegExpValidator instead of using the constructor.
 * @param {Object=} options an object literal used to provide the pattern, an optional hint and error
 * message.
 * @export
 * @constructor
 * @augments oj.Validator
 * @ojsignature [{target: "Type",
 *                value: "class RegExpValidator implements Validator<string|number>"},
 *               {target: "Type",
 *                value: "oj.RegExpValidator.ValidatorOptions",
 *                for: "options", jsdocOverride: true}
 *              ]
 * @since 0.6
 * @see oj.ValidatorFactory
 */
oj.RegExpValidator = function (options) {
  this.Init(options);
};

// Subclass from oj.Object or oj.Validator. It does not matter
oj.Object.createSubclass(oj.RegExpValidator, oj.Validator, 'oj.RegExpValidator');

// key to access required validator specific resources in the bundle
oj.RegExpValidator._BUNDLE_KEY_DETAIL = 'oj-validator.regExp.detail';
oj.RegExpValidator._BUNDLE_KEY_SUMMARY = 'oj-validator.regExp.summary';

/**
 * @typedef {object} oj.RegExpValidator.ValidatorOptions
 * @property {string=} pattern - a regexp pattern that the validator matches a value against.<p>
 * Example:<br/>
 * '\\d{10}'
 * @property {string=} hint - an optional hint text. There is no default hint provided by the
 * validator. It is generally not recommended to show the actual pattern in the hint as it might be
 * confusing to end-user, but if you do, you can use the {pattern} token.<p>
 * <p>The hint string is passed as the 'pattern' parameter to
 * [oj.Translations.html#applyParameters]{@link oj.Translations}. As stated in
 * that documentation, if you are using a reserved character, you need to escape it with
 * a dollar character ('$').
 * </p>
 * Tokens: <br/>
 * {pattern} - the pattern to enforce<p>
 * Example:<br/>
 * "value must meet this pattern {pattern}"
 * @property {string=} messageSummary - a custom error message summarizing the error when the
 * users input does not match the specified pattern. When not present, the default summary is the
 * resource defined with the key <code class="prettyprint">oj-validator.regExp.summary</code>.
 * It is generally not recommended to show the actual pattern in the message as it might be
 *  confusing to end-user. <p>
 * <p>The messageSummary string is passed as the 'pattern' parameter to
 * [oj.Translations.html#applyParameters]{@link oj.Translations}. As stated in
 * that documentation, if you are using a reserved character, you need to escape it with
 * a dollar character ('$').
 * </p>
 * Tokens: <p>
 * {label} - label of the component for which this message applies. The label may not always be
 * available depending on the usage of the validator. <br/>
 * {pattern} - the pattern the value should match<br/>
 * {value} - value entered by user<p>
 * Examples:<br/>
 * "'{label}' Format Incorrect" // translating to 'Phone Number' Format Incorrect
 * @property {string=} messageDetail - a custom error message to be used for creating detail
 * part of message, when the users input does not match the specified pattern. When not present, the
 * default detail message is the resource defined with the key
 * <code class="prettyprint">oj-validator.regExp.detail</code>.<p>
 * <p>The messageDetail string is passed as the 'pattern' parameter to
 * [oj.Translations.html#applyParameters]{@link oj.Translations}. As stated in
 * that documentation, if you are using a reserved character, you need to escape it with
 * a dollar character ('$').
 * </p>
 * Tokens:<br/>
 * {label} - label text of the component for which this message applies. <br/>
 * {pattern} the 'pattern' that the value should match <br/>
 * {value} value entered by the user <p>
 * Examples:<br/>
 * "The value {value} must contain at least 3 alphanumeric characters"<br/>
 */
/**
 * Initializes validator instance with the set options
 * @param {Object=} options
 * @memberof oj.RegExpValidator
 * @instance
 * @ignore
 */
oj.RegExpValidator.prototype.Init = function (options) {
  oj.RegExpValidator.superclass.Init.call(this);
  this._options = options;
};

/**
 * Validates value for matches using the regular expression provided by the pattern. This method
 * does not raise an error when value is the empty string or null; the method returns true indicating
 * that the validation was successful. If the application wants the empty string to fail validation,
 * then the application should chain in the required validator (e.g., set required on the input).
 *
 * @param {string|number} value that is being validated
 * @ojdeprecated {since: '6.2.0', target: "returnType", value: ["boolean"], description: 'This currently returns true
 * if successful. In v8.0 it will return nothing if successful.'}
 * @returns {void}
 * @ojsignature {target: "Type", for: "returns",
 *                value: "void"}
 *
 * @throws {Error} when there is no match
 * @memberof oj.RegExpValidator
 * @instance
 * @export
 */
oj.RegExpValidator.prototype.validate = function (value) {
  var detail;
  var label;
  var summary;
  var pattern = (this._options && this._options.pattern) || '';

  // don't validate null or empty string; per 
  // There are one of two ways we could handle the empty string:
  // 1) blow up on null and then require that customers wrap the validator with one that
  // succeeds on null if they don’t like the behavior
  // 2) Accept null and expect that the application will chain in the required checked if necessary
  // As a team we decided 2) was better than 1).
  if (value === null || value === undefined || value === '') {
    return true;
  }

  // when using digits as input values parseString becomes a integer type, so get away with it.
  var valueString = value.toString();

  // We intend that the pattern provided is matched exactly
  var exactPattern = '^(' + pattern + ')$';
  var valid = false;

  var matchArr = valueString.match(exactPattern);
  if ((matchArr !== null) && (matchArr[0] === valueString)) {
    valid = true;
  } else {
    if (this._options) {
      summary = this._options.messageSummary || null;
      detail = this._options.messageDetail || null;
      label = (this._options && this._options.label) || '';
    }

    var params = { label: label, pattern: pattern, value: valueString };
    var localizedSummary = summary ?
    Translations.applyParameters(summary, params) :
    Translations.getTranslatedString(this._getSummaryKey(), params);
    var localizedDetail = (detail) ?
    Translations.applyParameters(detail, params) :
    Translations.getTranslatedString(this._getDetailKey(), params);

    throw new oj.ValidatorError(localizedSummary, localizedDetail);
  }

  return valid;
};

/**
 * A message to be used as hint, when giving a hint on the expected pattern. There is no default
 * hint for this property.
 *
 * @returns {string|null} a hint message or null if no hint is available in the options
 * @memberof oj.RegExpValidator
 * @instance
 * @export
 */
oj.RegExpValidator.prototype.getHint = function () {
  var hint = null;
  var params = {};
  if (this._options && (this._options.hint)) {
    params = { pattern: this._options.pattern };
    hint = Translations.applyParameters(this._options.hint, params);
  }

  return hint;
};

oj.RegExpValidator.prototype._getSummaryKey = function () {
  return oj.RegExpValidator._BUNDLE_KEY_SUMMARY;
};

oj.RegExpValidator.prototype._getDetailKey = function () {
  return oj.RegExpValidator._BUNDLE_KEY_DETAIL;
};

/**
 * Copyright (c) 2008, 2013, Oracle and/or its affiliates.
 * All rights reserved.
 */
/* global Translations:false */
/**
 * Constructs a RequiredValidator that ensures that the value provided is not empty. In
 * most use cases you will use the ValidatorFactory to get an instance of a
 * RequiredValidator instead of using the constructor.
 * @param {Object=} options an object literal used to provide an optional hint and error message.<p>
 *
 * @export
 * @constructor
 * @augments oj.Validator
 * @ojsignature [{target: "Type", value: "class RequiredValidator implements Validator<object|string|number>"},
 *               {target: "Type", value: "oj.RequiredValidator.ValidatorOptions", for: "options", jsdocOverride: true}]
 * @since 0.6
 * @see oj.ValidatorFactory
 *
 */
oj.RequiredValidator = function (options) {
  this.Init(options);
};

// Subclass from oj.Object or oj.Validator. It does not matter
oj.Object.createSubclass(oj.RequiredValidator, oj.Validator, 'oj.RequiredValidator');

// key to access required validator specific resources in the bundle
oj.RequiredValidator._BUNDLE_KEY_DETAIL = 'oj-validator.required.detail';
oj.RequiredValidator._BUNDLE_KEY_SUMMARY = 'oj-validator.required.summary';

/**
 * @typedef {object} oj.RequiredValidator.ValidatorOptions
 * @property {string=} hint an optional hint text. There is no default hint provided by this
 * validator.
 * @property {string=} messageSummary - an optional custom error message summarizing the
 * error. When not present, the default message summary is the resource defined with the key
 * <code class="prettyprint">oj-validator.required.summary</code>.<p>
 * <p>The messageSummary string is passed as the 'pattern' parameter to
 * [oj.Translations.html#applyParameters]{@link oj.Translations}. As stated in
 * that documentation, if you are using a reserved character, you need to escape it with
 * a dollar character ('$').
 * </p>
 * Tokens: {label} - this token can be used to substitute the label of the component at runtime. </p>
 * <p>
 * Example:<br/>
 * "'{label}' Required"<br/>
 * </p>
 * @property {string=} messageDetail - a custom error message used for creating detail part
 * of message, when the value provided is empty. When not present, the default message detail is the
 * resource defined with the key <code class="prettyprint">oj-validator.required.detail</code>.
 * <p>The messageDetail string is passed as the 'pattern' parameter to
 * [oj.Translations.html#applyParameters]{@link oj.Translations}. As stated in
 * that documentation, if you are using a reserved character, you need to escape it with
 * a dollar character ('$').
 * </p>
 * <p>Tokens: {label} - this token can be used to substitute the label of the component at runtime.</p>
 * <p>
 * Example:<br/>
 * "A value is required for the field '{label}'."<br/>
 * </p>
 */
/**
 * Initializes validator instance with the set options
 * @param {Object=} options
 * @memberof oj.RequiredValidator
 * @instance
 * @ignore
 */
oj.RequiredValidator.prototype.Init = function (options) {
  oj.RequiredValidator.superclass.Init.call(this);
  this._options = options;
};

/**
 * Validates value to be non-empty
 *
 * @param {Object|string|number} value that is being validated
 * @returns {void}
 * @ojdeprecated {since: '6.2.0', target: "returnType", value: ["boolean"], description: 'This currently returns true
 * if successful. In v8.0 it will return nothing if successful.'}
 * @throws {Error} when fails required-ness check
 * @ojsignature {target: "Type", for: "returns",
 *                value: "void"}
 * @memberof oj.RequiredValidator
 * @instance
 * @export
 */
oj.RequiredValidator.prototype.validate = function (value) {
  var detail;
  var label = '';
  var localizedDetail;
  var localizedSummary;
  var summary;
  var params = {};

  // checks for empty arrays and String. Objects are considered non-null.
  // Need to specifically test for if value is 0 first if number is passed on.
  if ((typeof value === 'number' && value === 0) || (value && value.length !== 0)) {
    return true;
  }

  if (this._options) {
    // we have deprecated support for message param and instead use messageDetail.
    detail = this._options.messageDetail || this._options.message || null;
    summary = this._options.messageSummary || null;
    label = this._options.label || '';
  }
  params = { label: label };
  localizedSummary = (summary) ? Translations.applyParameters(summary, params) :
    Translations.getTranslatedString(this._getSummaryKey(), params);
  localizedDetail = (detail) ?
    Translations.applyParameters(detail, params) :
    Translations.getTranslatedString(this._getDetailKey(), params);

  throw new oj.ValidatorError(localizedSummary, localizedDetail);
};

/**
 * A message to be used as hint, when giving a hint on the expected pattern. There is no default
 * hint for this property.
 *
 * @returns {string|null} a hint message or null if no hint is available in the options
 * @memberof oj.RequiredValidator
 * @instance
 * @export
 */
oj.RequiredValidator.prototype.getHint = function () {
  var hint = '';
  if (this._options && (this._options.hint)) {
    hint = Translations.getTranslatedString(this._options.hint);
  }

  return hint;
};

oj.RequiredValidator.prototype._getSummaryKey = function () {
  return oj.RequiredValidator._BUNDLE_KEY_SUMMARY;
};

oj.RequiredValidator.prototype._getDetailKey = function () {
  return oj.RequiredValidator._BUNDLE_KEY_DETAIL;
};

/**
 * Copyright (c) 2017, Oracle and/or its affiliates.
 * All rights reserved.
 */

/* jslint browser: true,devel:true*/

/**
 * Validator Contract
 */

/**
 * The oj.AsyncValidator interface is a duck-typing interface for creating
 * asynchronous validators that can be set on the EditableValue (aka JET form) components
 * that have the <code class="prettyprint">async-validators</code> attribute.
 * @example <caption>Create an Object that duck-types the oj.AsyncValidator interface.
 * Bind the Object to the JET form component's async-validators attribute. The
 * validator's 'validate' method will be called when the user changes the input.</caption>
 *  self.asyncValidator1 = {
 *    // required validate method
 *    'validate': function(value) {
 *      return new Promise(function(resolve, reject) {
 *        var successful = someBackendMethod();
 *        if (successful) {
 *          resolve();
 *        } else {
 *          reject(new Error('The amount of purchase is too high. It is ' + value));
 *        }
 *      });
 *    },
 *    // optional hint attribute. hint shows up when user sets focus to input.
 *    'hint': new Promise(function (resolve, reject) {
 *      var formattedMaxPurchase = getSomeBackendFormattedMaxPurchase();
 *      resolve(maxPurchase + " is the maximum.");
 *    });
 *  };
 *  -- HTML --
 *  &lt;oj-input-text value="{{value1}}"
 *  async-validators="[[[asyncValidator1]]]">&lt;/oj-input-text>
 * @interface oj.AsyncValidator
 * @ojsignature {target: "Type", value: "interface AsyncValidator<V>",
 *               genericParameters: [{"name": "V", "description": "Type of value to be validated"}]}
 * @export
 * @since 5.2.0
 * @ojstatus preview
 *
 */

/**
 * <p>A method that validates the value.
 * The function returns a Promise that resolves to void
 * if the validation passes or a Promise that rejects with an error if it fails. The error will
 * be shown on the component.
 * </p>
 * <p>
 * It is recommended that you show the value you are validating in the error message
 * because if the async operation takes a while, the user could be typing in a new
 * value when the error message comes back and might be confused what value the error is for.
 * </p>
 * <p>If you need to format the value for the error message, you
 * can use oj.IntlConverterUtils.getConverterInstance(converterOption); to get the
 * converter instance, then call converter.format(value);
 * </p>
 *
 * @example <caption>Create an asynchronous validator and use it on an EditableValue
 * component. First, create an Object with 'validate' method that returns a Promise.
 * Then, bind it to the JET form component's async-validators attribute.</caption>
 *  self.asyncValidator1 = {
 *    'validate': function(value) {
 *      return new Promise(function(resolve, reject) {
 *        var successful = someBackendMethod();
 *        if (successful) {
 *          resolve();
 *        } else {
 *          //NOTE: if you need to format the value using a converter, you can call
 *          // oj.IntlConverterUtils.getConverterInstance(converterOption); to get the
 *          // converter instance, then call converter.format(value);
 *          reject(new Error('The amount of purchase ('+value+') is too high.'));
 *        }
 *      });
 *    }
 *  };
 *  -- HTML --
 *  &lt;oj-input-text value="{{value}}"
 *  async-validators="[[[asyncValidator1]]]">&lt;/oj-input-text>
 * @param {any} value to be validated
 * @return {Promise<void>} A Promise that resolves to nothing if validation passes or
 *  rejects with an Error if validation fails.
 * @method validate
 * @export
 * @expose
 * @ojsignature { target: "Type", value: "(value: V): Promise<void>" }
 * @memberof oj.AsyncValidator
 * @instance
 * @ojstatus preview
 */

 /**
 * hint is an optional attribute. It is a Promise that resolves to the hint string or null.
 * @example <caption>Create an Object that duck-types the oj.AsyncValidator interface.
 * Bind the Object to the JET form component's async-validators attribute. The
 * validator's 'hint'  will be called when the user focuses on the input and it
 * shows up as a notewindow giving the user a hint to what the validator will do.</caption>
 *  self.asyncValidator1 = {
 *    // required validate method
 *    'validate': function(value) {
 *      return new Promise(function(resolve, reject) {
 *        var successful = someBackendMethod();
 *        if (successful) {
 *          resolve();
 *        } else {
 *          reject(new Error('The amount of purchase ' + value +' is too high.'));
 *        }
 *      });
 *    },
 *    // optional hint attribute. hint shows up when user tabs to input.
 *    'hint': new Promise(function (resolve, reject) {
 *      // resolve the credit score REST call, and figure out what
 *      // is the maximum purchase dollar amount.
 *      var formattedMaxPurchase = getFormattedMaxPurchase();
 *      resolve('Your max purchase amount is ' + formattedMaxPurchase);
 *    });
 *  };
 *  -- HTML --
 *  &lt;oj-input-text value="{{value}}"
 *  async-validators="[[[asyncValidator1]]]">&lt;/oj-input-text>
 * @export
 * @expose
 * @memberof oj.AsyncValidator
 * @instance
 * @name hint
 * @type {Promise<string|null>=}
 * @ojstatus preview
 */

 /**
 * End of jsdoc
 */

/**
 * Copyright (c) 2016, Oracle and/or its affiliates.
 * All rights reserved.
 */
/* global Translations:false */

/**
 * oj.ColorConverter Contract.
 */


(function () {
  /**
   * In most use cases you will use the ConverterFactory to get an instance of a
   * ColorConverter.
   * @export
   * @constructor
   * @augments oj.Converter
   * @name oj.ColorConverter
   * @ojtsimport {module: "ojcolor", type: "AMD", importName: "Color"}
   * @ojsignature [{target: "Type",
   *                value: "class ColorConverter implements Converter<oj.Color>"},
   *               {target: "Type",
   *                value: "oj.ColorConverter.ConverterOptions",
   *                for: "options",
   *                jsdocOverride: true}
   *              ]

   * @classdesc An {@link oj.Color} object format converter.
   * @desc Creates a Converter that allows any color format to be obtained from an {@link oj.Color} object.
   * @since 0.6
   * @param {Object=} options - an object literal used to provide optional information to
   * initialize the converter.
   * @example <caption>Create a color converter to convert an rgb specification to hsl format</caption>
   * var cvFactory = oj.Validation.converterFactory(oj.ConverterFactory.CONVERTER_TYPE_COLOR);
   * var cv        = cv.createConverter({format: "hsl");
   * var color     = new oj.Color("rgb(30, 87, 236)") ;
   * var hsl       = cv.format(color);   -->  "hsl(223, 84%, 52%)"
   * @see oj.ConverterFactory
   */
  oj.ColorConverter = function (options) {
    this.Init(options);
  };

  /**
   * @typedef {object} oj.ColorConverter.ConverterOptions
   * @property {('rgb'|'hsl'|'hsv'|'hex'|'hex3')=} format - sets the format of the converted color specification.
   * Allowed values are "rgb" (the default, if omitted), "hsl", "hsv" "hex", and "hex3". "hex" returns six
   * hex digits ('#rrggbb'), and "hex3" returns three hex digits if possible ('#rgb') or six hex
   * digits if the value cannot be converted to three.
   */

// Subclass from oj.Converter
  oj.Object.createSubclass(oj.ColorConverter, oj.Converter, 'oj.ColorConverter');

  /**
   * Initializes the color converter instance with the set options.
   * @param {Object=} options an object literal used to provide an optional information to
   * initialize the converter.<p>
   * @export
   * @ignore
   */
  oj.ColorConverter.prototype.Init = function (options) {
    // eslint-disable-next-line no-param-reassign
    options = options || {};
    // eslint-disable-next-line no-param-reassign
    options.format = options.format || 'rgb';
    oj.ColorConverter.superclass.Init.call(this, options);
  };

  /**
   * Formats the color using the options provided into a string.
   *
   * @param {oj.Color} color the {@link oj.Color} instance to be formatted to a color specification string
   * @return {(string | null)} the color value formatted to the color specification defined in the options.
   * @throws {Error} a ConverterError if formatting fails, or the color option is invalid.
   * @export
   */
  oj.ColorConverter.prototype.format = function (color) {
    var fmt = this._getFormat();
    var ret = null;

    if (fmt === 'rgb') {
      ret = color.toString();
    } else if (fmt === 'hsl') {
      ret = _toHslString(color);
    } else if (fmt === 'hex') {
      ret = _toHexString(color);
    } else if (fmt === 'hex3') {
      ret = _toHexString(color, true);
    } else if (fmt === 'hsv') {
      ret = _toHsvString(color);
    } else {
      _throwInvalidColorFormatOption();
    }

    return ret || oj.ColorConverter.superclass.format.call(this, color);
  };


  /**
   * Parses a CSS3 color specification string and returns an oj.Color object.</br>
   * (Note that the "format" option used to create the Converter is not used
   * by this method, since the oj.Color object created is color agnostic.)
   * @param {string} value The color specification string to parse.
   * @return {oj.Color} the parsed value as an {@link oj.Color} object.
   * @throws {Error} a ConverterError if parsing fails
   * @export
   */
  oj.ColorConverter.prototype.parse = function (value) {
    try {
      return new oj.Color(value);   // throws error if invalid
    } catch (e) {
      throw _throwInvalidColorSyntax();  // This throw is not reachable.  THe function does it. Done for ESLint
    }
  };


  /**
   * Returns a hint that describes the color converter format.
   * @return {string} The expected format of a converted color.
   * @export
   */
  oj.ColorConverter.prototype.getHint = function () {
    return this._getFormat();
  };


  /**
   * Returns an object literal with properties reflecting the color formatting options computed based
   * on the options parameter.
   *
   * @ojsignature {target: "Type", for: "returns", value: "oj.ColorConverter.ConverterOptions"}
   * @return {Object} An object literal containing the resolved values for the following options.
   * <ul>
   * <li><b>format</b>: A string value with the format of the color specification.
   * for formatting.</li>
   * </ul>
   * @export
   */
  oj.ColorConverter.prototype.resolvedOptions = function () {
    return {
      format: this._getFormat()
    };
  };


  /**
   *   @private
   */
  oj.ColorConverter.prototype._getFormat = function () {
    return oj.ColorConverter.superclass.getOptions.call(this).format;
  };


  /**-------------------------------------------------------------*/
  /*   Helpers                                                    */
  /**-------------------------------------------------------------*/


  /**
   *  Converts an oj.Color object to a 3 or 6 hex character string
   *  @param {Object} color  The oj.Color object to be converted to a hex string.
   *  @param {boolean=} allow3Char  If true the representation is 3 hex characters
   *  (if possible). If false, or omitted, 6 hex characters are used.
   *  @return {string} The hex string representation of the color object.
   *  @private
   */
  function _toHexString(color, allow3Char) {
    return '#' + _toHex(color, allow3Char);
  }


  /**
   *  Converts an oj.Color object to an hsl/hsla string
   *  @param {Object} color  The oj.Color object to be converted to an hsl/hsla string.
   *  @return {string} The hsl/hsla representation of the color object.
   *  @private
   */
  function _toHslString(color) {
    var hsl = _rgbToHsl(color._r, color._g, color._b);
    var h = Math.round(hsl.h * 360);
    var s = Math.round(hsl.s * 100);
    var l = Math.round(hsl.l * 100);

    return (color._a === 1) ?
      'hsl(' + h + ', ' + s + '%, ' + l + '%)' :
      'hsla(' + h + ', ' + s + '%, ' + l + '%, ' + color._a + ')';
  }

  /**
   *  Converts an oj.Color object to a 3 or 6 hex character string
   *  @param {Object} color  The oj.Color object to be converted to a hex string.
   *  @param {boolean=} allow3Char  If true the representation is 3 hex characters
   *                   (if possible). If false, or omitted, 6 hex characters are used.
   *  @return {string} The hex string representation of the color object.
   *  @private
   */
  function _toHex(color, allow3Char) {
    return _rgbToHex(color._r, color._g, color._b, allow3Char);
  }

  /**
   *  Converts an oj.Color object to an hsv/hsva string
   *  @param {Object} color  The oj.Color object to be converted to an hsv/hsva string.
   *  @return {string} The hsv/hsva representation of the color object.
   *  @private
   */
  function _toHsvString(color) {
    var hsv = _rgbToHsv(color._r, color._g, color._b);

    var h = Math.round(hsv.h * 360);
    var s = Math.round(hsv.s * 100);
    var v = Math.round(hsv.v * 100);

    return (color._a === 1) ?
      'hsv(' + h + ', ' + s + '%, ' + v + '%)' :
      // "hsva(" + h + ", " + s + "%, " + v + "%, "+ this._roundA + ")";
      'hsva(' + h + ', ' + s + '%, ' + v + '%, ' + color._a + ')';
  }


  /**
   * Converts RGB color values to hex
   * @param {number} r the red value in the set [0,255]
   * @param {number} g the green value in the set [0,255]
   * @param {number} b the blue value in the set [0,255]
   * @param {boolean=} allow3Char  If true the representation is 3 hex characters
   *                   (if possible). If false, or omitted, 6 hex characters are used.
   * @returns {string} a 3 or 6 hex character string.
   * @private
   */
  function _rgbToHex(r, g, b, allow3Char) {
    var hex = [
      _pad2(Math.round(r).toString(16)),
      _pad2(Math.round(g).toString(16)),
      _pad2(Math.round(b).toString(16))
    ];

    // Return a 3 character hex if possible
    if (allow3Char && hex[0].charAt(0) === hex[0].charAt(1) &&
    hex[1].charAt(0) === hex[1].charAt(1) && hex[2].charAt(0) === hex[2].charAt(1)) {
      return hex[0].charAt(0) + hex[1].charAt(0) + hex[2].charAt(0);
    }

    return hex.join('');
  }


  /**
   * Converts an RGB color value to HSL.
   * Handle bounds/percentage checking to conform to CSS color spec, and returns
   * an object containg the h,s,l values.
   * <http://www.w3.org/TR/css3-color/>
   * Assumes:  r, g, b in [0, 255] or [0, 1]
   * @param {number} r the red value
   * @param {number} g the green value
   * @param {number} b the blue value
   * @returns {Object} Object with properties h, s, l, in [0, 1].
   * @private
   */
  function _rgbToHsl(r, g, b) {
    // eslint-disable-next-line no-param-reassign
    r = _bound01(r, 255);
    // eslint-disable-next-line no-param-reassign
    g = _bound01(g, 255);
    // eslint-disable-next-line no-param-reassign
    b = _bound01(b, 255);

    var max = Math.max(r, g, b);
    var min = Math.min(r, g, b);
    var h;
    var s;
    var l = (max + min) / 2;

    if (max === min) {
      h = 0; // achromatic
      s = 0; // achromatic
    } else {
      var d = max - min;
      s = l > 0.5 ? d / (2 - max - min) : d / (max + min);
      switch (max) {
        case r:
          h = ((g - b) / d) + (g < b ? 6 : 0);
          break;
        case g:
          h = ((b - r) / d) + 2;
          break;
        case b:
          h = ((r - g) / d) + 4;
          break;
        default:
          break;
      }

      h /= 6;
    }

    return { h: h, s: s, l: l };
  }

  /**
   * Converts an RGB color value to HSV.
   * Handle bounds/percentage checking to conform to CSS color spec, and returns
   * an object containg the h,s,v values.
   * <http://www.w3.org/TR/css3-color/>
   * Assumes:  r, g, and b are contained in the set [0,255] or [0,1]
   * @param {number} r the red value
   * @param {number} g the green value
   * @param {number} b the blue value
   * @returns {Object} Object with properties h, s, v, in [0,1].
   * @private
   */
  function _rgbToHsv(r, g, b) {
    // eslint-disable-next-line no-param-reassign
    r = _bound01(r, 255);
    // eslint-disable-next-line no-param-reassign
    g = _bound01(g, 255);
    // eslint-disable-next-line no-param-reassign
    b = _bound01(b, 255);

    var max = Math.max(r, g, b);
    var min = Math.min(r, g, b);
    var h;
    var v = max;

    var d = max - min;
    var s = max === 0 ? 0 : d / max;

    if (max === min) {
      h = 0; // achromatic
    } else {
      switch (max) {
        case r:
          h = ((g - b) / d) + (g < b ? 6 : 0);
          break;
        case g:
          h = ((b - r) / d) + 2;
          break;
        case b:
          h = ((r - g) / d) + 4;
          break;
        default:
          break;
      }
      h /= 6;
    }
    return { h: h, s: s, v: v };
  }


  /**
   * Converts an RGBA color plus alpha transparency to hex
   * Assumes r, g, b and a are contained in the set [0, 255]
   * @param {number} r the red value in the set [0, 255]
   * @param {number} g the green value in the set [0, 255]
   * @param {number} b the blue value in the set [0, 255]
   * @param {number} a the alpha value in the set [0,1]
   * Returns an 8 character hex string
   */
  /*    NOT USED currently
   function rgbaToHex(r, g, b, a)
   {
   var hex = [
   pad2(convertDecimalToHex(a)),
   pad2(mathRound(r).toString(16)),
   pad2(mathRound(g).toString(16)),
   pad2(mathRound(b).toString(16))
   ];

   return hex.join("");
   }
   */


  /**
   * Take input from [0, n] and return it as [0, 1]
   * @private
   */
  function _bound01(n, max) {
    if (_isOnePointZero(n)) {
      // eslint-disable-next-line no-param-reassign
      n = '100%';
    }

    var processPercent = _isPercentage(n);
    // eslint-disable-next-line no-param-reassign
    n = Math.min(max, Math.max(0, parseFloat(n)));

    // Automatically convert percentage into number
    if (processPercent) {
      // eslint-disable-next-line no-param-reassign
      n = parseInt(n * max, 10) / 100;
    }

    // Handle floating point rounding errors
    if ((Math.abs(n - max) < 0.000001)) {
      return 1;
    }

    // Convert into [0, 1] range if it isn't already
    return (n % max) / parseFloat(max);
  }


  /**
   *   Need to handle 1.0 as 100%, since once it is a number, there is no difference between it and 1
   *   <http://stackoverflow.com/questions/7422072/javascript-how-to-detect-number-as-a-decimal-including-1-0>
   *   @private
   */
  function _isOnePointZero(n) {
    return typeof n === 'string' && n.indexOf('.') !== -1 && parseFloat(n) === 1;
  }

  /**
   *  Check to see if string passed in is a percentage
   *  @param {string}  n  The number string
   *  @return {boolean}  True if the string contains a '%' character.
   *  @private
   */
  function _isPercentage(n) {
    return typeof n === 'string' && n.indexOf('%') !== -1;
  }


  /**
   *  Force a hex value string to have 2 characters by inserting a preceding zero
   *  if neccessary.  e.g. 'a' -> '0a'
   *  @param {string} c  The hex character(s) to be tested.
   *  @return {string} A two character hex string.
   *  @private
   */
  function _pad2(c) {
    return c.length === 1 ? '0' + c : '' + c;
  }


  /*
   *   Throw an invalid color specfication error.
   */
  function _throwInvalidColorSyntax() {
    var summary = Translations.getTranslatedString('oj-converter.color.invalidSyntax.summary');
    var detail = Translations.getTranslatedString('oj-converter.color.invalidSyntax.detail');

    var ce = new oj.ConverterError(summary, detail);

    throw ce;
  }


  /*
   *   Throw an invalid converter specfication error.
   */
  function _throwInvalidColorFormatOption() {
    var summary = Translations.getTranslatedString('oj-converter.color.invalidFormat.summary');
    var detail = Translations.getTranslatedString('oj-converter.color.invalidFormat.detail');

    var ce = new oj.ConverterError(summary, detail);

    throw ce;
  }
}());


/**
 * Copyright (c) 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */
/* global Logger:false, Translations:false, Promise:false */

/**
 * @export
 * @namespace oj.IntlConverterUtils
 * @classdesc Utility function for converters
 * @hideconstructor
 * @since 0.7
 */
oj.IntlConverterUtils = {};

/**
 * Parses the isoString and returns a JavaScript Date object
 *
 * @export
 * @param {string} isoString isoString to parse and to return Date of
 * @return {Date} the parsed JavaScript Date Object
 */
oj.IntlConverterUtils.isoToDate = function (isoString) {
  return oj.OraI18nUtils.isoToDate(isoString);
};

/**
 * Returns a local Date object provided a local isoString
 *
 * @param {string} isoString date in isoString format
 * @returns {Date} localDate
 * @export
 * @since 0.7
 */
oj.IntlConverterUtils.isoToLocalDate = function (isoString) {
  return oj.OraI18nUtils.isoToLocalDate(isoString);
};

/**
 * Returns a local isoString provided a Date object
 *
 * @param {Date} date
 * @returns {string} isoString
 * @export
 * @since 0.7
 */
oj.IntlConverterUtils.dateToLocalIso = function (date) {
  return oj.OraI18nUtils.dateToLocalIso(date);
};

// This private method is not called, commenting out for now to remove it from code coverage calculations
// oj.IntlConverterUtils._getTimeZone = function(isoString)
// {
//  return oj.OraI18nUtils._getTimeZone(isoString);
// };

/**
 * Returns the timezone offset between UTC and the local time in Etc/GMT[+-]hh:mm syntax.
 * The offset is positive if the local timezone is behind UTC and negative if
 * it is ahead. The offset range is between Etc/GMT-14 and Etc/GMT+12 (UTC-12 and UTC+14)
 *
 * @example <caption>The local time is UTC-7 (Pacific Daylight Time)</caption>
 * oj.IntlConverterUtils.getLocalTimeZoneOffset() will return the string "Etc/GMT+07:00"
 *
 * @example <caption>The local time is UTC+1 (Central European Standard Time)</caption>
 * oj.IntlConverterUtils.getLocalTimeZoneOffset() will return the string "Etc/GMT-01:00"
 *
 * @export
 * @returns {string}
 */
oj.IntlConverterUtils.getLocalTimeZoneOffset = function () {
  return oj.OraI18nUtils.getLocalTimeZoneOffset();
};

/**
 * Given either an Object literal representing a 'converter' option (used in components) or a
 * converter instance of type oj.Converter, this method returns the converter instance.
 * You can also pass in a string. In this case, it will return you an instance of a converter
 * registered with that type.
 * @param {string| Object} converterOption
 * @returns {Object|null} converterInstance or null if a converter cannot be determined
 * @ojsignature {
 *        target: "Type", value: "<T>(converterOption: string|oj.Validation.RegisteredConverter|oj.Converter<T>): oj.Converter<T>|null"}
 * @export
 * @since 0.6
 */
oj.IntlConverterUtils.getConverterInstance = function (converterOption) {
  var cTypeStr = '';
  var cOptions = {};
  var converterInstance = null;

  if (converterOption) {
    if (typeof converterOption === 'object') {
      // TODO: Should we check that it duck types oj.Converter?
      if (converterOption instanceof oj.Converter ||
          (converterOption.parse && typeof converterOption.parse === 'function') ||
          (converterOption.format && typeof converterOption.format === 'function')) {
        // we are dealing with a converter instance
        converterInstance = converterOption;
      } else {
        // check if there is a type set
        cTypeStr = converterOption.type;
        cOptions = converterOption.options || {};
      }
    }

    if (!converterInstance) {
      // either we have an object literal or just plain string.
      cTypeStr = cTypeStr || converterOption;
      if (cTypeStr && typeof cTypeStr === 'string') {
        // if we are passed a string get registered type.
        var cf = oj.Validation.converterFactory(cTypeStr);
        if (cf === null) {
          Logger.error(
            'Converter of type ' + cTypeStr +
              ' cannot be found. Make sure the correct converter/validation module is included.'
          );
        }
        return cf.createConverter(cOptions);
      }
    }
  }

  return converterInstance;
};

/**
 * So the requirement is if min or max lacks date portion and value contains it, then min + max should use
 * value's date portion
 *
 * @param {string} minMax date in isoString format
 * @param {string} value date in isoString format
 * @returns {string} merged date in isoString format
 * @export
 * @ignore
 * @since 1.2
 */
oj.IntlConverterUtils._minMaxIsoString = function (minMax, value) {
  if (minMax) {
    // eslint-disable-next-line no-param-reassign
    value = value || this.dateToLocalIso(new Date());

    var vTindex = value.indexOf('T');

    if (minMax.indexOf('T') === 0 && vTindex > 0) {
      // meaning only time exists for minMax and value contains date
      // eslint-disable-next-line no-param-reassign
      minMax = value.substring(0, vTindex) + minMax;
    }
  }

  return minMax;
};

// PACKAGE PRIVATE

/**
 * Processes an converter option error and returns a oj.ConverterERror instance.
 * @param {string} errorCode
 * @param {Object} parameterMap
 * @return {Object} an oj.ConverterError instance
 * @private
 */
oj.IntlConverterUtils.__getConverterOptionError = function (errorCode, parameterMap) {
  oj.Assert.assertObject(parameterMap);
  var summary = '';
  var detail = '';
  var propName = parameterMap.propertyName;
  var propValueValid;

  if (errorCode === 'optionTypesMismatch') {
    var reqPropName = parameterMap.requiredPropertyName;
    propValueValid = parameterMap.requiredPropertyValueValid;
    // Summary: A value for the property '{requiredPropertyName}' is required when the property
    // '{propertyName}' is set to '{propertyValue}'.
    summary = Translations.getTranslatedString('oj-converter.optionTypesMismatch.summary',
      { propertyName: propName,
        propertyValue: parameterMap.propertyValue,
        requiredPropertyName: reqPropName });

    detail = oj.IntlConverterUtils._getOptionValueDetailMessage(reqPropName, propValueValid);
  } else if (errorCode === 'optionTypeInvalid') {
    // Summary: A value of the expected type was not provided for '{propertyName}'.
    propName = parameterMap.propertyName;
    propValueValid = parameterMap.propertyValueValid;
    summary = Translations.getTranslatedString('oj-converter.optionTypeInvalid.summary',
      { propertyName: propName });

    detail = oj.IntlConverterUtils._getOptionValueDetailMessage(propName, propValueValid);
  } else if (errorCode === 'optionOutOfRange') {
    // The value {propertyValue} is out of range for the option '{propertyName}'.
    summary = Translations.getTranslatedString('oj-converter.optionOutOfRange.summary',
      { propertyName: propName,
        propertyValue: parameterMap.propertyValue });

    propValueValid = parameterMap.propertyValueValid;
    detail = oj.IntlConverterUtils._getOptionValueDetailMessage(propName, propValueValid);
  } else if (errorCode === 'optionValueInvalid') {
    // An invalid value '{propertyValue}' was specified for the option '{propertyName}'..
    summary = Translations.getTranslatedString('oj-converter.optionValueInvalid.summary',
      { propertyName: propName,
        propertyValue: parameterMap.propertyValue });

    propValueValid = parameterMap.propertyValueHint;
    detail = oj.IntlConverterUtils._getOptionValueDetailMessage(propName, propValueValid);
  }

  return new oj.ConverterError(summary, detail);
};

/**
 * Builds the detail message for possible converter option values. Only applicable when errorInfo is
 * returned from JET converter implementation.
 *
 * @param {string} propName name of the property
 * @param {Array|string} propValueValid valid value or values expected.
 *
 * @return {string} the localized message
 * @private
 */
oj.IntlConverterUtils._getOptionValueDetailMessage = function (propName, propValueValid) {
  // Detail: An accepted value for '{propertyName}' is '{propertyValueValid}'. or
  // Accepted values for '{propertyName}' are '{propertyValueValid}'.
  var resourceKey;

  if (propValueValid) {
    if (typeof propValueValid === 'string') {
      resourceKey = 'oj-converter.optionHint.detail';
    } else {
      // we have an array of values
      resourceKey = 'oj-converter.optionHint.detail-plural';
      // eslint-disable-next-line no-param-reassign
      propValueValid =
         propValueValid.join(Translations.getTranslatedString('oj-converter.plural-separator'));
    }
    return Translations.getTranslatedString(resourceKey,
      { propertyName: propName,
        propertyValueValid: propValueValid });
  }

  return '';
};

/**
 * Returns the default value for non-truthy values.
 *
 * @returns {string} an empty string
 * @private
 */
oj.IntlConverterUtils.__getNullFormattedValue = function () {
  return '';
};

/**
 * Will return an updated toIsoString using the timePortion from the fromIsoString or from the default
 * oj.OraI18nUtils.DEFAULT_TIME_PORTION
 *
 * @private
 * @expose
 * @param {string} fromIsoString isoString that may not be a complete isoString
 * @param {string} toIsoString isoString that may not be a complete isoString
 * @returns {string} modified toIsoString with original date portion and the time portion from the fromIsoString
 * @since 1.1
 */
oj.IntlConverterUtils._copyTimeOver = function (fromIsoString, toIsoString) {
  return oj.OraI18nUtils._copyTimeOver(fromIsoString, toIsoString);
};

/**
 * Clears the time portion of the isoString
 *
 * @private
 * @expose
 * @param {string} isoString isoString that may not be a complete isoString
 * @returns {string} an updated isoString
 * @since 1.1
 */
oj.IntlConverterUtils._clearTime = function (isoString) {
  return oj.OraI18nUtils._clearTime(isoString);
};

/**
 * Will accept an isoString and perform a get operation or a set operation depending on whether param is an Array
 * or a JSON
 *
 * The keys for the get and set operation are defined in oj.OraI18nUtils's _DATE_TIME_KEYS.
 *
 * Note the handling of month starting with 0 in Date object and being 1 based in isoString will be handled by the function
 * with the usage of doParseValue. Meaning when you doParseValue and you are getting the value it will automatically
 * decrement the value and when you are setting the param it will check if the value is of number and if so will
 * increment it.
 *
 * @private
 * @expose
 * @param {string} isoString isoString that may not be a complete isoString
 * @param {Array|Object} actionParam if an Array will be a get operation, if a JSON will be a set operation
 * @param {boolean=} doParseValue whether one should parseInt the value during the get request
 * @returns {Object|string} an Object when a get operation and a string when a set operation
 * @since 1.1
 */
oj.IntlConverterUtils._dateTime = function (isoString, actionParam, doParseValue) {
  return oj.OraI18nUtils._dateTime(isoString, actionParam, doParseValue);
};

/**
 * So the problem is Jet uses incomplete isoString which causes issues in different browsers.
 *
 * For instance for a new Date().toISOString() => 2015-02-02T18:00:37.007Z
 * ojInputDate stores 2015-02-02
 * ojInputTime stores T18:00:37.007Z
 *
 * yet constructing new Date(val) on above causes different results or errors in different browsers, so
 * this function is to normalize them. Note it is assumed that the point is creating the Date object from the
 * normalized isoString. Meaning if both contain only the time portion today's date will appended to it.
 *
 * @private
 * @expose
 * @param {string} isoString isoString that may not be a complete isoString
 * @returns {string} a normalized isoString
 * @since 1.1
 */
oj.IntlConverterUtils._normalizeIsoString = function (isoString) {
  return oj.OraI18nUtils._normalizeIsoString(isoString);
};

/**
 * Returns a person's initials
 *
 * @param {string=} firstName first name
 * @param {string=} lastName last name
 * @returns {string|undefined} uppercase concatenation of first letter of first name and first letter
 * of last name.
 * There are the following special cases:
 * - If the name is Arabic characters, it returns empty string.
 * - If the name is Hindi characters, it returns the first letter of the first name.
 * - If the name is Thai characters, it returns the first letter of the first name.
 * - If the name is Korean characters, it returns the first name.
 * - If the name is Japanese or Chinese characters, it returns the last name.
 *
 * @export
 * @since 4.0.0
 */
oj.IntlConverterUtils.getInitials = function (firstName, lastName) {
  return oj.OraI18nUtils.getInitials(firstName, lastName);
};

/**
 * Copyright (c) 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */
/* global Translations:false */

/**
 * Constructs a LengthValidator that ensures the value entered is within a given length.
 * In most use cases you will use the ValidatorFactory to get an instance of a
 * LengthValidator instead of using the constructor.
 * <p>
 * By default this uses Javascript's String length property
 * which counts a UTF-16 surrogate pair as length === 2.
 * If you need to count surrogate pairs as length === 1, then set the
 * <code class="prettyprint">countBy</code> option to <code class="prettyprint">"codePoint"</code>
 * or use
 * <code class="prettyprint">oj.LengthValidator.defaults.countBy = "codePoint";</code>
 * to set the page-wide default.
 * </p>
 * <p>
 * You can customize the default messages of all validators including this one
 * using the messageDetail and messageSummary options.
 * <p>
 * @param {Object=} options an object literal used to provide:<p>
 * @export
 * @constructor
 * @augments oj.Validator
 * @ojsignature [{target: "Type",
 *                value: "class LengthValidator implements Validator<number|string>"},
 *               {target: "Type",
 *                value: "oj.LengthValidator.ValidatorOptions",
 *                for: "options",
 *                jsdocOverride: true}
 *              ]
 * @since 0.7
 * @see oj.ValidatorFactory
 */
oj.LengthValidator = function (options) {
  this.Init(options);
};

/**
 * @typedef {object} oj.LengthValidator.ValidatorOptions
 * @property {('codeUnit'|'codePoint')=} countBy - A string that specifies how to count the length. Valid values are
 * <code class="prettyprint">"codeUnit"</code> and <code class="prettyprint">"codePoint"</code>.
 * Defaults to <code class="prettyprint">oj.LengthValidator.defaults.countBy</code> which defaults
 * to <code class="prettyprint">"codeUnit"</code>.<br/>
 * <code class="prettyprint">"codeUnit"</code> uses javascript's length function which counts the
 * number of UTF-16 code units. Here a Unicode surrogate pair has a length of two. <br/>
 * <code class="prettyprint">"codePoint"</code>
 * counts the number of Unicode code points.
 * Here a Unicode surrogate pair has a length of one.<br/>
 * @property {number=} min - a number 0 or greater that is the minimum length of the value.
 * @property {number=} max - a number 1 or greater that is the maximum length of the value.
 * @property {Object=} hint - an optional object literal of hints to be used.
 * <p>The hint strings (e.g., hint.min) are  passed as the 'pattern' parameter to
 * [oj.Translations.html#applyParameters]{@link oj.Translations}. As stated in
 * that documentation, if you are using a reserved character, you need to escape it with
 * a dollar character ('$').
 * </p>
 * @property {string=} hint.max - a hint message to be used to indicate the allowed maximum.
 * When not present, the default hint is the resource defined with the key
 * <code class="prettyprint">oj-validator.length.hint.max</code>.<p>
 * Tokens: <br/>
 * {max} - the maximum<p>
 * Usage: <br/>
 * Enter {max} or fewer characters
 * @property {string=} hint.min - a hint message to be used to indicate the allowed minimum.
 * When not present, the default hint is the resource defined with the key
 * <code class="prettyprint">oj-validator.length.hint.min</code>.<p>
 * Tokens: <br/>
 * {min} the minimum<p>
 * Usage: <br/>
 * Enter {min} or more characters
 * @property {string=} hint.inRange - a hint message to be used to indicate the allowed range.
 * When not present, the default hint is the resource defined with the key
 * <code class="prettyprint">oj-validator.length.hint.inRange</code>.<p>
 * Tokens: <br/>
 * {min} the minimum<p>
 * {max} - the maximum<p>
 * Usage: <br/>
 * Enter between {min} and {max} characters
 * @property {string=} hint.exact - a hint message to be used, to indicate the exact length.
 * When not present, the default hint is the resource defined with the key
 * <code class="prettyprint">oj-validator.length.hint.exact</code>.<p>
 * Tokens: <br/>
 * {length} the length<p>
 * Usage: <br/>
 * Enter {length} characters
 * @property {Object=} messageDetail - an optional object literal of custom error messages to
 * be used.
 * <p>The messageDetail strings (e.g., messageDetail.tooLong) are  passed as the 'pattern'
 * parameter to [oj.Translations.html#applyParameters]{@link oj.Translations}. As stated in
 * that documentation, if you are using a reserved character, you need to escape it with
 * a dollar character ('$').
 * </p>
 * @property {string=} messageDetail.tooLong - the detail error message to be used as the error
 * message, when the length of the input value exceeds the maximum value set. When not present, the
 * default detail message is the resource defined with the key
 * <code class="prettyprint">oj-validator.length.messageDetail.tooLong</code>.<p>
 * Tokens:<br/>
 * {value} - value entered by the user<br/>
 * {max} - the maximum allowed value<p>
 * Usage: <br/>
 * The {value} has too many characters. Enter {max} or fewer characters, not more.
 * @property {string=} messageDetail.tooShort - the detail error message to be used as the error
 * message, when the length of the input value is less the minimum value set. When not present, the
 * default detail message is the resource defined with the key
 * <code class="prettyprint">oj-validator.length.messageDetail.tooShort</code>.<p>
 * Tokens:<br/>
 * {value} - value entered by the user<br/>
 * {min} - the minimum allowed value<p>
 * Usage: <br/>
 * The {value} has too few characters. Enter {min} or more characters, not less.
 * @property {Object=} messageSummary - optional object literal of custom error summary message
 * to be used.
 * <p>The messageSummary strings (e.g., messageSummary.tooLong) are  passed as the 'pattern'
 * parameter to [oj.Translations.html#applyParameters]{@link oj.Translations}. As stated in
 * that documentation, if you are using a reserved character, you need to escape it with
 * a dollar character ('$').
 * </p>
 * @property {string=} messageSummary.tooLong - the message to be used as the summary error
 * message, when the length of the input value exceeds the maximum value set. When not present, the
 * default message summary is the resource defined with the key
 * <code class="prettyprint">oj-validator.length.messageSummary.tooLong</code>.
 * @property {string=} messageSummary.tooShort - the message to be used as the summary error
 * message, when input value is less than the set minimum value. When not present, the default
 * message summary is the resource defined with the key
 * <code class="prettyprint">oj-validator.length.messageSummary.tooShort</code>.
 */

/**
 * The set of attribute/value pairs that serve as default values
 * when new LengthValidator objects are created.
 * <p>
 * LengthValidator's <code class="prettyprint">countBy</code> option may be changed
 * for the entire application after the 'ojs/ojvalidation' module is loaded
 * (each form control module includes the 'ojs/ojvalidation' module). If the
 * <code class="prettyprint">options.countBy</code> is specifically set,
 * it will take precedence over this default.
 * </p>
 * <p>For example:
 * <code class="prettyprint">
 * oj.LengthValidator.defaults.countBy = 'codePoint';
 * </code></p>
 * @property {string} countBy count the length by <code class="prettyprint">"codeUnit"</code> or
 * <code class="prettyprint">"codePoint"</code>.
 * Defaults to <code class="prettyprint">"codeUnit"</code>.
 * @export
 * @memberof oj.LengthValidator
 * @since 2.1.0
 */
oj.LengthValidator.defaults =
{
  countBy: 'codeUnit'
};

// Subclass from oj.Validator
oj.Object.createSubclass(oj.LengthValidator, oj.Validator, 'oj.LengthValidator');

/**
 * Initializes validator instance with the set options
 * @param {Object=} options
 * @ignore
 */
oj.LengthValidator.prototype.Init = function (options) {
  var countByOptions = options.countBy;

  oj.LengthValidator.superclass.Init.call(this);

  this._min = options.min !== undefined ? parseInt(options.min, 10) : null;
  this._max = options.max !== undefined ? parseInt(options.max, 10) : null;

  // check that the min/max make sense, otherwise throw an error
  if (isNaN(this._min)) {
    throw new Error("length validator's min option is not a number. min option is " + options.min);
  }
  if (isNaN(this._max)) {
    throw new Error("length validator's max option is not a number. max option is " + options.min);
  }
  if (this._min !== null && this._min < 0) {
    throw new Error("length validator's min option cannot be less than 0. min option is " +
                    options.min);
  }
  if (this._max !== null && this._max < 1) {
    throw new Error("length validator's max option cannot be less than 1. max option is " +
                    options.max);
  }

  this._countBy = (countByOptions === undefined) ?
  oj.LengthValidator.defaults.countBy : countByOptions;

  if (options) {
    this._hint = options.hint || {};
    this._customMessageSummary = options.messageSummary || {};
    this._customMessageDetail = options.messageDetail || {};
  }
};

/**
 * A message to be used as hint, when giving a hint about the expected length. There is no default
 * hint for this property.
 *
 * @returns {string|null} a hint message or null if no hint is available in the options
 * @memberof oj.LengthValidator
 * @instance
 * @export
 */
oj.LengthValidator.prototype.getHint = function () {
  var hint = null;
  var hints = this._hint;
  var hintExact = hints.exact;
  var hintInRange = hints.inRange;
  var hintMaximum = hints.max;
  var hintMinimum = hints.min;

  var max = this._max;
  var min = this._min;
  var params;
  var translations = Translations;

  if (min !== null && max !== null) {
    if (min !== max) {
      params = { min: min, max: max };
      hint = hintInRange ? translations.applyParameters(hintInRange, params) :
      translations.getTranslatedString('oj-validator.length.hint.inRange', params);
    } else {
      params = { length: min };
      hint = hintExact ? translations.applyParameters(hintExact, params) :
      translations.getTranslatedString('oj-validator.length.hint.exact', params);
    }
  } else if (min !== null) {
    params = { min: min };
    hint = hintMinimum ? translations.applyParameters(hintMinimum, params) :
    translations.getTranslatedString('oj-validator.length.hint.min', params);
  } else if (max !== null) {
    params = { max: max };
    hint = hintMaximum ? translations.applyParameters(hintMaximum, params) :
    translations.getTranslatedString('oj-validator.length.hint.max', params);
  }

  return hint;
};

/**
 * Validates the length of value is greater than minimum and/or less than maximum.
 * @param {string|number} value that is being validated
 * @returns {void}
 * @ojsignature {target: "Type", for: "returns",
 *                value: "void"}
 * @throws {Error} when the length is out of range.
 * @ojdeprecated [{since: '6.2.0', target: "returnType", value: ["string"], description: 'This currently returns the original value
 * in string form if successful. In v8.0 it will return nothing if successful.'}]
 * @export
 */
oj.LengthValidator.prototype.validate = function (value) {
  var customMessageDetail = this._customMessageDetail;
  var customMessageSummary = this._customMessageSummary;
  var detail = '';
  var max = this._max;
  var messageSummaryTooLong = customMessageSummary.tooLong;
  var messageSummaryTooShort = customMessageSummary.tooShort;
  var messageTooLong = customMessageDetail.tooLong;
  var messageTooShort = customMessageDetail.tooShort;
  var min = this._min;
  var params;
  var summary = '';
  var translations = Translations;
  var string = '' + value;
  var length = this._getLength(string);

  // If only min is set and length is at least min, or
  // if only max is set and length is at most max, or
  // if length is between min and max or
  // if neither min or max is set return with no error.
  if ((min === null || length >= this._min) &&
  ((max === null) || (length <= this._max))) {
    return string;
  }

  if (length < this._min) { // too short
    params = { value: value, min: min };
    summary = messageSummaryTooShort ?
      translations.applyParameters(messageSummaryTooShort, params) :
      translations.getTranslatedString('oj-validator.length.messageSummary.tooShort');
    detail = messageTooShort ?
      translations.applyParameters(messageTooShort, params) :
      translations.getTranslatedString('oj-validator.length.messageDetail.tooShort', params);
  } else { // too long
    params = { value: value, max: max };
    summary = messageSummaryTooLong ?
      translations.applyParameters(messageSummaryTooLong, params) :
      translations.getTranslatedString('oj-validator.length.messageSummary.tooLong');
    detail = messageTooLong ?
      translations.applyParameters(messageTooLong, params) :
      translations.getTranslatedString('oj-validator.length.messageDetail.tooLong', params);
  }

  throw new oj.ValidatorError(summary, detail);
};

/**
 * @returns {number} the length of the text counted by UTF-16 codepoint
 *  or codeunit as specified in the countBy option.
 * @private
 */
oj.LengthValidator.prototype._getLength = function (text) {
  var countBy = this._countBy.toLowerCase();
  var codeUnitLength = text.length;
  var length;
  var surrogateLength = 0;

  switch (countBy) {
    case 'codepoint' :
      // if countBy is "codePoint", then count supplementary characters as length of one
      // For UTF-16, a "Unicode  surrogate pair" represents a single supplementary character.
      // The first (high) surrogate is a 16-bit code value in the range U+D800 to U+DBFF.
      // The second (low) surrogate is a 16-bit code value in the range U+DC00 to U+DFFF.
      // This code figures out if a charCode is a high or low surrogate and if so,
      // increments surrogateLength
      for (var i = 0; i < codeUnitLength; i++) {
        // eslint-disable-next-line no-bitwise
        if ((text.charCodeAt(i) & 0xF800) === 0xD800) {
          surrogateLength += 1;
        }
      }
      // e.g., if the string is two supplementary characters, codeUnitLength is 4, and the
      // surrogateLength is 4, so we will return two.
      oj.Assert.assert(surrogateLength % 2 === 0,
        'the number of surrogate chars must be an even number.');
      length = (codeUnitLength - (surrogateLength / 2));
      break;
    case 'codeunit' :
    default :
      // Javascript's length function counts # of code units.
      // A supplementary character has a length of 2 code units.
      length = codeUnitLength;
  }
  return length;
};

/**
 * Copyright (c) 2008, 2013, Oracle and/or its affiliates.
 * All rights reserved.
 */

/**
 * A factory implementation to create the built-in color converter of type
 * {@link oj.ColorConverter}.
 *
 * @name oj.ColorConverterFactory
 * @public
 * @class
 * @ojtsnoexport
 * @ojtsexportastype
 * @hideconstructor
 * @example <caption>create an instance of the jet color converter using the options provided</caption>
 * var ccf = oj.Validation.converterFactory(oj.ConverterFactory.CONVERTER_TYPE_COLOR);
 * var options = {format: 'hsl'};
 * var colorConverter = ccf.createConverter(options);
 * @since 0.6
 *
 */
oj.ColorConverterFactory = (function () {
  function _createColorConverter(options) {
    return new oj.ColorConverter(options);
  }

  /**
   *
   * @public
   */
  return {
    /**
     * Creates an immutable (jet) color converter instance.
     *
     * @param {Object=} options an object literal used to provide an optional information to
     * initialize the jet color converter. For details on what to pass for options, refer to
     * {@link oj.ColorConverter}.
     *
     * @return {oj.ColorConverter}
     * @ojsignature {target: "Type", for: "options", value: "oj.ColorConverter.ConverterOptions"}
     * @instance
     * @memberOf oj.ColorConverterFactory
     * @public
     */
    createConverter: function (options) {
      return _createColorConverter(options);
    }
  };
}()); // notice immediate invocation of anonymous function

/** Register the default factory provider function */
oj.Validation.__registerDefaultConverterFactory(oj.ConverterFactory.CONVERTER_TYPE_COLOR, // factory name
                                                oj.ColorConverterFactory);

// JET VALIDATOR FACTORIES

/**
 * A factory implementation to create an instance of the built-in required validator of type
 * {@link oj.RequiredValidator}.
 *
 * @example <caption>create an instance of the required validator using the factory</caption>
 * var rvf = oj.Validation.validatorFactory(oj.ValidatorFactory.VALIDATOR_TYPE_REQUIRED);
 * var options = {'hint' : 'a value is required for this field'};
 * var requiredValidator = rvf.createValidator(options);
 *
 * @name oj.RequiredValidatorFactory
 * @hideconstructor
 * @ojtsnoexport
 * @ojtsexportastype
 * @class
 * @public
 * @since 0.6
 *
 */
oj.RequiredValidatorFactory = (function () {
  function _createRequiredValidator(options) {
    return new oj.RequiredValidator(options);
  }

  return {
    /**
     * Creates an immutable validator instance of type @link oj.RequiredValidator that ensures that
     * the value provided is not empty.
     *
     * @param {Object=} options an object literal used to provide an optional hint and error
     * message. See {@link oj.RequiredValidator} for details.<p>
     *
     * @return {oj.RequiredValidator}
     * @ojsignature {target: "Type", for: "options", value: "oj.RequiredValidator.ValidatorOptions"}
     * @memberOf oj.RequiredValidatorFactory
     * @instance
     * @public
     */
    createValidator: function (options) {
      return _createRequiredValidator(options);
    }
  };
}()); // notice immediate invocation of anonymous function

/** Register the default factory provider function */
oj.Validation.__registerDefaultValidatorFactory(oj.ValidatorFactory.VALIDATOR_TYPE_REQUIRED,
                                                oj.RequiredValidatorFactory);

/**
 * A factory implementation that creates an instance of the built-in regExp validator of type
 * {@link oj.RegExpValidator}.
 *
 * @example <caption>create an instance of the regExp validator using the factory</caption>
 * var rvf = oj.Validation.validatorFactory(oj.ValidatorFactory.VALIDATOR_TYPE_REGEXP);
 * var usernameValidator = rvf.createValidator(
 *  {
 *    'pattern': '[a-zA-Z0-9]{3,}',
 *    'messageDetail': 'You must enter at least 3 letters or numbers'}
 *  });
 *
 * @name oj.RegExpValidatorFactory
 * @ojtsnoexport
 * @ojtsexportastype
 * @class
 * @public
 * @hideconstructor
 * @since 0.6
 *
 */
oj.RegExpValidatorFactory = (function () {
  function _createRegExpValidator(options) {
    return new oj.RegExpValidator(options);
  }

  return {
    /**
     * Creates an immutable validator instance of type {@link oj.RegExpValidator} that ensures the value
     * matches the provided pattern.
     *
     * @param {Object} options an object literal used to provide the pattern, an optional hint, error
     * message among others. See {@link oj.RegExpValidator} for details.<p>
     *
     *
     * @return {oj.RegExpValidator}
     * @memberOf oj.RegExpValidatorFactory
     * @ojsignature {target: "Type", for: "options", value: "oj.RegExpValidator.ValidatorOptions"}
     * @instance
     * @public
     */
    createValidator: function (options) {
      return _createRegExpValidator(options);
    }
  };
}()); // notice immediate invocation of anonymous function

/** Register the default factory provider function */
oj.Validation.__registerDefaultValidatorFactory(oj.ValidatorFactory.VALIDATOR_TYPE_REGEXP,
                                                oj.RegExpValidatorFactory);


/**
 * Returns an instance of oj.LengthValidatorFactory that provides a factory method to create an
 * instance of a length validator.
 *
 * @example <caption>create an instance of the length validator using the factory</caption>
 * var lvf = oj.Validation.validatorFactory(oj.ValidatorFactory.VALIDATOR_TYPE_LENGTH);
 * var options = {hint: {max: 'Enter {max} or fewer characters'}, max: 10};
 * var lValidator = lvf.createValidator(options);
 *
 * @name oj.LengthValidatorFactory
 * @hideconstructor
 * @ojtsnoexport
 * @ojtsexportastype
 * @class
 * @public
 * @since 0.6
 *
 */
oj.LengthValidatorFactory = (function () {
  function _createLengthValidator(options) {
    return new oj.LengthValidator(options);
  }

  return {
    /**
     * Creates an immutable validator instance of type oj.LengthValidator that ensures that the
     * value provided is withing a given length.
     *
     * @param {Object=} options an object literal used to provide the 'minimum', 'maximum' and other
     * optional values. See {@link oj.LengthValidator} for details.<p>
     *
     * @return {oj.LengthValidator}
     * @memberOf oj.LengthValidatorFactory
     * @ojsignature {target: "Type", for: "options", value: "oj.LengthValidator.ValidatorOptions"}
     * @instance
     * @public
     */
    createValidator: function (options) {
      return _createLengthValidator(options);
    }
  };
}()); // notice immediate invocation of anonymous function

/** Register the default factory provider function */
oj.Validation.__registerDefaultValidatorFactory(oj.ValidatorFactory.VALIDATOR_TYPE_LENGTH,
                                                oj.LengthValidatorFactory);

/* global LocaleData:false */
// LocaleData was refactored into it's own module which  module does not register anymore the LocaleData object into the oj namespace.
// But to remain backward compatible, bleed back LocaleData here.
oj.LocaleData = LocaleData;

var __ValidationBase = {};
__ValidationBase.ColorConverter = oj.ColorConverter;
__ValidationBase.IntlConverterUtils = oj.IntlConverterUtils;
__ValidationBase.LengthValidator = oj.LengthValidator;
__ValidationBase.RegExpValidator = oj.RegExpValidator;
__ValidationBase.RequiredValidator = oj.RequiredValidator;
__ValidationBase.Validation = oj.Validation;

  ;return __ValidationBase;
});