// Copyright (c) 2001, 2003, Oracle Corporation.  All rights reserved.  
// File: Sample_AdaptiveBayesNetworkApply.java

// Generic Java Imports
import java.io.*;
import java.sql.*;
import java.util.*;
import java.lang.*;
import oracle.jdbc.driver.*;
import oracle.sql.*;

// ODM Imports
import oracle.dmt.odm.ODMException;
import oracle.dmt.odm.DataMiningServer;
import oracle.dmt.odm.LocationAccessData;
import oracle.dmt.odm.InvalidArgumentException;
import oracle.dmt.odm.MiningObjectException;
import oracle.dmt.odm.DataType;
import oracle.dmt.odm.AttributeType;
import oracle.dmt.odm.Category;
import oracle.dmt.odm.ComparisonFunction;
import oracle.dmt.odm.DataFormatType;

import oracle.dmt.odm.data.PhysicalDataSpecification;
import oracle.dmt.odm.data.TransactionalDataSpecification;
import oracle.dmt.odm.data.NonTransactionalDataSpecification;
import oracle.dmt.odm.data.Attribute;
import oracle.dmt.odm.data.AttributeInstance;
import oracle.dmt.odm.data.MiningAttribute;

import oracle.dmt.odm.result.MiningApplyOutput;
import oracle.dmt.odm.result.ApplyRuleItem;
import oracle.dmt.odm.result.ApplySourceAttributeItem;
import oracle.dmt.odm.result.RecordInstance;
import oracle.dmt.odm.result.MiningApplyResult;
import oracle.dmt.odm.result.ApplyMultipleScoringItem;
import oracle.dmt.odm.result.ApplyTargetProbabilityItem;

import oracle.dmt.odm.rule.ABNMiningRule;
import oracle.dmt.odm.rule.CompoundPredicate;
import oracle.dmt.odm.rule.Predicate;
import oracle.dmt.odm.rule.PredicateRuleComponent;
import oracle.dmt.odm.rule.SimplePredicate;

import oracle.dmt.odm.model.SupervisedModel;
import oracle.dmt.odm.model.AdaptiveBayesNetworkModel;
import oracle.dmt.odm.task.MiningTask;
import oracle.dmt.odm.task.MiningApplyTask;
import oracle.dmt.odm.task.MiningTaskException;
import oracle.dmt.odm.task.MiningTaskStatus;

/**
 * This sample program demonstrates the use of ODM API to score a
 * previously built Adaptive Bayes Network model.
 *
 * SAMPLE PROGRAM DESIGN
 * 
 * ODM Sample Programs are designed to demonstrate the ODM functionaly; 
 * they can also be used as quick start programs to perform data mining operations
 * in the Oracle database.
 *
 * This program has two input property files:
 * 
 * 1. Sample_AdaptiveBayesNetworkApply.property
 * 2. Sample_Global.property
 * 
 *"Sample_AdaptiveBayesNetworkApply.property" specifies the inputs required
 * to perform model scoring functionality.
 *
 * "Sample_Global.property" file specifies the Data Mining Server connectivity
 * information.
 * 
 * EXECUTION REQUIRMENTS
 *
 * This program applies the model built by Sample_AdaptiveBayesNetworkBuild.java.
 */

public class Sample_AdaptiveBayesNetworkApply extends Object {
  // Tag: miningServer
  private static String m_MiningServerUrl; // URL for the DataMiningServer
  private static String m_MiningServerUserName; // User name for the DataMiningServer
  private static String m_MiningServerPassword; // Password for the DataMiningServer

  // Tag: inputDataSchemaName
  private static String m_InputDataSchemaName; // Schema where the input dataset is read from
  // Tag: outputDataSchemaName
  private static String m_outputSchemaName; // Schema where the output database object/s will be created

  // Tag: enableCleanup
  private static String m_EnableCleanup; // Indicates what kind of cleanup to be performed before running

  // Tag: applyTask
  private static String m_ModelName; // Model name to be used for Apply
  private static String m_ApplyTaskType; // Type of apply: record or table
  private static String m_TaskName; // Name of the apply mining task

  // Read only if m_ApplyTaskType = "record"
  // Tag: applyRecord
  private static String m_AttributeNames[]; // List of "," separated attribute names
  private static String m_AttributeValues[]; // List of "," separated attribute values
  private static String m_AttributeTypes[]; // List of "," separated attribute types

  // Read only if m_ApplyTaskType = "table"
  // Tag: applyInput
  private static DataFormatType m_ApplyInputDataType; // Transactional or nonTransactional data
  private static String m_ApplyInputDataTable; // Input table name
  private static String m_ApplyOutputOption; // Apply output option
  
  // Read only if m_ApplyTaskType = "table" AND applyInput.dataType = transactional
  // Tag: applyInput
  private static String m_ApplyInputDataSequenceId;
  private static String m_ApplyInputDataAttributeName;
  private static String m_ApplyInputDataValueName;

  // Read only if m_ApplyTaskType = "table/record"
  // Tag: miningApplyOutput
  private static String m_SourceAttributeNames[]; // List of "," separated attribute names
  private static String m_SourceAttributeAliases[]; // List of "," separated attribute alias
  private static String m_PredictionColumnName;
  private static String m_ProbabilityColumnName;
  private static String m_NumberOfPredictions;
  private static String m_TopToBottom;
  private static String m_RankAttribute;
  private static String m_RuleColumnName;
  private static String m_TargetDataType;
  private static String m_TargetDisplayNames[]; // List of "," separated target display names
  private static String m_TargetValues[]; // List of "," separated target values

  // Read only if m_ApplyTaskType = "table"
  // Tag: applyOutputResult
  private static String m_ApplyOutputResultTableName;
  private static String m_ApplyOutputResultResultName;

  // Following variables are not directly read from the property file:
  private static boolean m_IsApplyRecordType;
  private static boolean m_IsInputTransactional;
  private static PhysicalDataSpecification m_InputPhysicalDataSpecification;
  private static MiningApplyOutput m_MiningApplyOutput;
  private static RecordInstance m_record;

  /**
  **
  ** This is the code entry point
  **
  */
  public static void main ( String[] args ) {
    if ( args.length != 2 ) {
      System.out.println("Usage: java SampleAdaptiveBayesNetworkApply <Global Property file name> <Apply Property file name>");
      return;
    }
    System.out.println("*************************************************************");
    System.out.println("ODM 9.2.0.4 Sample Program: Sample_AdaptiveBayesNetworkApply");
    System.out.println("*************************************************************");
    System.out.println("Property Files:");
    System.out.println("    Global: " + args[0]);
    System.out.println("    Sample: " + args[1]);
    System.out.println("**********************************************************");
    System.out.println("Start: " + new java.util.Date());

    DataMiningServer dms = null;
    oracle.dmt.odm.Connection dmsConnection = null;
    try {
      // Initialize the class varaibles from the property file.
      initializeClassVariables(args[0],args[1]);

      System.out.println("\nInitialization Phase:");
      System.out.println("    Data Mining Server:");
      System.out.println("        JDBC URL: " + m_MiningServerUrl);
      System.out.println("        Username: " + m_MiningServerUserName);
      // Log into the MiningServer
      System.out.println("\nLogin Phase:");
      dms = new DataMiningServer(
        m_MiningServerUrl,
        m_MiningServerUserName,
        m_MiningServerPassword);
      dmsConnection = dms.login();
      System.out.println("    Completed MiningServer login");

      // Performs appropriate cleanup as indicated by enableCleanup flag
      boolean continueRun = performCleanup ( dmsConnection);
      if (continueRun == false) {
        // cleanup flag value = cleanupOnly
        System.out.println("Exiting before executing the program. Cleanup only");
        return;
      }

      // Invoke Apply
      if ( m_IsApplyRecordType == false) { // Table type
        // Create PhysicalDataSpecification for input table
        createPhysicalDataSpecification(dmsConnection);

        // Create MiningApplyOutput
        createMiningApplyOutput(dmsConnection);

        // LocationAccessData for output table
        LocationAccessData ladOutput =
          new LocationAccessData(m_ApplyOutputResultTableName,m_outputSchemaName);

        System.out.println("\nModel Apply Task Phase:");
        // Invoke SupervisedModel.apply
        System.out.println("    Invoking AdaptiveBayesNetwork Model apply");
        MiningApplyTask task = new MiningApplyTask(
          m_InputPhysicalDataSpecification, // Physical Data Spec for input data
          m_ModelName, // Name of the model to be applied
          m_MiningApplyOutput, // MiningApplyOutput
          ladOutput, // Output table location
          m_ApplyOutputResultResultName); // Result name in odm_apply_result table
        task.store(dmsConnection, m_TaskName);
        task.execute(dmsConnection);
        MiningTaskStatus status = task.waitForCompletion(dmsConnection);
        System.out.println("    Status: " + status.getTaskState().getEnum());
        long applyDuration = MiningApplyTask.getExecutionDuration(dmsConnection, m_TaskName)/1000;
        System.out.println("    Duration: " + applyDuration + " seconds");

        // Display Apply Results
        displayApplyResults(dmsConnection);

      } else { // RecordInstance
        // Create RecordInstance object
        createRecordInstance(dmsConnection);

        // Create MiningApplyOutput
        createMiningApplyOutput(dmsConnection);
        
        System.out.println("\nModel Apply Phase:");
        System.out.println("    Invoking AdaptiveBayesNetwork Model apply.");
        
        java.util.Date startTime = new java.util.Date();
        RecordInstance outRecord = AdaptiveBayesNetworkModel.apply(
                            dmsConnection, // MiningServer handle
                            m_record, // Record Instance
                            m_ModelName, // Name of the model to be applied
                            m_MiningApplyOutput);
        java.util.Date endTime = new java.util.Date();
        long applyDuration = (endTime.getTime() - startTime.getTime());
        System.out.println("    Duration: " + applyDuration + " milli seconds");

        // Display Record results
        System.out.println("\nDisplay Apply Record Result Phase:");
        int numberOfPredictions = 0;
        if (m_ApplyOutputOption.compareTo("multipleScoring") == 0)
          numberOfPredictions = Integer.parseInt(m_NumberOfPredictions);
        else if (m_ApplyOutputOption.compareTo("targetProbability") == 0)
          numberOfPredictions = m_TargetDisplayNames.length;
        displayApplyRecordResults(dmsConnection, outRecord, numberOfPredictions);
      }
    } catch ( Exception e) {
      System.out.println("    Received an exception in main: " + e.getLocalizedMessage());
    } finally {
      System.out.println("\nLogout Phase:");
      try {
        dms.logout(dmsConnection);
        System.out.println("    Completed MiningServer logout");
      } catch (Exception e) {
        System.out.println("    Failed to logout. Ingnoring the exception. Continuing ...");
      }
    }
    System.out.println("End: " + new java.util.Date());
    System.out.println("**********************************************************");
  }

  /**
  ** This function removes white space from both ends of the input property values
  **/
  public static Properties trimStrings(Properties inputProps) {
    Enumeration keys = inputProps.keys();
    Properties outputProps = new Properties();
    while(keys.hasMoreElements()) {
      String key = (String)keys.nextElement();
      String value = inputProps.getProperty(key);
      outputProps.put(key, value.trim());
    }
    return outputProps;
  }

  /**
  ** performCleanup
  ** Reads the value in m_EableCleanup and performs appropriate action.
  ** This function catches and ignores all the exceptions: after printing the exception.
  ** This function will not remove the scored output table.
  ** @param dmsConnection DataMiningServer handle
  **/
  public static boolean performCleanup (oracle.dmt.odm.Connection dmsConnection)
  {
    System.out.println("\nCleanup Phase:" + m_EnableCleanup);
    System.out.println("    Started Cleanup ");
    boolean continueExecution = true;
    if ( m_EnableCleanup.compareTo("cleanupOnly") == 0 ){
      continueExecution = false;
    }
    if ( m_EnableCleanup.compareTo("cleanupBeforeExecution") == 0 ||
      m_EnableCleanup.compareTo("cleanupOnly") == 0) {
      // Remove ODM objects created (identified by the current property file)
      try {
        // Remove apply result object
        MiningApplyResult.remove(dmsConnection,m_ApplyOutputResultResultName,true);
      } catch (Exception e) {}
      try {
        // Remove MiningTask object
        MiningTask.remove(dmsConnection, m_TaskName);
      } catch (Exception e) {}
    }
    System.out.println("    Ended Cleanup ");
    return continueExecution;
  }

  /**
  ** createPhysicalDataSpecification
  ** Create a Transactional or NonTransactional PhysicalDataSpecification object.
  ** Initialize the static variable m_InputPhysicalDataSpecification.
  ** @param dmsConnection DataMiningServer handle
  **/
  public static void createPhysicalDataSpecification (oracle.dmt.odm.Connection dmsConnection)
  throws InvalidArgumentException, ODMException
  {
    System.out.println("\nData Setup Creation Phase:");
    //  Make appropriate instance of PhysicalDataSpecification
    LocationAccessData lad = new LocationAccessData(m_ApplyInputDataTable, m_InputDataSchemaName);
    if ( m_IsInputTransactional == true ) {
      m_InputPhysicalDataSpecification = new TransactionalDataSpecification(
            m_ApplyInputDataSequenceId, // Column name for "sequence id"
            m_ApplyInputDataAttributeName, // Column name for "attribute name"
            m_ApplyInputDataValueName, // Column name for "value"
            lad);
      System.out.println("    Created Transactional PDS");
      System.out.println("        Sequence id: " + m_ApplyInputDataSequenceId);
      System.out.println("        Attribute name: " + m_ApplyInputDataAttributeName);
      System.out.println("        Value: " + m_ApplyInputDataValueName);
    } else {
      m_InputPhysicalDataSpecification = new NonTransactionalDataSpecification(lad);
      System.out.println("    Created NonTransactional PDS");
    }
    System.out.println("    Input schema: " + m_InputDataSchemaName);
    System.out.println("    Input table: " + m_ApplyInputDataTable);
  }

  private static String convertRuleToString(ABNMiningRule rule) 
  {
    PredicateRuleComponent antecedent = (PredicateRuleComponent)rule.getAntecedent();
    PredicateRuleComponent consequent = (PredicateRuleComponent)rule.getConsequent();
    StringBuffer buffer = new StringBuffer();
    Predicate pred = antecedent.getPredicate();
    buffer.append("If ");
    if ( pred instanceof CompoundPredicate ) {
      CompoundPredicate cpred = (CompoundPredicate)pred;
      String oper = cpred.getBooleanOperator().getEnum();
      Predicate[] preds = cpred.getPredicates();
      for (int i=0; i < preds.length; i++) {
        SimplePredicate spred = (SimplePredicate)preds[i];
        String comparison = spred.getCompareFunction().getEnum();
        Category[] cats = spred.getValue();
        StringBuffer valueBuf = new StringBuffer();
        //if (spred.getCompareFunction().isEqual(ComparisonFunction.in))
          valueBuf.append("(");
        for (int j = 0; j < cats.length; j++) {
           valueBuf.append(cats[j].getDisplayName());
           if (j < cats.length - 1)
             valueBuf.append(", ");
        }
        //if (spred.getCompareFunction().isEqual(ComparisonFunction.in))
          valueBuf.append(")");
        String v = spred.getItem().getName() + " " + comparison + " " + valueBuf.toString();
        buffer.append(v);
        if (i < preds.length - 1)
          buffer.append(" " + oper + " ");
      }
    }
    else if ( pred instanceof SimplePredicate ) {
      SimplePredicate spred = (SimplePredicate)pred;
      String comparison = spred.getCompareFunction().getEnum();
      Category[] cats = spred.getValue();
      StringBuffer valueBuf = new StringBuffer();
      //if (spred.getCompareFunction().isEqual(ComparisonFunction.in))
        valueBuf.append("(");
      for (int i = 0; i < cats.length; i++) {
        valueBuf.append(cats[i].getDisplayName());
        if (i < cats.length - 1)
          valueBuf.append(", ");
      }
      //if (spred.getCompareFunction().isEqual(ComparisonFunction.in))
        valueBuf.append(")");
      String v = spred.getItem().getName() + " " + comparison + " " + valueBuf.toString();
      buffer.append(v);
    }
    else
      buffer.append("Unexpected predicate in antecedent");

    buffer.append(", then ");
    pred = consequent.getPredicate();
    if ( pred instanceof SimplePredicate ) {
      SimplePredicate spred = (SimplePredicate)pred;
      String comparison = spred.getCompareFunction().getEnum();
      Category[] cats = spred.getValue();
      StringBuffer valueBuf = new StringBuffer();
      //if (spred.getCompareFunction().isEqual(ComparisonFunction.in))
        valueBuf.append("(");
      for (int i = 0; i < cats.length; i++) {
        valueBuf.append(cats[i].getDisplayName());
        if (i < cats.length - 1)
          valueBuf.append(", ");
      }
      //if (spred.getCompareFunction().isEqual(ComparisonFunction.in))
        valueBuf.append(")");
      String v = spred.getItem().getName() + " " + comparison + " " + valueBuf.toString();
      buffer.append(v);
    }
    else
      buffer.append("Unexpected predicate in consequent");

    return buffer.toString();
  }
  
  /** displayApplyRecordResults
  ** Display the record apply results.
  ** @param dmsConnection DataMiningServer handle
  ** @param outRecord RecordInstance returned by the apply 
  ** @param numberOfPredictions number of predictions returned by the apply
  **/
  public static void displayApplyRecordResults (oracle.dmt.odm.Connection dmsConnection, RecordInstance outRecord, int numberOfPredictions)
  throws InvalidArgumentException, MiningObjectException, SQLException, Exception
  {
    for (int i=0; i < numberOfPredictions; i++) {
      String predictionColumnName = m_PredictionColumnName;
      if (numberOfPredictions > 1)
        predictionColumnName += (i+1);
      AttributeInstance ai = outRecord.getValue(predictionColumnName);
      String prediction = null;
      if (ai.getDataType() == DataType.intType)
        prediction = new Integer(ai.getIntValue()).toString();
      else if (ai.getDataType() == DataType.floatType)
        prediction = new Float(ai.getFloatValue()).toString();
      else if (ai.getDataType() == DataType.stringType)
        prediction = ai.getStringValue();
      String probabilityColumnName = m_ProbabilityColumnName;
      if (numberOfPredictions > 1)
        probabilityColumnName += (i+1);
      ai = outRecord.getValue(probabilityColumnName);
      String probability = null;
      if (ai.getDataType() == DataType.intType)
        probability = new Integer(ai.getIntValue()).toString();
      else if (ai.getDataType() == DataType.floatType)
        probability = new Float(ai.getFloatValue()).toString();
      else if (ai.getDataType() == DataType.stringType)
        probability = ai.getStringValue();
      System.out.print("    " + m_PredictionColumnName + " = " + prediction);
      System.out.println(" with " + m_ProbabilityColumnName + " = " + probability);
      System.out.println("        because");
      ai = outRecord.getValue(m_RuleColumnName);
      if (ai != null) {
        int ruleId = ai.getIntValue();
        ABNMiningRule rule = AdaptiveBayesNetworkModel.getRuleById(dmsConnection, m_ModelName, ruleId);
        System.out.println("            " + convertRuleToString(rule));
      }
    }
  }
  
  /** displayApplyResults
  ** Display the MiningApplyResult object.
  ** @param dmsConnection DataMiningServer handle
  **/
  public static void displayApplyResults (oracle.dmt.odm.Connection dmsConnection)
  throws InvalidArgumentException, MiningObjectException, SQLException, Exception
  {
    System.out.println("\nDisplay Apply Table Result Phase:");
    MiningApplyResult miningApplyResult =
      MiningApplyResult.restore( dmsConnection, m_ApplyOutputResultResultName);

    if (miningApplyResult == null)
      throw new Exception("    Apply result: " + m_ApplyOutputResultResultName + " not found.");
    System.out.println(miningApplyResult.toString());
  }

  /**
  ** createRecordInstance
  ** Create a RecordInstance object.
  ** Initialize m_record class variable.
  ** @param dmsConnection DataMiningServer handle
  **/
  public static void createRecordInstance (oracle.dmt.odm.Connection dmsConnection)
  throws InvalidArgumentException, ODMException
  {
    System.out.println("\nRecordInstance Creation Phase:");
    m_record = new RecordInstance();
    for ( int i=0; i < m_AttributeNames.length; i++) {
      AttributeInstance temp = null;
      if (m_AttributeTypes[i].compareToIgnoreCase("int") == 0)
        temp = new AttributeInstance(m_AttributeNames[i], Integer.parseInt(m_AttributeValues[i]));
      else if (m_AttributeTypes[i].compareToIgnoreCase("string") == 0)
        temp = new AttributeInstance(m_AttributeNames[i], m_AttributeValues[i]);
      else if (m_AttributeTypes[i].compareToIgnoreCase("float") == 0)
        temp = new AttributeInstance(m_AttributeNames[i], Float.parseFloat( m_AttributeValues[i] ));
      m_record.addAttributeInstance(temp);
    }
    System.out.println("    Created RecordInstance");
  }

  /**
  ** createMiningApplyOutput
  ** Create a MiningApplyOutput object.
  ** Initialize the static variable m_MiningApplyOutput.
  ** @param dmsConnection DataMiningServer handle
  **/
  public static void createMiningApplyOutput (oracle.dmt.odm.Connection dmsConnection)
  throws InvalidArgumentException, MiningObjectException, ODMException
  {
    System.out.println("\nMiningApplyOutput Creation Phase:");
    m_MiningApplyOutput = new MiningApplyOutput();
    // Add all the source attributes.
    for (int i=0;i < m_SourceAttributeNames.length; i++) {
      MiningAttribute sourceAttribute =
        new MiningAttribute(m_SourceAttributeNames[i],DataType.intType,
        AttributeType.notApplicable);
      Attribute destinationAttribute =
        new Attribute(m_SourceAttributeAliases[i],DataType.intType);
      ApplySourceAttributeItem temp =
        new ApplySourceAttributeItem(sourceAttribute,destinationAttribute);
      m_MiningApplyOutput.addItem(temp);
      System.out.println("    Added source attributes");
    }
    if (m_RuleColumnName != null) {
      Attribute ruleAttribute = new Attribute(m_RuleColumnName, DataType.stringType);
      ApplyRuleItem temp = new ApplyRuleItem(ruleAttribute);
      m_MiningApplyOutput.addItem(temp);
      System.out.println("    Added detail rule attributes");
    }
    if (m_ApplyOutputOption.compareTo("multipleScoring") == 0) {
      Attribute predictionAttribute = new Attribute(m_PredictionColumnName, DataType.stringType);
      Attribute probabilityAttribute = new Attribute(m_ProbabilityColumnName, DataType.stringType);
      int numberOfPredictions = Integer.parseInt(m_NumberOfPredictions);
      boolean isFromTop = m_TopToBottom.equalsIgnoreCase("true");
      ApplyMultipleScoringItem temp =
        new ApplyMultipleScoringItem(predictionAttribute, probabilityAttribute, numberOfPredictions, isFromTop);
      m_MiningApplyOutput.addItem(temp);
      System.out.println("    Added multipleScoring attributes");
    }
    else if (m_ApplyOutputOption.compareTo("targetProbability") == 0) {
      Attribute predictionAttribute = new Attribute(m_PredictionColumnName, DataType.stringType);
      Attribute probabilityAttribute = new Attribute(m_ProbabilityColumnName, DataType.stringType);
      Attribute rankAttr = new Attribute(m_RankAttribute, DataType.stringType);
      ApplyTargetProbabilityItem aTargetAttrItem =
        new ApplyTargetProbabilityItem(predictionAttribute, probabilityAttribute, rankAttr);
      for (int i=0;i < m_TargetDisplayNames.length; i++) {
        Category target_category = new Category(
          m_TargetDisplayNames[i],
          m_TargetValues[i],
          DataType.getInstance(m_TargetDataType));
        aTargetAttrItem.addTarget(target_category);
      }
      m_MiningApplyOutput.addItem(aTargetAttrItem);
      System.out.println("    Added targetProbability attributes");
    }
    m_MiningApplyOutput.validate();
    System.out.println("    Created MiningApplyOutput");
  }

  /**
  ** Initialize all the static variables using the input property file.
  ** @param globalFileName Name of the global property file.
  ** @param fileName Name of the initialization property file.
  **/

 protected static void initializeClassVariables (String globalFileName,String fileName)
  throws FileNotFoundException, IOException, Exception {

     // Load ODM Global properties
    FileInputStream globalFileStream = new FileInputStream(globalFileName);
    Properties globalPropertyDetails = new Properties();
    globalPropertyDetails.load(globalFileStream);
    globalPropertyDetails = trimStrings(globalPropertyDetails);

    // Read MiningServer details.
    String tagName = "miningServer";
    m_MiningServerUrl = globalPropertyDetails.getProperty(tagName + ".url");
    m_MiningServerUserName = globalPropertyDetails.getProperty(tagName + ".userName");
    m_MiningServerPassword = globalPropertyDetails.getProperty(tagName + ".password");

    // Read input data schema name.
    m_InputDataSchemaName = globalPropertyDetails.getProperty("inputDataSchemaName");
    m_outputSchemaName = globalPropertyDetails.getProperty("outputSchemaName");

    // Load operation specific properties.
    FileInputStream inputFileStream = new FileInputStream(fileName);
    Properties propertyDetails = new Properties();
    propertyDetails.load(inputFileStream);
    propertyDetails = trimStrings(propertyDetails);

    // Read cleanup option
    m_EnableCleanup = propertyDetails.getProperty("enableCleanup");
    if ( m_EnableCleanup == null || m_EnableCleanup.length() == 0 )
      m_EnableCleanup = "cleanupOnly";
    tagName = "apply";
    m_ModelName = propertyDetails.getProperty(tagName + ".modelName");
    m_TaskName = propertyDetails.getProperty(tagName + ".miningTaskName");
    m_ApplyTaskType = propertyDetails.getProperty(tagName + ".type");
    if ( m_ApplyTaskType.compareToIgnoreCase("table") == 0 ) {
      m_IsApplyRecordType = false;
    } else if ( m_ApplyTaskType.compareToIgnoreCase("record") == 0 ) {
      m_IsApplyRecordType = true;
    } else
      throw new Exception("apply.type can only be of record or table");
      
    m_ApplyOutputOption = propertyDetails.getProperty("applyInput.ApplyOutputOption");
    // Read sourceAttribute
    tagName = "miningApplyOutput.sourceAttribute";
    String allNames = propertyDetails.getProperty(tagName + ".sourceAttributeNames");
    StringTokenizer attributeNamesToken = new StringTokenizer(allNames, ",");
    int numAttributes = attributeNamesToken.countTokens();
    m_SourceAttributeNames  = new String[numAttributes];
    for(int i=0; i < numAttributes; i++) {
      m_SourceAttributeNames[i] = attributeNamesToken.nextToken();
    }
    allNames = propertyDetails.getProperty(tagName + ".sourceAttributeAliases");
    StringTokenizer attributeAliasesToken = new StringTokenizer(allNames, ",");
    numAttributes = attributeAliasesToken.countTokens();
    m_SourceAttributeAliases  = new String[numAttributes];
    for(int i=0; i < numAttributes; i++) {
      m_SourceAttributeAliases[i] = attributeAliasesToken.nextToken();
    }

    // Read rule
    tagName = "miningApplyOutput.rule";
    m_RuleColumnName = propertyDetails.getProperty(tagName + ".ruleColumnName");

    // Read multipleScoring
    tagName = "miningApplyOutput.multipleScoring";
    m_PredictionColumnName = propertyDetails.getProperty(tagName + ".predictionColumnName");
    m_ProbabilityColumnName = propertyDetails.getProperty(tagName + ".probabilityColumnName");
    m_NumberOfPredictions = propertyDetails.getProperty(tagName + ".numberOfPredictions");
    m_TopToBottom = propertyDetails.getProperty(tagName + ".topToBottom");

    // Read targetProbability
    tagName = "miningApplyOutput.targetProbability";
    m_PredictionColumnName = propertyDetails.getProperty(tagName + ".predictionColumnName");
    m_ProbabilityColumnName = propertyDetails.getProperty(tagName + ".probabilityColumnName");
    m_RankAttribute = propertyDetails.getProperty(tagName + ".rankAttribute");
    m_TargetDataType = propertyDetails.getProperty(tagName + ".targetDataType");
    allNames = propertyDetails.getProperty(tagName + ".targetDisplayNames");
    StringTokenizer targetDisplayNamesToken = new StringTokenizer(allNames, ",");
    int numTargetDisplayNames = targetDisplayNamesToken.countTokens();
    m_TargetDisplayNames  = new String[numTargetDisplayNames];
    for(int i=0; i < numTargetDisplayNames; i++) {
      m_TargetDisplayNames[i] = targetDisplayNamesToken.nextToken();
    }
    allNames = propertyDetails.getProperty(tagName + ".targetValues");
    StringTokenizer targetValuesToken = new StringTokenizer(allNames, ",");
    int numTargetValues = targetValuesToken.countTokens();
    m_TargetValues  = new String[numTargetValues];
    for(int i=0; i < numTargetValues; i++) {
      m_TargetValues[i] = targetValuesToken.nextToken();
    }
      
    // If the apply type is record
    if ( m_IsApplyRecordType == true) {
      tagName = "applyRecord";
      // Read AttributeNames
      allNames = propertyDetails.getProperty(tagName + ".attributeNames");
      attributeNamesToken = new StringTokenizer(allNames,",");
      numAttributes = attributeNamesToken.countTokens();
      m_AttributeNames  = new String[numAttributes];
      for(int i=0; i < numAttributes; i++) {
        m_AttributeNames[i] = attributeNamesToken.nextToken();
      }
      // Read AttributeValues
      String allValues = propertyDetails.getProperty(tagName + ".attributeValues");
      StringTokenizer attributeValuesToken = new StringTokenizer(allValues,",");
      int numValues = attributeValuesToken.countTokens();
      if ( numAttributes != numValues) {
        throw new Exception("Number of Attribute names should match the number of values.");
      }
      m_AttributeValues = new String[numValues];
      for(int i=0; i < numValues; i++) {
        m_AttributeValues[i] = attributeValuesToken.nextToken();
      }
      // Read AttributeTypes
      String allTypes = propertyDetails.getProperty(tagName + ".attributeTypes");
      StringTokenizer attributeTypesToken = new StringTokenizer(allTypes,",");
      int numTypes = attributeTypesToken.countTokens();
      if ( numAttributes != numTypes) {
        throw new Exception("Number of Attribute names should match the number of types.");
      }
      m_AttributeTypes = new String[numTypes];
      for(int i=0; i < numTypes; i++) {
        String attrType = attributeTypesToken.nextToken();
        if( attrType.compareToIgnoreCase( "int" ) != 0 &&
            attrType.compareToIgnoreCase( "float" ) != 0 &&
            attrType.compareToIgnoreCase( "string" ) != 0 )
          throw new Exception( "Invalid attribute data type " + attrType );
        m_AttributeTypes[i] = attrType;
      }
    } else {
      tagName = "applyInput";
      String dataType = propertyDetails.getProperty(tagName + ".type");
      if ( dataType==null || dataType.length() == 0 )
        throw new Exception ( "applyInput.type should be transactional or nonTransactional.");
      m_ApplyInputDataType = DataFormatType.getInstance(dataType);

      if ( m_ApplyInputDataType.isEqual(DataFormatType.transactional) ) {
        // Transactional input data
        m_IsInputTransactional = true;
        m_ApplyInputDataSequenceId = propertyDetails.getProperty(tagName + ".sequenceId");
        m_ApplyInputDataAttributeName = propertyDetails.getProperty(tagName + ".attributeName");
        m_ApplyInputDataValueName = propertyDetails.getProperty(tagName + ".valueName");
      } else if ( m_ApplyInputDataType.isEqual(DataFormatType.nonTransactional) ) {
        // Non transactional input data
        m_IsInputTransactional = false;
      } else
        throw new Exception ( "applyInput.type should be transactional or nonTransactional.");
        
      m_ApplyInputDataTable = propertyDetails.getProperty(tagName + ".tableName");

      // Read ApplyResult details
      tagName = "applyOutputResult";
      m_ApplyOutputResultTableName = propertyDetails.getProperty(tagName + ".tableName");
      m_ApplyOutputResultResultName = propertyDetails.getProperty(tagName + ".resultName");
    }
  }
}
