// Copyright (c) 2001, 2003, Oracle Corporation.  All rights reserved.  
// File: Sample_AdaptiveBayesNetworkBuild.java

// Generic Java Imports
import java.io.*;
import java.util.*;
import java.sql.SQLException;

// ODM Imports
import oracle.dmt.odm.ABNModelBuildType;
import oracle.dmt.odm.AttributeUsage;
import oracle.dmt.odm.Category;
import oracle.dmt.odm.CategoryMatrix;
import oracle.dmt.odm.DataType;
import oracle.dmt.odm.DataMiningServer;
import oracle.dmt.odm.ODMException;
import oracle.dmt.odm.PriorProbabilities;
import oracle.dmt.odm.LocationAccessData;
import oracle.dmt.odm.InvalidArgumentException;
import oracle.dmt.odm.MiningObjectException;
import oracle.dmt.odm.AttributeType;
import oracle.dmt.odm.DataPreparationStatus;
import oracle.dmt.odm.DataFormatType;
import oracle.dmt.odm.MiningTaskState;

import oracle.dmt.odm.data.PhysicalDataSpecification;
import oracle.dmt.odm.data.TransactionalDataSpecification;
import oracle.dmt.odm.data.NonTransactionalDataSpecification;
import oracle.dmt.odm.data.LogicalDataSpecification;

import oracle.dmt.odm.settings.algorithm.AdaptiveBayesNetworkSettings;

import oracle.dmt.odm.settings.function.ClassificationFunctionSettings;

import oracle.dmt.odm.model.MiningModel;
import oracle.dmt.odm.model.AdaptiveBayesNetworkModel;

import oracle.dmt.odm.data.MiningAttribute;

import oracle.dmt.odm.rule.ABNMiningRule;
import oracle.dmt.odm.rule.CompoundPredicate;
import oracle.dmt.odm.rule.MiningRuleSet;
import oracle.dmt.odm.rule.Predicate;
import oracle.dmt.odm.rule.PredicateRuleComponent;
import oracle.dmt.odm.rule.SimplePredicate;
import oracle.dmt.odm.rule.TargetItem;

import oracle.dmt.odm.task.MiningTask;
import oracle.dmt.odm.task.MiningBuildTask;
import oracle.dmt.odm.task.MiningTaskException;
import oracle.dmt.odm.task.MiningTaskStatus;

/**
 * This sample program demonstrates the use of the ODM API to build an Adaptive
 * Bayes Network model.
 *
 * SAMPLE PROGRAM DESIGN
 * 
 * ODM Sample Programs are designed to demonstrate the ODM functionality;
 * they can also be used as quick start programs to perform data mining operations
 * in the Oracle database.
 * 
 * This program has two input property files:
 * 
 * 1. Sample_AdaptiveBayesNetworkBuild.property
 * 2. Sample_Global.property
 * 
 * "Sample_AdaptiveBayesNetworkBuild.property" file specifies the inputs required
 * to perform model build.
 * 
 * "Sample_Global.property" file specifies the Data Mining Server connectivity
 * information.
 *
 * EXECUTION REQUIRMENTS
 *
 * This program can be executed independently from the other sample programs.
 */

public class Sample_AdaptiveBayesNetworkBuild extends Object {
 // Tag: miningServer
  private static String m_MiningServerUrl; // URL for the DataMiningServer
  private static String m_MiningServerUserName; // User name for the DataMiningServer
  private static String m_MiningServerPassword; // Password for the DataMiningServer

  // Tag: inputDataSchemaName
  private static String m_InputDataSchemaName; // Schema where the input dataset is read from

  // Tag: enableCleanup
  private static String m_EnableCleanup; // Indicates what kind of cleanup to be performed before running

  // Tag: buildData
  private static DataFormatType m_BuildDataType; // Transactional or non transactional
  private static String m_BuildDataTableName; // Data to build the model with

  // Tag: transactionalData
  // Valid only for transactional data
  private static String m_SequenceIdName; // Name of the "sequence id" column
  private static String m_AttributeName; // Name of the "attribute name" column
  private static String m_ValueName; // Name of the "value" column

  // Tag: categoryMatrix
  private static String[] m_cmTargetDisplayValues; // array of target display names
  private static String[] m_cmTargetValues; // array of target values
  private static String m_cmTargetDataType; // target type
  private static String[] m_cmCostValues; // costs in the cost matrix

  // Tag: priorProbabilities
  private static String[] m_targetDisplayValues; // array of target display names
  private static String[] m_targetValues; // array of target values
  private static String[] m_targetProbabilities; // target probabilities
  private static String m_targetDataType; // target type

  // Tag: adaptiveBayesNetwork
  // Adaptive Bayes Network function settings
  private static int m_treeDepth = 0; // Maximum depth of any NetworkFeature
  //private static int m_numOfNetworkFeature = 0; // Maximum number of seed NetworkFeatures trees that are candidates for extension - deprecated in 10i
  private static int m_numOfConsecutivePrunedNF = 0; // Maximum number of consecutive pruned features before halting the stepwise selection process
  private static int m_maxBuildTime = 0; // Maximum build time
  private static int m_maxPredictors = 0; // Maximum number of predictors chosen during predictor selection phase
  private static int m_NBNumberPredictors = 0; // Number of predictors in the NB model
  private static int m_modelBuildType = 0; // Model build type - multiFeatureBuild

  private static String m_targetAttrName; // Name of the target attribute
  private static DataPreparationStatus m_defaultDataPrep; // Attribute default data preparation status
  private static String[] m_inactiveAttributeNames; // List of inactive attriibutes
  
  // Tag: adaptiveBayesNetworkOutput
  // Output Names
  private static String m_MiningSettingsName; // Name of the MiningSettings object
  private static String m_TaskName; // Name of the build mining task
  private static String m_ModelName; // Name of the Association Rules model

  // Following variables are not directly read from the property file
  private static boolean m_IsTransactional; // whether the build dataset is 2D or transaction format
  private static PhysicalDataSpecification m_PhysicalDataSpecification;
  private static ClassificationFunctionSettings m_ClassificationFunctionSettings;
  private static CategoryMatrix m_costMatrix = null;
  private static PriorProbabilities m_priorProbability = null;

  /**
  **
  ** This is the code entry point
  **
  */
  public static void main ( String[] args ) {
    if ( args.length != 2 ) {
      System.out.println("Usage: java Sample_AdaptiveBayesNetworkBuild <Global Property file name> <Build Property file name>");
      return;
    }
    System.out.println("*************************************************************");
    System.out.println("ODM 9.2.0.4 Sample Program: Sample_AdaptiveBayesNetworkBuild");
    System.out.println("*************************************************************");
    System.out.println("Property Files:");
    System.out.println("    Global: " + args[0]);
    System.out.println("    Sample: " + args[1]);
    System.out.println("**********************************************************");
    System.out.println("Start: " + new java.util.Date());

    DataMiningServer dms = null;
    oracle.dmt.odm.Connection dmsConnection = null;
    try {
      // Initialize the class varaibles from the property file.
      initializeClassVariables(args[0],args[1]);

      System.out.println("\nInitialization Phase:");
      System.out.println("    Data Mining Server:");
      System.out.println("        JDBC URL: " + m_MiningServerUrl);
      System.out.println("        Username: " + m_MiningServerUserName);
      // Log into the MiningServer
      System.out.println("\nLogin Phase:");
      dms = new DataMiningServer(
        m_MiningServerUrl,
        m_MiningServerUserName,
        m_MiningServerPassword);
      dmsConnection = dms.login();
      System.out.println("    Completed MiningServer login");

      // Performs appropriate cleanup - as indicated by enableCleanup flag
      boolean continueRun = performCleanup ( dmsConnection);
      if (continueRun == false) {
        // cleanup flag value = cleanupOnly
        System.out.println("Exiting before executing the program. Cleanup only");
        return;
      }

      // Create PhysicalDataSpecification object.
      createPhysicalDataSpecification(dmsConnection);

      // Creates MiningFunctionSettings object
      createMiningFunctionSettings(dmsConnection);

      // Invoke the build function.
      boolean success = buildAdaptiveBayesNetworkModel(dmsConnection);

      if (success)
        displayAdaptiveBayesNetworkRules(dmsConnection);
      
    } catch ( Exception e) {
      System.out.println("    Received an exception in main: " + e.getMessage());
    } finally {
      System.out.println("\nLogout Phase:");
      try {
        dms.logout(dmsConnection);
        System.out.println("    Completed MiningServer logout");
      } catch (Exception e) {
        System.out.println("    Failed to logout. Ingnoring the exception. Continuing ...");
      }
    }
    System.out.println("End: " + new java.util.Date());
    System.out.println("**********************************************************");
  }

  /**
  ** This function removes white space from both ends of the input property values
  **/
  public static Properties trimStrings(Properties inputProps) {
    Enumeration keys = inputProps.keys();
    Properties outputProps = new Properties();
    while(keys.hasMoreElements()) {
      String key = (String)keys.nextElement();
      String value = inputProps.getProperty(key);
      outputProps.put(key, value.trim());
    }
    return outputProps;
  }

  /**
  ** performCleanup
  ** Reads the value in m_EableCleanup and performs appropriate action.
  ** This function catches and ignores all the exceptions: after printing the exception.
  ** @param dmsConnection DataMiningServer handle
  **/
  public static boolean performCleanup (oracle.dmt.odm.Connection dmsConnection)
  {
    
    System.out.println("\nCleanup Phase:" + m_EnableCleanup);
    System.out.println("    Started Cleanup ");

    boolean continueExecution = true;
    if ( m_EnableCleanup.compareTo("cleanupOnly") == 0 ){
      continueExecution = false;
    }
    if ( m_EnableCleanup.compareTo("cleanupBeforeExecution") == 0 ||
      m_EnableCleanup.compareTo("cleanupOnly") == 0) {
      // Remove ODM objects created (identified by the current property file)
      try {
        // Remove MiningFunctionSettings object
        ClassificationFunctionSettings.remove(dmsConnection,m_MiningSettingsName);
      } catch (Exception e) {}
      try {
        // Remove MiningModel object
        MiningModel.remove(dmsConnection,m_ModelName);
      } catch (Exception e) {}
      try {
        // Remove MiningTask object
        MiningTask.remove(dmsConnection, m_TaskName);
      } catch (Exception e) {}
    }
    System.out.println("    Ended Cleanup ");    
    return continueExecution;
  }

  /**
  ** createPhysicalDataSpecification
  ** Create a Transactional or NonTransactional PhysicalDataSpecification object.
  ** Initializes the static variable m_PhysicalDataSpecification.
  ** @param dmsConnection DataMiningServer handle
  **/
  public static void createPhysicalDataSpecification (oracle.dmt.odm.Connection dmsConnection)
  throws InvalidArgumentException, ODMException
  {
    System.out.println("\nData Setup Creation Phase:");
    //  Make appropriate instance of PhysicalDataSpecification
    LocationAccessData lad = new LocationAccessData(m_BuildDataTableName, m_InputDataSchemaName);
    if ( m_IsTransactional == true ) {
      m_PhysicalDataSpecification = new TransactionalDataSpecification(
            m_SequenceIdName, // Column name for "sequence id"
            m_AttributeName, // Column name for "attribute name"
            m_ValueName, // Column name for "value"
            lad);
      System.out.println("    Created Transactional PDS");
      System.out.println("        Sequence id: " + m_SequenceIdName);
      System.out.println("        Attribute name: " + m_AttributeName);
      System.out.println("        Value: " + m_ValueName);
    } else {
      m_PhysicalDataSpecification = new NonTransactionalDataSpecification(lad);
      System.out.println("    Created NonTransactional PDS");
    }
    System.out.println("    Input schema: " + m_InputDataSchemaName);
    System.out.println("    Input table: " + m_BuildDataTableName);
  }

  /**
   * This method is used to tokenize the string
   */
  protected static String[] getStringTokens(String str, String delimiter) {
    StringTokenizer strTokenizer = new StringTokenizer( str, delimiter );
    int noOfTokens = strTokenizer.countTokens();
    String[] strTokens = new String[noOfTokens];
    for(int iToken=0; iToken < noOfTokens; iToken++) {
      strTokens[iToken] = strTokenizer.nextToken();
    }
    return strTokens;
  }
  
  /**
  ** createCostMatrix
  ** Make a CostMatrix object.
  ** Initialize the static variable m_costMatrix.
  ** @param dmsConnection DataMiningServer handle
  */
  public static void createCostMatrix (oracle.dmt.odm.Connection dmsConnection)
  throws InvalidArgumentException, ODMException
  {
    if (m_cmTargetValues != null) {
      //Build category matrix
      m_costMatrix = new CategoryMatrix();
      int i = 0;
      for(int iRow=0; iRow < m_cmTargetValues.length; iRow++) {
        Category actualCat = new Category(
          m_cmTargetDisplayValues[iRow],
          m_cmTargetValues[iRow],
          DataType.getInstance( m_cmTargetDataType ) );
        for(int iCol=0; iCol < m_cmTargetValues.length; iCol++) {
          Category predictedCat = new Category(
            m_cmTargetDisplayValues[iCol],
            m_cmTargetValues[iCol],
            DataType.getInstance( m_cmTargetDataType ) );
          float costValue = Float.parseFloat( m_cmCostValues[i++] );
          m_costMatrix.addEntry(actualCat, predictedCat, costValue);
        }
      }
    }
  }

  /**
  ** createPriorProbabilities
  ** Make a PriorProbabilities object.
  ** Initialize the static variable m_priorProbability.
  ** @param dmsConnection DataMiningServer handle
  */
  public static void createPriorProbabilities (oracle.dmt.odm.Connection dmsConnection)
  throws InvalidArgumentException, ODMException
  {
    if (m_targetValues != null) {
      m_priorProbability = new PriorProbabilities();
      for (int i=0; i < m_targetValues.length; i++) {
        Category probCat = new Category(
          m_targetDisplayValues[i],
          m_targetValues[i],
          DataType.getInstance( m_targetDataType ) );
        float probValue = Float.parseFloat( m_targetProbabilities[i] );
        m_priorProbability.addEntry(probCat, probValue);
      }
    }
  }

  /**
  ** createMiningFunctionSettings
  ** Make a MiningFunctionSettings object.
  ** Initialize the static variable m_AssociationRulesFunctionSettings.
  ** @param dmsConnection DataMiningServer handle
  */
  public static void createMiningFunctionSettings (oracle.dmt.odm.Connection dmsConnection)
  throws InvalidArgumentException, MiningObjectException,
    SQLException, IOException, ODMException
  {
    System.out.println("\nMining Settings Creation Phase:");
    // Create and store AssociationRulesFunctionSettings.
    AdaptiveBayesNetworkSettings algorithmSetting = new AdaptiveBayesNetworkSettings();
    if (m_treeDepth != 0)
      algorithmSetting.setMaximumNetworkFeatureDepth(m_treeDepth);
    if (m_numOfConsecutivePrunedNF != 0)
      algorithmSetting.setMaximumConsecutivePrunedNetworkFeatures(m_numOfConsecutivePrunedNF);
    if (m_maxPredictors != 0)
      algorithmSetting.setMaximumPredictors(m_maxPredictors);
    if (m_NBNumberPredictors != 0)
      algorithmSetting.setNaiveBayesNumberPredictors(m_NBNumberPredictors);
    if (m_maxBuildTime != 0)
      algorithmSetting.setMaximumBuildTime(m_maxBuildTime);
    if (m_modelBuildType != 0)
      algorithmSetting.setModelBuildType(ABNModelBuildType.getInstance(m_modelBuildType));
    m_ClassificationFunctionSettings = ClassificationFunctionSettings.create(
      dmsConnection,
      algorithmSetting,
      m_PhysicalDataSpecification,
      m_targetAttrName,
      AttributeType.categorical,
      m_defaultDataPrep);
    if (m_inactiveAttributeNames != null) {
      m_ClassificationFunctionSettings.adjustAttributeUsage(
        m_inactiveAttributeNames,
        AttributeUsage.inactive);
      System.out.println("    Added inactive attribute(s)");
    }
    createCostMatrix(dmsConnection);
    if (m_costMatrix != null) {
      m_ClassificationFunctionSettings.setCostMatrix(m_costMatrix);
      System.out.println("    Added CostMatrix");
    }
    createPriorProbabilities(dmsConnection);
    if (m_priorProbability != null) {
      m_ClassificationFunctionSettings.setPriors(m_priorProbability);
      System.out.println("    Added Prior");
    }
    m_ClassificationFunctionSettings.validate();
    m_ClassificationFunctionSettings.store(dmsConnection,m_MiningSettingsName);
    System.out.println("    Persisted MiningFunctionSettings");
    System.out.println("    Name: " + m_MiningSettingsName);
  }

  private static String convertRuleToString(ABNMiningRule rule) 
  {
    PredicateRuleComponent antecedent = (PredicateRuleComponent)rule.getAntecedent();
    PredicateRuleComponent consequent = (PredicateRuleComponent)rule.getConsequent();
    StringBuffer buffer = new StringBuffer();
    Predicate pred = antecedent.getPredicate();
    buffer.append("If ");
    if ( pred instanceof CompoundPredicate ) {
      CompoundPredicate cpred = (CompoundPredicate)pred;
      String oper = cpred.getBooleanOperator().getEnum();
      Predicate[] preds = cpred.getPredicates();
      for (int i=0; i < preds.length; i++) {
        SimplePredicate spred = (SimplePredicate)preds[i];
        String comparison = spred.getCompareFunction().getEnum();
        Category[] cats = spred.getValue();
        StringBuffer valueBuf = new StringBuffer();
        valueBuf.append("(");
        for (int j = 0; j < cats.length; j++) {
           valueBuf.append(cats[j].getDisplayName());
           if (j < cats.length - 1)
             valueBuf.append(", ");
        }
        valueBuf.append(")");
        String v = spred.getItem().getName() + " " + comparison + " " + valueBuf.toString();
        buffer.append(v);
        if (i < preds.length - 1)
          buffer.append(" " + oper + " ");
      }
    }
    else if ( pred instanceof SimplePredicate ) {
      SimplePredicate spred = (SimplePredicate)pred;
      String comparison = spred.getCompareFunction().getEnum();
      Category[] cats = spred.getValue();
      StringBuffer valueBuf = new StringBuffer();
      valueBuf.append("(");
      for (int i = 0; i < cats.length; i++) {
        valueBuf.append(cats[i].getDisplayName());
        if (i < cats.length - 1)
          valueBuf.append(", ");
      }
      valueBuf.append(")");
      String v = spred.getItem().getName() + " " + comparison + " " + valueBuf.toString();
      buffer.append(v);
    }
    else
      buffer.append("Unexpected predicate in antecedent");

    buffer.append(", then ");
    pred = consequent.getPredicate();
    if ( pred instanceof SimplePredicate ) {
      SimplePredicate spred = (SimplePredicate)pred;
      String comparison = spred.getCompareFunction().getEnum();
      Category[] cats = spred.getValue();
      StringBuffer valueBuf = new StringBuffer();
      valueBuf.append("(");
      for (int i = 0; i < cats.length; i++) {
        valueBuf.append(cats[i].getDisplayName());
        if (i < cats.length - 1)
          valueBuf.append(", ");
      }
      valueBuf.append(")");
      String v = spred.getItem().getName() + " " + comparison + " " + valueBuf.toString();
      buffer.append(v);
    }
    else
      buffer.append("Unexpected predicate in consequent");

    return buffer.toString();
  }
  
  /**
  ** Displays the rules using variaous interfaces available
  ** This method uses static variables as inputs. These static
  ** variables must be initialized before this function is invoked.
  ** @param dmsConnection DataMiningServer handle
  **/
  public static void displayAdaptiveBayesNetworkRules (
    oracle.dmt.odm.Connection dmsConnection)
  throws SQLException, InvalidArgumentException,
    MiningObjectException, ODMException
  {
    // Get all rules.
    System.out.println("\nDisplay Result Phase:");
    MiningRuleSet ruleSet = AdaptiveBayesNetworkModel.getRules(dmsConnection, m_ModelName);
    if (ruleSet != null) {
      ABNMiningRule rule;
      while ((rule = (ABNMiningRule)ruleSet.getNextRule()) != null) {
        System.out.println("    Rule Id: "+rule.getRuleId());
        System.out.println("    Support: "+rule.getSupport());
        TargetItem[] items = rule.getTargetItems();
        for (int i=0; i < items.length; i++)
          System.out.println("    Prob("+items[i].getTarget().getDisplayName()+") = "+items[i].getProbability()+", count = "+items[i].getCount());
        System.out.println("    Rule: "+convertRuleToString(rule));
        System.out.println("\n");
      }
    }
    else {
      System.out.println("    No rule is generated.");      
    }
  }

  /**
  ** Initialize all the static variables using the input property file.
  ** @param fileName Name of the initialization property file.
  **/
  protected static void initializeClassVariables (String globalFileName, String fileName)
  throws FileNotFoundException, IOException, Exception {
    // Load ODM Global properties
    FileInputStream globalFileStream = new FileInputStream(globalFileName);
    Properties globalPropertyDetails = new Properties();
    globalPropertyDetails.load(globalFileStream);
    globalPropertyDetails = trimStrings(globalPropertyDetails);

    // Read MiningServer details.
    String tagName = "miningServer";
    m_MiningServerUrl = globalPropertyDetails.getProperty(tagName + ".url");
    m_MiningServerUserName = globalPropertyDetails.getProperty(tagName + ".userName");
    m_MiningServerPassword = globalPropertyDetails.getProperty(tagName + ".password");

    // Read input data schema name.
    m_InputDataSchemaName = globalPropertyDetails.getProperty("inputDataSchemaName");

    // Load operation specific properties.
    FileInputStream inputFileStream = new FileInputStream(fileName);
    Properties propertyDetails = new Properties();
    propertyDetails.load(inputFileStream);
    propertyDetails = trimStrings(propertyDetails);

    // Read cleanup option
    m_EnableCleanup = propertyDetails.getProperty("enableCleanup");
    if ( m_EnableCleanup == null || m_EnableCleanup.length() == 0 )
      m_EnableCleanup = "cleanupOnly";

    // Read build data details.
    tagName = "buildData";
    m_BuildDataTableName = propertyDetails.getProperty(tagName + ".tableName");

    String dataType = propertyDetails.getProperty(tagName + ".type");
    if ( dataType==null || dataType.length() == 0 )
      throw new Exception ( "applyInput.type should be transactional or nonTransactional.");
    m_BuildDataType = DataFormatType.getInstance(dataType);

    if ( m_BuildDataType.isEqual(DataFormatType.transactional) ) {
      m_IsTransactional = true;
      // Read transactional data specifications.
      tagName = "transactionalData";
      m_SequenceIdName = propertyDetails.getProperty(tagName + ".sequenceId");
      m_AttributeName = propertyDetails.getProperty(tagName + ".attributeName");
      m_ValueName = propertyDetails.getProperty(tagName + ".valueName");
    } else if ( m_BuildDataType.isEqual(DataFormatType.nonTransactional) ) {
      m_IsTransactional = false;
    } else {
      throw new Exception("Invalid build data type. Valid values are: transactional/nonTransactional.");
    }

    // Read cost matrix details.
    tagName = "categoryMatrix";
    String allNames = propertyDetails.getProperty(tagName + ".targetDisplayValues");
    if (allNames != null) {
      StringTokenizer targetDisplayValuesToken = new StringTokenizer(allNames, ",");
      int numTargetDisplayValues = targetDisplayValuesToken.countTokens();
      m_cmTargetDisplayValues  = new String[numTargetDisplayValues];
      for(int i=0; i < numTargetDisplayValues; i++) {
        m_cmTargetDisplayValues[i] = targetDisplayValuesToken.nextToken();
      }
    }
    allNames = propertyDetails.getProperty(tagName + ".targetValues");
    if (allNames != null) {
      StringTokenizer targetValuesToken = new StringTokenizer(allNames, ",");
      int numTargetValues = targetValuesToken.countTokens();
      m_cmTargetValues  = new String[numTargetValues];
      for(int i=0; i < numTargetValues; i++) {
        m_cmTargetValues[i] = targetValuesToken.nextToken();
      }
    }
    m_cmTargetDataType = propertyDetails.getProperty(tagName + ".targetDataType");
    allNames = propertyDetails.getProperty(tagName + ".costValues");
    if (allNames != null) {
      StringTokenizer costValuesToken = new StringTokenizer(allNames, ",");
      int numCostValues = costValuesToken.countTokens();
      m_cmCostValues  = new String[numCostValues];
      for(int i=0; i < numCostValues; i++) {
        m_cmCostValues[i] = costValuesToken.nextToken();
      }
    }

    // Read Prior Probability details.
    tagName = "priorProbabilities";
    allNames = propertyDetails.getProperty(tagName + ".targetDisplayValues");
    if (allNames != null) {
      StringTokenizer targetDisplayValuesToken = new StringTokenizer(allNames, ",");
      int numTargetDisplayValues = targetDisplayValuesToken.countTokens();
      m_targetDisplayValues  = new String[numTargetDisplayValues];
      for(int i=0; i < numTargetDisplayValues; i++) {
        m_targetDisplayValues[i] = targetDisplayValuesToken.nextToken();
      }
    }
    allNames = propertyDetails.getProperty(tagName + ".targetValues");
    if (allNames != null) {
      StringTokenizer targetValuesToken = new StringTokenizer(allNames, ",");
      int numTargetValues = targetValuesToken.countTokens();
      m_targetValues  = new String[numTargetValues];
      for(int i=0; i < numTargetValues; i++) {
        m_targetValues[i] = targetValuesToken.nextToken();
      }
    }
    allNames = propertyDetails.getProperty(tagName + ".targetProbabilities");
    if (allNames != null) {
      StringTokenizer targetProbabilitiesToken = new StringTokenizer(allNames, ",");
      int numTargetProbabilities = targetProbabilitiesToken.countTokens();
      m_targetProbabilities  = new String[numTargetProbabilities];
      for(int i=0; i < numTargetProbabilities; i++) {
        m_targetProbabilities[i] = targetProbabilitiesToken.nextToken();
      }
    }
    m_targetDataType = propertyDetails.getProperty(tagName + ".targetDataType");

    // classification function settings
    tagName = "classificationFunctionSettings";
    m_MiningSettingsName = propertyDetails.getProperty(tagName + ".miningSettingsName");
    m_targetAttrName = propertyDetails.getProperty(tagName + ".targetAttributeName");
    String dataPrepStatus = propertyDetails.getProperty(tagName + ".dataPrepStatus");
    m_defaultDataPrep = DataPreparationStatus.getInstance(dataPrepStatus);
    allNames = propertyDetails.getProperty(tagName + ".inactiveAttributes");
    if (allNames != null) {
      StringTokenizer attributeNamesToken = new StringTokenizer(allNames,",");
      int numAttributes = attributeNamesToken.countTokens();
      m_inactiveAttributeNames  = new String[numAttributes];
      for(int i=0; i < numAttributes; i++) {
        m_inactiveAttributeNames[i] = attributeNamesToken.nextToken();
      }
    }

    // Adaptive Bayes Network algorithm settings
    tagName = "adaptiveBayesNetwork";
    try {
      m_treeDepth = Integer.parseInt(propertyDetails.getProperty(tagName + ".maximumNetworkFeatureDepth"));
    } catch (Exception e) {}
    //try {
    //  m_numOfNetworkFeature = Integer.parseInt(propertyDetails.getProperty(tagName + ".maximumNumberOfNetworkFeatures"));
    //} catch (Exception e) {}
    try {
      m_numOfConsecutivePrunedNF = Integer.parseInt(propertyDetails.getProperty(tagName + ".maximumConsecutivePrunedNetworkFeatures"));
    } catch (Exception e) {}
    try {
      m_maxBuildTime = Integer.parseInt(propertyDetails.getProperty(tagName + ".maximumBuildTime"));
    } catch (Exception e) {}
    try {
      m_maxPredictors = Integer.parseInt(propertyDetails.getProperty(tagName + ".maximumPredictors"));
    } catch (Exception e) {}
    try {
      m_NBNumberPredictors = Integer.parseInt(propertyDetails.getProperty(tagName + ".naiveBayesNumberPredictors"));
    } catch (Exception e) {}
    try {
      m_modelBuildType = Integer.parseInt(propertyDetails.getProperty(tagName + ".modelBuildType"));
    } catch (Exception e) {}

    // Output Names
    tagName = "adaptiveBayesNetworkOutput";
    m_TaskName = propertyDetails.getProperty(tagName + ".miningTaskName");
    m_ModelName = propertyDetails.getProperty(tagName + ".modelName");
  }
  /**
  ** Build a Adaptive Bayes Network Model.
  ** This method uses static variables as inputs. These static
  ** variables have to be initialized before this function is invoked.
  ** @param dmsConnection MiningServer connection handle.
  **/
  public static boolean buildAdaptiveBayesNetworkModel (
    oracle.dmt.odm.Connection dmsConnection)
  throws InvalidArgumentException, SQLException,
    MiningObjectException, ODMException,
    IOException, MiningTaskException
  {
    System.out.println("\nModel Build Task Phase:");
    //  Create a LocationAccessData object for data to be used for model build.
    LocationAccessData buildDataAccessData = m_PhysicalDataSpecification.getLocationAccessData();

    // Invoke AdaptiveBayesNetwork Build.
    System.out.println("    Invoking AdaptiveBayesNetwork Model build.");
    MiningBuildTask task = new MiningBuildTask(
      m_PhysicalDataSpecification,
      m_MiningSettingsName,
      m_ModelName);
    task.store(dmsConnection, m_TaskName);
    task.execute(dmsConnection);
    MiningTaskStatus status = task.waitForCompletion(dmsConnection);

    System.out.println("    Built AdaptiveBayesNetwork Model");
    System.out.println("    Name: " + m_ModelName);
    System.out.println("    Status: " + status.getTaskState().getEnum());
    long buildDuration = task.getExecutionDuration(dmsConnection, m_TaskName)/1000;
    System.out.println("    Duration: " + buildDuration + " seconds");
    
    return (status.getTaskState().equals(MiningTaskState.success) == true);
  }
}
