
// Copyright (c) 2001, 2002, Oracle Corporation.  All rights reserved.  
// Modification log
// BB:1/10/02 -- Fixed for new string check design

// Generic Java Imports
import java.io.*;
import java.util.*;
import java.sql.SQLException;

// ODM Imports
import oracle.dmt.odm.ODMException;
import oracle.dmt.odm.DataMiningServer;
import oracle.dmt.odm.LocationAccessData;
import oracle.dmt.odm.InvalidArgumentException;
import oracle.dmt.odm.MiningObjectException;
import oracle.dmt.odm.DistanceFunction;
import oracle.dmt.odm.AttributeType;
import oracle.dmt.odm.MiningFunction;
import oracle.dmt.odm.MiningAlgorithm;
import oracle.dmt.odm.DataPreparationStatus;
import oracle.dmt.odm.MiningTaskState;
import oracle.dmt.odm.DataFormatType;
import oracle.dmt.odm.task.*;

import oracle.dmt.odm.data.PhysicalDataSpecification;
import oracle.dmt.odm.data.TransactionalDataSpecification;
import oracle.dmt.odm.data.NonTransactionalDataSpecification;
import oracle.dmt.odm.data.LogicalDataSpecification;
import oracle.dmt.odm.data.MiningAttribute;
 
import oracle.dmt.odm.settings.function.DataUsageSpecification;
import oracle.dmt.odm.settings.algorithm.KMeansAlgorithmSettings;
import oracle.dmt.odm.settings.function.ClusteringFunctionSettings;
import oracle.dmt.odm.settings.algorithm.OClusterAlgorithmSettings;
import oracle.dmt.odm.settings.algorithm.ClusteringAlgorithmSettings;

import oracle.dmt.odm.model.MiningModel;




/**
 * This sample program demonstrates the use of ODM API to build a Clustering
 * model. 
 *
 * This program requires two property files.
 * The required inputs for building the model are specified in a property file
 * "Sample_ClusteringBuild.property".
 * Connectivity information is specified in "Sample_Global.property".
 *
 * This program be used for both transactional and nontransactional data.
 * After a model is built, the name of the model can be passed to
 * "Sample_ClusteringApply.java" to score data.
 */
public class Sample_ClusteringBuild extends Object {

 // Tag: miningServer
  private static String m_MiningServerUrl; // URL for the DataMiningServer
  private static String m_MiningServerUserName; // User name for the DataMiningServer
  private static String m_MiningServerPassword; // Password for the DataMiningServer

  // Tag: buildData
  private static DataFormatType m_BuildDataType; // Transactional or nontransactional
  private static String m_BuildDataTableName; // Data to build the model with
  //private static String m_BuildDataSchemaName; // Schema where the data exists

  // Tag: transactionalData
  // Read only if m_buildDataType = "transactional"
  private static String m_SequenceIdName; // Name of the "sequence id" column
  private static String m_AttributeName; // Name of the "attribute name" column
  private static String m_ValueName; // Name of the "value" column

  
  // Tag: clusteringOutput
  // Output Names
  private static String m_MiningSettingsName; // Name of the MiningSettings object
  private static String m_ModelName; // Name of the Association Rules model

  // Tag: clustering
  private static float m_Error;
  private static float m_Accuracy;
  private static int m_Iterations;
  private static String m_TargetAttributeName;
  private static String m_StoppingCriterion;
  private static int m_AlgorithmType;
  private static float m_Sensitivity;
  private static String[] m_NumericAttributes = null;
  private static String[] m_CategoricAttributes = null;

  // Tag: autobinning
  protected static boolean m_AutoBinning;

  // Tag: buildTask
  protected static String m_BuildTaskName;

  // Tag: null
  private static String m_EnableCleanup; // Indicates what kind of cleanup to be performed before running

  // Tag: null
  private static String m_InputDataSchemaName; // Schema where the input dataset is read from
  private static String m_outputSchemaName; // Schema where the output database object/s will be created


  // Following variables are not directly read from the property file
  // Initialized from the m_buildDataType above.
  // private static boolean m_IsTransactional;
  // private static boolean m_PerformDiscretization;
  private static PhysicalDataSpecification m_PhysicalDataSpecification;
  private static ClusteringFunctionSettings m_FunctionSettings;
  
  // Following variables are not directly read from the property file
  // Initialized from the m_buildDataType above.
  private static boolean m_IsTransactional;

  // Not sure were to get this one from
  private static int m_Clusters;

 /**
  **
  ** This is the code entry point
  **
  */

  public static void main ( String[] args ) {

    if ( args.length != 2 ) {
      System.out.println("Usage: java Sample_ClusteringBuild <Global Property file name> <Build Property file name>");
      return;
    }

    // Print header
    System.out.println("****************************************************************");
    System.out.println("ODM 9.2.0 Sample Program: Sample_ClusteringBuild ");
    System.out.println("****************************************************************");
    System.out.println("Property Files:");
    System.out.println("    Global property file: " + args[0]);
    System.out.println("    Sample property file: " + args[1]);
    System.out.println("****************************************************************");
    System.out.println("Start: " + new java.util.Date());

    DataMiningServer dms = null;
    oracle.dmt.odm.Connection dmsConnection = null;

    try {
      // Initialize the class varaibles from the property file.
      initializeClassVariables(args[0],args[1]);
      System.out.println("\nInitialization Phase:");
      System.out.println("    Data Mining Server:");
      System.out.println("        JDBC URL: " + m_MiningServerUrl);
      System.out.println("        Username: " + m_MiningServerUserName);

      System.out.println("Login Phase:");
      // Log into the MiningServer
      dms = new DataMiningServer(
        m_MiningServerUrl,
        m_MiningServerUserName,
        m_MiningServerPassword);
      dmsConnection = dms.login();
      System.out.println("    Completed MiningServer login");


      // Perform clean up if nessesary
      boolean continueRun = performCleanup ( dmsConnection);
      if (continueRun == false) {
        // cleanup flag value = cleanupOnly
        System.out.println("Exiting before executing the program. Cleanup only");
        return;
      }

      // Create PhysicalDataSpecification object.
      createPhysicalDataSpecification(dmsConnection);

      // Creates MiningFunctionSettings object
      createMiningFunctionSettings(dmsConnection);

      // Invoke the build function.
      buildClusteringModel(dmsConnection);



      
    } catch ( Exception e) {
      System.out.println("Received an exception in main: " + e.getMessage());
      e.printStackTrace();
    } finally {
      System.out.println("\nLogout Phase:");
      try {
        dms.logout(dmsConnection);
        System.out.println("    Logged out of mining server.");
        System.out.println("\nEnd: " + new java.util.Date());
        System.out.println("\n****************************************************************\n");
      } catch ( Exception e) {
        System.out.println("    Received an exception while logging out of MiningServer.");
        System.out.println("    Exception: " + e.getMessage());
      }
    }
  }

    /**
  ** performCleanup
  ** Reads the value in m_EableCleanup and performs appropriate action.
  ** This function catches and ignores all the exceptions: after printing the exception.
  ** @param dataMiningServerHandle DataMiningServer handle
  **/
  public static boolean performCleanup (oracle.dmt.odm.Connection dataMiningServerHandle)
  {
    System.out.println("\nCleanup Phase:");
    System.out.println("    Cleanup flag: " + m_EnableCleanup);
    boolean continueExecution = true;
    if ( m_EnableCleanup.compareTo("cleanupOnly") == 0 ){
      continueExecution = false;
    }
    if ( m_EnableCleanup.compareTo("cleanupBeforeExecution") == 0 ||
      m_EnableCleanup.compareTo("cleanupOnly") == 0) {
      // Remove ODM objects created (identified by the current property file)
      try {
        // Remove MiningFunctionSettings object
        System.out.println("    Removing function settings object: " + m_MiningSettingsName);
        ClusteringFunctionSettings.remove(dataMiningServerHandle,m_MiningSettingsName);
      } catch (Exception e) {
        System.out.println("    Failed to remove function settings object. Exception: " + e.getLocalizedMessage());
        System.out.println("    Ingnoring the exception. Continuing ...");
      }
      try {
        // Remove MiningModel object
        System.out.println("    Removing mining model object: " + m_ModelName);
        MiningModel.remove(dataMiningServerHandle,m_ModelName);
      } catch (Exception e) {
        System.out.println("    Failed to remove mining model object. Exception: " + e.getLocalizedMessage());
        System.out.println("    Ingnoring the exception. Continuing ...");
      }
      try {
        System.out.println("    Removing mining build task: " + m_BuildTaskName);
        MiningTask.remove(dataMiningServerHandle, m_BuildTaskName);
      } catch (Exception e) {
        System.out.println("    Failed to remove mining build task object. Exception: " + e.getLocalizedMessage());
        System.out.println("    Ingnoring the exception. Continuing ...");
      }

    }
    return continueExecution;
  }



 /**
  ** Build a Clustering Model.
  ** This method uses static variables as inputs. These static
  ** variables have to be initialized before this function is invoked.
  ** @param dataMiningServerHandle MiningServer connection handle.
  **/
  public static String buildClusteringModel (
    oracle.dmt.odm.Connection dataMiningServerHandle)
  throws InvalidArgumentException, SQLException,
    MiningObjectException, IOException,
    MiningTaskException,
    ODMException
  {
    System.out.println("\nModel build Phase:");
    //  Create a LocationAccessData object for data to be used for model build.
    LocationAccessData buildDataAccessData =  new LocationAccessData(
      m_BuildDataTableName,
      m_InputDataSchemaName);

    // Invoke  Build.
    System.out.println("    Invoking Clustering Model build.");
    java.util.Date startTime = new java.util.Date();
    System.out.println("    Start time: " + startTime.toString());

    // Create mining task
    MiningBuildTask miningbuildTast = new MiningBuildTask(
      m_PhysicalDataSpecification,
      m_MiningSettingsName,
      m_ModelName);

    // Store mining task
    miningbuildTast.store(dataMiningServerHandle, m_BuildTaskName);

    // Execute mining task
    miningbuildTast.execute(dataMiningServerHandle);

    // Wait for completion
    MiningTaskStatus miningTaskStatus =
      miningbuildTast.waitForCompletion(dataMiningServerHandle);

    // Check status
    MiningTaskState miningTaskState = miningTaskStatus.getTaskState();

    if(miningTaskState.isEqual( MiningTaskState.success ))
    {
        System.out.println("    Build Task suceessfully completed");
        java.util.Date endTime = new java.util.Date();
        System.out.println("    End time: " + endTime.toString());
        System.out.println("    Built (and persisted) Clustering Model. Name: " + m_ModelName);
    }else  if(miningTaskState.isEqual( MiningTaskState.error))
    {
        System.out.println( "Build failed due to: " + miningTaskStatus.getStateDescription() );
    } else
    {
        System.out.println( "Unexpected error occured:   " +  miningTaskStatus.getStateDescription() );
    }
    
    return m_ModelName;
  }

 /**
  ** createPhysicalDataSpecification
  ** Create a Transactional or NonTransactional PhysicalDataSpecification object.
  ** Initializes the static variable m_PhysicalDataSpecification.
  ** @param dataMiningServerHandle DataMiningServer handle
  **/
  public static void createPhysicalDataSpecification (oracle.dmt.odm.Connection dataMiningServerHandle)
  throws InvalidArgumentException, ODMException
  {
    System.out.println("\nPhysicalDataSpecification creation Phase:");
    //  Make appropriate instance of PhysicalDataSpecification
    if ( m_IsTransactional == true ) {
      LocationAccessData lad = new LocationAccessData(m_BuildDataTableName,m_InputDataSchemaName);
      m_PhysicalDataSpecification = new TransactionalDataSpecification(
            m_SequenceIdName, // Column name for "sequence id"
            m_AttributeName, // Column name for "attribute name"
            m_ValueName,
            lad); // Column name for "value"
      System.out.println("    Created Transactional PDS");
    } else {
      LocationAccessData lad = new LocationAccessData(m_BuildDataTableName,m_InputDataSchemaName);
      m_PhysicalDataSpecification = new NonTransactionalDataSpecification(lad);
      System.out.println("    Created NonTransactional PDS");
    }
  }


 /**
  ** createMiningFunctionSettings
  ** Make a MiningFunctionSettings object.
  ** @param dataMiningServerHandle DataMiningServer handle
  */
  public static void createMiningFunctionSettings (oracle.dmt.odm.Connection dmsConnection)
    throws InvalidArgumentException, MiningObjectException,
           SQLException, IOException, ODMException
  {
    System.out.println("\nMining settings creation Phase: ");
    ClusteringAlgorithmSettings algorithmSetting = null;
    if ( m_AlgorithmType == 1 )
    {
      if(m_StoppingCriterion.compareToIgnoreCase("errorAndIterations") == 0 )
        algorithmSetting = new KMeansAlgorithmSettings(
          m_Iterations,
          m_Error,
          DistanceFunction.euclidean
          );
      else if(m_StoppingCriterion.compareToIgnoreCase("error") == 0 )
        algorithmSetting = new KMeansAlgorithmSettings(
          m_Error,
          DistanceFunction.euclidean
          );
      else if(m_StoppingCriterion.compareToIgnoreCase("iterations") == 0 )
        algorithmSetting = new KMeansAlgorithmSettings(
          m_Iterations,
          DistanceFunction.euclidean
          );
    }
    else if ( m_AlgorithmType == 2 )
      algorithmSetting = new OClusterAlgorithmSettings(m_Sensitivity);

    ClusteringFunctionSettings clMFS = ClusteringFunctionSettings.create (
      dmsConnection, algorithmSetting, m_PhysicalDataSpecification,
       ( m_AutoBinning == true ? DataPreparationStatus.unprepared : DataPreparationStatus.discretized),
       m_Clusters, m_CategoricAttributes, m_NumericAttributes, null );

    clMFS.validate();       // validate before store
    clMFS.store(dmsConnection, m_MiningSettingsName);
        System.out.println("    Persisted MiningFunctionSettings. Name: " + m_MiningSettingsName);
  }


  /**
  * read property file
  */
  protected static void initializeClassVariables (String globalFileName, String fileName)
  throws FileNotFoundException, IOException, Exception
  {

    // Load ODM Global properties
    FileInputStream globalFileStream = new FileInputStream(globalFileName);
    Properties globalPropertyDetails = new Properties();
    globalPropertyDetails.load(globalFileStream);

    // Read MiningServer details.
    String tagName = "miningServer";
    m_MiningServerUrl = globalPropertyDetails.getProperty(tagName + ".url");
    m_MiningServerUserName = globalPropertyDetails.getProperty(tagName + ".userName");
    m_MiningServerPassword = globalPropertyDetails.getProperty(tagName + ".password");

    // Read input data schema name.
    m_InputDataSchemaName = globalPropertyDetails.getProperty("inputDataSchemaName");
    m_outputSchemaName = globalPropertyDetails.getProperty("outputSchemaName");

     // Load properties.
    FileInputStream inputFileStream = new FileInputStream(fileName);
    Properties propertyDetails = new Properties();
    propertyDetails.load(inputFileStream);

    // Read build data details.
    tagName = "buildData";
    m_BuildDataTableName = propertyDetails.getProperty(tagName + ".tableName");
    //m_BuildDataSchemaName = propertyDetails.getProperty(tagName + ".schemaName");
    String dataType = propertyDetails.getProperty(tagName + ".type");
    if ( dataType != null && dataType.length() != 0 )
      m_BuildDataType = DataFormatType.getInstance(dataType);
    else
      throw new Exception("Invalid build data type. Valid values are: transactional/nonTransactional.");

    if ( m_BuildDataType.isEqual(DataFormatType.transactional) ) {
      m_IsTransactional = true;
      // Read transactional data specifications.
      tagName = "transactionalData";
      m_SequenceIdName = propertyDetails.getProperty(tagName + ".sequenceId");
      m_AttributeName = propertyDetails.getProperty(tagName + ".attributeName");
      m_ValueName = propertyDetails.getProperty(tagName + ".valueName");
    } else if ( m_BuildDataType.isEqual(DataFormatType.nonTransactional) ){
      m_IsTransactional = false;
    } else {
      throw new Exception("Invalid build data type. Valid values are: transactional/nonTransactional.");
    }

    // Read build task name
    tagName = "buildTask";
    m_BuildTaskName = propertyDetails.getProperty(tagName + ".taskName");


    // Naive Bayes function settings
    tagName = "clustering";
    m_TargetAttributeName = propertyDetails.getProperty(tagName + ".targetAttributeName");
    m_Error = Float.parseFloat(propertyDetails.getProperty(tagName + ".error"));
    m_Iterations = Integer.parseInt(propertyDetails.getProperty(tagName + ".iterations"));
    m_StoppingCriterion = propertyDetails.getProperty(tagName + ".stoppingCriterion");
    m_Clusters = Integer.parseInt(propertyDetails.getProperty(tagName + ".clusters"));
    m_AlgorithmType = Integer.parseInt(propertyDetails.getProperty(tagName + ".algorithmType"));
    m_Sensitivity = Float.parseFloat(propertyDetails.getProperty(tagName + ".sensitivity"));
    // Read arrays of numeric and categorical attributes
    String sAttributeNames = propertyDetails.getProperty(tagName+".numericAttributes");
    if(sAttributeNames != null && sAttributeNames.length() != 0 )
       m_NumericAttributes = getStringTokens( sAttributeNames, "," );

    sAttributeNames = propertyDetails.getProperty(tagName+".categoricalAttributes");
    if(sAttributeNames != null && sAttributeNames.length() != 0 )
       m_CategoricAttributes = getStringTokens( sAttributeNames, "," );

    String dataPrepStatus = propertyDetails.getProperty(tagName + ".dataPrepStatus");
    if ( dataPrepStatus.compareToIgnoreCase("unprepared") == 0)
      m_AutoBinning = true;
    else if ( dataPrepStatus.compareToIgnoreCase("discretized") == 0)
      m_AutoBinning = false;
    else
      throw new Exception("Invalid build data type. Valid values are: unprepared/discretized.");

    // Output Names
    tagName = "clusteringOutput";
    m_MiningSettingsName = propertyDetails.getProperty(tagName + ".miningSettingsName");
    m_ModelName = propertyDetails.getProperty(tagName + ".modelName");

    // Read cleanup option
    m_EnableCleanup = propertyDetails.getProperty("enableCleanup");
    if ( m_EnableCleanup == null || m_EnableCleanup.length() == 0 )
      m_EnableCleanup = "cleanupBeforeExecution";
  }
  /**
   * This method is used to tokenize the string
   */
  protected static String[] getStringTokens(String str, String delimiter) {
    StringTokenizer strTokenizer = new StringTokenizer( str, delimiter );
    int noOfTokens = strTokenizer.countTokens();
    String[] strTokens = new String[noOfTokens];
    for(int iToken=0; iToken < noOfTokens; iToken++) {
      strTokens[iToken] = strTokenizer.nextToken();
    }
    return strTokens;
  }
}

