
// Copyright (c) 2001, 2002, Oracle Corporation.  All rights reserved.  

// Generic Java Imports
import java.io.*;
import java.util.*;
import java.sql.*;

// ODM Imports
import oracle.dmt.odm.DataType;
import oracle.dmt.odm.AttributeType;
import oracle.dmt.odm.data.MiningAttribute;
import oracle.dmt.odm.DataMiningServer;
import oracle.dmt.odm.LocationAccessData;
import oracle.dmt.odm.InvalidArgumentException;
import oracle.dmt.odm.RuleSortCriteria;
import oracle.dmt.odm.DistanceFunction;
import oracle.dmt.odm.MiningFunction;
import oracle.dmt.odm.DataPreparationStatus;
import oracle.dmt.odm.AttributeHistogram;
import oracle.dmt.odm.Category;
import oracle.dmt.odm.ComparisonFunction;
import oracle.dmt.odm.MiningObjectException;

import oracle.dmt.odm.data.PhysicalDataSpecification;
import oracle.dmt.odm.data.TransactionalDataSpecification;
import oracle.dmt.odm.data.NonTransactionalDataSpecification;
import oracle.dmt.odm.data.LogicalDataSpecification;

import oracle.dmt.odm.settings.function.DataUsageSpecification;
import oracle.dmt.odm.settings.function.MiningFunctionSettings;
import oracle.dmt.odm.settings.function.ClusteringFunctionSettings;

import oracle.dmt.odm.settings.algorithm.ClusteringAlgorithmSettings;
import oracle.dmt.odm.settings.algorithm.OClusterAlgorithmSettings;
import oracle.dmt.odm.settings.algorithm.KMeansAlgorithmSettings;

import oracle.dmt.odm.rule.Predicate;

import oracle.dmt.odm.model.MiningModel;
import oracle.dmt.odm.model.ClusteringModel;
import oracle.dmt.odm.model.Cluster;
import oracle.dmt.odm.model.ClusterCentroid;
import oracle.dmt.odm.model.ClusterCentroidEntry;

import oracle.dmt.odm.rule.MiningRule;
import oracle.dmt.odm.rule.MiningRuleSet;
import oracle.dmt.odm.internal.util.EISUtils;

/**
 * This sample program demonstrates the use of ODM API to obtain information
 * about a Clustering model. The name of the model is specified the property
 * file "Sample_Clustering_Results.property". Connection information is
 * specified in "Sample_Global.property". 
 * The model used here needs to be already built in the MiningServer
 * schema being used in this program. You can use "Sample_ClusteringBuild.java"
 * to build a Clustering model.
 */


public class Sample_Clustering_Results extends Object {

 // Tag: miningServer
  private static String m_MiningServerUrl; // URL for the DataMiningServer
  private static String m_MiningServerUserName; // User name for the DataMiningServer
  private static String m_MiningServerPassword; // Password for the DataMiningServer

  // Tag: null
  private static String m_InputDataSchemaName; // Schema where the input dataset is read from
  private static String m_outputSchemaName; // Schema where the output database object/s will be created

  // Tag: null
  private static String m_ModelName; // Model name to be used for Apply


  /**
   * Constructor
   */
  public Sample_Clustering_Results()
  {
  }

  public static void main(String[] args)
  {


        if ( args.length != 2 ) {
      System.out.println("Please enter property file name.");
      return;
    }

    // Print header
    System.out.println("****************************************************************");
    System.out.println("ODM 9.2.0 Sample Program: Sample_ClusteringResults");
    System.out.println("****************************************************************");
    System.out.println("Property Files:");
    System.out.println("    Global property file: " + args[0]);
    System.out.println("    Sample property file: " + args[1]);
    System.out.println("****************************************************************");
    System.out.println("Start: " + new java.util.Date());



    DataMiningServer dms = null;
    oracle.dmt.odm.Connection dmsConnection = null;

    try {
      // Initialize the class varaibles from the property file.
      initializeClassVariables(args[0],args[1]);
      System.out.println();
      System.out.println("Initialization Phase:");
      System.out.println("    Data Mining Server:");
      System.out.println("        JDBC URL: " + m_MiningServerUrl);
      System.out.println("        Username: " + m_MiningServerUserName);

      System.out.println();
      System.out.println("Login Phase:");
      // Log into the MiningServer
      dms = new DataMiningServer(
        m_MiningServerUrl,
        m_MiningServerUserName,
        m_MiningServerPassword);
      dmsConnection = dms.login();
      System.out.println("    Completed MiningServer login");

      // Print model details
      getModelDetails(m_ModelName, dmsConnection);

    } catch ( Exception e) {
      System.out.println("Received an exception in main: " + e.getMessage());
      e.printStackTrace();
    } finally {
      System.out.println();
      System.out.println("Logout Phase:");
      try {
        dms.logout(dmsConnection);
        System.out.println("    Logged out of mining server.");
        System.out.println("\nEnd: " + new java.util.Date());
        System.out.println("\n****************************************************************\n");
      } catch ( Exception e) {
        System.out.println("    Received an exception while logging out of MiningServer.");
        System.out.println("    Exception: " + e.getMessage());
        e.printStackTrace();
      }
    }
  }


  /*
  * get model details
  */
  public static  void getModelDetails(
    String modelName,
    oracle.dmt.odm.Connection dmsConnection
  )
  {

    try {
      // Restore model
      ClusteringModel clModel = (ClusteringModel)ClusteringModel.restore(dmsConnection, modelName);

      // Print general model information
      System.out.println();
      System.out.println("Model general information:");
      System.out.println("    Model Name: " + modelName);
      System.out.println("    NumberOfClusters: " + clModel.getNumberOfClusters());
      System.out.println("    InputDataRecordCount: " + clModel.getInputDataRecordCount());
      System.out.println("    NumberOfLeaves: " + clModel.getNumberOfLeaves());
      System.out.println("    NumberOfLevels: " + clModel.getNumberOfLevels());
      System.out.println("    Root ID: " + clModel.getRootID());

      // Get Clusters
      Vector vCluster = clModel.getClusters(dmsConnection);

      // Display number of attributes
      MiningFunctionSettings mfs = clModel.getMiningFunctionSettings();
      MiningAttribute[] maArray = mfs.getLogicalDataSpecification().getMiningAttributes();
      int attrNum = maArray.length;
      Cluster cluster = (Cluster)vCluster.elementAt(vCluster.size()-1);
      System.out.println("       Number of attributes: " + attrNum);



      System.out.println();
      System.out.println("Cluster information:");
      for ( int na = 0; na < vCluster.size(); na++){
        System.out.println();
        Cluster clCluster = (Cluster)vCluster.elementAt(na);
        System.out.println("    Cluster : ");
        System.out.println("        ID: " + clCluster.getId() );
        Cluster clParent = clCluster.getParent();
        System.out.println("        Parent: " + ( clParent != null ?  (new Integer(clParent.getId())).toString(): "null" ) );
        Vector vChilden = clCluster.getChildren();
        for ( int nCh = 0; nCh < vChilden.size(); nCh++ ){
          Cluster clChild = (Cluster)vChilden.elementAt(nCh);
          System.out.println("        Child[" + nCh + "]: " + ( clChild != null ? (new Integer(clChild.getId())).toString(): "null" ) );
        }
        System.out.println("        Level: " + clCluster.getLevel() );
        System.out.println("        RecordCount: " + clCluster.getRecordCount() );

        ClusterCentroid clCentroid = clCluster.getClusterCentroid();
        Vector vCentroidEntries = clCentroid.getEntries();
        System.out.println("        Centroid Values\n");
        for ( int nj = 0; nj < vCentroidEntries.size(); nj++ ){
          ClusterCentroidEntry clCentEntry = (ClusterCentroidEntry)vCentroidEntries.elementAt(nj);
          Category value = clCentEntry.getValue();
          System.out.print("            " + clCentEntry.getAttribute().getName());
          System.out.print(" : " + value.getDisplayName());
          System.out.print("\n");
        }
      }

      System.out.println();
      System.out.println("   Print out rules: FILTER = FALSE");
      MiningRuleSet ruleSet = clModel.getClusterRules(dmsConnection, 3, false);
      MiningRule[] rules = ruleSet.getRules();
      for ( int ni = 0; ni < rules.length; ni++){
        System.out.println("      Rule [" + (ni+1) + "]: " + rules[ni].toString());
      }

      System.out.println();
      System.out.println("   Print out rules: FILTER = TRUE");
      ruleSet = clModel.getClusterRules(dmsConnection, 3, true);
      rules = ruleSet.getRules();
      for ( int ni = 0; ni < rules.length; ni++){
        System.out.println("      Rule [" + (ni+1) + "]: " + rules[ni].toString());
      }

      System.out.println();
      System.out.println("   Print out ALL rules:");
      ruleSet = clModel.getClusterRules(dmsConnection);
      rules = ruleSet.getRules();
      for ( int ni = 0; ni < rules.length; ni++){
        System.out.println("      Rule [" + (ni+1) + "]: " + rules[ni].toString());
      }


      MiningAttribute maCatMiningAttribute = null;
      MiningAttribute maNumMiningAttribute = null;
      for ( int ni = 0; ni < maArray.length; ni++ ){
        AttributeType aType = maArray[ni].getAttributeType();
        if ( true == aType.isEqual(AttributeType.categorical) )
          maCatMiningAttribute = maArray[ni];
        else if ( true == aType.isEqual(AttributeType.numerical) )
          maNumMiningAttribute = maArray[ni];
        if ( maCatMiningAttribute != null && maNumMiningAttribute != null )
          break;
      }
      
      if ( maCatMiningAttribute != null ){
        System.out.println("\n   Get Attribute Histogram for a categorical mining attribute");
        System.out.println("     Cluster ID: " + cluster.getId());
        System.out.println("     Attribute Name: " + maCatMiningAttribute.getName());

        AttributeHistogram aHistogram = clModel.getAttributeHistogram(
              dmsConnection, cluster.getId(), maCatMiningAttribute.getName() );
        int numberOfBeans = aHistogram.getNumberOfBins();
        System.out.println("      BIN\tDISPLAY NAME\tFREQUENCY");
        for(int i=1; i<=numberOfBeans; i++)
        {
          System.out.println("      " + i + "   " + aHistogram.getDisplayName(i) + "    " + aHistogram.getFrequency(i));
        }
      }
      
      if ( maNumMiningAttribute != null ){
        System.out.println("\n   Get Attribute Histogram for a numeric mining attribute");
        System.out.println("     Cluster ID: " + cluster.getId());
        System.out.println("     Attribute Name: " + maNumMiningAttribute.getName());

        AttributeHistogram aHistogram = clModel.getAttributeHistogram(
              dmsConnection, cluster.getId(), maNumMiningAttribute.getName() );
        int numberOfBeans = aHistogram.getNumberOfBins();
        System.out.println("      BIN\tDISPLAY NAME\tFREQUENCY");
        for(int i=1; i<=numberOfBeans; i++)
        {
          System.out.println("      " + i + "   " + aHistogram.getDisplayName(i) + "    " + aHistogram.getFrequency(i));
        }
      }
    } catch(Exception e) {
      e.printStackTrace();
    }

  }


  /**
  * read property file
  */
  protected static void initializeClassVariables (String globalFileName, String fileName)
  throws FileNotFoundException, IOException, Exception
  {

    // Load ODM Global properties
    FileInputStream globalFileStream = new FileInputStream(globalFileName);
    Properties globalPropertyDetails = new Properties();
    globalPropertyDetails.load(globalFileStream);

    // Read MiningServer details.
    String tagName = "miningServer";
    m_MiningServerUrl = globalPropertyDetails.getProperty(tagName + ".url");
    m_MiningServerUserName = globalPropertyDetails.getProperty(tagName + ".userName");
    m_MiningServerPassword = globalPropertyDetails.getProperty(tagName + ".password");

    // Read input data schema name.
    m_InputDataSchemaName = globalPropertyDetails.getProperty("inputDataSchemaName");
    m_outputSchemaName = globalPropertyDetails.getProperty("outputSchemaName");


    // Load properties
    FileInputStream inputFileStream = new FileInputStream(fileName);
    Properties propertyDetails = new Properties();
    propertyDetails.load(inputFileStream);

    m_ModelName = propertyDetails.getProperty("modelName");


  }
}
