// Copyright (c) 2001, 2002, Oracle Corporation.  All rights reserved.  
// Modification log -- Please add a line comment about the changes
// 2/21/02 -- Add code to trim blanks from property values.
// 2/19/02 -- Impact of Javadoc changes for MH comments.
// 11/28/01 -- Edited sample code for Model Seeker to make it easier to refererence from FS.

// Generic Java Imports
import java.io.*;
import java.util.*;
import java.sql.SQLException;

// ODM Imports
import oracle.dmt.odm.AttributeType;
import oracle.dmt.odm.AttributeUsage;
import oracle.dmt.odm.Category;
import oracle.dmt.odm.CategoryMatrix;
import oracle.dmt.odm.DataType;
import oracle.dmt.odm.DataMiningServer;
import oracle.dmt.odm.DataPreparationStatus;
import oracle.dmt.odm.InvalidArgumentException;
import oracle.dmt.odm.LocationAccessData;
import oracle.dmt.odm.MiningAlgorithm;
import oracle.dmt.odm.MiningTaskState;
import oracle.dmt.odm.ODMException;
import oracle.dmt.odm.PairCombinationsOption;
import oracle.dmt.odm.PriorProbabilities;
import oracle.dmt.odm.DataFormatType;

import oracle.dmt.odm.data.LogicalDataSpecification;
import oracle.dmt.odm.data.NonTransactionalDataSpecification;
import oracle.dmt.odm.data.MiningAttribute;
import oracle.dmt.odm.data.PhysicalDataSpecification;
import oracle.dmt.odm.data.TransactionalDataSpecification;

import oracle.dmt.odm.model.AdaptiveBayesNetworkModel;
import oracle.dmt.odm.model.MiningModel;

import oracle.dmt.odm.result.ModelSeekerResult;
import oracle.dmt.odm.result.ModelSeekerResultEntry;

import oracle.dmt.odm.rule.MiningRule;
import oracle.dmt.odm.rule.MiningRuleSet;

import oracle.dmt.odm.settings.algorithm.AdaptiveBayesNetworkSettings;
import oracle.dmt.odm.settings.algorithm.CombinationAdaptiveBayesNetworkSettings;
import oracle.dmt.odm.settings.algorithm.CombinationNaiveBayesSettings;
import oracle.dmt.odm.settings.algorithm.MiningAlgorithmSettings;
import oracle.dmt.odm.settings.algorithm.ModelSeekerClassificationAlgorithmSettings;
import oracle.dmt.odm.settings.algorithm.NaiveBayesSettings;

import oracle.dmt.odm.settings.function.MiningFunctionSettings;
import oracle.dmt.odm.settings.function.DataUsageSpecification;
import oracle.dmt.odm.settings.function.ClassificationFunctionSettings;

import oracle.dmt.odm.task.MiningTask;
import oracle.dmt.odm.task.MiningTaskStatus;
import oracle.dmt.odm.task.ModelSeekerTask;
import oracle.dmt.odm.internal.task.DMSModelSeekerTask;

/**
 * This sample program demonstrates the use of ODM API to pick the best model by
 * running model seeker.
 *
 * The required inputs are specified in two property files:
 * "Sample_Global.property": Contains ODM connectivity, data location
 * "Sample_ModelSeeker.property": Contains model seeker specific parameters
 *
 * This program can be used for both transactional and non transactional data.
 */

public class Sample_ModelSeeker extends Object
{
 // Tag: miningServer
  private static String m_MiningServerUrl; // URL for the DataMiningServer
  private static String m_MiningServerUserName; // User name for the DataMiningServer
  private static String m_MiningServerPassword; // Password for the DataMiningServer

  // Tag: null
  private static String m_outputSchemaName; // Schema where the output database object/s will be created

  // Tag: buildData
  private static DataFormatType m_BuildDataType; // Transactional or non transactional
  private static String m_BuildDataTableName; // Data to build the model with
  private static String m_BuildDataSchemaName; // Schema where the data exists

  // Valid only for transactional data
  private static String m_BuildSequenceIdName; // Name of the "sequence id" column
  private static String m_BuildAttributeName; // Name of the "attribute name" column
  private static String m_BuildValueName; // Name of the "value" column

  // Tag: testData
  private static String m_TestDataType; // Transactional or non transactional
  private static String m_TestDataTableName; // Data to build the model with
  private static String m_TestDataSchemaName; // Schema where the data exists
  // Read only if m_TestDataType = "transactional"
  private static String m_TestSequenceIdName; // Name of the "sequence id" column
  private static String m_TestAttributeName; // Name of the "attribute name" column
  private static String m_TestValueName; // Name of the "value" column

  // Tag: modelSeekerAlgorithmSetting
  private static String m_NumberOfAlgorithms;
  private static ModelSeekerClassificationAlgorithmSettings m_ModelSeekerClassificationAlgorithmSettings;

  // Tag: classificationFunctionSetting
  private static DataPreparationStatus m_DataPreparationStatus;
  private static String m_TargetAttributeName;
  private static AttributeType m_TargetAttributeType;
  private static String m_SupplementalAttributes; // Name of the key column
  private static ClassificationFunctionSettings m_ClassificationFunctionSettings;

  // Tag: positiveCategoryTarget
  private static String m_DisplayName;
  private static String m_DataType;
  private static String m_Value;
  private static Category m_PositiveCategoryTarget;

  // Tag: modelSeeker
  private static String m_Prefix;
  private static String m_ClassificationFunctionSettingName;
  private static String m_ResultName;
  private static int m_LiftQuantiles;
  private static float m_Weight;

  // Tag: modelSeekerTask
  private static String m_TaskName;
  private static String m_EnableCleanup;

  // Following variables are not directly read from the property file
  private static Properties m_PropertyDetails;
  private static boolean m_IsBuildTransactional;
  private static PhysicalDataSpecification m_BuildPhysicalDataSpecification;
  private static boolean m_IsTestTransactional;
  private static PhysicalDataSpecification m_TestPhysicalDataSpecification;

  /**
  **
  ** This is the code entry point
  **
  */
  public static void main ( String[] args )
  {
    if ( args.length != 2 )
    {
      System.out.println("Usage: java Sample_ModelSeeker <Global Property file name> <Model Seeker Property file name>");
      return;
    }
    System.out.println("*****************************************************");
    System.out.println("ODM 9.2.0 Sample Program: Sample_ModelSeeker");
    System.out.println("*****************************************************");
    System.out.println("Property Files:");
    System.out.println("    Global: " + args[0]);
    System.out.println("    Sample: " + args[1]);
    System.out.println("*****************************************************");
    System.out.println("Start: " + new java.util.Date());

    DataMiningServer dms = null;
    oracle.dmt.odm.Connection dmsConnection = null;
    try
    {
      // Initialize the class varaibles from the property file.
      System.out.println(" ");
      System.out.println("Initialization Phase:");
      initializeClassVariables(args[0],args[1]);
      System.out.println("    Data Mining Server:");
      System.out.println("        JDBC URL:" + m_MiningServerUrl);
      System.out.println("        User Name:" + m_MiningServerUserName);

      System.out.println(" ");
      System.out.println("Login Phase:");
      // Log into the MiningServer
      dms = new DataMiningServer(
        m_MiningServerUrl,
        m_MiningServerUserName,
        m_MiningServerPassword);
      dmsConnection = dms.login();
      System.out.println("    Completed Mining Server login");

      System.out.println(" ");
      System.out.println("Cleanup Phase:");
      System.out.println("    Cleanup flag: " + m_EnableCleanup);
      if (m_EnableCleanup.equalsIgnoreCase( "cleanupOnly" ))
      {
        cleanUp(dmsConnection);
        System.out.println("    Exiting before executing the program. Cleanup only.");
      }
      else
      {
        if (m_EnableCleanup.equalsIgnoreCase( "cleanpBeforeExecute" ))
          cleanUp(dmsConnection);
      }

      System.out.println(" ");
      System.out.println("Data Setup Creation Phase:");
      // Create PhysicalDataSpecification object for building a model.
      createBuildPhysicalDataSpecification(dmsConnection);
      System.out.println("    Created PhysicalDataSpecification for model building");

      // Create PhysicalDataSpecification object for testing a model.
      createTestPhysicalDataSpecification(dmsConnection);
      System.out.println("    Created PhysicalDataSpecification for model testing");

      System.out.println(" ");
      System.out.println("Mining Function Settings Creation Phase:");
      // Create ModelSeekerClassificationAlgorithmSettings object.
      createModelSeekerClassificationAlgorithmSettings();
      System.out.println("    Created ModelSeekerClassificationAlgorithmSettings");
      // Create ClassificationFunctionSettings object.
      createClassificationFunctionSettings(dmsConnection);
      System.out.println("    Created ClassificationFunctionSettings");
      // Create Category object for positive target value.
      createPositiveCategoryTarget();
      System.out.println("    Created Category object for positive target value");

      // Store, execute, and display results for Model Seeker Task
      System.out.println(" ");
      System.out.println("Model Seeker Build Phase:");
      System.out.println("    Invoking Model Seeker build task");
      executeModelSeeker(dmsConnection);
    }
    catch ( Exception e) {
      System.out.println("Received an exception in main: " + e.getMessage());
    }
    finally {
      System.out.println(" ");
      System.out.println("Logout Phase:");
      try {
        dms.logout(dmsConnection);
        System.out.println("    Logged out of mining server.");
        System.out.println("End: " + new java.util.Date());
        System.out.println("*****************************************************");
      }
      catch ( Exception e) {
        System.out.println("Received an exception while logging out of MiningServer.");
        System.out.println("Exception: " + e.getMessage());
      }
    } // finally
  } // main

  /**
  ** This function removes white space from both ends of the input property values
  **/
  public static Properties trimStrings(Properties inputProps) {
    Enumeration keys = inputProps.keys();
    Properties outputProps = new Properties();
    while(keys.hasMoreElements()) {
      String key = (String)keys.nextElement();
      String value = inputProps.getProperty(key);
      outputProps.put(key, value.trim());
    }
    return outputProps;
  }

  /**
  ** Create a Transactional or NonTransactional PhysicalDataSpecification object for building a model.
  ** Initializes the static variable m_BuildPhysicalDataSpecification.
  ** @param dataMiningServerHandle DataMiningServer handle
  **/
  private static void createBuildPhysicalDataSpecification (oracle.dmt.odm.Connection dataMiningServerHandle)
  throws Exception
  {
    //  Make appropriate instance of PhysicalDataSpecification for building a model
    LocationAccessData lad = new LocationAccessData(m_BuildDataTableName, m_BuildDataSchemaName);
    if ( m_IsBuildTransactional == true )
    {
      m_BuildPhysicalDataSpecification = new TransactionalDataSpecification(
            m_BuildSequenceIdName, // Column name for "sequence id"
            m_BuildAttributeName, // Column name for "attribute name"
            m_BuildValueName, // Column name for "value"
            lad);
    }
    else
    {
      m_BuildPhysicalDataSpecification = new NonTransactionalDataSpecification(lad);
    }
  } // createBuildPhysicalDataSpecification

  /**
  ** Create a Transactional or NonTransactional PhysicalDataSpecification object for testing a model.
  ** Initializes the static variable m_TestPhysicalDataSpecification.
  ** @param dataMiningServerHandle DataMiningServer handle
  **/
  private static void createTestPhysicalDataSpecification (oracle.dmt.odm.Connection dataMiningServerHandle)
  throws Exception
  {
    //  Make appropriate instance of PhysicalDataSpecification for testing a model
    LocationAccessData lad = new LocationAccessData(m_TestDataTableName, m_TestDataSchemaName);
    if ( m_IsTestTransactional == true )
    {
      m_TestPhysicalDataSpecification = new TransactionalDataSpecification(
            m_TestSequenceIdName, // Column name for "sequence id"
            m_TestAttributeName, // Column name for "attribute name"
            m_TestValueName, // Column name for "value"
            lad);
    }
    else
    {
      m_TestPhysicalDataSpecification = new NonTransactionalDataSpecification(lad);
    }
  } // createTestPhysicalDataSpecification

  /**
  ** Create a Model Seeker ModelSeekerClassificationAlgorithmSettings object.
  ** Initializes the static variable m_ModelSeekerClassificationAlgorithmSettings.
  **/
   private static void createModelSeekerClassificationAlgorithmSettings()
   throws Exception
   {
    String tagName = "modelSeekerAlgorithmSetting";
    m_NumberOfAlgorithms = m_PropertyDetails.getProperty(tagName + ".numberOfAlgorithms");

    if(m_NumberOfAlgorithms != null)
    {
      // Construct an array to store all pre-defined algorithm settings
      int numberOfAlgorithms = Integer.parseInt(m_NumberOfAlgorithms);
      MiningAlgorithmSettings[] miningAlgorithmSettingsArray =
          new MiningAlgorithmSettings[numberOfAlgorithms];

      for(int i=0; i < numberOfAlgorithms; i++)
      {
        //Get algorithm type
        String algorithmTag = tagName + "_" + (i+1);
        String algorithm = m_PropertyDetails.getProperty(algorithmTag + ".algorithm");
        if(algorithm != null)
        {
          if(algorithm.equalsIgnoreCase("null"))
          {
            miningAlgorithmSettingsArray[i] = null;
          }
          else
          {
            MiningAlgorithm miningAlgorithm = MiningAlgorithm.getInstance(algorithm);

            if(miningAlgorithm.isEqual(MiningAlgorithm.naiveBayes))
            {
              miningAlgorithmSettingsArray[i] = createNaiveBayesSettings(algorithmTag);
            }
            else if(miningAlgorithm.isEqual(MiningAlgorithm.adaptiveBayesNetwork))
            {
              miningAlgorithmSettingsArray[i] = createAdaptiveBayesNetworkSettings(algorithmTag);
            }
            else if(miningAlgorithm.isEqual(MiningAlgorithm.combinationNaiveBayes))
            {
              miningAlgorithmSettingsArray[i] = createCombinationNaiveBayesSettings(algorithmTag);
            }
            else
            if(miningAlgorithm.isEqual(MiningAlgorithm.combinationAdaptiveBayesNetwork))
            {
              miningAlgorithmSettingsArray[i] = createCombinationAdaptiveBayesNetworkSettings(algorithmTag);
            }
          }
        }
        else
        {
          miningAlgorithmSettingsArray[i] = null;
        }
      }
      m_ModelSeekerClassificationAlgorithmSettings = new ModelSeekerClassificationAlgorithmSettings(miningAlgorithmSettingsArray);
    }
    else
    {
      m_ModelSeekerClassificationAlgorithmSettings = null;
    }
  } // createModelSeekerClassificationAlgorithmSettings

  /**
  ** Create a NaiveBayesSettings object.
  ** @param String algorithmTag
  **/
   private static NaiveBayesSettings createNaiveBayesSettings(String algorithmTag) {
    String sigletonThreshold_str = m_PropertyDetails.getProperty(algorithmTag + ".singletonThreshold");
    String pairwiseThreshold_str = m_PropertyDetails.getProperty(algorithmTag + ".pairwiseThreshold");
    float sigletonThreshold, pairwiseThreshold;
    if(sigletonThreshold_str != null && pairwiseThreshold_str != null)
    {
      try
      {
        sigletonThreshold = Float.parseFloat(sigletonThreshold_str);
        pairwiseThreshold = Float.parseFloat(pairwiseThreshold_str);
        return new NaiveBayesSettings(sigletonThreshold, pairwiseThreshold);
      }
      catch(Exception exp)
      {
        return null;
      }
    }
    else
    {
      return null;
    }
  } // createNaiveBayesSettings

  /**
  ** Create a AdaptiveBayesNetworkSettings object.
  ** @param String algorithmTag
  **/
  private static AdaptiveBayesNetworkSettings createAdaptiveBayesNetworkSettings(String algorithmTag) {
    String maximumNetworkFeatureDepth_str = m_PropertyDetails.getProperty(algorithmTag + ".maximumNetworkFeatureDepth");
    String maximumNumberOfNetworkFeatures_str = m_PropertyDetails.getProperty(algorithmTag + ".maximumNumberOfNetworkFeatures");
    String maximumConsecutivePrunedNetworkFeatures_str = m_PropertyDetails.getProperty(algorithmTag + ".maximumConsecutivePrunedNetworkFeatures");
    String maximumBuildTime_str = m_PropertyDetails.getProperty(algorithmTag + ".maximumBuildTime");
    int maximumNetworkFeatureDepth;
    int maximumNumberOfNetworkFeatures;
    int maximumConsecutivePrunedNetworkFeatures;
    int maximumBuildTime;
    if( (maximumNetworkFeatureDepth_str != null) &&
        (maximumNumberOfNetworkFeatures_str != null) &&
        (maximumConsecutivePrunedNetworkFeatures_str != null) &&
        (maximumBuildTime_str != null) )
    {
      try
      {
        maximumNetworkFeatureDepth = Integer.parseInt(maximumNetworkFeatureDepth_str);
        maximumNumberOfNetworkFeatures = Integer.parseInt(maximumNumberOfNetworkFeatures_str);
        maximumConsecutivePrunedNetworkFeatures = Integer.parseInt(maximumConsecutivePrunedNetworkFeatures_str);
        maximumBuildTime = Integer.parseInt(maximumBuildTime_str);
        return new AdaptiveBayesNetworkSettings(
              maximumNetworkFeatureDepth,
              maximumNumberOfNetworkFeatures,
              maximumConsecutivePrunedNetworkFeatures,
              maximumBuildTime );
      }
      catch(Exception exp)
      {
        return null;
      }
    }
    else
    {
      return null;
    }
  } // createAdaptiveBayesNetworkSettings

  /**
  ** Create a CombinationNaiveBayesSettings object.
  ** @param String algorithmTag
  **/
  private static CombinationNaiveBayesSettings createCombinationNaiveBayesSettings(String algorithmTag) {
    String singletonThreshold_str = m_PropertyDetails.getProperty(algorithmTag + ".singletonThreshold");
    String pairwiseThreshold_str = m_PropertyDetails.getProperty(algorithmTag + ".pairwiseThreshold");
    String pairCombinationOption_str = m_PropertyDetails.getProperty(algorithmTag + ".pairCombinationOption");
    float singletonThreshold[], pairwiseThreshold[];
    PairCombinationsOption pairCombinationOption = null;
    if(pairCombinationOption_str != null && singletonThreshold_str != null && pairwiseThreshold_str != null)
    {
      try
      {
        pairCombinationOption = PairCombinationsOption.getInstance(pairCombinationOption_str);
        singletonThreshold = getFloatTokens(singletonThreshold_str, ",");
        pairwiseThreshold = getFloatTokens(pairwiseThreshold_str, ",");
        return new CombinationNaiveBayesSettings( pairCombinationOption, singletonThreshold, pairwiseThreshold);
      }
      catch(Exception exp)
      {
        return null;
      }
    }
    else
    {
      return null;
    }
  } // createCombinationNaiveBayesSettings

  /**
  ** Create a CombinationAdaptiveBayesNetworkSettings object.
  ** @param String algorithmTag
  **/
   private static CombinationAdaptiveBayesNetworkSettings createCombinationAdaptiveBayesNetworkSettings(String algorithmTag) {
    String maximumNetworkFeatureDepth_str = m_PropertyDetails.getProperty(algorithmTag + ".maximumNetworkFeatureDepth");
    int[] maximumNetworkFeatureDepth;
    if( maximumNetworkFeatureDepth_str != null)
    {
      try
      {
        maximumNetworkFeatureDepth = getIntTokens( maximumNetworkFeatureDepth_str, ",");
        return new CombinationAdaptiveBayesNetworkSettings( maximumNetworkFeatureDepth );
      }
      catch(Exception exp)
      {
        return null;
      }
    }
    else
    {
      return null;
    }
   } // createCombinationAdaptiveBayesNetworkSettings

  /**
  ** Create a ClassificationFunctionSettings object.
  ** @param dataMiningServerHandle DataMiningServer handle
  **/
  private static void createClassificationFunctionSettings (oracle.dmt.odm.Connection dataMiningServerHandle)
  throws Exception
  {
    MiningAlgorithmSettings miningAlgorithmSettings = m_ModelSeekerClassificationAlgorithmSettings;
    DataPreparationStatus dataPreparationStatus = m_DataPreparationStatus;
    PhysicalDataSpecification physicalDataSpecification = m_BuildPhysicalDataSpecification;
    String targetAttributeName = m_TargetAttributeName;
    AttributeType targetAttributeType = m_TargetAttributeType;
    m_ClassificationFunctionSettings =
    ClassificationFunctionSettings.create(
          dataMiningServerHandle,
          miningAlgorithmSettings,
          physicalDataSpecification,
          targetAttributeName,
          targetAttributeType,
          dataPreparationStatus);
    if ((m_SupplementalAttributes != null) && (m_SupplementalAttributes.length() > 0))
    {
      m_ClassificationFunctionSettings.adjustAttributeUsage(
          new String[]{m_SupplementalAttributes},
          AttributeUsage.supplementary);
    }
      
  } // createClassificationFunctionSettings

  /**
   * This method is used to tokenize the string
   */
  private static String[] getStringTokens(String str, String delimiter)
  {
    StringTokenizer strTokenizer = new StringTokenizer( str, delimiter );
    int noOfTokens = strTokenizer.countTokens();
    String[] strTokens = new String[noOfTokens];
    for(int iToken=0; iToken < noOfTokens; iToken++)
    {
      strTokens[iToken] = strTokenizer.nextToken();
    }
    return strTokens;
  } // getStringTokens

  /**
   * This method is used to tokenize the float
   */
  private static float[] getFloatTokens(String str, String delimiter)
  {
    StringTokenizer strTokenizer = new StringTokenizer( str, delimiter );
    int noOfTokens = strTokenizer.countTokens();
    float[] floatTokens = new float[noOfTokens];
    for(int iToken=0; iToken < noOfTokens; iToken++)
    {
      floatTokens[iToken] = Float.parseFloat( strTokenizer.nextToken() );
    }
    return floatTokens;
  } // getFloatTokens

  /**
   * This method is used to tokenize the int
   */
  private static int[] getIntTokens(String str, String delimiter)
  {
    StringTokenizer strTokenizer = new StringTokenizer( str, delimiter );
    int noOfTokens = strTokenizer.countTokens();
    int[] intTokens = new int[noOfTokens];
    for(int iToken=0; iToken < noOfTokens; iToken++)
    {
      intTokens[iToken] = Integer.parseInt( strTokenizer.nextToken() );
    }
    return intTokens;
  } // getIntTokens

  /**
   * Creates Category
   */
  private static void createPositiveCategoryTarget()
  throws Exception
  {
    String displayName = m_DisplayName;
    String dataType_str = m_DataType;
    String value_str = m_Value;

    if((displayName != null) && (dataType_str != null) && (value_str != null))
    {
      DataType dataType = DataType.getInstance(dataType_str);
      m_PositiveCategoryTarget =  new Category(displayName, value_str, dataType );
    }
    else if((displayName != null) && (value_str != null))
    {
      m_PositiveCategoryTarget = new Category(displayName, value_str);
    }
    else if(value_str != null)
    {
      m_PositiveCategoryTarget = new Category(value_str);
    }
    else
    {
      m_PositiveCategoryTarget = null;
    }
  } // createPositiveCategoryTarget

  /**
  ** Clean up all generated data from a previous execution
  ** @param dataMiningServerHandle MiningServer connection handle.
  **/
  private static void cleanUp(oracle.dmt.odm.Connection dataMiningServerHandle)
  {
    try
    {
      System.out.println("    Removing Mining Function Settings object: " + m_ClassificationFunctionSettingName);
      MiningFunctionSettings.remove(dataMiningServerHandle, m_ClassificationFunctionSettingName);
    }
    catch (Exception e)
    {
      //System.out.println("\tFailed to remove function settings object. Ignoring the exception in cleanup. Continuing ...");
    }
    try
    {
      System.out.println("    Removing Model Seeker result object: " + m_ResultName);
      ModelSeekerResult.remove(dataMiningServerHandle, m_ResultName);
    }
    catch (Exception e)
    {
      //System.out.println("\tFailed to remove model seeker result object. Ignoring the exception in cleanup. Continuing ...");
    }
    try
    {
      System.out.println("    Removing Mining Task object: " + m_TaskName);
      MiningTask.remove(dataMiningServerHandle, m_TaskName);
    }
    catch (Exception e)
    {
      //System.out.println("\tFailed to remove mining task object. Ignoring the exception in cleanup. Continuing ...");
    }
  } // cleanUp

  /**
  ** Execute Model Seeker to select the best model from a list of pre-defined algorithms.
  ** This method uses static variables as inputs. These static
  ** variables have to be initialized before this function is invoked.
  ** @param dataMiningServerHandle MiningServer connection handle.
  **/
  private static void executeModelSeeker(oracle.dmt.odm.Connection dataMiningServerHandle)
  throws Exception
  {
      // Store ClassificationFunctionSettings object into ODM schema.
      m_ClassificationFunctionSettings.store(dataMiningServerHandle, m_ClassificationFunctionSettingName);

      // Create ModelSeekerTask object.
      ModelSeekerTask modelSeekerTask = new ModelSeekerTask(
          m_ResultName,
          m_Prefix,
          m_BuildPhysicalDataSpecification,
          m_TestPhysicalDataSpecification,
          m_ClassificationFunctionSettingName,
          m_LiftQuantiles,
          m_PositiveCategoryTarget,
          m_Weight);

      // Store ModelSeekerTask object.
      modelSeekerTask.store( dataMiningServerHandle, m_TaskName );

      // Execute ModelSeekerTask object.
      modelSeekerTask.execute(dataMiningServerHandle);

      // Wait for completion.
      MiningTaskStatus miningTaskStatus = modelSeekerTask.waitForCompletion( dataMiningServerHandle );

      // Print out Build Duration
      System.out.println("    Built all Model Seeker models");
      long buildDuration = modelSeekerTask.getExecutionDuration(dataMiningServerHandle, m_TaskName)/1000;
      System.out.println("    Task Executed Duration: " + buildDuration + " seconds");

      // Check the completion status of the task.
      MiningTaskState miningTaskState = miningTaskStatus.getTaskState();

      if(miningTaskState.isEqual( MiningTaskState.success ))
      {
        // If successful completion, display results.
        printModelSeekerResult(dataMiningServerHandle);
      }
      else
      {
        // If not successful, display error information.
        System.out.println("Model Seeker Task failed to complete, final status details of this task:");
        System.out.println("    Task State: " + miningTaskState.getEnum() );
        System.out.println("    Task State Text: " + miningTaskStatus.getStateDescription() );
      }
  } // executeModelSeeker

  /**
  ** Print out Model Seeker result.
  ** @param dataMiningServerHandle MiningServer connection handle.
  **/
  private static void printModelSeekerResult(oracle.dmt.odm.Connection dataMiningServerHandle)
  throws Exception
  {
    System.out.println(" ");
    System.out.println("Display Model Seeker Result Phase:");

    // Restore ModelSeekerResult object.
    ModelSeekerResult restoredResult = (ModelSeekerResult)ModelSeekerResult.restore(dataMiningServerHandle, m_ResultName);
    System.out.println("    Restored Model Seeker Result: " + m_ResultName);

    // Get the array of ModelSeekerResultEntry objects.
    ModelSeekerResultEntry [] resultArray = restoredResult.getModelSeekerEntryArray();

    // Get the integer that identifies the best model.
    int iBestModel = restoredResult.getBestModelIndex();
    Integer IBest = new Integer(iBestModel);
    System.out.println("    Index of the best model is " + IBest.toString());

    // Display results for the best model.
    System.out.println(" ");
    System.out.println("        Model " + IBest.toString());
    System.out.println("            Model Name: " + resultArray[IBest.intValue()].getModelName());
    System.out.println("            Algorithm Type: " + resultArray[IBest.intValue()].getModelType().getEnum());
    System.out.println("            Classification Function Settings Name: " + resultArray[IBest.intValue()].getClassificationFunctionSettingsName());
    System.out.println("            Build Duration: " + (resultArray[IBest.intValue()].getBuildDuration())/1000 + " seconds");
    System.out.println("            Test Result Name: " + resultArray[IBest.intValue()].getTestResultName());
    System.out.println("            Lift Result Name: " + resultArray[IBest.intValue()].getComputeLiftResultName());
    System.out.println("            Figure Of Merit: " + resultArray[IBest.intValue()].getFigureOfMerit());

  } // printModelSeekerResult

  /**
  ** Initialize all the static variables using the input property file.
  ** @param fileName Name of the initialization property file.
  **/
  private static void initializeClassVariables (String globalFileName, String fileName)
  throws FileNotFoundException, IOException, Exception
  {
    // Load ODM Global properties
    FileInputStream globalFileStream = new FileInputStream(globalFileName);
    Properties globalPropertyDetails = new Properties();
    globalPropertyDetails.load(globalFileStream);
    globalPropertyDetails = trimStrings(globalPropertyDetails); // Added 2/21/02

    // Read MiningServer details.
    String tagName = "miningServer";
    m_MiningServerUrl = globalPropertyDetails.getProperty(tagName + ".url");
    m_MiningServerUserName = globalPropertyDetails.getProperty(tagName + ".userName");
    m_MiningServerPassword = globalPropertyDetails.getProperty(tagName + ".password");

    // Read output data schema name.
    m_outputSchemaName = globalPropertyDetails.getProperty("outputSchemaName");

    // Load operation specific properties.
    FileInputStream inputFileStream = new FileInputStream(fileName);
    Properties propertyDetails = new Properties();
    propertyDetails.load(inputFileStream);
    propertyDetails = trimStrings(propertyDetails); // Added 2/21/02
    m_PropertyDetails = propertyDetails;

    // Read build data details.
    tagName = "buildData";
    m_BuildDataTableName = propertyDetails.getProperty(tagName + ".tableName");
    m_BuildDataSchemaName = globalPropertyDetails.getProperty("inputDataSchemaName");

    String dataType = propertyDetails.getProperty(tagName + ".type");
    if ( dataType==null || dataType.length() == 0 )
      throw new Exception ( "applyInput.type should be transactional or nonTransactional.");
    m_BuildDataType= DataFormatType.getInstance(dataType);

    if (m_BuildDataType.isEqual(DataFormatType.transactional) ) {
      m_IsBuildTransactional = true;
      // Read transactional data specifications.
      tagName = "buildTransactionalData";
      m_BuildSequenceIdName = propertyDetails.getProperty(tagName + ".sequenceId");
      m_BuildAttributeName = propertyDetails.getProperty(tagName + ".attributeName");
      m_BuildValueName = propertyDetails.getProperty(tagName + ".valueName");
    } else if (m_BuildDataType.isEqual(DataFormatType.nonTransactional) ) {
      m_IsBuildTransactional = false;
    } else
      throw new Exception("Invalid build data type. Valid values are: transactional/nonTransactional.");

     // Read test data details.
    tagName = "testData";
    m_TestDataTableName = propertyDetails.getProperty(tagName + ".tableName");
    m_TestDataSchemaName = globalPropertyDetails.getProperty( "inputDataSchemaName");
    m_TestDataType = propertyDetails.getProperty(tagName + ".type");

    if ( m_TestDataType.compareToIgnoreCase("transactional") == 0 )
    {
      m_IsTestTransactional = true;
      // Read transactional data specifications.
      tagName = "testTransactionalData";
      m_TestSequenceIdName = propertyDetails.getProperty(tagName + ".sequenceId");
      m_TestAttributeName = propertyDetails.getProperty(tagName + ".attributeName");
      m_TestValueName = propertyDetails.getProperty(tagName + ".valueName");
    }
    else if ( m_TestDataType.compareToIgnoreCase("nonTransactional") == 0)
    {
      m_IsTestTransactional = false;
    }
    else
    {
      throw new Exception("Invalid test data type. Valid values are: transactional/nonTransactional.");
    }

    // Classification function settings
    tagName = "classificationFunctionSetting";
    String dataPreparationStatus = propertyDetails.getProperty(tagName + ".dataPreparationStatus");
    m_DataPreparationStatus = DataPreparationStatus.getInstance(dataPreparationStatus);
    m_TargetAttributeName = propertyDetails.getProperty(tagName + ".targetAttributeName");
    String targetAttributeType = propertyDetails.getProperty(tagName + ".targetAttributeType");
    m_TargetAttributeType = AttributeType.getInstance(targetAttributeType);
    m_SupplementalAttributes = propertyDetails.getProperty(tagName + ".supplementalAttributes");

    // Positive Category Target Values
    tagName = "positiveCategoryTarget";
    m_DisplayName = propertyDetails.getProperty(tagName + ".displayName");
    m_DataType = propertyDetails.getProperty(tagName + ".dataType");
    m_Value = propertyDetails.getProperty(tagName + ".value");

    // Model Seeker settings
    tagName = "modelSeeker";
    m_Prefix = propertyDetails.getProperty(tagName + ".prefix");
    m_ClassificationFunctionSettingName = propertyDetails.getProperty(tagName + ".classificationFunctionSettingName");
    m_ResultName = propertyDetails.getProperty(tagName + ".resultName");
    String m_LiftQuantiles_str = propertyDetails.getProperty(tagName + ".liftQuantiles");
    m_LiftQuantiles = Integer.parseInt(m_LiftQuantiles_str);
    String m_Weight_str= propertyDetails.getProperty(tagName + ".weight");
    m_Weight = Float.parseFloat(m_Weight_str);

    // Model Seeker task settings
    tagName = "modelSeekerTask";
    m_TaskName = propertyDetails.getProperty(tagName + ".taskName");
    m_EnableCleanup = propertyDetails.getProperty(tagName + ".enableCleanup");
  } // initializeClassVariables
  
} // end of class Sample_ModelSeeker
