// Copyright (c) 2001, 2003, Oracle Corporation.  All rights reserved.  
// File: Sample_NaiveBayesApply_short.java

/**
 * This sample program demonstrates the use of ODM API to apply the already built Naive
 * Bayes model. The model used here needs to be already built in the MiningServer
 * schema being used in this program. You can use "Sample_NaiveBayesBuild_short.java"
 * to build a Naive Bayes model.
 *
 * Note that this program does not require any property files.
 * 
 * This program is described in detail in ODM Concepts.
 */

// Generic Java Imports
import java.io.*;
import java.util.*;
import java.lang.*;
import java.sql.SQLException;

// ODM Imports
import oracle.dmt.odm.*;
import oracle.dmt.odm.data.*;
import oracle.dmt.odm.settings.function.*;
import oracle.dmt.odm.settings.algorithm.*;
import oracle.dmt.odm.model.*;
import oracle.dmt.odm.task.*;
import oracle.dmt.odm.result.*;

public class Sample_NaiveBayesApply_short extends Object {
  public static void main ( String[] args ) {
    System.out.println("Start: " + new java.util.Date());

    DataMiningServer dms = null;
    oracle.dmt.odm.Connection dmsConnection = null;
    try {
      // Create an instance of the data mining server and get a connection 
      // The mining server URL, user_name and password need to be specified
      dms = new DataMiningServer("DB_URL", "odm", "password");

      // Get the actual connection
      dmsConnection = dms.login();

      // Perform cleanup first before execute the program
      try {
        // Remove apply result object
        MiningApplyResult.remove(dmsConnection, "Sample_NB_APPLY_RESULT_Short" ,true);
        System.out.println("    Removed apply result object: " + "Sample_NB_APPLY_RESULT_Short");
      } catch (Exception e) {}
      try {
        // Remove MiningTask object
        MiningTask.remove(dmsConnection, "Sample_NB_APPLY_TASK_Short");
        System.out.println("    Removed mining task: " + "Sample_NB_APPLY_TASK_Short");
      } catch (Exception e) {}

      // This program illustrates how to score a whole table. 
      // Create PhysicalDataSpecification for input table
      // First create a LocationAccessData using the table name (e.g., CENSUS_2D_APPLY_UNBINNED)
      // and schema name
      LocationAccessData lad = new LocationAccessData("CENSUS_2D_APPLY_UNBINNED", "odm_mtr");

      // Create a NonTransactionalDataSpecification object since the dataset is nontransactional
      PhysicalDataSpecification m_PhysicalDataSpecification =
         new NonTransactionalDataSpecification(lad);

      // LocationAccessData for output table to store the apply results
      LocationAccessData ladOutput = new LocationAccessData("Sample_NB_APPLY_RESULT_Short", "odm");

      // Create MiningApplyOutput object
      MiningApplyOutput m_MiningApplyOutput = new MiningApplyOutput();

      // Create Attribute objects that specifies the names and data types
      // of the prediction, probability, and rank columns in the output
      Attribute predictionAttribute  = new Attribute("myprediction", DataType.stringType);
      Attribute probabilityAttribute = new Attribute("myprobability", DataType.stringType);
      Attribute rankAttr             = new Attribute("myrank", DataType.stringType);

      // Create the ApplyTargetProbabilityItem instance
      ApplyTargetProbabilityItem aTargetAttrItem = new ApplyTargetProbabilityItem(predictionAttribute, probabilityAttribute, rankAttr);

      // Create Category objects to represent the target values
      // to be included in the apply output table. In this example two target
      // values are specified.
      Category target_category  = new Category("negative_class", "0", DataType.getInstance("int"));
      Category target_category1 = new Category("negative_class", "1", DataType.getInstance("int"));

      // Add the target values to the ApplyTargetProbabilityItem instance
      aTargetAttrItem.addTarget(target_category);
      aTargetAttrItem.addTarget(target_category1);

      // Add the ApplyTargetProbabilityItem to the MiningApplyOutput object
      m_MiningApplyOutput.addItem(aTargetAttrItem);


      // Add all the source attributes that are to be returned with the scored result.
      // In this example, attribute "PERSON_ID" from the source table will be returned
      // with the result aliased as "ID"
      MiningAttribute sourceAttribute = new MiningAttribute("PERSON_ID", DataType.intType,
                               AttributeType.notApplicable);
      Attribute destinationAttribute = new Attribute("ID",DataType.intType);

      ApplySourceAttributeItem m_ApplySourceAttributeItem =
         new ApplySourceAttributeItem(sourceAttribute,destinationAttribute);

      // Add the ApplySourceAttributeItem object to the MiningApplyOutput object
      m_MiningApplyOutput.addItem(m_ApplySourceAttributeItem);

      // validate the MiningApplyOutput object
      m_MiningApplyOutput.validate();

      // Create a Naive Bayes apply task and execute it.
      MiningApplyTask task = new MiningApplyTask(
          m_PhysicalDataSpecification, // Physical Data Spec for input data
          "Sample_NB_Model_Short", // Name of the model to be applied
          m_MiningApplyOutput, // MiningApplyOutput
          ladOutput, // Output table location
          "Sample_NB_APPLY_RESULT_Short"); // Result name in odm_apply_result table
      task.store(dmsConnection, "Sample_NB_APPLY_TASK_Short");
      task.execute(dmsConnection);
      MiningTaskStatus status = task.waitForCompletion(dmsConnection);

      // Print the status and the time taken for the apply
      System.out.println("    Status: " + status.getTaskState().getEnum());
      long applyDuration = task.getExecutionDuration(dmsConnection, "Sample_NB_APPLY_TASK_Short")/1000;
      System.out.println("    Duration: " + applyDuration + " seconds");

      // Display Apply Results
      MiningApplyResult miningApplyResult = MiningApplyResult.restore(dmsConnection, "Sample_NB_APPLY_RESULT_Short");

      if (miningApplyResult == null)
        throw new Exception("    Apply result: Sample_NB_APPLY_RESULT_Short not found.");
      System.out.println(miningApplyResult.toString());
    } catch ( Exception e) {
      System.out.println("    Received an exception in main: " + e.getMessage());
    } finally {
      System.out.println("\nLogout Phase:");
      try {
        dms.logout(dmsConnection);
        System.out.println("    Logging out from the MiningServer");
      } catch (Exception e) {
        System.out.println("    Failed to logout. Ingnoring the exception. Continuing ...");
      }
    }
  }
}
