// Copyright (c) 2001, 2002, Oracle Corporation.  All rights reserved.  
// File: Sample_NaiveBayesBuild_short.java

/**
 * This sample program illustrates the use of ODM API to build a Naive
 * Bayes model. After a model is built, the name of the model can be 
 * passed to the following programs:
 * Sample_NaiveBayesLiftAndTest.java
 * Sample_NaiveBayesApply.java
 * Sample_NaiveBayesApply_short.java
 *
 * This program does not use any property files.
 *
 * This program is described in detail in the ODM Concepts manual.
*/

// Generic Java Imports
import java.io.*;
import java.util.*;
import java.sql.SQLException;

// ODM Imports
import oracle.dmt.odm.*;
import oracle.dmt.odm.data.*;
import oracle.dmt.odm.settings.function.*;
import oracle.dmt.odm.settings.algorithm.*;
import oracle.dmt.odm.model.*;
import oracle.dmt.odm.task.*;

public class Sample_NaiveBayesBuild_short extends Object {
  public static void main ( String[] args ) {
    System.out.println("Start: " + new java.util.Date());

    DataMiningServer dms = null;
    oracle.dmt.odm.Connection dmsConnection = null;
    try {
      // Create an instance of the data mining server and get a connection 
      // The mining server URL, user_name and password need to be specified
      dms = new DataMiningServer("DB_URL", "odm", "password");
      dmsConnection = dms.login();

      // Perform cleanup first before execute the program
      try {
        // Remove MiningFunctionSettings object
        ClassificationFunctionSettings.remove(dmsConnection, "Sample_NB_MFS_Short");
        System.out.println("    Removed function settings object: " + "Sample_NB_MFS_Short");
      } catch (Exception e) {}
      try {
        // Remove MiningModel object
        MiningModel.remove(dmsConnection, "Sample_NB_Model_Short");
        System.out.println("    Removed mining model object: " + "Sample_NB_Model_Short");
      } catch (Exception e) {}
      try {
        // Remove MiningTask object
        MiningTask.remove(dmsConnection, "Sample_NB_Build_Task_Short");
        System.out.println("    Removed mining task: " + "Sample_NB_Build_Task_Short");
      } catch (Exception e) {}   

      // Create PhysicalDataSpecification object
      // First create a LocationAccessData using the table name and schema name
      LocationAccessData lad = new LocationAccessData("CENSUS_2D_BUILD_UNBINNED", "odm_mtr");
      // Create a NonTransactionalDataSpecification object since the dataset is nontransactional
      PhysicalDataSpecification m_PhysicalDataSpecification = new NonTransactionalDataSpecification(lad);

      // Create the Naive Bayes algorithm settings by setting the thresholds to 0.01
      NaiveBayesSettings algorithmSetting = new NaiveBayesSettings(0.01f, 0.01f);

      // Specify "class" as the target attribute name and set the DataPreparationStatus
      // to unprepared. Automatic binning will be applied in this case
      ClassificationFunctionSettings m_ClassificationFunctionSettings = ClassificationFunctionSettings.create(
         dmsConnection, algorithmSetting, m_PhysicalDataSpecification,
         "class", AttributeType.categorical, DataPreparationStatus.getInstance("unprepared"));
    
      // Validate and store the ClassificationFunctionSettings object with the name "Sample_NB_MFS_Short"
      m_ClassificationFunctionSettings.validate();
      m_ClassificationFunctionSettings.store(dmsConnection, "Sample_NB_MFS_Short");

      // Create a Naive Bayes build task and execute it.
      // MiningFunctionsSettings name and the ModelName need to be specified
      MiningBuildTask task = new MiningBuildTask(m_PhysicalDataSpecification, "Sample_NB_MFS_Short", "Sample_NB_Model_Short");
      // Store the task under the name "Sample_NB_Build_Task_Short"
      task.store(dmsConnection, "Sample_NB_Build_Task_Short");
      task.execute(dmsConnection);
      MiningTaskStatus status = task.waitForCompletion(dmsConnection);

      // Print the status and the time taken for the build
      System.out.println("    Status: " + status.getTaskState().getEnum());
      long buildDuration = task.getExecutionDuration(dmsConnection, "Sample_NB_Build_Task_Short")/1000;
      System.out.println("    Duration: " + buildDuration + " seconds");
    } catch ( Exception e) {
      System.out.println("    Received an exception in main: " + e.getMessage());
    } finally {
      System.out.println("\nLogout Phase:");
      try {
        dms.logout(dmsConnection);
        System.out.println("    Logging out from the MiningServer");
      } catch (Exception e) {
        System.out.println("    Failed to logout. Ingnoring the exception. Continuing ...");
      }
    }
  }
}
