// Copyright (c) 2001, 2002, Oracle Corporation.  All rights reserved.  
// File: Sample_PMMLExport.java

/**
 * This sample program demonstrates the use of ODM API to export a mining
 * model conforming to the PMML standard.
 *
 * PMML MODEL EXPORT
 * 
 * This program exports a mining model from the ODM schema
 * into an XMLType column in a database table. In order
 * for this program to run, an Association Rules or
 * Naive Bayes model must have been created in the ODM schema.
 *
 * SAMPLE PROGRAM DESIGN
 * 
 * This program has two input property files:
 *  1. Sample_PMMLExport.property
 *  2. Sample_Global.property
 *
 *  "Sample_PMMLExport.property" specifies the inputs required
 * to perform the model export.
 * 
 *  "Sample_Global.property" specifies the Data Mining Server connectivity
 * information.
 *
 * EXECUTION REQUIRMENTS
 * 
 *  This test program requires that an Association Rules or Naive-Bayes
 *  mining model exists in the ODM schema. These models can be created
 *  using other included sample programs.
 *  
 *  NOTE: If the Sample_PMMLExport.property file has enableCleanup=cleanupBeforeExecution,
 *  then even the PMML Model output table will be removed at re-execution. 
 */
// Generic Java Imports
import java.sql.*;
import javax.jms.*;
import java.io.*;
import java.util.*;

// Oracle Imports
import oracle.jms.*;
import oracle.AQ.*;
import oracle.jdbc.*;
import oracle.dmt.odm.task.ModelExportTask;
import oracle.dmt.odm.LocationCellAccessData;
import oracle.dmt.odm.MiningStandardType;
import oracle.dmt.odm.DataMiningServer;
import oracle.dmt.odm.model.MiningModel;
import oracle.dmt.odm.task.MiningTask;
import oracle.dmt.odm.task.MiningTaskStatus;
import oracle.dmt.odm.MiningTaskState;

public class Sample_PMMLExport extends Object {

  // Tag: miningServer
  private static String m_MiningServerUrl; // URL for the DataMiningServer
  private static String m_MiningServerUserName; // User name for the DataMiningServer
  private static String m_MiningServerPassword; // Password for the DataMiningServer

  // Tag: null
  private static String m_InputDataSchemaName; // Schema where the input dataset is read from
  private static String m_OutputSchemaName; // Schema where the output database object/s will be created

  // Tag: miningTask
  private static String m_TaskName; // Mining Task name

  // Tag: modelExport
  private static String m_ModelName; // Name of Model to export
  private static String m_ModelExportSchemaName; // Schema to export model to
  private static String m_ModelExportTableName; // Table to export model to
  private static String m_ModelExportColumnName; // Column to export model to
  private static String m_ModelExportRowid; // Rowid to export model to

  // Tag: null
  private static String m_EnableCleanup = "cleanupBeforeExecution"; //This memeber is used to decide whether to cleanup the task related stuff before executing

  //Output formatting
  private static String INDENT = "    ";
  private static String DIVIDER = "***********************************************************************************";

  // Following variables are not directly read from the property file:
  private static oracle.dmt.odm.Connection m_dmsConnection = null;
  private static ModelExportTask m_ModelExportTask;

  /**
  **
  ** This is the code entry point
  **
  */
  public static void main(String[] args)
  {
    if ( args.length < 2 ) {
      System.out.println("Please enter global and sample " +
        "property file names. i.e., Sample_Global.property " +
        "& Sample_PMMLExport.property" );
      return;
    }
    System.out.println(DIVIDER);
    System.out.println("ODM 9.2.0 Sample Program: Sample_PMMLExport");
    System.out.println(DIVIDER);
    System.out.println("Property Files:");
    System.out.println(INDENT + "Global: " + args[0]);
    System.out.println(INDENT + "Sample: " + args[1]);
    System.out.println(DIVIDER);
    System.out.println("Start: " + new java.util.Date());

    DataMiningServer dms = null;
    try {
      // Initialize the class varaibles from the property file.
      initializeClassVariables(args[0], args[1]);
      System.out.println("\nInitialization Phase:");
      System.out.println(INDENT + "Data Mining Server:");
      System.out.println(INDENT + INDENT + "JDBC URL: " + m_MiningServerUrl);
      System.out.println(INDENT + INDENT + "Username: " + m_MiningServerUserName);

      // Log into the MiningServer
      System.out.println("\nLogin Phase:");
      dms = new DataMiningServer(
        m_MiningServerUrl,
        m_MiningServerUserName,
        m_MiningServerPassword);
      m_dmsConnection = dms.login();
      System.out.println(INDENT + "Completed MiningServer login");

      //Check the cleanup type cleanupBeforeExecution, cleanupOnly, noCleanup
      if(m_EnableCleanup.equalsIgnoreCase("noCleanup")) {
        //do nothing
      } else if(m_EnableCleanup.equalsIgnoreCase("cleanupOnly")) {
        performCleanup();
        return;
      } else {
        performCleanup();
      }
      // Form a LocationCellAccessData object
      System.out.println("\nData Setup Creation Phase:");
      oracle.sql.ROWID rowID = null;
      if (m_ModelExportRowid.equals(""))
      {
        try {
          String sCmd1 = "create table " + m_ModelExportSchemaName +
                        "." + m_ModelExportTableName +
                        " (" + m_ModelExportColumnName +
                        " SYS.XMLTYPE)";
          Statement stmt = m_dmsConnection.getDBConnection().createStatement();
          stmt.execute(sCmd1);
          //insert a row into the table
          String sCmd2 = "insert into " + m_ModelExportSchemaName +
                          "." + m_ModelExportTableName +
                          " values(null)";
          stmt.execute(sCmd2);
          //select the rowid
          String sCmd3 = "select rowid from " + m_ModelExportSchemaName +
                          "." + m_ModelExportTableName;
          OracleResultSet ors = (OracleResultSet) stmt.executeQuery(sCmd3);
          ors.next();
          rowID = ors.getROWID(1);
          ors.close();
          stmt.close();
        } catch(Exception e) {
          String expMsg = "Please provide a ROWID for the table: " +
                          m_ModelExportSchemaName + "." +
                          m_ModelExportTableName;
          throw new Exception(expMsg);
        }
      }
      else
      {
        rowID = new oracle.sql.ROWID(m_ModelExportRowid.getBytes());
      }
      LocationCellAccessData outputDataLocation = new LocationCellAccessData(
        m_ModelExportTableName,
        m_ModelExportSchemaName,
        m_ModelExportColumnName,
        rowID);
      System.out.println(INDENT + "Created LocationCellAccessData");
      System.out.println(INDENT + INDENT + "table name: " + m_ModelExportTableName);
      System.out.println(INDENT + INDENT + "schema name: " + m_ModelExportSchemaName);
      System.out.println(INDENT + INDENT + "column name: " + m_ModelExportColumnName);
      String s = new String(rowID.getBytes());
      System.out.println(INDENT + INDENT + "rowid: " + s);

      //Create the Model Export task
      System.out.println("\nModel Export Task Phase:");
      m_ModelExportTask = new ModelExportTask(
        m_ModelName,
        outputDataLocation,
        MiningStandardType.PMML20);
      System.out.println(INDENT + "Created Model Export Task");

      //Store the task
      m_ModelExportTask.store(m_dmsConnection, m_TaskName);
      System.out.println(INDENT + "Persisted Model Export Task.");
      System.out.println(INDENT + INDENT + " Name: " + m_TaskName);

      //Execute the task
      m_ModelExportTask.execute(m_dmsConnection);
      System.out.println(INDENT + "Executing Model Export Task");

      //Wait for completion
      long timeoutInterval = 120;
      MiningTaskStatus miningTaskStatus =
        m_ModelExportTask.waitForCompletion(m_dmsConnection, timeoutInterval);
      long exportDuration = m_ModelExportTask.getExecutionDuration(m_dmsConnection, m_TaskName)/1000;

      //Check the success of the task
      MiningTaskState miningTaskState =
        miningTaskStatus.getTaskState();
      if(miningTaskState.isEqual(MiningTaskState.success))
      {
        System.out.println(INDENT + "Status: SUCCESS");
        System.out.println(INDENT + "Duration: " + exportDuration + " seconds");
     } else if (miningTaskState.isEqual(MiningTaskState.error))
     {
        System.out.println("Model Export Task failed to complete, final status details of this task:");
        System.out.println(INDENT + "Task State: " + miningTaskState.getEnum() );
        System.out.println(INDENT + "Task State Text: " + miningTaskStatus.getStateDescription() );
        System.out.println(INDENT + "Duration: " + exportDuration + " seconds");
      } else
      {
        System.out.println("Model Export Task failed to complete, final status details of this task:");
        System.out.println(INDENT + "Task State: " + miningTaskState.getEnum() );
        System.out.println(INDENT + "Task waitForCompletion timed out. Consider increasing timeout interval." );
      }
    } catch ( Exception e) {
      System.out.println("Received an exception in main: " + e.getMessage());
    } finally {
      System.out.println("\nLogout Phase:");
      try {
        dms.logout(m_dmsConnection);
        System.out.println(INDENT + "Completed MiningServer logout");
      } catch ( Exception e) {
        System.out.println(INDENT + "MiningServer logout failed");
      }
      System.out.println("\nEnd: " + new java.util.Date());
      System.out.println(DIVIDER);
    }
  }

  /**
  ** Initialize all the static variables using the input property file.
  ** @param fileName Name of the initialization property file.
  **/
 protected static void initializeClassVariables (String globalFileName, String fileName)
  throws FileNotFoundException, IOException, Exception {
    // Load ODM Global properties
    FileInputStream globalFileStream = new FileInputStream(globalFileName);
    Properties globalPropertyDetails = new Properties();
    globalPropertyDetails.load(globalFileStream);
    trimStrings(globalPropertyDetails);

    // Read MiningServer details.
    String tagName = "miningServer";
    m_MiningServerUrl = globalPropertyDetails.getProperty(tagName + ".url");
    m_MiningServerUserName = globalPropertyDetails.getProperty(tagName + ".userName");
    m_MiningServerPassword = globalPropertyDetails.getProperty(tagName + ".password");

    // Read input data schema name.
    m_InputDataSchemaName = globalPropertyDetails.getProperty("inputDataSchemaName");
    m_OutputSchemaName = globalPropertyDetails.getProperty("outputSchemaName");

    // Load operation specific properties.
    FileInputStream inputFileStream = new FileInputStream(fileName);
    Properties propertyDetails = new Properties();
    propertyDetails.load(inputFileStream);
    trimStrings(propertyDetails);

    // Read cleanup option
    m_EnableCleanup = propertyDetails.getProperty("enableCleanup");
    if ( m_EnableCleanup == null || m_EnableCleanup.length() == 0 )
      m_EnableCleanup = "cleanupOnly";

    // Mining task related
    tagName = "miningTask";
    m_TaskName = propertyDetails.getProperty(tagName + ".name");
    m_ModelName = propertyDetails.getProperty(tagName + ".modelName");

    // Model Export related
    tagName = "modelExport";
    m_ModelExportTableName = propertyDetails.getProperty(tagName + ".tableName");
    m_ModelExportSchemaName = propertyDetails.getProperty(tagName + ".schemaName");
    m_ModelExportColumnName = propertyDetails.getProperty(tagName + ".columnName");
    m_ModelExportRowid = propertyDetails.getProperty(tagName + ".rowid");
  }

  /**
  ** performCleanup
  ** Reads the value in m_EableCleanup and performs appropriate action.
  ** This function catches and ignores all the exceptions: after printing the exception.
  ** @param dataMiningServerHandle DataMiningServer handle
  **/
  public static void performCleanup ()
  {
    System.out.println("\nCleanup Phase:");
    System.out.println(INDENT + "Cleanup flag: " + m_EnableCleanup);

    //Remove MiningTask
    try {
      MiningTask.remove(m_dmsConnection, m_TaskName);
    } catch(Exception e) {}
    //Remove output table
    try {
      String sCmd1 = "drop table " + m_ModelExportSchemaName +
                        "." + m_ModelExportTableName;
      Statement stmt = m_dmsConnection.getDBConnection().createStatement();
      stmt.execute(sCmd1);
      stmt.close();
    } catch(Exception e) {}
  }

  public static Properties trimStrings(Properties inputProps)
  {
    Enumeration keys = inputProps.keys();
    Properties outputProps = new Properties();
    while(keys.hasMoreElements())
    {
      String key = (String)keys.nextElement();
      String value = inputProps.getProperty(key);
      outputProps.put(key, value.trim());
    }
    return outputProps;
  }
}


