#
# Copyright (c) 2001, 2006, Oracle. All rights reserved.  
#
#    NAME
#    Apply.pm
#
#    DESCRIPTION
#    This file contains the methods for the apply command to opatch.
#
#    NOTES: 1. The reporting needs to be improved.
#           2. An undo file needs to be implemented. The commands should
#              be written out to the file just before the file is patched.
#              This is _after_ the back-up is done. This will allow a
#              back-out for novices who may have an broken environment
#              and can't raise Oracle's Support.
#           3. To reapply an existing patch it needs to be done as a
#              rollback or apply -force.
#
#    BUGS
#    * <if you know of any bugs, put them here!>
#
#    MODIFIED   (MM/DD/YY)
#    phnguyen    01/12/06 - Add note for auto-rollback case.
#    vsriram     01/11/06 - Search for FAIL in make output with case sensitive.
#    vsriram     12/15/05 - Bug 4883881, look for java in jdk/jre/bin/java before searching.
#    vsriram     12/01/05 - Take care of make actions in is_noop_patch.
#                           Print the exact make targets that is run.
#    vsriram     11/11/05 - Skip verify_patch for -no_inventory.
#    vsriram     10/26/05 - Use escape_special_chars, to escape $ before writing to rollback.sh
#    vsriram     10/10/05 - Invoke clean_up_inventory() to clean dangling filemaps just before update inventory.
#                           Do live printing for pre/post scripts.
#                           Check if patch is a no-op patch.
#    vsriram     09/26/05 - Bug-4628014 - Save the original Oracle Home path and use it for make.
#                           Make error handling algorithm is changed as per the new design.
#    vsriram     09/01/05 - Correcting the typo.
#    vsriram     08/02/05 - Bug-4243203 : Added few log/debug messages.
#    vsriram     07/29/05 - Bug-4478852 : Delete new files introduced by archive actions.
#    vsriram     06/26/05 - Bug-4450313 : Decouple fuser from pre script. Error out
#				          if there are active instances, before doing conflict checks.
#    shgangul    06/21/05 - Do not perform mkdir -p for windows 
#    vsriram     06/16/05 - 
#    vsriram     06/09/05 - Print the "What is the cause" of answering Y or N during errors.
#    vsriram     05/18/05 - Use get_abs_path.
#    phnguyen    05/07/05 - Special note on Windows if can't load inventory
#                              due to command.exe not in PATH
#                         - Pass in OPATCH_IS_SHARED (-DOPatch.is_shared)
#                             as true, false, or detect
#       phnguyen 05/05/05 - In the case of auto-rollback with rolling or min
#                             donwtime, instead of error out, change to
#                             all-node mode.
#       vsriram  04/27/05 - Convert Oracle Home to absolute path
#				Log for "next if".
#       phnguyen 04/26/05 - Jar pre-req check
#                           Error out if rolling or min dt with auto-rb
#	vsubrahm 04/05/05 - Bug 4256736: "-silent -force" is not working in patch conflicts and both patch conflict and superset case.
#       vsubrahm 04/05/05 - Bug 4247450: regular expression causing
#			 OPatch failing to parse comp. in "actions" file
#       shgangul 02/15/05 - Bug 4088969: Do not execute pre/post in report 
#                           mode 

#    phnguyen    01/24/05 - If error out because of RAC && rolling && auto-rollback,
#                              remove lock file.
#                           Remove no-support warning on -report mode.
#    phnguyen    01/20/05 - Put new dirs. that are created into a file
#    shgangul    01/18/05 - Add debug messages for make errors 
#    shgangul    01/18/05 - grabtrans 'phnguyen_fix_emptyline' 
#    shgangul    01/17/05 - Bug 4124475: Do not report error for non-zero 
#                           STDERR 
#    phnguyen    01/14/05 - Fix bug in ^M in 'actions' file.
#    shgangul    01/11/05 - Change message before copy comps.xml 
#    phnguyen    01/08/05 - Back up comps.xml after applying as comps.xml.afterapply
#    shgangul    01/06/05 - Fix the logging issue with auto rollback 
#    vsriram     01/05/05 - Rollback:Delete new files 
#    shgangul    01/05/05 - Control filemap backup feature using env var 
#                           OPATCH_BACKUP_FILEMAP 
#    phnguyen    01/05/05 - 
#                           Put timestamp check point into
#                             Start of log file
#                             Start of system patching
#                             Start of inventory update
#                             Start of RAC file propagation
#                             End of session
#    phnguyen    01/04/05 - Use getuid() to check for root, overriden by
#                                OPATCH_SKIP_ROOT_CHECK
#    shgangul    01/04/05 - Copy filemap with comps.xml 
#    shgangul    01/03/05 - Change logging info for optional comp checks 
#    shgangul    12/31/04 - Bug 4085436: Added new error for make failure 
#    phnguyen    12/30/04 - Set up .patch_storage even in -report, fix bug 4088969
#                           User the env. var. 'USER' to check if OPatch is 
#                              invoked as 'root'.  This 'USER' should be avai.
#                              on all platforms since Java uses it as system
#                              property 'user.name'.  Tested on Solaris & Linux.
#                              Fix has no effect on Windows (tested on Windows).
#                              (bug 4085419)
#                           (COMMENT OUT)
#    shgangul    12/17/04 - Use getArchiveCommand() to get ar command 
#    shgangul    12/16/04 - Display make error to stdout 
#    phnguyen    12/03/04 - Change patch conflict/superpset message AGAIN, bug 4002593
#                         - Comment out code to warn usrs
#                           about auto-rolling back --> user, if roll back this patch,
#                           has to re-apply auto-rolled back patches
#                         - Correctly parse componet/version given back
#                            from Java, use both comp and ver to
#                            decide if a comp actions should be skipped
#    shgangul    11/26/04 - Further changes to rollback.sh script 
#    shgangul    11/22/04 - Add ORACLE_HOME to rollback.txt and make.txt 
#    phnguyen    11/15/04 - Change message text for conflict/superset cases, bug 4002593 fix
#    phnguyen    11/12/04 - Use $ORACLE_HOME instead of ORACLE_HOME in <ID>_make.txt
#    phnguyen    11/11/04 - Add -no_relink
#    phnguyen    10/13/04 - Implement getMakeCommand().  Error out if 'make'
#                              is required but not available in predefined
#                              path or in PATH
#    shgangul    10/07/04 - Added help for opatch_pre_end and opatch_post_end
#    shgangul    10/07/04 - Bug 3718438: Perl hash issue
#    phnguyen    10/06/04 - fuser errmg touch-up
#    phnguyen    10/04/04   - Make $patch_inventory_location to be absolute
#                               path instead of a relative path to where
#                               we launch OPatch
#    phnguyen    10/02/04   - Do not support auto-rollback with min. downtime 
#                               or rolling patch on RAC.
#    phnguyen    10/02/04   - Print out warning about checking binaries on 
#                               remote nodes
#    phnguyen    09/29/04   - print out clearly that patch won't be propagated
#                                if CFS is detected
#                           - remove extra check on $local_node_only because
#                                shouldPropagatePatch() would look into that
#                                Boolean (this is prior to calling
#                                propagate_patch_to_RAC_nodes())
#    phnguyen    09/20/04 - Use  backupCompsXML() function to back up
#                              OH comps.xml to .patch_storage/<ID> after
#                              all pre-reg. check.
#
#    phnguyen    06/07/04 - chanege env. var. to OPATCH_REMOTE_SHELL
#    shgangul    05/20/04 - expose env variables for patch operation and add 
#    phnguyen    05/28/04 - check for env. var. OPATCH_RSH
#                           new parameters to pre/post 
#    shgangul    05/18/04 - add -opatch_pre_end and -opatch_post_end options 
#    phnguyen    04/20/04 - Eliminate dup. call to get_os_id(), merge with Shamik
#    shgangul    04/13/04 - Add -pre and -post options to Apply and Rollback 
#    phnguyen    04/13/04 - reverse Shamik's change to <ID>_make.txt
#                           since this is used in Command / RollBack
#    phnguyen    04/12/04 - fix bug in <ID>_make.txt file naming
#    phnguyen    04/02/04 - use -oh if supplied
#    phnguyen    03/08/04 - change the -silent behavior to error out on
#                              Apply::apply_patches_to_files()
#                              Apply::create_and_run_make_command_list()
#                           The following error-handling routines stays unchanged:
#                              Apply::backup_files_before_patching()
#                              RollBack::remove_an_applied_patch()
#                              RollBack::get_and_run_make_commands()
#                              RollBack::remove_an_applied_patch()
#                           Re-release as 1.48
#    phnguyen    03/02/04 - initialize PATCH_VERSION, PATCH_DESC and PATCH_CREATION_DATE
#    phnguyen    02/11/04 - use built-in substr function
#    phnguyen    02/10/04 - fix Unicode problem on Win with \p in OH
#    phnguyen    02/02/04 - Work-around for Active State Perl
#    phnguyen    01/30/04 - remove extra "" in Win, fix cdrom prob
#    phnguyen     1/07/03 - remove syntax check for (-local && -no_inventory)
#    phnguyen    12/30/03 - bug 3344568 3344464 
#    shgangul    12/23/03 - bug 3334888: added a new error message 
#    phnguyen    12/18/03 - touch up output in case of pre-script and RAC 
#                           check if -silent is used on RAC
#    phnguyen    12/15/03 - pass in -Dopatch.debug & cluster debug if OPATCH_DEBUG is set to TRUE
#    phnguyen    12/10/03 - fault-tolerant: default to continue 
#    phnguyen    12/09/03 - process stderr of make
#    shgangul    12/08/03 - bug 3257294 
#    phnguyen    12/08/03 - fix Windows problem with class path
#    phnguyen    12/02/03 - break up Unix chain cmd "cd $path; jar..."
#                            into separate cmds so that it works on Win.
#    phnguyen    12/01/03 - move debug & log file upfront
#                           add retry/delay options with defaul 10/30
#    phnguyen    11/26/03 - Shamik's feedback 
#    phnguyen    11/18/03 - all Java classes are in opatch package
#                           so we need to run as opatch/LsInventory or
#                           opatch/LsHome (the / is also used for Windows)
#    shgangul    11/17/03 - bug 3252933: Return error for RAC 
#    shgangul    10/30/03 - touchup custom copy changes 
#    phnguyen    10/29/03 - fix 'custom' copy
#    phnguyen    10/27/03 - remove wrong text message regarding min-downtime
#    phnguyen    10/22/03 - adjust RAC & fuser prompt depending on -local,
#                              rolling, all-node or min. downtime.
#    phnguyen    10/21/03 - touch up stdout
#    phnguyen    10/20/03 - correct prev. merge error
#    shgangul    10/17/03 - changes to copy everything under custom 
#    phnguyen    10/10/03 - touch up help 
#    phnguyen    10/08/03 - touch up non-verbose output, fix bug
#                           to have backup_jar_file() use -jdk option
#    shgangul    10/07/03 - rollback.sh fix, part of bug 3170021 
#    shgangul    10/07/03 - display pre/post.txt irrespective of verbose 
#    phnguyen    10/06/03 - Add -jre and -jdk option
#    phnguyen    10/03/03 - Pre/post processing & copy to .patch storage
#    phnguyen    09/29/03 - Merge bug-superset and file conflict
#    shgangul    09/26/03 - Further changes to pre post functionality
#    shgangul    09/24/03 - Add changes for pre post script
#    shgangul    09/19/03 - add error_on_conflict option
#    shgangul    09/11/03 - Logging changes for 10G
#    shgangul    08/13/03 - Code changes for BUG 2899335
#    phnguyen    08/05/03  v1.43, merge in rollback fix from Shamik
#    phnguyen    06/10/03  v1.39
#
#    phnguyen    05/30/03  v1.37
#    phnguyen    05/28/03  v1.36
#
#    phnguyen    05/28/03  v1.35
#                          final_extract_of_file_names(): the
#                             name is already fully-qualified
#    phnguyen    05/25/03  Implement a primitive hashCode()
#                            so that we can use filename as
#                            key to the hash table
#                            Instead of using the filename directly
#                            as key to hash, we now go thru
#                            Command::encodeName()/decodeName()
#                          "mkdir -p" if the action entry is a new
#                            directory
#    phnguyen    05/21/03  Move RAC chcking code to Command.pm
#                          Check cmd. arg. conflicts
#                          Use util. func. is_option_set()
#                          No need to call CheckRACDetails
#    phnguyen    05/16/03  Get Comp name and ver. during xml parsing
#                            (watch for multiple comp. in inv. file)
#                          Version 32 to support EM
#    phnguyen    05/15/03  Give 1st priority to OH/oui, version 31
#                          Output RAC info. only in case of errors
#    phnguyen    05/14/03  Check to make sure all must-have commands
#                            are in PATH
#                          Check Cluster health to detect known issues
#                            such as "can't get local node name",
#                            zero node count, etc.
#    phnguyen    04/29/03  Fall back to no_inventory if best_try
#                            is specified
#                            is specified
#                          Use ERROR_CODE_INVENTORY_PROBLEM and
#                              ERROR_CODE_PERL_PROBLEM in
#                              opatchIO->print_message_and_die()
#    phnguyen    04/14/03  fix problem when ORACLE_HOME is a sym. link
#                          check if ORACLE_HOME exist, warn if it's
#                          a symbolic link
#
#    phnguyen  	 04/04/03  regular expression problem on Win (2751325)
#                          CHILD_ERROR on Linux (2887659)
#                          Need detailed error message if lock file
#                              exists (2885756)
#                          Roll back AIX 5L as 212 -- not 211 (2889700)
#    daevans     03/07/03  Minor fix for error conditions.
#    daevans     02/28/03  Fix for bug 2835213.
#    daevans     02/28/03  Fixed "local node" issue for RAC installations,
#                          implemented rolling RAC patching and added
#                          clustered file system support.
#    daevans     01/31/03  Added support for the rollback command to
#                          identify when "rolling RAC" is allowed.
#    daevans     12/20/02  Fix issues with "-n" not running make and leaving
#                          the lock flag behind.
#    daevans     12/06/02  Add support for installations that used the
#                          unsupported "invPtrLoc" flag.
#    daevans     12/05/02  Add more log output to the 'apply' phase.
#    daevans     11/27/02  Fix for bug 2681319.
#    daevans     10/30/02  Fix for bug 2582512.
#    daevans     10/17/02  Fix for bugs 2607816, 2626988, 2582512, 2569528,
#                          25629289 (may need to cover more cases) and
#                          introduce support for Win98 clinets.
#    daevans     09/30/02  Bug 2601400 - wrong print statement for $^V.
#    daevans     08/01/02  Many bug fixes, added file based locking.
#    daevans     06/01/02  Many code changes. Stable check-in.
#    daevans     12/03/01  Inital code.
#
##########################################################################

###############################################################################
###############################################################################
#
#  ------------------------ INITIALIZATION ------------------------
#
###############################################################################
###############################################################################

package Apply;
@ISA = ("Command");

######
#
# Standard modules:
#
use English;         # Let us say "$CHILD_ERROR" instead of "$?", etc.
use strict;          # Enforce strict variables, refs, subs, etc.

use Cwd();
use Cwd;
use File::Copy();
use File::Basename();
use File::Copy;
use File::Find;
use File::Path;

# to call getuid()
use POSIX;

# These constants are used for navigating the patch area.
use constant PATCH_BASE  => "files";

# These constants are used for the file details map.
use constant AR_STRING   => "archive";
use constant COPY_STRING => "copy";
use constant JAR_STRING  => "jar";
use constant MAKE_STRING => "make";

# list of "must-have" commands.  Need to add more such as "ar", "jar",
#    "make", etc
use constant APPLY_COMMAND_LIST => "fuser";

# These constants are for identifying files for recovery or rollback.
use constant BACKUP      => "backup";
use constant END_PATCH   => "done";
use constant PRE_PATCH   => "pre";
use constant OPATCH_NEW_FILE    => "opatch_new";

###############################################################################
#
#  ------------------------ DATA STRUCTURES ------------------------
#
###############################################################################
#
# Global data structures are defined in Command.pm.
#
###############################################################################


###############################################################################
#
#  ------------------------ SUBROUTINES ------------------------
#
###############################################################################

###############################################################################
#
# NAME   : abdomen
#
# PURPOSE: Run the command.
#
# INPUTS : $$ARG[1]{command}   - The command that is to be executed.
#          $$ARG[1]{arguments} - The arguments to the command.
#
# OUTPUTS: NONE
#
# NOTES  : 1. This functions as a main body. Since this is OO perl and perl
#             reads in the main body of the packages during initialization
#             to avoid any side effects the functionality of the body has
#             moved to this method.
#          2. The name abdomen was chosen as its a simile for main body and
#             it'll hopefully remain near the top of the file since it starts
#             with "ab". Unless someone wants to really wreck this scheme.
#
###############################################################################
sub abdomen {

    my $debug = $ENV{OPATCH_DEBUG};
    if ( ! ($debug eq "TRUE") ) {
      $Command::DEBUG = "";
    } else {
      $Command::DEBUG = "-DTRACING.ENABLED=TRUE -DTRACING.LEVEL=2 -Dopatch.debug=true ";
    }
 
    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $command    = $$rh_arguments{command};
    my $arg_string = $$rh_arguments{arguments};
    my $rh_command_list = $$rh_arguments{rh_command_list};
     
    my $system_command   = "";
    my $sys_call_result = "";
    # $patch_storage_directory will be OH/.patch_storage/<patch ID>
    my $patch_storage_directory = "";

    my $patch_myself_flag = "";

    my @arguments = split ( " ", $arg_string ) ;

    # Set the operation, so that it can be accessed from anywhere
    $Command::opatch_operation = "apply";

    # Now parse the arguments for this command.
    my $rh_arg_check =
            $this_class -> parse_arguments( { ra_arguments => \@arguments } );

    my $error_flag = $$rh_arg_check{error_flag};

    # For any errors display the problem, give command usage then exit.
    if ( $error_flag ) {
        $this_class -> display_help( { command => $command,
                                       rh_command_list => $rh_command_list } );
       opatchIO->print_message_and_die({
          exit_val => $this_class->ERROR_CODE_COMMAND_LINE_ARGUMENT,
          message=>
            "Error parsing arguments: $error_flag\n"
	      });
    }

    # At least $Oracle_Home will be valid at this point.
    my $force_flag  = $$rh_arg_check{force_flag};
    my $general_options = $$rh_arg_check{general_options};
    my $noop_flag   = $$rh_arg_check{noop_flag};
    my $Oracle_Home = $$rh_arg_check{Oracle_Home};
    my $RAC_options = $$rh_arg_check{RAC_options};
    my $ship_home   = $$rh_arg_check{ship_home};

    my $rh_file_names = $this_class -> build_required_XML_filenames ();
    my $actions_file = $$rh_file_names{actions};
    my $inventory_file = $$rh_file_names{inventory};
    my $GenericActions_file = $$rh_file_names{GenericActions};

    my $patch_inventory_location = $ship_home || &Cwd::cwd();

    # this $patch_inventory_location needs to be an absolute path
    # so we have to 'cd' to $patch_inventory_location, get its absolute path,
    # then 'cd' back
    my $curDir = $this_class->get_cwd();
    chdir $patch_inventory_location;
    my $patch_inventory_location = $this_class->get_cwd();
    chdir $curDir;

    # User the env. var. 'user' to make sure invoker is not Root
    # Java refers to this as user.name
    # User the env. var. 'user' to make sure invoker is not Root
    # Java refers to this as user.name
    my $skipRootCheck = $ENV{OPATCH_SKIP_ROOT_CHECK};
    if ( $skipRootCheck eq "" ) {
      if ( ! ($OSNAME =~ m#Win32#) ) {    
         my $uid = POSIX::getuid();
         if ( $uid == 0 ) { 
           opatchIO -> print_message_and_die ( {
                 exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
                 message  => $this_class->INVOKED_BY_ROOT_ERROR_MESSAGE
                 } );      
         }
      }
    }

    $this_class -> get_os_id ( { Oracle_home => $Oracle_Home } );
    my $os_id = $Command::OS_ID;
    
    # Process env. var. OPATCH_IS_SHARED
    $Command::OPATCH_IS_SHARED = $ENV{"OPATCH_IS_SHARED"} || "";
    
    # From this point any error messages are processing errors and will
    # be sent to STERR, information messages to STDOUT.

    my $verbose = $this_class -> is_option_set ( {
                                    general_options => $general_options,
                                    option => $this_class -> verbose
                                    } );

    # Set the global verbose option
    $Command::verbose_mode = $verbose;

    my $opatch_ver = $this_class -> version;
    opatchIO -> print_message ( {
		    message => "\nOPatch Version $opatch_ver\n" .
		               "Perl Version $]\n"
	    } );

    # save the location of this action file
    $Command::ACTION_FILE = File::Spec -> catfile ($patch_inventory_location,
                                                   $actions_file);

    opatchIO -> print_message ( { message => "Performing pre-patch " .
                                             "installation checks.\n" } );

    # Check command line args.
    my $no_bug_superset = $this_class -> is_option_set ( {
                                    general_options => $general_options,
                                    option => $this_class -> no_bug_superset
                                    } );

    my $local_node_only = $this_class -> is_option_set ( {
                                    general_options => $general_options,
                                    option => $this_class -> local_node_only
                                    } );
    if ($local_node_only) {
        $Command::LOCAL_NODE_ONLY = 1;
    }
    my $no_inventory_update = $this_class -> is_option_set ( {
                                    general_options => $general_options,
                                    option => $this_class -> no_inventory_update
                                    } );

    my $report_only = $this_class -> is_option_set ( {
		                    general_options => $general_options,
				    option => $this_class -> report_only
			            } );

    my $silent = $this_class -> is_option_set ( {
                                    general_options => $general_options,
                                    option => $this_class -> silent
                                    } );

    if ($silent) {
        $Command::SILENT_MODE = 1;
        opatchIO->print_message ({ message =>
           "-silent: auto-answer 'Y' to any question.\n"
        });
    }
                                 
    if ($Command::NO_RELINK_MODEk) { 
        opatchIO->print_message ({ message =>
           "-no_relink: OPatch will not execute any 'make' action.\n"
        });
    }
    
    #if ($Command::LOCAL_NODE_ONLY && $Command::LOCAL_NODE_NAME_MODE) { 
    #   opatchIO->print_message_and_die({
    #      exit_val => $this_class->ERROR_CODE_COMMAND_LINE_ARGUMENT,
    #      message=> "-local and -local_node <node name> cannot be used together.\n"
    #	      });
    #  
    #}
    
    opatchIO->debug({ message =>
           "general_options is set to $general_options\n"
    });
    

    # Semantic check for all option
    my $badRetry = $this_class -> isBadRetryDelaySemantics();
    if ( $badRetry ) {
       opatchIO->print_message_and_die({
          exit_val => $this_class->ERROR_CODE_COMMAND_LINE_ARGUMENT,
          message=> $this_class->RETRY_ERROR_MESSAGE
	      });
    }

    if ($report_only) {
       opatchIO->print_message({message=>
		      "Running in -report mode...\n"
	       });
    }

#    if ($local_node_only && $no_inventory_update) {
#       opatchIO->print_message_and_die({
#          exit_val => $this_class->ERROR_CODE_COMMAND_LINE_ARGUMENT,
#          message=>
#            "-local and -no_inventory can not be used together.\n"
#	      });
#    }

    if ( $local_node_only && $Command::MINIMIZE_DOWNTIME_OPTION ) {
       opatchIO->print_message_and_die({
          exit_val => $this_class->ERROR_CODE_COMMAND_LINE_ARGUMENT,
          message=>
            "-local and -minimize_downtime can not be used together.\n"
	      });
    }


    if ( $Command::JRE_LOC ne "" && $Command::JDK_LOC ne "" ) {
       opatchIO->print_message_and_die({
          exit_val => $this_class->ERROR_CODE_COMMAND_LINE_ARGUMENT,
          message=>
            "Please specify either -jre or -jdk but not both.\n"
	      });
    }

    # Make sure the JDK or JRE LOCs are valid
    my $target = "";
    my $start_point = "";
    my $result = "";
    my $skip_jre = 0;
    my $search_OH_jar = 1;

    if ( $Command::JDK_LOC ne "" ) {
      # If -jdk is used, look for "jar"
      $search_OH_jar = 0;
      $start_point = $Command::JDK_LOC;
      $target  = "jar";
      if ( $OSNAME =~ m#Win32# ) {
         $target = "jar.exe";
      }
      opatchIO -> print_message ( { message => "-jdk specified, need to find " .
                                               "executable jar from $Command::JDK_LOC\n"
                                } );

      $result = $this_class -> find_executable_file ( { start_point => $start_point,
                                      target => $target } );
      if ( $result ne "" ) {
        $Command::JDK_LOC = $result;
        opatchIO -> print_message ( { message => "JAR set to $result\n" } );
      }
      # Then look for "java" as well
      $target  = "java";
      if ( $OSNAME =~ m#Win32# ) {
         $target = "java.exe";
      }
      opatchIO -> print_message ( { message => "-jdk specified, need to find " .
                                               "executable java from $Command::JDK_LOC\n"
                                } );
      my $default_path_1 = File::Spec -> catfile ($start_point,
                                      "bin", $target);
      my $default_path_2 = File::Spec -> catfile ($start_point,
                                      "jre", "bin", $target);

      # Bug 4883881 : First look into <user specified path>/bin/java, if not then look for
      # <user specified path>/jre/bin/java, if not found
      # then use find and look for the path as usual.
      if ( -x $default_path_1 ) {
          $result = $default_path_1;
      } elsif ( -x $default_path_2 ) {
          $result = $default_path_2;
      } else {
          $result = $this_class -> find_executable_file ( { start_point => $start_point,
                                          target => $target } );  
      }

      if ( $result ne "" ) {
        $Command::JRE_LOC = $result;
        opatchIO -> print_message ( { message => "JAVA set to $result\n" } );
        $skip_jre = 1;
      }
    } else {
      # -jdk not used, make sure users have OH/jdk/bin/jar
      $search_OH_jar = 1;
    }

    # Note: $Command::JRE_LOC could have been set by -jdk option above ($skip_jre set to 1)
    if ( $Command::JRE_LOC ne "" && ! $skip_jre ) {
      $start_point = $Command::JRE_LOC;
      $target  = "java";
      if ( $OSNAME =~ m#Win32# ) {
         $target = "java.exe";
      }
      opatchIO -> print_message ( { message => "-jre specified, need to find " .
                                               "executable java from $Command::JRE_LOC\n"
                                } );

      my $default_path = File::Spec -> catfile ($start_point,
                                      "bin", $target);

      # Bug 4883881 : First look into <user specified path>/bin/java, if not 
      # then use find and look for the path as usual.
      if ( -x $default_path ) {
          $result = $default_path;
      } else {
          $result = $this_class -> find_executable_file ( { start_point => $start_point,
                                          target => $target } );
      }

      if ( $result ne "" ) {
        $Command::JRE_LOC = $result;
        opatchIO -> print_message ( { message => "JAVA set to $result" } );
      }
    }

    # Make sure all commands are in path
    my $has_commands = $this_class -> has_commands_in_path(
	                                  $this_class -> APPLY_COMMAND_LIST
				                            );

    if ( $has_commands == 0 ) {
       my $list = $this_class -> APPLY_COMMAND_LIST;
       opatchIO -> print_message_and_die ( {
		       exit_val => $this_class -> ERROR_CODE_PATH_PROBLEM,
		       message =>
		       "Some commands below are not in your path:\n" .
		       "  Command= $list\n" .
           "  Extra Search Path= /sbin/fuser, /usr/sbin/fuser\n" .
		       "  Path= $ENV{PATH}\n"
	       });
    }


    # Check for env. var. OPATCH_REMOTE_SHELL
    my $isRSH_SSH = $ENV{OPATCH_REMOTE_SHELL};
    if ( $isRSH_SSH eq "" ) {
      $Command::RSH_SSH = "rsh";
    } else {
      $Command::RSH_SSH = $isRSH_SSH;
      my $rsh_msg = "Using user-supplied $Command::RSH_SSH to run make " .
                    "on remote nodes if applicable.\n";
      opatchIO->print_message ({ message => $rsh_msg });
    }

    # Begin to check patch validity
    # Check for where the patch is located.
    my $rh_return_values = $this_class -> check_patch_area_validity ( {
                                      p_i_l => $patch_inventory_location,
                                      rh_required_files => $rh_file_names } );
    if ( $rh_return_values eq "" ) {
        opatchIO -> print_message_and_die ( {
                   exit_val=> $this_class->ERROR_CODE_PATCH_AREA,
                   message => "Not a valid patch area: some files are missing or patch ID is not a number.\n"
                   } );
    }
    $patch_inventory_location = $$rh_return_values{patch_inventory_location};
    
    # Check and remember if pre-script is there for this apply session
    #  so that we don't have to do fuser
    my $script_loc = "";
    my $script_name= "";
    
    if ( $OSNAME =~ m#Win# )
    {
        $script_loc = File::Spec -> catfile ($patch_inventory_location,
                                      "custom", "scripts", $Command::WIN_PRE_SCRIPT);
        $script_name = $Command::WIN_PRE_SCRIPT;
    }
    else
    {
        $script_loc = File::Spec -> catfile ($patch_inventory_location,
                                      "custom", "scripts", $Command::UNIX_PRE_SCRIPT);
        $script_name = $Command::UNIX_PRE_SCRIPT;
    }
    if ( -e $script_loc ) {
      $Command::APPLY_PRE_SCRIPT_IS_THERE = 1;
    } else {
      $Command::APPLY_PRE_SCRIPT_IS_THERE = 0;
    }

    # Get the patch id number part (can be <number>_<text>).
    my ( $patch_id ) = ( File::Basename::basename( $patch_inventory_location )
                                                              =~ m#^(\d+)# );

    # Make sure patch dir. is a number
    if ( $patch_id !~ m#^(\d+)$# ) {
        opatchIO -> print_message_and_die ({
                    exit_val=> $this_class->ERROR_CODE_PATCH_AREA,
                    message => "The patch directory area must be a number.\n"
                  } );
    }

    # Check validity of patch location and directory name.
    if ( ( ! $patch_inventory_location ) ) {
        opatchIO -> print_message_and_die ( {
                   exit_val=> $this_class->ERROR_CODE_PATCH_AREA,
                   message => "Not a valid patch area.\n" .
                              "Check the command line, current " .
                              "directory location and permissions of " .
                              "the patch directory." } );
    }

    # Now assign the full patch id.
    $patch_id = File::Basename::basename ( $patch_inventory_location );
    # done checking patch validity

    # build the file patch representing the OH/.patch_storage/<patch ID> area
    $patch_storage_directory   = File::Spec -> catfile
                                         ( "$Oracle_Home",
                                           $this_class -> save_dir,
                                           $patch_id );

    # This is needed for invoking and running the java VM.
    my $rh_OUI_file_names = ();

    # Setup an save area even if this is report mode 
    my $save_area_errors = $this_class -> create_save_area ( {
                                              patch_id    => $patch_id,
                                              Oracle_Home => $Oracle_Home } );

    if ( $save_area_errors ) {
        opatchIO -> print_message_and_die ( {
                exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
                message  => "Problems with creating the backup " .
                            "area:\n $save_area_errors" } );
    }
    
    # Create the lock file.  
    #
    my $lock_file_errors = "";
    my $patched_RAC_nodes = "";
    my $lock_class        = "";
    if ($report_only) {
      opatchIO -> print_message({message =>
		      "-report mode: skip creating lock file\n"
	      });
    } else {
      my $rh_lock_check_details = $this_class -> check_patch_lock_status ( {
                                              patch_id    => $patch_id,
                                              Oracle_Home => $Oracle_Home } );

      $lock_file_errors  = $$rh_lock_check_details{lock_file_errors}  || "";
      $patched_RAC_nodes = $$rh_lock_check_details{patched_RAC_nodes} || "";
      $lock_class        = $$rh_lock_check_details{lock_class}        || "";

      if ( $lock_file_errors ) {
        $lock_file_errors = "Problems with the lock file\n" .
                                                            $lock_file_errors;

        opatchIO -> print_message_and_die ( {
              exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
              message  => $lock_file_errors
            } );
      }
    }    
    
    # Setup the logging area.
    my $rh_log_creation_details = $this_class -> create_logging_file ( {
                                              patch_id    => $patch_id,
                                              Oracle_Home => $Oracle_Home } );

    my $log_failure   = $$rh_log_creation_details{error_flag};
    my $fh_log_file   = $$rh_log_creation_details{fh_log_file};
    my $log_file_name = $$rh_log_creation_details{log_file_name};
    $OUTPUT_AUTOFLUSH = 1;

    if ( $log_failure ) {
        
        $lock_file_errors = $this_class -> free_the_lock_file ( {
                                        Oracle_Home     => $Oracle_Home,
                                        patch_id        => $patch_id } );
      

        opatchIO -> print_message_and_die ( {
                      exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
                      message  => "Problems with creating the log file:\n " .
                                              "$log_failure" } );
    }

    my $timestamp = $this_class -> get_current_timestamp();
    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "Starting OPatch Apply session at " .
                                              "$timestamp.\n" } );

    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "Command arguments are: " .
                                              "$command $arg_string\n" } );

    my $version_number = $this_class -> version;
    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "OPatch version is: " .
                                              "$version_number\n" } );
                                              
    # If this is a RAC instance that had a rollback partially done
    # reapply the patch from scratch.
    if ( ( $lock_class ) && ( $this_class ne $lock_class ) ) {
        $patched_RAC_nodes = "";
    }


    # If users do not specify no_inventory_update, we
    # need to build OUI and check RAC
    if ( ! $no_inventory_update ) {

       $rh_OUI_file_names = $this_class -> build_required_OUI_filenames ( {
                                              Oracle_home => $Oracle_Home } );

       my $rh_java_paths = $this_class -> make_java_paths ( {
	        Oracle_Home       => $Oracle_Home,
			    rh_OUI_file_names => $rh_OUI_file_names } );

       # Due to so many problems with RAC, check its sanity upfront
       #
       $this_class -> validate_OUI_and_RAC( {
          Oracle_Home => $Oracle_Home,
          rh_OUI_file_names => $rh_OUI_file_names,
          rh_java_paths => $rh_java_paths,
          fh_log_file => $fh_log_file
       } );
    }
    if ( ( ! defined $$rh_OUI_file_names{error_flag} ) &&
         ( ! $no_inventory_update ) ) {

        my $error_message = $$rh_OUI_file_names{error_flag} . "\n\n" .
                            "You may need to add the -n option. Please " .
                            "contact support as this my put the\n" .
                            "database into an unsupported state.\n";

        opatchIO -> print_message_and_die ( {
                        exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
                        message => $error_message
                        } );
    }

    # Set-up a "rollback" file.
    my $rh_rollback_file_details = $this_class -> create_rollback_file ( {
                                              patch_id    => $patch_id,
                                              Oracle_Home => $Oracle_Home } );

    my $rollback_failure   = $$rh_rollback_file_details{error_flag};
    my $fh_rollback        = $$rh_rollback_file_details{fh_log_file};
    my $rollback_file_name = $$rh_rollback_file_details{log_file_name};

    if ( $rollback_failure ) {
        opatchIO -> print_message ( { f_handle => $fh_log_file,
                                      message  => "Problems with the " .
                                                  "rollback file\n" .
                                                  "$rollback_failure\n" } );

        $lock_file_errors = $this_class -> free_the_lock_file ( {
                                        Oracle_Home     => $Oracle_Home,
                                        patch_id        => $patch_id } );

        opatchIO -> print_message_and_die ( {
              exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
              message  => "Problems with creating the rollback file:\n" .
                                              $rollback_failure } );
    }

    # Save the rollback_<ID>.sh file details in a global location
    $Command::fh_rollback = $fh_rollback;
    $Command::rollback_file_name = $rollback_file_name;

    opatchIO->print_message({ message => "Interim Patch ID: $patch_id"});

    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "Checking the patch " .
                                              "inventory.\n" } );

    # The following crudely parse the XML files. This gets action to
    # command mapping.
    # Read etc/config/inventory file
    my $rh_global_details = $this_class-> read_and_check_reference_details ( {
                                        inventory_file => $inventory_file,
                                        Oracle_Home    => $Oracle_Home,
                                        p_i_l => $patch_inventory_location
				} );

    my $check_reference_error = $$rh_global_details{error_flag};
    my $rh_bug_list          = $$rh_global_details{bug_list};
    my $rh_components        = $$rh_global_details{components};
    my $rh_executable_files  = $$rh_global_details{executable};
    my $shutdown_needed_flag = $$rh_global_details{shutdown};
    my $shutdown_RAC_flag    = $$rh_global_details{RAC_shutdown};
    my $os_platforms          = $$rh_global_details{os_platforms};

    if ( $check_reference_error ) {
        $lock_file_errors = $this_class -> free_the_lock_file ( {
                                        Oracle_Home     => $Oracle_Home,
                                        patch_id        => $patch_id } );

        opatchIO -> print_message ( { f_handle => $fh_log_file,
                              message => $$rh_global_details{error_flag} } );
        opatchIO -> print_message_and_die ( {
                              exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
                              message => $$rh_global_details{error_flag} } );
    }

    # Command::ROLLING_PATCH is set by read_and_check_reference_details()
    if ( $Command::ROLLING_PATCH ) {
         opatchIO->print_message({ message =>
			 "This is a rolling patch.\n"
		 });
    }

    # Output list of bug fixes
    my @bug_keys = keys   %$rh_bug_list;
    my @bug_vals = values %$rh_bug_list;
    my $list = "   ";
    while (@bug_keys) {
      my $bug = pop(@bug_keys);
      my $des = pop(@bug_vals);
      $list .= $bug . " : " . $des . "\n   "; 
    } 
    opatchIO->print_message({message => "Bugs fixed by this patch $patch_id:"});
    opatchIO->print_message({message => "$list"});

    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "Reading patch XML files and " .
                                              "doing sanity checks.\n" } );

    my $matching_os_flag = $this_class -> check_patch_OS_against_machine ( {
                     current_os_id   => $os_id,
                     rh_os_platforms => $os_platforms} );

    # Read the file that has the "command type" and the associated action
    # (GenericActions.xml)
    my $rh_action_map = $this_class -> get_command_to_action_mapping ( {
                              GenericActions => $GenericActions_file,
                              Oracle_Home    => $Oracle_Home,
                              os_id          => $os_id,
                              patch_location => $patch_inventory_location } );

    my $action_map_error = $$rh_action_map{error_flag};
    my $rh_actions_list = $$rh_action_map{rh_actions};
    
    if ( $action_map_error ) {
        if ( ! $lock_class ) {
            $lock_file_errors = $this_class -> free_the_lock_file ( {
                                        Oracle_Home     => $Oracle_Home,
                                        patch_id        => $patch_id } );
        }

        opatchIO -> print_message ( { f_handle => $fh_log_file,
                              message => $action_map_error } );
        opatchIO -> print_message_and_die ( {
                          exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
                          message => $action_map_error } );
    }

    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "Read the command to action " .
                                              "file map.\n" } );

    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "Performing initial safety " .
                                              "check.\n" } );

    # Check the details about database (RAC, running, collisions, etc).
    # This new CheckConflict class will check file conflict as well as bug conflict

    my $rh_initial_checks = $this_class -> initial_safety_check ( {
                              fh_log_file       => $fh_log_file,
                              no_inventory_update   => $no_inventory_update,
                              Oracle_Home       => $Oracle_Home,
                              patch_id          => $patch_id,
                              patch_location    => $patch_inventory_location,
                              rh_bug_list       => $rh_bug_list,
                              rh_executables    => $rh_executable_files,
                              rh_OUI_file_names => $rh_OUI_file_names } );
    
    # output of initial_safety_check() are:
    # {files_in_use}       = $active_files;
    # {error_flag}         = $error_flag;
    # {conflict_patches}   = $conflict_patches; (list of patches that have conflicts)
    # {skip_components}    = skip_components;   (list of optional components to be skipped)
    # {bug_conflict}       = 1 if this patch has conflicts with other installed
    #                        patches on system
    # {file_conflict}      = 1 if this patch has conflicts with other installed
    #                        patches on system
    # {missed_component}    = 1 if this patch requires some comp/ver that
    #                        do not exist in inventory
    # {bug_superset}       = 1 if this patch's bugsToFix is a super-set or
    #                           same set of some installed patch's bugsFixed
    # {bug_subset_patches} = list of subset/same patches if bug_superset is 1

    my $initial_checks_error = $$rh_initial_checks{error_flag};
    my $active_files       = $$rh_initial_checks{files_in_use};
    my $conflict_patches   = $$rh_initial_checks{conflict_patches};
    my $skip_components    = $$rh_initial_checks{skip_components};
    my $bug_conflict       = $$rh_initial_checks{bug_conflict};
    my $file_conflict      = $$rh_initial_checks{file_conflict};
    my $missed_component   = $$rh_initial_checks{missed_component};

    my $bug_superset       = $$rh_initial_checks{bug_superset};
    my $bug_subset_patches = $$rh_initial_checks{bug_subset_patches};
        
    if ( $initial_checks_error ) {
        if ( ! $lock_class ) {
            $lock_file_errors = $this_class -> free_the_lock_file ( {
                                        Oracle_Home     => $Oracle_Home,
                                        patch_id        => $patch_id } );
        }

        opatchIO -> print_message_and_die ( {
                             exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
                             message => $initial_checks_error } );
    }

    # If there is any activity on critical files it's time to exit.
    if ( $active_files ) {

        if ( ! $lock_class ) {
            $lock_file_errors = $this_class -> free_the_lock_file ( {
                                        Oracle_Home     => $Oracle_Home,
                                        patch_id        => $patch_id } );
        }

        my $message = "Problems when checking for files that are active.\n" .
                                                        "$active_files\n\n" ;
        opatchIO -> print_message ( { f_handle => $fh_log_file,
                                      message  => $message } );
        opatchIO -> print_message_and_die ( {
                      exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
                      message  => $message } );
    }

    # Required components/versions check
    my $patchMsg = "";
    my $answer = "";
    my $question = "";

    if ( $missed_component ) {
      $patchMsg = "This Oracle Home does not have components/versions required by the patch.\n";
      $this_class -> free_the_lock_file ( {
                                        Oracle_Home     => $Oracle_Home,
                                        patch_id        => $patch_id } );
      opatchIO -> print_message_and_die ( {
                      exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
                      message  => $patchMsg } );

    }

    # Patch conflict check
    $patchMsg = "";
    $answer = "";

    my $patchConflict = $bug_conflict || $file_conflict;
    my $patchSuperset = $bug_superset;
        
    if ( $patchConflict && $patchSuperset ) {
        my $conflictingPatchList = "";
        my $subsetPatchList = "";
        my $combinedList = "";
        
        foreach my $p ( @$bug_subset_patches ) {
            $combinedList .= "$p, ";
            $subsetPatchList.= "$p, ";
        }
        
        foreach my $p ( @$conflict_patches ) {
            $combinedList .= "$p, ";
            $conflictingPatchList.= "$p, ";
        }
        
        $question =  "Subset patches:  $subsetPatchList\n" .
                     "Conflict patches: $conflictingPatchList\n\n" .
                     
                     "The fix for Patch $subsetPatchList are included in the patch currently \n" .
                     "being installed ($patch_id)\n\n" .
                     
                     "Patch(es) $conflictingPatchList conflict with the patch currently \n" .
                     "being installed ($patch_id)\n\n" .
                     "If you continue, all patches will be rolled back and the new patch \n" .
                     "($patch_id) will be installed.\n\n" .
                     
                     "Note: If the patch currently being installed ($patch_id) is rolled back, \n" .
                     "it is recommended that the fixes being rolled back here ($combinedList) are reinstalled.\n\n" .
                     
                     "If a merge of the new patch ($patch_id) and the conflicting patch(es)\n" .
                     "$conflictingPatchList is required, contact Oracle Support Services and request\n" .
                     "a Merged patch.\n\n" .
                     
                     "Do you want to STOP?" ;

	if( $Command::SILENT_MODE && $force_flag ) {
		opatchIO -> print_message_noverbose ( {
               		 message => $question ."\nPlease respond Y|N >"
              	  });
	        opatchIO  -> print_message_noverbose ( {
               		 message => "N (auto-answered by -silent -force)\n"
              	  });
	}
	else {
	        $answer = opatchIO -> check_to_stop_processing ( {
                	        message =>  $question 
        	             } );
        	if ( $answer eq "" ) {
	              $this_class -> free_the_lock_file ( {
        	                                Oracle_Home     => $Oracle_Home,
                	                        patch_id        => $patch_id } );
	              opatchIO -> print_message_and_die ( {
        	              exit_val => $this_class->ERROR_CODE_BUG_FILE_CONFLICT,
                	      message  => "Bug or File Conflict, patch application stopped on user's request" } );
	        } 
	}
	 
        
        # Comment out since we decide not to warn usrs
        #  about auto-rolling back --> user, if roll back this patch,
        #  has to re-apply auto-rolled back patches
        # Create the file to  hold a list of patches that will be rolled back
        # $this_class -> generateRollbackedPatches ( {
        #      Oracle_Home => $Oracle_Home,
        #      patch_id    => $patch_id,
        #      patch_list  => $combinedList
        # } );
        
    } elsif ( $patchConflict ) {
        #
        # file and/or bug conflicts 
        #        

        my $conflictingPatchList = "";
        foreach my $conflictingPatch ( @$conflict_patches ) {
          $conflictingPatchList .= "$conflictingPatch, ";
        }

        $question = "Conflicting patches: $conflictingPatchList\n\n" .
        
                     "Patch(es) $conflictingPatchList conflict with the patch currently " .
                     "being installed ($patch_id).\n\n" .
                     
                     "If you continue, patch(es) $conflictingPatchList will be rolled back and the\n" .
                     "new patch ($patch_id) will be installed.\n\n" .
                     
                     "Note: If the patch currently being installed ($patch_id) is rolled back, \n" .
                     "it is recommended that the fixes being rolled back here ($conflictingPatchList) are reinstalled.\n\n" .
 
                     "If a merge of the new patch ($patch_id) and the conflicting patch(es)\n" .
                     "$conflictingPatchList is required, contact Oracle Support Services and request\n" .
                     "a Merged patch.\n\n" .
                     
                     "Do you want to STOP?" ;
        
                
        # if there is conflict, -silent will error out
        #   while -silent -force will silently roll back, then apply
        if ( $Command::SILENT_MODE &&  $force_flag ) {
        	opatchIO -> print_message_noverbose ( {
                	message => $question ."\nPlease respond Y|N >"
              	  });
                opatchIO  -> print_message_noverbose ( {
                	message => "N (auto-answered by -silent -force)\n"
              	 });
	
        } else {
           $answer = opatchIO -> check_to_stop_processing ( {
                        message =>  $question 
                     } );
           if ( $answer eq "" ) {
              $this_class -> free_the_lock_file ( {
                                        Oracle_Home     => $Oracle_Home,
                                        patch_id        => $patch_id } );
              opatchIO -> print_message_and_die ( {
                      exit_val => $this_class->ERROR_CODE_BUG_FILE_CONFLICT,
                      message  => "Bug or File Conflict, patch application stopped on user's request" } );
           }
        }
        # Create the file to  hold a list of patches that will be rolled back
        # $this_class -> generateRollbackedPatches ( {
        #      Oracle_Home => $Oracle_Home,
        #      patch_id    => $patch_id,
        #      patch_list  => $conflictingPatchList
        # } );
    }

    elsif ( $patchSuperset ) {
       # 
       # super-set
       #
       my $conflictingPatchList = "";
       foreach my $conflictingPatch ( @$bug_subset_patches ) {
          $conflictingPatchList .= "$conflictingPatch, ";
       }

       $question =
                   "Subset patches:  $conflictingPatchList\n\n" .
                   "The fixes for Patch $conflictingPatchList are included in the patch currently\n" .
                   "being installed ($patch_id).  If you continue, they will be rolled back and\n" .
                   "the new patch ($patch_id) will be installed.\n\n" .

                   "Note: If the patch currently being installed ($patch_id) is rolled back, \n" .
                   "it is recommended that the fixes being rolled back here ($conflictingPatchList) are reinstalled.\n\n" .

                   "Do you want to continue?" ;

       # if there is bug-superset case, -silent will silently continue with auto-rollback
       #   while -silent -no_bug_superset will error out
       if ( $no_bug_superset ) {
         # clear lock file first
         $this_class -> free_the_lock_file ( {
                              Oracle_Home => $Oracle_Home,
                              patch_id => $patch_id } );
         # then die
         opatchIO -> print_message_and_die ( {
              exit_val => $this_class->ERROR_CODE_ERROR_ON_CONFLICT,
              message => $question . " (no_bug_superset)\n"
              });
       } else {
         $answer = opatchIO -> check_to_stop_processing ( {
                      message => $question 
                   } );
         if ( $answer eq "N" ) {
            $this_class -> free_the_lock_file ( {
                              Oracle_Home => $Oracle_Home,
                              patch_id => $patch_id } );
            opatchIO -> print_message_and_die ( {
                   exit_val => $this_class->ERROR_CODE_BUG_FILE_CONFLICT,
                   message => "Patch application stopped on user's request" } );
         }
       }
       # Create the file to  hold a list of patches that will be rolled back
       # $this_class -> generateRollbackedPatches ( {
       #       Oracle_Home => $Oracle_Home,
       #       patch_id    => $patch_id,
       #       patch_list  => $conflictingPatchList
       # } );
    }  

    # If there's remote nodes propagation AND there's auto-rollback AND
    #    (users requested -min_downtime || rolling patch), we will
    #    say 'sorry, no support'
    if ( $patchConflict || $patchSuperset ) {
        my $shouldPropagateChange = $this_class -> shouldPropagateChange();
        if ( $shouldPropagateChange ) {
          if ($Command::MINIMIZE_DOWNTIME_OPTION || $Command::ROLLING_PATCH) { 

             #$this_class -> free_the_lock_file ( {
             #                 Oracle_Home => $Oracle_Home,
             #                 patch_id => $patch_id } );

             #opatchIO -> print_message_and_die ( {
             #      exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
             #      message => 
             #       "OPatch detected conflict and needed to roll back " .
             #       "conflicting patches.  However, auto-rollback on " .
             #       "RAC environment for rolling patch or minimum downtime " .
             #       "is not supported.  Please use OPatch to roll back reported " .
             #       "conflicting patches first, then invoke OPatch to apply " .
             #       "this patch.  Alternatively, you can use -local on each of " .
             #       "the node in which case, OPatch will " .
             #       "automatically roll back conflicting patch(es), then apply " .
             #       "this patch."
             #      } );
             
            $question = "OPatch detected conflict and needed to roll back ".
                        "conflicting patches.  Please use OPatch to roll back reported " .
                        "conflicting patches first, then invoke OPatch to apply " .
                        "this patch.  Alternatively, you can use -local on each of " .
                        "the node in which case, OPatch will " .
                        "automatically roll back conflicting patch(es), then apply " .
                        "this patch.\n" . 
                        "If you continue, OPatch will switch to all-node mode.  " .
                        "Do you want to continue?";
                        
            $answer = opatchIO -> check_to_stop_processing ( {
                        message =>  $question 
                     } );
            if ( $answer eq "N" ) {
              $this_class -> free_the_lock_file ( {
                                        Oracle_Home     => $Oracle_Home,
                                        patch_id        => $patch_id } );
              opatchIO -> print_message_and_die ( {
                      exit_val => $this_class->ERROR_CODE_BUG_FILE_CONFLICT,
                      message  => "Auto-rollback not supported with rolling " .
                        "or min. downtime, patch application stopped on user's request" } );
            } else {
              $Command::ROLLING_PATCH = 0;
              $Command::MINIMIZE_DOWNTIME_OPTION = 0;
            }
          }
        }
    }
    

    # Read the etc/config/action file & filter out uninstalled optional components
    my $rh_file_map_details = $this_class -> build_file_details ( {
                              actions_file   => $actions_file,
                              Oracle_Home    => $Oracle_Home,
                              skip_components=> $skip_components,
                              patch_location => $patch_inventory_location } );

    my $file_map_error = $$rh_file_map_details{error_flag};
    my $rh_file_details  = $$rh_file_map_details{rh_files};

    my $prereq_errStr = $this_class -> prereq_patch_actions ( {
                               fh_log_file     => $fh_log_file,
                               fh_rollback     => $fh_rollback,
                               noop_flag       => $noop_flag,
                               Oracle_Home     => $Oracle_Home,
                               patch_id        => $patch_id,
                               patch_location  => $patch_inventory_location,
                               rh_actions_list => $rh_actions_list,
                               rh_file_details => $rh_file_details,
                               search_OH_jar   => $search_OH_jar
                               } );    

    if( $prereq_errStr ) {
        opatchIO -> print_message_and_die ( {
                      exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
                      message  => $prereq_errStr } );
    }

    if ( $file_map_error ) {
        if ( ! $lock_class ) {
            $lock_file_errors = $this_class -> free_the_lock_file ( {
                                        Oracle_Home     => $Oracle_Home,
                                        patch_id        => $patch_id } );
        }

        opatchIO -> print_message ( { f_handle => $fh_log_file,
                             message => $file_map_error } );
        opatchIO -> print_message_and_die ( {
                             exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
                             message => $file_map_error } );
    }

    # Check to see if the patch is a no-op patch.
    my $noop_patch = $this_class -> is_noop_patch ( {
                               rh_file_details => $rh_file_details,
                               } );

    if( $noop_patch ) {
        if ( ! $lock_class ) {
            $lock_file_errors = $this_class -> free_the_lock_file ( {
                                        Oracle_Home     => $Oracle_Home,
                                        patch_id        => $patch_id } );
        }


        opatchIO -> print_message_noverbose( { 
			message => "\nNone of the patch actions is applicable to the Oracle Home.\n" .
				   "OPatch will not apply the patch, exiting...\n" } );
        opatchIO -> print_message_and_die ( {
                      exit_val => $this_class->ERROR_CODE_NO_ERROR } );
    }


    my $all_RAC_nodes = $Command::NODE_LIST;
    my $local_RAC_node= $Command::LOCAL_NODE_NAME;
    my $shouldPropagateChange = $this_class -> shouldPropagateChange();
    my $is_CFS = $this_class -> is_CFS();

    # Clustered file systems are treated as a single system.
    # I'm told shutting down one node shuts all instances down.
    if ( $is_CFS == 1 ) {
        $all_RAC_nodes = "";
        $Command::NODE_COUNT = 0;
        $Command::NODE_LIST  = "";
        opatchIO->debug({message=>
            "CFS detected, setting NODE_LIST to be empty\n"});
    }

    # Double check everything is down for a cluster file system or a
    # "non-rolling" upgrade. This needs to be changed to just check the
    # active nodes when the APIs are available.

    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "RAC double check.\n" } );
    
    if ( ( $shouldPropagateChange ) ) {
      if ( $Command::ROLLING_PATCH  || $Command::MINIMIZE_DOWNTIME_OPTION ) {
        if ( $silent ) {
           my $message = "-silent cannot be used on multi-node RAC environment with rolling or minimize_downtime. " .
                      "You can patch the system with -local -silent on each of the node.";
           $this_class -> free_the_lock_file ( {
                                        Oracle_Home     => $Oracle_Home,
                                        patch_id        => $patch_id } );
            opatchIO -> print_message_and_die ( { message  => $message ,
                          exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM } );
        }
      }
    }
    
    # If user supplied -local_node, it must match one of the element
    #   in the node list
    if ( $shouldPropagateChange && $Command::LOCAL_NODE_NAME_MODE ) {
        my $found = 0;
        my @nodes          = split ( " ", $all_RAC_nodes );
        foreach my $target ( @nodes ) {
          if ( $target eq $local_RAC_node ) { 
            $found = 1;
          }
        }
        # Exit if the local node name doesn't match one in the node list
        if ( !$found ) { 
            my $message =
            "The supplied \"$local_RAC_node\" does not match one of " .
            "the nodes in node list \"$all_RAC_nodes\"." ;
            opatchIO -> print_message_and_die ( { message  => $message ,
                          exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM } );
        
        }
      
    }


    # Check for unconditional failures. This could be done as a single
    # check but then you'd still need the subsequent checks to send the
    # correct message out.
    my $failure_flag = "";
    # Does the patch match the opererating system?
    if ( ! $matching_os_flag ) {
        my $platform_list = "   ";
        foreach my $os_number ( keys %$os_platforms ) {
          my $os_name = $$os_platforms{$os_number};
          $platform_list .= $os_number . " ($os_name)\n" . "   ";
        }
        $failure_flag .= "\n" . 
                         "OPatch detects your platform as $Command::OS_ID while " .
                         "this patch $patch_id supports platforms:\n" .
                         "$platform_list\n" .
                         "This patch is not suitable for this operating system.\n" .
                         "Please contact support for the correct patch.\n" ;
    }


    # Conditional failures need to be checked next.
    # If a database shutdown is need are there any active processes?
    # This isn't 100% reliable.
    if ( ( $shutdown_needed_flag ) && ( $active_files ) ){
        $failure_flag .= "\n" . "$active_files" . "\n";
    }

    # On failure exit with error messages.
    if ( $failure_flag ) {
        if ( ! $lock_class ) {
            $lock_file_errors = $this_class -> free_the_lock_file ( {
                                        Oracle_Home     => $Oracle_Home,
                                        patch_id        => $patch_id } );
        }
        opatchIO -> print_message ( { f_handle => $fh_log_file,
                                      message  => $failure_flag } );
        opatchIO -> print_message_and_die ( {
          exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
          message => $failure_flag } );
    }

    # Back up original comps.xml to .patch_storage as "comps.xml.b4apply"
    my $backup_filemap = $ENV{OPATCH_BACKUP_FILEMAP};
    if ( $backup_filemap eq "TRUE" )
    {
        opatchIO->print_message_noverbose ( { 
               message => 
                          "Backing up comps.xml and filemap ...\n"      
                                            } );
    }
    else
    {
        opatchIO->print_message_noverbose ( { 
               message => 
                          "Backing up comps.xml ...\n"      
                                            } );
    }
    $this_class->backupCompsXML({ Oracle_Home => $Oracle_Home,
                                  patch_id    => $patch_id,
                                  fh_log_file => $fh_log_file,
                                  target_name => "comps.xml.b4apply",
                                  target_filemap_name => "filemap.b4apply"
                               });
   
    # Call pre Apply script if available
    # For Windows call pre.bat and for non windows call pre
    my $script_txt_loc = "";
    my $script_txt_handle;
    my @script_txt_lines;
    my $script_line = "";
    my $script_result = "";

    # First copy the whole custom directory to .patch_storage
    my $source_dir_custom = File::Spec->catfile($patch_inventory_location, "custom");
    my $target_dir_custom = File::Spec->catfile($patch_storage_directory, "custom");
    my $targetdir = "";
    my $target = "";
    $source_dir_custom =~ s/\\/\//g;
    $target_dir_custom =~ s/\\/\//g;

    # note: for the copy sub, $_ is set to the current file being 
    # tested, and we are chdir'd to the directory that file resides in.
    sub copy_files 
    {
        $targetdir = $File::Find::dir;
        $target = $targetdir;
        $targetdir =~ s/$source_dir_custom/$target_dir_custom/o;
    
        mkpath( $targetdir ) if not -e $targetdir;
        
        my $source = "$_";
        $source = "./" . $source;
        my $dest   = "$targetdir/$_";
        
        if ( -f $source ) {
          copy ($source, $dest);
        }
    }

    # If custom directory exists, copy it to $OH/.patch_storage/<id>
    if ( -e $source_dir_custom )
    {
        opatchIO->print_message( {message => "Copying $source_dir_custom to $target_dir_custom"} );
        find( \&copy_files, $source_dir_custom );

        # Provide execute permission to pre and post files if they exist
        if ( -e File::Spec->catfile($patch_storage_directory,
                                 "custom", "scripts", $Command::UNIX_PRE_SCRIPT) )
        {
            chmod 0755, File::Spec->catfile($patch_storage_directory,
                                 "custom", "scripts", $Command::UNIX_PRE_SCRIPT);
        }
        if ( -e File::Spec->catfile($patch_storage_directory,
                                 "custom", "scripts", $Command::UNIX_POST_SCRIPT) )
        {
            chmod 0755, File::Spec->catfile($patch_storage_directory,
                                 "custom", "scripts", $Command::UNIX_POST_SCRIPT);
        }
        if ( -e File::Spec->catfile($patch_storage_directory,
                                 "custom", "scripts", $Command::WIN_PRE_SCRIPT) )
        {
            chmod 0755, File::Spec->catfile($patch_storage_directory,
                                 "custom", "scripts", $Command::WIN_PRE_SCRIPT);
        }
        if ( -e File::Spec->catfile($patch_storage_directory,
                                 "custom", "scripts", $Command::WIN_POST_SCRIPT) )
        {
            chmod 0755, File::Spec->catfile($patch_storage_directory,
                                 "custom", "scripts", $Command::WIN_POST_SCRIPT);
        }
    }

    $script_txt_loc = File::Spec -> catfile ($patch_inventory_location,
                                      "custom", $Command::PRE_README);

    opatchIO->print_message ( { message => "Looking for pre-patch readme $script_txt_loc\n" } );
    #
    # If there's pre.txt, we'll cat the content first, then ask if the system 
    #   is ready for updating
    #
    if ( -e $script_txt_loc )
    {
        opatchIO->print_message ( { message => "  printing content of $script_txt_loc to stdout\n" } );
        local *script_txt_handle;
        opatchIO->print_message ( { f_handle => $fh_log_file, message => $Command::PRE_README_HEADER } );
        open(script_txt_handle, $script_txt_loc) || opatchIO->print_message ( { message => "Could not open file $script_txt_loc" } );
        @script_txt_lines = <script_txt_handle>;
        foreach $script_line (@script_txt_lines)
        {
            chomp $script_line;
            opatchIO->print_message_noverbose ( { message => $script_line } );
        }
        opatchIO->print_message ( { f_handle => $fh_log_file, message => $Command::PRE_README_TAIL } );
        close script_txt_handle;

    } 
    
    # Now give appropriate dire warnings after having catted the pre.txt
    $question = "";
    opatchIO->print_message_noverbose ( {
      message => $Command::RAC_MESSAGE
    } );
    if ( ( $shouldPropagateChange ) ) {
        $question =
           "This node is part of an Oracle Real Application Cluster.\n" .
           "  Local node is \"$local_RAC_node\"\n" .
           "  Nodes are: $all_RAC_nodes\n" ;
           
        if ( $local_node_only ) {
           # actually, if it's -local option, shouldPropagateChange()
           #   has been turned off, so we will never get to this line
        } elsif ( $Command::ROLLING_PATCH ) {
           $question = $question .
           "(patching mode = rolling)\n" .
           "Please shut down Oracle instances running out of this ORACLE_HOME on this node.\n" .
           "(Oracle Home = $Oracle_Home)\n" .
           "Is this node ready for updating?";
        } elsif ( $Command::MINIMIZE_DOWNTIME_OPTION ) {
           $question = $question .
           "(patching mode = minimize down-time)\n" .
           "Please shut down Oracle instances running out of this ORACLE_HOME on this node.\n" .
           "(Oracle Home = $Oracle_Home)\n" .
           "Is this node ready for updating?";
        } else {
           $question = $question .
           "(patching mode = all-node)\n" .
           "Please shut down all Oracle instances running out of this ORACLE_HOME on all nodes.\n" .
           "(Oracle Home = $Oracle_Home)\n" .
           "Are all the nodes ready for updating?";
        }
    } else {
      $question =
        "\n" .
        "Please shut down Oracle instances running out of this ORACLE_HOME\n" .
        "(Oracle Home = $Oracle_Home)" .
        "\n" .
        "Is this system ready for updating?";
    }
    # Ask if user is ready?    
    $answer = opatchIO -> check_to_stop_processing ( {
                                                message => $question } );

    if ( $answer eq "N" ) {
        # User requested the process stop.
        $lock_file_errors = $this_class -> free_the_lock_file ( {
                                        Oracle_Home     => $Oracle_Home,
                                        patch_id        => $patch_id } );

        my $message = "Patch application stopped at user's request.";
        opatchIO -> print_message ( { f_handle => $fh_log_file,
                                      message  => $message } );
        opatchIO -> print_message_and_die ( { message  => $message,
                                              exit_val => $this_class->ERROR_CODE_PATCHING_STOPPED_ON_USER_REQUEST } );
    }
            
    # Set the environment variables for the pre scripts to access
    $ENV{OPATCH_OPERATION} = $Command::opatch_operation;
    $ENV{OPATCH_PATCHID} = $patch_id;

    my $non_standard_inventory_location = "";
    if ( $Command::inventory_location_ptr ) {
        ( $non_standard_inventory_location ) =
               ( $Command::inventory_location_ptr =~
                                    m#^-Doracle.installer.invPtrLoc=(.+)$# );
    }

    # Backup the log file to restore after auto rollback
    my $auto_rollback_logging_error_flag = $Command::logging_error_flag;
    my $auto_rollback_logging_fh_log_file = $Command::logging_fh_log_file;
    my $auto_rollback_logging_log_file_name = $Command::logging_log_file_name;

    # Roll back bug/file conflicting patches
    foreach my $patch_id ( @$conflict_patches ) {
        opatchIO->print_message_noverbose ( { message => "Rolling back patch $patch_id...\n" } );
        $Command::ROLLBACK_CALLED_BY_APPLY = 1;

        my $message = "Rolling back patch $patch_id.\n\n";
        opatchIO -> print_message ( { f_handle => $fh_log_file,
                                      message  => $message } );

        # the argument_list could be "-id rollback_patch -ph current_patch".
        #
        my $argument_list = "-id $patch_id -ph $patch_inventory_location";

        # if users supply -invPtrLoc /tmp/oraInst.loc, $non_standard_inventory_location
        #   will contain /tmp/oraInst.loc
        if ($non_standard_inventory_location ne "") {
          $argument_list .= " -invPtrLoc $non_standard_inventory_location";
        }
        if ( $noop_flag ) { $argument_list .= " -no_op"; }
        opatchIO -> print_message ( {
                                      message  => "rollback $argument_list" } );

        RollBack -> abdomen ( { arguments => $argument_list } );
    }

    # Roll back bug-superset conflicting patches
    foreach my $patch_id ( @$bug_subset_patches ) {
        opatchIO->print_message_noverbose ( { message => "Rolling back patch $patch_id...\n" } );
        $Command::ROLLBACK_CALLED_BY_APPLY = 1;

        my $message = "Rolling back patch $patch_id.\n\n";
        opatchIO -> print_message ( { f_handle => $fh_log_file,
                                      message  => $message } );

        # the argument_list could be "-id rollback_patch -ph current_patch".
        #
        my $argument_list = "-id $patch_id -ph $patch_inventory_location";

        # if users supply -invPtrLoc /tmp/oraInst.loc, $non_standard_inventory_location
        #   will contain /tmp/oraInst.loc
        if ($non_standard_inventory_location ne "") {
          $argument_list .= " -invPtrLoc $non_standard_inventory_location";
        }
        if ( $noop_flag ) { $argument_list .= " -no_op"; }
        
        # No need to pass in -jre or -jdk to RollBack.pm because it will just
        #   use the global $Command::JRE_LOC and $Command::JDK_LOC
        
        opatchIO -> print_message ( {
                                      message  => "rollback $argument_list" } );

        RollBack -> abdomen ( { arguments => $argument_list } );
    }

    # Restore the log file to restore after auto rollback
    $Command::logging_error_flag = $auto_rollback_logging_error_flag;
    $Command::logging_fh_log_file = $auto_rollback_logging_fh_log_file;
    $Command::logging_log_file_name = $auto_rollback_logging_log_file_name;

    my $timestamp = $this_class -> get_current_timestamp();
    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "\nStarting to apply patch to the system at " .
                                              "$timestamp.\n" } );

    if ( $Command::ROLLBACK_CALLED_BY_APPLY  ) {
       my $backup_filemap = $ENV{OPATCH_BACKUP_FILEMAP};
       if ( $backup_filemap eq "TRUE" )
       {
            opatchIO->print_message_noverbose ( { 
               message => "\nBack to applying patch $patch_id...\n\n" .
                          "Backing up 2nd copy of comps.xml and filemap ...\n"      
                                                } );
       }
       else
       {
            opatchIO->print_message_noverbose ( { 
               message => "\nBack to applying patch $patch_id...\n\n" .
                          "Backing up 2nd copy of comps.xml ...\n"      
                                                } );
       }
       # Back up the comps.xml which has been modified by auto-rollback
       #   to .patch_storage as "comps.xml.after_autorb"
       $this_class->backupCompsXML({ Oracle_Home => $Oracle_Home,
                                  patch_id    => $patch_id,
                                  fh_log_file => $fh_log_file,
                                  target_name => "comps.xml.after_autorb",
                                  target_filemap_name => "filemap.after_autorb"
                               });
    } 
    #else {
    #   opatchIO->print_message_noverbose ( { message => "Patching..." } );    
    #}
    
    # Execute pre script
    opatchIO->print_message ( { message => "Looking for pre-patch script $script_loc\n" } );
    if ( (-e $script_loc) && (!$report_only) )
    {
        opatchIO->print_message ( { message => "  preparing to execute $script_loc\n" } );
        $system_command = "$script_loc -apply $patch_id $Command::PRE_PARAMETERS";
        opatchIO->print_message_noverbose ( { message => "Executing the Apply pre-patch script ($script_loc)..." } );
        #$script_result = qx/$system_command/;

        $this_class -> run_cmd_print_live ( { sys_command => $system_command,
             			              level => 2 });

        my $return_code = $this_class -> get_child_error ( {
          CHILD_ERROR     => $CHILD_ERROR
        } );


        opatchIO -> print_message ( { f_handle => $fh_log_file,
                                      message  => $system_command
                                  } );
        opatchIO->print_message ( { f_handle => $fh_log_file, message => $Command::PRE_SCRIPT_HEADER } );
        opatchIO->print_message_noverbose ( { message => $script_result } );
        opatchIO->print_message ( { f_handle => $fh_log_file, message => "exit code = \"$return_code\"" } );
        opatchIO->print_message ( { f_handle => $fh_log_file, message => $Command::PRE_SCRIPT_TAIL } );
        
        # this would remember that pre-script was run and remember the return code as well
        $this_class -> save_apply_prescript_return_code ( {
           return_code => $return_code
        } );
        
        #
        # if pre-script returns error, we ask users if they want to exit
        #
        if ( $Command::APPLY_PRE_SCRIPT_STATUS == $Command::PREPOST_ERROR ) {
          
          $question =
            "The pre patch script returned an error.  \nDo you want to STOP?" ;

          $answer = opatchIO -> check_to_stop_processing ( {
                                                message => $question } );
          if ( $answer eq "" ) {
            $this_class -> free_the_lock_file ( {
                                    Oracle_Home => $Oracle_Home,
                                    patch_id    => $patch_id } );
            opatchIO->print_message_and_die({
                exit_val => $this_class->ERROR_CODE_PRE_SCRIPT,
                message=> "Pre-patch script error"
            });
          }
        }
        
        # Assert: either pre script did not fail or users chose to 
        #  ignore pre script failure to continue
        #
        # We need to write to rollback_<ID>.sh as
        #  OH/.patch_storage/$script_loc -rollback

        my $patch_storage_directory_rbf = "";
        if ( $OSNAME =~ m#Win32# )
        {
            $patch_storage_directory_rbf = substr($patch_storage_directory, length($Oracle_Home));
        }
        else
        {
            $patch_storage_directory_rbf = substr($patch_storage_directory, length($Oracle_Home));
        }

        if ( $OSNAME =~ m#Win32# )
        {
            $patch_storage_directory_rbf = File::Spec -> catfile (
                                            "\%OH\%",
                                            $patch_storage_directory_rbf );
        }
        else
        {
            $patch_storage_directory_rbf = File::Spec -> catfile (
                                            "\$OH",
                                            $patch_storage_directory_rbf );
        }

        my $rollback_command = "";
        my $rolltemp = $Command::opatch_operation;
        if ( $OSNAME =~ m#Win32# )
        {
            $rollback_command = "set OPATCH_OPERATION=$rolltemp\n" .
                                "set OPATCH_PATCHID=$patch_id\n";
        }
        else
        {
            $rollback_command = "OPATCH_OPERATION=$rolltemp\n" .
                                "OPATCH_PATCHID=$patch_id\n" .
                                "export OPATCH_OPERATION\n" .
                                "export OPATCH_PATCHID\n";
        }

        $rollback_command .= File::Spec -> catfile (
                            $patch_storage_directory_rbf, 
                            "custom", "scripts", $script_name);
        $rollback_command = "$rollback_command -rollback $patch_id";
        opatchIO -> print_message ( { f_handle => $fh_rollback,
                                      message  => $rollback_command
                                  } );
    }

    opatchIO -> print_message ( { message => "Backing-up files before " .
                                             "patching.\n" } );

    # Make backup copies of all files so rollback or recover can be done.
    # files that can't be backed up will be saved by this function
    #   to a global arry called FAILED_BACKUP_JAR_FILES, 
    #   FAILED_BACKUP_AR_FILES 
    # We won't have FAILED_BACKUP_COPY_FILES because if the copy file
    #   is not under OH, we treat it as a new file 
    my $pre_patch_failure = $this_class -> backup_files_before_patching ( {
                                  fh_log_file     => $fh_log_file,
                                  noop_flag       => $noop_flag,
                                  Oracle_Home     => $Oracle_Home,
                                  patch_id        => $patch_id,
                                  rh_actions_list => $rh_actions_list,
                                  rh_file_details => $rh_file_details } );


    # The back-up process failed. Send the details to the user and exit.
    if ( $pre_patch_failure ) {
        # Fault-tolerant: allow users to decide whether to go on
        #   or not.  Default answer is to stop.
        my $message = "OPatch encounters the following issues during file back-up:\n" .
                      "$pre_patch_failure" . "\n" .
                      "Do you want to continue?";

        $answer = opatchIO -> check_to_stop_processing ( {
                                 message => $message
                              } );
        if ( $answer eq "" ) {
          my $message = "Ignore back-up issues and continue...\n"; 
          opatchIO -> print_message ( {
                         message => $message
                      } ); 
        } else {
          opatchIO -> print_message ( { f_handle => $fh_log_file,
                                      message  => $message } );
          opatchIO -> print_message_and_die ( {
                      exit_val => $this_class->ERROR_CODE_PATCH_PROBLEM,
                      message => "File Back-up Errors!" } );
        }
    }

    # For the rest of the program ignore the interrupt signal. It's much
    # easier to rollback. However this isn't a guaranteed method. The
    # following subroutines still must check for intrupted processing.
    $SIG{INT}='IGNORE';

    opatchIO -> print_message_noverbose ( { f_handle => $fh_log_file,
                               message => "Applying patch $patch_id...\n" } );

    # Use rh_file_details (built from etc/config/actions) to build up
    #   a list of files to be patched
    my $rh_file_list = $this_class -> extract_of_file_names ( {
                                  Oracle_Home     => $Oracle_Home,
                                  patch_id        => $patch_id,
                                  rh_file_details => $rh_file_details } );

    # Apply the patches. Takes a long time to get here doesn't it.
    #
    if ($report_only) {
        opatchIO->print_message({ message =>
          "-report mode: skip apply_patches_to_files().  OPatch ends here\n"
          });
        opatchIO->debug({ message => "-report mode: OPatch ends here\n" });
        exit 1;
    }
    my $patching_failure = $this_class -> apply_patches_to_files ( {
                               fh_log_file     => $fh_log_file,
                               fh_rollback     => $fh_rollback,
                               noop_flag       => $noop_flag,
                               Oracle_Home     => $Oracle_Home,
                               patch_id        => $patch_id,
                               patch_location  => $patch_inventory_location,
                               rh_actions_list => $rh_actions_list,
                               rh_file_details => $rh_file_details } );

    # Lack of time here. Need a function to rollback the changes on error
    # at this point and exit. Use the Rollback methods where possible.

    if ( $patching_failure ) {
          my $question = "OPatch encounters the following issues during file patching:\n" .
                         "$patching_failure" . "\n" .
			 "Replying 'Y' will terminate the patch installation immediately. " .
 			 "It WILL NOT restore any updates that have been performed to this point. " .
			 "It WILL NOT update the inventory." . "\n" .
			 "Replying 'N' will update the inventory showing the patch has been applied." . "\n" .
			 "NOTE: After replying either 'Y' or 'N' it is critical to review: " . "\n" .
			 "      Metalink Note 312767.1 How to rollback a failed Interim patch installation." . "\n" . 
                         "Do you want to STOP?";
          my $answer = opatchIO->check_to_stop_processing({
                                    message => $question
                                 });
          if ($answer eq "") {
             opatchIO->print_message_and_die ({
                         exit_val => $this_class->ERROR_CODE_PATCH_PROBLEM,
                                     f_handle => *STDERR,
                                     message => "File Patching Error!" });
          } else {
             my $message = "Ignore file patching issues and continue...\n"; 
             opatchIO->print_message ({ f_handle=>$fh_log_file,
                                        message =>$message });
          }
    }

    my $make_failure = $this_class -> create_and_run_make_command_list ( {
                               fh_log_file     => $fh_log_file,
                               noop_flag       => $noop_flag,
                               Oracle_Home     => $Oracle_Home,
                               patch_id        => $patch_id,
                               patch_location  => $patch_inventory_location,
                               rh_actions_list => $rh_actions_list,
                               rh_file_details => $rh_file_details } );
    
    if ( $make_failure ) {
           my $question = "OPatch encounters the following issues during Make:\n" .
                          "$make_failure" . "\n" .
			  "Replying 'Y' will terminate the patch installation immediately." .
                          "It WILL NOT restore any updates that have been performed to this point. " .
                          "It WILL NOT update the inventory." . "\n" .
			  "Replying 'N' will continue with updating the inventory showing the patch has been applied." . "\n" .
			  "NOTE: After replying either 'Y' or 'N' it is critical to review: " . "\n" .
                          "      Metalink Note 312767.1 How to rollback a failed Interim patch installation." . "\n" .
                          "Do you want to STOP?";
           my $answer = opatchIO->check_to_stop_processing({
                           message => $question
                        });
           if ($answer eq "") {
             opatchIO->print_message_and_die ({
                         exit_val => $this_class->ERROR_CODE_PATCH_PROBLEM,
                                     f_handle => *STDERR,
                                     message => "Make Error!" });  
           } else {
              my $message = "Ignore Make issues and continue...\n"; 
              opatchIO->print_message ({ f_handle=>$fh_log_file,
                                        message =>$message });
           }
    }

    if( !$no_inventory_update ) {
        # Before proceeding further, lets try to clean up the filemap
        # if it has some dangling filemap entries.
        my $cleanup_result = $this_class -> clean_up_inventory( { f_handle => $fh_log_file,
                                                         Oracle_Home => $Oracle_Home } );

        if( $cleanup_result ) {
                opatchIO->print_message_noverbose( { message => "Could not find if the inventory is clean or not.\n" } );
        }
    }

    my $message = "Performing post-patch inventory update and " .
                      "removing working files.\n";

    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                      message  => $message } );

    # If there were no patching errors update the inventory and clean-up.
    if ( ! $noop_flag ) {

            my $clean_up_failure = $this_class -> clean_up_after_patching ( {
                                 Oracle_Home     => $Oracle_Home,
                                 rh_actions_list => $rh_actions_list,
                                 patch_id        => $patch_id,
                                 rh_file_details => $rh_file_details } );
            if ( $clean_up_failure ) {
                opatchIO -> print_message ( {
                                        message => "\n$clean_up_failure" } );
            } else {
                my $message = "";
                if ( $shouldPropagateChange ) {
                    $message = "Patch $patch_id has been successfully " .
                               "applied to this node: $local_RAC_node.\n";
                } else {
                    $message = "Patch $patch_id has been applied " .
                               "successfully.\n";
                }
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                              message  => $message } );
            }
    }

    # If it's a RAC installation propagate the changes to the other
    # nodes. The only thing that will be an issue are the log files
    # as they are still open and growing.
    my $rac_errors = "";
    if ( $shouldPropagateChange ) {
        my $timestamp = $this_class -> get_current_timestamp();
        opatchIO -> print_message ( { f_handle => $fh_log_file,
                                      message  => "Starting to propagate patches to nodes for Apply at " .
                                                  "$timestamp.\n" } );
        # The following function returns:
        #       empty string if success
        #       empty string if there are errors and users decide to go on
        #       sys_call_result (result of calling Java programs) if there are
        #          fatal fatal errors or (errors, prompted, users want to quite)
        $rac_errors = $this_class -> propagate_patch_to_RAC_nodes ( {
                       all_nodes         => $all_RAC_nodes,
                       fh_log_file       => $fh_log_file,
                       local_node        => $local_RAC_node,
                       general_options   => $general_options,
                       general_shutdown  => $shutdown_needed_flag,
                       Oracle_Home       => $Oracle_Home,
                       patch_id          => $patch_id,
                       patched_RAC_nodes => $patched_RAC_nodes,
                       RAC_options       => $RAC_options,
                       rh_file_list      => $rh_file_list,
                       rh_OUI_file_names => $rh_OUI_file_names,
                       shutdown_RAC_flag => $shutdown_RAC_flag } );
    } elsif ( $local_node_only ) {
         opatchIO->print_message({message =>
              "-local mode: patch is not propagated to other nodes.\n"
          });
    } elsif (  $is_CFS == 1 ) { 
         opatchIO->print_message({message =>
              "CFS detected: patch is not propagated to other nodes.\n"
          });
    }

    my $inventory_update_error = "";
    if ( $rac_errors ) {
    
        # Report the RAC errors and finish.
        my $final_message = 
            "Problems during RAC propagation:\n" .
            "$rac_errors" . "\n" .
            "Some nodes might have been patched but OPatch will not update " .
            "inventory.  \nPlease run OH/.patch_storage/<ID>/rollback_<ID>.sh, then " . 
            "you might try one of the following suggestions:\n" .
            "1) Run 'opatch apply -local' on both patched and unpatched nodes.\n" .
            "2) Try 'opatch apply' again on this node.\n"
            ;

        opatchIO -> print_message_and_die({
                              exit_val => $this_class -> ERROR_RAC_INSTALL_PROBLEM,
                              message  => $final_message
                              });

    } else {
    
        if ( ! $no_inventory_update ) {            
                my $timestamp = $this_class -> get_current_timestamp();
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                              message  => "Starting to update inventory for Apply at " .
                                                          "$timestamp.\n" } );
                opatchIO->print_message_noverbose({message =>
                  "\nUpdating inventory..."
                });
                if ($local_node_only) {
                   opatchIO->debug({message=>
                        "now calling update_inventory() with with -local option set\n"
                   });
                } else {
                    opatchIO->debug({message=>
                        "propagate_patch_to_RAC_nodes() done, now update_inventory()\n"
                  });
                }

                $inventory_update_error =
                          $this_class -> update_inventory ( {
                              fh_log_file       => $fh_log_file,
                              Oracle_Home       => $Oracle_Home,
                              patch_id          => $patch_id,
                              patch_location    => $patch_inventory_location,
                              rh_bug_list       => $rh_bug_list,
                              rh_OUI_file_names => $rh_OUI_file_names } );

                # inventory_update_error will be filled with output from
                #   the UpdateInventory.java if there's error.  Otherwise,
                #   it'll be blank
                if ( $inventory_update_error ) {
                    my $inventory_update_msg = 
                    "Files on system are patched but Inventory Update " . 
                    "has failed.  Please run 'opatch lsinventory' " .
                    "to check if the patch has been recorded in Inventory.\n" .
                    "(" .
                    $inventory_update_error .
                    ")\n" ;

                    opatchIO -> print_message_noverbose ( { 
                                f_handle => $fh_log_file,
                                message  => $inventory_update_msg
                    } );
                }
        }

        # Invoke verify_patch to verify the patch application.
        # Should I skip verification?
        my $skipVerify = $ENV{OPATCH_SKIP_VERIFY};

        if( $skipVerify eq "TRUE" || $skipVerify eq "true" ) {
              opatchIO->print_message_noverbose({message =>
                          "\nSkip verifying patch..."
              });

        } elsif( ! $no_inventory_update ) {
              # Skip verification if no_inventory.
              $this_class -> verify_patch ( {
                              fh_log_file       => $fh_log_file,
                              Oracle_Home       => $Oracle_Home,
                              patch_id          => $patch_id,
                              patch_location    => $patch_inventory_location,
                              rh_OUI_file_names => $rh_OUI_file_names, 
                              no_inventory      => $no_inventory_update } );
        }

        # Back up comps.xml one more time.  Use-case: cust. applies,
        #   update inv. fails, and cust. goes ahead applying other
        #   patches.  This back up of comps.xml ensures they have
        #   a good copy to return the system to.
        #
        # (we are backing up comps.xml one more time even if cust.
        #  invoked with -no_inventory).
        #
        my $backup_filemap = $ENV{OPATCH_BACKUP_FILEMAP};
        if ( $backup_filemap eq "TRUE" )
        {
        opatchIO->print_message_noverbose ( {
               message =>
                          "Backing up comps.xml and filemap ...\n"
        } );
        }
        else
        {
            opatchIO->print_message_noverbose ( { 
                   message => 
                              "Backing up comps.xml ...\n"      
                                                } );
        }
        $this_class->backupCompsXML({ Oracle_Home => $Oracle_Home,
                                  patch_id    => $patch_id,
                                  fh_log_file => $fh_log_file,
                                  target_name => "comps.xml.afterapply",
                                  target_filemap_name => "filemap.afterapply"
                               });

        my $free_lock_error = $this_class -> free_the_lock_file ( {
                                        Oracle_Home     => $Oracle_Home,
                                        patch_id        => $patch_id } );
        # Exit with error
        if($inventory_update_error) {
          opatchIO -> print_message_and_die({
                              exit_val => $this_class -> ERROR_CODE_INVENTORY_PROBLEM,
                              message => "Update Inventory error"
                            });
        }
     }


    # Call post Apply script if available
    # For Windows call post.bat and for non windows call post
    $script_loc = "";
    $script_txt_loc = "";
    $script_line = "";
    $script_result = "";

    if ( $OSNAME =~ m#Win# )
    {
        $script_loc = File::Spec -> catfile ($patch_inventory_location,
                                        "custom", "scripts", $Command::WIN_POST_SCRIPT);
        $script_name = $Command::WIN_POST_SCRIPT;
    }
    else
    {
        $script_loc = File::Spec -> catfile ($patch_inventory_location,
                                        "custom", "scripts", $Command::UNIX_POST_SCRIPT);
        $script_name = $Command::UNIX_POST_SCRIPT;
    }
    $script_txt_loc = File::Spec -> catfile ($patch_inventory_location,
                                        "custom", $Command::POST_README);

    # cat post readme to stdout & save to .patch_storage
    opatchIO->print_message ( { message => "Looking for post-patch readme $script_txt_loc\n" } );
    if ( -e $script_txt_loc )
    {
        opatchIO->print_message ( { message => "  printing content of $script_txt_loc to stdout\n" } );
        local *script_txt_handle;
        opatchIO->print_message ( { f_handle => $fh_log_file, message => $Command::POST_README_HEADER } );
        open(script_txt_handle, $script_txt_loc) || opatchIO->print_message ( { message => "Could not open file $script_txt_loc" } );
        @script_txt_lines = <script_txt_handle>;
        foreach $script_line (@script_txt_lines)
        {
            chomp $script_line;
            opatchIO->print_message_noverbose ( { message => $script_line } );
        }
        opatchIO->print_message ( { f_handle => $fh_log_file, message => $Command::POST_README_TAIL } );
        close script_txt_handle;
    }
    # exec() post script
    opatchIO->print_message ( { message => "Looking for post-patch script $script_loc\n" } );
    if ( (-e $script_loc) && (!$report_only) )
    {
        opatchIO->print_message ( { message => "  preparing to execute $script_loc\n" } );
        $system_command = "$script_loc -apply $patch_id $Command::POST_PARAMETERS";
        opatchIO->print_message_noverbose ( { message => "Executing the Apply post-patch script ($script_loc)..." } );        
        #$script_result = qx/$system_command/;

        $this_class -> run_cmd_print_live ( { sys_command => $system_command,
                               level => 2 } );


        my $return_code = $this_class -> get_child_error ( {
          CHILD_ERROR     => $CHILD_ERROR
        } );
        opatchIO -> print_message ( { f_handle => $fh_log_file, 
                                      message  => $system_command
                                  } );
        opatchIO->print_message ( { f_handle => $fh_log_file, message => $Command::POST_SCRIPT_HEADER } );
        opatchIO->print_message_noverbose ( { message => $script_result } );
        opatchIO->print_message ( { f_handle => $fh_log_file, message => "exit code = \"$return_code\"" } );
        opatchIO->print_message ( { f_handle => $fh_log_file, message => $Command::POST_SCRIPT_TAIL } );

        # this would remember that post-script was run
        $this_class -> save_apply_postscript_return_code ( {
           return_code => $return_code
        } );
        
        # if post script returns error, we need to warn users but won't exit
        if ( $Command::APPLY_POST_SCRIPT_STATUS == $Command::PREPOST_ERROR ) {
          my $message = "The post-patch script $script_loc has returned an error.\n" ;
                        
          opatchIO -> print_message_noverbose ( { f_handle => $fh_rollback,
                                      message  => $message
                                  } );
        }

        # We need to write to rollback_<ID>.sh as
        #  OH/.patch_storage/$script_loc -rollback

        my $patch_storage_directory_rbf = "";
        if ( $OSNAME =~ m#Win32# )
        {
            $patch_storage_directory_rbf = substr($patch_storage_directory, length($Oracle_Home));
        }
        else
        {
            $patch_storage_directory_rbf = substr($patch_storage_directory, length($Oracle_Home));
        }

        if ( $OSNAME =~ m#Win32# )
        {
            $patch_storage_directory_rbf = File::Spec -> catfile (
                                            "\%OH\%",
                                            $patch_storage_directory_rbf );
        }
        else
        {
            $patch_storage_directory_rbf = File::Spec -> catfile (
                                            "\$OH",
                                            $patch_storage_directory_rbf );
        }

        my $rollback_command = "";
        my $rolltemp = $Command::opatch_operation;
        if ( $OSNAME =~ m#Win32# )
        {
            $rollback_command = "set OPATCH_OPERATION=$rolltemp\n" .
                                "set OPATCH_PATCHID=$patch_id\n";
        }
        else
        {
            $rollback_command = "OPATCH_OPERATION=$rolltemp\n" .
                                "OPATCH_PATCHID=$patch_id\n" .
                                "export OPATCH_OPERATION\n" .
                                "export OPATCH_PATCHID\n";
        }

        $rollback_command .= File::Spec -> catfile ($patch_storage_directory_rbf, 
                                                      "custom", "scripts", $script_name);
        $rollback_command = "$rollback_command -rollback $patch_id";
        opatchIO -> print_message ( { f_handle => $fh_rollback,
                                      message  => $rollback_command
                                  } );
    }

    # Remind users to check binaries on remote nodes if there's remote make
    if ( ! ($Command::REMOTE_MAKE_COMMAND_IS_INVOKED eq "") ) {
      opatchIO -> print_message_noverbose ( {
          f_handle => $fh_log_file,
          message  => 
          "***************************************************************\n" .
          "OPatch invoked the following commands\n" .
          "  on the remote nodes to relink Oracle binaries.\n"      .
          $Command::REMOTE_MAKE_COMMAND_INVOKED .
          "\n" .
          "  Please check the binaries' timestamp and size\n"       .
          "  to make sure they are relinked correctly.\n"           .
          "***************************************************************\n" 
      } );
    }
    
    # Remind users to invoke make manually because some has failed
    if ( ! ($Command::REMOTE_MAKE_COMMAND_FAILED eq "") ) {
      opatchIO -> print_message_noverbose ( {
          f_handle => $fh_log_file,
          message  => 
          "***************************************************************\n" .
          "                       ERROR                                   \n" .
          "                                                               \n" .
          "You have to manually run make on remote nodes\n" .
          "  because the following commands failed:\n"      .
          $Command::REMOTE_MAKE_COMMAND_FAILED .
          "\n" .
          "  After running make, please check the binaries' timestamp and size\n" .
          "  to make sure they are relinked correctly.\n"           .
          "***************************************************************\n" 
      } );
    }
    
    my $timestamp = $this_class -> get_current_timestamp();
    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "Ending Apply session at " .
                                              "$timestamp.\n" } );
    opatchIO -> print_message_and_die({
          exit_val => $this_class -> ERROR_CODE_NO_ERROR,
          message => "Apply done"
        });


}   # End of abdomen().

###############################################################################
#
# NAME   : patch_archive_files
#
# PURPOSE: Patch (that is update) the contents of a file archive.
#
# INPUTS : $$ARG[1]{fh_log_file}      - The file handle for the log file.
#          $$ARG[1]{fh_rollback}      - The file handle for the rollback
#                                       commands.
#          $$ARG[1]{noop_flag}        - Just report the actions.
#          $$ARG[1]{Oracle_Home}      - The location of the area to be patched.
#          $$ARG[1]{patch_id}         - The patch identification string.
#          $$ARG[1]{patch_location}   - Where the patches are located.
#          $$ARG[1]{archive_files}    - The hash of files that are to be
#                                       patched and which archive to use.
#          $$ARG[1]{rh_actions_list}  - The action require for a type of file.
#
# OUTPUTS: $error_messages - A string containing any errors encountered
#                            during backup the process.
#
# NOTES  : 1. The update command for an archive file is currently only for
#             a system call. If this changes then logic will need to be added
#             to find what sort of call to use and dispatch the call in an
#             appropriate manner.
#          2. System commands are assumed to be available via the user's $PATH.
#          3. $parameter_count not used but may be needed at some future point.
#          4. The rollback material is a quick hack. This needs to be done
#             properly.
#
###############################################################################
sub patch_archive_files {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file      = $$rh_arguments{fh_log_file};
    my $fh_rollback      = $$rh_arguments{fh_rollback};
    my $noop_flag        = $$rh_arguments{noop_flag};
    my $Oracle_Home      = $$rh_arguments{Oracle_Home};
    my $patch_id         = $$rh_arguments{patch_id};
    my $patch_location   = $$rh_arguments{patch_location};
    my $archive_files    = $$rh_arguments{archive_files};
    my $rh_actions_list  = $$rh_arguments{rh_actions_list};

    my $error_messages   = "";
    my $parameter_count  = 0;
    my $initial_command  = "";
    my $rollback_command = "";
    my $system_command   = "";

   
    if( scalar(%$archive_files ))
    {
    opatchIO -> print_message_noverbose ( { f_handle => $fh_log_file,
                                  message => "Patching archive files...\n" } );
    }
    else
    {
	return ( "" );
    }

    $patch_location = File::Spec -> catfile (
                                     "$patch_location", &Apply::PATCH_BASE );

    my $archive_command = $this_class->getArchiveCommand();
    if( $archive_command eq "" ) { 
       opatchIO -> print_message_and_die ( {
		       exit_val => $this_class -> ERROR_CODE_PATH_PROBLEM,
		       message =>
		       "'archive' is not in your PATH or is not executable.\n" .
           "  Extra Search Path= /usr/bin/ar (Linux), /usr/ccs/bin/ar (others)\n" .
		       "  Path= $ENV{PATH}\n" 
	       });    
    }

    # Extract the apply command out of the actions list.
    foreach my $command ( keys %{$$rh_actions_list{archive_lib}{apply}} ) {
        $initial_command = "$archive_command ";
        $initial_command .= $$rh_actions_list{archive_lib}{apply}{$command}
                                                                {parameters};

        my @array_to_count = split ( /%/, $initial_command );
        $parameter_count = ( scalar (@array_to_count) - 1 );

        # This next regex should leave "ar -rc ".
        ( $initial_command ) = ( $initial_command =~ m#^([^%]+)%# );
        $rollback_command = $initial_command;
    }

    # Emergency rollback.
    my ( $copy_command ) = ( $OSNAME =~ m#Win32# ) ? "copy" : "cp" ;
    my ( $del_command )  = ( $OSNAME =~ m#Win32# ) ? "del"  : "rm" ;

    # rbf vars to be used for rollback_ID.sh/cmd
    my $patch_directory_rbf = "";
    my $finished_name_rbf = "";
    if ( $OSNAME =~ m#Win32# )
    {
        $patch_directory_rbf = File::Spec -> catfile
                                         ( "\%OH\%",
                                           $this_class -> save_dir,
                                           $patch_id );
    }
    else
    {
        $patch_directory_rbf = File::Spec -> catfile
                                         ( "\$OH",
                                           $this_class -> save_dir,
                                           $patch_id );
    }

    my $patch_directory   = File::Spec -> catfile
                                         ( "$Oracle_Home",
                                           $this_class -> save_dir,
                                           $patch_id );

    my $finished_name = "";
    my $error_counter = 1;
    foreach my $file ( keys %$archive_files ) {

        my $simple_name = &File::Basename::basename( $file );
        my $target = $$archive_files{$file};

        my $target_rbf = "";
        if ( $OSNAME =~ m#Win32# )
        {
            $target_rbf = substr ($target, length($Oracle_Home));
        }
        else
        {
            $target_rbf = substr ($target, length($Oracle_Home));
        }

        if ( $OSNAME =~ m#Win32# )
        {
            $target_rbf = File::Spec -> catfile (
                                            "\%OH\%",
                                            $target_rbf );
        }
        else
        {
            $target_rbf = File::Spec -> catfile (
                                            "\$OH",
                                            $target_rbf );
        }

        my $library = File::Basename::basename( $target );
        $file = File::Spec -> catfile ( "$patch_location", "$file" );
        $system_command = "$initial_command" . "$target " . "$file";

        # Test to see if the file has already been patched on a prior run
        # that was interrupted.
        # This kind of reg. exp. matching might give Unicode error if
        # there is \p in Oracle_Home, as in C:\phi
        # Since there is no Archive action in Windows, we will leave it alone
        # for now
         $finished_name = &File::Basename::dirname( $target );
         $finished_name = substr($finished_name, length($Oracle_Home));
         # This variable is for the rollback_ID.sh/cmd
         $finished_name_rbf = File::Spec -> catfile ( 
                                                  "$patch_directory_rbf" ,
                                                  "$finished_name",
                                                  "$simple_name" );
         $finished_name = File::Spec -> catfile ( "$patch_directory" ,
                                                  "$finished_name",
                                                  "$simple_name" );

        my $original_name = $finished_name;
        my $original_name_rbf = $finished_name_rbf;
        $finished_name .= "_" . $library . "_" . &Apply::END_PATCH .
                                                                "_$patch_id";
        $finished_name_rbf .= "_" . $library . "_" . &Apply::END_PATCH .
                                                                "_$patch_id";
        my $rollback_name = $original_name . "_" . $library . "_" . &Apply::PRE_PATCH . "_$patch_id";
        my $rollback_name_rbf = $original_name_rbf . "_" . $library . "_" . &Apply::PRE_PATCH . "_$patch_id";

        if ( -e $finished_name ) {
	        opatchIO -> print_message ( {f_handle => $fh_log_file, 
					     message => "File patched on previous run, will be skipped : " .
        	                                         $finished_name } );
		next;
	}

        # Check if the file/directory exists. If not there is something wrong
        # with the system or the patch. Yes, this was tested for earlier.
        if ( ! -e $target ) {
            # If this file is one of the backup-failed files users have OKed,
            #  we can ignore it.
            my $ok = 0;
            foreach my $file ( @Command::FAILED_BACKUP_AR_FILES ) {
               if ( "$target" eq "$file" ) {
                  opatchIO->print_message({
                     message => "User has chosen to ignore failure of " .
                                "backing up archive file $target. " .
                                "It is OK to go on.\n"
                  });
                  $ok = 1;
               }
            }
            if (! $ok) {
               $error_messages .= "$error_counter.\t$target\n" .
                               "\t[ No archive archive file found. ]\n";
               $error_counter++;
               next;
            } else {
               # skip this file
               next;
            }
        }

        opatchIO -> print_message ( { message => "  archiving $target " .
                                                 "using $simple_name.\n" } );

        if ( ! $noop_flag ) {
            # Make a copy of the archive in case things go horribly wrong.
            # It'll take manual intervention to fix this but its a good safety
            # measure. If every thing goes OK this file is removed.
            my $backup_target = $target . "_" . &Apply::BACKUP . "_$patch_id";
            opatchIO -> print_message ( { f_handle => $fh_log_file,
                                          message  => "  backing up " .
                                          "$target as $backup_target\n" } );
            File::Copy::copy ( $target, $backup_target ) || (
                    ( $error_messages .= "$error_counter.\t$target\n" .
                                         "\t[ Couldn't make backup copy of ".
                                         "file. ]\n"), $error_counter++,
                      next );

            # First lets escape any special characters.
            $original_name_rbf = $this_class->escape_special_chars({ name => $original_name_rbf });
            $target_rbf = $this_class->escape_special_chars({ name => $target_rbf });
            $rollback_name_rbf = $this_class->escape_special_chars({ name => $rollback_name_rbf });

            # Write out the emergency rollback details.
            my $message = "    $copy_command $rollback_name_rbf $original_name_rbf\n" .
                          "    $rollback_command $target_rbf $original_name_rbf\n" .
                          "    $del_command $original_name_rbf\n" ;
            if ( $OSNAME !~ m#Win32# ) {
                $message = "if [ -f $rollback_name_rbf ]; then\n" . $message .
                           "fi\n\n";
            }

            opatchIO -> print_message ( { f_handle => $fh_rollback,
                                          message  => $message } );

            # So lets patch the file.
            my ( $sys_call_result ) = qx/$system_command/;
            chomp $sys_call_result;
            opatchIO -> print_message ( { f_handle => $fh_log_file,
                                          message  => "  " .
                                          $system_command . "\nResult:  " .
                                          "$sys_call_result\n" } );
            my $status = $this_class -> get_child_error ( {
                CHILD_ERROR     => $CHILD_ERROR
            } );
            if ( $status ) {
                $error_messages .= "$error_counter.\t$target\n" .
                                   "\t[ object: " .
                                   &File::Basename::basename( $file ) . " ]\n";
                $error_counter++;
            }

            # Now create an empty file to signal this operation was
            # successful. Since file creation is OS dependent this is being
            # done within perl.
            if ( ! $status ) {
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                              message  => "  creating " .
                                                       "$finished_name " .
                                                       "to signal successful ops.\n"
                                          } );
                open ( SUCCESS, ">$finished_name" ) || (
                    ( $error_messages .= "$error_counter.\t$finished_name\n" .
                                         "\t[ Non-critical error message: " .
                                         "$OS_ERROR ]\n"),
                      $error_counter++ );
                close SUCCESS;
                unlink $backup_target;
            }
        }
    }

    if ( $error_messages ) {
        $error_messages = "The following problems were encountered when " .
                          "applying patches:\n" . $error_messages;
    }

    return ( $error_messages );

}   # End of patch_archive_files().

###############################################################################
#
# NAME   : patch_copy_files
#
# PURPOSE: Patch (that is copy) the listed files into place.
#          Mkdir -p if the directories do not exist
#          Record all dirs. that were mkdir'ed into a file
#
# INPUTS : $$ARG[1]{fh_log_file}      - The file handle for the log file.
#          $$ARG[1]{fh_rollback}      - The file handle for the rollback
#                                       commands.
#          $$ARG[1]{noop_flag}        - Just report the actions.
#          $$ARG[1]{Oracle_Home}      - The location of the area to be patched.
#          $$ARG[1]{patch_id}         - The patch identification string.
#          $$ARG[1]{patch_location}   - Where the patches are located.
#          $$ARG[1]{copy_files}       - The hash of files that are to be
#                                       patched and which archive to use.
#          $$ARG[1]{rh_actions_list}  - The action require for a type of file.
#
# OUTPUTS: $error_messages - A string containing any errors encountered
#                            during backup the process.
#
# NOTES  : 1. The patch command for a file copy is currently only for a perl
#             call. If this changes then logic will need to be added to find
#             what sort of call to use and dispatch the call in an
#             appropriate manner.
#          2. $parameter_count not used but may be needed at some future point.
#          3. The rollback material is a quick hack. This needs to be done
#             properly.
#
###############################################################################
sub patch_copy_files {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file      = $$rh_arguments{fh_log_file};
    my $fh_rollback      = $$rh_arguments{fh_rollback};
    my $noop_flag        = $$rh_arguments{noop_flag};
    my $Oracle_Home      = $$rh_arguments{Oracle_Home};
    my $rh_actions_list  = $$rh_arguments{rh_actions_list};
    my $copy_files       = $$rh_arguments{copy_files};
    my $patch_id         = $$rh_arguments{patch_id};
    my $patch_location   = $$rh_arguments{patch_location};

    my $error_messages   = "";
    my $parameter_count  = 0;
    my $initial_command  = "";

    if( scalar(%$copy_files ))
    {
    opatchIO -> print_message_noverbose ( { f_handle => $fh_log_file,
                                  message => "Patching copy files...\n" } );
    }
    else
    {
	return ( "" );
    }

    my $patch_base = File::Spec -> catfile (
                                     "$patch_location", &Apply::PATCH_BASE );

    # Extract the apply command out of the actions list.
    foreach my $command ( keys %{$$rh_actions_list{copy}{apply}} ) {

        my $parameter_count = $$rh_actions_list{copy}{apply}{$command}
                                                                 {param_num};
        my $required_module = $$rh_actions_list{copy}{apply}{$command}{module};
    }


    # Only covering Unix and MicroSoft at this time.
    my ( $copy_command ) = ( $OSNAME =~ m#Win32# ) ? "copy" : "cp" ;

    # rbf vars to be used for rollback_ID.sh/cmd
    my $patch_directory_rbf = "";
    my $finished_name_rbf = "";
    if ( $OSNAME =~ m#Win32# )
    {
        $patch_directory_rbf = File::Spec -> catfile
                                         ( "\%OH\%",
                                           $this_class -> save_dir,
                                           $patch_id );
    }
    else
    {
        $patch_directory_rbf = File::Spec -> catfile
                                         ( "\$OH",
                                           $this_class -> save_dir,
                                           $patch_id );
    }

    my $patch_directory   = File::Spec -> catfile
                                         ( "$Oracle_Home",
                                           $this_class -> save_dir,
                                           $patch_id );

    # Have to hard code it. Due to binding issues and variable interpolation
    # "do", "require", and "use" doesn't find what's in $required_module
    # or doesn't find the module (depending on usage). So it's hardwired as
    # a bare word.

    my $finished_name = "";
    my $error_counter = 1;

    # this list holds all directories that we created
    my $list_of_dirs_to_mkdir = "";

    foreach my $file ( keys %$copy_files ) {

        my $patch = $$copy_files{$file};

        my $file_rbf = "";
        if ( $OSNAME =~ m#Win32# )
        {
            $file_rbf = substr($file, length($Oracle_Home));
        }
        else
        {
            $file_rbf = substr($file, length($Oracle_Home));
        }

        if ( $OSNAME =~ m#Win32# )
        {
            $file_rbf = File::Spec -> catfile (
                                            "\%OH\%",
                                            $file_rbf );
        }
        else
        {
            $file_rbf = File::Spec -> catfile (
                                            "\$OH",
                                            $file_rbf );
        }

        $patch = File::Spec -> catfile ( "$patch_base", "$patch" );

        if ( $OSNAME =~ m#Win32# ) {
           $file  = lc ( $file );
           $patch = lc ( $patch );
           $patch =~ s#/#\\#g;
        }

        # The reg. exp. matching on Oracle_Home below could have failed
        #   with Unicode error if there is a \p as in C:\phi
        #   Fortunately, we have  $Oracle_Home =~ s/\\/\\\\/g;  
        #   before entering this loop.
        my $simple_name = &File::Basename::basename( $file );
        if ( $OSNAME =~ m#Win32# )
        {
            $finished_name_rbf = File::Spec -> catfile ( 
                                         "$patch_directory_rbf",
                                         substr($file, length($Oracle_Home)));
        }
        else
        {
            $finished_name_rbf = File::Spec -> catfile ( 
                                         "$patch_directory_rbf",
                                         substr($file, length($Oracle_Home)));
        }
        $finished_name = File::Spec -> catfile ( "$patch_directory",
                                         substr($file, length($Oracle_Home)));
    
        my $original_name = $finished_name;
        my $original_name_rbf = $finished_name_rbf;
        $finished_name .=  "_" .  &Apply::END_PATCH . "_" . $patch_id;
        $finished_name_rbf .=  "_" .  &Apply::END_PATCH . "_" . $patch_id;
        my $rollback_name = $finished_name;
        my $rollback_name_rbf = $finished_name_rbf;
        $rollback_name = $original_name . "_" . &Apply::PRE_PATCH .
                                                             "_$patch_id";
        $rollback_name_rbf = $original_name_rbf . "_" . &Apply::PRE_PATCH .
                                                             "_$patch_id";

        # Test to see if the file was patched on a prior run.
        if ( -e $finished_name ) {
                opatchIO -> print_message ( {f_handle => $fh_log_file,
                                             message => "File patched on previous run, will be skipped : " .
                                                         $finished_name } );
                next;
        }


        # So lets patch the file.
        my $status = 0;
        my $message = "  replacing file $simple_name";
        if ( ! -e $file ) { $message = "  copying $simple_name"; }
        opatchIO -> print_message ( { message => $message } );

        if ( ! $noop_flag ) {

            # First lets escape any special characters.
            $file_rbf = $this_class->escape_special_chars({ name => $file_rbf });
            $rollback_name_rbf = $this_class->escape_special_chars({ name => $rollback_name_rbf });

            # Write out the emergency rollback details.
            my $message = "if [ -f $rollback_name_rbf ]; then\n" .
                          "    $copy_command $rollback_name_rbf $file_rbf\n" .
                          "fi\n\n";
            if ( $OSNAME =~ m#Win32# ) {
                $message = "$copy_command $rollback_name_rbf $file_rbf\n";
            }

            # if the file exists in OH, then its good to copy it back from backup
            # if the file is a new file, then no need to restore during rollback from script.
            # this is done, because for .bat files, we don't put "if" file exists in backup area,
            # as a result, the rollback.bat script may fail. But it can be generalised for .sh files also.
            # This is not required for ar and jar actions.
            if( -e $file ) {
               opatchIO -> print_message ( { f_handle => $fh_rollback,
                                              message  => $message } );
            }

            $file = &File::Basename::dirname( $file );

            opatchIO -> print_message ( { f_handle => $fh_log_file,
                                          message  => "  copying file " .
                                                     "$patch to $file" } );

            # Create the directory if it's not there
            #
            #my $directory = &File::Basename::dirname($file);
            my $directory = $file;
            if (! -e $directory) 
            {
                opatchIO->print_message_noverbose({ message =>
                        "Creating new directory \"$directory\"" }); 

                if ( $OSNAME =~ m#Win32# )
                {
                    my $system_command = "cmd.exe \/C mkdir $directory";
                    my $result = qx/$system_command/;
                }
                else
                {
                    my $system_command = "mkdir -p $directory";
                    qx/$system_command/;
                }

                # remember this entry to write to a file later
                $list_of_dirs_to_mkdir .= "$directory" . "\n";
            }
            my $cur_dir = cwd();
            File::Copy::copy ( $patch, $file ) || (
                    ( $error_messages .= "$error_counter.\t$file\n\t[ " .
                                         "Couldn't copy $patch to " .
                                         "$file from $cur_dir. ]\n" ),
                                          $error_counter++, $status = 1,
                      next );


            # Now create an empty file to signal this operation was
            # successful.
            if ( ! $status ) {
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                              message  => "  creating " .
                                                       "$finished_name " .
                                                       "to signal a successful ops."
                                            } );
                open ( SUCCESS, ">$finished_name" ) || (
                    ( $error_messages .= "$error_counter.\t$finished_name\n" .
                                         "\t[ Non-critical error message: " .
                                         "$OS_ERROR ]\n"),
                      $error_counter++ );
                close SUCCESS;
            }
        }
    }

    # Create a file under .patch_storage/<ID> to hold the list of dirs.
    #   there were created above.
    #
    # 
    my $dirListName  = File::Spec -> catfile ( "$Oracle_Home",
                                               ".patch_storage",
                                               "$patch_id" ,
                                              "$Command::DIRS_TO_MKDIR" );

    if ($list_of_dirs_to_mkdir ne "" ) {
       opatchIO->print_message_noverbose({ message =>
             "Creating file to hold list of directories that were mkdir'ed: " .
             "\"$dirListName\"" });

       if( -e $dirListName ) {
          opatchIO->print_message
            ({  f_handle => $fh_log_file, 
                message => "  removing the old file to write new content to it: " .
                                 "\"$dirListName\"" });
          unlink ( $dirListName );
        }

        local *OUTPUT_FILE;
        open ( OUTPUT_FILE, ">$dirListName" ) or do {
                opatchIO -> print_message ( {
                  f_handle => $fh_log_file,
                  message => "Cannot open file to write: " .
                             "\"$dirListName\"\n"
                } );
                $error_messages .= "\n" .
                   "Cannot open file to write \"$dirListName\"\n" ; 
        };
        # Write to it list of files to be propagated, already separated by "\n"
        print OUTPUT_FILE ("$list_of_dirs_to_mkdir");
        # Then close the file
        close OUTPUT_FILE;
        opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  =>
            "List of directories to be mkdir'ed: $list_of_dirs_to_mkdir" } );

    } # list of dirs. is not empty
 
    if ( $error_messages ) {
        $error_messages = "The following files had problems with being " .
                          "patched:\n" . $error_messages;
    }

    return ( $error_messages );

}   # End of patch_copy_files().

###############################################################################
#
# NAME   : patch_jar_files_on_Unix
#
# PURPOSE: Patch (that is update) the contents of a java archive.
#
# INPUTS : $$ARG[1]{fh_log_file}      - The file handle for the log file.
#          $$ARG[1]{fh_rollback}      - The file handle for the rollback
#                                       commands.
#          $$ARG[1]{noop_flag}        - Just report the actions.
#          $$ARG[1]{Oracle_Home}      - The location of the area to be patched.
#          $$ARG[1]{rh_actions_list}  - The action require for a type of file.
#          $$ARG[1]{jar_files}        - The hash of files that are to be
#                                       patched and which archive to use.
#          $$ARG[1]{patch_id}         - The patch identification string.
#          $$ARG[1]{patch_location}   - Where the patches are located.
#
# OUTPUTS: $error_messages - A string containing any errors encountered
#                            during backup the process.
#
# NOTES  : 1. The patch command for an archive file is currently only for
#             a system call. If this changes then logic will need to be added
#             to find what sort of call to use and dispatch the call in an
#             appropriate manner.
#          2. This call assumes the appropriate JDK is installed with the
#             product so it should be in $ORACLE_HOME.
#          3. $parameter_count not used but may be needed at some future point.
#          4. The design documents specify a -m as an argument (for a manifest
#             file) but I've not seen an XML file with it included and it's
#             not in the packaging tool. The result is it's being ignored for
#             the time being.
#          5. Need to rationalize this and Command::extract_file_names().
#          6. The rollback material is a quick hack. This needs to be done
#             properly.
#
###############################################################################
sub patch_jar_files_on_Unix {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file      = $$rh_arguments{fh_log_file};
    my $fh_rollback      = $$rh_arguments{fh_rollback};
    my $noop_flag        = $$rh_arguments{noop_flag};
    my $Oracle_Home      = $$rh_arguments{Oracle_Home};
    my $patch_id         = $$rh_arguments{patch_id};
    my $patch_location   = $$rh_arguments{patch_location};
    my $rh_actions_list  = $$rh_arguments{rh_actions_list};
    my $jar_files        = $$rh_arguments{jar_files};

    my $error_messages   = "";
    my $parameter_count  = 0;
    my $initial_command  = "";
    my $initial_command_rbf  = "";
    my $rollback_command = "";
    my $system_command   = "";

    if( scalar(%$jar_files ))
    {
	opatchIO -> print_message_noverbose ( { f_handle => $fh_log_file,
                                  message => "Patching jar files...\n" } );
    }
    else
    {
	return ( "" );
    }

    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message => "Patching jar files on Unix.\n" } );

    my $patch_base = File::Spec -> catfile (
                                     "$patch_location", &Apply::PATCH_BASE );

    # Extract the apply command out of the actions list.
    foreach my $command ( keys %{$$rh_actions_list{archive_jar}{apply}} ) {
        $initial_command = $$rh_actions_list{archive_jar}{apply}{jar}{path};

        if ( $OSNAME =~ m#Win32# )
        {
            $initial_command_rbf = substr($initial_command, length($Oracle_Home));
        }
        else
        {
            $initial_command_rbf = substr($initial_command, length($Oracle_Home));
        }

        if ( $OSNAME =~ m#Win32# )
        {
            $initial_command_rbf = File::Spec -> catfile (
                                            "\%OH\%",
                                            $initial_command_rbf );
        }
        else
        {
            $initial_command_rbf = File::Spec -> catfile (
                                            "\$OH",
                                            $initial_command_rbf );
        }

        $initial_command = File::Spec -> catfile
                                         ( "$initial_command" , "$command" );
        $initial_command_rbf = File::Spec -> catfile
                                         ( "$initial_command_rbf" , "$command" );

        $initial_command .= " " . $$rh_actions_list{archive_jar}{apply}
                                                      {$command}{parameters};
        $initial_command_rbf .= " " . $$rh_actions_list{archive_jar}{apply}
                                                      {$command}{parameters};

        my @array_to_count = split ( /%/, $initial_command );
        $parameter_count = ( scalar (@array_to_count) - 1 );

        # This next regex should leave "jar -uf ".
        ( $initial_command ) = ( $initial_command =~ m#^([^%]+)%# );
        ( $initial_command_rbf ) = ( $initial_command_rbf =~ m#^([^%]+)%# );
    }
    # Apply pre-req. checks has ensured that JDK_LOC points to
    #  an executable jar.exe
    if ( $Command::JDK_LOC ne "" ) {
        opatchIO->print_message ({ message => "  use jar from supplied -jdk option.\n" } );
        $initial_command = "$Command::JDK_LOC -uf ";
        $initial_command_rbf = "$Command::JDK_LOC -uf ";
    }

    # Emergency rollback.
    my ( $copy_command ) = ( $OSNAME =~ m#Win32# ) ? "copy" : "cp" ;
    my ( $del_command )  = ( $OSNAME =~ m#Win32# ) ? "del"  : "rm" ;

    # rbf vars to be used for rollback_ID.sh/cmd
    my $patch_directory_rbf = "";
    my $finished_name_rbf = "";
    if ( $OSNAME =~ m#Win32# )
    {
        $patch_directory_rbf = File::Spec -> catfile
                                         ( "\%OH\%",
                                           $this_class -> save_dir,
                                           $patch_id );
    }
    else
    {
        $patch_directory_rbf = File::Spec -> catfile
                                         ( "\$OH",
                                           $this_class -> save_dir,
                                           $patch_id );
    }

    my $patch_directory   = File::Spec -> catfile
                                         ( "$Oracle_Home",
                                           $this_class -> save_dir,
                                           $patch_id );

    my $curr_dir = "";
    my $finished_name = "";
    my $error_counter = 1;
    my $directory_separator = File::Spec -> catfile ( "", "bin" );
    ( $directory_separator ) = ( $directory_separator =~ m#^(.+)bin$# );
    # Watch out for Windows dir. sep. which is an escape in reg. exp.

    foreach my $file ( keys %$jar_files ) {

        my $path = $$jar_files{$file};

        my $path_rbf = "";
        if ( $OSNAME =~ m#Win32# )
        {
            $path_rbf = substr($path, length($Oracle_Home));
        }
        else
        {
            $path_rbf = substr($path, length($Oracle_Home));
        }

        if ( $OSNAME =~ m#Win32# )
        {
            $path_rbf = File::Spec -> catfile (
                                            "\%OH\%",
                                            $path_rbf );
        }
        else
        {
            $path_rbf = File::Spec -> catfile (
                                            "\$OH",
                                            $path_rbf );
        }

        my $target = &File::Basename::basename ( $path );
        my ( $patch_loc ) = ( $file =~ m#(.+)$target# );
        $patch_location = File::Spec -> catfile (
                           "$patch_base", "$patch_loc", "$target" );
        
        # $system_command = "cd $patch_location; $initial_command" . "$path ";
        # the above chain cmd does not work for Windows --> use Perl chdir
        
        $system_command = "$initial_command $path "; 
 
        my $reg_exp = $target . $directory_separator;
        $reg_exp =~ s/\\/\\\\/g;

        my ( $class_file ) = ( $file =~ m#$reg_exp(.+)$# );
        # class_file might be Foo$InnerClassName.class, so we need to 'escape'
        #   the $ in class file
        my $inner_class_file = $this_class->getInnerClassName({ class_file => $class_file });
        $system_command .= $inner_class_file;
        
        # Test to see if the file was patched on a prior run.
        $finished_name = File::Spec -> catfile ( "$patch_directory",
                                                 ( $file =~ m#(.+)$target# ),
                                                             "$class_file" );
        $finished_name_rbf = File::Spec -> catfile ( "$patch_directory_rbf",
                                                 ( $file =~ m#(.+)$target# ),
                                                             "$class_file" );
        my $original_name = $finished_name;
        my $original_name_rbf = $finished_name_rbf;
        $finished_name = "$finished_name" . "_" . $target . "_" .
                                            &Apply::END_PATCH . "_$patch_id";
        $finished_name_rbf = "$finished_name_rbf" . "_" . $target . "_" .
                                            &Apply::END_PATCH . "_$patch_id";
        my $rollback_name = $original_name . "_" . $target . "_" .
                                            &Apply::PRE_PATCH . "_$patch_id";
        my $rollback_name_rbf = $original_name_rbf . "_" . $target . "_" .
                                            &Apply::PRE_PATCH . "_$patch_id";

        #my $rollback_location = File::Spec -> catfile ( "$Oracle_Home",
        my $rollback_location = File::Spec -> catfile ( $patch_directory,
                                              ( $file =~ m#(.+)$target# ) );                                         
        my $rollback_location_rbf = File::Spec -> catfile ( $patch_directory_rbf,
                                              ( $file =~ m#(.+)$target# ) );                                         
        if ( -e $finished_name ) {
                opatchIO -> print_message ( {f_handle => $fh_log_file,
                                             message => "File patched on previous run, will be skipped : " .
                                                         $finished_name } );
                next;
        }


        # Check if the file/directory exists. If not there is something wrong
        # with the system or the patch. Yes, this was tested for earlier.
        if ( ! -e $path ) {
            # If this file is one of the backup-failed files users have OKed,
            #  we can ignore it.
            my $ok = 0;
            foreach my $file ( @Command::FAILED_BACKUP_JAR_FILES ) {
               if ( "$path" eq "$file" ) {
                  opatchIO->print_message({
                     message => "User has chosen to ignore failure of " .
                                "backing up jar file $path. " . 
                                "It is OK to go on.\n"
                  });
                  $ok = 1;
               } 
            } 
            if (! $ok) { 
               $error_messages .= "$error_counter.\t$path\n" .
                               "\t[ No java archive file found. ]\n";
               $error_counter++;
               next;
            } else {
               # skip this file
               next;
            }
        }

        # Make a copy of the Jar file in case things go horribly wrong.
        # It'll take manual intervention to fix this but its a good safety
        # measure.
        my $backup_target = $path . "_" . &Apply::BACKUP . "_$patch_id";

        # So lets patch the file.
        opatchIO -> print_message ( { message => "  patching $path " .
                                                 "with $class_file" } );

        if ( ! $noop_flag ) {

            # First lets escape any special characters.
            $original_name_rbf = $this_class->escape_special_chars({ name => $original_name_rbf });
            $rollback_name_rbf = $this_class->escape_special_chars({ name => $rollback_name_rbf });
            $path_rbf = $this_class->escape_special_chars({ name => $path_rbf });
            $class_file = $this_class->escape_special_chars({ name => $class_file });

            # Write out the emergency rollback details.
            my $message = "if [ -f $rollback_name_rbf ]; then\n" .
                          "    $copy_command $rollback_name_rbf $original_name_rbf\n" .
                          "    cd $rollback_location_rbf\n" .
                          "    $initial_command_rbf $path_rbf $class_file\n" .
                          "    $del_command $original_name_rbf\n" .
                          "fi\n\n";
            opatchIO -> print_message ( { f_handle => $fh_rollback,
                                          message  => $message } );
            opatchIO -> print_message ( { f_handle => $fh_log_file,
                                          message  => "  backing up file " .
                                          "$path as $backup_target" } );
            File::Copy::copy ( $path, $backup_target ) || (
                ( $error_messages .= "$error_counter.\t$target\n" .
                                     "\t[ Couldn't make backup copy of ".
                                     "file. ]\n"), $error_counter++,
                  next );

            $curr_dir = $this_class->get_cwd();
            opatchIO->print_message ({ f_handle => $fh_log_file,
                                    message  => "  chdir to $patch_location" });
            chdir $patch_location;
            my $sys_call_result = qx/$system_command/;
            # jump back to where we were
            chdir $curr_dir;

            chomp $sys_call_result;
            opatchIO -> print_message ( { f_handle => $fh_log_file,
                                          message  => "  " .
                                          $system_command . "\n  Result: " .
                                          "$sys_call_result" } );
            opatchIO->print_message ({ f_handle => $fh_log_file,
                                          message  => "  chdir back to $curr_dir" });
    
            my $status = $this_class -> get_child_error ( {
                      CHILD_ERROR     => $CHILD_ERROR
            } );
            if ( $status ) {
                $error_messages .= "$error_counter.\t$target\n" .
                                   "\t[ object: " .
                                   &File::Basename::basename( $file ) .
                                   " ]\n";
                $error_counter++;
            } else {

                # Now create an empty file to signal this operation was
                # successful.
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                              message  => "  creating " .
                                                       "$finished_name " .
                                                       "to signal a successful ops.\n"
                                            } );
                open ( SUCCESS, ">$finished_name" ) || (
                    ( $error_messages .= "$error_counter.\t$finished_name\n" .
                                         "\t[ Non-critical error message: " .
                                         "$OS_ERROR ]\n"),
                      $error_counter++ );
                close SUCCESS;
                unlink $backup_target;
            }
        }
    }

    if ( $error_messages ) {
        $error_messages = "The following java archives had problems with " .
                          "being patched:\n" . $error_messages;
    }

    return ( $error_messages );

}   # End of patch_jar_files().

###############################################################################
#
# NAME   : patch_jar_files_on_Windows
#
# PURPOSE: Patch (that is update) the contents of a java archive.
#
# INPUTS : $$ARG[1]{fh_log_file}      - The file handle for the log file.
#          $$ARG[1]{fh_rollback}      - The file handle for the rollback
#                                       commands.
#          $$ARG[1]{noop_flag}        - Just report the actions.
#          $$ARG[1]{Oracle_Home}      - The location of the area to be patched.
#          $$ARG[1]{rh_actions_list}  - The action require for a type of file.
#          $$ARG[1]{jar_files}        - The hash of files that are to be
#                                       patched and which archive to use.
#          $$ARG[1]{patch_id}         - The patch identification string.
#          $$ARG[1]{patch_location}   - Where the patches are located.
#
# OUTPUTS: $error_messages - A string containing any errors encountered
#                            during backup the process.
#
# NOTES  : 1. The patch command for an archive file is currently only for
#             a system call. If this changes then logic will need to be added
#             to find what sort of call to use and dispatch the call in an
#             appropriate manner.
#          2. This call assumes the appropriate JDK is installed with the
#             product so it should be in $ORACLE_HOME.
#          3. $parameter_count not used but may be needed at some future point.
#          4. The design documents specify a -m as an argument (for a manifest
#             file) but I've not seen an XML file with it included and it's
#             not in the packaging tool. The result is it's being ignored for
#             the time being.
#          5. Need to rationalize this and Command::extract_file_names().
#          6. The rollback material is a quick hack. This needs to be done
#             properly.
#
###############################################################################
sub patch_jar_files_on_Windows {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file      = $$rh_arguments{fh_log_file};
    my $fh_rollback      = $$rh_arguments{fh_rollback};
    my $noop_flag        = $$rh_arguments{noop_flag};
    my $Oracle_Home      = $$rh_arguments{Oracle_Home};
    my $patch_id         = $$rh_arguments{patch_id};
    my $patch_location   = $$rh_arguments{patch_location};
    my $rh_actions_list  = $$rh_arguments{rh_actions_list};
    my $jar_files        = $$rh_arguments{jar_files};

    my $error_messages   = "";
    my $parameter_count  = 0;
    my $initial_command  = "";
    my $initial_command_rbf  = "";
    my $rollback_command = "";
    my $system_command   = "";

    opatchIO -> print_message_noverbose ( { f_handle => $fh_log_file,
                                  message => "Patching jar files...\n" } );

    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message => "Patching jar files on Windows.\n" } );

    my $patch_base = File::Spec -> catfile (
                                     "$patch_location", &Apply::PATCH_BASE );

    # Extract the apply command out of the actions list.
    foreach my $command ( keys %{$$rh_actions_list{archive_jar}{apply}} ) {
        $initial_command = $$rh_actions_list{archive_jar}{apply}{jar}{path};

        $initial_command = File::Spec -> catfile
                                         ( "$initial_command" , "$command" );

        $initial_command .= " " . $$rh_actions_list{archive_jar}{apply}
                                                      {$command}{parameters};

        my @array_to_count = split ( /%/, $initial_command );
        $parameter_count = ( scalar (@array_to_count) - 1 );

        # This next regex should leave "jar -uf ".
        ( $initial_command ) = ( $initial_command =~ m#^([^%]+)%# );
        $initial_command = lc ( $initial_command );

        if ( $OSNAME =~ m#Win32# )
        {
            $initial_command_rbf = substr($initial_command, length($Oracle_Home));
        }
        else
        {
            $initial_command_rbf = substr($initial_command, length($Oracle_Home));
        }

        if ( $OSNAME =~ m#Win32# )
        {
            $initial_command_rbf = File::Spec -> catfile (
                                            "\%OH\%",
                                            $initial_command_rbf );
        }
        else
        {
            $initial_command_rbf = File::Spec -> catfile (
                                            "\$OH",
                                            $initial_command_rbf );
        }

    }
    # Apply pre-req. checks has ensured that JDK_LOC points to
    #  an executable jar.exe
    if ( $Command::JDK_LOC ne "" ) {
        opatchIO->print_message ({ message => "  use jar from supplied -jdk option.\n" } );
        $initial_command = "$Command::JDK_LOC -uf ";
        $initial_command_rbf = "$Command::JDK_LOC -uf ";
    }

    # Emergency rollback.
    my ( $copy_command ) = ( $OSNAME =~ m#Win32# ) ? "copy" : "cp" ;
    my ( $del_command )  = ( $OSNAME =~ m#Win32# ) ? "del"  : "rm" ;

    # rbf vars to be used for rollback_ID.sh/cmd
    my $patch_directory_rbf = "";
    my $finished_name_rbf = "";
    if ( $OSNAME =~ m#Win32# )
    {
        $patch_directory_rbf = File::Spec -> catfile
                                         ( "\%OH\%",
                                           $this_class -> save_dir,
                                           $patch_id );
    }
    else
    {
        $patch_directory_rbf = File::Spec -> catfile
                                         ( "\$OH",
                                           $this_class -> save_dir,
                                           $patch_id );
    }

    my $patch_directory   = File::Spec -> catfile
                                         ( "$Oracle_Home",
                                           $this_class -> save_dir,
                                           $patch_id );

    my $curr_dir = "";
    my $finished_name = "";
    my $error_counter = 1;
    my $directory_separator = File::Spec -> catfile ( "", "bin" );
    ( $directory_separator ) = ( $directory_separator =~ m#^(.+)bin$# );
    # Watch out for Windows dir. sep. which is an escape in reg. exp.

    foreach my $file ( keys %$jar_files ) {

        # path will be "$OH\lib\xsu12.jar"
        # target will be "xsu12.jar"
        # class_file will be "oracle/sql/xml/foo.class
        # patch_directory will be "$OH\.patch_storage\<ID>
        my $path = $$jar_files{$file};

        my $path_rbf = "";
        if ( $OSNAME =~ m#Win32# )
        {
            $path_rbf = substr($path, length($Oracle_Home));
        }
        else
        {
            $path_rbf = substr($path, length($Oracle_Home));
        }

        if ( $OSNAME =~ m#Win32# )
        {
            $path_rbf = File::Spec -> catfile (
                                            "\%OH\%",
                                            $path_rbf );
        }
        else
        {
            $path_rbf = File::Spec -> catfile (
                                            "\$OH",
                                            $path_rbf );
        }

        my $target = &File::Basename::basename ( $path );
        my ( $patch_loc ) = ( $file =~ m#(.+)$target# );
        $patch_location = File::Spec -> catfile (
                           "$patch_base", "$patch_loc", "$target" );
        
        # $system_command = "cd $patch_location; $initial_command" . "$path ";
        # the above chain cmd does not work for Windows --> use Perl chdir
        
        $system_command = "$initial_command $path "; 
 
        #my $reg_exp = $target . $directory_separator;
        #$reg_exp =~ s/\\/\\\\/g;
        # Win-specific
        my $reg_exp = $target . "/";

        my ( $class_file ) = ( $file =~ m#$reg_exp(.+)$# );
                my $tmp = ( $file =~ m#(.+)$target# );
        
        # class_file might be Foo$InnerClassName.class, so we need to 'escape'
        #   the $ in class file
        my $inner_class_file = $this_class->getInnerClassName({ class_file => $class_file });
        $system_command .= $inner_class_file;
        
        # Test to see if the file was patched on a prior run.
        $finished_name = File::Spec -> catfile ( "$patch_directory",
                                                 ( $file =~ m#(.+)$target# ),
                                                             "$class_file" );
        $finished_name_rbf = File::Spec -> catfile ( "$patch_directory_rbf",
                                                 ( $file =~ m#(.+)$target# ),
                                                             "$class_file" );
                                                          
        my $original_name = $finished_name;
        my $original_name_rbf = $finished_name_rbf;
        $finished_name = "$finished_name" . "_" . $target . "_" .
                                            &Apply::END_PATCH . "_$patch_id";
        $finished_name_rbf = "$finished_name_rbf" . "_" . $target . "_" .
                                            &Apply::END_PATCH . "_$patch_id";

        my $rollback_name = $original_name . "_" . $target . "_" .
                                            &Apply::PRE_PATCH . "_$patch_id";
        my $rollback_name_rbf = $original_name_rbf . "_" . $target . "_" .
                                            &Apply::PRE_PATCH . "_$patch_id";
        $rollback_name = $this_class->convert_path_separator({
                              path => $rollback_name
                         });
        $original_name = $this_class->convert_path_separator({
                              path => $original_name
                         });
        $rollback_name_rbf = $this_class->convert_path_separator({
                              path => $rollback_name_rbf
                         });
        $original_name_rbf = $this_class->convert_path_separator({
                              path => $original_name_rbf
                         });

        my $rollback_location = File::Spec -> catfile ( $patch_directory,
                                              ( $file =~ m#(.+)$target# ) );  
        my $rollback_location_rbf = File::Spec -> catfile ( $patch_directory_rbf,
                                              ( $file =~ m#(.+)$target# ) );  
        $rollback_location_rbf = $this_class->convert_path_separator({
                              path => $rollback_location_rbf
                         });
                 
        $rollback_location = substr($rollback_location, 0, -1);
        $rollback_location_rbf = substr($rollback_location_rbf, 0, -1);
                 
        # Win-specific    
        # finished_name will be 
        #    "$OH\.patch_storage\<ID>\lib\oracle\sql\xml\foo.class_xsu.jar_done_<ID>
        $finished_name = $this_class->convert_path_separator({
                              path => $finished_name
                         });
        $finished_name_rbf = $this_class->convert_path_separator({
                              path => $finished_name_rbf
                         });

        if ( -e $finished_name ) {
                opatchIO -> print_message ( {f_handle => $fh_log_file,
                                             message => "File patched on previous run, will be skipped : " .
                                                         $finished_name } );
                next;
        }
 
        # Check if the file/directory exists. If not there is something wrong
        # with the system or the patch. Yes, this was tested for earlier.
        if ( ! -e $path ) {
            # If this file is one of the backup-failed files users have OKed,
            #  we can ignore it.
            my $ok = 0;
            foreach my $file ( @Command::FAILED_BACKUP_JAR_FILES ) {
               if ( "$path" eq "$file" ) {
                  opatchIO->print_message({
                     message => "User has chosen to ignore failure of " .
                                "backing up jar file $path. " . 
                                "It is OK to go on.\n"
                  });
                  $ok = 1;
               } 
            } 
            if (! $ok) { 
               $error_messages .= "$error_counter.\t$path\n" .
                               "\t[ No java archive file found. ]\n";
               $error_counter++;
               next;
            } else {
               # skip this file
               next;
            }
        }

        # Make a copy of the archive in case things go horribly wrong.
        # It'll take manual intervention to fix this but its a good safety
        # measure.
        my $backup_target = $path . "_" . &Apply::BACKUP . "_$patch_id";

        if ( ! $noop_flag ) {

            # Write out the emergency rollback details.
            my $message = "$copy_command $rollback_name_rbf $original_name_rbf\n" .
                          "cd $rollback_location_rbf\n" .
                          "$initial_command_rbf $path_rbf $class_file\n" .
                          "$del_command $original_name_rbf\n";
            opatchIO -> print_message ( { f_handle => $fh_rollback,
                                          message  => $message } );

            opatchIO -> print_message ( { f_handle => $fh_log_file,
                                          message  => "  backing up jar file " .
                                          "$path as $backup_target" } );
            File::Copy::copy ( $path, $backup_target ) || (
                ( $error_messages .= "$error_counter.\t$target\n" .
                                     "\t[ Couldn't make backup copy of ".
                                     "file. ]\n"), $error_counter++,
                  next );

            $curr_dir = $this_class->get_cwd();
            opatchIO->print_message ({ f_handle => $fh_log_file,
                                    message  => "  chdir to $patch_location" });
            # So lets patch the file.
            opatchIO -> print_message ( { message => "  updating $path " .
                                                 "using $class_file" } );
            chdir $patch_location;
            my $sys_call_result = qx/$system_command/;
            # jump back to where we were
            chdir $curr_dir;

            chomp $sys_call_result;
            opatchIO -> print_message ( { f_handle => $fh_log_file,
                                          message  => "  " .
                                          $system_command . "\n  Result: " .
                                          "$sys_call_result" } );
            opatchIO->print_message ({ f_handle => $fh_log_file,
                                          message  => "  chdir back to $curr_dir" });
    
            my $status = $this_class -> get_child_error ( {
                      CHILD_ERROR     => $CHILD_ERROR
            } );
            if ( $status ) {
                $error_messages .= "$error_counter.\t$target\n" .
                                   "\t[ object: " .
                                   &File::Basename::basename( $file ) .
                                   " ]\n";
                $error_counter++;
            } else {

                # Now create an empty file to signal this operation was
                # successful.
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                              message  => "  creating " .
                                                       "$finished_name " .
                                                       "to signal successful ops.\n"
                                           } );
                open ( SUCCESS, ">$finished_name" ) || (
                    ( $error_messages .= "$error_counter.\t$finished_name\n" .
                                         "\t[ Non-critical error message: " .
                                         "$OS_ERROR ]\n"),
                      $error_counter++ );
                close SUCCESS;
                unlink $backup_target;
            }
        }
    }

    if ( $error_messages ) {
        $error_messages = "The following java archives on Windows had problems with " .
                          "being patched:\n" . $error_messages;
    }

    return ( $error_messages );

}   # End of patch_jar_files_on_Windows().

###############################################################################
#
# NAME   : prereq_patch_actions 
#
# PURPOSE: Look at all Patch Actions defined in "actions" file.  For now,
#            just warn users if there is Archive Action but there's no
#            Make Action.
#
# INPUTS : $$ARG[1]{fh_log_file}     - The file handle for the log file.
#          $$ARG[1]{fh_rollback}     - The file handle for the rollback
#                                      commands.
#          $$ARG[1]{noop_flag}       - Just report the actions.
#          $$ARG[1]{Oracle_Home}     - The location of the area to be patched.
#          $$ARG[1]{patch_id}        - The patch identification string.
#          $$ARG[1]{patch_location}  - The location of the patch directory.
#          $$ARG[1]{rh_actions_list} - The action require for a type of file.
#          $$ARG[1]{rh_file_details} - A list of files that are the patches.
#          $$ARG[1]{search_OH_jar}   = 1 if no -jdk is used
#                                      0 if -jdk is used
#
# OUTPUTS: $return_messages - A string containing any errors encountered
#                             during backup the process.
#
# NOTES  : 1. This is similar in structure to backup_files_before_patching.
#
###############################################################################
sub prereq_patch_actions  {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file     = $$rh_arguments{fh_log_file};
    my $fh_rollback     = $$rh_arguments{fh_rollback};
    my $noop_flag       = $$rh_arguments{noop_flag};
    my $Oracle_Home     = $$rh_arguments{Oracle_Home};
    my $patch_id        = $$rh_arguments{patch_id};
    my $patch_location  = $$rh_arguments{patch_location};
    my $rh_actions_list = $$rh_arguments{rh_actions_list};
    my $rh_file_details = $$rh_arguments{rh_file_details};
    my $search_OH_jar   = $$rh_arguments{search_OH_jar};

    # Extract the files that need patched by action type.
    my $hasArchiveAction = 0;
    my $hasMakeAction    = 0;
    my $hasCopyAction    = 0;
    my $hasJarAction     = 0;
    my $makeActionCount  = 0;
    
    # looked at rh_file_details to see if there's Archive & Make Actions
    foreach my $action_type ( keys %$rh_file_details ) {
        
        if ( $action_type eq &Apply::AR_STRING ) {
          $hasArchiveAction = 1;
          
        } elsif ( $action_type eq &Apply::JAR_STRING ) {
          $hasJarAction = 1;
          
        } elsif ( $action_type eq &Apply::COPY_STRING ) {
          $hasCopyAction = 1;

        } elsif ( $action_type eq &Apply::MAKE_STRING ) { 
          $makeActionCount = scalar ( @{$$rh_file_details{$action_type}} );
        
        }
    }
    # Due to a bug when we initialize Make hash set, we always have
    #   one entry tagged as "MAKE_STRING" even if there's no make
    #   action in "actions" file.  So we have to rely on the counter.
    #   If counter > 1, then we have Make action.  If counter is 1,
    #   it means there's no Make action.
    if ( $makeActionCount > 0 ) { 
      $hasMakeAction = 1;
    }
    
    # if -no_relink is not used, then see if has archive archive action
    #    but no make action
    if ( $Command::NO_RELINK_MODE == 0 ) {
    if ( $hasArchiveAction == 1 && $hasMakeAction == 0 ) { 
      opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => 
      "-------------------------------------------------------------------------\n" .                          
      "This patch $patch_id had some Archive Actions but there was no " .
      "Make Action.\n" .
       "------------------------------------------------------------------------\n" 
                                 } );
    }
}

    # if -jdk is not used, then see if we have jar action but OH/jdk/bin/jar
    #    is not there
    if ( $hasJarAction == 1 && $search_OH_jar == 1 ) {
      my $jarCheck = $this_class -> hasOracleHomeJdkBinJar( {
                                         Oracle_Home => $Oracle_Home
                                      } );
      if( $jarCheck == 0 ) {
       return  "This patch will update Jar file(s) but " .
               "OPatch was not able to locate executable jar under $Oracle_Home/jdk/bin.";
      }    
    }

}

###############################################################################
#
# NAME   : apply_patches_to_files
#
# PURPOSE: The title says it all.
#
# INPUTS : $$ARG[1]{fh_log_file}     - The file handle for the log file.
#          $$ARG[1]{fh_rollback}     - The file handle for the rollback
#                                      commands.
#          $$ARG[1]{noop_flag}       - Just report the actions.
#          $$ARG[1]{Oracle_Home}     - The location of the area to be patched.
#          $$ARG[1]{patch_id}        - The patch identification string.
#          $$ARG[1]{patch_location}  - The location of the patch directory.
#          $$ARG[1]{rh_actions_list} - The action require for a type of file.
#          $$ARG[1]{rh_file_details} - A list of files that are the patches.
#
# OUTPUTS: $return_messages - A string containing any errors encountered
#                             during backup the process.
#
# NOTES  : 1. This is similar in structure to backup_files_before_patching.
#
###############################################################################
sub apply_patches_to_files {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file     = $$rh_arguments{fh_log_file};
    my $fh_rollback     = $$rh_arguments{fh_rollback};
    my $noop_flag       = $$rh_arguments{noop_flag};
    my $Oracle_Home     = $$rh_arguments{Oracle_Home};
    my $patch_id        = $$rh_arguments{patch_id};
    my $patch_location  = $$rh_arguments{patch_location};
    my $rh_actions_list = $$rh_arguments{rh_actions_list};
    my $rh_file_details = $$rh_arguments{rh_file_details};

    my %archive_files   = ();
    my %jar_files   = ();
    my %copy_files      = ();

    # Extract the files that need patched by action type.
    foreach my $action_type ( keys %$rh_file_details ) {
        if ( $action_type eq &Apply::AR_STRING ) {
            foreach my $object ( keys %{$$rh_file_details{$action_type}} ) {
                my $path   = $$rh_file_details{$action_type}{$object}{path};
                my $target = $$rh_file_details{$action_type}{$object}{target};
                my $comp_name = $$rh_file_details{$action_type}{$object}{component_name};

                $path = File::Spec -> catfile ( "$path", "$target" );
                $archive_files{$object} = $path;
            }

        } elsif ( $action_type eq &Apply::JAR_STRING ) {
            foreach my $object ( keys %{$$rh_file_details{$action_type}} ) {
                my $path   = $$rh_file_details{$action_type}{$object}{path};
                my $target = $$rh_file_details{$action_type}{$object}{object};
                my $comp_name = $$rh_file_details{$action_type}{$object}{component_name};

                $path = File::Spec -> catfile ( "$path", "$target" );
                $jar_files{$object} = $path;
            }

        } elsif ( $action_type eq &Apply::COPY_STRING ) {
            # Note that the key of the hash is not the original
            # file name.  It's been encoded.
            # Ex:
            #   patch is to copy Foo.class to OH/lib
            #   $file below was encoded as OH/lib/Foo.class
            #   it's then decoded to Foo.class
            #   $path will be OH/lib
            #   $patch will be lib/Foo.class

            foreach my $file ( keys %{$$rh_file_details{$action_type}} ) {
                my $path   = $$rh_file_details{$action_type}{$file}{path};
                my $patch  = $$rh_file_details{$action_type}{$file}{object};
                my $comp_name = $$rh_file_details{$action_type}{$file}{component_name};

                # restore the name back to its orignal name
                $file = $this_class->decodeName({
                    encodedName => $file
                });
                $path = File::Spec -> catfile ( "$path", "$file" );
                $copy_files{$path} = $patch;
            }
        }
    }

    # Now call the methods to apply the patch to each file.
    opatchIO->print_message({message=>"Archive Action...\n"});
    my $archive_error_message = $this_class-> patch_archive_files ( {
                                   fh_log_file      => $fh_log_file,
                                   fh_rollback      => $fh_rollback,
                                   noop_flag        => $noop_flag,
                                   Oracle_Home      => $Oracle_Home,
                                   rh_actions_list  => $rh_actions_list,
                                   archive_files    => \%archive_files,
                                   patch_id         => $patch_id,
                                   patch_location   => $patch_location } );

    my $java_archive_error_message = "";
 
    opatchIO->print_message({message=>"Jar Action...\n"});
    if ( $OSNAME =~ m#Win32# ) {
   
        $java_archive_error_message = $this_class-> patch_jar_files_on_Windows ( {
                                   fh_log_file      => $fh_log_file,
                                   fh_rollback      => $fh_rollback,
                                   noop_flag        => $noop_flag,
                                   Oracle_Home      => $Oracle_Home,
                                   rh_actions_list  => $rh_actions_list,
                                   jar_files        => \%jar_files,
                                   patch_id         => $patch_id,
                                   patch_location   => $patch_location } );
    } else {
    
        $java_archive_error_message = $this_class-> patch_jar_files_on_Unix ( {
                                   fh_log_file      => $fh_log_file,
                                   fh_rollback      => $fh_rollback,
                                   noop_flag        => $noop_flag,
                                   Oracle_Home      => $Oracle_Home,
                                   rh_actions_list  => $rh_actions_list,
                                   jar_files        => \%jar_files,
                                   patch_id         => $patch_id,
                                   patch_location   => $patch_location } );
    }
    
    opatchIO->print_message({message=>"Copy Action...\n"});
    my $copy_error_message = $this_class-> patch_copy_files ( {
                                   fh_log_file      => $fh_log_file,
                                   fh_rollback      => $fh_rollback,
                                   noop_flag        => $noop_flag,
                                   Oracle_Home      => $Oracle_Home,
                                   rh_actions_list  => $rh_actions_list,
                                   copy_files       => \%copy_files,
                                   patch_id         => $patch_id,
                                   patch_location   => $patch_location } );

    my $return_messages = $archive_error_message . $copy_error_message .
                                               $java_archive_error_message;

    # Write a message in rollback.sh or .bat, saying the completion.
    opatchIO -> print_message ( { f_handle => $fh_rollback,
                                  message  => "echo Rollback completed." } );


    return ( $return_messages );

}   # End of apply_patches_to_files().

###############################################################################
#
# NAME   : backup_archive_files
#
# PURPOSE: Backup files that reside inside an archive file.
#
# INPUTS : $$ARG[1]{fh_log_file}      - The file handle for the log file.
#          $$ARG[1]{noop_flag}        - Just report the actions.
#          $$ARG[1]{Oracle_Home}      - The location of the area to be patched.
#          $$ARG[1]{patch_id}         - The patch identification string.
#          $$ARG[1]{archive_files} - The hash of files that are to be
#                                       backed-up and which archive to use.
#          $$ARG[1]{rh_actions_list}  - The action require for a type of file.
#
# OUTPUTS: $error_messages - A string containing any errors encountered
#                            during backup the process.
#          add failed files to global FAILED_BACKUP_AR_FILES
#
# NOTES  : 1. The backup command for an archive file is currently only for
#             a system call. If this changes then logic will need to be added
#             to find what sort of call to use and dispatch the call in an
#             appropriate manner.
#          2. System commands are assumed to be available via the user's $PATH.
#          3. $parameter_count not used but may be needed at some future point.
#
###############################################################################
sub backup_archive_files {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file      = $$rh_arguments{fh_log_file};
    my $noop_flag        = $$rh_arguments{noop_flag};
    my $Oracle_Home      = $$rh_arguments{Oracle_Home};
    my $patch_id         = $$rh_arguments{patch_id};
    my $archive_files = $$rh_arguments{archive_files};
    my $rh_actions_list  = $$rh_arguments{rh_actions_list};

    my $error_messages   = "";
    my $parameter_count  = 0;
    my $initial_command  = "";
    my $system_command   = "";

    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                 message => "Backing-up archive files.\n" } );

    # Extract the backup command out of the actions list.
    foreach my $command ( keys %{$$rh_actions_list{archive_lib}{backup}} ) {
        $initial_command = "$command ";
        $initial_command .= $$rh_actions_list{archive_lib}{backup}{$command}
                                                                {parameters};
        my @array_to_count = split ( /%/, $initial_command );
        $parameter_count = ( scalar (@array_to_count) - 1 );
        # This next regex should leave "ar -x " or similar depending on OS.
        ( $initial_command ) = ( $initial_command =~ m#^([^%]+)%# );
    }

    my $patch_directory = File::Spec -> catfile
                                         ( "$Oracle_Home",
                                           $this_class -> save_dir,
                                           $patch_id );
    my $curr_dir = "";
    my $new_name = "";
    my $error_counter = 1;
    foreach my $file ( keys %$archive_files ) {

        my $path = $$archive_files{$file}{path};
        my $target = $$archive_files{$file}{target};
        $target = File::Spec -> catfile ( "$path", "$target" );

        my $backup_path = File::Spec -> catfile ( "$patch_directory",
                                        substr($path, length($Oracle_Home)));

        $system_command = "$initial_command $target ";
        $system_command .= &File::Basename::basename( $file );
        $system_command .= " 2>&1";

        # Test to see if the file has already been extracted on a prior run.
        my $library = File::Basename::basename ( $target );
        $new_name = &File::Basename::basename( $file ) . "_" . $library . "_";
        $new_name = "$new_name" . &Apply::PRE_PATCH . "_$patch_id";
        $new_name = File::Spec -> catfile ( "$backup_path", "$new_name" );
        next if ( -e $new_name );

        # Check if the file/directory exists. If not there is something wrong
        # with the system or the patch.
        if ( ! -e $target ) {
            # This happens when some OH do not have certain files.
            my $failed_file = $target;
            push ( @Command::FAILED_BACKUP_AR_FILES, $failed_file);
 
            $error_messages .= "$error_counter.\t$target\n" .
                               "\t[ No file found. ]\n";
            $error_counter++;
            next;
        }

        if ( ! $noop_flag ) {

            my $make_dir = "";
            # Make sure the backup directory exists first.
            my $backup_dir = &File::Basename::dirname ( $new_name );
            if ( ! -e $backup_dir ) {
                $make_dir =
                              $this_class -> make_directories_recursively ( {
                                              patch_dir => $patch_directory,
                                              path      => $backup_dir } );
            }

            # So lets extract the file, if there have been no errors.
            if ( ! $make_dir ) {

                $curr_dir = $this_class->get_cwd();
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                              message  => "  chdir to $backup_path" });
                chdir $backup_path;
                my ( $sys_call_result ) = qx/$system_command/;
                chdir $curr_dir;

                chomp $sys_call_result;
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                              message  => "System Command:" .
                                              "\n\t$system_command\n" .
                                              "Result:\n" .
                                              "$sys_call_result\n" } );
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                              message  => "  chdir back to $curr_dir" });
                 my $status = $this_class -> get_child_error ( {
                    CHILD_ERROR     => $CHILD_ERROR
                } );
                if ( $status ) {
                    my $ar_file = &File::Basename::basename( $file );
                    # Check for it being a new file.
                    if ( $sys_call_result !~ m#ar: $ar_file not found# ) {
                        $error_messages .= "$error_counter.\t$target\n" .
                                       "\t[ object: " . $ar_file . " ]\n";
                        $error_counter++;
                    } else {
                        # Probably a new file. Create a 0 length file
                        # for rollback.
                        open ( NEW_FILE, ">$new_name" ) || (
                            ( $error_messages .= "$error_counter.\t$target\n" .
                                                "\t[ Couldn't create file: " .
                                                "$new_name. ]\n"),
                              $error_counter++ );
                        close ( NEW_FILE );
                    }
                } else {
                    # Now rename the files so they have the patch ID as part
                    # of the name.
                    my $extracted_name = File::Spec -> catfile (
                                         "$backup_path",
                                         &File::Basename::basename( $file ) );
                    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                                  message  => "  backing up file " .
                                         "$extracted_name as $new_name\n" } );
                    if ( -e $extracted_name ) {
                        rename ( $extracted_name, $new_name ) || (
                            ( $error_messages .= "$error_counter.\t$target\n" .
                                   "\t[ object: " . $extracted_name . " ]\n"),
                          $error_counter++ );
                    }
		    else { # creating a 0byte file <filename>_<archive name>_opatch_new_<patchid> in backup.
		        my $new_file_name = "";
			$new_file_name = "$extracted_name" . "_" . $library ."_" .
						&Apply::OPATCH_NEW_FILE . "_$patch_id";
                        opatchIO -> debug ( {
                                    message => "  creating \"$new_file_name\" to indicate it as a new file."
                                    } );
			
			open (NFH, ">$new_file_name") or 
					opatchIO -> print_message_noverbose ({
						message => "   Couldn't create $new_file_name in backup area" });
			
			close(NFH);
		    }
                }
            }
        }
    }

    if ( $error_messages ) {
        $error_messages = "The following problems were encountered " .
                          "when backing-up the static archive files:\n" .
                          $error_messages;
    }

    return ( $error_messages );

}   # End of backup_archive_files().

###############################################################################
#
# NAME   : backup_copy_files
#
# PURPOSE: Backup files that are to be copied into place.
#
# INPUTS : $$ARG[1]{fh_log_file}      - The file handle for the log file.
#          $$ARG[1]{noop_flag}        - Just report the actions.
#          $$ARG[1]{Oracle_Home}      - The location of the area to be patched.
#          $$ARG[1]{patch_id}         - The patch identification string.
#          $$ARG[1]{copy_files}       -
#          $$ARG[1]{rh_actions_list}  - The action require for a type of file.
#                                       backed-up.
#
# OUTPUTS: $error_messages - A string containing any errors encountered
#                            during backup the process.
#
# NOTES  : 1. The backup command for a file copy is currently only for a
#             perl call. If this changes then logic will need to be added
#             to find what sort of call to use and dispatch the call in an
#             appropriate manner.
#          2. $parameter_count not used but may be needed at some future point.
#
###############################################################################
sub backup_copy_files {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file      = $$rh_arguments{fh_log_file};
    my $noop_flag        = $$rh_arguments{noop_flag};
    my $Oracle_Home      = $$rh_arguments{Oracle_Home};
    my $patch_id         = $$rh_arguments{patch_id};
    my $copy_files       = $$rh_arguments{copy_files};
    my $rh_actions_list  = $$rh_arguments{rh_actions_list};

    my $error_messages   = "";
    my $parameter_count  = 0;
    my $backup_command   = "";
    my $required_module  = "";

    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message => "Backing-up copy files.\n" } );

    # Extract the backup command out of the actions list.
    foreach my $command ( keys %{$$rh_actions_list{copy}{backup}} ) {
        $backup_command = "$command ";
        $parameter_count = $$rh_actions_list{copy}{backup}{$command}
                                                                 {param_num};
        $required_module = $$rh_actions_list{copy}{backup}{$command}{module};
        $required_module .= "()";
    }

    if ( $OSNAME =~ m#Win32# ) {
         $Oracle_Home =~ s/\\/\\\\/g;
    }

    # Have to hard code it. Due to binding issues and variable interpolation
    # "do", "require", and "use" doesn't find what's in $required_module
    # or doesn't find the module (depending on usage). So it's hardwired as
    # a bare word.

    my $patch_directory = File::Spec -> catfile
                                         ( "$Oracle_Home",
                                           $this_class -> save_dir,
                                           $patch_id );
    my $new_name = "";
    my $error_counter = 1;

    foreach my $file ( @$copy_files ) {

        if ( $OSNAME =~ m#Win32# ) {
             $file = lc ( $file );
        }

        # Test to see if the file has already been extracted on a prior run.
        ( $new_name ) = ( $file =~ m#$Oracle_Home(.+)# );
        $new_name = File::Spec -> catfile ( "$patch_directory", "$new_name" );
        $new_name = "$new_name" . "_". &Apply::PRE_PATCH . "_$patch_id";

        # Make sure the backup directory exists first.
        my $backup_dir = &File::Basename::dirname ( $new_name );
        if ( ! -e $backup_dir ) {
            $error_messages = $this_class -> make_directories_recursively ( {
                                                patch_dir => $patch_directory,
                                                path      => $backup_dir } );
        }

        # See if the original file exists. If its a new file a backup can't
        # be made. Need to make a 0 length file to allow rollback to happen.

        # Commenting this, to create a 0 byte file <filname>_opatch_new_<patchid>
        # next if ( ! -e $file );

	# Creating 0 byte file in .patch_storage, to indicate it as new file.
        if( ! -e $file)
        {
           ( $new_name ) = ( $file =~ m#$Oracle_Home(.+)# );
           $new_name = File::Spec -> catfile ( "$patch_directory", "$new_name" );
           $new_name = "$new_name" . "_". &Apply::OPATCH_NEW_FILE . "_$patch_id";

           if ( $OSNAME =~ m#Win32# ) {
               $new_name =~ s/\\\\/\\/g;
           }

           opatchIO -> debug ( {
                       message => "  creating \"$new_name\" to indicate it as a new file."
                       } );

           open(NFH, ">$new_name") or  die "Couldn't create file in backup area";
           close(NFH);

           next;
        }

        # Microsoft OSs choke on duplicate directory separators. This is
        # generated by the catfile call.
        if ( $OSNAME =~ m#Win32# ) {
             $new_name =~ s/\\\\/\\/g;
        }

        next if ( -e $new_name );

        # Check if the file/directory exists. If not there is something wrong
        # with the system or the patch.
        if ( ! -e $file ) {
            $error_messages .= "$error_counter.\t$file\n";
            $error_counter++;
            next;
        }

        # Now copy the file to the new name.
        if ( ( ! $noop_flag ) && ( ! $error_messages ) ) {

            # If there have been no errors we can finally copy the file.
            opatchIO -> print_message ( { f_handle => $fh_log_file,
                                          message => "  backing-up file " .
                                          "$file as $new_name\n" } );
            File::Copy::copy ( $file, $new_name) || (
                ( $error_messages .= "$error_counter.\t$file\n" ),
                                                  $error_counter++ );
        }
    }

    if ( $error_messages ) {
        $error_messages = "The following problems were encountered " .
                          "when backing-up files for copying:\n" .
                          $error_messages;
    }

    return ( $error_messages );

}   # End of backup_copy_files().

###############################################################################
#
# NAME   : backup_jar_files_on_Unix
#
# PURPOSE: Backup files that reside inside a Java archive file.
#
# INPUTS : $$ARG[1]{fh_log_file}      - The file handle for the log file.
#          $$ARG[1]{noop_flag}        - Just report the actions.
#          $$ARG[1]{Oracle_Home}      - The location of the area to be patched.
#          $$ARG[1]{rh_actions_list}  - The action require for a type of file.
#          $$ARG[1]{jar_files}        - The hash of files that are to be
#                                       backed-up and which archive to use.
#          $$ARG[1]{patch_id}         - The patch identification string.
#
# OUTPUTS: $error_messages - A string containing any errors encountered
#                            during backup the process.
#          add failed files to global FAILED_BACKUP_JAR_FILES
#
# NOTES  : 1. The backup command for a Java archive file is currently only for
#             a system call. If this changes then logic will need to be added
#             to find what sort of call to use and dispatch the call in an
#             appropriate manner.
#          2. This call assumes the appropriate JDK is installed with the
#             product so it should be in $ORACLE_HOME.
#          3. $parameter_count not used but may be needed at some future point.
#
###############################################################################
sub backup_jar_files_on_Unix {
    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file      = $$rh_arguments{fh_log_file};
    my $noop_flag        = $$rh_arguments{noop_flag};
    my $Oracle_Home      = $$rh_arguments{Oracle_Home};
    my $patch_id         = $$rh_arguments{patch_id};
    my $jar_files        = $$rh_arguments{jar_files};
    my $rh_actions_list  = $$rh_arguments{rh_actions_list};

    my $error_messages   = "";
    my $parameter_count  = 0;
    my $initial_command  = "";
    my $system_command   = "";

    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message => "Backing-up jar files.\n" } );

    # Extract the backup command out of the actions list.
    foreach my $command ( keys %{$$rh_actions_list{archive_jar}{backup}} ) {
        $initial_command = $$rh_actions_list{archive_jar}{backup}{jar}{path};
        $initial_command = File::Spec -> catfile
                                         ( "$initial_command" , "$command" );
        $initial_command .= " " . $$rh_actions_list{archive_jar}{backup}
                                                      {$command}{parameters};

        my @array_to_count = split ( /%/, $initial_command );
        $parameter_count = ( scalar (@array_to_count) - 1 );

        # This next regex should leave "$ORACLE_HOME/jdk/bin/jar -xf ".
        ( $initial_command ) = ( $initial_command =~ m#^([^%]+)%# );
    }
    # Apply pre-req. checks has ensured that JDK_LOC points to
    #  an executable jar.exe
    if ( $Command::JDK_LOC ne "" ) {
        opatchIO->print_message ({ message => "backup_jar_files(): use jar from supplied -jdk option.\n" } );
        $initial_command = "$Command::JDK_LOC -xf ";
    }

    my $patch_directory = File::Spec -> catfile
                                         ( "$Oracle_Home",
                                           $this_class -> save_dir,
                                           $patch_id );

    my $curr_dir = "";
    my $new_name = "";
    my $error_counter = 1;
    my $directory_separator = File::Spec -> catfile ( "", "bin" );
    ( $directory_separator ) = ( $directory_separator =~ m#^(.+)bin$# );

    # watch out for dir. sep. on Windows which is an escape in reg. exp.

    foreach my $file ( keys %$jar_files ) {

        my $path = $$jar_files{$file}{path};
        my $target = $$jar_files{$file}{target};

        my $reg_exp1 = $target . $directory_separator;
        $reg_exp1  =~ s/\\/\\\\/g;

        my ( $jar_file ) = ( $file =~ m#$reg_exp1(.+)$# );
	
	      # class_file might be Foo$InnerClassName.class, so we need to 'escape'
        #   the $ in class file
        my $inner_jar_file = $this_class->getInnerClassName({ class_file => $jar_file });
	
        my $reg_exp2 = $Oracle_Home . $directory_separator;
        $reg_exp2 =~ s/\\/\\\\/g;

        my ( $oracle_path ) =
                      ( $path =~ m#$reg_exp2(.+)$# );

        my $backup_path = File::Spec -> catfile ( "$patch_directory",
                                         "$oracle_path",
                                          "$jar_file" );
        my $backup_dir = File::Basename::dirname ( $backup_path );

        # Make sure the backup directory exists.
        if ( ! -e $backup_dir ) {
            $error_messages = $this_class -> make_directories_recursively ( {
                                              patch_dir => $patch_directory,
                                              path      => $backup_dir } );
        }

        # Now reset it to where the archive will start from.
        $backup_dir = File::Spec -> catfile ( "$patch_directory",
                                                 "$oracle_path" );

        $system_command  = "$initial_command";
        $system_command .= File::Spec -> catfile ( "$path", "$target" );
        $system_command .= " " . $inner_jar_file . " 2>&1";

        # Test to see if the file has already been extracted on a prior run.
        $new_name = $backup_path;
        $new_name = "$new_name" . "_" . $target . "_" .
                                           &Apply::PRE_PATCH . "_$patch_id";
        next if ( -e $new_name );

        # Check if the file/directory exists. If not there is something wrong
        # with the system or the patch.
        my $check_existence = File::Spec -> catfile ( "$path", "$target" );
        if ( ! -e $check_existence ) {
            # This happens when some OH do not have certain files, i.e.
            #   thin JDBC
            my $failed_file = $check_existence;
            push ( @Command::FAILED_BACKUP_JAR_FILES, $failed_file);

            $error_messages .= "$error_counter.\t$check_existence\n" .
                               "\t[ No file found. ]\n";
            $error_counter++;
            next;
        }

        if ( ! $noop_flag )  {

            # So lets extract the file, if there have been no errors.
                $curr_dir = cwd(); 

                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                              message  => "  chdir to $backup_dir" });
                chdir $backup_dir; 
                my ( $sys_call_result ) = qx/$system_command/;
                chdir $curr_dir;
 
                chomp $sys_call_result;
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                              message  => "  " .
                                              "$system_command\n" .
                                              "  Result:" .
                                              "$sys_call_result" } );
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                              message  => "  chdir back to $curr_dir" });

                # Not much point. jar returns 0 for most actions.
                my $status = $this_class -> get_child_error ( {
                    CHILD_ERROR     => $CHILD_ERROR
                } );

                if ( $status ) {
                    $error_messages .= "$error_counter.\t$check_existence\n" .
                                       "\t[ object: " .
                                       &File::Basename::basename( $file ) .
                                       " ]\n";
                    $error_counter++;

                } else {
                    # Now rename the files so they have the patch ID as part
                    # of the name. For Java archive files also add the archive
                    # they came from. Several archives have the same class
                    # with different details (it's valid).

                    my $extracted_file = File::Spec -> catfile (
                                         "$patch_directory",
                                         "$oracle_path",
                                         "$jar_file" );

                    # If there is no file extracted it's a new file.
                    if ( -e $extracted_file ) {

                        opatchIO -> print_message ( {
                                               f_handle => $fh_log_file,
                                               message  => "  re-naming file " .
                                         "$extracted_file as $new_name\n" } );

                        rename ( $backup_path, $new_name ) || (
                            ( $error_messages .=
                                       "$error_counter.\t$check_existence\n" .
                                       "\t[ object: " . $jar_file . " ]\n"),
                              $error_counter++ );
                    } else {
                        # Probably a new file. Create a 0 length file
                        # for rollback.
                        open ( NEW_FILE, ">$new_name" ) || (
                            ( $error_messages .= "$error_counter.\t$target\n" .
                                                "\t[ Couldn't create file: " .
                                                "$new_name. ]\n"),
                              $error_counter++ );

                    }
            }
        }
    }

    if ( $error_messages ) {
        $error_messages = "The following problems were encountered when " .
                          "trying to backup Java class files:\n" .
                          $error_messages;
    }

    return ( $error_messages );
}  # End of backup_jar_files_on_Unix()

################################################################################
#
# NAME   : backup_jar_files_on_Windows
#
# PURPOSE: Backup files that reside inside a Java archive file.
#
# INPUTS : $$ARG[1]{fh_log_file}      - The file handle for the log file.
#          $$ARG[1]{noop_flag}        - Just report the actions.
#          $$ARG[1]{Oracle_Home}      - The location of the area to be patched.
#          $$ARG[1]{rh_actions_list}  - The action require for a type of file.
#          $$ARG[1]{jar_files}        - The hash of files that are to be
#                                       backed-up and which archive to use.
#          $$ARG[1]{patch_id}         - The patch identification string.
#
# OUTPUTS: $error_messages - A string containing any errors encountered
#                            during backup the process.
#          add failed files to global FAILED_BACKUP_JAR_FILES
#
# NOTES  : 1. The backup command for a Java archive file is currently only for
#             a system call. If this changes then logic will need to be added
#             to find what sort of call to use and dispatch the call in an
#             appropriate manner.
#          2. This call assumes the appropriate JDK is installed with the
#             product so it should be in $ORACLE_HOME.
#          3. $parameter_count not used but may be needed at some future point.
#
###############################################################################
sub backup_jar_files_on_Windows {
    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file      = $$rh_arguments{fh_log_file};
    my $noop_flag        = $$rh_arguments{noop_flag};
    my $Oracle_Home      = $$rh_arguments{Oracle_Home};
    my $patch_id         = $$rh_arguments{patch_id};
    my $jar_files        = $$rh_arguments{jar_files};
    my $rh_actions_list  = $$rh_arguments{rh_actions_list};

    my $error_messages   = "";
    my $parameter_count  = 0;
    my $initial_command  = "";
    my $system_command   = "";

    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message => "Backing-up jar files on Windows.\n" } );

    # Extract the backup command out of the actions list.
    foreach my $command ( keys %{$$rh_actions_list{archive_jar}{backup}} ) {
        $initial_command = $$rh_actions_list{archive_jar}{backup}{jar}{path};
        $initial_command = File::Spec -> catfile
                                         ( "$initial_command" , "$command" );
        $initial_command .= " " . $$rh_actions_list{archive_jar}{backup}
                                                      {$command}{parameters};

        my @array_to_count = split ( /%/, $initial_command );
        $parameter_count = ( scalar (@array_to_count) - 1 );

        # This next regex should leave "$ORACLE_HOME/jdk/bin/jar -xf ".
        ( $initial_command ) = ( $initial_command =~ m#^([^%]+)%# );
    }
    # Apply pre-req. checks has ensured that JDK_LOC points to
    #  an executable jar.exe
    if ( $Command::JDK_LOC ne "" ) {
        opatchIO->print_message ({ message => "backup_jar_files(): use jar from supplied -jdk option.\n" } );
        $initial_command = "$Command::JDK_LOC -xf ";
    }

    my $patch_directory = File::Spec -> catfile
                                         ( "$Oracle_Home",
                                           $this_class -> save_dir,
                                           $patch_id );

    my $curr_dir = "";
    my $new_name = "";
    my $error_counter = 1;
    my $directory_separator = File::Spec -> catfile ( "", "bin" );
    ( $directory_separator ) = ( $directory_separator =~ m#^(.+)bin$# );

    # watch out for dir. sep. on Windows which is an escape in reg. exp.

    foreach my $file ( keys %$jar_files ) {

        # path will be "$OH\lib"
        # target will be "xsu12.jar"
        # jar_file will be "oracle/sql/xml/foo.class"
        # oracle_path will be "lib"
        #
        my $path = $$jar_files{$file}{path};
        $path = $this_class->convert_path_separator({path=>$path});
        my $target = $$jar_files{$file}{target};
        
        my $reg_exp1 = $target . "/";
        my ( $jar_file ) = ( $file =~ m#$reg_exp1(.+)$# );

        # class_file might be Foo$InnerClassName.class, so we need to 'escape'
        #   the $ in class file
        my $inner_jar_file = $this_class->getInnerClassName({ class_file => $jar_file });

        opatchIO->print_message({message=>"  extract $jar_file from $target to back up"});
        
        my $reg_exp2 = $Oracle_Home . $directory_separator;
        $reg_exp2 =~ s/\\/\\\\/g;
        my ( $oracle_path ) = ( $path =~ m#$reg_exp2(.+)$# );

        my $backup_path = "$patch_directory" . "/" . "$oracle_path" . "/" . "$jar_file";
        my $backup_path = $this_class->convert_path_separator ({ path => $backup_path });
        my $backup_dir = File::Basename::dirname ( $backup_path );

        # Make sure the backup directory exists.
        if ( ! -e $backup_dir ) {
            $error_messages = $this_class -> make_directories_recursively ( {
                                              patch_dir => $patch_directory,
                                              path      => $backup_dir } );
        }

        # Now reset it to where the archive will start from.
        $backup_dir = File::Spec -> catfile ( "$patch_directory",
                                                 "$oracle_path" );

        # system_command will be "$path\jar -xf "
        $system_command  = "$initial_command";   
        # then "$path\jar -xf $OH\lib\xsu12.jar"
        $system_command .= File::Spec -> catfile ( "$path", "$target" );
        # then "$path\jar -xf $OH\lib\xsu12.jar oracle/sql/xml/foo.class 2>&1"
        $system_command .= " " . $inner_jar_file . " 2>&1";
        

        # $new_name will be $OH\.patch_storage\<id>\oracle\sql\xml\Foo.class_xsu12.jar_pre_8801
        # $backup_path is "$OH\.patch_storage\<id>\oracle\sql\xml\Foo.class"
        #
        $new_name = "$backup_path" . "_" . $target . "_" .
                                           &Apply::PRE_PATCH . "_$patch_id";

        # Skip if the file has already been extracted on a prior run. Otherwise, 
        #  we will run
        #    "jar -xf $OH\lib\xsu12.jar oracle/sql/xml/foo.class 2>&1"
        #  then rename "$OH\.patch_storage\<id>\oracle\sql\xml\Foo.class" to
        #              "$OH\.patch_storage\<id>\oracle\sql\xml\Foo.class_xsu12.jar_pre_8801"
        next if ( -e $new_name );

        opatchIO->print_message({message=> "  run jar -xf... then save extracted class file as $new_name"});
        
        # Check if the file/directory exists. If not there is something wrong
        # with the system or the patch.
        my $check_existence = File::Spec -> catfile ( "$path", "$target" );
        if ( ! -e $check_existence ) {
            # This happens when some OH do not have certain files, i.e.
            #   thin JDBC
            my $failed_file = $check_existence;
            push ( @Command::FAILED_BACKUP_JAR_FILES, $failed_file);

            $error_messages .= "$error_counter.\t$check_existence\n" .
                               "\t[ No file found. ]\n";
            $error_counter++;
            next;
        }

        if ( ! $noop_flag )  {

            # So lets extract the file, if there have been no errors.
                $curr_dir = $this_class->get_cwd(); 

                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                              message  => "  chdir to $backup_dir" });
                chdir $backup_dir; 
                my ( $sys_call_result ) = qx/$system_command/;
                chdir $curr_dir;
 
                chomp $sys_call_result;
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                              message  => "  " .
                                              "$system_command\n" .
                                              "  Result: " .
                                              "$sys_call_result" } );
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                              message  => "  chdir back to $curr_dir" });

                # Not much point. jar returns 0 for most actions.
                my $status = $this_class -> get_child_error ( {
                    CHILD_ERROR     => $CHILD_ERROR
                } );

                if ( $status ) {
                    $error_messages .= "$error_counter.\t$check_existence\n" .
                                       "\t[ object: " .
                                       &File::Basename::basename( $file ) .
                                       " ]\n";
                    $error_counter++;

                } else {
                    # Now rename the files so they have the patch ID as part
                    # of the name. For Java archive files also add the archive
                    # they came from. Several archives have the same class
                    # with different details (it's valid).

                    my $extracted_file = $backup_path;


                    # If there is no file extracted it's a new file.
                    if ( -e $extracted_file ) {

                        opatchIO -> print_message ( {
                                               f_handle => $fh_log_file,
                                               message  => "  re-name " .
                                         "$extracted_file to $new_name\n" } );
                        
                        rename ( $backup_path, $new_name ) || (
                            ( $error_messages .=
                                       "$error_counter.\t$check_existence\n" .
                                       "\t[ object: " . $jar_file . " ]\n"),
                              $error_counter++ );

                    } else {
                        # Probably a new file. Create a 0 length file
                        # for rollback.
                        open ( NEW_FILE, ">$new_name" ) || (
                            ( $error_messages .= "$error_counter.\t$target\n" .
                                                "\t[ Couldn't create file: " .
                                                "$new_name. ]\n"),
                              $error_counter++ );

                    }
            }
        }
    }

    if ( $error_messages ) {
        $error_messages = "The following problems were encountered when " .
                          "trying to backup Java class files:\n" .
                          $error_messages;
    }

    return ( $error_messages );

}  # End of backup_jar_files_on_Windows().

###############################################################################
#
# NAME   : backup_files_before_patching
#
# PURPOSE: The title says it all.
#
# INPUTS : $$ARG[1]{fh_log_file}     - The file handle for the log file.
#          $$ARG[1]{noop_flag}       - Just report the actions.
#          $$ARG[1]{Oracle_Home}     - The location of the area to be patched.
#          $$ARG[1]{patch_id}        - The patch identification string.
#          $$ARG[1]{rh_actions_list} - The action require for a type of file.
#          $$ARG[1]{rh_file_details} - A list of files that are the patches.
#
# OUTPUTS: $return_messages - A string containing any errors encountered
#                             during backup the process.
#
# NOTES  : 1. For backing-out a patch the inventory should be queried. This
#             will new files that need to be removed. In theory these new
#             files can stay 'in situ' but it's much cleaner to remove them.
#             Also creating a zero lenth file as an identifier for a new file
#             is a bad idea as one of the groups may have a zero lenth file
#             for some reason.
#
###############################################################################
sub backup_files_before_patching {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file     = $$rh_arguments{fh_log_file};
    my $noop_flag       = $$rh_arguments{noop_flag};
    my $Oracle_Home     = $$rh_arguments{Oracle_Home};
    my $patch_id        = $$rh_arguments{patch_id};
    my $rh_actions_list = $$rh_arguments{rh_actions_list};
    my $rh_file_details = $$rh_arguments{rh_file_details};

    my %archive_files   = ();
    my %jar_files   = ();
    my @copy_files      = ();

    # Extract the files that need backing-up by action type.
    foreach my $action_type ( keys %$rh_file_details ) {
        if ( $action_type eq &Apply::AR_STRING ) {
            foreach my $object ( keys %{$$rh_file_details{$action_type}} ) {
                my $path   = $$rh_file_details{$action_type}{$object}{path};
                my $target = $$rh_file_details{$action_type}{$object}{target};
                my $comp_name = $$rh_file_details{$action_type}{$object}{component_name};

                $archive_files{$object}{path} = $path;
                $archive_files{$object}{target} = $target;
            }

        } elsif ( $action_type eq &Apply::JAR_STRING ) {
            foreach my $object ( keys %{$$rh_file_details{$action_type}} ) {
                my $path   = $$rh_file_details{$action_type}{$object}{path};
                my $target = $$rh_file_details{$action_type}{$object}{object};
                my $comp_name = $$rh_file_details{$action_type}{$object}{component_name};

                $jar_files{$object}{path} = $path;
                $jar_files{$object}{target} = $target;
            }

        } elsif ( $action_type eq &Apply::COPY_STRING ) {
            foreach my $file ( keys %{$$rh_file_details{$action_type}} ) {
                my $path   = $$rh_file_details{$action_type}{$file}{path};
                my $target = $$rh_file_details{$action_type}{$file}{object};
                my $comp_name = $$rh_file_details{$action_type}{$file}{component_name};

                $path = File::Spec -> catfile (
                              "$path", &File::Basename::basename( $target ) );
                push ( @copy_files, $path );
            }
        }
    }

    # Now call the methods to back-up the files for each action type.
    # (failed files will be saved to FAILED_BACKUP_AR_FILES)
    my $archive_error_message = $this_class-> backup_archive_files ( {
                                   fh_log_file      => $fh_log_file,
                                   noop_flag        => $noop_flag,
                                   Oracle_Home      => $Oracle_Home,
                                   rh_actions_list  => $rh_actions_list,
                                   archive_files    => \%archive_files,
                                   patch_id         => $patch_id } );

    # (failed files will be saved to FAILED_BACKUP_JAR_FILES)
    my $java_archive_error_message = "";
    
    if ( $OSNAME =~ m#Win32# ) {
        $java_archive_error_message = $this_class-> backup_jar_files_on_Windows ( {
                                   fh_log_file      => $fh_log_file,
                                   noop_flag        => $noop_flag,
                                   Oracle_Home      => $Oracle_Home,
                                   rh_actions_list  => $rh_actions_list,
                                   jar_files        => \%jar_files,
                                   patch_id         => $patch_id } );
    } else {
        $java_archive_error_message = $this_class-> backup_jar_files_on_Unix ( {
                                   fh_log_file      => $fh_log_file,
                                   noop_flag        => $noop_flag,
                                   Oracle_Home      => $Oracle_Home,
                                   rh_actions_list  => $rh_actions_list,
                                   jar_files        => \%jar_files,
                                   patch_id         => $patch_id } );
    }
    # (failed files will be saved to FAILED_BACKUP_COPY_FILES)
    my $copy_error_message = $this_class-> backup_copy_files ( {
                                   fh_log_file      => $fh_log_file,
                                   noop_flag        => $noop_flag,
                                   Oracle_Home      => $Oracle_Home,
                                   rh_actions_list  => $rh_actions_list,
                                   copy_files => \@copy_files,
                                   patch_id         => $patch_id } );

    my $return_messages = "";
    if ( $archive_error_message ) {
        $return_messages .= "\n" . $archive_error_message . "\n";
    }
    if ( $java_archive_error_message ) {
        $return_messages .= "\n" . $java_archive_error_message . "\n";
    }
    if ( $copy_error_message ) {
        $return_messages .= "\n" . $copy_error_message . "\n";
    }

    return ( $return_messages );

}   # End of backup_files_before_patching().


###############################################################################
#
# NAME   : check_patch_area_validity
#
# PURPOSE: Check that the patch area is a patch area.
#
# INPUTS : $$ARG[1]{p_i_l} - The location to check for patches.
#
# OUTPUTS: $return_values{patch_inventory_location} -
#                         The string passed in as $$ARG{p_i_l}
#          empty string if patch area is invalid
#
# NOTES  :
#             patch_area
#                     |-------etc
#                     |       |-------config
#                     |       |       |-------actions
#                     |       |       `-------inventory
#                     |       `-------xml
#                     |               |-------GenericActions.xml
#                     |               `-------ShiphomeDirectoryStructure.xml
#                     |-------files
#                     `-------README.txt
#
#
#
###############################################################################
sub check_patch_area_validity {

    # Because this was called in an OO manner the first argument is the
    # class name.
    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    # $rh_required_files has four entries. Add two for the "bare files".
    use constant MAX_CHECKS => 6;

    my $patch_inventory_location  = $$rh_arguments{p_i_l};
    my $rh_required_files         = $$rh_arguments{rh_required_files};
    my $check_count = 0;

    my $actions_file  =  $$rh_required_files{actions};
    my $inventory_file = $$rh_required_files{inventory};
    my $GenericActions_file = $$rh_required_files{GenericActions};
    my $ShiphomeDirectoryStructure_file =
                             $$rh_required_files{ShiphomeDirectoryStructure};

    my $Readme_file      = "README.txt";
    my $files_directory  = "files";
    my $tmp_patch_inventory_location = $patch_inventory_location;

    if ( ! -e File::Spec -> catfile( "$patch_inventory_location",
                                   "$actions_file" ) ) {
        $tmp_patch_inventory_location = "";
    }

    if( ! -e File::Spec -> catfile( "$patch_inventory_location",
                                   "$inventory_file" ) ) {
        $tmp_patch_inventory_location = "";
    }

    if ( ! -e File::Spec -> catfile( "$patch_inventory_location",
                                   "$GenericActions_file" ) ) {
        $tmp_patch_inventory_location = "";
    }

    if ( ! -e File::Spec -> catfile( "$patch_inventory_location",
                                   "$ShiphomeDirectoryStructure_file" ) ) {
        $tmp_patch_inventory_location = "";
    }

    if ( ! -e File::Spec -> catfile( "$patch_inventory_location",
                                   "$Readme_file" ) ) {
        $tmp_patch_inventory_location = "";
    }

    if ( ! -e File::Spec -> catfile( "$patch_inventory_location",
                                   "$files_directory" ) ) {
        $tmp_patch_inventory_location = "";
    }

    my %return_values = ();
    $return_values{patch_inventory_location} = $tmp_patch_inventory_location;

    # Now check the directory starts with digits.
    my $test_name = &File::Basename::basename ( $patch_inventory_location );
    if ( $test_name !~ m#^(\d+_?)# ) {
        $tmp_patch_inventory_location = "";
        $return_values{patch_inventory_location} = $tmp_patch_inventory_location;
    }

    return ( \%return_values );

}   # End of check_patch_area_validity().

###############################################################################
#
# NAME   : clean_up_after_patching
#
# PURPOSE: Remove work files from the patch process.
#
# INPUTS : $$ARG[1]{Oracle_Home}      - The location of the area to be patched.
#          $$ARG[1]{rh_actions_list} - The action require for a type of file.
#          $$ARG[1]{patch_id}        - The patch identification string.
#          $$ARG[1]{rh_file_details} - A list of files that are the patches.
#
# OUTPUTS: $return_messages - A string containing any errors encountered
#                             during cleaning the process.
#
# NOTES  : 1. Currently only removes the <file>_done_<patch_id> files.
#
###############################################################################
sub clean_up_after_patching {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $Oracle_Home     = $$rh_arguments{Oracle_Home};
    my $rh_actions_list = $$rh_arguments{rh_actions_list};
    my $patch_id        = $$rh_arguments{patch_id};
    my $rh_file_details = $$rh_arguments{rh_file_details};

    my $error_message = "";
    my @file_list     = ();

    my $patch_directory   = File::Spec -> catfile
                                         ( "$Oracle_Home",
                                           $this_class -> save_dir,
                                           $patch_id );

    # Extract the files that need backing-up by action type.
    foreach my $action_type ( keys %$rh_file_details ) {
        if ( $action_type eq &Apply::AR_STRING ) {
            foreach my $object ( keys %{$$rh_file_details{$action_type}} ) {

                my $path   = $$rh_file_details{$action_type}{$object}{path};
                my $target = $$rh_file_details{$action_type}{$object}{target};
                $object = &File::Basename::basename( $object );
                $path = substr($path, length($Oracle_Home));

                my $finished_name = File::Spec->catfile ( "$patch_directory",
                                                          "$path", $object );
                $finished_name .=  "_". $target . "_" . &Apply::END_PATCH .
                                                                 "_$patch_id";
                push ( @file_list, $finished_name );
            }

        } elsif ( $action_type eq &Apply::JAR_STRING ) {
            foreach my $object ( keys %{$$rh_file_details{$action_type}} ) {
                my $target = $$rh_file_details{$action_type}{$object}{object};

                #( $object ) = ( $object =~ m#^(.+)$target(.+)$# );
                my $path = "";
                ( $path, $object ) = ( $object =~ m#^(.+)$target(.+)$# );

                my $finished_name = File::Spec->catfile ( "$patch_directory",
                                                          "$path", $object );
                $finished_name .=  "_" . $target . "_" . &Apply::END_PATCH .
                                                             "_" . $patch_id;
                push ( @file_list, $finished_name );
            }


        } elsif ( $action_type eq &Apply::COPY_STRING ) {
            foreach my $object ( keys %{$$rh_file_details{$action_type}} ) {
                my $path   = $$rh_file_details{$action_type}{$object}{path};
                my $target = $$rh_file_details{$action_type}{$object}{object};
                my $finished_name = File::Spec -> catfile ( "$patch_directory",
                                                            "$target" );
                $finished_name .= "_" . &Apply::END_PATCH . "_" . $patch_id;
                push ( @file_list, $finished_name );

            }
        }
    }

    foreach my $file ( @file_list ) {
        # if a file was not backed up and users chose to go on,
        #   we will generate error messages for that file here. 
        # Improve later: check and ignore that file  
        my $count = unlink ( $file );
        if ( $count != 1 ) {
            $error_message .= $file . "\n";
        }
    }

    if ( $error_message ) {
        $error_message = "The following work files couldn't be removed:\n" .
                         $error_message;
    }

    return ( $error_message );

}   # End of clean_up_after_patching().

###############################################################################
#
# NAME   : command_details
#
# PURPOSE: Provide the details about the command.
#
# INPUTS : $$ARG[1]{rh_detail} - A reference to a hash structure of the class
#                                Command to store the attributes in.
#
# OUTPUTS: NONE
#
# NOTES  : 1. The return values are in hash so the caller will get the new
#             details when this subroutine is finished.
#
###############################################################################
sub command_details {

    # Because this was called in an OO manner the first argument is the
    # class name.
    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_detail    = $$rh_arguments{rh_detail};

    my @options = ( "delay",
                    "force", "invPtrLoc", "jdk", "jre",
                    "local", "local_node", "minimize_downtime", "no_relink",
                    "no_bug_superset", "no_inventory", "oh", "patch_location",
                    "retry", "report",
                    "silent", "verbose", "error_on_conflict",
                    "pre", "post", "opatch_pre_end", 
                    "opatch_post_end" );

    my $command = lc ( $this_class );
    $$rh_detail{$command}  = Command -> new_command();

    $$rh_detail{$command}{_description} = "apply <patch_location> ";

    $$rh_detail{$command}{_helpText}    =
               "Applies an interim patch to an ORACLE_HOME from the " .
               "current \n" .
               "directory. The patch location can be changed by using " .
               "patch_location. \n" .
               "A different installation " .
               "can be specified by using ORACLE_HOME.";


    $$rh_detail{$command}{_options} =
               $this_class -> build_option_details ( {
                          rh_options     => $$rh_detail{$command}{_options},
                          ra_option_list => \@options } );

    $this_class -> add_options_to_command_description
                                ( { rh_command => $$rh_detail{$command} } );

    return ( 0 );

}   # End of command_detail().

###############################################################################
#
# NAME   : create_and_run_make_command_list
#
# PURPOSE: Run make command
#          Save commands to .patch_storage/<ID>_make.txt
#
# INPUTS : $$ARG[1]{fh_log_file}      - The file handle for the log file.
#          $$ARG[1]{noop_flag}        - Just report the actions.
#          $$ARG[1]{Oracle_Home}      - The location of the area to be patched.
#          $$ARG[1]{patch_id}         - The patch identification string.
#          $$ARG[1]{patch_location}   - Where the patch sources are located.
#          $$ARG[1]{rh_actions_list}  - The action require for a type of file.
#          $$ARG[1]{rh_file_details}  - The hash that has the details for
#                                       the make commands included in the
#                                       data.
#
# OUTPUTS: $error_messages - A string containing any errors encountered.
#
# NOTES  : Run make command directly. 
#          Do not write to <patch_ID>_make.txt under the 
#             patch location so that patch can be installed from CD-ROM.
#             Instead, write directly to .patch_storage area.
#          On Windows, skip all steps above (there's no make on Windows).
#             In other words, there is no .patch_storage/<ID>_make.txt on Win.
#
###############################################################################
sub create_and_run_make_command_list {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file      = $$rh_arguments{fh_log_file};
    my $noop_flag        = $$rh_arguments{noop_flag};
    my $Oracle_Home      = $$rh_arguments{Oracle_Home};
    my $patch_id         = $$rh_arguments{patch_id};
    my $patch_location   = $$rh_arguments{patch_location};
    my $rh_actions_list  = $$rh_arguments{rh_actions_list};
    my $rh_file_details  = $$rh_arguments{rh_file_details};

    my $error_messages   = "";
    my $initial_command  = "";
    
    if ( $Command::NO_RELINK_MODE ) { 
       opatchIO -> print_message_noverbose ( {
           f_handle => $fh_log_file,
		       message =>
           "---------------------------------------------------\n" .
		       "Skip executing 'make' on local system during apply.\n" .
           "Please invoke 'make' manually after patching.\n" .
		       "---------------------------------------------------\n" 
	       });
         
         # Do not return, need to go thru to generate <ID>_make.txt
         # return "";
    }
    
    my $undo_directory   = File::Spec -> catfile
                                         ( "$Oracle_Home",
                                           $this_class -> save_dir,
                                           $patch_id );
                                           
    my $make_command = $this_class->getMakeCommand();
    if( $make_command eq "" ) { 
       opatchIO -> print_message_and_die ( {
		       exit_val => $this_class -> ERROR_CODE_PATH_PROBLEM,
		       message =>
		       "'make' is not in your PATH or is not executable.\n" .
           "  Extra Search Path= /usr/bin/make (Linux), /usr/ccs/bin/make (others)\n" .
		       "  Path= $ENV{PATH}\n" 
	       });    
    } 

    # Create a <ID>_rollback.sh to contain manual roll-back commands
    my $make_command_file = "$patch_id" . "_make.txt";
    $make_command_file = File::Spec -> catfile ( "$undo_directory", "$make_command_file" );

    #opatchIO -> print_message_noverbose ( { f_handle => $fh_log_file,
    #                              message => "\nUsing \"$make_command\" to make...\n" } );

    # Extract the make command out of the actions list.
    # foreach my $command ( keys %{$$rh_actions_list{make}{apply}} ) {

        # David Evans: 
        # OK, I'm going to cheat here. The XML document describes this
        # as two sub-commands, a cd followed by a make. And the cd command
        # is defined as an earlier perl command. This won't work and the
        # DTD/XML file needs to be updated. So to get around this I'm
        # using the "make" from the file and assuming the OS supports a
        # 'cd' command to change directories. This will be revised and
        # addressed in a revision of this tool.
    #    if ( defined $$rh_actions_list{$command}{apply}{$command} ) {
    #        if ( $$rh_actions_list{$command}{apply}{$command}
    #                                        {command_type} eq "system" ) {
    #            $initial_command = $command . " ";
    #            $initial_command .= $$rh_actions_list{$command}{apply}
    #                                                   {$command}{parameters};
    #        }
    #    }
    #}

    # This next regex should leave "make -f ".
    # ( $initial_command ) = ( $initial_command =~ m#^([^%]+)%# );
    
    # Phi: use the make command from getMakeCommand().  This should leave
    # "/validpath/validmake -f "
    ( $initial_command ) = "$make_command -f ";

    # Now extract the details into a format suitable for printing.
    my $directory   = "";
    my $make_file   = "";
    my $target      = "";
    my $array_index = 0;
    my @commands    = ();
    my @command_targets  = ();

    foreach my $action_type ( keys %$rh_file_details ) {
        next if ( $action_type ne &Apply::MAKE_STRING );

        $array_index = scalar ( @{$$rh_file_details{$action_type}} );

        # The data is accumulated in file order. Extract the commands and
        # sort them so they are in order of client libraries first.

        # for ( my $i = $array_index -1 ; $i >= 0; $i-- ) {
        # (the above reversed traversal causes the make to be carried out in reversed sequence)

        for ( my $i = 0; $i <= $array_index; $i++ ) {

            foreach my $file ( keys %{$$rh_file_details{$action_type}[$i]} ) {
                $make_file = $file;
                $directory = $$rh_file_details{$action_type}[$i]{$file}
                                                                {directory};
                $target    = $$rh_file_details{$action_type}[$i]{$file}
                                                                {target};

                my $subDir = $directory;
                #$subDir =~ s#$Oracle_Home#\$ORACLE_HOME#;
                $subDir = File::Spec -> catfile (
                                        "\$ORACLE_HOME",
                                        substr($subDir, length($Oracle_Home)));
                
                my $output_string = "cd $subDir\n" . "$initial_command " .
                                    "$make_file $target\n";

                # Strip out duplicate entries.
                my @test = grep /$output_string/, @commands;

                if ( ! scalar ( @test ) ) {
                    if ( $target =~ m#^client_sharedlib$# ) {
                        unshift ( @commands, $output_string );
                        unshift ( @command_targets, $target );
                    } else {
                        push ( @commands, $output_string );
                        push ( @command_targets, $target );
                    }
                }
            }
        }
    }

    # Create some files to put the commands if people want to do it manually.
    my $output_string    = "# Either run these commands by hand or use " .
                           "this script.\n# This is the last stage of " .
                           "applying patch $patch_id.\n\n";
    my $make_file_error = 0;
    
    if ( $OSNAME =~ m#Win32# ) {
      opatchIO->print_message({ message=>"Skipping make file $make_command_file" });
    } else {  
      opatchIO->print_message({ message=>"Creating temporary make file $make_command_file" });
      open ( MAKE_COMMANDS, ">$make_command_file" ) || (
                    ( $error_messages .= "Couldn't create file for the make " .
                                         "commands. Error is: $OS_ERROR\n" .
                                             "file is \"$make_command_file\"\n"),
                      $make_file_error = 1 );
    if ( ! $make_file_error ) {
        opatchIO -> print_message ( { 
                                      f_handle => *MAKE_COMMANDS,
                                      message  => $output_string } );
    }
    }

    # Now lets run the commands and write out the files.
    my $ctr = 0;
    foreach my $command ( @commands ) {

        # This has literal "ORACLE_HOME" in the "cd" part
        my $cmdToMakeTxt = $command;

        # This has "ORACLE_HOME" substituted by the value $ORACLE_HOME
        $command =~ s#\$ORACLE_HOME#$Command::UNRESOLVED_ORACLE_HOME#;
        chomp $command;

        # Must parse target at this point
        #my ( $target ) = ( $command =~ m# ([^ ]+)$# );
        my $target = $command_targets[$ctr];
        $ctr++;

        # This make command doesn't have the part "ORACLE_HOME=<value>"
        #   as in Rollback and RAC make, so we will put it there
        #   to be consistent
        $command = $command . " ORACLE_HOME=\"$Command::UNRESOLVED_ORACLE_HOME\" "; 
        
        opatchIO -> print_message_noverbose ( { message => "Running make for target " .
                                                         "$target.\n" } );
        
        if ( ! $make_file_error ) {
            opatchIO -> print_message ( { f_handle => *MAKE_COMMANDS,
                                          message  => "$cmdToMakeTxt\n" } );
        }

        my $filename = "";
        if ( ! $noop_flag ) {
            my $extend_cmd = "";
            if ( $OSNAME =~ m#Win32# ) {
                $extend_cmd = "$command";
            } else {
                # Redirect stderr to a file called
                $filename = File::Spec -> catfile ( 
                                              "$undo_directory",
                                              "make_local.stderr" );
                # First remove the file if it's there
                if ( -e $filename ) {
                  unlink $filename;
                }
                
                if ( $this_class->is_newline_terminated({string => $command}) ) {
                   # let extract the command w/o the very last "\n"
                   my $len = length ($command) - 1;
                   $extend_cmd = substr($command, 0, $len);
                   # append "2>&1" to catch stderr
                   $extend_cmd = $extend_cmd . " 2>$filename";
                } else {
                  $extend_cmd = $command . " 2>$filename";
                }
            }
            
            if ( $Command::NO_RELINK_MODE ) {
                opatchIO -> print_message_noverbose ( {
                                      f_handle => $fh_log_file,
                                      message  => "-no_relink: Skip running command\n" .
                                      "\"$extend_cmd\"" 
                } );
                next;
            }
            my $sys_call_result = qx/$extend_cmd/;
            chomp $sys_call_result;
            my $status = $this_class -> get_child_error ( {
                CHILD_ERROR     => $CHILD_ERROR
            } );
            opatchIO -> print_message ( {
                                      f_handle => $fh_log_file,
                                      message  => "System Command:\n" .
                                      "\"$extend_cmd\"" . "\nResult:\n" .
                                      "$sys_call_result\n" } );

            # Small note: status may not be enough. The return value is for
            # the last make target invoked. Can't trust it really.
            if ( $status ) {
                $error_messages .= "Error for command: $extend_cmd";
            } elsif ( $sys_call_result =~ m#\bFATAL\b# ) {
                $error_messages = "FATAL ERROR: $sys_call_result\n";
            } elsif ( $sys_call_result =~ m#\bERROR\b# ) {
                $error_messages = "ERROR: $sys_call_result\n";
            } elsif ( $sys_call_result =~ m#\bFAIL\b# ) {
                $error_messages = "FAIL: $sys_call_result\n";
            }
                                
            # Read the content of redirected stderr file to check for error
            # assert: non-zero byte file size means error

            # Don't do this for now because warnings in Alert 69
            #  may generate tons of support calls.  Need more research
           
            opatchIO -> debug ( {
                message => "Display make errors from: \"$filename\""
                            } );
            if ( -e $filename && -s $filename ) 
            {

                local *MAKE_ERROR_FILE;
                open ( MAKE_ERROR_FILE, $filename ) or do {
                            opatchIO -> print_message ( {
                              f_handle => $fh_log_file,
                              message => "ERROR: Cannot open error file to read: \"$filename\"\n"
                            } ); 
                };
                my @file_data = <MAKE_ERROR_FILE>;

                my $print_stderr = 0;
                foreach my $line ( @file_data ) {
                   chomp $line;
                   if ( $line =~ m#FATAL#i ) {
                          $print_stderr = 1;
                   } elsif ( $line =~ m#ERROR#i && ($print_stderr == 0 || $print_stderr > 2)) {
                          $print_stderr = 2;
                   } elsif ( $line =~ m#FAIL#i && ($print_stderr == 0 || $print_stderr > 3)) {
                          $print_stderr = 3;
                   }
                }

                if($print_stderr) {
                     my $stderr_msg = "OPatch found the word \"";
                     if($print_stderr == 1) {
                          $stderr_msg = $stderr_msg . "fatal\" in stderr, please look into it." ;
                     } elsif($print_stderr == 2) {
                          $stderr_msg = $stderr_msg . "error\" in stderr, please look into it." ;
                     } else {
                          $stderr_msg = $stderr_msg . "fail\" in stderr, please look into it." ;
                     }

                     $stderr_msg = $stderr_msg . "\nYou may have to run the make again.";

                     opatchIO -> print_message_noverbose ( {
                            f_handle => $fh_log_file,
                            message => $stderr_msg 
                     } );
                     opatchIO -> print_message_noverbose ( {
                            f_handle => $fh_log_file,
                            message => "Stderr Output \(from $filename\):"
                     } );
                }

                if($print_stderr || $status) {
                     foreach my $line ( @file_data ) {
                              chomp $line;
                              opatchIO -> print_message_noverbose ( {
                                    f_handle => $fh_log_file,
                                    message => "$line\n"
                              } ); 
                      }
                }
                close MAKE_ERROR_FILE;
            } # end of if the stderr output is non-zero in size

            opatchIO -> debug ( {
                message => "Display make errors from: \"$filename\" Complete"
                            } );

            if ($status)
            {
                my $question =
                            "Make failed.  If you continue, " .
                            "you will have to run make manually on the local system.  " . "\n" .
			    "Replying 'Y' will terminate the patch installation immediately. " .
                            "It WILL NOT restore any updates that have been performed to this point. " .
                            "It WILL NOT update the inventory." . "\n" .
                            "Replying 'N' will continue with additional make operations and " .
			    "update the inventory showing the patch has been applied." . "\n" .
			    "NOTE: After replying either 'Y' or 'N' it is critical to review: " . "\n" .
                            "      Metalink Note 312767.1 How to rollback a failed Interim patch installation." . "\n" .
                            "\nDo you want to STOP?" ;

                my $answer = opatchIO -> check_to_stop_processing ( {
                                                message => $question } );
                if ( $answer eq "" ) {
                          opatchIO -> print_message_and_die ( {
                            exit_val => $this_class->ERROR_CODE_MAKE_FAILURE, 
                            message => "Local make failed.  See log file for details.\n"
                          } ); 
                }
            } # It make returns non-zero, error out


        }
    }

    if ( ! $make_file_error ) {
        if (!($OSNAME =~ m#Win32#)) {
        close MAKE_COMMANDS;
        }
    }


    if ( $error_messages ) {
        $error_messages = "There were problems running make commands.  " .
                          "The errors encounted were:\n"   .
                          "$error_messages\n";
    }

    return ( $error_messages );

}   # End of create_and_run_make_command_list().

###############################################################################
#
# NAME   : extract_of_file_names
#
# PURPOSE: Create the fully qualified path names from a data structure
#          made by build_file_details().
#
# INPUTS : $$ARG[1]{rh_file_details} - A reference to the data structure made
#                                      by build_file_details().
#          $$ARG[1]{Oracle_Home}     - The root of the patch hierarchy.
#          $$ARG[1]{patch_id}        - The id of the patch being applied.
#
# OUTPUTS: \%file_names - An hash of arrays. The key is the fully qualified
#                         path name. Values, if any, are dependent files.
#
# NOTES  : 1. All hardcoded.
#          2. Need to rationalize this and Apply::patch_jar_files() ?
#          3. This also identifies the new files.
#          4. This and Command::extract_file_names needs to be rationalized.
#
###############################################################################
sub extract_of_file_names {

    my $this_class      = shift @ARG;
    my $rh_arguments    = $ARG[0];

    my $Oracle_Home     = $$rh_arguments{Oracle_Home};
    my $patch_id        = $$rh_arguments{patch_id};
    my $rh_file_details = $$rh_arguments{rh_file_details};

    my %rh_file_list = ();

    my $directory_separator = File::Spec -> catfile ( "", "bin" );
    ( $directory_separator ) = ( $directory_separator =~ m#^(.+)bin$# );

    # Watch out for Window dir. sep. which is an escape in reg. exp.

    # Where is the patch directory in general?
        my $patch_directory = File::Spec -> catfile ( "$Oracle_Home",
                                                      $this_class -> save_dir,
                                                      $patch_id );

    foreach my $action ( keys %$rh_file_details ) {
        next if ( ! $action );
        next if ( $action eq &Apply::MAKE_STRING );
        next if ( ($action eq &Apply::AR_STRING) && ($OSNAME =~ m#Win32#) );

        if ( $action eq &Apply::AR_STRING ) {
            foreach my $object ( keys %{$$rh_file_details{$action}} ) {
                # Reassigned only for readability.
                my $path      = ${$$rh_file_details{$action}}{$object}{path};
                my $target    = ${$$rh_file_details{$action}}{$object}{target};
                my $component = ${$$rh_file_details{$action}}{$object}
                                                            {component_name};

                # Check if this is a new file.
                my $library = File::Basename::basename ( $target );
                my $backup_path = File::Spec -> catfile ( "$patch_directory",
                                     ( $path =~ m#$Oracle_Home(.+)# ) );
                my $check_name = &File::Basename::basename( $object ) . "_" .
                                                               $library . "_";
                $check_name = "$check_name" . &Apply::PRE_PATCH . "_$patch_id";
                $check_name = File::Spec -> catfile (
                                                "$backup_path", "$check_name" );

                # Due to a small outstanding OUI bug a simple name cannot
                # be used. So we need the directory separator to be prepended
                # to $object.
                $object = &File::Basename::basename( $object );
                my $full_name = File::Spec -> catfile( "$path", "$target" );

                $object = File::Spec -> catfile ( "$full_name", "$object" );
                ( $object ) = ( $object =~ m#$full_name(.+)$# );
                if ( ! -e $check_name ) {
                    $object .= " +> $component";
                }
                push ( @{$rh_file_list{$full_name}}, "$object" );
            }

        } elsif ( $action eq &Apply::JAR_STRING ) {
            foreach my $class ( keys %{$$rh_file_details{$action}} ) {
                # Reassigned only for readability.
                my $path      = ${$$rh_file_details{$action}}{$class}{path};
                my $target    = ${$$rh_file_details{$action}}{$class}{object};
                my $component = ${$$rh_file_details{$action}}{$class}
                                                            {component_name};
                # Check if this is a new file.
                my ( $patch_path ) = ( $class =~ m#$target(.+)# );
                # If Oracle_Home is C:\phi, path will be C:\phi\lib.
                # The func. below will return \lib
                my $tmp =  substr ($path, length($Oracle_Home));

#                my $check_name = File::Spec -> catfile ( "$patch_directory",
#                              ( $path =~ m#$Oracle_Home(.+)# ), $patch_path );
#        
#                This kind of reg. exp. code will give Unicode error if there is
#                     a \p in Oracle_Home, as in C:\phi
#
                my $check_name = File::Spec -> catfile ( "$patch_directory",
                              $tmp, $patch_path );

                $check_name = $check_name . "_" . $target . "_";
                $check_name = "$check_name" . &Apply::PRE_PATCH . "_$patch_id";

                my $reg_exp = $target . $directory_separator;
                $reg_exp =~ s/\\/\\\\/g;

                ( $class ) = ( $class =~ m#$reg_exp(.+)$# );
                if ( ! -e $check_name ) {
                    $class .= " +> $component";
                }
                my $full_name = File::Spec -> catfile( "$path", "$target" );
                push ( @{$rh_file_list{$full_name}}, $class );
            }
        } elsif ( $action eq &Apply::COPY_STRING ) {
            foreach my $target ( keys %{$$rh_file_details{$action}} ) {

                next if ( ! $target );

                # Reassigned only for readability.
                my $path      = ${$$rh_file_details{$action}}{$target}{path};
                my $object    = ${$$rh_file_details{$action}}{$target}{object};
                my $component = ${$$rh_file_details{$action}}{$target}
                                                              {component_name};

                my $check_name = File::Spec -> catfile ( $patch_directory,
                                                                     $object );
                $check_name .= "_" . &Apply::PRE_PATCH . "_$patch_id";

                $object = &File::Basename::basename( $object );
                if ( ! -e $check_name ) {
                    $object .= " +> $component";
                }
                # my $full_name = File::Spec -> catfile( "$path", "$target" );
                # Name is already fully-qualified.
                my $full_name = $target;
                push ( @{$rh_file_list{$full_name}}, $object );
            }
        }
    }

    return ( \%rh_file_list );

}   # End of extract_of_file_names().
###############################################################################
#
# NAME   : get_command_to_action_mapping
#
# PURPOSE: Parse an XML file to get the command string associated with
#          an 'action' tag.
#
# INPUTS : $$ARG[1]{GenericActions} - Name and location of the file that
#                                     has the map.
#          $$ARG[1]{Oracle_Home}    - The location of the area to be patched.
#          $$ARG[1]{os_id}          - The identification number of this OS.
#          $$ARG[1]{patch_location} - Where the patch sources are located.
#
# OUTPUTS: \%action_map             - A reference to hash that contains XML
#                                     action tags and the command to invoke
#                                     for the given tag. The key is is the
#                                     command name. Sub-keys as needed (i.e
#                                     apply or backup) have the values of the
#                                     command to run.
#
# NOTES  : 1. The XML modules (not part of the standard perl sources) haven't
#             been cleared by the legal department for use so this is a
#             quick and dirty replacement for the functions we need.
#          2. See get_file_and_action_map() for what files are to be operated
#             on for a given action.
#          4. The assumption is each entry in the XML file is on a seperate
#             line.
#          5. Not using constants for checks as names yet to be confirmed.
#
###############################################################################
sub get_command_to_action_mapping {

    # Because this was called in an OO manner the first argument is the
    # class name.
    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $os_id             = $$rh_arguments{os_id};
    my $Oracle_Home       = $$rh_arguments{Oracle_Home};
    my $patch_location    = $$rh_arguments{patch_location};
    my $xml_file_location = $$rh_arguments{GenericActions};
    my $full_location     = File::Spec->catfile
                                 ( "$patch_location", "$xml_file_location" );

    my $error_flag       = "";
    my $information_flag = "";
    my %action_map       = ();
    my @action_file_data = ();

    local *ACTION_FILE;
    open ( ACTION_FILE, $full_location ) or do {
        $error_flag .= "Cannot open action map file for reading: " .
                       "$full_location.\n[$OS_ERROR]";
    };

    if ( ! $error_flag ) {
        @action_file_data = <ACTION_FILE>;

        close ACTION_FILE or do {
            opatchIO -> print_message ( { f_handle => *STDERR,
                                          message  => "Cannot close the " .
                                   "file after reading: $full_location\n" .
                                    "[$OS_ERROR]\nProcessing continuing" } );
        };

    }

    my $perl_string   = "perl";
    my $system_string = "system";

    my $file_size = scalar ( @action_file_data );

    # This is a long loop as I only want to parse the file once. Since
    # the XML sections can be in any order (including sections we're not
    # parsing this seems to be the best. If there is time the individual
    # line splits can be handed off to other methods.
    for ( my $i =0; $i < $file_size; $i++ ) {

        chomp $action_file_data[$i];
        $action_file_data[$i] =~ s#%ORACLE_HOME%#$Oracle_Home#;

        # Yes, making it elsif is more efficient. But this is going to a
        # be redone in Java and I don't want to highlight the regex's and
        # the data structure usage.

        # <env> tags used for future functionality.
        if ( $action_file_data[$i] =~ m#<env>$# ) {
            while ( $action_file_data[$i] !~ m#</env>$# ) {
                $i++;
            }
        }

        if ( $action_file_data[$i] =~ m#<globalcmd name="# ) {
            my ( $command_name ) =
                      ( $action_file_data[$i] =~ m#<globalcmd name="(.+)"# );
            my $action_type = "";
            $i++;
            while ( $action_file_data[$i] !~ m#</globalcmd>$# ) {

                if ( ( $action_file_data[$i] =~ m#<apply>$# ) ||
                     ( $action_file_data[$i] =~ m#<backup>$# ) ) {
                    ( $action_type ) = ( $action_file_data[$i] =~ m#<(.+)># );
                    $i++;
                }

                if ( $action_file_data[$i] =~ m#<env>$# ) {
                    while ( $action_file_data[$i] !~ m#</env># ) {
                        $i++;
                    }
                    $i++;
                }

                # Currently only "system" or "perl".
                my ( $command_type ) =
                                     ( $action_file_data[$i] =~ m#<(\w+)\s# );

                # Because the XML format changes and lines of the same
                # command type may have different requirements the regex
                # matching is on the keyword and not the whole string. It also
                # relies on the surrounding quotes to and XML atom.

                if ( $command_type eq "$perl_string" ) {

                    my ( $name ) =
                             ( $action_file_data[$i] =~ m#\sname="([^"]+)"# );
                    my ( $module ) =
                           ( $action_file_data[$i] =~ m#\smodule="([^"]+)"# );
                    my ( $cmd ) =
                          ( $action_file_data[$i] =~ m#\sperlcmd="([^"]+)"# );
                    my ( $param_num ) =
                        ( $action_file_data[$i] =~ m#\sparam_num="([^"]+)"# );
                    my ( $parameters ) =
                        ( $action_file_data[$i] =~ m#\sparameters='([^"]+)'# );
                    my ( $path ) =
                         ( $action_file_data[$i] =~ m#\spath='([^']+)'# );

                    if ( ! $action_type ) {
                        if ( ! $parameters ) {
                            # A "simple" command, no apply or backup rules.
                            $action_map{$command_name}{command_type} =
                                                             "$perl_string";
                            $action_map{$command_name}{module} = "$module";
                            $action_map{$command_name}{command} = "$cmd";
                            $action_map{$command_name}{param_num} =
                                                              "$param_num";
                        }
                    } else {
                        $action_map{$command_name}{$action_type}
                                     {$name}{command_type} = "$perl_string";
                        if ( $cmd ) {
                            $action_map{$command_name}{$action_type}
                                                  {$name}{command} = "$cmd";
                        }
                        if ( $module ) {
                            $action_map{$command_name}{$action_type}
                                                {$name}{module} = "$module";
                        }
                        if ( $param_num ) {
                            $action_map{$command_name}{$action_type}
                                           {$name}{param_num} = "$param_num";
                        }
                        if ( $parameters ) {
                            $action_map{$command_name}{$action_type}
                                         {$name}{parameters} = "$parameters";
                        }
                        if ( $path ) {
                            $action_map{$command_name}{$action_type}
                                                     {$name}{path} = "$path";
                        }
                    }

                } elsif ( $command_type eq "$system_string" ) {
                    my ( $name ) =
                         ( $action_file_data[$i] =~ m#\scommand="([^"]+)"# );
                    if ( ! $name ) {
                        ( $name ) =
                         ( $action_file_data[$i] =~ m#\sname="([^"]+)"# );
                    }
                    my ( $path ) =
                         ( $action_file_data[$i] =~ m#\spath='([^']+)'# );
                    $path =~ s#%ORACLE_HOME%#$Oracle_Home#;
                    my ( $parameters ) =
                        ( $action_file_data[$i] =~ m#\sparameters='([^']+)'# );

                    if ( ! $action_type ) {
                        # All examples I have to test have apply & backup
                        # but I'm sure this will change.
                    } else {
                        $action_map{$command_name}{$action_type}
                                   {$name}{command_type} = "$system_string";
                        $action_map{$command_name}{$action_type}
                                                    {$name}{path} = "$path";
                        $action_map{$command_name}{$action_type}
                                        {$name}{parameters} = "$parameters";

                    }
                }
                $i++;
            }
        }   # End of "global commands" section.

        # This is similar to the global commands section but adds commands
        # specific to the current OS or overrides global commands with
        # OS specific variants. According to theory these should all be
        # "system" type commands. That's all there is at the moment.
        if ( $action_file_data[$i] =~ m#<os osname="[^"]+"\sid="$os_id"# ) {
            while ( $action_file_data[$i] !~ m#</os>$# ) {

                if ( $action_file_data[$i] =~ m#<env>$# ) {
                    while ( $action_file_data[$i] !~ m#</env># ) {
                        $i++;
                    }
                    $i++;
                }

                my $action_name = "";
                my $action_type = "";
                if ( $action_file_data[$i] =~ m#<action name="# ) {

                    ( $action_name ) =
                       ( $action_file_data[$i] =~ m#<action name="([^"]+)"# );
                    $i++;

                    $action_map{$action_name} = ();
                    while ( $action_file_data[$i] !~ m#</action>$# ) {

                        if ( $action_file_data[$i] =~ m#<env>$# ) {
                            while ( $action_file_data[$i] !~ m#</env># ) {
                                $i++;
                            }
                            $i++;
                        }

                        if ( ( $action_file_data[$i] =~ m#<apply>$# ) ||
                             ( $action_file_data[$i] =~ m#<backup>$# ) ) {
                            ( $action_type ) =
                                     ( $action_file_data[$i] =~ m#<(.+)># );
                            $i++;
                        }

                        my ( $name ) =
                                ( $action_file_data[$i] =~
                                                    m#\scommand="([^"]+)"# );
                        my ( $path ) =
                                ( $action_file_data[$i] =~
                                                       m#\spath='([^']+)'# );
                        $path =~ s#%ORACLE_HOME%#$Oracle_Home#;
                        my ( $parameters ) =
                                ( $action_file_data[$i] =~
                                                 m#\sparameters='([^']+)'# );

                        # Want to ignore the lines with the closing tag for
                        # $action_type.
                        if ( $name ) {
                            $action_map{$action_name}{$action_type}
                                   {$name}{command_type} = "$system_string";
                            $action_map{$action_name}{$action_type}
                                                    {$name}{path} = "$path";
                            $action_map{$action_name}{$action_type}
                                        {$name}{parameters} = "$parameters";
                        }
                        $i++;
                    }
                }
                $i++;
            }
        }   # End of "OS" override section.
    }

    my %return_values = ();
    $return_values{rh_actions} = \%action_map;
    $return_values{error_flag} = $error_flag;

    return ( \%return_values );

}   # End of get_command_to_action_mapping().

###############################################################################
#
# NAME   : build_file_details
#
# PURPOSE: Parse an XML file to get a list 'action' tags and associated files.
#
# INPUTS : $$ARG[1]{actions_file}   - Name and location of the file that
#                                     has the map.
#          $$ARG[1]{Oracle_Home}    - The location of the area to be patched.
#          $$ARG[1]{patch_location} - Where the patch sources are located.
#
# OUTPUTS: \%file_map               - A reference to hash that contains the
#                                     XML data in a usable form. The keys
#                                     have different formats according to use.
#
# NOTES  : 1. The XML modules (not part of the standard perl sources) haven't
#             been cleared by the legal department for use so this is a
#             quick and dirty replacement for the functions we need.
#          2. For each action tag a list of files should be associated with
#             it. See get_command_to_action_mapping() for the command to run
#             as the action.
#          3. The assumption is each entry in the XML file is on a seperate
#             line.
#          4. The data needs to be parsed twice. The first time is for the
#             check of the inventory. The second time is build the data set
#             for the patch based on the results of the inventory check.
#
###############################################################################
sub build_file_details {

    # Because this was called in an OO manner the first argument is the
    # class name.
    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $error_flag       = "";
    my %file_map         = ();
    my @file_data        = ();

    my $actions_file      = $$rh_arguments{actions_file};
    my $Oracle_Home       = $$rh_arguments{Oracle_Home};
    my $patch_location    = $$rh_arguments{patch_location};
    my $skip_components = $$rh_arguments{skip_components};

    my $full_location     = File::Spec->catfile
                                ( "$patch_location", "$actions_file" );

    opatchIO->debug({message=>"Opening actions.xml..."});

    local *DATA_FILE;
    open ( DATA_FILE, $full_location ) or do {
        $error_flag .= "Cannot open file list map for reading: " .
                       "$full_location.\n[$OS_ERROR]";
    };

    if ( ! $error_flag ) {
        @file_data = <DATA_FILE>;

        close DATA_FILE or do {
            opatchIO -> print_message ( { f_handle => *STDERR,
                                          message  => "Cannot close the " .
                                   "file after reading: $full_location\n" .
                                    "[$OS_ERROR]\nProcessing continuing" } );
        };
    }

    my $file_size = scalar ( @file_data );
    @{$file_map{&MAKE_STRING}} = ();

    # component_name and component_version must not be re-init. inside the loop
    my $component_name = "";
    my $component_version = "";
    
    # variable $skip must not be re-init. inside the loop
    my $skip = 0;
    for ( my $i =0; $i < $file_size; $i++ ) {

       my $line = $file_data[$i];
       chomp $line;
       my $componentLine = 0;
       
       opatchIO->debug({ message =>
               "Processing $line"
       });
       
       # Need to store the component name for new files, if it exists.
      # if ( $line =~ m#.\w+\sversion="[\d\.]+"\sopt_req="[O|R]"># ) {
	if ( $line =~ m#.\w+\sversion=\"(.)+\"\sopt_req="[O|R]"># ) {
         # This is the line that represents a component
         $componentLine = 1;
                
          # ( $component_name ) = ( $line =~
          #            m#<([\w\.]+)\sversion="[\d\.]+"\sopt_req="[O|R]">$# );
          
          ( $component_name, $component_version ) = ( $line =~
                       m#<([\w\.]+)\sversion=\"(.+)\"\sopt_req="[O|R]"># );
          opatchIO->debug({ message =>
             "  Got \"$component_name,$component_version\""
          });         
       }
       
       # Continue till component and version is set
       if (($component_name eq "") && ($component_version eq "")) { 
          opatchIO->debug({ message => " not getting a 'component' line yet, skip" });
          next;
       }

       if ($componentLine) {
          opatchIO->debug({ message =>
               "Check whether to skip component <$component_name, $component_version>\n"
          });
          $skip = $this_class->skipThisComponent({
                  component_name => $component_name,
                  component_version => $component_version,
                  skip_components => $skip_components
              });
       }
       # Assert: once $skip is set to something, it won't be reset until
       #  we process the next 'component' line.
       if ( $skip == 1) {
          opatchIO->debug({ 
            message => "This line belongs to a skipped component, " .
                       "do not process it"
          });
          next;
       } else {
          opatchIO->debug({ message =>
                 "This line belongs to a present component, " .
                 "process it"
          });
       }
       # Lets expand $ORACLE_HOME if its used.
       if ( $line =~ m#%ORACLE_HOME%# ) {
           $line =~ s#%ORACLE_HOME%#$Oracle_Home#;
       }

       if ( $line =~ m#<archive name="# ) {
           opatchIO->debug({ message => "  add archive action" });
           my ( $target, $path, $object ) = ( $line =~
            m#<archive\sname="([^"]+)"\spath="([^"]+)"\sobject_name="([^"]+)"# );

           # For static libraries the object being updated is unique so that is
           # the key to use.
           $file_map{&AR_STRING}{$object}{target} = $target;
           $file_map{&AR_STRING}{$object}{path}   = $path;
           $file_map{&AR_STRING}{$object}{component_name} = $component_name;

       } elsif ( $line =~ m#<copy name="# ) {
           opatchIO->debug({ message => "  add copy action" });
           my ( $name, $path, $object ) = ( $line =~
              m#<copy\sname="([^"]+)"\spath="([^"]+)"\sfile_name="([^"]+)"# );

           # the above hash will over-write $name if it appears twice
           # in "actions" file.  For example:
           # <copy name="foo.xml" path= .... file_name= ...>
           # <copy name="foo.xml" path= .... file_name= ...>
           #
           # So we need to encode the "foo.xml" as something else...
           my $encodedName = $this_class->encodeName ({
             name => $name,
             path => $path
           });

           $file_map{&COPY_STRING}{$encodedName}{object} = $object;
           $file_map{&COPY_STRING}{$encodedName}{path}   = $path;
           $file_map{&COPY_STRING}{$encodedName}{component_name} =
	                                              $component_name;
           # $file_map{&COPY_STRING}{$name}{object} = $object;
           # $file_map{&COPY_STRING}{$name}{path}   = $path;
           # $file_map{&COPY_STRING}{$name}{component_name} = $component_name;

       } elsif ( $line =~ m#<jar name="# ) {
           opatchIO->debug({ message => "  add jar action" });
           my ( $target, $path, $class ) = ( $line =~
              m#<jar\sname="([^"]+)"\spath="([^"]+)"\sclass_name="([^"]+)"# );
           $file_map{&JAR_STRING}{$class}{object} = $target;
           $file_map{&JAR_STRING}{$class}{path}   = $path;
           $file_map{&JAR_STRING}{$class}{component_name} = $component_name;

       } elsif ( $line =~ m#<make change_dir="# ) {    
           opatchIO->debug({ message => "  add make action" });
           # Just splitting it up to keep it under 80 columns. Yes, I'm a
           # dinosaur.
           my ( $directory ) = ( $line =~
                              m#<make\schange_dir="([^"]+)"\smake_file="# );
           my ( $file, $target ) = ( $line =~
                          m#\smake_file="([^"]+)"\smake_target="([^"]+)"# );
           # Small cheat here. The tuple is unique but all elements may not
           # be. So this needs to be an array of hashes. Also the makes
           # may be dependant on order so they must be kept in that order.
           my $array_index = scalar ( @{$file_map{&MAKE_STRING}} );
           $file_map{&MAKE_STRING}[$array_index]{$file}{directory} =
                                                                  $directory;
           $file_map{&MAKE_STRING}[$array_index]{$file}{target} = $target;

       }
    }

    my %return_values = ();
    $return_values{rh_files} = \%file_map;
    $return_values{error_flag} = $error_flag;

    return ( \%return_values );

}   # End of build_file_details().


##############################################################################
#
# NAME   : parse_arguments
#
# PURPOSE: A check, validation and variable assignments based on the
#          the arguments for this sub-command.
#
# INPUTS : $$ARG[1]{rh_option} - A reference to a hash where the details about
#                                the option are to be stored.
#
# OUTPUTS: A reference to a hash.
#
# NOTES  : 1. Well parse_arguments() is used for the command line check. But
#             at that point the class is the super-class: Command. So we can
#             re-use the name space here for specific processing for the class
#             at the correct time.
#          2. I know the "*_arguments" will confuse people so let me explain it
#             at this point in the module. The command line is parsed initially
#             at the top level (by parse_arguments) that retruns a string. The
#             string contains the arguments. This is turned into an array. The
#             array is an argument to this method. Since the argument to this
#             method is a the local arguments are rh_arguments. The initial
#             command line items, now an array, is located at
#             $$rh_arguments{ra_arguments}.
#          3. The above may be confusing but it keeps all the calls in the
#             same format: a single hash contains all the parameters for any
#             method.
#          4. While "force" can be given multiple times an error is raised if
#             other arguments appear more than once.
#
###############################################################################
sub parse_arguments {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $ra_arguments = $$rh_arguments{ra_arguments};

    my $array_size  = scalar( @$ra_arguments );
    my $error_flag  = "";
    my $force_flag  = 0;
    my $noop_flag   = 0;
    my $Oracle_Home = "";
    my $Ship_Home   = "";
    my $general_options = 0;
    my $RAC_options     = 0;

    for ( my $i = 0; $i < $array_size; $i++ ) {

        if ( $$ra_arguments[$i] =~ m#^-oh$# ) {
            # Override the location if given on the command line.
            if ( ! $Oracle_Home ) {
                $Command::UNRESOLVED_ORACLE_HOME = $$ra_arguments[$i +1];
                $Oracle_Home = $this_class -> get_abs_path( { Path =>  $Command::UNRESOLVED_ORACLE_HOME });
                $ENV{ORACLE_HOME} = $Oracle_Home;
                $i++;
            } else {
                $error_flag .= "Argument -oh appears more than once.\n";
                $i++;
            }

        } elsif ( $$ra_arguments[$i] =~ m#^-force# ) {
            $force_flag = 1;
            $general_options = $general_options ^
                               ( $this_class -> force_patch_installation );
		
        } elsif ( $$ra_arguments[$i] =~ m#^-h(elp)?$# ) {
            $force_flag = 1;
            $general_options = $general_options ^
                               ( $this_class -> display_usage );

        } elsif ( $$ra_arguments[$i] =~ m#^-i(nvPtrLoc)?$# ) {
            my $inv_ptr_file = $$ra_arguments[$i + 1];
            if ( &File::Basename::basename ( $inv_ptr_file ) ne
                                                          "oraInst.loc" ) {
                $inv_ptr_file = File::Spec -> catfile ( "$inv_ptr_file" ,
                                                            "oraInst.loc" );
            }

            $Command::inventory_location_ptr =
                    "-Doracle.installer.invPtrLoc=" . $$ra_arguments[$i + 1];

            $i++;

        } elsif ( $$ra_arguments[$i] =~ m#^-minimize_downtime# ) {
            $RAC_options     = $general_options ^
                               ( $this_class -> minimize_downtime );
            $Command::MINIMIZE_DOWNTIME_OPTION = 1;
            
        } elsif ( $$ra_arguments[$i] =~ m#^-no_inventory# ) {
            # No effect yet.
            $general_options = $general_options ^
                               ( $this_class -> no_inventory_update );
        } elsif ( $$ra_arguments[$i] =~ m#^-local_node# ) {

            $Command::LOCAL_NODE_NAME = $$ra_arguments[$i + 1];
            $Command::LOCAL_NODE_NAME_MODE = 1;
            $i++;

        } elsif ( $$ra_arguments[$i] =~ m#^-local# ) {
            $general_options = $general_options ^
	                       ( $this_class -> local_node_only );

        } elsif ( $$ra_arguments[$i] =~ m#^-r(eport)?$# ) {
            $noop_flag = 1;
            $general_options = $general_options ^
                               ( $this_class -> report_only );

        } elsif ( $$ra_arguments[$i] =~ m#^-silent# ) {
            $general_options = $general_options ^
                               ( $this_class -> silent );

        } elsif ( $$ra_arguments[$i] =~ m#^-no_relink# ) {
            
            $Command::NO_RELINK_MODE = 1;
            
        } elsif ( $$ra_arguments[$i] =~ m#^-no_bug_superset# ) {
            $general_options = $general_options ^
                               ( $this_class -> no_bug_superset );

        } elsif ( $$ra_arguments[$i] =~ m#^-verbose# ) {
            $general_options = $general_options ^
                               ( $this_class -> verbose );

        } elsif ( $$ra_arguments[$i] =~ m#^-error_on_conflict$# ) {
            # Do nothing, this is a no-op

        } elsif ( $$ra_arguments[$i] =~ m#^-jdk# ) {
            $Command::JDK_LOC = $$ra_arguments[$i + 1];
            $Command::JDK_ARG = $Command::JDK_LOC;
            $i++;


        } elsif ( $$ra_arguments[$i] =~ m#^-jre# ) {
            $Command::JRE_LOC = $$ra_arguments[$i + 1];
            $Command::JRE_ARG = $Command::JRE_LOC;
            $i++;

        } elsif ( $$ra_arguments[$i] =~ m#^-retry# ) {
            $Command::RETRY = $$ra_arguments[$i + 1];
            $i++;

        } elsif ( $$ra_arguments[$i] =~ m#^-delay# ) {
            $Command::DELAY = $$ra_arguments[$i + 1];
            $i++;

        } elsif ( $$ra_arguments[$i] =~ m#^-pre# ) {
            # get the whole parameter list for pre, the command
            # reaches end or -opatch_pre_end explicitly shows the
            # end of -pre options
            my $concat_arg = "";
            my $arg_no = ($i + 1);
            while (($arg_no < $array_size) && ($$ra_arguments[$arg_no] ne "-opatch_pre_end"))
            {
                $concat_arg .= $$ra_arguments[$arg_no] . " ";
                $arg_no ++;
            }
            chomp $concat_arg;
            $Command::PRE_PARAMETERS = $concat_arg;
            $i = $arg_no;

        } elsif ( $$ra_arguments[$i] =~ m#^-post# ) {
            # get the whole parameter list for post, the command
            # reaches end or -opatch_post_end explicitly shows the
            # end of -post options
            my $concat_arg = "";
            my $arg_no = ($i + 1);
            while (($arg_no < $array_size) && ($$ra_arguments[$arg_no] ne "-opatch_post_end"))
            {
                $concat_arg .= $$ra_arguments[$arg_no] . " ";
                $arg_no ++;
            }
            chomp $concat_arg;
            $Command::POST_PARAMETERS = $concat_arg;
            $i = $arg_no;

        } else {

            if ( ! $Ship_Home ) {
                $Ship_Home = $$ra_arguments[$i];
                opatchIO->debug({message => "Ship_Home is $Ship_Home\n" });
            } else {
                $error_flag .= "OracleHome/ShipHome appears more than once.\n";
            }
        }
    }

    # If not set get it from the envrionment;
    if ( ! $Oracle_Home ) {
        $Oracle_Home = $ENV{ORACLE_HOME} || "";
        
        # save the original value of OH in case it is a symbolic link and get
        #   resolved below
        $Command::UNRESOLVED_ORACLE_HOME = $Oracle_Home;
	
        # Convert it into absolute path.
        $Oracle_Home = $this_class -> get_abs_path( { Path => $Oracle_Home } );
    }

    # If $Oracle_Home isn't a directory raise an error.
    if ( ! -d $Oracle_Home || ! -e $Oracle_Home ) {
        $error_flag .= "Oracle's home is not a directory: \n" .
                          "$Oracle_Home.\n";
    }

    # If there is still no $Oracle_Home raise an error. If there is an
    # $Oracle_Home and it's MicroSoft OS lowercase the string.
    if ( ! $Oracle_Home ) {
        $error_flag .= "There is no \$ORACLE_HOME to apply the patch to.\n";
    } elsif ( $OSNAME =~ m#Win32# ) {
        $Oracle_Home = lc ( $Oracle_Home );
    } else {
        # If $Oracle_Home is a symbolic link, convert it to the real path
        my $is_link = $this_class -> is_symbolic_link ( $Oracle_Home );
        if ( $is_link == 1 ) {
            my $real_OH = $this_class->resolve_symbolic_link ($Oracle_Home);

            opatchIO->print_message( {message=> "\n-----------------------------------------\n"} );
            opatchIO->print_message( {message=> "\nWARNING: Oracle_Home is a symbolic link:\n"} );
            opatchIO->print_message( {message=> "$Oracle_Home\n"} );
            opatchIO->print_message( {message=> "--->\n"} );
            opatchIO->print_message( {message=> "$real_OH"} );
            opatchIO->print_message( {message=> "\n-----------------------------------------\n"} );
            # Begin Add: BUG 2899335
            my $is_internal_link = $this_class -> verify_internal_link ($Oracle_Home);
            if ($is_internal_link == 0 ) {
                $Oracle_Home = $real_OH;
            }
            # End Add: BUG 2899335
        }
    }

    # If there is an $Oracle_Home strip any trailing "/" or "\" from it.
    # This upsets the OUI APIs for some reason.
    if ( $Oracle_Home =~ m#[\\/]$# ) {
        ( $Oracle_Home ) = ( $Oracle_Home =~ m#(.+)[\\/]\s?$# );
    }

    # Where the patches are located.
    if ( $Ship_Home ) {
      if ( (! -d $Ship_Home) || (! -e $Ship_Home) ) {
        $error_flag .= "The patch location is not a directory: $Ship_Home.\n";
      }
    }

    my %return_values = ();
    $return_values{error_flag}  = $error_flag;
    $return_values{force_flag}  = $force_flag;
    $return_values{general_options}  = $general_options;
    $return_values{noop_flag}   = $noop_flag;
    $return_values{Oracle_Home} = $Oracle_Home;
    $return_values{RAC_options} = $RAC_options;
    $return_values{ship_home}   = $Ship_Home;

    return ( \%return_values );

}   # End of parse_arguments().

###############################################################################
#
# NAME   : option_force
#
# PURPOSE: Provide the details the "force" option.
#
# INPUTS : $$ARG[1]{rh_option} - A reference to a hash where the details about
#                                the option are to be stored.
#
# OUTPUTS: NONE
#
# NOTES  : 1. The return values are in hash so the caller will get the new
#             details when this subroutine is finished.
#          2. The formatting is used in the help methods.
#
###############################################################################
sub option_force {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

    $$rh_details{_defaultValue} = "false";
    $$rh_details{_description}  = "-force";
    $$rh_details{_flag}         = "yes";
    $$rh_details{_helpText}     = "If a conflict exist which prevents the " .
                                  "patch \n" .
                                  "from being applied, the -force flag " .
                                  "can be \n" .
                                  "used to apply the patch.\n" .
                                  "*** This removes the conflicting " .
                                  "patches from \n" .
                                  "the system.";
    $$rh_details{_type}         = "install";

}   # End of option_force().

###############################################################################
#
# NAME   : option_minimize_downtime
#
# PURPOSE: Provide the details the "minimize_downtime" option.
#
# INPUTS : $$ARG[1]{rh_option} - A reference to a hash where the details about
#                                the option are to be stored.
#
# OUTPUTS: NONE
#
# NOTES  : 1. The return values are in hash so the caller will get the new
#             details when this subroutine is finished.
#          2. The formatting is used in the help methods.
#
###############################################################################
sub option_minimize_downtime {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

    $$rh_details{_defaultValue} = "false";
    $$rh_details{_description}  = "-minimize_downtime";
    $$rh_details{_flag}         = "yes";
    $$rh_details{_helpText}    = "Only applied to RAC instances. User supplies\n" .
                                 "the order of nodes to be patched.\n" .
                                 "This option can not be used in\n" .
                                 "conjunction with -local option\n" .
                                 "or with a rolling patch.\n";
    $$rh_details{_type}        = "install";

}   # End of option_minimize_downtime().

##############################################################################
#
# NAME   : option_no_inventory
#
# PURPOSE: Set the Oracle home directory path.
#
# INPUTS : $$ARG[1]{rh_option} - A reference to a hash where the details about
#                                the option is to be stored.
#
# OUTPUTS: NONE
#
# NOTES  : 1. The return values are in hash so the caller will get the new
#             details when this subroutine is finished.
#          2. The difference between this and the rollback one is the
#             formatting of the strings.
#
###############################################################################
sub option_no_inventory {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

    $$rh_details{_defaultValue} = "false";
    $$rh_details{_description}    = "-no_inventory";
    $$rh_details{_helpText}       = "Bypass the inventory for reading and updates.\n" .
				    "This option can not be used in\n" .
				    "conjunction with -local option\n" .
                                    "*** This puts the installation ***\n"  .
                                    "*** into an unsupported state. ***\n";
    $$rh_details{_required}       = "no";

}   # End of option_no_inventory().

##############################################################################
#
# NAME: option_local
#
# PURPOSE: Set the local option
#
# ############################################################################
sub option_local {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];

   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-local";
   $$rh_details{_helpText}     =
                                 "Patch then update\n" .
                                 "inventory of the local node. \n" .
                                 "Do not propagate the patch or\n" .
                                 "inventory update to other nodes.\n" .
                                  "This option cannot be used together with -local_node." ;
   $$rh_details{_required}       = "no";

}

##############################################################################
#
# NAME: option_silent
#
# PURPOSE: Set the silent option
#
# ############################################################################
sub option_silent {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-silent";
   $$rh_details{_helpText}     =
                                 "This suppresses any user-interaction and defaults\n" .
                                 "any Y|N question to Y.\n" .
                                 "This option is not supported on RAC yet.\n" ;
   $$rh_details{_required}       = "no";

}

##############################################################################
#
# NAME: option_report
#
# PURPOSE: Set the report option
#
# ############################################################################
sub option_report {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-report";
   $$rh_details{_helpText}     =
                               "Invokes OPatch in report only mode";
   $$rh_details{_required}       = "no";

}

##############################################################################
#
# NAME: option_local_node
#
# PURPOSE: Set local node name instead of detecting it 
#
# ############################################################################
sub option_local_node {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-local_node <Name of local node>";
   $$rh_details{_helpText}     =
                                 "OPatch will use the supplied local node name\n" .
                                 "instead of detecting it.  OPatch will trust\n" .
                                 "that the supplied local node name is correct.\n" .
                                 "This option cannot be used together with -local." ;
   $$rh_details{_required}       = "no";

}

##############################################################################
#
# NAME: option_no_relink
#
# PURPOSE: No relink for make actions 
#
# ############################################################################
sub option_no_relink {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-no_relink";
   $$rh_details{_helpText}     =
                                 "OPatch will not execute 'make' actions during Apply.\n" .
                                 "Users will run 'make' manually after patching.\n" .
                                 "This option is valid on Unix platforms only." ;
   $$rh_details{_required}       = "no";

}

##############################################################################
#
# NAME: option_no_bug_superset
#
# PURPOSE: Set the no_bug_superset option
#
# ############################################################################
sub option_no_bug_superset {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-no_bug_superset";
   $$rh_details{_helpText}     =
                                 "This option tells OPatch to error out if the\n" .
                                 "current patch's bugs-to-fix is a super-set\n" .
                                 "(or same set) of an installed patch's bugs-fixed\n" .
                                 "in the Oracle Home.\n";

   $$rh_details{_required}       = "no";

}

##############################################################################
#
# NAME: option_verbose
#
# PURPOSE: Set the verbose option
#
# ############################################################################
sub option_verbose {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-verbose";
   $$rh_details{_helpText}     =
                                 "This option prints more OPatch output\n" .
                                 "to the screen as well as to the log file\n" ;
   $$rh_details{_required}       = "no";

}

##############################################################################
#
# NAME   : option_oh
#
# PURPOSE: Set the Oracle home directory path.
#
# INPUTS : $$ARG[1]{rh_option} - A reference to a hash where the details about
#                                the option is to be stored.
#
# OUTPUTS: NONE
#
# NOTES  : 1. The return values are in hash so the caller will get the new
#             details when this subroutine is finished.
#          2. The difference between this and the rollback one is the
#             formatting of the strings.
#
###############################################################################
#sub option_oh {
#
#    my $this_class   = shift @ARG;
#    my $rh_arguments = $ARG[0];
#
#    my $rh_details   = $$rh_arguments{rh_option_detail};
#
#    $$rh_details{_description}    = "-oh <ORACLE_HOME>";
#    $$rh_details{_helpText}       = "The directory to use instead of the\n" .
#                                    "default of \$ORACLE_HOME.\n" .
#				                            "This option is deprecated.\n";
#    $$rh_details{_required}       = "yes";
#    $$rh_details{_type}           = "directory";
#    $$rh_details{_validityChecks} = [ "read", "write", "execute" ];

#}   # End of option_oh()

##############################################################################
#
# NAME: option_jdk
#
# PURPOSE: Set the jdk option
#
# ############################################################################
sub option_jdk {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-jdk <LOC>";
   $$rh_details{_helpText}     =
                                 "This option tells OPatch to use JDK (jar)\n" .
                                 "from the specified location instead of\n" .
                                 "the default location under Oracle Home.\n";
    $$rh_details{_required}       = "no";
    $$rh_details{_type}           = "directory";
}

##############################################################################
#
# NAME: option_jre
#
# PURPOSE: Set the jre option
#
# ############################################################################
sub option_jre {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-jre <LOC>";
   $$rh_details{_helpText}     =
                                 "This option tells OPatch to use JRE (java)\n" .
                                 "from the specified location instead of\n" .
                                 "the default location under Oracle Home.\n";
    $$rh_details{_required}       = "no";
    #$$rh_details{_type}           = "directory";
}

##############################################################################
#
# NAME: option_pre
#
# PURPOSE: Set the pre option
#
# ############################################################################
sub option_pre {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "";
   $$rh_details{_description}  = "-pre <options to be passed into pre> [-opatch_pre_end]";
   $$rh_details{_helpText}     =
                                 "This option specifies the parameters\n" .
                                 "to be passed inside pre script apart\n" .
                                 "from the standard ones.\n" .
                                 "Without this option OPatch calls pre\n" .
                                 "as 'pre -<operation> <patch-id>'\n" .
                                 "With this option OPatch will call pre\n" .
                                 "as 'pre -<operation> <patch-id> <pre options>'\n";
    $$rh_details{_required}       = "no";
    #$$rh_details{_type}           = "directory";
}

##############################################################################
#
# NAME: option_opatch_pre_end
#
# PURPOSE: Set the opatch_pre_end option
#
# ############################################################################
sub option_opatch_pre_end {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-opatch_pre_end";
   $$rh_details{_helpText}     =
                                 "This option is used to mark the\n".
                                 "end of pre options. Without this option\n".
                                 "everything after pre till end of\n".
                                 "the command is passed into pre.\n".
                                 "This option is meaningless without\n".
                                 "-pre option\n";
   $$rh_details{_required}     = "no";
    #$$rh_details{_type}           = "directory";
}

##############################################################################
#
# NAME: option_post
#
# PURPOSE: Set the post option
#
# ############################################################################
sub option_post {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "";
   $$rh_details{_description}  = "-post <options to be passed into post> [-opatch_post_end]";
   $$rh_details{_helpText}     =
                                 "This option specifies the parameters\n" .
                                 "to be passed inside post script apart\n" .
                                 "from the standard ones.\n" .
                                 "Without this option OPatch calls post\n" .
                                 "as 'post -<operation> <patch-id>'\n" .
                                 "With this option OPatch will call post\n" .
                                 "as 'post -<operation> <patch-id> <post options>'\n";
    $$rh_details{_required}       = "no";
    #$$rh_details{_type}           = "directory";
}

##############################################################################
#
# NAME: option_opatch_post_end
#
# PURPOSE: Set the opatch_post_end option
#
# ############################################################################
sub option_opatch_post_end {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-opatch_post_end";
   $$rh_details{_helpText}     =
                                 "This option is used to mark the\n".
                                 "end of post options. Without this option\n".
                                 "everything after post till end of\n".
                                 "the command is passed into post.\n".
                                 "This option is meaningless without\n".
                                 "-post option\n";
   $$rh_details{_required}     = "no";
    #$$rh_details{_type}           = "directory";
}

##############################################################################
#
# NAME: option_retry
#
# PURPOSE: Set the retry option for OPatch to retry locking inventory
#
# ############################################################################
sub option_retry {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-retry <value>";
   $$rh_details{_helpText}     =
                                 "This option tells OPatch how many times\n" .
                                 "it should retry in case of an inventory\n" .
                                 "lock failure.";
    $$rh_details{_required}       = "no";
    #$$rh_details{_type}           = "directory";
}

##############################################################################
#
# NAME: option_delay
#
# PURPOSE: Set the number of seconds to retry locking inventory
#
# ############################################################################
sub option_delay {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-delay <value>";
   $$rh_details{_helpText}     =
                                 "If -retry is specified, this option tells\n" .
                                 "OPatch how many seconds it should wait before\n" .
                                 "attempting to lock inventory again in case\n" .
                                 "of a previous failure.";
    $$rh_details{_required}       = "no";
    #$$rh_details{_type}           = "directory";
}

###############################################################################
#
# NAME   : option_patch_location
#
# PURPOSE: Provide the details the "patch_location" option.
#
# INPUTS : $$ARG[1]{rh_option} - A reference to a hash where the details about
#                                the option are to be stored.
#
# OUTPUTS: NONE
#
# NOTES  : 1. The return values are in hash so the caller will get the new
#             details when this subroutine is finished.
#          2. The formatting is used in the help methods.
#
###############################################################################
sub option_patch_location {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

    $$rh_details{_defaultValue} = "false";
    $$rh_details{_description}  = "";
    $$rh_details{_flag}         = "yes";
    $$rh_details{_helpText}     = "Where to install the interim patch " .
                                  "from.\n" .
                                  "This should be a directory with the " .
                                  "same name as\n" .
                                  "the interim patch.";
    $$rh_details{_type}         = "install";

}   # End of option_patch_location().

# NO-OP for option -error_on_conflict
sub option_error_on_conflict {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

    $$rh_details{_defaultValue} = "false";
    $$rh_details{_description}  = "";
    $$rh_details{_flag}         = "yes";
    $$rh_details{_helpText}     = "Deprecated.  Has no effect.";
    $$rh_details{_type}         = "install";
 
     
}

###############################################################################
#
# NAME   : read_and_check_reference_details
#
# PURPOSE: Provide the details the "force" option.
#
# INPUTS : $$ARG[1]{inventory_file} - The path to the file that has the top
#                                     level reference details. The path is
#                                     realative to the patch directory.
#          $$ARG[1]{Oracle_Home}    - The path to the Oracle installation.
#          $$ARG[1]{p_i_l}          - The path to the patch directory.
#
# OUTPUTS: $rh_reference_details - A reference to a hash that has the data
#                                  needed.
#          {bug_list}            - A hash of base bugs, subkey is:
#                {$number}       - The number of the base and the description.
#          {date}                - Date the patch was made, subkeys are:
#                {day}           - Self-evident.
#                {month}         - Self-evident.
#                {time}          - Self-evident.
#                {year}          - Self-evident.
#                {zone}          - The time-zone of the computer.
#          {description}         - The title of the bug.
#          {error_flag}          - On success it's a NULL string. Otherwise
#                                  the details of failure are stored here.
#          {executable}          - Paths to binaries that are needed, key is:
#                {name}          - The name of the binary and the path.
#          {online_rac}          - Flag to say if the RAC needs to be online
#                                  when dealing with real application clusters.
#          {os_platforms}        - What OSs this patch applies to, subkey is:
#                {$number}       - OS id number and descriptive string.
#          {reference_id}        - The patch id, may have text after the
#                                  number.
#          {components}          - A hash of componets, subkey is:
#                {$name}         - The name of the component that has the
#                                  sub-subkeys {version} and {option_flag}.
#          {shutdown}            - Flag to say if the database needs to be
#                                  shutdown for the patch to be installed.
#
# NOTES  : 1. This list may be out of date. Check the code below for
#             what is really returned.
#          2. Remember this is hard-coded as we don't have legal clearance
#             to use XML modules as they are outside of the standard perl
#             distribution.
#          3. Some of this code has been duplicated into the function
#             RollBack::get_initial_details as an initial hack for rollback
#             of rolling RAC.
#
###############################################################################
sub read_and_check_reference_details {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $inventory_file   = $$rh_arguments{inventory_file};
    my $Oracle_Home      = $$rh_arguments{Oracle_Home};
    my $patch_inventory_location = $$rh_arguments{p_i_l};
    my $error_flag = "";
    my $information_flag = "";
    my @file_data = ();

    my $full_inventory_path = File::Spec -> catfile(
                                              "$patch_inventory_location",
                                              "$inventory_file" );
    local *INVENTORY_FILE;
    open ( INVENTORY_FILE, $full_inventory_path ) or do {
        $error_flag = "Cannot open data file for reading: " .
                      "$full_inventory_path.\n[$OS_ERROR]";
    };

    if ( ! $error_flag ) {
        @file_data = <INVENTORY_FILE>;
        close INVENTORY_FILE or do {
            opatchIO -> print_message ( { f_handle => *STDERR,
                                          message  => "Cannot close the " .
                             "file after reading: $full_inventory_path\n" .
                                    "[$OS_ERROR]\nProcessing continuing" } );
        };
    }

    my %reference_details = ();
    $reference_details{RAC_shutdown} = 1;

    my $file_size = scalar ( @file_data );

    for ( my $i =0; $i < $file_size; $i++ ) {
        chomp $file_data[$i];

        # Yes, making it elsif is more efficient. But this is going to a
        # be redone in Java and I don't want to highlight the regex's and
        # the data structure usage.
        if ( $file_data[$i] =~ m#<reference_id number="# ) {
           ( $reference_details{reference_id} ) =
                        ( $file_data[$i] =~ m#<reference_id number="(.+)"# );
        }

        if ( $file_data[$i] =~ m#<date_of_patch year="# ) {
            ( $reference_details{date}{year}  ) =
                                       ( $file_data[$i] =~ m#year="(\d+)"# );
            ( $reference_details{date}{month} ) =
                                      ( $file_data[$i] =~ m#month="(\w+)"# );
            ( $reference_details{date}{day}   ) =
                                        ( $file_data[$i] =~ m#day="(\d+)"# );
            ( $reference_details{date}{time}  ) =
                               ( $file_data[$i] =~ m#time="([:\d]+)\shrs"# );
            ( $reference_details{date}{zone}  ) =
                                     ( $file_data[$i] =~ m#zone="(.+)"# );
        }

        if ( $file_data[$i] =~ m#<base_bugs>$# ) {
            $i++;
            while ( $file_data[$i] !~ m#</base_bugs>$# ) {
                my ( $number, $detail ) = ( $file_data[$i] =~
                           m#<bug number="(\d+)"\sdescription="(.+)"\s?/>$# );
                $reference_details{bug_list}{$number} = $detail;
                $i++;
            }
        }

        if ( $file_data[$i] =~ m#<required_components>$# ) {
            $i++;
            while ( $file_data[$i] !~ m#</required_components>$# ) {
                # Some component names may have spaces and digits and some
                # version numbers contain letters.
                my ( $name, $version ) = ( $file_data[$i] =~
                       m#<component internal_name="(.+)"\sversion="(.+)"\s# );
                my ( $option_flag ) = ( $file_data[$i] =~
                                                    m#opt_req="(\w)"\s?/>$# );
                $reference_details{components}{$name}{version} = $version;
                $reference_details{components}{$name}
                                                 {option_flag} = $option_flag;
                $Command::COMPONENT_VERSION = $version;
                $Command::COMPONENT_NAME = $name;
                opatchIO -> print_message ( {
                  message =>
                    "Component Name: $Command::COMPONENT_NAME\n" .
                    "Component Version: $Command::COMPONENT_VERSION\n"
                } );

                $i++;
            }
        }

        if ( $file_data[$i] =~ m#<os_platforms>$# ) {
            $i++;
            while ( $file_data[$i] !~ m#</os_platforms>$# ) {
                my ( $description, $number ) = ( $file_data[$i] =~
                           m#<platform name="(.+)"\sid="(\d+)"\s?/>$# );
                $reference_details{os_platforms}{$number} = $description;
                $i++;
            }
        }

        if ( $file_data[$i] =~ m#<executables>$# ) {
            $i++;
            while ( $file_data[$i] !~ m#</executables>$# ) {
                my ( $path ) = ( $file_data[$i] =~
                           m#<executable path="(.+)"\s?/>$# );
                my $name = &File::Basename::basename ( $path );
                $path =~ s/\%ORACLE_HOME\%/$Oracle_Home/ ;
                $reference_details{executable}{$name} = $path;
                $i++;
            }
        }

        if ( $file_data[$i] =~ m#<instance_shutdown># ) {
            $reference_details{shutdown} = 0;
            my ( $shutdown ) =
                 ( $file_data[$i] =~
                       m#<instance_shutdown>(\w+)</instance_shutdown>$# );
            # Now turn a "true" into a 1.
            if ( $shutdown eq "true" ) {
                $reference_details{shutdown} = 1;
            }
        }

        # This is for rolling RAC upgrades. If its rolling RAC installable
        # then the RAC doesn't need to be shutdown.
        if ( $file_data[$i] =~ m#<online_rac_installable># ) {
            my ( $rolling_RAC ) =
                 ( $file_data[$i] =~
             m#<online_rac_installable>(\w+)</online_rac_installable>$# );

            if ( $rolling_RAC eq "true" ) {
                $reference_details{RAC_shutdown} = 0;
                $Command::ROLLING_PATCH = 1;
            }
        }
    }  # Finished handling the inventory file.

    my $save_inventory_location = File::Spec -> catfile
                                         ( "$Oracle_Home",
                                           $this_class -> save_dir,
                                           $reference_details{reference_id},
                                           &File::Basename::basename (
                                           $full_inventory_path ) );
    if ( $OSNAME =~ m#Win32# ) {
        $save_inventory_location =
                        &File::Basename::dirname( $save_inventory_location );
    }

    # Used for rollback. Mainly to get the rolling RAC flag back.
    File::Copy::copy ( $full_inventory_path, $save_inventory_location ) ||
                ( $error_flag .= "Couldn't copy file from " .
                                 "$full_inventory_path to " .
                                 "$save_inventory_location.\n" ),

    # Check that the my $patch_inventory_location matches this patch.
    my $base_id = &File::Basename::basename ( $patch_inventory_location );
    if ( $base_id ne $reference_details{reference_id} ) {
        if ( ! $error_flag ) {
            $error_flag = "The patch directory $base_id doesn't match the patch id.";
        } else {
            $error_flag .= "\n\nThe patch directory $base_id doesn't match the patch id.";
        }
    }

    if ( $error_flag ) { $reference_details{error_flag} = $error_flag; }

    return ( \%reference_details );

}   # End of read_and_check_reference_details().


###############################################################################
#
# NAME   : validate_options
#
# PURPOSE: A method to validate the command line.
#
# INPUTS : $$ARG[1]{rh_command_line} - A reference to a hash that has the
#                                      command and options to be validated.
#          $$ARG[1]{rh_command_list} - A reference to a hash containg details
#                                      about the valid commands and options.
#
# OUTPUTS: Nothing directly. If validation fails, the attibute {invalid} is
#          defined for rh_command_line.
#
# NOTES  : 1. Check the given arguments (in $rh_command_line) against the list
#             of valid arguments for the command. If the argument has a
#             an attribute _validityChecks then the value from the command
#             line is validated by the appropriate method.
#          2. If you're trying to debug this (or follow it in a debugger) the
#             following arguments to 'keys' should help:
#             a. %${$$rh_command_list{$command}{_options}}, and
#             b. ${$${$$rh_command_list{$command}{_options}}{$type}}{$attribute}
#             but the loops to see all the values are left to the reader.
#          3. Assumes that the return value from any validation method is
#             a null string for success or a descriptive string of the failure.
#             This is stored in $error_string.
#
###############################################################################
sub validate_options {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_command_line = $$rh_arguments{rh_command_line};
    my $rh_command_list = $$rh_arguments{rh_command_list};

    my $command         = $$rh_command_line{command};
    my @arguments       = split ( " ", $$rh_command_line{arguments});


    my $upper_bound = scalar ( @arguments );
    for ( my $i = 0; $i < $upper_bound; $i++ ) {

        my $arg_name = $arguments[$i];

        my $invalid_flag = 1;

        # Skip global options, already validated.
        next if ( exists $$rh_command_list{$arg_name} );

        # Skip if it doesn't begin with a "-" symbol. Apply allows a single
        # string for OralceHome/ShipHome in the processesing.
        next if ( $arg_name !~ m#^-# );

        # Now check for bogus arguments. Because of perl's autovivification
        # this test needs be done first as the attribute would be created
        # by the validity check if it gets that far.

        foreach my $valid_argument
                ( keys %${$$rh_command_list{$command}{_options}} ) {
            if ( index ($valid_argument, $arg_name ) > -1 ) {
                $invalid_flag = 0;
            }
        }

        if ( $invalid_flag ) {
            if ( exists $$rh_command_line{invalid} ) {
                $$rh_command_line{invalid} .= "\nInvalid argument: $arg_name.";
            } else {
                $$rh_command_line{invalid} = "Invalid argument: $arg_name.";
            }
            next;
        }

        # Does this option, now valid, require a check on the value?
        # ( Yes Virgina, you can split hashes like this to avoid line wraps. )
        if ( ${$${$$rh_command_list{$command}{_options}}
                                           {$arg_name}}{_validityChecks} ) {
            my $check_type = ${$${$$rh_command_list{$command}{_options}}
                                                        {$arg_name}}{_type};
            my $rule_set =   ${$${$$rh_command_list{$command}{_options}}
                                              {$arg_name}}{_validityChecks};

            # Out of hubris I've assumed all validation methods to start
            # with "validate_".
            my $validate_method = "validate_" . $check_type;
            my $error_string = $this_class -> $validate_method ( {
                                        item_to_validate => $arguments[$i+1],
                                        validation_rules => $rule_set } );
            $i++;
            if ( $error_string ) {
                $$rh_command_line{invalid} .= $error_string;
            }
        }
    }
}   # End of validate_options().

##############################################################################
#
# Name : is_noop_patch
#
# Inputs:
#    files touched
#
# Output:
#    1 - Patch is a noop patch
#    0 - Patch is not a noop patch
#
##############################################################################
sub is_noop_patch {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_file_details = $$rh_arguments{rh_file_details};

    my $noop_patch = 1;

    my $make_actions = 0;

    foreach my $action_type ( keys %$rh_file_details ) {

        if ( $action_type eq &Apply::AR_STRING ) {
          $noop_patch = 0;

        } elsif ( $action_type eq &Apply::JAR_STRING ) {
          $noop_patch = 0;

        } elsif ( $action_type eq &Apply::COPY_STRING ) {
          $noop_patch = 0;

        } elsif ( $action_type eq &Apply::MAKE_STRING ) {
          $make_actions = scalar ( @{$$rh_file_details{$action_type}} );

        }
    }

    # This is a small hack. This is because of the cheat in build_files_details
    # method. Hence we depend on the size of tha array.
    if( $make_actions > 0) {
       return 0;
    }

    return $noop_patch;

}
###############################################################################
#
# NAME   : dummy
#
# PURPOSE: A place to use variable a second time to suppress warnings
#          that are bogus.
#
# INPUTS : None.
#
# OUTPUTS: None.
#
###############################################################################
sub dummy {

    my $dummy = $Command::do_not_update_inventory;
}   # Dummy.

#############################################################################
# Perl packages need to return a true value.
1;
#############################################################################
