#
# Copyright (c) 2001, 2003, Oracle Corporation.  All rights reserved.  
#
#    NAME
#    AttachHome.pm
#
#    DESCRIPTION
#    This file contains the methods for the attach command to opatch.
#
#    NOTES
#    1. Philosophically this is really belongs to OUI. However this
#       needs to be made available in a shorter time period than the
#       OUI development cycle. So it's being made available via the
#       patch tool.
#
#    BUGS
#    * <if you know of any bugs, put them here!>
#
#    MODIFIED   (MM/DD/YY)
#    phnguyen    12/30/04 - Take out 'attach' help
#    shgangul    09/11/03 - Logging changes for 10G 
#    phnguyen    08/28/03 - phnguyen_installable 
#    phnguyen    04/14/03  Check if ORACLE_HOME exists
#                          Warn if it's a symbolic link.
# 
#       daevans  02/11/03 - daevans_relocate_directories
#    daevans     01/31/03  Remove some duplicate declarations.
#    daevans     12/24/02  First release.
#    daevans     12/10/02  Initial code.
#
##########################################################################

###############################################################################
###############################################################################
#
#  ------------------------ INITIALIZATION ------------------------
#
###############################################################################
###############################################################################

package AttachHome;
@ISA = ("Command");

######
#
# Standard modules:
#
use English;         # Let us say "$CHILD_ERROR" instead of "$?", etc.
use strict;          # Enforce strict variables, refs, subs, etc.

use Cwd();
use File::Basename();

###############################################################################
#
#  ------------------------ DATA STRUCTURES ------------------------
#
###############################################################################
#
# Global data structures are defined in Command.pm.
#
###############################################################################


###############################################################################
#
#  ------------------------ SUBROUTINES ------------------------
#
###############################################################################

###############################################################################
#
# NAME   : abdomen
#
# PURPOSE: Run the command.
#
# INPUTS : $$ARG[command]   - The command that is to be executed.
#          $$ARG[arguments] - The arguments to the command.
#
# OUTPUTS: NONE
#
# NOTES  : 1. This functions as a main body. Since this is OO perl and perl
#             reads in the main body of the packages during initialization
#             to avoid any side effects the functionality of the body has
#             moved to this method.
#          2. The name abdomen was chosen as its a simile for main body and
#             it'll hopefully remain near the top of the file since it starts
#             with "ab". Unless someone wants to really wreck this scheme.
#          3. This can be invoked directly of via the "apply" command when
#             it must force a rollback of conflicting patches. When invoked
#             by "apply" most of the values will have been checked by the
#             "apply" command so they shouldn't fail at this point. If
#             invoked directly by a user then the checks should fail. In
#             either event the program should (and does) the same actions.
#          4. Point 3 is why caller() is used so we don't keep rechecking
#             the database state.
#
###############################################################################
sub abdomen {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];


    my $command   = $$rh_arguments{command};
    my $arguments = $$rh_arguments{arguments};
    my $rh_command_list = $$rh_arguments{rh_command_list};

    my @arguments = split ( " ", $arguments ) ;

    # Now parse the arguments for this command.
    my $rh_arg_check =
            $this_class -> parse_arguments( { ra_arguments => \@arguments } );

    my $error_flag = $$rh_arg_check{error_flag};

    # For any errors display the problem, give command usage then exit.
    if ( $error_flag ) {
        opatchIO->print_message ( {message => "$error_flag"} );
        $this_class -> display_help( { command => $command,
                                       rh_command_list => $rh_command_list } );
        exit 0;
    }

    my $Oracle_Home = $$rh_arg_check{Oracle_Home};
    my $home_name   = $$rh_arg_check{home_name};
    my $node_list   = $$rh_arg_check{node_list};

    if ( $OSNAME =~ m#Win32# ) { $Oracle_Home = lc ( $Oracle_Home ); }

    my $invoked_location  = &File::Basename::dirname($PROGRAM_NAME);

    my $rh_OUI_file_names = $this_class -> build_required_OUI_filenames ( {
                                              Oracle_home => $Oracle_Home } );

    my $path_to_java        = $$rh_OUI_file_names{path_to_java};

    if ( $$rh_OUI_file_names{error_flag} ) {
        opatchIO -> print_message_and_die ( {
                              message => $$rh_OUI_file_names{error_flag} } );
    }

    # Now set-up the Java class for the inventory read.
    my $rh_java_paths = $this_class -> make_java_paths ( {
                                Oracle_Home       => $Oracle_Home,
                                rh_OUI_file_names => $rh_OUI_file_names } );

    my $class_path = $$rh_java_paths{class};
    my $oui_path   = $$rh_java_paths{oui_path};

    # Now set-up the call and print the results. Not using IPC as
    # this is trying to keep it simple.
    my $class_to_run = "AttachHome";
    my $system_command = "$path_to_java $class_path " .
                             "$Command::inventory_location_ptr " .
                             "$class_to_run $Oracle_Home $home_name " .
                             "$node_list";

    opatchIO -> print_message ( { message => "Updating \$ORACLE_HOME " .
                                             "details.\n" } );

    my $sys_call_result = qx/$system_command/;

    opatchIO -> print_message ( { message => $sys_call_result } );

    # What sort of errors can be expected? Currently all output is
    # returned to the user.

    return ( 0 );

}   # End of abdomen().

###############################################################################
#
# NAME   : command_details
#
# PURPOSE: Provide the details about the command.
#
# INPUTS : $$ARG[rh_detail] - A reference to a hash structure of the class
#                             Command to store the attributes in.
#
# OUTPUTS: NONE
#
# NOTES  : 1. The return values are in hash so the caller will get the new
#             details when this subroutine is finished.
#
###############################################################################
sub command_details {

    # Because this was called in an OO manner the first argument is the
    # class name.
    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_detail    = $$rh_arguments{rh_detail};

    # Put options here for easy reference.
    my @options = ( "invPtrLoc", "name", "nodes", "oh" );

    #my $command = lc ( $this_class );
    my $command = "attach";

    $$rh_detail{$command}  = Command -> new_command();

    $$rh_detail{$command}{_description} = "attach";

    $$rh_detail{$command}{_helpText}    =
               # "Attach an installed database to a central inventory.",
               "Command is no longer supported.  Please use OUI instead.",

    # $$rh_detail{$command}{_options} =
    #           $this_class -> build_option_details ( {
    #                      rh_options     => $$rh_detail{$command}{_options},
    #                      ra_option_list => \@options } );

    # $this_class -> add_options_to_command_description
    #                             ( { rh_command => $$rh_detail{$command} } );

    return ( 0 );

}   # End of command_detail().

##############################################################################
#
# NAME   : parse_arguments
#
# PURPOSE: A check, validation and variable assignments based on the
#          the arguments for this sub-command.
#
# INPUTS : $$ARG[1]{rh_option} - A reference to a hash where the details about
#                                the option are to be stored.
#
# OUTPUTS: A reference to a hash.
#
# NOTES  : 1. Well parse_arguments() is used for the command line check. But
#             at that point the class is the super-class: Command. So we can
#             re-use the name space here for specific processing for the class
#             at the correct time.
#          2. I know the "*_arguments" will confuse people so let me explain it
#             at this point in the module. The command line is parsed initially
#             at the top level (by parse_arguments) that retruns a string. The
#             string contains the arguments. This is turned into an array. The
#             array is an argument to this method. Since the argument to this
#             method is a the local arguments are rh_arguments. The initial
#             command line items, now an array, is located at
#             $$rh_arguments{ra_arguments}.
#          3. The above may be confusing but it keeps all the calls in the
#             same format: a single hash contains all the parameters for any
#             method.
#          4. This is the first time a patch program has interacted with the
#             inventory so only rolling back one patch at a time to keep it
#             robust.
#
###############################################################################
sub parse_arguments {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $ra_arguments = $$rh_arguments{ra_arguments};

    my $array_size   = scalar( @$ra_arguments );
    my $error_flag   = "";
    my $noop_flag    = 0;
    my $Oracle_Home  = "";
    my $home_name    = "";
    my $node_list    = "";
    my $node_flag    = 0;

    for ( my $i = 0; $i < $array_size; $i++ ) {
        if ( $$ra_arguments[$i] =~ m#^-oh$# ) {
            if ( ! $Oracle_Home ) {
                $Oracle_Home = $$ra_arguments[$i +1];
                $node_flag = 0;
                $i++;
            } else {
                $error_flag .= "Argument -oh appears more than once.\n";
                $i++;
            }

        } elsif ( $$ra_arguments[$i] =~ m#^-na(me)?$# ) {
            if ( ! $home_name ) {
                $home_name = $$ra_arguments[$i +1];
                $node_flag = 0;
                $i++;
            } else {
                $error_flag .= "Argument -name is used more than once.\n";
                $i++;
            }

        } elsif ( $$ra_arguments[$i] =~ m#^-no(des)?$# ) {
            $node_list .= $$ra_arguments[$i +1] . " ";
            $node_flag = 1;
            $i++;

        } elsif ( $$ra_arguments[$i] =~ m#^-i(nvPtrLoc)?$# ) {
            my $inv_ptr_file = $$ra_arguments[$i + 1];
            if ( &File::Basename::basename ( $inv_ptr_file ) ne
                                                          "oraInst.loc" ) {
                $inv_ptr_file = File::Spec -> catfile ( "$inv_ptr_file" , 
                                                            "oraInst.loc" );
            }

            $Command::inventory_location_ptr = 
                    "-Doracle.installer.invPtrLoc=" . $$ra_arguments[$i + 1];

            $i++;
            $node_flag = 0;

        } elsif ( $node_flag ) {
            $node_list .= $$ra_arguments[$i] . " ";

        } elsif ( $$ra_arguments[$i] =~ m#^-no_op$# ) {
            $noop_flag = 1;
            $node_flag = 0;

        } else {
            $error_flag .= "Problem with the command line arguments.\n";
        }
    }

    # If $Oracle_Home isn't set get it from the environment. Need to check
    # for empty string to cover Microsoft's OSs.
    if ( ( ( ! $Oracle_Home ) && ( $ENV{ORACLE_HOME} ) ) ||
         ( $Oracle_Home eq "" ) ) {
        $Oracle_Home = $ENV{ORACLE_HOME};
        if ( ! -d $Oracle_Home || ! -e $Oracle_Home ) {
            $error_flag .= "Oracle's home is not a directory: \n" .
                           "$Oracle_Home.\n";
        }
    }

    # If there is still no $Oracle_Home raise an error. IF there is an
    # $Oracle_Home and it's MicroSoft OS lowercase the string.
    if ( ! $Oracle_Home ) {
        $error_flag .= "There is no \$ORACLE_HOME to remove the patch from.\n";
    } elsif ( $OSNAME =~ m#Win32# ){
      $Oracle_Home = lc ($Oracle_Home );
    } else {
        my $is_link = $this_class -> is_symbolic_link ( $Oracle_Home );
        if ( $is_link == 1 ) {
            my $real_OH = $this_class->resolve_symbolic_link($Oracle_Home);

            opatchIO->print_message ( {message => "\n-----------------------------------------\n"} );
            opatchIO->print_message ( {message => "\nWARNING: Oracle_Home is a symbolic link:\n"} );
            opatchIO->print_message ( {message => "$Oracle_Home\n"} );
            opatchIO->print_message ( {message => "--->\n"} );
            opatchIO->print_message ( {message => "$real_OH"} );
            opatchIO->print_message ( {message => "\n-----------------------------------------\n"} );
        }
    }

    # If there is an $Oracle_Home strip any trailing "/" or "\" from it.
    # This upsets the OUI APIs for some reason.
    if ( $Oracle_Home =~ m#[\\/]$# ) {
        ( $Oracle_Home ) = ( $Oracle_Home =~ m#(.+)[\\/]\s?$# );
    }

    # Now make sure mandatory args have been supplied.
    if ( ! $home_name ) {
        $error_flag .= "The string to use for ORACLE_HOME_NAME " .
                       "was not supplied. -name is mandatory.\n";
    }

    my %return_values = ();
    $return_values{error_flag}  = $error_flag;
    $return_values{noop_flag}   = $noop_flag;
    $return_values{Oracle_Home} = $Oracle_Home;
    $return_values{home_name}   = $home_name;
    $return_values{node_list}   = $node_list;

    return ( \%return_values );

}    # End of parse_arguments().


##############################################################################
#
# NAME   : option_name
#
# PURPOSE: Provide the details the patch id option.
#
# INPUTS : $$ARG[rh_option] - A reference to a hash where the details about
#                             the option is to be stored.
#
# OUTPUTS: NONE
#
# NOTES  : 1. The return values are in hash so the caller will get the new
#             details when this subroutine is finished.
#
###############################################################################
sub option_name {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

    $$rh_details{_description}  = "-na[me] <ORACLE_HOME_NAME>";
    $$rh_details{_helpText}     =    
                                 "The name to use when attaching the " .
                                 "product back to the \ndefault central " . 
                                 "inventory";
    $$rh_details{_required}     = "yes";
    $$rh_details{_type}         = "string";

}   # End of option_name().

##############################################################################
#
# NAME   : option_nodes
#
# PURPOSE: Provide the details the patch id option.
#
# INPUTS : $$ARG[rh_option] - A reference to a hash where the details about
#                             the option is to be stored.
#
# OUTPUTS: NONE
#
# NOTES  : 1. The return values are in hash so the caller will get the new
#             details when this subroutine is finished.
#
###############################################################################
sub option_nodes {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

    $$rh_details{_description}  = "-no[des] <node1 node2 ... >";
    $$rh_details{_helpText}     = "The nodes to register in a RAC " .
                                  "environment when attacting \na " .
                                  "product back to the default central " . 
                                  "inventory";
    $$rh_details{_required}     = "no";
    $$rh_details{_type}         = "string";

}   # End of option_nodes().

#
# PURPOSE: Update the products inventory by removing details of the
#          patch.
#
# INPUTS : $$ARG[1]{fh_log_file}      - The file handle for the log file.
#          $$ARG[1]{Oracle_Home}    - The area that is to be patched.
#          $$ARG[1]{patch_id}       - The patch identification string.
#          $$ARG[1]{rh_OUI_file_names}
#                                   - A list of files that are needed when
#                                     invoking OUI through the JVM.
#
# OUTPUTS: $error_messages          - A string containing any errors that
#                                     were encountered.
#
# NOTES  :
#
###############################################################################
sub update_inventory {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file       = $$rh_arguments{fh_log_file};
    my $Oracle_Home       = $$rh_arguments{Oracle_Home};
    my $patch_id          = $$rh_arguments{patch_id};
    my $rh_OUI_file_names = $$rh_arguments{rh_OUI_file_names};

    my %check_results  = ();
    my $error_messages = "";

    my $path_to_java = $$rh_OUI_file_names{path_to_java};

    # First set-up the Java class for the inventory read.
    my $rh_java_paths = $this_class -> make_java_paths ( {
                                Oracle_Home       => $Oracle_Home,
                                rh_OUI_file_names => $rh_OUI_file_names } );

    my $class_path = $$rh_java_paths{class};
    my $oui_path   = $$rh_java_paths{oui_path};

    opatchIO -> print_message ( { message => "Removing details of patch ".
                                             "$patch_id from the " .
                                             "inventory.\n" } );

    # Now set-up the call and capture the results. Quoting $file_names to
    # make the $file_names a single argument to the Java program.
    # This may need some tuning.
    my $class_to_run = "RemovePatchDetails";
    my $program_name = &File::Basename::basename( $PROGRAM_NAME );
    my $version_num  = $this_class -> version;
    my $system_command = "$path_to_java $Command::inventory_location_ptr " .
                         "$class_path $class_to_run $oui_path " .
                         "$Oracle_Home $program_name $version_num " .
                         "$patch_id";

    my $sys_call_result = qx/$system_command/;
    chomp $sys_call_result;
    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "System Command:\n\t" .
                                          $system_command . "\nResult:\n\t" .
                                          "$sys_call_result\n" } );
    my $status = $CHILD_ERROR >> 8;

    if ( $status ) {
            $error_messages .= "Error when updating the inventory:\n" .
                               "[ $sys_call_result ]\n";
    }

    return ( $error_messages );

}   # End of update_inventory().

###############################################################################
#
# NAME   : validate_options
#
# PURPOSE: A method to validate the command line.
#
# INPUTS : $$ARG[1]{rh_command_line} - A reference to a hash that has the
#                                      command and options to be validated.
#          $$ARG[1]{rh_command_list} - A reference to a hash containg details
#                                      about the valid commands and options.
#
# OUTPUTS: Nothing directly. If validation fails, the attibute {invalid} is
#          defined for rh_command_line.
#
# NOTES  : 1. Check the given arguments (in $rh_command_line) against the list
#             of valid arguments for the command. If the argument has a
#             an attribute _validityChecks then the value from the command
#             line is validated by the appropriate method.
#          2. If you're trying to debug this (or follow it in a debugger) the
#             following arguments to 'keys' should help:
#             a. %${$$rh_command_list{$command}{_options}}, and
#             b. ${$${$$rh_command_list{$command}{_options}}{$type}}{$attribute}#             but the loops to see all the values are left to the reader.
#          3. Assumes that the return value from any validation method is
#             a null string for success or a descriptive string of the failure.
#             This is stored in $error_string.
#
###############################################################################
sub validate_options {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_command_line = $$rh_arguments{rh_command_line};
    my $rh_command_list = $$rh_arguments{rh_command_list};

    my $command         = $$rh_command_line{command};
    my @arguments       = split ( " ", $$rh_command_line{arguments});


    my $upper_bound = scalar ( @arguments );
    for ( my $i = 0; $i < $upper_bound; $i++ ) {

        my $arg_name = $arguments[$i];
        my $invalid_flag = 1;

        # Skip global options, already validated.
        next if ( exists $$rh_command_list{$arg_name} );

        # Skip if it doesn't begin with a "-" symbol. Apply allows a single
        # string for OralceHome/ShipHome in the processesing.
        next if ( $arg_name !~ m#^-# );

        # Now check for bogus arguments. Because of perl's autovivification
        # this test needs be done first as the attribute would be created
        # by the validity check if it gets that far.

        foreach my $valid_argument
                ( keys %${$$rh_command_list{$command}{_options}} ) {
            if ( index ($valid_argument, $arg_name ) > -1 ) {
                $invalid_flag = 0;
            }
        }

        if ( $invalid_flag ) {
            if ( exists $$rh_command_line{invalid} ) {
                $$rh_command_line{invalid} .= "\nInvalid argument: $arg_name.";
            } else {
                $$rh_command_line{invalid} = "Invalid argument: $arg_name.";
            }
            next;
        }

        # Does this option, now valid, require a check on the value?
        # ( Yes Virgina, you can split hashes like this to avoid line wraps. )
        if ( ${$${$$rh_command_list{$command}{_options}}
                                           {$arg_name}}{_validityChecks} ) {
            my $check_type = ${$${$$rh_command_list{$command}{_options}}
                                                        {$arg_name}}{_type};
            my $rule_set =   ${$${$$rh_command_list{$command}{_options}}
                                              {$arg_name}}{_validityChecks};

            # Out of hubris I've assumed all validation methods to start
            # with "validate_".
            my $validate_method = "validate_" . $check_type;
            my $error_string = $this_class -> $validate_method ( {
                                        item_to_validate => $arguments[$i+1],
                                        validation_rules => $rule_set } );
            $i++;
            if ( $error_string ) {
                $$rh_command_line{invalid} .= $error_string;
            }
        }
    }
}   # End of validate_options().

#############################################################################
# Perl packages need to return a true value.
1;
#############################################################################
