#
# Copyright (c) 2001, 2005, Oracle. All rights reserved.  
#
#    NAME
#    LsInventory.pm
#
#    DESCRIPTION
#    This file contains the methods for the lsinventory command to opatch.
#
#    NOTES
#
#    BUGS
#    * <if you know of any bugs, put them here!>
#
#    MODIFIED   (MM/DD/YY)
#    vsriram     12/15/05 - Bug 4883881: Look for jre/bin/java, if not found then use find.
#    vsriram     12/01/05 - Error out if -local or -local_node option is specified.
#    phnguyen    05/06/05 - Process OPATCH_IS_SHARED
#    shgangul    01/03/05 - Remove no_verbose for system command 
#    phnguyen    12/03/04 - -local and -local_node can be used
#                                together
#    phnguyen    09/24/04 - error out if -no_inventory
#    phnguyen    07/22/04 - call create_save_area() to mkdir .patch_storage
#                              on virgin home
#    phnguyen    04/02/04 - use -oh if supplied
#                           do not print error if no log file on virgin home
#    phnguyen    02/02/04 - work-around for Active State Perl
#    phnguyen    01/30/04 - remove extra "" in Win
#    phnguyen    12/24/03 - remove redundant debug setting
#    phnguyen    12/15/03 - pass in -Dopatch.debug & cluster debug if OPATCH_DEBUG is set to TRUE
#    phnguyen    12/01/03 - pass in -Dopatch.debug & cluster debug if OPATCH_DEBUG is set to TRUE
#    shgangul    11/25/03 - java, library path problems 
#    phnguyen    11/18/03 - all Java classes are in opatch package
#                           so we need to run as opatch/LsInventory or
#                           opatch/LsHome (the / is also used for Windows)
#    phnguyen    10/10/03 - touch up help 
#    phnguyen    10/08/03 - Add -jre option
#    shgangul    09/11/03 - Logging changes for 10G 
#    shgangul    08/13/03 - Code changes for BUG 2899335 
#    phnguyen 07/31/03 - 
#    phnguyen 04/14/03     Check if ORACLE_HOME exists
#                          Warn if it's a symbolic link
#  
#       daevans  02/11/03 - daevans_relocate_directories
#    daevans     12/06/02  Add support for installations that used the
#                          unsupported "invPtrLoc" flag.
#    daevans     10/26/02  Fix for bug 2637212.
#    praghuna    09/01/02  Remove path_to_oI_loc check. Variable never used.
#    daevans     06/01/02  Many code changes. Stable check-in.
#    daevans     03/12/01  Inital code.
#
##########################################################################

###############################################################################
###############################################################################
#
#  ------------------------ INITIALIZATION ------------------------
#
###############################################################################
###############################################################################

package LsInventory;
@ISA = ("Command");

######
#
# Standard modules:
#
use English;         # Let us say "$CHILD_ERROR" instead of "$?", etc.
use strict;          # Enforce strict variables, refs, subs, etc.

use File::Basename();
use File::Spec();

# will be set to 1 if users specify -detail
use constant SHOW_DETAIL        => 0;

###############################################################################
#
#  ------------------------ DATA STRUCTURES ------------------------
#
###############################################################################
#
# Global data structures are defined in Command.pm.
#
###############################################################################


###############################################################################
#
#  ------------------------ SUBROUTINES ------------------------
#
###############################################################################

###############################################################################
#
# NAME   : abdomen
#
# PURPOSE: Run the command.
#
# INPUTS : $$ARG[1]{command}   - The command that is to be executed.
#          $$ARG[1]{arguments} - The arguments to the command.
#
# OUTPUTS: NONE
#
# NOTES  : 1. This functions as a main body. Since this is OO perl and perl
#             reads in the main body of the packages during initialization
#             to avoid any side effects the functionality of the body has
#             moved to this method.
#          2. The name abdomen was chosen as its a simile for main body and
#             it'll hopefully remain near the top of the file since it starts
#             with "ab". Unless someone wants to really wreck this scheme.
#
###############################################################################
sub abdomen {

    # Set up debug
    my $debug = $ENV{OPATCH_DEBUG};
    if ( $debug eq "TRUE" ) {
      $Command::DEBUG = "-DTRACING.ENABLED=TRUE -DTRACING.LEVEL=2 -Dopatch.debug=true ";
    } else {
      $Command::DEBUG = "";
    }
    $debug = $Command::DEBUG;

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];
   
    my $command   = $$rh_arguments{command};
    my $arguments = $$rh_arguments{arguments};
    my $rh_command_list = $$rh_arguments{rh_command_list};

    # Set the operation, so that it can be accessed from anywhere
    $Command::opatch_operation = "lsinventory";

    my @arguments = split ( " ", $arguments ) ;

    # Now parse the arguments for this command.
    my $rh_arg_check =
            $this_class -> parse_arguments( { ra_arguments => \@arguments } );

    my $error_flag = $$rh_arg_check{error_flag};

    # For any errors display the problem, command usage then exit.
    if ( $error_flag ) {
        opatchIO->print_message ( {message => "$error_flag"} );
        $this_class -> display_help( { command => $command,
                                       rh_command_list => $rh_command_list } );
        
        opatchIO->print_message_and_die({
          exit_val => $this_class->ERROR_CODE_COMMAND_LINE_ARGUMENT,
          message=>
            "Error parsing arguments\n"
	      });
    }
    
    # Process -retry & -delay options
    my $badRetry = $this_class -> isBadRetryDelaySemantics();
    if ( $badRetry ) {
       opatchIO->print_message_and_die({
          exit_val => $this_class->ERROR_CODE_COMMAND_LINE_ARGUMENT,
          message=> $this_class->RETRY_ERROR_MESSAGE
	      });
    }

    my $Oracle_Home       = $$rh_arg_check{Oracle_Home};
    if ( $OSNAME =~ m#Win32# ) { $Oracle_Home = lc ( $Oracle_Home ); }

    my $report_all_flag   = $$rh_arg_check{report_flag};
    my $invoked_location  = &File::Basename::dirname($PROGRAM_NAME);

    # Processing -jre option
    my $target = "";
    my $start_point = "";
    my $result = "";
       
    #if ($Command::LOCAL_NODE_ONLY && $Command::LOCAL_NODE_NAME_MODE) { 
    #   opatchIO->print_message_and_die({
    #      exit_val => $this_class->ERROR_CODE_COMMAND_LINE_ARGUMENT,
    #      message=> "-local and -local_node <node name> cannot be used together.\n"
    #	      });
    #  
    #}

   if ( $Command::JRE_LOC ne "" ) {
      $start_point = $Command::JRE_LOC;
      $target  = "java";
      if ( $OSNAME =~ m#Win32# ) {
         $target = "java.exe";
      }
      opatchIO -> print_message ( { message => "-jre specified, need to find " .
                                               "executable java from $Command::JRE_LOC\n"
                                } );

      my $default_path = File::Spec -> catfile ($start_point,
                                      "bin", $target);

      # Bug 4883881 : First look into <user specified path>/bin/java, if not
      # then use find and look for the path as usual.
      if ( -x $default_path ) {
            $result = $default_path;
      } else {
            $result = $this_class -> find_executable_file ( { start_point => $start_point,
                                                   target => $target } );
      }
      
      if ( $result ne "" ) {
        $Command::JRE_LOC = $result;
        opatchIO -> print_message ( { message => "JAVA set to $result" } );                 
      }
    }
    $this_class -> get_os_id ( { Oracle_home => $Oracle_Home } );
    my $os_id = $Command::OS_ID;
    my $rh_OUI_file_names = $this_class -> build_required_OUI_filenames ( {
                                              Oracle_home => $Oracle_Home } );

    my $path_to_java        = $$rh_OUI_file_names{path_to_java};

    if ( $$rh_OUI_file_names{error_flag} ) {
        opatchIO -> print_message_and_die ( {
                              message => $$rh_OUI_file_names{error_flag} } );
    }

    # Now set-up the Java class for the inventory read.
    my $rh_java_paths = $this_class -> make_java_paths ( {
                                Oracle_Home       => $Oracle_Home,
                                rh_OUI_file_names => $rh_OUI_file_names } );

    my $class_path = $$rh_java_paths{class};
    my $oui_path   = $$rh_java_paths{oui_path};
    
    my $detail = "FALSE";
    if ( $LsInventory::SHOW_DETAIL ) {
       $detail = "TRUE";
    }
    my $version_num  = $this_class -> version;
    my $program_name = &File::Basename::basename($PROGRAM_NAME);

    # Setup the logging area.
    # (LsInventory doesn't have patch ID)
    my $patch_id = ""; 
    my $save_area_errors = $this_class -> create_save_area ( {
                                              patch_id    => $patch_id,
                                              Oracle_Home => $Oracle_Home } );

    my $rh_log_creation_details = $this_class -> create_logging_file ( {
                                              patch_id    => $patch_id,
                                              Oracle_Home => $Oracle_Home } );

    my $log_failure   = $$rh_log_creation_details{error_flag};
    my $fh_log_file   = $$rh_log_creation_details{fh_log_file};
    my $log_file_name = $$rh_log_creation_details{log_file_name};
    $OUTPUT_AUTOFLUSH = 1;

    if ( $log_failure ) {
      
        $this_class -> free_the_lock_file ( {
                                        Oracle_Home     => $Oracle_Home,
                                        patch_id        => $patch_id } );
        
        # Do not create the .patch_storage directory on virgin home.
        # Suppress error.
        # opatchIO -> print_message ( {
        #              message  => "Problems with creating the log file:\n " .
        #                                      "$log_failure" } );
    }

    # Process env. var. OPATCH_IS_SHARED
    $Command::OPATCH_IS_SHARED = $ENV{"OPATCH_IS_SHARED"} || "";
  
    my $is_shared_flag = "";
    # User wanna turn off CFS detection?
    #    OPATCH_IS_SHARED = true   -> CFS=true, no detect
    #                       false        =false, no detect
    #                       detect       =<detected value>, yes detect
    #                       not set      =false, no detect
    
    if ($Command::OPATCH_IS_SHARED eq "TRUE") {
        $is_shared_flag = "-Dopatch.is_shared=true";
		    opatchIO -> print_message ( {
                   f_handle => $fh_log_file,
               		 message => "No CFS detection, CFS set to true."
              	  });        
    } elsif ( $Command::OPATCH_IS_SHARED eq "FALSE" ) {
        $is_shared_flag = "-Dopatch.is_shared=false";
		    opatchIO -> print_message ( {
                   f_handle => $fh_log_file,
               		 message => "No CFS detection, CFS set to false."
              	  });            
    } elsif ( $Command::OPATCH_IS_SHARED eq "DETECT" ) {
        $is_shared_flag = "-Dopatch.is_shared=detect";
		    opatchIO -> print_message ( {
                   f_handle => $fh_log_file,
               		 message => "Perform CFS detection, CFS set to detected value."
              	  });        
    } elsif ( $Command::OPATCH_IS_SHARED eq "" ) {
        $is_shared_flag = "";
        # Env. variable OPATCH_IS_SHARED is not set, so ignore it.
    } else {
        $is_shared_flag = "";
		    opatchIO -> print_message ( {
                   f_handle => $fh_log_file,
               		 message => "Unknown OPATCH_IS_SHARED value, no CFS detection, CFS set to false."
              	  });        
    } 
    
    my $class_to_run = "opatch/LsInventory";

    my $local_node_only_flag = "";
    if ($Command::LOCAL_NODE_ONLY) {
      $local_node_only_flag = "-Dopatch.local_node_only";
    }
    
    my $local_node_name_flag = "";
    if ($Command::LOCAL_NODE_NAME_MODE) {
      $local_node_name_flag = "-Dopatch.local_node_name=$Command::LOCAL_NODE_NAME"; 
    }

    if ( $report_all_flag ) {
        # Report what we can find.

        # Now set-up the call and print the results. Not using IPC as
        # this is trying to keep it simple.
 
        my $class_to_run = "opatch/LsHome";
        my $system_command = join(' ', $path_to_java, $class_path,
                      $local_node_only_flag,
                      $is_shared_flag,
                      $local_node_name_flag,
                      $Command::inventory_location_ptr,
                      $Command::RETRY_OPTION,
                      $Command::DELAY_OPTION,
                      $debug,
                      $class_to_run, $oui_path,
                      qq(").$Oracle_Home.qq("),
			          $program_name, $version_num, $detail);
        $system_command = $this_class->getDependentPerlCommand({ system_command => $system_command });

        $this_class->warn_about_retry();
        
        opatchIO -> print_message ( { message => "$system_command\n" .
                                                 "Retrieving \$ORACLE_HOME details" 
                                   } );

        opatchIO -> print_message ( { f_handle => $fh_log_file,
                    message => "Invoking command \"$system_command\"" } );

        my $sys_call_result = qx/$system_command/;

        # Java program will format and output to stdout
        opatchIO -> print_message_noverbose ( { f_handle => $fh_log_file,
                                      message => $sys_call_result } );
        
        my $error = $this_class->catch_and_print_Java_error({
          sys_call_result => $sys_call_result
        });
        if ($error) {
          opatchIO->print_message_and_die({
            exit_val => $this_class->ERROR_CODE_INVENTORY_PROBLEM,
            message=>
              "Can not get a list of inventory on this home.\n"
          });
        }
    } else {
        # Report on what is in a given $ORACLE_HOME. This is similar to
        # to the block above.
        
        my $system_command = join(' ', $path_to_java, $class_path,
                      $local_node_only_flag,
                      $is_shared_flag,
                      $local_node_name_flag,
                      $Command::inventory_location_ptr,
                      $Command::RETRY_OPTION,
                      $Command::DELAY_OPTION,
                      $debug,
                      $class_to_run, $oui_path,
                      qq(").$Oracle_Home.qq("),
			          $program_name, $version_num, $detail);
        $system_command = $this_class->getDependentPerlCommand({ system_command => $system_command });

        $this_class->warn_about_retry();
          
        opatchIO -> print_message( { f_handle => $fh_log_file,
                    message => "Invoking command \"$system_command\"" } );
                    
        my $sys_call_result = "";
        $sys_call_result = qx/$system_command/;
        my $status = $this_class -> get_child_error ( {
                CHILD_ERROR     => $CHILD_ERROR
        } );

        # Java program will format and output to stdout
        opatchIO -> print_message_noverbose ( { f_handle => $fh_log_file,
                                      message => "Result: \n\n$sys_call_result" } );

        my $error = $this_class->catch_and_print_Java_error({
          sys_call_result => $sys_call_result
        });
        if ($error) {
          opatchIO->print_message_and_die({
            exit_val => $this_class->ERROR_CODE_INVENTORY_PROBLEM,
            message=>
              "Can not get a list of inventory on this home.\n"
          });
        }
   }

   opatchIO->print_message_and_die({
            exit_val => $this_class->ERROR_CODE_NO_ERROR,
            message =>
               "LsInventory done"
   });
                                       
}   # End of abdomen().

###############################################################################
#
# NAME   : command_details
#
# PURPOSE: Provide the details about the command.
#
# INPUTS : $$ARG[1]{rh_detail} - A reference to a hash structure of the class
#                                Command to store the attributes in.
#
# OUTPUTS: NONE
#
# NOTES  : 1. The return values are in hash so the caller will get the new
#             details when this subroutine is finished.
#
###############################################################################
sub command_details {

    # Because this was called in an OO manner the first argument is the
    # class name.
    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_detail    = $$rh_arguments{rh_detail};
  
    # Put options here for easy reference.
    my @options = ( "all", "delay", "detail", "invPtrLoc", "jre", 
                    "local", "local_node", "oh", "retry", );

    my $command = lc ( $this_class );
    $$rh_detail{$command}  = Command -> new_command();

    $$rh_detail{$command}{_description} = 
               "lsinventory";

    $$rh_detail{$command}{_helpText}    = 
               "List the inventory for a particular \$ORACLE_HOME or display " .
               "all installations that can be found.";

    $$rh_detail{$command}{_options} =
               $this_class -> build_option_details ( { 
                          rh_options     => $$rh_detail{$command}{_options},
                          ra_option_list => \@options } );

    $this_class -> add_options_to_command_description
                                ( { rh_command => $$rh_detail{$command} } );

    return ( 0 );

}   # End of command_detail().

###############################################################################
#
# NAME   : option_all
#
# PURPOSE: Insert the details for reporting on all installations on this
#          machine.
#
# INPUTS : $$ARG[1]{rh_option} - A reference to a hash where the details about
#                                the option are to be stored.
#
# OUTPUTS: NONE
#
# NOTES  : 1. The return values are in hash so the caller will get the new
#             details when this subroutine is finished.
#          2. The formatting is used in the help methods.
#
###############################################################################
sub option_all {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

    $$rh_details{_description}    = "-all"; 
    $$rh_details{_helpText}       = "Report the name and installation " .
                                    "directory for each \n\$ORACLE_HOME " .
                                    "found.";
    $$rh_details{_required}       = "no";

}   # End of option_all().

##############################################################################
#
# NAME: option_retry
#
# PURPOSE: Set the retry option for OPatch to retry locking inventory
#
# ############################################################################
sub option_retry {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-retry <value>";
   $$rh_details{_helpText}     =
                                 "This option tells OPatch how many times\n" .
                                 "it should retry in case of an inventory\n" .
                                 "lock failure.";
    $$rh_details{_required}       = "no";
    #$$rh_details{_type}           = "directory";
}

##############################################################################
#
# NAME: option_delay
#
# PURPOSE: Set the number of seconds to retry locking inventory
#
# ############################################################################
sub option_delay {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-delay <value>";
   $$rh_details{_helpText}     =
                                 "If -retry is specified, this option tells\n" .
                                 "OPatch how many seconds it should wait before\n" .
                                 "attempting to lock inventory again in case\n" .
                                 "of a previous failure.";
    $$rh_details{_required}       = "no";
    #$$rh_details{_type}           = "directory";
}

##############################################################################
#
# NAME: option_jre
#
# PURPOSE: Set the jre option
#
# ############################################################################
sub option_jre {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-jre <LOC>";
   $$rh_details{_helpText}     = 
                                 "This option tells OPatch to use JRE (java)\n" .
                                 "from the specified location instead of\n" .
                                 "the default location under Oracle Home.\n";
    $$rh_details{_required}       = "no";
    #$$rh_details{_type}           = "directory";
}

###############################################################################
#
# NAME   : option_detail
#
# PURPOSE: 
#
# INPUTS : $$ARG[1]{rh_option} - A reference to a hash where the details about
#                                the option are to be stored.
#
# OUTPUTS: NONE
#
#
###############################################################################
sub option_detail {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

    $$rh_details{_description}    = "-detail"; 
    $$rh_details{_helpText}       = "Report installed products and other " .
                                    "details.\n" .
                                    "This option can not be used in " .
                                    "conjunction with -all option \n" ;
    $$rh_details{_required}       = "no";

}   # End of option_all().

##############################################################################
#
# NAME: option_local
#
# PURPOSE: Set the local option
#
# ############################################################################
sub option_local {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];

   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-local";
   $$rh_details{_helpText}     =
                    "Look up the inventory of the local system.\n" .
                    "Do not attempt to detect whether it is RAC or not.\n" .
                    "This option cannot be used together with -local_node." ;
   $$rh_details{_required}       = "no";

}

##############################################################################
#
# NAME: option_local_node
#
# PURPOSE: Set local node name instead of detecting it 
#
# ############################################################################
sub option_local_node {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-local_node <Name of local node>";
   $$rh_details{_helpText}     =
                                 "OPatch will use the supplied local node name\n" .
                                 "when it tries to detect whether the current\n" .
                                 "Oracle Home is on a RAC or not.\n" .
                                 "This option cannot be used together with -local." ;
   $$rh_details{_required}       = "no";

}

##############################################################################
#
# NAME   : parse_arguments
#
# PURPOSE: Parse the object's arguements (or attributes if you prefer).
#
# INPUTS : $$ARG[1]{rh_option} - A reference to a hash where the details about
#                                the option are to be stored.
#
# OUTPUTS: A reference to a hash.
#
# NOTES  : 1. For full notes see Apply::parse_arguments().
#          2. Only need to check for a different location for ORACLE_HOME 
#             as given by the "-oh" (home path) argument. If not given use
#             the environmental variable ORACLE_HOME, otherwise raise an
#             error.
#
###############################################################################
sub parse_arguments {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $ra_arguments = $$rh_arguments{ra_arguments};

    my $array_size  = scalar( @$ra_arguments );
    my $error_flag  = "";
    my $Oracle_Home = "";
    my $report_flag = "";

    for ( my $i = 0; $i < $array_size; $i++ ) {
        if ( $$ra_arguments[$i] =~ m#^-oh$# ) {
            if ( -d $$ra_arguments[$i + 1] ) {
                $Oracle_Home = $$ra_arguments[$i + 1];
                $ENV{ORACLE_HOME} = $Oracle_Home;
            } else {
                $error_flag .= "Argument is not a directory: " .
                               "$$ra_arguments[$i].\n";
            }
            $i++;

        } elsif ( $$ra_arguments[$i] =~ m#^-all$# ) {
            $report_flag = 1;

        } elsif ( $$ra_arguments[$i] =~ m#-detail# ) {
            $LsInventory::SHOW_DETAIL = 1;
        
        } elsif ( $$ra_arguments[$i] =~ m#^-jre# ) {
            $Command::JRE_LOC = $$ra_arguments[$i + 1];
            $i++;
 
        } elsif ( $$ra_arguments[$i] =~ m#^-i(nvPtrLoc)?$# ) {
            my $inv_ptr_file = $$ra_arguments[$i + 1];
            if ( &File::Basename::basename ( $inv_ptr_file ) ne
                                                          "oraInst.loc" ) {
                $inv_ptr_file = File::Spec -> catfile ( "$inv_ptr_file" ,
                                                            "oraInst.loc" );
            }

            $Command::inventory_location_ptr = 
                    "-Doracle.installer.invPtrLoc=" . $$ra_arguments[$i + 1];

            $i++;

        } elsif ( $$ra_arguments[$i] =~ m#^-retry# ) {
            $Command::RETRY = $$ra_arguments[$i + 1];
            $i++;

        } elsif ( $$ra_arguments[$i] =~ m#^-delay# ) {
            $Command::DELAY = $$ra_arguments[$i + 1];
            $i++;
        } elsif ( $$ra_arguments[$i] =~ m#^-no_inventory# ) { 
          opatchIO->print_message_and_die({
            exit_val => $this_class->ERROR_CODE_COMMAND_LINE_ARGUMENT,
            message=>
              "lsinventory does not support no_inventory option\n"
          });    
        } elsif ( $$ra_arguments[$i] =~ m#^-local_node# ) {
        # Do not support -local_node option.
        #    $Command::LOCAL_NODE_NAME = $$ra_arguments[$i + 1];
        #    $Command::LOCAL_NODE_NAME_MODE = 1;
        #    $i++;
          opatchIO->print_message_and_die({
            exit_val => $this_class->ERROR_CODE_COMMAND_LINE_ARGUMENT,
            message=>
              "lsinventory does not support -local_node option\n" } );
        } elsif ( $$ra_arguments[$i] =~ m#^-local# ) {
        #    $Command::LOCAL_NODE_ONLY = 1;
          opatchIO->print_message_and_die({
            exit_val => $this_class->ERROR_CODE_COMMAND_LINE_ARGUMENT,
            message=>
              "lsinventory does not support -local option\n" });
        } elsif ( $$ra_arguments[$i] =~ m#^-verbose# ) {
            $Command::verbose_mode = 1;
        }
    }

    # If $Oracle_Home isn't set get it from the environment if possible.
    if ( ( ( ! $Oracle_Home ) && ( $ENV{ORACLE_HOME} ) ) ||
         ( $Oracle_Home eq "" ) ) {
        $Oracle_Home = $ENV{ORACLE_HOME};
        if ( ! -d $Oracle_Home || ! -e $Oracle_Home ) {
            $error_flag .= "Oracle's home is not a directory: \n" .
                           "$Oracle_Home.\n";
        }
    }
    # save the original value of OH in case it is a symbolic link and get
    #   resolved below
    $Command::UNRESOLVED_ORACLE_HOME = $Oracle_Home;

    # If there is still no $Oracle_Home raise an error. IF there is an
    # $Oracle_Home and it's MicroSoft OS lowercase the string.
    if ( ! $Oracle_Home ) {
        $error_flag .= "There is no \$ORACLE_HOME to remove the patch from.\n";
    } elsif ( $OSNAME =~ m#Win32# ){
      $Oracle_Home = lc ($Oracle_Home );
    } else {
        my $is_link = $this_class -> is_symbolic_link ( $Oracle_Home );
        if ( $is_link == 1 ) {
            my $real_OH = $this_class->resolve_symbolic_link($Oracle_Home);

            opatchIO->print_message ( {message => "\n-----------------------------------------\n"} );
            opatchIO->print_message ( {message => "\nWARNING: Oracle_Home is a symbolic link:\n"} );
            opatchIO->print_message ( {message => "$Oracle_Home\n"} );
            opatchIO->print_message ( {message => "--->\n"} );
            opatchIO->print_message ( {message => "$real_OH"} );
            opatchIO->print_message ( {message => "\n-----------------------------------------\n"} );
            # Begin Add: BUG 2899335
            my $is_internal_link = $this_class -> verify_internal_link ($Oracle_Home);
            if ($is_internal_link == 0 ) {
                $Oracle_Home = $real_OH;
                opatchIO->debug ( {message => "Oracle_Home is resolved to $Oracle_Home"} );
                opatchIO->debug ( {message => "User-supplied OH is saved as $Command::UNRESOLVED_ORACLE_HOME"} );
            }
            # End Add: BUG 2899335
        }
    }

    # If there is an $Oracle_Home strip any trailing "/" or "\" from it.
    # This upsets the OUI APIs for some reason.
    if ( $Oracle_Home =~ m#[\\/]$# ) {
        ( $Oracle_Home ) = ( $Oracle_Home =~ m#(.+)[\\/]\s?$# );
    }

    my %return_values = ();
    $return_values{error_flag}  = $error_flag;
    $return_values{Oracle_Home} = $Oracle_Home;
    $return_values{report_flag} = $report_flag;

    return ( \%return_values );

}   # End of parse_arguments().

###############################################################################
#
# NAME   : validate_options
#
# PURPOSE: A method to validate the command line.
#
# INPUTS : $$ARG[1]{rh_command_line} - A reference to a hash that has the
#                                      command and options to be validated.
#          $$ARG[1]{rh_command_list} - A reference to a hash containg details
#                                      about the valid commands and options.
#
# OUTPUTS: Nothing directly. If validation fails, the attibute {invalid} is
#          defined for rh_command_line.
#
# NOTES  : 1. Check the given arguments (in $rh_command_line) against the list
#             of valid arguments for the command. If the argument has a
#             an attribute _validityChecks then the value from the command
#             line is validated by the appropriate method.
#          2. If you're trying to debug this (or follow it in a debugger) the
#             following arguments to 'keys' should help:
#             a. %${$$rh_command_list{$command}{_options}}, and
#             b. ${$${$$rh_command_list{$command}{_options}}{$type}}{$attribute}
#             but the loops to see all the values are left to the reader.
#          3. Assumes that the return value from any validation method is
#             a null string for success or a descriptive string of the failure.
#             This is stored in $error_string.
#
###############################################################################
sub validate_options {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_command_line = $$rh_arguments{rh_command_line};
    my $rh_command_list = $$rh_arguments{rh_command_list};

    my $command         = $$rh_command_line{command};
    my @arguments       = split ( " ", $$rh_command_line{arguments});


    my $upper_bound = scalar ( @arguments );
    for ( my $i = 0; $i < $upper_bound; $i++ ) {

        my $arg_name = $arguments[$i];
        my $invalid_flag = 1;

        # Skip global options, already validated.
        next if ( exists $$rh_command_list{$arg_name} );

        # Skip if it doesn't begin with a "-" symbol. Apply allows a single
        # string for OralceHome/ShipHome in the processesing.
        next if ( $arg_name !~ m#^-# );

        # Now check for bogus arguments. Because of perl's autovivification
        # this test needs be done first as the attribute would be created
        # by the validity check if it gets that far.

        foreach my $valid_argument
                ( keys %${$$rh_command_list{$command}{_options}} ) {
            if ( index ($valid_argument, $arg_name ) > -1 ) {
                $invalid_flag = 0;
            }
        }

        if ( $invalid_flag ) {
            if ( exists $$rh_command_line{invalid} ) {
                $$rh_command_line{invalid} .= "\nInvalid argument: $arg_name.";
            } else {
                $$rh_command_line{invalid} = "Invalid argument: $arg_name.";
            }
            next;
        }

        # Does this option, now valid, require a check on the value?
        # ( Yes Virgina, you can split hashes like this to avoid line wraps. )
        if ( ${$${$$rh_command_list{$command}{_options}}
                                           {$arg_name}}{_validityChecks} ) {
            my $check_type = ${$${$$rh_command_list{$command}{_options}}
                                                        {$arg_name}}{_type};
            my $rule_set =   ${$${$$rh_command_list{$command}{_options}}
                                              {$arg_name}}{_validityChecks};

            # Out of hubris I've assumed all validation methods to start
            # with "validate_".
            my $validate_method = "validate_" . $check_type;
            my $error_string = $this_class -> $validate_method ( {
                                        item_to_validate => $arguments[$i+1],
                                        validation_rules => $rule_set } );
            $i++;
            if ( $error_string ) {
                $$rh_command_line{invalid} .= $error_string;
            }
        }
    }
}   # End of validate_options().

#############################################################################
# Perl packages need to return a true value.
1;
#############################################################################

