#
# Copyright (c) 2001, 2005, Oracle. All rights reserved.  
#
#    NAME
#    Query.pm
#
#    DESCRIPTION
#    Place holder to query info. about patch & system being patched
#    This is to query about a patch, so try to stay off other modules' 
#    dependency as much as possible.  For example, it doesn't make sense
#    to require ORACLE_HOME to query patch <xyz>
#
#    BUGS
#    * <if you know of any bugs, put them here!>
#
#    MODIFIED   (MM/DD/YY)
#    vsriram     11/15/05 - Do not make -verify a part of -all.
#                           If user enter wrong patch location, give a prompt to exit.
#    phnguyen    11/03/05 - Add option 'opatch query -verify' 
#    vsriram     08/15/05 - Bug-4541352: Flush stdout before reading user input.
#    phnguyen    01/21/05 - 
#    phnguyen    01/03/05 - minor text change when bad option is given
#    phnguyen    10/06/04 - help text
#    phnguyen    02/04/04 - fix bug where req. comp. is shown as optional
#    phnguyen    10/23/03 - modify output per Siemens' feedback.
#                             make clear that is_shutdown and get_system_change
#                                 are not supported yet.
#                             fix bugs to explicitly say whether a comp. is
#                                 required or not.
#                             add built date
#    phnguyen    10/08/03 - fix problem in pre/post readme cat
#                           touch up output
#    shgangul    10/07/03 - touchup OPatch display: bug 3170021 
#    shgangul    09/24/03 - Add changes for pre post script 
#    phnguyen    08/28/03 - Initial Code 
##########################################################################

package Query;
@ISA = ("Command");

######
# Standard modules:
use English;         # Let us say "$CHILD_ERROR" instead of "$?", etc.
use strict;          # Enforce strict variables, refs, subs, etc.

use Cwd();
use File::Basename();
use File::Spec();

# Local modules.
use Command();
use Apply();
use AttachHome();
use LsInventory();
use RollBack();
use Version();
use opatchIO();

# Flags set by users
use constant ALL_OPTION => 0;
use constant IS_ROLLING_OPTION => 0;
use constant GET_COMPONENT_OPTION => 0;
use constant GET_BASE_BUG_OPTION => 0;
use constant GET_DATE_OPTION => 0;
use constant GET_OS_OPTION => 0;
use constant IS_SHUTDOWN_OPTION => 0;
use constant VERIFY_OPTION => 0;
###############################################################################
#
# NAME   : global_options
#
# PURPOSE: Install global options so they appear _like_ commands.
#
# INPUTS : $$ARG[1]{rh_detail} - A reference to a hash structure of the class
#                                Command to store the attributes in.
#
# OUTPUTS: NONE
#
# NOTES  : 1. The return values are in hash so the caller will get the new
#             details when this subroutine is finished.
#          2. The design specification has "-help" as a command while "-n"
#             "-x" are options to an instance of Command->global. In reality
#             either they are all options or all commands. These have been
#             coded at the level of a command and are identified as global
#             options by the leading "-" symbol. This removes the dichotomy
#             of the usage and design specification. Of course making a
#             command "global" and having all three as options achieves the
#             same result.
#
###############################################################################
sub global_options {

    # Because this was called in an OO manner the first argument is the
    # class name.
    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_detail    = $$rh_arguments{rh_detail};

    my @option_list = ( "help" );

    foreach my $option ( @option_list ) {
        $$rh_detail{-$option}  = Query -> new_command();
        $$rh_detail{-$option}{description} = "-$option [ <command> ]";
    }

    $$rh_detail{-help}{_defaultValue}  = "false";

    $$rh_detail{-help}{_description}  = "Print the help and usage message.";

    # Occassionally everbody has to hardwire something.
    $$rh_detail{-help}{_helpText}  = "Displays the help message for the " .
                                     "command.";

    $$rh_detail{-help}{_flag}  = "yes";

    $$rh_detail{-help}{_requiredOption}  = "yes";


}   # End of command_details().


###############################################################################
#
# NAME   : command_details
#
# PURPOSE: Provide the details about the command.
#
# INPUTS : $$ARG[1]{rh_detail} - A reference to a hash structure of the class
#                                Command to store the attributes in.
#
# OUTPUTS: NONE
#
# NOTES  : 1. The return values are in hash so the caller will get the new
#             details when this subroutine is finished.
#
# CALLER : called by opatch.pl before abdomen is called
#
###############################################################################
sub command_details {

    # Because this was called in an OO manner the first argument is the
    # class name.
    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_detail    = $$rh_arguments{rh_detail};
 
    # Put options here for easy reference. They don't need to be global.
    my @options = ( "all", "is_rolling", "is_shutdown",
                    "get_base_bug", "get_date",
                    "get_component", "get_os",
                    "get_system_change", "verify",
                    "invPtrLoc"      
    );

    my $command = lc ( $this_class );
    $$rh_detail{$command}  = Command -> new_command();

    $$rh_detail{$command}{_description} = "query ";

    $$rh_detail{$command}{_helpText}    =
          "Query various info. about a patch and the system being patched.\n";

    $$rh_detail{$command}{_options} =
               $this_class -> build_option_details ( {
                          rh_options     => $$rh_detail{$command}{_options},
                          ra_option_list => \@options } );

    $this_class -> add_options_to_command_description
                                ( { rh_command => $$rh_detail{$command} } );

    return ( 0 );

}   # End of command_detail().

#################################################################################
#
# option_[actions supported]
#
#################################################################################
sub option_all {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

    $$rh_details{_defaultValue} = "false";
    $$rh_details{_description}  = "-all";
    $$rh_details{_flag}         = "yes";
    $$rh_details{_helpText}     =
             "Get everything available about a patch.\n" .
             "This is equivalent to setting all\n" .
             "available options.\n";
    $$rh_details{_type}         = "install";

}
sub option_verify {
    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

    $$rh_details{_defaultValue} = "false";
    $$rh_details{_description}  = "-verify";
    $$rh_details{_flag}         = "yes";
    $$rh_details{_helpText}     =
             "Verify if a patch has been applied on Oracle Home.\n"; 
    $$rh_details{_type}         = "install";

}
sub option_invPtrLoc {
    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

    $$rh_details{_defaultValue} = "false";
    $$rh_details{_description}  = "-invPtrLoc <Path to oraInst.loc>";
    $$rh_details{_flag}         = "yes";
    $$rh_details{_helpText}     = "Used to locate the oraInst.loc file.\n" .
                                  "Needed when the installation used the\n" .
                                  "invPtrLoc flag. This should " .
                                  "\nbe the path to the oraInst.loc file";
    $$rh_details{_type}         = "install";

}
sub option_is_rolling {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

    $$rh_details{_defaultValue} = "false";
    $$rh_details{_description}  = "-is_rolling";
    $$rh_details{_flag}         = "yes";
    $$rh_details{_helpText}     =
             "Is it a rolling patch?\n"; 
    $$rh_details{_type}         = "install";

}
sub option_is_shutdown {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

    $$rh_details{_defaultValue} = "false";
    $$rh_details{_description}  = "-is_shutdown";
    $$rh_details{_flag}         = "yes";
    $$rh_details{_helpText}     =
             "Should Oracle instances be down\n" .
             "during patching process\n" .
             "(currently not available, use -all)."; 
    $$rh_details{_type}         = "install";

}
sub option_get_base_bug {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

    $$rh_details{_defaultValue} = "false";
    $$rh_details{_description}  = "-get_base_bug";
    $$rh_details{_flag}         = "yes";
    $$rh_details{_helpText}     =
          "Get the base bugs fixed by a patch.\n";
    $$rh_details{_type}         = "install";

}
sub option_get_date {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

    $$rh_details{_defaultValue} = "false";
    $$rh_details{_description}  = "-get_date";
    $$rh_details{_flag}         = "yes";
    $$rh_details{_helpText}     =
          "Get the built date of a patch.\n";
    $$rh_details{_type}         = "install";

}
sub option_get_component {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

    $$rh_details{_defaultValue} = "false";
    $$rh_details{_description}  = "-get_component";
    $$rh_details{_flag}         = "yes";
    $$rh_details{_helpText}     =
          "Get the Oracle components either optional\n".
          "or required by a patch.\n";
    $$rh_details{_type}         = "install";

}
sub option_get_os {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

    $$rh_details{_defaultValue} = "false";
    $$rh_details{_description}  = "-get_os";
    $$rh_details{_flag}         = "yes";
    $$rh_details{_helpText}     =
          "Get the Operating System description\n" .
          "supported by a patch.\n";
    $$rh_details{_type}         = "install";

}
sub option_get_system_change {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

    $$rh_details{_defaultValue} = "false";
    $$rh_details{_description}  = "-get_system_change";
    $$rh_details{_flag}         = "yes";
    $$rh_details{_helpText}     =
          "Get a description of what changes will\n" .
          "be made to this system by a patch. \n" .
          "(currently not available, use -all)."; 
    $$rh_details{_type}         = "install";

}

################################################################################
#
#
#
################################################################################
sub parse_arguments {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $ra_arguments = $$rh_arguments{ra_arguments};

    my $array_size  = scalar( @$ra_arguments );
    my $error_flag  = "";
    my $force_flag  = 0;
    my $noop_flag   = 0;
    my $Oracle_Home = "";
    my $Ship_Home   = "";
    my $general_options = 0;
    my $RAC_options     = 0;

    for ( my $i = 0; $i < $array_size; $i++ ) {
        if ( $$ra_arguments[$i] =~ m#^-all$# ) {
            $Query::ALL_OPTION = 1;
        } elsif ( $$ra_arguments[$i] =~ m#^-is_rolling$# ) {
            $Query::IS_ROLLING_OPTION = 1;
        } elsif ( $$ra_arguments[$i] =~ m#^-get_component$# ) {
            $Query::GET_COMPONENT_OPTION = 1;
        } elsif ( $$ra_arguments[$i] =~ m#^-get_base_bug$# ) {
            $Query::GET_BASE_BUG_OPTION = 1;
        } elsif ( $$ra_arguments[$i] =~ m#^-get_date$# ) {
            $Query::GET_DATE_OPTION = 1;
        } elsif ( $$ra_arguments[$i] =~ m#^-get_os$# ) {
            $Query::GET_OS_OPTION = 1;
        } elsif ( $$ra_arguments[$i] =~ m#^-is_shutdown$# ) {
            $Query::IS_SHUTDOWN_OPTION = 1;
        } elsif ( $$ra_arguments[$i] =~ m#^-get_system_change$# ) {
            $Query::GET_SYSTEM_CHANGE_OPTION = 1;
        } elsif ( $$ra_arguments[$i] =~ m#^-verify$# ) {
	    $Query::VERIFY_OPTION = 1;
        } elsif ( $$ra_arguments[$i] =~ m#^-i(nvPtrLoc)?$# ) {
            if( $Query::VERIFY_OPTION == 1) {
                 my $inv_ptr_file = $$ra_arguments[$i + 1];
                 if ( &File::Basename::basename ( $inv_ptr_file ) ne
                                                             "oraInst.loc" ) {
                       $inv_ptr_file = File::Spec -> catfile ( "$inv_ptr_file" ,
                                                                "oraInst.loc" );
                 }

                 $Command::inventory_location_ptr =
                       "-Doracle.installer.invPtrLoc=" . $$ra_arguments[$i + 1];
           }
        }
    } 
    
    my %return_values = ();
    $return_values{error_flag}  = $error_flag;
    $return_values{force_flag}  = $force_flag;
    $return_values{Oracle_Home} = $Oracle_Home;
    $return_values{ship_home}   = $Ship_Home;

    return ( \%return_values );

}

################################################################################
#
# read xml files
#    input:  file name
#            patch_inventory_location
#    output: file data (content of the file) or NULL if errors
#
################################################################################
sub read_file {
    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $file_path = $$rh_arguments{file_path};
    my $file_name = $$rh_arguments{file_name};

    my $file = File::Spec->catfile("$file_path", "$file_name");
    
    local *INVENTORY_FILE;
    open ( INVENTORY_FILE, $file ) or do {
        opatchIO->print_message({
             message =>
               "Cannot open data file $file for reading: \n" .
                      "[$OS_ERROR]."
        });
        return ();
    };
    my @file_data = <INVENTORY_FILE>;
    close INVENTORY_FILE or do {
        opatchIO -> print_message ( { 
                  message  => "Cannot close the file after reading: \n" .
                         "[$OS_ERROR]." 
        });
    };
    return @file_data;
}

################################################################################
#
# dump a content of a given file to screen
#    input:  file name
#
################################################################################
sub dump_file {
    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my @file_data = $$rh_arguments{file_data};
    print @file_data;
    my $file_size = scalar ( @file_data );
    my $line = "";
    for ( my $i =0; $i < $file_size; $i++ ) {
        $line = $file_data[$i];
        chomp $line;
        print $line;
    }
}
################################################################################
#
# read patch xml files
#
################################################################################
sub read_and_check_reference_details {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $inventory_file   = $$rh_arguments{inventory_file};
    my $Oracle_Home      = $$rh_arguments{Oracle_Home};
    my $patch_inventory_location = $$rh_arguments{p_i_l};
    my $error_flag = "";
    my $information_flag = "";
    my @file_data = ();

    my $full_inventory_path = File::Spec -> catfile(
                                              "$patch_inventory_location",
                                              "$inventory_file" );
   
    local *INVENTORY_FILE;
    open ( INVENTORY_FILE, $full_inventory_path ) or do {
        $error_flag = "Cannot open data file for reading: " .
                      "$full_inventory_path.\n[$OS_ERROR]";
    };

    if ( ! $error_flag ) {
        @file_data = <INVENTORY_FILE>;
        close INVENTORY_FILE or do {
            opatchIO -> print_message ( { f_handle => *STDERR,
                                          message  => "Cannot close the " .
                             "file after reading: $full_inventory_path\n" .
                                    "[$OS_ERROR]\nProcessing continuing" } );
        };
    }

    my %reference_details = ();
    $reference_details{RAC_shutdown} = 1;

    my $file_size = scalar ( @file_data );
    for ( my $i =0; $i < $file_size; $i++ ) {
        chomp $file_data[$i];

        if ( $file_data[$i] =~ m#<reference_id number="# ) {
           ( $reference_details{reference_id} ) =
                        ( $file_data[$i] =~ m#<reference_id number="(.+)"# );
        }

        if ( $file_data[$i] =~ m#<date_of_patch year="# ) {
            ( $reference_details{date}{year}  ) =
                                       ( $file_data[$i] =~ m#year="(\d+)"# );
            ( $reference_details{date}{month} ) =
                                      ( $file_data[$i] =~ m#month="(\w+)"# );
            ( $reference_details{date}{day}   ) =
                                        ( $file_data[$i] =~ m#day="(\d+)"# );
            ( $reference_details{date}{time}  ) =
                               ( $file_data[$i] =~ m#time="([:\d]+)\shrs"# );
            ( $reference_details{date}{zone}  ) =
                                     ( $file_data[$i] =~ m#zone="(.+)"# );
        }

        if ( $file_data[$i] =~ m#<base_bugs>$# ) {
            $i++;
            while ( $file_data[$i] !~ m#</base_bugs>$# ) {
                my ( $number, $detail ) = ( $file_data[$i] =~
                           m#<bug number="(\d+)"\sdescription="(.+)"\s?/>$# );
                $reference_details{bug_list}{$number} = $detail;
                $i++;
            }
        }

        if ( $file_data[$i] =~ m#<required_components>$# ) {
            $i++;
            while ( $file_data[$i] !~ m#</required_components>$# ) {
                # Some component names may have spaces and digits and some
                # version numbers contain letters.
                my ( $name, $version ) = ( $file_data[$i] =~
                       m#<component internal_name="(.+)"\sversion="(.+)"\s# );
                my ( $option_flag ) = ( $file_data[$i] =~
                                                    m#opt_req="(\w)"\s?/>$# );
                $reference_details{components}{$name}{version} = $version;
                $reference_details{components}{$name}
                                                 {option_flag} = $option_flag;

                $i++;
            }
        }

        if ( $file_data[$i] =~ m#<os_platforms>$# ) {
            $i++;
            while ( $file_data[$i] !~ m#</os_platforms>$# ) {
                my ( $description, $number ) = ( $file_data[$i] =~
                           m#<platform name="(.+)"\sid="(\d+)"\s?/>$# );
                $reference_details{os_platforms}{$number} = $description;
                $i++;
            }
        }

        if ( $file_data[$i] =~ m#<executables>$# ) {
            $i++;
            while ( $file_data[$i] !~ m#</executables>$# ) {
                my ( $path ) = ( $file_data[$i] =~
                           m#<executable path="(.+)"\s?/>$# );
                my $name = &File::Basename::basename ( $path );
                $path =~ s/\%ORACLE_HOME\%/$Oracle_Home/ ;
                $reference_details{executable}{$name} = $path;
                $i++;
            }
        }

        if ( $file_data[$i] =~ m#<instance_shutdown># ) {
            $reference_details{shutdown} = 0;
            my ( $shutdown ) =
                 ( $file_data[$i] =~
                       m#<instance_shutdown>(\w+)</instance_shutdown>$# );
            # Now turn a "true" into a 1.
            if ( $shutdown eq "true" ) {
                $reference_details{shutdown} = 1;
            }
        }

        # This is for rolling RAC upgrades. If its rolling RAC installable
        # then the RAC doesn't need to be shutdown.
        if ( $file_data[$i] =~ m#<online_rac_installable># ) {
            my ( $rolling_RAC ) =
                 ( $file_data[$i] =~
             m#<online_rac_installable>(\w+)</online_rac_installable>$# );

            if ( $rolling_RAC eq "true" ) {
                $reference_details{RAC_shutdown} = 0;
                $Command::ROLLING_PATCH = 1;
            }
        }
    }  # Finished handling the inventory file.

    return ( \%reference_details );
}


################################################################################
#
# main entry
#
################################################################################
sub abdomen {

    opatchIO->print_message({ message =>
         "Query ...\n"
    });

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $command    = $$rh_arguments{command};
    my $arg_string = $$rh_arguments{arguments};
    my $rh_command_list = $$rh_arguments{rh_command_list};

    my $patch_myself_flag = "";

    my @arguments = split ( " ", $arg_string ) ;

    # Now parse the arguments for this command.
    my $rh_arg_check =
            $this_class -> parse_arguments( { ra_arguments => \@arguments } );

    my $error_flag = $$rh_arg_check{error_flag};

    # For any errors display the problem, give command usage then exit.
    if ( $error_flag ) {
        $this_class -> display_help( { command => $command,
                                       rh_command_list => $rh_command_list } );
       opatchIO->print_message_and_die({
          exit_val => $this_class->ERROR_CODE_COMMAND_LINE_ARGUMENT,
          message=>
            "Error parsing arguments\n"
	      });
    }

    if ($Query::ALL_OPTION) {
        $Query::IS_ROLLING_OPTION = 1;
        $Query::GET_COMPONENT_OPTION = 1;
        $Query::GET_BASE_BUG_OPTION = 1;
        $Query::GET_DATE_OPTION = 1;
        $Query::GET_OS_OPTION = 1;
    }
    my $any_option_set = 
             $Query::IS_ROLLING_OPTION || 
             $Query::GET_COMPONENT_OPTION ||
             $Query::GET_BASE_BUG_OPTION ||
             $Query::GET_DATE_OPTION ||
             $Query::GET_OS_OPTION ||
             $Query::VERIFY_OPTION; 
    
    if ( $Query::GET_SYSTEM_CHANGE_OPTION ) {
       opatchIO->print_message ( {message => "Option -get_system_change has not been implemented. " .
                                             "Please use -all to get all available options."} );
       return;
    }
    if ( $Query::IS_SHUTDOWN_OPTION) {
       opatchIO->print_message ( {message => "Option -is_shutdown has not been implmented. " .
                                             "Please use -all to get all available options."} );
       return;
    }
    if (!$any_option_set) {
       opatchIO->print_message ( {message => "No options (or bad options) given to 'opatch query': nothing to do...\n"} );
       return;
    }
    
    # At least $Oracle_Home will be valid at this point.
    # my $force_flag  = $$rh_arg_check{force_flag};
    my $general_options = $$rh_arg_check{general_options};
    # my $noop_flag   = $$rh_arg_check{noop_flag};
    # my $Oracle_Home = $$rh_arg_check{Oracle_Home};
    # my $RAC_options = $$rh_arg_check{RAC_options};
    my $ship_home   = $$rh_arg_check{ship_home};

    # Get list of XML files
    my $rh_file_names = $this_class -> build_required_XML_filenames ();
    my $actions_file = $$rh_file_names{actions};
    my $inventory_file = $$rh_file_names{inventory};
    my $GenericActions_file = $$rh_file_names{GenericActions};
    my $ShiphomeDirectoryStructure_file = 
          $$rh_file_names{ShiphomeDirectoryStructure_file};
    my $patch_inventory_location = $ship_home || &Cwd::cwd();
    

    
    my $f_handle = *STDOUT;
    my $message = "Please enter the patch location: ";
    my $reply = "";
    my $file_path = "";
    my $file_name = "";
    my @file_data = ();

    my $continue = 1;

    my $patch_loc = "";
    while ( $continue ) {
      # print out question to get file path
      opatchIO->print_message ( {message => "\n"} );
      opatchIO->print_message ( {message => $message} );

      # Flush the stdout first
      select(STDOUT);
      $|=1;

      $reply = <STDIN> || "";
      chomp $reply;
      $patch_loc = $reply;

      # Turnoff autoflush
      $|=0;

      opatchIO->print_message ( {message => "\n"} );
      
      my $notDone = 1;
      while ( $notDone ) {
          $file_path = $reply;
          $file_name = $inventory_file;
          @file_data = $this_class->read_file ({ file_path => $file_path,
                                                 file_name => $file_name });
                                               
          if ( ! @file_data ) {
              # Do users want to re-try to enter the patch location? 
              opatchIO->print_message ( {message => "\n"} );
              my $question = "Do you want to proceed?";
              my $answer = opatchIO->check_to_stop_processing({ message => $question });
              if ($answer eq "N") {
                   opatchIO -> print_message_and_die({
                               exit_val => $this_class -> ERROR_CODE_NO_ERROR,
                               message => "Query stops."
                   });
              }else {
                   opatchIO->print_message ( {message => "\n\n" .
                           "Please enter the patch location again: "} );

                   # Flush the stdout first
                   select(STDOUT);
                   $|=1;

                   $reply = <STDIN> || "";
                   chomp $reply;

                   $patch_loc = $reply;
                   # Turnoff autoflush
                   $|=0;
             }
          } else {
              #print "File content is:\n";
              #print @file_data;
              $notDone = 0;
          }
      }
      # Read file_data and check if it's rolling patch
      my $patchID = "";
      my $is_rolling = 0;
      my $comp_ver_list = "";
      my $baseBugList = "";
      my $osList = "";
      my $patchDate = "";
      
      my $file_size = scalar ( @file_data );
      my $line = "";
      for ( my $i =0; $i < $file_size; $i++ ) {
          $line = $file_data[$i];
          chomp $line;
          
          if ( $line =~ m#<date_of_patch year="# ) {
              my ( $year  ) = ( $line =~ m#year="(\d+)"# );
              my ( $month ) = ( $line =~ m#month="(\w+)"# );
              my ( $day   ) = ( $line =~ m#day="(\d+)"# );
              my ( $time  ) = ( $line =~ m#time="([:\d]+)\shrs"# );
              my ( $zone  ) = ( $line =~ m#zone="(.+)"# );
              
              $patchDate = $month . "/" . $day . "/" . $year . " at $time $zone";
                          
          } elsif ( $line =~ m#<os_platforms>$# ) {
              $i++;
              $line = $file_data[$i];
              while ( $line !~ m#</os_platforms>$# ) {
                  my ( $description, $number ) = ( $line =~
                             m#<platform name="(.+)"\sid="(\d+)"\s?/>$# );
                 
                  $osList = $osList .
                            "\t" .
                            $number . ": " . $description . "\n";
                  $i++;
                  $line = $file_data[$i];
               }
          } elsif ( $line =~ m#<online_rac_installable># ) {
              my ( $rolling_RAC ) =
                   ( $line =~
               m#<online_rac_installable>(\w+)</online_rac_installable>$# );

              if ( $rolling_RAC eq "true" ) {
                  $is_rolling = 1;
              }
          } elsif ( $line =~ m#<required_components>$# ) {
              $i++;
              while ( $file_data[$i] !~ m#</required_components>$# ) {
                  # Some component names may have spaces and digits and some
                  # version numbers contain letters.
                  my ( $name, $version ) = ( $file_data[$i] =~
                         m#<component internal_name="(.+)"\sversion="(.+)"\s# );
                  my ( $option_flag ) = ( $file_data[$i] =~
                                                      m#opt_req="(\w)"\s?/>$# );
                  my $required = "";
                  if ( $option_flag eq "R" ) {
                    $required = "(required)";
                  } else {
                    $required = "(optional)";
                  }
                  $comp_ver_list = $comp_ver_list .
                                   "\t" .
                                   $name . ": " . $version . "  " . $required .
                                   "\n";
                  $i++;
              }
          } elsif ( $line =~ m#<base_bugs>$# ) {
              $i++;
              while ( $file_data[$i] !~ m#</base_bugs>$# ) {
                  my ( $number, $detail ) = ( $file_data[$i] =~
                             m#<bug number="(\d+)"\sdescription="(.+)"\s?/>$# );
                  $baseBugList = $baseBugList . 
                                 "\t" .
                                 $number . ": " . $detail . "\n";
                  $i++;
              }
         } elsif ( $line =~ m#<reference_id number="# ) {
               ( my $id ) =
                        ( $line =~ m#<reference_id number="(.+)"# );
               $patchID = $id;
         } 
      }
      # Answers to users' queries
      opatchIO->print_message ( {message => "---------- Query starts ------------------\n"} );
      # opatchIO->print_message ( {message => "\n"} );
      opatchIO->print_message ( {message => "Patch ID: " . $patchID . "\n" } );
      
      # Print out the pre.txt if one exists
      my $script_txt_loc = "";
      my $script_txt_handle;
      my @script_txt_lines;
      my $script_line = "";

      $script_txt_loc = File::Spec -> catfile ($file_path, "custom", "pre.txt");
      if ( -e $script_txt_loc )
      {
          local *script_txt_handle;
          opatchIO->print_message_noverbose ( { message => $Command::PRE_README_HEADER } );
          open(script_txt_handle, $script_txt_loc) || opatchIO->print_message ( { message => "Could not open file $script_txt_loc" } );
          @script_txt_lines = <script_txt_handle>;
          foreach $script_line (@script_txt_lines)
          {
            chomp $script_line;
            opatchIO->print_message_noverbose ( { message => $script_line } );
          }
          opatchIO->print_message_noverbose ( { message => $Command::PRE_README_TAIL } );
          close script_txt_handle;          
      }

      # Print out the post.txt if one exists
      $script_txt_loc = File::Spec -> catfile ($file_path, "custom", "post.txt");
      if ( -e $script_txt_loc )
      {
          local *script_txt_handle;
          opatchIO->print_message_noverbose ( { message => $Command::POST_README_HEADER } );
          open(script_txt_handle, $script_txt_loc) || opatchIO->print_message ( { message => "Could not open file $script_txt_loc" } );
          @script_txt_lines = <script_txt_handle>;
          foreach $script_line (@script_txt_lines)
          {
            chomp $script_line;
            opatchIO->print_message_noverbose ( { message => $script_line } );
          }
          opatchIO->print_message_noverbose ( { message => $Command::POST_README_TAIL } );
          close script_txt_handle;      
      }

      if ($Query::IS_ROLLING_OPTION) {
          # opatchIO->print_message ( {message => "\n"} );
          if ($is_rolling) {
            opatchIO->print_message ( {message => "Rolling Patch: True.\n"} );
          } else {
            opatchIO->print_message ( {message => "Rolling Patch: False.\n"} );
          }
      }
      if ($Query::GET_DATE_OPTION) {
          opatchIO->print_message ( {message => "Built Date: $patchDate\n"} );
      }
      if ($Query::GET_OS_OPTION) {
          opatchIO->print_message ( {message => "Platform ID / Name:\n" .
                                                    "$osList\n"} );
      }
      if ($Query::GET_COMPONENT_OPTION) {
          # opatchIO->print_message ( {message => "\n"} );
          opatchIO->print_message ( {message => "Components / Versions:\n" .
                                                    "$comp_ver_list\n"} );
      }
      if ($Query::GET_BASE_BUG_OPTION) {
          # opatchIO->print_message ( {message => "\n"} );
          opatchIO->print_message ( {message => "Base Bugs:\n" . 
                "$baseBugList"} );
      }

      if ($Query::VERIFY_OPTION) { 
          my $Oracle_Home = $this_class -> get_abs_path( { 
               Path => ($$rh_arguments{Oracle_Home} || $ENV{ORACLE_HOME} ) } );

          my $rh_OUI_file_names = $this_class -> build_required_OUI_filenames({
                                              Oracle_home => $Oracle_Home});

          my $fh_log_file = ""; # No log file.
          my $no_inventory_update = 0; # No support for no_inventory

          my $debug = $ENV{OPATCH_DEBUG};
          if ( ! ($debug eq "TRUE") ) {
            $debug = "";
          } else {
            $debug = "-Dopatch.debug=true ";
          }
          $Command::DEBUG = $debug;

          $this_class -> verify_patch ( {
                              fh_log_file       => $fh_log_file,
                              Oracle_Home       => $Oracle_Home,
                              patch_id          => $patchID,
                              patch_location    => $patch_loc,
                              rh_OUI_file_names => $rh_OUI_file_names,
                              no_inventory      => $no_inventory_update } );
      }

      opatchIO->print_message ( {message => "---------- Query ends -------------------"}  );

      # Do users want to query another patch?
      opatchIO->print_message ( {message => "\n"} );
      my $question = "Do you want to check out another patch?";
      my $answer = opatchIO->check_to_stop_processing({ message => $question });
      if ($answer eq "N") {
        opatchIO->print_message ( {message => "Query stops.\n"} );
        $continue = 0;
      }
  } # main while loop
  opatchIO -> print_message_and_die({
              exit_val => $this_class -> ERROR_CODE_NO_ERROR,
              message => "Query done"
        });

}
###################################################################################
1;
###################################################################################
