#
# Copyright (c) 2001, 2006, Oracle. All rights reserved.  
#
#    NAME
#    RollBack.pm
#
#    DESCRIPTION
#    This file contains the methods for the rollback command to opatch.
#
#    NOTES
#
#    BUGS
#    * 1. It is possible to rollback a partially applied patch. The
#         system will be corrupted if you try to apply a partially
#         rolled back patch. Be warned. This is in FAQ.
#
#    MODIFIED   (MM/DD/YY)
#    vsriram     01/11/06 - Search for whole case sensitive words in make output.
#    vsriram     12/15/05 - Bug 4883881: Look for jdk/jre/bin/java before searching for it.
#    vsriram     12/02/05 - Error out if the -id and -ph does not match.
#                           If multiple make targets are specified, then run them all.
#    vsriram     10/10/05 - Invoke clean_up_inventory to clean up the dangling filemaps after updating the inventory.
#                           Do live printing for pre/post/java commands.
#                           Check if the file exists before deleting it, for Copy action new files.
#    vsriram     09/26/05 - Bug-4628014 - Save the original Oracle Home path and use it for make.
#                           Bug-4628240 - If -jdk/jre is specified, then use it for auto-rollback.
#                           Make error handling algorithm is changed as per the new design.
#                           Bug-4640904 - Run fuser check after checking if the patch is present in OH.
#    vsriram     08/02/05 - Bug-4243203 : Added few log/debug messages.
#    vsriram     07/29/05 - Bug-4478852 : Delete new files introduced by archive actions.
#    vsriram     06/27/05 - Bug-4450313 : Decouple fuser from pre script. Error out if files are active.
#    vsriram     06/16/05 - 
#    vsriram     06/09/05 - Print the "What is the cause" of answering Y or N during errors. 
#    vsriram     05/18/05 - Use get_abs_path
#    phnguyen    05/06/05 - Process OPATCH_IS_SHARED
#    phnguyen    05/05/05 - If called from apply, then use all-node mode
#                             
#    vsriram     04/27/05 - Convert Oracle Home to absolute path 
#			    Log for "next if"
#    phnguyen    04/26/05 - Jar pre-req.  No shutdown prompt on auto-rollback
#                           
#    shgangul    02/15/05 - Bug 4088969: Do not execute pre/post in report 
#                           mode 
#    phnguyen    01/19/05 - Back up the file before removing new file
#    shgangul    01/18/05 - Add debug messages for make errors 
#    shgangul    01/17/05 - Bug 4124475: Do not report error for non-zero 
#                           STDERR 
#    vsriram     01/12/05 - Adding Log for handling new files 
#    shgangul    01/11/05 - Change message before copy comps.xml 
#    phnguyen    01/10/05 - Back up 2nd copy of comps.xml as comps.xml.afterrollback
#    shgangul    01/06/05 - Cleanup work files in windows 
#    vsriram     01/06/05 - Rollback:Delete new files
#    shgangul    01/05/05 - Control filemap backup feature using env var 
#                           OPATCH_BACKUP_FILEMAP 
#    phnguyen    01/05/05 - Fix issues with restoration of new files
#                           Put timestamp check point into
#                             Start of log file
#                             Start of system patching
#                             Start of inventory update
#                             Start of RAC file propagation
#                             End of session
#    phnguyen    01/04/05 - Use getuid() to check for root, overriden by
#                                OPATCH_SKIP_ROOT_CHECK
#    shgangul    01/04/05 - Copy filemap with comps.xml 
#    shgangul 12/31/04 - Bug 4085436: Added new error for make failure 
#    phnguyen    12/30/04 - 
#                           User the env. var. 'USER' to check if OPatch is 
#                              invoked as 'root'.  This 'USER' should be avai.
#                              on all platforms since Java uses it as system
#                              property 'user.name'.  Tested on Solaris & Linux.
#                              Fix has no effect on Windows (tested on Windows).
#                              (bug 4085419)
#                           (COMMENT OUT)
#    shgangul 12/22/04 - Display make error properly to stdout 
#    shgangul 12/17/04 - Use getArchiveCommand() to get ar command 
#    shgangul 12/17/04 - Do not display version info 
#    phnguyen 12/03/04 - take out restriction on -local and
#                           -local_node being used together
#                      - comment out the new code where we can
#                           warn users that if they apply P,
#                           we have to roll back P1, P2...
#                      - We comment it out because of uncertainty
#                           in case of applying same P multiple times
#                      - -local and -local_node can be used together
#
#    phnguyen    10/29/04   Prepare for 1.52 Release
#
#                           - fix "make" regular expression
#    ----------------------------------------------------------------
#    phnguyen    11/12/04 - Use $ORACLE_HOME instead of ORACLE_HOME in <ID>_make.txt
#    phnguyen    11/11/04 - Add -no_relink
#    phnguyen    10/14/04 - Use getMakeCommand() to correctly reg-exp.
#                             the 'make' entry in make.txt which was
#                             put there by Apply.
#                           Do not use clean_up_error to decide to call
#                             propagate_patch_to_RAC_nodes() or not.  Use
#                             only shouldPropagateChange() so that RB is
#                             symmetrical to Apply
#    phnguyen    10/07/04 - Add has_command_in_path() for RollBack as well
#    shgangul    10/07/04 - Added help for opatch_pre_end and opatch_post_end
#    phnguyen    10/02/04   - Print out warning about checking binaries on 
#                               remote nodes
#    phnguyen    09/29/04   - print out clearly that patch won't be propagated
#                                if CFS is detected
#                           - remove extra check on $local_node_only because
#                                shouldPropagatePatch() would look into that
#                                Boolean (this is prior to calling
#                                propagate_patch_to_RAC_nodes())
#    phnguyen    09/20/04 - Use  backupCompsXML() function to back up
#                              OH comps.xml to .patch_storage/<ID> after
#                              all pre-reg. check. (only when not called
#                              from Apply)
#
#    phnguyen    06/07/04 - change env. var. to OPATCH_REMOTE_SHELL
#    phnguyen    05/28/04 - check for env. var. OPATCH_RSH_SSH
#    shgangul    05/20/04 - expose env variables for patch operation and add 
#                           new parameters to pre/post 
#    shgangul    05/18/04 - add -opatch_pre_end and -opatch_post_end options 
#    shgangul    04/13/04 - Add -pre and -post options to Apply and Rollback 
#    phnguyen    04/02/04 - use -oh if supplied
#    shgangul    02/19/04 - bug 3436586
#    phnguyen    02/11/04 - use built-in substr for \p problem
#    phnguyen    02/10/04 - fix Unicode prob. in Win. with \p in OH 
#    phnguyen    02/02/04 - work-around for Active State Perl
#    phnguyen    01/30/04 - remove extra "" in Win, fix cdrom probl.
#    phnguyen    12/30/03 - bug 3344568 3344464 
#    shgangul    12/23/03 - bug 3334888: added a new error message 
#    phnguyen    12/18/03 - touch up output in case of pre-script and RAC 
#    phnguyen    12/15/03 - pass in -Dopatch.debug & cluster debug if OPATCH_DEBUG is set to TRUE
#    phnguyen    12/10/03 - fault-tolerant: default to continue 
#    phnguyen    12/09/03 - process stderr of make
#    shgangul    12/09/03 - bug 3257294 
#    phnguyen    12/02/03 - break the Unix chain cmd "cd $path; jar..."
#                           into separate cmds so that it will work on Win.
#    phnguyen    12/01/03 - add retry/delay options
#    shgangul    11/25/03 - java, library path problems 
#    phnguyen    11/25/03 - remove extra / in top-level class file
#    phnguyen    11/18/03 - use opatch package
#    phnguyen    10/22/03 - adjust RAC & fuser prompt depending on -local,
#                              rolling, all-node or min. downtime.
#                         - check & print out RAC error (w/o exit & error code)
#    phnguyen    10/21/03 - touch up stdout
#    phnguyen    10/20/03 - prompt after catting pre.txt (similar to Apply)
#                           similar to Apply in processing pre-patch script
#    shgangul    10/20/03 - 
#    phnguyen    10/10/03 - touch up help 
#    phnguyen    10/08/03 - Add -jre and -jdk option, touch up non-verbose output
#                           rollback_id.sh bug in calling pre/post script
#    shgangul    10/07/03 - touch up OPatch output, bug 3170021 
#    shgangul    10/07/03 - display pre/post.txt irrespective of verbose 
#    shgangul    10/06/03 - fixed typo in pre post install scripts feature 
#    phnguyen    10/03/03 - Pre/post processing & copy to .patch storage
#    shgangul    09/26/03 - Further changes to pre post functionality
#    shgangul    09/24/03 - Add changes for pre post script
#    shgangul    09/19/03 - add -silent option to rollback
#    shgangul    09/11/03 - Logging changes for 10G
#    shgangul    08/13/03 - Code changes for BUG 2899335
#    phnguyen    06/10/03 - Remove call to CheckRACDetails, use
#                              cached info. instead.  v 1.39
#    phnguyen    05/30/03 - Pass in debug to java code
#                           Pre-req checking for OUI and RAC v 1.37
#    phnguyen    04/11/03 - Check if ORACLE_HOME exists, warn
#                           if it's a symbolic link
#                           Resolve the symbolic link ORACLE_HOME
#                             if necessary to be used during
#                             restore & clean-up
#                           If ORACLE_HOME is sym. link, resolve it
#                             in the following function:
#                             - cleanup_after_patch_removal()
#                             - get_and_run_make_commands()
#                             - restore_archive_files()
#                             - restore_copied_files()
#                             - restore_jar_files()
#
#    daevans     02/28/03 - Rolling RAC working, lock mechanism mainly
#                           honored, fix for clustered file systems.
#    daevans     01/30/03 - Initial cut of "rolling RAC" rollback.
#                           To be tested by Siemens.
#    daevans     12/06/02 - Add support for installations that used the
#                           unsupported "invPtrLoc" flag.
#    daevans     10/17/02 - Fix for bugs 2607816, 2569528 and introduce
#                           support for Win98 clinets.
#    daevans     09/30/02 - Bug 2601400 - wrong print statement for $^V.
#    daevans     09/20/02 - Fix for bug 2580726.
#    daevans     09/10/02 - Fix to support sym links that are ignored by
#                           the OUI APIs.
#    daevans     06/01/02 - Many code changes. Stable check-in.
#    daevans     03/22/02 - Inital code.
#
##########################################################################

###############################################################################
###############################################################################
#
#  ------------------------ INITIALIZATION ------------------------
#
###############################################################################
###############################################################################

package RollBack;
@ISA = ("Command");

######
#
# Standard modules:
#
use English;         # Let us say "$CHILD_ERROR" instead of "$?", etc.
use strict;          # Enforce strict variables, refs, subs, etc.

use Cwd();
use File::Basename();

###############################################################################
#
#  ------------------------ DATA STRUCTURES ------------------------
#
###############################################################################
#
# Global data structures are defined in Command.pm.
#
###############################################################################

# list of "must-have" commands.  Need to add more such as "ar", "jar",
#    "make", etc
use constant ROLLBACK_COMMAND_LIST => "fuser";

###############################################################################
#
#  ------------------------ SUBROUTINES ------------------------
#
###############################################################################

###############################################################################
#
# NAME   : abdomen
#
# PURPOSE: Run the command.
#
# INPUTS : $$ARG[command]   - The command that is to be executed.
#          $$ARG[arguments] - The arguments to the command.
#
# OUTPUTS: NONE
#
# NOTES  : 1. This functions as a main body. Since this is OO perl and perl
#             reads in the main body of the packages during initialization
#             to avoid any side effects the functionality of the body has
#             moved to this method.
#          2. The name abdomen was chosen as its a simile for main body and
#             it'll hopefully remain near the top of the file since it starts
#             with "ab". Unless someone wants to really wreck this scheme.
#          3. This can be invoked directly of via the "apply" command when
#             it must force a rollback of conflicting patches. When invoked
#             by "apply" most of the values will have been checked by the
#             "apply" command so they shouldn't fail at this point. If
#             invoked directly by a user then the checks should fail. In
#             either event the program should (and does) the same actions.
#          4. Point 3 is why caller() is used so we don't keep rechecking
#             the database state.
#
###############################################################################
sub abdomen {

    my $debug = $ENV{OPATCH_DEBUG};
    if ( ! ($debug eq "TRUE") ) {
      $Command::DEBUG = "";
    } else {
      $Command::DEBUG = "-DTRACING.ENABLED=TRUE -DTRACING.LEVEL=2 -Dopatch.debug=true ";
    }

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $command   = $$rh_arguments{command};
    my $arguments = $$rh_arguments{arguments};
    my $rh_command_list = $$rh_arguments{rh_command_list};
    my $system_command = "";

    my @arguments = split ( " ", $arguments ) ;

    # Set the operation, so that it can be accessed from anywhere
    if ( ! $Command::ROLLBACK_CALLED_BY_APPLY )
    {
        $Command::opatch_operation = "rollback";
    }

    # Now parse the arguments for this command.
    my $rh_arg_check =
            $this_class -> parse_arguments( { ra_arguments => \@arguments } );

    my $opatch_ver = $this_class -> version;
    opatchIO -> print_message ( {
		    message => "\nOPatch Version $opatch_ver\n" .
		               "Perl Version $]\n"
	    } );

    my $error_flag = $$rh_arg_check{error_flag};

    my $local_node_only = $Command::LOCAL_NODE_ONLY;


    my $search_OH_jar = 1;
    if ( $Command::ROLLBACK_CALLED_BY_APPLY  ) {
      # -jdk / -jre if supplied, have been processed by Apply.pm
       if ( $Command::JDK_LOC ne "" || $Command::JRE_LOC ne "") {
            $search_OH_jar = 0;
       }
    } else {
        # Make sure 'fuser' is in PATH
        my $has_commands = $this_class -> has_commands_in_path(
                                          $this_class -> ROLLBACK_COMMAND_LIST
                                                            );

        if ( $has_commands == 0 ) {
           my $list = $this_class -> ROLLBACK_COMMAND_LIST;
           opatchIO -> print_message_and_die ( {
                       exit_val => $this_class -> ERROR_CODE_PATH_PROBLEM,
                       message =>
                       "Some commands below are not in your path:\n" .
                       "  Command= $list\n" .
           "  Extra Search Path= /sbin/fuser, /sbin/fuser\n" .
                       "  Path= $ENV{PATH}\n"
               });
         }

        # Check cmd. line syntax
        if ( $Command::JRE_LOC ne "" && $Command::JDK_LOC ne "" ) {
           opatchIO->print_message_and_die({
              exit_val => $this_class->ERROR_CODE_COMMAND_LINE_ARGUMENT,
              message=>
                "Please specify either -jre or -jdk but not both.\n"
            });
        }
       
        # Make sure the JDK or JRE LOCs are valid
        my $target = "";
        my $start_point = "";
        my $result = "";
        my $skip_jre = 0;
        
        if ( $Command::JDK_LOC ne "" ) {
          # If -jdk is used, look for "jar"
          $search_OH_jar = 0;
          
          $start_point = $Command::JDK_LOC;
          $target  = "jar";
          if ( $OSNAME =~ m#Win32# ) {
             $target = "jar.exe";
          }
          opatchIO -> print_message ( { message => "-jdk specified, need to find " .
                                                   "executable jar from $Command::JDK_LOC\n"
                                    } );
          
          $result = $this_class -> find_executable_file ( { start_point => $start_point,
                                          target => $target } );
          if ( $result ne "" ) {
            $Command::JDK_LOC = $result;
            opatchIO -> print_message ( { message => "JAR set to $result\n" } );                 
          }
          # Then look for "java" as well
          $target  = "java";
          if ( $OSNAME =~ m#Win32# ) {
             $target = "java.exe";
          }
          opatchIO -> print_message ( { message => "-jdk specified, need to find " .
                                                   "executable java from $Command::JDK_LOC\n"
                                    } );

          my $default_path_1 = File::Spec -> catfile ($start_point,
                                      "bin", $target);
          my $default_path_2 = File::Spec -> catfile ($start_point,
                                      "jre", "bin", $target);

          # Bug 4883881 : First look into <user specified path>/bin/java, if not then look for 
          # <user specified path>/jre/bin/java, if not found
          # then use find and look for the path as usual.
          if ( -x $default_path_1 ) {
                $result = $default_path_1;
          } elsif ( -x $default_path_2 ) {
                $result = $default_path_2;
          } else {
                $result = $this_class -> find_executable_file ( { start_point => $start_point,
                                                   target => $target } );
          }
          
          if ( $result ne "" ) {
            $Command::JRE_LOC = $result;
            opatchIO -> print_message ( { message => "JAVA set to $result\n" } );  
            $skip_jre = 1;
          }      
        } else {
          $search_OH_jar = 1;
        }
    
        # Note: $Command::JRE_LOC could have been set by -jdk option above ($skip_jre set to 1)
        if ( $Command::JRE_LOC ne "" && ! $skip_jre ) {
          $start_point = $Command::JRE_LOC;
          $target  = "java";
          if ( $OSNAME =~ m#Win32# ) {
             $target = "java.exe";
          }
          opatchIO -> print_message ( { message => "-jre specified, need to find " .
                                                   "executable java from $Command::JRE_LOC\n"
                                    } );
          
          my $default_path = File::Spec -> catfile ($start_point,
                                      "bin", $target);

          # Bug 4883881 : First look into <user specified path>/bin/java, if not 
          # then use find and look for the path as usual.
          if ( -x $default_path ) {
                $result = $default_path;
          } else {
                $result = $this_class -> find_executable_file ( { start_point => $start_point,
                                                   target => $target } );
          }
          
          if ( $result ne "" ) {
            $Command::JRE_LOC = $result;
            opatchIO -> print_message ( { message => "JAVA set to $result" } );                 
          }
        }
    }
    
    if ($Command::NO_RELINK_MODEk) { 
        opatchIO->print_message ({ message =>
           "-no_relink: OPatch will not execute any 'make' action.\n"
        });
    }
    
    #if ($Command::LOCAL_NODE_ONLY && $Command::LOCAL_NODE_NAME_MODE) { 
    #  opatchIO->print_message_and_die({
    #      exit_val => $this_class->ERROR_CODE_COMMAND_LINE_ARGUMENT,
    #      message=> "-local and -local_node <node name> cannot be used together.\n"
    #	      });
    #  
    #}

    my $badRetry = $this_class -> isBadRetryDelaySemantics();
    if ( $badRetry ) {
       opatchIO->print_message_and_die({
          exit_val => $this_class->ERROR_CODE_COMMAND_LINE_ARGUMENT,
          message=> $this_class->RETRY_ERROR_MESSAGE
	      });
    }

    # For any errors display the problem, give command usage then exit.
    if ( $error_flag ) {
        opatchIO->print_message( {message => "$error_flag"} );
        $this_class -> display_help( { command => $command,
                                       rh_command_list => $rh_command_list } );
        opatchIO->print_message_and_die({
          exit_val => $this_class->ERROR_CODE_COMMAND_LINE_ARGUMENT,
          message=>
            "Error parsing arguments\n"
	      });
    }

    my $safety_check = 1;
    my ( $package, $filename, $line ) = caller;
    if ( $package =~ m#^Apply$# ) {
        $safety_check = 0;
    }

    my $noop_flag   = $$rh_arg_check{noop_flag};
    my $Oracle_Home = $$rh_arg_check{Oracle_Home};
    my $patch_inventory_location = $$rh_arg_check{patch_home} || &Cwd::cwd();
    my $patch_id = $$rh_arg_check{patch_id};

    # If rollback is called directly, then check.
    if( $package !~ m#^Apply$# ) {
        # If the path to patch location exists, then check if patch_id is there in its path.
        if( $patch_inventory_location && ( length($patch_inventory_location) >= length($patch_id) ) &&
                             ( $patch_inventory_location !~ m#$patch_id# ) ) {
            opatchIO -> print_message_and_die ( {
                  exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
                  message => "The patch location specfied is not of the patch $patch_id." } );
        }
    }


    my $rh_file_names = $this_class -> build_required_XML_filenames ();
    my $actions_file = $$rh_file_names{actions};
    my $inventory_file = $$rh_file_names{inventory};
    my $GenericActions_file = $$rh_file_names{GenericActions};

    # User the env. var. 'user' to make sure invoker is not Root
    # Java refers to this as user.name
    # User the env. var. 'user' to make sure invoker is not Root
    # Java refers to this as user.name
    my $skipRootCheck = $ENV{OPATCH_SKIP_ROOT_CHECK};
    if ( $skipRootCheck eq "" ) {
      if ( ! ($OSNAME =~ m#Win32#) ) {    
         my $uid = POSIX::getuid();
         if ( $uid == 0 ) { 
           opatchIO -> print_message_and_die ( {
                 exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
                 message  => $this_class->INVOKED_BY_ROOT_ERROR_MESSAGE
                 } );      
         }
      }
    }

    $this_class -> get_os_id();
    my $os_id = $Command::OS_ID;
        
    # Process env. var. OPATCH_IS_SHARED
    $Command::OPATCH_IS_SHARED = $ENV{"OPATCH_IS_SHARED"} || "";
     
    my $rh_return_values = Apply -> check_patch_area_validity ( {
                                      p_i_l => $patch_inventory_location,
                                      rh_required_files => $rh_file_names } );

    $patch_inventory_location = $$rh_return_values{patch_inventory_location};

    if ( ! $patch_inventory_location ) {
        opatchIO -> print_message_and_die ( {
                        exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
                        message => "The patch area isn't valid. " } );
    }

    # Check the backup directory exists.
    my $rollback_area = File::Spec -> catfile ( "$Oracle_Home",
                                                 $this_class -> save_dir,
                                                 $patch_id );
    my $patch_storage_directory = $rollback_area;
    if ( ! -e $rollback_area ) {
        opatchIO -> print_message_and_die ( {
              exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
              message => "The backup area [ $rollback_area ] is missing." } );
    }

    # Check if the pre script is there during this roll back session so that
    #   we don't have to do fuser
    my $script_loc = "";
    my $script_name= "";
    
    if ( $OSNAME =~ m#Win# )
    {
        $script_loc = File::Spec -> catfile ($patch_storage_directory,
                                      "custom", "scripts", $Command::WIN_PRE_SCRIPT);
        $script_name = $Command::WIN_PRE_SCRIPT;
    }
    else
    {
        $script_loc = File::Spec -> catfile ($patch_storage_directory,
                                      "custom", "scripts", $Command::UNIX_PRE_SCRIPT);
        $script_name = $Command::UNIX_PRE_SCRIPT;
    }
    if ( -e $script_loc ) {
      $Command::ROLLBACK_PRE_SCRIPT_IS_THERE = 1;
    } else {
      $Command::ROLLBACK_PRE_SCRIPT_IS_THERE = 0;
    }


    # This is needed for invoking and running the java VM.
    my $rh_OUI_file_names = $this_class -> build_required_OUI_filenames ( {
                                              Oracle_home => $Oracle_Home } );

    if ( $$rh_OUI_file_names{error_flag} ) {
        opatchIO -> print_message_and_die ( {
                              exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
                              message => $$rh_OUI_file_names{error_flag} } );
    }

    my $rh_java_paths = $this_class -> make_java_paths ( {
	    Oracle_Home       => $Oracle_Home,
	    rh_OUI_file_names => $rh_OUI_file_names } );

    # Due to so many problems with RAC, check its sanity upfront
    #
    $this_class -> validate_OUI_and_RAC( {
          Oracle_Home => $Oracle_Home,
          rh_OUI_file_names => $rh_OUI_file_names,
          rh_java_paths => $rh_java_paths
    } );

    # Setup the logging area.
    my $rh_log_creation_details = $this_class -> create_logging_file ( {
                                              patch_id    => $patch_id,
                                              Oracle_Home => $Oracle_Home } );

    my $log_failure   = $$rh_log_creation_details{error_flag};
    my $fh_log_file   = $$rh_log_creation_details{fh_log_file};
    my $log_file_name = $$rh_log_creation_details{log_file_name};

    if ( $log_failure ) {
        opatchIO -> print_message_and_die ( {
            exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
            message  => "Problems with creating the log file:\n " .
                                              "$log_failure" } );
    }

    my $timestamp = $this_class -> get_current_timestamp();
    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "Starting OPatch Rollback session at " .
                                              "$timestamp.\n" } );

    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "Command arguments are: " .
                                              "$arguments\n" } );

    # Create the lock file.
    my $rh_lock_check_details = $this_class -> check_patch_lock_status ( {
                                              patch_id  => $patch_id,
                                              Oracle_Home => $Oracle_Home } );

    my $lock_file_errors  = $$rh_lock_check_details{lock_file_errors}  || "";
    my $patched_RAC_nodes = $$rh_lock_check_details{patched_RAC_nodes} || "";
    my $lock_class        = $$rh_lock_check_details{lock_class}        || "";

    if ( $lock_file_errors ) {
        $lock_file_errors = "Problems with the lock file\n" .
                                                            $lock_file_errors;
        opatchIO -> print_message_and_die ( {
            exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
            message  => $lock_file_errors} );
    }

    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "Checking the patch " .
                                              "inventory.\n" } );

    # Check validity of patch location and directory name.
    if ( ( ! $patch_inventory_location ) || ( $patch_id !~ m#^(\d+)$# ) ) {
        if ( ! $lock_class ) {
            $lock_file_errors = $this_class -> free_the_lock_file ( {
                                              Oracle_Home => $Oracle_Home,
                                              patch_id    => $patch_id } );
        }

        opatchIO -> print_message_and_die ( {
                       exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
                       message => "Not a valid patch area.\n" .
                              "Check the command line, current " .
                              "directory location and permissions of " .
                              "the patch directory." } );
    }

    # Read the file that has the "command type" and the associated action.
    my $rh_action_map = Apply -> get_command_to_action_mapping ( {
                              GenericActions => $GenericActions_file,
                              Oracle_Home    => $Oracle_Home,
                              os_id          => $os_id,
                              patch_location => $patch_inventory_location } );

    if ( $$rh_action_map{error_flag} ) {
        if ( ! $lock_class ) {
            $lock_file_errors = $this_class -> free_the_lock_file ( {
                                              Oracle_Home => $Oracle_Home,
                                              patch_id    => $patch_id } );
       }

        opatchIO -> print_message ( { f_handle => $fh_log_file,
                              message => $$rh_action_map{error_flag} } );
        opatchIO -> print_message_and_die ( {
                              exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
                              message => $$rh_action_map{error_flag} } );
    }

    my $rh_actions_list = $$rh_action_map{rh_actions};

    # If a safety check is needed get the patch details at the same time.
    # Otherwise just call get_patch_details by itself.
    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "Checking system activity." .
                                              "\n" } );

    opatchIO -> print_message ( { message => "Retrieving details of patch " .
                                             "$patch_id.\n" } );

    # Retrieve the details of the patch to rollback and more details to check.
    my $patch_details = $this_class -> get_rollback_details ( {
                                  fh_log_file       => $fh_log_file,
                                  Oracle_Home       => $Oracle_Home,
                                  patch_id          => $patch_id,
                                  rh_OUI_file_names => $rh_OUI_file_names } );

    my $ra_file_list = $$patch_details{ra_files};
    my $patch_found  = $$patch_details{patch_found};
    
    opatchIO -> print_message ( { message => "Retrieving patch to roll back: patch_found = $patch_found." } );
    
    if ( $patch_found == 0 ) {
        if ( ! $lock_class ) {
            $lock_file_errors = $this_class -> free_the_lock_file ( {
                                              Oracle_Home => $Oracle_Home,
                                              patch_id    => $patch_id } );
        }

        opatchIO -> print_message_and_die ( { exit_val => $this_class->ERROR_CODE_ROLLBACK_PROBLEM,
                  message => 
                  "OPatch didn't find patch $patch_id in inventory.\n" .
                  "No files are associated with this patch.\n" } );
    }

    # Doing fuser check after checking if patch is present in the OH
    # Bug-4640904
    my $active_files = "";
    if ( $safety_check ) {
        opatchIO -> print_message ( { message => "Performing pre-backout " .
                                                             "checks.\n" } );

        # Check the details about database.
        # This will call Apply::read_and_check_reference_details()
        #   which loads the "inventory" file pointed to by -ph and modifies
        #   the ROLLING_PATCH flag.
        
        my $rh_rollback_checks = $this_class -> rollback_safety_check ( {
                           fh_log_file       => $fh_log_file,
                           inventory_file    => $inventory_file,
                           Oracle_Home       => $Oracle_Home,
                           patch_id          => $patch_id,
                           patch_location    => $patch_inventory_location } );

        $active_files = $$rh_rollback_checks{files_in_use};
    }

    # If there is any activity on critical files it's time to exit.
    if ( $active_files ) {

        if ( ! $lock_class ) {
            $lock_file_errors = $this_class -> free_the_lock_file ( {
                                        Oracle_Home     => $Oracle_Home,
                                        patch_id        => $patch_id } );
        }

        my $message = "Problems when checking for files that are active.\n" .
                                                        "$active_files\n\n" ;
        opatchIO -> print_message ( { f_handle => $fh_log_file,
                                      message  => $message } );
        opatchIO -> print_message_and_die ( {
                      exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
                      message  => $message } );
    }


    my $ra_rac_nodes = $Command::NODE_LIST;
    my $rh_file_list = $this_class -> parse_rollback_data ( {
                                         ra_file_list    => $ra_file_list,
                                         search_OH_jar   => $search_OH_jar,
                                         Oracle_Home     => $Oracle_Home,
                                         patch_id        => $patch_id
                                         } );

    # Partially installed RAC patch or no inventory.
    if ( ! keys %$rh_file_list ) {
        $Command::patch_missing_from_inventory = 1;
        $ra_file_list = $this_class -> check_for_noninventory_files ( {
                                  Oracle_Home       => $Oracle_Home,
                                  patch_id          => $patch_id } );

        $rh_file_list = $this_class -> parse_rollback_data ( {
                                         ra_file_list    => $ra_file_list } );

    }

    my $RAC_check_error_flag = "";
    my $shutdown_needed_flag = 0;
    my $shutdown_RAC_flag    = 0;

    # Check if user defines OPATCH_REMOTE_SHELL
    my $isRSH_SSH = $ENV{OPATCH_REMOTE_SHELL};
    if ( $isRSH_SSH eq "" ) { 
      $Command::RSH_SSH = "rsh";
    } else {
      $Command::RSH_SSH = $isRSH_SSH;
      my $rsh_msg = "Using user-supplied $Command::RSH_SSH to run make " .
                    "on remote nodes for Rollback if applicable.\n";
      opatchIO->print_message ({ message => $rsh_msg }); 
    }

    if ( $this_class -> shouldPropagateChange() ) {
       # The rest of this conditon is hack to support rolling RAC for
       # backing out a patch. The XML inventory is copied into the
       # patch area so we can read it later. Consider moving the
       # details into the inventory - investigation needed.
       my $rh_RAC_patch_details = $this_class -> get_initial_details ( {
                              Oracle_Home       => $Oracle_Home,
                              patch_id          => $patch_id } );
       $shutdown_needed_flag = $$rh_RAC_patch_details{shutdown};
       $shutdown_RAC_flag    = $$rh_RAC_patch_details{RAC_shutdown};

       if ( ( exists $$rh_RAC_patch_details{error_flag} ) ) {
             $RAC_check_error_flag =  $$rh_RAC_patch_details{error_flag};
       }
       opatchIO->debug({message =>
		     "Rollback will propagate changes to other nodes:\n" .
		     "  shutdown_needed_flag = $shutdown_needed_flag\n" .
		     "  shutdown_RAC_flag    = $shutdown_RAC_flag\n" .
		     "  RAC_check_error_flag = $RAC_check_error_flag\n"
	     });
    } else {
      opatchIO->debug({message =>
         "Rollback will not propagate changes to other nodes\n"
      });
    }
    # If the lock file was left by another class (Apply) negate the list
    # of nodes to be patched.
    if ( ( $lock_class ) && ( $this_class ne $lock_class ) ) {

        # OK, it may not be in the inventory at this point. Ugly.
        $Command::patch_missing_from_inventory = 1;
        my $temp_node_list = "";
        my @nodeList = split ' ', $Command::NODE_LIST;
        foreach my $node ( @nodeList ) {
            next if ( $patched_RAC_nodes =~ m#$node# );
            $temp_node_list .= "$node ";
        }

        $patched_RAC_nodes = $temp_node_list;

        opatchIO->debug({message=>
          "Lock file left by another class (Apply), negate the list\n" .
          "  of nodes to be patched:\n" .
          "  temp_node_list = \"$temp_node_list\"\n"
                      });
         }

        if ( ! $RAC_check_error_flag eq "" ) {
          if ( ! $lock_class ) {
            $lock_file_errors = $this_class -> free_the_lock_file ( {
                                              Oracle_Home => $Oracle_Home,
                                              patch_id    => $patch_id } );
        }

        opatchIO -> print_message ( { f_handle => $fh_log_file,
                        $RAC_check_error_flag });
        opatchIO -> print_message_and_die ({
                      exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
                      message => $RAC_check_error_flag });
    }

    my $all_RAC_nodes = $Command::NODE_LIST;
    my $local_RAC_node= $Command::LOCAL_NODE_NAME;
    my $shouldPropagateChange = $this_class -> shouldPropagateChange();
    my $is_CFS = $this_class -> is_CFS();


    # Clustered file systems are treated as a single system.
    # I'm told shutting down one node shuts all instances down.
    if ( $is_CFS == 1 ) {
        $all_RAC_nodes = "";
        $Command::NODE_COUNT = 0;
        $Command::NODE_LIST  = "";
        opatchIO->debug({message=> "CFS detected, setting NODE_LIST to be empty\n"});
    }

    # If user supplied -local_node, it must match one of the element
    #   in the node list
    if ( $shouldPropagateChange && $Command::LOCAL_NODE_NAME_MODE ) {
        my $found = 0;
        my @nodes          = split ( " ", $all_RAC_nodes );
        foreach my $target ( @nodes ) {
          if ( $target eq $local_RAC_node ) { 
            $found = 1;
          }
        }
        # Exit if the local node name doesn't match one in the node list
        if ( !$found ) { 
            my $message =
            "The supplied \"$local_RAC_node\" does not match one of " .
            "the nodes in node list \"$all_RAC_nodes\"." ;
            opatchIO -> print_message_and_die ( { message  => $message ,
                          exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM } );
        
        }
      
    }

    # Currently all we can reliably do is warn the user to check processing
    # has stopped on other nodes. APIs are being written but probably won't
    # be ready until after this is delivered.
    if ( $shouldPropagateChange ) {

        my $message = "All processes associated with this instance need to " .
                      "be inactive.\n" . "The machines involved in this " .
                      "RAC installation are:\n" . "   $all_RAC_nodes\n" .
                      "Please check the status of these instances before " .
                      "responding to the following question.\n\n";
        opatchIO -> print_message ( { f_handle => $fh_log_file,
                                      message  => $message } );
    }


    # Back up comps.xml as comps.xml.b4rollback
    if (! $Command::ROLLBACK_CALLED_BY_APPLY) {    

    # Back up original comps.xml to .patch_storage as "comps.xml.b4rollback"
      my $backup_filemap = $ENV{OPATCH_BACKUP_FILEMAP};
      if ( $backup_filemap eq "TRUE" )
      {
    opatchIO->print_message_noverbose ( { 
                   message =>
                             "Backing up comps.xml and filemap ...\n"
                                         } );
      }
      else
      {
        opatchIO->print_message_noverbose ( { 
                   message =>
                             "Backing up comps.xml ...\n"
                                         } );
      }

      $this_class->backupCompsXML({ Oracle_Home => $Oracle_Home,
                                  patch_id    => $patch_id,
                                  fh_log_file => $fh_log_file,
                                  target_name => "comps.xml.b4rollback",
                                  target_filemap_name => "filemap.b4rollback"
                               });
    }

    # Call pre Rollback script if available
    # For Windows call pre.bat and for non windows call pre
    my $script_txt_loc = "";
    my $script_txt_handle;
    my @script_txt_lines;
    my $script_line = "";
    my $script_result = "";

    # script_loc and script_name have been set up previously
    $script_txt_loc = File::Spec -> catfile ($patch_storage_directory,
                                      "custom", $Command::PRE_README);

    # cat pre readme to stdout
    opatchIO->print_message ( { message => "Looking for pre-patch readme $script_txt_loc\n" } );
    if ( -e $script_txt_loc )
    {
        opatchIO->print_message ( { message => "  printing content of $script_txt_loc to stdout\n" } );
        local *script_txt_handle;
        opatchIO->print_message ( { f_handle => $fh_log_file, message => $Command::PRE_README_HEADER } );
        open(script_txt_handle, $script_txt_loc) || opatchIO->print_message ( { message => "Could not open file $script_txt_loc" } );
        @script_txt_lines = <script_txt_handle>;
        foreach $script_line (@script_txt_lines)
        {
            chomp $script_line;
            opatchIO->print_message_noverbose ( { message => $script_line } );
        }
        opatchIO->print_message ( { f_handle => $fh_log_file, message => $Command::PRE_README_TAIL } );
        opatchIO->print_message_noverbose ( { message => "\n" } );
        close script_txt_handle;
    }

    # Ask if users want to continue
    my $question = "";
    my $answer = "";
    
    #
    # When this patch was applied, if we had rolled back some other patches,
    #   we need to inform user about it.
    #
    # my $hasRolledBack = $this_class -> printRollbackedPatches ( {
    #          Oracle_Home => $Oracle_Home,
    #          patch_id    => $patch_id,
    #          f_handle => $fh_log_file
    # } );
    
    # if ($hasRolledBack) { 
    #  $question = "Do you want to continue?";
    #  $answer = opatchIO -> check_to_stop_processing ( {
    #                                              message => $question } );
    #
    # 
      # User requested the process stop.
    #  if ( $answer eq "N" ) {
    #      if ( ! $lock_class ) {
    #          $lock_file_errors = $this_class -> free_the_lock_file ( {
    #                                            Oracle_Home => $Oracle_Home,
    #                                            patch_id    => $patch_id } );
    #      }
  
    #      my $message = "Patch removal stopped at user's request.";
    #      opatchIO -> print_message ( { f_handle => $fh_log_file,
    #                                    message  => $message } );
    #      opatchIO -> print_message_and_die ( {
    #                    exit_val => $this_class->ERROR_CODE_PATCHING_STOPPED_ON_USER_REQUEST,
    #                    message => $message } );
    #  }
    #}

    # Do not support min. down-time or rolling mode in auto-rollback
    if ( $Command::ROLLBACK_CALLED_BY_APPLY ) {
          $Command::ROLLING_PATCH = 0;
          $Command::MINIMIZE_DOWNTIME_OPTION = 0;
          opatchIO -> print_message_noverbose ( {
            message => "Rolling back with all-node mode."
          } );
    }

    opatchIO->print_message_noverbose ( {
      message => $Command::RAC_MESSAGE
    } );

    if (  $shouldPropagateChange ) {
        $question =
           "This node is part of an Oracle Real Application Cluster.\n" .
           "  Local node is \"$local_RAC_node\"\n" .
           "  Nodes are: $all_RAC_nodes\n";
           
            if ( $local_node_only ) {
              # actually, if it's -local option, shouldPropagateChange()
              #   has been turned off, so we will never get to this line
            } elsif ( $Command::ROLLING_PATCH ) {
              $question = $question .
              "(patching mode = rolling)\n" .
              "Please shut down Oracle instances running out of this ORACLE_HOME on this node.\n" .
              "(Oracle Home = $Oracle_Home)\n" .
              "Is this node ready for updating?";
            } elsif ( $Command::MINIMIZE_DOWNTIME_OPTION ) {
              # there is no minimize_downtime option for RAC RollBack, but this is just in
              # case RollBack is called by Apply
              $question = $question .
              "(patching mode = minimize down-time)\n" .
              "Please shut down Oracle instances running out of this ORACLE_HOME on this node.\n" .
              "(Oracle Home = $Oracle_Home)\n" .
              "Is this node ready for updating?";
            } else {
              $question = $question .
              "(patching mode = all-node)\n" .
              "Please shut down all Oracle instances running out of this ORACLE_HOME on all nodes.\n" .
              "(Oracle Home = $Oracle_Home)\n" .
              "Are all the nodes ready for updating?";
            }
    } else { 
           $question =
           "Please shut down Oracle instances running out of this ORACLE_HOME\n" .
           "(Oracle Home = $Oracle_Home)\n" .
           "Is this system ready for updating?";

    } 
    
    #
    # Do not prompt for shut down during auto-rollback
    #
    if ($Command::ROLLBACK_CALLED_BY_APPLY) {
        my $message = "Oracle instances have been shut down, proceeding with auto-rollback.\n";
        opatchIO -> print_message_noverbose ( { f_handle => $fh_log_file,
                                      message  => $message } );
        $answer = "";
    } else {
       $answer = opatchIO -> check_to_stop_processing ( {
                                                message => $question } );
    }

    # User requested the process stop.
    if ( $answer eq "N" ) {
        if ( ! $lock_class ) {
            $lock_file_errors = $this_class -> free_the_lock_file ( {
                                              Oracle_Home => $Oracle_Home,
                                              patch_id    => $patch_id } );
        }

        my $message = "Patch removal stopped at user's request.";
        opatchIO -> print_message ( { f_handle => $fh_log_file,
                                      message  => $message } );
        opatchIO -> print_message_and_die ( {
                      exit_val => $this_class->ERROR_CODE_PATCHING_STOPPED_ON_USER_REQUEST,
                      message => $message } );
    }

    # Set the environment variables for the pre scripts to access
    $ENV{OPATCH_OPERATION} = $Command::opatch_operation;
    if (!$Command::ROLLBACK_CALLED_BY_APPLY)
    {
        $ENV{OPATCH_PATCHID} = $patch_id;
    }

    my $timestamp = $this_class -> get_current_timestamp();
    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "\nStarting to remove patch from the system at " .
                                              "$timestamp.\n" } );

   # Execute pre script
    opatchIO->print_message ( { message => "Looking for pre-patch script $script_loc\n" } );
    if ( (-e $script_loc) && (!$noop_flag) )
    {
        opatchIO->print_message ( { message => "  preparing to execute $script_loc\n" } );
        $system_command = "$script_loc -rollback $patch_id $Command::PRE_PARAMETERS";
        opatchIO->print_message_noverbose ( { message => "Executing the Roll-back pre-patch script ($script_loc)..." } );        
        #$script_result = qx/$system_command/;

	$this_class -> run_cmd_print_live ( { sys_command => $system_command,
                               level => 2 });


        my $return_code = $this_class -> get_child_error ( {
          CHILD_ERROR     => $CHILD_ERROR
        } );
        opatchIO -> print_message ( { f_handle => $fh_log_file,
                                      message  => $system_command
                                  } );
        opatchIO->print_message( { f_handle => $fh_log_file, message => $Command::PRE_SCRIPT_HEADER } );
        opatchIO->print_message_noverbose ( { message => $script_result } );
        opatchIO->print_message( { f_handle => $fh_log_file, message => "exit code = \"$return_code\"" } );
        opatchIO->print_message( { f_handle => $fh_log_file, message => $Command::PRE_SCRIPT_TAIL });
                                  
        # this would remember that pre-script was run and remember the return code as well
        $this_class -> save_rollback_prescript_return_code ( {
           return_code => $return_code
        } );
        
        #
        # if pre-script returns error, we ask users if they want to exit
        #
        if ( $Command::ROLLBACK_PRE_SCRIPT_STATUS == $Command::PREPOST_ERROR ) {
          
          $question =
            "The pre patch script returned an error.  Do you want to STOP?" ;

          $answer = opatchIO -> check_to_stop_processing ( {
                                                message => $question } );
          if ( $answer eq "" ) {
            $this_class -> free_the_lock_file ( {
                                    Oracle_Home => $Oracle_Home,
                                    patch_id    => $patch_id } );
            opatchIO->print_message_and_die({
                exit_val => $this_class->ERROR_CODE_PRE_SCRIPT,
                message=> "Pre-patch script error"
            });
          }
        }
      
    }

    opatchIO->print_message_noverbose ( { message => "Removing patch $patch_id...\n" } );

    # The values in this are adjusted later for cleaning. Need to copy this
    # for RAC installations.
    my @rac_file_list = ();
    foreach my $rac_file (@$ra_file_list) {
        push ( @rac_file_list, $rac_file);
    }

    # Now lets remove the patch from the files.
    my $patch_removal_errors = $this_class -> remove_an_applied_patch ( {
                                        fh_log_file     => $fh_log_file,
                                        noop_flag       => $noop_flag,
                                        Oracle_Home     => $Oracle_Home,
                                        patch_id        => $patch_id,
                                        rh_actions_list => $rh_actions_list,
                                        ra_file_list    => $ra_file_list } );

   my $question = "";
   my $answer = "";

   if ( $patch_removal_errors ) {

        $question = "OPatch encounters the following file roll-back issues:\n" . 
                    $patch_removal_errors . "\n" .
		    "Replying 'Y' will terminate the patch roll-back immediately. " .
                    "It WILL NOT restore any updates that have been performed to this point. " .
                    "It WILL NOT update the inventory." . "\n" .
                    "Replying 'N' will update the inventory showing the patch has been removed." . "\n" .
                    "\nDo you want to STOP?";
        $answer = opatchIO->check_to_stop_processing({message=>$question});
        
        if($answer eq "") {
          opatchIO -> print_message_and_die ( {
                                        exit_val => $this_class->ERROR_CODE_PATCH_PROBLEM,
                                        message => "File Back-up Errors!"} );
        } else {
          opatchIO -> print_message ({ f_handle => $fh_log_file,
                                       message => "Ignore file roll-back issues and continue...\n" }); 
        }
    }

    my $make_failure = $this_class -> get_and_run_make_commands ( {
                                        fh_log_file     => $fh_log_file,
                                        noop_flag       => $noop_flag,
                                        Oracle_Home     => $Oracle_Home,
                                        patch_id        => $patch_id } );

    if ( $make_failure ) {

        $question = "OPatch encounters the following Make issues:\n" . $make_failure . "\n" .
		    "Replying 'Y' will terminate the patch rollback immediately. " .
                    "It WILL NOT restore any updates that have been performed to this point. " .
                    "It WILL NOT update the inventory." . "\n" .
                    "Replying 'N' will continue with updating the inventory showing the patch has been removed." . "\n" .
                    "\nDo you want to STOP?";
        $answer = opatchIO->check_to_stop_processing({message=>$question});
        if($answer eq "") {
          opatchIO -> print_message_and_die ( {
                                        exit_val => $this_class->ERROR_CODE_PATCH_PROBLEM,
                                        message => "Make Error!"} );
        } else {
          opatchIO -> print_message ({ f_handle => $fh_log_file, 
                                       message => "Ignore Make issues and continue...\n" }); 
        }
    }

    if ( ! $noop_flag ) {
        my $clean_up_errors = "";
        if ( $OSNAME =~ m#Win32# ) {
             $clean_up_errors = $this_class -> cleanup_after_patch_removal_windows ( {
                                                Oracle_Home  => $Oracle_Home,
                                                ra_file_list => $ra_file_list,
                                                patch_id     => $patch_id } );
        } else {
             $clean_up_errors = $this_class -> cleanup_after_patch_removal_unix ( {
                                                Oracle_Home  => $Oracle_Home,
                                                ra_file_list => $ra_file_list,
                                                patch_id     => $patch_id } );
        }
        if ( $clean_up_errors ) {
            opatchIO -> print_message ( { f_handle => $fh_log_file,
                                          message  => $clean_up_errors } );
        }

        my $rac_errors = "";
        # if ( ( scalar(@$ra_rac_nodes) )  && ( ! $clean_up_errors ) ) {

        my $rac_errors = "";
        if ( $shouldPropagateChange )  {

            my $timestamp = $this_class -> get_current_timestamp();
            opatchIO -> print_message ( { f_handle => $fh_log_file,
                                          message  => "\nStarting to propagate patches for Rollback to nodes at " .
                                                      "$timestamp.\n" } );
                                                      
            # (daevans' comment) Due to tasking for several weeks the pre-req checks
            # needed to really make this work safely are missing.
            # Also see the Apply call for check_patch_lock_status().

            opatchIO->debug({message=>
              "Rollback::propagate_patch_to_RAC_nodes():\n" .
              "  all_nodes = \"$all_RAC_nodes\"\n" .
              "  local_node= \"$local_RAC_node\"\n" .
              "  patched_RAC_nodes = \"$patched_RAC_nodes\"\n"
              });

            # The following function returns:
            #       empty string if success
            #       empty string if there are errors and users decide to go on
            #       sys_call_result (result of calling Java programs) if there are
            #          fatal fatal errors or (errors, prompted, users want to quite)
            $rac_errors = $this_class -> propagate_patch_to_RAC_nodes ( {
                                  all_nodes         => $all_RAC_nodes,
                                  fh_log_file       => $fh_log_file,
                                  local_node        => $local_RAC_node,
                                  general_options   => 0,
                                  general_shutdown  => $shutdown_needed_flag,
                                  Oracle_Home       => $Oracle_Home,
                                  patch_id          => $patch_id,
                                  patched_RAC_nodes => $patched_RAC_nodes,
                                  RAC_options       => 0,
                                  rh_file_list      => $rh_file_list,
                                  rh_OUI_file_names => $rh_OUI_file_names,
                                  shutdown_RAC_flag => $shutdown_RAC_flag } );
        } elsif ( $local_node_only ) {
            opatchIO->print_message({message =>
                "-local mode: patch is not propagated to other nodes\n"
            });
        } elsif (  $is_CFS == 1 ) { 
           opatchIO->print_message({message =>
              "CFS detected: patch is not propagated to other nodes.\n"
           });
        }

        # Due to bug in auto-rollback (when Apply detects conflict
        #   but users decide to continue, which causes a rollback,
        #   the bug is about busy lock which somehow causes
        #   $Command::patch_missing_from_inventory to be set to 1,
        #   therefore Rollback::update_inventory() is not called.
        # Fix is to check for $Command::ROLLBACK_CALLED_BY_APPLY)
        #   and proceed with Rollback::update_inventory() if this
        #   $Command::ROLLBACK_CALLED_BY_APPLY) is set

        my $inventory_update_error = "";
        if ( $rac_errors ) {
          opatchIO->print_message_noverbose({message =>
                       "There are errors during RAC patching, " .
                       "OPatch will not update inventory"
                       });
        # if ( ! $rac_errors ) {
        } else {
            if ( ! $Command::do_not_update_inventory ) {
                if ( ! $Command::patch_missing_from_inventory ||
                       $Command::ROLLBACK_CALLED_BY_APPLY) {
                       
                        my $timestamp = $this_class -> get_current_timestamp();
                        opatchIO -> print_message ( { f_handle => $fh_log_file,
                                              message  => "Starting to update inventory for Rollback at " .
                                                          "$timestamp.\n" } );

                        opatchIO->print_message_noverbose({message =>
                          "\nUpdating inventory..."
                        });
                        
                        if ($local_node_only) {
                            opatchIO->debug({message=>
                                "now calling update_inventory() with -local option set\n"
                            });
                        } else {
                            opatchIO->debug({message=>
                                "propagate_patch_to_RAC_nodes() done, now " .
                                "update_inventory()\n"
                            });
                        }
                        # inventory_update_error will be filled with output from
                        #   the UpdateInventory.java if there's error.  Otherwise,
                        #   it'll be blank
                        $inventory_update_error =
                               $this_class -> update_inventory ( {
                                  fh_log_file       => $fh_log_file,
                                  Oracle_Home       => $Oracle_Home,
                                  patch_id          => $patch_id,
                                  rh_OUI_file_names => $rh_OUI_file_names } );

                        if ( $inventory_update_error ) {
                          opatchIO -> print_message_noverbose ( { f_handle => $fh_log_file,
                                              message  =>
                                                  $inventory_update_error } );
                        }
                }

                # Back up comps.xml one more time.  Use-case: cust. rolls back,
                #   update inv. fails, and cust. goes ahead rolling back other
                #   patches.  This back up of comps.xml ensures they have
                #   a good copy to return the system to.
                #
                # (we are backing up comps.xml one more time even if cust.
                #  invoked with -no_inventory).
                #
                if (! $Command::ROLLBACK_CALLED_BY_APPLY) {    
            
                  # Back up original comps.xml to .patch_storage as "comps.xml.afterrollback"
                  my $backup_filemap = $ENV{OPATCH_BACKUP_FILEMAP};
                  if ( $backup_filemap eq "TRUE" )
                  {
                  opatchIO->print_message_noverbose ( { 
                               message =>
                                         "Backing up comps.xml and filemap ...\n"
                                                     } );
                  }
                  else
                  {
                    opatchIO->print_message_noverbose ( { 
                               message =>
                                         "Backing up comps.xml ...\n"
                                                     } );
                  }
            
                  $this_class->backupCompsXML({ Oracle_Home => $Oracle_Home,
                                              patch_id    => $patch_id,
                                              fh_log_file => $fh_log_file,
                                              target_name => "comps.xml.afterrollback",
                                              target_filemap_name => "filemap.afterrollback"
                                           });
                }
    
                if($inventory_update_error) {
                    opatchIO -> print_message_and_die({
                              exit_val => $this_class -> ERROR_CODE_INVENTORY_PROBLEM,
                              message => "Update Inventory error"
                            });
                }
            }
        }
    }

    if( ! $Command::ROLLBACK_CALLED_BY_APPLY ) {
        # Before continuing with post script, lets
        # clean up the inventory for any dangling filemap entries.
        my $cleanup_result = $this_class->clean_up_inventory( { f_handle => $fh_log_file, 
						       Oracle_Home => $Oracle_Home } );

        if( $cleanup_result ) {
      	  opatchIO->print_message_noverbose( { message => "Could not find if the inventory is clean or not.\n" } );
        }
    }

    # Call post RollBack script if available
    # For Windows call post.bat and for non windows call post
    $script_loc = "";
    $script_txt_loc = "";
    $script_line = "";
    $script_result = "";

    if ( $OSNAME =~ m#Win# )
    {
        $script_loc = File::Spec -> catfile ($patch_storage_directory,
                                        "custom", "scripts", $Command::WIN_POST_SCRIPT);
         $script_name = $Command::WIN_POST_SCRIPT;                                
    }
    else
    {
        $script_loc = File::Spec -> catfile ($patch_storage_directory,
                                        "custom", "scripts", $Command::UNIX_POST_SCRIPT);
         $script_name = $Command::UNIX_POST_SCRIPT;                                
    }
    $script_txt_loc = File::Spec -> catfile ($patch_storage_directory,
                                        "custom", $Command::POST_README);

    opatchIO->print_message ( { message => "Looking for post-patch readme $script_txt_loc\n" } );
    # cat post readme to stdout
    if ( -e $script_txt_loc )
    {
        opatchIO->print_message ( { message => "  printing content of $script_txt_loc to stdout\n" } );
        local *script_txt_handle;
        opatchIO->print_message ( { f_handle => $fh_log_file, message => $Command::POST_README_HEADER } );
        open(script_txt_handle, $script_txt_loc) || opatchIO->print_message ( { message => "Could not open file $script_txt_loc" } );
        @script_txt_lines = <script_txt_handle>;
            foreach $script_line (@script_txt_lines)
            {
                chomp $script_line;
                opatchIO->print_message_noverbose ( { message => $script_line } );
            }
        opatchIO->print_message ( { f_handle => $fh_log_file, message => $Command::POST_README_TAIL } );
        
        close script_txt_handle;
        }
    # Execute post script
    opatchIO->print_message ( { message => "Looking for post-patch script $script_loc\n" } );
    if ( (-e $script_loc) && (!$noop_flag) )
    {
        opatchIO->print_message ( { message => "  preparing to execute $script_loc\n" } );
        $system_command = $system_command = "$script_loc -rollback $patch_id $Command::POST_PARAMETERS";
        opatchIO->print_message_noverbose ( { message => "Executing the Roll-back post-patch script ($script_loc)..." } );        
        #$script_result = qx/$system_command/;

        $this_class -> run_cmd_print_live ( { sys_command => $system_command,
                               level => 2 } );


        my $return_code = $this_class -> get_child_error ( {
          CHILD_ERROR     => $CHILD_ERROR
        } );
        opatchIO -> print_message ( { f_handle => $fh_log_file,
                                      message  => $system_command
                                  } );
        opatchIO->print_message( { f_handle => $fh_log_file, message => $Command::POST_SCRIPT_HEADER } );
        opatchIO->print_message_noverbose ( { message => $script_result } );
        opatchIO->print_message( { f_handle => $fh_log_file, message => "exit code = \"$return_code\"" } );
        opatchIO->print_message( { f_handle => $fh_log_file, message => $Command::POST_SCRIPT_TAIL } );

        # this would remember that post-script was run
        $this_class -> save_rollback_postscript_return_code ( {
           return_code => $return_code
        } );
        
        # if post script returns error, we need to warn users but won't exit
        if ( $Command::ROLLBACK_POST_SCRIPT_STATUS == $Command::PREPOST_ERROR ) {
          my $message = "The post-patch script $script_loc has returned an error.\n" ;
                      
          opatchIO -> print_message_noverbose ( { f_handle => $fh_log_file,
                                      message  => $message
                                  } );
        }

   }

    # Free lock file
    $lock_file_errors = $this_class -> free_the_lock_file ( {
                                              Oracle_Home => $Oracle_Home,
                                              patch_id    => $patch_id } );

    # Remind users to check binaries on remote nodes if there's remote make
    if (! $Command::ROLLBACK_CALLED_BY_APPLY) {
       if ( ! ($Command::REMOTE_MAKE_COMMAND_IS_INVOKED eq "") ) {
        opatchIO -> print_message_noverbose ( {
          f_handle => $fh_log_file,
          message  => 
          "***************************************************************\n" .
          "OPatch invoked the following commands\n" .
          "  on the remote nodes to relink Oracle binaries.\n"      .
          $Command::REMOTE_MAKE_COMMAND_INVOKED .
          "\n" .
          "  Please check the binaries' timestamp and size\n"       .
          "  to make sure they are relinked correctly.\n"           .
          "***************************************************************\n"
        } );
      }
    }
    
    # Exit
    my $timestamp = $this_class -> get_current_timestamp();
    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "Ending Rollback session at " .
                                              "$timestamp.\n" } );
    
    if ( $Command::ROLLBACK_CALLED_BY_APPLY == 1 ) {
       return ( 0 );
    } else {
        opatchIO -> print_message_and_die({
            exit_val => $this_class -> ERROR_CODE_NO_ERROR,
            message => "Roll back done"
            });
    }
}   # End of abdomen().

###############################################################################
#
# NAME   : build_executable_checklist
#
# PURPOSE: Create a list of files that are to be checked for active
#          processes.
#
# INPUTS : $$ARG[1]{Oracle_Home} - The path to $ORACLE_HOME to use.
#
# OUTPUTS: \%file_list           - A hash of names to check. The key is
#                                  basename of the file, the value is
#                                  full path to the file.
#
# NOTES  : 1. The return value is kept in the same form as the subroutine
#             Apply::read_and_check_reference_details() so other processing
#             can stay in the same format as used by "apply".
#          2. This is a best-guess method only. It means the real data in
#             original patch directory is not available.
#
###############################################################################
sub build_executable_checklist {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $Oracle_Home  = $$rh_arguments{Oracle_Home};

    my @base_list    = ();
    push ( @base_list, File::Spec -> catfile ( "bin", "lsnrctl" ) );
    push ( @base_list, File::Spec -> catfile ( "bin", "oracle" ) );
    push ( @base_list, File::Spec -> catfile ( "bin", "tnslsnr" ) );
    push ( @base_list, File::Spec -> catfile ( "ctx", "bin", "ctxhx" ) );

    my %executable_files = {};

    foreach my $file ( @base_list ) {
        my $key = &File::Basename::basename ( $file );
        $file = File::Spec -> catfile ( "$Oracle_Home", "$file" );

        $executable_files{$key} = $file;
    }

    return ( \%executable_files );

}   # End of build_executable_checklist().


###############################################################################
#
# NAME   : check_for_noninventory_files
#
# PURPOSE: Return details of a given patch (files, base bugs, etc.).
#
# INPUTS : $$ARG[1]{Oracle_Home}  - The area that is to be patched.
#          $$ARG[1]{patch_id}     - The patch identification string.
#
# OUTPUTS: \@return_data          - Reference to an array of files.
#
# NOTES  : 1. No errors are checked for or returned.
#
###############################################################################
sub check_for_noninventory_files {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $Oracle_Home  = $$rh_arguments{Oracle_Home};
    my $patch_id     = $$rh_arguments{patch_id};

    my $error_message = "";
    my %file_list     = ();

    my $patch_storage_area = File::Spec -> catfile ( "$Oracle_Home",
                                                    $this_class -> save_dir,
                                                    $patch_id );

   # I have no choice, this is the most effective way to do it. Sorry.
   use File::Find();

   my @files = ();
   File::Find::find sub
        { push @files, $File::Find::name if m#pre_$patch_id$# },
                                                      $patch_storage_area;

   my $index = scalar (  @files );
   my $extention = "_pre_$patch_id";

   for ( my $i = 0; $i < $index; $i++ ) {

       my ( $bare_name ) =
                    ( $files[$i] =~ m#^$patch_storage_area(.+)$extention$# );

       my $test_name = File::Spec -> catfile ( "$Oracle_Home", $bare_name );

       if ( -f $test_name ) {
           $files[$i] = $test_name;
       } else {
           # Have to strip the parent off this parent-child type file.
           my $parent = "";
           ( $bare_name, $parent ) = ( $bare_name =~ m#^(.+)_(.+?)$# );

           $test_name = File::Spec -> catfile ( "$Oracle_Home",
                                      &File::Basename::dirname ( $bare_name ),
                                                                    $parent );
           if ( -f $test_name ) {
               $files[$i] = "$test_name -> " .
                                     &File::Basename::basename ( $bare_name );
           } else {
               # Using delete would be nice but this code is compatible with
               # earlier versions so it's not an option.
               $files[$i] = "";
           }
       }
   }

    # Now remove any empty elements. Again due to the undelete function
    # not being available on earlier versions of perl for arrays the
    # clean way is to use a new array.
    my @new_array = ();
    foreach my $entry ( @files ) {
        next if ( ! $entry );
        push ( @new_array, $entry );
    }

    return ( \@new_array );

}   # End of check_for_noninventory_files().

###############################################################################
#
# NAME   : cleanup_after_patch_removal_unix
#
# PURPOSE: Return details of a given patch (files, base bugs, etc.).
#
# INPUTS : $$ARG[1]{Oracle_Home}  - The area that is to be patched.
#          $$ARG[1]{ra_file_list} - The base files that need to have the
#                                   the associated work files removed.
#          $$ARG[1]{patch_id}     - The patch identification string.
#
# OUTPUTS: $error_messages - A string containing any errors encountered.
#
# NOTES  : 1. Using a hash for the list of files instead of the array because
#             Java class files can have duplicate details for different
#             classes. So the file isn't deleted twice (and raise a false
#             error message) the hash is used.
#
###############################################################################
sub cleanup_after_patch_removal_unix {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $Oracle_Home  = $$rh_arguments{Oracle_Home};
    my $patch_id     = $$rh_arguments{patch_id};
    my $ra_file_list = $$rh_arguments{ra_file_list};

    my $error_message = "";
    my %file_list     = ();

    if ( $OSNAME !~ m#Win32# ) {
       my $check = $this_class -> is_symbolic_link ($Oracle_Home);
       if ( $check == 1 ) {
           my $real_OH = $this_class -> resolve_symbolic_link ($Oracle_Home);
           $Oracle_Home= $real_OH;
           opatchIO->print_message( {message => "\n using real Oracle_Home:\n"} );
           opatchIO->print_message( {message => "$Oracle_Home\n"} );
       }
    }

    my $patch_directory   = File::Spec -> catfile ( "$Oracle_Home",
                                                    $this_class -> save_dir,
                                                    $patch_id );

    if ( $OSNAME =~ m#Win32# ) {
        $patch_directory = lc ( $patch_directory );
        $Oracle_Home =~ s/\\/\\\\/g;
    }

    # Sort the files into appropriate groups and remove the files.
    my $new_file_flag = 0;
    foreach my $file ( @$ra_file_list ) {

        if ( $file =~ m# \+> NEW$# ) {
            ( $file ) = ( $file =~ m#(.+) \+> NEW$# );
            #
            #  With filemap being empty, every copy action turns out to be
            #  'new' anyway.
            #  $new_file_flag = 1;
        }


        # Need to address this locally and adjust the global data values.
        my $local_file = $file;

        my $dir_name = &File::Basename::dirname( $file );

        if ( $local_file !~ m# -> # ) {

            # If it was a new file fix the name
            if ( $file =~ m# \+> NEW$# ) {
                ( $file ) = ( $file =~ m#(.+) \+> NEW$# );
            }

            if ( $OSNAME =~ m#Win32# ) {
                $local_file = lc ( $file );
                $local_file =~ s/\//\\/g;
            }

            #$local_file =~ s#$Oracle_Home#$patch_directory#;
	    $local_file = $this_class -> replace_oh_with_patchdir ({
							Oracle_Home => $Oracle_Home,
	                                                patch_dir   => $patch_directory,
							target_path => $local_file } );


            my $post_name = $local_file . "_" . &Apply::END_PATCH .
                                                               "_$patch_id";
            my $pre_name  = $local_file . "_" . &Apply::PRE_PATCH .
                                                               "_$patch_id";
            # Do not look into 'new' or not.  Just blindly add the name to 
            # the list to be removed, then at removal time, check for
            # existence before removal.
            #if ( ! $new_file_flag ) {
            #  $file_list{$pre_name} = "";
            #}
            
            $file_list{$pre_name} = "";
            $file_list{$post_name} = "";

        } else {
            # Parent -> child relation.
            my ( $parent, $child ) = ( $file =~ m#^(.+) -> (.+)$# );
            my $dir_name = &File::Basename::dirname( $parent );
            my $library  = &File::Basename::basename( $parent );

            #$dir_name =~ s#$Oracle_Home#$patch_directory#;
            $dir_name = $this_class -> replace_oh_with_patchdir ({
                                                        Oracle_Home => $Oracle_Home,
                                                        patch_dir   => $patch_directory,
                                                        target_path => $dir_name } );




            if ( ( $parent =~ m#\.jar$# ) || ( $parent =~ m#\.zip$# ) ) {

                $local_file = File::Spec -> catfile ( "$dir_name", "$child" );

                my $jar_file = &File::Basename::basename( $parent );
                $child = $child . "_" . $jar_file . "_" . &Apply::PRE_PATCH .
                                                              "_" . $patch_id;
                $local_file = File::Spec -> catfile ( "$dir_name", "$child" );
                $file_list{$local_file} = "";

                ( $local_file ) = ( $local_file =~ m#^(.+)$jar_file# );
                $local_file .= $jar_file . "_" . &Apply::END_PATCH .
                                                                "_$patch_id";
                $file_list{$local_file} = "";

            } else {
                my $object = &File::Basename::basename( $child );
                if ( $object ) { $child = $object; }
                $local_file = File::Spec -> catfile ( "$dir_name", "$child" );
                $file_list{$local_file} = "";
                $child = $child . "_" . $library . "_" .
                                            &Apply::END_PATCH . "_$patch_id";
                $local_file = File::Spec -> catfile ( "$dir_name", "$child" );
                $file_list{$local_file} = "";
            }
        }
    }

    foreach my $file ( keys %file_list ) {
        # make sure the file exists before attempting to remove to avoid
        # warning on new file
        if ( -e $file ) {
          my $count = unlink ( $file );
          if ( $count != 1 ) {
              $error_message .= $file . "\n";
          }
        }
    }

    if ( $error_message ) {
        $error_message = "The following work files couldn't be removed:\n" .
                         $error_message;
    }

    return ( $error_message );

}   # End of cleanup_after_patch_removal_unix().

#####################################
#
#####################################
sub cleanup_after_patch_removal_windows {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $Oracle_Home  = $$rh_arguments{Oracle_Home};
    my $patch_id     = $$rh_arguments{patch_id};
    my $ra_file_list = $$rh_arguments{ra_file_list};

    my $error_message = "";
    my %file_list     = ();

    if ( $OSNAME !~ m#Win32# ) {
       my $check = $this_class -> is_symbolic_link ($Oracle_Home);
       if ( $check == 1 ) {
           my $real_OH = $this_class -> resolve_symbolic_link ($Oracle_Home);
           $Oracle_Home= $real_OH;
           opatchIO->print_message( {message => "\n using real Oracle_Home:\n"} );
           opatchIO->print_message( {message => "$Oracle_Home\n"} );
       }
    }

    my $patch_directory   = File::Spec -> catfile ( "$Oracle_Home",
                                                    $this_class -> save_dir,
                                                    $patch_id );

#    if ( $OSNAME =~ m#Win32# ) {
#        $patch_directory = lc ( $patch_directory );
#        $Oracle_Home =~ s/\\/\\\\/g;
#    }

    # Sort the files into appropriate groups and remove the files.
    my $new_file_flag = 0;
    foreach my $file ( @$ra_file_list ) {

        if ( $file =~ m# \+> NEW$# ) {
            ( $file ) = ( $file =~ m#(.+) \+> NEW$# );
            # $new_file_flag = 1;
        }


        # Need to address this locally and adjust the global data values.
        my $local_file = $file;

        my $dir_name = &File::Basename::dirname( $file );

        if ( $local_file !~ m# -> # ) {

            # If it was a new file fix the name
            if ( $file =~ m# \+> NEW$# ) {
                ( $file ) = ( $file =~ m#(.+) \+> NEW$# );
            }

            if ( $OSNAME =~ m#Win32# ) {
                $local_file = lc ( $file );
                $local_file =~ s/\//\\/g;
            }

            # $local_file =~ s#$Oracle_Home#$patch_directory#;
            #$local_file = File::Spec -> catfile ( $patch_directory,
            #                       substr($local_file, $Oracle_Home));
            $local_file = $this_class -> replace_oh_with_patchdir ({
                                                        Oracle_Home => $Oracle_Home,
                                                        patch_dir   => $patch_directory,
                                                        target_path => $local_file } );



            my $post_name = $local_file . "_" . &Apply::END_PATCH .
                                                               "_$patch_id";
            my $pre_name  = $local_file . "_" . &Apply::PRE_PATCH .
                                                               "_$patch_id";
            #if ( ! $new_file_flag ) {
            #    $file_list{$pre_name} = "";
            #}
             
            $file_list{$pre_name} = "";
            $file_list{$post_name} = "";

        } else {
            # Parent -> child relation.
            my ( $parent, $child ) = ( $file =~ m#^(.+) -> (.+)$# );
            my $dir_name = &File::Basename::dirname( $parent );
            my $library  = &File::Basename::basename( $parent );

            # class_path is "oracle/sql/xml/foo.class"
            my $class_path = $this_class->convert_path_separator({path=>$child});
            my $dir_name = $this_class->convert_path_separator({path=>$dir_name});
            my $parent = $this_class->convert_path_separator({path=>$parent});
            
            # $dir_name is "OH/<jar_path>, i.e.
            #   u:\windows\home\lib
            # 
            # we will extract <jar_path> out (the "lib" part)
            my $len = length($Oracle_Home);
            my $jar_path = substr($dir_name, $len);                      
            $dir_name = $patch_directory . "\\" . $jar_path;

            #
            #  dest_path is OH/lib/xsu12.jar
            my $dest_path = $parent;
            # parent will be just "xsu12.jar"
            $len = length($Oracle_Home) + length($jar_path) + 1;
            $parent = substr($parent, $len);

            if ( ( $parent =~ m#\.jar$# ) || ( $parent =~ m#\.zip$# ) ) {
                
                $local_file = $dir_name . "\\" . $child . 
                              "_" .
                              $parent . 
                              "_" .
                              &Apply::PRE_PATCH .
                              "_" .
                              $patch_id;
                $file_list{$local_file} = "";
                
                $local_file = $dir_name . "\\" . $child . 
                              "_" .
                              $parent .
                              "_" .
                              &Apply::END_PATCH .
                              "_" .
                              $patch_id;
                
                $file_list{$local_file} = "";

            } else {
                my $object = &File::Basename::basename( $child );
                if ( $object ) { $child = $object; }
                $local_file = File::Spec -> catfile ( "$dir_name", "$child" );
                $file_list{$local_file} = "";
                $child = $child . "_" . $library . "_" .
                                            &Apply::END_PATCH . "_$patch_id";
                $local_file = File::Spec -> catfile ( "$dir_name", "$child" );
                $file_list{$local_file} = "";
            }
        }
    }

    foreach my $file ( keys %file_list ) {
        if ( -e $file ) {
          my $count = unlink ( $file );
          if ( $count != 1 ) {
              $error_message .= $file . "\n";
          }
        }
    }

    if ( $error_message ) {
        $error_message = "The following work files couldn't be removed:\n" .
                         $error_message;
    }

    return ( $error_message );

}   # End of cleanup_after_patch_removal_windows().

###############################################################################
#
# NAME   : command_details
#
# PURPOSE: Provide the details about the command.
#
# INPUTS : $$ARG[rh_detail] - A reference to a hash structure of the class
#                             Command to store the attributes in.
#
# OUTPUTS: NONE
#
# NOTES  : 1. The return values are in hash so the caller will get the new
#             details when this subroutine is finished.
#
###############################################################################
sub command_details {
 
    # Because this was called in an OO manner the first argument is the
    # class name.
    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_detail    = $$rh_arguments{rh_detail};

    # Put options here for easy reference.
    my @options = ( "delay", "id", "invPtrLoc",  "jdk", 
                    "jre", "local", "local_node", "no_relink", "oh", "ph", 
                    "retry", "silent", "verbose",
                    "pre", "post",
                    "opatch_pre_end", "opatch_post_end", "report");

    my $command = lc ( $this_class );
    $$rh_detail{$command}  = Command -> new_command();

    $$rh_detail{$command}{_description} = "rollback -id <ID> -ph <dir> ";

    $$rh_detail{$command}{_helpText}    =
               "Rollback an existing one-off patch indicated by the " .
               "reference-id.",

    $$rh_detail{$command}{_options} =
               $this_class -> build_option_details ( {
                          rh_options     => $$rh_detail{$command}{_options},
                          ra_option_list => \@options } );

    $this_class -> add_options_to_command_description
                                 ( { rh_command => $$rh_detail{$command} } );

    return ( 0 );

}   # End of command_detail().

###############################################################################
#
# NAME   : get_and_run_make_commands
#
# PURPOSE: Run the make commands needed after restoring a patch.
#
# INPUTS : $$ARG[1]{fh_log_file} - The file handle for the log file.
#          $$ARG[1]{noop_flag}   - Do the checks but not the actions.
#          $$ARG[1]{Oracle_Home} - The area that is to be patched.
#          $$ARG[1]{patch_id}    - The patch identification string.
#
# OUTPUTS: $error_messages - A string containing any errors encountered.
#
# NOTES  : 1. This bundles-up the details needed to minimize the number of
#             invokations to the Java virtual machine.
#          2. This call also returns other details such as if this is a
#             RAC install for the caller to check.
#          3. Need to see if this can be merged with the method
#             Command::run_make_on_remote_nodes().
#
###############################################################################
sub get_and_run_make_commands {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file   = $$rh_arguments{fh_log_file};
    my $noop_flag     = $$rh_arguments{noop_flag};
    my $Oracle_Home   = $$rh_arguments{Oracle_Home};
    my $patch_id      = $$rh_arguments{patch_id};

    my $error_messages = "";
    my @file_data      = ();

    if ( $Command::NO_RELINK_MODE ) { 
       opatchIO -> print_message_noverbose ( {
           f_handle => $fh_log_file,
		       message =>
           "-------------------------------------------------------\n" .
		       "Skip executing 'make' on local system during roll-back.\n" .
           "Please invoke 'make' manually after patching.\n" .
		       "-------------------------------------------------------\n" 
	       });  
         return "";
    }

    # get the "make" command
    my $make_command = $this_class->getMakeCommand();
    if( $make_command eq "" ) {
       opatchIO -> print_message_and_die ( {
                       exit_val => $this_class -> ERROR_CODE_PATH_PROBLEM,
                       message =>
                       "'make' is not in your PATH or is not executable.\n" .
           "  Extra Search Path= /usr/bin/make (Linux), /usr/ccs/bin/make (others)\n" .
                       "  Path= $ENV{PATH}\n" .
           "You have to manually run make on all remote nodes or use -local option.\n"
               });
    }

    #opatchIO -> print_message_noverbose ( { f_handle => $fh_log_file,
    #                              message => "Using \"$make_command\" to make...\n" } );

    if ( $OSNAME !~ m#Win32# ) {
       my $check = $this_class -> is_symbolic_link ($Oracle_Home);
       if ( $check == 1 ) {
           my $real_OH = $this_class -> resolve_symbolic_link ($Oracle_Home);
           $Oracle_Home= $real_OH;
           opatchIO->print_message( {message => "\n  : using real Oracle_Home:\n"} );
           opatchIO->print_message( {message => "$Oracle_Home\n"} );
       }
    }

    my $patch_directory   = File::Spec -> catfile ( "$Oracle_Home",
                                                    $this_class -> save_dir,
                                                    $patch_id );

    my $make_command_file = $patch_id . "_make.txt";
    $make_command_file = File::Spec -> catfile ( "$patch_directory",
                                                 "$make_command_file" );
    my $status_ok = 1;
    if ( -e $make_command_file ) {

        local *MAKE_COMMANDS;
        open ( MAKE_COMMANDS, $make_command_file ) or do {
            opatchIO -> print_message ( { f_handle => *STDERR,
                                          message  => "Cannot open file " .
                            "$make_command_file for reading. Cannot undo " .
                            "make commands.\n" } ),
            $status_ok = 0
        };

        if ( $status_ok ) {
            @file_data = <MAKE_COMMANDS>;
            close MAKE_COMMANDS or do {
            opatchIO -> print_message ( { f_handle => *STDERR,
                                          message  => "Cannot close the " .
                             "file after reading: $make_command_file\n" .
                                    "[$OS_ERROR]\nProcessing continuing" } );
            }
        }
    } else {
        $status_ok = 0;
    }
    my $sys_call_result = "OK";
    if ( $status_ok ) {
        my $system_command = "";
        my $changeDirCommand = "";

        foreach my $line ( @file_data ) {

            # Strip out comments and blank lines.
            next if ( $line =~ m/^#/ );
            next if ( $line =~ m/^\s+#/ );
            next if ( $line =~ m#^$# );
            
            if ( $line =~ m#^cd# ) {
              # this is the "cd /private/phi_local/9201home/rdbms/lib/" line
              #   or        "cd $ORACLE_HOME/rdbms/lib/"
              if ( $line =~ m#\$ORACLE_HOME# ) {
                $changeDirCommand = $line;
                $changeDirCommand =~ s#\$ORACLE_HOME#$Command::UNRESOLVED_ORACLE_HOME#;
              } else {
                $changeDirCommand = $line;
              }
              chomp $changeDirCommand;
              next;
              # go to next line, expecting a "make" command there...
            }
            
            #
            # This use of regular expression might fail if the "make_command"
            #   picked up when applying this patch is different from the
            #   "make_command" picked up now during rollback.
            #
            # if ( $line =~ m#^$make_command\s# ) {
            #
            if ( $line =~ m#make\s# ) {

                # Because various vendors do _strange_ things in rsh
                # and $ORACLE_HOME can dissapear during the make
                # invokation it's neccessary to nail $ORACLE_HOME into
                # place.
                #   $system_command =~
                #      s#;$make_command#;$make_command ORACLE_HOME=\"\$ORACLE_HOME\" #;
                
                # line is
                # make -f  ins_rdbms.mk ioracle OR
                # /usr/bin/make -f ins_rdbms.mk ioracle
                my @list = split ( ' ', $line );
                my $cmdPart = $list[0];
                my $cmdArgPart = $list[1];
                my $makeFilePart = $list[2];

                # Read all the targets and not just one.
                my $targetPart = "";
                my $tCtr = 0;
                my $no_elems_list = scalar(@list);
                for($tCtr = 3; $tCtr < $no_elems_list; $tCtr++ ) {
                    $targetPart .= " " . $list[$tCtr];
                }

                $system_command = "$changeDirCommand; $make_command $cmdArgPart $makeFilePart $targetPart " .
                                  "ORACLE_HOME=\"$Command::UNRESOLVED_ORACLE_HOME\"";
                
                #my ( $target ) = ( $line =~ m#\s(\w+)$# );
                opatchIO -> print_message_noverbose ( { message => "Running make " .
                                                         "for target $targetPart.\n" } );

                my $filename = "";
                if ( ! $noop_flag ) {
                    my $extend_cmd = "";
                    if ( $OSNAME =~ m#Win32# ) {
                        $extend_cmd = "$system_command";
                    } else {
                        # Redirect stderr to a file called
                        $filename = File::Spec -> catfile ( 
                                              "$patch_directory",
                                              "make_local.stderr" );
                        # First remove the file if it's there
                        if ( -e $filename ) {
                          unlink $filename;
                        }
                        if ( $this_class->is_newline_terminated({string => $system_command}) ) {
                           # let extract the command w/o the very last "\n"
                           my $len = length ($system_command) - 1;
                           $extend_cmd = substr($system_command, 0, $len);
                           # append "2>&1" to catch stderr
                           $extend_cmd = $extend_cmd . " 2>$filename";
                        } else {
                          $extend_cmd = $system_command . " 2>$filename";
                        }
                    }
                    $sys_call_result = qx/$extend_cmd/;
                    chomp $sys_call_result;
                    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                          message  => "System Command:\n\t" .
                                          "\"$extend_cmd\"" . "\nResult:\n" .
                                          "$sys_call_result\n" } );
                    my $status = $this_class -> get_child_error ( {
                        CHILD_ERROR     => $CHILD_ERROR
                    } );

                    # Small note: status may not be enough. The return value
                    # is for the last make target invoked.
                    if ( $status ) {
                        $error_messages = "Error for command: $extend_cmd";
                    } elsif ( $sys_call_result =~ m#\bFATAL\b# ) {
                        $error_messages = "FATAL ERROR: $sys_call_result\n";
                    } elsif ( $sys_call_result =~ m#\bERROR\b# ) {
                        $error_messages = "ERROR: $sys_call_result\n";
                    } elsif ( $sys_call_result =~ m#\bFAIL\b# ) {
                        $error_messages = "FAIL: $sys_call_result\n";
                    }

                    # reset system_command to empty string
                    $system_command = "";
                    
                    
                    # Read the content of redirected stderr file to check for error
                    # assert: non-zero byte file size means error
                        
                    opatchIO -> debug ( {
                        message => "Display make errors from: \"$filename\""
                                } ); 

                    if ( -e $filename && -s $filename ) 
                    {
                        local *MAKE_ERROR_FILE;
                        open ( MAKE_ERROR_FILE, $filename ) or do {
                                    opatchIO -> print_message ( {
                                       f_handle => $fh_log_file,
                                      message => "ERROR: Cannot open error file to read: \"$filename\"\n"
                                    } ); 
                        };
                        my @file_data = <MAKE_ERROR_FILE>;
                        my $print_stderr = 0;
                        foreach my $line ( @file_data ) {
                           chomp $line;
                           if ( $line =~ m#FATAL#i ) {
                                 $print_stderr = 1;
                           } elsif ( $line =~ m#ERROR#i && ( $print_stderr == 0 || $print_stderr > 2 )) {
                                 $print_stderr = 2;
                           } elsif ( $line =~ m#FAIL#i && ( $print_stderr == 0 || $print_stderr > 3 )) {
                                 $print_stderr = 3;
                           }
                        }

                        if($print_stderr) {
                             my $stderr_msg = "OPatch found the word \"";
                             if($print_stderr == 1) {
                                  $stderr_msg = $stderr_msg . "fatal\" in stderr, please look into it." ;
                             } elsif($print_stderr == 2) {
                                  $stderr_msg = $stderr_msg . "error\" in stderr, please look into it." ;
                             } else {
                                  $stderr_msg = $stderr_msg . "fail\" in stderr, please look into it." ;
                             }

                             $stderr_msg = $stderr_msg . "\nYou may have to run the make again.";

                             opatchIO -> print_message_noverbose ( {
                                    f_handle => $fh_log_file,
                                    message => $stderr_msg
                             } );
                             opatchIO -> print_message_noverbose ( {
                                    f_handle => $fh_log_file,
                                    message => "Stderr Output \(from $filename\):"
                             } );
                          }

                          if($print_stderr || $status) {
                             foreach my $line ( @file_data ) {
                                      chomp $line;
                                      opatchIO -> print_message_noverbose ( {
                                            f_handle => $fh_log_file,
                                            message => "$line\n"
                                      } );
                             }
                          }
                          close MAKE_ERROR_FILE;
                      } # end of if the stderr output is non-zero in size


                    opatchIO -> debug ( {
                        message => "Display make errors from: \"$filename\" Complete"
                            } );

                    if ($status)
                    {
                        my $question =
                                    "Make failed.  If you continue, " .
                                    "you will have to run make manually on the local system.  " . "\n" .
				    "Replying 'Y' will terminate the patch rollback immediately. " . 
                                    "It WILL NOT restore any updates that have been performed to this point. " .
                                    "It WILL NOT update the inventory." . "\n" .
		                    "Replying 'N' will continue with the additional make operations and " . 
				    "update the inventory showing the patch has been removed." . "\n" .
                                    "Do you want to STOP?" ;
        
                        my $answer = opatchIO -> check_to_stop_processing ( {
                                                        message => $question } );
                        if ( $answer eq "" ) {
                                  opatchIO -> print_message_and_die ( {
                                    exit_val => $this_class->ERROR_CODE_MAKE_FAILURE, 
                                    message => "Local make failed.  See log file for details.\n"
                                  } ); 
                        }
                    } # If make returns error, error out

                }
            }
        }
    }

    return ( $error_messages );

}   # End of get_and_run_make_commands().

###############################################################################
#
# NAME   : get_initial_details
#
# PURPOSE: Return details of a patch's installation rules.
#
# INPUTS : $$ARG[1]{Oracle_Home}    - The area that is to be patched.
#          $$ARG[1]{patch_id}       - The patch identification string.
#
# OUTPUTS: $rh_reference_details - A reference to a hash that has the data
#                                  needed.
#          {bug_list}            - A hash of base bugs, subkey is:
#                {$number}       - The number of the base and the description.
#          {date}                - Date the patch was made, subkeys are:
#                {day}           - Self-evident.
#                {month}         - Self-evident.
#                {time}          - Self-evident.
#                {year}          - Self-evident.
#                {zone}          - The time-zone of the computer.
#          {description}         - The title of the bug.
#          {error_flag}          - On success it's a NULL string. Otherwise
#                                  the details of failure are stored here.
#          {executable}          - Paths to binaries that are needed, key is:
#                {name}          - The name of the binary and the path.
#          {online_rac}          - Flag to say if the RAC needs to be online
#                                  when dealing with real application clusters.
#          {os_platforms}        - What OSs this patch applies to, subkey is:
#                {$number}       - OS id number and descriptive string.
#          {reference_id}        - The patch id, may have text after the
#                                  number.
#          {components}          - A hash of componets, subkey is:
#                {$name}         - The name of the component that has the
#                                  sub-subkeys {version} and {option_flag}.
#          {shutdown}            - Flag to say if the database needs to be
#                                  shutdown for the patch to be installed.
#
# NOTES  : 1. This is a cut down version of
#             Apply::read_and_check_reference_details and the functions
#             need to be amalgamated and generalized.
#          2. This is another hack for supporting rolling RAC. In this
#             case the rollback of a patch.
#
###############################################################################
sub get_initial_details {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $Oracle_Home  = $$rh_arguments{Oracle_Home};
    my $patch_id     = $$rh_arguments{patch_id};

    my $error_flag = "";
    my @file_data  = ();

    # The details are stored in the file "inventory" in XML format.
    my $full_inventory_path = File::Spec -> catfile( $Oracle_Home,
                                              ".patch_storage",
                                              "$patch_id", "inventory" );

    local *INVENTORY_FILE;
    open ( INVENTORY_FILE, $full_inventory_path ) or do {
        $error_flag = "Cannot open data file for reading: " .
                      "$full_inventory_path.\n[$OS_ERROR]";
    };

    if ( ! $error_flag ) {
        @file_data = <INVENTORY_FILE>;
        close INVENTORY_FILE or do {
            opatchIO -> print_message ( { f_handle => *STDERR,
                                          message  => "Cannot close the " .
                             "file after reading: $full_inventory_path\n" .
                                    "[$OS_ERROR]\nProcessing continuing" } );
        };
    }

    my %reference_details = ();
    $reference_details{RAC_shutdown} = 1;

    my $file_size = scalar ( @file_data );

    for ( my $i =0; $i < $file_size; $i++ ) {
        chomp $file_data[$i];

        # Yes, making it elsif is more efficient. But this is going to a
        # be redone in Java and I don't want to highlight the regex's and
        # the data structure usage.
        if ( $file_data[$i] =~ m#<reference_id number="# ) {
           ( $reference_details{reference_id} ) =
                        ( $file_data[$i] =~ m#<reference_id number="(.+)"# );
        }

        if ( $file_data[$i] =~ m#<date_of_patch year="# ) {
            ( $reference_details{date}{year}  ) =
                                       ( $file_data[$i] =~ m#year="(\d+)"# );
            ( $reference_details{date}{month} ) =
                                      ( $file_data[$i] =~ m#month="(\w+)"# );
            ( $reference_details{date}{day}   ) =
                                        ( $file_data[$i] =~ m#day="(\d+)"# );
            ( $reference_details{date}{time}  ) =
                               ( $file_data[$i] =~ m#time="([:\d]+)\shrs"# );
            ( $reference_details{date}{zone}  ) =
                                     ( $file_data[$i] =~ m#zone="(.+)"# );
        }

        if ( $file_data[$i] =~ m#<base_bugs>$# ) {
            $i++;
            while ( $file_data[$i] !~ m#</base_bugs>$# ) {
                my ( $number, $detail ) = ( $file_data[$i] =~
                           m#<bug number="(\d+)"\sdescription="(.+)"\s?/>$# );
                $reference_details{bug_list}{$number} = $detail;
                $i++;
            }
        }

        if ( $file_data[$i] =~ m#<required_components>$# ) {
            $i++;
            while ( $file_data[$i] !~ m#</required_components>$# ) {
                # Some component names may have spaces and digits and some
                # version numbers contain letters.
                my ( $name, $version ) = ( $file_data[$i] =~
                       m#<component internal_name="(.+)"\sversion="(.+)"\s# );
                my ( $option_flag ) = ( $file_data[$i] =~
                                                    m#opt_req="(\w)"\s?/>$# );
                $reference_details{components}{$name}{version} = $version;
                $reference_details{components}{$name}
                                                 {option_flag} = $option_flag;
                $i++;
            }
        }

        if ( $file_data[$i] =~ m#<os_platforms>$# ) {
            $i++;
            while ( $file_data[$i] !~ m#</os_platforms>$# ) {
                my ( $description, $number ) = ( $file_data[$i] =~
                           m#<platform name="(.+)"\sid="(\d+)"\s?/>$# );
                $reference_details{os_platforms}{$number} = $description;
                $i++;
            }
        }

        if ( $file_data[$i] =~ m#<executables>$# ) {
            $i++;
            while ( $file_data[$i] !~ m#</executables>$# ) {
                my ( $path ) = ( $file_data[$i] =~
                           m#<executable path="(.+)"\s?/>$# );
                my $name = &File::Basename::basename ( $path );
                $path =~ s/\%ORACLE_HOME\%/$Oracle_Home/ ;
                $reference_details{executable}{$name} = $path;
                $i++;
            }
        }

        if ( $file_data[$i] =~ m#<instance_shutdown># ) {
            $reference_details{shutdown} = 0;
            my ( $shutdown ) =
                 ( $file_data[$i] =~
                       m#<instance_shutdown>(\w+)</instance_shutdown>$# );
            # Now turn a "true" into a 1.
            if ( $shutdown eq "true" ) {
                $reference_details{shutdown} = 1;
            }
        }

        # This is for rolling RAC upgrades. If its rolling RAC installable
        # then the RAC doesn't need to be shutdown.
        if ( $file_data[$i] =~ m#<online_rac_installable># ) {
            my ( $rolling_RAC ) =
                 ( $file_data[$i] =~
             m#<online_rac_installable>(\w+)</online_rac_installable>$# );

            if ( $rolling_RAC eq "true" ) {
                $reference_details{RAC_shutdown} = 0;
            }
        }
    }  # Finished handling the inventory file.

    if ( $error_flag ) { $reference_details{error_flag} = $error_flag; }

    return ( \%reference_details );

}   # End of get_initial_details().

###############################################################################
#
# NAME   : get_rollback_details
#
# PURPOSE: Return details of a given patch (files, base bugs, etc.).
#
# INPUTS : $$ARG[1]{fh_log_file}    - The file handle for the log file.
#          $$ARG[1]{Oracle_Home}    - The area that is to be patched.
#          $$ARG[1]{patch_id}       - The patch identification string.
#          $$ARG[1]{rh_OUI_file_names}
#                                   - A list of files that are needed when
#                                     invoking OUI through the JVM.
#
# OUTPUTS: \%return_data            - A hash to return the data, keys are:
#              {ra_files}           - A hash containing the names of the files
#                                     were patched. The key is the file name,
#                                     the value (if any) is the included
#                                     file (for archives, etc.).
#              {ra_rac_nodes}       - A hash containing the names of any RAC
#                                     nodes.
#                                     (phnguyen 6/9/03: no ra_rac_nodes output)
#
# NOTES  : 1. This bundles-up the details needed to minimize the number of
#             invokations to the Java virtual machine.
#          2. This call also returns other details such as if this is a
#             RAC install for the caller to check.
#
###############################################################################
sub get_rollback_details {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file       = $$rh_arguments{fh_log_file};
    my $Oracle_Home       = $$rh_arguments{Oracle_Home};
    my $patch_id          = $$rh_arguments{patch_id};
    my $rh_OUI_file_names = $$rh_arguments{rh_OUI_file_names};

    my %check_results  = ();

    my $path_to_java = $$rh_OUI_file_names{path_to_java};

    # First set-up the Java class for the inventory read.
    my $rh_java_paths = $this_class -> make_java_paths ( {
                                Oracle_Home       => $Oracle_Home,
                                rh_OUI_file_names => $rh_OUI_file_names } );

    my $class_path = $$rh_java_paths{class};
    my $oui_path   = $$rh_java_paths{oui_path};

    # Now set-up the call and capture the results. Quoting $file_names to
    # make the $file_names a single argument to the Java program.
    # This may need some tuning.
    my $class_to_run = "opatch/GetPatchDetails";
    my $debug = $Command::DEBUG;
    my $localNode = $Command::LOCAL_NODE;
    my $nodeList = $Command::NODE_LIST;
    my $racCode  = $Command::RAC_CODE;

    # GetPatchDetails will invoke O2O, so we will pass in -local and -local_node
    #   to be consistent
    my $local_node_only_flag = "";
    if ($Command::LOCAL_NODE_ONLY) {
        $local_node_only_flag = "-Dopatch.local_node_only";
    }
    my $local_node_name_flag = "";
    if ($Command::LOCAL_NODE_NAME_MODE) {
        $local_node_name_flag = "-Dopatch.local_node_name=$Command::LOCAL_NODE_NAME";
    }

    my $is_shared_flag = "";
    # User wanna turn off CFS detection?
    #    OPATCH_IS_SHARED = true   -> CFS=true, no detect
    #                       false        =false, no detect
    #                       detect       =<detected value>, yes detect
    #                       not set      =false, no detect
    if ($Command::OPATCH_IS_SHARED eq "TRUE") {
        $is_shared_flag = "-Dopatch.is_shared=true";
		    opatchIO -> print_message ( {
               		 message => "No CFS detection, CFS set to true."
              	  });        
    } elsif ( $Command::OPATCH_IS_SHARED eq "FALSE" ) {
        $is_shared_flag = "-Dopatch.is_shared=false";
		    opatchIO -> print_message ( {
               		 message => "No CFS detection, CFS set to false."
              	  });            
    } elsif ( $Command::OPATCH_IS_SHARED eq "DETECT" ) {
        $is_shared_flag = "-Dopatch.is_shared=detect";
		    opatchIO -> print_message ( {
               		 message => "PerformCFS detection, CFS set to detected value."
              	  });        
    } elsif ( $Command::OPATCH_IS_SHARED eq "" ) {
        $is_shared_flag = "";
	# Env. variable OPATCH_IS_SHARED is not set, so ignore it.
    } else {
        $is_shared_flag = "";
		    opatchIO -> print_message ( {
               		 message => "Unknown OPATCH_IS_SHARED value, no CFS detection, CFS set to false."
              	  });     
    } 
    
    my $program_name = &File::Basename::basename( $PROGRAM_NAME );
    my $version_num  = $this_class -> version;
    my $system_command = join(' ', $path_to_java, $local_node_only_flag,
                        $is_shared_flag,
                        $local_node_name_flag,
                        $Command::inventory_location_ptr,
                        $debug,                          
                        $class_path, $class_to_run,
                        $oui_path,
                        qq(").$Oracle_Home.qq("), $program_name, 
                        $version_num, $patch_id);
    $system_command = $this_class->getDependentPerlCommand({ system_command => $system_command });

    $this_class -> warn_about_retry( { f_handle => $fh_log_file } );
    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "\nSystem Command: $system_command"
                              } );

    #my $sys_call_result = qx/$system_command/;
    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "\nResult:\n" 
                               } );

    my $sys_call_result = $this_class -> run_cmd_print_live ( { sys_command => $system_command,
                                              level => 1,
                                              f_handle => $fh_log_file } );


    chomp $sys_call_result;

    my $rh_rollback_data = $this_class -> process_rollback_data ( {
                                          patch_id    => $patch_id,
                                          system_data => $sys_call_result } );

    my %rh_return_data = ();
    my $patch_found = $$rh_rollback_data{patch_found};
        
    $rh_return_data{ra_files}    = $$rh_rollback_data{ra_file_list};
    $rh_return_data{patch_found} = $patch_found;

    return ( \%rh_return_data );

}   # End of get_rollback_details().

##############################################################################
#
# NAME   : parse_arguments
#
# PURPOSE: A check, validation and variable assignments based on the
#          the arguments for this sub-command.
#
# INPUTS : $$ARG[1]{rh_option} - A reference to a hash where the details about
#                                the option are to be stored.
#
# OUTPUTS: A reference to a hash.
#
# NOTES  : 1. Well parse_arguments() is used for the command line check. But
#             at that point the class is the super-class: Command. So we can
#             re-use the name space here for specific processing for the class
#             at the correct time.
#          2. I know the "*_arguments" will confuse people so let me explain it
#             at this point in the module. The command line is parsed initially
#             at the top level (by parse_arguments) that retruns a string. The
#             string contains the arguments. This is turned into an array. The
#             array is an argument to this method. Since the argument to this
#             method is a the local arguments are rh_arguments. The initial
#             command line items, now an array, is located at
#             $$rh_arguments{ra_arguments}.
#          3. The above may be confusing but it keeps all the calls in the
#             same format: a single hash contains all the parameters for any
#             method.
#          4. This is the first time a patch program has interacted with the
#             inventory so only rolling back one patch at a time to keep it
#             robust.
#
###############################################################################
sub parse_arguments {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $ra_arguments = $$rh_arguments{ra_arguments};

    my $array_size   = scalar( @$ra_arguments );
    my $error_flag   = "";
    my $noop_flag    = 0;
    my $Oracle_Home  = "";
    my $patch_home   = "";
    my $patch_id     = "";

    for ( my $i = 0; $i < $array_size; $i++ ) {
        if ( $$ra_arguments[$i] =~ m#^-oh$# ) {
            if ( ! $Oracle_Home ) {
                $Command::UNRESOLVED_ORACLE_HOME =  $$ra_arguments[$i +1];
                $Oracle_Home = $this_class -> get_abs_path( { Path => $Command::UNRESOLVED_ORACLE_HOME } );
                $ENV{ORACLE_HOME} = $Oracle_Home;
                $i++;
            } else {
                $error_flag .= "Argument -oh appears more than once.\n";
                $i++;
            }

        } elsif ( $$ra_arguments[$i] =~ m#^-id$# ) {
            if ( ! $patch_id ) {
                $patch_id = $$ra_arguments[$i +1];
                $i++;
            } else {
                $error_flag .= "Argument -id appears more than once.\n";
                $i++;
            }

        } elsif ( $$ra_arguments[$i] =~ m#^-i(nvPtrLoc)?$# ) {
            my $inv_ptr_file = $$ra_arguments[$i + 1];
            if ( &File::Basename::basename ( $inv_ptr_file ) ne
                                                          "oraInst.loc" ) {
                $inv_ptr_file = File::Spec -> catfile ( "$inv_ptr_file" ,
                                                            "oraInst.loc" );
            }

            $Command::inventory_location_ptr =
                    "-Doracle.installer.invPtrLoc=" . $$ra_arguments[$i + 1];

            $i++;

        } elsif ( $$ra_arguments[$i] =~ m#^-no_op$# ) {
            $noop_flag = 1;

        } elsif ( $$ra_arguments[$i] =~ m#^-report$# ) {
            $noop_flag = 1;

        } elsif ( $$ra_arguments[$i] =~ m#^-ph$# ) {
            if ( ! $patch_home ) {
                $patch_home = $$ra_arguments[$i +1];
                $i++;
            } else {
                $error_flag .= "Argument -ph appears more than once.\n";
                $i++;
            }

        } elsif ( $$ra_arguments[$i] =~ m#^-local$# ) {
            # this could have been set by Apply, but setting it twice won't hurt
            opatchIO->debug({ message => "-local option specified\n" });
            $Command::LOCAL_NODE_ONLY = 1;

        } elsif ( $$ra_arguments[$i] =~ m#^-silent$# ) {
            opatchIO->print_message ({ message =>
                "-silent: auto-answer 'Y' to any question.\n" });
            $Command::SILENT_MODE = 1;

        } elsif ( $$ra_arguments[$i] =~ m#^-no_relink# ) {
            
            $Command::NO_RELINK_MODE = 1;
                              
        } elsif ( $$ra_arguments[$i] =~ m#^-local_node# ) {
            $Command::LOCAL_NODE_NAME_MODE = 1;
            $Command::LOCAL_NODE_NAME = $$ra_arguments[$i + 1];
            $i++;
 
        } elsif ( $$ra_arguments[$i] =~ m#^-verbose$# ) {
            $Command::verbose_mode = 1;

        } elsif ( $$ra_arguments[$i] =~ m#^-jdk# ) {
            $Command::JDK_LOC = $$ra_arguments[$i + 1];
            $i++;
            
        } elsif ( $$ra_arguments[$i] =~ m#^-jre# ) {
            $Command::JRE_LOC = $$ra_arguments[$i + 1];
            $i++;

        } elsif ( $$ra_arguments[$i] =~ m#^-pre# ) {
            # get the whole parameter list for pre, the command
            # reaches end or -opatch_pre_end explicitly shows the
            # end of -pre options
            my $concat_arg = "";
            my $arg_no = ($i + 1);
            while (($arg_no < $array_size) && ($$ra_arguments[$arg_no] ne "-opatch_pre_end"))
            {
                $concat_arg .= $$ra_arguments[$arg_no] . " ";
                $arg_no ++;
            }
            chomp $concat_arg;
            $Command::PRE_PARAMETERS = $concat_arg;
            $i = $arg_no;

        } elsif ( $$ra_arguments[$i] =~ m#^-post# ) {
            # get the whole parameter list for post, the command
            # reaches end or -opatch_post_end explicitly shows the
            # end of -post options
            my $concat_arg = "";
            my $arg_no = ($i + 1);
            while (($arg_no < $array_size) && ($$ra_arguments[$arg_no] ne "-opatch_post_end"))
            {
                $concat_arg .= $$ra_arguments[$arg_no] . " ";
                $arg_no ++;
            }
            chomp $concat_arg;
            $Command::POST_PARAMETERS = $concat_arg;
            $i = $arg_no;

        } elsif ( $$ra_arguments[$i] =~ m#^-retry# ) {
            $Command::RETRY = $$ra_arguments[$i + 1];
            $i++;

        } elsif ( $$ra_arguments[$i] =~ m#^-delay# ) {
            $Command::DELAY = $$ra_arguments[$i + 1];
            $i++;

        } else {
            $error_flag .= "Problem with the command line arguments.\n";
        }
    }

    # If $Oracle_Home isn't set get it from the environment. Need to check
    # for empty string to cover Microsoft's OSs.
    if ( ( ( ! $Oracle_Home ) && ( $ENV{ORACLE_HOME} ) ) ||
         ( $Oracle_Home eq "" ) ) {
        # save the original value of OH in case it is a symbolic link and get
        #   resolved below
        $Command::UNRESOLVED_ORACLE_HOME = $ENV{ORACLE_HOME};
        
        # Convert into absolute path
        $Oracle_Home = $this_class -> get_abs_path( { Path => $Command::UNRESOLVED_ORACLE_HOME } );
        if ( ! -d $Oracle_Home || ! -e $Oracle_Home ) {
            $error_flag .= "Oracle's home is not a directory: \n" .
                           "$Oracle_Home.\n";
        }
    }

    # If there is still no $Oracle_Home raise an error. IF there is an
    # $Oracle_Home and it's MicroSoft OS lowercase the string.
    if ( ! $Oracle_Home ) {
        $error_flag .= "There is no \$ORACLE_HOME to remove the patch from.\n";
    } elsif ( $OSNAME =~ m#Win32# ){
      $Oracle_Home = lc ($Oracle_Home );
    } else {
        # If $Oracle_Home is a symbolic link, convert it to the real path
        my $is_link = $this_class -> is_symbolic_link ( $Oracle_Home );
        if ( $is_link == 1 ) {
            my $real_OH = $this_class->resolve_symbolic_link($Oracle_Home);

            opatchIO->print_message( {message => "\n-----------------------------------------\n"} );
            opatchIO->print_message( {message => "\nWARNING: Oracle_Home is a symbolic link:\n"} );
            opatchIO->print_message( {message => "$Oracle_Home\n"} );
            opatchIO->print_message( {message => "--->\n"} );
            opatchIO->print_message( {message => "$real_OH"} );
            opatchIO->print_message( {message => "\n-----------------------------------------\n"} );
            # Begin Add: BUG 2899335
            my $is_internal_link = $this_class -> verify_internal_link ($Oracle_Home);
            if ($is_internal_link == 0 ) {
                $Oracle_Home = $real_OH;
            }
            # End Add: BUG 2899335
        }
    }

    # If there is an $Oracle_Home strip any trailing "/" or "\" from it.
    # This upsets the OUI APIs for some reason.
    if ( $Oracle_Home =~ m#[\\/]$# ) {
        ( $Oracle_Home ) = ( $Oracle_Home =~ m#(.+)[\\/]\s?$# );
    }

    # If there is still no $patch_id raise an error.
    if ( ! $patch_id ) {
        $error_flag .= "This command requires a patch identifier to be " .
                       "given.\n";
    }

    # If there is still no $patch_id raise an error.
    if ( ( $patch_home ) && ( ! -d $patch_home ) ) {
        $error_flag .= "The patch location is not a directory: $patch_home.\n";
    }

    my %return_values = ();
    $return_values{error_flag}  = $error_flag;
    $return_values{noop_flag}   = $noop_flag;
    $return_values{Oracle_Home} = $Oracle_Home;
    $return_values{patch_home}  = $patch_home;
    $return_values{patch_id}    = $patch_id;

    return ( \%return_values );

}    # End of parse_arguments().


##############################################################################
#
# NAME   : process_rollback_data
#
# PURPOSE: Organize data from a system call that queries the OUI APIs via
#          Java.
#
# INPUTS : $$ARG[1]{system_data} - The data to process (I hope).
#          $$ARG[1]{patch_id}    - The id of the patch to be rolled-back.
#
# OUTPUTS: %rh_rollback_data     - A reference to hash with that groups the
#                                  the data. Keys are:
#           {ra_one_off_files}   - A list of files
#           {ra_rac_nodes}       - A list of RAC nodes.
#           phnguyen 6/9/03: no ra_rac_nodes output
#
# NOTES  : 1. The patch id is needed because the Java call I used prepends
#             the string "<patch_id>: " to each file.
#          2. Composite files (that is the files that belong in an archive)
#             have the magic string " -> " between the parent (achive) file
#             and the child (object/class) file.
#
###############################################################################
sub process_rollback_data {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $patch_id     = $$rh_arguments{patch_id};
    my $system_data  = $$rh_arguments{system_data};
    my @file_list    = ();
    my $patch_found = 0;
    
    # First need to change it from a possibly very long string.
    my @unprocessed_data = split ( "\n", $system_data );
   
    foreach my $line ( @unprocessed_data ) {

        if ( $line =~ m#^$patch_id:\s+# ) {
            my ( $file_detail ) = ( $line =~ m#^$patch_id:\s+(.+)$# );
            push ( @file_list, $file_detail );
            $patch_found = 1;
        }
    }

    my %rh_installed_data = ();

    $rh_installed_data{ra_file_list} = \@file_list;
    $rh_installed_data{patch_found} = $patch_found;

    return ( \%rh_installed_data );

}   # End of process_rollback_data().

##############################################################################
#
# NAME   : parse_rollback_data 
#
# PURPOSE: Transform the file list in rollback into a hash for use by
#          propagate_patch_for_RAC.
#
# INPUTS : $$ARG[ra_file_list] - A reference to an array that contains the
#                                details about the files.
#
# OUTPUTS: \%files             - A reference to the hash.
#
# NOTES  : 1. This just massages the data so it's in the same form as the
#             other calls to propagate_patch_for_RAC. Yes, it's possible
#             to use 'ref($var)' in p_p_f_RAC but I don't want to confuse
#             people.
#
###############################################################################
sub parse_rollback_data {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $ra_file_list = $$rh_arguments{ra_file_list};
    my $search_OH_jar = $$rh_arguments{search_OH_jar};
    my $Oracle_Home   = $$rh_arguments{Oracle_Home};
    my $patch_id      = $$rh_arguments{patch_id};
    
    my %files = ();

    my $hasJarAction = 0;
    foreach my $file_detail ( @$ra_file_list ) {
        # Remove the unwanted +> NEW charactes.
        if( $file_detail =~ m#(.+) \+> NEW$# )
        {
              ( $file_detail ) = ( $file_detail =~ m#(.+) \+> NEW$# );
        }

        my ( $parent, $child ) = ( $file_detail =~ m#^(.+) -> (.+)$# );
        if ( $child ) {
            # If child is a top-level class file with the form /name.class,
            #   we need to strip away the prefix "/" 
            my @list = split '/', $child;
            my $token= $list[0];
            if($token eq "") {
               if($child =~ m#\.class$#) {
                  $child = $list[1];
                  opatchIO->print_message({message=>"adjust top-level class name to \"$child\""});
               }
            }
            $files{$parent} = $child;
            
            # $target will be something like "foo.jar" or "foo.JAR"
            my $target = &File::Basename::basename ( $parent );
            if ( $target =~ m#\.jar#i) {
               $hasJarAction = 1;
            }
        } else {
            $files{$file_detail} = "";
        }
    }

    # if -jdk is not used, then see if we have jar action but OH/jdk/bin/jar
    #    is not there
    if ( $hasJarAction == 1 && $search_OH_jar == 1 ) {
      
      my $jarCheck = $this_class -> hasOracleHomeJdkBinJar( {
                                         Oracle_Home => $Oracle_Home
                                      } );
      if( $jarCheck == 0 ) {
          $this_class -> free_the_lock_file ( {
                                      Oracle_Home => $Oracle_Home,
                                      patch_id    => $patch_id } );         
                                      
          my $message =  "This patch will update Jar file(s) but " .
          "OPatch was not able to locate executable jar under $Oracle_Home/jdk/bin.";
          opatchIO -> print_message_and_die ( { 
                exit_val => $this_class->ERROR_CODE_PREREQ_PROBLEM,
                message  => $message
          } );
      }    
    }

    return ( \%files );

}   # End of parse_rollback_data().

##############################################################################
#
# NAME: option_retry
#
# PURPOSE: Set the retry option for OPatch to retry locking inventory
#
# ############################################################################
sub option_retry {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-retry <value>";
   $$rh_details{_helpText}     =
                                 "This option tells OPatch how many times\n" .
                                 "it should retry in case of an inventory\n" .
                                 "lock failure.";
    $$rh_details{_required}       = "no";
    #$$rh_details{_type}           = "directory";
}

##############################################################################
#
# NAME: option_delay
#
# PURPOSE: Set the number of seconds to retry locking inventory
#
# ############################################################################
sub option_delay {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "[ -delay <value> ]";
   $$rh_details{_helpText}     =
                                 "If -retry is specified, this option tells\n" .
                                 "OPatch how many seconds it should wait before\n" .
                                 "attempting to lock inventory again in case\n" .
                                 "of a previous failure.";
    $$rh_details{_required}       = "no";
    #$$rh_details{_type}           = "directory";
}

##############################################################################
#
# NAME   : option_id
#
# PURPOSE: Provide the details the patch id option.
#
# INPUTS : $$ARG[rh_option] - A reference to a hash where the details about
#                             the option is to be stored.
#
# OUTPUTS: NONE
#
# NOTES  : 1. The return values are in hash so the caller will get the new
#             details when this subroutine is finished.
#
###############################################################################
sub option_id {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

    $$rh_details{_description}  = "";
    $$rh_details{_helpText}     = "Use 'lsinventory' option to display all patch\n" .
                                  "id's. Each one-off patch is indicated by it's\n" .
                                  "id. To rollback a patch the id for that patch\n" .
                                  "must be supplied.";
    $$rh_details{_required}     = "yes";
    $$rh_details{_type}         = "number";
    $$rh_details{_validityChecks} = [ "eq 0" ];

}   # End of option_id().

##############################################################################
#
# NAME: option_jdk
#
# PURPOSE: Set the jdk option
#
# ############################################################################
sub option_jdk {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-jdk <LOC>";
   $$rh_details{_helpText}     = 
                                 "This option tells OPatch to use JDK (jar)\n" .
                                 "from the specified location instead of\n" .
                                 "the default location under Oracle Home.\n";
    $$rh_details{_required}       = "no";
    $$rh_details{_type}           = "directory";
}

##############################################################################
#
# NAME: option_jre
#
# PURPOSE: Set the jre option
#
# ############################################################################
sub option_jre {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-jre <LOC>";
   $$rh_details{_helpText}     = 
                                 "This option tells OPatch to use JRE (java)\n" .
                                 "from the specified location instead of\n" .
                                 "the default location under Oracle Home.\n";
    $$rh_details{_required}       = "no";
    #$$rh_details{_type}           = "directory";
}

##############################################################################
#
# NAME: option_pre
#
# PURPOSE: Set the pre option
#
# ############################################################################
sub option_pre {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "";
   $$rh_details{_description}  = "-pre <options to be passed into pre> [-opatch_pre_end]";
   $$rh_details{_helpText}     =
                                 "This option specifies the parameters\n" .
                                 "to be passed inside pre script apart\n" .
                                 "from the standard ones.\n" .
                                 "Without this option OPatch calls pre\n" .
                                 "as 'pre -<operation> <patch-id>'\n" .
                                 "With this option OPatch will call pre\n" .
                                 "as 'pre -<operation> <patch-id> <pre options>'\n";
    $$rh_details{_required}       = "no";
    #$$rh_details{_type}           = "directory";
}

##############################################################################
#
# NAME: option_opatch_pre_end
#
# PURPOSE: Set the opatch_pre_end option
#
# ############################################################################
sub option_opatch_pre_end {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-opatch_pre_end";
   $$rh_details{_helpText}     =
                                 "This option is used to mark the\n".
                                 "end of pre options. Without this option\n".
                                 "everything after pre till end of\n".
                                 "the command is passed into pre.\n".
                                 "This option is meaningless without\n".
                                 "-pre option\n";
   $$rh_details{_required}     = "no";
    #$$rh_details{_type}           = "directory";
}

##############################################################################
#
# NAME: option_post
#
# PURPOSE: Set the post option
#
# ############################################################################
sub option_post {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "";
   $$rh_details{_description}  = "-post <options to be passed into post> [-opatch_post_end]";
   $$rh_details{_helpText}     =
                                 "This option specifies the parameters\n" .
                                 "to be passed inside post script apart\n" .
                                 "from the standard ones.\n" .
                                 "Without this option OPatch calls post\n" .
                                 "as 'post -<operation> <patch-id>'\n" .
                                 "With this option OPatch will call post\n" .
                                 "as 'post -<operation> <patch-id> <post options>'\n";
    $$rh_details{_required}       = "no";
    #$$rh_details{_type}           = "directory";
}

##############################################################################
#
# NAME: option_opatch_post_end
#
# PURPOSE: Set the opatch_post_end option
#
# ############################################################################
sub option_opatch_post_end {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-opatch_post_end";
   $$rh_details{_helpText}     =
                                 "This option is used to mark the\n".
                                 "end of post options. Without this option\n".
                                 "everything after post till end of\n".
                                 "the command is passed into post.\n".
                                 "This option is meaningless without\n".
                                 "-post option\n";
   $$rh_details{_required}     = "no";
    #$$rh_details{_type}           = "directory";
}

##############################################################################
#
# NAME   : option_ph
#
# PURPOSE: Provide the details the patch home option.
#
# INPUTS : $$ARG[rh_option] - A reference to a hash where the details about
#                             the option is to be stored.
#
# OUTPUTS: NONE
#
# NOTES  : 1. The return values are in hash so the caller will get the new
#             details when this subroutine is finished.
#
###############################################################################
sub option_ph {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

    $$rh_details{_description}    = "";
    $$rh_details{_helpText}       = "Specify the valid patch directory area.\n".
                                    "Rollback will use the command types \n" .
                                    "found in the patch directory to\n" .
                                    "identify what commands are to be used\n" .
                                    "for the current operating system.";
    $$rh_details{_required}       = "no";
    $$rh_details{_type}           = "directory";
    $$rh_details{_validityChecks} = [ "read" ];

}   # End of option_ph().

##############################################################################
#
# NAME   : option_local
#
# PURPOSE: Provide the details the patch home option.
#
# INPUTS : $$ARG[rh_option] - A reference to a hash where the details about
#                             the option is to be stored.
#
# OUTPUTS: NONE
#
# NOTES  : 1. The return values are in hash so the caller will get the new
#             details when this subroutine is finished.
#
###############################################################################
sub option_local {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-local";
   $$rh_details{_helpText}     =
                                 "Roll back then update " .
                                 "inventory of the local node. \n" .
                                 "Do not propagate the patch or \n" .
                                 "inventory update to other nodes.\n" .
                                 "This option cannot be used together with -local_node." ;
   $$rh_details{_required}       = "no";

}   # End of option_local().

##############################################################################
#
# NAME: option_silent
#
# PURPOSE: Set the silent option
#
# ############################################################################
sub option_silent {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-silent";
   $$rh_details{_helpText}     =
                                 "This suppresses any user-interaction and defaults\n" .
                                 "any Y|N question to Y.\n" .
                                 "This option is not supported on RAC yet.\n" ;
   $$rh_details{_required}       = "no";

}

##############################################################################
#
# NAME: option_report
#
# PURPOSE: Set the report option
#
# ############################################################################
sub option_report {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-report";
   $$rh_details{_helpText}     =
                                 "Invokes OPatch in report only mode";
   $$rh_details{_required}       = "no";

}

##############################################################################
#
# NAME: option_local_node
#
# PURPOSE: Set local node name instead of detecting it 
#
# ############################################################################
sub option_local_node {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-local_node <Name of local node>";
   $$rh_details{_helpText}     =
                                 "OPatch will use the supplied local node name\n" .
                                 "instead of detecting it.  OPatch will trust\n" .
                                 "that the supplied local node name is correct.\n" .
                                 "This option cannot be used together with -local." ;
   $$rh_details{_required}       = "no";

}

##############################################################################
#
# NAME: option_no_relink
#
# PURPOSE: No relink for make actions
#
# ############################################################################
sub option_no_relink {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-no_relink";
   $$rh_details{_helpText}     =
                                 "OPatch will not execute 'make' actions during roll-back.\n" .
                                 "Users will run 'make' manually after patching.\n" .
                                 "This option is valid on Unix platforms only." ;
   $$rh_details{_required}       = "no";

}


##############################################################################
#
# NAME: option_verbose
#
# PURPOSE: Set the verbose option
#
# ############################################################################
sub option_verbose {

   my $this_class = shift @ARG;
   my $rh_arguments = $ARG[0];
   my $rh_details   = $$rh_arguments{rh_option_detail};

   $$rh_details{_defaultValue} = "false";
   $$rh_details{_description}  = "-verbose";
   $$rh_details{_helpText}     =
                                 "This option prints more OPatch output\n" .
                                 "to the screen as well as to the log file\n" ;
   $$rh_details{_required}       = "no";

}


###############################################################################
#
# NAME   : remove_an_applied_patch
#
# PURPOSE: Remove a patch from the appropriate $ORACLE_HOME.
#
#
# INPUTS : $$ARG[1]{fh_log_file}     - The file handle for the log file.
#          $$ARG[1]{noop_flag}       - Just report the actions.
#          $$ARG[1]{Oracle_Home}     - The area that is to be un-patched.
#          $$ARG[1]{patch_id}        - The patch identification string.
#          $$ARG[1]{rh_actions_list} - A hash that contains the details of
#                                      what commands are to be run.
#          $$ARG[1]{ra_file_list}    - The files that need to be un-patched.
#
# OUTPUTS: $error_messages           - A string, possibly over multiple lines,
#                                      containing any errors that were found.
#
# NOTES  : 1. The sort algorithym is:
#                 If it doesn't have " -> " in the name it's a copy file,
#                 For everything else:
#                     If $parent ends in ".jar" its a Java archive,
#                     Else it's an archive file.
#
###############################################################################
sub remove_an_applied_patch {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file     = $$rh_arguments{fh_log_file};
    my $noop_flag       = $$rh_arguments{noop_flag};
    my $Oracle_Home     = $$rh_arguments{Oracle_Home};
    my $patch_id        = $$rh_arguments{patch_id};
    my $ra_file_list    = $$rh_arguments{ra_file_list};
    my $rh_actions_list = $$rh_arguments{rh_actions_list};

    my %archive_files   = ();
    my @copy_files      = ();
    my %jar_files       = ();

    # Sort the files into appropriate groups.
    foreach my $file ( @$ra_file_list ) {
        if ( $file !~ m# -> # ) {
            push ( @copy_files, $file );
        } else {
            # Parent -> child relation.
            my ( $parent, $child ) = ( $file =~ m#^(.+) -> (.+)$# );

            if ( ( $parent =~ m#\.jar$# ) || ( $parent =~ m#\.zip$# ) ) {

              # If child is a top-level class file with the form /name.class,  
              #   we need to strip away the prefix "/"                          
              my @list = split '/', $child;                                     
              my $token= $list[0];                                              
              if($token eq "") {                                               
               if($child =~ m#\.class$#) {                                      
                  $child = $list[1];                                            
                  opatchIO->print_message({
                     message=>"adjust top-level class name to \"$child\""
                  });                                                              
               }                                                                
              }                                                                 
              push ( @{$jar_files{$parent}}, $child );
            } else {
              push ( @{$archive_files{$parent}}, $child );
            }
        }
    }
    opatchIO->print_message({message=>"Restore Archive File"});
    my $archive_file_errors = $this_class -> restore_archive_files ( {
                                      fh_log_file     => $fh_log_file,
                                      noop_flag       => $noop_flag,
                                      Oracle_Home     => $Oracle_Home,
                                      patch_id        => $patch_id,
                                      rh_actions_list => $rh_actions_list,
                                      rh_file_list    => \%archive_files } );

    opatchIO->print_message({message=>"Restore Copy File"});
    my $copy_file_errors = $this_class -> restore_copied_files ( {
                                         fh_log_file     => $fh_log_file,
                                         noop_flag       => $noop_flag,
                                         Oracle_Home     => $Oracle_Home,
                                         patch_id        => $patch_id,
                                         rh_actions_list => $rh_actions_list,
                                         ra_file_list    => \@copy_files } );

    opatchIO->print_message({message=>"Restore Jar File"});
    my $jar_file_errors = "";
    
    if ( $OSNAME =~ m#Win32# ) {
      $jar_file_errors = $this_class -> restore_jar_files_on_Windows ( {
                                         fh_log_file     => $fh_log_file,
                                         noop_flag       => $noop_flag,
                                         Oracle_Home     => $Oracle_Home,
                                         patch_id        => $patch_id,
                                         rh_actions_list => $rh_actions_list,
                                         rh_file_list    => \%jar_files } );
    } else {
      $jar_file_errors = $this_class -> restore_jar_files_on_Unix ( {
                                         fh_log_file     => $fh_log_file,
                                         noop_flag       => $noop_flag,
                                         Oracle_Home     => $Oracle_Home,
                                         patch_id        => $patch_id,
                                         rh_actions_list => $rh_actions_list,
                                         rh_file_list    => \%jar_files } );
    
    }
    my $error_messages = $archive_file_errors . $copy_file_errors .
                                                            $jar_file_errors;
    return ( $error_messages );

}   # End of remove_an_applied_patch().

###############################################################################
#
# NAME   : restore_archive_files
#
# PURPOSE: Restore objcets that were saved during a previous patch.
#
# INPUTS : $$ARG[1]{fh_log_file}     - The file handle for the log file.
#          $$ARG[1]{noop_flag}       - Just report the actions.
#          $$ARG[1]{Oracle_Home}     - The location of the area to be patched.
#          $$ARG[1]{patch_id}        - The patch identification string.
#          $$ARG[1]{ra_file_list}    - The hash of objects to restore. The
#                                          key is an array reference and
#                                          array index values are the objects.
#          $$ARG[1]{rh_actions_list} - A hash that contains the details of
#                                      what commands are to be run.
#
# OUTPUTS: $error_message           - A string that contains the details of
#                                     any errors.
#
# NOTES  :
#
###############################################################################
sub restore_archive_files {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file     = $$rh_arguments{fh_log_file};
    my $noop_flag       = $$rh_arguments{noop_flag};
    my $Oracle_Home      = $$rh_arguments{Oracle_Home};
    my $patch_id        = $$rh_arguments{patch_id};
    my $rh_actions_list = $$rh_arguments{rh_actions_list};
    my $rh_file_list    = $$rh_arguments{rh_file_list};

    my $error_messages  = "";
    my $initial_command = "";

    if ( scalar ( %$rh_file_list ) )
    {
        opatchIO -> print_message_noverbose ( { f_handle => $fh_log_file,
                                  message  => "Restoring archive files...\n" } );
    }
    else
    {
	return ( "" );
    }

    my $archive_command = $this_class->getArchiveCommand();
    # Extract the apply command out of the actions list.
    foreach my $command ( keys %{$$rh_actions_list{archive_lib}{apply}} ) {
        $initial_command = "$archive_command ";
        $initial_command .= $$rh_actions_list{archive_lib}{apply}{$command}
                                                                {parameters};

        my @array_to_count = split ( /%/, $initial_command );
        my $parameter_count = ( scalar (@array_to_count) - 1 );

        # This next regex should leave "ar -rc ".
        ( $initial_command ) = ( $initial_command =~ m#^([^%]+)%# );
    }

    if ( $OSNAME !~ m#Win32# ) {
       my $check = $this_class -> is_symbolic_link ($Oracle_Home);
       if ( $check == 1 ) {
           my $real_OH = $this_class -> resolve_symbolic_link ($Oracle_Home);
           $Oracle_Home= $real_OH;
           opatchIO->print_message( {message => "\nrestore_archive_files: using real Oracle_Home:\n"} );
           opatchIO->print_message( {message => "$Oracle_Home\n"} );
       }
    }

    my $patch_directory   = File::Spec -> catfile ( "$Oracle_Home",
                                                    $this_class -> save_dir,
                                                    $patch_id );

    my $finished_name = "";
    my $error_counter = 1;
    foreach my $file ( keys %$rh_file_list ) {

        my $archive_name = &File::Basename::basename( $file );
        my $dir_name = &File::Basename::dirname( $file );

        # Hack to get around OUI bug that maps sym link names to real paths
        # when being stored in the inventory.
        my $intervening_elements = "";
        my $test_dir = File::Spec -> catfile
                                    ( "$dir_name", "$this_class -> save_dir" );
        while ( ! -e $test_dir ) {
            # Strip off $this_class -> save_dir.
            $test_dir = &File::Basename::dirname( $test_dir );
            $intervening_elements = File::Spec -> catfile (
                                       &File::Basename::basename( $test_dir ),
                                       $intervening_elements );
            $test_dir = &File::Basename::dirname( $test_dir );
            $test_dir = File::Spec -> catfile
                                    ( "$test_dir", $this_class -> save_dir );
        }


        foreach my $object ( @{$$rh_file_list{$file}} ) {
            # In some cases, OUI returns the item name with a leading
            # directory separator. This is a known bug.

            my $test_object = &File::Basename::basename( $object );
            if ( $test_object ) { $object = $test_object; }
            # Has this item been done?
            $finished_name = $object . "_" . $archive_name . "_" .
                                             &Apply::END_PATCH . "_$patch_id";
            $finished_name = File::Spec -> catfile (
                                              "$dir_name", "$finished_name" );
            #$finished_name =~ s#$Oracle_Home#$patch_directory#;
            $finished_name = $this_class -> replace_oh_with_patchdir ({
                                                        Oracle_Home => $Oracle_Home,
                                                        patch_dir   => $patch_directory,
                                                        target_path => $finished_name } );


            if ( -e $finished_name ) {
                     opatchIO -> print_message ( {f_handle => $fh_log_file,
                                                  message => "File patched on previous run, will be skipped : " .
                                                         $finished_name } );
                     next;
            }

            # Now we need to move the file back to the original name and
            # test to see if the file is still available.
            my ( $test_name ) = ( $dir_name =~ m#$Oracle_Home(.+)$# );

             # See above about OUI APIs and sym link handling.
            if ( ! $test_name ) { $test_name = $intervening_elements; }


            $test_name = File::Spec -> catfile ( "$patch_directory",
                                                 "$test_name", "$object" );
            my $original_name = $test_name;
            $test_name .= "_" . $archive_name . "_" . &Apply::PRE_PATCH .
                                                             "_" . $patch_id;

            # Taking care of new files.
            if( ! -e $test_name ) {
                my $new_file_name = "";
                $new_file_name = "$original_name". "_" . $archive_name . "_" . 
					&Apply::OPATCH_NEW_FILE . "_" .$patch_id;
                if ( -e $new_file_name) {

                     opatchIO -> print_message_noverbose({
                                 message => "OPatch found \"$new_file_name\". " .
                                            " So, deleting the file from the archive.\n"
                     });
                 
                     my $sys_command = "";
                     $sys_command = "ar -d " . "$file" . " " .
                                        &File::Basename::basename( $original_name );
                     my ( $sys_call_result ) = qx/$sys_command/;
                     opatchIO -> print_message ( { f_handle => $fh_log_file,
                                              message  => "System Command:" .
                                              "\n\t$sys_command\n" .
                                              "Result:\n" .
                                              "$sys_call_result\n" } );
                     opatchIO -> debug ( {
                            message => "  deleting \"$new_file_name\" ."
                     } );
                     unlink $new_file_name;
                }
                next;
            }

            if ( ( ! -e $test_name ) && ( ! -e $original_name ) ) {
                my $message = "Couldn't restore object $object into the " .
                              "archive $archive_name.\n";
                $error_messages .= $message;
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                              message  => $message } );
                next;
            }

            # So lets restore the object file.
            opatchIO -> print_message ( { message => "Restoring $object to " .
                                                     "$archive_name.\n" } );
            if ( ! $noop_flag ) {
                rename ( $test_name, $original_name );

                my $system_command = $initial_command . "$file " .
                                                               $original_name;
                my ( $sys_call_result ) = qx/$system_command/;
                chomp $sys_call_result;
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                          message  => "System Command:\n\t" .
                                          $system_command . "\nResult:\n" .
                                          "$sys_call_result\n" } );
                my $status = $this_class -> get_child_error ( {
                    CHILD_ERROR     => $CHILD_ERROR
                } );
                if ( $status ) {
                    $error_messages .= "$error_counter.\t$archive_name" .
                                       "\t[ object: " . $object . " ]\n";
                    $error_counter++;
                }

                # Now create an empty file to show this operation was
                # successful.
                if ( ! $status ) {

                    open ( SUCCESS, ">$finished_name" ) || (
                        ( $error_messages .= "$error_counter.\t" .
                                             "$finished_name\n" .
                                             "\t[ Non-critical error " .
                                             "message: $OS_ERROR ]\n"),
                          $error_counter++ );
                    close SUCCESS;
                }
            }
        }
    }

    if ( $error_messages ) {
        $error_messages = "The following archives had problems with being " .
                          "restored:\n" . $error_messages;
    }

    return ( $error_messages );

}   # End of restore_archive_files().

###############################################################################
#
# NAME   : restore_copied_files
#
# PURPOSE: Restore files that were copied into place during a previous patch.
#
# INPUTS : $$ARG[1]{fh_log_file}     - The file handle for the log file.
#          $$ARG[1]{noop_flag}       - Just report the actions.
#          $$ARG[1]{Oracle_Home}     - The location of the area to be patched.
#          $$ARG[1]{patch_id}        - The patch identification string.
#          $$ARG[1]{ra_file_list}    - The list of files to copy back.
#          $$ARG[1]{rh_actions_list} - A hash that contains the details of
#                                      what commands are to be run.
#
# OUTPUTS: $error_message           - A string that contains the details of
#                                     any errors.
#
# NOTES  :
#
###############################################################################
sub restore_copied_files {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file     = $$rh_arguments{fh_log_file};
    my $noop_flag       = $$rh_arguments{noop_flag};
    my $Oracle_Home     = $$rh_arguments{Oracle_Home};
    my $patch_id        = $$rh_arguments{patch_id};
    my $rh_actions_list = $$rh_arguments{rh_actions_list};
    my $ra_file_list    = $$rh_arguments{ra_file_list};

    my $error_message = "";

    if ( scalar (@$ra_file_list))
    {
        opatchIO -> print_message_noverbose ( { f_handle => $fh_log_file,
                                message  => "Restoring copied files...\n" } );
    }
    else
    {
	return ( "" );
    }

    # Extract the apply command out of the actions list.
    foreach my $command ( keys %{$$rh_actions_list{copy}{apply}} ) {
        my $parameter_count = $$rh_actions_list{copy}{apply}{$command}
                                                                 {param_num};
        my $required_module = $$rh_actions_list{copy}{apply}{$command}{module};
    }

    use File::Copy();

    if ( $OSNAME !~ m#Win32# ) {
       my $check = $this_class -> is_symbolic_link ($Oracle_Home);
       if ( $check == 1 ) {
           my $real_OH = $this_class -> resolve_symbolic_link ($Oracle_Home);
           $Oracle_Home= $real_OH;
           opatchIO->print_message( {message => "\nrestore_copied_files: using real Oracle_Home:\n"} );
           opatchIO->print_message( {message => "$Oracle_Home\n"} );
       }
    }


    my $patch_directory   = File::Spec -> catfile ( "$Oracle_Home",
                                                    $this_class -> save_dir,
                                                    $patch_id );
    if ( $OSNAME =~ m#Win32# ) {
        $patch_directory = lc ( $patch_directory );
        $Oracle_Home =~ s/\\/\\\\/g;
    }

    my $error_counter = 1;

    my $new_flag = 0;

    foreach my $file ( @$ra_file_list ) {

        $new_flag = 0;

        # If it was a new file remove it.
        if ( $file =~ m# \+> NEW$# ) {
            ( $file ) = ( $file =~ m#(.+) \+> NEW$# );
            # $new_flag = 1;
            # Bug 3436586: Do not remove to take of the case where
            # the file in OH but not in filemap inventory
            # unlink $file;
        }

        # The inventory returns the details with the a unix style directory
        # separator.
        if ( $OSNAME =~ m#Win32# ) {
            $file =~ s/\//\\/g;
            $file = lc ( $file );
        }

        my $dirname = &File::Basename::dirname ( $file );
        my $test_name = $file;
        #$test_name =~ s#$Oracle_Home#$patch_directory#;
        $test_name = $this_class -> replace_oh_with_patchdir ({
                                                   Oracle_Home => $Oracle_Home,
                                                   patch_dir   => $patch_directory,
                                                   target_path => $test_name } );

        my $finished_name = $test_name;
        $test_name = $test_name . "_" . &Apply::PRE_PATCH . "_" .
                                                                   $patch_id;

        # Have we done this on a previous run?
        $finished_name .= "_" . &Apply::END_PATCH .  "_$patch_id";

        if ( -e $finished_name ) {
                opatchIO -> print_message ( {f_handle => $fh_log_file,
                                             message => "File patched on previous run, will be skipped : " .
                                                         $finished_name } );
                next;
        }

        # Test to see if the file is still available.
        # Bug 3436586: Check for the case where the file in OH but
        # not present in the filemap
        if ( ( ! -e $test_name ) ) {
		# Check if opatch_new file is present in .patch_storage,
		# if not then instead of removing the file
                # let's back it up to the same
                # location with a new extension using patch ID.
                # The logic below ensures that a new file introduced by an
                # one-off will be removed, and existing file will not.
                # The extra back-up using patch ID below covers the error case
                # when the file is existing file but somehow the .patch_storage
                # doesn't have a back-up copy of it.
                # 
                my $new_name = "";
                ( $new_name ) = ( $file =~ m#$Oracle_Home(.+)# );
                $new_name = File::Spec -> catfile ( "$patch_directory", "$new_name" );
                $new_name = "$new_name" . "_". &Apply::OPATCH_NEW_FILE . "_$patch_id";

                if ( $OSNAME =~ m#Win32# ) {
                      $new_name =~ s/\\\\/\\/g;
                }

                if( -e $new_name) {
                    opatchIO->print_message( {message => 
                         "OPatch found the file in backup area \n\"$new_name\""
                    } ); 
                    opatchIO->print_message( {
                          message => 
                     "So, \"$file\" is treated as a new file, and removing it\n"
                    } ); 

                    unlink $new_name;

                    if( -e $file ) {
                        # Just backup the file in .patch_storage/<patchID>/<filename>_moved_from_OH_<patchID>
                        # to make sure we are always on the safer side, coz we still haven't got complete control
                        # of this legacy code. :)
                        my $backup_name = "";
                        ( $backup_name ) = ( $file =~ m#$Oracle_Home(.+)# );
                        $backup_name = File::Spec -> catfile ( "$patch_directory", "$backup_name" );
                        $backup_name = "$backup_name" . "_". "moved_from_OH" . "_$patch_id";

                        opatchIO->debug( {
                                message => "   file \"$file\" is backed up as \n    \"$backup_name\" \n   before deleting.\n"
                        } );

                        File::Copy::copy( $file, $backup_name );
                        unlink $file;
                   }
                   else {
                        opatchIO -> print_message ( { message => "File \"$file\" does not exist. Nothing to delete.\n" } );
                   }
                }
                else {
                  if ( -e $file ) {
                    my $removedFile = $file . "_removed_$patch_id";
                    if ( -e $removedFile ) {
                      unlink $removedFile;
                    }
                    File::Copy::copy ( $file, $removedFile );
                    unlink $file;
		    opatchIO->print_message_noverbose( {
                       message => "\nWarning: Cannot restore the file"} ); 
		    opatchIO->print_message_noverbose( {
                       message => "\"$file\""} ); 
		    opatchIO->print_message_noverbose( {
                       message => "as OPatch can't find backup file, so it is backed up as"} ); 
		    opatchIO->print_message_noverbose( {
                       message => "\"$removedFile\""} ); 
                  }
                }
            next;
        }

        # So lets restore the file.
        my $status = 0;
        opatchIO -> print_message ( { message => "Restoring $file.\n" } );

        if ( ! $noop_flag ) {
            # if ( ! $new_flag ) {
            # Just blindly restore the file if it exists in .patch_storage
            if ( -e $test_name ) {
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                          message  => "Restoring file: " .
                                          "$test_name -> $file\n" } );
                File::Copy::copy ( $test_name, $file ) || (
                    ( $error_message .= "$error_counter.\t$file\n" ),
                                          $error_counter++, $status = 1,
                      next );

            }

            # Now create an empty file to signify this operation was
            # successful.
            if ( ! $status ) {

                open ( SUCCESS, ">$finished_name" ) || (
                    ( $error_message .= "$error_counter.\t$finished_name\n" .
                                         "\t[ Non-critical error message: " .
                                         "$OS_ERROR ]\n"),
                      $error_counter++ );
                close SUCCESS;
            }
        }
    }

    if ( $error_message ) {
        $error_message = "The following files had problems with being " .
                         "restored:\n" . $error_message;
    }

    return ( $error_message );

}   # End of restore_copied_files().

###############################################################################
#
# NAME   : restore_jar_files_on_Unix
#
# PURPOSE: Restore objects and classes that were saved during a previous patch.
#
# INPUTS : $$ARG[1]{fh_log_file}     - The file handle for the log file.
#          $$ARG[1]{noop_flag}       - Just report the actions.
#          $$ARG[1]{Oracle_Home}     - The location of the area to be patched.
#          $$ARG[1]{patch_id}        - The patch identification string.
#          $$ARG[1]{ra_file_list}    - The hash of objects to restore. The
#                                          key is an array reference and
#                                          array index values are the objects.
#          $$ARG[1]{rh_actions_list} - A hash that contains the details of
#                                      what commands are to be run.
#
# OUTPUTS: $error_message           - A string that contains the details of
#                                     any errors.
#
# NOTES  :
#
###############################################################################
sub restore_jar_files_on_Unix {

    use File::Copy();
    use Cwd;

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file     = $$rh_arguments{fh_log_file};
    my $noop_flag       = $$rh_arguments{noop_flag};
    my $Oracle_Home     = $$rh_arguments{Oracle_Home};
    my $patch_id        = $$rh_arguments{patch_id};
    my $rh_actions_list = $$rh_arguments{rh_actions_list};
    my $rh_file_list    = $$rh_arguments{rh_file_list};

    my $error_messages = "";
    my $initial_command = "";

    if( scalar (%$rh_file_list))
    {
        opatchIO -> print_message_noverbose ( { f_handle => $fh_log_file,
                                  message  => "Restoring jar files...\n" } );
    }
    else
    {
	return ( "" );
    }

    # Extract the apply command out of the actions list.
    foreach my $command ( keys %{$$rh_actions_list{archive_jar}{apply}} ) {
        $initial_command = $$rh_actions_list{archive_jar}{apply}{jar}{path};
        $initial_command = File::Spec -> catfile
                                         ( "$initial_command" , "$command" );
        $initial_command .= " " . $$rh_actions_list{archive_jar}{apply}
                                                      {$command}{parameters};

        # This next regex should leave "$ORACLE_HOME/jdk/bin/jar -uf ".
        ( $initial_command ) = ( $initial_command =~ m#^([^%]+)%# );
    }
    # Either RollBack main or Apply (if it's auto-rollback) pre-req. checks 
    #  has ensured that JDK_LOC points to an executable jar.exe
    if ( $Command::JDK_LOC ne "" ) {
        opatchIO->print_message ({ message => "  use jar from supplied -jdk option.\n" } );
        $initial_command = "$Command::JDK_LOC -uf ";
    }

    if ( $OSNAME !~ m#Win32# ) {
       my $check = $this_class -> is_symbolic_link ($Oracle_Home);
       if ( $check == 1 ) {
           my $real_OH = $this_class -> resolve_symbolic_link ($Oracle_Home);
           $Oracle_Home= $real_OH;
           opatchIO->print_message( {message => "\nrestore_jar_files: using real Oracle_Home:\n"} );
           opatchIO->print_message( {message => "$Oracle_Home\n"} );
       }
    }
    my $patch_directory   = File::Spec -> catfile ( "$Oracle_Home",
                                                    $this_class -> save_dir,
                                                    $patch_id );

    my $finished_name = "";
    my $error_counter = 1;
    foreach my $file ( keys %$rh_file_list ) {

        my $archive_name = &File::Basename::basename( $file );
        my $dir_name = &File::Basename::dirname( $file );

        # *.zip files don't use compression.
        my $real_command = $initial_command;
        if ( $archive_name =~ m#\.zip$# ) {
            $real_command =~ s#-uf#-uf0#;
        }

        foreach my $object ( @{$$rh_file_list{$file}} ) {

            # Now we need to move the file back to the original name.
            my $original_name = File::Spec -> catfile (
                                                    "$dir_name", "$object" );
            #$original_name =~ s#$Oracle_Home#$patch_directory#;
            $original_name = $this_class -> replace_oh_with_patchdir ({
                                                        Oracle_Home => $Oracle_Home,
                                                        patch_dir   => $patch_directory,
                                                        target_path => $original_name } );



            # Test to see if the file is still available.
            my $test_name = $original_name . "_" . $archive_name . "_" .
                                         &Apply::PRE_PATCH . "_" . $patch_id;

            $finished_name = $original_name . "_" . $archive_name . "_" .
                                         &Apply::END_PATCH . "_" . $patch_id;

            # Have we already done this file?
            if ( -e $finished_name ) {
                     opatchIO -> print_message ( {f_handle => $fh_log_file,
                                                  message => "File patched on previous run, will be skipped : " .
                                                              $finished_name } );
                     next;
            }

            if ( ! -e $test_name ) {
                $error_messages .= "$test_name does not exist.\n";
                $error_messages .= "Couldn't restore object $object into " .
                                   "the jar file $archive_name.\n";
                next;
            }

            # So lets restore the object file.
            opatchIO -> print_message ( { f_handle => $fh_log_file,
                                          message  => "Restoring $object to " .
                                                     "$archive_name.\n" } );

            if ( ! $noop_flag ) {
                File::Copy::copy ( $test_name, $original_name );

                my $patch_loc = $dir_name;
                #$patch_loc =~ s#$Oracle_Home#$patch_directory#;
                $patch_loc = $this_class -> replace_oh_with_patchdir ({
                                                        Oracle_Home => $Oracle_Home,
                                                        patch_dir   => $patch_directory,
                                                        target_path => $patch_loc } );



                # class_file might be Foo$InnerClassName.class, so we need to 'escape'
                #   the $ in class file
                my $class_file = $object;     
                my $inner_class_file = $this_class->getInnerClassName({ class_file => $class_file });

                # my $system_command = "cd $patch_loc; " . 
                #   $real_command . "$file " . $inner_class_file;
                # the above chain cmd doesn't work for Windows.  Use Perl chdir

                my $system_command = "$real_command" . "$file " . 
                                          "$inner_class_file";
                my $curr_dir = cwd();
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                          message  => "  chdir to $patch_loc" });
                chdir $patch_loc;               
                my ( $sys_call_result ) = qx/$system_command/;
                chdir $curr_dir;

                chomp $sys_call_result;
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                          message  => "  " .
                                          $system_command . "\n  Result: " .
                                          "$sys_call_result" } );
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                          message  => "  chdir back to $curr_dir\n" });
                my $status = $this_class -> get_child_error ( {
                    CHILD_ERROR     => $CHILD_ERROR
                } );
                if ( $status ) {
                    $error_messages .= "$error_counter.\t$archive_name" .
                                       "\t[ object: " . $object . " ]\n";
                    $error_counter++;
                }

                # Now create an empty file to show this operation was
                # successful.
                if ( ! $status ) {

                    open ( SUCCESS, ">$finished_name" ) || (
                        ( $error_messages .= "$error_counter.\t" .
                                             "$finished_name\n" .
                                             "\t[ Non-critical error " .
                                             "message: $OS_ERROR ]\n"),
                          $error_counter++ );
                    close SUCCESS;
                }
            }
        }
    }

    if ( $error_messages ) {
        $error_messages = "The following jar files had problems with being " .
                          "restored:\n" . $error_messages;
    }

    return ( $error_messages );

}   # End of restore_jar_files_on_Unix().

###############################################################################
#
# NAME   : restore_jar_files_on_Windows
#
# PURPOSE: Restore objects and classes that were saved during a previous patch.
#
# INPUTS : $$ARG[1]{fh_log_file}     - The file handle for the log file.
#          $$ARG[1]{noop_flag}       - Just report the actions.
#          $$ARG[1]{Oracle_Home}     - The location of the area to be patched.
#          $$ARG[1]{patch_id}        - The patch identification string.
#          $$ARG[1]{ra_file_list}    - The hash of objects to restore. The
#                                          key is an array reference and
#                                          array index values are the objects.
#          $$ARG[1]{rh_actions_list} - A hash that contains the details of
#                                      what commands are to be run.
#
# OUTPUTS: $error_message           - A string that contains the details of
#                                     any errors.
#
# NOTES  :
#
###############################################################################
sub restore_jar_files_on_Windows {

    use File::Copy();
    use Cwd;

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file     = $$rh_arguments{fh_log_file};
    my $noop_flag       = $$rh_arguments{noop_flag};
    my $Oracle_Home     = $$rh_arguments{Oracle_Home};
    my $patch_id        = $$rh_arguments{patch_id};
    my $rh_actions_list = $$rh_arguments{rh_actions_list};
    my $rh_file_list    = $$rh_arguments{rh_file_list};

    my $error_messages = "";
    my $initial_command = "";

    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "Restoring jar files...\n\n" } );

    # Extract the apply command out of the actions list.
    foreach my $command ( keys %{$$rh_actions_list{archive_jar}{apply}} ) {
        $initial_command = $$rh_actions_list{archive_jar}{apply}{jar}{path};
        $initial_command = File::Spec -> catfile
                                         ( "$initial_command" , "$command" );
        $initial_command .= " " . $$rh_actions_list{archive_jar}{apply}
                                                      {$command}{parameters};

        # This next regex should leave "$ORACLE_HOME/jdk/bin/jar -uf ".
        ( $initial_command ) = ( $initial_command =~ m#^([^%]+)%# );
    }
    # Either RollBack main or Apply (if it's auto-rollback) pre-req. checks 
    #  has ensured that JDK_LOC points to an executable jar.exe
    if ( $Command::JDK_LOC ne "" ) {
        opatchIO->print_message ({ message => "  use jar from supplied -jdk option.\n" } );
        $initial_command = "$Command::JDK_LOC -uf ";
    }

    if ( $OSNAME !~ m#Win32# ) {
       my $check = $this_class -> is_symbolic_link ($Oracle_Home);
       if ( $check == 1 ) {
           my $real_OH = $this_class -> resolve_symbolic_link ($Oracle_Home);
           $Oracle_Home= $real_OH;
           opatchIO->print_message( {message => "\nrestore_jar_files: using real Oracle_Home:\n"} );
           opatchIO->print_message( {message => "$Oracle_Home\n"} );
       }
    }
    my $patch_directory   = File::Spec -> catfile ( "$Oracle_Home",
                                                    $this_class -> save_dir,
                                                    $patch_id );

    my $finished_name = "";
    my $error_counter = 1;
    foreach my $file ( keys %$rh_file_list ) {

        # archive_name will be "xsu12.jar"
        # dir_name will be "OH\lib"
        my $archive_name = &File::Basename::basename( $file );
        my $dir_name = &File::Basename::dirname( $file );
        $dir_name = $this_class->convert_path_separator({path=>$dir_name});
        
        # *.zip files don't use compression.
        my $real_command = $initial_command;
        if ( $archive_name =~ m#\.zip$# ) {
            $real_command =~ s#-uf#-uf0#;
        }

        foreach my $object ( @{$$rh_file_list{$file}} ) {

            # Now we need to move the file back to the original name.
            # patch_directory will be "OH\.patch_storage\<ID>
            
            # object is "oracle\xml\sql\foo.class"
            # class_path will be "oracle/xml/sql/foo.class"
            my @list = split "\\\\", $object;
            my $class_path = "";
            foreach my $token ( @list ) {
              if ($class_path eq "") {
                $class_path = $token;
              } else {
                $class_path = $class_path . "/" . $token;
              }
            }
            
            my $original_name = File::Spec -> catfile (
                                                    "$dir_name", "$object" );
            $original_name = $this_class->convert_path_separator({path=>$original_name});
            # original_name is "OH/<jar_path>/class_path, i.e.
            #   u:\windows\home\lib\xsu12.jar
            # 
            # we will extract <jar_path> out (the "lib" part)
            my $len = length($Oracle_Home);
            $original_name = substr($original_name, $len + 1);
            my $jar_path_len = length($original_name) - length($class_path);
            my $jar_path = substr($original_name, 0, $jar_path_len - 1 );

            # Test to see if the file is still available.
            my $test_name = $patch_directory . "\\" .
                             $jar_path . "\\" .
                             $object . 
                             "_" . $archive_name . 
                             "_" .
                              &Apply::PRE_PATCH  . 
                             "_" . 
                             $patch_id;
                                               
            $finished_name = $patch_directory . "\\" .
                             $jar_path . "\\" .
                             $object . 
                             "_" . $archive_name . 
                             "_" .
                             &Apply::END_PATCH . 
                             "_" . 
                             $patch_id;

            # Have we already done this file?
            if ( -e $finished_name ) {
                     opatchIO -> print_message ( {f_handle => $fh_log_file,
                                                  message => "File patched on previous run, will be skipped : " .
                                                              $finished_name } );
                     next;
	    }

            if ( ! -e $test_name ) {
                $error_messages .= "$test_name does not exist.\n";
                $error_messages .= "Couldn't restore object $class_path into " .
                                   "the jar file $archive_name.\n";
                next;
            }

            # So lets restore the object file.
            opatchIO -> print_message ( { f_handle => $fh_log_file,
                                          message  => "  restoring $class_path to " .
                                                     "$archive_name.\n" } );

            if ( ! $noop_flag ) {
                $original_name = $patch_directory . "\\" . $original_name;
                opatchIO->print_message({message=>"  copy $test_name  $original_name\n"});
                File::Copy::copy ( $test_name, $original_name );

                my $patch_loc = $dir_name;
      
                # The command below fails with Unicode error if there is a \p
                #   in Oracle_Home, as in C:\phi
                # $patch_loc =~ s#$Oracle_Home#$patch_directory#;
                # What we tried to do here is to convert patch_loc from
                #   C:\phi\lib to C:\phi\.patch_storage\lib
                $patch_loc =  substr ($patch_loc, length($Oracle_Home));
                $patch_loc = $patch_directory . "\\" . $patch_loc;

                # class_file might be Foo$InnerClassName.class, so we need to 'escape'
                #   the $ in class file
                my $class_file = $class_path;     
                my $inner_class_file = $this_class->getInnerClassName({ class_file => $class_file });

                # my $system_command = "cd $patch_loc; " . 
                #   $real_command . "$file " . $inner_class_file;
                # the above chain cmd doesn't work for Windows.  Use Perl chdir

                my $win_file =  $this_class->convert_path_separator({path=>$file});
                my $system_command = "$real_command" . "$win_file " . 
                                          "$inner_class_file";
                my $curr_dir = cwd();
                # $curr_dir = $this_class->convert_path_separator({path=>$curr_dir});
                $patch_loc = $patch_directory . "\\" . $jar_path;
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                          message  => "  chdir to $patch_loc" });
                chdir $patch_loc;               
                my ( $sys_call_result ) = qx/$system_command/;
                chdir $curr_dir;

                chomp $sys_call_result;
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                          message  => "  " .
                                          $system_command . "\n  Result: " .
                                          "$sys_call_result" } );
                opatchIO -> print_message ( { f_handle => $fh_log_file,
                                          message  => "  chdir back to $curr_dir\n" });
                my $status = $this_class -> get_child_error ( {
                    CHILD_ERROR     => $CHILD_ERROR
                } );
                if ( $status ) {
                    $error_messages .= "$error_counter.\t$archive_name" .
                                       "\t[ object: " . $object . " ]\n";
                    $error_counter++;
                }

                # Now create an empty file to show this operation was
                # successful.
                if ( ! $status ) {

                    open ( SUCCESS, ">$finished_name" ) || (
                        ( $error_messages .= "$error_counter.\t" .
                                             "$finished_name\n" .
                                             "\t[ Non-critical error " .
                                             "message: $OS_ERROR ]\n"),
                          $error_counter++ );
                    close SUCCESS;
                }
            }
        }
    }

    if ( $error_messages ) {
        $error_messages = "The following jar files had problems with being " .
                          "restored:\n" . $error_messages;
    }

    return ( $error_messages );

}   # End of restore_jar_files_on_Windows().
###############################################################################
#
# NAME   : rollback_safety_check
#
# PURPOSE: Return details of the database to so the calling function can
#          determine if it's safe to continue.
#
# INPUTS : $$ARG[1]{fh_log_file}     - The file handle for the log file.
#          $$ARG[1]{inventory_file} - The file that should contain which
#                                     files are to be checked for activity.
#          $$ARG[1]{Oracle_Home}    - The area that is to be un-patched.
#          $$ARG[1]{patch_id}       - The patch identification string.
#          $$ARG[1]{patch_location} - The location of a patch directory.
#
# OUTPUTS: %rh_db_report            - A hash containing the details of the
#                                     database with the keys of:
#                   {RAC}           - Details of any RAC nodes.
#                   {active}        - Which files have processes attached
#                                     when they should be quite.
#                   {files}         - Which files have been patched.
#
# NOTES  : 1. This bundles-up the details needed to minimize the number of
#             invokations to the Java virtual machine.
#          2. Checks are for:
#             a. Active processes on "critical files".
#             c. Is this a RAC installation and return any nodes found.
#          3. Java only takes strings for arguments so arrays and hashes need
#             to be reformatted into single strings.
#          4. See Command::initial_safety_check() for a similar subroutine.
#
###############################################################################
sub rollback_safety_check {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file     = $$rh_arguments{fh_log_file};
    my $inventory_file  = $$rh_arguments{inventory_file};
    my $Oracle_Home     = $$rh_arguments{Oracle_Home};
    my $patch_id        = $$rh_arguments{patch_id};
    my $patch_location  = $$rh_arguments{patch_location};

    my %check_results  = ();
    my $rh_executable_files = "";

    # Check 1: Active processes on critical files.
    if ( $patch_location =~ m#$patch_id$# ) {
        my $rh_global_details = Apply -> read_and_check_reference_details ( {
                                        inventory_file => $inventory_file,
                                        Oracle_Home    => $Oracle_Home,
                                        p_i_l          => $patch_location } );

        $rh_executable_files = $$rh_global_details{executable};

    } else {
        $rh_executable_files = $this_class -> build_executable_checklist (
                                         { Oracle_Home    => $Oracle_Home } );
    }
    
    my $active_files = "";
    $active_files  = $this_class -> check_if_database_running ( {
                                 Oracle_Home    => $Oracle_Home,
                                 rh_executables => $rh_executable_files } );
    
    $check_results{files_in_use} = $active_files;

    return ( \%check_results );

}   # End of rollback_safety_check().

###############################################################################
#
# NAME   : update_inventory
#
# PURPOSE: Update the products inventory by removing details of the
#          patch.
#
# INPUTS : $$ARG[1]{fh_log_file}      - The file handle for the log file.
#          $$ARG[1]{Oracle_Home}    - The area that is to be patched.
#          $$ARG[1]{patch_id}       - The patch identification string.
#          $$ARG[1]{rh_OUI_file_names}
#                                   - A list of files that are needed when
#                                     invoking OUI through the JVM.
#
# OUTPUTS: $error_messages          - A string containing any errors that
#                                     were encountered.
#
# NOTES  :
#
###############################################################################
sub update_inventory {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $fh_log_file       = $$rh_arguments{fh_log_file};
    my $Oracle_Home       = $$rh_arguments{Oracle_Home};
    my $patch_id          = $$rh_arguments{patch_id};
    my $rh_OUI_file_names = $$rh_arguments{rh_OUI_file_names};

    my %check_results  = ();
    my $error_messages = "";

    my $path_to_java = $$rh_OUI_file_names{path_to_java};

    # First set-up the Java class for the inventory read.
    my $rh_java_paths = $this_class -> make_java_paths ( {
                                Oracle_Home       => $Oracle_Home,
                                rh_OUI_file_names => $rh_OUI_file_names } );

    my $class_path = $$rh_java_paths{class};
    my $oui_path   = $$rh_java_paths{oui_path};

    opatchIO -> print_message ( { message => "Removing details of patch ".
                                             "$patch_id from the " .
                                             "inventory.\n" } );

    # Now set-up the call and capture the results. Quoting $file_names to
    # make the $file_names a single argument to the Java program.
    # This may need some tuning.
    my $debug = $Command::DEBUG;
    my $class_to_run = "opatch/RemovePatchDetails";
    my $program_name = &File::Basename::basename( $PROGRAM_NAME );
    my $version_num  = $this_class -> version;
    my $local_node_only_flag = "";

    if ($Command::LOCAL_NODE_ONLY) {
        $local_node_only_flag = "-Dopatch.local_node_only";
    }
    my $local_node_name_flag = "";
    if ($Command::LOCAL_NODE_NAME_MODE) {
        $local_node_name_flag = "-Dopatch.local_node_name=$Command::LOCAL_NODE_NAME";
    }

    my $is_shared_flag = "";
    # User wanna turn off CFS detection?
    #    OPATCH_IS_SHARED = true   -> CFS=true, no detect
    #                       false        =false, no detect
    #                       detect       =<detected value>, yes detect
    #                       not set      =false, no detect
    if ($Command::OPATCH_IS_SHARED eq "TRUE") {
        $is_shared_flag = "-Dopatch.is_shared=true";
		    opatchIO -> print_message ( {
               		 message => "No CFS detection, CFS set to true."
              	  });        
    } elsif ( $Command::OPATCH_IS_SHARED eq "FALSE" ) {
        $is_shared_flag = "-Dopatch.is_shared=false";
		    opatchIO -> print_message ( {
               		 message => "No CFS detection, CFS set to false."
              	  });            
    } elsif ( $Command::OPATCH_IS_SHARED eq "DETECT" ) {
        $is_shared_flag = "-Dopatch.is_shared=detect";
		    opatchIO -> print_message ( {
               		 message => "PerformCFS detection, CFS set to detected value."
              	  });        
    } elsif ( $Command::OPATCH_IS_SHARED eq "" ) {
        $is_shared_flag = "";
        # Env. variable OPATCH_IS_SHARED is not set, so ignore it.
    } else {
        $is_shared_flag = "";
		    opatchIO -> print_message ( {
               		 message => "Unknown OPATCH_IS_SHARED value, no CFS detection, CFS set to false."
              	  });        
    } 
    
    my $system_command = join(' ', $path_to_java, $local_node_only_flag,
                        $is_shared_flag,
                        $local_node_name_flag, 
                        $Command::inventory_location_ptr,
                        $debug, 
                        $class_path, $class_to_run,
                        $oui_path,
                        qq(").$Oracle_Home.qq("), $program_name, 
                        $version_num, $patch_id);
    $system_command = $this_class->getDependentPerlCommand({ system_command => $system_command });

    $this_class -> warn_about_retry( { f_handle => $fh_log_file } );
    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "\nSystem Command: $system_command"
                              } );

    #my $sys_call_result = qx/$system_command/;

    opatchIO -> print_message ( { f_handle => $fh_log_file,
                                  message  => "\nResult:\n" 
                              } );

    my $sys_call_result = $this_class -> run_cmd_print_live ( { sys_command => $system_command,
                                              level => 1,
                                              f_handle => $fh_log_file } );


    chomp $sys_call_result;

    my $status = $this_class -> get_child_error ( {
          CHILD_ERROR     => $CHILD_ERROR
    } );

    if ( $status ) {
            $error_messages .= "Error when updating the inventory:\n" .
                               "[ $sys_call_result ]\n";
    }

    return ( $error_messages );

}   # End of update_inventory().

###############################################################################
#
# NAME   : validate_options
#
# PURPOSE: A method to validate the command line.
#
# INPUTS : $$ARG[1]{rh_command_line} - A reference to a hash that has the
#                                      command and options to be validated.
#          $$ARG[1]{rh_command_list} - A reference to a hash containg details
#                                      about the valid commands and options.
#
# OUTPUTS: Nothing directly. If validation fails, the attibute {invalid} is
#          defined for rh_command_line.
#
# NOTES  : 1. Check the given arguments (in $rh_command_line) against the list
#             of valid arguments for the command. If the argument has a
#             an attribute _validityChecks then the value from the command
#             line is validated by the appropriate method.
#          2. If you're trying to debug this (or follow it in a debugger) the
#             following arguments to 'keys' should help:
#             a. %${$$rh_command_list{$command}{_options}}, and
#             b. ${$${$$rh_command_list{$command}{_options}}{$type}}{$attribute}#             but the loops to see all the values are left to the reader.
#          3. Assumes that the return value from any validation method is
#             a null string for success or a descriptive string of the failure.
#             This is stored in $error_string.
#
###############################################################################
sub validate_options {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $rh_command_line = $$rh_arguments{rh_command_line};
    my $rh_command_list = $$rh_arguments{rh_command_list};

    my $command         = $$rh_command_line{command};
    my @arguments       = split ( " ", $$rh_command_line{arguments});


    my $upper_bound = scalar ( @arguments );
    for ( my $i = 0; $i < $upper_bound; $i++ ) {

        my $arg_name = $arguments[$i];
        my $invalid_flag = 1;

        # Skip global options, already validated.
        next if ( exists $$rh_command_list{$arg_name} );

        # Skip if it doesn't begin with a "-" symbol. Apply allows a single
        # string for OralceHome/ShipHome in the processesing.
        next if ( $arg_name !~ m#^-# );

        # Now check for bogus arguments. Because of perl's autovivification
        # this test needs be done first as the attribute would be created
        # by the validity check if it gets that far.

        foreach my $valid_argument
                ( keys %${$$rh_command_list{$command}{_options}} ) {
            if ( index ($valid_argument, $arg_name ) > -1 ) {
                $invalid_flag = 0;
            }
        }

        if ( $invalid_flag ) {
            if ( exists $$rh_command_line{invalid} ) {
                $$rh_command_line{invalid} .= "\nInvalid argument: $arg_name.";
            } else {
                $$rh_command_line{invalid} = "Invalid argument: $arg_name.";
            }
            next;
        }

        # Does this option, now valid, require a check on the value?
        # ( Yes Virgina, you can split hashes like this to avoid line wraps. )
        if ( ${$${$$rh_command_list{$command}{_options}}
                                           {$arg_name}}{_validityChecks} ) {
            my $check_type = ${$${$$rh_command_list{$command}{_options}}
                                                        {$arg_name}}{_type};
            my $rule_set =   ${$${$$rh_command_list{$command}{_options}}
                                              {$arg_name}}{_validityChecks};

            # Out of hubris I've assumed all validation methods to start
            # with "validate_".
            my $validate_method = "validate_" . $check_type;
            my $error_string = $this_class -> $validate_method ( {
                                        item_to_validate => $arguments[$i+1],
                                        validation_rules => $rule_set } );
            $i++;
            if ( $error_string ) {
                $$rh_command_line{invalid} .= $error_string;
            }
        }
    }
}   # End of validate_options().

###############################################################################
#
# NAME   : dummy
#
# PURPOSE: A place to use variable a second time to suppress warnings
#          that are bogus.
#
# INPUTS : None.
#
# OUTPUTS: None.
#
###############################################################################
sub dummy {

    my $dummy = $Command::do_not_update_inventory;
}   # Dummy.
#############################################################################
# Perl packages need to return a true value.
1;
#############################################################################

