#
# Copyright (c) 2001, 2005, Oracle. All rights reserved.  
#
#    NAME
#    opatchIO
#
#    DESCRIPTION
#    This file contains the I/O methods for opatch.
#
#    NOTES
#    1.    This could be included in another file but it is much easier to
#          maintain as a seperate module.
#    2.    This was mainly done to support XML processing. The existing perl
#          modules all require expat.pm which is a C program. Since opatch
#          is meant to be OS independent we need a pure perl module or to 
#          call the Oracle libraries directly. Using the DynaLoader module
#          (or ExtUtils::DynaLib, PDL::CallExt, etc ...) isn't being used
#          
#
#    BUGS
#    * <if you know of any bugs, put them here!>
#
#    MODIFIED   (MM/DD/YY)
#    vsriram     12/01/05 - New error code for verification failure.
#    vsriram     08/15/05 - Bug-4541352: Flush stdout before reading user input.
#    phnguyen    01/21/05 - 
#    shgangul    01/03/05 - Remove no_verbose for system command 
#    shgangul    12/31/04 - Bug 4085436: Added new error for make failure 
#    shgangul    12/17/04 - No copyright message at the end 
#    shgangul    06/07/04 - verbose message also in debug mode 
#    shgangul    12/23/03 - bug 3334888: added a new error message 
#    shgangul    12/10/03 - bug 3257294: Added new error code 
#    shgangul    11/17/03 - bug 3252933: Return error for RAC 
#    shgangul    10/20/03 - alter print_message to write to any filehandle 
#                           passed to it 
#    shgangul    10/07/03 - rollback.sh fix, part of bug 3170021 
#    shgangul    10/06/03 - Add a new function to print in non-verbose mode 
#    phnguyen    09/29/03 - Merge bug-superset and file conflict
#    shgangul    09/26/03 - Add verbose option to Rollback
#    shgangul    09/11/03 - Logging changes for 10G 
#    phnguyen    06/13/03 - Add silent_mode
#    phnguyen    05/28/03 - Add debug()
#    phnguyen    04/28/03 - Use if (!defined) instead of bugged if (undef) 
#    daevans     03/07/03 - Fix exit value in print_and_die().
#    daevans     02/11/03 - Relocate directories.
#    daevans     06/01/02 - Many code changes. Stable check-in.
#    daevans     03/12/01 - Inital code.
#
##########################################################################

###############################################################################
###############################################################################
#
#  ------------------------ INITIALIZATION ------------------------
#
###############################################################################
###############################################################################

package opatchIO;
@ISA = ("Command");

######
#
# Standard modules:
#
use English;         # Let us say "$CHILD_ERROR" instead of "$?", etc.
use strict;          # Enforce strict variables, refs, subs, etc.

###############################################################################
#
#  ------------------------ DATA STRUCTURES ------------------------
#
###############################################################################
#
# Global data structures are defined in Command.pm.
#
###############################################################################


###############################################################################
#
#  ------------------------ SUBROUTINES ------------------------
#
###############################################################################

###############################################################################
#
# NAME   : check_to_stop_processing
#
# PURPOSE: Print a supplied message and checks the user's input from the
#          message.
#
# INPUTS : $$ARG[1]{message}  - The message to display.
#          $$ARG[1]{f_handle} - The filehandle to use. Defaults to STDOUT.
#
# OUTPUTS: $reply             - The response "N" for No or "" for Yes.
#
# NOTES  : 1. The file handle is either STDOUT, STDERR or an existing file
#             handle supplied by the caller.
#          2. Using 'printf' to suppress the new line on output of prompt.
#
###############################################################################
sub check_to_stop_processing {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $message  = $$rh_arguments{message}  || "";
    my $f_handle = $$rh_arguments{f_handle} || *STDOUT;

    $message .= "\nPlease respond Y|N > ";

    opatchIO -> print_message_noverbose ( { message => $message } );

    my $reply = "";
    if ( $Command::SILENT_MODE ) {
        $message = " Y (auto-answered by -silent)\n";
        opatchIO -> print_message_noverbose ( { message => $message } );
        $reply = "";
        sleep 3;
        return ( $reply ); 
    }

    # Flush the stdout first
    select(STDOUT);
    $|=1;

    $reply = uc <STDIN> || "";
    chomp $reply;

    # Turnoff autoflush
    $|=0;

    while (  $reply !~ m#^[NY]$# ) {

        opatchIO -> print_message_noverbose ( { message => $message } );

        # Flush the stdout first
        select(STDOUT);
        $|=1;
        
        $reply = uc <STDIN> || "";
        chomp $reply;
    
        # Turnoff autoflush
        $|=0;
        
    }

    if ( $reply =~ m#^[Y]$# ) {
        $reply = "";
    }
    
    return ( $reply );

}   # End of check_to_stop_processing().

###############################################################################
#
# NAME   : print_message_noverbose
#
# PURPOSE: Print a message irrespective of the verbose option.
#
# INPUTS : $$ARG[1]{message}  - The message to display.
#          $$ARG[1]{f_handle} - The filehandle to use. Defaults to STDOUT.
#
# OUTPUTS: write to stdout even if it's non-verbose
#          if log file exists, write to log file as well
#
# NOTES  : 1. The file handle is either STDOUT, STDERR or an existing file
#             handle supplied by the caller.
# 
#
###############################################################################
sub print_message_noverbose{

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $message  = $$rh_arguments{message};
    my $f_handle = $$rh_arguments{f_handle} || *STDOUT;

    if ( $message ) {
        # First display to STDOUT
        print ( "$message" . "\n" );

        # If log file is created write to the logfile
        if ( ($Command::logging_error_flag eq "") &&
            ($Command::logging_log_file_name ne "") )
        {
            $f_handle = $Command::logging_fh_log_file;
            print $f_handle ( "$message" . "\n" );
        }
    }
   
    return;

}   # End of print_message_noverbose().

###############################################################################
#
# NAME   : write_lock_file
#
# PURPOSE: Print a message to the LOCK file under .patch_storage
#
# INPUTS : $$ARG[1]{message}  - The message to display.
#          $$ARG[1]{f_handle} - The filehandle to use.
#
# OUTPUTS: NONE
#

###############################################################################
sub write_lock_file {
    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $message  = $$rh_arguments{message};
    my $f_handle = $$rh_arguments{lock_file};
    
    print $f_handle ( "$message" . "\n" );
}
###############################################################################
#
# NAME   : print_message
#
# PURPOSE: Print a message.
#
# INPUTS : $$ARG[1]{message}  - The message to display.
#          $$ARG[1]{f_handle} - The filehandle to use. Defaults to STDOUT.
#
# OUTPUTS: NONE
#
# NOTES  : 1. The file handle is either STDOUT, STDERR or an existing file
#             handle supplied by the caller.
#        write to stdout if it's verbose
#        write to log file if it exists
#
###############################################################################
sub print_message {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $message  = $$rh_arguments{message};
    my $f_handle = 0;
    if ( ($$rh_arguments{f_handle}) )
    {
        $f_handle = $$rh_arguments{f_handle};
    }

    if ( $message )
    {
        # Write to file handle if it is not the logging file handle.
        # We write to log file later anyways
        if ( ($f_handle ne 0) && 
             ($f_handle ne $Command::logging_fh_log_file) ) 
        {
            print $f_handle ( "$message" . "\n" );
        }
        else 
        {
            # Verbose mode is only in apply command
            if ( (($Command::opatch_operation ne "apply") && 
                  ($Command::opatch_operation ne "rollback") &&
                  ($Command::opatch_operation ne "lsinventory")) ||
                 ($Command::verbose_mode) || 
                 ($this_class->isDebug())
               )
            {
                # First display to STDOUT
                print ( "$message" . "\n" );
            }
    
            # If log file is created write to the logfile
            if ( ($Command::logging_error_flag eq "") &&
                 ($Command::logging_log_file_name ne "") )
            {
                $f_handle = $Command::logging_fh_log_file;
                print $f_handle ( "$message" . "\n" );
            }
        }
    }
   
    return;

}   # End of print_message().

###############################################################################
#
# NAME   : print_message_and_die
#
# PURPOSE: Print a message and exit.
#
# INPUTS : $$ARG[1]{message}  - The message to display.
#          $$ARG[1]{f_handle} - The filehandle to use. Defaults to STDERR.
#          $$ARG[1]{exit_val} - The return value to the send to the system.
#                            Defaults to 1.
#
# OUTPUTS: NONE
#
# NOTES  : 1. The file handle is either STDOUT, STDERR or an existing file
#             handle supplied by the caller.
#
###############################################################################
sub print_message_and_die {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $message  = $$rh_arguments{message}  || "Unknown problem.";
    my $f_handle = $$rh_arguments{f_handle} || *STDERR;
    my $exit_val = $$rh_arguments{exit_val};

    if ( ! defined $exit_val ) { $exit_val = 1; }
    if ( $exit_val == $this_class->ERROR_CODE_NO_ERROR ) {
       $message = "";
    }
    my $version = $this_class -> version;
    my $copyright = &Command::copyright;
    my $exit_msg  = $this_class -> get_error_text ({
                     error_code => $exit_val
                  });
    my $f_handle_err = *STDERR;
#    print $f_handle_err ( "$PROGRAM_NAME version: $version" . "\n" );
#    print $f_handle_err ( "$copyright" . "\n\n" );
    print $f_handle_err ( "$message" . "\n" );

    print $exit_msg;

    # If log file is created, write to logfile as well
    if ( ($Command::logging_error_flag eq "") &&
         ($Command::logging_log_file_name ne "") )
    {
        $f_handle = $Command::logging_fh_log_file;
#        print $f_handle ( "$PROGRAM_NAME version: $version" . "\n" );
#        print $f_handle ( "$copyright" . "\n\n" );
        print $f_handle ( "$message" . "\n" );

        print $f_handle ($exit_msg);
    }
    
    exit ( $exit_val );

}   # End of print_message_and_die().

##########################################################
#
# get error text
#    INPUT:  error code
#    OUTPUT: error message with prefix "ERROR: " 
##########################################################
sub get_error_text {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $error_code = $$rh_arguments{error_code};
    my $msg = "";

    # Error message will be "ERROR: ..." with exit code != 0 
    # to help automation / wrapper scripts
    if ( $error_code == $this_class->ERROR_CODE_NO_ERROR ) {
        $msg = "OPatch succeeded."; 
    } elsif ( $error_code == $this_class->ERROR_CODE_INVENTORY_PROBLEM ) {
        $msg =  "ERROR: OPatch failed because of Inventory problem.";
    } elsif ( $error_code == $this_class->ERROR_CODE_PATH_PROBLEM ) {
        $msg = "ERROR: OPatch failed because of Path problem.";
    } elsif ( $error_code == $this_class->ERROR_CODE_CLUSTER_PROBLEM ) {
        $msg = "ERROR: OPatch failed because of Cluster problem.";
    } elsif ( $error_code == $this_class->ERROR_CODE_COMMAND_LINE_ARGUMENT ) {
        $msg = "ERROR: OPatch failed because of cmd. args. problem.";
    } elsif ( $error_code == $this_class->ERROR_CODE_ERROR_ON_CONFLICT ) {
        $msg = "ERROR: OPatch failed because it detected conflicts."; 
    } elsif ( $error_code == $this_class->ERROR_CODE_PREREQ_PROBLEM ) {
        $msg = "ERROR: OPatch failed during pre-reqs check.";
    } elsif ( $error_code == $this_class->ERROR_CODE_ROLLBACK_PROBLEM ) {
        $msg = "ERROR: OPatch failed because of problems during roll back.";
    } elsif ( $error_code == $this_class->ERROR_CODE_PATCH_AREA ) {
        $msg = "ERROR: OPatch failed because of problems in patch area.";
    } elsif ( $error_code == $this_class->ERROR_CODE_PERL_PROBLEM ) {
        $msg = "ERROR: Perl is on the system but you need to run the command manually.\n" .
               "You probably need to specify \$PERL5LIB on the command line.\n" .
               " and run opatch as \"perl opatch.pl ...\"";
    } elsif ( $error_code == $this_class->ERROR_CODE_PRE_SCRIPT ) {
        $msg = "ERROR: OPatch failed because of problems with pre-patch script.\n";
    } elsif ( $error_code == $this_class->ERROR_RAC_INSTALL_PROBLEM ) {
        $msg = "ERROR: OPatch failed because of RAC problems.\n";
    } elsif ( $error_code == $this_class->ERROR_CODE_PATCH_PROBLEM ) {
        $msg = "ERROR: OPatch failed during patching, possibly due to " .
               "missing files.\n";
    } elsif ( $error_code == $this_class->ERROR_CODE_BUG_FILE_CONFLICT ) {
        $msg = "ERROR: OPatch failed due to bug/file conflict";
    } elsif ( $error_code == $this_class->ERROR_CODE_PATCHING_STOPPED_ON_USER_REQUEST ) {
        $msg = "ERROR: Patching process stopped on user's request";
    } elsif ( $error_code == $this_class->ERROR_CODE_MAKE_FAILURE ) {
        $msg = "ERROR: OPatch failed as make operation did not succeed";
    } elsif ( $error_code == $this_class->ERROR_CODE_VERIFY_FAILURE ) {
        $msg = "ERROR: OPatch failed as verification of the patch failed.";
    }

    return $msg;
}

##########################################################
#
# print out to stdout if $Command::DEBUG is set
#
##########################################################
sub debug {

    my $this_class   = shift @ARG;
    my $rh_arguments = $ARG[0];

    my $message  = $$rh_arguments{message};

    if ( $message && $this_class->isDebug() ) {
        print ( "$message" . "\n" );

        # If log file is created write to the logfile
        if ( ($Command::logging_error_flag eq "") &&
             ($Command::logging_log_file_name ne "") )
        {
            my $f_handle = $Command::logging_fh_log_file;
            print $f_handle ( "$message" . "\n" );
        }
    }

    return;

}   # End of print_message().

#############################################################################
# Perl packages need to return a true value.
1;
#############################################################################

