CREATE OR REPLACE PACKAGE BODY OLAP_API_SESSION_INIT AS
   /* Define local function, available only inside package. */
  FUNCTION getRoleID(roleName VARCHAR2) RETURN INT IS
    user_num INT;
  BEGIN
    SELECT USER# 
      INTO user_num
      FROM SYS.USER$
     WHERE TYPE# = 0 
       AND DEFROLE = 1
       AND NAME = roleName;
    RETURN user_num;
    EXCEPTION
      WHEN NO_DATA_FOUND THEN
        RAISE; 
  END getRoleId;

  FUNCTION newRole(roleNum INT, roleName VARCHAR2) return INT IS
    num INT;
  BEGIN
    SELECT COUNT(*)
      INTO num
      FROM SYS.OLAP$ALTER_SESSION
     WHERE ROLE# = roleNum
       AND TEXT = roleName;
     RETURN num;
  END newRole;

  /* Fully define subprograms specified in package. */
  PROCEDURE add_alter_session (roleName VARCHAR2, sqlStmt VARCHAR2) IS
    role_num INT;
    isNewRole INT;
    alterStmt VARCHAR2(4000);
  BEGIN
    DECLARE
    BEGIN
      role_num := getRoleID(roleName);
      alterStmt := 'ALTER SESSION ' || UPPER(sqlStmt);
      isNewRole := newRole(role_num, alterStmt);
      IF isNewRole = 0 THEN
        INSERT INTO SYS.OLAP$ALTER_SESSION VALUES (role_num, alterStmt);
        DBMS_OUTPUT.PUT_LINE('Row inserted');
        COMMIT;
      ELSE
        RAISE duplicate_role;
      END IF;  
    END;
    EXCEPTION
      WHEN NO_DATA_FOUND THEN
        RAISE_APPLICATION_ERROR(-20001, roleName || ' is not a valid role');
        RAISE invalid_role;
      WHEN duplicate_role THEN
        RAISE_APPLICATION_ERROR(-20002, 
          'ALTER SESSION command already exists for ' || roleName);
        RAISE duplicate_role;
      WHEN OTHERS THEN
        DBMS_OUTPUT.PUT_LINE('Unknown exception occured');
  END add_alter_session;

  PROCEDURE delete_alter_session (roleName VARCHAR2, sqlStmt VARCHAR2) IS
    role_num INT;
    alterStmt VARCHAR2(4000);
  BEGIN
    DECLARE
    BEGIN
      role_num := getRoleID(roleName);
      alterStmt := 'ALTER SESSION ' || UPPER(sqlStmt);
      DELETE FROM SYS.OLAP$ALTER_SESSION 
        WHERE ROLE# = role_num AND TEXT = alterStmt;
      COMMIT;
    END;
    EXCEPTION
      WHEN NO_DATA_FOUND THEN
        RAISE_APPLICATION_ERROR(-20001, roleName || ' is not a valid role');
        RAISE invalid_role;
      WHEN OTHERS THEN
         DBMS_OUTPUT.PUT_LINE('Unknown exception occured');
  END delete_alter_session;

  PROCEDURE clean_alter_session IS
  BEGIN
    DELETE FROM olap$alter_session where role# NOT IN (
      SELECT user# FROM SYS.USER$ WHERE TYPE# = 0 AND DEFROLE = 1);
  END clean_alter_session;

--  BEGIN  -- initialization part starts here
END OLAP_API_SESSION_INIT;
/
CREATE OR REPLACE PUBLIC SYNONYM OLAP_API_SESSION_INIT FOR SYS.OLAP_API_SESSION_INIT
/

