Rem
Rem   HISTORY
Rem     mmuppago    03/02/05 -  Backup and patch owa packages in DB 
Rem

set define on

set pagesize 65
set serverout on size 100000
set verify off
set echo off
set termout on
set feedback off
set trimspool on
set trimout on

whenever oserror exit 32767

variable owa_backup_file   varchar2(200);
variable owa_patch_file   varchar2(200);
variable owa_message     varchar2(1000);


Rem
Rem always initialize owa_backup_file and owa_patch_file to some dummy value.
Rem
begin :owa_backup_file := 'owadummy.sql'; end;
/
begin :owa_patch_file := 'owadummy.sql'; end;
/

DECLARE
  /*
   * This next line must be updated whenever 
   * OWA_UTIL.owa_version is updated.
   */
  shipped_30_owa_version    VARCHAR2(80) := '3.0.9.8.7';
  shipped_90_owa_version    VARCHAR2(80) := '9.0.4.0.6';
  shipped_101_owa_version    VARCHAR2(80) := '10.1.2.0.3';
  installed_owa_version  VARCHAR2(80);
  shipped_owa_version  VARCHAR2(80);
  dir_number           number := 30;
  new_line               VARCHAR2(4)  := '
';
  install_pkgs           BOOLEAN;

 --
 -- takes a string of the form 'num1.num2.num3.....'
 -- returns "num1" AND updates string to 'num2.num3...'
 --
 FUNCTION get_next_int_and_advance(str IN OUT varchar2)
      RETURN PLS_INTEGER is
  loc pls_integer;
  ans pls_integer;
 BEGIN
  loc := instr(str, '.', 1);
  if (loc > 0) then
   ans := to_number(substr(str, 1, loc - 1));
   str := substr(str, loc + 1, length(str) - loc);
  else
   ans := to_number(str);
   str := '';
  end if;
  return ans;
 END;

  --
  -- If shipped version of OWA packages is higher than the 
  -- pre-installed version of the OWA packages, then
  -- we need to reinstall the OWA packages.
  -- 
  FUNCTION needs_reinstall(shipped_owa_version   IN VARCHAR2,
                           installed_owa_version IN VARCHAR2) 
        RETURN BOOLEAN is

     shp_str VARCHAR2(80) := shipped_owa_version;
     shp_vsn PLS_INTEGER;
     ins_str VARCHAR2(80) := installed_owa_version;
     ins_vsn PLS_INTEGER;

  BEGIN
    --
    -- either OWA pkgs are not already installed (as can happen
    -- with a new DB) or an older version of the pkg is installed
    -- where version numbering was not implemented.
    --
    IF (installed_owa_version is NULL) THEN
      return TRUE;
    END IF;

    -- If version is the same, then we don't install it again to avoid 
    -- recompiling all dependent packages.
    --
    IF (installed_owa_version = shipped_owa_version) THEN
      return FALSE;
    END IF;

    --
    -- Check if shipped version is higher.
    --
    -- The OWA_UTIL version number format is V1.V2.V3.V4.V5.
    -- Lets compare versions by comparing Vi's from left to right.
    --
    FOR i in 1..5 LOOP 

     -- parse "shipped_version" one int at a time, from L to R
     shp_vsn := get_next_int_and_advance(shp_str);

     -- parse "installed_version" one int at a time, from L to R
     ins_vsn := get_next_int_and_advance(ins_str);
 
     IF (shp_vsn > ins_vsn) THEN
       return TRUE;
     END IF;

     IF (shp_vsn < ins_vsn) THEN
       return FALSE;
     END IF;

    END LOOP;

    -- 
    -- Should never come here. Return TRUE in this case as well.
    --
    RETURN TRUE;
  END;

  FUNCTION get_installed_owa_version RETURN VARCHAR2 IS
    owa_version VARCHAR2(80);
    l_cursor    INTEGER;
    l_stmt      VARCHAR2(256);
    l_status    INTEGER;
  BEGIN

    --
    -- Run this block via dynamic SQL and not static SQL
    -- because compilation of this block could fail as OWA_UTIL
    -- might be non-existant. Doing it from dynamic SQL allows
    -- us to catch the compile error as a run-time exception
    -- and proceed.
    --
    l_stmt := 'select OWA_UTIL.get_version from dual';
    l_cursor := dbms_sql.open_cursor;
    dbms_sql.parse(l_cursor, l_stmt, dbms_sql.native);
    dbms_sql.define_column( l_cursor, 1, owa_version, 80 );
    l_status := dbms_sql.execute(l_cursor);

    loop
       if dbms_sql.fetch_rows (l_cursor) > 0 then
          dbms_sql.column_value(l_cursor, 1, owa_version);
       else
          exit; 
       end if;
    end loop;
    dbms_sql.close_cursor(l_cursor);

    return owa_version;

  EXCEPTION
    --
    -- Either OWA pkgs have not been preinstalled
    -- Or, they are older set of OWA pkgs which
    -- a.) did not implement the OWA_UTIL.get_version method
    -- b.) resulted in ORA-6571 : ignore it
    -- 
    WHEN OTHERS THEN
     if dbms_sql.is_open(l_cursor) then
         dbms_sql.close_cursor(l_cursor);
     end if;
     return NULL;
  END;

 FUNCTION get_owa_dirnumber(p_version IN varchar2)
      RETURN NUMBER is
    ans            NUMBER;
   l_version      VARCHAR2(32);

 BEGIN

   -- Convert string to a number
   l_version := p_version;
   ans := 0;
   FOR i in 1..2 LOOP
     ans := 10 * ans + get_next_int_and_advance(l_version);
   END LOOP;

   RETURN ans;

 END;

BEGIN

 -- Get the version of OWA packages installed in the database
 installed_owa_version := get_installed_owa_version;

 -- Format a message for display
 IF (installed_owa_version is NULL) THEN

    :owa_message := 'No older OWA packages detected or OWA packages too old';
    -- In this case we default to owa_version: '3.0.0.0.0'
    -- so that we install latest of 3.0 version owa packages. 
    installed_owa_version := '3.0.0.0.0';
    shipped_owa_version := shipped_30_owa_version;
    :owa_patch_file := 'owa_patch.sql';
    :owa_backup_file := 'owa_backup.sql';

 ELSE

    :owa_message := 'Installed OWA version is: ' || installed_owa_version;

    dir_number := get_owa_dirnumber(installed_owa_version);
    if dir_number >= 101 then
        shipped_owa_version := shipped_101_owa_version;
    elsif dir_number = 90 then
        shipped_owa_version := shipped_90_owa_version;
    else
        shipped_owa_version := shipped_30_owa_version;
    end if;

    -- Check if we need to install the OWA packages?
    install_pkgs := needs_reinstall(shipped_owa_version, installed_owa_version);
   
    IF (install_pkgs) THEN
      :owa_patch_file := 'owa_patch.sql';
      :owa_backup_file := 'owa_backup.sql';
    ELSE
      :owa_patch_file := 'owadummy.sql';
      :owa_backup_file := 'owadummy.sql';
      :owa_message := :owa_message || new_line || 
      	   'You already have a newer version of the OWA packages' ||
	         new_line || 'No install is required';
    END IF;

 END IF;


END;
/

print :owa_message;

COLUMN :owa_patch_file NEW_VALUE owa_patch_var NOPRINT;
SELECT :owa_patch_file FROM DUAL;
COLUMN :owa_backup_file NEW_VALUE owa_backup_var NOPRINT;
SELECT :owa_backup_file FROM DUAL;

@@&owa_backup_var;
@@&owa_patch_var;

