--
-- $Header: onepclfy.pls 03-feb-2004.10:38:30 dbardwel Exp $
--
-- onepclfy.pls
--
-- Copyright (c) Oracle Corporation 2000. All Rights Reserved.
--
--   NAME
--     onepclfy.pls - cwm$classify
--
--   DESCRIPTION
--     Provides methods for classifying metadata, including categorizing
--     measures into catalogs and attaching descriptor values to metadata.
--
--   RETURNS
--
--   NOTES
--    
--
--   MODIFIED   (MM/DD/YY)
--   dbardwel    02/03/04 - Remove stray bad character hex 0x92
--   dthompso    06/22/00 - Add optional secondary_child_entity_name parameter to support level attribute descriptors.
--   dthompso    04/27/00 - Initial Version
--   dthompso    01/00/00 - Creation
--


create or replace package cwm$classify as
 
  -- Create a catalog node, returns the node id
  --
  -- param catalog_name          the name of the catalog
  -- param catalog_description   the description for the new catalog
  -- param parent_catalog_id     the parent for the new catalog, null indicates root
  --
  -- raise invalid_name             if catalog name is invalid
  -- raise catalog_already_exists   if catalog with this name already exists
  -- raise parent_catalog_not_found if catalog cannot be found for parent_catalog_id
  function create_catalog(catalog_name varchar2
                        , catalog_description varchar2
                        , parent_catalog_id number := null)
                          return number;


  -- Add a measure to a catalog node entry
  --
  -- param catalog_id            the catalog id to update
  -- param entity_owner          the owner(user) for the entity
  -- param entity_name           the name of the cube containing the measure
  -- param child_entity_name     the name of the measure within the cube
  --
  -- raise catalog_not_found      if catalog does not exist
  -- raise measure_not_found      if measure does not exist
  -- raise element_already_exists if the entity to add is already in the catalog
  procedure add_catalog_entity(catalog_id number
                             , entity_owner varchar2
                             , entity_name varchar2
                             , child_entity_name varchar2 := null);

  -- Add an entity to a descriptor
  --
  -- param descriptor_id           the descriptor id to update
  -- param entity_type             the type of the entity
  -- param entity_owner            the owner(user) for the entity
  -- param entity_name             the name of the entity containing the child
  -- param child_entity_name       the name of the child
  -- param secondary_child_entity_name  the optional grandchild name
  --
  -- raise descriptor_not_found     if descriptor does not exist
  -- raise element_already_exists   if the entity to add is already in the catalog
  -- raise entity_type_not_allowed  if entity type does not match descriptor types
  procedure add_entity_descriptor_use(descriptor_id number
                                    , entity_type varchar2
                                    , entity_owner varchar2
                                    , entity_name varchar2
				    , child_entity_name varchar2 := NULL
				    , secondary_child_entity_name VARCHAR2 := null);

  -- Set a catalog's parent identifier, cannot create cyclic recursive references
  --
  -- param catalog_id             the catalog id to update
  -- param parent_catalog_id      the identifier of the parent catalog to reference
  --
  -- raise catalog_not_found        if catalog does not exist
  -- raise parent_catalog_not_found if parent catalog does not exist
  procedure set_catalog_parent(catalog_id number
                             , parent_catalog_id number);

  -- Set a catalog's description
  --
  -- param catalog_id             the catalog id to update
  -- param catalog_description    the description which will be set for the catalog
  --
  -- raise catalog_not_found        if catalog does not exist
  procedure set_catalog_description(catalog_id number
                             , catalog_description varchar2);

  -- Drop a catalog, if catalog has children exception is raised, cascade forces drop.
  --
  -- param catalog_id             the catalog id to update
  -- param cascade                this indicates all children will be dropped to for catalog
  --
  -- raise catalog_not_found        if catalog does not exist
  -- raise catalog_has_sub_catalogs if catalog has sub-catalogs and cascade (TRUE) is not specified
  procedure drop_catalog(catalog_id number
                             , cascade boolean := false);

  -- Remove a measure from a catalog node entry
  --
  -- param catalog_id             the catalog id to update
  -- param cascade                this indicates all children will be dropped to for catalog
  --
  -- raise catalog_not_found        if catalog does not exist
  -- raise measure_not_found        if measure does not exist, or is not accessible to the user
  -- raise element_does_not_exist   if the entity passed does not exist in the catalog
  procedure remove_catalog_entity(catalog_id number
                             , entity_owner varchar2
                             , entity_name varchar2
                             , child_entity_name varchar2);


  -- Lock a catalog node for update
  --
  -- param catalog_id            id of the catalog node to lock
  -- param wait_for_lock         wait for lock if acquired by other user
  --
  -- raise no_access_privileges  if no privileges to edit the catalog
  -- raise catalog_not_found     if attribute doesn't exist
  -- raise failed_to_gain_lock   if lock could not be acquired
  procedure lock_catalog(catalog_id number, wait_for_lock boolean := false);
  
  -- Create a descriptor type, name must be unique
  --
  -- param descriptor_type        a string representing the new descriptor type
  --
  -- raise descriptor_type_already_exists if the type already exists
  -- raise no_access_privileges           need to be an OLAP_DBA to execute this
  procedure create_descriptor_type(descriptor_type varchar2);


  -- Drop a descriptor type
  --
  -- param  descriptor_type           the value for the type of descriptor
  --
  -- raise descriptor_type_not_found  if the type does not exist
  -- raise descriptor_type_in_use     if the type is in use
  -- raise no_access_privileges       need to be an OLAP_DBA to execute this
  procedure drop_descriptor_type(descriptor_type varchar2);


  -- Add an entity type to the descriptor type
  --
  -- param  descriptor_type           the value for the type of descriptor
  -- param  entity_type               the value for the type of entity
  --
  -- raise descriptor_type_not_found  if the type does not exist
  -- raise entity_type_not_allowed    if the type is invalid
  procedure add_descriptor_entity_type(descriptor_type varchar2
                                      ,entity_type varchar2);


  -- Remove an entity type from a the descriptor type
  --
  -- param  descriptor_type           the value for the type of descriptor
  -- param  entity_type               the value for the type of entity
  --
  -- raise descriptor_type_not_found  if the type does not exist
  -- raise entity_type_not_allowed    if the type is invalid
  procedure remove_descriptor_entity_type(descriptor_type varchar2
                                      ,entity_type varchar2);

  -- Function to create a new descriptor based on a descriptor type, id returned
  --
  -- param  descriptor_type           the value for the type of descriptor
  -- param  descriptor_value          the value for the descriptor
  -- param  description               the description for the new descriptor
  --
  -- raise descriptor_type_not_found    if the type does not exist
  -- raise descriptor_with_value_exists if the descriptor value for this type exists
  -- raise no_access_privileges         need to be an OLAP_DBA to execute this
  function create_descriptor(descriptor_type varchar2
                            ,descriptor_value varchar2
                            ,description varchar2) return number;


  -- Procedure to drop a descriptor given its id
  --
  -- param  descriptor_id             the id of the descriptor created in create_descriptor
  --
  -- raise descriptor_undefined       if the descriptor does not exist
  -- raise no_access_privileges       need to be an OLAP_DBA to execute this
  procedure drop_descriptor(descriptor_id number);


  -- Remove an entity from a descriptor
  --
  -- param descriptor_id         the descriptor id to update
  -- param entity_type           the type of the entity
  -- param entity_owner          the owner(user) for the entity
  -- param entity_name           the name of the entity containing the child
  -- param child_entity_name     the name of the child
  -- param secondary_child_entity_name the optional grandchild name
  --
  -- raise descriptor_not_found      if descriptor does not exist
  -- raise element_already_exists if the entity to add is already in the catalog
  procedure remove_entity_descriptor_use(descriptor_id number
                                     ,entity_type varchar2
                                     ,entity_owner varchar2
                                     ,entity_name varchar2
				     ,child_entity_name varchar2 := NULL
				     , secondary_child_entity_name VARCHAR2 := null);

end;
/
