#
# $Header: frmlnupd.tcl@@/main/TOOLS_DEV2K_994/3 \
# Checked in on Thu Jan 13 11:55:52 PST 2000 by vobadm \
# Copyright (c) 2000, 2003, Oracle Corporation.  All rights reserved.  
# $
#
#
# Copyright (c) 1998, 2003, Oracle Corporation.  All rights reserved.  
#
# NAME:
#
#  frmlnupd.tcl -- Forms up/down event script
#                  detects up/down event for Listener, Metrics server/client
#                  returns: WARNING if some instaces are down
#                           ALERT if all instances are down
#
# ARGUMENTS:
#
#  argv(0) - connect_string
#
# RETURNS:
#
#  OEM Event types
#
# MODIFIED     (MM/DD/YY)
#  mkiran       07/25/03 - Don't use oracommit after orasql that executes a DML statement
#  mkiran       07/25/03 - Consider "portno" column instead of "pid" column 
#                          of smp_forms_instances table to determine
#                          "active" forms listeners and "dead" ones
#  clowes       11/07/01 - #2019562: oraautocom is not being recognised as a valid
#                           tcl command in this script. We should investigate why 
#                           this is the case. However, we can implement a fix by 
#                           replacing this command with an oracommit after the 
#                           only DML statement in the script.
#  jbellemo     01/11/01 - #1554211: take full connect string
#  ayu          11/04/99 -  implemented a new way of reporting up/down events
#  ayu          03/06/99 -  took out platform dependent code
#  ayu          10/06/98 -  Creation
#

oradefine event /oracle/forms/fault/frmlnupd
oraarguments connect_params
oravardesc connect_params string
oradefine end

#
# Note: the connect string arguments are: userid pwd protocol host port sid
# We have to do this to get around the limit of 80 char in the $argv string
# do the following once the 80 char limit is fixed:
set connect_str [lindex $argv 0]
#
# set cp [split [lindex $argv 0] ","]
# set connect_str "[lindex $cp 0]/[lindex $cp 1]@(description=(address=(protocol=[lindex $cp 2])(host=[lindex $cp 3])(port=[lindex $cp 4]))(connect_data=(sid=[lindex $cp 5])))"

source [AGENT_ORACLE_HOME]/[NETWORK]/agent/tcl/frminit.tcl
frmdebug "frmlnupd.tcl start --> [oratime]"
frmdebug $connect_str
set last_report $NO_EVENT
set last_output ""

set instance_type 0
set exename "^$_lnrexename|^$_lnr45exename"
set nodename $oramsg(nodename)
#
# remove the domain name from nodename
#
if {[string first . $nodename] != -1 } {
    set nodename [string range $nodename 0 [expr [string first . $nodename]-1]]
}

#
# The main event checking functions
#
proc EvalEvent {} {

    global connect_str instance_type exename last_output nodename
    global output last_report
    global oramsg
    global SCRIPT_FAIL CLEAR_EVENT NO_EVENT WARNING_EVENT ALERT_EVENT

    set ret_code $NO_EVENT
    set output ""
    set downtarget {}

    if {[catch {oralogon $connect_str} lda]} {
        lappend output "Logon failed : MESSAGE - $oramsg(rc)"
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }

    # (2019562)
    # Commenting out this statement to overcome bug 2019562 as the 
    # the oraautocom is not being recognised and is causing this script 
    # to fail.
    #  oraautocom $lda on

    set cur1 [oraopen $lda]

    set db_plist {}
    set db_nlist {}

    # db_portnumlist holds the port number of each forms listener process
    set db_portnumlist {}

    #
    # get list of instance pids on the local node to monitor 
    #
    set select_sql "select pid, name, portno from smp_forms_instances where type = $instance_type and node = '$nodename'"
    orasql $cur1 $select_sql
    set result [orafetch $cur1]
    while {$oramsg(rc) == 0} {
        lappend db_plist [lindex $result 0]
        lappend db_nlist [lindex $result 1]
        lappend db_portnumlist [lindex $result 2]
        set result [orafetch $cur1]
    }

    # db_distinct_portnumlist = "db_portnumlist with duplicate port numbers removed"
    # Example: If db_portnumlist = "1234 1234 1111 2222 1111 3333 4444",
    #             db_distinct_portnumlist = "1234 1111 2222 3333 4444"
    set db_distinct_portnumlist {}
    if {[llength $db_portnumlist] > 0} {
        foreach el $db_portnumlist {

            # If an element of db_portnumlist is not present in db_distinct_portnumlist,
            # add that element to db_distinct_portnumlist
            if {[lsearch -exact $db_distinct_portnumlist $el] == -1} {
                lappend db_distinct_portnumlist $el
            }
        } 
    }

    # current_plist = "list of all OS pid's of forms listeners, running on the machine currently"
    set current_plist [frm_getpidlist $exename]

    # current_portnumlist = "list of all port numbers at which forms listeners are currently 
    #                        running on the machine"
    set current_portnumlist {}
    if {[llength $current_plist] > 0} {
        foreach el $current_plist {

            # Get port_number of forms listener currently running at OS pid $el,
            # by calling frm_getportnumber procedure, present in frminit.tcl
            lappend current_portnumlist [frm_getportnumber f60srvm $el]    
        }
    } else {
        # No forms listeners are currently running on the machine.
        # That means all forms listeners in list db_nlist are down.
        # So, return an Alert message to OEM console that all listeners are down.
        set downtarget $db_nlist
        set ret_code $ALERT_EVENT
    }

    # If a port number is present only in list db_distinct_portnumlist and 
    # not in list current_portnumlist, that means all forms listeners in list db_nlist, 
    # which were running before at that port number are DOWN now.
    # Collect all such DEAD port numbers in list db_dead_portnumlist
    set db_dead_portnumlist {}
    if {[llength $db_distinct_portnumlist] > 0 && [llength $current_portnumlist] > 0} {
        foreach el $db_distinct_portnumlist {
            if {[lsearch -exact $current_portnumlist $el] == -1} {
                lappend db_dead_portnumlist $el
            }
        }
    }

    # From list db_plist, get OS pid's of all those forms listeners, whose port numbers 
    # are present in list db_dead_portnumlist, meaning dead forms listeners.
    if {[llength $db_portnumlist] > 0 && [llength $db_dead_portnumlist] > 0} {
        set i 0
        foreach el $db_portnumlist {
            if {[lsearch -exact $db_dead_portnumlist $el] != -1} {
                # Port number $el is present in list db_dead_portnumlist.
                # That means forms listener running at that port number is down now.
                # So, append it's name (from list db_nlist) to list of down targets
                lappend downtarget [lindex $db_nlist $i]

                # User at OEM console will receive atleast warning message if not an alert. 
                set ret_code $WARNING_EVENT

                # dead_pid is the DEAD forms listener's OS pid.
                set dead_pid [lindex $db_plist $i]
                if {$dead_pid > 0} {
                    # dead_pid value is positive and represents an OS pid. 
                    # '-2' value for a "pid" column in smp_forms_instances table
                    # represents a DEAD forms listener
                    # So, make this forms listener's pid value '-2'.
                    set update_sql "update smp_forms_instances set pid = -2 where type = $instance_type and node = '$nodename' and pid = $dead_pid"
                    orasql $cur1 $update_sql

                }
            }
            incr i
        }
    }

        if {$ret_code != $NO_EVENT} {
            if {[llength $db_plist] == [llength $downtarget]} {
                # all forms listeners in list db_nlist are DOWN
                set ret_code $ALERT_EVENT
            }
            set output "$downtarget on node \'$nodename\' went DOWN."
        } elseif {[llength $downtarget] == 0 && $last_output != ""} {
            # all forms listeners in list db_nlist are UP and currently running.
            set ret_code $CLEAR_EVENT
            set output "Event Cleared - Instance(s) came back UP."
        }

    if {[string compare $last_output $output] != 0} {
        # previous ouput and current output are not the same.
        # so, set last_report to NO_EVENT, so that
        # an Alert or Warning will be triggered this time.
        set last_output $output
        set last_report $NO_EVENT
    }

    oraclose $cur1
    oralogoff $lda

    frmdebug "frmlnupd : [oratime] : $ret_code : $output : $downtarget"
    ORATCL_DEBUG "frmlnupd : [oratime] : $ret_code : $output"

    if {$last_report == $ret_code} {
        return $NO_EVENT
    } else {
        set last_report $ret_code
        return $ret_code 
    }
}
