Rem
Rem $Header: catcr.sql 13-apr-2006.14:59:54 rburns Exp $
Rem
Rem catcr.sql
Rem
Rem Copyright (c) 2001, 2006, Oracle. All rights reserved.  
Rem
Rem    NAME
Rem      catcr.sql - CATalog Component Registry
Rem
Rem    DESCRIPTION
Rem      This script creates the data dictionary elements and package for
Rem      the registry of components that have been loaded into the database.
Rem
Rem    NOTES
Rem      Use SQLPLUS
Rem      Conned AS SYSDBA
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    rburns      04/13/06 - Backport rburns_bug-5095748 from 
Rem                           st_rdbms_9.2.0.1 
Rem    rburns      04/05/06 - fix time_stamp 
Rem    rburns      11/18/05 - Backport rburns_bug-4200878 from main 
Rem    yuli        08/18/05 - raise version number to 9.2.0.8 
Rem    yuli        02/23/05 - raise version number to 9.2.0.7 
Rem    rburns      01/06/05 - only SYS validation procedures 
Rem    rburns      10/13/04 - no validation for OPTION OFF 
Rem    rburns      10/01/04 - move procedure 
Rem    yuli        04/27/04 - raise version number to 9.2.0.6 
Rem    rburns      03/25/04 - bug 3430486: special patch script 
Rem    rburns      10/03/03 - check options, add status value 
Rem    yuli        09/15/03 - raise to 9.2.0.5 
Rem    yuli        06/24/03 - raise to 9.2.0.4
Rem    rkazak      01/20/03 - convert for 9.2.0.3
Rem    rburns      05/14/02 - convert for 9.2.0.2
Rem    rburns      04/10/02 - always use full path for nothing.sql
Rem    rburns      04/09/02 - No script if removed or removing
Rem    rburns      03/08/02 - fix Intermedia populate
Rem    rburns      02/14/02 - change AMD name and fix ORDIM and SDO names
Rem    rburns      02/11/02 - add registry version
Rem    rburns      02/06/02 - add MGW component
Rem    rburns      02/04/02 - fix ODM populate
Rem    rburns      01/09/02 - fix intermedia populate and permission check
Rem    rburns      12/15/01 - add catjava
Rem    rburns      12/12/01 - MDSYS for Spatial
Rem    rburns      12/10/01 - fix validate procedure
Rem    rburns      12/06/01 - add other components
Rem    rburns      10/26/01 - add registry validation procedure
Rem    smavris     11/08/01 - Update interMedia registry
Rem    rburns      10/23/01 - fix views, add drop_user
Rem    rburns      10/15/01 - add owm, ols, and new registry columns
Rem    rburns      10/02/01 - add JAVAVM and new interfaces
Rem    rburns      09/20/01 - add flags column for restart, new interfaces
Rem    rburns      08/30/01 - Merged rburns_component_registry
Rem    rburns      08/15/01 - Created
Rem
Rem -----------------------------------------------------------------
Rem REGISTRY$ table
Rem -----------------------------------------------------------------

CREATE TABLE registry$ (
             cid      VARCHAR2(30),                 /* component identifier */
             cname    VARCHAR2(255),                      /* component name */
             schema#  NUMBER NOT NULL,             /* user# of schema owner */
             invoker# NUMBER NOT NULL,           /* user# of script invoker */
             version  VARCHAR2(30),             /* component version loaded */
             status   NUMBER NOT NULL,                  /* component status */
                                                         /* 0 = INVALID     */
                                                         /* 1 = VALID       */
                                                         /* 2 = LOADING     */
                                                         /* 3 = LOADED      */
                                                         /* 4 = UPGRADING   */
                                                         /* 5 = UPGRADED    */
                                                         /* 6 = DOWNGRADING */
                                                         /* 7 = DOWNGRADED  */ 
                                                         /* 8 = REMOVING    */
                                                         /* 9 = OPTION OFF  */
                                                         /* 99 = REMOVED    */
             flags    NUMBER NOT NULL,        /* additional component flags */
                                                   /* 0x01 STARTUP REQUIRED */
             modified DATE,                       /* last modified datetime */
             pid      VARCHAR2(30),          /* parent component identifier */
             banner   VARCHAR2(80),             /* component display banner */
             vproc    VARCHAR2(61),                  /* validation procedure */
             date_invalid       DATE,              /* last INVALID datetime */
             date_valid         DATE,                /* last VALID datetime */
             date_loading       DATE,              /* last LOADING datetime */
             date_loaded        DATE,               /* last LOADED datetime */
             date_upgrading     DATE,            /* last UPGRADING datetime */
             date_upgraded      DATE,             /* last UPGRADED datetime */
             date_downgrading   DATE,          /* last DOWNGRADING datetime */
             date_downgraded    DATE,           /* last DOWNGRADED datetime */
             date_removing      DATE,             /* last REMOVING datetime */
             date_removed       DATE,              /* last REMOVED datetime */
             CONSTRAINT registry_pk  PRIMARY KEY (cid),
             CONSTRAINT registry_parent_fk FOREIGN KEY (pid)
                        REFERENCES registry$ (cid) ON DELETE CASCADE);

Rem -------------------------------------------------------------------------
Rem DBA_REGISTRY view
Rem -------------------------------------------------------------------------

CREATE OR REPLACE VIEW dba_registry (
            comp_id, comp_name, version, status,
            modified, control, schema, procedure,
            startup, parent_id)
AS
SELECT r.cid, r.cname, r.version,
       DECODE(r.status, 0, 'INVALID',
                        1, 'VALID',
                        2, 'LOADING',
                        3, 'LOADED',
                        4, 'UPGRADING',
                        5, 'UPGRADED',
                        6, 'DOWNGRADING',
                        7, 'DOWNGRADED',
                        8, 'REMOVING',
                        9, 'OPTION OFF',
                        99, 'REMOVED',
                        NULL),
       TO_CHAR(r.modified,'DD-MON-YYYY HH24:MI:SS'), 
       i.name, s.name, r.vproc,
       DECODE(bitand(r.flags,1),1,'REQUIRED',NULL), r.pid
FROM registry$ r, user$ s, user$ i
WHERE r.schema# = s.user# AND r.invoker#=i.user#;
         
CREATE OR REPLACE PUBLIC SYNONYM dba_registry FOR dba_registry;
GRANT SELECT ON dba_registry TO SELECT_CATALOG_ROLE;

Rem -------------------------------------------------------------------------
Rem USER_REGISTRY view
Rem -------------------------------------------------------------------------

CREATE OR REPLACE VIEW user_registry (
            comp_id, comp_name, version, status,
            modified, control, schema, procedure,
            startup, parent_id)
AS
SELECT r.cid, r.cname, r.version,
       DECODE(r.status, 0, 'INVALID',
                        1, 'VALID',
                        2, 'LOADING',
                        3, 'LOADED',
                        4, 'UPGRADING',
                        5, 'UPGRADED',
                        6, 'DOWNGRADING',
                        7, 'DOWNGRADED',
                        8, 'REMOVING',
                        9, 'OPTION OFF',
                        99, 'REMOVED',
                        NULL),
       TO_CHAR(r.modified,'DD-MON-YYYY HH24:MI:SS'), 
       i.name, s.name, r.vproc,
       DECODE(bitand(r.flags,1),1,'REQUIRED',NULL), r.pid
FROM registry$ r, user$ s, user$ i
WHERE (r.schema# = USERENV('SCHEMAID') OR r.invoker# = USERENV('SCHEMAID'))
      AND r.schema# = s.user# AND r.invoker#=i.user#;

CREATE OR REPLACE PUBLIC SYNONYM user_registry FOR user_registry;
GRANT SELECT ON user_registry TO PUBLIC;

Rem -------------------------------------------------------------------------
Rem DBA_REGISTRY_HIERARCHY view
Rem -------------------------------------------------------------------------

CREATE OR REPLACE VIEW dba_registry_hierarchy (
            comp_id, version, status, modified)
AS
SELECT LPAD(' ',2*(LEVEL-1)) || LEVEL || ' ' || cid, version,
       DECODE(status, 0, 'INVALID',
                      1, 'VALID',
                      2, 'LOADING',
                      3, 'LOADED',
                      4, 'UPGRADING',
                      5, 'UPGRADED',
                      6, 'DOWNGRADING',
                      7, 'DOWNGRADED',
                      8, 'REMOVING',
                      9, 'OPTION OFF',
                      99, 'REMOVED',
                      NULL),
       TO_CHAR(modified,'DD-MON-YYYY HH24:MI:SS')
FROM registry$ 
START WITH pid IS NULL
CONNECT BY PRIOR cid = pid;

CREATE OR REPLACE PUBLIC SYNONYM dba_registry_hierarchy 
                  FOR dba_registry_hierarchy;
GRANT SELECT ON dba_registry_hierarchy TO SELECT_CATALOG_ROLE;

Rem -------------------------------------------------------------------------
Rem ALL_REGISTRY_BANNERS view
Rem    Public view of valid components in the database
Rem -------------------------------------------------------------------------

CREATE OR REPLACE VIEW all_registry_banners 
AS
SELECT banner FROM registry$
WHERE status = 1; 

CREATE OR REPLACE PUBLIC SYNONYM all_registry_banners
                  FOR all_registry_banners;
GRANT SELECT ON all_registry_banners TO PUBLIC;

Rem
Rem ------------------------------------------------------------------------
Rem REGISTRY$HISTORY table
Rem
Rem Contains a time-stamped entry for each upgrade/downgrade performed 
Rem and for each CPU applied to the database.
Rem ------------------------------------------------------------------------

CREATE TABLE registry$history (
             action_time     DATE,                          /* time stamp */
             action          VARCHAR2(30),              /* name of action */
             namespace       VARCHAR2(30),           /* upgrade namespace */
             version         VARCHAR(30),               /* server version */
             id              NUMBER,                   /* CPU or Patch ID */
             comments        VARCHAR2(255)                    /* comments */
             );
Rem -------------------------------------------------------------------------
Rem CREATE history VIEW
Rem -------------------------------------------------------------------------

CREATE OR REPLACE VIEW dba_registry_history (
            action_time, action, namespace, version, id, comments)
AS
SELECT action_time, action, namespace, version, id, comments
FROM registry$history;

CREATE OR REPLACE PUBLIC SYNONYM dba_registry_history FOR dba_registry_history;
GRANT SELECT ON dba_registry_history TO SELECT_CATALOG_ROLE;

Rem -------------------------------------------------------------------------
Rem DBMS REGISTRY PACKAGE
Rem -------------------------------------------------------------------------

CREATE OR REPLACE PACKAGE dbms_registry AS

release_version       CONSTANT registry$.version%type := '9.2.0.8.0';
release_status        CONSTANT VARCHAR2(30) := 'Production';
nothing_script        CONSTANT VARCHAR2(35) := '?/rdbms/admin/nothing.sql';

PROCEDURE invalid     (comp_id      IN VARCHAR2);

PROCEDURE valid       (comp_id      IN VARCHAR2);

PROCEDURE loading     (comp_id      IN VARCHAR2,
                       comp_name    IN VARCHAR2,
                       comp_proc    IN VARCHAR2 DEFAULT NULL,
                       comp_schema  IN VARCHAR2 DEFAULT NULL,
                       comp_parent  IN VARCHAR2 DEFAULT NULL);

PROCEDURE loaded      (comp_id      IN VARCHAR2,
                       comp_version IN VARCHAR2 DEFAULT NULL,
                       comp_banner  IN VARCHAR2 DEFAULT NULL);

PROCEDURE upgrading   (comp_id      IN VARCHAR2,
                       new_name     IN VARCHAR2 DEFAULT NULL,
                       new_proc     IN VARCHAR2 DEFAULT NULL,
                       new_schema   IN VARCHAR2 DEFAULT NULL,
                       new_parent   IN VARCHAR2 DEFAULT NULL);

PROCEDURE upgraded     (comp_id      IN VARCHAR2,
                       new_version   IN VARCHAR2 DEFAULT NULL,
                       new_banner    IN VARCHAR2 DEFAULT NULL);

PROCEDURE downgrading (comp_id      IN VARCHAR2,
                       old_name     IN VARCHAR2 DEFAULT NULL,
                       old_proc     IN VARCHAR2 DEFAULT NULL,
                       old_schema   IN VARCHAR2 DEFAULT NULL,
                       old_parent   IN VARCHAR2 DEFAULT NULL);

PROCEDURE downgraded  (comp_id      IN VARCHAR2,
                       old_version  IN VARCHAR2);

PROCEDURE removing    (comp_id      IN VARCHAR2);

PROCEDURE removed     (comp_id      IN VARCHAR2);

PROCEDURE startup_required (comp_id IN VARCHAR2);

PROCEDURE startup_complete (comp_id IN VARCHAR2);

FUNCTION  status      (comp_id IN VARCHAR2) RETURN VARCHAR2;

FUNCTION  version     (comp_id IN VARCHAR2) RETURN VARCHAR2;

FUNCTION  comp_name   (comp_id IN VARCHAR2) RETURN VARCHAR2;

FUNCTION  script      (comp_id IN VARCHAR2, 
                       script_name IN VARCHAR2) RETURN VARCHAR2;

FUNCTION  patch_script (comp_id IN VARCHAR2, 
                        script_name IN VARCHAR2) RETURN VARCHAR2;

FUNCTION  cpu_script (comp_id IN VARCHAR2,
                      script_name IN VARCHAR2) RETURN VARCHAR2;


FUNCTION  time_stamp  (comp_id IN VARCHAR2) RETURN VARCHAR2; 

FUNCTION  is_loaded   (comp_id IN VARCHAR2, 
                       version IN VARCHAR2 DEFAULT NULL) RETURN NUMBER;

FUNCTION  is_valid   (comp_id IN VARCHAR2, 
                       version IN VARCHAR2 DEFAULT NULL) RETURN NUMBER;

FUNCTION  is_startup_required (comp_id IN VARCHAR2) RETURN NUMBER;

END dbms_registry;
/

CREATE OR REPLACE PACKAGE BODY dbms_registry 
AS

-- STATUS
 
  s_invalid     NUMBER :=0;
  s_valid       NUMBER :=1;
  s_loading     NUMBER :=2;
  s_loaded      NUMBER :=3;
  s_upgrading   NUMBER :=4;
  s_upgraded    NUMBER :=5;
  s_downgrading NUMBER :=6;
  s_downgraded  NUMBER :=7;
  s_removing    NUMBER :=8; 
  s_option_off  NUMBER :=9; 
  s_removed     NUMBER :=99; 

-- FLAGS

  f_startup_req NUMBER :=1;  /* 0x01 */

-- EXCEPTIONS

not_invoker     EXCEPTION;
PRAGMA          EXCEPTION_INIT(not_invoker, -01031);

-- GLOBAL

g_null         CHAR(1);

----------------------------------------------------------------------
-- PRIVATE FUNCTIONS
----------------------------------------------------------------------

------------------------- exists_comp --------------------------------

FUNCTION exists_comp (id IN VARCHAR2) RETURN BOOLEAN
IS
BEGIN
  SELECT NULL INTO g_null FROM sys.registry$
  WHERE cid = id;
  RETURN TRUE;
EXCEPTION
  WHEN NO_DATA_FOUND THEN
    RETURN FALSE;
END exists_comp;

------------------------- get_user --------------------------------

FUNCTION get_user RETURN NUMBER
IS

p_user# NUMBER;

BEGIN
  SELECT user# INTO p_user# FROM sys.user$
  WHERE name = SYS_CONTEXT ('USERENV', 'SESSION_USER');
  RETURN p_user#;
END get_user;

--------------------------------------------------------------------

FUNCTION get_user(usr IN VARCHAR2) RETURN NUMBER
IS

p_user# NUMBER;

BEGIN
  SELECT user# INTO p_user# FROM sys.user$
  WHERE name = usr;
  RETURN p_user#;
END get_user;

------------------------- check_invoker --------------------------------

PROCEDURE check_invoker (id IN VARCHAR2, usr# IN NUMBER)
IS 
BEGIN
  SELECT NULL into g_null from sys.registry$
  WHERE id = cid AND (usr# = invoker# OR usr# = schema#);
EXCEPTION
  WHEN NO_DATA_FOUND THEN
     RAISE not_invoker;
END check_invoker;

------------------------- new_comp --------------------------------

PROCEDURE new_comp (st     IN VARCHAR2,
                    id     IN VARCHAR2, 
                    nme    IN VARCHAR2, 
                    prc    IN VARCHAR2,
                    par    IN VARCHAR2,
                    inv#   IN NUMBER,
                    sch#   IN NUMBER)
IS 
BEGIN

  INSERT INTO sys.registry$ (modified, status, cid, cname, 
                             pid, vproc, invoker#, schema#,  flags)
         VALUES (SYSDATE, st, id, nme, par, prc, inv#, sch#, 0);

END new_comp;


------------------------- update_comp --------------------------------

PROCEDURE update_comp (st     IN VARCHAR2,
                       id     IN VARCHAR2, 
                       nme    IN VARCHAR2, 
                       prc    IN VARCHAR2,
                       par    IN VARCHAR2,
                       ver    IN VARCHAR2,
                       ban    IN VARCHAR2)

IS 
BEGIN

  UPDATE sys.registry$ SET status = st, modified = SYSDATE WHERE id = cid; 

  IF nme IS NOT NULL THEN
     UPDATE sys.registry$ SET cname = nme WHERE id = cid;
  END IF;
  IF par IS NOT NULL THEN
     UPDATE sys.registry$ SET pid = par WHERE id = cid;
  END IF;
  IF prc IS NOT NULL THEN
     UPDATE sys.registry$ SET vproc = prc WHERE id = cid;
  END IF;
  IF ver IS NOT NULL THEN
     UPDATE sys.registry$ SET version = ver WHERE id = cid;
  END IF;
  IF ban IS NOT NULL THEN
     UPDATE sys.registry$ SET banner = ban WHERE id = cid;
  END IF;

END update_comp;

------------------------- set_flag -----------------------------------

FUNCTION set_flag(flag NUMBER, bit NUMBER) RETURN NUMBER 
IS

BEGIN
   IF bitand(flag, bit) = bit THEN
      RETURN(flag);
   ELSE
      RETURN(flag + bit);
   END IF;
END set_flag;

------------------------- clear_flag ---------------------------------

FUNCTION clear_flag(flag NUMBER, bit NUMBER) RETURN NUMBER 
IS

BEGIN
   IF bitand(flag, bit) = bit THEN
      RETURN(flag - bit);
   ELSE
      RETURN(flag);
   END IF;
END clear_flag;

------------------------- option_OK ---------------------------------

FUNCTION option_OK (option_name VARCHAR2) RETURN BOOLEAN
IS

option_value v$option.value%type;
OK           BOOLEAN := FALSE;

BEGIN
   -- Get option name 

   IF option_name IS NULL THEN
      OK := TRUE;   -- No option required
   ELSE
      SELECT value INTO option_value FROM v$option
      WHERE parameter = option_name;
      IF option_value = 'TRUE' THEN
         OK := TRUE;
      END IF;
   END IF;      

   RETURN OK; 

END option_OK;


------------------------- disable DDL triggers -------------------------

PROCEDURE disable_ddl_triggers (schema varchar2)
IS
PRAGMA AUTONOMOUS_TRANSACTION;  -- executes DDL 

p_schema       VARCHAR2(30) := NLS_UPPER(schema);
p_trigger_name VARCHAR2(30);
TYPE CurTyp    IS REF CURSOR;  
trigger_cv     CurTyp;  

BEGIN

  IF p_schema IN ('SYS','SYSTEM') THEN
     RETURN;  -- don't disable any triggers
  END IF;
  -- Use dynamic SQL to avoid errors during DB creation
  OPEN trigger_cv FOR
         'SELECT trigger_name FROM dba_triggers
          WHERE owner = :p_schema AND
                status = ''ENABLED'' AND
                trigger_type IN (''BEFORE EVENT'', ''AFTER EVENT'')'
        USING p_schema;

  LOOP
      FETCH trigger_cv INTO p_trigger_name;  
      EXIT WHEN trigger_cv%NOTFOUND; 
      EXECUTE IMMEDIATE 'ALTER TRIGGER ' || 
                         p_schema || '.' || p_trigger_name ||
                         ' DISABLE';
  END LOOP;

END disable_ddl_triggers;

--------------------------- comp_option_off -----------------------------
-- for use in dbms_registry.script

PROCEDURE comp_option_off (comp_id IN VARCHAR2)
IS
PRAGMA AUTONOMOUS_TRANSACTION;  -- executes DML from query

p_id registry$.cid%type := NLS_UPPER(comp_id);

BEGIN
  update registry$ set status = s_option_off where cid=p_id;
  commit;
END comp_option_off;

--------------------------- comp_invalid -------------------------------
-- for use in dbms_registry.patch_script

PROCEDURE comp_invalid (comp_id IN VARCHAR2)
IS
PRAGMA AUTONOMOUS_TRANSACTION;  -- executes DML from query

p_id registry$.cid%type := NLS_UPPER(comp_id);

BEGIN
  update registry$ set status = s_invalid where cid=p_id;
  commit;
END comp_invalid;

----------------------------------------------------------------------
-- PUBLIC FUNCTIONS
----------------------------------------------------------------------

-------------------------- INVALID -----------------------------------

PROCEDURE invalid     (comp_id      IN VARCHAR2)
IS

p_id       VARCHAR2(30) :=NLS_UPPER(comp_id);
p_invoker# NUMBER       :=get_user;

BEGIN

IF exists_comp(p_id) THEN
   check_invoker(p_id, p_invoker#);
   update_comp(s_invalid, p_id, NULL, NULL, NULL, NULL, NULL);
   update registry$ set date_invalid = SYSDATE where cid=p_id;
   commit;
ELSE
   RAISE NO_DATA_FOUND;
END IF;

END invalid;

-------------------------- VALID -----------------------------------

PROCEDURE valid     (comp_id      IN VARCHAR2)
IS

p_id       VARCHAR2(30) :=NLS_UPPER(comp_id);
p_invoker# NUMBER       :=get_user;

BEGIN

IF exists_comp(p_id) THEN
   check_invoker(p_id, p_invoker#);
   update_comp(s_valid, p_id, NULL, NULL, NULL, NULL, NULL);
   update registry$ set date_valid = SYSDATE where cid=p_id;
   commit;
ELSE
   RAISE NO_DATA_FOUND;
END IF;

END valid;

------------------------- LOADING ------------------------------------

PROCEDURE loading     (comp_id      IN VARCHAR2,
                       comp_name    IN VARCHAR2,
                       comp_proc    IN VARCHAR2 DEFAULT NULL,
                       comp_schema  IN VARCHAR2 DEFAULT NULL,
                       comp_parent  IN VARCHAR2 DEFAULT NULL)
IS

p_id      VARCHAR2(30) :=NLS_UPPER(comp_id);
p_name    VARCHAR2(255):=comp_name;
p_proc    VARCHAR2(61) :=NLS_UPPER(comp_proc);
p_schema  VARCHAR2(30) :=NLS_UPPER(comp_schema);
p_parent  VARCHAR2(30) :=NLS_UPPER(comp_parent);
p_invoker#   NUMBER    :=get_user;
p_schema#    NUMBER;

BEGIN
  IF p_schema IS NOT NULL then
     p_schema#:=get_user(p_schema);
  ELSE
     p_schema#:=p_invoker#;
  END IF;

  IF exists_comp(p_id) THEN
     check_invoker(p_id,p_invoker#);
     update_comp(s_loading, p_id, p_name, p_proc, p_parent, NULL, NULL);
     update registry$ set schema# = p_schema#, 
                          date_loading = SYSDATE where cid=p_id;
     commit;
  ELSE
     new_comp(s_loading, p_id, p_name, p_proc, p_parent, 
              p_invoker#, p_schema#); 
     update registry$ set date_loading = SYSDATE where cid=p_id;
     commit;
END IF;

END loading;

-------------------------- LOADED -------------------------------------

PROCEDURE loaded      (comp_id      IN VARCHAR2,
                       comp_version IN VARCHAR2 DEFAULT NULL,
                       comp_banner  IN VARCHAR2 DEFAULT NULL)
IS

p_id      VARCHAR2(30) :=NLS_UPPER(comp_id);
p_version VARCHAR2(16) :=NLS_UPPER(comp_version);
p_banner  VARCHAR2(80) :=comp_banner;
p_invoker#   NUMBER       :=get_user;

BEGIN
  
IF exists_comp(p_id) THEN
   check_invoker(p_id, p_invoker#);
   IF p_version IS NULL THEN
      p_version := sys.dbms_registry.release_version;
   END IF;
   IF p_banner IS NULL THEN
      p_banner:= dbms_registry.comp_name(p_id) || ' Release ' ||
          p_version || ' - ' || sys.dbms_registry.release_status;
   END IF;
   update_comp(s_loaded, p_id, NULL, NULL, NULL, p_version, p_banner);
   update registry$ set date_loaded = SYSDATE where cid=p_id;
   commit;
ELSE
   RAISE NO_DATA_FOUND;
END IF;

END loaded;

-------------------------- UPGRADING ----------------------------------

PROCEDURE upgrading   (comp_id     IN VARCHAR2,
                       new_name    IN VARCHAR2 DEFAULT NULL,
                       new_proc    IN VARCHAR2 DEFAULT NULL,
                       new_schema  IN VARCHAR2 DEFAULT NULL,
                       new_parent  IN VARCHAR2 DEFAULT NULL)
IS

p_id      VARCHAR2(30) :=NLS_UPPER(comp_id);
p_name    VARCHAR2(255):=new_name;
p_proc    VARCHAR2(61):=NLS_UPPER(new_proc);
p_schema  VARCHAR2(30):=NLS_UPPER(new_schema);
p_par     VARCHAR2(30):=NLS_UPPER(new_parent);
p_invoker#   NUMBER       :=get_user;
p_schema#    NUMBER;

BEGIN
  
IF exists_comp(p_id) THEN
   check_invoker(p_id, p_invoker#);
   update_comp(s_upgrading, p_id, p_name, p_proc, p_par, NULL, NULL);
   IF p_schema IS NOT NULL then
      p_schema#:=get_user(p_schema);
      update registry$ set schema# = p_schema# where cid=p_id;
   END IF;
   update registry$ set date_upgrading = SYSDATE where cid=p_id;
   commit;
ELSE
   RAISE NO_DATA_FOUND;
END IF;

END upgrading;

-------------------------- UPGRADED -------------------------------------

PROCEDURE upgraded    (comp_id      IN VARCHAR2,
                       new_version  IN VARCHAR2 DEFAULT NULL,
                       new_banner   IN VARCHAR2 DEFAULT NULL)
IS

p_id      VARCHAR2(30) :=NLS_UPPER(comp_id);
p_version VARCHAR2(16) :=NLS_UPPER(new_version);
p_banner  VARCHAR2(80) :=new_banner;
p_invoker#   NUMBER    :=get_user;

BEGIN
  
IF exists_comp(p_id) THEN
   check_invoker(p_id, p_invoker#);
   IF p_version IS NULL THEN
      p_version := sys.dbms_registry.release_version;
   END IF;
   IF p_banner IS NULL THEN
      p_banner:= dbms_registry.comp_name(p_id) || ' Release ' ||
          p_version || ' - ' || sys.dbms_registry.release_status;
   END IF;
   update_comp(s_upgraded, p_id, NULL, NULL, NULL, p_version, p_banner);
   update registry$ set date_upgraded = SYSDATE where cid=p_id;
   commit;
ELSE
   RAISE NO_DATA_FOUND;
END IF;

END upgraded;

-------------------------- DOWNGRADING -------------------------------

PROCEDURE downgrading (comp_id      IN VARCHAR2,
                       old_name     IN VARCHAR2 DEFAULT NULL,
                       old_proc     IN VARCHAR2 DEFAULT NULL,
                       old_schema   IN VARCHAR2 DEFAULT NULL,
                       old_parent   IN VARCHAR2 DEFAULT NULL)
IS

p_id      VARCHAR2(30) :=NLS_UPPER(comp_id);
p_name    VARCHAR2(255):=old_name;
p_proc    VARCHAR2(61) :=NLS_UPPER(old_proc);
p_schema  VARCHAR2(30) :=NLS_UPPER(old_schema);
p_par     VARCHAR2(30) :=NLS_UPPER(old_parent);
p_invoker#   NUMBER       :=get_user;
p_schema#    NUMBER;

BEGIN

IF exists_comp(p_id) THEN
   check_invoker(p_id, p_invoker#);
   update_comp(s_downgrading, p_id, p_name, p_proc, p_par, NULL, NULL);
   IF p_schema IS NOT NULL then
      p_schema#:=get_user(p_schema);
      update registry$ set schema# = p_schema# where cid=p_id;
   END IF;
   update registry$ set date_downgrading = SYSDATE where cid=p_id;
   commit;
ELSE
   RAISE NO_DATA_FOUND;
END IF;

END downgrading;


-------------------------- DOWNGRADED --------------------------------

PROCEDURE downgraded  (comp_id      IN VARCHAR2,
                       old_version  IN VARCHAR2)
IS

p_id      VARCHAR2(30) :=NLS_UPPER(comp_id);
p_version VARCHAR2(30) :=old_version;
p_invoker#   NUMBER       :=get_user;

BEGIN

IF exists_comp(p_id) THEN
   check_invoker(p_id, p_invoker#);
   update_comp(s_downgraded, p_id, NULL, NULL, NULL, p_version, NULL);
   update registry$ set date_downgraded = SYSDATE where cid=p_id;
   commit;
ELSE
   RAISE NO_DATA_FOUND;
END IF;

END downgraded;

-------------------------- REMOVING ---------------------------------

PROCEDURE removing      (comp_id      IN VARCHAR2)
IS

p_id      VARCHAR2(30) :=NLS_UPPER(comp_id);
p_invoker#   NUMBER       :=get_user;

BEGIN

IF exists_comp(p_id) THEN
   check_invoker(p_id, p_invoker#);
   update_comp(s_removing, p_id, NULL, NULL, NULL, NULL, NULL);
   update registry$ set date_removing = SYSDATE where cid=p_id;
   commit;
ELSE
   RAISE NO_DATA_FOUND;
END IF;

END removing;

-------------------------- REMOVED -----------------------------------

PROCEDURE removed      (comp_id      IN VARCHAR2)
IS

p_id      VARCHAR2(30) :=NLS_UPPER(comp_id);
p_invoker#   NUMBER       :=get_user;

BEGIN

IF exists_comp(p_id) THEN
   check_invoker(p_id, p_invoker#);
   update_comp(s_removed, p_id, NULL, NULL, NULL, NULL, NULL);
   update registry$ set date_removed = SYSDATE where cid=p_id;
   commit;
ELSE
   RAISE NO_DATA_FOUND;
END IF;

END removed;

-------------------------- STARTUP_REQUIRED ----------------------------

PROCEDURE startup_required  (comp_id      IN VARCHAR2)
IS

p_id      VARCHAR2(30) :=NLS_UPPER(comp_id);
p_invoker#   NUMBER       :=get_user;
p_flags   NUMBER;

BEGIN

IF exists_comp(p_id) THEN
   check_invoker(p_id, p_invoker#);
   select flags into p_flags from registry$ where cid=p_id;
   p_flags := set_flag(p_flags,f_startup_req);
   update registry$ set flags = p_flags where cid=p_id;
   commit;
ELSE
   RAISE NO_DATA_FOUND;
END IF;

END startup_required;

----------------------- STARTUP_COMPLETE --------------------------

PROCEDURE startup_complete  (comp_id      IN VARCHAR2)
IS

p_id      VARCHAR2(30) :=NLS_UPPER(comp_id);
p_invoker#   NUMBER       :=get_user;
p_flags   NUMBER;
BEGIN

IF exists_comp(p_id) THEN
   check_invoker(p_id, p_invoker#);
   select flags into p_flags from registry$ where cid=p_id;
   p_flags := clear_flag(p_flags,f_startup_req);
   update registry$ set flags = p_flags where cid=p_id;
   commit;
ELSE
   RAISE NO_DATA_FOUND;
END IF;

END startup_complete;

-------------------------- STATUS ---------------------------

FUNCTION  status       (comp_id IN VARCHAR2) RETURN VARCHAR2
IS 

p_id          VARCHAR2(30) :=NLS_UPPER(comp_id);
p_status_name VARCHAR2(30);

BEGIN
   SELECT DECODE(
              status, 0, 'INVALID',
                      1, 'VALID',
                      2, 'LOADING',
                      3, 'LOADED',
                      4, 'UPGRADING',
                      5, 'UPGRADED',
                      6, 'DOWNGRADING',
                      7, 'DOWNGRADED',
                      8, 'REMOVING',
                      99, 'REMOVED',
                      NULL) INTO p_status_name
    FROM registry$ WHERE cid=p_id; 
    RETURN p_status_name;
EXCEPTION
   WHEN NO_DATA_FOUND THEN
     RETURN NULL;

END status;

-------------------------- VERSION --------------------------

FUNCTION  version      (comp_id IN VARCHAR2) RETURN VARCHAR2
IS

p_id      VARCHAR2(30) :=NLS_UPPER(comp_id);
p_version VARCHAR2(30);

BEGIN
   SELECT version INTO p_version FROM registry$ WHERE cid=p_id;
   RETURN p_version;
EXCEPTION
   WHEN NO_DATA_FOUND THEN
     RETURN NULL;

END version;

----------------------- COMP_NAME --------------------------

FUNCTION  comp_name  (comp_id IN VARCHAR2) RETURN VARCHAR2
IS

p_id      registry$.cid%TYPE :=NLS_UPPER(comp_id);
p_name    registry$.cname%TYPE;

BEGIN
   SELECT cname INTO p_name FROM registry$ WHERE cid=p_id;
   RETURN p_name;
EXCEPTION
   WHEN NO_DATA_FOUND THEN
     RETURN NULL;

END comp_name;

-------------------------- SCRIPT --------------------------------

FUNCTION script(comp_id IN VARCHAR2, 
                script_name IN VARCHAR2) RETURN VARCHAR2 
IS

p_cid  VARCHAR2(30) := NLS_UPPER(comp_id);
p_schema VARCHAR2(30);

BEGIN
  SELECT schema INTO p_schema FROM dba_registry
  WHERE comp_id = p_cid AND status NOT IN ('REMOVED','REMOVING');
  -- look for components that require options (SE database and OLS)
  IF (p_cid = 'OLS' AND NOT option_OK('Oracle Label Security')) OR
     (p_cid IN ('APS','XOQ','AMD') AND NOT option_OK('OLAP')) OR
     (p_cid = 'ODM' AND NOT option_OK('Oracle Data Mining')) THEN
       IF p_cid = 'OLS' THEN    -- login trigger causes errors
          disable_ddl_triggers('LBACSYS');
       END IF;
       comp_option_off(p_cid);  -- set status to OPTION OFF
       RETURN dbms_registry.nothing_script;
  ELSE
     RETURN (script_name);
  END IF;
EXCEPTION
  WHEN NO_DATA_FOUND THEN 
     RETURN dbms_registry.nothing_script;
END script;

-------------------- PATCH_SCRIPT --------------------------------

FUNCTION patch_script(comp_id IN VARCHAR2, 
                script_name IN VARCHAR2) RETURN VARCHAR2 
IS

p_cid  VARCHAR2(30) := NLS_UPPER(comp_id);
p_schema VARCHAR2(30);

BEGIN
  SELECT schema INTO p_schema FROM dba_registry
  WHERE comp_id = p_cid AND status NOT IN ('REMOVED','REMOVING');
  -- look for components that require options (SE database and OLS)
  IF (p_cid = 'OLS' AND NOT option_OK('Oracle Label Security')) OR
     (p_cid IN ('APS','XOQ','AMD') AND NOT option_OK('OLAP')) OR
     (p_cid = 'ODM' AND NOT option_OK('Oracle Data Mining')) THEN
       IF p_cid = 'OLS' THEN    -- login trigger causes errors
          disable_ddl_triggers('LBACSYS');
       END IF;
       comp_option_off(p_cid);  -- set option off status
       RETURN dbms_registry.nothing_script;
  ELSE -- check that the component version can be patched
     IF SUBSTR(dbms_registry.version(p_cid),1,5) =
        SUBSTR(dbms_registry.release_version,1,5) THEN
        RETURN (script_name);
     ELSE
        comp_invalid(p_cid);  -- set status invalid
        RETURN dbms_registry.nothing_script;
     END IF;
  END IF;
EXCEPTION
  WHEN NO_DATA_FOUND THEN 
     RETURN dbms_registry.nothing_script;
END patch_script;


-------------------------- CPU_SCRIPT -------------------------

FUNCTION cpu_script(comp_id IN VARCHAR2,
                    script_name IN VARCHAR2) RETURN VARCHAR2 
IS

p_cid  VARCHAR2(30) := NLS_UPPER(comp_id);
p_schema VARCHAR2(30);

BEGIN
  SELECT schema INTO p_schema FROM dba_registry
  WHERE comp_id = p_cid AND status NOT IN ('REMOVED','REMOVING');
  -- look for components that require options (SE database and OLS)
  IF (p_cid = 'OLS' AND NOT option_OK('Oracle Label Security')) OR
     (p_cid IN ('APS','XOQ','AMD') AND NOT option_OK('OLAP')) OR
     (p_cid = 'ODM' AND NOT option_OK('Oracle Data Mining')) THEN
       IF p_cid = 'OLS' THEN    -- login trigger causes errors
          disable_ddl_triggers('LBACSYS');
       END IF;
       comp_option_off(p_cid);  -- set option off status
       RETURN dbms_registry.nothing_script;
  ELSE -- check that the component version can be patched
     IF SUBSTR(dbms_registry.version(p_cid),1,5) =
        SUBSTR(dbms_registry.release_version,1,5) THEN
        RETURN (script_name);
     ELSE
        comp_invalid(p_cid);  -- set status invalid
        RETURN dbms_registry.nothing_script;
     END IF;
  END IF;
EXCEPTION
  WHEN NO_DATA_FOUND THEN 
     RETURN dbms_registry.nothing_script;
END cpu_script;

-------------------------- TIME --------------------------------

FUNCTION time_stamp (comp_id IN VARCHAR2) RETURN VARCHAR2 
IS

p_cid  VARCHAR2(30) := NLS_UPPER(comp_id);
p_null CHAR(1);

BEGIN
  SELECT NULL INTO p_null FROM registry$
  WHERE cid = p_cid AND status NOT IN (s_removing, s_removed);
  RETURN ('COMP_TIMESTAMP ' 
           || RPAD(p_cid,10) || ' ' || SYSTIMESTAMP);
EXCEPTION
  WHEN NO_DATA_FOUND THEN 
     IF p_cid LIKE '%_BGN' OR p_cid LIKE '%_END' THEN
        RETURN ('COMP_TIMESTAMP ' 
           || RPAD(p_cid,10) || ' ' || SYSTIMESTAMP); 
     ELSE
        RETURN NULL;
     END IF;
END time_stamp;

-------------------------- IS_LOADED ----------------------------

FUNCTION  is_loaded    (comp_id IN VARCHAR2, 
                        version IN VARCHAR2 DEFAULT NULL) RETURN NUMBER
IS 

p_id      VARCHAR2(30) :=NLS_UPPER(comp_id);
p_version VARCHAR2(30) :=version;

BEGIN

IF exists_comp(p_id) THEN
   IF dbms_registry.status(p_id) IN ('LOADED','VALID') THEN
      IF p_version IS NULL THEN
         RETURN 1;
      ELSE
         IF dbms_registry.version(p_id) = p_version THEN
            RETURN 1;
         ELSE
            RETURN 0;
         END IF;
      END IF;
   END IF;
   RETURN 0;
END IF;
RETURN NULL;

END is_loaded;

-------------------------- IS_VALID ----------------------------

FUNCTION  is_valid    (comp_id IN VARCHAR2, 
                        version IN VARCHAR2 DEFAULT NULL) RETURN NUMBER
IS 

p_id      VARCHAR2(30) :=NLS_UPPER(comp_id);
p_version VARCHAR2(30) :=version;

BEGIN

IF exists_comp(p_id) THEN
   IF dbms_registry.status(p_id) = 'VALID' THEN
      IF p_version IS NULL THEN
         RETURN 1;
      ELSE
         IF dbms_registry.version(p_id) = p_version THEN
            RETURN 1;
         ELSE
            RETURN 0;
         END IF;
      END IF;
   END IF;
   RETURN 0;
END IF;
RETURN NULL;

END is_valid;

--------------------- IS_STARTUP_REQUIRED ------------------------

FUNCTION  is_startup_required (comp_id IN VARCHAR2) RETURN NUMBER
IS 

p_id      VARCHAR2(30) :=NLS_UPPER(comp_id);
p_flags   NUMBER;

BEGIN
IF exists_comp(p_id) THEN
   SELECT flags INTO p_flags FROM registry$ WHERE cid=p_id;
   IF bitand(p_flags, f_startup_req) = f_startup_req THEN
      RETURN 1;
   ELSE
      RETURN 0;
   END IF;
END IF;
RETURN NULL;

END is_startup_required;

END dbms_registry;
/

show errors

CREATE OR REPLACE PUBLIC SYNONYM dbms_registry FOR dbms_registry;

--------------------------------------------------------------------
--  Internal functions used by SYS during upgrade/downgrade
--------------------------------------------------------------------

CREATE OR REPLACE PACKAGE dbms_registry_sys
AS

PROCEDURE drop_user  (username IN VARCHAR2);
 
PROCEDURE validate_catalog;

PROCEDURE validate_catproc;

PROCEDURE validate_catjava;

PROCEDURE check_component_downgrades;

PROCEDURE populate;

PROCEDURE validate_components;

END dbms_registry_sys;
/

CREATE OR REPLACE PACKAGE BODY dbms_registry_sys
AS

--------------------------------------------------------------------
--  Internal functions used by SYS during upgrade/downgrade
--------------------------------------------------------------------

-- GLOBAL

g_null         CHAR(1);

------------------------ PRIVATE FUNCTIONS ------------------------

------------------------- get_user --------------------------------

FUNCTION get_user RETURN NUMBER
IS

p_user# NUMBER;

BEGIN
  SELECT user# INTO p_user# FROM sys.user$
  WHERE name = SYS_CONTEXT ('USERENV', 'SESSION_USER');
  RETURN p_user#;
END get_user;

--------------------------------------------------------------------

FUNCTION get_user(usr IN VARCHAR2) RETURN NUMBER
IS

p_user# NUMBER;

BEGIN
  SELECT user# INTO p_user# FROM sys.user$
  WHERE name = usr;
  RETURN p_user#;
END get_user;

------------------------ PUBLIC FUNCTIONS ---------------------------

-------------------------- DROP_USER --------------------------------
--
-- This procedure is called by the server whenever a user is dropped
-- with the cascade option (via athe duc$ table)
--
--------------------------------------------------------------------

PROCEDURE drop_user  (username IN VARCHAR2)
IS 

p_usr   VARCHAR2(30) := NLS_UPPER(username);
p_usr#  NUMBER;
BEGIN
   p_usr# := get_user(p_usr);
   DELETE FROM registry$ WHERE invoker# = p_usr# OR schema# = p_usr#;

END drop_user;

-------------------------- VALIDATE_CATPROC -------------------------
--
-- This function performs a short "sanity check" on the contents of 
-- the RDBMS data dictionary elements created by CATPROC and sets
-- the status to VALID or INVALID for the CATPROC component.
--
--------------------------------------------------------------------

PROCEDURE validate_catproc
IS

start_time DATE;
end_time   DATE;

BEGIN
   SELECT date_loading, date_loaded INTO start_time, end_time
   FROM registry$ WHERE cid = 'CATPROC';

   IF start_time IS NULL OR end_time IS NULL OR end_time < start_time THEN
       dbms_registry.invalid('CATPROC');
       RETURN;
   END IF;

   BEGIN
      SELECT NULL INTO g_null
      FROM obj$
      WHERE status > 1 AND
            (ctime BETWEEN start_time AND end_time OR
             mtime BETWEEN start_time AND end_time OR
             stime BETWEEN start_time AND end_time) AND
             ROWNUM <=1;
--    If any found, then invalid
      dbms_registry.invalid('CATPROC');
   EXCEPTION
      WHEN NO_DATA_FOUND THEN
         dbms_registry.valid('CATPROC');
   END;

END validate_catproc;

-------------------------- VALIDATE_CATJAVA -------------------------
--
-- This function performs a short "sanity check" on the contents of 
-- the RDBMS data dictionary elements created by CATJAVA and sets
-- the status to VALID or INVALID for the CATJAVA component.
--
--------------------------------------------------------------------

PROCEDURE validate_catjava
IS

start_time DATE;
end_time   DATE;

BEGIN
   SELECT date_loading, date_loaded INTO start_time, end_time
   FROM registry$ WHERE cid = 'CATJAVA';

   IF start_time IS NULL OR end_time IS NULL OR end_time < start_time THEN
       dbms_registry.invalid('CATJAVA');
       RETURN;
   END IF;

   BEGIN
      SELECT NULL INTO g_null
      FROM obj$
      WHERE status > 1 AND
            (ctime BETWEEN start_time AND end_time OR
             mtime BETWEEN start_time AND end_time OR
             stime BETWEEN start_time AND end_time) AND
             ROWNUM <=1;
--    If any found, then invalid
      dbms_registry.invalid('CATJAVA');
   EXCEPTION
      WHEN NO_DATA_FOUND THEN
         dbms_registry.valid('CATJAVA');
   END;

END validate_catjava;

-------------------------- VALIDATE_CATALOG --------------------------------
--
-- This function performs a short "sanity check" on the contents of 
-- the RDBMS dictionary views created by CATALOG.sql and sets the
-- registry status to VALID or INVALID;
--
--------------------------------------------------------------------

PROCEDURE  validate_catalog
IS

start_time DATE;
end_time   DATE;
p_count  NUMBER;

BEGIN

   SELECT date_loading, date_loaded INTO start_time, end_time 
   FROM registry$ WHERE cid = 'CATALOG';
  
   IF start_time IS NULL OR end_time IS NULL OR end_time < start_time THEN
      dbms_registry.invalid('CATALOG');
      RETURN;
   END IF;

   BEGIN
      SELECT NULL INTO g_null
      FROM obj$ 
      WHERE status > 1 AND 
            (ctime BETWEEN start_time AND end_time OR
             mtime BETWEEN start_time AND end_time OR
             stime BETWEEN start_time AND end_time) AND
             ROWNUM <=1;
--    Any rows found, then invalid
      dbms_registry.invalid('CATALOG');
   EXCEPTION
      WHEN NO_DATA_FOUND THEN
          dbms_registry.valid('CATALOG');
   END;
END validate_catalog;

--------------------CHECK_COMPONENT_DOWNGRADES ---------------------

PROCEDURE check_component_downgrades
IS

BEGIN
   SELECT NULL INTO g_null FROM dba_registry
   WHERE status NOT IN ('DOWNGRADED','REMOVED','INVALID') AND
         comp_id NOT IN ('CATPROC', 'CATJAVA', 'CATALOG') AND
         ROWNUM <=1;
   RAISE_APPLICATION_ERROR(-20000,
        'Some components have not been successfully downgraded.');
EXCEPTION
   WHEN NO_DATA_FOUND THEN NULL;
END check_component_downgrades;

-------------------------- POPULATE --------------------------------
--
-- This procedure looks in the database to determine if 
-- a component is installed.  If it is, then an attempt is made to
-- insert rows for the component into the registry.  If a row already
-- exists for the component identifier, then the new row is not
-- inserted, but no error is indicated.
--
--------------------------------------------------------------------
PROCEDURE populate
IS

s_loaded      NUMBER :=3;

p_schema#  NUMBER;
p_sdo#     NUMBER;
p_invoker# NUMBER := get_user;
p_stmt     VARCHAR2(500);
TYPE cur_t IS REF CURSOR;
p_cur      cur_t;
p_num      NUMBER;

o_id       VARCHAR2(30);
o_name     VARCHAR2(255);
o_version  VARCHAR2(30);
o_banner   VARCHAR2(80);
o_status   VARCHAR2(30);
o_modified DATE;
o_pid      VARCHAR2(30);

im_status   VARCHAR2(30);
vi_status   VARCHAR2(30);

BEGIN

-----------------------------------------------------------------------
-- Java
-----------------------------------------------------------------------

  SELECT COUNT(*) INTO p_num FROM obj$ WHERE type#=29 AND owner#=0;
  IF p_num != 0 THEN
      o_version := '8.1.5';
      BEGIN
         SELECT obj# INTO p_num FROM obj$ WHERE type#=29 AND owner#=0
             AND name='java/security/cert/CRL';
         o_version := '8.1.6';
         SELECT obj# INTO p_num FROM obj$ WHERE type#=29 AND owner#=0
             AND name='oracle/aurora/util/OS';
         o_version := '8.1.7';
         SELECT obj# INTO p_num FROM obj$ WHERE type#=29 AND owner#=0
             AND name='oracle/jdbc/Const';
         o_version := '9.0.1';
         SELECT obj# INTO p_num FROM obj$ WHERE type#=29 AND owner#=0
             AND name='oracle/aurora/genmsg/GenMsg';
         o_version := '9.2.0';
      EXCEPTION
         WHEN NO_DATA_FOUND THEN NULL;
      END;
      o_name := 'JServer JAVA Virtual Machine';
      BEGIN
        INSERT INTO registry$ 
           (status, modified, cid, cname, version, schema#, invoker#, 
            banner, flags, date_loaded)
         VALUES (s_loaded, SYSDATE, 'JAVAVM', o_name, o_version, 
              p_invoker#, p_invoker#, 
              o_name||' Release '||o_version||' - Production', 0, SYSDATE);
         COMMIT;
      EXCEPTION
         WHEN DUP_VAL_ON_INDEX THEN NULL;
      END;
      -- indicate CATJAVA also loaded 
      BEGIN
        INSERT INTO registry$ 
           (status, modified, cid, cname, version, schema#, invoker#, 
            banner, flags, date_loaded)
         VALUES (s_loaded, SYSDATE, 'CATJAVA', 'Java Packages', o_version, 
              p_invoker#, p_invoker#, 
              'Java Packages Release '||o_version||' - Production', 0, SYSDATE);
         COMMIT;
      EXCEPTION
         WHEN DUP_VAL_ON_INDEX THEN NULL;
      END;
    END IF;

-----------------------------------------------------------------------
-- XML
-----------------------------------------------------------------------

 SELECT COUNT(*) INTO p_num FROM obj$ WHERE type#=29 AND owner#=0
      AND name like 'oracle/xml/parser%';
  IF p_num != 0 THEN
      o_version := '8.1.7';
      BEGIN
         SELECT obj# INTO p_num FROM obj$ WHERE type#=29 AND owner#=0
             AND name='oracle/xml/comp/CXMLConstants';
         o_version := '9.0.1';
         SELECT obj# INTO p_num FROM obj$ WHERE type#=29 AND owner#=0
             AND name='oracle/xml/io/XMLObjectInput';
         o_version := '9.2.0';
      EXCEPTION
         WHEN NO_DATA_FOUND THEN NULL;
      END;
      o_name := 'Oracle XDK for Java';
      BEGIN
        INSERT INTO registry$ 
           (status, modified, cid, cname, version, schema#, invoker#, 
            banner, flags, date_loaded)
         VALUES (s_loaded, SYSDATE, 'XML', o_name, o_version, 
              p_invoker#, p_invoker#, 
              o_name||' Release '||o_version||' - Production', 0, SYSDATE);
         COMMIT;
      EXCEPTION
         WHEN DUP_VAL_ON_INDEX THEN NULL;
      END;
    END IF;

-----------------------------------------------------------------------
-- Text
-----------------------------------------------------------------------

 BEGIN
   SELECT user# INTO p_schema# FROM user$ WHERE name = 'CTXSYS';

   SELECT count(*) INTO p_num
     FROM obj$
    WHERE owner# = p_schema# AND name='CTX_VERSION' AND rownum <= 1;

   IF (p_num = 0) THEN
     -- no CTX_VERSION -- must be pre-8.1.5.  Since we cannot determine
     -- accurately exactly which version, assume 2.4.6
     o_name := 'Oracle ConText Option';
     o_version := '2.4.6.0.0';
     o_banner := 'Oracle ConText Option 2.4.6 - Production';
   ELSE
     BEGIN
       execute immediate 
        'SELECT ver_dict FROM ctxsys.ctx_version WHERE rownum <= 1'
         into o_version;
       IF (o_version LIKE '8.1.%') THEN
         o_name := 'Oracle interMedia Text';
         o_banner := 'Oracle interMedia Text '||o_version||' - Production';
       ELSE
         o_name := 'Oracle Text';
         o_banner := 'Oracle Text '||o_version||' - Production';
       END IF;
     EXCEPTION
       WHEN OTHERS THEN
         -- unexpected problem while selecting from CTX_VERSION
         -- assume 8.1.5
         o_version := '8.1.5';
         o_name := 'Oracle interMedia Text';
         o_banner := 'Oracle interMedia Text 8.1.5 - Production';
     END;
   END IF;

   BEGIN
      INSERT INTO registry$ 
        (status, modified, cid, cname, version, schema#, invoker#, 
         banner, flags, date_loaded)
      VALUES (s_loaded, SYSDATE, 'CONTEXT', o_name, o_version, 
              p_schema#, p_schema#, o_banner, 0, SYSDATE);
      COMMIT;
    EXCEPTION
         WHEN DUP_VAL_ON_INDEX THEN NULL;
    END;

 EXCEPTION
   WHEN NO_DATA_FOUND THEN NULL;
   -- if there is no CTXSYS, then text is not loaded at all
 END;

-----------------------------------------------------------------------
-- Real Application Clusters
-----------------------------------------------------------------------
   BEGIN
      SELECT null INTO g_null 
      FROM obj$
      WHERE owner#=0 AND name='V$PING' AND ROWNUM <=1;
--    If V$PING exists, then RAC is loaded
      BEGIN
--      put in registry without version or banner
        INSERT INTO registry$ 
           (status, modified, cid, cname, version, schema#, invoker#, 
            banner, flags, date_loaded)
         VALUES (s_loaded, SYSDATE, 'RAC', 'Real Application Clusters', NULL, 
              p_invoker#, p_invoker#, NULL, 0, SYSDATE);
         COMMIT;
      EXCEPTION
         WHEN DUP_VAL_ON_INDEX THEN NULL;
      END;
   EXCEPTION
      WHEN NO_DATA_FOUND THEN NULL;
   END;

-----------------------------------------------------------------------
-- Oracle Workspace Manager
-----------------------------------------------------------------------

  -- first find the env_vars table
  select count(*) into p_num from obj$ o, user$ u
  where u.name = 'WMSYS' and o.name = 'WM$ENV_VARS' and
        u.user#=o.owner# and o.type#=2;
  IF(p_num = 1)
  THEN
    p_schema#:=get_user('WMSYS');
    execute immediate 
      'select value from WMSYS.WM$ENV_VARS where name = ''OWM_VERSION''' 
       into o_version ;
  ELSE
    select count(*) into p_num from obj$ o, user$ u
    where u.name = 'SYSTEM' and o.name = 'WM$ENV_VARS' and 
          u.user#=o.owner# and o.type#=2;
    IF(p_num = 1) THEN
      BEGIN
        p_schema#:=get_user('SYSTEM');
        execute immediate 
         'select value from SYSTEM.WM$ENV_VARS where name = ''OWM_VERSION''' 
          into o_version ;
      EXCEPTION
        WHEN NO_DATA_FOUND THEN
          select count(*) into p_num from obj$ o, user$ u
          where u.name = 'SYSTEM' and o.name = 'WM$LOCKROWS_INFO' 
                and u.user# = o.owner# and o.type#=2;
          IF(p_num = 1)
          THEN
            o_version := '9.0.1.0.0';
          ELSE
            o_version := null;
          END IF;
        END ;
    ELSE
      -- neither of the table found, not installed
      o_version := null;
    END IF;
  END IF;

  IF (o_version is not null) THEN
      o_name := 'Oracle Workspace Manager';
      BEGIN
        INSERT INTO registry$
                (status, modified, cid, cname, version, 
                 invoker#, schema#, banner, flags, date_loaded)
         VALUES (s_loaded, SYSDATE, 'OWM', o_name, o_version, 
                 p_invoker#, p_schema#,
                 o_name||' Release '||o_version||' - Production', 0, SYSDATE);
         COMMIT;      
      EXCEPTION
         WHEN DUP_VAL_ON_INDEX THEN NULL;
      END;
  END IF;
 
-----------------------------------------------------------------------
-- Spatial and Intermedia
-----------------------------------------------------------------------

  BEGIN
    im_status:=NULL;
    vi_status:=NULL;
    --
    -- Find 8.1.5 and later versions
    --
    SELECT user# INTO p_schema# FROM user$ WHERE name='ORDSYS';
    im_status:='NOT_FOUND';  /* if ORDSYS user exists */
    vi_status:='NOT_FOUND';

    p_stmt := 'SELECT short_name, substr(name,1,255),' || 
             'version, description ' ||
             'FROM ordsys.ord_installations';
    BEGIN
      OPEN p_cur FOR p_stmt;
      LOOP
        FETCH p_cur INTO o_id, o_name, o_version, o_banner;
        EXIT WHEN p_cur%NOTFOUND;
        IF o_id ='SDO' THEN
          BEGIN
            SELECT user# INTO p_sdo# FROM user$ WHERE name='MDSYS';
            o_name:='Oracle Spatial';
            INSERT INTO sys.registry$ 
                   (status, modified, cid, cname, version, 
                    invoker#, schema#, banner, flags, date_loaded)
            VALUES (s_loaded,SYSDATE, o_id, o_name, o_version, 
                    p_invoker#, p_sdo#, o_banner, 0, SYSDATE);
            COMMIT;
          EXCEPTION
            WHEN DUP_VAL_ON_INDEX THEN NULL;
            WHEN NO_DATA_FOUND THEN NULL;  /* no MDSYS user */
          END;
        ELSIF o_id = 'ORDIM' OR o_id = 'ORDVIR' THEN
          BEGIN
            IF o_id = 'ORDIM' THEN 
               o_name := 'Oracle interMedia'; 
               im_status:='FOUND';
            END IF;
            IF o_id = 'ORDVIR' THEN 
               o_name := 'Oracle Visual Information Retrieval';
               vi_status:='FOUND';
            END IF;
            INSERT INTO sys.registry$ 
                   (status, modified, cid, cname, version, 
                    invoker#, schema#, banner, flags, date_loaded)
            VALUES (s_loaded,SYSDATE, o_id, o_name, o_version, 
                    p_invoker#, p_schema#, o_banner, 0, SYSDATE);
            COMMIT;
          EXCEPTION
            WHEN DUP_VAL_ON_INDEX THEN NULL;
          END;
        END IF;
      END LOOP;

    EXCEPTION
      WHEN OTHERS THEN
        im_status := 'NOT_FOUND'; /* no install table - pre 815*/
        vi_status := 'NOT_FOUND'; /* no install table - pre 815*/
    END;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN NULL;  /* no ORDSYS user */
  END;

  -- 
  -- Look for interMedia pre 8.1.5 versions
  --
  IF im_status = 'NOT_FOUND' THEN
      SELECT count(*) INTO p_num FROM obj$ o, user$ u
      WHERE o.name='ORDIMGB' AND u.name ='ORDSYS' AND o.owner# = u.user#;
      IF p_num > 0 THEN
        BEGIN
          INSERT INTO sys.registry$ 
                 (status, modified, cid, cname, version, 
                  invoker#, schema#, banner, flags, date_loaded)
          VALUES (s_loaded, SYSDATE, 'ORDIM', 'Oracle interMedia', 
                  '8.0.6.0.0', p_invoker#, p_schema#, 
                  'Oracle interMedia', 0, SYSDATE);
          COMMIT;
        EXCEPTION
            WHEN DUP_VAL_ON_INDEX THEN NULL;
        END;
      END IF;
  END IF;

  --
  -- Look for Visual Information Retrieval pre 8.1.5 versions
  --  
  IF vi_status = 'NOT_FOUND' THEN
      SELECT count(*) INTO p_num FROM obj$ o, user$ u
      WHERE o.name='ORDVIRB' AND u.name ='ORDSYS' AND o.owner# = u.user#;
      IF p_num > 0 THEN
        BEGIN
          INSERT INTO sys.registry$ 
                 (status, modified, cid, cname, version, 
                  invoker#, schema#, banner, flags, date_loaded)
          VALUES (s_loaded, SYSDATE, 'ORDVIR', 
                  'Oracle Visual Information Retrieval',
                  '8.0.6.0.0', p_invoker#, p_schema#, 
                  'Oracle Visual Information Retrieval', 0, SYSDATE);
          COMMIT;
        EXCEPTION
           WHEN DUP_VAL_ON_INDEX THEN NULL;
        END;
      END IF;      
  END IF;

-----------------------------------------------------------------------
-- Oracle Label Security
-----------------------------------------------------------------------

  BEGIN
    SELECT user# INTO p_schema# FROM user$ WHERE name='LBACSYS';
    o_id := 'OLS';
    o_name := 'Oracle Label Security';
    EXECUTE IMMEDIATE 'SELECT NLS_UPPER(version), installed ' ||
              'FROM lbacsys.lbac$installations ' ||
              'WHERE component = ''LBAC'''
      INTO o_version, o_modified;
    o_banner := o_name || ' ' || o_version || ' - Production'; 
    BEGIN
      INSERT INTO sys.registry$ 
            (status, modified, cid, cname, version, invoker#, schema#, 
             banner, flags, date_loaded)
         VALUES (s_loaded,o_modified, 
             o_id, o_name, o_version, p_invoker#, p_schema#, 
             o_banner, 0, SYSDATE);
      COMMIT;
    EXCEPTION
       WHEN DUP_VAL_ON_INDEX THEN NULL;
    END;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN NULL; /* no LBACSYS user or LBAC table */
  END;

  COMMIT;

-----------------------------------------------------------------------
-- Oracle Data Mining
-----------------------------------------------------------------------

 BEGIN
    SELECT user# INTO p_schema# FROM user$ WHERE name='ODM';
    SELECT NULL INTO g_null
    FROM obj$
    WHERE name = 'ODM_CONFIGURATION' AND type# = 2 AND
          owner# = p_schema#;
    o_id := 'ODM';
    o_name := 'Oracle Data Mining';
    o_version := '9.0.1.0.0'; 
    BEGIN 
        INSERT INTO registry$ 
                (status, modified, cid, cname, version, 
                 invoker#, schema#, banner, flags, date_loaded) 
         VALUES (s_loaded, SYSDATE, o_id, o_name, o_version, 
                 p_invoker#, p_schema#, 
                 o_name||' Release '||o_version||' - Production', 0, SYSDATE); 
         COMMIT;
     EXCEPTION 
         WHEN DUP_VAL_ON_INDEX THEN NULL; 
     END; 
  EXCEPTION
    WHEN NO_DATA_FOUND THEN NULL;  /* no ODM user or no ODM table */
  END;

-----------------------------------------------------------------------
-- Ultrasearch
-----------------------------------------------------------------------

 BEGIN
    SELECT user# INTO p_schema# FROM user$ WHERE name='WKSYS';
    SELECT NULL INTO g_null
    FROM obj$
    WHERE name = 'WK$SYSINFO' AND type# = 2 AND
          owner# = p_schema#;
    o_id := 'WK';
    o_name := 'Ultrasearch';
    o_version := '9.0.1.0.0'; 
    BEGIN 
        INSERT INTO registry$ 
                (status, modified, cid, cname, version, 
                 invoker#, schema#, banner, flags, date_loaded) 
         VALUES (s_loaded, SYSDATE, o_id, o_name, o_version, 
                 p_schema#, p_schema#,  /* install schema is owner */
                 o_name||' Release '||o_version||' - Production', 0, SYSDATE); 
         COMMIT;
     EXCEPTION 
         WHEN DUP_VAL_ON_INDEX THEN NULL; 
     END; 
  EXCEPTION
    WHEN NO_DATA_FOUND THEN NULL;  /* no WKSYS user or no WK$ table table */
  END;


-----------------------------------------------------------------------
-- CWMLite
-----------------------------------------------------------------------

  BEGIN
    SELECT user# INTO p_schema# FROM user$ WHERE name='OLAPSYS';
    SELECT NULL INTO g_null
    FROM obj$ 
    WHERE name = 'CWM$DIMENSION' AND type# = 2 AND
          owner# = p_schema#;
    o_id := 'AMD';
    o_name := 'OLAP Catalog';
    o_version := '9.0.1.0.0'; 
    BEGIN 
        INSERT INTO registry$ 
                (status, modified, cid, cname, version, 
                 invoker#, schema#, banner, flags, date_loaded) 
         VALUES (s_loaded, SYSDATE, o_id, o_name, o_version, 
                 p_invoker#, p_schema#,  /* install schema is owner */
                 o_name||' Release '||o_version||' - Production', 0, SYSDATE); 
         COMMIT;
     EXCEPTION 
         WHEN DUP_VAL_ON_INDEX THEN NULL; 
     END; 
  EXCEPTION
    WHEN NO_DATA_FOUND THEN NULL;  /* no OLAPSYS user or no CWM table */
  END;

-----------------------------------------------------------------------
-- Messaging Gateway
-----------------------------------------------------------------------

 BEGIN
    SELECT NULL INTO g_null
    FROM obj$
    WHERE name = 'MGW$_GATEWAY' AND type# = 2 AND
          owner# = p_invoker#;
    o_id := 'MGW';
    o_name := 'Messaging Gateway';
    o_version := '9.0.1.0.0'; 
    BEGIN 
        INSERT INTO registry$ 
                (status, modified, cid, cname, version, 
                 invoker#, schema#, banner, flags, date_loaded) 
         VALUES (s_loaded, SYSDATE, o_id, o_name, o_version, 
                 p_invoker#, p_invoker#, 
                 o_name||' Release '||o_version||' - Production', 0, SYSDATE); 
         COMMIT;
     EXCEPTION 
         WHEN DUP_VAL_ON_INDEX THEN NULL; 
     END; 
  EXCEPTION
    WHEN NO_DATA_FOUND THEN NULL;  /* no MGW table in SYS schema*/
  END;

  COMMIT;

END populate;

----------------------------- VALIDATE_COMPONENTS -------------------------

PROCEDURE validate_components
IS

BEGIN
   /* don't include removing, removed, downgraded, downgrading, option off */
   FOR comp IN (SELECT * FROM registry$ 
               WHERE status NOT IN (99,6,7,8,9)) LOOP
     
      IF comp.vproc IS NOT NULL THEN
         BEGIN
           EXECUTE IMMEDIATE 'BEGIN SYS.' || comp.vproc || '; END;';
         EXCEPTION
           WHEN OTHERS THEN NULL;
         END;
      END IF;
   END LOOP;

END validate_components;

END dbms_registry_sys;
/
show errors

Rem
Rem Set up drop user invocation
Rem

DELETE FROM sys.duc$ WHERE owner='SYS' AND pack='DBMS_REGISTRY_SYS';
INSERT INTO sys.duc$ (owner, pack, proc, operation#, seq, com)
  VALUES ('SYS','DBMS_REGISTRY_SYS','DROP_USER',1, 1,
          'Delete registry entries when schema or invoker is dropped');
COMMIT;
