Rem Copyright (c) 1987, 2006, Oracle. All rights reserved.  
Rem NAME
Rem    CATMETA.SQL - Object views of the Oracle dictionary for Metadata API.
Rem  FUNCTION
Rem     Creates an object model of the Oracle dictionary for use by the
Rem     DataPump's Metadata API.
Rem  NOTES
Rem     Must be run when connected to SYS or INTERNAL.
Rem     IMPORTANT! Keep catnomet.sql (used during downgrade) in synch with
Rem             this file.
Rem
Rem     All types must have EXECUTE granted to PUBLIC.
Rem     All top-level views used by the mdAPI to actually fetch full object
Rem     metadata (eg, KU$_TABLE_VIEW) must have SELECT granted to PUBLIC, but
Rem     must have CURRENT_USERID checking security clause.
Rem     All views subordinate to the top level views (eg, KU$_SCHEMAOBJ_VIEW)
Rem     must have SELECT granted to SELECT_CATALOG_ROLE.
Rem
Rem  MODIFIED
Rem     lbarton    05/16/06 - barton_rfi_backport_2816302_9.2.0.8
Rem     lbarton    03/22/06 - lbarton_rfi_backport_2944274_9.2.0.8: bitmap 
Rem                           join indexes 
Rem     araghava   11/11/05 - 4719142: rfi backport of 3448802 
Rem     lbarton    03/30/05 - backport bug 2844111: get_source_lines 
Rem     bmccarth   01/26/05 - xbranchmerge to 'st_rdbms_9.2.0.1' - Add 
Rem                           transportable_classic - and a few notes
Rem     lbarton    01/13/05 - lbarton_backport_9.2.0.6.0_3102211
Rem     lbarton    01/03/05 - backport 3225530: move get_domidx_metadata to 
Rem                           dbms_metadata 
Rem     htseng     12/16/04 - htseng_backport_9.2.0.6.0_353476
Rem     htseng     04/29/04 - htseng_backport_9.2.0.5.0_3534765
Rem     htseng     04/07/04 - bug 3534765 : comment with single quotes
Rem     htseng     09/29/03 - backport 3116063/3056720 use long2clob for view 
Rem			      and MV 
Rem     htseng     09/23/03 - backport 3108046 materialized view log
Rem     lbarton    05/14/03 - Domain index support
Rem     lbarton    05/12/03 - Bug 2818850: fetch null$ in prim_column
Rem     htseng     06/27/03 - bug 2967731:Trigger body > 4000
Rem     htseng     10/11/02 - fix bug 2616047 - when query length > 4000 
Rem                           in ku$_m_view_t
Rem     bmccarth   08/08/02 - change strip_username to emit_Schema
Rem     lbarton    07/24/02 - xmlschema work
Rem     lbarton    06/11/02 - add column lists to inserts
Rem     jdavison   04/27/02 - Fix termination on grant stmt
Rem     bmccarth   03/01/02 - add schemaoid to xmlschema
Rem     lbarton    02/14/02 - Fix iot issue
Rem     emagrath   01/31/02 - Complete support for REF constraints
Rem     lbarton    12/13/01 - filter out empty AUDIT_OBJ docs
Rem     dgagne     12/06/01 - remove not-null constraints from constraint view
Rem     dgagne     12/11/01 - fix security views so only DBA can use
Rem     htseng     11/05/01 - privilege checking.
Rem     htseng     10/22/01 - change AQ. to AQ_
Rem     lbarton    10/25/01 - xdb support
Rem     vmarwah    10/18/01 - LOB retention compatibility changes.
Rem     htseng     10/18/01 - change privilege for ku$_user_view..
Rem     lbarton    10/05/01 - Support Ordered Collections in Tables
Rem     vmarwah    10/08/01 - renaming spare1 and spare2 fields for LOB$.
Rem     lbarton    10/10/01 - Improve performance of mviews and mview logs
Rem     htseng     09/24/01 - fix missing kujob.xsl.
Rem     htseng     09/17/01 - add syn_long_name for java_source.
Rem     htseng     09/04/01 - add more objects support.
Rem     dgagne     09/14/01 - add subpartition template support
Rem     lbarton    09/24/01 - tablespace filter
Rem     dgagne     09/05/01 - add support for range/list composite support
Rem     lbarton    09/05/01 - package reorg
Rem     lbarton    07/11/01 - tweak query on source$ for perf
Rem     dgagne     06/14/01 - add more object support
Rem     lbarton    01/19/01 - add BUILTIN_COL filter
Rem     lbarton    01/12/01 - add comments
Rem     lbarton    01/10/01 - add USER_COL to constraint_col_t
Rem     lbarton    01/08/01 - support 8.1 compatibility
Rem     lbarton    12/01/00 - bugfix
Rem     gclaborn   11/22/00 - Update base dir for stylesheets
Rem     lbarton    11/10/00 - support long views
Rem     lbarton    10/24/00 - sort in views, not in xsl
Rem     lbarton    10/13/00 - bugfixes
Rem     gclaborn   11/03/00 - Change xsl script names to kufoo format
Rem     lbarton    10/05/00 - synonym support
Rem     gclaborn   10/11/00 - perf. enhancements
Rem     svivian    09/14/00 - add support for logical instantiation
Rem     lbarton    09/13/00 - version strings; new transform params
Rem     lbarton    08/18/00 - functions now in dbms_metadata_int
Rem     elu        08/15/00 - add replication xsl scripts for flavors
Rem     lbarton    07/28/00 - datapump: add metastylesheet
Rem     lbarton    06/23/00 - Multinested collections; populate dictionary
Rem     lbarton    06/12/00 - facility name change
Rem     rmurthy    06/20/00 - change objauth.option column to hold flag bits
Rem     gclaborn   05/04/00 - Update operators/indextypes to use multinesting
Rem     lbarton    05/18/00 - bugfix: piotable_view
Rem     gclaborn   04/24/00 - Update outline metadata to new 8.2 format
Rem     gclaborn   04/06/00 - Add outline support
Rem     lbarton    03/30/00 - Changes for partitioning
Rem     lbarton    03/17/00 - multinested collections
Rem     lbarton    03/01/00 - new table views, etc.
Rem     gclaborn   12/29/99 - Remove ref. to dbms_metadata_int
Rem     lbarton    12/03/99 - domain index support
Rem     gclaborn   11/24/99 - Add trigger support
Rem     akalra/lbarton/gclaborn
Rem                11/17/99 -  created


-------------------------------------------------------------------------------
--                              SCHEMA OBJECTS
-------------------------------------------------------------------------------

-- ADT for schema objects.  Based on obj$
create or replace type ku$_schemaobj_t as object
(
  obj_num       number,                                     /* object number */
  dataobj_num   number,                          /* data layer object number */
  owner_num     number,                                 /* owner user number */
  owner_name    varchar2(30),                                  /* owner name */
  name          varchar2(30),                                 /* object name */
  namespace     number,                               /* namespace of object */
  subname       varchar2(30),                     /* subordinate to the name */
  type_num      number,                                       /* object type */
  type_name     varchar2(30),                            /* object type name */
  ctime         date,                                /* object creation time */
  mtime         date,                               /* DDL modification time */
  stime         date,                   /* specification timestamp (version) */
  status        number,                                  /* status of object */
  remoteowner   varchar2(30),           /* remote owner name (remote object) */
  linkname      varchar2(128),                  /* link name (remote object) */
  flags         number,                                             /* flags */
  oid           raw(16),	/* OID for typed table, typed view, and type */
  spare1        number,                 
  spare2        number,
  spare3        number, 
  spare4        varchar2(1000),
  spare5        varchar2(1000),
  spare6        date 
)
/

grant execute on ku$_schemaobj_t  to public
/

-- view for schema objects
create or replace force view ku$_schemaobj_view of ku$_schemaobj_t
  with object identifier(obj_num) as
  select o.obj#, o.dataobj#, o.owner#, u.name, o.name, o.namespace, o.subname,
         o.type#,
         decode(o.type#,
                 0, 'CURSOR',                    1, 'INDEX',
                 2, 'TABLE',                     3, 'CLUSTER',
                 4, 'VIEW',                      5, 'SYNONYM',
                 6, 'SEQUENCE',                  7, 'PROCEDURE',
                 8, 'FUNCTION',                  9, 'PACKAGE',
                11, 'PACKAGE BODY',             12, 'TRIGGER',
                13, 'TYPE',                     14, 'TYPE BODY',
                15, 'OBJECT',                   16, 'USER',
                17, 'LINK',                     18, 'PIPE',
                19, 'TABLE PARTITION',          20, 'INDEX PARTITION',
                21, 'LOB',                      22, 'LIBRARY',
                23, 'DIRECTORY',                24, 'QUEUE',
                25, 'IOT',                      26, 'REPLICATION OBJECT GROUP',
                27, 'REPLICATION PROPAGATOR',   28, 'JAVA SOURCE',
                29, 'JAVA CLASS',               30, 'JAVA RESOURCE',
                31, 'JAVA JAR',                 32, 'INDEXTYPE',
                33, 'OPERATOR',                 34, 'TABLE SUBPARTITION',
                35, 'INDEX SUBPARTITION',       36, 'SERVER-SIDE REPAPI',
                37, 'REPLICATION INTERNAL PACKAGE',
                38, 'CONTEXT POLICY OBJECT',
                39, 'PUB_SUB INTERNAL INFORMATION',
                40, 'LOB PARTITION',            41, 'LOB SUBPARTITION',
                42, 'SUMMARY',                  43, 'DIMENSION',
                44, 'APP CONTEXT',              45, 'STORED OUTLINE',
                46, 'RULESET OBJECT',           47, 'SCHEDULER PLAN',
                48, 'SCHEDULER CLASS',          49, 'PENDING SCHEDULER PLAN',
                50, 'PENDING SCHEDULER CLASS',  51, 'SUBSCRIPTION INFORMATION',
                52, 'LOCATION INFORMATION',     53, 'REMOTE OBJECTS INFO',
                54, 'REPAPI SNAPSHOT METADATA', 55, 'IFS DATA',
                56, 'JAVA SHARED DATA',         57, 'SECURITY PROFILE',
                58, 'TRANSFORMATION',          'ERROR'),
         o.ctime, o.mtime, o.stime, o.status, o.remoteowner,
         o.linkname, o.flags, o.oid$, 
         o.spare1, o.spare2, o.spare3,
         o.spare4, o.spare5, o.spare6
  from obj$ o, user$ u
  where o.owner# = u.user#;

grant select on ku$_schemaobj_view to select_catalog_role
/

-------------------------------------------------------------------------------
--                              STORAGE
-------------------------------------------------------------------------------

-- ADT for storage characteristics
create or replace type ku$_storage_t as object
(
  file_num      number,                        /* segment header file number */
  block_num     number,                       /* segment header block number */
  type_num      number,                                      /* segment type */
  ts_num        number,                       /* tablespace for this segment */
  blocks        number,                /* blocks allocated to segment so far */
                                           /* zero for bitmapped tablespaces */
  extents       number,               /* extents allocated to segment so far */
  iniexts       number,                               /* initial extent size */
  minexts       number,                         /* minimum number of extents */
  maxexts       number,                         /* maximum number of extents */
  extsize       number,                          /* current next extent size */
                                           /* zero for bitmapped tablespaces */
  extpct        number,                             /* percent size increase */
  user_num      number,                        /* user who owns this segment */
  lists         number,                        /* freelists for this segment */
  groups        number,                  /* freelist groups for this segment */
  bitmapranges  number,                          /* ranges per bit map entry */
  cachehint     number,                                 /* hints for caching */
  scanhint      number,                                /* hints for scanning */
  hwmincr       number,                      /* Amount by which HWM is moved */
  flags         number,                                     /* Segment flags */
  spare2        number
)
/

grant execute on ku$_storage_t to public
/

-- view for storage ADT
create or replace force view ku$_storage_view of ku$_storage_t 
  with object identifier (file_num, block_num, ts_num) as
  select s.file#, s.block#, s.type#, s.ts#, s.blocks, s.extents,
         s.iniexts, s.minexts, s.maxexts, s.extsize, s.extpct,
         s.user#, decode(s.lists, 0, 1, s.lists), 
         decode(s.groups, 0, 1, s.groups), s.bitmapranges, s.cachehint,
         s.scanhint, s.hwmincr, s.spare1, s.spare2
  from seg$ s
/
grant select on ku$_storage_view to select_catalog_role
/

-------------------------------------------------------------------------------
--                              FILESPEC
-------------------------------------------------------------------------------

-- ADT for filespec

create or replace type ku$_file_t as object
(
  name          varchar2(513),                               /* raw filename */
  fname         varchar2(1026),                            /* fixed filename */
  fsize         number,                            /* size of file in blocks */
  maxextend     number,                                 /* maximum file size */
  inc           number,                                 /* increment ammount */
  ts_num        number                                  /* tablespace number */
)
/
grant execute on ku$_file_t to public
/
create or replace force view ku$_file_view of ku$_file_t
       with object identifier (name) as
        select  v.name,
                replace(v.name, '''', ''''''),
                f.blocks, f.maxextend, f.inc, f.ts#
        from    sys.v$dbfile v, sys.file$ f
        where   f.file# = v.file#
      union all
        select                                       /*+ ordered use_nl(hc) +*/
                v.fnnam,
                replace ((v.fnnam), '''', ''''''),
                DECODE(hc.ktfthccval, 0, hc.ktfthcsz, -1),
                DECODE(hc.ktfthccval, 0, hc.ktfthcmaxsz, NULL),
                DECODE(hc.ktfthccval, 0, hc.ktfthcinc, NULL),
                ts.ts#
        FROM    sys.x$kccfn v, sys.x$ktfthc hc, sys.ts$ ts
        WHERE   v.fntyp = 7 AND
                v.fnnam IS NOT NULL AND
                v.fnfno = hc.ktfthctfno AND
                hc.ktfthctsn(+) = ts.ts#
/
grant select on ku$_file_view to select_catalog_role
/

create or replace type ku$_file_list_t
  as table of ku$_file_t
/
grant execute on ku$_file_list_t to public
/

-------------------------------------------------------------------------------
--                              TABLESPACE
-------------------------------------------------------------------------------

-- ADT for tablespaces
create or replace type ku$_tablespace_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(2),                              /* UDT minor version # */
  ts_num        number,                      /* tablespace identifier number */
  name          varchar2(30),                          /* name of tablespace */
  owner_num     number,                               /* owner of tablespace */
  status        number,                                            /* status */
  contents      number,                      /* TEMPORARY = 1 / PERMANENT = 0*/
  undofile_num  number,  /* undo_off segment file number (status is OFFLINE) */
  undoblock_num number,               /* undo_off segment header file number */
  blocksize     number,                            /* size of block in bytes */
  inc_num       number,                      /* incarnation number of extent */
  scnwrp        number,     /* clean offline scn - zero if not offline clean */
  scnbas        number,              /* scnbas - scn base, scnwrp - scn wrap */
  dflminext     number,                 /* default minimum number of extents */
  dflmaxext     number,                 /* default maximum number of extents */
  dflinit       number,                       /* default initial extent size */
  dflincr       number,                          /* default next extent size */
  dflminlen     number,                       /* default minimum extent size */
  dflextpct     number,              /* default percent extent size increase */
  dflogging     number,                         /* default logging attribute */
  affstrength   number,                                 /* Affinity strength */
  bitmapped     number,      /* If not bitmapped, 0 else unit size in blocks */
  plugged       number,                                        /* If plugged */
  directallowed number,    /* Operation which invalidate standby are allowed */
  flags         number,                                     /* various flags */
                                     /* 0x01 = system managed allocation     */
                                     /* 0x02 = uniform allocation            */
                                /* if above 2 bits not set then user managed */
                                     /* 0x04 = migrated tablespace           */
                                     /* 0x08 = tablespace being migrated     */
                                     /* 0x10 = undo tablespace               */
                                     /* 0x20 = auto segment space management */
                       /* if above bit not set then freelist segment managed */
  pitrscnwrp    number,                      /* scn wrap when ts was created */
  pitrscnbas    number,                      /* scn base when ts was created */
  ownerinstance varchar(30),                          /* Owner instance name */
  backupowner   varchar(30),                   /* Backup owner instance name */
  groupname     varchar(30),                                   /* Group name */
  spare1        number,                                  /* plug-in SCN wrap */
  spare2        number,                                  /* plug-in SCN base */
  spare3        varchar2(1000),
  spare4        date,
  filespec      ku$_file_list_t
)
/

grant execute on ku$_tablespace_t to public
/

-- view for tablespaces
create or replace force view ku$_tablespace_view of ku$_tablespace_t
  with object identifier (ts_num) as
  select '1', '0',
	  t.ts#, t.name,  t.owner#, t.online$, t.contents$, t.undofile#, 
 	  t.undoblock#, t.blocksize, t.inc#, t.scnwrp, t.scnbas, t.dflminext,
	  t.dflmaxext, t.dflinit, t.dflincr, t.dflminlen, t.dflextpct, 
	  t.dflogging, t.affstrength, t.bitmapped, t.plugged, t.directallowed,
	  t.flags, t.pitrscnwrp, t.pitrscnbas, t.ownerinstance, t.backupowner,
	  t.groupname, t.spare1, t.spare2, t.spare3, t.spare4,      
          cast(multiset(select value(f) from ku$_file_view f
                       where f.ts_num = t.ts#
                      ) as ku$_file_list_t
             )
 from ts$ t
 where  (SYS_CONTEXT('USERENV','CURRENT_USERID') = 0
	OR EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_tablespace_view to public
/

-------------------------------------------------------------------------------
--                              TYPE
-------------------------------------------------------------------------------

-- adt for types
-- Note: ku$_source_list_t is defined in dbmsmetu.sql
create or replace type ku$_type_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(2),                              /* UDT minor version # */
  obj_num       number,                                     /* object number */
  schema_obj    ku$_schemaobj_t,                            /* schema object */
  oid           raw(16),	                                     /* toid */
  source_lines  ku$_source_list_t                            /* source lines */
)
/

grant execute on ku$_type_t to public
/

create or replace force view ku$_type_view of ku$_type_t
  with object identifier (obj_num) as
  select '1','1',
         oo.obj#,
         value(o),
         oo.oid$,
         sys.dbms_metadata_util.get_source_lines(oo.name,oo.obj#,oo.type#)
  from sys.obj$ oo, ku$_schemaobj_view o, type$ t
  where oo.type# = 13
    and oo.obj#  = o.obj_num
    and oo.subname is null      /* latest type version */
    and oo.oid$ = t.toid
        /* type$ properties bits:
           262144=0     - latest type version
           other bits=0 - not system-generated type
        */
    and bitand(t.properties,262144+2048+64+16)=0
         AND (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num, 0) OR 
              EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_type_view to public
/

create or replace type ku$_type_body_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(2),                              /* UDT minor version # */
  obj_num       number,                                     /* object number */
  schema_obj    ku$_schemaobj_t,                            /* schema object */
  source_lines  ku$_source_list_t                            /* source lines */
)
/

grant execute on ku$_type_body_t to public
/

create or replace force view ku$_type_body_view of ku$_type_body_t
  with object identifier (obj_num) as
  select '1','1',
         oo.obj#,
         value(o),
         sys.dbms_metadata_util.get_source_lines(oo.name,oo.obj#,oo.type#)
  from sys.obj$ oo, ku$_schemaobj_view o
  where oo.type# = 14
    and oo.obj#  = o.obj_num
         AND (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num, 0) OR 
              EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_type_body_view to public
/

create or replace type ku$_full_type_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(2),                              /* UDT minor version # */
  obj_num       number,                                     /* object number */
  schema_obj    ku$_schemaobj_t,                            /* schema object */
  type_t        ku$_type_t,                                   /* type object */
  type_body_t   ku$_type_body_t                          /* type body object */
)
/

grant execute on ku$_full_type_t to public
/

create or replace force view ku$_full_type_view of ku$_full_type_t
  with object identifier (obj_num) as
  select '1','0',
         oo.obj#,
         value(o),
         value(t),
         (select value(tb) from ku$_type_body_view tb
          where oo.name  = tb.schema_obj.name
          and o.owner_name  = tb.schema_obj.owner_name)
  from sys.obj$ oo, ku$_schemaobj_view o,
        ku$_type_view t
  where oo.type# = 13
    and oo.obj#  = o.obj_num
    and oo.obj#  = t.schema_obj.obj_num
         AND (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num, 0) OR 
              EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_full_type_view to public
/

-------------------------------------------------------------------------------
--                              SIMPLE COLUMNS
-------------------------------------------------------------------------------

-- UDT for simple column info. This is the foundation for a number of different
-- column variants and is sufficient for simple column name lists.

create or replace type ku$_simple_col_t as object
(
  obj_num       number,                      /* object number of base object */
  col_num       number,                          /* column number as created */
  intcol_num    number,                            /* internal column number */
  segcol_num    number,                          /* column number in segment */
  property      number,                     /* column properties (bit flags) */
  name          varchar2(30),                              /* name of column */
  attrname      varchar2(4000),/* name of type attr. column: null if != type */
  type_num      number,                               /* data type of column */
  deflength     number,    /* virtual column text length (for func. indexes) */
  default_val   varchar2(4000)             /* virtual column expression text */
)
/
grant execute on ku$_simple_col_t to public
/
create or replace type ku$_simple_col_list_t
 as table of ku$_simple_col_t
/
grant execute on ku$_simple_col_list_t to public
/
-- view to get a simple set of column attributes.
-- NOTE: Originally we had just an attribute called 'name' that was either
-- c.name or attrcol$.name selected via a DECODE of c.property bit 1.
-- However, for an as yet unexplained reason, this causes a full table scan
-- on col$ in outer views that used this view.

create or replace force view ku$_simple_col_view of ku$_simple_col_t 
  with object identifier (obj_num,intcol_num) as
  select c.obj#,
         c.col#,
         c.intcol#,
         c.segcol#,
         c.property,
         c.name,
         (select a.name from attrcol$ a where
                        a.obj#=c.obj# and a.intcol#=c.intcol#),
	 c.type#,
         c.deflength,
         sys.dbms_metadata_util.long2varchar(c.deflength,
                                        'SYS.COL$',
                                        'DEFAULT$',
                                        c.rowid)
  from col$ c
/
grant select on ku$_simple_col_view to select_catalog_role
/

-- view to get simple column information for a nested table setid column.
-- used soley for constraint column name resolution - sets the constraint
-- column name/attribute to the 'real' column/attribute name.

create or replace force view ku$_simple_setid_col_view of ku$_simple_col_t
  with object identifier (obj_num, intcol_num) as
  select c.obj#,
         c.col#,
         c.intcol#,
         c.segcol#,
         (c.property + BITAND(c2.property,1)),
         c2.name,
         (select a.name
          from attrcol$ a
          where a.obj# = c2.obj# and
                a.intcol# = c2.intcol#),
         c.type#,
         c.deflength,
         sys.dbms_metadata_util.long2varchar(c.deflength,
                                             'SYS.COL$',
                                             'DEFAULT$',
                                             c.rowid)
  from col$ c, col$ c2
  where BITAND(c.property, 1024) = 1024 and
        c2.obj# = c.obj# and
        c2.col# = c.col# and
        c2.intcol# = (c.intcol# - 1) and
        c2.segcol# = 0
/
grant select on ku$_simple_setid_col_view to select_catalog_role
/

-- view to get simple column information for a pkRef REF column.
-- used soley for constraint column name resolution - sets the constraint
-- column name/attribute to the 'real' column/attribute name for each
-- internal column referenced in the intcol#s ub2 intcol# array.
-- invoked only for columns with property REA (REF attribute).

create or replace force view ku$_simple_pkref_col_view of ku$_simple_col_t
  with object identifier (obj_num, intcol_num) as
  select c.obj#,
         c.col#,
         c.intcol#,
         c.segcol#,
         c.property,
         c2.name,
         (select a.name
          from attrcol$ a
          where a.obj# = c2.obj# and
                a.intcol# = c2.intcol#),
         c.type#,
         c.deflength,
         sys.dbms_metadata_util.long2varchar(c.deflength,
                                             'SYS.COL$',
                                             'DEFAULT$',
                                             c.rowid)
  from  col$ c, col$ c2, ccol$ cc, cdef$ cd, coltype$ ct   
  where cc.obj# = c.obj# and
        cc.intcol# = c.intcol# and
        cd.con# = cc.con# and
        ct.obj# = c.obj# and
        ct.col# = c.col# and
        ct.intcols = cd.intcols and
        UTL_RAW.CAST_TO_BINARY_INTEGER(
          SUBSTRB(ct.intcol#s, (cc.pos# * 2 - 1), 2), 3) = c.intcol# and
        c2.obj# = c.obj# and
        c2.intcol# = ct.intcol#
/

grant select on ku$_simple_pkref_col_view to select_catalog_role
/

-------------------------------------------------------------------------------
--                              INDEX COLUMNS
-------------------------------------------------------------------------------


-- ADT for index columns
create or replace type ku$_index_col_t as object
(
  obj_num       number,                               /* index object number */
  bo_num        number,                                /* base object number */
  intcol_num    number,                            /* internal column number */
  col           ku$_simple_col_t,                                  /* column */
  pos_num       number,                 /* column position number as created */
  segcol_num    number,                          /* column number in segment */
  segcollen     number,                      /* length of the segment column */
  offset        number,                                  /* offset of column */
  flags         number,                                             /* flags */
  spare2        number,
  spare3        number,
  spare4        varchar2(1000),
  spare5        varchar2(1000),
  spare6        date  
)
/

create or replace type ku$_index_col_list_t as table of ku$_index_col_t;
/

grant execute on ku$_index_col_t to public
/

grant execute on ku$_index_col_list_t to public
/

-- view for index columns
create or replace force view ku$_index_col_view of ku$_index_col_t
  with object identifier(obj_num, intcol_num) as
  select ic.obj#, ic.bo#, ic.intcol#,
          value(c), ic.pos#, 
          ic.segcol#, ic.segcollength, ic.offset, ic.spare1,
          ic.spare2, ic.spare3, ic.spare4, ic.spare5, ic.spare6
  from ku$_simple_col_view c, ind$ i, icol$ ic
  where ic.bo#     = c.obj_num
  and   i.obj# = ic.obj#
  and ((bitand(i.property, 1024) = 1024 and c.intcol_num = ic.spare2)
   or  (bitand(i.property, 1024) = 0    and c.intcol_num = ic.intcol#))
/
grant select on ku$_index_col_view to select_catalog_role
/

-------------------------------------------------------------------------------
--                              LOB COLUMNS
-------------------------------------------------------------------------------
-- ADT for LOB columns in non-partitioned tables
create or replace type ku$_lob_t as object
(
  obj_num       number,                                /* obj# of base table */
  col_num       number,                                     /* column number */
  intcol_num    number,                            /* internal column number */
  schema_obj    ku$_schemaobj_t,                        /* LOB schema object */
  storage       ku$_storage_t,                          /* LOB storage       */
  ind_num       number,                                 /* LOB index obj #   */
  ts_name       varchar2(30),                             /* tablespace name */
  blocksize     number,                            /* size of block in bytes */
  chunk         number,                    /* oracle blocks in one lob chunk */
  pctversion    number,                                      /* version pool */
  flags         number,                                             /* flags */
                                                 /* 0x0001 = NOCACHE LOGGING */
                                               /* 0x0002 = NOCACHE NOLOGGING */
                                             /* 0x0008 = CACHE READS LOGGING */
                                           /* 0x0010 = CACHE READS NOLOGGING */
  property      number,                    /* 0x00 = user defined lob column */
                                    /* 0x01 = kernel column(s) stored as lob */
                                     /* 0x02 = user lob column with row data */
                                            /* 0x04 = partitioned LOB column */
                                   /* 0x0008 = LOB In Global Temporary Table */
                                          /* 0x0010 = Session-specific table */
  retention     number,                              /* LOB Retention period */
  freepools     number,	                            /* LOB segment FREEPOOLS */
  spare1        number,
  spare2        number,
  spare3        varchar2(255)
)
/
grant execute on ku$_lob_t to public
/

-- view for lobs in nonpartitioned tables
create or replace force view ku$_lob_view of ku$_lob_t
  with object OID(obj_num, intcol_num)
  as select l.obj#, l.col#, l.intcol#, 
        (select value(o) from ku$_schemaobj_view o
         where o.obj_num = l.lobj#),
        (select value(s) from ku$_storage_view s
         where s.file_num  = l.file#
         and   s.block_num = l.block#
         and   s.ts_num    = l.ts#),
        l.ind#, 
        (select ts.name from ts$ ts where l.ts# = ts.ts#),
        (select ts.blocksize from ts$ ts where l.ts# = ts.ts#),
        l.chunk, l.pctversion$, l.flags, l.property,
        l.retention, l.freepools, l.spare1, l.spare2, l.spare3
  from lob$ l
/
grant select on ku$_lob_view to select_catalog_role
/

-- ADT for table level defaults for LOBs (from partlob$)
create or replace type ku$_partlob_t as object
(
  obj_num       number,                                /* obj# of base table */
  intcol_num    number,                            /* internal column number */
  schema_obj    ku$_schemaobj_t,            /* LOB schema object (for lobj#) */
  defts_name    varchar2(30),                     /* default tablespace name */
  defblocksize  number,                    /* default size of block in bytes */
  defchunk      number,                    /* oracle blocks in one lob chunk */
  defpctversion number,                              /* default version pool */
  defflags      number,                                             /* flags */
                                                 /* 0x0001 = NOCACHE LOGGING */
                                               /* 0x0002 = NOCACHE NOLOGGING */
                                             /* 0x0008 = CACHE READS LOGGING */
                                           /* 0x0010 = CACHE READS NOLOGGING */
  defpro        number,                        /* default partition property */
                                             /* 0x02 = enable storage in row */
  definiexts    number,  /* default INITIAL extent size; NULL if unspecified */
  defextsize    number,     /* default NEXT extent size; NULL if unspecified */
  defminexts    number,           /* default MINEXTENTS; NULL if unspecified */
  defmaxexts    number,           /* default MAXEXTENTS; NULL if unspecified */
  defextpct     number,          /* default PCTINCREASE; NULL if unspecified */
  deflists      number,      /* default FREELISTS value; NULL if unspecified */
  defgroups     number,      /* default FREELIST GROUPS; NULL if unspecified */
  defbufpool    number,          /* default BUFFER_POOL; NULL if unspecified */
  spare1        number,
  spare2        number,
  spare3        number
)
/
grant execute on ku$_partlob_t to public
/

create or replace force view ku$_partlob_view of ku$_partlob_t
  with object OID(obj_num, intcol_num)
  as select l.tabobj#, l.intcol#, 
        (select value(o) from ku$_schemaobj_view o
         where o.obj_num = l.lobj#),
        (select ts.name from ts$ ts where l.defts# = ts.ts#),
        NVL(
          (select ts.blocksize from ts$ ts where l.defts# = ts.ts#),
          NVL(         /* should be avail. thru lobcompart, lobfrag if null */
            (select ts.blocksize
             from   ts$ ts, lobfrag$ lf
             where  l.lobj# = lf.parentobj# and
                    lf.ts# = ts.ts# and rownum < 2),
            (select ts.blocksize
             from   ts$ ts, lobcomppart$ lcp, lobfrag$ lf
             where  l.lobj# = lcp.lobj# and 
                    lcp.partobj# = lf.parentobj# and 
                    lf.ts# = ts.ts# and rownum < 2))),
        l.defchunk, l.defpctver$, l.defflags, l.defpro,
        l.definiexts, l.defextsize, l.defminexts, l.defmaxexts,
        l.defextpct, l.deflists, l.defgroups, l.defbufpool,
        l.spare1, l.spare2, l.spare3
  from partlob$ l
/

grant select on ku$_partlob_view to select_catalog_role
/

-- ADT for actual partition- or subpartition-level LOB attributes
create or replace type ku$_lobfrag_t as object
(
  obj_num       number,                            /* fragment object number */
  parent_obj_num number,                             /* parent object number */
  part_obj_num  number,                           /* partition object number */
  base_obj_num  number,                               /* obj# of base table */
  intcol_num    number,                            /* internal column number */
  part_num      number,                                  /* partition number */
  schema_obj    ku$_schemaobj_t,                        /* LOB schema object */
  storage       ku$_storage_t,                          /* LOB storage       */
  ind_num       number,                                 /* LOB index obj #   */
  ts_name       varchar2(30),                             /* tablespace name */
  blocksize     number,                            /* size of block in bytes */
  chunk         number,                    /* oracle blocks in one lob chunk */
  pctversion    number,                                      /* version pool */
  flags         number,                                             /* flags */
                                                 /* 0x0001 = NOCACHE LOGGING */
                                               /* 0x0002 = NOCACHE NOLOGGING */
                                             /* 0x0008 = CACHE READS LOGGING */
                                           /* 0x0010 = CACHE READS NOLOGGING */
  property      number,                    /* 0x00 = user defined lob column */
                                    /* 0x01 = kernel column(s) stored as lob */
                                     /* 0x02 = user lob column with row data */
                                            /* 0x04 = partitioned LOB column */
                                   /* 0x0008 = LOB In Global Temporary Table */
                                          /* 0x0010 = Session-specific table */
  spare1        number,
  spare2        number,
  spare3        number       
)
/
grant execute on ku$_lobfrag_t to public
/

create or replace type ku$_lobfrag_list_t as table of ku$_lobfrag_t
/
grant execute on ku$_lobfrag_list_t to public
/

-- view for partition-level LOB attributes
create or replace force view ku$_lobfrag_view of ku$_lobfrag_t
  with object OID(obj_num)
  as select lf.fragobj#, lf.parentobj#, lf.tabfragobj#,
        l.obj#, l.intcol#, lf.frag#,
        (select value(o) from ku$_schemaobj_view o
         where o.obj_num = lf.fragobj#),
        (select value(s) from ku$_storage_view s
         where s.file_num  = lf.file#
         and   s.block_num = lf.block#
         and   s.ts_num    = lf.ts#),
        lf.indfragobj#, 
        (select ts.name from ts$ ts where lf.ts# = ts.ts#),
        (select ts.blocksize from ts$ ts where lf.ts# = ts.ts#),
        lf.chunk, lf.pctversion$, lf.fragflags, lf.fragpro,
        lf.spare1, lf.spare2, lf.spare3
  from  lob$ l, lobfrag$ lf
        where l.lobj#=lf.parentobj#
/
grant select on ku$_lobfrag_view to select_catalog_role
/

-- view for partition-level LOB attributes in partitioned IOTs
create or replace force view ku$_piotlobfrag_view of ku$_lobfrag_t
  with object OID(obj_num)
  as select lf.fragobj#, lf.parentobj#, pl.tabobj#,
        l.obj#, l.intcol#, lf.frag#,
        (select value(o) from ku$_schemaobj_view o
         where o.obj_num = lf.fragobj#),
        (select value(s) from ku$_storage_view s
         where s.file_num  = lf.file#
         and   s.block_num = lf.block#
         and   s.ts_num    = lf.ts#),
        lf.indfragobj#, 
        (select ts.name from ts$ ts where lf.ts# = ts.ts#),
        (select ts.blocksize from ts$ ts where lf.ts# = ts.ts#),
        lf.chunk, lf.pctversion$, lf.fragflags, lf.fragpro,
        lf.spare1, lf.spare2, lf.spare3
  from  lob$ l, partlob$ pl, lobfrag$ lf
        where l.lobj#=lf.parentobj# and pl.lobj#=lf.parentobj#
/
grant select on ku$_piotlobfrag_view to select_catalog_role
/

-- view for subpartition-level LOB attributes
create or replace force view ku$_sublobfrag_view of ku$_lobfrag_t
  with object OID(obj_num)
  as select lf.fragobj#, lf.parentobj#, lf.tabfragobj#,
        l.obj#, l.intcol#, lf.frag#,
        (select value(o) from ku$_schemaobj_view o
         where o.obj_num = lf.fragobj#),
        (select value(s) from ku$_storage_view s
         where s.file_num  = lf.file#
         and   s.block_num = lf.block#
         and   s.ts_num    = lf.ts#),
        lf.indfragobj#, 
        (select ts.name from ts$ ts where lf.ts# = ts.ts#),
        (select ts.blocksize from ts$ ts where lf.ts# = ts.ts#),
        lf.chunk, lf.pctversion$, lf.fragflags, lf.fragpro,
        lf.spare1, lf.spare2, lf.spare3
  from  lob$ l, lobcomppart$ lc, lobfrag$ lf
        where lc.partobj#=lf.parentobj#
          and l.lobj#=lc.lobj#
/
grant select on ku$_sublobfrag_view to select_catalog_role
/

-- ADT for partition-level defaults in composite partitioned tables
-- (from lobcomppart$)
create or replace type ku$_lobcomppart_t as object
(
  obj_num       number,                             /* obj# of LOB partition */
  part_obj_num  number,                     /* table partition object number */
  part_num      number,                                  /* partition number */
  intcol_num    number,                            /* internal column number */
  schema_obj    ku$_schemaobj_t,            /* LOB schema object (for lobj#) */
  defts_name    varchar2(30),                     /* default tablespace name */
  defblocksize  number,                    /* default size of block in bytes */
  defchunk      number,                    /* oracle blocks in one lob chunk */
  defpctversion number,                              /* default version pool */
  defflags      number,                                             /* flags */
                                                           /* 0x0000 = CACHE */
                                                 /* 0x0001 = NOCACHE LOGGING */
                                               /* 0x0002 = NOCACHE NOLOGGING */
                                             /* 0x0008 = CACHE READS LOGGING */
                                           /* 0x0010 = CACHE READS NOLOGGING */
  defpro        number,                        /* default partition property */
                                             /* 0x02 = enable storage in row */
  definiexts    number,  /* default INITIAL extent size; NULL if unspecified */
  defextsize    number,     /* default NEXT extent size; NULL if unspecified */
  defminexts    number,           /* default MINEXTENTS; NULL if unspecified */
  defmaxexts    number,           /* default MAXEXTENTS; NULL if unspecified */
  defextpct     number,          /* default PCTINCREASE; NULL if unspecified */
  deflists      number,      /* default FREELISTS value; NULL if unspecified */
  defgroups     number,      /* default FREELIST GROUPS; NULL if unspecified */
  defbufpool    number,          /* default BUFFER_POOL; NULL if unspecified */
  spare1        number,
  spare2        number,
  spare3        number       
)
/
grant execute on ku$_lobcomppart_t to public
/

create or replace type ku$_lobcomppart_list_t as table of ku$_lobcomppart_t
/
grant execute on ku$_lobcomppart_list_t to public
/

create or replace force view ku$_lobcomppart_view of ku$_lobcomppart_t
  with object OID(obj_num)
  as select lc.partobj#, lc.tabpartobj#, lc.part#,
        (select l.intcol# from lob$ l where l.lobj#=lc.lobj#),
        (select value(o) from ku$_schemaobj_view o
         where o.obj_num = lc.partobj#),
        (select ts.name from ts$ ts where lc.defts# = ts.ts#),
        (select ts.blocksize from ts$ ts where lc.defts# = ts.ts#),
        lc.defchunk, lc.defpctver$, lc.defflags, lc.defpro,
        lc.definiexts, lc.defextsize, lc.defminexts, lc.defmaxexts,
        lc.defextpct, lc.deflists, lc.defgroups, lc.defbufpool,
        lc.spare1, lc.spare2, lc.spare3
  from lobcomppart$ lc
/
grant select on ku$_lobcomppart_view to select_catalog_role
/
Rem
Rem Lob information for template subpartition lob store as clause
Rem
create or replace type ku$_tlob_comppart_t as object
(
  base_objnum    number,                           /* object number of table */
  colname        varchar2(30),
  intcol_num     number,                      /* column number of lob column */
  spart_pos      number,                            /* subpartition position */
  flags          number,          /* Type of lob column - 1 varray, 2 opaque */
  lob_spart_name varchar2(34),          /* segment name for lob subpartition */
  ts_name        varchar2(30),          /* tablespace name (if any) assigned */
  ts_num         number                      /* tablespace (if any) assigned */
)
/
grant execute on ku$_tlob_comppart_t to public
/

create or replace type ku$_tlob_comppart_list_t as table of ku$_tlob_comppart_t
/

grant execute on ku$_tlob_comppart_list_t to public
/

create or replace force view ku$_tlob_comppart_view of ku$_tlob_comppart_t
  with object OID(base_objnum)
  as select  dspl.bo#, '"'||c.name||'"', dspl.intcol#,
             dspl.spart_position, dspl.flags, dspl.lob_spart_name,
             (select(t.name) from sys.ts$ t where t.ts# = dspl.lob_spart_ts#),
             dspl.lob_spart_ts#
     from    sys.col$ c, sys.defsubpartlob$ dspl
     where   dspl.bo# = c.obj#
         and dspl.intcol# = c.col#
/
grant select on ku$_tlob_comppart_view to select_catalog_role
/

-------------------------------------------------------------------------------
--                              PARTITIONS
-------------------------------------------------------------------------------

-- ADT for index partitions.  Based on indpart$
create or replace type ku$_ind_part_t as object
(
  obj_num       number,                                 /* obj# of partition */
  schema_obj    ku$_schemaobj_t,                            /* schema object */
  ts_name       varchar2(30),                                  /* tablespace */
  blocksize     number,                            /* size of block in bytes */
  storage       ku$_storage_t,                    /* storage characteristics */
  dataobj_num   number,                          /* data layer object number */
  base_obj_num  number,                                        /* base index */
  part_num      number,                                  /* partition number */
  hiboundlen    number,                     /* len. of high bound val. expr. */
  hiboundval    varchar2(4000),                                 /* text of " */
-- hiboundval   long,                                           /* text of " */
  flags         number,                                             /* flags */
  pct_free      number,                   /* min. free space %age in a block */
  pct_thres     number,                 /* iot overflow thresh. null if !iot */
  initrans      number,                         /* initial # of transactions */
  maxtrans      number,                            /* max. # of transactions */
  analyzetime   date,                             /* timestamp last analyzed */
  samplesize    number,                                     /* for histogram */
  rowcnt        number,                                         /* # of rows */
  blevel        number,                                      /* B-tree level */
  leafcnt       number,                             /* number of leaf blocks */
  distkey       number,                                /* # of distinct keys */
  lblkkey       number,                      /* av. # of leaf blocks per key */
  dblkkey       number,                      /* av. # of data blocks per key */
  clufac        number,                                 /* clustering factor */
  spare1        number,
  spare2        number,
  spare3        number,
  inclcol       number                     /* iot include col#, null if !iot */
)
/

grant execute on ku$_ind_part_t to public
/

create or replace type ku$_ind_part_list_t as table of ku$_ind_part_t
/

grant execute on ku$_ind_part_list_t to public
/

-- view for index partitions
create or replace force view ku$_ind_part_view of ku$_ind_part_t
  with object identifier(obj_num) as
  select ip.obj#, 
         (select value(so) from ku$_schemaobj_view so
          where so.obj_num = ip.obj#),
         ts.name, ts.blocksize, value(s), ip.dataobj#, ip.bo#,
         ip.part#, ip.hiboundlen, 
         sys.dbms_metadata_util.long2varchar(ip.hiboundlen,
                                    'SYS.INDPART$',
                                    'HIBOUNDVAL',
                                     ip.rowid),
         ip.flags, ip.pctfree$, ip.pctthres$,
         ip.initrans, ip.maxtrans, ip.analyzetime, ip.samplesize,
         ip.rowcnt, ip.blevel, ip.leafcnt, ip.distkey, ip.lblkkey,
         ip.dblkkey, ip.clufac, ip.spare1, ip.spare2, ip.spare3,
         ip.inclcol
  from  ku$_storage_view s, indpart$ ip, ts$ ts
  where ip.ts# = s.ts_num
        AND ip.file# = s.file_num
        AND ip.block# = s.block_num
        AND ip.ts# = ts.ts#
/
grant select on ku$_ind_part_view to select_catalog_role
/

-- ADT for PIOT partitions.  Based on indpart$
create or replace type ku$_piot_part_t as object
(
  obj_num       number,                                 /* obj# of partition */
  schema_obj    ku$_schemaobj_t,                            /* schema object */
  ts_name       varchar2(30),                                  /* tablespace */
  blocksize     number,                            /* size of block in bytes */
  storage       ku$_storage_t,                    /* storage characteristics */
  dataobj_num   number,                          /* data layer object number */
  base_obj_num  number,                                        /* base index */
  part_num      number,                                  /* partition number */
  hiboundlen    number,                     /* len. of high bound val. expr. */
  hiboundval    varchar2(4000),                                 /* text of " */
-- hiboundval   long,                                           /* text of " */
  lobs          ku$_lobfrag_list_t,                                  /* lobs */
  flags         number,                                             /* flags */
  pct_free      number,                   /* min. free space %age in a block */
  pct_thres     number,                 /* iot overflow thresh. null if !iot */
  initrans      number,                         /* initial # of transactions */
  maxtrans      number,                            /* max. # of transactions */
  analyzetime   date,                             /* timestamp last analyzed */
  samplesize    number,                                     /* for histogram */
  rowcnt        number,                                         /* # of rows */
  blevel        number,                                      /* B-tree level */
  leafcnt       number,                             /* number of leaf blocks */
  distkey       number,                                /* # of distinct keys */
  lblkkey       number,                      /* av. # of leaf blocks per key */
  dblkkey       number,                      /* av. # of data blocks per key */
  clufac        number,                                 /* clustering factor */
  spare1        number,
  spare2        number,
  spare3        number,
  inclcol       number                     /* iot include col#, null if !iot */
)
/

grant execute on ku$_piot_part_t to public
/

create or replace type ku$_piot_part_list_t as table of ku$_piot_part_t
/
grant execute on ku$_piot_part_list_t to public
/

-- view for PIOT partitions
create or replace force view ku$_piot_part_view of ku$_piot_part_t
  with object identifier(obj_num) as
  select ip.obj#, 
         (select value(so) from ku$_schemaobj_view so
          where so.obj_num = ip.obj#),
         ts.name, ts.blocksize, value(s), ip.dataobj#, ip.bo#,
         ip.part#, ip.hiboundlen, 
         sys.dbms_metadata_util.long2varchar(ip.hiboundlen,
                                    'SYS.INDPART$',
                                    'HIBOUNDVAL',
                                     ip.rowid),
         cast(multiset(select lf.* from ind$ i, ku$_piotlobfrag_view lf
                        where lf.part_num=ip.part#
                          and ip.bo#=i.obj# and i.bo#=lf.base_obj_num
                        order by lf.intcol_num
                      ) as ku$_lobfrag_list_t
             ),
         ip.flags, ip.pctfree$, ip.pctthres$,
         ip.initrans, ip.maxtrans, ip.analyzetime, ip.samplesize,
         ip.rowcnt, ip.blevel, ip.leafcnt, ip.distkey, ip.lblkkey,
         ip.dblkkey, ip.clufac, ip.spare1, ip.spare2, ip.spare3,
         ip.inclcol
  from  ku$_storage_view s, indpart$ ip, ts$ ts
  where ip.ts# = s.ts_num
        AND ip.file# = s.file_num
        AND ip.block# = s.block_num
        AND ip.ts# = ts.ts#
/
grant select on ku$_piot_part_view to select_catalog_role
/


-- ADT for table partitions.  Based on tabpart$
create or replace type ku$_tab_part_t as object
(
  obj_num       number,                                     /* object number */
  schema_obj    ku$_schemaobj_t,                            /* schema object */
  ts_name       varchar2(30),                                  /* tablespace */
  blocksize     number,                            /* size of block in bytes */
  storage       ku$_storage_t,                    /* storage characteristics */
  dataobj_num   number,                          /* data layer object number */
  base_obj_num  number,                                 /* base table object */
  part_num      number,                                  /* partition number */
  hiboundlen    number,                   /* length of high bound value expr */
  hiboundval    varchar2(4000),             /* text of high bound value expr */
-- hiboundval   long,                       /* text of high bound value expr */
  lobs          ku$_lobfrag_list_t,                                  /* lobs */
  pct_free      number,                   /* min. free space %age in a block */
  pct_used      number,                   /* min. used space %age in a block */
  initrans      number,                         /* initial # of transactions */
  maxtrans      number,                            /* max. # of transactions */
  flags         number,                                             /* flags */
  analyzetime   date,                             /* timestamp last analyzed */
  samplesize    number,                          /* samplesize for histogram */
  rowcnt        number,                                    /* number of rows */
  blkcnt        number,                                  /* number of blocks */
  empcnt        number,                            /* number of empty blocks */
  avgspc        number,                      /* average available free space */
  chncnt        number,                            /* number of chained rows */
  avgrln        number,                                /* average row length */
  spare1        number,
  spare2        number,
  spare3        number
)
/

grant execute on ku$_tab_part_t to public
/

create or replace type ku$_tab_part_list_t as table of ku$_tab_part_t
/

grant execute on ku$_tab_part_list_t to public
/

-- view for table partitions
create or replace force view ku$_tab_part_view of ku$_tab_part_t
  with object identifier(obj_num) as
  select tp.obj#, value(o), ts.name, ts.blocksize,
         value(s), tp.dataobj#, tp.bo#,
         tp.part#, tp.hiboundlen,
         sys.dbms_metadata_util.long2varchar(tp.hiboundlen,
                                        'SYS.TABPART$',
                                        'HIBOUNDVAL',
                                        tp.rowid),
         cast(multiset(select * from ku$_lobfrag_view lf
                        where lf.part_obj_num=tp.obj#
                        order by lf.intcol_num
                      ) as ku$_lobfrag_list_t
             ),
         tp.pctfree$, tp.pctused$, tp.initrans,
         tp.maxtrans, tp.flags, tp.analyzetime, tp.samplesize, tp.rowcnt,
         tp.blkcnt, tp.empcnt, tp.avgspc, tp.chncnt, tp.avgrln, tp.spare1,
         tp.spare2, tp.spare3
  from ku$_schemaobj_view o, ku$_storage_view s, tabpart$ tp, ts$ ts
  where tp.obj# = o.obj_num
        AND tp.ts# = s.ts_num
        AND tp.file# = s.file_num
        AND tp.block# = s.block_num
        AND tp.ts# = ts.ts#
/
grant select on ku$_tab_part_view to select_catalog_role
/
-- ADT for table subpartitions.  Based on tabsubpart$.  These hang off
-- of their parents: table composite partitions.
create or replace type ku$_tab_subpart_t as object
(
  obj_num       number,                              /* obj# of subpartition */
  schema_obj    ku$_schemaobj_t,                            /* schema object */
  ts_name       varchar2(30),                             /* tablespace name */
  blocksize     number,                            /* size of block in bytes */
  storage       ku$_storage_t,                    /* storage characteristics */
  dataobj_num   number,                                   /* data layer obj# */
  pobj_num      number,                          /* obj# of parent partition */
  subpart_num   number,                                     /* subpartition# */
  lobs          ku$_lobfrag_list_t,                                  /* lobs */
  flags         number,                                             /* flags */
  pct_free      number,                      /* min. free space % in a block */
  pct_used      number,                      /* min. used spare % in a block */
  initrans      number,                         /* initial # of transactions */
  maxtrans      number,                            /* max. # of transactions */
  analyzetime   date,                        /* timestamp when last analyzed */
  samplesize    number,                      /* # of rows sampled by analyze */
  rowcnt        number,                                    /* number of rows */
  blkcnt        number,                                  /* number of blocks */
  empcnt        number,                            /* number of empty blocks */
  avgspc        number,                      /* average available free space */
  chncnt        number,                            /* number of chained rows */
  avgrln        number,                                /* average row length */
  spare1        number,
  spare2        number,
  spare3        number,
  hiboundlen    number,                     /* len. of high bound val. expr. */
  hiboundval    varchar2(4000)                                 /* text of " */
)
/

grant execute on ku$_tab_subpart_t to public
/

create or replace type ku$_tab_subpart_list_t as table of ku$_tab_subpart_t
/

grant execute on ku$_tab_subpart_list_t to public
/

-- view for table subpartitions
create or replace force view ku$_tab_subpart_view of ku$_tab_subpart_t
  with object identifier(obj_num) as
  select tsp.obj#, value(o), ts.name, ts.blocksize, value(s), tsp.dataobj#,
         tsp.pobj#, tsp.subpart#,
         cast(multiset(select * from ku$_sublobfrag_view lf
                        where lf.part_obj_num=tsp.obj#
                        order by lf.intcol_num
                      ) as ku$_lobfrag_list_t
             ),
         tsp.flags, tsp.pctfree$, tsp.pctused$,
         tsp.initrans, tsp.maxtrans, tsp.analyzetime, tsp.samplesize,
         tsp.rowcnt, tsp.blkcnt, tsp.empcnt, tsp.avgspc, tsp.chncnt,
         tsp.avgrln, tsp.spare1, tsp.spare2, tsp.spare3, tsp.hiboundlen,
         sys.dbms_metadata_util.long2varchar(tsp.hiboundlen,
                                    'SYS.TABSUBPART$',
                                    'HIBOUNDVAL',
                                     tsp.rowid)
  from ku$_schemaobj_view o, ku$_storage_view s, tabsubpart$ tsp, ts$ ts
  where tsp.obj# = o.obj_num
        AND tsp.ts# = s.ts_num
        AND tsp.file# = s.file_num
        AND tsp.block# = s.block_num
        AND tsp.ts# = ts.ts#
/
grant select on ku$_tab_subpart_view to select_catalog_role
/

-- ADT for table template subpartitions.  Based on defsubpart$.  These hang off
-- of their parents: table template subpartitions.
create or replace type ku$_tab_tsubpart_t as object
(
  base_objnum   number,                            /* Object number of table */
  spart_pos     number,                             /* subpartition position */
  spart_name    varchar2(34),                       /* name assigned by user */
  ts_name       varchar2(30),                            /* tablespace name */
  ts_num        number,                   /* Default tablespace NULL if none */
  flags         number,
  hiboundlen    number,              /* high bound text of this subpartition */
  hiboundval    varchar2(4000),                        /* length of the text */
  tlobs         ku$_tlob_comppart_list_t                             /* lobs */
)
/

grant execute on ku$_tab_tsubpart_t to public
/

create or replace type ku$_tab_tsubpart_list_t as table of ku$_tab_tsubpart_t
/

grant execute on ku$_tab_tsubpart_list_t to public
/

-- view for table template subpartitions

create or replace force view ku$_tab_tsubpart_view of ku$_tab_tsubpart_t
  with object identifier (base_objnum,spart_name,spart_pos) as
  select  dsp.bo#, dsp.spart_position, dsp.spart_name,
          (select( ts.name) from sys.ts$ ts where ts.ts# = dsp.ts#),
          dsp.ts#, dsp.flags, dsp.hiboundlen,
          sys.dbms_metadata_util.long2varchar( dsp.hiboundlen,
                                              'SYS.DEFSUBPART$',
                                              'HIBOUNDVAL',
                                              dsp.rowid),
          cast(multiset(select * from ku$_tlob_comppart_view tlcv
                        where  tlcv.base_objnum = dsp.bo#
                           and tlcv.spart_pos = dsp.spart_position
                        order by tlcv.intcol_num
                      ) as ku$_tlob_comppart_list_t)
  from    sys.defsubpart$ dsp
/
grant select on ku$_tab_tsubpart_view to select_catalog_role
/

-- ADT for table composite partitions.  Based on tabcompart$
create or replace type ku$_tab_compart_t as object
(
  obj_num       number,                           /* obj# of comp. partition */
  schema_obj    ku$_schemaobj_t,                            /* schema object */
  dataobj_num   number,                          /* data layer object number */
  base_obj_num  number,                                     /* obj# of table */
  part_num      number,                                  /* partition number */
  hiboundlen    number,             /* length of high bound value expression */
  hiboundval    varchar2(4000),       /* text of high-bound value expression */
  subpartcnt    number,                           /* number of subpartitions */
  subparts      ku$_tab_subpart_list_t,                     /* subpartitions */
  lobs          ku$_lobcomppart_list_t,                              /* lobs */
  flags         number,                                             /* flags */
  defts_name    varchar2(30),                          /* default TABLESPACE */
  defblocksize  number,          /* blocksize in bytes of default TABLESPACE */
  defpctfree    number,                                   /* default PCTFREE */
  defpctused    number,                                   /* default PCTUSED */
  definitrans   number,                                  /* default INITRANS */
  defmaxtrans   number,                                  /* default MAXTRANS */
  definiexts    number,  /* default INITIAL extent size; NULL if unspecified */
  defextsize    number,     /* default NEXT extent size; NULL if unspecified */
  defminexts    number,           /* default MINEXTENTS; NULL if unspecified */
  defmaxexts    number,           /* default MAXEXTENTS; NULL if unspecified */
  defextpct     number,          /* default PCTINCREASE; NULL if unspecified */
  deflists      number,      /* default FREELISTS value; NULL if unspecified */
  defgroups     number,      /* default FREELIST GROUPS; NULL if unspecified */
  deflogging    number,                        /* default LOGGING attribute  */
  defbufpool    number,                         /* default BUFFER_POOL value */
  analyzetime   date,                        /* timestamp when last analyzed */
  samplesize    number,                          /* samplesize for histogram */
  rowcnt        number,                                    /* number of rows */
  blkcnt        number,                                  /* number of blocks */
  empcnt        number,                            /* number of empty blocks */
  avgspc        number,                      /* average available free space */
  chncnt        number,                            /* number of chained rows */
  avgrln        number,                                /* average row length */
  spare1        number,                       
  spare2        number,                       
  spare3        number
)
/

grant execute on ku$_tab_compart_t to public
/

create or replace type ku$_tab_compart_list_t as table of ku$_tab_compart_t
/

grant execute on ku$_tab_compart_list_t to public
/

-- view for table composite partitions
create or replace force view ku$_tab_compart_view of ku$_tab_compart_t
  with object identifier(obj_num) as
  select tcp.obj#, value(o), tcp.dataobj#, tcp.bo#, tcp.part#,
         tcp.hiboundlen, 
         sys.dbms_metadata_util.long2varchar(tcp.hiboundlen,
                                    'SYS.TABCOMPART$',
                                    'HIBOUNDVAL',
                                    tcp.rowid),
         tcp.subpartcnt,
         cast(multiset(select * from ku$_tab_subpart_view tsp
                       where tsp.pobj_num = tcp.obj#
                        order by tsp.subpart_num
                      ) as ku$_tab_subpart_list_t
             ),
         cast(multiset(select * from ku$_lobcomppart_view lc
                       where lc.part_obj_num = tcp.obj#
                        order by lc.intcol_num
                      ) as ku$_lobcomppart_list_t
             ),
         tcp.flags, ts.name, ts.blocksize,
         tcp.defpctfree, tcp.defpctused, tcp.definitrans,
         tcp.defmaxtrans, tcp.definiexts, tcp.defextsize, tcp.defminexts,
         tcp.defmaxexts, tcp.defextpct, tcp.deflists, tcp.defgroups,
         tcp.deflogging, tcp.defbufpool, tcp.analyzetime, tcp.samplesize,
         tcp.rowcnt, tcp.blkcnt, tcp.empcnt, tcp.avgspc, tcp.chncnt,
         tcp.avgrln, tcp.spare1, tcp.spare2, tcp.spare3
  from ku$_schemaobj_view o, tabcompart$ tcp, ts$ ts
  where tcp.obj# = o.obj_num
        AND tcp.defts# = ts.ts#
/
grant select on ku$_tab_compart_view to select_catalog_role
/

-- ADT for index subpartitions.  Based in indsubpart$.  These hang off of
-- their parents: index composite partitions.
create or replace type ku$_ind_subpart_t as object
(
  obj_num       number,                              /* obj# of subpartition */
  schema_obj    ku$_schemaobj_t,                            /* schema object */
  ts_name       varchar2(30),                             /* tablespace name */
  blocksize     number,                            /* size of block in bytes */
  storage       ku$_storage_t,                                    /* storage */
  dataobj_num   number,                          /* data layer object number */
  pobj_num      number,                              /* parent object number */
  subpart_num   number,                               /* subpartition number */
  flags         number,                                             /* flags */
  pct_free      number,          /* minimum free space percentage in a block */
  initrans      number,                    /* initial number of transactions */
  maxtrans      number,                    /* maximum number of transactions */
  analyzetime   date,                        /* timestamp when last analyzed */
  samplesize    number,                          /* samplesize for histogram */
  rowcnt        number,                                    /* number of rows */
  blevel        number,                                      /* B-tree level */
  leafcnt       number,                             /* number of leaf blocks */
  distkey       number,                           /* number of distinct keys */
  lblkkey       number,             /* average number of leaf blocks per key */
  dblkkey       number,             /* average number of data blocks per key */
  clufac        number,                                 /* clustering factor */
  spare1        number,
  spare2        number,
  spare3        number
)
/

grant execute on ku$_ind_subpart_t to public
/

create or replace type ku$_ind_subpart_list_t as table of ku$_ind_subpart_t
/

grant execute on ku$_ind_subpart_list_t to public
/

-- view for index subpartitions
create or replace force view ku$_ind_subpart_view of ku$_ind_subpart_t
  with object identifier(obj_num) as
  select isp.obj#, value(o), ts.name, ts.blocksize, value(s), isp.dataobj#,
         isp.pobj#, isp.subpart#, isp.flags, isp.pctfree$, isp.initrans,
         isp.maxtrans, isp.analyzetime, isp.samplesize, isp.rowcnt,
         isp.blevel, isp.leafcnt, isp.distkey, isp.lblkkey, isp.dblkkey,
         isp.clufac, isp.spare1, isp.spare2, isp.spare3
  from ku$_schemaobj_view o, ku$_storage_view s, indsubpart$ isp, ts$ ts
  where isp.obj# = o.obj_num
        AND isp.ts# = s.ts_num
        AND isp.file# = s.file_num
        AND isp.block# = s.block_num
        AND isp.ts# = ts.ts#
/
grant select on ku$_ind_subpart_view to select_catalog_role
/

-- ADT for index composite partitions.  Based on indcompart$
create or replace type ku$_ind_compart_t as object
(
  obj_num       number,                           /* obj# of comp. partition */
  schema_obj    ku$_schemaobj_t,                            /* schema object */
  dataobj_num   number,                          /* data layer object number */
  base_obj_num  number,                                     /* obj# of table */
  part_num      number,                                  /* partition number */
  hiboundlen    number,             /* length of high bound value expression */
  hiboundval    varchar2(4000),       /* text of high-bound value expression */
  subpartcnt    number,                           /* number of subpartitions */
  subparts      ku$_ind_subpart_list_t,                     /* subpartitions */
  flags         number,                                     /* for any flags */
  defts_name    varchar2(30),     /* default TABLESPACE; NULL if unspecified */
  defblocksize  number,          /* blocksize in bytes of default TABLESPACE */
  defpctfree    number,                                   /* default PCTFREE */
  definitrans   number,                                  /* default INITRANS */
  defmaxtrans   number,                                  /* default MAXTRANS */
  definiexts    number,  /* default INITIAL extent size; NULL if unspecified */
  defextsize    number,     /* default NEXT extent size; NULL if unspecified */
  defminexts    number,           /* default MINEXTENTS; NULL if unspecified */
  defmaxexts    number,           /* default MAXEXTENTS; NULL if unspecified */
  defextpct     number,          /* default PCTINCREASE; NULL if unspecified */
  deflists      number,      /* default FREELISTS value; NULL if unspecified */
  defgroups     number,         /* default FREELIST GROUPS (N/A for indexes) */
  deflogging    number,                         /* default LOGGING attribute */
  defbufpool    number,                         /* default BUFFER_POOL value */
  analyzetime   date,                        /* timestamp when last analyzed */
  samplesize    number,                          /* samplesize for histogram */
  rowcnt        number,                                    /* number of rows */
  blevel        number,                                      /* B-tree level */
  leafcnt       number,                             /* number of leaf blocks */
  distkey       number,                           /* number of distinct keys */
  lblkkey       number,             /* average number of leaf blocks per key */
  dblkkey       number,             /* average number of data blocks per key */
  clufac        number,                                 /* clustering factor */
  spare1        number,
  spare2        number,
  spare3        number
)
/

grant execute on ku$_ind_compart_t to public
/

create or replace type ku$_ind_compart_list_t as table of ku$_ind_compart_t
/

grant execute on ku$_ind_compart_list_t to public
/

-- view for index composite partitions
create or replace force view ku$_ind_compart_view of ku$_ind_compart_t
  with object identifier(obj_num) as
  select icp.obj#, value(o), icp.dataobj#, icp.bo#, icp.part#,
         icp.hiboundlen, 
         sys.dbms_metadata_util.long2varchar(icp.hiboundlen,
                                    'SYS.INDCOMPART$',
                                    'HIBOUNDVAL',
                                    icp.rowid),
         icp.subpartcnt,
         cast(multiset(select * from ku$_ind_subpart_view isp
                       where isp.pobj_num = icp.obj#
                        order by isp.subpart_num
                      ) as ku$_ind_subpart_list_t
             ),
          icp.flags, 
          -- hoist the next 2 queries up here because icp.defts# may be null
          -- and this avoids an outer join which is slooooow
          (select ts.name from ts$ ts where icp.defts# = ts.ts#),
          (select ts.blocksize from ts$ ts where icp.defts# = ts.ts#),
          icp.defpctfree, icp.definitrans,
          icp.defmaxtrans, icp.definiexts, icp.defextsize, icp.defminexts,
          icp.defmaxexts, icp.defextpct, icp.deflists, icp.defgroups,
          icp.deflogging, icp.defbufpool, icp.analyzetime, icp.samplesize,
          icp.rowcnt, icp.blevel, icp.leafcnt, icp.distkey, icp.lblkkey,
          icp.dblkkey, icp.clufac, icp.spare1, icp.spare2, icp.spare3
  from ku$_schemaobj_view o, indcompart$ icp
  where icp.obj# = o.obj_num
/
grant select on ku$_ind_compart_view to select_catalog_role
/

-- ADT for partitioning columns.  Based on partcol$
create or replace type ku$_part_col_t as object
(
  obj_num       number,                      /* object number of base object */
  intcol_num    number,                            /* internal column number */
  col           ku$_simple_col_t,       /* the column object */
  pos_num       number,                 /* position of col. in key */
  spare1        number                  /* spare column */
)
/

grant execute on ku$_part_col_t to public
/

create or replace type ku$_part_col_list_t as table of ku$_part_col_t
/
grant execute on ku$_part_col_list_t to public
/

-- view for table partitioning columns

create or replace force view ku$_tab_part_col_view of
  ku$_part_col_t with object identifier (obj_num, intcol_num) as
  select pc.obj#, pc.intcol#, value(c), pc.pos#, pc.spare1
  from ku$_simple_col_view c, partcol$ pc
  where   pc.obj#=c.obj_num
  and     pc.intcol#=c.intcol_num
/
grant select on ku$_tab_part_col_view to select_catalog_role
/

-- view for table subpartitioning columns. Based on tabsubpart$. Also uses
-- ku$_part_col_t.

create or replace force view ku$_tab_subpart_col_view of
  ku$_part_col_t with object identifier(obj_num, intcol_num) as
  select sc.obj#, sc.intcol#, value(c), sc.pos#, sc.spare1
  from ku$_simple_col_view c, subpartcol$ sc
  where  sc.obj#=c.obj_num
  and    sc.intcol#=c.intcol_num
/
grant select on ku$_tab_subpart_col_view to select_catalog_role
/


-- view for index partitioning columns

create or replace force view ku$_ind_part_col_view of
  ku$_part_col_t with object identifier (obj_num, intcol_num) as
  select pc.obj#, pc.intcol#, value(c), pc.pos#, pc.spare1
  from ku$_simple_col_view c, ind$ i, partcol$ pc
  where   pc.obj#=i.obj#
  and     i.bo#=c.obj_num
  and     pc.intcol#=c.intcol_num
/
grant select on ku$_ind_part_col_view to select_catalog_role
/

-- view for index subpartitioning columns. Based on tabsubpart$. Also uses
-- ku$_part_col_t.

create or replace force view ku$_ind_subpart_col_view of
  ku$_part_col_t with object identifier(obj_num, intcol_num) as
  select sc.obj#, sc.intcol#, value(c), sc.pos#, sc.spare1
  from ku$_simple_col_view c, ind$ i, subpartcol$ sc
  where  sc.obj#=i.obj#
  and    i.bo#=c.obj_num
  and    sc.intcol#=c.intcol_num
/
grant select on ku$_ind_subpart_col_view to select_catalog_role
/

-- ADT for partitioned objects; included in table/index-specific types.
-- Based on partobj$

create or replace type ku$_partobj_t as object
(
  obj_num       number,                 /* obj# of partitioned tab. or index */
  parttype      number,                                 /* partitioning type */
  partcnt       number,                              /* number of partitions */
  partkeycols   number,                  /* # of columns in partitioning key */
  flags         number,                                             /* flags */
  defts_name    varchar2(30),                     /* default tablespace name */
  defblocksize  number,          /* blocksize in bytes of default TABLESPACE */
  defpctfree    number,                                   /* default PCTFREE */
  defpctused    number,                 /* default PCTUSED (N/A for indexes) */
  defpctthres   number,             /* default PCTTHRESHOLD (N/A for tables) */
  definitrans   number,                                  /* default INITRANS */
  defmaxtrans   number,                                  /* default MAXTRANS */
  definiexts    number,                       /* default INITIAL extent size */
  defextsize    number,                          /* default NEXT extent size */
  defminexts    number,                                /* default MINEXTENTS */
  defmaxexts    number,                                /* default MAXEXTENTS */
  defextpct     number,                               /* default PCTINCREASE */
  deflists      number,                           /* default FREELISTS value */
  defgroups     number,         /* default FREELIST GROUPS (N/A for indexes) */
  deflogging    number,           /* default logging attribute of the object */
  defbufpool    number,                         /* default BUFFER_POOL value */
  spare2        number,                              /* subpartitioning info */
  spare3        number,                                     /* spare column  */
  definclcol    number                           /* default iot include col# */
)
/

grant execute on ku$_partobj_t to public
/

-- view for partitioned objects
create or replace force view ku$_partobj_view 
  of ku$_partobj_t with object identifier (obj_num) as
  select po.obj#, po.parttype, po.partcnt,
         po.partkeycols,
         po.flags, 
         -- hoist the next 2 queries up here because po.defts# may be null
         -- and this avoids an outer join which is slooooow
         (select ts.name from ts$ ts where po.defts# = ts.ts#),
         (select ts.blocksize from ts$ ts where po.defts# = ts.ts#),
         po.defpctfree, po.defpctused, po.defpctthres,
         po.definitrans, po.defmaxtrans, po.deftiniexts, po.defextsize, 
         po.defminexts, po.defmaxexts, po.defextpct, po.deflists,
         po.defgroups, po.deflogging, po.spare1, po.spare2, po.spare3,
         po.definclcol
  from partobj$ po
/
grant select on ku$_partobj_view to select_catalog_role
/

-- ADT for partitioned tables

create or replace type ku$_tab_partobj_t as object
(
  obj_num       number,                         /* obj# of partitioned table */
  partobj       ku$_partobj_t,                     /* Base partitioning info */
  partcols      ku$_part_col_list_t,         /* list of partitioning columns */
  subpartcols   ku$_part_col_list_t,      /* list of subpartitioning columns */
  part_list     ku$_tab_part_list_t,                 /* table partition list */
  compart_list  ku$_tab_compart_list_t,    /* table composite partition list */
  tsubparts     ku$_tab_tsubpart_list_t        /* template subpartition list */
)
/
grant execute on ku$_tab_partobj_t to public
/
create or replace force view ku$_tab_partobj_view 
  of ku$_tab_partobj_t with object identifier (obj_num) as
  select po.obj_num, value(po),
         cast(multiset(select * from ku$_tab_part_col_view pc
                       where pc.obj_num = po.obj_num
                        order by pc.pos_num
                      ) as ku$_part_col_list_t
             ),
         cast(multiset(select * from ku$_tab_subpart_col_view sc
                       where sc.obj_num = po.obj_num
                        order by sc.pos_num
                      ) as ku$_part_col_list_t
             ),
         cast(multiset(select * from ku$_tab_part_view tp
                       where tp.base_obj_num = po.obj_num
                        order by tp.part_num
                      ) as ku$_tab_part_list_t
             ),
         cast(multiset(select * from ku$_tab_compart_view tcp
                       where tcp.base_obj_num = po.obj_num
                        order by tcp.part_num
                      ) as ku$_tab_compart_list_t
             ),
         cast(multiset(select * from ku$_tab_tsubpart_view ttsp
                       where ttsp.base_objnum = po.obj_num
                        order by ttsp.spart_pos
                      ) as ku$_tab_tsubpart_list_t
             )
  from ku$_partobj_view po
/
grant select on ku$_tab_partobj_view to select_catalog_role
/
-- ADT for partitioned indexes;

create or replace type ku$_ind_partobj_t as object
(
  obj_num       number,                 /* obj# of partitioned table */
  partobj       ku$_partobj_t,          /* Base partitioning info */
  partcols      ku$_part_col_list_t,         /* list of partitioning columns */
  subpartcols   ku$_part_col_list_t,      /* list of subpartitioning columns */
  part_list     ku$_ind_part_list_t,                 /* index partition list */
  compart_list  ku$_ind_compart_list_t    /* index composite partition list */
)
/
grant execute on ku$_ind_partobj_t to public
/
create or replace force view ku$_ind_partobj_view 
  of ku$_ind_partobj_t with object identifier (obj_num) as
  select po.obj_num, value(po),
         cast(multiset(select * from ku$_ind_part_col_view pc
                       where pc.obj_num = i.obj#
                        order by pc.pos_num
                      ) as ku$_part_col_list_t
             ),
         cast(multiset(select * from ku$_ind_subpart_col_view sc
                       where sc.obj_num = i.obj#
                        order by sc.pos_num
                      ) as ku$_part_col_list_t
             ),
         cast(multiset(select * from ku$_ind_part_view ip
                       where ip.base_obj_num = po.obj_num
                        order by ip.part_num
                      ) as ku$_ind_part_list_t
             ),
         cast(multiset(select * from ku$_ind_compart_view icp
                       where icp.base_obj_num = po.obj_num
                        order by icp.part_num
                      ) as ku$_ind_compart_list_t
             )
  from ind$ i, ku$_partobj_view po
        where i.obj#=po.obj_num
/
grant select on ku$_ind_partobj_view to select_catalog_role
/

-------------------------------------------------------------------------------
--                              DOMAIN INDEX
-------------------------------------------------------------------------------
-- ADT for domain index plsql code
create or replace type ku$_domidx_plsql_t as object
(
  obj_num       number,                 /* object # */
  plsql         ku$_vcnt                /* plsql code */
)
/

grant execute on  ku$_domidx_plsql_t to public
/

-- view for domain index plsql code
create or replace force view ku$_domidx_plsql_view of ku$_domidx_plsql_t
  with object identifier(obj_num) as
  select i.obj#,
        sys.dbms_metadata.get_domidx_metadata(o.name, u.name,
                o2.name, u2.name, 0)
   from obj$ o, obj$ o2, ind$ i, user$ u, user$ u2, indtypes$ it
   where i.type# = 9
         AND o.obj# = i.obj#
         AND o.owner# = u.user#
         AND i.indmethod# = it.obj#
         AND o2.obj# = it.implobj#
         AND o2.owner# = u2.user#
         AND bitand(i.property, 2) != 2         /* non-partitioned */
   UNION ALL
  select i.obj#,
        sys.dbms_metadata.get_domidx_metadata(o.name, u.name,
                o2.name, u2.name, 
                DECODE(BITAND (i.property, 512), 512, 64,0)+   /*0x200=iot di*/
                DECODE(BITAND(po.flags, 1), 1, 1, 0) +          /* 1 = local */
                DECODE(po.parttype, 1, 2, 2, 4, 0)    /* 1 = range, 2 = hash */
                )
   from obj$ o, obj$ o2, ind$ i, user$ u, user$ u2, partobj$ po, indtypes$ it
   where i.type# = 9
         AND o.obj# = i.obj#
         AND o.owner# = u.user#
         AND i.indmethod# = it.obj#
         AND o2.obj# = it.implobj#
         AND o2.owner# = u2.user#
         AND bitand(po.flags, 8) = 8            /* domain index */
         AND po.obj# = i.obj#
         AND bitand(i.property, 2) = 2          /* partitioned */
/
grant select on ku$_domidx_plsql_view to public
/

-------------------------------------------------------------------------------
--                              INDEX
-------------------------------------------------------------------------------

-- UDTs/views for bitmap join index information

create or replace type ku$_jijoin_table_t as object
(
  obj_num       number,                                          /* object # */
  tabobj_num    number,                                  /* table obj number */
  owner_name    varchar2(30),                                 /* table owner */
  name          varchar2(30)                                 /* table name  */
)
/
grant execute on ku$_jijoin_table_t to public
/
create or replace type ku$_jijoin_table_list_t as table of  ku$_jijoin_table_t
/
grant execute on ku$_jijoin_table_list_t to public
/
create or replace force view ku$_jijoin_table_view of ku$_jijoin_table_t
  with object identifier(obj_num,tabobj_num)
  as select j.obj#, o.obj_num, o.owner_name, o.name
  from sys.ku$_schemaobj_view o, sys.jijoin$ j
  where o.obj_num in (j.tab1obj#, j.tab2obj#)
  group by j.obj#, o.obj_num, o.owner_name, o.name
/
grant select on ku$_jijoin_table_view to select_catalog_role
/

create or replace type ku$_jijoin_t as object
(
  obj_num       number,                                          /* object # */
  tab1obj_num   number,                                /* table 1 obj number */
  tab1col_num   number,                /* internal column number for table 1 */
  tab2obj_num   number,                                /* table 2 obj number */
  tab2col_num   number,                /* internal column number for table 2 */
  tab1col       ku$_simple_col_t,                          /* table 1 column */
  tab2col       ku$_simple_col_t,                          /* table 2 column */
  joinop        number,                /* Op code as defined in opndef.h (=) */
  flags         number,                                        /* misc flags */
  tab1inst_num  number,           /* instance of table 1 (for multiple refs) */
  tab2inst_num  number            /* instance of table 2 (for multiple refs) */
)
/
grant execute on ku$_jijoin_t to public
/
create or replace type ku$_jijoin_list_t as table of ku$_jijoin_t
/
grant execute on ku$_jijoin_list_t to public
/
create or replace force view ku$_jijoin_view of ku$_jijoin_t
  with object identifier(obj_num,tab1obj_num,tab1col_num,tab2obj_num,tab2col_num)
  as select j.obj#, j.tab1obj#, j.tab1col#, j.tab2obj#, j.tab2col#,
            (select value(c) from sys.ku$_simple_col_view c
             where c.obj_num = j.tab1obj# and c.intcol_num = j.tab1col#),
            (select value(c) from sys.ku$_simple_col_view c
             where c.obj_num = j.tab2obj# and c.intcol_num = j.tab2col#),
            j.joinop, j.flags, j.tab1inst#, tab2inst#
  from sys.jijoin$ j
/
grant select on ku$_jijoin_view to select_catalog_role
/

-- UDT for indexes
create or replace type ku$_index_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(2),                              /* UDT minor version # */
  obj_num       number,                                          /* object # */
  schema_obj    ku$_schemaobj_t,                            /* schema object */
  col_list      ku$_index_col_list_t,               /* list of index columns */
  ts_name       varchar2(30),                                  /* tablespace */
  blocksize     number,                            /* size of block in bytes */
  storage       ku$_storage_t,                                    /* storage */
  dataobj_num   number,                          /* data layer object number */
  base_obj      ku$_schemaobj_t,                              /* base object */
  indmethod_num number,             /* object # for cooperative index method */
  indtype_name  varchar2(30),                              /* indextype name */
  indtype_owner varchar2(30),                             /* indextype owner */
  plsql_code    ku$_domidx_plsql_t,               /* domain index plsql code */
  jijoin_tabs   ku$_jijoin_table_list_t, /* jijoin tables if bitmap join idx */
  jijoin        ku$_jijoin_list_t,       /* jijoin data if bitmap join index */
  cols          number,                                 /* number of columns */
  pct_free      number,          /* minimum free space percentage in a block */
  initrans      number,                     /* initial number of transaction */
  maxtrans      number,                    /* maximum number of transactions */
  pct_thres     number,           /* iot overflow threshold, null if not iot */
  type_num      number,                       /* what kind of index is this? */
  flags         number,                                     /* mutable flags */
  property      number,             /* immutable flags for life of the index */
  blevel        number,                                       /* btree level */
  leafcnt       number,                                  /* # of leaf blocks */
  distkey       number,                                   /* # distinct keys */
  lblkkey       number,                          /* avg # of leaf blocks/key */
  dblkkey       number,                          /* avg # of data blocks/key */
  clufac        number,                                 /* clustering factor */
  analyzetime   date,                        /* timestamp when last analyzed */
  samplesize    number,                 /* number of rows sampled by Analyze */
  rowcnt        number,                       /* number of rows in the index */
  intcols       number,                        /* number of internal columns */
  degree        number,           /* # of parallel query slaves per instance */
  instances     number,             /* # of OPS instances for parallel query */
  trunccnt      number,                        /* re-used for iots 'inclcol' */
  numcolsdep    number,         /* number of columns depended on, >= intcols */
  numkeycols    number,             /* # of key columns in compressed prefix */
  part_obj      ku$_ind_partobj_t,                /* null if not partitioned */
  spare3        number,
  spare4        varchar2(1000),     /* used for parameter str for domain idx */
  spare5        varchar2(1000),
  spare6        date
)
/

grant execute on ku$_index_t to public
/

create or replace type ku$_index_list_t as table of ku$_index_t;
/

grant execute on ku$_index_list_t to public
/

-- view for indexes
create or replace force view ku$_index_view of ku$_index_t
  with object identifier(obj_num) as
  select '1','0',
         i.obj#, value(o),
         cast(multiset(select * from ku$_index_col_view ic
                       where ic.obj_num = i.obj#
                        order by ic.pos_num
                      ) as ku$_index_col_list_t
             ),
         ts.name, ts.blocksize,
         (select value(s) from ku$_storage_view s
          where i.file#  = s.file_num
          and   i.block# = s.block_num
          and   i.ts#    = s.ts_num),
         i.dataobj#,
         (select value(so) from ku$_schemaobj_view so
          where so.obj_num = i.bo#),
         i.indmethod#,
         (select o2.name from obj$ o2 where i.indmethod# = o2.obj#),
         (select u2.name from user$ u2 
                where u2.user# = (select o3.owner# from obj$ o3
                                        where i.indmethod# = o3.obj#)),
         (select value(pl) from ku$_domidx_plsql_view pl
                where pl.obj_num = i.obj#),
         -- include bitmap join index info if this is a bji
         decode(bitand(i.property, 1024), 1024,
           cast(multiset(select * from ku$_jijoin_table_view j
                        where j.obj_num = i.obj#
                        order by j.tabobj_num
                        ) as ku$_jijoin_table_list_t
                ),
           null),
         decode(bitand(i.property, 1024), 1024,
           cast(multiset(select * from ku$_jijoin_view j
                        where j.obj_num = i.obj#
                        order by j.tab1obj_num, j.tab2obj_num,
                                 j.tab1col_num, j.tab2col_num
                        ) as ku$_jijoin_list_t
                ),
           null),
         i.cols, i.pctfree$,
         i.initrans, i.maxtrans, i.pctthres$, i.type#, i.flags, i.property,
         i.blevel, i.leafcnt, i.distkey, i.lblkkey, i.dblkkey, i.clufac,
         i.analyzetime, i.samplesize, i.rowcnt, i.intcols, i.degree,
         i.instances, i.trunccnt, i.spare1, i.spare2,
         (select value(po) from ku$_ind_partobj_view po
          where i.obj# = po.obj_num),
         i.spare3, i.spare4, i.spare5, i.spare6
   from  ku$_schemaobj_view o, ind$ i, ts$ ts
   where o.obj_num = i.obj#
         AND  i.ts# = ts.ts#
         AND (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num, 0) OR 
              EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_index_view to public
/


-------------------------------------------------------------------------------
--                              CONSTRAINTS
-------------------------------------------------------------------------------
-- type for columns in constraint.  Only includes attributes
-- required to generate appropriate DDL.
create or replace type ku$_constraint_col_t as object
(
  con_num       number,                                 /* constraint number */
  obj_num       number,                      /* object number of base object */
  intcol_num    number,                            /* internal column number */
  pos_num       number,                 /* column position number as created */
  oid_or_setid  number, /* 1 = either OID column or nested table col's setid */
  col           ku$_simple_col_t                                  /* column */
)
/

grant execute on ku$_constraint_col_t to public
/

create or replace type ku$_constraint_col_list_t
 as TABLE of ku$_constraint_col_t;
/

-- view to get columns in a constraint.
create or replace force view ku$_constraint_col_view of ku$_constraint_col_t
  with object identifier (obj_num,intcol_num) as
  select cc.con#, cc.obj#, cc.intcol#, cc.pos#,
  decode(bitand(c.property,1024+2),0,0,1),
  decode(bitand(c.property,2097152+1024),
         2097152,(select value(c1) from ku$_simple_pkref_col_view c1
                  where c1.obj_num    = cc.obj# and
                        c1.intcol_num = cc.intcol#),
            1024,(select value(c2) from ku$_simple_setid_col_view c2
                  where c2.obj_num    = cc.obj# and
                        c2.intcol_num = cc.intcol#),
            value(c))
  from ku$_simple_col_view c, ccol$ cc
  where c.obj_num    = cc.obj#
    and c.intcol_num = cc.intcol#
/
grant select on ku$_constraint_col_view to select_catalog_role
/

grant execute on ku$_constraint_col_list_t to public
/

-- Types for column constraints. Based on con$ and cdef$.  Does not include
-- the spare columns in con$.
-- We distinguish 3 types of constraint:
-- (1) those where we don't need column information to generate DDL
--     (this is really just NOT NULL);
-- (2) those with one set of columns (primary key, unique key, CHECK) --
--     this kind of constraint is implemented with an index for prim/unique;
-- (3) those with 2 sets of columns (referential).

create or replace type ku$_constraint0_t as object
(
  owner_num     number,                                      /* owner user # */
  name          varchar2(30),                             /* constraint name */
  con_num       number,                                 /* constraint number */
  obj_num       number,                  /* object number of base table/view */
  numcols       number,                   /* number of columns in constraint */
  contype       number,                                   /* constraint type */
  enabled       number,           /* is constraint enabled? NULL if disabled */
  intcols       number,              /* #  of internal columns in constraint */
  mtime         date,      /* date this constraint was last enabled-disabled */
  flags         number                                              /* flags */
)
/

grant execute on ku$_constraint0_t to public
/

create or replace type ku$_constraint0_list_t as table of ku$_constraint0_t;
/

grant execute on ku$_constraint0_list_t to public
/

-- unique/primary key constraint

create or replace type ku$_constraint1_t as object
(
  owner_num     number,                                      /* owner user # */
  name          varchar2(30),                             /* constraint name */
  con_num       number,                                 /* constraint number */
  obj_num       number,                  /* object number of base table/view */
  numcols       number,                   /* number of columns in constraint */
  contype       number,                                   /* constraint type */
  enabled       number,           /* is constraint enabled? NULL if disabled */
  condlength    number,                 /* table check condition text length */
  condition     varchar2(4000),                /* table check condition text */
  intcols       number,              /* #  of internal columns in constraint */
  mtime         date,      /* date this constraint was last enabled-disabled */
  flags         number,                                             /* flags */
  col_list      ku$_constraint_col_list_t,                        /* columns */
  ind           ku$_index_t                                /* index metadata */
)
/

grant execute on ku$_constraint1_t to public
/

create or replace type ku$_constraint1_list_t as table of ku$_constraint1_t;
/

grant execute on ku$_constraint1_list_t to public
/

create or replace type ku$_constraint2_t as object
(
  owner_num     number,                                      /* owner user # */
  name          varchar2(30),                             /* constraint name */
  con_num       number,                                 /* constraint number */
  obj_num       number,                  /* object number of base table/view */
  numcols       number,                   /* number of columns in constraint */
  contype       number,                                   /* constraint type */
  robj_num      number,                 /* object number of referenced table */
  rcon_num      number,                 /* constraint# of referenced columns */
  rrules        varchar2(3),                 /* future: use this for pendant */
  match_num     number,                 /* referential constraint match type */
  refact        number,                                /* referential action */
  enabled       number,           /* is constraint enabled? NULL if disabled */
  intcols       number,              /* #  of internal columns in constraint */
  mtime         date,      /* date this constraint was last enabled-disabled */
  flags         number,                                             /* flags */
  schema_obj    ku$_schemaobj_t,                  /* referenced table object */
  src_col_list  ku$_constraint_col_list_t,                 /* source columns */
  tgt_col_list  ku$_constraint_col_list_t                  /* target columns */
)
/

grant execute on ku$_constraint2_t to public
/

create or replace type ku$_constraint2_list_t as table of ku$_constraint2_t;
/

grant execute on ku$_constraint2_list_t to public
/

-- views for constraints
--  This first view is for all but primary key, unique, check and referential.
-- In reality, it's only used for NOT NULL.
--  It has no column lists.

create or replace force view ku$_constraint0_view of ku$_constraint0_t
   with object identifier(con_num) as
   select c.owner#, c.name, c.con#, cd.obj#, cd.cols, cd.type#,
          NVL(cd.enabled,0),
          cd.intcols, cd.mtime, cd.defer
   from con$ c, cdef$ cd
   where c.con# = cd.con#
     and cd.type# not in (1,2,3,4)  -- not check (1), primary key (2),
                                    -- unique (3), referential (4)
/
grant select on ku$_constraint0_view to select_catalog_role
/

-- view for check, primary key/unique constraints: includes index metadata
-- for the latter two types.
create or replace force view ku$_constraint1_view of ku$_constraint1_t
   with object identifier(con_num) as
   select c.owner#, c.name, c.con#, cd.obj#, cd.cols, cd.type#,
          NVL(cd.enabled,0),
          cd.condlength, 
          sys.dbms_metadata_util.long2varchar(cd.condlength,
                                        'SYS.CDEF$',
                                        'CONDITION',
                                        cd.rowid),
          cd.intcols, cd.mtime, cd.defer,
          cast( multiset(select * from ku$_constraint_col_view col
                        where col.con_num = c.con#
                        order by col.pos_num
                        ) as ku$_constraint_col_list_t
                ),
          ( select value(i) from ku$_index_view i
                where i.obj_num=cd.enabled )
--              where i.schema_obj.owner_num=c.owner#
--                and i.schema_obj.name=c.name )
   from  con$ c, cdef$ cd
   where c.con# = cd.con#
     and cd.type# in (1,2,3)    -- check (1), primary key (2) or unique (3)
/
grant select on ku$_constraint1_view to select_catalog_role
/

-- view for referential constraints
create or replace force view ku$_constraint2_view of ku$_constraint2_t
   with object identifier(con_num) as
   select c.owner#, c.name, c.con#, cd.obj#, cd.cols, cd.type#,
          cd.robj#, cd.rcon#, cd.rrules, cd.match#, cd.refact, 
          NVL(cd.enabled,0),
          cd.intcols, cd.mtime, cd.defer,
          (select value(o) from ku$_schemaobj_view o 
                where o.obj_num = cd.robj#),
          cast( multiset(select * from ku$_constraint_col_view col
                        where col.con_num = c.con#
                        order by col.pos_num
                        ) as ku$_constraint_col_list_t
                ),
          cast( multiset(select * from ku$_constraint_col_view col
                        where col.con_num = cd.rcon#
                        order by col.pos_num
                        ) as ku$_constraint_col_list_t
                )
   from con$ c, cdef$ cd
   where c.con# = cd.con#
     and cd.type# = 4           -- referential constraint
/
grant select on ku$_constraint2_view to select_catalog_role
/


-- For all REF constraints
create or replace type ku$_pkref_constraint_t as object
(
  obj_num       number,                      /* object number of base object */
  col_num       number,                                     /* column number */
  intcol_num    number,                            /* internal column number */
  reftyp        number,                                     /* REF type flag */
                                                     /* 0x01 = REF is scoped */
                                             /* 0x02 = REF stored with rowid */
                                             /* 0x04 = Primary key based ref */
                            /* 0x08 = Unscoped primary key based ref allowed */
  property      number,                     /* column properties (bit flags) */
  name          varchar2(30),                              /* name of column */
  attrname      varchar2(4000),/* name of type attr. column: null if != type */
  schema_obj    ku$_schemaobj_t,                         /* referenced table */
  foreignkey    number,               /* 1= scoped REF is also a foreign key */
  pk_col_list   ku$_simple_col_list_t   /* any pkREF refd. pkey constr. cols */
)
/
grant execute on ku$_pkref_constraint_t to public
/

create or replace type ku$_pkref_constraint_list_t
 as table of ku$_pkref_constraint_t;
/

grant execute on ku$_pkref_constraint_list_t to public
/

create or replace force view ku$_pkref_constraint_view
 of ku$_pkref_constraint_t with object identifier(obj_num, intcol_num) as
 select rf.obj#, rf.col#, rf.intcol#, rf.reftyp, c.property, c.name,
        (select a.name from attrcol$ a
                 where a.obj#=rf.obj# and a.intcol#=rf.intcol#),
        (select value(o) from ku$_schemaobj_view o, obj$ oo
                 where rf.stabid = oo.oid$
                    and oo.obj#  = o.obj_num),
        nvl((select 1
             from coltype$ fct, ccol$ fcc, cdef$ fcd  
             where fct.obj# = rf.obj# and
                   fct.intcol# = rf.intcol# and
                   fcc.obj# = rf.obj# and
                   fcc.intcol# =
                     UTL_RAW.CAST_TO_BINARY_INTEGER(
                       SUBSTRB(fct.intcol#s, 1, 2), 3) and
                   fcd.con# = fcc.con# and
                   fcd.type# = 4), 0),
        decode(bitand(rf.reftyp,4),
                       4, cast(multiset
                                (select rc.*
                                 from   ku$_simple_col_view rc, ccol$ rcc 
                                 where  rcc.con# =
                                            (select con#
                                             from   obj$ ro, cdef$ rcd
                                             where  ro.oid$ = rf.stabid and
                                                    rcd.obj# = ro.obj# and
                                                    rcd.type# = 2)       and
                                          rc.obj_num = rcc.obj# and
                                          rc.intcol_num = rcc.intcol#
                                  order by rcc.pos#
                                ) as ku$_simple_col_list_t),
                       null)
 from refcon$ rf, col$ c
 where c.obj#=rf.obj# and c.intcol#=rf.intcol# 
/
grant select on ku$_pkref_constraint_view to select_catalog_role
/

create or replace type ku$_constraint_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(1),                              /* UDT minor version # */
  con_num       number,                                 /* constraint number */
  owner         varchar2(30),                         /* owner of constraint */
  name          varchar2(30),                          /* name of constraint */
  base_obj      ku$_schemaobj_t,            /* base table/view schema object */
  con0          ku$_constraint0_t,                       /* con 0 constraint */
  con1          ku$_constraint1_t                        /* con 1 constraint */
)
/
grant execute on ku$_constraint_t to public
/

create or replace force view ku$_constraint_view
  of ku$_constraint_t with object identifier(con_num) as
  select '1', '0',
         c.con#,
         u.name,
         c.name,
         value(sov),
         (select value(con) from ku$_constraint0_view con
          where  con.con_num = c.con#
             and con.contype not in (7,11)),
         (select value(con) from ku$_constraint1_view con
          where  con.con_num = c.con#)
  from  obj$ o, con$ c, cdef$ cd, user$ u, ku$_schemaobj_view sov
  where o.obj# = sov.obj_num and
        cd.obj# = o.obj# and
        c.con# = cd.con# and
        u.user# = c.owner# and
        cd.type# not in (4, 7, 11)
	AND (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (sov.owner_num, 0) 
	OR EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_constraint_view to public
/

-- For stand alone (ALTER TABLE) referential (foreign key) constraints
create or replace type ku$_ref_constraint_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(1),                              /* UDT minor version # */
  con_num       number,                                 /* constraint number */
  owner         varchar2(30),                         /* owner of constraint */
  name          varchar2(30),                          /* name of constraint */
  base_obj      ku$_schemaobj_t,            /* base table/view schema object */
  con2          ku$_constraint2_t                       /* type 2 constraint */
)
/
grant execute on ku$_ref_constraint_t to public
/

create or replace force view ku$_ref_constraint_view
  of ku$_ref_constraint_t with object identifier(con_num) as
  select '1', '0',
         c.con#,
         u.name,
         c.name,
         value(sov),
         value(con)
  from  obj$ o, con$ c, cdef$ cd, user$ u, ku$_schemaobj_view sov,
        ku$_constraint2_view con
  where o.obj# = sov.obj_num and
        cd.obj# = o.obj# and
        c.con# = cd.con# and
        u.user# = c.owner# and
	con.con_num = c.con# and
  	(SYS_CONTEXT('USERENV','CURRENT_USERID') IN (sov.owner_num, 0) 
	OR EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_ref_constraint_view to public
/

---
-------------------------------------------------------------------------------
--                              PRIMITIVE COLUMNS
-------------------------------------------------------------------------------

-- ADT for primitive columns (with builtin datatypes that are not LOBs)
create or replace type ku$_prim_column_t as object
(
  obj_num       number,                      /* object number of base object */
  col_num       number,                          /* column number as created */
  intcol_num    number,                            /* internal column number */
  segcol_num    number,                          /* column number in segment */
  property      number,                     /* column properties (bit flags) */
  name          varchar2(30),                              /* name of column */
  type_num      number,                               /* data type of column */
  length        number,                         /* length of column in bytes */
  precision_num number,                                         /* precision */
  scale         number,                                             /* scale */
  not_null      number,                               /* 0 = nulls permitted */
                                                 /* > 0 = no NULLs permitted */
  deflength     number,                   /* default value expr. text length */
  default_val   varchar2(4000),             /* default value expression text */
  charsetid     number,                              /* NLS character set id */
  charsetform   number,                 
  con           ku$_constraint0_t,                    /* not null constraint */
  spare1        number,                      /* fractional seconds precision */
  spare2        number,                  /* interval leading field precision */
  spare3        number,                 
  spare4        varchar2(1000),          /* NLS settings for this expression */
  spare5        varchar2(1000),
  spare6        date
)
/

grant execute on ku$_prim_column_t to public
/

create or replace type ku$_prim_column_list_t
 as table of ku$_prim_column_t
/
grant execute on ku$_prim_column_list_t to public
/

-- view to get all primitive columns (columns that are not LOBs or UDTs).
-- Each column has a constraint ADT embedded in it
-- if the column has a not-null constraint.  If there is no such constraint,
-- the constraint attribute evaluates to null.  Other constraints on columns
-- are in the table UDT.
create or replace force view ku$_prim_column_view of ku$_prim_column_t 
  with object identifier (obj_num,intcol_num) as
  select c.obj#, c.col#, c.intcol#, c.segcol#, c.property,
         c.name, c.type#, c.length,
         c.precision#, c.scale, c.null$, c.deflength,
         sys.dbms_metadata_util.long2varchar(c.deflength,
                                        'SYS.COL$',
                                        'DEFAULT$',
                                        c.rowid),
         c.charsetid, c.charsetform,
         ( select value(cv)
             from ku$_constraint_col_view ccv, ku$_constraint0_view cv
             where c.intcol# = ccv.intcol_num
             and c.obj# = ccv.obj_num
             and ccv.con_num = cv.con_num
             and cv.contype in (7,11)
         ),
         c.spare1, c.spare2, c.spare3, c.spare4, c.spare5, c.spare6
  from col$ c
/
grant select on ku$_prim_column_view to select_catalog_role
/

-------------------------------------------------------------------------------
--                              TYPED COLUMNS
-------------------------------------------------------------------------------

-- UDTs for typed columns

create or replace type ku$_subcoltype_t as object
(
  obj_num       number,                               /* obj# of base object */
  intcol_num    number,                            /* internal column number */
  toid          raw(16),            	                             /* toid */
  version       number,		             /* internal type version number */
  intcols       number,                        /* number of internal columns */
                                          /* storing the exploded ADT column */
  intcol_nums   raw(2000),	      /* list of intcol#s of columns storing */
                          /* the unpacked ADT column; stored in packed form; */
                                          /* each intcol# is stored as a ub2 */
  flags         number)                                             /* flags */
                          /* 0x01 - This type was stated in the IS OF clause */
                          /* 0x02 - This type has ONLY in the IS OF clause   */
/
grant execute on ku$_subcoltype_t to public
/
create or replace type ku$_subcoltype_list_t as table of ku$_subcoltype_t
/
grant execute on ku$_subcoltype_list_t to public
/

create or replace force view ku$_subcoltype_view of ku$_subcoltype_t
  with object identifier (obj_num,intcol_num,toid) as
  select sct.obj#, sct.intcol#, sct.toid,
	 sct.version#, sct.intcols, sct.intcol#s, sct.flags
    from subcoltype$ sct
/
grant select on ku$_subcoltype_view to select_catalog_role
/

create or replace type ku$_coltype_t as object
(
  obj_num       number,                               /* obj# of base object */
  col_num       number,                                     /* column number */
  intcol_num    number,                            /* internal column number */
  flags         number,                                             /* flags */
  toid          raw(16),            	                             /* toid */
  version       number,		             /* internal type version number */
  packed        number,		                 /* 0 = unpacked, 1 = packed */
  intcols       number,                        /* number of internal columns */
                                          /* storing the exploded ADT column */
  intcol_nums   raw(2000),	      /* list of intcol#s of columns storing */
                          /* the unpacked ADT column; stored in packed form; */
                                          /* each intcol# is stored as a ub2 */
  typidcol_num  number,           /* intcol# of the type discriminant column */
  synobj_num    number,              /* obj# of type synonym of the col type */
  subtype_list  ku$_subcoltype_list_t,			 /* subtype metadata */
  schema_obj    ku$_schemaobj_t                             /* schema object */
)
/
grant execute on ku$_coltype_t to public
/

-- view for typed column metadata
create or replace force view ku$_coltype_view
  of ku$_coltype_t with object identifier(obj_num, intcol_num) as
  select ct.obj#, ct.col#, ct.intcol#, ct.flags, ct.toid,
	 ct.version#, ct.packed, ct.intcols, ct.intcol#s,
	 ct.typidcol#, ct.synobj#,
	 /* look up stuff in subcoltype$ only if column is substitutable */
	 decode(bitand(ct.flags, 512), 512,
 	   cast(multiset(select sct.* from ku$_subcoltype_view sct
		where ct.obj#    = sct.obj_num
		and   ct.intcol# = sct.intcol_num
                       ) as ku$_subcoltype_list_t
                ),
	   null),
        (select value(o) from ku$_schemaobj_view o, obj$ oo
         where ct.toid = oo.oid$
         and o.obj_num = oo.obj#)
    from coltype$ ct
/

grant select on ku$_coltype_view to select_catalog_role
/

-------------------------------------------------------------------------------
--                              XML SCHEMA / OPAQUE TYPE
-------------------------------------------------------------------------------

-- For reasons having to do with compatibility, the XDB objects
-- can't be created by catproc.sql; they must instead be created
-- by a separate script catqm.sql.  Since catmeta.sql is run
-- by catproc.sql, we here create real UDTs for ku$_xmlschema_t
-- and ku$_xmlschema_elmt_t but the corresponding object views
-- are fake.  The real object views are defined in catmetx.sql
-- which is invoked by catqm.sql.

-- UDT for XML Schema
create or replace type ku$_xmlschema_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(2),                              /* UDT minor version # */
  owner_num     number,                                 /* owner user number */
  owner_name    varchar2(30),                                  /* owner name */
  url		varchar2(700),				       /* schema URL */
  schemaoid     raw(16),                                       /* schema OID */
  local		number,						/* 1 = local */
  schema_val	clob,				     /* the XMLSchema itself */
  stripped_val	clob		     /* the XMLSchema with username stripped */
)
/
grant execute on ku$_xmlschema_t to public
/
-- fake object view
create or replace force view ku$_xmlschema_view of ku$_xmlschema_t
  with object identifier (owner_name, url) as
  select '1','0', 0, NULL, NULL, NULL, 0, NULL, NULL
    from dual where 1=0
/
grant select on ku$_xmlschema_view to public
/

-- UDT for XML Schema element in opaque column
create or replace type ku$_xmlschema_elmt_t as object
(
  schemaoid     raw(16),                                   /* schema oid col */
  xmlschema     varchar2(700),                     /* The name of the schema */
  elemnum       number,                                    /* element number */
  element_name  varchar2(256)                     /* The name of the element */
)
/
grant execute on ku$_xmlschema_elmt_t to public
/
-- fake object view
create or replace view ku$_xmlschema_elmt_view
  of ku$_xmlschema_elmt_t with object identifier(schemaoid, elemnum) as
  select NULL, NULL, 0, NULL  
    from dual where 1=0
/
grant select on ku$_xmlschema_elmt_view to select_catalog_role
/

-- UDT for opaque type
create or replace type ku$_opqtype_t as object
(
  obj_num       number,                               /* obj# of base object */
  intcol_num    number,                            /* internal column number */
  type          number,                            /* The opaque type - type */
                                                           /* 0x01 - XMLType */
  flags         number,                         /* flags for the opaque type */
                              /* -------------- XMLType flags ---------
                               * 0x0001 (1) -- XMLType stored as object
                               * 0x0002 (2) -- XMLType schema is specified
                               * 0x0004 (4) -- XMLType stored as lob 
                               * 0x0008 (8) -- XMLType stores extra column
                               */
  /* Flags for XMLType (type == 0x01). Override them when necessary  */
  lobcol        number,                                        /* lob column */
  objcol        number,                                    /* obj rel column */
  extracol      number,                                    /* extra info col */
  schemaoid     raw(16),                                   /* schema oid col */
  elemnum       number,                                    /* element number */
  schema_elmt	ku$_xmlschema_elmt_t
)
/
grant execute on ku$_opqtype_t to public
/
create or replace force view ku$_opqtype_view of ku$_opqtype_t
  with object identifier(obj_num, intcol_num) as
  select opq.obj#, opq.intcol#, opq.type, opq.flags, opq.lobcol,
	 opq.objcol, opq.extracol, opq.schemaoid, opq.elemnum,
	 decode(bitand(opq.flags,2),0,NULL,
	   (select value (xe) from ku$_xmlschema_elmt_view xe
		where opq.schemaoid = xe.schemaoid
		  and opq.elemnum   = xe.elemnum))
  from sys.opqtype$ opq
/
grant select on ku$_opqtype_view to select_catalog_role
/

-------------------------------------------------------------------------------
--                              OID INDEX
-------------------------------------------------------------------------------

-- ADT for OID index (for object tables)
create or replace type ku$_oidindex_t as object
(
  obj_num       number,                              /* obj# of owning table */
  intcol_num    number,                            /* internal column number */
  name          varchar2(30),                             /* constraint name */
  flags         number,                                       /* defer flags */
  storage       ku$_storage_t,                                    /* storage */
  ts_name       varchar2(30),                             /* tablespace name */
  blocksize     number,                            /* size of block in bytes */
  pct_free      number,                   /* min. free space %age in a block */
  initrans      number,                     /* initial number of transaction */
  maxtrans      number                      /* maximum number of transaction */
)
/

grant execute on ku$_oidindex_t to public
/

-- view for OID index (derived from sys.exu8oid in catexp.sql)
create or replace force view ku$_oidindex_view
  of ku$_oidindex_t with object identifier(obj_num, intcol_num) as
       SELECT cc.obj#,cc.intcol#,co.name,cd.defer,
              ( select value(s) 
                from ku$_storage_view s
                where i.file#  = s.file_num
                and   i.block# = s.block_num
                and   i.ts#    = s.ts_num),
              ts.name, ts.blocksize,
              i.pctfree$,i.initrans,i.maxtrans
       FROM   sys.obj$ o, sys.ccol$ cc, sys.con$ co, sys.cdef$ cd,
              sys.ind$ i, sys.ts$ ts  
       WHERE  cc.con# = co.con# AND
              cc.con# = cd.con# AND
              cd.type# = 3 AND
              co.owner# = o.owner# AND 
              co.name = o.name AND
              o.obj# = i.obj# AND
              i.ts# = ts.ts#
/

grant select on ku$_oidindex_view to select_catalog_role
/

-------------------------------------------------------------------------------
--                              COLUMNS
-------------------------------------------------------------------------------

-- ADT for columns
create or replace type ku$_column_t as object
(
  obj_num       number,                      /* object number of base object */
  col_num       number,                          /* column number as created */
  segcol_num    number,                          /* column number in segment */
  segcollength  number,                      /* length of the segment column */
  offset        number,                                  /* offset of column */
  name          varchar2(30),                              /* name of column */
  attrname      varchar2(4000),/* name of type attr. column: null if != type */
  type_num      number,                               /* data type of column */
  length        number,                         /* length of column in bytes */
  fixedstorage  number,             /* flags: 0x01 = fixed, 0x02 = read-only */
  precision_num number,                                         /* precision */
  scale         number,                                             /* scale */
  is_null       number,                                   /* nulls permitted */
  deflength     number,                   /* default value expr. text length */
  default_val   varchar2(4000),             /* default value expression text */
  intcol_num    number,                            /* internal column number */
  property      number,                     /* column properties (bit flags) */
  charsetid     number,                              /* NLS character set id */
  charsetform   number,                 
  con           ku$_constraint0_t,                    /* not null constraint */
  typemd        ku$_coltype_t,     /* type metadata. null if not a typed col */
  lobmd         ku$_lob_t,            /* lob metadata. null if not a lob col */
  opqmd		ku$_opqtype_t,	 /* opaque metadata. null if not type opaque */
  oidindex      ku$_oidindex_t,   /*oidindex if col is OID$ col of obj table */
  spare1        number,                      /* fractional seconds precision */
  spare2        number,                  /* interval leading field precision */
  spare3        number,                 
  spare4        varchar2(1000),          /* NLS settings for this expression */
  spare5        varchar2(1000),
  spare6        date
)
/
grant execute on ku$_column_t to public
/

create or replace type ku$_column_list_t as table of ku$_column_t;
/
grant execute on ku$_column_list_t to public
/

-- view to get all columns.  Each column has a constraint ADT embedded in it
-- if the column has a not-null constraint.  If there is no such constraint,
-- the constraint attribute evaluates to null.  Other constraints on columns
-- are in the table ADT.
create or replace force view ku$_column_view of ku$_column_t 
  with object identifier (obj_num,intcol_num) as
  select c.obj#, c.col#, c.segcol#,
         c.segcollength, c.offset, c.name,
         (select a.name from attrcol$ a where
                        a.obj#=c.obj# and a.intcol#=c.intcol#),
         c.type#, c.length, c.fixedstorage,
         c.precision#, c.scale, c.null$, c.deflength,
         sys.dbms_metadata_util.long2varchar(c.deflength,
                                        'SYS.COL$',
                                        'DEFAULT$',
                                        c.rowid),
         c.intcol#,
         c.property, c.charsetid, c.charsetform,
         ( select value(cv)
             from ku$_constraint0_view cv, ku$_constraint_col_view ccv
             where c.intcol# = ccv.intcol_num
             and c.obj# = ccv.obj_num
             and ccv.con_num = cv.con_num
             and cv.contype in (7,11)
         ),
         ( select value(ctv)
             from ku$_coltype_view ctv
             where c.type# in ( 121,    -- DTYADT  (user-defined type)
                                122,    -- DTYNTB  (nested table)
                                123,    -- DTYNAR  (varray)
                                111,    -- DTYIREF (REF)
                                 58)    -- DTYOPQ  (opaque type)
             and   c.obj#  = ctv.obj_num
             and   c.intcol# = ctv.intcol_num
         ),
         ( select value(lv)
             from ku$_lob_view lv
             where (c.type# in (112,    -- CLOB
                                113,    -- BLOB
                                123)    -- DTYNAR  (varray)
             and   c.obj#  = lv.obj_num
             and   c.intcol# = lv.intcol_num)
             or    (c.type# = 58        -- DTYOPQ (opaque type)
             and   c.obj#  = lv.obj_num
             and   lv.intcol_num = (select op.lobcol from sys.opqtype$ op
                                    where op.obj# = c.obj#
                                    and   bitand(op.flags,4) != 0
                                    and   op.intcol# = c.intcol#)
                )
         ),
	 ( select value(opq) from sys.ku$_opqtype_view opq
	     where c.type# = 58        -- DTYOPQ (opaque type)
             and   c.obj# = opq.obj_num
             and   c.intcol# = opq.intcol_num
         ),
         ( select value(oi)
             from ku$_oidindex_view oi
             where bitand(c.property, 2) = 2
             and   c.obj# = oi.obj_num
             and   c.intcol# = oi.intcol_num
         ),
         c.spare1, c.spare2, c.spare3, c.spare4, c.spare5, c.spare6
  from col$ c
/

grant select on ku$_column_view to select_catalog_role
/

-- ADT for columns in partitioned tables
create or replace type ku$_pcolumn_t as object
(
  obj_num       number,                      /* object number of base object */
  col_num       number,                          /* column number as created */
  segcol_num    number,                          /* column number in segment */
  segcollength  number,                      /* length of the segment column */
  offset        number,                                  /* offset of column */
  name          varchar2(30),                              /* name of column */
  attrname      varchar2(4000),/* name of type attr. column: null if != type */
  type_num      number,                               /* data type of column */
  length        number,                         /* length of column in bytes */
  fixedstorage  number,             /* flags: 0x01 = fixed, 0x02 = read-only */
  precision_num number,                                         /* precision */
  scale         number,                                             /* scale */
  is_null       number,                                   /* nulls permitted */
  deflength     number,                   /* default value expr. text length */
  default_val   varchar2(4000),             /* default value expression text */
  intcol_num    number,                            /* internal column number */
  property      number,                     /* column properties (bit flags) */
  charsetid     number,                              /* NLS character set id */
  charsetform   number,                 
  con           ku$_constraint0_t,                    /* not null constraint */
  typemd        ku$_coltype_t,     /* type metadata. null if not a typed col */
  lobmd         ku$_partlob_t,        /* lob metadata. null if not a lob col */
  oidindex      ku$_oidindex_t,   /*oidindex if col is OID$ col of obj table */
  spare1        number,                      /* fractional seconds precision */
  spare2        number,                  /* interval leading field precision */
  spare3        number,                 
  spare4        varchar2(1000),          /* NLS settings for this expression */
  spare5        varchar2(1000),
  spare6        date
)
/
grant execute on ku$_pcolumn_t to public
/

create or replace type ku$_pcolumn_list_t as table of ku$_pcolumn_t;
/
grant execute on ku$_pcolumn_list_t to public
/

-- view to get all columns.  Each column has a constraint ADT embedded in it
-- if the column has a not-null constraint.  If there is no such constraint,
-- the constraint attribute evaluates to null.  Other constraints on columns
-- are in the table ADT.
create or replace force view ku$_pcolumn_view of ku$_pcolumn_t 
  with object identifier (obj_num,intcol_num) as
  select c.obj#, c.col#, c.segcol#,
         c.segcollength, c.offset, c.name,
         (select a.name from attrcol$ a where
                        a.obj#=c.obj# and a.intcol#=c.intcol#),
         c.type#, c.length, c.fixedstorage,
         c.precision#, c.scale, c.null$, c.deflength,
         sys.dbms_metadata_util.long2varchar(c.deflength,
                                        'SYS.COL$',
                                        'DEFAULT$',
                                        c.rowid),
         c.intcol#,
         c.property, c.charsetid, c.charsetform,
         ( select value(cv)
             from ku$_constraint0_view cv, ku$_constraint_col_view ccv
             where c.intcol# = ccv.intcol_num
             and c.obj# = ccv.obj_num
             and ccv.con_num = cv.con_num
             and cv.contype in (7,11)
         ),
         ( select value(ctv)
             from ku$_coltype_view ctv
             where c.type# in ( 121,    -- DTYADT  (user-defined type)
                                122,    -- DTYNTB  (nested table)
                                123,    -- DTYNAR  (varray)
                                111,    -- DTYIREF (REF)
                                 58)    -- DTYOPQ  (opaque type)
             and   c.obj#  = ctv.obj_num
             and   c.intcol# = ctv.intcol_num
         ),
         ( select value(lv)
             from ku$_partlob_view lv
             where c.type# in ( 112,    -- CLOB
                                113,    -- BLOB
                                123)    -- DTYNAR  (varray)
             and   c.obj#  = lv.obj_num
             and   c.intcol# = lv.intcol_num
         ),
         ( select value(oi)
             from ku$_oidindex_view oi
             where bitand(c.property, 2) = 2
             and   c.obj# = oi.obj_num
             and   c.intcol# = oi.intcol_num
         ),
         c.spare1, c.spare2, c.spare3, c.spare4, c.spare5, c.spare6
  from col$ c
/

grant select on ku$_pcolumn_view to select_catalog_role
/

-------------------------------------------------------------------------------
--                              NESTED TABLE
-------------------------------------------------------------------------------


-- ADT for IOT overflow table
create or replace type ku$_ov_table_t as object
(
  obj_num       number,                                              /* obj# */
  bobj_num      number,                           /* base obj# (cluster/iot) */
  storage       ku$_storage_t,                                    /* storage */
  ts_name       varchar2(30),                             /* tablespace name */
  blocksize     number,                            /* size of block in bytes */
  pct_free      number,                   /* min. free space %age in a block */
  pct_used      number,                   /* min. used space %age in a block */
  initrans      number,                     /* initial number of transaction */
  maxtrans      number,                     /* maximum number of transaction */
  flags         number                                              /* flags */
)
/
grant execute on ku$_ov_table_t to public
/

-- view for IOT overflow table
create or replace force view ku$_ov_table_view of ku$_ov_table_t 
  with object OID(obj_num)
  as select t.obj#, t.bobj#,
         (select value(s) from ku$_storage_view s
          where t.file# = s.file_num
          and t.block#  = s.block_num
          and t.ts#     = s.ts_num),
         ts.name, ts.blocksize,
         t.pctfree$, t.pctused$, t.initrans, t.maxtrans, t.flags
  from  tab$ t, ts$ ts
  where bitand(t.property,512) = 512
        and t.ts# = ts.ts#
/
grant select on ku$_ov_table_view to select_catalog_role
/


-- UDT for table data for heap nested table
create or replace type ku$_hnt_t as object
(
  obj_num       number,                              /* obj# of nested table */
  property      number,                                  /* table properties */
  storage       ku$_storage_t,                                    /* storage */
  ts_name       varchar2(30),                             /* tablespace name */
  blocksize     number,                            /* size of block in bytes */
  pct_free      number,                   /* min. free space %age in a block */
  pct_used      number,                   /* min. used space %age in a block */
  initrans      number,                     /* initial number of transaction */
  maxtrans      number,                     /* maximum number of transaction */
  flags         number,                                             /* flags */
  con0_list     ku$_constraint0_list_t,               /* list of constraints */
  con1_list     ku$_constraint1_list_t,               /* list of constraints */
  con2_list     ku$_constraint2_list_t,               /* list of constraints */
  pkref_list    ku$_pkref_constraint_list_t  /* list of table ref constraints*/
)
/
grant execute on ku$_hnt_t to public
/

-- view for table data for heap nested table
create or replace force view ku$_hnt_view
  of ku$_hnt_t with object identifier(obj_num) as
  select t.obj#, t.property,
        (select value(s) from ku$_storage_view s
         where     t.file#  = s.file_num
             and   t.block# = s.block_num
             and   t.ts#    = s.ts_num),
        (select ts.name from ts$ ts where t.ts# = ts.ts#),
        (select ts.blocksize from ts$ ts where t.ts# = ts.ts#),
        t.pctfree$, t.pctused$, t.initrans, t.maxtrans, t.flags,
         cast( multiset(select * from ku$_constraint0_view con
                        where con.obj_num = t.obj#
                        and con.contype not in (7,11)
                       ) as ku$_constraint0_list_t
             ),
         cast( multiset(select * from ku$_constraint1_view con
                        where con.obj_num = t.obj#
                       ) as ku$_constraint1_list_t
             ),
         cast( multiset(select * from ku$_constraint2_view con
                        where con.obj_num = t.obj#
                       ) as ku$_constraint2_list_t
             ),
         cast( multiset(select * from ku$_pkref_constraint_view con
                        where con.obj_num = t.obj#
                       ) as ku$_pkref_constraint_list_t
             )
  from tab$ t where bitand(t.property,64+512) = 0 -- skip IOT and overflow segs
/
grant select on ku$_hnt_view to select_catalog_role
/

-- UDT for table data for index organized nested table
create or replace type ku$_iont_t as object
(
  obj_num       number,                              /* obj# of nested table */
  property      number,                                  /* table properties */
  storage       ku$_storage_t,                                    /* storage */
  ts_name       varchar2(30),                             /* tablespace name */
  blocksize     number,                            /* size of block in bytes */
  pct_free      number,                   /* min. free space %age in a block */
  initrans      number,                     /* initial number of transaction */
  maxtrans      number,                     /* maximum number of transaction */
  flags         number,                                             /* flags */
  pct_thresh    number,                            /* pctthreshold for IOTs. */
  numkeycols    number,      /* # of key columns in compressed prefix (IOTs) */
  inclcol_name  varchar(30),/* column where IOT splits into overflow segment */
  con0_list     ku$_constraint0_list_t,               /* list of constraints */
  con1_list     ku$_constraint1_list_t,               /* list of constraints */
  con2_list     ku$_constraint2_list_t,               /* list of constraints */
  pkref_list    ku$_pkref_constraint_list_t, /* list of table ref constraints*/
  iov           ku$_ov_table_t                             /* overflow table */
)
/
grant execute on ku$_iont_t to public
/

-- view for table data for index organized nested table
create or replace force view ku$_iont_view
  of ku$_iont_t with object identifier(obj_num) as
  select t.obj#, t.property,
        (select value(s) from ku$_storage_view s
         where     i.file#  = s.file_num
             and   i.block# = s.block_num
             and   i.ts#    = s.ts_num),
        (select ts.name from ts$ ts where i.ts# = ts.ts#),
        (select ts.blocksize from ts$ ts where i.ts# = ts.ts#),
        i.pctfree$, i.initrans, i.maxtrans, t.flags,
        mod(i.pctthres$,256), i.spare2,
        (select c.name from col$ c
                 where c.obj# = i.bo#
                 and   c.col# = i.trunccnt and i.trunccnt != 0),
         cast( multiset(select * from ku$_constraint0_view con
                        where con.obj_num = t.obj#
                        and con.contype not in (7,11)
                       ) as ku$_constraint0_list_t
             ),
         cast( multiset(select * from ku$_constraint1_view con
                        where con.obj_num = t.obj#
                       ) as ku$_constraint1_list_t
             ),
         cast( multiset(select * from ku$_constraint2_view con
                        where con.obj_num = t.obj#
                       ) as ku$_constraint2_list_t
             ),
         cast( multiset(select * from ku$_pkref_constraint_view con
                        where con.obj_num = t.obj#
                       ) as ku$_pkref_constraint_list_t
             ),
        (select value(ov) from ku$_ov_table_view ov
         where ov.bobj_num = t.obj#
         and bitand(t.property, 128) = 128)  -- IOT has overflow
  from tab$ t, ind$ i
  where bitand(t.property,64+512) = 64  -- IOT but not overflow
    and t.pctused$ = i.obj#             -- for IOTs, pctused has index obj#
/
grant select on ku$_iont_view to select_catalog_role
/

-- UDT for nested table
create or replace type ku$_nt_t as object
(
  obj_num       number,                                /* obj# of base table */
  intcol_num    number,              /* internal column number in base table */
  ntab_num      number,              /* object number of nested table object */
  schema_obj    ku$_schemaobj_t,           /* schema object for nested table */
  col           ku$_simple_col_t,                                  /* column */
  property      number,                                  /* table properties */
  flags         number,                                             /* flags */
  hnt           ku$_hnt_t,                                /* heap table data */
  iont          ku$_iont_t,                                      /* iot data */
  col_list      ku$_column_list_t                         /* list of columns */
)
/
grant execute on ku$_nt_t to public
/
create or replace type ku$_nt_list_t as table of ku$_nt_t
/
grant execute on ku$_nt_list_t to public
/

-- UDT for collection of nested tables of a parent table
create or replace type ku$_nt_parent_t as object
(
  obj_num       number,                                /* obj# of base table */
  nts           ku$_nt_list_t                               /* nested tables */
)
/
grant execute on ku$_nt_parent_t to public
/

-- view for collection of nested tables of a parent table
create or replace force view ku$_nt_parent_view
  of ku$_nt_parent_t with object identifier(obj_num) as
  select t.obj#,
    cast(multiset(select
                nt.obj#, nt.intcol#, nt.ntab#,
                (select value(o) from ku$_schemaobj_view o
                 where o.obj_num = nt.ntab#),
                (select value(c) from ku$_simple_col_view c
                 where c.obj_num = nt.obj#
                 and   c.intcol_num = nt.intcol#),
                (select t.property from tab$ t where t.obj# = nt.ntab#),
                (select ct.flags from coltype$ ct
                        where ct.obj# = nt.obj#
                        and   ct.intcol# = nt.intcol#),
                (select value(h) from ku$_hnt_view h
                 where h.obj_num = nt.ntab#),
                (select value(i) from ku$_iont_view i
                 where i.obj_num = nt.ntab#),
                (cast(multiset(select * from ku$_column_view c 
                                where c.obj_num = nt.ntab#
                                order by c.col_num, c.intcol_num
                        ) as ku$_column_list_t
                        ))
          from ntab$ nt start with nt.obj#=t.obj#
                        connect by prior nt.ntab#=nt.obj#
                ) as ku$_nt_list_t
        )
  from tab$ t where bitand(t.property,4) = 4    -- has nested table columns
/
grant select on ku$_nt_parent_view to select_catalog_role
/

-------------------------------------------------------------------------------
--                              CLUSTERED TABLE
-------------------------------------------------------------------------------
-- UDT for table clustering info
create or replace type ku$_tabcluster_t as object
(
  obj_num       number,                            /* object number of table */
  schema_obj    ku$_schemaobj_t,                  /* cluster's schema object */
  col_list      ku$_simple_col_list_t           /* list of clustered columns */
)
/
grant execute on ku$_tabcluster_t to public
/

create or replace force view ku$_tabcluster_col_view of ku$_simple_col_t 
  with object identifier (obj_num,intcol_num) as
  select c.obj#,
         c.col#,
         c.intcol#,
         c.segcol#,
         c.property,
         c.name,
         (select a.name from attrcol$ a where
                        a.obj#=c.obj# and a.intcol#=c.intcol#),
	 c.type#,
         c.deflength, ''
  from col$ c, col$ cc, tab$ t
  where c.obj#  = t.obj#
    and cc.obj# = t.bobj#
    and cc.segcol# = c.segcol#
/
grant select on ku$_tabcluster_col_view to select_catalog_role
/

create or replace force view ku$_tabcluster_view of ku$_tabcluster_t
  with object identifier (obj_num) as
  select t.obj#,
         value(cl),
         cast(multiset(select * from ku$_tabcluster_col_view c
                       where c.obj_num = t.obj#
                        order by c.col_num, c.intcol_num
                      ) as ku$_simple_col_list_t
             )
  from  ku$_schemaobj_view cl, sys.tab$ t
  where bitand(t.property,1024) = 1024          -- clustered table
    and cl.obj_num = t.bobj#
/
grant select on ku$_tabcluster_view to select_catalog_role
/

-------------------------------------------------------------------------------
--                              TABLE
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
-- Specialized table UDTs/views
-------------------------------------------------------------------------------

create or replace type ku$_extloc_t as object
(
  obj_num       number,                          /* base table object number */
  position      number,                               /* this location index */
  dir           varchar2(30),                   /* location directory object */
  name          varchar2(4000)                              /* location name */
)
/
grant execute on ku$_extloc_t to public
/ 

create or replace type ku$_extloc_list_t
 as table of ku$_extloc_t
/
grant execute on ku$_extloc_list_t to public
/

create or replace type ku$_exttab_t as object
(
  obj_num       number,                          /* base table object number */
  default_dir   varchar2(30),                           /* default directory */
  type          varchar2(30),                          /* access driver type */
  nr_locations  number,                               /* number of locations */
  reject_limit  number,                                      /* reject limit */
  par_type      number,             /* access parameter type: blob=1, clob=2 */
  param_clob    clob,                      /* access parameters in clob form */
  location      ku$_extloc_list_t                      /* external locations */
)
/
grant execute on ku$_exttab_t to public
/

create or replace force view ku$_exttab_view of ku$_exttab_t
  with object OID(obj_num)
  as select
        et.obj#,
        et.default_dir,
        et.type$,
        et.nr_locations,
        et.reject_limit,
        et.par_type,
        et.param_clob,
        cast( multiset(select el.obj#, el.position, el.dir, el.name
                       from   sys.external_location$ el
                       where  el.obj# = et.obj#
                       order by el.obj#,el.position
                      ) as ku$_extloc_list_t
              )
      from sys.external_tab$ et
/
grant select on ku$_exttab_view to public
/

-- ADT for primitive, non-partitioned Heap TABLEs
create or replace type ku$_htable_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(2),                              /* UDT minor version # */
  obj_num       number,                                              /* obj# */
  schema_obj    ku$_schemaobj_t,                            /* schema object */
  storage       ku$_storage_t,                                    /* storage */
  ts_name       varchar2(30),                             /* tablespace name */
  blocksize     number,                            /* size of block in bytes */
  dataobj_num   number,                                /* data layer object# */
  bobj_num      number,                           /* base obj# (cluster/iot) */
  tab_num       number,                  /* # in cluster, null if !clustered */
  cols          number,                                      /* # of columns */
  clucols       number,                 /* # of clustered cols, if clustered */
  tabcluster    ku$_tabcluster_t,        /* cluster info, null if !clustered */
  pct_free      number,                   /* min. free space %age in a block */
  pct_used      number,                   /* min. used space %age in a block */
  initrans      number,                     /* initial number of transaction */
  maxtrans      number,                     /* maximum number of transaction */
  flags         number,                                             /* flags */
  audit_val     varchar2(38),                            /* auditing options */
  rowcnt        number,                                    /* number of rows */
  blkcnt        number,                                  /* number of blocks */
  empcnt        number,                            /* number of empty blocks */
  avgspc        number,                      /* average available free space */
  chncnt        number,                            /* number of chained rows */
  avgrln        number,                                /* average row length */
  avgspc_flb    number,       /* avg avail free space of blocks on free list */
  flbcnt        number,                             /* free list block count */
  analyzetime   date,                        /* timestamp when last analyzed */
  samplesize    number,                 /* number of rows sampled by Analyze */
  degree        number,                       /* # of PQ slaves per instance */
  instances     number,                         /* # of OPS instances for PQ */
  intcols       number,                             /* # of internal columns */
  kernelcols    number,                   /* number of REAL (kernel) columns */
  property      number,                                  /* table properties */
  trigflag      number,                              /* inline trigger flags */
  spare1        number,                       /* used to store hakan_kqldtvc */
  spare2        number,         /* committed partition # used by drop column */
  spare3        number,                           /* summary sequence number */
  spare4        varchar2(1000),         /* committed RID used by drop column */
  spare5        varchar2(1000),         
  spare6        date,                                       /* dml timestamp */
  col_list      ku$_prim_column_list_t,                   /* list of columns */
  con0_list     ku$_constraint0_list_t,               /* list of constraints */
  con1_list     ku$_constraint1_list_t,               /* list of constraints */
  con2_list     ku$_constraint2_list_t,               /* list of constraints */
  exttab        ku$_exttab_t                      /* external table metadata */
)
/
grant execute on ku$_htable_t to public
/

create or replace force view ku$_htable_view of ku$_htable_t 
  with object OID(obj_num)
  as select '1','0',
         t.obj#,
         value(o),
         (select value(s) from ku$_storage_view s
          where t.file# = s.file_num
          and t.block#  = s.block_num
          and t.ts#     = s.ts_num),
         ts.name, ts.blocksize,
         t.dataobj#, t.bobj#, t.tab#, t.cols,
         t.clucols,
         (select value(cl) from ku$_tabcluster_view cl
          where cl.obj_num = t.obj#),
         t.pctfree$, t.pctused$, t.initrans, t.maxtrans, t.flags,
         t.audit$, t.rowcnt, t.blkcnt, t.empcnt, t.avgspc, t.chncnt, t.avgrln,
         t.avgspc_flb, t.flbcnt, t.analyzetime, t.samplesize, t.degree,
         t.instances, t.intcols, t.kernelcols, t.property, t.trigflag,
         t.spare1, t.spare2, t.spare3, t.spare4, t.spare5, t.spare6,
         cast( multiset(select * from ku$_prim_column_view c 
                        where c.obj_num = t.obj#
                        order by c.col_num, c.intcol_num
                        ) as ku$_prim_column_list_t
              ),
         cast( multiset(select * from ku$_constraint0_view con
                        where con.obj_num = t.obj#
                        and con.contype not in (7,11)
                       ) as ku$_constraint0_list_t
             ),
         cast( multiset(select * from ku$_constraint1_view con
                        where con.obj_num = t.obj#
                       ) as ku$_constraint1_list_t
             ),
         cast( multiset(select * from ku$_constraint2_view con
                        where con.obj_num = t.obj#
                       ) as ku$_constraint2_list_t
             ),
         (select value(etv) from ku$_exttab_view etv
                        where etv.obj_num = o.obj_num)
  from  ku$_schemaobj_view o, tab$ t, ts$ ts
  where t.obj# = o.obj_num
        AND bitand(t.property,1607991295)     -- mask off bits 0x20280000
                in (0,1024)                   -- can be clustered table (1024)
        AND t.ts# = ts.ts#
        AND (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num, 0) OR 
                EXISTS ( SELECT * FROM session_roles 
                        WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_htable_view to public
/

-- ADT for primitive, Partitioned Heap TABLEs
create or replace type ku$_phtable_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(2),                              /* UDT minor version # */
  obj_num       number,                                              /* obj# */
  schema_obj    ku$_schemaobj_t,                            /* schema object */
  storage       ku$_storage_t,                                    /* storage */
  ts_name       varchar2(30),                             /* tablespace name */
  blocksize     number,                            /* size of block in bytes */
  dataobj_num   number,                                /* data layer object# */
  bobj_num      number,                           /* base obj# (cluster/iot) */
  tab_num       number,                  /* # in cluster, null if !clustered */
  cols          number,                                      /* # of columns */
  clucols       number,                 /* # of clustered cols, if clustered */
  tabcluster    ku$_tabcluster_t,        /* cluster info, null if !clustered */
  pct_free      number,                   /* min. free space %age in a block */
  pct_used      number,                   /* min. used space %age in a block */
  initrans      number,                     /* initial number of transaction */
  maxtrans      number,                     /* maximum number of transaction */
  flags         number,                                             /* flags */
  audit_val     varchar2(38),                            /* auditing options */
  rowcnt        number,                                    /* number of rows */
  blkcnt        number,                                  /* number of blocks */
  empcnt        number,                            /* number of empty blocks */
  avgspc        number,                      /* average available free space */
  chncnt        number,                            /* number of chained rows */
  avgrln        number,                                /* average row length */
  avgspc_flb    number,       /* avg avail free space of blocks on free list */
  flbcnt        number,                             /* free list block count */
  analyzetime   date,                        /* timestamp when last analyzed */
  samplesize    number,                 /* number of rows sampled by Analyze */
  degree        number,                       /* # of PQ slaves per instance */
  instances     number,                         /* # of OPS instances for PQ */
  intcols       number,                             /* # of internal columns */
  kernelcols    number,                   /* number of REAL (kernel) columns */
  property      number,                                  /* table properties */
  trigflag      number,                              /* inline trigger flags */
  spare1        number,                       /* used to store hakan_kqldtvc */
  spare2        number,         /* committed partition # used by drop column */
  spare3        number,                           /* summary sequence number */
  spare4        varchar2(1000),         /* committed RID used by drop column */
  spare5        varchar2(1000),         
  spare6        date,                                       /* dml timestamp */
  col_list      ku$_prim_column_list_t,                   /* list of columns */
  con0_list     ku$_constraint0_list_t,               /* list of constraints */
  con1_list     ku$_constraint1_list_t,               /* list of constraints */
  con2_list     ku$_constraint2_list_t,               /* list of constraints */
  part_obj      ku$_tab_partobj_t,                         /* partition info */
  exttab        ku$_exttab_t                      /* external table metadata */
)
/
grant execute on ku$_phtable_t to public
/

create or replace force view ku$_phtable_view of ku$_phtable_t 
  with object OID(obj_num)
  as select '1','0',
         t.obj#,
         value(o),
         (select value(s) from ku$_storage_view s
          where t.file# = s.file_num
          and t.block#  = s.block_num
          and t.ts#     = s.ts_num),
         ts.name, ts.blocksize,
         t.dataobj#, t.bobj#, t.tab#, t.cols,
         t.clucols,
         (select value(cl) from ku$_tabcluster_view cl
          where cl.obj_num = t.obj#),
         t.pctfree$, t.pctused$, t.initrans, t.maxtrans, t.flags,
         t.audit$, t.rowcnt, t.blkcnt, t.empcnt, t.avgspc, t.chncnt, t.avgrln,
         t.avgspc_flb, t.flbcnt, t.analyzetime, t.samplesize, t.degree,
         t.instances, t.intcols, t.kernelcols, t.property, t.trigflag,
         t.spare1, t.spare2, t.spare3, t.spare4, t.spare5, t.spare6,
         cast( multiset(select * from ku$_prim_column_view c 
                        where c.obj_num = t.obj#
                        order by c.col_num, c.intcol_num
                        ) as ku$_prim_column_list_t
              ),
         cast( multiset(select * from ku$_constraint0_view con
                        where con.obj_num = t.obj#
                        and con.contype not in (7,11)
                       ) as ku$_constraint0_list_t
             ),
         cast( multiset(select * from ku$_constraint1_view con
                        where con.obj_num = t.obj#
                       ) as ku$_constraint1_list_t
             ),
         cast( multiset(select * from ku$_constraint2_view con
                        where con.obj_num = t.obj#
                       ) as ku$_constraint2_list_t
             ),
         (select value(po) from ku$_tab_partobj_view po
          where t.obj# = po.obj_num),
         (select value(etv) from ku$_exttab_view etv
                        where etv.obj_num = o.obj_num)
  from  ku$_schemaobj_view o, tab$ t, ts$ ts
  where t.obj# = o.obj_num
        AND bitand(t.property,1607983103)     -- mask off bits 0x20282000
                                 = 32   /* simple, partitioned tables */
                                        /* no CLOBs, UDTs, nested cols*/
        AND t.ts# = ts.ts#
        AND     (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num,0) OR 
                EXISTS ( SELECT * FROM session_roles 
                        WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_phtable_view to public
/

-- ADT for Full (i.e., non-primitive), non-partitioned Heap TABLEs
create or replace type ku$_fhtable_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(2),                              /* UDT minor version # */
  obj_num       number,                                              /* obj# */
  schema_obj    ku$_schemaobj_t,                            /* schema object */
  storage       ku$_storage_t,                                    /* storage */
  ts_name       varchar2(30),                             /* tablespace name */
  blocksize     number,                            /* size of block in bytes */
  dataobj_num   number,                                /* data layer object# */
  bobj_num      number,                           /* base obj# (cluster/iot) */
  tab_num       number,                  /* # in cluster, null if !clustered */
  cols          number,                                      /* # of columns */
  clucols       number,                 /* # of clustered cols, if clustered */
  tabcluster    ku$_tabcluster_t,        /* cluster info, null if !clustered */
  pct_free      number,                   /* min. free space %age in a block */
  pct_used      number,                   /* min. used space %age in a block */
  initrans      number,                     /* initial number of transaction */
  maxtrans      number,                     /* maximum number of transaction */
  flags         number,                                             /* flags */
  audit_val     varchar2(38),                            /* auditing options */
  rowcnt        number,                                    /* number of rows */
  blkcnt        number,                                  /* number of blocks */
  empcnt        number,                            /* number of empty blocks */
  avgspc        number,                      /* average available free space */
  chncnt        number,                            /* number of chained rows */
  avgrln        number,                                /* average row length */
  avgspc_flb    number,       /* avg avail free space of blocks on free list */
  flbcnt        number,                             /* free list block count */
  analyzetime   date,                        /* timestamp when last analyzed */
  samplesize    number,                 /* number of rows sampled by Analyze */
  degree        number,                       /* # of PQ slaves per instance */
  instances     number,                         /* # of OPS instances for PQ */
  intcols       number,                             /* # of internal columns */
  kernelcols    number,                   /* number of REAL (kernel) columns */
  property      number,                                  /* table properties */
  trigflag      number,                              /* inline trigger flags */
  spare1        number,                       /* used to store hakan_kqldtvc */
  spare2        number,         /* committed partition # used by drop column */
  spare3        number,                           /* summary sequence number */
  spare4        varchar2(1000),         /* committed RID used by drop column */
  spare5        varchar2(1000),         
  spare6        date,                                       /* dml timestamp */
  col_list      ku$_column_list_t,                        /* list of columns */
  nt            ku$_nt_parent_t,                            /* nested tables */
  con0_list     ku$_constraint0_list_t,               /* list of constraints */
  con1_list     ku$_constraint1_list_t,               /* list of constraints */
  con2_list     ku$_constraint2_list_t,               /* list of constraints */
  pkref_list    ku$_pkref_constraint_list_t, /* list of table ref constraints*/
  exttab        ku$_exttab_t                      /* external table metadata */
)
/
grant execute on ku$_fhtable_t to public
/

create or replace force view ku$_fhtable_view of ku$_fhtable_t 
  with object OID(obj_num)
  as select '1','0',
         t.obj#,
         value(o),
         (select value(s) from ku$_storage_view s
          where t.file# = s.file_num
          and t.block#  = s.block_num
          and t.ts#     = s.ts_num),
         ts.name, ts.blocksize,
         t.dataobj#, t.bobj#, t.tab#, t.cols,
         t.clucols,
         (select value(cl) from ku$_tabcluster_view cl
          where cl.obj_num = t.obj#),
         t.pctfree$, t.pctused$, t.initrans, t.maxtrans, t.flags,
         t.audit$, t.rowcnt, t.blkcnt, t.empcnt, t.avgspc, t.chncnt, t.avgrln,
         t.avgspc_flb, t.flbcnt, t.analyzetime, t.samplesize, t.degree,
         t.instances, t.intcols, t.kernelcols, t.property, t.trigflag,
         t.spare1, t.spare2, t.spare3, t.spare4, t.spare5, t.spare6,
         cast( multiset(select * from ku$_column_view c 
                        where c.obj_num = t.obj#
                        order by c.col_num, c.intcol_num
                        ) as ku$_column_list_t
              ),
         (select value(nt) from ku$_nt_parent_view nt
          where nt.obj_num = t.obj#),
         cast( multiset(select * from ku$_constraint0_view con
                        where con.obj_num = t.obj#
                        and con.contype not in (7,11)
                       ) as ku$_constraint0_list_t
             ),
         cast( multiset(select * from ku$_constraint1_view con
                        where con.obj_num = t.obj#
                       ) as ku$_constraint1_list_t
             ),
         cast( multiset(select * from ku$_constraint2_view con
                        where con.obj_num = t.obj#
                       ) as ku$_constraint2_list_t
             ),
         cast( multiset(select * from ku$_pkref_constraint_view con
                        where con.obj_num = t.obj#
                       ) as ku$_pkref_constraint_list_t
             ),
         (select value(etv) from ku$_exttab_view etv
                        where etv.obj_num = o.obj_num)
  from ku$_schemaobj_view o, tab$ t, ts$ ts
  where t.obj# = o.obj_num
        AND t.ts# = ts.ts#
        AND bitand(t.property, 32+64+128+256+512+8192) = 0
                                                /* not IOT, partitioned   */
                                                /* or nested table        */
        AND bitand(t.property,1607983103)     -- mask off bits 0x20282000
                NOT in (0,1024,8192) -- don't include those in ku$_htable_view
        AND     (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num, 0) OR 
                EXISTS ( SELECT * FROM session_roles 
                        WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_fhtable_view to public
/

-- ADT for Partitioned, Full (i.e., non-primitive) Heap TABLEs
create or replace type ku$_pfhtable_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(2),                              /* UDT minor version # */
  obj_num       number,                                              /* obj# */
  schema_obj    ku$_schemaobj_t,                            /* schema object */
  storage       ku$_storage_t,                                    /* storage */
  ts_name       varchar2(30),                             /* tablespace name */
  blocksize     number,                            /* size of block in bytes */
  dataobj_num   number,                                /* data layer object# */
  bobj_num      number,                           /* base obj# (cluster/iot) */
  tab_num       number,                  /* # in cluster, null if !clustered */
  cols          number,                                      /* # of columns */
  clucols       number,                 /* # of clustered cols, if clustered */
  tabcluster    ku$_tabcluster_t,        /* cluster info, null if !clustered */
  pct_free      number,                   /* min. free space %age in a block */
  pct_used      number,                   /* min. used space %age in a block */
  initrans      number,                     /* initial number of transaction */
  maxtrans      number,                     /* maximum number of transaction */
  flags         number,                                             /* flags */
  audit_val     varchar2(38),                            /* auditing options */
  rowcnt        number,                                    /* number of rows */
  blkcnt        number,                                  /* number of blocks */
  empcnt        number,                            /* number of empty blocks */
  avgspc        number,                      /* average available free space */
  chncnt        number,                            /* number of chained rows */
  avgrln        number,                                /* average row length */
  avgspc_flb    number,       /* avg avail free space of blocks on free list */
  flbcnt        number,                             /* free list block count */
  analyzetime   date,                        /* timestamp when last analyzed */
  samplesize    number,                 /* number of rows sampled by Analyze */
  degree        number,                       /* # of PQ slaves per instance */
  instances     number,                         /* # of OPS instances for PQ */
  intcols       number,                             /* # of internal columns */
  kernelcols    number,                   /* number of REAL (kernel) columns */
  property      number,                                  /* table properties */
  trigflag      number,                              /* inline trigger flags */
  spare1        number,                       /* used to store hakan_kqldtvc */
  spare2        number,         /* committed partition # used by drop column */
  spare3        number,                           /* summary sequence number */
  spare4        varchar2(1000),         /* committed RID used by drop column */
  spare5        varchar2(1000),         
  spare6        date,                                       /* dml timestamp */
  col_list      ku$_pcolumn_list_t,                       /* list of columns */
  nt            ku$_nt_parent_t,                            /* nested tables */
  con0_list     ku$_constraint0_list_t,               /* list of constraints */
  con1_list     ku$_constraint1_list_t,               /* list of constraints */
  con2_list     ku$_constraint2_list_t,               /* list of constraints */
  pkref_list    ku$_pkref_constraint_list_t, /* list of table ref constraints*/
  part_obj      ku$_tab_partobj_t,                         /* partition info */
  exttab        ku$_exttab_t                      /* external table metadata */
)
/
grant execute on ku$_pfhtable_t to public
/

create or replace force view ku$_pfhtable_view of ku$_pfhtable_t 
  with object OID(obj_num)
  as select '1','0',
         t.obj#,
         value(o),
         (select value(s) from ku$_storage_view s
          where t.file# = s.file_num
          and t.block#  = s.block_num
          and t.ts#     = s.ts_num),
         ts.name, ts.blocksize,
         t.dataobj#, t.bobj#, t.tab#, t.cols,
         t.clucols,
         (select value(cl) from ku$_tabcluster_view cl
          where cl.obj_num = t.obj#),
         t.pctfree$, t.pctused$, t.initrans, t.maxtrans, t.flags,
         t.audit$, t.rowcnt, t.blkcnt, t.empcnt, t.avgspc, t.chncnt, t.avgrln,
         t.avgspc_flb, t.flbcnt, t.analyzetime, t.samplesize, t.degree,
         t.instances, t.intcols, t.kernelcols, t.property, t.trigflag,
         t.spare1, t.spare2, t.spare3, t.spare4, t.spare5, t.spare6,
         cast( multiset(select * from ku$_pcolumn_view c 
                        where c.obj_num = t.obj#
                        order by c.col_num, c.intcol_num
                        ) as ku$_pcolumn_list_t
              ),
         (select value(nt) from ku$_nt_parent_view nt
          where nt.obj_num = t.obj#),
         cast( multiset(select * from ku$_constraint0_view con
                        where con.obj_num = t.obj#
                        and con.contype not in (7,11)
                       ) as ku$_constraint0_list_t
             ),
         cast( multiset(select * from ku$_constraint1_view con
                        where con.obj_num = t.obj#
                       ) as ku$_constraint1_list_t
             ),
         cast( multiset(select * from ku$_constraint2_view con
                        where con.obj_num = t.obj#
                       ) as ku$_constraint2_list_t
             ),
         cast( multiset(select * from ku$_pkref_constraint_view con
                        where con.obj_num = t.obj#
                       ) as ku$_pkref_constraint_list_t
             ),
         (select value(po) from ku$_tab_partobj_view po
          where t.obj# = po.obj_num),
         (select value(etv) from ku$_exttab_view etv
                        where etv.obj_num = o.obj_num)
  from ku$_schemaobj_view o, tab$ t, ts$ ts
  where t.obj# = o.obj_num
        AND t.ts# = ts.ts#
        AND bitand(t.property, 32+64+128+256+512+8192) = 32
                                                /* partitioned (32)       */
                                                /* but not IOT            */
                                                /* or nested table        */
        AND bitand(t.property,1607983103)     -- mask off bits 0x20282000
             not in (32,32+8192)  /* Mutually exclusive of ku$_phtable_view */
        AND     (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num,0) OR 
                EXISTS ( SELECT * FROM session_roles 
                        WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_pfhtable_view to public
/

-- ADT for non-partitioned Index-Organized TABLEs
create or replace type ku$_iotable_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(2),                              /* UDT minor version # */
  obj_num       number,                                              /* obj# */
  schema_obj    ku$_schemaobj_t,                            /* schema object */
  storage       ku$_storage_t,                                    /* storage */
  ts_name       varchar2(30),                             /* tablespace name */
  blocksize     number,                            /* size of block in bytes */
  dataobj_num   number,                                /* data layer object# */
  bobj_num      number,                           /* base obj# (cluster/iot) */
  tab_num       number,                  /* # in cluster, null if !clustered */
  cols          number,                                      /* # of columns */
  clucols       number,                 /* # of clustered cols, if clustered */
  pct_free      number,                   /* min. free space %age in a block */
  initrans      number,                     /* initial number of transaction */
  maxtrans      number,                     /* maximum number of transaction */
  pct_thresh    number,                            /* pctthreshold for IOTs. */
  numkeycols    number,      /* # of key columns in compressed prefix (IOTs) */
  flags         number,                                             /* flags */
  audit_val     varchar2(38),                            /* auditing options */
  rowcnt        number,                                    /* number of rows */
  blkcnt        number,                                  /* number of blocks */
  empcnt        number,                            /* number of empty blocks */
  avgspc        number,                      /* average available free space */
  chncnt        number,                            /* number of chained rows */
  avgrln        number,                                /* average row length */
  avgspc_flb    number,       /* avg avail free space of blocks on free list */
  flbcnt        number,                             /* free list block count */
  analyzetime   date,                        /* timestamp when last analyzed */
  samplesize    number,                 /* number of rows sampled by Analyze */
  degree        number,                       /* # of PQ slaves per instance */
  instances     number,                         /* # of OPS instances for PQ */
  intcols       number,                             /* # of internal columns */
  kernelcols    number,                   /* number of REAL (kernel) columns */
  property      number,                                  /* table properties */
  trigflag      number,                              /* inline trigger flags */
  spare1        number,                       /* used to store hakan_kqldtvc */
  spare2        number,         /* committed partition # used by drop column */
  spare3        number,                           /* summary sequence number */
  spare4        varchar2(1000),         /* committed RID used by drop column */
  spare5        varchar2(1000),         
  spare6        date,                                       /* dml timestamp */
  inclcol_name  varchar(30),/* column where IOT splits into overflow segment */
  col_list      ku$_column_list_t,                        /* list of columns */
  nt            ku$_nt_parent_t,                            /* nested tables */
  con0_list     ku$_constraint0_list_t,               /* list of constraints */
  con1_list     ku$_constraint1_list_t,               /* list of constraints */
  con2_list     ku$_constraint2_list_t,               /* list of constraints */
  iov           ku$_ov_table_t,                            /* overflow table */
  exttab        ku$_exttab_t                      /* external table metadata */
)
/
grant execute on ku$_iotable_t to public
/

create or replace force view ku$_iotable_view of ku$_iotable_t 
  with object OID(obj_num)
  as select '1','0',
         t.obj#,
         value(o),
         (select value(s) from ku$_storage_view s
          where i.file# = s.file_num
          and i.block#  = s.block_num
          and i.ts#     = s.ts_num),
         ts.name, ts.blocksize,
         t.dataobj#, t.bobj#, t.tab#, t.cols,
         t.clucols, i.pctfree$, i.initrans, i.maxtrans, 
         mod(i.pctthres$,256), i.spare2, t.flags,
         t.audit$, t.rowcnt, t.blkcnt, t.empcnt, t.avgspc, t.chncnt, t.avgrln,
         t.avgspc_flb, t.flbcnt, t.analyzetime, t.samplesize, t.degree,
         t.instances, t.intcols, t.kernelcols, t.property, t.trigflag,
         t.spare1, t.spare2, t.spare3, t.spare4, t.spare5, t.spare6,
         (select c.name from col$ c
                 where c.obj# = t.obj#
                 and   c.col# = i.trunccnt and i.trunccnt != 0
		 and   bitand(c.property,1)=0),
         cast( multiset(select * from ku$_column_view c 
                        where c.obj_num = t.obj#
                        order by c.col_num, c.intcol_num
                        ) as ku$_column_list_t
              ),
         (select value(nt) from ku$_nt_parent_view nt
          where nt.obj_num = t.obj#),
         cast( multiset(select * from ku$_constraint0_view con
                        where con.obj_num = t.obj#
                        and con.contype not in (7,11)
                       ) as ku$_constraint0_list_t
             ),
         cast( multiset(select * from ku$_constraint1_view con
                        where con.obj_num = t.obj#
                       ) as ku$_constraint1_list_t
             ),
         cast( multiset(select * from ku$_constraint2_view con
                        where con.obj_num = t.obj#
                       ) as ku$_constraint2_list_t
             ),
         (select value(ov) from ku$_ov_table_view ov
          where ov.bobj_num = t.obj#
          and bitand(t.property, 128) = 128),  -- IOT has overflow
         (select value(etv) from ku$_exttab_view etv
                        where etv.obj_num = o.obj_num)
  from ku$_schemaobj_view o, tab$ t, ind$ i, ts$ ts
  where t.obj# = o.obj_num
        and t.pctused$ = i.obj#          -- For IOTs, pctused has index obj#
        and bitand(t.property, 64+512) = 64  -- IOT but not overflow
        and bitand(t.property, 32+8192) = 0     /* but not partitioned    */
                                                /* or nested table        */
        and  i.ts# = ts.ts#
        AND     (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num, 0) OR 
                EXISTS ( SELECT * FROM session_roles 
                        WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_iotable_view to public
/

-- ADT for partitioned IOT overflow table partition
create or replace type ku$_ov_tabpart_t as object
(
  obj_num       number,                                              /* obj# */
  bobj_num      number,                           /* base obj# (cluster/iot) */
  part_num      number,                                  /* partition number */
  storage       ku$_storage_t,                                    /* storage */
  ts_name       varchar2(30),                             /* tablespace name */
  blocksize     number,                            /* size of block in bytes */
  pct_free      number,                   /* min. free space %age in a block */
  pct_used      number,                   /* min. used space %age in a block */
  initrans      number,                     /* initial number of transaction */
  maxtrans      number,                     /* maximum number of transaction */
  flags         number                                              /* flags */
)
/
grant execute on ku$_ov_tabpart_t to public
/
create or replace type ku$_ov_tabpart_list_t as table of ku$_ov_tabpart_t;
/
grant execute on ku$_ov_tabpart_list_t to public
/

create or replace force view ku$_ov_tabpart_view of ku$_ov_tabpart_t 
  with object OID(obj_num)
  as select t.obj#, t.bo#, t.part#,
         (select value(s) from ku$_storage_view s
          where t.file# = s.file_num
          and t.block#  = s.block_num
          and t.ts#     = s.ts_num),
         ts.name, ts.blocksize,
         t.pctfree$, t.pctused$, t.initrans, t.maxtrans, t.flags
  from  tabpart$ t, ts$ ts
  where t.ts# = ts.ts#
/
grant select on ku$_ov_tabpart_view to select_catalog_role
/

-- ADT for partitioned IOTs

create or replace type ku$_iot_partobj_t as object
(
  obj_num       number,                                     /* obj# of table */
  tabpartobj    ku$_partobj_t,                       /* table partition info */
  partcols      ku$_part_col_list_t,         /* list of partitioning columns */
  subpartcols   ku$_part_col_list_t,      /* list of subpartitioning columns */
  indpartobj    ku$_partobj_t,                       /* index partition info */
  ovpartobj     ku$_partobj_t,                    /* overflow partition info */
  part_list     ku$_piot_part_list_t,                      /* partition list */
  iov_list      ku$_ov_tabpart_list_t            /* overflow part table list */
)
/
grant execute on ku$_iot_partobj_t to public
/
create or replace force view ku$_iot_partobj_view of ku$_iot_partobj_t
  with object identifier (obj_num) as
  select t.obj#,
         (select value (tpo) from ku$_partobj_view tpo
          where t.obj# = tpo.obj_num),
         cast(multiset(select * from ku$_tab_part_col_view pc
                       where pc.obj_num = t.obj#
                        order by pc.pos_num
                      ) as ku$_part_col_list_t
             ),
         cast(multiset(select * from ku$_tab_subpart_col_view sc
                       where sc.obj_num = t.obj#
                        order by sc.pos_num
                      ) as ku$_part_col_list_t
             ),
         (select value (ipo) from ku$_partobj_view ipo
          where i.obj# = ipo.obj_num),
         (select value (ovpo) from ku$_partobj_view ovpo
          where t.bobj# = ovpo.obj_num),
         cast(multiset(select * from ku$_piot_part_view ip
                       where ip.base_obj_num = i.obj#
                        order by ip.part_num
                      ) as ku$_piot_part_list_t
             ),
         cast(multiset(select * from ku$_ov_tabpart_view ovp
                       where ovp.bobj_num = t.bobj#
                        order by ovp.part_num
                      ) as ku$_ov_tabpart_list_t
             )
  from tab$ t, ind$ i 
  where i.bo#=t.obj#
    and i.type#=4                       -- iot index
    and bitand(t.property,32)=32        -- partitioned table
/
grant select on ku$_iot_partobj_view to select_catalog_role
/

-- ADT for Partitioned Index-Organized TABLEs
create or replace type ku$_piotable_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(2),                              /* UDT minor version # */
  obj_num       number,                                              /* obj# */
  schema_obj    ku$_schemaobj_t,                            /* schema object */
  storage       ku$_storage_t,                                    /* storage */
  ts_name       varchar2(30),                             /* tablespace name */
  blocksize     number,                            /* size of block in bytes */
  dataobj_num   number,                                /* data layer object# */
  bobj_num      number,                           /* base obj# (cluster/iot) */
  tab_num       number,                  /* # in cluster, null if !clustered */
  cols          number,                                      /* # of columns */
  clucols       number,                 /* # of clustered cols, if clustered */
  pct_free      number,                   /* min. free space %age in a block */
  initrans      number,                     /* initial number of transaction */
  maxtrans      number,                     /* maximum number of transaction */
  pct_thresh    number,                            /* pctthreshold for IOTs. */
  numkeycols    number,      /* # of key columns in compressed prefix (IOTs) */
  flags         number,                                             /* flags */
  audit_val     varchar2(38),                            /* auditing options */
  rowcnt        number,                                    /* number of rows */
  blkcnt        number,                                  /* number of blocks */
  empcnt        number,                            /* number of empty blocks */
  avgspc        number,                      /* average available free space */
  chncnt        number,                            /* number of chained rows */
  avgrln        number,                                /* average row length */
  avgspc_flb    number,       /* avg avail free space of blocks on free list */
  flbcnt        number,                             /* free list block count */
  analyzetime   date,                        /* timestamp when last analyzed */
  samplesize    number,                 /* number of rows sampled by Analyze */
  degree        number,                       /* # of PQ slaves per instance */
  instances     number,                         /* # of OPS instances for PQ */
  intcols       number,                             /* # of internal columns */
  kernelcols    number,                   /* number of REAL (kernel) columns */
  property      number,                                  /* table properties */
  trigflag      number,                              /* inline trigger flags */
  spare1        number,                       /* used to store hakan_kqldtvc */
  spare2        number,         /* committed partition # used by drop column */
  spare3        number,                           /* summary sequence number */
  spare4        varchar2(1000),         /* committed RID used by drop column */
  spare5        varchar2(1000),         
  spare6        date,                                       /* dml timestamp */
  inclcol_name  varchar(30),/* column where IOT splits into overflow segment */
  col_list      ku$_pcolumn_list_t,                       /* list of columns */
  nt            ku$_nt_parent_t,                            /* nested tables */
  con0_list     ku$_constraint0_list_t,               /* list of constraints */
  con1_list     ku$_constraint1_list_t,               /* list of constraints */
  con2_list     ku$_constraint2_list_t,               /* list of constraints */
  iov           ku$_ov_table_t,                            /* overflow table */
  part_obj      ku$_iot_partobj_t,                         /* partition info */
  exttab        ku$_exttab_t                      /* external table metadata */
)
/
grant execute on ku$_piotable_t to public
/
create or replace force view ku$_piotable_view of ku$_piotable_t 
  with object OID(obj_num)
  as select '1','0',
         t.obj#,
         value(o),
         (select value(s) from ku$_storage_view s
          where i.file# = s.file_num
          and i.block#  = s.block_num
          and i.ts#     = s.ts_num),
         ts.name, ts.blocksize,
         t.dataobj#, t.bobj#, t.tab#, t.cols,
         t.clucols, i.pctfree$, i.initrans, i.maxtrans, 
         mod(i.pctthres$,256), i.spare2, t.flags,
         t.audit$, t.rowcnt, t.blkcnt, t.empcnt, t.avgspc, t.chncnt, t.avgrln,
         t.avgspc_flb, t.flbcnt, t.analyzetime, t.samplesize, t.degree,
         t.instances, t.intcols, t.kernelcols, t.property, t.trigflag,
         t.spare1, t.spare2, t.spare3, t.spare4, t.spare5, t.spare6,
         (select c.name from col$ c
                 where c.obj# = t.obj#
                 and   c.col# = i.trunccnt and i.trunccnt != 0
		 and   bitand(c.property,1)=0),
         cast( multiset(select * from ku$_pcolumn_view c 
                        where c.obj_num = t.obj#
                        order by c.col_num, c.intcol_num
                        ) as ku$_pcolumn_list_t
              ),
         (select value(nt) from ku$_nt_parent_view nt
          where nt.obj_num = t.obj#),
         cast( multiset(select * from ku$_constraint0_view con
                        where con.obj_num = t.obj#
                        and con.contype not in (7,11)
                       ) as ku$_constraint0_list_t
             ),
         cast( multiset(select * from ku$_constraint1_view con
                        where con.obj_num = t.obj#
                       ) as ku$_constraint1_list_t
             ),
         cast( multiset(select * from ku$_constraint2_view con
                        where con.obj_num = t.obj#
                       ) as ku$_constraint2_list_t
             ),
         (select value(ov) from ku$_ov_table_view ov
          where ov.bobj_num = t.obj#),
         (select value (po) from ku$_iot_partobj_view po
          where t.obj# = po.obj_num),
         (select value(etv) from ku$_exttab_view etv
                        where etv.obj_num = o.obj_num)
  from ku$_schemaobj_view o, tab$ t, ind$ i, ts$ ts
  where t.obj# = o.obj_num
        and t.pctused$ = i.obj#          -- For IOTs, pctused has index obj#
        and bitand(t.property, 64+512)  = 64  -- IOT but not overflow
        and bitand(t.property, 32+8192) = 32  -- partitioned but not nested tbl
        and  i.ts# = ts.ts#
        AND     (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num,0) OR 
                EXISTS ( SELECT * FROM session_roles 
                        WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_piotable_view to public
/
-------------------------------------------------------------------------------
--                     TABLE, MVIEW, MVIEW_LOG PROPERTIES
-------------------------------------------------------------------------------

-- View to retrieve table properties; dbms_metadata.get_object uses
--  this to figure out which object_flags to set for a table
-- Note that the order of tables in the from-clause is important:
-- putting tab$ last avoids a full scan of user$ for queries of
-- the form 'schema not in (...)'

create or replace force view ku$_tabprop_view
 (obj_num,name,schema,flags,property)
 as
 select o.obj#, o.name, u.name, o.flags, t.property
 from obj$ o, user$ u, tab$ t
 where o.owner# = u.user#
 and   o.obj#   = t.obj#
/
grant select on ku$_tabprop_view to select_catalog_role
/

-- View to do the same for materialized views.

create or replace force view sys.ku$_mvprop_view
 (obj_num,name,schema,flags,property)
 as
 select o.obj#, o.name, u.name, o.flags, t.property
 from obj$ o, tab$ t, user$ u, snap$ s
 where o.owner# = u.user#
 and   o.obj#   = t.obj#
 and   s.sowner = u.name
 and   s.tname  = o.name
/
grant select on sys.ku$_mvprop_view to select_catalog_role
/

-- ...and for materialized view logs.

create or replace force view sys.ku$_mvlprop_view
 (obj_num,name,schema,flags,property)
 as
 select o.obj#, m.master, m.mowner, o.flags, t.property
 from obj$ o, tab$ t, user$ u, mlog$ m
 where o.owner# = u.user#
 and   o.obj#   = t.obj#
 and   m.mowner = u.name
 and   m.log = o.name
/
grant select on sys.ku$_mvlprop_view to select_catalog_role
/


-------------------------------------------------------------------------------
--                              TABLES BY TABLESPACE
-------------------------------------------------------------------------------

-- Partitioned and non-partitioned tables in tablespaces.
-- This view is used to implement the TABLESPACE filter.
--
-- tab$.property bits:
--	32	table is partitioned
--	64	index only table
--	512	IOT overflow segment
--	8192	is a nested table

create or replace view sys.ku$_tts_view ( owner_num, obj_num, ts_name ) as
  select o.owner#, t.obj#, ts.name		-- unpartitioned heap tables
  from   sys.obj$ o, sys.tab$ t, sys.ts$ ts
  where  t.ts#  = ts.ts#
  and    o.obj# = t.obj#
  and    bitand(t.property, 32+64+512+8192) = 0
 UNION ALL
  select o.owner#, t.obj#, ts.name		-- simple partitions
  from   sys.obj$ o, sys.tab$ t, sys.tabpart$ tp, sys.ts$ ts
  where  tp.ts# = ts.ts#
  and    t.obj# = tp.bo#
  and    o.obj# = t.obj#
  and    bitand(t.property, 32+64+512+8192) = 32
 UNION ALL
  select o.owner#, t.obj#, ts.name		-- composite partitions
  from   sys.obj$ o, sys.tab$ t,
         sys.tabcompart$ tcp, sys.tabsubpart$ tsp, sys.ts$ ts
  where  tsp.ts#  = ts.ts#
  and    tcp.obj# = tsp.pobj#
  and    t.obj#   = tcp.bo#
  and    o.obj# = t.obj#
  and    bitand(t.property, 32+64+512+8192) = 32
 UNION ALL
  select o.owner#, t.obj#, ts.name		-- unpartitioned IOTs
  from   sys.obj$ o, sys.tab$ t, sys.ind$ i, sys.ts$ ts
  where  i.ts#    = ts.ts#
    and  i.obj#   = t.pctused$
  and    o.obj# = t.obj#
  and    bitand(t.property, 32+64+512+8192) = 64
 UNION ALL
  select o.owner#, t.obj#, ts.name		-- PIOTs
  from   sys.obj$ o, sys.tab$ t, sys.indpart$ ip, sys.ts$ ts
  where  ip.ts#   = ts.ts#
    and  ip.bo#   = t.pctused$
  and    o.obj# = t.obj#
  and    bitand(t.property, 32+64+512+8192) = 32 + 64
/
grant select on sys.ku$_tts_view to select_catalog_role;

create or replace view sys.ku$_tab_ts_view ( owner_num, obj_num, ts_name ) as
  select t.owner_num, t.obj_num, t.ts_name
  from   sys.ku$_tts_view t
  where  (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (t.owner_num,0) OR 
                EXISTS ( SELECT * FROM session_roles 
                        WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on sys.ku$_tab_ts_view to public
/

-------------------------------------------------------------------------------
--                              PROC./FUNC./PACKAGE
-------------------------------------------------------------------------------

-- common adt for procedures, functions, packages and package bodies
create or replace type ku$_proc_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(2),                              /* UDT minor version # */
  obj_num       number,                                     /* object number */
  type_num      number,                                       /* type number */
  schema_obj    ku$_schemaobj_t,                            /* schema object */
  source_lines  ku$_source_list_t                            /* source lines */
)
/
grant execute on ku$_proc_t to public
/

-- base view for procedures, functions, packages and package bodies

create or replace force view ku$_base_proc_view of ku$_proc_t
  with object identifier (obj_num) as
  select '1','1',
         oo.obj#,
         oo.type#,
         value(o),
         sys.dbms_metadata_util.get_source_lines(oo.name,oo.obj#,oo.type#)
  from  ku$_schemaobj_view o, sys.obj$ oo
  where (oo.type# = 7 or oo.type# = 8 or oo.type# = 9 or oo.type# = 11)
    and oo.obj#  = o.obj_num
         AND (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num, 0) OR 
              EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_base_proc_view to select_catalog_role
/
-- particular views for procedures, functions, packages and package bodies

create or replace force view ku$_proc_view of ku$_proc_t
  with object identifier (obj_num) as
  select t.vers_major, t.vers_minor, t.obj_num, t.type_num,
         t.schema_obj, t.source_lines
  from ku$_base_proc_view t
  where t.type_num = 7
/
grant select on ku$_proc_view to public
/

create or replace force view ku$_func_view of ku$_proc_t
  with object identifier (obj_num) as
  select t.vers_major, t.vers_minor, t.obj_num, t.type_num,
         t.schema_obj, t.source_lines
  from ku$_base_proc_view t
  where t.type_num = 8
/
grant select on ku$_func_view to public
/

create or replace force view ku$_pkg_view of ku$_proc_t
  with object identifier (obj_num) as
  select t.vers_major, t.vers_minor, t.obj_num, t.type_num,
         t.schema_obj, t.source_lines
  from ku$_base_proc_view t
  where t.type_num = 9
/
grant select on ku$_pkg_view to public
/

create or replace force view ku$_pkgbdy_view of ku$_proc_t
  with object identifier (obj_num) as
  select t.vers_major, t.vers_minor, t.obj_num, t.type_num,
         t.schema_obj, t.source_lines
  from ku$_base_proc_view t
  where t.type_num = 11
/
grant select on ku$_pkgbdy_view to public
/

create or replace type ku$_full_pkg_t as object
(
  vers_major            char(1),                      /* UDT major version # */
  vers_minor            char(2),                      /* UDT minor version # */
  obj_num               number,                             /* object number */
  schema_obj            ku$_schemaobj_t,                    /* schema object */
  package_t             ku$_proc_t,                        /* package header */
  package_body_t        ku$_proc_t                           /* package body */
)
/

grant execute on ku$_full_pkg_t to public
/

create or replace force view ku$_full_pkg_view of ku$_full_pkg_t
  with object identifier (obj_num) as
  select '1','0',
         oo.obj#,
         value(o),
         value(p),
         (select value(pb) from ku$_pkgbdy_view pb
          where oo.name  = pb.schema_obj.name
          and o.owner_name  = pb.schema_obj.owner_name)
  from   sys.obj$ oo, ku$_schemaobj_view o, ku$_pkg_view p
  where oo.type# = 9
    and oo.obj#  = o.obj_num
    and oo.obj#  = p.schema_obj.obj_num
         AND (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num, 0) OR 
              EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_full_pkg_view to public
/

-------------------------------------------------------------------------------
--                              OPERATOR
-------------------------------------------------------------------------------

-- ADT for operator arguments
create or replace type ku$_oparg_t as object
(
  obj_num       number,                            /* operator object number */
  bind_num      number,                      /* binding this arg. belongs to */
  position      number,                   /* position of the arg in the bind */
  type          varchar2(61)                          /* datatype of the arg */
)
/
grant execute on ku$_oparg_t to public
/
create or replace type ku$_oparg_list_t as TABLE of ku$_oparg_t
/
grant execute on ku$_oparg_list_t to public
/
-- NOTE: We don't need a view of ku$_oparg_t's because they're instantiated
-- directly from sys.oparg$ in the higher level views that follow.

-- Simplified ADT for listing primary operators for ancillary operators
create or replace type ku$_opancillary_t as object
(
  obj_num       number,                  /* object number of ANCILLARY oper. */
  bind_num      number,                  /* bind number for the ancillary op */
  primop_num    number,          /* object number of PRIMARY for this ancil. */
  primop_obj    ku$_schemaobj_t,                /* schema object for PRIMARY */
  args          ku$_oparg_list_t               /* arguments for this primary */
)
/
grant execute on ku$_opancillary_t to public
/
create or replace type ku$_opancillary_list_t as TABLE of ku$_opancillary_t
/
grant execute on ku$_opancillary_list_t to public
/

-- View for primaries of ancillaries
create or replace force view ku$_opancillary_view of ku$_opancillary_t
  with object identifier (obj_num, bind_num, primop_num) as
  select oa.obj#, oa.bind#, oa.primop#,
         (select value(o) from sys.ku$_schemaobj_view o
                 where o.obj_num=oa.primop#),
         cast(multiset(select * from sys.oparg$ arg
                       where arg.obj#=oa.primop# and arg.bind#=oa.primbind#
                       order by arg.position
                      ) as ku$_oparg_list_t
             )
  from sys.opancillary$ oa
/
grant select on ku$_opancillary_view to select_catalog_role
/
-- ADT for operator bindings
create or replace type ku$_opbinding_t as object
(
  obj_num       number,                            /* operator object number */
  bind_num      number,                            /* number of this binding */
  functionname  varchar2(92),                /* func that impl. this binding */
  returnschema  varchar2(30),              /* schema of return type (if ADT) */
  returntype    varchar2(30),                     /* return type of function */
  impschema     varchar2(30),             /* indextype implementation schema */
  imptype       varchar2(30),               /* indextype implementation type */
  property      number,                                    /* property flags */
  spare1        varchar2(30),
  spare2        varchar2(30),
  spare3        number,
  args          ku$_oparg_list_t,              /* arguments for this binding */
  ancillaries   ku$_opancillary_list_t  /* list of primary ops for this ancil*/
)
/
grant execute on ku$_opbinding_t to public
/
create or replace type ku$_opbinding_list_t as TABLE of ku$_opbinding_t
/
grant execute on ku$_opbinding_list_t to public
/

-- View for operator bindings
create or replace force view ku$_opbinding_view of ku$_opbinding_t
  with object identifier(obj_num, bind_num) as
  select ob.obj#, ob.bind#, ob.functionname, ob.returnschema, 
	 ob.returntype, ob.impschema, ob.imptype, ob.property, 
	 ob.spare1, ob.spare2, ob.spare3,
         cast(multiset(select * from sys.oparg$ oa
                       where oa.obj#=ob.obj# and oa.bind#=ob.bind#
                       order by oa.position
                      ) as ku$_oparg_list_t
             ),
         cast(multiset(select value(a) from sys.ku$_opancillary_view a
                       where ob.obj#=a.obj_num and ob.bind#=a.bind_num
                      ) as ku$_opancillary_list_t
             )
  from sys.opbinding$ ob
/
grant select on ku$_opbinding_view to select_catalog_role
/

-- ADT for operators
create or replace type ku$_operator_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(2),                              /* UDT minor version # */
  obj_num       number,                            /* operator object number */
  schema_obj    ku$_schemaobj_t,                    /* base schema obj. info */
  property      number,                                    /* property flags */
  bindings      ku$_opbinding_list_t       /* List of bindings for this oper */
)
/
grant execute on ku$_operator_t to public
/
-- View for operators
create or replace force view ku$_operator_view of ku$_operator_t
  with object identifier(obj_num) as
  select '1','0',
         op.obj#, value(o), op.property,
         cast(multiset(select value(ob) from ku$_opbinding_view ob
                       where ob.obj_num=op.obj#
                       order by ob.bind_num
                      ) as ku$_opbinding_list_t
             )
  from  ku$_schemaobj_view o, sys.operator$ op
  where op.obj#=o.obj_num
        and (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num, 0) OR 
              EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_operator_view to public
/

-------------------------------------------------------------------------------
--                              INDEXTYPE OPERATORS
-------------------------------------------------------------------------------

-- ADT for operators supported by indextypes. This is significantly
-- simplified from the full-blown ku$_operator_t above. Notice it also does
-- not include the filter or property info. found in indop$ as this
-- functionality was dropped.
create or replace type ku$_indexop_t as object
(
  obj_num       number,                          /* obj# of parent indextype */
  oper_num      number,                 /* obj# of operator for this binding */
  bind_num      number,                            /* number of this binding */
  oper_obj      ku$_schemaobj_t,             /* sch. info. for this operator */
  args          ku$_oparg_list_t           /* arguments for this op. binding */
)
/
grant execute on ku$_indexop_t to public
/
create or replace type ku$_indexop_list_t as TABLE of ku$_indexop_t
/
grant execute on ku$_indexop_list_t to public
/

create or replace force view ku$_indexop_view of ku$_indexop_t
  with object identifier(obj_num, oper_num, bind_num) as
  select io.obj#, io.oper#, io.bind#,
         (select value(o) from sys.ku$_schemaobj_view o
          where io.oper#=o.obj_num),
         cast(multiset(select * from sys.oparg$ oa
                       where oa.obj#=io.oper# and oa.bind#=io.bind#
                       order by oa.position
                      ) as ku$_oparg_list_t
             )
  from sys.indop$ io
/
grant select on ku$_indexop_view to select_catalog_role
/

create or replace type ku$_indextype_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(2),                              /* UDT minor version # */
  obj_num       number,                                 /* obj# of indextype */
  schema_obj    ku$_schemaobj_t,                    /* base schema obj. info */
  impl_obj      ku$_schemaobj_t,        /* sch. info for implementation type */
  operators     ku$_indexop_list_t
)
/
grant execute on ku$_indextype_t to public
/
create or replace force view ku$_indextype_view of ku$_indextype_t
  with object identifier(obj_num) as
  select '1','0',
         it.obj#, value(o),
         (select value(oit) from sys.ku$_schemaobj_view oit
          where it.implobj#=oit.obj_num),
          cast(multiset(select value(io) from sys.ku$_indexop_view io
                        where it.obj#=io.obj_num
                        /* the following order-by clause exists solely
                           to assure repeatable regression tests results */
                        order by io.oper_obj.owner_name, io.oper_obj.name
                       ) as ku$_indexop_list_t
              )
  from  sys.ku$_schemaobj_view o, sys.indtypes$ it
  where it.obj#=o.obj_num
        AND (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num, 0) OR 
              EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_indextype_view to public
/

-------------------------------------------------------------------------------
--                              OBJECT GRANTS
-------------------------------------------------------------------------------

-- Do Object Grants
create or replace type ku$_objgrant_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(2),                              /* UDT minor version # */
  obj_num       number,                                 /* obj# of base obj. */
  base_obj      ku$_schemaobj_t,                           /* base obj. info */
  grantor       varchar2(30),
  grantee       varchar2(30),
  privname      varchar2(40),
  sequence      number,                        /* Unique seq# for this grant */
  wgo           number,                             /* with grant option = 1 */
  colname       varchar2(30))          /* column name if col grant else null */
/

grant execute on ku$_objgrant_t to public
/

-- View for object grants
create or replace force view ku$_objgrant_view of ku$_objgrant_t
  with object identifier (sequence) as
  select '1','0',
         g.obj#, value(o), u1.name, u2.name, p.name, g.sequence#,
         mod(NVL(g.option$,0),2),
         (select c.name from sys.col$ c where g.obj#=c.obj# and g.col#=c.col#)
  from sys.ku$_schemaobj_view o, sys.objauth$ g, sys.user$ u1, sys.user$ u2,
       sys.table_privilege_map p
  where g.obj#=o.obj_num and
        g.grantor#=u1.user# and
        g.grantee#=u2.user# and
        g.privilege#=p.privilege and
        (SYS_CONTEXT('USERENV','CURRENT_USERID')
                IN (g.grantor#, g.grantee#, o.owner_num, 0) OR
                g.grantee#=1 OR
                EXISTS ( SELECT * FROM session_roles 
                        WHERE role='SELECT_CATALOG_ROLE' ))
/
-- The security clause allows a non-privileged user the same access
-- as the catalog views. Note comment on ALL_TAB_PRIVS:
-- 'Grants on objects for which the user is the grantor, grantee, owner, 
-- or an enabled role or PUBLIC is the grantee'

grant select on ku$_objgrant_view to public
/

-------------------------------------------------------------------------------
--                              SYSTEM GRANTS
-------------------------------------------------------------------------------

create or replace type ku$_sysgrant_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(1),                              /* UDT minor version # */
  grantee       varchar2(30),
  privname      varchar2(40),
  sequence      number,
  wgo           number)
/
grant execute on ku$_sysgrant_t to public
/

-- View for system priv. grants
create or replace force view ku$_sysgrant_view of ku$_sysgrant_t
  with object identifier (sequence) as
  select '1','0',
         u.name, p.name, g.sequence#, NVL(g.option$,0)
  from sys.sysauth$ g, sys.user$ u, sys.system_privilege_map p
  where g.grantee#=u.user# and
        g.privilege#=p.privilege and
        bitand(p.property, 1) != 1         /* Don't show non-SQL sys. grants */
        and
        (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (g.grantee#, 0) OR 
                g.grantee#=1 OR
                EXISTS ( SELECT * FROM session_roles 
                        WHERE role='SELECT_CATALOG_ROLE' ))
/
-- The security clause allows a non-privileged user *almost* the same access
-- as the catalog views DBA/USER_SYS_PRIVS: system privs granted to user
-- or to PUBLIC.
-- The exception is that the catalog views include grants with
-- the "no-export" property.

-- Do we need the roles enumerated in exu8spv in catexp excluded in
-- the above where clause?

grant select on ku$_sysgrant_view to public
/

-------------------------------------------------------------------------------
--                              TRIGGERS
-------------------------------------------------------------------------------
create or replace type ku$_trigger_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(2),                              /* UDT minor version # */
  obj_num       number,                                   /* obj# of trigger */
  schema_obj    ku$_schemaobj_t,                 /* object info. for trigger */
  base_obj      ku$_schemaobj_t,    /* object info. for base obj: May be null*/
  type_num      number,                                     /* trigger type: */
   /* 0=before table, 1=before row, 2=after table, 3=after row, 4=instead of */
  act_update    number,                                    /* fire on update */
  act_insert    number,                                    /* fire on insert */
  act_delete    number,                                    /* fire on delete */
  definition    varchar2(4000),                /* text of trigger definition */
  whenclause    varchar2(4000),                       /* text of when clause */
  body          varchar2(4000),                      /* text of trigger body */
  body_vcnt     ku$_vcnt,                            /* text of trigger body */
  body_len      number,                            /* length of trigger body */
  enabled       number,                         /* 0 = DISABLED, 1 = ENABLED */
  property      number,                   /* trigger properties: see sql.bsq */
  sys_evts      number                          /* system events for trigger */
)
/
grant execute on ku$_trigger_t to public
/
create or replace force view ku$_trigger_view of ku$_trigger_t
  with object identifier(obj_num) as
  select '1','0',
         t.obj#, value(o),
         (select value(bo) from ku$_schemaobj_view bo
          where t.baseobject = bo.obj_num),
         t.type#, t.update$, t.insert$, t.delete$, t.definition, t.whenclause,
         sys.dbms_metadata_util.long2varchar(t.actionsize,
                                        'SYS.TRIGGER$',
                                        'ACTION#',
                                        t.rowid),
         case when t.actionsize > 4000
           then
             sys.dbms_metadata_util.long2vcmax(t.actionsize,
                                        'SYS.TRIGGER$',
                                        'ACTION#',
                                        t.rowid)
           else NULL
         end,
         t.actionsize,                          
         t.enabled, t.property, t.sys_evts
  from   sys.ku$_schemaobj_view o, sys.trigger$ t
  where  t.obj# = o.obj_num AND
         (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num, 0) OR 
                EXISTS ( SELECT * FROM session_roles 
                        WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_trigger_view to public
/

-------------------------------------------------------------------------------
--                              VIEWS
-------------------------------------------------------------------------------

create or replace type ku$_view_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(2),                              /* UDT minor version # */
  obj_num       number,                                              /* obj# */
  schema_obj    ku$_schemaobj_t,                            /* schema object */
  audit_val     varchar2(38),                            /* auditing options */
  cols          number,                                      /* # of columns */
  intcols       number,                             /* # of internal columns */
  property      number,                                  /* table properties */
                /* 0x0001 =       1 = this is typed view                     */
                /* 0x0002 =       2 = view has ADT column(s)                 */
                /* 0x0004 =       4 = view has nested table column(s)        */
                /* 0x0008 =       8 = view has REF column(s)                 */
                /* 0x0010 =      16 = view has array column(s)               */
                /* 0x1000 =    4096 = view has primary key-based oid         */
                /* 0x4000 =   16384 = view is read-only                      */
            /* 0x08000000 =         = view is a sub view                     */
            /* 0x10000000 =         = view is packed object view             */
  flags         number,                                             /* flags */
                /* 0x0800 =    2048 = view/table has security policy         */
                /* 0x1000 =    4096 = view is insertable via trigger         */
                /* 0x2000 =    8192 = view is updatable via trigger          */
                /* 0x4000 =   16384 = view is deletable via trigger          */
             /* 0x0400000 =         = view has sub views defined under it    */
  textlength    number,                               /* length of view text */
  text          clob,                                           /* view text */
  textvcnt      ku$_vcnt,                                       /* view text */
  col_list      ku$_simple_col_list_t,    /* list of relational view columns */
  col_list2     ku$_column_list_t,            /* list of object view columns */
  owner_name    varchar2(30),                           /* owner of row type */
  name          varchar2(30),                            /* name of row type */
  typetextlength  number,                         /* length of row type text */
  typetext        varchar2(4000),                               /* type text */
  oidtextlength   number,                              /* length of oid text */
  oidtext         varchar2(4000),                                /* oid text */
  transtextlength number,                 /* length of transformed view text */
  transtext       varchar2(4000),
  con1_list     ku$_constraint1_list_t,               /* list of constraints */
  con2_list     ku$_constraint2_list_t                /* list of constraints */
)
/
grant execute on ku$_view_t to public
/
-- This version of the view uses long2clob to populate the text attribute.
-- long2clob can return view text of arbitrary length;
-- the old textvcnt attribute is retained for compatibility
-- but is always set to NULL.

create or replace force view ku$_view_view of ku$_view_t
  with object identifier (obj_num) as
  select '1','0',
         v.obj#,
         value(o),
         v.audit$, v.cols, v.intcols, v.property, v.flags, v.textlength,
         sys.dbms_metadata_util.long2clob(v.textlength,
                                        'SYS.VIEW$',
                                        'TEXT',
                                        v.rowid),
	 NULL,
         cast(multiset(select * from ku$_simple_col_view c
                       where c.obj_num = v.obj#
			 and (bitand(v.property,1)=0)
                        order by c.intcol_num
                      ) as ku$_simple_col_list_t
             ),
         cast(multiset(select * from ku$_column_view c
                       where c.obj_num = v.obj#
			 and (bitand(v.property,1)=1)
                        order by c.intcol_num
                      ) as ku$_column_list_t
             ),
         tv.typeowner, tv.typename, tv.typetextlength, tv.typetext,
         tv.oidtextlength, tv.oidtext, tv.transtextlength,
         sys.dbms_metadata_util.long2varchar(tv.transtextlength,
                                        'SYS.TYPED_VIEW$',
                                        'TRANSTEXT',
                                        tv.rowid),
         cast( multiset(select * from ku$_constraint1_view con
                        where con.obj_num = v.obj#
                       ) as ku$_constraint1_list_t
             ),
         cast( multiset(select * from ku$_constraint2_view con
                        where con.obj_num = v.obj#
                       ) as ku$_constraint2_list_t
             )
  from ku$_schemaobj_view o, sys.obj$ oo, sys.view$ v, sys.typed_view$ tv
  where oo.obj# = o.obj_num
    and oo.obj# = v.obj#
    and oo.obj# = tv.obj# (+)
         AND (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num, 0) OR 
              EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_view_view to public
/

-------------------------------------------------------------------------------
--                              OUTLINES
-------------------------------------------------------------------------------
-- Types and views to support OUTLINEs

create or replace type ku$_outline_hint_t as object
( name              varchar2(30),                           /* outline name */
  hint              number,               /* which hint for a given outline */
  category          varchar2(30),               /* collection/grouping name */
  hint_type         number,                                 /* type of hint */
  hint_text         varchar2(512),             /* hint specific information */
  stage             number,            /* stage of hint generation/applic'n */
  node              number,                                  /* QBC node id */
  table_name        varchar2(30),                       /* for ORDERED hint */
  table_tin         number,                        /* table instance number */
  table_pos         number,                             /* for ORDERED hint */
  ref_id            number,        /* node id that this hint is referencing */
  user_table_name   varchar2(30),  /* table name to which this hint applies */
  cost              double precision,    /* optimizer estimated cost of the
                                                           hinted operation */
  cardinality       double precision,    /* optimizer estimated cardinality
                                                    of the hinted operation */
  bytes             double precision,     /* optimizer estimated byte count
                                                    of the hinted operation */
  hint_textoff      number,             /* offset into the SQL statement to
                                                    which this hint applies */
  hint_textlen      number,     /* length of SQL to which this hint applies */
  join_pred         varchar2(2000)      /* join predicate (applies only for 
                                                         join method hints) */
)
/
grant execute on ku$_outline_hint_t to public
/
create or replace type ku$_outline_hint_list_t
 as table of ku$_outline_hint_t
/
grant execute on ku$_outline_hint_list_t to public
/
create or replace type ku$_outline_node_t as object
(
  name          varchar2(30),                              /* outline name  */
  category      varchar2(30),                           /* outline category */
  node          number,                              /* qbc node identifier */
  parent        number,      /* node id of the parent node for current node */ 
  node_type     number,                                    /* qbc node type */
  node_textlen  number,         /* length of SQL to which this node applies */ 
  node_textoff  number       /* offset into the SQL statement to which this
                                                               node applies */
)
/
grant execute on ku$_outline_node_t to public
/
create or replace type ku$_outline_node_list_t
 as table of ku$_outline_node_t
/
grant execute on ku$_outline_node_list_t to public
/
create or replace type ku$_outline_t as object
(
  vers_major        char(1),                          /* UDT major version # */
  vers_minor        char(1),                          /* UDT minor version # */
  name              varchar2(30),         /* named is potentially generated */
  sql_text          varchar2(4000),          /* the SQL stmt being outlined */
  textlen           number,                           /* length of SQL stmt */
  signature         raw(16),                       /* signature of sql_text */
  hash_value        number,                  /* KGL's calculated hash value */
  category          varchar2(30),                          /* category name */
  version           varchar2(64),          /* db version @ outline creation */
  creator           varchar2(30),         /* user from whom outline created */
  timestamp         date,                               /* time of creation */
  flags             number,              /* e.g. everUsed, bindVars, dynSql */
  hintcount         number,               /* number of hints on the outline */
  hints             ku$_outline_hint_list_t,/* list of hints for this outln */
  nodes             ku$_outline_node_list_t           /* list of qbc blocks */
)
/
grant execute on ku$_outline_t to public
/
create or replace force view ku$_outline_view of ku$_outline_t
  with object identifier(name) as
  select '1','0',
         ol.ol_name,
         sys.dbms_metadata_util.long2varchar(ol.textlen,
                                    'OUTLN.OL$',
                                    'SQL_TEXT',
                                    ol.rowid),
         ol.textlen, ol.signature, ol.hash_value, ol.category, ol.version,
         ol.creator, ol.timestamp, ol.flags, ol.hintcount,
         cast(multiset(select ol_name,hint#,category,hint_type,hint_text,
                              stage#,node#,table_name,table_tin,table_pos,
                              ref_id,user_table_name,cost,cardinality,
                              bytes,hint_textoff,hint_textlen,join_pred 
                       from outln.ol$hints h
                       where h.ol_name = ol.ol_name order by h.stage#,h.node#
                      ) as ku$_outline_hint_list_t
             ),
         cast(multiset(select ol_name,category,node_id,parent_id,node_type,
                              node_textlen,node_textoff
                       from outln.ol$nodes n
                       where n.ol_name = ol.ol_name
                      ) as ku$_outline_node_list_t
             )
  from outln.ol$ ol, sys.user$ u
  where ol.creator=u.name
        AND (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (u.user#, 0) OR 
              EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_outline_view to public
/

-------------------------------------------------------------------------------
--                              SYNONYMS
-------------------------------------------------------------------------------
-- 
create or replace type ku$_synonym_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(2),                              /* UDT minor version # */
  obj_num       number,                             /* synonym object number */
  schema_obj    ku$_schemaobj_t,                    /* synonym schema object */
  syn_long_name varchar2(4000),                         /* synonym long name */
  db_link       varchar2(128),                                /* object node */
  owner_name    varchar2(30),                                /* object owner */
  name          varchar2(30),                                 /* object name */
  obj_long_name varchar2(4000)                           /* object long name */
)
/
grant execute on ku$_synonym_t to public
/
create or replace force view ku$_synonym_view of ku$_synonym_t
  with object identifier(obj_num) as
  select '1','0',
         s.obj#, value(o),
         (select j.longdbcs from sys.javasnm$ j where j.short = o.name),
         s.node, s.owner, s.name,
         (select j.longdbcs from sys.javasnm$ j where j.short = s.name)
  from  sys.ku$_schemaobj_view o, sys.syn$ s
  where s.obj# = o.obj_num AND
         (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num, 0) OR 
                o.owner_name='PUBLIC' or
                EXISTS ( SELECT * FROM session_roles 
                        WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_synonym_view to public
/

-------------------------------------------------------------------------------
--                              DIRECTORY
-------------------------------------------------------------------------------
-- 
create or replace type ku$_directory_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(1),                              /* UDT minor version # */
  obj_num       number,                           /* directory object number */
  schema_obj    ku$_schemaobj_t,                  /* directory schema object */
  audit_val     varchar2(38),                            /* auditing options */
  os_path       varchar2(4000)                             /* OS path string */
)
/
grant execute on ku$_directory_t to public
/
create or replace force view ku$_directory_view of ku$_directory_t
  with object identifier(obj_num) as
  select '1','0',
         o.obj#, value(sov),
         d.audit$,
         d.os_path
  from   sys.obj$ o, sys.ku$_schemaobj_view sov, sys.dir$ d
  where  o.obj# = sov.obj_num AND
         o.obj# = d.obj#
	 AND (SYS_CONTEXT('USERENV','CURRENT_USERID')= 0
		OR EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_directory_view to public
/

-------------------------------------------------------------------------------
--                           ROLLBACK SEGMENTS
-------------------------------------------------------------------------------
-- 
create or replace type ku$_rollback_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(1),                              /* UDT minor version # */
  us_num        number,                               /* undo segment number */
  name          varchar2(30),                       /* rollback segment name */
  user_num      number,                /* Owner: 0 = SYS(PRIVATE) 1 = PUBLIC */
  optimal       number,                                     /* optimal value */
  iniexts       number,                               /* initial extent size */
  minexts       number,                         /* minimum number of extents */
  maxexts       number,                         /* maximum number of extents */
  extsize       number,                          /* current next extent size */
                                           /* zero for bitmapped tablespaces */
  tablespace    ku$_tablespace_t                   /* tablespace information */
)
/
grant execute on ku$_rollback_t to public
/
create or replace force view ku$_rollback_view of ku$_rollback_t
  with object identifier(us_num) as
  select '1','0',
         u.us#,
         u.name,
         u.user#,
         u.optimal,
         s.iniexts,
         s.minexts,
         s.maxexts,
         s.extsize,
         value(tsv)
  from   sys.ku$_tablespace_view tsv, sys.seg$ s, sys.undo$ u
  where  u.ts# = tsv.ts_num
    and  u.file#  = s.file#
    and  u.block# = s.block#
    and  u.ts#    = s.ts#
    and (SYS_CONTEXT('USERENV','CURRENT_USERID') = 0
	OR EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_rollback_view to public
/

-------------------------------------------------------------------------------
--                           DATABASE LINKS
-------------------------------------------------------------------------------
-- 
create or replace type ku$_dblink_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(1),                              /* UDT minor version # */
  owner_name    varchar2(30),
  owner_num     number,
  name          varchar2(128),
  ctime         date,
  host          varchar2(2000),
  userid        varchar2(30),
  password      varchar2(30),
  flag          number,
  authusr       varchar2(30),
  authpwd       varchar2(30)
)
/
grant execute on ku$_dblink_t to public
/
create or replace force view ku$_dblink_view of ku$_dblink_t
  with object identifier(owner_num,name) as
  select '1','0',
         u.name,
	 l.owner#, l.name, l.ctime, l.host, l.userid,
	 l.password, l.flag, l.authusr, l.authpwd
  from   sys.user$ u, sys.link$ l
  where  u.user# = l.owner#
     AND (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (u.user#, 0) OR 
          EXISTS ( SELECT * FROM session_roles 
                   WHERE role='SELECT_CATALOG_ROLE' ))


/
grant select on ku$_dblink_view to public
/

-------------------------------------------------------------------------------
--                           TRUSTED LINKS
-------------------------------------------------------------------------------
-- 
create or replace type ku$_trlink_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(1),                              /* UDT minor version # */
  name          varchar2(132),                          /* trusted link name */
  function      varchar2(45),                                    /* function */
  type          number                               /* type of trusted link */
)
/
grant execute on ku$_trlink_t to public
/
create or replace force view ku$_trlink_view of ku$_trlink_t
  with object identifier(name) as
  select '1','0',
         tl.dbname,
         decode(tl.dbname, '+*', 'DBMS_DISTRIBUTED_TRUST_ADMIN.ALLOW_ALL',
                           '-*', 'DBMS_DISTRIBUTED_TRUST_ADMIN.DENY_ALL',
                fdef.function),
         decode(tl.dbname, '+*', 0, '-*', 0, 1)
  from   sys.trusted_list$ tl,
         ( select  decode (dbname,
                           '+*', 'DBMS_DISTRIBUTED_TRUST_ADMIN.DENY_SERVER',
                           '-*', 'DBMS_DISTRIBUTED_TRUST_ADMIN.ALLOW_SERVER')
                           function
           from    sys.trusted_list$
           where   dbname like '%*') fdef
  where (SYS_CONTEXT('USERENV','CURRENT_USERID') = 0 
	OR EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_trlink_view to public
/

-------------------------------------------------------------------------------
--                           Fine Grained Auditing
-------------------------------------------------------------------------------
-- 
create or replace type ku$_fga_policy_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(1),                              /* UDT minor version # */
  obj_num       number,                              /* parent object number */
  name          varchar2(30),                              /* name of policy */
  ptxt          varchar2(4000),                          /* policy predicate */
  pfschema      varchar2(30),                     /* schema of event handler */
  ppname        varchar2(30),                          /* event package name */
  pfname        varchar2(30),                         /* event funciton name */
  pcol          varchar2(30),                             /* relevent column */
  enable_flag   number,                         /* 0 = disabled, 1 = enabled */
  base_obj      ku$_schemaobj_t                        /* base Schema object */
)
/
grant execute on ku$_fga_policy_t to public
/
create or replace force view ku$_fga_policy_view of ku$_fga_policy_t
  with object identifier(obj_num, pfschema, name) as
  select '1','0',
         f.obj#,
         pname,
         replace ((ptxt), '''', ''''''),
         pfschma, ppname, pfname, pcol, enable_flag,
         value(sov)
  from   sys.ku$_schemaobj_view sov, sys.fga$ f
  where  f.obj# = sov.obj_num
     and (SYS_CONTEXT('USERENV','CURRENT_USERID') = 0 OR
          EXISTS ( SELECT * FROM session_roles
                   WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_fga_policy_view to public
/

-------------------------------------------------------------------------------
--              Fine Grained Access Control Administrative Interface
-------------------------------------------------------------------------------

create or replace type ku$_rls_policy_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(1),                              /* UDT minor version # */
  base_obj      ku$_schemaobj_t,                       /* base schema object */
  obj_num       number,                              /* parent object number */
  gname         varchar2(30),                        /* name of policy group */
  name          varchar2(30),                              /* name of policy */
  stmt_type     number,                        /* applicable statement type: */
  check_opt     number,                                 /* with check option */
  enable_flag   number,                         /* 0 = disabled, 1 = enabled */
  pfschma       varchar2(30),                   /* schema of policy function */
  ppname        varchar2(30),                         /* policy package name */
  pfname        varchar2(30),                        /* policy function name */
  ptype         number                                        /* policy type */
)
/
grant execute on ku$_rls_policy_t to public
/

create or replace force view ku$_rls_policy_view of ku$_rls_policy_t
  with object identifier (obj_num,name) as
  select '1','0',
          value(sov),
	  r.obj#, r.gname, r.pname, r.stmt_type, r.check_opt,
	  r.enable_flag, r.pfschma, r.ppname, r.pfname, r.ptype
  from    ku$_schemaobj_view sov, sys.rls$ r
  where   r.obj# = sov.obj_num
     and (SYS_CONTEXT('USERENV','CURRENT_USERID') = 0 OR
          EXISTS ( SELECT * FROM session_roles
                   WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_rls_policy_view to PUBLIC
/

create or replace type ku$_rls_group_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(1),                              /* UDT minor version # */
  base_obj      ku$_schemaobj_t,                       /* base schema object */
  obj_num       number,                              /* parent object number */
  name          varchar2(30)                         /* name of policy group */
)
/
grant execute on ku$_rls_group_t to public
/

create or replace force view ku$_rls_group_view of ku$_rls_group_t
  with object identifier (name,obj_num) as
  select '1','0',
          value(sov),
          rg.obj# , rg.gname
  from    ku$_schemaobj_view sov, sys.rls_grp$ rg
  where   rg.obj# = sov.obj_num
      and (SYS_CONTEXT('USERENV','CURRENT_USERID') = 0 OR
           EXISTS ( SELECT * FROM session_roles
                    WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_rls_group_view to PUBLIC
/

create or replace type ku$_rls_context_t as object
(
  vers_major    char(1),                              /* UDT major version # */
  vers_minor    char(1),                              /* UDT minor version # */
  base_obj      ku$_schemaobj_t,                       /* base schema object */
  obj_num       number,                              /* parent object number */
  name          varchar2(30),                                   /* namespace */
  attr          varchar2(30)                                    /* attribute */
)
/
grant execute on ku$_rls_context_t to public
/

create or replace force view ku$_rls_context_view of ku$_rls_context_t
  with object identifier (name,obj_num) as
  select '1','0',
          value(sov),
          rc.obj# ,rc.ns ,rc.attr 
  from    ku$_schemaobj_view sov, sys.rls_ctx$ rc
  where   rc.obj# = sov.obj_num
      and (SYS_CONTEXT('USERENV','CURRENT_USERID') = 0 OR
              EXISTS ( SELECT * FROM session_roles
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_rls_context_view to PUBLIC
/

-------------------------------------------------------------------------------
--                           Materialized View
-------------------------------------------------------------------------------
--
create or replace type ku$_m_view_srt_t as object
(
  tablenum          number,           /* order of Master table in snap query */
  snaptime          varchar2(21),          /* time of last refresh for table */
  masflag           number,                 /* additional master information */
  masobj_num        number,                    /* obj number of master table */
  loadertime        varchar2(21),      /* last refresh w.r.t. SQL*Loader log */
  refscn            number, /* scn of latest info used to refresh this table */
  lastsuccess       varchar2(21),   /* time of last known successful refresh */
  fcmaskvec         raw(255), 	               /* filter columns mask vector */
  ejmaskvec         raw(255),       	    /* equi-join columns mask vector */
  sub_handle        number,            /* subscription handle (if using CDC) */
  change_view       varchar2(30)          /* change view name (if using CDC) */
)
/
grant execute on ku$_m_view_srt_t to public
/

create or replace type ku$_m_view_srt_list_t
  as table of ku$_m_view_srt_t
/
grant execute on ku$_m_view_srt_list_t to public
/

create or replace type ku$_m_view_scm_t as object
(
  snacol            varchar2(30),                 /* name of snapshot column */
  mascol            varchar2(30),                      /* master column name */
  maspos            number,            /* position of master column (intcol) */
  colrole           number,                       /* how is this column used */
  snapos            integer             /* position of col in snapshot table */
)
/
grant execute on ku$_m_view_scm_t to public
/
create or replace type ku$_m_view_scm_list_t
  as table of ku$_m_view_scm_t
/
grant execute on ku$_m_view_scm_list_t to public
/
create or replace type ku$_m_view_t as object
(
  vers_major        char(1),                          /* UDT major version # */
  vers_minor        char(1),                          /* UDT minor version # */
  sowner            varchar2(30),                       /* Owner of snapshot */
  vname             varchar2(30),                        /* name of snapshot */
  tname             varchar2(30),                              /* Table name */
  mowner            varchar2(30),                         /* owner of master */
  master            varchar2(30),                          /* name of master */
  mlink             varchar2(128),           /* database link to master site */
  snapshot          varchar2(21), /* used by V7 masters to identify snapshot */
  snapid            integer,  /* used by V8 masters to identify the snapshot */
  auto_fast         varchar2(8),      /* date function for automatic refresh */
  auto_fun          varchar2(200),                  /* next time for refresh */
  auto_date         date,                          /* start time for refresh */
  uslog             varchar2(30),             /* log for updatable snapshots */
  status            integer,                           /* refresh operations */
  master_version    integer,            /* Oracle version of the master site */
  tables            integer,                 /* Number of tables in snapshot */
  flag              number,
  flag2             number,
  lobmaskvec        raw(255),	                  /* lob columns mask vector */
  mas_roll_seg      varchar2(30),            /* master-side rollback segment */
  rscn              number,                              /* last refresh scn */
  instsite          integer,                           /* instantiating site */
  flavor_id         number,                                     /* flavor id */
  objflag           number,                 /* object properties of snapshot */
  sna_type_owner    varchar2(30),                    /* object MV type owner */
  sna_type_name     varchar2(30),                     /* object MV type name */
  mas_type_owner    varchar2(30),          /* master object table type owner */
  mas_type_name     varchar2(30),           /* master object table type name */
  parent_sowner     varchar2(30),                   /* parent snapshot owner */
  parent_vname      varchar2(30),                    /* parent snapshot name */
  query_len         number,                                  /* query length */
  query_txt         clob,              /* query which this view instantiates */
  query_vcnt        ku$_vcnt,          /* store the query when length > 4000 */
  rel_query         clob,              /* relational transformation of query */
  loc_roll_seg      varchar2(30),             /* local side rollback segment */
  global_db_name    varchar2(4000),                  /* Global database Name */
  srt_list          ku$_m_view_srt_list_t,
  scm_count         number,
  scm_list          ku$_m_view_scm_list_t
)
/
grant execute on ku$_m_view_t to public
/

create or replace type ku$_m_view_h_t as object
(
  vers_major        char(1),                          /* UDT major version # */
  vers_minor        char(1),                          /* UDT minor version # */
  sowner            varchar2(30),                       /* Owner of snapshot */
  vname             varchar2(30),                        /* name of snapshot */
  mview             ku$_m_view_t,
  mview_tab         ku$_htable_t,
  mview_idx_list    ku$_index_list_t
)
/
grant execute on ku$_m_view_h_t to public
/

create or replace type ku$_m_view_ph_t as object
(
  vers_major        char(1),                          /* UDT major version # */
  vers_minor        char(1),                          /* UDT minor version # */
  sowner            varchar2(30),                       /* Owner of snapshot */
  vname             varchar2(30),                        /* name of snapshot */
  mview             ku$_m_view_t,
  mview_tab         ku$_phtable_t,
  mview_idx_list    ku$_index_list_t
)
/
grant execute on ku$_m_view_ph_t to public
/

create or replace type ku$_m_view_fh_t as object
(
  vers_major        char(1),                          /* UDT major version # */
  vers_minor        char(1),                          /* UDT minor version # */
  sowner            varchar2(30),                       /* Owner of snapshot */
  vname             varchar2(30),                        /* name of snapshot */
  mview             ku$_m_view_t,
  mview_tab         ku$_fhtable_t,
  mview_idx_list    ku$_index_list_t
)
/
grant execute on ku$_m_view_fh_t to public
/

create or replace type ku$_m_view_pfh_t as object
(
  vers_major        char(1),                          /* UDT major version # */
  vers_minor        char(1),                          /* UDT minor version # */
  sowner            varchar2(30),                       /* Owner of snapshot */
  vname             varchar2(30),                        /* name of snapshot */
  mview             ku$_m_view_t,
  mview_tab         ku$_pfhtable_t,
  mview_idx_list    ku$_index_list_t
)
/
grant execute on ku$_m_view_pfh_t to public
/

create or replace type ku$_m_view_iot_t as object
(
  vers_major        char(1),                          /* UDT major version # */
  vers_minor        char(1),                          /* UDT minor version # */
  sowner            varchar2(30),                       /* Owner of snapshot */
  vname             varchar2(30),                        /* name of snapshot */
  mview             ku$_m_view_t,
  mview_tab         ku$_iotable_t,
  mview_idx_list    ku$_index_list_t
)
/
grant execute on ku$_m_view_iot_t to public
/

create or replace type ku$_m_view_piot_t as object
(
  vers_major        char(1),                          /* UDT major version # */
  vers_minor        char(1),                          /* UDT minor version # */
  sowner            varchar2(30),                       /* Owner of snapshot */
  vname             varchar2(30),                        /* name of snapshot */
  mview             ku$_m_view_t,
  mview_tab         ku$_piotable_t,
  mview_idx_list    ku$_index_list_t
)
/
grant execute on ku$_m_view_piot_t to public
/
create or replace force view ku$_m_view_view of ku$_m_view_t
  with object identifier (sowner, vname) as
  select '1','0',
         s.sowner,
         s.vname,
         s.tname,
         s.mowner,
         s.master,
         s.mlink,
         TO_CHAR(s.snapshot,'YYYY-MM-DD:HH24:MI:SS'),
         s.snapid,
         DECODE(s.auto_fast, 'C', 'COMPLETE', 'F', 'FAST', '?', 'FORCE',
                NULL, 'FORCE', 'N', 'NEVER', 'ERROR'),
         s.auto_fun,
         s.auto_date,
         s.uslog,
         s.status,
         s.master_version,
         s.tables,
         s.flag,
         s.flag2,
         s.lobmaskvec,
         s.mas_roll_seg,
         s.rscn,
         s.instsite,
         NVL(s.flavor_id, 0),
         s.objflag,
         s.sna_type_owner,
         s.sna_type_name,
         s.mas_type_owner,
         s.mas_type_name,
         s.parent_sowner,
         s.parent_vname,
         s.query_len,
         sys.dbms_metadata_util.long2clob(s.query_len, 'sys.snap$',
                                            'query_txt', s.rowid),
         NULL,
         s.rel_query,
         (select rg.rollback_seg
          from   sys.rgroup$ rg
          where  rg.owner = s.sowner
             and rg.name = s.vname),
         p.value$,
         cast(multiset(select srt.tablenum,
                              TO_CHAR(srt.snaptime,'YYYY-MM-DD:HH24:MI:SS'),
                              srt.masflag,
                              srt.masobj#,
                              TO_CHAR(srt.loadertime,'YYYY-MM-DD:HH24:MI:SS'),
                              srt.refscn,
                              TO_CHAR(srt.lastsuccess,'YYYY-MM-DD:HH24:MI:SS'),
                              srt.fcmaskvec,
                              srt.ejmaskvec,
                              srt.sub_handle,
                              srt.change_view
                       from   sys.snap_reftime$ srt
                       where  s.vname    = srt.vname
                          and s.sowner   = srt.sowner
                          and s.instsite = srt.instsite)
                       as ku$_m_view_srt_list_t),
         (select count(*)
                       from   sys.snap_colmap$ scm
                       where  s.vname = scm.vname
                          and s.sowner = scm.sowner
                          and s.instsite = scm.instsite),
         cast(multiset(select scm.snacol,
                              scm.mascol,
                              scm.maspos,
                              scm.colrole,
                              scm.snapos
                       from   sys.snap_colmap$ scm
                       where  s.vname = scm.vname
                          and s.sowner = scm.sowner
                          and s.instsite = scm.instsite
                          order by scm.snacol)
                       as ku$_m_view_scm_list_t)
  from   snap$ s, sys.props$ p
  where  p.name  = 'GLOBAL_DB_NAME'
/
grant select on ku$_m_view_view to public
/

create or replace force view ku$_m_view_h_view of ku$_m_view_h_t
  with object identifier (sowner,vname) as
  select '1','0',
         mvv.sowner,
         mvv.vname,
         value(mvv),
         value(htv),
         cast(multiset(select value(iv) from sys.ku$_index_view iv, sys.ind$ i
                       where i.bo# = ot.obj# and
                             bitand(i.property,8192) = 8192 and
                             iv.obj_num = i.obj#) as ku$_index_list_t)
  from   sys.obj$ ot, sys.user$ u, sys.ku$_htable_view htv,
         sys.ku$_m_view_view mvv
  where  ot.name     = mvv.tname
     and ot.owner#   = u.user#
     and u.name      = mvv.sowner
     and ot.type#    = 2
     and htv.obj_num = ot.obj#
     and BITAND(mvv.flag,33554432) != 33554432
     and (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (ot.owner#, 0) OR 
          EXISTS ( SELECT * FROM session_roles 
                   WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_m_view_h_view to public
/
create or replace force view ku$_m_view_ph_view of ku$_m_view_ph_t
  with object identifier (sowner, vname) as
  select '1','0',
         mvv.sowner,
         mvv.vname,
         value(mvv),
         value(phtv),
         cast(multiset(select value(iv) from sys.ku$_index_view iv, sys.ind$ i
                       where i.bo# = ot.obj# and
                             bitand(i.property,8192) = 8192 and
                             iv.obj_num = i.obj#) as ku$_index_list_t)
  from   sys.obj$ ot, sys.user$ u, sys.ku$_phtable_view phtv,
         sys.ku$_m_view_view mvv
  where  ot.name     = mvv.tname
     and ot.owner#   = u.user#
     and u.name      = mvv.sowner
     and ot.type#    = 2
     and phtv.obj_num = ot.obj#
     and BITAND(mvv.flag,33554432) != 33554432
     and (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (ot.owner#, 0) OR 
          EXISTS ( SELECT * FROM session_roles 
                   WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_m_view_ph_view to public
/
create or replace force view ku$_m_view_fh_view of ku$_m_view_fh_t
  with object identifier (sowner, vname) as
  select '1','0',
         mvv.sowner,
         mvv.vname,
         value(mvv),
         value(fhtv),
         cast(multiset(select value(iv) from sys.ku$_index_view iv, sys.ind$ i
                       where i.bo# = ot.obj# and
                             bitand(i.property,8192) = 8192 and
                             iv.obj_num = i.obj#) as ku$_index_list_t)
  from   sys.obj$ ot, sys.user$ u, sys.ku$_fhtable_view fhtv,
         sys.ku$_m_view_view mvv
  where  ot.name     = mvv.tname
     and ot.owner#   = u.user#
     and u.name      = mvv.sowner
     and ot.type#    = 2
     and fhtv.obj_num = ot.obj#
     and BITAND(mvv.flag,33554432) != 33554432
     and (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (ot.owner#, 0) OR 
          EXISTS ( SELECT * FROM session_roles 
                   WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_m_view_fh_view to public
/
create or replace force view ku$_m_view_pfh_view of ku$_m_view_pfh_t
  with object identifier (sowner, vname) as
  select '1','0',
         mvv.sowner,
         mvv.vname,
         value(mvv),
         value(pfhtv),
         cast(multiset(select value(iv) from sys.ku$_index_view iv, sys.ind$ i
                       where i.bo# = ot.obj# and
                             bitand(i.property,8192) = 8192 and
                             iv.obj_num = i.obj#) as ku$_index_list_t)
  from   sys.obj$ ot, sys.user$ u, sys.ku$_pfhtable_view pfhtv,
         sys.ku$_m_view_view mvv
  where  ot.name     = mvv.tname
     and ot.owner#   = u.user#
     and u.name      = mvv.sowner
     and ot.type#    = 2
     and pfhtv.obj_num = ot.obj#
     and BITAND(mvv.flag,33554432) != 33554432
     and (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (ot.owner#, 0) OR 
          EXISTS ( SELECT * FROM session_roles 
                   WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_m_view_pfh_view to public
/
create or replace force view ku$_m_view_iot_view of ku$_m_view_iot_t
  with object identifier (sowner, vname) as
  select '1','0',
         mvv.sowner,
         mvv.vname,
         value(mvv),
         value(iotv),
         cast(multiset(select value(iv) from sys.ku$_index_view iv, sys.ind$ i
                       where i.bo# = ot.obj# and
                             bitand(i.property,8192) = 8192 and
                             iv.obj_num = i.obj#) as ku$_index_list_t)
  from   sys.obj$ ot, sys.user$ u, sys.ku$_iotable_view iotv,
         sys.ku$_m_view_view mvv
  where  ot.name     = mvv.tname
     and ot.owner#   = u.user#
     and u.name      = mvv.sowner
     and ot.type#    = 2
     and iotv.obj_num = ot.obj#
     and BITAND(mvv.flag,33554432) = 33554432
     and (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (ot.owner#, 0) OR 
          EXISTS ( SELECT * FROM session_roles 
                   WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_m_view_iot_view to public
/
create or replace force view ku$_m_view_piot_view of ku$_m_view_piot_t
  with object identifier (sowner, vname) as
  select '1','0',
         mvv.sowner,
         mvv.vname,
         value(mvv),
         value(piotv),
         cast(multiset(select value(iv) from sys.ku$_index_view iv, sys.ind$ i
                       where i.bo# = ot.obj# and
                             bitand(i.property,8192) = 8192 and
                             iv.obj_num = i.obj#) as ku$_index_list_t)
  from   sys.obj$ ot, sys.user$ u, sys.ku$_piotable_view piotv,
         sys.ku$_m_view_view mvv
  where  ot.name     = mvv.tname
     and ot.owner#   = u.user#
     and u.name      = mvv.sowner
     and ot.type#    = 2
     and piotv.obj_num = ot.obj#
     and BITAND(mvv.flag,33554432) = 33554432
     and (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (ot.owner#, 0) OR 
              EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_m_view_piot_view to public
/

-------------------------------------------------------------------------------
--                             Materialized View Log
-------------------------------------------------------------------------------
--
create or replace type ku$_refcol_t as object
(
  colname         varchar2(30),                  /* master table column name */
  oldest          varchar2(21),  /* maximum age of information in the column */
  flag            number                          /* column meta information */
)
/
grant execute on ku$_refcol_t to public
/
create or replace type ku$_refcol_list_t
  as table of ku$_refcol_t
/
grant execute on ku$_refcol_list_t to public
/

create or replace type ku$_slog_t as object
(
  snapid          integer,                        /* identifies V8 snapshots */
  snaptime        varchar2(21)                        /* when last refreshed */
)
/
grant execute on ku$_slog_t to public
/
create or replace type ku$_slog_list_t
  as table of ku$_slog_t
/
grant execute on ku$_slog_list_t to public
/

create or replace type ku$_m_view_log_t as object
(
  vers_major      char(1),                            /* UDT major version # */
  vers_minor      char(1),                            /* UDT minor version # */
  mowner          varchar2(30),                           /* owner of master */
  master          varchar2(30),                            /* name of master */
  oldest          date,       /* maximum age of rowid information in the log */
  oldest_pk       varchar2(21),  /* maximum age of PK information in the log */
  oscn            number,                                   /* scn of oldest */
  youngest        varchar2(21),             /* most recent snaptime assigned */
  yscn            number,                                      /* set-up scn */
  log             varchar2(30),                               /* name of log */
  trig            varchar2(30),                 /* trigger on master for log */
  flag            number,    /* 0x01, rowid         0x02, primary key values */
                             /* 0x04, column values 0x08, log is imported    */
                             /* 0x10, log is created with temp table         */
  mtime           varchar2(21),                     /* DDL modification time */
  temp_log        varchar2(30),      /* temp table as updatable snapshot log */
  oldest_oid      varchar2(21), /* maximum age of OID information in the log */
  oldest_new      varchar2(21),      /* maximum age of new values in the log */
  global_db_name  varchar2(4000),                    /* Global database Name */
  fc_count        number,                        /* number of filter columns */
  fc_list         ku$_refcol_list_t,                   /* filter column list */
  lm_count        number,                         /* number of local masters */
  lm_list         ku$_slog_list_t                       /* local master list */
)
/
grant execute on ku$_m_view_log_t to public
/

create or replace type ku$_m_view_log_h_t as object
(
  vers_major        char(1),                          /* UDT major version # */
  vers_minor        char(1),                          /* UDT minor version # */
  tabobj_num        number,                       /* log table object number */
  mviewlog          ku$_m_view_log_t,
  mviewlog_tab      ku$_htable_t
)
/
grant execute on ku$_m_view_log_h_t to public
/

create or replace type ku$_m_view_log_ph_t as object
(
  vers_major        char(1),                          /* UDT major version # */
  vers_minor        char(1),                          /* UDT minor version # */
  tabobj_num        number,                       /* log table object number */
  mviewlog          ku$_m_view_log_t,
  mviewlog_tab      ku$_phtable_t
)
/
grant execute on ku$_m_view_log_ph_t to public
/

create or replace type ku$_m_view_log_fh_t as object
(
  vers_major        char(1),                          /* UDT major version # */
  vers_minor        char(1),                          /* UDT minor version # */
  tabobj_num        number,                       /* log table object number */
  mviewlog          ku$_m_view_log_t,
  mviewlog_tab      ku$_fhtable_t
)
/
grant execute on ku$_m_view_log_fh_t to public
/

create or replace type ku$_m_view_log_pfh_t as object
(
  vers_major        char(1),                          /* UDT major version # */
  vers_minor        char(1),                          /* UDT minor version # */
  tabobj_num        number,                       /* log table object number */
  mviewlog          ku$_m_view_log_t,
  mviewlog_tab      ku$_pfhtable_t
)
/
grant execute on ku$_m_view_log_pfh_t to public
/

create or replace force view ku$_m_view_log_view of ku$_m_view_log_t
  with object identifier (mowner,master) as
  select '1','0',
         m.mowner, m.master, m.oldest,
         TO_CHAR(m.oldest_pk, 'YYYY-MM-DD:HH24:MI:SS'),
         m.oscn,
         TO_CHAR(m.youngest, 'YYYY-MM-DD:HH24:MI:SS'),
         m.yscn, m.log, m.trig, m.flag,
         TO_CHAR(m.mtime, 'YYYY-MM-DD:HH24:MI:SS'),
         m.temp_log,
         TO_CHAR(m.oldest_oid, 'YYYY-MM-DD:HH24:MI:SS'),
         TO_CHAR(m.oldest_new, 'YYYY-MM-DD:HH24:MI:SS'),
         p.value$,
         (select count(*)
                       from   sys.mlog_refcol$ r
                       where  m.mowner = r.mowner
                          and m.master = r.master),
         cast(multiset(select r.colname,
                              to_char(r.oldest, 'YYYY-MM-DD:HH24:MI:SS'),
                              r.flag
                       from   sys.mlog_refcol$ r
                       where  m.mowner = r.mowner
                          and m.master = r.master)
                       as ku$_refcol_list_t),
         (select count(*)
                       from   sys.slog$ s
                       where  m.mowner = s.mowner
                          and m.master = s.master),
         cast(multiset(select s.snapid,
                              to_char(s.snaptime, 'YYYY-MM-DD:HH24:MI:SS')
                       from   sys.slog$ s
                       where  m.mowner = s.mowner
                          and m.master = s.master)
                       as ku$_slog_list_t)
  from   sys.mlog$ m, sys.props$ p
  where  p.name  = 'GLOBAL_DB_NAME'
/
grant select on ku$_m_view_log_view to select_catalog_role;
/

create or replace force view ku$_m_view_log_h_view of ku$_m_view_log_h_t
  with object identifier (tabobj_num) as
  select '1','0',
         htv.obj_num,
         value(mvlv),
         value(htv)
  from   obj$ o, user$ u, sys.ku$_htable_view htv, ku$_m_view_log_view mvlv
  where  mvlv.mowner = u.name
     and mvlv.log    = o.name
     and o.owner#    = u.user#
     and o.type#     = 2
     and o.obj#      = htv.schema_obj.obj_num
     and (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (u.user#, 0) OR
          EXISTS ( SELECT * FROM session_roles
                   WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_m_view_log_h_view to public
/

create or replace force view ku$_m_view_log_ph_view of ku$_m_view_log_ph_t
  with object identifier (tabobj_num) as
  select '1','0',
         phtv.obj_num,
         value(mvlv),
         value(phtv)
  from   obj$ o, user$ u, sys.ku$_phtable_view phtv, ku$_m_view_log_view mvlv
  where  mvlv.mowner = u.name
     and mvlv.log    = o.name
     and o.owner#    = u.user#
     and o.type#     = 2
     and o.obj#      = phtv.schema_obj.obj_num
     and (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (u.user#, 0) OR 
          EXISTS ( SELECT * FROM session_roles 
                   WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_m_view_log_ph_view to public
/

create or replace force view ku$_m_view_log_fh_view of ku$_m_view_log_fh_t
  with object identifier (tabobj_num) as
  select '1','0',
         fhtv.obj_num,
         value(mvlv),
         value(fhtv)
  from   obj$ o, user$ u, sys.ku$_fhtable_view fhtv, ku$_m_view_log_view mvlv
  where  mvlv.mowner = u.name
     and mvlv.log    = o.name
     and o.owner#    = u.user#
     and o.type#     = 2
     and o.obj#      = fhtv.schema_obj.obj_num
     and (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (u.user#, 0) OR 
          EXISTS ( SELECT * FROM session_roles 
                   WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_m_view_log_fh_view to public
/

create or replace force view ku$_m_view_log_pfh_view of ku$_m_view_log_pfh_t
  with object identifier (tabobj_num) as
  select '1','0',
         pfhtv.obj_num,
         value(mvlv),
         value(pfhtv)
  from   obj$ o, user$ u, sys.ku$_pfhtable_view pfhtv, ku$_m_view_log_view mvlv
  where  mvlv.mowner = u.name
     and mvlv.log    = o.name
     and o.owner#    = u.user#
     and o.type#     = 2
     and o.obj#      = pfhtv.schema_obj.obj_num
     and (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (u.user#, 0) OR
          EXISTS ( SELECT * FROM session_roles 
                   WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_m_view_log_pfh_view to public
/
-------------------------------------------------------------------------------
--				LIBRARY
-------------------------------------------------------------------------------
-- 
create or replace type ku$_library_t as object
(
  vers_major    char(1),			      /* UDT major version # */
  vers_minor    char(1),			      /* UDT minor version # */
  obj_num	number,				    /* library object number */
  schema_obj	ku$_schemaobj_t,		    /* library schema object */
  filespec	varchar2(2000),  				 /* filename */
  lib_audit	varchar2(2000),
  property      number
)
/
grant execute on ku$_library_t to public
/
create or replace force view ku$_library_view of ku$_library_t
  with object identifier(obj_num) as
  select '1','0',
	 lb.obj#, value(o),
	 lb.filespec, 
	 lb.audit$,
         lb.property
  from sys.ku$_schemaobj_view o, sys.library$ lb
  where	o.type_num=22 AND
	lb.obj# = o.obj_num AND
         (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num, 0) OR 
		EXISTS ( SELECT * FROM session_roles 
                	WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_library_view to public
/
-------------------------------------------------------------------------------
--				USER
-------------------------------------------------------------------------------
-- 
create or replace type ku$_user_t as object
(
  vers_major    char(1),			     /* UDT major version # */
  vers_minor    char(1),			     /* UDT minor version # */
  user_id 	number,				                /* user id  */
  name          varchar2(30),                               /* name of user */
  type_num      number ,                              /* 0 = role, 1 = user */
  password      varchar2(30),                         /* encrypted password */
  datats        varchar2(30), 		         /* user default tablespace  */
  tempts        varchar2(30),  	  /* default tablespace for temporary tables */
  ctime         date,                          /* user account creation time */
  ptime         date,                                /* password change time */
  exptime       date,                     /* actual password expiration time */
  ltime         date,                         /* time when account is locked */
  profnum       number,                                 /* resource profile# */
  profname         varchar2(30),                            /* profile name  */
  user_audit        varchar2(38),                     /* user audit options */
  defrole       number,                           /* default role indicator: */
  defgrp_num       number,                             /* default undo group */
  defgrp_seq_num   number,             /* global sequence number for the grp */
  astatus       number,                             /* status of the account */
  lcount        number,                    /* count of failed login attempts */
  defschclass   varchar2(30),                      /* initial consumer group */
  ext_username  varchar2(400),                          /* external username */
  spare1        number,
  spare2        number,
  spare3        number,
  spare4        varchar2(1000),
  spare5        varchar2(1000),
  spare6        date
)
/
grant execute on ku$_user_t to public
/
create or replace force view ku$_user_view of ku$_user_t
  with object identifier(user_id) as
  select '1','0',
	  u.user#,
	  u.name,
	  u.type#,
	  u.password,
	  ts1.name,
	  ts2.name,
	  u.ctime,
	  u.ptime,
	  u.exptime,
	  u.ltime,
	  u.resource$,
	  p.name,
	  u.audit$,
	  u.defrole,
	  u.defgrp#,
	  u.defgrp_seq#,
	  u.astatus,
	  u.lcount,
	  u.defschclass,
	  u.ext_username,
	  u.spare1,
	  u.spare2,
	  u.spare3,
	  u.spare4,
	  u.spare5,
	  u.spare6
  from sys.user$ u,        
       sys.ts$ ts1, sys.ts$ ts2, sys.profname$ p
  where	  u.datats# = ts1.ts# AND
          u.tempts# = ts2.ts# AND
          u.type# = 1 AND
          u.resource$ = p.profile# 
	  AND (SYS_CONTEXT('USERENV','CURRENT_USERID') = 0
		OR EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_user_view to public
/
-------------------------------------------------------------------------------
--				ROLE
-------------------------------------------------------------------------------
-- 
create or replace type ku$_role_t as object
(
  vers_major    char(1),			      /* UDT major version # */
  vers_minor    char(1),			      /* UDT minor version # */
  user_id 	number,				                 /* role id  */
  name          varchar2(30),                                /* name of role */
  password      varchar2(30),                                   /*  password */
  schema        varchar2(30),                                      /* schema */
  package       varchar2(30)                                      /* package */
)
/
grant execute on ku$_role_t to public
/
create or replace force view ku$_role_view of ku$_role_t
  with object identifier(user_id) as
  select '1','0',
  	u.user#,
  	u.name,
  	u.password,
  	(select r.schema from sys.approle$ r  where r.role#=u.user#), 
  	(select r.package from sys.approle$ r  where r.role#=u.user#)
  from sys.user$ u 
  where	  u.type# = 0 
  AND (SYS_CONTEXT('USERENV','CURRENT_USERID') = 0
	OR EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_role_view to public
/
-------------------------------------------------------------------------------
--				PROFILE
-------------------------------------------------------------------------------
-- 
create or replace type ku$_profile_attr_t as object
(
  profile_id 	number,				              /* profile id  */
  resource_num  number, 				       /* resource id*/
  resname       varchar2(30), 				    /* resource name */
  type_num		number, 				     /* type */
  limit_num		number					    /* limit */
)
/
grant execute on ku$_profile_attr_t to public
/ 
create or replace type ku$_profile_list_t as TABLE of ku$_profile_attr_t
/
create or replace force view ku$_profile_attr_view of ku$_profile_attr_t
  with object identifier(profile_id) as
  select  p.profile#,
	  p.resource#, 
	  r.name, 
	  p.type#, 
	  p.limit# 
  from sys.resource_map r, sys.profile$ p 
  where p.resource# = r.resource# and p.type# = r.type#  and 
        p.profile# !=0  
/
grant select on ku$_profile_attr_view to select_catalog_role
/ 
grant execute on ku$_profile_list_t to public
/ 
create or replace type ku$_profile_t as object
(
  vers_major    char(1),			      /* UDT major version # */
  vers_minor    char(1),			      /* UDT minor version # */
  profile_id 	number,				              /* profile id  */
  profile_name	varchar2(30),				     /* profile name */
  pass_func_name varchar2(30),  	         /* password verify function */
  profile_list  ku$_profile_list_t		       /* profile attributes */
)
/
grant execute on ku$_profile_t to public
/
create or replace force view ku$_profile_view of ku$_profile_t
  with object identifier(profile_id) as
  select '1','0',
  	n.profile#,
  	n.name, 
  	(select distinct o.name
            from    sys.obj$ o, sys.ku$_profile_attr_view p
            where   o.type# = 8 AND
                    o.owner# = 0 AND
		    o.obj# = p.limit_num and
		    p.profile_id = n.profile# and
		    p.resource_num =4 and    -- res# 4, type# 1 =
		    p.type_num =1 ),         -- PASSWORD_VERIFY_FUNCTION
  	cast(multiset (select * from ku$_profile_attr_view pl
	    where pl.profile_id = n.profile# ) as ku$_profile_list_t 
	)
  from sys.profname$ n 
  where n.profile# !=0  
	AND (SYS_CONTEXT('USERENV','CURRENT_USERID') = 0
	OR EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_profile_view to public
/
-------------------------------------------------------------------------------
--				DEFAULT_ROLE
-------------------------------------------------------------------------------
--
create or replace type ku$_defrole_item_t as object
(
  user_id 	number,				                 /* user id  */
  user_name	varchar2(30), 				        /* user name */
  role	        varchar2(30),  				 /* role source name */
  role_id 	number				                  /* role id */
)
/
grant execute on ku$_defrole_item_t to public
/
create or replace type ku$_defrole_list_t as TABLE of ku$_defrole_item_t
/
create or replace force view ku$_defrole_list_view of ku$_defrole_item_t
  with object identifier(user_id) as
  select  d$.user#, u$.name, u1$.name, d$.role#
  from    sys.user$ u$, sys.user$ u1$, sys.defrole$ d$
  where   u$.user# = d$.user# AND
          u1$.user# = d$.role#
/
grant select on ku$_defrole_list_view to select_catalog_role
/ 
grant execute on ku$_defrole_list_t to public
/ 
create or replace type ku$_defrole_t as object
(
  vers_major    char(1),			      /* UDT major version # */
  vers_minor    char(1),			      /* UDT minor version # */
  user_id 	number,				              /* profile id  */
  user_name     varchar2(30),				        /* user name */
  defrole	number,					/* default role type */
  role_list     ku$_defrole_list_t			        /* role list */
)
/
grant execute on ku$_defrole_t to public
/
create or replace force view ku$_defrole_view of ku$_defrole_t
  with object identifier(user_id) as
  select '1','0',
	  u.user#,
	  u.name, 
	  u.defrole,
	  cast(multiset (select * from ku$_defrole_list_view df
		where df.user_id = u.user#) as ku$_defrole_list_t 
		)
  from sys.user$ u 
  where u.user# !=0 and
	(SYS_CONTEXT('USERENV','CURRENT_USERID') = 0 
		OR EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_defrole_view to public
/
-------------------------------------------------------------------------------
--				PROXY
-------------------------------------------------------------------------------
--
create or replace type ku$_proxy_role_item_t as object
(
  role_id	number,						  /* role id */
  client 	varchar2(30),				      /* client name */
  proxy		varchar2(30), 				       /* proxy name */
  role		varchar2(20)				     /* role context */
)
/
grant execute on ku$_proxy_role_item_t to public
/
create or replace type ku$_proxy_role_list_t as TABLE of ku$_proxy_role_item_t
/
create or replace force view ku$_proxy_role_list_view of ku$_proxy_role_item_t
  with object identifier(client, proxy, role_id) as
  select  prd.role#, uc.name, up.name, ur.name
  from    sys.user$ ur, sys.user$ uc, 
          sys.user$ up ,sys.proxy_role_data$ prd
  where   prd.role#   = ur.user# AND
                prd.client# = uc.user# AND
                prd.proxy#  = up.user#
/
grant select on ku$_proxy_role_list_view to select_catalog_role
/ 
grant execute on ku$_proxy_role_list_t to public
/ 
create or replace type ku$_proxy_t as object
(
  user_id	number,					          /* role id */
  client_name 	varchar2(30),				      /* client name */
  proxy_name 	varchar2(30), 				       /* proxy name */
  flags 	number,	              /* Mask flags of associated with entry */
  cred_type     number,	               /* Type of credential passed by proxy */
  cred_ver	number,                  /* Version number of the credential */
  cred_minor	number,                   /* Minor credential version number */
  proxy_role_list  ku$_proxy_role_list_t                  /* proxy role list */
)
/
grant execute on ku$_proxy_t to public
/
create or replace force view ku$_proxy_view of ku$_proxy_t
  with object identifier(user_id) as
  select  u.user#, u.name, up.name, pd.flags,
          pd.credential_type#,
          pd.credential_version#,
          pd.credential_minor#,
          cast(multiset (select * from ku$_proxy_role_list_view pr
		where pr.client= u.name AND pr.proxy=up.name) 
		as ku$_proxy_role_list_t) 
  from   sys.user$ u, sys.user$ up, sys.proxy_data$ pd
  where  pd.client# = u.user# AND
         pd.proxy# = up.user#
	 AND (SYS_CONTEXT('USERENV','CURRENT_USERID') = 0
		OR EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_proxy_view to public
/ 
-- 
-------------------------------------------------------------------------------
--				ROLE_GRANT
-------------------------------------------------------------------------------
--
create or replace type ku$_rogrant_t as object
(
  vers_major    char(1),			      /* UDT major version # */
  vers_minor    char(1),			      /* UDT minor version # */
  grantee_id 	number,				                /* user id  */
  grantee       varchar2(30),                                    /* grantee */
  role          varchar2(30),                                       /* role */
  role_id       number,						 /* role id */
  admin         number,					   /*  admin option */
  sequence	number   		           /* unique grant sequence */
)
/
grant execute on ku$_rogrant_t to public
/
create or replace force view ku$_rogrant_view of ku$_rogrant_t
  with object identifier(grantee_id, role_id) as
  select '1','0',
          u1.user#, u1.name, u2.name, u2.user#, NVL(g.option$, 0), g.sequence#
  from    sys.user$ u1, sys.user$ u2, sys.sysauth$ g
  where   u1.user# = g.grantee# AND
          u2.user# = g.privilege# AND
          g.privilege# > 0 
	  AND (SYS_CONTEXT('USERENV','CURRENT_USERID') =0 
		OR EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_rogrant_view to public
/
-------------------------------------------------------------------------------
--				TABLESPACE_QUOTA
-------------------------------------------------------------------------------
-- 
create or replace type ku$_tsquota_t as object
(
  vers_major    char(1),			      /* UDT major version # */
  vers_minor    char(1),			      /* UDT minor version # */
  user_id 	number,				                 /* user id  */
  user_name     varchar2(30),                                /* name of user */
  ts_name       varchar2(30),                             /* tablespace name */
  ts_id	        number,					    /* tablespace id */
  maxblocks     number,				 	       /* max blocks */
  blocksize	number,					      /* blocks size */
  grantor_num   number, 	                               /* grantor id */
  grantor       varchar2(30), 	                             /* grantor name */
  blocks        number,                  /* number of blocks charged to user */
  priv1         number,                     /* reserved for future privilege */
  priv2         number,                     /* reserved for future privilege */
  priv3         number                      /* reserved for future privilege */
)
/
grant execute on ku$_tsquota_t to public
/
create or replace force view ku$_tsquota_view of ku$_tsquota_t
  with object identifier(user_id, ts_id) as
  select  '1','0',
	  u.user#, u.name, t.name, q.ts#, q.maxblocks, t.blocksize, q.grantor#,
          (select us.name from sys.user$ us  where us.user# =q.grantor#),
	  q.blocks, q.priv1, q.priv2, q.priv3
  from    sys.user$ u, sys.tsq$ q, sys.ts$ t
  where   q.user# = u.user# AND 
          q.ts# = t.ts# AND
          q.maxblocks != 0 AND
          t.online$ IN (1, 2, 4) AND 
	  (SYS_CONTEXT('USERENV','CURRENT_USERID') = 0 
		OR EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_tsquota_view to public
/
-------------------------------------------------------------------------------
--				RESOURCE_COST 
-------------------------------------------------------------------------------
-- 
create or replace type ku$_resocost_item_t as object
(
  resource_id       number,				      /* resource id */
  resource_name     varchar2(30),                                /* resource */
  resource_type     number,					     /* type */
  unit_cost         number					/* unit cost */
)
/
grant execute on ku$_resocost_item_t to public
/
create or replace type ku$_resocost_list_t as TABLE of ku$_resocost_item_t
/
create or replace force view ku$_resocost_list_view of ku$_resocost_item_t
  with object identifier(resource_id) as
  select  m.resource#, m.name,m.type#, c.cost
  from    sys.resource_cost$ c, sys.resource_map m
  where   c.resource# = m.resource#
/
grant select on ku$_resocost_list_view to select_catalog_role
/
grant execute on ku$_resocost_list_t to public
/
create or replace type ku$_resocost_t as object
(
  vers_major    char(1),			     /* UDT major version # */
  vers_minor    char(1),			     /* UDT minor version # */
  cost_list     ku$_resocost_list_t	              /* resource cost info */
)
/ 
grant execute on ku$_resocost_t to public
/
create or replace force view ku$_resocost_view of ku$_resocost_t
  with object identifier (vers_major) as 
  select '1','0',
         cast(multiset (select * from  ku$_resocost_list_view
	     ) as  ku$_resocost_list_t
	 )
  from dual
/
grant select on ku$_resocost_view to  select_catalog_role
/
-------------------------------------------------------------------------------
--				SEQUENCE
-------------------------------------------------------------------------------
-- 
create or replace type ku$_sequence_t as object
(
  vers_major    char(1),			      /* UDT major version # */
  vers_minor    char(1),			      /* UDT minor version # */
  obj_num	number,				   /* sequence object number */
  schema_obj	ku$_schemaobj_t,		   /* sequence schema object */
  incre		number,            	    /* the sequence number increment */
  minvalue      number,                         /* minimum value of sequence */
  maxvalue      number,                         /* maximum value of sequence */
  cycle         number,    	                      /* 0 = FALSE, 1 = TRUE */
  seq_order     number,                               /* 0 = FALSE, 1 = TRUE */
  cache         number,                          /* how many to cache in sga */
  highwater     number,                              /* disk high water mark */
  seq_audit     varchar2(38),                            /* auditing options */
  flags         number                               /* 0x08 LOGICAL STANDBY */
)
/
grant execute on ku$_sequence_t to public
/
create or replace force view ku$_sequence_view of ku$_sequence_t
  with object identifier(obj_num) as
  select '1','0',
         s.obj#, value(o), 
         s.increment$, s.minvalue, s.maxvalue, s.cycle#, s.order$, s.cache, 
         s.highwater,  s.audit$, s.flags
  from  sys.ku$_schemaobj_view o, sys.seq$ s
  where s.obj# = o.obj_num AND
        (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num, 0) OR 
	 EXISTS ( SELECT * FROM session_roles 
	 WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_sequence_view to public
/
-------------------------------------------------------------------------------
--				CONTEXT
-------------------------------------------------------------------------------
-- 
create or replace type ku$_context_t as object
(
  vers_major    char(1),			      /* UDT major version # */
  vers_minor    char(1),			      /* UDT minor version # */
  obj_num	number,				   /* context object number */
  schema_obj	ku$_schemaobj_t,		   /* context schema object */
  schema_name	varchar2(30),				      /* schema name */
  package_name	varchar2(30),				     /* package name */
  flags		number
)
/
grant execute on ku$_context_t to public
/
create or replace force view ku$_context_view of ku$_context_t
  with object identifier(obj_num) as
  select '1','0',
	 c.obj#, value(o),
         c.schema, c.package, c.flags
  from   sys.ku$_schemaobj_view o, sys.context$ c
  where  o.obj_num = c.obj# AND
         (SYS_CONTEXT('USERENV','CURRENT_USERID') = 0 OR 
		EXISTS ( SELECT * FROM session_roles 
                	WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_context_view to public
/
-------------------------------------------------------------------------------
--				DIMENSION
-------------------------------------------------------------------------------
-- 
create or replace type ku$_dimension_t as object
(
  vers_major    char(1),			      /* UDT major version # */
  vers_minor    char(1),			      /* UDT minor version # */
  obj_num	number,				  /* dimension object number */
  schema_obj	ku$_schemaobj_t,		  /* dimension schema object */
  dimtextlen	number,				        /* length of dimtext */
  dimtext	varchar2(4000)				/* text of dimension */
)
/
grant execute on ku$_dimension_t to public
/
create or replace force view ku$_dimension_view of ku$_dimension_t
  with object identifier(obj_num) as
  select '1','0',
	 d.obj#, value(o),
 	 d.dimtextlen,
	 sys.dbms_metadata_util.long2varchar(d.dimtextlen,
					'SYS.DIM$',
					'DIMTEXT',
					d.rowid)
  from sys.ku$_schemaobj_view o, sys.dim$ d
  where d.obj# = o.obj_num  AND
         (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num, 0) OR 
		EXISTS ( SELECT * FROM session_roles 
                	WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_dimension_view to public
/
-------------------------------------------------------------------------------
--				ASSOCIATION
-------------------------------------------------------------------------------
create or replace type ku$_assoc_t as object
(
  vers_major    char(1),			      /* UDT major version # */
  vers_minor    char(1),			      /* UDT minor version # */
  obj_num	number,				     /* association object # */
  base_obj	ku$_schemaobj_t,			      /* base object */
  obj_type	number, 				 /* association type */
  objcol	varchar2(30), 			       /* association column */
  stats_obj     ku$_schemaobj_t,			    /* statistic obj */
  selectivity	number, 				      /* selectivity */
  cpu_cost	number, 					  /* cpu cost*/
  io_cost	number, 					  /* io cost */
  net_cost	number,						 /* net cost */
  interface_version number,
  spare2	number
) 
/
grant execute on ku$_assoc_t to public
/
create or replace force view ku$_assoc_view of ku$_assoc_t
  with object identifier(obj_num) as
  select '1','0',
	a.obj#,
	value(so),
  	a.property,
        (select c.name from  sys.col$ c where c.obj# = a.obj# and
	   c.intcol# = a.intcol#),
	(select value(ss) from ku$_schemaobj_view ss where
		   ss.obj_num = a.statstype#), 
        NVL(a.default_selectivity, 0), NVL(a.default_cpu_cost, 0),
        NVL(a.default_io_cost, 0), NVL(a.default_net_cost, 0),
	a.interface_version#, a.spare2
   from  sys.ku$_schemaobj_view so,
	 sys.association$ a
   where   a.obj# =so.obj_num
	 AND (SYS_CONTEXT('USERENV','CURRENT_USERID')=0  or
	      EXISTS ( SELECT * FROM session_roles 
                      WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_assoc_view to public
/
-------------------------------------------------------------------------------
--				PASSWORD_VERIFY_FUNCTION 
-------------------------------------------------------------------------------
--
create or replace force view ku$_pwdvfc_view of ku$_proc_t
  with object identifier (obj_num) as
  select t.vers_major, t.vers_minor, t.obj_num, t.type_num,
         t.schema_obj, t.source_lines
  from ku$_base_proc_view t, profile$ p
  where t.type_num = 8 and t.obj_num = p.limit#
		 	and p.resource# =4 and p.type# =1 
/
grant select on ku$_pwdvfc_view to select_catalog_role 
/
-------------------------------------------------------------------------------
--				COMMENT
-------------------------------------------------------------------------------
-- 
create or replace type ku$_comment_t as object
(
  vers_major    char(1),			      /* UDT major version # */
  vers_minor    char(1),			      /* UDT minor version # */
  obj_num	number,				            /* object number */
  base_obj	ku$_schemaobj_t,		       /* base schema object */
  colno         number, 				        /* column id */
  colname       varchar2(30),   			      /* column name */
  cmnt		varchar2(4000)			             /* comment text */
)
/
grant execute on ku$_comment_t to public
/
create or replace force view ku$_comment_view of ku$_comment_t
  with object identifier(obj_num) as
  select '1','0',
	 cm.obj#, value(o),
         cm.col#,
	 (select c.name 
  		 from  sys.col$ c 
		 where  c.obj#=cm.obj# and c.intcol# = cm.col# ),
         replace(cm.comment$,'''','''''')
  from   sys.ku$_schemaobj_view o, 
		sys.com$ cm 
  where  o.obj_num = cm.obj# AND
         (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num, 0) OR 
	     EXISTS ( SELECT * FROM session_roles 
                WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_comment_view to public;
/
-------------------------------------------------------------------------------
--				CLUSTER
-------------------------------------------------------------------------------
--
create or replace type ku$_cluster_t as object
(
  vers_major    char(1),			      /* UDT major version # */
  vers_minor    char(1),			      /* UDT minor version # */
  obj_num	number,				         /* cluster object # */
  schema_obj	ku$_schemaobj_t,		       /* cluster schema obj */
  col_list      ku$_column_list_t,
  ts_name	varchar2(30),		 		       /* tablespace */
  tsno		number,				        /* tablespace number */
  fileno	number, 	               /* segment header file number */
  blockno	number,                       /* segment header block number */
  pct_free	number,          /* minimum free space percentage in a block */
  pct_used	number,          /* minimum used space percentage in a block */
  initrans	number, 	            /* initial number of transaction */
  maxtrans	number,                     /* maximum number of transaction */
  size_t	number,
  hashfunc      varchar2(30),              /* if hashed, function identifier */
  hashkeys	number,                                    /* hash key count */
  function	number, /* function: 0 (key is function), 1 (system default) */
  extind        number,             /* extent index value of fixed hash area */
  flags	        number,                                      /* 0x08 = CACHE */
                                          /* 0x010000 = Single Table Cluster */
                                                /* 0x00800000 = DEPENDENCIES */
  degree        number,      /* number of parallel query slaves per instance */
  instances     number,       /*  number of OPS instances for parallel query */
  avgchn        number,          /* average chain length - previously spare4 */
  funclen	number, 
  functxt	varchar2(4000),
  storage       ku$_storage_t,
  spare1        number,
  spare2        number,
  spare3        number,
  spare4        number,
  spare5        varchar2(1000),
  spare6        varchar2(1000),
  spare7        date
)
/
grant execute on ku$_cluster_t to public
/
create or replace force view ku$_cluster_view of ku$_cluster_t
  with object identifier(obj_num) as
  select '1','0',
	o.obj_num,
        value(o),
        cast(multiset(select * from ku$_column_view col
		       where col.obj_num = c.obj#
                      ) as ku$_column_list_t
	),
 	ts.name,
        ts.ts#, c.file#, c.block#,
        c.pctfree$, c.pctused$, c.initrans, c.maxtrans,NVL(c.size$, -1), 
        c.hashfunc, NVL(c.hashkeys, 0), NVL(c.func, 1), c.extind, 
        c.flags,
        NVL(c.degree, 1), NVL(c.instances, 1),
        NVL(c.avgchn, -1),
        (select condlength from sys.cdef$ co  where co.obj# = c.obj#),
	(select sys.dbms_metadata_util.long2varchar(cd.condlength,
					'SYS.CDEF$',
					'CONDITION',
					cd.rowid)
	      from sys.cdef$ cd  where cd.obj# = c.obj#),
        (select value(s) from  ku$_storage_view s 
	     where s.file_num = c.file#  and  s.block_num= c.block#
		 and s.ts_num = c.ts#),
	c.spare1, c.spare2, c.spare3, c.spare4, c.spare5, c.spare6, c.spare7
   from     sys.ku$_schemaobj_view o, sys.ts$ ts, sys.clu$ c
   where    o.obj_num = c.obj# AND
                c.ts# = ts.ts# AND
            (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num, 0) OR 
	     EXISTS ( SELECT * FROM session_roles 
                WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_cluster_view to public;
/
-------------------------------------------------------------------------------
--				AUDIT
-------------------------------------------------------------------------------
create or replace type ku$_audit_t as object
(
  vers_major    char(1),
  vers_minor    char(1),	
  user_num	number,		  		   /* user identifier number */
  user_name 	varchar2(30),			                /* user name */
  proxy_num     number, 			    /* UID of the proxy user */
  audit_option  varchar2(40), 			          /* auditing option */
  property      number, 		  /* 0x01 = do not export this audit */
  success	number, 				/* audit on success? */
  failure       number 					/* audit on failure? */
)
/
grant execute on ku$_audit_t to public
/
create or replace force view ku$_audit_view of ku$_audit_t
  with object identifier(user_num) as
  select '1','0',
	a.user#,
        u.name,
	a.proxy#,
        m.name, m.property, 
	NVL(a.success, 0), 
	NVL(a.failure, 0)
  from     sys.audit$ a, sys.stmt_audit_option_map m,
	   sys.user$ u 
  where    a.user# = u.user# and
           a.option# = m.option# 
	 AND (SYS_CONTEXT('USERENV','CURRENT_USERID') = 0 
		OR EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on sys.ku$_audit_view to public;
/
-------------------------------------------------------------------------------
--				AUDIT_OBJ
-------------------------------------------------------------------------------
create or replace type ku$_audit_obj_t as object
(
  vers_major	char(1),
  vers_minor	char(1),	
  obj_num	number,					    /* object number */
  base_obj      ku$_schemaobj_t,			  /* base obj schema */
  audit_val     varchar2(38),                            /* auditing options */
  audit_list	sys.ku$_audit_list_t		               /* audit list */
)
/
grant execute on ku$_audit_obj_t to public
/
create or replace force view ku$_audit_obj_base_view of ku$_audit_obj_t
  with object identifier(obj_num) as
  select '1','0',
         o.obj_num,
         value(o), 
	 case when o.type_num = 2 
	then (SELECT t.audit$ from sys.tab$ t where t.obj# = o.obj_num)
	when o.type_num = 4
	then (SELECT v.audit$ from sys.view$ v where v.obj# = o.obj_num)
	when o.type_num = 6
	then (SELECT s.audit$ from sys.seq$ s where s.obj# = o.obj_num)
	when o.type_num = 7
	then (SELECT p.audit$ from sys.procedure$ p where p.obj# = o.obj_num)
	when o.type_num = 13
	then (SELECT ty.audit$ from sys.type_misc$ ty where ty.obj# = o.obj_num)
	when o.type_num = 22
	then (SELECT l.audit$ from sys.library$ l where l.obj# = o.obj_num)
	when o.type_num = 23
	then (SELECT d.audit$ from sys.dir$ d where d.obj# = o.obj_num)
	else null end,
         sys.dbms_metadata_util.get_audit(o.obj_num,o.type_num)
  from   ku$_schemaobj_view o 
/
grant select on sys.ku$_audit_obj_base_view to select_catalog_role;
/
create or replace force view ku$_audit_obj_view of ku$_audit_obj_t
  with object identifier(obj_num) as
  select value(ku$) from sys.ku$_audit_obj_base_view ku$
  where ku$.audit_val!='--------------------------------'
  and  (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (ku$.base_obj.owner_num, 0) OR 
	      EXISTS ( SELECT * FROM session_roles 
                       WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on sys.ku$_audit_obj_view to public;
/
-------------------------------------------------------------------------------
--				JAVA_SOURCE
-------------------------------------------------------------------------------
-- 
create or replace type ku$_java_source_t as object
(
  vers_major    char(1),			      /* UDT major version # */
  vers_minor    char(1),			      /* UDT minor version # */
  obj_num	number,				/* java source object number */
  schema_obj	ku$_schemaobj_t,		/* java source schema object */
  syn_long_name varchar2(4000),                         /* synonym long name */
  source_lines  ku$_source_list_t			     /* source lines */
)
/
grant execute on ku$_java_source_t to public
/
create or replace force view ku$_java_source_view of ku$_java_source_t
  with object identifier(obj_num) as
  select '1','0',
	 o.obj_num, value(o),
         (select j.longdbcs from sys.javasnm$ j where j.short = o.name),
         cast(multiset(select s.joxftobn, s.joxftlno, NULL,NULL, s.joxftsrc
		from  x$joxfs s
                       where s.joxftobn = o.obj_num order by s.joxftlno
                       ) as ku$_source_list_t
             )
  from sys.ku$_schemaobj_view o
  where o.type_num = 28 and
            (SYS_CONTEXT('USERENV','CURRENT_USERID') IN (o.owner_num, 0) OR 
	     EXISTS ( SELECT * FROM session_roles 
                WHERE role='SELECT_CATALOG_ROLE' ))
/
grant select on ku$_java_source_view to public
/
--
-------------------------------------------------------------------------------
--				AQ_QUEUE_TABLE
-------------------------------------------------------------------------------
-- 
create or replace type ku$_queue_table_t as object
(
  vers_major    char(1),			      /* UDT major version # */
  vers_minor    char(1),			      /* UDT minor version # */
  obj_num	number,				/* java source object number */
  schema_obj	ku$_schemaobj_t,		    /* object of queue table */
  udata_type	number, 			            /* userdata type */
  object_type	varchar2(61),			       /* userdata type name */
  sort_cols	number, 			   /* sort order for dequeue */
  flags		number, 			   /* queue table properties */
  table_comment 	varchar2(2000),		             /* user comment */
  primary_instance  	number,		       /*  primary owner instance-id */
  secondary_instance	number,		      /* secondary owner instance-id */
  owner_instance 	number		        /* current owner instance-id */
)
/
grant execute on ku$_queue_table_t to public
/
create or replace force view ku$_queue_table_view of ku$_queue_table_t
  with object identifier(obj_num) as
  select '1','0',
	  t.objno, 
          (select value(qo) from  sys.ku$_schemaobj_view qo where 
	     qo.obj_num=t.objno),
          t.udata_type,
         (select u.name || '.' || o.name  
	    from sys.ku$_schemaobj_view o, sys.user$ u,
	         sys.col$ c, sys.coltype$ ct  
	  where c.intcol# = ct.intcol# 
	        and c.obj# = ct.obj# 
	        and c.name = 'USER_DATA' 
	        and t.objno = c.obj# 
	        and o.oid = ct.toid
	        and o.type_num = 13
	        and o.owner_num = u.user#),
         t.sort_cols,
         t.flags,
         t.table_comment,
         aft.primary_instance,
         aft.secondary_instance,
         aft.owner_instance
  from   system.aq$_queue_tables t, 
         sys.aq$_queue_table_affinities aft
  where  t.objno = aft.table_objno 
/
grant select on ku$_queue_table_view to public
/
--
-------------------------------------------------------------------------------
--				AQ_QUEUE
-------------------------------------------------------------------------------
-- 
create or replace type ku$_queues_t as object
(
  vers_major    char(1),			      /* UDT major version # */
  vers_minor    char(1),			      /* UDT minor version # */
  obj_num	number,				/* queue table object number */
  qid           number, 		           /* queue obj number */ 
  schema_obj	ku$_schemaobj_t,		      /* queue object schema */
  base_obj	ku$_schemaobj_t,		   /* queue table obj schema */
  tflags        number, 			   /* queue table properties */
  usage         number,				       /* usage of the queue */
  max_retries   number, 			/* maximum number of retries */
  retry_delay   number,			     /* delay before retrying (secs) */
  enqueue_enabled number,			          /*  queue enabled? */
  properties    number,			     /*  various properties of queue */
  retention     number,			      /* retention time (in seconds) */
  queue_comment  varchar2(2000)			             /* user comment */
)
/
grant execute on ku$_queues_t to public
/
create or replace force view ku$_queues_view of ku$_queues_t
  with object identifier(obj_num) as
  select '1','0',
	t.objno, 
        q.eventid,
     (select value(o) from sys.ku$_schemaobj_view o where o.obj_num=q.eventid),
     (select value(b) from sys.ku$_schemaobj_view b where b.obj_num=t.objno),
	t.flags,
	q.usage, q.max_retries, q.retry_delay,
        q.enable_flag, 
        q.properties, q.ret_time, q.queue_comment
from system.aq$_queues q, system.aq$_queue_tables t
where   q.table_objno = t.objno 
/
grant select on ku$_queues_view to public
/
--
-------------------------------------------------------------------------------
--				AQ_TRANSFORM
-------------------------------------------------------------------------------
-- 
create or replace type ku$_qtrans_t as object
(
  vers_major    char(1),			      /* UDT major version # */
  vers_minor    char(1),			      /* UDT minor version # */
  transformation_id    number,   		        /* transformation id */
  schema_name          varchar2(30),			           /* schema */
  transform_name       varchar2(30), 		      /* transformation name */
  from_obj	       ku$_schemaobj_t,			  /* from obj schema */
  to_obj	       ku$_schemaobj_t,			    /* to obj schema */
  attribute_num	       number,					
  sql_expression       varchar2(4000)
)
/
grant execute on ku$_qtrans_t to public
/
create or replace force view ku$_qtrans_view of ku$_qtrans_t
  with object identifier(transformation_id) as
  select '1','0',
	  t.transformation_id,
	  u.name, t.name, 
	  (select value (f) from ku$_schemaobj_view f where f.oid=t.from_toid),
	  (select value (o) from ku$_schemaobj_view o where o.oid=t.to_toid),
          at.attribute_number,
          replace(at.sql_expression, '''', '''''')
  FROM sys.user$ u , transformations$ t, attribute_transformations$ at
  WHERE  u.name = t.owner and t.transformation_id = at.transformation_id
/
grant select on ku$_qtrans_view to	 public
/
--
-------------------------------------------------------------------------------
--				JOB
-------------------------------------------------------------------------------
-- 
create or replace type ku$_job_t as object
(
  vers_major    char(1),			      /* UDT major version # */
  vers_minor    char(1),			      /* UDT minor version # */
  powner_id	number,				                 /* owner id */
  powner	varchar2(30),			               /* owner name */
  lowner	varchar2(30),				   /* logged in user */
  cowner	varchar2(30),				         /* parsing  */
  job_id        number, 		                          /* job id  */
  last_date     varchar2(19),                /* when this job last succeeded */
  this_date     varchar2(19),   /* when current execute started,usually null */
  next_date     varchar2(19),                /* when to execute the job next */
  flag          number, 			 /* 0x01, this job is broken */
  failures        number,           /* number of failures since last success */
  interval_num  varchar2(200),	              /* function for next next_date */
  what		varchar2(4000),              /* PL/SQL text, what is the job */
  nlsenv	varchar2(4000),                            /* nls parameters */
  env	 	raw(32),                      /* other environment variables */
  field1   	number,            /* instance number restricted to run  job */
  charenv       varchar2(4000)                /* Reserved for Trusted Oracle */
)
/
grant execute on ku$_job_t to public
/
create or replace force view ku$_job_view of ku$_job_t
  with object identifier(powner_id) as
  select '1','0',
         u.user#, j.powner, j.lowner, j.cowner, j.job,
         TO_CHAR(j.last_date, 'YYYY-MM-DD:HH24:MI:SS'),
         TO_CHAR(j.this_date, 'YYYY-MM-DD:HH24:MI:SS'),
         TO_CHAR(j.next_date, 'YYYY-MM-DD:HH24:MI:SS'),
         j.flag, j.failures,
 	 REPLACE(j.interval#, '''', ''''''),
         REPLACE(j.what, '''', ''''''),
         REPLACE(j.nlsenv, '''', ''''''),
 	 j.env, j.field1, j.charenv
  from   sys.job$ j, sys.user$ u
  where  j.powner = u.name
/
grant select on ku$_job_view to public
/
--
-- POPULATE THE DICTIONARY TABLES WITH THE ORACLE MODEL INFORMATION
-- The Metadata API is table-driven. The following inserts set up the ORACLE
-- object model of the dictionary. Other models (eg, CWM, ANSI, etc) can be
-- plugged in by emulating what we do here. First, clear out old stuff.
--
DELETE FROM sys.metaview$       WHERE model='ORACLE';
DELETE FROM sys.metafilter$     WHERE model='ORACLE';
DELETE FROM sys.metaxslparam$   WHERE model='ORACLE';
DELETE FROM sys.metaxsl$        WHERE model='ORACLE';
DELETE FROM sys.metastylesheet  WHERE model='ORACLE';

-- Populate metaview$: This table identifies which view (object or relational)
-- to generate a SELECT against for a given object type. Might be multiple rows
-- in this table per object type if the object type's space has been
-- partitioned into multiple views. We did this for tables for performance:
-- Simple, relational, non-partitioned tables are in a high performance object
-- view with a relatively simple UDT.
-- NOTE: The following table flags are internal to the mdAPI and do *not*
-- correspond to the 'property' bits in tab$.

rem  table flags:   1 = heap tables
rem                 2 = index-organized tables
rem                 4 = unused
rem                 8 = unused
rem                16 = non-partitioned tables
rem                32 = partitioned tables
rem                64 = unused
rem               128 = unused
rem               256 = relational tables
rem               512 = object tables
rem              1024 = unused
rem              2048 = unused
rem              4096 = tables with builtin, non-LOB datatypes
rem              8192 = tables with LOB columns
rem             16384 = tables with UDT columns
rem  The next flags designate the particular view.  This allows a filter
rem  to explicitly specify a desired set of views.
rem
rem             65536 = HTABLE
rem            131072 = FHTABLE
rem            262144 = PHTABLE
rem            524288 = PFHTABLE
rem           1048576 = IOTABLE
rem           2097152 = PIOTABLE
rem Like tables, materialized views and materialized view logs
rem have multiple UDTs and views. Each mv and mvl UDT contains
rem one of the table UDTs. Note that not all table types are represented.

rem heap, non-partitioned, relational, no LOB or UDT columns
insert into metaview$ (type, flags, properties, model, version, xmltag,
                       udt, schema, viewname) values
        ('TABLE',1+16+256+4096+65536,1,'ORACLE',0801000000,'TABLE_T',
         'KU$_HTABLE_T','SYS','KU$_HTABLE_VIEW');

insert into metaview$ (type, flags, properties, model, version, xmltag,
			udt, schema, viewname) values
        ('MATERIALIZED_VIEW_LOG',1+16+256+4096+65536,0,'ORACLE',0801000000,
         'M_VIEW_LOG_T', 'KU$_M_VIEW_LOG_H_T','SYS','KU$_M_VIEW_LOG_H_VIEW');

rem heap, non-partitioned, relational or object, any datatype
rem FHTABLE_VIEW does NOT set the bit for builtin, non-LOB types so that it
rem will be mutually exclusive of view HTABLE_VIEW. In other words, we DON'T
rem want this view used for simple relational tables because it's slow.

insert into metaview$ (type, flags, properties, model, version, xmltag,
 			udt, schema, viewname) values 
       ('TABLE',1+16+256+512+8192+16384+131072,1,'ORACLE',0801000000,'TABLE_T',
         'KU$_FHTABLE_T','SYS','KU$_FHTABLE_VIEW');

insert into metaview$ (type, flags, properties, model, version, xmltag,
 			udt, schema, viewname) values 
     ('MATERIALIZED_VIEW',1+16+256+512+8192+16384+131072,1,'ORACLE',0801000000,
         'M_VIEW_T','KU$_M_VIEW_FH_T','SYS','KU$_M_VIEW_FH_VIEW');

insert into metaview$ (type, flags, properties, model, version, xmltag,
 			udt, schema, viewname) values 
 ('MATERIALIZED_VIEW_LOG',1+16+256+512+8192+16384+131072,0,'ORACLE',0801000000,
         'M_VIEW_LOG_T','KU$_M_VIEW_LOG_FH_T','SYS','KU$_M_VIEW_LOG_FH_VIEW');

rem heap, partitioned, relational, no LOB or UDT columns
insert into metaview$ (type, flags, properties, model, version, xmltag,
 			udt, schema, viewname) values 
        ('TABLE',1+32+256+4096+262144,1,'ORACLE',0801000000,'TABLE_T',
         'KU$_PHTABLE_T','SYS','KU$_PHTABLE_VIEW');

insert into metaview$ (type, flags, properties, model, version, xmltag,
 			udt, schema, viewname) values 
        ('MATERIALIZED_VIEW_LOG',1+32+256+4096+262144,0,'ORACLE',0801000000,
         'M_VIEW_LOG_T', 'KU$_M_VIEW_LOG_PH_T','SYS','KU$_M_VIEW_LOG_PH_VIEW');

rem heap, partitioned, relational or object, any datatype
rem PFHTABLE_VIEW does NOT set the bit for builtin, non-LOB types so that it
rem will be mutually exclusive of view PHTABLE_VIEW. In other words, we DON'T
rem want this view used for simple, partitioned tables because it's slow.
insert into metaview$ (type, flags, properties, model, version, xmltag,
 			udt, schema, viewname) values 
       ('TABLE',1+32+256+512+8192+16384+524288,1,'ORACLE',0801000000,'TABLE_T',
         'KU$_PFHTABLE_T','SYS','KU$_PFHTABLE_VIEW');

insert into metaview$ (type, flags, properties, model, version, xmltag,
 			udt, schema, viewname) values 
     ('MATERIALIZED_VIEW',1+32+256+512+8192+16384+524288,1,'ORACLE',0801000000,
         'M_VIEW_T', 'KU$_M_VIEW_PFH_T','SYS','KU$_M_VIEW_PFH_VIEW');

insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values 
        ('MATERIALIZED_VIEW_LOG',1+32+256+512+8192+16384+524288,0,
         'ORACLE',0801000000,
         'M_VIEW_LOG_T', 'KU$_M_VIEW_LOG_PFH_T','SYS',
         'KU$_M_VIEW_LOG_PFH_VIEW');

rem IOT, non-partitioned, relational, any datatype
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values 
         ('TABLE',2+16+256+512+4096+8192+16384+1048576,1,'ORACLE',
         0801000000,'TABLE_T',
         'KU$_IOTABLE_T','SYS','KU$_IOTABLE_VIEW');

insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values 
        ('MATERIALIZED_VIEW',2+16+256+512+4096+8192+16384+1048576,1,'ORACLE',
          0801000000,'M_VIEW_T','KU$_M_VIEW_IOT_T','SYS',
          'KU$_M_VIEW_IOT_VIEW');

rem IOT, partitioned, relational, any datatype
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values 
        ('TABLE',2+32+256+512+4096+8192+16384+2097152,1,'ORACLE',
         0801000000,'TABLE_T',
         'KU$_PIOTABLE_T','SYS','KU$_PIOTABLE_VIEW');

insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values 
        ('MATERIALIZED_VIEW',2+32+256+512+4096+8192+16384+2097152,1,'ORACLE',
         0801000000,'M_VIEW_T', 'KU$_M_VIEW_PIOT_T','SYS',
         'KU$_M_VIEW_PIOT_VIEW');

rem
rem  type flags:  1 = type specification
rem               2 = type body
rem

insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
	 ('TYPE',1,1,'ORACLE',0801000000,'TYPE_T',
                      'KU$_TYPE_T','SYS','KU$_TYPE_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
	 ('TYPE',2,1,'ORACLE',0801000000,'TYPE_BODY_T',
                      'KU$_TYPE_BODY_T','SYS','KU$_TYPE_BODY_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
	 ('TYPE',3,1,'ORACLE',0801000000,'FULL_TYPE_T',
                      'KU$_FULL_TYPE_T','SYS','KU$_FULL_TYPE_VIEW');

insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
	 ('TYPE_SPEC',0,1,'ORACLE',0801000000,'TYPE_T',
                     'KU$_TYPE_T','SYS','KU$_TYPE_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
	 ('TYPE_BODY',0,1,'ORACLE',0801000000,
 	'TYPE_BODY_T', 'KU$_TYPE_BODY_T','SYS','KU$_TYPE_BODY_VIEW');

rem
rem  package flags:  1 = package specification
rem                  2 = package body
rem

insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('PACKAGE',1,1,'ORACLE',0801000000,'PACKAGE_T',
                      'KU$_PROC_T','SYS','KU$_PKG_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
               ('PACKAGE',2,1,'ORACLE',0801000000,'PACKAGE_BODY_T',
                      'KU$_PROC_T','SYS','KU$_PKGBDY_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
               ('PACKAGE',3,1,'ORACLE',0801000000,'FULL_PACKAGE_T',
                'KU$_FULL_PKG_T','SYS','KU$_FULL_PKG_VIEW');

insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('PACKAGE_SPEC',0,1,'ORACLE',0801000000,
		'PACKAGE_T','KU$_PROC_T','SYS','KU$_PKG_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
               ('PACKAGE_BODY',0,1,'ORACLE',0801000000,'PACKAGE_BODY_T',
                      'KU$_PROC_T','SYS','KU$_PKGBDY_VIEW');

insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('INDEX',0,1,'ORACLE',0801000000,'INDEX_T',
                      'KU$_INDEX_T','SYS','KU$_INDEX_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		('PROCEDURE',0,1,'ORACLE',0801000000,'PROCEDURE_T',
                      'KU$_PROC_T','SYS','KU$_PROC_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('FUNCTION',0,1,'ORACLE',0801000000,'FUNCTION_T',
                      'KU$_PROC_T','SYS','KU$_FUNC_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('OPERATOR',0,1,'ORACLE',0801000000,'OPERATOR_T',
                      'KU$_OPERATOR_T','SYS','KU$_OPERATOR_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		('INDEXTYPE',0,1,'ORACLE',0801000000,'INDEXTYPE_T',
                      'KU$_INDEXTYPE_T','SYS','KU$_INDEXTYPE_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('OBJECT_GRANT',0,0,'ORACLE',0801000000,
                      'OBJGRANT_T','KU$_OBJGRANT_T','SYS','KU$_OBJGRANT_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('SYSTEM_GRANT',0,0,'ORACLE',0801000000,
                      'SYSGRANT_T','KU$_SYSGRANT_T','SYS','KU$_SYSGRANT_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('TRIGGER',0,1,'ORACLE',0801000000,'TRIGGER_T',
                      'KU$_TRIGGER_T','SYS','KU$_TRIGGER_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('VIEW',0,1,'ORACLE',0801000000,'VIEW_T',
                      'KU$_VIEW_T','SYS','KU$_VIEW_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('OUTLINE',0,0,'ORACLE',0801000000,'OUTLINE_T',
                      'KU$_OUTLINE_T','SYS','KU$_OUTLINE_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('SYNONYM',0,0,'ORACLE',0801000000,'SYNONYM_T',
                      'KU$_SYNONYM_T','SYS','KU$_SYNONYM_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('DIRECTORY',0,0,'ORACLE',0801000000,
                      'DIRECTORY_T', 'KU$_DIRECTORY_T','SYS',
                      'KU$_DIRECTORY_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('TABLESPACE',0,0,'ORACLE',0801000000,
                      'TABLESPACE_T', 'KU$_TABLESPACE_T','SYS',
                      'KU$_TABLESPACE_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('ROLLBACK_SEGMENT',0,0,'ORACLE',0801000000,
                      'ROLLBACK_T', 'KU$_ROLLBACK_T', 'SYS',
                      'KU$_ROLLBACK_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('DB_LINK',0,1,'ORACLE',0801000000, 'DBLINK_T',
                      'KU$_DBLINK_T','SYS', 'KU$_DBLINK_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('TRUSTED_DB_LINK',0,0,'ORACLE',0801000000,
                      'TRLINK_T', 'KU$_TRLINK_T','SYS', 'KU$_TRLINK_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('FGA_POLICY',0,0,'ORACLE',0801000000,
                      'FGA_POLICY_T', 'KU$_FGA_POLICY_T','SYS',
                      'KU$_FGA_POLICY_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('RLS_POLICY',0,0,'ORACLE',0801000000,
                      'RLS_POLICY_T', 'KU$_RLS_POLICY_T','SYS',
                      'KU$_RLS_POLICY_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('RLS_GROUP',0,0,'ORACLE',0801000000,
                      'RLS_GROUP_T', 'KU$_RLS_GROUP_T','SYS',
                      'KU$_RLS_GROUP_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('RLS_CONTEXT',0,0,'ORACLE',0801000000,
                      'RLS_CONTEXT_T', 'KU$_RLS_CONTEXT_T','SYS',
                      'KU$_RLS_CONTEXT_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('CONSTRAINT',0,1,'ORACLE',0801000000,
                      'CONSTRAINT_T', 'KU$_CONSTRAINT_T','SYS',
                      'KU$_CONSTRAINT_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('REF_CONSTRAINT',0,1,'ORACLE',0801000000,
                      'REF_CONSTRAINT_T', 'KU$_REF_CONSTRAINT_T','SYS',
                      'KU$_REF_CONSTRAINT_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('LIBRARY',0,1,'ORACLE',0801000000,'LIBRARY_T',
                      'KU$_LIBRARY_T','SYS','KU$_LIBRARY_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('PROFILE',0,0,'ORACLE',0801000000,'PROFILE_T',
                      'KU$_PROFILE_T','SYS','KU$_PROFILE_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('ROLE',0,0,'ORACLE',0801000000,'ROLE_T',
                      'KU$_ROLE_T','SYS','KU$_ROLE_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('USER',0,0,'ORACLE',0801000000,'USER_T',
                      'KU$_USER_T','SYS','KU$_USER_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('DEFAULT_ROLE',0,0,'ORACLE',0801000000,
                      'DEFROLE_T','KU$_DEFROLE_T','SYS','KU$_DEFROLE_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('PROXY',0,0,'ORACLE',0801000000,
                      'PROXY_T','KU$_PROXY_T','SYS','KU$_PROXY_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('ROLE_GRANT',0,0,'ORACLE',0801000000,
                      'ROGRANT_T','KU$_ROGRANT_T','SYS','KU$_ROGRANT_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('TABLESPACE_QUOTA',0,0,'ORACLE',0801000000,
                      'TSQUOTA_T','KU$_TSQUOTA_T','SYS','KU$_TSQUOTA_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('RESOURCE_COST',0,0,'ORACLE',0801000000,
                      'RESOCOST_T','KU$_RESOCOST_T','SYS','KU$_RESOCOST_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('SEQUENCE',0,1,'ORACLE',0801000000,'SEQUENCE_T',
                      'KU$_SEQUENCE_T','SYS','KU$_SEQUENCE_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('CONTEXT',0,0,'ORACLE',0801000000,'CONTEXT_T',
                      'KU$_CONTEXT_T','SYS','KU$_CONTEXT_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('DIMENSION',0,1,'ORACLE',0801000000,
		'DIMENSION_T','KU$_DIMENSION_T','SYS','KU$_DIMENSION_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('ASSOCIATION',0,0,'ORACLE',0801000000,'ASSOC_T',
                'KU$_ASSOC_T','SYS','KU$_ASSOC_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('PASSWORD_VERIFY_FUNCTION',0, 0,'ORACLE',
		0801000000,'PWDVFC_T','KU$_PROC_T','SYS','KU$_PWDVFC_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('COMMENT',0,0,'ORACLE',0801000000,'COMMENT_T',
                'KU$_COMMENT_T','SYS','KU$_COMMENT_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('CLUSTER',0,1,'ORACLE',0801000000,
		'CLUSTER_T','KU$_CLUSTER_T','SYS','KU$_CLUSTER_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('AUDIT',0,0,'ORACLE',0801000000,'AUDIT_T',
                'KU$_AUDIT_T','SYS','KU$_AUDIT_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('AUDIT_OBJ',0,0,'ORACLE',0801000000,'AUDITOBJ_T',
                'KU$_AUDIT_OBJ_T','SYS','KU$_AUDIT_OBJ_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('JAVA_SOURCE',0,1,'ORACLE',0801000000,
		'JAVASRC_T','KU$_JAVA_SOURCE_T','SYS','KU$_JAVA_SOURCE_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag, 
			udt, schema, viewname) values
		 ('AQ_QUEUE_TABLE',0,0,'ORACLE',0801000000,
		'QUETAB_T','KU$_QUEUE_TABLE_T','SYS','KU$_QUEUE_TABLE_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag,
 			udt, schema, viewname) values
		 ('AQ_QUEUE',0,0,'ORACLE',0801000000,
		'QUEUES_T','KU$_QUEUES_T','SYS','KU$_QUEUES_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag,
 			udt, schema, viewname) values
		 ('AQ_TRANSFORM',0,0,'ORACLE',0801000000,
		'QTRANS_T','KU$_QTRANS_T','SYS','KU$_QTRANS_VIEW');
insert into metaview$ (type, flags, properties, model, version, xmltag,
 			udt, schema, viewname) values
		 ('JOB',0,0,'ORACLE',0801000000,
		'JOB_T','KU$_JOB_T','SYS','KU$_JOB_VIEW');

-- new in 9.2
insert into metaview$ (type, flags, properties, model, version, xmltag,
 			udt, schema, viewname) values
		 ('XMLSCHEMA',0,1,'ORACLE',0900000000,
		'XMLSCHEMA_T', 'KU$_XMLSCHEMA_T','SYS','KU$_XMLSCHEMA_VIEW');

-- Populate metafilter$: This table defines all the valid SELECT filters for
-- dictionary object types that can be set through the SET_FILTER interface.
-- Common filters used by many object types are NAME / SCHEMA. grants support
-- GRANTOR / GRANTEE. Tables support TABLESPACE, HEAP or IOT, RELATIONAL or
-- OBJECT, etc. Each one of these turns into a WHERE clause predicate on the
-- SELECT stmts built against the views in metaview$. The attrname field
-- specifies the UDT attribute or rel. column used in the where clause.

insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'FUNCTION', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'INDEX', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'INDEXTYPE', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'OPERATOR', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'OUTLINE', 'ORACLE', 0, 0, 'NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'PACKAGE', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'PACKAGE_SPEC', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'PACKAGE_BODY', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'PROCEDURE', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'SYNONYM', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'TABLE', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'TRIGGER', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'TYPE', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'TYPE_SPEC', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'TYPE_BODY', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'VIEW', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'MATERIALIZED_VIEW', 'ORACLE', 0, 0, 'MVIEW.VNAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'MATERIALIZED_VIEW_LOG', 'ORACLE', 0, 0, 'MVIEWLOG.LOG', NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'DIRECTORY', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'ROLLBACK_SEGMENT', 'ORACLE', 0, 0, 'NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'DB_LINK', 'ORACLE', 0, 0, 'NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'TRUSTED_DB_LINK', 'ORACLE', 0, 0, 'NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'TABLESPACE', 'ORACLE', 0, 0, 'NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'CONSTRAINT', 'ORACLE', 0, 0, 'NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'REF_CONSTRAINT', 'ORACLE', 0, 0, 'NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'USER', 'ORACLE', 0, 0, 'NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'LIBRARY', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'PROFILE', 'ORACLE', 0, 0, 'PROFILE_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'ROLE', 'ORACLE', 0, 0, 'NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'SEQUENCE', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'CONTEXT', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'DIMENSION', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'CLUSTER', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'JAVA_SOURCE', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'AQ_QUEUE_TABLE', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'AQ_QUEUE', 'ORACLE', 0, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'AQ_TRANSFORM', 'ORACLE', 0, 0, 'TRANSFORM_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'JOB', 'ORACLE', 0, 0, 'POWNER',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME', 'XMLSCHEMA', 'ORACLE', 0, 0, 'URL',NULL);

insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'FUNCTION', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'INDEX', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'INDEXTYPE', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'OPERATOR', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'OUTLINE', 'ORACLE', 2, 0, 'NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'PACKAGE', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'PACKAGE_SPEC', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'PACKAGE_BODY', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'PROCEDURE', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'SYNONYM', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'TABLE', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'TRIGGER', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'TYPE', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'TYPE_SPEC', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'TYPE_BODY', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'VIEW', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'MATERIALIZED_VIEW', 'ORACLE', 2, 0, 'MVIEW.VNAME', NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'MATERIALIZED_VIEW_LOG', 'ORACLE', 2, 0, 'MVIEWLOG.LOG', NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'DIRECTORY', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'ROLLBACK_SEGMENT', 'ORACLE', 2, 0, 'NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'DB_LINK', 'ORACLE', 2, 0, 'NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'TRUSTED_DB_LINK', 'ORACLE', 2, 0, 'NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'TABLESPACE', 'ORACLE', 2, 0, 'NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'CONSTRAINT', 'ORACLE', 2, 0, 'NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'REF_CONSTRAINT', 'ORACLE', 2, 0, 'NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'LIBRARY', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'PROFILE', 'ORACLE', 2, 0, 'PROFILE_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'ROLE', 'ORACLE', 2, 0, 'NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'SEQUENCE', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'USER', 'ORACLE', 2, 0, 'NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'CONTEXT', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'DIMENSION', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'CLUSTER', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'JAVA_SOURCE', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'AQ_QUEUE_TABLE', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'AQ_QUEUE', 'ORACLE', 2, 0, 'SCHEMA_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'AQ_TRANSFORM', 'ORACLE', 2, 0, 'TRANSFORM_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'JOB', 'ORACLE', 2, 0, 'POWNER',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NAME_EXPR', 'XMLSCHEMA', 'ORACLE', 2, 0, 'URL',NULL);


insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'FUNCTION', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'INDEX', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'INDEXTYPE', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'OPERATOR', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'PACKAGE', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'PACKAGE_SPEC', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'PACKAGE_BODY', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'PROCEDURE', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'SYNONYM', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'TABLE', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'TRIGGER', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'TYPE', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'TYPE_SPEC', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'TYPE_BODY', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'VIEW', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'MATERIALIZED_VIEW', 'ORACLE', 0, 0, 'MVIEW.SOWNER', NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'MATERIALIZED_VIEW_LOG', 'ORACLE', 0, 0, 'MVIEWLOG.MOWNER', 
  NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'CONSTRAINT', 'ORACLE', 0, 0, 'OWNER', NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'REF_CONSTRAINT', 'ORACLE', 0, 0, 'OWNER', NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'DB_LINK', 'ORACLE', 0, 0, 'OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'LIBRARY', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'SEQUENCE', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'DIMENSION', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'CLUSTER', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'JAVA_SOURCE', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'AQ_TRANSFORM', 'ORACLE', 0, 0, 'SCHEMA_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'AQ_QUEUE_TABLE', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
 insert into metafilter$ (filter, type, model, properties, view_attr
,                         attrname, default_val) values
 ('SCHEMA', 'AQ_QUEUE', 'ORACLE', 0, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA', 'XMLSCHEMA', 'ORACLE', 0, 0, 'OWNER_NAME',NULL);

insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'FUNCTION', 'ORACLE', 2, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'INDEX', 'ORACLE', 2, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'INDEXTYPE', 'ORACLE', 2, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'OPERATOR', 'ORACLE', 2, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'PACKAGE', 'ORACLE', 2, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'PACKAGE_SPEC', 'ORACLE', 2, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'PACKAGE_BODY', 'ORACLE', 2, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'PROCEDURE', 'ORACLE', 2, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'SYNONYM', 'ORACLE', 2, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'TABLE', 'ORACLE', 2, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'TRIGGER', 'ORACLE', 2, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'TYPE', 'ORACLE', 2, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'TYPE_SPEC', 'ORACLE', 2, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'TYPE_BODY', 'ORACLE', 2, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'VIEW', 'ORACLE', 2, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'MATERIALIZED_VIEW', 'ORACLE', 2, 0, 'MVIEW.SOWNER', NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'MATERIALIZED_VIEW_LOG', 'ORACLE', 2, 0, 'MVIEWLOG.MOWNER', 
  NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'CONSTRAINT', 'ORACLE', 2, 0, 'OWNER', NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'REF_CONSTRAINT', 'ORACLE', 2, 0, 'OWNER', NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'DB_LINK', 'ORACLE', 2, 0, 'OWNER_NAME', NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'LIBRARY', 'ORACLE', 2, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'SEQUENCE', 'ORACLE', 2, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'DIMENSION', 'ORACLE', 2, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'CLUSTER', 'ORACLE', 2, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'JAVA_SOURCE', 'ORACLE', 2, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'AQ_TRANSFORM', 'ORACLE', 2, 0, 'SCHEMA_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'AQ_QUEUE_TABLE','ORACLE', 2, 0,'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'AQ_QUEUE', 'ORACLE', 2, 0, 'SCHEMA_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMA_EXPR', 'XMLSCHEMA', 'ORACLE', 2, 0, 'OWNER_NAME',NULL);

insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SPECIFICATION', 'PACKAGE', 'ORACLE', 1, 1, NULL,NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SPECIFICATION', 'PACKAGE_SPEC', 'ORACLE', 1, 1, NULL,NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SPECIFICATION', 'TYPE', 'ORACLE', 1, 1, NULL,NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SPECIFICATION', 'TYPE_SPEC', 'ORACLE', 1, 1, NULL,NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BODY', 'PACKAGE', 'ORACLE', 1, 2, NULL,NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BODY', 'PACKAGE_BODY', 'ORACLE', 1, 2, NULL,NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BODY', 'TYPE', 'ORACLE', 1, 2, NULL,NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BODY', 'TYPE_BODY', 'ORACLE', 1, 2, NULL,NULL);

insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('TABLESPACE', 'TABLE', 'ORACLE', 0, 0,
  '(EXISTS (SELECT 1 FROM SYS.KU$_TAB_TS_VIEW TTS ' ||
  'WHERE TTS.OBJ_NUM = KU$.OBJ_NUM AND TTS.TS_NAME :1))',
  NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('TABLESPACE_EXPR', 'TABLE', 'ORACLE', 2, 0,
  '(EXISTS (SELECT 1 FROM SYS.KU$_TAB_TS_VIEW TTS ' ||
  'WHERE TTS.OBJ_NUM = KU$.OBJ_NUM AND TTS.TS_NAME :1))',
  NULL);

insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_NAME', 'INDEX', 'ORACLE', 0, 0, 'BASE_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_NAME', 'OBJECT_GRANT', 'ORACLE', 0, 0, 'BASE_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_NAME', 'TRIGGER', 'ORACLE', 0, 0, 'BASE_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_NAME', 'FGA_POLICY', 'ORACLE', 0, 0, 'BASE_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_NAME', 'RLS_POLICY', 'ORACLE', 0, 0, 'BASE_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_NAME', 'RLS_GROUP', 'ORACLE', 0, 0, 'BASE_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_NAME', 'RLS_CONTEXT', 'ORACLE', 0, 0, 'BASE_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_NAME', 'MATERIALIZED_VIEW_LOG', 'ORACLE', 0, 0,
  'MVIEWLOG.MASTER', NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_NAME', 'CONSTRAINT', 'ORACLE', 0, 0, 'BASE_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_NAME', 'REF_CONSTRAINT', 'ORACLE', 0, 0, 'BASE_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_NAME', 'COMMENT', 'ORACLE', 0, 0, 'BASE_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_NAME', 'ASSOCIATION', 'ORACLE', 0, 0, 'BASE_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_NAME', 'AUDIT', 'ORACLE', 0, 0, 'AUDIT_OPTION',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_NAME', 'AUDIT_OBJ', 'ORACLE', 0, 0, 'BASE_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_NAME', 'AQ_QUEUE', 'ORACLE', 0, 0, 'BASE_OBJ.NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_NAME', 'AQ_QUEUE_TABLE', 'ORACLE', 0, 0,
 'SCHEMA_OBJ.NAME',NULL);

insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_SCHEMA', 'INDEX', 'ORACLE', 0, 0, 'BASE_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_SCHEMA', 'OBJECT_GRANT', 'ORACLE', 0, 0, 'BASE_OBJ.OWNER_NAME',
   NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_SCHEMA', 'TRIGGER', 'ORACLE', 0, 0, 'BASE_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_SCHEMA', 'FGA_POLICY', 'ORACLE', 0, 0, 'BASE_OBJ.OWNER_NAME',
  NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_SCHEMA', 'RLS_POLICY', 'ORACLE', 0, 0, 'BASE_OBJ.OWNER_NAME',
  NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_SCHEMA', 'RLS_GROUP', 'ORACLE', 0, 0, 'BASE_OBJ.OWNER_NAME',
  NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_SCHEMA', 'RLS_CONTEXT', 'ORACLE', 0, 0, 'BASE_OBJ.OWNER_NAME',
  NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_SCHEMA', 'MATERIALIZED_VIEW_LOG', 'ORACLE', 0, 0,
  'MVIEWLOG.MOWNER', NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_SCHEMA', 'CONSTRAINT', 'ORACLE', 0, 0, 'BASE_OBJ.OWNER_NAME',
   NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_SCHEMA', 'REF_CONSTRAINT', 'ORACLE', 0, 0,
  'BASE_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_SCHEMA', 'ASSOCIATION', 'ORACLE', 0, 0, 
  'BASE_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_SCHEMA', 'COMMENT', 'ORACLE', 0, 0, 'BASE_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_SCHEMA', 'AUDIT_OBJ', 'ORACLE', 0, 0, 
  'BASE_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_SCHEMA', 'AQ_QUEUE', 'ORACLE', 0, 0, 
  'BASE_OBJ.OWNER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BASE_OBJECT_SCHEMA', 'AQ_QUEUE_TABLE', 'ORACLE', 0, 0, 
  'SCHEMA_OBJ.OWNER_NAME',NULL);

insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SYSTEM_GENERATED', 'INDEX', 'ORACLE', 9, 0,
                                        'BITAND(KU$.SCHEMA_OBJ.FLAGS,4)=4',1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SYSTEM_GENERATED', 'TRIGGER', 'ORACLE', 9, 0,
                                        'BITAND(KU$.SCHEMA_OBJ.FLAGS,4)=4',1);

-- If BASE_OBJECT_NAME=DATABASE, we set NON_DB filter false to exclude
--  all triggers that are not database triggers.
-- Similarly, if BASE_OBJECT_NAME=SCHEMA,we set NON_SCHEMA filter false.
-- (A bit clumsy but it works.)

insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NON_DB', 'TRIGGER', 'ORACLE', 9, 0, 'BITAND(KU$.PROPERTY,8)=0',1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NON_SCHEMA', 'TRIGGER', 'ORACLE', 9, 0, 'BITAND(KU$.PROPERTY,16)=0',1);

insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('LONGNAME', 'SYNONYM', 'ORACLE', 0, 0, 'SYN_LONG_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('LONGNAME_EXPR', 'SYNONYM', 'ORACLE', 2, 0, 'SYN_LONG_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('LONGNAME', 'JAVA_SOURCE', 'ORACLE', 0, 0, 'SYN_LONG_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('LONGNAME_EXPR', 'JAVA_SOURCE', 'ORACLE', 2, 0, 'SYN_LONG_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('LONGNAME', 'XMLSCHEMA', 'ORACLE', 0, 0, 'URL',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('LONGNAME_EXPR', 'XMLSCHEMA', 'ORACLE', 2, 0, 'URL',NULL);

insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('GRANTOR', 'OBJECT_GRANT', 'ORACLE', 0, 0, 'GRANTOR',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('GRANTEE', 'OBJECT_GRANT', 'ORACLE', 0, 0, 'GRANTEE',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('GRANTEE', 'SYSTEM_GRANT', 'ORACLE', 0, 0, 'GRANTEE',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('GRANTEE', 'DEFAULT_ROLE', 'ORACLE', 0, 0, 'USER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('GRANTEE', 'PROXY', 'ORACLE', 0, 0, 'CLIENT_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('GRANTEE', 'ROLE_GRANT', 'ORACLE', 0, 0, 'GRANTEE',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('GRANTEE', 'TABLESPACE_QUOTA', 'ORACLE', 0, 0, 'USER_NAME',NULL);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('GRANTEE', 'AUDIT', 'ORACLE', 0, 0, 'USER_NAME',NULL);


insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('SCHEMAOID', 'XMLSCHEMA', 'ORACLE', 0, 0, 'SCHEMAOID',NULL);

insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('CUSTOM_FILTER', '*', 'ORACLE', 4, 0, NULL,NULL);

/* additional filters for tables */
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('HEAP', 'TABLE', 'ORACLE', 9, 1, NULL,1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('IOT', 'TABLE', 'ORACLE', 9, 2, NULL,1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NON_PARTITIONED', 'TABLE', 'ORACLE', 9, 16, 'BITAND(KU$.PROPERTY,32)=0',1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('PARTITIONED', 'TABLE', 'ORACLE', 9, 32, 'BITAND(KU$.PROPERTY,32)=32',1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('RELATIONAL', 'TABLE', 'ORACLE', 9, 256, 'BITAND(KU$.PROPERTY,1)=0',1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('OBJECT', 'TABLE', 'ORACLE', 9, 512, 'BITAND(KU$.PROPERTY,1)=1',1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BUILTIN_COL', 'TABLE', 'ORACLE', 9, 4096, NULL, 1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('LOB_COL', 'TABLE', 'ORACLE', 9, 8192,
        'BITAND(KU$.PROPERTY,2048+32768+262144)!=0',1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('UDT_COL', 'TABLE', 'ORACLE',9, 16384, 'BITAND(KU$.PROPERTY,2+4+8+16)!=0',1);
Rem
Rem Boolean filters to pick out specific views.
Rem By default the filters are on, i.e., you get all views by default.
Rem
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('HTABLE', 'TABLE', 'ORACLE', 9, 65536, NULL,1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('FHTABLE', 'TABLE', 'ORACLE', 9, 131072, NULL,1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('PHTABLE', 'TABLE', 'ORACLE', 9, 262144, NULL,1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('PFHTABLE', 'TABLE', 'ORACLE', 9, 524288, NULL,1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('IOTABLE', 'TABLE', 'ORACLE', 9, 1048576, NULL,1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('PIOTABLE', 'TABLE', 'ORACLE', 9, 2097152, NULL,1);

/* additional filters for materialized views */
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('HEAP', 'MATERIALIZED_VIEW', 'ORACLE', 9, 1, NULL,1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('IOT', 'MATERIALIZED_VIEW', 'ORACLE', 9, 2, NULL,1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NON_PARTITIONED', 'MATERIALIZED_VIEW', 'ORACLE', 9, 16,
  'BITAND(KU$.MVIEW_TAB.PROPERTY,32)=0',1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('PARTITIONED', 'MATERIALIZED_VIEW', 'ORACLE', 9, 32,
  'BITAND(KU$.MVIEW_TAB.PROPERTY,32)=32',1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('RELATIONAL', 'MATERIALIZED_VIEW', 'ORACLE', 9, 256,
  'BITAND(KU$.MVIEW_TAB.PROPERTY,1)=0',1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('OBJECT', 'MATERIALIZED_VIEW', 'ORACLE', 9, 512,
  'BITAND(KU$.MVIEW_TAB.PROPERTY,1)=1',1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BUILTIN_COL', 'MATERIALIZED_VIEW', 'ORACLE', 9, 4096, NULL, 1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('LOB_COL', 'MATERIALIZED_VIEW', 'ORACLE', 9, 8192,
  'BITAND(KU$.MVIEW_TAB.PROPERTY,2048+32768+262144)!=0',1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('UDT_COL', 'MATERIALIZED_VIEW', 'ORACLE',9, 16384,
  'BITAND(KU$.MVIEW_TAB.PROPERTY,2+4+8+16)!=0',1);

Rem
Rem Boolean filters to pick out specific views.
Rem By default the filters are on, i.e., you get all views by default.
Rem
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('HTABLE', 'MATERIALIZED_VIEW', 'ORACLE', 9, 65536, NULL,0);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('FHTABLE', 'MATERIALIZED_VIEW', 'ORACLE', 9, 131072, NULL,1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('PHTABLE', 'MATERIALIZED_VIEW', 'ORACLE', 9, 262144, NULL,0);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('PFHTABLE', 'MATERIALIZED_VIEW', 'ORACLE', 9, 524288, NULL,1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('IOTABLE', 'MATERIALIZED_VIEW', 'ORACLE', 9, 1048576, NULL,1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('PIOTABLE', 'MATERIALIZED_VIEW', 'ORACLE', 9, 2097152, NULL,1);

/* additional filters for materialized view logs */
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('HEAP', 'MATERIALIZED_VIEW_LOG', 'ORACLE', 9, 1, NULL,1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('IOT', 'MATERIALIZED_VIEW_LOG', 'ORACLE', 9, 2, NULL,0);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('NON_PARTITIONED', 'MATERIALIZED_VIEW_LOG', 'ORACLE', 9, 16,
  'BITAND(KU$.MVIEWLOG_TAB.PROPERTY,32)=0',1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('PARTITIONED', 'MATERIALIZED_VIEW_LOG', 'ORACLE', 9, 32,
  'BITAND(KU$.MVIEWLOG_TAB.PROPERTY,32)=32',1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('RELATIONAL', 'MATERIALIZED_VIEW_LOG', 'ORACLE', 9, 256,
  'BITAND(KU$.MVIEWLOG_TAB.PROPERTY,1)=0',1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('OBJECT', 'MATERIALIZED_VIEW_LOG', 'ORACLE', 9, 512,
  'BITAND(KU$.MVIEWLOG_TAB.PROPERTY,1)=1',1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('BUILTIN_COL', 'MATERIALIZED_VIEW_LOG', 'ORACLE', 9, 4096, NULL, 1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('LOB_COL', 'MATERIALIZED_VIEW_LOG', 'ORACLE', 9, 8192,
  'BITAND(KU$.MVIEWLOG_TAB.PROPERTY,2048+32768+262144)!=0',1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('UDT_COL', 'MATERIALIZED_VIEW_LOG', 'ORACLE',9, 16384,
  'BITAND(KU$.MVIEWLOG_TAB.PROPERTY,2+4+8+16)!=0',1);
Rem
Rem Boolean filters to pick out specific views.
Rem By default the filters are on, i.e., you get all views by default.
Rem
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('HTABLE', 'MATERIALIZED_VIEW_LOG', 'ORACLE', 9, 65536, NULL,1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('FHTABLE', 'MATERIALIZED_VIEW_LOG', 'ORACLE', 9, 131072, NULL,1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('PHTABLE', 'MATERIALIZED_VIEW_LOG', 'ORACLE', 9, 262144, NULL,1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('PFHTABLE', 'MATERIALIZED_VIEW_LOG', 'ORACLE', 9, 524288, NULL,1);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('IOTABLE', 'MATERIALIZED_VIEW_LOG', 'ORACLE', 9, 1048576, NULL,0);
insert into metafilter$ (filter, type, model, properties, view_attr,
                         attrname, default_val) values
 ('PIOTABLE', 'MATERIALIZED_VIEW_LOG', 'ORACLE', 9, 2097152, NULL,0);

/* TBD: 'INCOMPLETE'??? */

-- Populate metaxsl$: This table provides the URI of the top level stylesheet
-- for a given XML document and transform. The XML document is identified by
-- xmltag and model (from metaview$) and the transform name (eg, DDL, CREATE,
-- PARSE, etc) is in 'transform'. 'script' contains the stylesheet name.
-- For ORACLE model stylesheets, this matches the 'name' column in the
-- table metastylesheet; dbms_metadata_int converts this name to a URI.
-- For other models, 'name' is a URI.

insert into metaxsl$ (xmltag, transform, model, script) values
 ('TABLE_T', 'DDL', 'ORACLE', 'kutable');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('INDEX_T', 'DDL', 'ORACLE', 'kuindex');

-- full.xsl can process type spec and type body as well as full type
insert into metaxsl$ (xmltag, transform, model, script) values
 ('TYPE_T', 'DDL', 'ORACLE', 'kufull');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('TYPE_BODY_T', 'DDL', 'ORACLE', 'kufull');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('FULL_TYPE_T', 'DDL', 'ORACLE', 'kufull');

insert into metaxsl$ (xmltag, transform, model, script) values
 ('PROCEDURE_T', 'DDL', 'ORACLE', 'kuproc');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('FUNCTION_T', 'DDL', 'ORACLE', 'kuproc');

-- full.xsl can process type spec and type body as well as full type
insert into metaxsl$ (xmltag, transform, model, script) values
 ('PACKAGE_T', 'DDL', 'ORACLE', 'kufull');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('PACKAGE_BODY_T', 'DDL', 'ORACLE', 'kufull');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('FULL_PACKAGE_T', 'DDL', 'ORACLE', 'kufull');

insert into metaxsl$ (xmltag, transform, model, script) values
 ('OPERATOR_T', 'DDL', 'ORACLE', 'kuoper');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('INDEXTYPE_T', 'DDL', 'ORACLE', 'kuidxtyp');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('OBJGRANT_T', 'DDL', 'ORACLE', 'kugrant');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('SYSGRANT_T', 'DDL', 'ORACLE', 'kugrant');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('SYNONYM_T', 'DDL', 'ORACLE', 'kusyn');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('TRIGGER_T', 'DDL', 'ORACLE', 'kutrig');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('VIEW_T', 'DDL', 'ORACLE', 'kuview');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('M_VIEW_T', 'DDL', 'ORACLE', 'kumview');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('M_VIEW_LOG_T', 'DDL', 'ORACLE', 'kumviewl');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('OUTLINE_T', 'DDL', 'ORACLE', 'kuoutln');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('DIRECTORY_T', 'DDL', 'ORACLE', 'kudir');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('ROLLBACK_T', 'DDL', 'ORACLE', 'kurollbk');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('DBLINK_T', 'DDL', 'ORACLE', 'kudblink');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('TRLINK_T', 'DDL', 'ORACLE', 'kutrlink');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('TABLESPACE_T', 'DDL', 'ORACLE', 'kutblspc');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('FGA_POLICY_T', 'DDL', 'ORACLE', 'kufga');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('RLS_POLICY_T', 'DDL', 'ORACLE', 'kurls');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('RLS_GROUP_T', 'DDL', 'ORACLE', 'kurls');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('RLS_CONTEXT_T', 'DDL', 'ORACLE', 'kurls');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('CONSTRAINT_T', 'DDL', 'ORACLE', 'kualter');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('REF_CONSTRAINT_T', 'DDL', 'ORACLE', 'kualter');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('LIBRARY_T', 'DDL', 'ORACLE', 'kulib');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('PWDVFC_T', 'DDL', 'ORACLE', 'kuproc');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('USER_T', 'DDL', 'ORACLE', 'kuuser');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('PROFILE_T', 'DDL', 'ORACLE', 'kuprofil');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('SEQUENCE_T', 'DDL', 'ORACLE', 'kuseq');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('ROLE_T', 'DDL', 'ORACLE', 'kurole');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('DEFROLE_T', 'DDL', 'ORACLE', 'kudefrol');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('PROXY_T', 'DDL', 'ORACLE', 'kuproxy');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('ROGRANT_T', 'DDL', 'ORACLE', 'kurolegr');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('TSQUOTA_T', 'DDL', 'ORACLE', 'kutsquo');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('RESOCOST_T', 'DDL', 'ORACLE', 'kurescos');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('CONTEXT_T', 'DDL', 'ORACLE', 'kuctx');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('DIMENSION_T', 'DDL', 'ORACLE', 'kudimen');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('ASSOC_T', 'DDL', 'ORACLE', 'kuassoc');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('COMMENT_T', 'DDL', 'ORACLE', 'kucoment');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('CLUSTER_T', 'DDL', 'ORACLE', 'kuclus');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('AUDIT_T', 'DDL', 'ORACLE', 'kuaudit');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('AUDITOBJ_T', 'DDL', 'ORACLE', 'kuaudobj');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('JAVASRC_T', 'DDL', 'ORACLE', 'kujvsrc');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('QUETAB_T', 'DDL', 'ORACLE', 'kuquetab');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('QUEUES_T', 'DDL', 'ORACLE', 'kuqueues');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('QTRANS_T', 'DDL', 'ORACLE', 'kuqtrans');
insert into metaxsl$ (xmltag, transform, model, script) values
 ('JOB_T', 'DDL', 'ORACLE', 'kujob');

insert into metaxsl$ (xmltag, transform, model, script) values
 ('XMLSCHEMA_T', 'DDL', 'ORACLE', 'kuxmlsch');

Rem placeholder for parse transform

insert into metaxsl$ (xmltag, transform, model, script) values
 ('*', 'PARSE', 'ORACLE', 'parse');

Rem -- Special hack to store directory where xsl files are located
insert into metaxsl$ (xmltag, transform, model, script) values
 ('XSL_BASE_DIR', 'TEMP', 'ORACLE', '/vobs/rdbms/xml/xsl');

-- Populate metaxslparam$: This table defines all the valid transformation
-- parameters for each dictionary object type and transform the user can set
-- through the SET_TRANSFORM_PARAM method. Eg, a table's creation DDL can be
-- generated with or without storage clause, OID, constraints, constraints as
-- ALTER TABLE cmns or column constraints, etc. The stylesheets themselves
-- define these parameters and the XSL engine sets them.

insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','*','PRETTY','1');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','*','SQLTERMINATOR','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','*','EMIT_SCHEMA','1');

insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','TABLE','SEGMENT_ATTRIBUTES','1');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','TABLE','STORAGE','1');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','TABLE','TABLESPACE','1');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','TABLE','CONSTRAINTS','1');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','TABLE','REF_CONSTRAINTS','1');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','TABLE','CONSTRAINTS_AS_ALTER','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','TABLE','OID','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','TABLE','SIZE_BYTE_KEYWORD','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','TABLE','EXPORT','0');
--
-- Backport bug 3575797 - Transportable classic
-- Note:  This stream (9.2....) does not support TRANSPORTABLE, the bug 
-- fix addressed an issue that was a problem with both TRANSPORTABLE and 
-- table mode export (classic).
-- In case TRANSPORTABLE mode is backported, this fix needed to be present.
-- See kutable.xsl for references
--
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','TABLE','TRANSPORTABLE_CLASSIC','0');

-- for logical standby
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','TABLE','INSERT','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','TABLE','DBLINK','0');

insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','MATERIALIZED_VIEW','EXPORT','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','MATERIALIZED_VIEW','START_WITH','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','MATERIALIZED_VIEW_LOG','EXPORT','0');

insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','INDEX','SEGMENT_ATTRIBUTES','1');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','INDEX','STORAGE','1');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','INDEX','TABLESPACE','1');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','INDEX','EXPORT','0');

insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','TYPE','SPECIFICATION','1');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','TYPE','BODY','1');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','TYPE','OID','0');
rem insert into metaxslparam$ (model, transform, type, param, default_val) values
rem ('ORACLE','DDL','TYPE','TIMESTAMP','0');

insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','PACKAGE','SPECIFICATION','1');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','PACKAGE','BODY','1');

insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','VIEW','FORCE','1');

insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','OUTLINE','INSERT','0');

insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','ROLE','REVOKE_FROM','""');

insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','AQ_QUEUE_TABLE','EXPORT','0');

insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','AQ_QUEUE','EXPORT','0');

insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','AQ_TRANSFORM','EXPORT','0');

insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','JOB','EXPORT','0');

insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','XMLSCHEMA','GEN_TYPES','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','XMLSCHEMA','GEN_BEAN','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','XMLSCHEMA','GEN_TABLES','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','XMLSCHEMA','FORCE','0');

Rem Dummy entries for object types without transform params.
Rem All object types need at least one DDL transform param.

insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','OPERATOR','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','INDEXTYPE','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','TRIGGER','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','PROCEDURE','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','FUNCTION','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','SYSTEM_GRANT','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','OBJECT_GRANT','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','DIRECTORY','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','TABLESPACE','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','ROLLBACK_SEGMENT','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','DB_LINK','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','TRUSTED_DB_LINK','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','FGA_POLICY','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','RLS_POLICY','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','RLS_GROUP','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','RLS_CONTEXT','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','CONSTRAINT','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','REF_CONSTRAINT','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','LIBRARY','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','USER','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','PROFILE','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','DEFAULT_ROLE','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','PROXY','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','RESOURCE_COST','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','ROLE_GRANT','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','SEQUENCE','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','TABLESPACE_QUOTA','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','CONTEXT','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','DIMENSION','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','ASSOCIATION','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','COMMENT','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','PASSWORD_VERIFY_FUNCTION','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','CLUSTER','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','AUDIT','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','AUDIT_OBJ','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','JAVA_SOURCE','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','AQ_QUEUE_TABLE','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','AQ_QUEUE','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','AQ_TRANSFORM','DUMMY','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','DDL','JOB','DUMMY','0');


Rem parse items

insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','PARSE','*','PRS_DDL','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','PARSE','*','PRS_DELIM','"\{]`"');

insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','PARSE','*','PRS_VERB','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','PARSE','*','PRS_OBJECT_TYPE','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','PARSE','*','PRS_SCHEMA','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','PARSE','*','PRS_NAME','0');

insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','PARSE','TABLE','PRS_TABLESPACE','0');
insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','PARSE','INDEX','PRS_TABLESPACE','0');

insert into metaxslparam$ (model, transform, type, param, default_val) values
 ('ORACLE','PARSE','TRIGGER','PRS_ENABLE','0');

-- Populate metastylesheet: This table actually stores the XSL stylesheets,
-- one row per. The CLOB column is loaded later during install / upgrade from
-- files via routine dbms_metadata_util.load_stylesheets, but it must be loaded
-- here with some dummy text so that the LOB locator won't be NULL.


insert into metastylesheet (name, model, stylesheet) values
	('kubody','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kucolumn','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kucommon','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuconstr','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kufull','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kugrant','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuindex','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuidxtyp','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuidxwrk','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuiot','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kulob','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuobjtbl','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuoper','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuoutln','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuprtobj','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuproc','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kusource','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kustorag','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kusyn','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kutable','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kutrig','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kutypspc','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuview','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kumview','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kudir','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kurollbk','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kudblink','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kutrlink','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kutblspc','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kufga','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kurls','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kumviewl','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kualter','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kulib','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kudefrol','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuproxy','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuprofil','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kurescos','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kurole','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kurolegr','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kutsquo','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuuser','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuseq','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuctx','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kudimen','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuassoc','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kucoment','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuclus','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuaudit','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuaudobj','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kujvsrc','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuquetab','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuqueues','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kuqtrans','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('kujob','ORACLE','not initialized');

insert into metastylesheet (name, model, stylesheet) values
	('kuxmlsch','ORACLE','not initialized');


-- replication stylesheets
insert into metastylesheet (name, model, stylesheet) values
	('knfcf1','ORACLE','not initialized');
insert into metastylesheet (name, model, stylesheet) values
	('knfcf2','ORACLE','not initialized');

commit;
