Rem
Rem $Header: dbmslmd.sql 04-may-2005.09:02:15 mtao Exp $
Rem
Rem dbmslogmnrd.sql
Rem
Rem Copyright (c) 1998, 2005, Oracle. All rights reserved.  
Rem
Rem    NAME
Rem      dbmslogmnrd.sql - <one-line expansion of the name>
Rem
Rem    DESCRIPTION
Rem	 This file contains system meta data creation statements
Rem      required by the Logminer component (dbms_logmnr & dbms_logmnr_d).
Rem      It also contains the LogMnr package header to create the
Rem      LogMnr dictionary file.
Rem
Rem    NOTES
Rem      Also see prvtlmd.sql.  dbmslmd.sql was split into this, user visible
Rem      portion and the non user visible portion which is now in prvtlmd.sql.
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    mtao        05/05/05 - reopen 3953131, add partitioning to i3gtlo
Rem    mtao        02/15/05 - bkpt 3953131, add indexes to c_gtlo, ckpt table
Rem    mtao        02/14/05 - backport 4137433, add index to logmnr_log$
Rem    dvoss       09/15/04 - bug 3878767 
Rem    abrown      07/06/04 - xbranchmerge to 'st_rdbms_9.2.0.1' 
Rem    abrown      07/01/04 - xbranchmerge to 'st_rdbms_9.2.0.1' 
Rem    ajadams     09/27/04 - 
Rem    abrown      08/31/04 - for bug 3785754, tweak logmnrc_gtcs on upgrade 
Rem    abrown      08/10/04 - handle transient duplicates in DDL apply 
Rem    jkundu      08/12/02 - upgrade/downgrade issue for restart_ckpt_txinfo
Rem    dvoss       07/30/02 - bug 2446679 - delete type 10 objects from logmnrc_gtlo
Rem    jkundu      08/05/02 - change constraint on txinfo$ table
Rem    abrown      05/21/02 - Do not recreate logminer metadata
Rem    ajadams     02/12/02 - fix LOGMNR_INTERESTING_COLS query
Rem    dvoss       02/04/02 - Add ORDER to CREATE SEQUENCE calls
Rem    dvoss       01/21/02 - spare cols missing from non-partition case
Rem    dvoss       01/10/02 - Add index history support
Rem    abrown      01/08/02 - for bug 2172631, always reload LOGMNR_INTERESTING_COLS
Rem    abrown      11/26/01 - trap errors with noexp
Rem    abrown      11/12/01 - bug 2104829 - eliminate logmnr from export set
Rem    celsbern    10/21/01 - merge LOG to MAIN
Rem    dvoss       10/15/01 - Add spare cols to gtlo and gtcs
Rem    dvoss       09/25/01 - Rename interval columns in persitent cache
Rem    abrown      09/17/01 - Add record global name option to krvxcs
Rem    abrown      09/17/01 - Add logmnr_buildlog table, again
Rem    smangala    08/28/01 - add valid column to logmnr_restart_ckpt table.
Rem    mtao        08/16/01 - LOCK_STATE flag added to LOGMNRG_DICTIONARY$.
Rem    abrown      08/15/01 - metadata for LOBs, IOTs and Replication
Rem    abrown      08/22/01 - add threadid to logmnr_dictstate
Rem    dvoss       08/01/01 - GTLO and GTCS must use local indexes for pk
Rem    dvoss       07/31/01 - Global name length can not exceed 128
Rem    dvoss       07/30/01 - Create persistant history/cache tables
Rem    qiwang      07/24/01 - add restart checkpoint tables
Rem    rguzman     08/09/01 - Add columns to ind$.
Rem    qiwang      05/29/01 - add age spill table
Rem    gkulkarn    04/17/01 - create sys.LOGMNR_INTERESTING_COLS if necessary
Rem    fayang      03/20/01 - add 3 columns in log$ table
Rem    abrown      03/12/01 - Handle partition option not present
Rem    abrown      01/08/01 - Split out private objects
Rem    abrown      12/12/00 - fix ff load to tables
Rem    abrown      12/11/00 - session_id is NUMBER, not binary_integer
Rem    abrown      12/05/00 - gtlo3, lvlcnt_out incorrectly set on missing obj
Rem    abrown      11/14/00 - fix buffer overflow error seen with SQL
Rem    abrown      11/16/00 - Improve dict load/delete performance
Rem    ajadams     11/03/00 - do not write blob columns to flatfile
Rem    abrown      10/24/00 - ls_trunc_gather_tables missing seed table
Rem    ajadams     10/16/00 - fix FF error handling
Rem    ajadams     09/26/00 - comment my previous fix
Rem    abrown      09/12/00 - bug 1399866: move long requests to DB
Rem    jkundu      08/23/00 - always create temporary tables
Rem    ajadams     09/07/00 -
Rem    ajadams     08/14/00 - support for loading FF to Logmnr tables
Rem    ajadams     08/08/00 - add encryption tables to FF
Rem    ajadams     07/25/00 - Flatfile - add 2 tables to support IOT and ADT
Rem    qiwang      07/24/00 - add more columns to logmnr_spill$
Rem    abrown      07/13/00 - additional index on logmnr_obj
Rem    lkaplan     06/25/00 - C style comments fail in sqlplus
Rem    abrown      06/19/00 - add more dictionary tables
Rem    abrown      06/14/00 - Add ADT related tables, etc.
Rem    ajadams     06/14/00 - fix file pathname when returning error
Rem    jkundu      06/21/00 -
Rem    aime        06/08/00 - disable ddl trigger creation since it is broken f
Rem    qiwang      06/07/00 - modify logmnr_spill$ schema
Rem    jkundu      05/31/00 - must not create table unless >= 8.2.0.0
Rem    jkundu      05/24/00 - creating logminer related tables
Rem    abrown      05/04/00 - Must have Logminer Dictionary tables created to l
Rem    jkundu      04/06/00 - adding marker and trigger code
Rem    abrown      04/03/00 - Dump dictionary to log stream
Rem    jkundu      06/28/99 - bug 798473: error message on invalid filename
Rem    gkulkarn    09/30/98 - Upcase the create dictionary table
Rem    doshaugh    06/17/98 - Improve exception handling for invalid file name
Rem    mcusson     05/12/98 - Add grant execute_catalog_role.
Rem    doshaugh    04/14/98 - Created
Rem
Rem
Rem
Rem  PUBLIC PROCEDURES
Rem
Rem     BUILD (FileName, FileLocation, Options)
Rem
Rem  PUBLIC CONSTANTS
Rem
Rem     STORE_IN_FLAT_FILE
Rem
Rem     STORE_IN_REDO_LOGS
Rem
Rem
Rem
-- -------------------------------------------------------------
Rem
Rem    NAME
Rem      dbmslogmnrdict.sql - Creates the various Logminer dictionary tables.
Rem
Rem    DESCRIPTION
Rem      The following contains the definitions for all Logminer
Rem      Dictionary tables.  Originally these SQL statements existed
Rem      in a stand-alone file.  Because the build() routine now
Rem      requires that these tables exist before it can be loaded
Rem      we create the tables here.  For now, if a table of the same
Rem      name as a dictionary table exists, it will be delete.  In the
Rem      this may not be desireable.
Rem
Rem    INPUTS
Rem
Rem    NOTES
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem
Rem
Rem  PUBLIC PROCEDURES
Rem
Rem
Rem  PUBLIC Views and Tables
Rem        
Rem
Rem  PRIVATE FUNCTIONS
Rem
Rem
Rem  NOTES
Rem    There are 3 flavors of most tables:
Rem    1) The permananent gather tables with names like SYS.LOGMNRG_*.
Rem       These tables are used exclusively on a source system for
Rem       gathering a source system catalog into the log stream.  They
Rem       are always truncated after use.
Rem    2) The permanent Logminer Dictionary with name like
Rem       SYSTEM.LOGMNR_*.  These tables store a persistant
Rem       set of meta data used by Logminer.
Rem    3) The temporary tables with names like SYS.LOGMNRT_*.  These
Rem       tables are used as part of the processes of mining a gathered
Rem       system catalog out of the log stream.
Rem
Rem     The "master" tables are the gather tables.  When it is necessary
Rem     to add additional columns to the dictionary typically they should
Rem     be added just to the gather table(s).  Both the Logminer Dictionary
Rem     and the temporary tables are based on the gather tables.  That is to
Rem     say, they are created using CREATE TABLE foo AS to ensure that all
Rem     flavors of a given table stay in sync.
Rem

Rem
Rem  The Logminer Gather definitions.
Rem

CREATE OR REPLACE PROCEDURE logmnr_create_replace_metadata(
   PartOn           IN BOOLEAN  DEFAULT false,
   LoggingOn        IN BOOLEAN  DEFAULT true,
   DictTablespace   IN VARCHAR2 DEFAULT 'SYSTEM',
   SpillTablespace  IN VARCHAR2 DEFAULT 'SYSTEM',
   InitialCreation  IN BOOLEAN  DEFAULT FALSE)

/***************************************************************************
 *
 *    NAME
 *      logmnr_create_replace_metadata
 *
 *    DESCRIPTION
 *      logmnr_create_replace_metadata causes most metadata required by
 *      Logminer to be defined.  Any existing meta data is deleted.  This
 *      procedure is intended for INTERNAL use only.
 *
 *    PARAMETERS
 *      PartOn           FALSE indicates that the Partition option is not
 *                             available and that no tables are created
 *                             as partitioned
 *                       TRUE  indicates that the Partition option is
 *                             available and that some tables are created
 *                             as partitioned and their indecies are created
 *                             LOCAL.
 *      LoggingOn        Control default logging attribute for Logminer
 *                       dictionary Indecies.
 *                       FALSE indicates that writes to indecies will not
 *                             be logged.
 *                       TRUE indicates that writes to indecies will be logged.
 *      DictTablespace   Name of tablespace in which all Logminer dictionary
 *                       tables are stored.
 *      SpillTablespace  Name of tablespace in which spill and log record
 *                       tables are stored.
 *      InitialCreation  Only TRUE when invoked from unnamed block in
 *                       dbmslmd.sql.  When FALSE Gather tables are not
 *                       created.
 *
 *    TRANSACTION
 *
 *    EMBEDDED PROCEDURES
 *       logmnrRunSQL
 *       logmnrCreateAGatherTable
 *       logmnrCreateATempTable
 *       logmnrCreateADicTable
 *       logmnrCreateADicIndex
 *       logmnrCreateGatherTables
 *       logmnrCreateTempTables
 *       logmnrCreateDictTables
 *       logmnrCreateDictIndecies
 *       logmnrCreateSpillTables
 *
 *    ERRORS
 *       Error conditions are raised or ignored dependent on ignoreErrors
 *       input parameter.
 */


IS

  /* global variables, if any, go here */
  DumpDebugMsgs    CONSTANT BOOLEAN        := FALSE;
  loc_DictTablespace   varchar2(32);
  loc_SpillTablespace  varchar2(32);
  tscnt                number;

  /* exception declarations and pragma, if any, go here */
  invalid_dict_ts exception;
  PRAGMA EXCEPTION_INIT(invalid_dict_ts, -1351); 
  invalid_spill_ts exception;
  PRAGMA EXCEPTION_INIT(invalid_spill_ts, -1352); 

  /* Type declarations, if any, go here */

  /* Local functions and procedures go here */

  /***************************************************************************
   *
   *    NAME
   *      logmnrRunSQL
   *
   *    DESCRIPTION
   *       Dynamically execute a SQL command.
   *
   *    PARAMETERS
   *       command       IN: The SQL command to be executed.
   *       ignoreErrors  IN: FALSE (defualt) causes errors to be raised
   *                         TRUE causes errors to be ignored
   *    TRANSACTION
   *
   *    ERRORS
   *       Error conditions are raised or ignored dependent on ignoreErrors
   *       input parameter.
   */
  PROCEDURE logmnrRunSQL( command   IN VARCHAR2,
                          ignoreErrors IN BOOLEAN DEFAULT FALSE )
  IS
    cur       NUMBER;
    rowcnt    INTEGER;

  BEGIN
    /* execute the command  */
    IF DumpDebugMsgs THEN
      DBMS_OUTPUT.PUT_LINE('Running ' || substr(command, 1, 245));
    END IF;

    cur := DBMS_SQL.OPEN_CURSOR;
    BEGIN
      DBMS_SQL.PARSE( cur, command, DBMS_SQL.V7 );
      rowcnt := DBMS_SQL.EXECUTE(cur);
    EXCEPTION
    WHEN OTHERS THEN
      /* Putlines are for debugging only  */
      IF DumpDebugMsgs THEN
        DBMS_OUTPUT.PUT_LINE(substr(command, 1, 254));
        DBMS_OUTPUT.PUT_LINE ('ERROR ' || SQLCODE || '  ' || SQLERRM);
      END IF;

      IF NOT ignoreErrors THEN
        BEGIN
          /* silently try to close the cursor */
          DBMS_SQL.CLOSE_CURSOR( cur );
        EXCEPTION
        WHEN OTHERS THEN
          NULL;
        END;

        RAISE;
      END IF;
    END;

    BEGIN
      /* silently try to close the cursor */
      DBMS_SQL.CLOSE_CURSOR( cur );
    EXCEPTION
    WHEN OTHERS THEN
      NULL;
    END;

  END logmnrRunSQL;


  /***************************************************************************
   *
   *    NAME
   *      logmnrCreateAGatherTable
   *
   *    DESCRIPTION
   *      
   *
   *    PARAMETERS
   *       tableSpaceName - Tablespace for gather tables
   *       tableName      - Name of the gather table
   *       tableColumns   - SQL string containing column definitions
   *       
   *    TRANSACTION
   *
   *    ERRORS
   *       
   *       
   */

  PROCEDURE logmnrCreateAGatherTable(tableSpaceName   IN VARCHAR2,
                                     tableName        IN VARCHAR2,
                                     tableColumns     IN VARCHAR2)
  IS
  BEGIN

    logmnrRunSQL('TRUNCATE TABLE SYS.' || tableName, TRUE);
    logmnrRunSQL('DROP TABLE SYS.' || tableName, TRUE);
    logmnrRunSQL('CREATE TABLE SYS.' || tableName || ' ( ' ||
            tableColumns || ' ) TABLESPACE ' ||
            tableSpaceName || ' LOGGING');

  END logmnrCreateAGatherTable;

  /***************************************************************************
   *
   *    NAME
   *      logmnrCreateATempTable
   *
   *    DESCRIPTION
   *      Create/Replace a temp table in SYS based on some other table
   *      from SYS.
   *
   *    PARAMETERS
   *       newTableName - name of temp table to be created or replaced
   *       basedOnTableName - name of table on which to base it
   *
   *    TRANSACTION
   *
   *    NOTES:
   *      It appears that for temp tables one can not specify a tablespace.
   *
   *    ERRORS
   *       A failure on table create is rasied.  Any failure to remove
   *       the table (which might not exist in the first place) is ignored.
   *       
   */

  PROCEDURE logmnrCreateATempTable(
                                   newTableName     IN VARCHAR2,
				   basedOnTableName IN VARCHAR2)
  IS
  BEGIN

    logmnrRunSQL('TRUNCATE TABLE SYS.' || newTableName, TRUE);
    logmnrRunSQL('DROP TABLE SYS.' || newTableName, TRUE);
    logmnrRunSQL('CREATE GLOBAL TEMPORARY TABLE SYS.' || newTableName ||
           ' ON COMMIT PRESERVE ROWS AS SELECT * FROM SYSTEM.' ||
           basedOnTableName || ' WHERE 1 = 2', FALSE);

  END logmnrCreateATempTable;

  /***************************************************************************
   *
   *    NAME
   *      logmnrCreateADicTable
   *
   *    DESCRIPTION
   *      Create/Replace one table that stores the Logminer Dictionary.
   *      Most tables used to store the Logminer Dictionary are created
   *      based on the corresponding Gather table with several additional
   *      columns.
   *
   *    PARAMETERS
   *       mustPartition - True to create a partitioned table, False to
   *                       create a normal table
   *       tableSpaceName - The tablespace in which this table is to reside
   *       newTableName - Name of the table to be created
   *       basedOnTableName - Name of the table on which this table's
   *                       definition is based.
   *       indexColumns - comma separated string which lists the column
   *                      name or names which compise the primary key.
   *                      A Null value indicates that no primary key is
   *                      to be created.
   *                      Primary keys are required for all tables to which
   *                      DDL events may be applied.  However, overhead of
   *                      a primary key can be excessive so all are created
   *                      disabled.
   *       
   *    TRANSACTION
   *
   *    ERRORS
   *       All errors are signalled.
   *       
   */

  PROCEDURE logmnrCreateADicTable( mustPartition    IN BOOLEAN,
                                   tableSpaceName   IN VARCHAR2,
                                   newTableName     IN VARCHAR2,
				   basedOnTableName IN VARCHAR2,
                                   indexColumns     IN VARCHAR2)
  IS
  BEGIN
    /* first remove anything that is there already  */
    logmnrRunSQL('TRUNCATE TABLE SYSTEM.' || newTableName, TRUE);
    logmnrRunSQL('DROP TABLE SYSTEM.' || newTableName, TRUE);

    IF mustPartition THEN
      logmnrRunSql('CREATE TABLE SYSTEM.' || newTableName ||
                   ' PARTITION BY RANGE(logmnr_uid)' ||
                   '  ( PARTITION p_lessthan100 VALUES LESS THAN (100)) ' ||
                   ' TABLESPACE ' || tableSpaceName || ' LOGGING ' ||
                   '    AS SELECT * FROM SYS.' || basedOnTableName ||
                   ', SYSTEM.LOGMNR_HEADER1$, SYSTEM.LOGMNR_HEADER2$' ||
                   '    WHERE 1 = 2');
    ELSE
      logmnrRunSql ('CREATE TABLE SYSTEM.' || newTableName ||
                   ' TABLESPACE ' || tableSpaceName || ' LOGGING' ||
                   '    AS SELECT * FROM SYS.' || basedOnTableName ||
                   ', SYSTEM.LOGMNR_HEADER1$, SYSTEM.LOGMNR_HEADER2$' ||
                   '    WHERE 1 = 2');
    END IF;

    IF NOT indexColumns IS NULL THEN
      logmnrRunSql ('alter table SYSTEM.' ||  newTableName ||
                    ' add primary key ( ' || indexColumns ||
                    ' ) disable');
    END IF;

  END logmnrCreateADicTable;

  /***************************************************************************
   *
   *    NAME
   *      logmnrCreateADicIndex
   *
   *    DESCRIPTION
   *      Given an existing table, create an index for that table.
   *      The index is created in the named tablespace.
   *      When partitioning is enabled, the table to be indexed must
   *      be partitioned and the index is created as LOCAL.
   *      For performance reasons, all indecies are created as not unique.
   *
   *    PARAMETERS
   *       mustPartition - True to create a partitioned table, False to
   *                       create a normal table
   *       tableSpaceName - The tablespace in which this table is to reside
   *       newIndexName - Name of the table to be created
   *       tableName - Name of the table which is to be indexed.
   *       indexColumns - comma separated string which lists the column
   *                      name or names which compise the index.
   *       logging - FALSE indicats a NOLOGGING index.
   *                 TRUE indicates a LOGGING enabled index
   *       
   *    TRANSACTION
   *
   *    ERRORS
   *       All errors are signalled.
   *       
   */

  PROCEDURE logmnrCreateADicIndex( mustPartition    IN BOOLEAN,
                                   tableSpaceName   IN VARCHAR2,
                                   logging          IN BOOLEAN,
                                   newIndexName     IN VARCHAR2,
				   tableName        IN VARCHAR2,
                                   indexColumns     IN VARCHAR2)
  IS
    loggingStr  VARCHAR2(64) := '';
    localStr    VARCHAR2(64) := '';

  BEGIN
    IF mustPartition THEN
      localStr := ' LOCAL ';
    END IF;

    IF logging THEN
      loggingStr := ' LOGGING ';
    ELSE
      loggingStr := ' NOLOGGING ';
    END IF;

    logmnrRunSql('CREATE INDEX SYSTEM.' || newIndexName || ' ON ' ||
                   ' SYSTEM.' || tableName || ' ( ' ||
                   indexColumns || ' ) ' || ' TABLESPACE ' ||
                   tableSpaceName || localStr || loggingStr);

  END logmnrCreateADicIndex;

  /***************************************************************************
   *
   *    NAME
   *      logmnrCreateGatherTables
   *
   *    DESCRIPTION
   *      Creates all gather tables from definition given here.
   *
   *    PARAMETERS
   *       tableSpaceName - Tablespace for gather tables
   *       
   *    TRANSACTION
   *
   *    ERRORS
   *       All errors are signalled.
   *       
   */

  PROCEDURE logmnrCreateGatherTables( tableSpaceName   IN VARCHAR2)
  IS
  BEGIN

    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_SEED$',
     'SEED_VERSION NUMBER(22),
      GATHER_VERSION NUMBER(22),
      SCHEMANAME VARCHAR2(30),
      OBJ# NUMBER,
      OBJV# NUMBER(22),
      TABLE_NAME VARCHAR2(30),
      COL_NAME VARCHAR2(30),
      COL# NUMBER,
      INTCOL# NUMBER,
      SEGCOL# NUMBER,
      TYPE# NUMBER,
      LENGTH NUMBER,
      PRECISION# NUMBER,
      SCALE NUMBER,
      NULL$ NUMBER
     ');

    /*
     *       DB_DICT_SCN NUMBER(22),    scn at time of db_thread_map
     *       DB_THREAD_MAP RAW(8),
     *       DB_TXN_SCNBAS NUMBER(22),  scn when gather Txn started
     *       DB_TXN_SCNWRP NUMBER(22),
     */
    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_DICTIONARY$',
     'DB_NAME VARCHAR2(9),
      DB_ID NUMBER(20),
      DB_CREATED VARCHAR2(20),
      DB_DICT_CREATED VARCHAR2(20),
      DB_DICT_SCN NUMBER(22),
      DB_THREAD_MAP RAW(8),
      DB_TXN_SCNBAS NUMBER(22),
      DB_TXN_SCNWRP NUMBER(22),
      DB_RESETLOGS_CHANGE# NUMBER(22),
      DB_RESETLOGS_TIME VARCHAR2(20),
      DB_VERSION_TIME VARCHAR2(20),
      DB_REDO_TYPE_ID VARCHAR2(8),
      DB_REDO_RELEASE VARCHAR2(60),
      DB_CHARACTER_SET VARCHAR2(30),
      DB_VERSION VARCHAR2(64),
      DB_STATUS VARCHAR2(64),
      DB_GLOBAL_NAME VARCHAR(128),
      DB_DICT_MAXOBJECTS NUMBER(22),
      DB_DICT_OBJECTCOUNT NUMBER(22)');


    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_OBJ$',
     'OBJV# NUMBER(22),
      OWNER# NUMBER(22),
      NAME VARCHAR2(30),
      NAMESPACE NUMBER(22),
      SUBNAME VARCHAR2(30),
      TYPE# NUMBER(22),
      OID$  RAW(16),
      REMOTEOWNER VARCHAR2(30),
      LINKNAME VARCHAR(128),
      OBJ# NUMBER(22)');

    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_TAB$',
     'TS# NUMBER(22),
      COLS NUMBER(22),
      PROPERTY NUMBER(22),
      INTCOLS NUMBER(22),
      OBJ# NUMBER(22)');

    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_COL$',
     'COL# NUMBER(22),
      SEGCOL# NUMBER(22),
      NAME VARCHAR2(30),
      TYPE# NUMBER(22),
      LENGTH NUMBER(22),
      PRECISION# NUMBER(22),
      SCALE NUMBER(22),
      NULL$ NUMBER(22),
      INTCOL# NUMBER(22),
      PROPERTY NUMBER(22),
      CHARSETID NUMBER(22),
      CHARSETFORM NUMBER(22),
      SPARE1 NUMBER(22),
      SPARE2 NUMBER(22),
      OBJ# NUMBER(22)');

    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_ATTRCOL$',
     'INTCOL#   number,
      NAME      varchar2(4000),
      OBJ#      number');

    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_TS$',
     'TS# NUMBER(22),
      NAME VARCHAR2(30),
      OWNER# NUMBER(22),
      BLOCKSIZE NUMBER(22)');

    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_IND$',
     'BO#     NUMBER(22),
     COLS     NUMBER(22),
     TYPE#    NUMBER(22),
     FLAGS    NUMBER,
     PROPERTY NUMBER,
     OBJ#     NUMBER(22)');

    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_USER$',
     'USER# NUMBER(22),
      NAME VARCHAR2(30)');

    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_TABPART$',
     'OBJ# NUMBER(22),
      BO# NUMBER(22),
      TS# NUMBER(22)');

    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_TABSUBPART$',
     'OBJ# NUMBER(22),
      DATAOBJ# NUMBER(22),
      POBJ# NUMBER(22),
      SUBPART# NUMBER(22),
      TS# NUMBER(22)');

    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_TABCOMPART$',
     'OBJ# NUMBER(22),
      BO# NUMBER(22),
      PART# NUMBER(22)');

    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_TYPE$',
     'VERSION#   NUMBER(22),
      TVOID      RAW(16),
      PROPERTIES NUMBER(22),
      ATTRIBUTES NUMBER(22),
      TOID       RAW(16)');

    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_COLTYPE$',
     'COL#     NUMBER(22),
      INTCOL#  NUMBER(22),
      TOID     RAW(16),
      VERSION# NUMBER(22),
      INTCOLS  NUMBER(22),
      TYPIDCOL# NUMBER(22),
      OBJ#     NUMBER(22)');

    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_ATTRIBUTE$',
     'VERSION#   NUMBER(22),
      NAME       VARCHAR2(30),
      ATTRIBUTE# NUMBER(22),
      ATTR_TOID  RAW(16),
      ATTR_VERSION# NUMBER(22),
      PROPERTIES NUMBER(22),
      TOID       RAW(16)');

    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_LOB$',
     'OBJ#          NUMBER,
      INTCOL#       NUMBER,
      COL#          NUMBER,
      LOBJ#         NUMBER,
      CHUNK         NUMBER');

    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_CDEF$',
     'CON#          NUMBER,
      OBJ#          NUMBER,
      COLS          NUMBER,
      TYPE#         NUMBER,
      ROBJ#         NUMBER, 
      RCON#         NUMBER, 
      ENABLED       NUMBER,
      DEFER         NUMBER');

    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_CCOL$',
     'CON#          NUMBER,
      OBJ#          NUMBER,
      COL#          NUMBER,
      POS#          NUMBER,
      INTCOL#       NUMBER');

    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_ICOL$',
     'OBJ#          NUMBER,
      BO#           NUMBER,
      COL#          NUMBER,
      POS#          NUMBER,
      SEGCOL#       NUMBER,
      INTCOL#       NUMBER');

    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_LOBFRAG$',
     'FRAGOBJ#      NUMBER,
      PARENTOBJ#    NUMBER,
      TABFRAGOBJ#   NUMBER,
      INDFRAGOBJ#   NUMBER,
      FRAG#         NUMBER');

    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_INDPART$',
     'OBJ# NUMBER,
      BO#  NUMBER,
      TS#  NUMBER');

    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_INDSUBPART$',
     'OBJ# NUMBER(22),
      DATAOBJ# NUMBER(22),
      POBJ# NUMBER(22),
      SUBPART# NUMBER(22),
      TS# NUMBER(22)');

    logmnrCreateAGatherTable( tableSpaceName, 'LOGMNRG_INDCOMPART$',
     'OBJ#     NUMBER,
      DATAOBJ# NUMBER,
      BO#      NUMBER,
      PART#    NUMBER');

  END;

  /***************************************************************************
   *
   *    NAME
   *      logmnrCreateTempTables
   *
   *    DESCRIPTION
   *      Create all temp tables needed by Logminer.
   *
   *    PARAMETERS
   *       
   *       
   *    TRANSACTION
   *
   *    ERRORS
   *       All errors are raised.
   *       
   */

  PROCEDURE logmnrCreateTempTables
  IS
  BEGIN

    /*
     * SEED$ is special.  Because there is no dictionary counterpart
     * its temp table definition comes from the gather definition.
     */
    logmnrRunSQL('TRUNCATE TABLE SYS.LOGMNRT_SEED$', TRUE);
    logmnrRunSQL('DROP TABLE SYS.LOGMNRT_SEED$', TRUE);
    logmnrRunSQL('CREATE GLOBAL TEMPORARY TABLE SYS.LOGMNRT_SEED$
                  ON COMMIT PRESERVE ROWS AS SELECT * FROM SYS.LOGMNRG_SEED$
                  WHERE 1 = 2', FALSE);

    /*
     * SYSTEM.LOGMNRT_MDDL$ is also special.  Its rows persist only for the
     * duration of the transaction.  Added 10-AUG-2004 to address bug 3785754.
     * Note this is in SYSTEM not SYS to avoid issue with cursors on 
     * SYS.LOGMNRT_ objects.
     */
    logmnrRunSQL('TRUNCATE TABLE SYSTEM.LOGMNRT_MDDL$', TRUE);
    logmnrRunSQL('DROP TABLE SYSTEM.LOGMNRT_MDDL$', TRUE);
    logmnrRunSQL('CREATE GLOBAL TEMPORARY TABLE system.logmnrt_mddl$ (
                  source_obj#     NUMBER,
                  source_rowid    ROWID,
                  dest_rowid      ROWID NOT NULL,
                    CONSTRAINT logmnrt_mddl$_pk
                      PRIMARY KEY(source_obj#, source_rowid)
                  ) on commit delete rows', FALSE);

    /*
     * The rest of the temp tables are based on their Logminer Dictionary
     * counterparts.
     */
    logmnrCreateATempTable('LOGMNRT_DICTIONARY$', 'LOGMNR_DICTIONARY$');
    logmnrCreateATempTable('LOGMNRT_OBJ$', 'LOGMNR_OBJ$');
    logmnrCreateATempTable('LOGMNRT_TAB$', 'LOGMNR_TAB$');
    logmnrCreateATempTable('LOGMNRT_COL$', 'LOGMNR_COL$');
    logmnrCreateATempTable('LOGMNRT_ATTRCOL$', 'LOGMNR_ATTRCOL$');
    logmnrCreateATempTable('LOGMNRT_TS$', 'LOGMNR_TS$');
    logmnrCreateATempTable('LOGMNRT_IND$', 'LOGMNR_IND$');
    logmnrCreateATempTable('LOGMNRT_USER$', 'LOGMNR_USER$');
    logmnrCreateATempTable('LOGMNRT_TABPART$', 'LOGMNR_TABPART$');
    logmnrCreateATempTable('LOGMNRT_TABSUBPART$', 'LOGMNR_TABSUBPART$');
    logmnrCreateATempTable('LOGMNRT_TABCOMPART$', 'LOGMNR_TABCOMPART$');
    logmnrCreateATempTable('LOGMNRT_TYPE$', 'LOGMNR_TYPE$');
    logmnrCreateATempTable('LOGMNRT_COLTYPE$', 'LOGMNR_COLTYPE$');
    logmnrCreateATempTable('LOGMNRT_ATTRIBUTE$', 'LOGMNR_ATTRIBUTE$');
    logmnrCreateATempTable('LOGMNRT_LOB$', 'LOGMNR_LOB$');
    logmnrCreateATempTable('LOGMNRT_CDEF$', 'LOGMNR_CDEF$');
    logmnrCreateATempTable('LOGMNRT_CCOL$', 'LOGMNR_CCOL$');
    logmnrCreateATempTable('LOGMNRT_ICOL$', 'LOGMNR_ICOL$');
    logmnrCreateATempTable('LOGMNRT_LOBFRAG$', 'LOGMNR_LOBFRAG$');
    logmnrCreateATempTable('LOGMNRT_INDPART$', 'LOGMNR_INDPART$');
    logmnrCreateATempTable('LOGMNRT_INDSUBPART$', 'LOGMNR_INDSUBPART$');
    logmnrCreateATempTable('LOGMNRT_INDCOMPART$', 'LOGMNR_INDCOMPART$');



  END logmnrCreateTempTables;

  /***************************************************************************
   *
   *    NAME
   *      logmnrCreateDictTables
   *
   *    DESCRIPTION
   *      Creates/Replaces all core Logminer Dictionary tables
   *      
   *
   *    PARAMETERS
   *       
   *       
   *       
   *    TRANSACTION
   *
   *    ERRORS
   *       
   *       
   */

  PROCEDURE logmnrCreateDictTables( mustPartition    IN BOOLEAN,
                                    tableSpaceName   IN VARCHAR2)
  IS
  BEGIN
    /* First recreate the abnormal tables   */
    logmnrRunSQL('TRUNCATE TABLE SYSTEM.LOGMNR_UID$', TRUE);
    logmnrRunSQL('DROP TABLE SYSTEM.LOGMNR_UID$', TRUE);

    logmnrRunSQL('TRUNCATE TABLE SYS.LOGMNR_BUILDLOG', TRUE);
    logmnrRunSQL('DROP TABLE SYS.LOGMNR_BUILDLOG', TRUE);

    logmnrRunSQL('TRUNCATE TABLE SYSTEM.LOGMNR_USER$', TRUE);
    logmnrRunSQL('DROP TABLE SYSTEM.LOGMNR_USER$', TRUE);

    logmnrRunSQL('TRUNCATE TABLE SYSTEM.LOGMNR_OBJ$', TRUE);
    logmnrRunSQL('DROP TABLE SYSTEM.LOGMNR_OBJ$', TRUE);

    logmnrRunSQL('TRUNCATE TABLE SYSTEM.LOGMNR_DICTIONARY$', TRUE);
    logmnrRunSQL('DROP TABLE SYSTEM.LOGMNR_DICTIONARY$', TRUE);

    logmnrRunSQL('TRUNCATE TABLE SYSTEM.LOGMNR_DICTSTATE$', TRUE);
    logmnrRunSQL('DROP TABLE SYSTEM.LOGMNR_DICTSTATE$', TRUE);

    logmnrRunSQL('TRUNCATE TABLE SYSTEM.LOGMNRC_DBNAME_UID_MAP', TRUE);
    logmnrRunSQL('DROP TABLE SYSTEM.LOGMNRC_DBNAME_UID_MAP', TRUE);

    logmnrRunSQL('TRUNCATE TABLE SYSTEM.LOGMNRC_GTLO', TRUE);
    logmnrRunSQL('DROP TABLE SYSTEM.LOGMNRC_GTLO', TRUE);

    logmnrRunSQL('TRUNCATE TABLE SYSTEM.LOGMNRC_GTCS', TRUE);
    logmnrRunSQL('DROP TABLE SYSTEM.LOGMNRC_GTCS', TRUE);

    logmnrRunSQL('TRUNCATE TABLE SYSTEM.LOGMNRC_GSII', TRUE);
    logmnrRunSQL('DROP TABLE SYSTEM.LOGMNRC_GSII', TRUE);


    /*
     * For version 9.0.1 there is a one-one mapping between entries
     * in LOGMNT_UID$ and LOGMNR_SESSION$.  That is, each session
     * has its own dictionary as denoted by a unique LOGMNR_UID.
     * In 9.0.2 Logminer Sessions are able to share dictionaries.
     * That is, there are potentially multiple records in LOGMNR_UID$ with
     * a common LOGMNR_UID, but each with distinct SESSION#.
     */

    logmnrRunSql('CREATE TABLE SYSTEM.LOGMNR_UID$ (
                    LOGMNR_UID              NUMBER(22),
                    SESSION#                NUMBER,
                      PRIMARY KEY (SESSION#))  TABLESPACE ' ||
                      tableSpaceName || ' LOGGING');

    logmnrRunSql('CREATE TABLE SYS.LOGMNR_BUILDLOG (
                    INITIAL_XID VARCHAR2(22),
                    BUILD_DATE VARCHAR2(20),
                    DB_TXN_SCNBAS NUMBER,
                    DB_TXN_SCNWRP NUMBER,
                    CURRENT_BUILD_STATE NUMBER,
                    COMPLETION_STATUS NUMBER,
                       PRIMARY KEY (INITIAL_XID))
                    TABLESPACE ' || 
                      tableSpaceName || ' LOGGING');

    logmnrRunSql('CREATE TABLE SYSTEM.LOGMNRC_DBNAME_UID_MAP(
                      GLOBAL_NAME        VARCHAR2(128) NOT NULL,
                        CONSTRAINT LOGMNRC_DBNAME_UID_MAP_PK  PRIMARY KEY(GLOBAL_NAME),
                      LOGMNR_UID         NUMBER,
                      FLAGS              NUMBER ) TABLESPACE ' ||
                      tableSpaceName || ' LOGGING');
    

    
    IF mustPartition THEN
      /*
       * SYSTEM.LOGMNR_DICTSTATE$
       *      START_SCNBAS NUMBER,   Not valid for translation of
       *      START_SCNWRP NUMBER,   of redo records written before this SCN.
       *      END_SCNBAS NUMBER,     Not valid for translation of redo
       *      END_SCNWRP NUMBER,     records written on or after this SCN.
       *      REDO_THREAD NUMBER,    OPS thread which generated the DDL event
       *                             corresponding to this end scn.  Note
       *                             that redo_thread is only useful in the
       *                             event that multiple threads commit DDL
       *                             events on the same SCN.
       *      RBASQN NUMBER,         Added in 10.1.0.3 to handle DDLs that
       *      RBABLK NUMBER,         commit at the same SCN.
       *      RBABYTE NUMBER)
       */
      logmnrRunSql('CREATE TABLE SYSTEM.LOGMNR_DICTSTATE$ (
                      LOGMNR_UID NUMBER(22),
                      START_SCNBAS NUMBER,
                      START_SCNWRP NUMBER,
                      END_SCNBAS NUMBER,
                      END_SCNWRP NUMBER,
                      REDO_THREAD NUMBER,
                      RBASQN NUMBER,
                      RBABLK NUMBER,
                      RBABYTE NUMBER)
                   PARTITION BY RANGE(logmnr_uid)
                      ( PARTITION p_lessthan100 VALUES LESS THAN (100))
                   TABLESPACE ' || tableSpaceName || ' LOGGING');

      logmnrRunSql('CREATE TABLE SYSTEM.LOGMNR_DICTIONARY$
                     PARTITION BY RANGE(logmnr_uid)
                     ( PARTITION p_lessthan100 VALUES LESS THAN (100))
                     TABLESPACE ' || tableSpaceName || ' LOGGING
                    AS SELECT * FROM SYS.LOGMNRG_DICTIONARY$,
                                     SYSTEM.LOGMNR_HEADER1$
                    WHERE 1 = 2');

      logmnrRunSql('CREATE TABLE SYSTEM.LOGMNR_OBJ$
                      PARTITION BY RANGE(logmnr_uid)
                        ( PARTITION p_lessthan100 VALUES LESS THAN (100))
                      TABLESPACE ' || tableSpaceName || ' LOGGING
                    AS SELECT * FROM SYS.LOGMNRG_OBJ$, SYSTEM.LOGMNR_HEADER1$
                    WHERE 1 = 2');

      logmnrRunSql('CREATE TABLE SYSTEM.LOGMNR_USER$
                      PARTITION BY RANGE(logmnr_uid)
                        ( PARTITION p_lessthan100 VALUES LESS THAN (100))
                      TABLESPACE ' || tableSpaceName || ' LOGGING
                   AS SELECT * FROM SYS.LOGMNRG_USER$, SYSTEM.LOGMNR_HEADER1$
                   WHERE 1 = 2');
    logmnrRunSql('CREATE TABLE SYSTEM.LOGMNRC_GTLO( 
                  LOGMNR_UID         NUMBER NOT NULL, 
                  KEYOBJ#            NUMBER NOT NULL,
                  LVLCNT             NUMBER NOT NULL,  /* level count */
                  BASEOBJ#           NUMBER NOT NULL,  /* base object number */
                  BASEOBJV#          NUMBER NOT NULL,  /* base object version */
                  LVL1OBJ#           NUMBER,  /* level 1 object number */
                  LVL2OBJ#           NUMBER,  /* level 2 object number */
                  LVL0TYPE#          NUMBER NOT NULL,  /* level 0 (base obj) type # */
                  LVL1TYPE#          NUMBER,  /* level 1 type # */
                  LVL2TYPE#          NUMBER,  /* level 2 type # */
                  OWNER#             NUMBER,  /* owner number */
                  OWNERNAME          VARCHAR2(30) NOT NULL,
                  LVL0NAME           VARCHAR2(30) NOT NULL, /* name of level 0 (base obj)  */
                  LVL1NAME           VARCHAR2(30), /* name of level 1 object */
                  LVL2NAME           VARCHAR2(30), /* name of level 2 object */
                  INTCOLS            NUMBER NOT NULL, /* for table object, number of cols  */
                  TS#                NUMBER, /* table space number */
                  TSNAME             VARCHAR2(30), /* table space name   */
                  PROPERTY           NUMBER,
                  /* Replication Dictionary Specific Columns  */
                  START_SCN          NUMBER NOT NULL,/* SCN at which existance begins */
                  DROP_SCN         NUMBER,         /* SCN at which existance end    */
                  XIDUSN             NUMBER,         /* src txn which created this object */
                  XIDSLT             NUMBER,
                  XIDSQN             NUMBER,
                  FLAGS              NUMBER,
                  LOGMNR_SPARE1             NUMBER,
                  LOGMNR_SPARE2             NUMBER,
                  LOGMNR_SPARE3             VARCHAR2(1000),
                  LOGMNR_SPARE4             DATE,
                    CONSTRAINT LOGMNRC_GTLO_PK PRIMARY KEY(LOGMNR_UID, KEYOBJ#, BASEOBJV#) USING INDEX LOCAL
                  /*  CONSTRAINT LOGMNRC_GTCS_SRFK FOREIGN KEY (LOGMNR_UID, BASEOBJ#, BASEOBJV#)
                        REFERENCES SYSTEM.LOGMNRC_GTLO (LOGMNR_UID, KEYOBJ#, BASEOBJV#)
                        ON DELETE CASCADE */ 
                  ) PARTITION BY RANGE(logmnr_uid)
                     ( PARTITION p_lessthan100 VALUES LESS THAN (100))
                    TABLESPACE ' || tableSpaceName || ' LOGGING');
      
     logmnrRunSql('CREATE TABLE SYSTEM.LOGMNRC_GTCS(
                   LOGMNR_UID                NUMBER NOT NULL,
                   OBJ#                      NUMBER NOT NULL, /* table (base) object number  */
                   OBJV#                     NUMBER NOT NULL, /* table object version  */
                   SEGCOL#                   NUMBER NOT NULL, /* segcol# of column */
                   INTCOL#                   NUMBER NOT NULL, /* intcol# of column */
                   COLNAME                   VARCHAR2(30) NOT NULL, /* name of column */
                   TYPE#                     NUMBER NOT NULL, /* column type */
                   LENGTH                    NUMBER, /* data length */
                   PRECISION                 NUMBER, /* data precision */
                   SCALE                     NUMBER, /* data scale */
                   INTERVAL_LEADING_PRECISION  NUMBER, /* Interval Leading Precision, if any */
                   INTERVAL_TRAILING_PRECISION NUMBER, /* Interval trailing precision, if any */
                   PROPERTY                  NUMBER,
                   TOID                      RAW(16),
                   CHARSETID                 NUMBER,
                   CHARSETFORM               NUMBER,
                   TYPENAME                  VARCHAR2(30),
                   FQCOLNAME                 VARCHAR2(4000), /* fully-qualified column name */
                   NUMINTCOLS                NUMBER, /* Number of Int Cols  */
                   NUMATTRS                  NUMBER,
                   ADTORDER                  NUMBER,
                   LOGMNR_SPARE1                    NUMBER,
                   LOGMNR_SPARE2                    NUMBER,
                   LOGMNR_SPARE3                    VARCHAR2(1000),
                   LOGMNR_SPARE4                    DATE,
                     constraint logmnrc_gtcs_pk primary key(logmnr_uid, obj#, objv#,segcol#) using index local
                  ) PARTITION BY RANGE(logmnr_uid)
                     ( PARTITION p_lessthan100 VALUES LESS THAN (100))
                    TABLESPACE ' || tableSpaceName || ' LOGGING');

     logmnrRunSql('CREATE TABLE SYSTEM.LOGMNRC_GSII(
                   LOGMNR_UID                NUMBER NOT NULL,
                   OBJ#                      NUMBER NOT NULL,
                   BO#                       NUMBER NOT NULL,
                   INDTYPE#                  NUMBER NOT NULL,
                   DROP_SCN                  NUMBER,
                   LOGMNR_SPARE1             NUMBER,
                   LOGMNR_SPARE2             NUMBER,
                   LOGMNR_SPARE3             VARCHAR2(1000),
                   LOGMNR_SPARE4             DATE,
                     constraint logmnrc_gsii_pk primary key(logmnr_uid, obj#)
                                 using index local
                  ) PARTITION BY RANGE(logmnr_uid)
                     ( PARTITION p_lessthan100 VALUES LESS THAN (100))
                    TABLESPACE ' || tableSpaceName || ' LOGGING');
      

    ELSE
      logmnrRunSql('CREATE TABLE SYSTEM.LOGMNR_DICTSTATE$ (
                      LOGMNR_UID NUMBER(22),
                      START_SCNBAS NUMBER,
                      START_SCNWRP NUMBER,
                      END_SCNBAS NUMBER,
                      END_SCNWRP NUMBER,
                      REDO_THREAD NUMBER,
                      RBASQN NUMBER,
                      RBABLK NUMBER,
                      RBABYTE NUMBER)
                    TABLESPACE ' || tableSpaceName || ' LOGGING');

      logmnrRunSql('CREATE TABLE SYSTEM.LOGMNR_DICTIONARY$ TABLESPACE ' ||
                      tableSpaceName || ' LOGGING
                    AS SELECT * FROM SYS.LOGMNRG_DICTIONARY$,
                                     SYSTEM.LOGMNR_HEADER1$
                    WHERE 1 = 2');

      logmnrRunSql('CREATE TABLE SYSTEM.LOGMNR_OBJ$ TABLESPACE ' ||
                      tableSpaceName || ' LOGGING
                    AS SELECT * FROM SYS.LOGMNRG_OBJ$, SYSTEM.LOGMNR_HEADER1$
                    WHERE 1 = 2');

      logmnrRunSql('CREATE TABLE SYSTEM.LOGMNR_USER$ TABLESPACE ' ||
                      tableSpaceName || ' LOGGING
                   AS SELECT * FROM SYS.LOGMNRG_USER$, SYSTEM.LOGMNR_HEADER1$
                   WHERE 1 = 2');

    logmnrRunSql('CREATE TABLE SYSTEM.LOGMNRC_GTLO( 
                  LOGMNR_UID         NUMBER NOT NULL, 
                  KEYOBJ#            NUMBER NOT NULL,
                  LVLCNT             NUMBER NOT NULL,  /* level count */
                  BASEOBJ#           NUMBER NOT NULL,  /* base object number */
                  BASEOBJV#          NUMBER NOT NULL,  /* base object version */
                  LVL1OBJ#           NUMBER,  /* level 1 object number */
                  LVL2OBJ#           NUMBER,  /* level 2 object number */
                  LVL0TYPE#          NUMBER NOT NULL,  /* level 0 (base obj) type # */
                  LVL1TYPE#          NUMBER,  /* level 1 type # */
                  LVL2TYPE#          NUMBER,  /* level 2 type # */
                  OWNER#             NUMBER,  /* owner number */
                  OWNERNAME          VARCHAR2(30) NOT NULL,
                  LVL0NAME           VARCHAR2(30) NOT NULL, /* name of level 0 (base obj)  */
                  LVL1NAME           VARCHAR2(30), /* name of level 1 object */
                  LVL2NAME           VARCHAR2(30), /* name of level 2 object */
                  INTCOLS            NUMBER NOT NULL, /* for table object, number of cols  */
                  TS#                NUMBER, /* table space number */
                  TSNAME             VARCHAR2(30), /* table space name   */
                  PROPERTY           NUMBER,
                  /* Replication Dictionary Specific Columns  */
                  START_SCN          NUMBER NOT NULL,/* SCN at which existance begins */
                  DROP_SCN         NUMBER,         /* SCN at which existance end    */
                  XIDUSN             NUMBER,         /* src txn which created this object */
                  XIDSLT             NUMBER,
                  XIDSQN             NUMBER,
                  FLAGS              NUMBER,
                  LOGMNR_SPARE1             NUMBER,
                  LOGMNR_SPARE2             NUMBER,
                  LOGMNR_SPARE3             VARCHAR2(1000),
                  LOGMNR_SPARE4             DATE,
                    CONSTRAINT LOGMNRC_GTLO_PK PRIMARY KEY(LOGMNR_UID, KEYOBJ#, BASEOBJV#)
                  /*  CONSTRAINT LOGMNRC_GTCS_SRFK FOREIGN KEY (LOGMNR_UID, BASEOBJ#, BASEOBJV#)
                        REFERENCES SYSTEM.LOGMNRC_GTLO (LOGMNR_UID, KEYOBJ#, BASEOBJV#)
                        ON DELETE CASCADE */ 
                  ) TABLESPACE ' || tableSpaceName || ' LOGGING');
     logmnrRunSql('CREATE TABLE SYSTEM.LOGMNRC_GTCS(
                   LOGMNR_UID                NUMBER NOT NULL,
                   OBJ#                      NUMBER NOT NULL, /* table (base) object number  */
                   OBJV#                     NUMBER NOT NULL, /* table object version  */
                   SEGCOL#                   NUMBER NOT NULL, /* segcol# of column */
                   INTCOL#                   NUMBER NOT NULL, /* intcol# of column */
                   COLNAME                   VARCHAR2(30) NOT NULL, /* name of column */
                   TYPE#                     NUMBER NOT NULL, /* column type */
                   LENGTH                    NUMBER, /* data length */
                   PRECISION                 NUMBER, /* data precision */
                   SCALE                     NUMBER, /* data scale */
                   INTERVAL_LEADING_PRECISION  NUMBER, /* Interval Leading Precision, if any */
                   INTERVAL_TRAILING_PRECISION NUMBER, /* Interval trailing precision, if any */
                   PROPERTY                  NUMBER,
                   TOID                      RAW(16),
                   CHARSETID                 NUMBER,
                   CHARSETFORM               NUMBER,
                   TYPENAME                  VARCHAR2(30),
                   FQCOLNAME                 VARCHAR2(4000), /* fully-qualified column name */
                   NUMINTCOLS                NUMBER, /* Number of Int Cols  */
                   NUMATTRS                  NUMBER,
                   ADTORDER                  NUMBER,
                   LOGMNR_SPARE1                    NUMBER,
                   LOGMNR_SPARE2                    NUMBER,
                   LOGMNR_SPARE3                    VARCHAR2(1000),
                   LOGMNR_SPARE4                    DATE,
                     constraint logmnrc_gtcs_pk primary key(logmnr_uid, obj#, objv#,segcol#)
                  ) TABLESPACE ' || tableSpaceName || ' LOGGING');

     logmnrRunSql('CREATE TABLE SYSTEM.LOGMNRC_GSII(
                   LOGMNR_UID                NUMBER NOT NULL,
                   OBJ#                      NUMBER NOT NULL,
                   BO#                       NUMBER NOT NULL,
                   INDTYPE#                  NUMBER NOT NULL,
                   DROP_SCN                  NUMBER,
                   LOGMNR_SPARE1             NUMBER,
                   LOGMNR_SPARE2             NUMBER,
                   LOGMNR_SPARE3             VARCHAR2(1000),
                   LOGMNR_SPARE4             DATE,
                     constraint logmnrc_gsii_pk primary key(logmnr_uid, obj#)
                  ) TABLESPACE ' || tableSpaceName || ' LOGGING');

    END IF;

    logmnrRunSql('alter table SYSTEM.LOGMNR_DICTSTATE$
                  add primary key (LOGMNR_UID) disable');

    logmnrRunSql('alter table SYSTEM.LOGMNR_DICTIONARY$
                  add primary key (LOGMNR_UID) disable');

    logmnrRunSql('ALTER TABLE SYSTEM.LOGMNR_OBJ$ ADD (
                    START_SCNBAS NUMBER, START_SCNWRP NUMBER)');

    logmnrRunSql('ALTER TABLE SYSTEM.LOGMNR_OBJ$
                    add primary key (LOGMNR_UID, OBJ#) disable');

    logmnrRunSql('alter table SYSTEM.LOGMNR_USER$
                   add primary key (LOGMNR_UID, USER#) disable');

    /* Create all of the "normal" dictionary tables  */

    logmnrCreateADicTable( mustPartition, tableSpaceName,
			   'LOGMNR_TAB$', 'LOGMNRG_TAB$',
                           'LOGMNR_UID, OBJ#');

    logmnrCreateADicTable( mustPartition, tableSpaceName,
                           'LOGMNR_COL$', 'LOGMNRG_COL$',
                           'LOGMNR_UID, OBJ#, INTCOL#');

    logmnrCreateADicTable( mustPartition, tableSpaceName, 
                           'LOGMNR_ATTRCOL$', 'LOGMNRG_ATTRCOL$',
                           'LOGMNR_UID, OBJ#, INTCOL#');

    logmnrCreateADicTable( mustPartition, tableSpaceName, 
                           'LOGMNR_TS$', 'LOGMNRG_TS$',
                           'LOGMNR_UID, TS#');

    logmnrCreateADicTable( mustPartition, tableSpaceName,
                           'LOGMNR_IND$', 'LOGMNRG_IND$',
                           'LOGMNR_UID, OBJ#');

    logmnrCreateADicTable( mustPartition, tableSpaceName, 
                           'LOGMNR_TABPART$', 'LOGMNRG_TABPART$',
                           'LOGMNR_UID, OBJ#, BO#');

    logmnrCreateADicTable( mustPartition, tableSpaceName,
                           'LOGMNR_TABSUBPART$', 'LOGMNRG_TABSUBPART$',
                           'LOGMNR_UID, OBJ#, POBJ#');

    logmnrCreateADicTable( mustPartition, tableSpaceName, 
                           'LOGMNR_TABCOMPART$', 'LOGMNRG_TABCOMPART$',
                           'LOGMNR_UID, OBJ#' );

    logmnrCreateADicTable( mustPartition, tableSpaceName, 
                           'LOGMNR_TYPE$', 'LOGMNRG_TYPE$',
                           'LOGMNR_UID, TOID, VERSION#' );

    logmnrCreateADicTable( mustPartition, tableSpaceName, 
                           'LOGMNR_COLTYPE$', 'LOGMNRG_COLTYPE$',
                           'LOGMNR_UID, OBJ#, INTCOL#');

    logmnrCreateADicTable( mustPartition, tableSpaceName, 
                           'LOGMNR_ATTRIBUTE$', 'LOGMNRG_ATTRIBUTE$',
                           'LOGMNR_UID, TOID, VERSION#, ATTRIBUTE#');

    logmnrCreateADicTable( mustPartition, tableSpaceName, 
                           'LOGMNR_LOB$', 'LOGMNRG_LOB$',
                           'LOGMNR_UID, OBJ#, INTCOL#');

    logmnrCreateADicTable( mustPartition, tableSpaceName, 
                           'LOGMNR_CDEF$', 'LOGMNRG_CDEF$',
                           'LOGMNR_UID, CON#' );

    logmnrCreateADicTable( mustPartition, tableSpaceName, 
                           'LOGMNR_CCOL$', 'LOGMNRG_CCOL$',
                           'LOGMNR_UID, CON#, INTCOL#');

    logmnrCreateADicTable( mustPartition, tableSpaceName, 
                           'LOGMNR_ICOL$', 'LOGMNRG_ICOL$',
                           'LOGMNR_UID, OBJ#, INTCOL#');

    logmnrCreateADicTable( mustPartition, tableSpaceName, 
                           'LOGMNR_LOBFRAG$', 'LOGMNRG_LOBFRAG$',
                           'LOGMNR_UID, FRAGOBJ#');

    logmnrCreateADicTable( mustPartition, tableSpaceName, 
                           'LOGMNR_INDPART$', 'LOGMNRG_INDPART$',
                           'LOGMNR_UID, OBJ#, BO#');

    logmnrCreateADicTable( mustPartition, tableSpaceName, 
                           'LOGMNR_INDSUBPART$', 'LOGMNRG_INDSUBPART$',
                           'LOGMNR_UID, OBJ#, POBJ#');

    logmnrCreateADicTable( mustPartition, tableSpaceName, 
                           'LOGMNR_INDCOMPART$', 'LOGMNRG_INDCOMPART$',
                           'LOGMNR_UID, OBJ#');

  END logmnrCreateDictTables;


  /***************************************************************************
   *
   *    NAME
   *      logmnrCreateDictIndecies
   *
   *    DESCRIPTION
   *      Creates all indecies for the Logminer Dictionary tables
   *
   *    PARAMETERS
   *      mustPartition  FALSE means that partition option is not available.
   *                     TRUE means that partition option is available.
   *                          Most tables are assumped to be partitioned and
   *                          therefore will have LOCAL indecies created.
   *                          Some tables are never partitioned and therefore
   *                          will have normal indecies defined.
   *      tableSpaceName  Names the tablespace attribute used for indexes.
   *      logging        FALSE do not log when creating/loading the indecies
   *                     TRUE  (default) do log when creating/loading the
   *                           indecies.
   *    TRANSACTION
   *
   *    ERRORS
   *       All errors are signalled.
   *       
   */

  PROCEDURE logmnrCreateDictIndecies( mustPartition    IN BOOLEAN,
                                      tableSpaceName   IN VARCHAR2,
                                      logging          IN BOOLEAN default TRUE)
  IS
  BEGIN
    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnr_i1obj$', 'logmnr_obj$',
                          'logmnr_uid, obj#');
    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnr_i1user$', 'logmnr_user$',
                          'logmnr_uid, user#');
    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnr_i1tab$', 'logmnr_tab$',
                          'logmnr_uid, obj#');
    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnr_i1ts$', 'logmnr_ts$',
                          'logmnr_uid, ts#');
    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnr_i1tabpart$', 'logmnr_tabpart$',
                          'logmnr_uid, obj#, bo#');
    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnr_i1tabsubpart$', 'logmnr_tabsubpart$',
                          'logmnr_uid, obj#, pobj#');
    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnr_i1tabcompart$', 'logmnr_tabcompart$',
                          'logmnr_uid, obj#');
    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnr_i1ind$', 'logmnr_ind$',
                          'logmnr_uid, obj#');
   /* gtcs */
    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnr_i1col$', 'logmnr_col$',
                          'logmnr_uid, obj#, intcol#');
  /*
   * gov - might add index for obj$ which includes uid, owner# and name,
   *       but query is such that it probably does full table scan anyway.
   */

  /* gtc  */
    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnr_i2col$', 'logmnr_col$',
                          'logmnr_uid, obj#, name');
  /*
   * gtc - might add index for SYSTEM.LOGMNR_ATTRCOL$, but unclear what
   * include.  Maybe LOGMNR_UID, OBJ#, NAME, INTCOL but appears that
   * full table scan may be done anyway.
   */

  /*
   * The rest are to ensure that each primary key def has an index.
   * The primary key def is used by DDL tracking.
   * Presumably these could be removed for faster startup, but slower
   * DDL tracking.
   */

    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnr_i1attrcol$', 'logmnr_attrcol$',
                          'logmnr_uid, obj#, intcol#');
    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnr_i1type$', 'logmnr_type$',
                          'logmnr_uid, toid, version#');
    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnr_i1coltype$', 'logmnr_coltype$',
                          'logmnr_uid, obj#, intcol#');
    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnr_i1lob$', 'logmnr_lob$',
                          'logmnr_uid, obj#, intcol#'); 
    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnr_i1cdef$', 'logmnr_cdef$',
                          'logmnr_uid, con#');
    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnr_i1ccol$', 'logmnr_ccol$',
                          'logmnr_uid, con#, intcol#');
    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnr_i1icol$', 'logmnr_icol$',
                          'logmnr_uid, obj#, intcol#'); 
    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnr_i1lobfrag$', 'logmnr_lobfrag$',
                          'logmnr_uid, fragobj#');
    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnr_i1indpart$', 'logmnr_indpart$',
                          'logmnr_uid, obj#, bo#');
    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnr_i1indsubpart$', 'logmnr_indsubpart$',
                          'logmnr_uid, obj#, pobj#');
    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnr_i1indcompart$', 'logmnr_indcompart$',
                          'logmnr_uid, obj#');
    
    /* Bug: 3953131 improve purge_scn performance */
    logmnrCreateADicIndex(mustPartition, tableSpaceName, logging,
                          'logmnrc_i3gtlo', 'logmnrc_gtlo',
                          'logmnr_uid, drop_scn');

  END logmnrCreateDictIndecies;

  /***************************************************************************
   *
   *    NAME
   *      logmnrCreateSpillTables
   *
   *    DESCRIPTION
   *      Creates/Replaces spill and log tables for Logminer in
   *      requested tablespace.
   *
   *    PARAMETERS
   *      mustPartition  FALSE means that partition option is not available.
   *                     TRUE means that partition option is available.
   *                          Currently spill, etc. tables are never
   *                          partitioned; this parameter is for future use.
   *     tableSpaceName -  Name of tablespace for tables to be created.
   *       
   *    TRANSACTION
   *
   *    ERRORS
   *       All errors are signalled.
   *       
   */

  PROCEDURE logmnrCreateSpillTables( mustPartition    IN BOOLEAN,
                                     tableSpaceName   IN VARCHAR2)
  IS
    /* Cursor for all non primary key indexes on logmnr_log$ */
   cursor del_ind is
       select i.owner, i.index_name from all_indexes i 
	 where i.table_name = 'LOGMNR_LOG$' and i.table_owner='SYSTEM' 
	 and (i.owner, i.index_name) 
	 not in (select c.owner, c.index_name from all_constraints c 
	 where c.owner='SYSTEM' and c.table_name = 'LOGMNR_LOG$' 
	 and c.constraint_type = 'P');
  BEGIN
    logmnrRunSQL('TRUNCATE TABLE SYSTEM.LOGMNR_LOG$', TRUE);
    logmnrRunSQL('DROP TABLE SYSTEM.LOGMNR_LOG$', TRUE);
    logmnrRunSQL('CREATE TABLE SYSTEM.LOGMNR_LOG$ (
              session#        number,
              thread#         number,
              sequence#       number,
              first_change#   number,
              next_change#    number,
              first_time      date,
              next_time       date,
              file_name       varchar2(513),
              status          number,
              info            varchar2(32),
              timestamp       date,
              dict_begin      varchar2(3),
              dict_end        varchar2(3),
              status_info     varchar2(32))
           TABLESPACE ' || tableSpaceName || ' LOGGING ');
    logmnrRunSQL('ALTER TABLE SYSTEM.LOGMNR_LOG$ ADD 
            PRIMARY KEY (SESSION#, THREAD#, SEQUENCE#)');

    /* 
     * Backport Bug: 4137433: Add index on main columns in log$ to speed up
     * queries to krvxalfft_active and other queries on log$.
     */
    FOR del_ind_rec in del_ind LOOP
      /* First drop all other indexes on log$ (except primary key) */
      logmnrRunSQL('DROP INDEX ' || del_ind_rec.owner || '.' || 
            	   del_ind_rec.index_name, TRUE);
    END LOOP;
    logmnrRunSQL('CREATE INDEX system.logmnr_log$_active ON system.logmnr_log$(
                  session#, thread#, sequence#, first_change#,
                  next_change#, status) TABLESPACE ' || tableSpaceName || 
                  ' LOGGING');

    logmnrRunSQL('TRUNCATE TABLE SYSTEM.logmnr_processed_log$', TRUE);
    logmnrRunSQL('DROP TABLE SYSTEM.logmnr_processed_log$', TRUE);
    logmnrRunSQL('CREATE TABLE SYSTEM.logmnr_processed_log$ (
              session#        number,
              thread#         number,
              sequence#       number,
              first_change#   number,
              next_change#    number,
              first_time      date,
              next_time       date,
              file_name       varchar2(513),
              status          number,
              info            varchar2(32),
              timestamp       date)
            TABLESPACE ' || tableSpaceName || ' LOGGING');
    logmnrRunSQL('ALTER TABLE SYSTEM.LOGMNR_PROCESSED_LOG$ 
            ADD PRIMARY KEY (SESSION#, THREAD#)');


    /*
     * New for 9.2.0.6 backport of bug fix 3464629
     *   PCTVERSION 0
     */
    logmnrRunSQL('TRUNCATE TABLE SYSTEM.logmnr_spill$', TRUE);
    logmnrRunSQL('DROP TABLE SYSTEM.logmnr_spill$', TRUE);
    logmnrRunSQL('CREATE TABLE SYSTEM.logmnr_spill$ (
                                                     session#     number,
                                                     xidusn       number,
                                                     xidslt       number,
                                                     xidsqn       number,
                                                     chunk        integer, 
                                                     sequence#    number,
                                                     offset       number,  
                                                     spill_data   blob)
                                LOB (spill_data) STORE AS (TABLESPACE ' ||
                                     tableSpaceName || ' NOCACHE NOLOGGING
                                     PCTVERSION 0 
                                     CHUNK 16k STORAGE (INITIAL 16K NEXT 16K))
                                TABLESPACE ' || tableSpaceName || ' LOGGING' );
    logmnrRunSQL('ALTER TABLE SYSTEM.LOGMNR_SPILL$ 
       ADD PRIMARY KEY (session#, xidusn, xidslt, xidsqn, chunk, sequence#)');

    /*
     * New for 9.2.0.6 backport of bug fix 3464629
     *   PCTVERSION 0
     */
    logmnrRunSQL('TRUNCATE TABLE SYSTEM.logmnr_age_spill$', TRUE);
    logmnrRunSQL('DROP TABLE SYSTEM.logmnr_age_spill$', TRUE);
    logmnrRunSQL('CREATE TABLE SYSTEM.logmnr_age_spill$ (
                                                     session#     number,
                                                     xidusn       number,
                                                     xidslt       number,
                                                     xidsqn       number,
                                                     chunk        integer, 
                                                     sequence#    number,
                                                     offset       number,  
                                                     spill_data   blob)
                                LOB (spill_data) STORE AS (TABLESPACE ' ||
                                     tableSpaceName || ' CACHE 
                                     PCTVERSION 0 
                                     CHUNK 16k STORAGE (INITIAL 16K NEXT 16K))
                                TABLESPACE ' || tableSpaceName || ' LOGGING' );
    logmnrRunSQL('ALTER TABLE SYSTEM.LOGMNR_AGE_SPILL$ 
        ADD PRIMARY KEY (session#, xidusn, xidslt, xidsqn, chunk, sequence#)');

    /*
     * New for 9.2.0.6 backport of bug fix 3464629
     *   PCTVERSION 0
     */
    logmnrRunSQL('TRUNCATE TABLE SYSTEM.logmnr_restart_ckpt$', TRUE);
    logmnrRunSQL('DROP TABLE SYSTEM.logmnr_restart_ckpt$', TRUE);
    logmnrRunSQL('CREATE TABLE SYSTEM.logmnr_restart_ckpt$ (
                                                     session#     number,
                                                     valid        number,
                                                     ckpt_scn     number,
                                                     xidusn       number,
                                                     xidslt       number,
                                                     xidsqn       number,
                                                     session_num  number,
                                                     serial_num   number,
                                                     offset       number,  
                                                     client_data  blob)
                                LOB (client_data) STORE AS (TABLESPACE ' ||
                                     tableSpaceName || ' CACHE PCTVERSION 0 
                                     CHUNK 16k STORAGE (INITIAL 16K NEXT 16K))
                                TABLESPACE ' || tableSpaceName || ' LOGGING' );
    /*
     * New for 9.2.0.6 backport of bug fix 3464629
     *   PCTVERSION 0
     */
    logmnrRunSQL('ALTER TABLE SYSTEM.LOGMNR_restart_ckpt$ 
            ADD PRIMARY KEY 
      (session#, ckpt_scn, xidusn, xidslt, xidsqn, session_num, serial_num)');

    logmnrRunSQL('TRUNCATE TABLE SYSTEM.logmnr_restart_ckpt_txinfo$', TRUE);
    logmnrRunSQL('DROP TABLE SYSTEM.logmnr_restart_ckpt_txinfo$', TRUE);
    logmnrRunSQL('CREATE TABLE SYSTEM.logmnr_restart_ckpt_txinfo$ (
                                                     session#     number,
                                                     xidusn       number,
                                                     xidslt       number,
                                                     xidsqn       number,
                                                     session_num  number,
                                                     serial_num   number,
                                                     flag         number,
                                                     start_scn    number,
                                                     effective_scn number,
                                                     offset       number,  
                                                     tx_data      blob)
                                LOB (tx_data) STORE AS (TABLESPACE ' ||
                                     tableSpaceName || ' CACHE PCTVERSION 0 
                                     CHUNK 16k STORAGE (INITIAL 16K NEXT 16K))
                                TABLESPACE ' || tableSpaceName || ' LOGGING' );
    logmnrRunSQL('ALTER TABLE system.logmnr_restart_ckpt_txinfo$ 
            ADD CONSTRAINT restart_ckpt_txinfo_pk PRIMARY KEY 
  (session#, xidusn, xidslt, xidsqn, session_num, serial_num, effective_scn)');

  END logmnrCreateSpillTables;

BEGIN  /* logmnr_create_replace_metadata */

  IF DumpDebugMsgs THEN
    /* When debugging need a very large output buffer  */
    DBMS_OUTPUT.ENABLE(90000);
  END IF;

  loc_DictTablespace :=
       replace(replace(upper(substr(DictTablespace,1,30)),'"'),' ');
  select count(1) into tscnt from sys.ts$
    where name = loc_DictTablespace and online$ = 1;
  if tscnt <> 1 then
    raise invalid_dict_ts;
  end if;
  loc_DictTablespace := '"' || loc_DictTablespace || '"';

  loc_SpillTablespace := 
       replace(replace(upper(substr(SpillTablespace,1,30)),'"'),' ');
  select count(1) into tscnt from sys.ts$ 
    where name = loc_SpillTablespace and online$ = 1;
  if tscnt <> 1 then
    raise invalid_spill_ts;
  end if;
  loc_SpillTablespace := '"' || loc_SpillTablespace || '"';

  /*
   * Do not move the gather tables out of initial tablespace.
   * The problem is that doing so invalidates dbms_logmnr_d package.
   * This means that the next invocation of build will cause an
   * error unless user explicitly recompiles the package after
   * running dbms_logmnr_d.set_tablespace.
   *     alter package sys.dbms_logmnr_d compile body;
   */
  IF InitialCreation THEN
    logmnrCreateGatherTables( loc_DictTablespace );
  END IF;

  logmnrCreateDictTables( PartOn, loc_DictTablespace );
  logmnrCreateTempTables;
  logmnrCreateDictIndecies( PartOn, loc_DictTablespace, LoggingOn );
  logmnrCreateSpillTables( PartOn, loc_SpillTablespace );

END logmnr_create_replace_metadata;
/

show errors;
grant execute on logmnr_create_replace_metadata to execute_catalog_role
/

DECLARE 

/***************************************************************************
 *
 *    NAME
 *      This is the main block.  It is unnamed.
 *
 *    DESCRIPTION
 *      Creates/Replaces all Logminer meta data.
 *
 *    PARAMETERS
 *       
 *    TRANSACTION
 *
 *    ERRORS
 *       All errors are signalled.
 *       
 */

  LOGMNR_UPGRADE_DUPCOL EXCEPTION;
  PRAGMA EXCEPTION_INIT(LOGMNR_UPGRADE_DUPCOL, -1430);

  /* LOGMNR_UPGRADE_DUPNAME needed for ajadams_backport_9.2.0.5.0_3785754. */
  LOGMNR_UPGRADE_DUPNAME EXCEPTION;
  PRAGMA EXCEPTION_INIT(LOGMNR_UPGRADE_DUPNAME, -955);

  output_warning boolean := FALSE;
  partOn boolean := FALSE;  /* Partitioning Option avaiable == TRUE */
  partString varchar2(25) := '';
  curLogmnrColColCnt integer := 0; 
  interestingColsTableCnt integer := 0;
  
  /***************************************************************************
   *
   *    NAME
   *      alter_table_add_1_col
   *
   *    DESCRIPTION
   *      add a column to a talbe, swallow the error if it is a duplicate.
   *
   *    PARAMETERS
   *       tablename - full table name (e.g. SYSTEM.LOGMNR_OBJ$) to be altered
   *       column_description - name and datatype of column to be added.
   *
   *    ERRORS
   *      Error conditions other than duplicate column are raised.
   *
   *    NOTES
   *      Only supply one column at a time in the column_description.
   *      Add column will not add any column if even just one is a duplicate.
   */
  PROCEDURE alter_table_add_1_col(tablename IN VARCHAR,
                                  column_description IN VARCHAR2)
  IS
    altbuf varchar2(2000);
  BEGIN
    altbuf := 'alter table ' || tablename || ' add  ( ' ||
               column_description || ' )';
    BEGIN
      EXECUTE IMMEDIATE altbuf;
    EXCEPTION
      WHEN LOGMNR_UPGRADE_DUPCOL THEN
        NULL;
    END;
  END alter_table_add_1_col;


  /*
   * awb: 21-sep-2004
   * PrepareGTCSforNewKey is added here for ajadams_backport_9.2.0.5.0_3785754.
   * When merging this txn note that this routine will be in addition to
   * other new code (e.g. 'alter_table_add_1_col') that has already been
   * added at this location for 9.2.0.6.  This block can follow the other
   * new code.
   */
  /****************************************************************************
   *
   *    NAME
   *      PrepareGTCSforNewKey
   *
   *    DESCRIPTION
   *      On upgrade from 9i to 10g, logmnrc_gtcs gets a new primary key of
   *    logmnr_uid, obj#, objv#, intcol#.
   *    For 9i the key was logmnr_uid, obj#, objv#, segcol#.
   *    Though the new key should always be unique, bug 3785754 could cause
   *    a given table entry to have multiple occurances of the same intcol#.
   *    This procedure "tweaks" (does not correct) all offending entries so
   *    that the creation of the new key will succeed.  "Tweaking" happens such
   *    that tweaked intcol# values are easily indentifed as having been
   *    tweaked.  They are all over 2000 whereas normal intcol# are typically
   *    less than 1000.
   *      Pre-tweaked values are first saved in uniquely named, generated
   *    tables of the form SYS.BUG$3785754_xxxxxx where xxxxxx is a unique
   *    integer that corresponds to the occation on which procedure was run.
   *      The source of logmnrc_gtcs corruption is always logmnr_col$. Streams
   *    sessions may not be fully caught up at the time of upgrade.  If a
   *    session still needed to mine the redo of a table for which the
   *    Logminer frontier metadata is wrong, subsquent corruption of
   *    logmnrc_gtcs is possible.
   *      In this post upgrade scenario, logmnr_gtcs will be corrupted
   *    differently than pre-upgrade.  Rather than just having incorrect
   *    duplicate intcol# values, in logmnr_gtcs the duplicates will now
   *    be collapsed into a single column description.  This type of corruption
   *    would likely cause streams apply to stop.
   *      To avoid this cascading problem, at the time of the logmnr_gtcs
   *    fixup a similar fixup will also be applied to logmnr_col$.  Note that
   *    this should ensure that the mining of subsequent DMLs that need this
   *    definition may continue to work if this adjustment is to a
   *    logmnr_gtcs entry that has only been corrupted by one DDL.  If a
   *    corrupt entry has been manipulated by subsequent DDLs, the results
   *    are unknown.  Further more, once logmnr_col$ has been adjusted by
   *    this script, the application of a subsequent DDL will likely result
   *    in an unusable definition.
   *      In otherwords, by applying the same tweak to corrupt logmnr_col$
   *    entries, we allow for the possibility of mining
   *    unprocessed DMLs that were generated pre-upgrade.  This change
   *    does not allow for the mining of unapplied DMLs that require the
   *    corrupt logmnr_col$ / logmnrc_gtcs entries if there is there
   *    is an intervening DDL.  At the time of upgrade, with our without
   *    this change, unprocessed DDLs that impact corrupt logmnr_col$ entries
   *    will likely cause problems when attempt to apply subsequent DMLs
   *    that need these definitions.
   *
   *    PARAMETERS
   *      None
   *
   *    TRANSACTION
   *      This routine performs DDL.  There should be no transaction open
   *    in the context in which this is called.
   *
   *    ERRORS
   *      No errors are expected.  All errors are signalled.
   */

  PROCEDURE PrepareGTCSforNewKey
  IS
  BugTableName  VARCHAR2(60);

    /*
     * MakeBugTableName
     *    Assuming that multiple callers are not invoking this simultaneously,
     *    returns a new, unique name to be used to name a table related to
     *    this bug.
     *
     * RETURNS:
     *           String, not to exceed 60 characters, of user.tablename.
     *           Example: 'SYS.BUG$3785754_xxxxxx'
     */
    FUNCTION MakeBugTableName
    RETURN VARCHAR2
    IS
      NewTableName    varchar2(30);
    BEGIN
      select 'SYS.BUG$3785754_' ||
             ltrim(to_char(1 + NVL(MAX(TO_NUMBER(LTRIM(o.name,
                                                      'BUG$3785754_'))),
                                   0),
                     '0999999'))
             into NewTableName
      from sys.obj$ o, sys.user$ u
      where u.name = 'SYS' and
            o.owner# = u.user# and
            o.type# = 2 and
            o.name like 'BUG$3785754_%';
      RETURN NewTableName;
    END MakeBugTableName;

    /*
     * GtcsTweakIsNeeded
     *   Looks at logmnrc_gtcs and returns TRUE if there are duplicates
     *   for the proposed new key of logmnr_uid, obj#, objv#, intcol#.
     *   FALSE is returned if there are no duplicates.
     */
    FUNCTION GtcsTweakIsNeeded
    RETURN BOOLEAN
    IS
      CntKeyViloations number;
    BEGIN
      /*
       * Use execute immediate to ensure that this does not recompile
       * every time a partition is added/dropped to logmnrc_gtcs.
       */
      execute immediate
        'select count(*)
         from (select logmnr_uid
               from system.logmnrc_gtcs
               group by logmnr_uid, obj#, objv#, intcol#
               having count(1) > 1)' into CntKeyViloations;
      return (CntKeyViloations > 0);
    END GtcsTweakIsNeeded;

    /*
     * ColTweakIsNeeded
     *   Looks at logmnr_col$ and returns TRUE if there are duplicates
     *   for a key of logmnr_uid, obj#, intcol#.
     *   FALSE is returned if there are no duplicates.
     */
    FUNCTION ColTweakIsNeeded
    RETURN BOOLEAN
    IS
      CntKeyViloations number;
    BEGIN
      /*
       * Use execute immediate to ensure that this does not recompile
       * every time a partition is added/dropped to logmnrc_gtcs.
       */
      execute immediate
        'select count(*)
         from (select logmnr_uid
               from system.logmnr_col$
               group by logmnr_uid, obj#, intcol#
               having count(1) > 1)' into CntKeyViloations;
      return (CntKeyViloations > 0);
    END ColTweakIsNeeded;

    /*
     * CreateTableOfTweakedGtcs
     *   Creates a table named by input parameter of all tuples from
     *   logmnrc_gtcs that will be tweaked to ensure successful creation
     *   of new primary key.
     *   PARAMETERS:
     *     BugTableName :  IN
     *                     A string of an unsued schema.tablename that is
     *                     to be created and loaded with logmnrc_gtcs entries
     *                     that need to be tweaked.
     */
    PROCEDURE CreateTableOfTweakedGtcs(BugTableName IN VARCHAR2)
    IS
    BEGIN
      execute immediate 'create table ' || BugTableName ||
        ' as select * 
          from system.logmnrc_gtcs x
          where (x.logmnr_uid, x.obj#, x.objv#, x.intcol#) IN
            (select logmnr_uid, obj#, objv#, intcol#
             from system.logmnrc_gtcs
             group by logmnr_uid, obj#, objv#, intcol#
             having count(1) > 1)';
    END CreateTableOfTweakedGtcs;

    /*
     * CreateTableOfTweakedCols
     *   Creates a table named by input parameter of all tuples from
     *   logmnr_col$ that will be tweaked to ensure that there are not
     *   duplicate intcol#s for the columns of a given table.
     *   PARAMETERS:
     *     BugTableName :  IN
     *                     A string of an unsued schema.tablename that is
     *                     to be created and loaded with logmnrc_gtcs entries
     *                     that need to be tweaked.
     */
    PROCEDURE CreateTableOfTweakedCols(BugTableName IN VARCHAR2)
    IS
    BEGIN
      execute immediate 'create table ' || BugTableName ||
        ' as select * 
          from system.logmnr_col$ x
          where (x.logmnr_uid, x.obj#, x.intcol#) IN
            (select logmnr_uid, obj#, intcol#
             from system.logmnr_col$
             group by logmnr_uid, obj#, intcol#
             having count(1) > 1)';
    END CreateTableOfTweakedCols;

    /*
     * MakeTheGtcsTweaks
     *   Finds every tuple in logmnrc_gtcs that participates in a collision
     *   of what is to be the new primary key and tweaks it's intcol# value
     *   to ensure no collisions.  The intcol# is set to 2000+segcol#.
     */
    PROCEDURE MakeTheGtcsTweaks
    IS
    BEGIN
      execute immediate
        'update system.logmnrc_gtcs x
         set x.intcol# = 2000 + x.segcol#
         where (x.logmnr_uid, x.obj#, x.objv#, x.intcol#) IN
           (select logmnr_uid, obj#, objv#, intcol#
            from system.logmnrc_gtcs
            group by logmnr_uid, obj#, objv#, intcol#
            having count(1) > 1)';
    END MakeTheGtcsTweaks;

    /*
     * MakeTheColTweaks
     *   Finds every tuple in logmnr_col$ that has a duplicate intcol#
     *   within a given table definition and tweaks intcol# values
     *   to eliminate collisions.  The intcol# is set to 2000+segcol#.
     *   Although this is imperfect as there are rare circumstances when
     *   a table may have multiple, equal values of segcol# == 0, these
     *   columns are not of interest when processing redo generated by 9i.
     */
    PROCEDURE MakeTheColTweaks
    IS
    BEGIN
      execute immediate
        'update system.logmnr_col$ x
         set x.intcol# = 2000 + x.segcol#
         where (x.logmnr_uid, x.obj#, x.intcol#) IN
           (select logmnr_uid, obj#, intcol#
            from system.logmnr_col$
            group by logmnr_uid, obj#, intcol#
            having count(1) > 1)';
    END MakeTheColTweaks;

  BEGIN  /* body of PrepareGTCSforNewKey */
    IF GtcsTweakIsNeeded THEN
      BugTableName := MakeBugTableName;
      CreateTableOfTweakedGtcs(BugTableName);
      MakeTheGtcsTweaks;
      Commit;
    END IF;
    IF ColTweakIsNeeded THEN
      BugTableName := MakeBugTableName;
      CreateTableOfTweakedCols(BugTableName);
      MakeTheColTweaks;
      Commit;
    END IF;
  END PrepareGTCSforNewKey;

BEGIN  /* main unnamed block */
  /*
   * For all post 9.2 upgrades we can no longer delete and recreate all
   * Logminer related metadata because there may exist sessions which
   * can not be dropped.  Only recreate Logminer metadata when we detect
   * that a metadata object, new for 9.2, is not present.
   * Note that this test must handle the case of an upgrade, downgrade,
   * upgrade scenario.  Because in 9.2 new columns were added to some
   * of the original 9.1 tables, we count columns.
   */

  SELECT count(c.name)
  INTO curLogmnrColColCnt
  FROM sys.obj$ o, sys.user$ u, sys.col$ c
  WHERE o.owner# = u.user# and o.obj# = c.obj# and (
    ( u.name = 'SYSTEM' and o.type# = 2 and o.name in
                                   (
                                   'LOGMNRC_DBNAME_UID_MAP',
                                   'LOGMNRC_GSII',
                                   'LOGMNRC_GTCS',
                                   'LOGMNRC_GTLO',
                                   'LOGMNR_AGE_SPILL$',
                                   'LOGMNR_ATTRCOL$',
                                   'LOGMNR_ATTRIBUTE$',
                                   'LOGMNR_CCOL$',
                                   'LOGMNR_CDEF$',
                                   'LOGMNR_COL$',
                                   'LOGMNR_COLTYPE$',
                                   'LOGMNR_DICTIONARY$',
                                   'LOGMNR_DICTSTATE$',
                                   'LOGMNR_HEADER1$',
                                   'LOGMNR_HEADER2$',
                                   'LOGMNR_ICOL$',
                                   'LOGMNR_IND$',
                                   'LOGMNR_INDCOMPART$',
                                   'LOGMNR_INDPART$',
                                   'LOGMNR_INDSUBPART$',
                                   'LOGMNR_LOB$',
                                   'LOGMNR_LOBFRAG$',
                                   'LOGMNR_LOG$',
                                   'LOGMNR_OBJ$',
                                   'LOGMNR_PROCESSED_LOG$',
                                   'LOGMNR_RESTART_CKPT$',
                                   'LOGMNR_RESTART_CKPT_TXINFO$',
                                   'LOGMNR_SESSION$',
                                   'LOGMNR_SPILL$',
                                   'LOGMNR_TAB$',
                                   'LOGMNR_TABCOMPART$',
                                   'LOGMNR_TABPART$',
                                   'LOGMNR_TABSUBPART$',
                                   'LOGMNR_TS$',
                                   'LOGMNR_TYPE$',
                                   'LOGMNR_UID$',
                                   'LOGMNR_USER$'
                                    )
    )
  OR
    (u.name = 'SYS' and o.type# = 2 and o.name in
                                   (
                                   'LOGMNRG_ATTRCOL$',
                                   'LOGMNRG_ATTRIBUTE$',
                                   'LOGMNRG_CCOL$',
                                   'LOGMNRG_CDEF$',
                                   'LOGMNRG_COL$',
                                   'LOGMNRG_COLTYPE$',
                                   'LOGMNRG_DICTIONARY$',
                                   'LOGMNRG_ICOL$',
                                   'LOGMNRG_IND$',
                                   'LOGMNRG_INDCOMPART$',
                                   'LOGMNRG_INDPART$',
                                   'LOGMNRG_INDSUBPART$',
                                   'LOGMNRG_LOB$',
                                   'LOGMNRG_LOBFRAG$',
                                   'LOGMNRG_OBJ$',
                                   'LOGMNRG_SEED$',
                                   'LOGMNRG_TAB$',
                                   'LOGMNRG_TABCOMPART$',
                                   'LOGMNRG_TABPART$',
                                   'LOGMNRG_TABSUBPART$',
                                   'LOGMNRG_TS$',
                                   'LOGMNRG_TYPE$',
                                   'LOGMNRG_USER$',
                                   'LOGMNRT_ATTRCOL$',
                                   'LOGMNRT_ATTRIBUTE$',
                                   'LOGMNRT_CCOL$',
                                   'LOGMNRT_CDEF$',
                                   'LOGMNRT_COL$',
                                   'LOGMNRT_COLTYPE$',
                                   'LOGMNRT_DICTIONARY$',
                                   'LOGMNRT_ICOL$',
                                   'LOGMNRT_IND$',
                                   'LOGMNRT_INDCOMPART$',
                                   'LOGMNRT_INDPART$',
                                   'LOGMNRT_INDSUBPART$',
                                   'LOGMNRT_LOB$',
                                   'LOGMNRT_LOBFRAG$',
                                   'LOGMNRT_OBJ$',
                                   'LOGMNRT_SEED$',
                                   'LOGMNRT_TAB$',
                                   'LOGMNRT_TABCOMPART$',
                                   'LOGMNRT_TABPART$',
                                   'LOGMNRT_TABSUBPART$',
                                   'LOGMNRT_TS$',
                                   'LOGMNRT_TYPE$',
                                   'LOGMNRT_USER$',
                                   'LOGMNR_BUILDLOG',
                                   'LOGMNR_INTERESTING_COLS'
                                   )
    )
  );

  /* 
   * There should be 680 Logminer related columns in a 9.2 catalog.
   * Anything less indicates that Logminer related meta data is not
   * 9.2 compliant and it is therefore safe to destroy and recreate it.
   * Anything more than 680 may indicate that we are being run as
   * part of a downgrade from 10i which means that we must not delete
   * existing data, hence the >= condition.
   * NOTE: An implication of this change is that the total number of columns
   *       employed in the 9.2 set of Logminer tables named above must not
   *       decrease.
   */
  IF curLogmnrColColCnt >= 680 THEN

    /* Bug 2446679 -- Type 10 objects must not be stored in     */
    /* system.logmnrc_gtlo.  This bug is fixed in 9.2.0.2, but  */
    /* in case this database hit this bug in 9.2.0.1 we need to */
    /* try to delete those rows now.  Because of the check      */
    /* above we assume the table exists.  If that table doesn't */
    /* exist and we somehow made it into this block it is a     */
    /* serious bug anyway and throwing an error would be fine.  */
    /* In the next major release we will move this delete into  */
    /* the upgraded/downgrade scripts.                          */
    EXECUTE IMMEDIATE ' DELETE FROM SYSTEM.LOGMNRC_GTLO ' || 
                      ' WHERE LVL0TYPE# = 10 ';
    COMMIT;

    /* 
     * system.logmnr_restart_ckpt_txinfo used to have a primary key
     * on (session#, xidusn, xidslt, xidsqn, session_num, serial_num)
     * prior to 9.2.0.2 We need to drop that primary key and create
     * a new one. Ideally we should check to see whether the named
     * primary key exists, before dropping and recreating the primary key.
     * The following delete will only remove 9.2.0.1 checkpoint data
     */
     execute immediate ' delete system.logmnr_restart_ckpt_txinfo$  ' || 
       ' where start_scn = effective_scn';
     commit;
     execute immediate ' ALTER TABLE system.logmnr_restart_ckpt_txinfo$ ' ||
                   ' DROP PRIMARY KEY';
     execute immediate ' ALTER TABLE system.logmnr_restart_ckpt_txinfo$ ' || 
            ' ADD CONSTRAINT restart_ckpt_txinfo_pk PRIMARY KEY' || 
  '(session#, xidusn, xidslt, xidsqn, session_num, serial_num, effective_scn)';

    /*
     * Backport Bug: 4137433: Add index on main columns in log$ to speed up
     * queries to krvxalfft_active and other queries on log$.
    */
    DECLARE
    /* First remove all other indexes on logmnr_log$, except primary key */
      cursor del_ind is
       select i.owner, i.index_name from all_indexes i 
	 where i.table_name = 'LOGMNR_LOG$' and i.table_owner='SYSTEM' 
	 and (i.owner, i.index_name) 
	 not in (select c.owner, c.index_name from all_constraints c 
	 where c.owner='SYSTEM' and c.table_name = 'LOGMNR_LOG$' 
	 and c.constraint_type = 'P');
    BEGIN
      FOR del_ind_rec in del_ind LOOP
	 execute immediate 'DROP INDEX ' || del_ind_rec.owner || '.' || 
   	   del_ind_rec.index_name;
      END LOOP;
    END;
    /* Create the index */
    execute immediate 'CREATE INDEX system.logmnr_log$_active ON ' ||
                      'system.logmnr_log$(session#, thread#, sequence#, ' ||
                      'first_change#, next_change#, status) LOGGING';


    /*
     * BACKPORT of bug fix 3381899 necessitates the addition of RBA columns
     * to logmnr_dictstate$.  Also, logmnr_dictstate$ end_scn no longer
     * has special meaning for a value of max_scn.
     * BACKPORT 9.2.0.6 backport of bug fix 3464629
     *   PCTVERSION 0
     */
    alter_table_add_1_col('SYSTEM.LOGMNR_DICTSTATE$', 'RBASQN NUMBER');
    alter_table_add_1_col('SYSTEM.LOGMNR_DICTSTATE$', 'RBABLK NUMBER');
    alter_table_add_1_col('SYSTEM.LOGMNR_DICTSTATE$', 'RBABYTE NUMBER');
    execute immediate '
      update system.logmnr_dictstate$ x
      set x.end_scnbas = x.start_scnbas, x.end_scnwrp = x.start_scnwrp
      where x.end_scnbas = 4294967295 and x.end_scnwrp = 65535';
    commit;
    execute immediate ' ALTER TABLE SYSTEM.LOGMNR_SPILL$
                        MODIFY LOB (SPILL_DATA) (PCTVERSION 0)';
    execute immediate ' ALTER TABLE SYSTEM.LOGMNR_AGE_SPILL$
                        MODIFY LOB (SPILL_DATA) (PCTVERSION 0)';
    execute immediate ' ALTER TABLE SYSTEM.LOGMNR_RESTART_CKPT$
                        MODIFY LOB (client_data) (PCTVERSION 0)';
    execute immediate ' ALTER TABLE SYSTEM.LOGMNR_RESTART_CKPT_TXINFO$
                        MODIFY LOB (TX_DATA) (PCTVERSION 0)';


    /* Backport Bug: 3953131 improve purge_scn performance */
    declare
      tablespacename VARCHAR2(32);
      localStr    VARCHAR2(64) := '';
      PartProperty     NUMBER;
    BEGIN
      SELECT bitand(t.property, 32) INTO PartProperty
      FROM sys.tab$ t, sys.obj$ o, sys.user$ u
      WHERE o.owner# = u.user# and t.obj# = o.obj# and o.name = 'LOGMNRC_GTCS'
            and u.name = 'SYSTEM';
      IF (32 = PartProperty) THEN
        localStr := ' LOCAL ';
      END IF; 
      BEGIN
        select s.name into tablespacename
        from sys.obj$ o, sys.ts$ s, sys.user$ u, sys.tab$ t
        where s.ts# = t.ts# and o.obj# = t.obj# and
              o.owner# = u.user# and u.name = 'SYSTEM' and
              o.name = 'LOGMNR_UID$' and rownum = 1;
      EXCEPTION
        WHEN OTHERS THEN
           tablespacename := 'SYSTEM';
      END;
      execute immediate
        'create index system.logmnrc_i3gtlo on system.logmnrc_gtlo(
            logmnr_uid, drop_scn)
            tablespace ' || tablespacename || localStr || ' LOGGING';

    exception
      when OTHERS THEN
         NULL;
    end;

    /*
     * awb: 21-sep-2004
     * Call to PrepareGTCSforNewKey and the creation of logmnrt_mddl$
     * is added here for ajadams_backport_9.2.0.5.0_3785754.
     * When merging this txn note that this code will be in addition to
     * other new code in this same location (e.g. that has already been
     * added for 9.2.0.6.  This block can follow the other new code.
     */
     /*
     * SYSTEM.LOGMNRT_MDDL$ it a temp table.  Its rows persist only for the
     * duration of the transaction.  Added 10-AUG-2004 to address bug 3785754.
     */
    begin
      execute immediate 'CREATE GLOBAL TEMPORARY TABLE system.logmnrt_mddl$ (
              source_obj#     NUMBER,
              source_rowid    ROWID,
              dest_rowid      ROWID NOT NULL,
                CONSTRAINT logmnrt_mddl$_pk
                  PRIMARY KEY(source_obj#, source_rowid)
              ) on commit delete rows';
    exception
      when LOGMNR_UPGRADE_DUPNAME THEN
         NULL;
    end;
    /*
     * 09/01/04 : To address bug 3785754, before creatring the new primary
     * key we must invoke PrepareGTCSforNewKey which attempts to ensure
     * uniqueness of the new primary key.
     * NOTE: In 9.2 there is no "new primary key".  However,
     * PrepareGTCSforNewKey; has been backported and is invoked because
     * it is part of the mechanism used to detect this bug.
     */
    PrepareGTCSforNewKey;

    RETURN;
  END IF;

  BEGIN

    /* Does this database support the partition option? */
    SELECT value INTO partString FROM v$option
    WHERE LOWER(parameter) = 'partitioning';

    IF LOWER(partString) = 'true' THEN
      partOn := TRUE;
    ELSE
      partOn := FALSE;
    END IF;

  EXCEPTION
    WHEN OTHERS THEN
      partOn := FALSE;
  END;

  /* Create/Replace fixed Logminer meta-data   */

  /*
   * LOGMINER SESSION related immutable objects
   * Always create SYSTEM.LOGMNR_SESSION$ in the system tablespace.
   * LOGMNR_HEADER*$ are never used for data, just for meta-data, so
   * create them only once with system tablespace attribute.
   */
  BEGIN
    EXECUTE IMMEDIATE
      'DROP TABLE SYSTEM.LOGMNR_SESSION$';
  EXCEPTION
  WHEN OTHERS THEN
    IF output_warning THEN 
      DBMS_OUTPUT.PUT_LINE ('Drop Table SYSTEM.LOGMNR_SESSION$  failed');
    END IF;
    NULL;
  END;
  EXECUTE IMMEDIATE 'CREATE TABLE system.logmnr_session$ (
    session#                number,
    client#                 number,
    session_name            varchar2(32),
    db_id                   number,
    resetlogs_change#       number,
    session_attr            number,
    session_attr_verbose    varchar2(400),
    start_scn               number,
    end_scn                 number,
    spill_scn               number,
    resume_scn              number)';

  BEGIN
    EXECUTE IMMEDIATE
      'DROP SEQUENCE system.logmnr_seq$';
  EXCEPTION
  WHEN OTHERS THEN
    IF output_warning THEN 
      DBMS_OUTPUT.PUT_LINE ('Drop SEQUENCE SYSTEM.LOGMNR_SEQ$ failed');
    END IF;
    NULL;
  END;
  EXECUTE IMMEDIATE 'CREATE SEQUENCE system.logmnr_seq$ ORDER';

  BEGIN
    EXECUTE IMMEDIATE
      'DROP TABLE system.logmnr_header1$';
  EXCEPTION
  WHEN OTHERS THEN
     IF output_warning THEN 
        DBMS_OUTPUT.PUT_LINE ('Drop Table SYSTEM.LOGMNR_HEADER1$ failed');
      END IF;
     NULL;
  END;
  EXECUTE IMMEDIATE
    'CREATE TABLE system.logmnr_header1$ (logmnr_uid NUMBER(22))';

  BEGIN
    EXECUTE IMMEDIATE
      'DROP TABLE system.logmnr_header2$';
  EXCEPTION
  WHEN OTHERS THEN
     IF output_warning THEN 
        DBMS_OUTPUT.PUT_LINE ('Drop Table SYSTEM.LOGMNR_HEADER2$ failed');
      END IF;
     NULL;
  END;
  EXECUTE IMMEDIATE
    'CREATE TABLE system.logmnr_header2$ ( objv# NUMBER(22) )';

  BEGIN
    EXECUTE IMMEDIATE
      'DROP SEQUENCE system.logmnr_uids$';
  EXCEPTION
  WHEN OTHERS THEN
    IF output_warning THEN 
        DBMS_OUTPUT.PUT_LINE ('Drop Table SYSTEM.LOGMNR_UIDS$ failed');
    END IF;
    NULL;
  END;
  EXECUTE IMMEDIATE
    'CREATE SEQUENCE system.logmnr_uids$ START WITH 100
     INCREMENT BY 1 NOMAXVALUE ORDER';


  /*
   * Create/Replace all mutable Logminer metadata.  Take default
   * tablespaces (SYSTEM) this time.
   */
  logmnr_create_replace_metadata(partOn, TRUE, 'SYSTEM', 'SYSTEM', TRUE);

  /* 
   * Create the "tabe of interesting" columns for implicit column group logging
   * if it does not exist.
   */
  SELECT COUNT(*) INTO interestingColsTableCnt 
  FROM sys.obj$ o, sys.user$ u 
  WHERE u.name = 'SYS' AND 
        o.owner# = u.user# AND
        o.name = 'LOGMNR_INTERESTING_COLS';

  /*
   * Bug 2172631 : abrown : 8-Jan-02
   *    Each new release of Oracle may include new tables to be
   * tracked in the Logminer Dictionary.  Therefore LOGMNR_INTERESTING_COLS
   * must be reloaded each time this script is run, as it will be on
   * upgrade.
   */
  IF (interestingColsTableCnt <> 0) THEN
    EXECUTE IMMEDIATE  'DROP TABLE SYS.LOGMNR_INTERESTING_COLS';
  END IF;

  EXECUTE IMMEDIATE  'CREATE TABLE SYS.LOGMNR_INTERESTING_COLS AS 
    SELECT  so.obj# obj#, sc.intcol# intcol#, so.name oname , sc.name cname
    FROM sys.obj$ so, sys.obj$ lmo, sys.col$ sc, sys.col$ lmc,
         sys.ccol$ lccol, sys.user$ susr, sys.user$ lusr
   WHERE lusr.name = ''SYSTEM'' AND
         lusr.user# = lmo.owner# AND
         susr.name = ''SYS'' AND
         susr.user# = so.owner# AND
         lmo.type# = 2 AND
         lmo.type# = so.type# AND
         lmo.name = concat(''LOGMNR_'', so.name) AND
         lmo.name not in (''LOGMNR_LOG$'', ''LOGMNR_SPILL$'',
                          ''LOGMNR_SESSION$'', ''LOGMNR_HEADER1$'',
                          ''LOGMNR_HEADER2$'', ''LOGMNR_UID$'',
                          ''LOGMNR_BUILDLOG'', ''LOGMNR_DICTSTATE$'',
                          ''LOGMNR_DICTIONARY$'', ''LOGMNR_PROCESSED_LOG$'',
                          ''LOGMNR_AGE_SPILL$'', ''LOGMNR_RESTART_CKPT$'',
                          ''LOGMNR_RESTART_CKPT_TXINFO$'',
                          ''LOGMNR_INTERESTING_COLS'') AND
         sc.obj# = so.obj# AND
         lmc.obj# = lmo.obj# AND
         lmc.name = sc.name AND
         lccol.obj# = lmc.obj# AND
         lccol.intcol# = lmc.intcol# AND
         lccol.con# =  ( SELECT MIN(con.con#)
                         FROM cdef$ con
                         WHERE con.obj# = lmo.obj# AND con.type# = 2)';

  /*
   * Bug 2104829
   * Ensure that no logminer tables get exported.  Export is not
   * a problem, however import could be a problem if the target
   * database already contained session numbers or uid numbers
   * which were also present in the source.  Also, cross platform
   * export/import could never work.
   * Ignore any error.
   */

   BEGIN
      execute immediate '
        INSERT INTO sys.noexp$
           SELECT u.name, o.name, o.type#
           FROM sys.obj$ o, sys.user$ u
           WHERE (o.type# = 2 or o.type# = 6) AND
                 o.owner# = u.user# AND
                 u.name = ''SYSTEM'' AND
                 o.name like ''LOGMNR%''
         MINUS
           SELECT x.owner, x.name, x.obj_type
           FROM sys.noexp$ x';
     COMMIT;
   EXCEPTION
   WHEN OTHERS THEN
     COMMIT;
  END;
END; /* main unnamed block */
/

-- --------------------------------------------------------------
--
CREATE or REPLACE PACKAGE dbms_logmnr_d AS
--
--    PACKAGE NAME
--      dbms_logmnr_d
--
--    DESCRIPTION
--      This package contains LogMnr Dictionary related procedures.
--      "build" is used to gather the LogMnr dictionary.
--
--      "set_tablespace" is used to alter the default tablespace of
--      Logminer tables.
--
--      BUILD
--      The LogMnr dictionary can be gathered
--      into a flat file (Logminer V1 behavior) or it can be gathered
--      into the redo log stream.
--
--      When creating a Flat File dictionary the procedure queries the
--      dictionary tables of the current database and creates a text based
--      file containing their contents. Each table is represented by
--      "pseudo" SQL statements. A description of the columns in a 
--      table is created by a "CREATE_TABLE" line (one statement for
--      table). It contains the name, datatype and length for each 
--      column. A "INSERT_INTO" statement is created for each row in a 
--      selected table. It contains the values for each row. The file
--      is created in preparation of future analysis of databases
--      log files using the LogMnr tool.
--
--      When gathering the system dictionary into the logstream the procedure
--      queries the dictionary tables inserting the results into a special
--      set of Logminer Gather tables (SYS.LOGMNRG_*).  A side effect of
--      each query is that the resultant inserts cause redo to be generated.
--      Down stream processing can mine this redo to determine the contents
--      of this system's system dictionary at the time this procedure was
--      executed.
--      NOTE:  Database log mode must be "Archive Mode" for this procedure
--             to run
--
--      BUILD INPUTS
--      dictionary_filename - name of the dictionary file
--      dictionary_location - path to file directory
--      options - To explicitly indicate flat file or log stream destination.
-- 
--      BUILD EXAMPLE1
--      Creating a dictionary file as:
--                   /usr/ora/dict.ora
--      Complete syntax, typed all on one line:
--
--      SQL> execute dbms_logmnr_d.build('dict.ora',
--                                       '/usr/ora',
--                                       DBMS_LOGMNR_D.STORE_IN_FLAT_FILE);
--
--      BUILD EXAMPLE2
--      Creating a dictionary file as:
--                   /usr/ora/dict.ora
--      Logminer V1 syntax.
--
--      SQL> execute dbms_logmnr_d.build('dict.ora', '/usr/ora');
--
--      BUILD EXAMPLE3
--      Gathering a dictionary into the log stream
--      Complete syntax, typed all on one line:
--
--      SQL> execute dbms_logmnr_d.build('', '',
--                                          DBMS_LOGMNR_D.STORE_IN_REDO_LOGS);
--
--      BUILD NOTES
--      The dictionary gather should be done after all dictionary
--      changes to a database and prior to the creation of any log
--      files that are to be analyzed.
--
--
--      SET_TABLESPACE
--      SET_TABLESPACE DESCRIPTION
--      By default all Logminer tables are created to use the SYSTEM
--      tablespace.  All users will find it desirable to alter Logminer
--      tables to employ an alternate tablespace.  Use this routine to
--      recreate all Logminer tables in an alternate tablespace.
--
--      SET_TABLESPACE INPUTS
--      new_tablespace         - a string naming a preexistant tablespace.
--                               Supply only this parameter to recreate all
--                               Logminer tables to employ this tablesapce.
--      dictionary_tablespace  - a string naming a preexistant tablespace.
--                               Supply this parameter when it is desired
--                               to place Logminer Dictionary data in a
--                               tablespace different from that where
--                               Logminer spill data is to be written.
--                               This parameter overrides the new_tablespace
--                               parameter with respect to Logminer Dictionary
--                               tables.
--      spill_tablespace       - a string naming a preexistant tablespace.
--                               Supply this parameter when it is desired
--                               to place Logminer spill data in a
--                               tablespace different from that where
--                               Logminer Dictionary data is to be written.
--                               This parameter overrides the new_tablespace
--                               parameter with respect to Logminer spill
--                               tables.
--      SET_TABLESPACE NOTES
--      1. There can be no Logminer sessions at the time this procedure
--         is run.  If the Logminer AdHoc tool has been terminated
--         abnormally it may be necessary to shutdown and re startup
--         Oracle before running this procedure.
--      2. Though the intent is that this routine is to be run exactly
--         once to configure Logminer for use by other products, it can
--         be run multiple times should it be necessary to redefine the
--         tablespaces that are to be employed.  However, rule 1 is still
--         enforced.  Because the techniques required to force layered products
--         to terminate their Logminer sessions may be non-trivial, Oracle
--         does not recommend that this routine be used more then once.
--      3. Certain layered products require that this routine be used to
--         alter the tablespace of all Logminer tables before the layered
--         product will operate.
--      4. Certain performance optimizations can be made when Logminer tables
--         do not employ the SYSTEM tablespace.  Specifically, certain
--         easily repeatable operations, such as memory spill, Logminer
--         Dictionary load and index creation, will not be logged.
--         This would have unacceptable implications with respect to the SYSTEM
--         tablespace in the event of a database recover.
--      5. User of this routine must supply an existent tablespace.
--         Information about tablespaces and how to create them is available
--         in "Oracle9i Concepts" manual and the "Oracle 9i SQL Reference"
--         manual.  A sample command for creating an alternate tablespace
--
--        create tablespace logmnrts$ datafile '/usr/oracle/dbs/logmnrts'
--             size 25 M reuse autoextend on maxsize unlimited;
--
--      6. Sample Usage:
--
--         execute dbms_logmnr_d.set_tablespace('logmnrts$');

STORE_IN_FLAT_FILE CONSTANT INTEGER := 1;
STORE_IN_REDO_LOGS CONSTANT INTEGER := 2;

PROCEDURE  build
		(dictionary_filename IN VARCHAR2 DEFAULT '',
		 dictionary_location IN VARCHAR2 DEFAULT '',
                 options IN NUMBER DEFAULT 0);
--
--
PROCEDURE set_tablespace( new_tablespace IN VARCHAR2 DEFAULT '',
                          dictionary_tablespace IN VARCHAR2 DEFAULT '',
                          spill_tablespace IN VARCHAR2 DEFAULT '');
--
END dbms_logmnr_d; -- End Definition of package
/

