create or replace package dbms_rcvman IS

----------------------------------------
-- PUBLIC VARIABLES AND TYPES SECTION --
----------------------------------------

TRUE#  CONSTANT number := 1;
FALSE# CONSTANT number := 0;

-- The values here must never be changed, because the 8.0 rman executables have
-- these values hard-coded in the krmkbt enum in krmk.h.  The setFrom procedure
-- in particular is using hard-coded values.

-- The 8.1.5 rman executable calls a procedure, set_package_constants, that
-- re-assigns these constants to whatever the package needs them to be, then
-- queries the package for their new values.  The 8.1.5 rman does not care
-- what the values are, however, the cursor used by reportGetDFDel used to use
-- these values to perform an order-by to return rows in preference order.
-- The preference order is used to decide which ones to delete.

-- As of 8.1.6, the order-by in reportGetDFDel is independant of these values.
-- The 8.1.6 rman does not use these values at all, except in setFrom.
-- However, for backwards compatibility with the 8.1.5 RMAN, these must remain
-- as public package constants.

COPY                  number := 1; -- any image copy of a file
FULL_DF_BACKUP        number := 2; -- datafile in a full backup set
INCREMENTAL_DF_BACKUP number := 3; -- datafile in an incr backup set
BACKUP                number := 4; -- any file in a backup set
OFFLINE_RANGE         number := 5; -- an offline range
CUMULATIVE            number := 6; -- cumulative incremental - for LIST only
PROXY                 number := 7; -- any proxy copy of a file

-- Recovery Action Kinds (Obsolete as of 8.1.6) --

implicitOfflRange CONSTANT NUMBER := 2**0;
cleanRange        CONSTANT NUMBER := 2**1;
applyOfflRange    CONSTANT NUMBER := 2**2;
dfCopy            CONSTANT NUMBER := 2**3;
proxyFull         CONSTANT NUMBER := 2**4;
buSet             CONSTANT NUMBER := 2**5;
applyIncremental  CONSTANT NUMBER := 2**6;
redo              CONSTANT NUMBER := 2**7;

-- kind masks
maxKind           CONSTANT NUMBER := redo;            -- last real kind above
allKind           CONSTANT NUMBER := (maxKind*2) - 1; -- all real backup types
fullKind          CONSTANT NUMBER := dfCopy + proxyFull + buSet;
tagKind           CONSTANT NUMBER := fullKind + applyIncremental;

-- pseudo kinds
deletedKind       CONSTANT NUMBER := maxKind*2;      -- action deleted

----------------------------------
-- Backupset Availability Masks --
----------------------------------

BSavailable     CONSTANT BINARY_INTEGER := 2**0;
BSunavailable   CONSTANT BINARY_INTEGER := 2**1;
BSdeleted       CONSTANT BINARY_INTEGER := 2**2;
BSexpired       CONSTANT BINARY_INTEGER := 2**3;
-- BSpartial_avail is a backupset validation mask and NOT a backuppiece
-- filter.  For eg. to get 'A', 'U', 'X' pieces and to enable validation
-- to succeed for partially available backupset use
-- BSpartial_avail + BSavailable + BSunavailable + BSexpired.
BSpartial_avail CONSTANT BINARY_INTEGER := 2**4;


----------------------
-- BackupType Mask ---
----------------------
BSdatafile_full  CONSTANT BINARY_INTEGER := 2**0;
BSdatafile_incr  CONSTANT BINARY_INTEGER := 2**1;
BSarchivelog     CONSTANT BINARY_INTEGER := 2**2;

---------------------------
-- ControlfileType Mask ---
---------------------------
BScfile_all      CONSTANT BINARY_INTEGER := 2**0;      -- shouldn't be altered
BScfile_auto     CONSTANT BINARY_INTEGER := 2**1;

---------------------
-- Datafile Record --
---------------------

TYPE dfRec_t IS RECORD
(
   dfNumber             number,
   dfCreationSCN        number,
   dfCreationTime       date,
   fileName             varchar2(1024),
   tsName               varchar2(30),
   tsNumber             number,
   status               number,
   blocks               number,
   blockSize            number,
   kbytes               number,
   unrecovSCN           number,
   stopSCN              number,
   readOnly             number,
   rfNumber             number,
   inBackup             number,     -- if greater than 0 then
                                    -- included_in_database_backup is set
   auxName              varchar2(1024),
   dbincKey             number
);

-------------------------
-- Archived Log Record --
-------------------------

TYPE alRec_t IS RECORD
(
   key                  number,
   recid                number,
   stamp                number,
   thread               number,
   sequence             number,
   fileName             varchar2(1024),
   lowSCN               number,
   lowTime              date,
   nextSCN              number,
   nextTime             date,
   rlgSCN               number,
   rlgTime              date,
   blocks               number,
   blockSize            number,
   status               char(1),
   compTime             date,
   duplicate            number
);

-- All of the queries which return data about a backup/imagecopy/proxycopy
-- select into a rcvRec_t record type.  We have standardized all of our
-- queries to have a common select-list and the results of the queries are
-- returned through a common public package function.  The reason for this is
-- so that krmk.pc can populate its internal data structures consistantly,
-- regardless of what particular procedure it has called to query the catalog.

-- By having all queries select into the same record type, we can ensure
-- that all queries use the same select list.  Any new fields that get added
-- to this record will require updating the select lists of all queries.
-- Failure to make the correct updates will result in PLSQL giving an error
-- when the package body is re-created, so the error will be easily detected
-- without the need to run any test suite.

-- The record is divided into three sections.  These correpond to
-- three krmk.h data structures which will be populated with the data
-- from this record.  Refer to krmk.h for a description of the purpose
-- of each of these three data strucutres.

-- Think of this as:  the container acts on the object.

---------------------
-- Recovery Record --
---------------------

TYPE rcvRec_t IS RECORD
(
   -- *** Recovery Container Section ***

   type_con             number(3),      -- recovery container type
   key_con              number(15),     -- primary key
   recid_con            number(10),     -- recid
   stamp_con            number(10),     -- stamp
   setStamp_con         number(10),     -- set count if backup set (null)
   setCount_con         number(10),     -- set stamp if backup set (null)
   bsRecid_con          number(10),     -- backup set recid (null)
   bsStamp_con          number(10),     -- backup set stamp (null)
   bsKey_con            number(15),     -- backup set key (null)
   bsLevel_con          number(1),      -- backup set level (null)
   bsType_con           char(1),        -- backup set type
   elapseSecs_con       number(10),     -- backup set elapse seconds (null)
   pieceCount_con       number(5),      -- backup set piece count (null)
   fileName_con         varchar2(1024), -- filename if a copy (or) piece (null)
   tag_con              varchar2(32),   -- tag (null)
                                        -- filled in by addAction() for
                                        -- backup sets
   copyNumber_con       number(3),      -- backup set copy# (null) maxlimit 256
                                        -- filled in by addAction() only
   status_con           char(1),        -- status (null)
   blocks_con           number(10),     -- size of file in blocks (null)
   blockSize_con        number(5),      -- block size (null)
   deviceType_con       varchar2(255),  -- device type required (null)
                                        -- filled in by addAction() for
                                        -- backup sets
   compTime_con         date,           -- completion time
   cfCreationTime_con   date,           -- controlfile creation time if
                                        -- offline range (null)
   pieceNumber_con      number,
   bpCompTime_con       date,

   -- *** Recovery Action Section ***

   type_act             number(2),      -- recovery action type
   fromSCN_act          number(15),
   toSCN_act            number(15),
   toTime_act           date,
   rlgSCN_act           number(15),
   rlgTime_act          date,
   dbincKey_act         number(15),
   level_act            number(1),

   -- *** Recovery Object Section ***

   dfNumber_obj         number(6),
   dfCreationSCN_obj    number(15),
   cfSequence_obj       number(15),    -- controlfile autobackup sequence
   cfDate_obj           date,          -- controlfile autobackup date
   logSequence_obj      number(10),
   logThread_obj        number(4),
   logRlgSCN_obj        number(15),
   logRlgTime_obj       date,
   logLowSCN_obj        number(15),
   logLowTime_obj       date,
   logNextSCN_obj       number(15),
   logNextTime_obj      date,

   -- *** Retention Policy Section ***
   keep_options         number(4),
   keep_until           date,

   -- *** Optimization Action Section ***

   afzSCN_act           number(15),
   rfzTime_act          date,
   rfzSCN_act           number(15)
);

------------------------------
-- Recovery Container Types --
------------------------------

-- NOTE:  Order is important, it is used in an ORDER BY.

offlineRangeRec_con_t   CONSTANT NUMBER := 2**0;
proxyCopy_con_t         CONSTANT NUMBER := 2**1;
imageCopy_con_t         CONSTANT NUMBER := 2**2;
backupSet_con_t         CONSTANT NUMBER := 2**3;
deleted_con_t           CONSTANT NUMBER := 2**8;

-- Masks
backupMask_con_t        CONSTANT NUMBER := proxyCopy_con_t + imageCopy_con_t +
                                           backupSet_con_t;
tagMask_con_t           CONSTANT NUMBER := proxyCopy_con_t + imageCopy_con_t +
                                           backupSet_con_t;

---------------------------
-- Recovery Action Types --
---------------------------

full_act_t              CONSTANT NUMBER := 2**0;
incremental_act_t       CONSTANT NUMBER := 2**1;
redo_act_t              CONSTANT NUMBER := 2**2;
offlineRange_act_t      CONSTANT NUMBER := 2**3;
cleanRange_act_t        CONSTANT NUMBER := 2**4;
implicitRange_act_t     CONSTANT NUMBER := 2**5;
spanningRange_act_t     CONSTANT NUMBER := 2**6;

-----------------------------------------
-- Recovery Record Returning Functions --
-----------------------------------------

-- These defines are used as the funCode arg to getRcvRec to tell it which
-- function it should call.  We do this so that krmk.pc can have a single
-- interface routine for getting a rcvRec_t.

getCfCopy               CONSTANT NUMBER := 0;
getDfCopy               CONSTANT NUMBER := 1;
getAnyProxy             CONSTANT NUMBER := 2;
getCfBackup             CONSTANT NUMBER := 3;
listCfCopy              CONSTANT NUMBER := 4;
listDfCopy              CONSTANT NUMBER := 5;
listCfBackup            CONSTANT NUMBER := 6;
listDfBackup            CONSTANT NUMBER := 7;
listAlBackup            CONSTANT NUMBER := 8;
listDfProxy             CONSTANT NUMBER := 9;
getRecovAction          CONSTANT NUMBER := 10;
getAlBackup             CONSTANT NUMBER := 11;
listAlCopy              CONSTANT NUMBER := 12;
listBSet                CONSTANT NUMBER := 13;
getSfBackup             CONSTANT NUMBER := 14;
listSfBackup            CONSTANT NUMBER := 15;
getAllBSet              CONSTANT NUMBER := 16;

-----------------------
-- Backup Set Record --
-----------------------

TYPE bsRec_t IS RECORD
(
   recid                number,
   stamp                number,
   key                  number,
   setStamp             number,
   setCount             number,
   bsType               char(1),
   level                number,
   elapseSecs           number,
   compTime             date,
   status               char(1),
   pieceCount           number,
   keep_options         number(4),
   keep_until           date
);

------------------------
-- Backup Piece Record --
-------------------------

TYPE bpRec_t IS RECORD
(
   recid                number,
   stamp                number,
   key                  number,
   bskey                number,
   setStamp             number,
   setCount             number,
   pieceNumber          number,
   copyNumber           number,
   status               char(1),
   compTime             date,
   handle               varchar2(1024),
   tag                  varchar2(32),
   deviceType           varchar2(255)
);

---------------------------------
-- Backupset Validation Record --
---------------------------------

TYPE validBackupSetRec_t IS RECORD
(
   deviceType   varchar2(255),
   tag          varchar2(32),                   -- may be null
   copyNumber   number,                         -- null if code 2 or 3
   code         number                          -- 1 => same copy#
                                                -- 2 => mix of copy#s, but
                                                --      same tag
                                                -- 3 => mix of copy#s and tags
);

--------------------
-- backup history --
--------------------
TYPE bhistoryRec_t IS RECORD
(
   key             number,
   device_type     varchar2(255),
   dfNumber        number,
   create_scn      number,
   reset_scn       number,
   reset_time      date,
   ckp_scn         number,
   stop_scn        number,
   logThread       number,
   logSequence     number,
   setStamp        number,
   setCount        number,
   compTime        date,
   nbackups        number
);



-----------------------------------------------------
-- PUBLIC FUNCTION/PROCEDURE SPECIFICATION SECTION --
-----------------------------------------------------

----------------------------------------
-- Debugging functions and procedures --
----------------------------------------

FUNCTION dumpState(
   lineno IN number)
RETURN varchar2;

PROCEDURE setDebugOn;

PROCEDURE setDebugOff;

----------------------------
-- Package Initialization --
----------------------------

-- This is a vestigal function that was released to customers in 8.1.3 Beta.
-- It is no longer called, and is no longer needed, but must still be here
-- because this version of the package may be called by an 8.1.3 rman
-- executable.

PROCEDURE initialize(rman_vsn IN number);

-- Used by 8.1.5 to re-assign the order of the backup_type constants to their
-- correct order.  This procedure is not called by 8.1.4-, so the constants
-- will reamin set to the above values for those executables.

PROCEDURE set_package_constants;

-----------------------
-- Utility functions --
-----------------------

FUNCTION stamp2date(stamp IN number) RETURN date;

------------------------------
-- Set Database Incarnation --
------------------------------

-- setDatabase selects which target database subsequent dbms_rcvman
-- procedures operate on. Note that only the current incarnation can be
-- selected. If the target database or its current incarnation is not
-- registered then setDatabase will fail.
-- setDatabase sets the package state variables to point to the selected
-- database and its current incarnation.
-- The settings will be valid until the end of the session unless setDatabase
-- is called again

-- When dbms_rcvman package executes against the target database controlfile,
-- setDatabase just returns without doing anything.

-- Input parameters:
--   db_id
--     the value of kccfhdbi from the controlfile of the target database
--   db_name
--     the name of the database
--   reset_scn
--     the resetlogs SCN of this database
--   reset_time
--     the resetlogs time
-- Exceptions:
--   DATABASE_NOT_FOUND (ORA-20001)
--     No database with the given db_id was found in the recovery catalog
--     The database must be registered using registerDatabase first
--   DATABASE_INCARNATION_NOT_FOUND (ORA-20003)
--     No database incarnation matches the given arguments
--     The database incarnation must be registered using resetDatabase first

PROCEDURE setDatabase(
   db_name    IN varchar2
  ,reset_scn  IN number
  ,reset_time IN date
  ,db_id      IN number);

-- setDbincKey used in lieu of setDatabase for when SET DBID command is
-- issued.

PROCEDURE setDbincKey(
   key IN number);

-- getParent Incarnation returns the parent incarnation.  If resetlogs_change#
-- is NULL on input, then the current incarnation is returned.  Returns TRUE
-- if a row was returned, otherwise returns FALSE.

FUNCTION getParentIncarnation(
   resetlogs_change# IN OUT number
  ,resetlogs_time    IN OUT date)
RETURN number;

-- getCheckpoint gets and returns the highest recovery catalog checkpoint SCN
-- for FULL checkpoints.  This SCN indicates how current the datafilenames and
-- lognames in the recovery catalog are.  This SCN can be compared with a
-- backup controlfile SCN to decide which name to use if they differ.

PROCEDURE getCheckpoint(
   scn OUT number
  ,seq OUT number);

-- This version of getCheckpoint is only used internally by
-- dbms_rcvcat.cleanupCKP, to find out which rows can't be deleted from ckp.

PROCEDURE getCheckpoint(
   scn       OUT number
  ,seq       OUT number
  ,ckp_key_1 OUT number
  ,ckp_key_2 OUT number);

-------------------
-- Query Filters --
-------------------

-- setCompletedRange sets completedBefore and/or completedAfter filters for
-- use by computeRecoveryActions.
-- setLikePattern sets fileName patter for computeRecoveryActions.
--
-- setUntilTime, setUntilScn, setUntilLog, resetUntil These procedures are
-- used to inform dbms_rcvman of an until_clause.  The setUntil remains in
-- effect until another setUntil has been called, or until resetUntil has been
-- called.  If none of these have been called, then all queries for name
-- translation, restore, and recovery should assume that a complete recovery
-- is being done.  Otherwise, all restore and recovery queries should limit
-- their replies to backup sets and datafile copies that are appropriate for
-- use in an incomplete recovery until the specified until condition.  Name
-- translations should be done relative to the specified epoch.
--
-- "appropriate" means that the fuzziness of the backup datafile or datafile
-- copy ends at an SCN less than the untilChange SCN (for untilChange), or the
-- low SCN of the specified log (for untilLog), or the fuzziness timestamp is
-- less than the specified time (for unttime).  Note that datafiles have three
-- kinds of fuzziness, all of which must be less than the specified SCN or
-- time.  If the fuzziness of a datafile is unknown, then it should be
-- ignored.
--
-- The setUntil procedures will signal an error when executed against
-- the target database controlfile. The resetUntil procedure can be
-- executed against the controlfile, it but doesn't have any effect.

-- Input parameters:
--   unttime
--     The incomplete recovery will stop when this timestamp is reached
--     in the redo log.
--   scn
--     The incomplete recovery will stop when this scn is reached in the redo
--     log.
--   sequence#, thread#
--     The incomplete recovery will stop when this log becomes the very next
--     log to be applied.
--
-- Exceptions:
--   NO_RECOVERY_CATALOG (ORA-20300)
--     this operation is not supported without the recovery catalog
--   SEQUENCE_IS_NULL (ORA-20205)
--     A null log sequence# was given
--   LOG_MISSING (ORA-20206)
--     No log with the give thread# and sequence# was found

PROCEDURE setCompletedRange(
   after  IN date
  ,before IN date);

PROCEDURE setLikePattern(
   pattern IN varchar2);

-- Obsolete as of 8.1.6
PROCEDURE setAllFlag(
   flag IN boolean);

PROCEDURE setAllIncarnations(
   flag IN boolean);

PROCEDURE setUntilTime(
   unttime IN date);

PROCEDURE setUntilScn(
   scn IN number);

PROCEDURE setUntilLog(
   sequence# IN number
  ,thread#   IN number);

FUNCTION getUntilTime return date;

FUNCTION getUntilScn return number;

PROCEDURE resetUntil;

-- setFrom is used to limit the potential restore candidates to either
-- backup sets or datafile copies, or to allow either kind of file to
-- be used.
--
-- Input parameters:
--   restorefrom
--     One of BACKUP, COPY, or NULL.

PROCEDURE setFrom(
   restorefrom IN number DEFAULT NULL);

-- setDeviceType specifies the type of an allocated device.  It is called 1 or
-- more (up to 8) times , depending on the number of different device types
-- that are allocated.  dbms_rcvman should return only files that can be
-- accessed through one of the device types specifed through this call.
--
-- Input parameters:
--   type
--     type of the device
-- Exceptions:
--   NULL_DEVICE_TYPE
--     A null device type was specied
--   TOO_MANY_DEVICE_TYPES
--     At most 8 device types can be specified

PROCEDURE setDeviceType(
   type IN varchar2);

-- setDeviceTypeAny is an alternative to setDeviceType.  It causes dbms_rcvman
-- to return a backup set on ANY device type.

PROCEDURE setStandby(
   stby IN boolean);

PROCEDURE setDeviceTypeAny;

-- resetDeviceType resets the list of device types to null.

PROCEDURE resetDeviceType;

-- setTag is used to limit the restore candidates to backups and copies with
-- the given tag. If the tag is NULL then all backups and copies are searched
-- by the find functions.
--
-- Input parameters:
--   tag
--     tag of the datafile copies to be translated
--     The tag must be passed in uppercase ### ok?

PROCEDURE setTag(tag IN varchar2 DEFAULT NULL);

-- resetAll calls resetUntil, setFrom, resetDevice and setTag to reset
-- everything.

PROCEDURE resetAll;

---------------------------
-- Backup Set Validation --
---------------------------

-- Use the findValidBackupSetRec public variable to save a backupset record
-- for later use as an input argument to this procedure.

findValidBackupSetRcvRec rcvRec_t;              -- place to save a rcvRec_t

PROCEDURE findValidBackupSet(
   backupSetRec            IN     rcvRec_t
  ,deviceType              IN     varchar2       DEFAULT NULL
  ,tag                     IN     varchar2       DEFAULT NULL
  ,available               IN     number         DEFAULT TRUE#  -- for compat.
  ,unavailable             IN     number         DEFAULT FALSE# -- for compat.
  ,deleted                 IN     number         DEFAULT FALSE# -- for compat.
  ,expired                 IN     number         DEFAULT FALSE# -- for compat.
  ,availableMask           IN     binary_integer DEFAULT NULL); -- for compat.

findValidBackupSetBsRec  bsRec_t;               -- place to save a bsRec_t

-- Obsolete as of 8.1.7
PROCEDURE findValidBackupSet(
   backupSetRec            IN     bsRec_t
  ,deviceType              IN     varchar2       DEFAULT NULL
  ,tag                     IN     varchar2       DEFAULT NULL
  ,available               IN     number         DEFAULT TRUE#  -- for compat.
  ,unavailable             IN     number         DEFAULT FALSE# -- for compat.
  ,deleted                 IN     number         DEFAULT FALSE# -- for compat.
  ,expired                 IN     number         DEFAULT FALSE# -- for compat.
  ,availableMask           IN     binary_integer DEFAULT NULL); -- for compat.

FUNCTION getValidBackupSet(
   validBackupSetRec            OUT validBackupSetRec_t
  ,checkDeviceIsAllocated       IN  number DEFAULT FALSE#)
RETURN number;                                  -- TRUE# -> got a record
                                                -- FALSE# -> no_data_found

---------------------
-- Get an rcvRec_t --
---------------------

-- This function is a cover function for all procedures/functions that
-- return a rcvRec_t.  It routes the call to the correct procedure.  It
-- is provided for the convienence of krmk.pc.  The function return value
-- is whatever the underlying function returns.  If we call a procedure,
-- then getRcvRec returns 0.  Refer to the funCode list above in the
-- types/variables section.

FUNCTION getRcvRec(
   funCode      IN number
  ,rcvRec       OUT rcvRec_t
  ,callAgain    OUT number)
RETURN number;

--------------------------
-- Datafile Translation --
--------------------------

-- translateTableSpace translates a tablespace name into a list of datafile
-- numbers.  translateDataBase translates the database into a list of datafile
-- numbers in the database excluding datafiles belonging to tablespaces
-- specified using skipTableSpace.  The translation is performed relative to
-- epoch setting currently in use.  getDataFile is used to obtain the datafile
-- numbers, one at a time until null is returned.

-- When doing the translation relative to current time the client should
-- ensure that recovery catalog is up-to-date. When doing translations
-- relative to an point-in-time in the past two potential anomalies may
-- show up.
--
-- 1) files belonging to a tablespace that was dropped before the point-in-time
-- may be returned since the drop_scn and drop_time are approximations.
-- As a result of this point-in-time recovery will restore and recover
-- a tablespace which will be dropped before the database is opened.
-- No real harm, just extra work for the recovery. And this won't happen
-- if rcvcat is resynced immediatly after dropping a tablespace.
-- 2) A tablespace which is created and dropped between two consecutive
-- recovery catalog resyncs will never be recorded in the rcvcat. It is
-- conceivable that such a tablespace existed at the intended point-in-time.
-- As a result the tablespace will not be recovered and must be dropped
-- after the database is opened. The worst case scenario is that a rollback
-- segment was also created in this tablespace. The recovered database
-- might fail to rollback some transactions. Again, this won't happen if
-- rcvcat is always resynced after creating a tablespace.
-- PS. These anomalies won't occur if the point-in-time is chosen to be
-- a rcvcat checkpoint.

-- Input parameters:
--   ts_name
--     name of the tablespace to be translated or skipped.
--     The name must be in uppercase
-- Exceptions:
--   TABLESPACE_DOES_NOT_EXIST (ORA-20202)
--     the tablespace to be translated does not exists (does not have any
--     datafiles). Check that the recovery catalog is current.
--   TRANSLATION_IN_PROGRESS (ORA-20203)
--     the previous translation conversation is still in progess.
--     To terminate get all datafiles with getDataFile.
--   TRANSLATION_NOT_IN_PROGRESS (ORA-20204)
--     getDataFile was called with no translation in progress

PROCEDURE translateDatabase(
   sinceUntilSCN IN number DEFAULT NULL);

PROCEDURE skipTableSpace(
   ts_name IN varchar2);

PROCEDURE translateTablespace(
   ts_name IN varchar2);

-- translateDataFile translates the datafile name/number into
-- a datafile number and creation SCN and filename.  getDataFile must
-- be called to obtain the translation info, just as for the other translate
-- functions.
-- Unlike the other translation functions, translateDatafile by name is always
-- performed relative to current time.  If an until setting is in effect,
-- and if the filename is ambiguous, then an exception is raised.  Ambiguous
-- means that the filename refers to different datafile at the until time than
-- it does at the current time.  This happens only when a filename has been
-- reused.  When fno and ckpscn are passed, the filename and other info as of
-- that scn is returned.

-- Input parameters:
--   fname
--     name of the datafile to be translated.
--     The name must be a normalized filename.
--   fno
--     The datafile number.  If the datafile number was not in use at the
--     until time, then an exception is raised.
-- Exceptions:
--   DATAFILE_DOES_NOT_EXIST (ORA-20201)
--     the datafile to be translated does not exists
--     Check that the recovery catalog is current.

PROCEDURE translateDataFile(
   fname IN varchar2);

PROCEDURE translateDatafile(
   fno   IN number);

PROCEDURE translateDatafile(
   fno    IN number
  ,ckpscn IN number);

-- translateAllDatafile returns a list of all datafiles that ever
-- existed in this database.

PROCEDURE translateAllDatafile;

PROCEDURE translateCorruptList;

-- Bug 2391697
PROCEDURE translateDatafileCancel;

PROCEDURE getDatafile(
   dfRec     OUT dfRec_t
  ,oldClient IN  boolean DEFAULT FALSE);

-- Obsolete as of 8.1.6
PROCEDURE getDataFile(
   file#        OUT number
  ,crescn       OUT number
  ,creation_time OUT date
  ,fname        OUT varchar2
  ,ts_name      OUT varchar2
  ,status       OUT number
  ,blksize      OUT number
  ,kbytes       OUT number
  ,blocks       OUT number
  ,unrecoverable_change# OUT number
  ,stop_change# OUT number
  ,read_only    OUT number);

----------------------------
-- Online Log Translation --
----------------------------

-- translateOnlineLogs translates the database to a list of online redo logs.
-- The translation is always performed relative to current epoch.

PROCEDURE translateOnlineLogs;

PROCEDURE getOnlineLog(
   fname        OUT varchar2
  ,thread#      OUT number
  ,group#       OUT number);

-----------------------------
-- Archivedlog Translation --
-----------------------------

-- translateArchivedLogKey translates the archived log key to a archived
-- log recid and stamp in V$ARCHIVED_LOG.

-- translateArchivedLogRange* procedures translate a specified
-- archive log range to a list of archived logs.
-- getArchivedLog is used to get the recid and stamp for each archived log,
-- one at a time until null is returned.

-- The available, unavailable and deleted parameters are used to limit
-- the translation to archived logs with the desired status. For example,
-- only available archived logs can be backed up, but unavailable and deleted
-- archived logs can be restored from backups.

-- The duplicates parameter controls whether the translation returns all
-- archived logs or eliminates duplicate ones. Archived logs that have the
-- same thread#, sequence# and low_scn are considered duplicates. (duplicate
-- archived logs are usually created by copying archived logs).

-- Note that only archived logs recorded in the recovery catalog or
-- controlfile file are returned. If there is an archived log that belongs
-- to the range but is not known, there will be a "hole" in the range.

-- Input parameters:
--    al_key
--      key of the archived log record in the recovery catalog
--   thread#
--     return only logs that belong to this thread#
--     if NULL return logs for all threads
--   fromseq#
--     lowest log sequence number in the range
--   toseq#
--     highest log sequence number in the range
--   fromtime
--     exclude logs that were switched out before fromtime
--   totime
--     exclude logs that were switched in after totime
--   fromscn
--     exclude logs that were switched out before fromscn
--   toscn
--     exclude logs that were switched in after toscn
--   pattern
--     return only archived logs whose filename match the pattern
--     The pattern is matched against normalized filenames ### ok?
--   available
--     if TRUE (1) return available archived logs
--   unavailable
--     if TRUE (1) return unavailable archived logs
--   deleted
--     if TRUE (1) return deleted archived logs
--   online
--     if TRUE (1) return also inspected online logs (in addition to
--     archived logs)
--   duplicates
--     if TRUE (1) return all archived logs
--     if FALSE (0) eliminate duplicate archived logs
-- Output parameters:
--    recid
--      recid of the archived log record (in V$ARCHIVED_LOG)
--    stamp
--      stamp of the archived log record (in V$ARCHIVED_LOG)
--    thread#
--    sequence#
--    low_scn
--    fname
--    reset_scn
--    block_size
-- Exceptions:
--  NO_RECOVERY_CATALOG (ORA-20300)
--    this operation is not supported without the recovery catalog
--  ARCHIVED_LOG_DOES_NOT_EXIST
--    the key does not match any archived log
--   TRANSLATION_IN_PROGRESS (ORA-20203)
--     the previous translation conversation is still in progess.
--     To terminate get all datafiles with getArchivedLog.
--   TRANSLATION_NOT_IN_PROGRESS (ORA-20204)
--     getArchivedLog was called with no translation in progress
--   THREAD_IS_NULL (ORA-20210)
--     a null thread# was passed to translateArchivedLogSeqRange
--   HIGH_SEQUENCE_IS_NULL
--     a null toseq# was passed to translateArchivedLogSeqRange
--   UNTIL_TIME_IS_NULL (ORA-20212)
--     a null totime was passed to translateArchivedLogTimeRange
--   UNTIL_SCN_IS_NULL (ORA-20213)
--     a null toscn was passed to translateArchivedLogSCNRange
--   ARCHIVED_LOG_RANGE_IS_EMPTY
--     the specified range doesn't contain any archived log

------------------------------
-- Archived Log Translation --
------------------------------

PROCEDURE getArchivedLog(
   alRec       OUT alRec_t
  ,closeCursor IN  boolean DEFAULT FALSE);

PROCEDURE translateArchivedLogKey(
   al_key       IN  number
  ,available    IN  number       DEFAULT 1 -- ignored (for compatability)
  ,unavailable  IN  number       DEFAULT 1 -- ignored (for compatability)
  ,deleted      IN  number       DEFAULT 1 -- ignored (for compatability)
  ,online       IN  number       DEFAULT 1 -- ignored (for compatability)
  ,recid        OUT number
  ,stamp        OUT number
  ,thread#      OUT number
  ,sequence#    OUT number
  ,low_scn      OUT number
  ,reset_scn    OUT number
  ,block_size   OUT number
  ,fname        OUT varchar2
  ,needstby     IN number        DEFAULT NULL);

PROCEDURE translateArchivedLogName(
   fname        IN varchar2
  ,available    IN number         DEFAULT NULL   -- for compatability
  ,unavailable  IN number         DEFAULT NULL   -- for compatability
  ,deleted      IN number         DEFAULT NULL   -- for compatability
  ,online       IN number                        -- ignored
  ,duplicates   IN number
  ,statusMask   IN binary_integer DEFAULT NULL   -- for compatability
  ,needstby     IN number         DEFAULT NULL);

PROCEDURE translateArchivedLogSeqRange(
   thread#      IN number
  ,fromseq#     IN number
  ,toseq#       IN number
  ,pattern      IN varchar2
  ,available    IN number         DEFAULT NULL   -- for compatability
  ,unavailable  IN number         DEFAULT NULL   -- for compatability
  ,deleted      IN number         DEFAULT NULL   -- for compatability
  ,online       IN number                        -- ignored
  ,duplicates   IN number
  ,statusMask   IN binary_integer DEFAULT NULL   -- for compatability
  ,needstby     IN number         DEFAULT NULL);

PROCEDURE translateArchivedLogTimeRange(
   thread#      IN number
  ,fromTime     IN date
  ,toTime       IN date
  ,pattern      IN varchar2
  ,available    IN number         DEFAULT NULL   -- for compatability
  ,unavailable  IN number         DEFAULT NULL   -- for compatability
  ,deleted      IN number         DEFAULT NULL   -- for compatability
  ,online       IN number                        -- ignored
  ,duplicates   IN number
  ,statusMask   IN binary_integer DEFAULT NULL   -- for compatability
  ,needstby     IN number         DEFAULT NULL);

PROCEDURE translateArchivedLogSCNRange(
   thread#      IN number
  ,fromSCN      IN number
  ,toSCN        IN number
  ,pattern      IN varchar2
  ,available    IN number         DEFAULT NULL   -- for compatability
  ,unavailable  IN number         DEFAULT NULL   -- for compatability
  ,deleted      IN number         DEFAULT NULL   -- for compatability
  ,online       IN number
  ,duplicates   IN number
  ,statusMask   IN binary_integer DEFAULT NULL   -- for compatability
  ,needstby     IN number         DEFAULT NULL);

PROCEDURE translateArchivedLogPattern(
   pattern      IN varchar2
  ,available    IN number         DEFAULT NULL   -- for compatability
  ,unavailable  IN number         DEFAULT NULL   -- for compatability
  ,deleted      IN number         DEFAULT NULL   -- for compatability
  ,online       IN number                        -- ignored
  ,duplicates   IN number
  ,statusMask   IN binary_integer DEFAULT NULL   -- for compatability
  ,needstby     IN number         DEFAULT NULL);

PROCEDURE translateArchivedLogCancel;

-- translateBackupPieceKey looks up a backup piece by primary key.
-- translateBackupPieceHandle looks up a backup piece by handle and deviceType.
-- translatebackupPieceTag looks up backup pieces by tag.

-- The available are unavailable parameters are used to limit the translation
-- to backup pieces with the desired status. For example, only available
-- backup pieces can be backed up, but unavailable pieces can be made
-- available.  Deleted backup pieces are never returned.

-- Input parameters:
--    bp_key
--      key of the backup piece record in the recovery catalog
--    handle
--      backup piece handle
--    device type
--      device type on which the backup piece resides
-- Exceptions:
--   NO_RECOVERY_CATALOG (ORA-20300)
--     this operation is not supported without the recovery catalog
--   BACKUP_PIECE_DOES_NOT_EXIST
--     the key does not match any backup piece
--   BACKUP_PIECE_HANDLE_IS_AMBIGUOUS
--     the key does not match any backup piece

-- Obsolete as of 8.1.6
PROCEDURE getArchivedLog(
   recid        OUT number
  ,stamp        OUT number
  ,thread#      OUT number
  ,sequence#    OUT number
  ,low_scn      OUT number
  ,nxt_scn      OUT number
  ,fname        OUT varchar2
  ,reset_scn    OUT number
  ,block_size   OUT number
  ,blocks       OUT number);

---------------------------------
-- Controlfilecopy Translation --
---------------------------------

-- translateControlFileCopyName translates a control file name into a list of
-- control file copies.
-- Input parameters:
--   fname
--     name of the controlfile copy to be translated.
--     The name must be a normalized filename
-- Exceptions:
--   CONTROLFILE_COPY_DOES_NOT_EXIST
--     The filename does not match any controlfile copy

PROCEDURE translateControlFileCopyName(
   fname        IN varchar2
  ,available    IN number         DEFAULT NULL -- for compatability
  ,unavailable  IN number         DEFAULT NULL -- for compatability
  ,duplicates   IN number
  ,statusMask   IN binary_integer DEFAULT NULL); -- for compatability

PROCEDURE getControlFileCopy(
   rcvRec       IN OUT rcvRec_t);

-- Obsolete as of 8.1.6
PROCEDURE getControlFileCopy(
   recid        OUT number
  ,stamp        OUT number
  ,reset_scn    OUT number
  ,ckp_scn      OUT number
  ,block_size   OUT number);

------------------------------
-- Datafilecopy Translation --
------------------------------

PROCEDURE getDataFileCopy(
   rcvRec       OUT rcvRec_t
  ,closeCursor  IN  boolean DEFAULT FALSE);

-- translateDataFileCopyKey translates the datafile copy key into a
-- datafile copy recid and stamp in V$DATAFILE_COPY.

-- translateDataFileCopyNumber translates a file number and (optional) tag
-- to a datafile copy recid and stamp. Not used currently in 8.0.

-- translatedDataFileCopyName translates the datafile copy name into a
-- a list of datafile copies and getDataFileCopy returns the recid and stamp
-- of each datafile copy. The duplicates parameter controls whether
-- getDataFileCopy returns all matching datafile copies or just the most
-- recent copy (highest stamp in rcvcat or highest recid in controlfile).

-- translateDataFileCopyTag translates the tag into a list of datafile
-- copies and getDataFileCopy returns the recid and stamp of each datafile copy
-- one at a time until null is returned.

-- translateDataFileCopyFno translates a file number into a list of datafile
-- copies.  getDataFileCopy returns the recid and stamp of each datafile
-- copy one at at time until null is returned.  The duplicates parameter
-- controls whether getDataFileCopy returns all matching datafile copies or
-- just the most recent copy (highest stamp in rcvcat or highest recid in
-- controlfile).

-- The available are unavailable parameters are used to limit the translation
-- to datafile copies with the desired status. For example, only available
-- datafile copies can be backed up, but unavailable copies can be made
-- available. Deleted copies are never returned.

-- The duplicates parameter controls whether getDataFileCopy returns all
-- datafile copies or just the most recent (highest checkpoint scn) copy
-- of each datafile (file#).

-- Input parameters:
--   cdf_key
--     key of the datafile copy record in the recovery catalog
--   fname
--     name of the datafile copy to be translated.
--     The name must be a normalized filename
--   tag
--     tag of the datafile copies to be translated
--     The tag must be passed exactly as stored in the controlfile,
--     it is not uppercased by translate
--   available
--     if TRUE (1) return available datafile copies
--   unavailable
--     if TRUE (1) return unavailable datafile copies
--   duplicates
--     if TRUE (1) return all datafile copies
--     if FALSE (0) eliminate duplicate datafile copies
--
--    The remaining parameters are returned for deleteDataFileCopy
--
--    file#
--    fname
--    reset_scn
--    create_scn
--    ckp_scn
--    blocks_size
--
-- Exceptions:
--   NO_RECOVERY_CATALOG (ORA-20300)
--     translation by key is not supported without the recovery catalog
--   DATAFILE_COPY_DOES_NOT_EXIST
--     the specified key or filename does not match any datafile copy
--   DATAFILE_COPY_NAME_AMBIGUOUS
--     the specified filename matches more than one datafile copy
--   TAG_DOES_NOT_MATCH
--     the specified tag doesn't match any datafile copies
--   TRANSLATION_IN_PROGRESS (ORA-20203)
--     the previous translation conversation is still in progess.
--     To terminate get all datafiles with getDataFileCopy.
--   TRANSLATION_NOT_IN_PROGRESS (ORA-20204)
--     getDataFileCopy was called with no translation in progress

PROCEDURE translateDataFileCopyKey(
   cdf_key      IN number
  ,available    IN number         DEFAULT NULL   -- for compatability
  ,unavailable  IN number         DEFAULT NULL   -- for compatability
  ,statusMask   IN binary_integer DEFAULT NULL); -- for compatability

-- Obsolete as of 8.1.6
PROCEDURE translateDataFileCopyKey(
   cdf_key      IN number
  ,available    IN number
  ,unavailable  IN number
  ,recid        OUT number
  ,stamp        OUT number
  ,file#        OUT number
  ,fname        OUT varchar2
  ,reset_scn    OUT number
  ,create_scn   OUT number
  ,ckp_scn      OUT number
  ,block_size   OUT number
  ,blocks       OUT number);

PROCEDURE translateDataFileCopyName(
   fname        IN varchar2
  ,available    IN number         DEFAULT NULL   -- for compatability
  ,unavailable  IN number         DEFAULT NULL   -- for compatability
  ,duplicates   IN number
  ,statusMask   IN binary_integer DEFAULT NULL); -- for compatability);

PROCEDURE translateDataFileCopyTag(
   tag          IN varchar2
  ,available    IN number         DEFAULT NULL   -- for compatability
  ,unavailable  IN number         DEFAULT NULL   -- for compatability
  ,duplicates   IN number
  ,statusMask   IN binary_integer DEFAULT NULL); -- for compatability);

PROCEDURE translateDataFileCopyFno(
   fno          IN number
  ,available    IN number         DEFAULT NULL
  ,unavailable  IN number         DEFAULT NULL
  ,duplicates   IN number
  ,statusMask   IN binary_integer DEFAULT NULL);

-- Obsolete as of 8.1.6
PROCEDURE getDataFileCopy(
   recid        OUT number
  ,stamp        OUT number
  ,file#        OUT number
  ,fname        OUT varchar2
  ,reset_scn    OUT number
  ,create_scn   OUT number
  ,ckp_scn      OUT number
  ,block_size   OUT number
  ,blocks       OUT number);

----------------------------
-- Proxy Copy Translation --
----------------------------

PROCEDURE getProxyCopy(
   rcvRec       OUT rcvRec_t
  ,closeCursor  IN  boolean DEFAULT FALSE);

PROCEDURE translateProxyCopyKey(
   pc_key       IN number
  ,deviceType   IN varchar2
  ,available    IN number           DEFAULT NULL   -- for compatability
  ,unavailable  IN number           DEFAULT NULL   -- for compatability
  ,deleted      IN number           DEFAULT NULL   -- for compatability
  ,expired      IN number           DEFAULT NULL   -- for compatability
  ,statusMask   IN binary_integer   DEFAULT NULL); -- for compatability

-- Obsolete as of 8.1.6
PROCEDURE translateProxyCopyKey(
   pc_key       IN number
  ,device_type  IN varchar2
  ,available    IN number
  ,unavailable  IN number
  ,deleted      IN number
  ,recid        OUT number
  ,stamp        OUT number
  ,handle       OUT varchar2);

PROCEDURE translateProxyCopyHandle(
   handle       IN varchar2
  ,deviceType   IN varchar2
  ,available    IN number           DEFAULT NULL   -- for compatability
  ,unavailable  IN number           DEFAULT NULL   -- for compatability
  ,deleted      IN number           DEFAULT NULL   -- for compatability
  ,expired      IN number           DEFAULT NULL   -- for compatability
  ,statusMask   IN binary_integer   DEFAULT NULL); -- for compatability

-- Obsolete as of 8.1.6
PROCEDURE translateProxyCopyHandle(
   handle       IN varchar2
  ,device_type  IN varchar2
  ,available    IN number
  ,unavailable  IN number
  ,deleted      IN number
  ,recid        OUT number
  ,stamp        OUT number);

PROCEDURE translateProxyCopyTag(
   tag          IN varchar2
  ,device_type  IN varchar2
  ,available    IN number           DEFAULT NULL   -- for compatability
  ,unavailable  IN number           DEFAULT NULL   -- for compatability
  ,deleted      IN number           DEFAULT NULL   -- for compatability
  ,statusMask   IN binary_integer   DEFAULT NULL); -- for compatability

-- translateProxyCopyKey translates a proxy copy key to a
-- recid and stamp in V$PROXY_DATAFILE/V$PROXY_ARCHIVEDLOG
-- translateProxyCopyHandle translates handle and device type to a
-- proxy copy recid and stamp.

-- getProxyCopy returns one proxy copy after calling translateProxyCopyTag.
-- keep calling getProxyCopy until recid is null.

-- The available and unavailable parameters are used to limit the
-- translation to backup pieces with the desired status.

-- Input parameters:
--    pc_key
--      key of the proxy copy record in the recovery catalog
--    handle
--      proxy copy handle
--    device type
--      device type on which the proxy copy resides
-- Output parameters:
--    recid
--      recid/stamp of the proxy copy record (in V$PROXY_DATAFILE or
--                                               V$PROXY_ARCHIVEDLOG)
-- Exceptions:
--   NO_RECOVERY_CATALOG (ORA-20300)
--     this operation is not supported without the recovery catalog
--   PROXY_COPY_DOES_NOT_EXIST
--     the key does not match any proxy copy
--   PROXY_COPY_HANDLE_IS_AMBIGUOUS
--     the key matches more than one proxy copy

-- Obsolete as of 8.1.6
PROCEDURE getProxyCopy(
   recid OUT number
  ,stamp OUT number
  ,handle OUT varchar2);

------------------------------
-- Backup Piece Translation --
------------------------------

PROCEDURE getBackupPiece(
   bpRec        OUT bpRec_t
  ,closeCursor  IN  boolean DEFAULT FALSE);

PROCEDURE translateBackupPieceKey(
   key         IN  number
  ,available   IN  number            DEFAULT TRUE#
  ,unavailable IN  number            DEFAULT TRUE#
  ,expired     IN  number            DEFAULT TRUE#
  ,statusMask  IN  binary_integer    DEFAULT NULL);   -- for compatability

PROCEDURE translateBackupPieceKey(                        -- only used in 8.1.6
   bp_key          IN  number
  ,available    IN  number
  ,unavailable IN  number
  ,recid           OUT number
  ,stamp           OUT number
  ,handle       OUT varchar2
  ,set_stamp    OUT number
  ,set_count    OUT number
  ,piece#          OUT number);

PROCEDURE translateBackupPieceHandle(
   handle      IN  varchar2
  ,deviceType  IN  varchar2
  ,available   IN  number            DEFAULT NULL     -- for compatability
  ,unavailable IN  number            DEFAULT NULL     -- for compatability
  ,expired     IN  number            DEFAULT NULL     -- for compatability
  ,statusMask  IN  binary_integer    DEFAULT NULL);   -- for compatability

PROCEDURE translateBackupPieceHandle(                     -- only used in 8.1.6
   handle          IN  varchar2
  ,device_type IN  varchar2
  ,available    IN  number
  ,unavailable IN  number
  ,recid           OUT number
  ,stamp           OUT number
  ,set_stamp    OUT number
  ,set_count    OUT number
  ,piece#          OUT number);

PROCEDURE translateBackupPieceTag(
   tag          IN varchar2
  ,available    IN number             DEFAULT NULL     -- for compatability
  ,unavailable  IN number             DEFAULT NULL     -- for compatability
  ,statusMask   IN binary_integer     DEFAULT NULL);   -- for compatability

PROCEDURE translateBackupPieceBSKey(
   key          IN number
  ,tag          IN varchar2           DEFAULT NULL
  ,deviceType   IN varchar2           DEFAULT NULL
  ,pieceCount   IN number
  ,duplicates   IN number             DEFAULT TRUE#
  ,copyNumber   IN number             DEFAULT NULL
  ,available    IN number             DEFAULT TRUE#
  ,unavailable  IN number             DEFAULT FALSE#
  ,deleted      IN number             DEFAULT FALSE#
  ,expired      IN number             DEFAULT FALSE#
  ,statusMask   IN binary_integer     DEFAULT NULL);   -- for compatability

-- Obsolete as of 8.1.6
PROCEDURE translateBackupSetKey(
   bs_key          IN  number
  ,device_type     IN  varchar2
  ,available       IN  number
  ,unavailable     IN  number
  ,deleted         IN  number
  ,duplicates      IN  number
  ,backup_type     OUT varchar2
  ,recid           OUT number
  ,stamp           OUT number
  ,set_stamp       OUT number
  ,set_count       OUT number
  ,bslevel         OUT number
  ,completion_time OUT date);

-- Obsolete as of 8.1
PROCEDURE translateBackupSetKey(
   bs_key      IN  number
  ,device_type IN  varchar2
  ,available   IN  number
  ,unavailable IN  number
  ,deleted     IN  number
  ,duplicates  IN  number
  ,backup_type OUT varchar2
  ,recid       OUT number
  ,stamp       OUT number);

-- Obsolete as of 8.1.6
PROCEDURE translateBackupSetRecid(
   recid       IN  number
  ,stamp       IN  number
  ,device_type IN  varchar2
  ,bs_key      OUT number
  ,bslevel     OUT number
  ,completed   OUT date);

-- Obsolete as of 8.1
PROCEDURE translateBackupSetRecid(
   recid       IN  number
  ,stamp       IN  number
  ,device_type IN  varchar2);

-- translateBackupPieceBSKey translates the specified backup set into a list of
-- backup pieces.  If there are multiple available copies of a piece then
-- only the latest (with highest stamp) is returned.  If there is no available
-- copy of a piece then raise an exception.

-- Input parameters:
--    key
--      key of the backup set record in the recovery catalog
--    recid
--      recid of the backup set record (in V$BACKUP_SET)
--    stamp
--      stamp of the backup set record (in V$BACKUP_SET)
-- Output parameters:
--   device_type
--     type of the device on which the piece resides
--   handle
--     handle to the backup piece
-- Exceptions:
--   BACKUP_SET_MISSING
--     no backup set with the specified recid and stamp found
--   NO_RECOVERY_CATALOG (ORA-20300)
--     translation by bs_key is not supported without the recovery catalog

-- Obsolete as of 8.1.6
PROCEDURE getBackupPiece(
   recid      OUT number
  ,stamp      OUT number
  ,bpkey      OUT number
  ,set_stamp  OUT number
  ,set_count  OUT number
  ,piece#     OUT number
  ,copy#      OUT number
  ,status     OUT varchar2
  ,completion OUT date
  ,handle     OUT varchar2);

-- Obsolete as of 8.1
PROCEDURE getBackupPiece(
   recid      OUT number
  ,stamp      OUT number
  ,set_stamp  OUT number
  ,set_count  OUT number
  ,piece#     OUT number
  ,handle     OUT varchar2);

----------------------------
-- Backup Set Translation --
----------------------------

PROCEDURE translateBackupSetKey(
   key        IN  number
  ,bsRec      OUT bsRec_t);

PROCEDURE translateAllBackupSet(
   backupType            IN  binary_integer
  ,tag                   IN  varchar2
  ,statusMask            IN  binary_integer
  ,completedAfter        IN  date
  ,completedBefore       IN  date);

PROCEDURE getAllBackupSet(
   rcvRec OUT rcvRec_t);

------------------------
-- Controlfile Backup --
------------------------

-- getControlfileBackup is not a public function, but needs to be here due
-- to bug 1269570.
FUNCTION getControlfileBackup(
   rcvRec     OUT rcvRec_t)
RETURN number;

-- findControlFileBackup finds the optimal copy or backup of the controlfile
-- based on the given criteria.
-- The optimal copy is the one with highest checkpoint SCN.  Returns one of:
-- SUCCESS, AVAILABLE, UNAVAILABLE.

-- This is for 8.0.4 thru 8.1.5 compatibility
FUNCTION findControlFileBackup(
   type         OUT number
  ,recid        OUT number
  ,stamp        OUT number
  ,fname        OUT varchar2
  ,device_type  OUT varchar2
  ,ckp_scn      OUT number)
RETURN number;

-- Obsolete as of 8.1.6 (8.1.5 uses this)
FUNCTION findControlFileBackup(
   type         OUT number
  ,recid        OUT number
  ,stamp        OUT number
  ,fname        OUT varchar2
  ,device_type  OUT varchar2
  ,ckp_scn      OUT number
  ,rlg_scn      OUT number
  ,blksize      OUT number)
RETURN number;

-------------------------
-- Archived Log Backup --
-------------------------

PROCEDURE findArchivedLogBackup(
   thread    IN number
  ,sequence  IN number
  ,lowSCN    IN number);

-- findArchivedLogBackup finds a backup set containing the given archived log.
-- getArchivedLogBackup returns the record for the backup set.  The return
-- value is one of:  SUCCESS, AVAILABLE, UNAVAILABLE.
--
-- Input Parameter:
--    thread#
--    sequence#
--    low_scn

FUNCTION getArchivedLogBackup(
   rcvRec       OUT rcvRec_t)
RETURN binary_integer;

-- Obsolete as of 8.1.6
FUNCTION findArchivedLogBackup(
   thread#    IN  number
  ,sequence#  IN  number
  ,low_scn    IN  number
  ,type       OUT number
  ,recid      OUT number
  ,stamp      OUT number
  ,device_type OUT varchar2)
RETURN number;

-------------------
-- SPFILE Backup --
-------------------

FUNCTION getSpfileBackup(
   rcvRec       OUT rcvRec_t)
RETURN number;

---------------
-- List Copy --
---------------

PROCEDURE listTranslateControlfileCopy(
   tag             IN  varchar2
  ,completedAfter  IN  date
  ,completedBefore IN  date
  ,statusMask      IN  binary_integer   DEFAULT
                       BSavailable+BSunavailable+BSexpired); -- default for 8.1

PROCEDURE listGetControlfileCopy(
   rcvRec OUT rcvRec_t);

-- Obsolete as of 8.1.6
FUNCTION listGetControlfileCopy(
   bcfkey     OUT number
  ,ckpscn     OUT number
  ,ckptime    OUT date
  ,status     OUT varchar2
  ,completion OUT date
  ,fname      OUT varchar2)
RETURN number;

PROCEDURE listTranslateDataFileCopy(
   file#             IN number
  ,creation_change#  IN number
  ,tag               IN varchar2        DEFAULT NULL
  ,file_name_pattern IN varchar2        DEFAULT NULL
  ,completedAfter    IN date            DEFAULT NULL
  ,completedBefore   IN date            DEFAULT NULL
  ,statusMask        IN binary_integer  DEFAULT BSavailable+BSunavailable);
                                                             -- default for 8.1

PROCEDURE listGetDataFileCopy(
   rcvRec OUT rcvRec_t);

-- Obsolete as of 8.1.6
FUNCTION listGetDataFileCopy(
   cdf_key            OUT number
  ,status             OUT varchar2
  ,fname              OUT varchar2
  ,completion_time    OUT date
  ,checkpoint_change# OUT number
  ,checkpoint_time    OUT date)
RETURN number;

PROCEDURE listTranslateArchivedLogCopy(
   thread#           IN number
  ,sequence#         IN number
  ,first_change#     IN number
  ,file_name_pattern IN varchar2        DEFAULT NULL
  ,completedAfter    IN date            DEFAULT NULL
  ,completedBefore   IN date            DEFAULT NULL
  ,statusMask        IN binary_integer  DEFAULT
                       BSavailable+BSunavailable+BSexpired  -- 8.0/8.1 defaults
  ,needstby          IN number          DEFAULT NULL);

PROCEDURE listGetArchivedLogCopy(
   rcvRec       OUT rcvRec_t);

-- Obsolete as of 8.1.6
FUNCTION listGetArchivedLogCopy(
   al_key          OUT number
  ,status          OUT varchar2
  ,fname           OUT varchar2
  ,completion_time OUT date)
RETURN number;

-----------------
-- List Backup --
-----------------

PROCEDURE listTranslateControlfileBackup(
   tag             IN  varchar2
  ,completedAfter  IN  date
  ,completedBefore IN  date
  ,statusMask      IN  binary_integer   DEFAULT
                      BSavailable+BSunavailable+BSexpired   -- 8.0/8.1 defaults
  ,autobackup      IN  binary_integer    DEFAULT BScfile_all);

PROCEDURE listGetControlfileBackup(
   rcvRec OUT rcvRec_t);

-- Obsolete as of 8.1.6
FUNCTION listGetControlfileBackup(
   bskey      OUT number,
   ckpscn     OUT number,
   ckptime    OUT date)
RETURN number;

PROCEDURE listTranslateSpfileBackup(
   completedAfter  IN  date
  ,completedBefore IN  date);

PROCEDURE listGetSpfileBackup(
   rcvRec OUT rcvRec_t);

PROCEDURE listTranslateDataFileBackup(
   file#             IN number
  ,creation_change#  IN number
  ,tag               IN varchar2        DEFAULT NULL
  ,completedAfter    IN date            DEFAULT NULL
  ,completedBefore   IN date            DEFAULT NULL
  ,statusMask        IN binary_integer  DEFAULT
                      BSavailable+BSunavailable+BSexpired); -- 8.0/8.1 defaults

PROCEDURE listGetDataFileBackup(
   rcvRec OUT rcvRec_t);

-- Obsolete as of 8.1.6
FUNCTION listGetDataFileBackup(
   bs_key             OUT number
  ,backup_type        OUT varchar2
  ,incremental_level  OUT number
  ,completion_time    OUT date
  ,checkpoint_change# OUT number
  ,checkpoint_time    OUT date)
RETURN number;

-- 8.1.5 LIST implementation
PROCEDURE translateBackupFile(
   bs_recid    IN  number
  ,bs_stamp    IN  number
  ,fno         IN  number
  ,bskey       OUT number
  ,inclevel    OUT number
  ,backup_type OUT varchar2
  ,completed   OUT date);

-- Used by 8.0 and 8.1.6, but not 8.1.5
PROCEDURE listTranslateArchivedLogBackup(
   thread#           IN number
  ,sequence#         IN number
  ,first_change#     IN number
  ,completedAfter    IN date           DEFAULT NULL
  ,completedBefore   IN date           DEFAULT NULL
  ,statusMask        IN binary_integer DEFAULT
                      BSavailable+BSunavailable+BSexpired); -- 8.0/8.1 defaults

PROCEDURE listGetArchivedLogBackup(
   rcvRec OUT rcvRec_t);

-- Obsolete as of 8.1
FUNCTION listGetArchivedLogBackup(
   bs_key          OUT number
  ,completion_time OUT date)
RETURN number;

-- Obsolete as of 8.1.6, but used in 9.0
PROCEDURE listTranslateArchivedLogBackup(
   thread#      IN number   DEFAULT NULL
  ,lowseq       IN number   DEFAULT NULL
  ,highseq      IN number   DEFAULT NULL
  ,lowscn       IN number   DEFAULT NULL
  ,highscn      IN number   DEFAULT NULL
  ,from_time    IN date     DEFAULT NULL
  ,until_time   IN date     DEFAULT NULL
  ,pattern      IN varchar2 DEFAULT NULL);

-- Obsolete as of 8.1.6
FUNCTION listGetArchivedLogBackup(
   bs_key          OUT number
  ,thread#         OUT number
  ,sequence#       OUT number
  ,first_change#   OUT number
  ,next_change#    OUT number
  ,first_time      OUT date
  ,next_time       OUT date)
RETURN number;

--------------------
-- List Backupset --
--------------------

PROCEDURE listTranslateBackupsetFiles(
   bs_key          IN  number);

PROCEDURE listGetBackupsetFiles(
   rcvRec          OUT rcvRec_t);

---------------------
-- List Proxy Copy --
---------------------

-- Note that this is used for both datafiles and the controlfile
PROCEDURE listTranslateProxyDataFile(
   file#             IN number
  ,creation_change#  IN number
  ,tag               IN varchar2        DEFAULT NULL
  ,handle_pattern    IN varchar2        DEFAULT NULL
  ,completedAfter    IN date            DEFAULT NULL
  ,completedBefore   IN date            DEFAULT NULL
  ,statusMask        IN binary_integer  DEFAULT
                       BSavailable+BSunavailable+BSexpired); -- default for 8.1

PROCEDURE listGetProxyDataFile(
   rcvRec OUT rcvRec_t);

-- Obsolete as of 8.1.6
FUNCTION listGetProxyDataFile(
   xdf_key            OUT number
  ,recid              OUT number
  ,stamp              OUT number
  ,status             OUT varchar2
  ,handle             OUT varchar2
  ,completion_time    OUT date
  ,checkpoint_change# OUT number
  ,checkpoint_time    OUT date)
RETURN number;

-- This procedure serves absolutely no purpose.  It is here only for
-- backwards compatbility with 8.1.5.  The only call to this is from
-- krmkafs(), which gets called from krmkgra().  Since the calls are always
-- in sequence, we can simply save the last record returned from
-- getRecoveryAction and avoid doing an extra query.
-- The only value this functions returns that krmkgra() didn't already have
-- in 8.1.5 is the xdf_key.  Completion time was being estimated from the
-- stamp.
PROCEDURE listTranslateProxyDFRecid(
   recid              IN number
  ,stamp              IN number
  ,xdf_key            OUT number
  ,file#              OUT number
  ,status             OUT varchar2
  ,handle             OUT varchar2
  ,completion_time    OUT date
  ,checkpoint_change# OUT number
  ,checkpoint_time    OUT date);

-------------------------------
-- List Database Incarnation --
-------------------------------

PROCEDURE listTranslateDBIncarnation(
   db_name      IN varchar2 DEFAULT NULL);

FUNCTION listGetDBIncarnation(
   db_key            OUT number
  ,dbinc_key         OUT number
  ,db_name           OUT varchar2
  ,db_id             OUT number
  ,current_inc       OUT varchar2
  ,resetlogs_change# OUT number
  ,resetlogs_time    OUT date)
RETURN number;

--------------------------------------
-- List Rollback Segment Tablespace --
--------------------------------------

PROCEDURE listRollbackSegTableSpace;

FUNCTION listGetTableSpace(
   ts#               OUT number
  ,ts_name           OUT varchar2)
RETURN number;

------------------------
-- Incremental Backup --
------------------------

-- getIncrementalScn returns the starting scn for an incremental backup.
-- Input Parameters:
--   file#
--     datafile number
--   reset_scn
--     the resetlogs SCN of the datafile
--   reset_time
--     the resetlogs time of the datafile
--   incr_scn
--     level of the incremental backup
--   cumulative
--      TRUE# if the backup is cumulative
--   first
--      TRUE open the cursor, otherwise just fetch from already opened cursor

-- Exceptions
--   DATAFILE_DOES_NOT_EXIST
--   INVALID_LEVEL
--   NO_PARENT_BACKUP_FOUND

FUNCTION getIncrementalScn(
   file#        IN number
  ,create_scn   IN number
  ,reset_scn    IN number
  ,reset_time   IN date
  ,incr_level   IN number
  ,cumulative   IN number)
RETURN number;

-- This one is an improved version of above. If you want to get
-- incremental scn for all datafiles by opening the cursor only once, then
-- using this will give enormous performance improvement.
--
-- NOTE!! NOTE!! NOTE!!
-- If you pass NULL to file# then it means all of the following
--  o all datafiles
--  o datafiles which has reset_scn and reset_time of current incarnation.
-- It is the callers responsibility to fetch the incremental scn of remaining
-- datafiles which doesn't have reset_scn and reset_time of current
-- incarnation.

PROCEDURE getIncrementalScn(
   first        IN  boolean                  -- open the cursor if this is TRUE
  ,file#        IN  number
  ,create_scn   IN  number
  ,reset_scn    IN  number
  ,reset_time   IN  date
  ,incr_level   IN  number
  ,cumulative   IN  number
  ,rcvRec       OUT rcvRec_t);


--------------------
-- Offline Ranges --
--------------------

PROCEDURE findOfflineRangeCopy(
   offr_recid   IN number
  ,offr_ckpscn  IN number
  ,cf_cretime   IN date
  ,dbinc_key    IN number);

PROCEDURE getOfflineRangeCopy(
   rcvRec       OUT     rcvRec_t);

-- Obsolete as of 8.1.6
FUNCTION getOfflineRangeCopy
RETURN varchar2;

-- findOfflineRangeCopy begins the search for a controlfile copy
-- containing a specified offline range.  getOfflinRangeCopy is called
-- to retrieve the controlfile names one by one.  NULL is returned at
-- end of fetch.

-- Input Parameters:
--   offr_recid
--     recid of offline range
--   offr_ckpscn
--     online checkpoint SCN (end) of offline range
--   dbinc_rlgscn
--     resetlogs SCN of the db incarnation that contains this range
-- Output Parameters:
--   offr_recid
--     recid of the offline range record
--   offr_stamp
--     stamp of the offline range record
--   type
--     type of the controlfile that contains the offline range.
--     COPY or BACKUP
--   recid
--     the recid of datafile copy record or
--     the recid of the backup set record
--   stamp
--     The timestamp associated with the recid in the controlfile.
--   fname
--     filename of the controlfile copy
--     NULL if a backup controlfile is returned

-- returns TRUE (1) if a copy or backup was found
-- returns FALSE (0) if no copy or backup was found

-- Exceptions:
--   OFFLINE_RANGE_NOT_FOUND (ORA-20250)
--     No offline range was found for the datafile starting at the offline SCN

---------------------------------------
-- Recovery Functions and Procedures --
---------------------------------------

PROCEDURE setComputeRecoveryActionMasks(
   containerMask        IN number
  ,actionMask           IN number
  ,allRecords           IN number
  ,availableMask        IN binary_integer);

--Obsolete as of 8.1.7
PROCEDURE setComputeRecoveryActionMasks(
   containerMask        IN number
  ,actionMask           IN number
  ,allRecords           IN number);

-- Obsolete as of 8.1.6
PROCEDURE setRAflags(
   kindMask    IN number
  ,allRecords  IN boolean);

FUNCTION computeRecoveryActions(
fno        IN number,   -- Datafile number.
crescn     IN number,   -- Datafile creation SCN.
df_rlgscn  IN number    -- Datafile resetlogs SCN.  Null if this is a RESTORE
   default null,        -- command, else this is the value in the datafile
                        -- header for the datafile we are RECOVERing.
df_rlgtime IN date      -- Datafile resetlogs time.  Null if df_rlgscn is
   default null,        -- null, else value from datafile header.
df_ckpscn  IN number    -- Datafile checkpoint SCN.  Null if df_rlgscn is
   default null,        -- null, else value from datafile header.
offlscn    IN number    -- kccfeofs (may be null).
   default 0,
onlscn     IN number    -- kccfeonc (null if offlscn is null).
   default 0,
onltime    IN date      -- kccfeonc_time
   default null,
cleanscn   IN number    -- kccfecps if either SOR or WCC set, else null.
   default 0,
clean2scn  IN number    -- CF ckpt SCN if WCC set, infinity if SOR bit set
   default 0,           -- else null.
clean2time IN date      -- cf ckpt time if WCC, SYSDATE if SOR
   default null,
allowfuzzy IN boolean   -- TRUE if can be fuzzy at until SCN/time, FALSE if
  default FALSE,        -- not.  default is FALSE.
partial_rcv IN boolean  -- TRUE if can do partial recovery, FALSE if not
  default FALSE,
cf_scn     IN number    -- controlfile checkpoint SCN (NULL if none mounted)
  default NULL,
cf_cretime IN date      -- controlfile creation time (NULL if none mounted)
  default NULL,
cf_offrrid IN number    -- recid of oldest offline range in controlfile
  default NULL,         -- (NULL if none mounted)
allCopies  IN boolean   -- if TRUE, then stack all valid copies of a bu set
  default FALSE
) return binary_integer;

-- Returns:
--   SUCCESS -> the file can be restored/recovered.
--   else one of RESTORABLE, AVAILABLE, UNAVAILABLE, NO_ACTION.

-- computeRecoveryActions return values --

SUCCESS     CONSTANT binary_integer := 0;
UNAVAILABLE CONSTANT binary_integer := 1;
AVAILABLE   CONSTANT binary_integer := 2;
RESTORABLE  CONSTANT binary_integer := 3;
NO_ACTION   CONSTANT binary_integer := 4;

-- SUCCESS:      A file has been found for RESTORE, or the file on disk
--               can be recovered.
-- UNAVAILABLE:  If RESTORE, then no datafilecopy or level 0 backup was found.
--               If RECOVER, then some incremental backup is missing, or the
--               datafile on disk is too old to recover.
-- AVAILABLE:    If RESTORE, then some level 0 or datafilecopy exists, but
--               the required device type is not allocated.
-- RESTORABLE:   This is returned only when doing a RECOVER.  It means that
--               the file on disk cannot be recovered, but there is some level
--               0 or datafilecopy that could be restored and then recovered.
-- NO_ACTION:    There are no incrementals or offline ranges to apply, but
--               the file should be recoverable with redo.  No guarantee is
--               made that the logs needed are actually available.

FUNCTION getRecoveryAction(
   action OUT rcvRec_t)
RETURN binary_integer;

-- Obsolete as of 8.1.6
FUNCTION getRecoveryAction(
   kind       OUT number
  ,set_stamp  OUT number
  ,set_count  OUT number
  ,recid      OUT number
  ,stamp      OUT number
  ,fname      OUT varchar2
  ,blocksize  OUT number
  ,blocks     OUT number
  ,devtype    OUT varchar2
  ,from_scn   OUT number
  ,to_scn     OUT number
  ,to_time    OUT date
  ,rlgscn     OUT number
  ,rlgtime    OUT date
  ,cfcretime  OUT date
  ,dbinc_key  OUT number)
RETURN binary_integer;

PROCEDURE printRecoveryActions;

PROCEDURE trimRecoveryActions(
   maxActions           IN number
  ,containerMask        IN number
  ,actionMask           IN number);

-- trimRecoveryActions will trim the stack down to the specified number
-- actions if it contains more.  This is used by report obsolete to implement
-- the redundancy count.  The reason for it is that getRecoveryActions
-- returns actions in LIFO order.  This means the oldest actions, which
-- were stacked most recently, are returned first.  However, report obsolete
-- wants to keep only the most recent backups when constructing the
-- "must keep" list.  We solve the problem by getting rid of any excess
-- actions first, and so the order in which getRecoveryActions returns them
-- won't matter.  Note that only actions whose type_con and type_act are
-- selected by the masks will be deleted.  Other actions are left on the
-- stack.

---------------------
-- Report Obsolete --
---------------------

PROCEDURE reportTranslateDFDel ;

-- pre 8.1.5 version
FUNCTION reportGetDFDel(
   file#               OUT number
  ,filetype            OUT number
  ,checkpoint_change#  OUT number
  ,checkpoint_time     OUT date
  ,resetlogs_change#   OUT number
  ,resetlogs_time      OUT date
  ,incremental_change# OUT number
  ,fuzzy_change#       OUT number
  ,recid               OUT number
  ,stamp               OUT number
  ,fname               OUT varchar2
  ,restorable          OUT number)
RETURN number;

-- 8.1.5+ version
FUNCTION reportGetDFDel(
   file#               OUT number
  ,filetype            OUT number
  ,checkpoint_change#  OUT number
  ,checkpoint_time     OUT date
  ,resetlogs_change#   OUT number
  ,resetlogs_time      OUT date
  ,incremental_change# OUT number
  ,fuzzy_change#       OUT number
  ,recid               OUT number
  ,stamp               OUT number
  ,fname               OUT varchar2
  ,restorable          OUT number
  ,key                 OUT number
  ,completion_time     OUT date)
RETURN number;

------------
-- TSPITR --
------------

FUNCTION getCloneName(
   fno IN number
  ,crescn IN number)
RETURN varchar2;


---------------
-- DUPLICATE --
---------------

FUNCTION wasFileOffline(
   fno    IN number
  ,untilscn IN number)
RETURN number;

-------------------------
-- RMAN Configuration ---
-------------------------

PROCEDURE getConfig(
   conf#          OUT    number
  ,name           IN OUT varchar2
  ,value          IN OUT varchar2
  ,first          IN     boolean);

------------------------------
-- Get max(copy#) --
------------------------------

FUNCTION getmaxcopyno(
   bsstamp         IN    number
  ,bscount         IN    number)
RETURN number;

--------------------------
-- Add Corruption Table --
--------------------------

PROCEDURE bmrAddCorruptTable(
   dfnumber    OUT number
  ,blknumber   OUT number
  ,range       OUT number
  ,first       IN  boolean);

------------------------
-- Get Backup History --
------------------------

PROCEDURE getDfBackupHistory(
   backedUpDev     IN   varchar2
  ,first           IN   boolean
  ,file#           IN   number  DEFAULT NULL
  ,crescn          IN   number  DEFAULT NULL
  ,reset_scn       IN   number  DEFAULT NULL
  ,reset_time      IN   date    DEFAULT NULL
  ,bhistoryRec     OUT  bhistoryRec_t);

PROCEDURE getAlBackupHistory(
   backedUpDev     IN   varchar2
  ,first           IN   boolean
  ,thread#         IN   number  DEFAULT NULL
  ,sequence#       IN   number  DEFAULT NULL
  ,bhistoryRec     OUT  bhistoryRec_t);

PROCEDURE getBsBackupHistory(
   backedUpDev     IN   varchar2
  ,first           IN   boolean
  ,set_stamp       IN   number  DEFAULT NULL
  ,set_count       IN   number  DEFAULT NULL
  ,bhistoryRec     OUT  bhistoryRec_t);

PROCEDURE getDcBackupHistory(
   backedUpDev     IN   varchar2
  ,first           IN   boolean
  ,file#           IN   number  DEFAULT NULL
  ,crescn          IN   number  DEFAULT NULL
  ,reset_scn       IN   number  DEFAULT NULL
  ,reset_time      IN   date    DEFAULT NULL
  ,bhistoryRec     OUT  NOCOPY bhistoryRec_t);

-- Obsolute as of 9.0.1
PROCEDURE getBackupHistory(
   dfRec            IN  dfRec_t
  ,backedUpDev      IN  varchar2
  ,nbackupsFlag     IN  number
  ,bscompletionFlag IN  number
  ,nbackups         OUT number
  ,bscompletion     OUT date);

-- Obsolute as of 9.0.1
PROCEDURE getBackupHistory(
   alRec            IN  alRec_t
  ,backedUpDev      IN  varchar2
  ,nbackupsFlag     IN  number
  ,bscompletionFlag IN  number
  ,nbackups         OUT number
  ,bscompletion     OUT date);

-- Obsolute as of 9.0.1
PROCEDURE getBackupHistory(
   bpRec            IN  bpRec_t
  ,backedUpDev      IN  varchar2
  ,nbackupsFlag     IN  number
  ,bscompletionFlag IN  number
  ,nbackups         OUT number
  ,bscompletion     OUT date);

------------------
-- Version Info --
------------------

FUNCTION getPackageVersion
RETURN varchar2;

FUNCTION isStatusMatch(status      IN VARCHAR2,
                       mask        IN NUMBER) RETURN NUMBER;
FUNCTION isDeviceTypeAllocated(deviceType IN varchar2)
                      RETURN NUMBER;
FUNCTION isBackupTypeMatch(btype       IN VARCHAR2,
                           mask        IN binary_integer)
                                              RETURN NUMBER;
pragma TIMESTAMP('2000-03-12:13:51:00');

-- getRequiredSCN --
-- Calculate the lowest gap for all destinations.  Calculate the highest
-- scn available on all valid standby destinations.  If no gap, return the
-- high scn, otherwise return the gap.
PROCEDURE getRequiredSCN(
   reqscn   OUT  number
  ,rlgscn   OUT  number);

END; -- dbms_rcvman or x$dbms_rcvman

/
