create or replace package dbms_stats_internal is

  -- remember DOP of the temp table
  sample_temp_tab_degree integer;


  -- All of the main cursors used in the dbms_stats package
  -- Forcing plans for a couple of reasons:
  --   users may not have statistics on the dictionary, but may be
  --    forcing the cbo with session parameters
  --   bootstrap problem exists in that this package handles the
  --    collection of statistics, and therefore must be prepared to
  --    run without, with partial, and with full statistics

  -- cursor to retrieve column statistics from the dictionary     
  -- *** PLEASE BE SURE TO KEEP export_colstats_direct IN SYNC WITH THIS ***
  -- *** WHEN YOU MAKE A CHANGE HERE.                                    ***
  cursor get_colstats_dict
    (owner varchar2, tabname varchar2, partname varchar2, colname varchar2)
    is
    select /*+ rule */
      u.name, ot.name, null, null, c.name,
      bitand(h.spare2, 2), bitand(h.spare2, 1),
      h.distcnt, h.density, h.spare1, h.sample_size, h.null_cnt,
      h.timestamp#, h.avgcln, 
      h.lowval, h.hival, h.minimum, h.maximum, null, null,
      h.cache_cnt, hg.bucket, hg.endpoint,
      hg.epvalue, bitand(h.spare2, 4)
    from sys.user$ u,  sys.obj$ ot, sys.col$ c,
	 sys.hist_head$ h, histgrm$ hg
    where
      partname is null and
      u.name = owner and ot.owner# = u.user# and
      ot.name = tabname and ot.type# = 2 and 
      c.obj# = ot.obj# and
      (colname is null or c.name = colname) and
      h.obj# = ot.obj# and h.intcol# = c.intcol# and
      hg.obj#(+) = h.obj# and hg.intcol#(+) = h.intcol#
    union all
    select 
      u.name, ot.name, op.subname, null, c.name,
      bitand(h.spare2, 2), bitand(h.spare2, 1),
      h.distcnt, h.density, h.spare1, h.sample_size, h.null_cnt,
      h.timestamp#, h.avgcln, 
      h.lowval, h.hival, h.minimum, h.maximum, null, null,
      h.cache_cnt, hg.bucket, hg.endpoint,
      hg.epvalue, bitand(h.spare2, 4)
    from sys.user$ u,  sys.obj$ ot, sys.col$ c,
	 sys.tabpart$ tp, sys.obj$ op,
	 sys.hist_head$ h, histgrm$ hg
    where
      u.name = owner and ot.owner# = u.user# and
      ot.name = tabname and ot.type# = 2 and 
      c.obj# = ot.obj# and
      (colname is null or c.name = colname) and
      tp.bo# = ot.obj# and tp.obj# = op.obj# and
      (partname is null or op.subname = partname) and
      h.obj# = op.obj# and h.intcol# = c.intcol# and
      hg.obj#(+) = h.obj# and hg.intcol#(+) = h.intcol#
    union all
    select 
      u.name, op.name, op.subname, null, c.name,
      bitand(h.spare2, 2), bitand(h.spare2, 1),
      h.distcnt, h.density, h.spare1, h.sample_size, h.null_cnt,
      h.timestamp#, h.avgcln, 
      h.lowval, h.hival, h.minimum, h.maximum, null, null,
      h.cache_cnt, hg.bucket, hg.endpoint,
      hg.epvalue, bitand(h.spare2, 4)
    from sys.user$ u, sys.col$ c,
	 sys.tabcompart$ tp, sys.obj$ op,
	 sys.hist_head$ h, histgrm$ hg
    where
      u.name = owner and op.owner# = u.user# and
      op.name = tabname and op.type# = 19 and 
      (partname is null or op.subname = partname) and
      tp.obj# = op.obj# and c.obj# = tp.bo# and
      (colname is null or c.name = colname) and
      h.obj# = op.obj# and h.intcol# = c.intcol# and
      hg.obj#(+) = h.obj# and hg.intcol#(+) = h.intcol#
    union all
    select 
      u.name, op.name, op.subname, os.subname, c.name,
      bitand(h.spare2, 2), bitand(h.spare2, 1),
      h.distcnt, h.density, h.spare1, h.sample_size, h.null_cnt,
      h.timestamp#, h.avgcln, 
      h.lowval, h.hival, h.minimum, h.maximum, null, null,
      h.cache_cnt, hg.bucket, hg.endpoint,
      hg.epvalue, bitand(h.spare2, 4)
    from sys.user$ u, sys.col$ c,
	 sys.tabcompart$ tp, sys.obj$ op,
	 sys.tabsubpart$ ts, sys.obj$ os,
	 sys.hist_head$ h, histgrm$ hg
    where
      u.name = owner and op.owner# = u.user# and
      op.name = tabname and op.type# = 19 and 
      tp.obj# = op.obj# and c.obj# = tp.bo# and
      (colname is null or c.name = colname) and
      ts.pobj# = tp.obj# and ts.obj# = os.obj# and
      (partname is null or op.subname = partname or os.subname = partname) and
      h.obj# = os.obj# and h.intcol# = c.intcol# and
      hg.obj#(+) = h.obj# and hg.intcol#(+) = h.intcol#
    order by 1,2,3,4,5,22;

  -- cursor to retrieve index statistics from the dictionary     
  cursor get_indstats_dict
    (owner varchar2, indname varchar2, partname varchar2)
    is
    select /*+ rule */
      u.name, oi.name, null, null,
      decode(bitand(i.flags, 2048),2048,2,0),
      decode(bitand(i.flags, 64),64,1,0),
      i.rowcnt, i.leafcnt, i.distkey, i.lblkkey, i.dblkkey, i.clufac,
      i.blevel, i.analyzetime, i.samplesize,
      decode(bitand(i.flags, 128), 128, mod(trunc(i.pctthres$/256),256),
              decode(i.type#, 4, mod(trunc(i.pctthres$/256),256), NULL))
    from sys.user$ u, sys.obj$ oi, sys.ind$ i
    where
      partname is null and
      u.name = owner and oi.owner# = u.user# and 
      oi.name = indname and oi.type# = 1 and
      oi.obj# = i.obj# and bitand(i.flags,2) = 2
    union all
    select 
      u.name, op.name, op.subname, null,
      decode(bitand(ip.flags, 16),16,2,0), 
      decode(bitand(ip.flags, 8),8,1,0),
      ip.rowcnt, ip.leafcnt, ip.distkey, ip.lblkkey, ip.dblkkey, ip.clufac,
      ip.blevel, ip.analyzetime, ip.samplesize,
      ip.pctthres$
    from sys.user$ u, sys.obj$ oi, sys.indpart$ ip, sys.obj$ op
    where
      u.name = owner and oi.owner# = u.user# and 
      oi.name = indname and oi.type# = 1 and
      ip.bo# = oi.obj# and op.obj# = ip.obj# and
      (partname is null or op.subname = partname) and
      bitand(ip.flags,2) = 2
    union all
    select 
      u.name, op.name, op.subname, null,
      decode(bitand(ip.flags, 16),16,2,0),
      decode(bitand(ip.flags, 8),8,1,0),
      ip.rowcnt, ip.leafcnt, ip.distkey, ip.lblkkey, ip.dblkkey, ip.clufac,
      ip.blevel, ip.analyzetime, ip.samplesize,
      null
    from sys.user$ u, sys.obj$ op, sys.indcompart$ ip
    where
      u.name = owner and op.owner# = u.user# and 
      op.name = indname and op.type# = 20 and
      ip.obj# = op.obj# and
      (partname is null or op.subname = partname) and
      bitand(ip.flags,2) = 2
    union all
    select 
      u.name, op.name, op.subname, os.subname,
      decode(bitand(isp.flags, 16),16,2,0),
      decode(bitand(isp.flags, 8),8,1,0),
      isp.rowcnt, isp.leafcnt, isp.distkey, isp.lblkkey, 
      isp.dblkkey, isp.clufac,
      isp.blevel, isp.analyzetime, isp.samplesize,
      null
    from sys.user$ u, sys.obj$ op, sys.indcompart$ ip,
	 sys.indsubpart$ isp, sys.obj$ os
    where
      u.name = owner and op.owner# = u.user# and 
      op.name = indname and op.type# = 20 and
      ip.obj# = op.obj# and 
      isp.pobj# = ip.obj# and os.obj# = isp.obj# and
      (partname is null or op.subname = partname or os.subname = partname) and
      bitand(isp.flags,2) = 2
    order by 1,2,3;

  -- cursor to retrieve table statistics from the dictionary     
  cursor get_tabstats_dict
    (owner varchar2, tabname varchar2, partname varchar2)
    is
    select /*+ rule */
      u.name, ot.name, null, null,
      decode(bitand(t.flags, 512),512,2,0), 
      decode(bitand(t.flags, 256),256,1,0),
      t.rowcnt, t.blkcnt, t.avgrln, t.analyzetime, t.samplesize
    from sys.user$ u, sys.obj$ ot, sys.tab$ t
    where
      partname is null and
      u.name = owner and ot.owner# = u.user# and 
      ot.name = tabname and ot.type# = 2 and
      ot.obj# = t.obj# and bitand(t.flags,16) = 16
    union all
    select 
      u.name, op.name, op.subname, null,
      decode(bitand(tp.flags, 16),16,2,0), 
      decode(bitand(tp.flags, 8),8,1,0),
      tp.rowcnt, tp.blkcnt, tp.avgrln, tp.analyzetime, tp.samplesize
    from sys.user$ u, sys.obj$ ot, sys.tabpart$ tp, sys.obj$ op
    where
      u.name = owner and ot.owner# = u.user# and 
      ot.name = tabname and ot.type# = 2 and
      tp.bo# = ot.obj# and op.obj# = tp.obj# and
      (partname is null or op.subname = partname) and
      bitand(tp.flags,2) = 2
    union all
    select 
      u.name, op.name, op.subname, null,
      decode(bitand(tp.flags, 16),16,2,0), 
      decode(bitand(tp.flags, 8),8,1,0),
      tp.rowcnt, tp.blkcnt, tp.avgrln, tp.analyzetime, tp.samplesize
    from sys.user$ u, sys.obj$ op, sys.tabcompart$ tp
    where
      u.name = owner and op.owner# = u.user# and 
      op.name = tabname and op.type# = 19 and
      tp.obj# = op.obj# and
      (partname is null or op.subname = partname) and
      bitand(tp.flags,2) = 2
    union all
    select 
      u.name, os.name, op.subname, os.subname,
      decode(bitand(ts.flags, 16),16,2,0), 
      decode(bitand(ts.flags, 8),8,1,0),
      ts.rowcnt, ts.blkcnt, ts.avgrln, ts.analyzetime, ts.samplesize
    from sys.user$ u, sys.obj$ op, sys.tabcompart$ tp,
	 sys.tabsubpart$ ts, sys.obj$ os
    where
      u.name = owner and op.owner# = u.user# and 
      op.name = tabname and op.type# = 19 and
      tp.obj# = op.obj# and 
      ts.pobj# = tp.obj# and os.obj# = ts.obj# and
      (partname is null or op.subname = partname or os.subname = partname) and
      bitand(ts.flags,2) = 2
    order by 1,2,3;


  -- cursor which retrieves index information for a given table
  cursor all_indexes
    (whose_tab varchar2, which_tab varchar2, global_idx_only varchar2) is
    select /*+ rule */
	'"'||ui.name||'"' ind_owner, '"'||oi.name||'"' ind_name, 
	oi.obj# obj_num, nvl(po.flags,0) locality
    from 
	sys.user$ ut, sys.obj$ ot, sys.ind$ i, sys.obj$ oi, 
	sys.user$ ui, sys.partobj$ po
    where
        ((global_idx_only is null) or
	   (po.flags is null) or    -- non-partitioned
	   (po.flags = 2)) and      -- globally partitioned
	ut.name = whose_tab and ut.user# = ot.owner# and
	ot.name = which_tab and ot.type# = 2 and ot.obj# = i.bo# and
	i.obj# = oi.obj# and oi.owner# = ui.user# and
	i.obj# = po.obj#(+);

  -- cursor which retrieves index information for a given table partition
  cursor all_local_index_parts 
    (whose_tab varchar2, which_tab varchar2, which_part varchar2) is
      select /*+ all_rows */
        '"'||ui.name||'"' ind_owner, '"'||oip.name||'"' ind_name, 
        '"'||oip.subname||'"' part_name, oip.obj# obj_num
      from
        sys.user$ ut, sys.obj$ otp, sys.tabpart$ tp,
        sys.ind$ i, sys.partobj$ po, sys.indpart$ ip,
        sys.obj$ oip, sys.user$ ui
      where
        ut.name = whose_tab and ut.user# = otp.owner# and
        otp.name = which_tab and otp.subname = which_part and
        otp.type# = 19 and otp.obj# = tp.obj# and
        tp.bo# = i.bo# and i.obj# = po.obj# and bitand(po.flags, 1) = 1 and
        i.obj# = ip.bo# and ip.part# = tp.part# and
        ip.obj# = oip.obj# and oip.owner# = ui.user#
      union all
      select 
        '"'||ui.name||'"' ind_owner, '"'||oip.name||'"' ind_name, 
        '"'||oip.subname||'"' part_name, oip.obj# obj_num
      from
        sys.user$ ut, sys.obj$ otp, sys.tabcompart$ tp,
        sys.ind$ i, sys.partobj$ po, sys.indcompart$ ip,
        sys.obj$ oip, sys.user$ ui
      where
        ut.name = whose_tab and ut.user# = otp.owner# and
        otp.name = which_tab and otp.subname = which_part and
        otp.type# = 19 and otp.obj# = tp.obj# and
        tp.bo# = i.bo# and i.obj# = po.obj# and bitand(po.flags, 1) = 1 and
        i.obj# = ip.bo# and ip.part# = tp.part# and
        ip.obj# = oip.obj# and oip.owner# = ui.user#
      union all
      select 
        '"'||ui.name||'"' ind_owner, '"'||ois.name||'"' ind_name, 
        '"'||ois.subname||'"' part_name, ois.obj# obj_num
      from
        sys.user$ ut, sys.obj$ ots, sys.tabsubpart$ tsp, sys.tabcompart$ tp,
        sys.ind$ i, sys.partobj$ po, sys.indcompart$ ip,
        sys.indsubpart$ isp, sys.obj$ ois, sys.user$ ui
      where
        ut.name = whose_tab and ut.user# = ots.owner# and
        ots.name = which_tab and ots.subname = which_part and
        ots.type# = 34 and ots.obj# = tsp.obj# and
        tsp.pobj# = tp.obj# and tp.bo# = i.bo# and
        i.obj# = po.obj# and bitand(po.flags, 1) = 1 and
        i.obj# = ip.bo# and ip.part# = tp.part# and
        ip.obj# = isp.pobj# and isp.subpart# = tsp.subpart# and
        isp.obj# = ois.obj# and ois.owner# = ui.user#;


  -- cursor to return information about all of a table's columns
  cursor tab_colinfo(whose_tab varchar2, which_tab varchar2) is
    select /*+ rule */
	c.name col_name, c.type# col_type, c.charsetform col_csf, 
        c.default$ col_def, c.null$ col_null, c.property col_prop,
        c.col# col_unum, c.intcol# col_inum, c.obj# col_obj,
	c.scale col_scale, h.bucket_cnt h_bcnt,
	h.distcnt h_pndv, c.length col_len, cu.timestamp cu_time, 
	cu.equality_preds cu_ep, cu.equijoin_preds cu_ejp, 
	cu.range_preds cu_rp, cu.like_preds cu_lp,
        cu.nonequijoin_preds cu_nejp, cu.null_preds np
    from 
	sys.user$ u, sys.obj$ o, sys.col$ c, sys.col_usage$ cu,
	sys.hist_head$ h
    where 
	u.name = whose_tab and o.owner# = u.user# and 
	o.type# = 2 and o.name = which_tab and
	o.obj# = c.obj# and 
	c.obj# = cu.obj#(+) and c.intcol# = cu.intcol#(+) and
	c.obj# = h.obj#(+) and c.intcol# = h.intcol#(+);

  -- cursor to get unique/primary key constraints for a column
  -- also check if this column is a prefix of an index
  cursor get_col_consind (onum number, inum number) is
    select count(unq) unq, count(pfx) pfx from
     (select /*+ first_rows(1) leading(cc) */
        cd.type# unq, null pfx
      from sys.ccol$ cc, sys.cdef$ cd
      where 
	cc.obj# = onum and cc.intcol# = inum and
	cd.con# = cc.con# and cd.obj# = cc.obj# and
	cd.enabled is not null and cd.intcols = 1 and cd.type# in (2,3) and
        rownum <= 1
      union all
      select /*+ first_rows(1) leading(i) */
        case when i.intcols = 1 and bitand(i.property,1) = 1 then 3
          else null end unq,
        case when ic.pos# = 1 then 1 else null end pfx
      from sys.ind$ i, sys.icol$ ic
      where
        i.bo# = onum and ic.intcol# = inum and
	i.obj# = ic.obj# and bitand(i.flags,1025) = 0 and
        rownum <= 1);

  -- cursor to return list of indexes containing column
  cursor col_indexes(whose_tab varchar2, which_tab varchar2,
                   which_col varchar2) is
    select /*+ rule */
	oi.name 
    from 
	sys.user$ u, sys.obj$ ot, sys.col$ c, sys.icol$ ic, sys.obj$ oi
    where 
	u.name = whose_tab and ot.owner# = u.user# and 
	ot.name = which_tab and ot.type# = 2 and
	ot.obj# = c.obj# and c.name = which_col and 
	ot.obj# = ic.bo# and ic.intcol# = c.intcol# and
	ic.obj# = oi.obj#;

  -- cursor to return information about all of an index's columns
  cursor ind_colinfo(whose_ind varchar2, which_ind varchar2) is
    select /*+ rule */
	h.distcnt dcnt, h.null_cnt ncnt, h.avgcln acln,
        c.length mcln, c.type# ctyp, c.name cnam
      from
	sys.user$ u, sys.obj$ o, sys.icol$ ic, sys.col$ c, sys.hist_head$ h
      where 
	u.name = whose_ind and o.owner# = u.user# and
	o.name = which_ind and o.type# = 1 and
	ic.obj# = o.obj# and ic.bo# = c.obj# and
	ic.intcol# = c.intcol# and
	h.col#(+) = c.intcol# and h.obj#(+) = c.obj#
      order by ic.pos#;

  -- cursor to acquire all of a table's partitions
  cursor tpart_cur (whose_tab in varchar2,which_tab varchar2,
		   which_part varchar2) is
    select /*+ rule */
	o.subname part_name, o.obj# obj_num
    from
	sys.user$ u, sys.obj$ o
    where 
	u.name = whose_tab and o.owner# = u.user# and
	o.name = which_tab and o.type# = 19 and
        ((which_part is null) or      -- all partitions
         (which_part = o.subname) or  -- partition name specified
          exists     -- subpartition name specified (return parent partition)
            (select *
               from
                 tabsubpart$ tsp, obj$ spo
               where
                 spo.type# = 34 and spo.subname = which_part and
                 tsp.obj#=spo.obj# and tsp.pobj# = o.obj#))
    order by part_name;

  -- cursor to acquire all of a table's subpartitions
  cursor tspart_cur (whose_tab in varchar2,which_tab varchar2,
		    which_part varchar2) is
    select /*+ rule */
	os.subname part_name, os.obj# obj_num, op.subname parent
    from
	sys.user$ u, sys.obj$ op, tabsubpart$ tsp, sys.obj$ os
    where 
	u.name = whose_tab and op.owner# = u.user# and
	op.name = which_tab and 
	op.obj# = tsp.pobj# and tsp.obj# = os.obj#
	and ((which_part is null) or (which_part = os.subname) or
	     (which_part = op.subname))
    order by part_name;

  -- cursor to acquire the name of the IOT_TOP index for an IOT
  cursor get_iot_top (owner varchar2, tabname varchar2) is
    select /*+ rule */
	'"'||oi.name||'"'
    from
	sys.user$ u, sys.obj$ ot, sys.ind$ i, sys.obj$ oi
    where 
	u.name = owner and ot.owner# = u.user# and
	ot.name = tabname and 
	i.bo# = ot.obj# and i.type# = 4 and
	i.obj# = oi.obj#;


  -- retrieve all tables and cluster indexes
  -- ignore IOT overflow segments, nested tables, snapshot logs
  -- and temporary tables (unless gather_temp is not null)
  -- set gather_sys to non-null to get the dictionary objects
  -- set gather_fixed to non-null to get the fixed tables
  cursor all_objects(ownname varchar2, gather_sys varchar2,
                     gather_temp varchar2, gather_fixed varchar2) is
    select /*+ ordered full(t) full(o) use_hash(o) full(u) use_hash(u) */
	'TABLE' obj_type, o.name obj_name, u.name user_name,
	o.obj# obj_num
    from
	sys.tab$ t, sys.obj$ o, sys.user$ u
    where
        ((ownname is null) or (u.name = ownname)) and
	((gather_sys is not null) or (u.name != 'SYS')) and
	bitand(t.property,1) != 1 and
	bitand(t.property,512) != 512 and
	((gather_temp is not null) or
         (bitand(t.property,4194304) != 4194304 and
          bitand(t.property,8388608) != 8388608)) and
        bitand(t.property,8192) != 8192 and
        (u.name,o.name) not in
          (select /*+ unnest */ mowner,log from mlog$) and
        (u.name,o.name) not in
          (select /*+ unnest */ mowner,temp_log
             from mlog$ where temp_log is not null) and
	(u.name != 'SYS' or o.name not like 'S$%') and
	t.obj# = o.obj# and o.owner# = u.user# and
	o.linkname is null
    union all
    select /*+ ordered use_nl(c) use_nl(o) use_nl(u) */
	'INDEX' obj_type, o.name obj_name, u.name user_name,
	o.obj# obj_num
    from
	sys.ind$ i, sys.clu$ c, sys.obj$ o, sys.user$ u
    where
        ((ownname is null) or (u.name = ownname)) and
	((gather_sys is not null) or (u.name != 'SYS')) and
	i.bo# = c.obj# and 
	i.obj# = o.obj# and o.owner# = u.user# and
	o.linkname is null
    union all
    -- Ideally the fixed tables should be analyzed before analyzing obj$,
    -- tab$, col$ etc as gather fixed table stats may generate new entries 
    -- for the dummy tables in dictionary
    select 
        'TABLE' obj_type, kqftanam obj_name, 'SYS' user_name, 0 obj_num
    from
        sys.x$kqfta fxt
    where
        gather_fixed is not null and 
        ((ownname is null) or (ownname = 'SYS')) and
        -- get rid of following predicate once bugs 2050824, 2014541 are fixed
        kqftanam not in ('X$KSFMELEM', 'X$KSMMEM')
    order by 3, 1 desc, 2;

  -- cursor to acquire the list of all objects with stale statistics
  -- set gather_sys to non-null to get the dictionary objects
  cursor all_stale_objects(ownname varchar2, gather_sys varchar2) is
    select /*+ ordered full(t) use_hash(t) use_nl(o) use_nl(u) */
	u.name own, o.name tab, null part, null spart
    from
	sys.mon_mods$ m, sys.tab$ t, sys.obj$ o, sys.user$ u
    where
        ((ownname is null) or (u.name = ownname)) and
	((gather_sys is not null) or (u.name != 'SYS')) and
	m.obj# = t.obj# and
        (u.name,o.name) not in
          (select /*+ unnest */ mowner,log from mlog$) and
        (u.name,o.name) not in
          (select /*+ unnest */ mowner,temp_log
             from mlog$ where temp_log is not null) and
	bitand(t.flags,16) = 16 and
	((bitand(m.flags,1) = 1) or
	 ((m.inserts + m.updates + m.deletes) > (.1 * t.rowcnt))) and
	(u.name != 'SYS' or o.name not like 'S$%') and
	t.obj# = o.obj# and o.owner# = u.user#
    union all
    select /*+ ordered use_hash(tp) use_nl(t) use_nl(o) use_nl(u) */
	u.name own, 
        /* bug 2953803 : retrieve based table name instead of iot-overflow */
        (case bitand(t.property ,512) 
        when  512
        then       (select oi.name from sys.obj$ oi where oi.obj#=t.bobj#)
        else      o.name
        end
               ) tab, 
        o.subname part, null spart
    from
	sys.mon_mods$ m, sys.tabpart$ tp, sys.tab$ t,sys.obj$ o, sys.user$ u
    where
        ((ownname is null) or (u.name = ownname)) and
	((gather_sys is not null) or (u.name != 'SYS')) and
	m.obj# = tp.obj# and
        tp.bo# =t.obj# and
	bitand(tp.flags,2) = 2 and
	((bitand(m.flags,1) = 1) or
	 ((m.inserts + m.updates + m.deletes) > (.1 * tp.rowcnt))) and
	tp.obj# = o.obj# and o.owner# = u.user#
    union all
    select /*+ ordered use_hash(tcp) use_nl(o) use_nl(u) */
	u.name own, o.name tab, o.subname part, null spart
    from
	sys.mon_mods$ m, sys.tabcompart$ tcp, sys.obj$ o, sys.user$ u
    where
        ((ownname is null) or (u.name = ownname)) and
	((gather_sys is not null) or (u.name != 'SYS')) and
	m.obj# = tcp.obj# and
	bitand(tcp.flags,2) = 2 and
	((bitand(m.flags,1) = 1) or
	 ((m.inserts + m.updates + m.deletes) > (.1 * tcp.rowcnt))) and
	tcp.obj# = o.obj# and o.owner# = u.user#
    union all
    select /*+ ordered use_hash(tsp) use_nl(o) use_nl(u) use_nl(op) */
	u.name own, o.name tab, op.subname part, o.subname spart
    from
	sys.mon_mods$ m, sys.tabsubpart$ tsp, sys.obj$ o, sys.user$ u,
	sys.obj$ op
    where
        ((ownname is null) or (u.name = ownname)) and
	((gather_sys is not null) or (u.name != 'SYS')) and
	m.obj# = tsp.obj# and
	bitand(tsp.flags,2) = 2 and
	((bitand(m.flags,1) = 1) or
	 ((m.inserts + m.updates + m.deletes) > (.1 * tsp.rowcnt))) and
	tsp.obj# = o.obj# and o.owner# = u.user# and tsp.pobj# = op.obj#
    order by 1,2,3,4;

  -- cursor to acquire the list of all objects with no statistics
  -- ignore temporary tables, indexes on temporary tables and snapshot logs
  -- set gather_sys to non-null to get the dictionary objects
  cursor all_empty_objects(ownname varchar2, gather_sys varchar2,
                           gather_temp varchar2) is
    select /*+ ordered */
	'TABLE' obj_type, u.name own, o.name tab, null part, null spart
    from 
	sys.tab$ t, sys.obj$ o, sys.user$ u
    where 
        ((ownname is null) or (u.name = ownname)) and
	((gather_sys is not null) or (u.name != 'SYS')) and
	bitand(t.property,1) != 1 and
	((gather_temp is not null) or
         (bitand(t.property,4194304) != 4194304 and
          bitand(t.property,8388608) != 8388608)) and
	bitand(t.flags,16) != 16 and
        (u.name,o.name) not in
          (select /*+ unnest */ mowner,log from mlog$) and
        (u.name,o.name) not in
          (select /*+ unnest */ mowner,temp_log
             from mlog$ where temp_log is not null) and
	(u.name != 'SYS' or o.name not like 'S$%') and
	t.obj# = o.obj# and o.owner# = u.user#
    union all
    select /*+ ordered */
	'TABLE' obj_type, u.name own, o.name tab, o.subname part, null spart
    from 
	sys.tabpart$ tp, sys.obj$ o, sys.user$ u
    where 
        ((ownname is null) or (u.name = ownname)) and
	((gather_sys is not null) or (u.name != 'SYS')) and
	bitand(tp.flags,2) != 2 and
	tp.obj# = o.obj# and o.owner# = u.user#
    union all
    select /*+ ordered */
	'TABLE' obj_type, u.name own, o.name tab, o.subname part, null spart
    from 
	sys.tabcompart$ tcp, sys.obj$ o, sys.user$ u
    where 
        ((ownname is null) or (u.name = ownname)) and
	((gather_sys is not null) or (u.name != 'SYS')) and
	bitand(tcp.flags,2) != 2 and
	tcp.obj# = o.obj# and o.owner# = u.user#
    union all
    select /*+ ordered */
	'TABLE' obj_type, u.name own, o.name tab, op.subname part, 
	o.subname spart
    from 
	sys.tabsubpart$ tsp, sys.obj$ o, sys.user$ u, sys.obj$ op
    where 
        ((ownname is null) or (u.name = ownname)) and
	((gather_sys is not null) or (u.name != 'SYS')) and
	bitand(tsp.flags,2) != 2 and
	tsp.obj# = o.obj# and o.owner# = u.user# and
	tsp.pobj# = op.obj#
    union all
    select /*+ ordered */
	'INDEX' obj_type, u.name own, o.name tab, null part, null spart
    from 
	sys.ind$ i, sys.obj$ o, sys.user$ u
    where 
        ((ownname is null) or (u.name = ownname)) and
	((gather_sys is not null) or (u.name != 'SYS')) and
	bitand(i.property,32) != 32 and
	bitand(i.property,64) != 64 and
	bitand(i.flags,2) != 2 and
	i.obj# = o.obj# and o.owner# = u.user#
    union all
    select /*+ ordered */
	'INDEX' obj_type, u.name own, o.name tab, o.subname part, null spart
    from 
	sys.indpart$ ip, sys.obj$ o, sys.user$ u
    where 
        ((ownname is null) or (u.name = ownname)) and
	((gather_sys is not null) or (u.name != 'SYS')) and
	bitand(ip.flags,2) != 2 and
	ip.obj# = o.obj# and o.owner# = u.user#
    union all
    select /*+ ordered */
	'INDEX' obj_type, u.name own, o.name tab, o.subname part, null spart
    from 
	sys.indcompart$ icp, sys.obj$ o, sys.user$ u
    where 
        ((ownname is null) or (u.name = ownname)) and
	((gather_sys is not null) or (u.name != 'SYS')) and
	bitand(icp.flags,2) != 2 and
	icp.obj# = o.obj# and o.owner# = u.user#
    union all
    select /*+ ordered */
	'INDEX' obj_type, u.name own, o.name tab, op.subname part, 
	o.subname spart
    from 
	sys.indsubpart$ isp, sys.obj$ o, sys.user$ u, sys.obj$ op
    where 
        ((ownname is null) or (u.name = ownname)) and
	((gather_sys is not null) or (u.name != 'SYS')) and
	bitand(isp.flags,2) != 2 and
	isp.obj# = o.obj# and o.owner# = u.user# and
	isp.pobj# = op.obj#
    order by 1 desc,2,3,4;


  -- cursor to acquire the list of all objects with nomonitoring
  -- ignore temp tables, external tables, bootstrap objects, mon_mods$
  -- and snapshot logs
  -- set gather_sys to non-null to get the dictionary objects
  cursor all_monitored_objects
    (ownname varchar2, monitoring varchar2, gather_sys varchar2)
      is
    select /*+ ordered full(t) use_nl(o) use_nl(u) */
	u.name own, o.name tab
    from
	sys.tab$ t, sys.obj$ o, sys.user$ u
    where
        ((ownname is null) or (u.name = ownname)) and
	((gather_sys is not null) or (u.name != 'SYS')) and
        bitand(t.property,512) != 512 and
        bitand(t.property,4194304) != 4194304 and
        bitand(t.property,8388608) != 8388608 and
	bitand(t.property,2147483648) != 2147483648 and
	((monitoring = 'Y' and bitand(t.flags,2097152) = 2097152) or
	(monitoring = 'N' and bitand(t.flags,2097152) != 2097152)) and
	o.obj# not in (select obj# from bootstrap$) and
        (u.name,o.name) not in
          (select /*+ unnest */ mowner,log from mlog$) and
        (u.name,o.name) not in
          (select /*+ unnest */ mowner,temp_log
             from mlog$ where temp_log is not null) and
        (u.name != 'SYS' or 
         o.name not in ('MON_MODS$', 'HIST_HEAD$', 'HISTGRM$',
                        'OBJAUTH$', 'OBJECT_USAGE')) and
	(u.name != 'SYS' or o.name not like 'S$%') and
	t.obj# = o.obj# and o.owner# = u.user#;


  -- cursor to acquire the statistics' table tablespace name
  cursor get_tblspc (statownU varchar2, stattabU varchar2) is
    select /*+ rule */
	ts.name
    from
	sys.user$ u, sys.ts$ ts, sys.tab$ t, sys.obj$ o
    where
	u.name = statownU and u.user# = o.owner# and
	o.name = stattabU and o.obj# = t.obj# and
	t.ts# = ts.ts#;


  -- figure out the tab partition name for a particular subpartition
  cursor get_tpart_name (owner varchar2, tabname varchar2, spname varchar2) is
    select /*+ rule */
	op.subname
    from 
	sys.user$ u, sys.obj$ os, sys.tabsubpart$ tsp, sys.obj$ op
    where
	u.name = owner and u.user# = op.owner# and
	op.name = tabname and op.obj# = tsp.pobj# and
	tsp.obj# = os.obj# and os.subname = spname;

  -- figure out the ind partition name for a particular subpartition
  cursor get_ipart_name (owner varchar2, indname varchar2, spname varchar2) is
    select /*+ rule */
	op.subname
    from 
	sys.user$ u, sys.obj$ os, sys.indsubpart$ isp, sys.obj$ op
    where
	u.name = owner and u.user# = op.owner# and
	op.name = indname and op.obj# = isp.pobj# and
	isp.obj# = os.obj# and os.subname = spname;


  -- get supporting information for generating statistics
  cursor get_generate_info (owner varchar2, objname varchar2) is
    select /*+ rule */
	tob.name, i.property, i.type#, i.cols, i.pctfree$,
	t.rowcnt, t.blkcnt, t.flags
    from 
	sys.ind$ i, sys.obj$ iob, sys.user$ u, sys.tab$ t, sys.obj$ tob
    where 
	u.name = owner and u.user# = iob.owner# and
	iob.name = objname and iob.obj# = i.obj# and
	i.bo# = t.obj# and t.obj# = tob.obj#;


  -- find out how large a data block is
  cursor get_block_size is
    select /*+ rule */ value
    from v$parameter
    where upper(name) = 'DB_BLOCK_SIZE';


  -- return index, underlying table and index partitioning information
  cursor get_index_info(owner varchar2, indname varchar2) is
    select /*+ ordered use_nl(u io i ito t itu po) */
        io.obj# iobjn,i.type# itype,i.flags iflags,i.property iprop,
        i.intcols iintcols, nvl(i.degree,1), i.analyzetime,
        itu.name town,ito.name tab,i.bo# tobjn,t.flags tflags,t.property tprop,
        case when bitand(t.flags, 16) = 0 then null else t.rowcnt end nrows,
        po.parttype ptype,po.partcnt pcnt,po.partkeycols ptkcols,
	po.flags pflags,mod(po.spare2,256) sptype
      from sys.user$ u,sys.obj$ io,sys.ind$ i,
           sys.obj$ ito,sys.user$ itu,sys.tab$ t,
	   partobj$ po
      where u.name=owner and io.name=indname and io.owner#=u.user#
	    and io.type#=1 and io.obj#=i.obj# and ito.obj#=i.bo#
	    and ito.owner#=itu.user# and ito.obj#=t.obj#
	    and io.obj#=po.obj#(+)
    union all
    select  /*+ ordered use_nl(u io i ito t itu) */
        io.obj# iobjn,i.type# itype,i.flags iflags,i.property iprop,
        i.intcols iintcols, nvl(i.degree,1), i.analyzetime,
        itu.name town,ito.name tab,i.bo# tobjn,0 tflags,0 tprop,
        null nrows,
        null ptype,null pcnt,null ptkcols,
	null pflags,null sptype
      from sys.user$ u,sys.obj$ io,sys.ind$ i,
           sys.obj$ ito,sys.user$ itu,sys.clu$ t
      where u.name=owner and io.name=indname and io.owner#=u.user#
	    and io.type#=1 and io.obj#=i.obj# and ito.obj#=i.bo#
	    and ito.owner#=itu.user# and ito.obj#=t.obj#;

  -- return index key column information
  --   if join index, return "tabname"."colname"
  cursor get_index_key_columns (iobjn pls_integer, join_index varchar2) is
    select /*+ all_rows */
        c.name icname,ic.pos# icpos,ic.spare1 icflags,
	c.property icprop,c.default$ icexpr, c.deflength icelen
      from icol$ ic,col$ c
      where (join_index is null)
            and ic.obj#=iobjn and ic.bo#=c.obj# and ic.intcol# = c.intcol#
     union all
     select
        '"'||bo.name||'"."'||c.name||'"' icname,
	ic.pos# icpos,ic.spare1 icflags,
	c.property icprop,c.default$ icexpr, c.deflength icelen
      from icol$ ic,col$ c, obj$ bo
      where (join_index is not null)
            and ic.obj#=iobjn and ic.bo#=c.obj# and ic.bo#=bo.obj#
	    and ic.spare2 = c.intcol#
    order by icpos;

  -- return name of ANY NON-INDEX KEY COLUMN of the underlying table
  cursor get_any_non_index_key_column (tobjn pls_integer, iobjn pls_integer) is
    select /*+ rule */
	   c.name cname
      from col$ c
      where c.obj#=tobjn and
            bitand(c.property,32) != 32 and        -- skip hidden columns
	    c.intcol# not in
	      (select ic.intcol# from icol$ ic where ic.obj#=iobjn) and
	    rownum < 2;


  -- return join conditions of the join index
  cursor get_index_join_cond(iobjn pls_integer) is
    select /*+ rule */
	   u1.name owner1,o1.name tab1,c1.name col1,
	   u2.name owner2,o2.name tab2,c2.name col2
      from jijoin$ j,
	   user$ u1,obj$ o1,tab$ t1,col$ c1,
	   user$ u2,obj$ o2,tab$ t2,col$ c2
      where j.obj#=iobjn
	    and j.tab1obj#=o1.obj# and u1.user#=o1.owner#
	    and j.tab1obj#=t1.obj# and j.tab1obj#=c1.obj#
	    and j.tab1col#=c1.intcol#
	    and j.tab2obj#=o2.obj# and u2.user#=o2.owner#
	    and j.tab2obj#=t2.obj# and j.tab2obj#=c2.obj#
	    and j.tab2col#=c2.intcol#;


  -- return either specified or all (when pname is null) subpartition info
  cursor get_index_subpartitions (
              whose_ind in varchar2, which_ind varchar2,
              which_part varchar2) is
    select /*+ all_rows */
        sp.obj# spobjn, sp.subpart# spn, spo.subname spname,
        sp.analyzetime last_analyzed,
        p.obj# pobjn, p.part# pn, po.subname pname
      from
        sys.indcompart$ p, sys.indsubpart$ sp, sys.obj$ spo,
        sys.obj$ po, sys.user$ u
      where
        u.name = whose_ind and po.owner# = u.user# and po.name = which_ind
        and p.obj# = sp.pobj# and sp.obj# = spo.obj#
	and po.obj# = p.obj#
	and ((which_part is null)
            or (spo.subname = which_part)
            or (po.subname = which_part))
      order by pn, spn;


  -- return partition info of either specified or all indexes
  cursor get_index_partitions(
        whose_ind in varchar2, which_ind varchar2,
        which_part varchar2) is
    select /*+ all_rows */
	o.obj# pobjn, nvl(ip.part#,icp.part#) pn, o.subname pname,
        nvl(ip.analyzetime,icp.analyzetime) last_analyzed
    from
	sys.indcompart$ icp, sys.indpart$ ip, sys.obj$ o, sys.user$ u
    where 
	u.name = whose_ind and o.owner# = u.user# and
	o.name = which_ind and o.type# = 20 and
        o.obj# = ip.obj#(+) and o.obj# = icp.obj#(+) and
        ((which_part is null) or      -- all partitions
         (which_part = o.subname) or  -- partition name specified
          exists     -- subpartition name specified (return parent partition)
            (select *
               from
                 indsubpart$ tsp, obj$ spo
               where
                 spo.type# = 35 and spo.subname = which_part and
                 tsp.obj#=spo.obj# and tsp.pobj# = o.obj#))
    order by pn;

  -- return lower/upper bounds of the specified index partition
  cursor get_index_partition_bounds (
        ipobjn pls_integer		-- index partition object number
	) is
    select /*+ ordered index(p0) use_nl(p0) */
	   p0.hiboundlen plblen,p0.hiboundval plbval,
	   p.hiboundlen publen,p.hiboundval pubval
      from sys.indcompart$ p,sys.indcompart$ p0
      where p.obj#=ipobjn and p0.bo#(+)=p.bo# and p0.part#(+)=p.part#-1
    union all
    select /*+ ordered index(p0) use_nl(p0) */
	   p0.hiboundlen plblen,p0.hiboundval plbval,
	   p.hiboundlen publen,p.hiboundval pubval
      from sys.indpart$ p,sys.indpart$ p0
      where p.obj#=ipobjn and p0.bo#(+)=p.bo# and p0.part#(+)=p.part#-1;


  -- return table sub/partition names for specified sub/partition#
  cursor get_tspart_names_by_number (
	 tobjn pls_integer,		-- table object number
	 pn pls_integer,		-- partition#
	 spn pls_integer		-- subpartition#
	 ) is
    select /*+ ordered use_nl(tsp tpo tspo) */
        tpo.subname tpname, tspo.subname tspname
      from sys.tabcompart$ tp,sys.tabsubpart$ tsp,sys.obj$ tpo,sys.obj$ tspo
      where tp.bo#=tobjn and tp.part#=pn and tsp.subpart#=spn
	    and tp.obj#=tsp.pobj# and tp.obj#=tpo.obj# and tsp.obj#=tspo.obj#;
            
  -- return table partition name for specified partition#
  cursor get_tpart_name_by_number (
	 tobjn pls_integer,		-- table object number
	 pn pls_integer			-- partition#
	 ) is
    select /*+ ordered use_nl(tpo) */
        tpo.subname tpname
      from sys.tabpart$ tp,sys.obj$ tpo
      where tp.bo#=tobjn and tp.part#=pn and tp.obj#=tpo.obj#
    union all
    select /*+ ordered use_nl(tpo) */
        tpo.subname tpname
      from sys.tabcompart$ tp,sys.obj$ tpo
      where tp.bo#=tobjn and tp.part#=pn and tp.obj#=tpo.obj#;
            
  -- return partitioning key columns info
  cursor get_index_part_cols (
        iobjn pls_integer,		-- index object number
        tobjn pls_integer		-- underlying table object number
	) is
    select /*+ ordered use_nl(pc c) */
	   c.name pcname
      from partcol$ pc,col$ c
      where pc.obj#=iobjn and c.obj#=tobjn and pc.intcol#=c.intcol#
      order by pc.pos#;

  -- cursor to retrieve system statistics from the dictionary 
  cursor get_sysstats_dict is
    select s.sname, s.pname, s.pval1, s.pval2
    from aux_stats$ s
    where s.sname like 'SYSSTATS%'
    order by s.sname;

  -- cursor to retrieve filestat info
  cursor get_filestat_dict is
    select
           sum(kcfiosbr) sblkrds, sum(kcfiosbt)*10 sblkrdtim,
           sum(kcfiombr) mblkrds, sum(kcfiombt)*10 mblkrdtim,
           0 as cpucycles, 0 as cputim, 0 as job, 0 as mbrtotal
    from x$kcfio;
  
  cursor get_mbrtotal_dict is
    select s1.value - s2.value mbreads
    from v$sysstat s1, v$sysstat s2 
    where s1.name = 'physical reads' and
          s2.name = 'physical reads direct';
  
  cursor get_job_queue_processes is
    select to_number(value) from v$system_parameter
    where name = 'job_queue_processes';

CURSOR all_tables (owner varchar2, size_threshold integer,
                    MIN_PARTITIONS integer) is
    select /*+ ordered full(t) full(o) use_hash(o) full(u) use_hash(u) */
        o1.name parent,
        o1.obj# pobj,
        o2.name child,
        o2.obj# cobj,
        po1.partcnt npartitions
    from
        sys.tab$ t1, sys.obj$ o1, sys.user$ u1, sys.partobj$ po1,
        sys.tab$ t2, sys.obj$ o2, sys.user$ u2, sys.partobj$ po2
    where
        bitand(t1.property,1) != 1 and -- not a typed table
        bitand(t1.property,512) != 512 and -- not an IOT w/ row CLustering
        bitand(t1.property,4194304) != 4194304 and -- has no user-defined lob columns
        bitand(t1.property,8388608) != 8388608 and -- table does not contains unused columns
        bitand(t1.property,8192) != 8192 and -- is not at nested table
        t1.obj# = o1.obj# and o1.owner# = u1.user# and
        t1.rowcnt * t1.avgrln > (size_threshold * 1073741824) and
        u1.name = owner and
        o1.linkname is null and
        po1.obj# = t1.obj# and
        po1.parttype = 1 and
        po1.partcnt >= MIN_PARTITIONS and
        po1.partcnt = po2.partcnt and
        po1.partkeycols = 1 and
        bitand(t2.property,1) != 1 and
        bitand(t2.property,512) != 512 and
        bitand(t2.property,4194304) != 4194304 and
        bitand(t2.property,8388608) != 8388608 and
        bitand(t2.property,8192) != 8192 and
        t2.obj# = o2.obj# and o2.owner# = u2.user# and
        t2.rowcnt * t1.avgrln > (size_threshold * 1073741824) and
        u2.name = owner and
        o2.linkname is null and
        o1.name <> o2.name and
        po2.obj# = t2.obj# and
        po2.parttype = 1 and
        po2.partcnt >= MIN_PARTITIONS and
        po2.partkeycols = 1
     group by o1.name,
              o1.obj#,
              o2.name,
              o2.obj#,
              po1.partcnt;

cursor unique_ind_old (owner varchar2, tablename varchar2) is
        select o2.name index_name
        from ind$ i, sys.obj$ o1, sys.obj$ o2, sys.user$ u
        where u.name = owner and
              o1.owner# = u.user# and
              i.bo# = o1.obj# and
              o1.name = tablename and
              i.obj# = o2.obj#  and
              bitand(i.property,1) = 1 and
              i.cols = 1;

cursor unique_ind (owner varchar2, tablename varchar2) is
        select o2.name index_name, c.name col_name 
        from ind$ i, sys.obj$ o1, sys.obj$ o2, sys.user$ u, sys.icol$ ic, 
             sys.col$ c, sys.hist_head$ h 
        where u.name = owner and 
              o1.owner# = u.user# and 
              i.bo# = o1.obj# and 
              o1.name = tablename and 
              i.obj# = o2.obj# and 
              bitand(i.property,1) = 1 and 
              i.cols = 1 and 
              ic.obj# = o2.obj# and 
              ic.bo# = c.obj# and 
              ic.intcol# = c.intcol# and 
              h.col#(+) = c.intcol# and 
              h.obj#(+) = c.obj# 
        order by ic.pos#;
-- change i.property = 3 and

cursor nonunique_ind (owner varchar2, tablename varchar2) is
        select o2.name index_name, c.name col_name 
        from ind$ i, sys.obj$ o1, sys.obj$ o2, 
             sys.user$ u, sys.icol$ ic, sys.col$ c, 
             sys.hist_head$ h 
        where u.name = owner and 
              o1.owner# = u.user# and 
              i.bo# = o1.obj# and 
              o1.name = tablename and 
              i.obj# = o2.obj# and 
              bitand(i.property,1) <> 1 and 
              i.cols = 1 and 
              ic.obj# = o2.obj# and 
              ic.bo# = c.obj# and 
              ic.intcol# = c.intcol# and 
              h.col#(+) = c.intcol# and 
              h.obj#(+) = c.obj# 
        order by ic.pos#;


cursor index_cols (owner varchar2, indexname varchar2) is
        select /*+ rule */
               c.name cname
        from
               sys.user$ u, sys.obj$ o, sys.icol$ ic, sys.col$ c, sys.hist_head$ h
        where
               u.name = owner and o.owner# = u.user# and
               o.name = indexname and o.type# = 1 and
               ic.obj# = o.obj# and ic.bo# = c.obj# and
               ic.intcol# = c.intcol# and
               h.col#(+) = c.intcol# and h.obj#(+) = c.obj#
        order by ic.pos#;

cursor table_cols (objnm integer) is
        select
              name
        from col$
        where obj# = objnm;

-- get domain index property, and indexed table and columns names
cursor idx_info_qry(idx_owner varchar2, idx_name varchar2) is
select idx_inf.property, tab_own.name, tab_obj.name, col_inf.name
from obj$ idx_obj, user$ idx_own, ind$ idx_inf, icol$ idx_col, 
     col$ col_inf, obj$ tab_obj, user$ tab_own
where idx_own.name = idx_owner
and   idx_obj.name = idx_name
and   idx_obj.owner# = idx_own.user#
and   idx_obj.obj# = idx_inf.obj#
and   idx_obj.obj# = idx_col.obj#
and   idx_col.bo#  = col_inf.obj#
and   idx_col.intcol# = col_inf.intcol#
and   idx_col.bo# = tab_obj.obj#
and   tab_obj.owner# = tab_own.user#
and   idx_obj.type# = 1;

-- get column type name and type code
cursor col_type_qry(tbl_owner varchar2, tbl_name varchar2, 
                    col_name varchar2) is
select typ_own.name, 
       case bitand(col_typ.flags,16) when 16 then 'REF ' end || typ_obj.name,
       col_inf.type#
from obj$ typ_obj, user$ typ_own, col$ col_inf, coltype$ col_typ,
     obj$ tab_obj, user$ tab_own
where tab_own.name = tbl_owner
and   tab_obj.name = tbl_name
and   tab_obj.type# = 2
and   col_inf.name = col_name
and   tab_obj.owner# = tab_own.user#
and   tab_obj.obj# = col_inf.obj#
and   col_inf.obj# = col_typ.obj# (+)
and   col_inf.intcol# = col_typ.intcol# (+)
and   col_typ.toid = typ_obj.oid$ (+)
and   typ_obj.owner# = typ_own.user# (+);

-- get intcol# of column
cursor col_info_qry(tbl_owner varchar2, tbl_name varchar2,
                    col_name varchar2) is
select col_inf.intcol#, col_inf.name
from obj$ tab_obj, user$ tab_own, col$ col_inf
where tab_obj.owner#=tab_own.user#
and   tab_obj.obj#=col_inf.obj#
and   tab_obj.type#=2
and   tab_own.name=tbl_owner
and   tab_obj.name=tbl_name
and   (col_inf.name=col_name or col_name is null);

-- get table or index or type obj# or a tbl/idx partition obj number
cursor obj_part_qry(obj_owner varchar2, obj_name varchar2, 
                    part_name varchar2) is
select o.obj#
from obj$ o, user$ u
where o.owner#=u.user#
and   o.name=obj_name
and   u.name=obj_owner
and   ((o.type# in (19,20) and o.subname=part_name) or
       (o.type# in (1,2,13) and part_name is null));

-- get one or all partition names of a table or index
cursor part_name_qry(obj_owner varchar2, obj_name varchar2,
                     part_name varchar2) is
select base_obj.subname
from obj$ base_obj, user$ base_own
where base_obj.name=obj_name
and   base_own.name=obj_owner
and   base_obj.owner#=base_own.user#
and   base_obj.type# in (19,20)
and   (base_obj.subname=part_name or 
       (part_name is null and base_obj.subname is not null));

-- get object name using the object number
cursor obj_name_qry(obj_num number) is
select u.name, o.name
from obj$ o, user$ u
where o.owner#=u.user#
and   o.obj#=obj_num;

-- get user statistics from ustats$
cursor ustats_qry(obj_num number, intcol number) is
select statstype#, property, statistics
from ustats$
where obj#=obj_num
and   nvl(intcol#,0)=nvl(intcol,0);

-- get column intcol#
cursor col_num_qry(tbl_owner varchar2, tbl_name varchar2,
                   col_name varchar2) is
select c.intcol#
from obj$ o, user$ u, col$ c
where o.owner#=u.user#
and   o.obj#=c.obj#
and   o.type#=2
and   u.name=tbl_owner
and   o.name=tbl_name
and   c.name=col_name;

-- get column direct association info
cursor col_dir_assoc_qry(tbl_owner varchar2, tbl_name varchar2,
                         col_name varchar2) is
select su.name, so.name, a.statstype#, a.intcol#
from association$ a, obj$ o, user$ u, col$ c, obj$ so, user$ su
where o.owner#=u.user#
and   a.obj#=o.obj#
and   o.obj#=c.obj#
and   c.intcol#=a.intcol#
and   a.statstype#=so.obj#
and   so.owner#=su.user#
and   o.type#=2
and   u.name=tbl_owner
and   o.name=tbl_name
and   c.name=col_name;

-- get column implicit association info
cursor col_imp_assoc_qry(tbl_owner varchar2, tbl_name varchar2,
                         col_name varchar2) is
select su.name, so.name, a.statstype#, c.intcol#
from association$ a, obj$ o, user$ u, col$ c, obj$ so, user$ su,
     coltype$ ct, obj$ ty
where o.owner#=u.user#
and   a.obj#=ty.obj#
and   o.obj#=c.obj#
and   c.intcol#=ct.intcol#
and   o.obj#=ct.obj#
and   ct.toid=ty.oid$
and   a.statstype#=so.obj#
and   so.owner#=su.user#
and   o.type#=2
and   o.name=tbl_name
and   u.name=tbl_owner
and   c.name=col_name;

-- get index direct association info
cursor idx_dir_assoc_qry(idx_owner varchar2, idx_name varchar2) is
select su.name, so.name, a.statstype#
from association$ a, obj$ o, user$ u, obj$ so, user$ su
where o.owner#=u.user#
and   a.obj#=o.obj#
and   a.statstype#=so.obj#
and   so.owner#=su.user#
and   o.type#=1
and   o.name=idx_name
and   u.name=idx_owner;

-- get index implicit association info
cursor idx_imp_assoc_qry(idx_owner varchar2, idx_name varchar2) is
select su.name, so.name, a.statstype#
from association$ a, obj$ o, user$ u, obj$ so, user$ su, ind$ i
where o.owner#=u.user#
and   o.obj#=i.obj#
and   a.obj#=i.indmethod#
and   a.statstype#=so.obj#
and   so.owner#=su.user#
and   o.type#=1
and   o.name=idx_name
and   u.name=idx_owner;

  -- Procedures and functions

  procedure get_default_dop(dop OUT binary_integer);

  function schema_exists(uname varchar2) return boolean;

  function table_exists(owner varchar2, tabname varchar2) return boolean;

  function is_fixed_tab(owner varchar2, tabname varchar2) return boolean;

  procedure create_fxt_dummy(
        fxtname  varchar2,              -- name of fixed table
	dummyfxtname  varchar2          -- name of dummy table to be created
  );

  function is_fxt_dummy_ok(
        fxtname varchar2,               -- name of fixed table
	dummyfxtname varchar2           -- name of dummy table to be verified
  ) return boolean;

  procedure quick_estimate_rowcnt (
	sizetxt varchar2,
	n IN OUT number
  );

  function get_objnum (owner varchar2, objname varchar2, sname varchar2,
    objtype varchar2) return number;

  procedure set_temp_dop(
      ttabname varchar2,
      degree integer,
      alter_table boolean
  );


  procedure create_temp(
	seltxt varchar2,		-- select clause
	fromtxt varchar2,               -- from clause to use later
	ttabname OUT NOCOPY varchar2,   -- temp table name
	uname varchar2,			-- calling user name
	degree integer,			-- degree of parallelism
	populateit boolean		-- also populate the table
  );

  procedure populate_temp_insert(
	seltxt varchar2,		-- select clause
	fromtxt IN OUT varchar2,	-- from clause to use later
	ttabname IN OUT varchar2	-- temp table name
  );

  procedure truncate_temp(
	ttabname varchar2		-- temp table name
  );

  procedure drop_temp(
	ttabname varchar2		-- temp table name
  );


  function is_ext_tab(owner varchar2, tabname varchar2) return boolean;

  function get_table_degree(owner varchar2, tabname varchar2) return number;

  -- return current parallel execution settings flags
  function pqflags return binary_integer;
  -- pqflags bits (defined in ksu.h: ksuse.ksusepfl)
  DSC_PDML_ENABLED	CONSTANT BINARY_INTEGER := 16;
  DSC_PDDL_DISABLED	CONSTANT BINARY_INTEGER := 128;
  DSC_PQ_DISABLED	CONSTANT BINARY_INTEGER := 256;

  -- enable or disable each individual parallel execution settings
  procedure control_parallel(
        enable boolean,			-- enable or disable
	pqflags binary_integer		-- types of PQ opelations
  );


  -- check unique/primary key constraints/indexes whose key columns
  -- are complete subset of the specified index's and return non-null
  -- if it is unique
  function is_index_unique (bobjn number, iobjn number, icols number)
      return number;

  -- check if we can collect ndk with sys_op_descend
  -- (it can exceeds 4000 byte limit if it is too large)
  function index_max_keysize_ok(owner varchar2, indname varchar2)
      return boolean;

  -- export column statistics from the dictionary
  procedure export_colstats_direct
    (owner varchar2, tabname varchar2, colname varchar2, partname varchar2,
     stattab varchar2, statid varchar2, statown varchar2, version number);

  -- create additional pruning table procedure
  procedure cct (ownerin varchar2);

  -- insert or update ustats
  procedure insert_into_ustats( obj_num number, intcol number,
    stat_obj number, prop number, stats raw );
  procedure delete_from_ustats( obj_num number, intcol number);

end dbms_stats_internal;
/
show errors
create or replace package body dbms_stats_internal is

  GRANT_SELF exception;
  pragma EXCEPTION_INIT(GRANT_SELF, -1749);  -- granting privs to self
  CANT_EXTEND exception;
  pragma EXCEPTION_INIT(CANT_EXTEND, -1652);  -- can't extend tablespace
  NOT_ENOUGH_VALUES exception;
  pragma EXCEPTION_INIT(NOT_ENOUGH_VALUES, -947);  -- not enough values
  ALREADY_USED exception;
  pragma EXCEPTION_INIT(ALREADY_USED, -955); -- table already exists
  NOT_EXIST0 exception;
  pragma EXCEPTION_INIT(NOT_EXIST0, -942);  -- table doesn't exist
  TOO_LOW_COMPAT exception;
  pragma EXCEPTION_INIT(TOO_LOW_COMPAT, -406);  -- COMPATIBLE param too low


  procedure get_default_dop(
    dop      out    binary_integer
  ) is
  EXTERNAL NAME "kdapdefdop" WITH CONTEXT
  PARAMETERS(context,
    dop      sb4
  )
  LIBRARY DBMS_STATS_LIB;


  function schema_exists(uname varchar2) return boolean is
    retval boolean := false;
    n number;
  begin
    select count(*) into n from user$ where name = uname;
    if (n > 0) then
      retval := true;
    end if;
    return retval;
  end schema_exists;


  function table_exists(owner varchar2, tabname varchar2) return boolean is
    retval boolean := false;
    n number;
  begin
    select /*+ use_nl(u,o,t) */ count(*) into n
      from user$ u, obj$ o, tab$ t
      where u.name = owner and o.name = tabname and
        u.user# = o.owner# and o.obj# = t.obj#;
    if (n > 0) then
      retval := true;
    end if;
    return retval;
  end table_exists;


  procedure quick_estimate_rowcnt (
	sizetxt varchar2,
	n IN OUT number
	) is
  begin
    execute immediate sizetxt into n;
  end quick_estimate_rowcnt;


  -- return true if it is a fixed table
  function is_fixed_tab(owner varchar2, tabname varchar2) return boolean is
    is_fxt integer;
  begin

    if (owner != 'SYS' or tabname not like 'X$%') then
      return false;
    end if;

    is_fxt := -1;

    begin
      select /*+ rule */
        indx into is_fxt
        from sys.x$kqfta fxt
        where fxt.kqftanam = tabname and
        rownum < 2;
    exception
      when NO_DATA_FOUND then
        return false;
    end;

    return (is_fxt != -1);

  end is_fixed_tab;


  procedure create_fxt_dummy(
        fxtname IN varchar2,            -- name of fixed table
	dummyfxtname IN  varchar2       -- name of dummy table to be created
	) is
    gentxt_create varchar2(256);
    gentxt_drop varchar2(256);
  begin

    gentxt_create := 'create table sys.' || dummyfxtname ||
                     ' as select * from sys.' || fxtname || ' where 1 = 2';

    begin
      execute immediate gentxt_create;
    exception
      when ALREADY_USED then
        if (is_fxt_dummy_ok(fxtname, dummyfxtname) = false) then
          gentxt_drop := 'drop table sys.' || dummyfxtname;
          begin
            execute immediate gentxt_drop;
          end;
          begin
            execute immediate gentxt_create;
          end;
        end if;
    end;

  end create_fxt_dummy;


  function is_fxt_dummy_ok(
        fxtname IN varchar2,            -- name of fixed table
        dummyfxtname IN  varchar2       -- name of dummy table to be verified
	) return boolean is
    gentxt varchar2(256);
    retval boolean;
  begin

    retval := true;

    commit;

    gentxt := 'insert into sys.' || dummyfxtname ||
              '  select * from sys.' || fxtname || ' where rownum < 2';

    begin
      execute immediate gentxt;
    exception
      when others then
        retval := false;
    end;

    rollback;

    return retval;

  end is_fxt_dummy_ok;


  -- get the obj# for improved progression monitoring information
  function get_objnum (owner varchar2, objname varchar2, sname varchar2,
    objtype varchar2) return number is
    objn number;
  begin

    begin
      select /*+ ordered index(u) index(o) */
          o.obj# into objn
      from
          sys.user$ u, sys.obj$ o
      where
  	  u.name = owner and o.owner# = u.user# and
	  o.name = objname and
	  ((objtype = 'TABLE' and o.type# in (2,19,34)) or
	   (objtype = 'INDEX' and o.type# in (1,20,35))) and
          o.remoteowner is null and o.linkname is null and
	  ((o.subname is null and sname is null) or
	   (o.subname = sname and sname is not null));
    exception
      when NO_DATA_FOUND then
        objn := 0;
      when others then
        raise;
    end;

    return objn;

  end get_objnum;


  procedure create_temp(
	seltxt varchar2,		-- select clause
	fromtxt varchar2,               -- from clause to use later
	ttabname OUT NOCOPY varchar2,   -- temp table name
	uname varchar2,			-- calling user name
	degree integer,			-- degree of parallelism
	populateit boolean		-- also populate the table
	) is
    gentxt varchar2(32767);
    instnum number;
    partxt varchar2(20);
    whtxt varchar2(20);
    ttabname_loc varchar2(32767);
  begin

    select userenv('INSTANCE') into instnum from dual;

    if (degree <= 1) then
      partxt := ' noparallel ';
    elsif (degree is not null) then
      partxt := ' parallel ' || degree;
    end if;

    if (populateit = false) then
      whtxt := ' where 1 = 2';
    end if;

    while (ttabname_loc is null) loop
      -- don't have a valid, populated temp table
      select ora_tq_base$.nextval into ttabname_loc from dual;
      ttabname_loc := 'sys.ora_temp_' || instnum || '_ds_' || ttabname_loc;
      gentxt := 'create global temporary table ' || ttabname_loc ||
		' on commit preserve rows cache' || partxt ||
		' as select '
                || DBMS_ASSERT.NOOP(seltxt) || DBMS_ASSERT.NOOP(fromtxt)
                || whtxt;
      begin
        execute immediate gentxt;
      exception
	when ALREADY_USED then
	  ttabname := null;
	  ttabname_loc := null;
	when others then
	  ttabname := null;
	  raise;
      end;
      if (ttabname_loc is not null) then
        gentxt := 'grant select,insert on ' || ttabname_loc
                  || ' to ' || DBMS_ASSERT.SIMPLE_SQL_NAME(uname);
        begin
          execute immediate gentxt;
        exception
          when GRANT_SELF then null;
        end;
      end if;
    end loop;

    ttabname := ttabname_loc;

  end create_temp;

  procedure populate_temp_insert(
	seltxt varchar2,		-- select clause
	fromtxt IN OUT varchar2,	-- from clause to use later
	ttabname IN OUT varchar2	-- temp table name
	) is
    gentxt varchar2(32767);
  begin
    gentxt := 'insert /*+ APPEND */ into '
              || DBMS_ASSERT.QUALIFIED_SQL_NAME(ttabname) ||
	      ' select '
              || DBMS_ASSERT.NOOP(seltxt) || DBMS_ASSERT.NOOP(fromtxt);


    execute immediate gentxt;
    -- we have to commit after PDML to avoid ORA-12838
    commit;

  end populate_temp_insert;


 procedure truncate_temp(
        ttabname varchar2               -- temp table name
        ) is
    gentxt varchar2(32767);
  begin
    gentxt := 'truncate table ' || DBMS_ASSERT.QUALIFIED_SQL_NAME(ttabname);
    execute immediate gentxt;
  exception
    when NOT_EXIST0 then
      null;
  end truncate_temp;

  procedure drop_temp(
        ttabname varchar2               -- temp table name
        ) is
    gentxt varchar2(32767);
  begin
    truncate_temp(ttabname);
    gentxt := 'drop table ' || DBMS_ASSERT.QUALIFIED_SQL_NAME(ttabname);
    execute immediate gentxt;
  exception
    when NOT_EXIST0 then
      null;
    when others then
      raise;
  end drop_temp;

  procedure set_temp_dop(
      ttabname varchar2,
      degree integer,
      alter_table boolean
      ) is
    partxt varchar2(100);
    gentxt varchar2(32767);
  begin
    if (degree = sample_temp_tab_degree) then
      null;
    elsif (degree is null) then
      sample_temp_tab_degree := null;
    else
      if (alter_table) then
        if (degree > 1) then
          partxt := ' parallel ' || degree;
        else
          partxt := ' noparallel';
        end if;
        gentxt := 'alter table '
            || DBMS_ASSERT.QUALIFIED_SQL_NAME(ttabname) || partxt;
        execute immediate gentxt;
      end if;
      sample_temp_tab_degree := degree;
    end if;
  end set_temp_dop;


  -- return true if it is an external table
  function is_ext_tab(owner varchar2, tabname varchar2) return boolean is
    is_xtb integer;
  begin
    select /*+ rule */
      decode(bitand(t.property,2147483648), 0, 0, 1)
        into is_xtb
      from sys.obj$ tob, sys.user$ u, sys.tab$ t
      where tob.name = tabname and u.name = owner
        and tob.obj# = t.obj# and tob.owner# = u.user#;
    return (is_xtb != 0);
  end is_ext_tab;


  -- return DOP of the specified table
  function get_table_degree(owner varchar2, tabname varchar2) return number is
    degree number;
  begin
    select /*+ use_nl(u,o,t) */ nvl(t.degree,1)
      into degree
      from user$ u,obj$ o,tab$ t
      where u.name=owner and o.name=tabname and
        o.obj#=t.obj# and u.user#=o.owner#;
    return degree;
  end get_table_degree;


  -- return current value of the flags for parallel query processing
  function pqflags return binary_integer is
    flags binary_integer;
    sesid pls_integer;                  -- session id
    sesser pls_integer;                 -- session serial#
  begin
    select to_number(substrb(dbms_session.unique_session_id(),1,4),'XXXX'),
	   to_number(substrb(dbms_session.unique_session_id(),5,4),'XXXX')
	     into sesid, sesser from dual;
    select ksusepfl into flags
      from x$ksuse
      where ksusenum = sesid and ksuseser = sesser;
    return flags;
  end pqflags;


  -- enable/disable parallel operations (pq/pdml/pddl) specified by the flags
  procedure control_parallel(
        enable boolean,
	pqflags binary_integer
	) is
    optxt varchar2(10);
  begin
    if (enable = true) then
      optxt := ' enable ';
    else
      optxt := ' disable ';
    end if;

    if (bitand(pqflags,DSC_PQ_DISABLED) = DSC_PQ_DISABLED) then
      execute immediate 'alter session'||optxt||'parallel query';
    end if;
    if (bitand(pqflags,DSC_PDDL_DISABLED) = DSC_PDDL_DISABLED) then
      execute immediate 'alter session'||optxt||'parallel ddl';
    end if;
    if (bitand(pqflags,DSC_PDML_ENABLED) != DSC_PDML_ENABLED) then
      execute immediate 'alter session'||optxt||'parallel dml';
    end if;
  end control_parallel;


  -- check unique/primary key constraints/indexes whose key columns
  -- are complete subset of the specified index's and return non-null
  -- if it is unique
  function is_index_unique (bobjn number, iobjn number, icols number)
      return number is
    TYPE CurType IS REF CURSOR;
    cur CurType;
    retval number;
  begin
    with unqidx as
        (select /*+ index(cc) */ cd.con#,cc.intcol#
         from sys.ccol$ cc, sys.cdef$ cd
         where 
           cc.obj# = bobjn and cd.con# = cc.con# and cd.obj# = cc.obj# and
           cd.enabled is not null and
           cd.intcols <= icols and cd.type# in (2,3)
         union all
         select /*+ index(i) index(ic) */ i.obj#,ic.intcol#
         from sys.ind$ i, sys.icol$ ic
         where
           i.bo# = bobjn and i.obj# = ic.obj# and i.intcols <= icols and
           bitand(property,1) = 1 and bitand(flags,1025) = 0)
    select con#
        into retval
    from unqidx
    where con# not in
          (select /*+ no_unnest */ con#
           from unqidx
           where intcol# not in
                 (select /*+ no_unnest index(ic) */
                         intcol# from icol$ ic where obj#=iobjn)) and
                 rownum <= 1;
    return retval;

  exception
    when NO_DATA_FOUND then
      return null;
    when others then
      raise;
  end is_index_unique;


  function index_max_keysize_ok(owner varchar2, indname varchar2)
      return boolean is
    len number;
  begin
    -- column data [  + delimiter + column data + ...] + terminater
    select
      (sum(length)+count(*))*2
      into len
    from
      user$ u, obj$ o, ind$ i, icol$ ic, col$ c
    where
      u.name = owner and
      o.name = indname and
      u.user# = o.owner# and
      o.obj# = i.obj# and o.obj# = ic.obj# and
      ic.bo# = c.obj# and ic.intcol# = c.intcol#;
    if (len > 4000) then
      return false;  -- NG
    end if;
    return true;  -- OK
  end index_max_keysize_ok;


  -- export column statistics from the dictionary
  -- this doesn't delete existing entries in the stattab.
  -- *** PLEASE BE SURE TO KEEP get_colstats_dict IN SYNC WITH THIS      ***
  -- *** WHEN YOU MAKE A CHANGE HERE.                                    ***
  procedure export_colstats_direct
    (owner varchar2, tabname varchar2, colname varchar2, partname varchar2,
     stattab varchar2, statid varchar2, statown varchar2, version number) is
    alltxt varchar2(32767);
  begin

    alltxt := 'insert into ' || DBMS_ASSERT.QUALIFIED_SQL_NAME(
      '"' || statown || '"."' || stattab || '"') ||
      ' select /' || '*+ rule *' || '/ ' ||
      ':5 statid, ''C'' type, :6 version, bitand(h.spare2,7) flags,
      ot.name c1, null c2, null c3, c.name c4, u.name c5,
      h.distcnt n1, h.density n2, h.spare1 n3, h.sample_size n4, h.null_cnt n5,
      h.minimum n6, h.maximum n7, h.avgcln n8,
      decode(h.cache_cnt,0,null,1) n9, hg.bucket n10, hg.endpoint n11,
      null n12,
      h.timestamp# d1,
      h.lowval r1, h.hival r2, hg.epvalue ch1
    from sys.user$ u,  sys.obj$ ot, sys.col$ c,
	 sys.hist_head$ h, histgrm$ hg
    where
      :3 is null and
      u.name = :1 and ot.owner# = u.user# and
      ot.name = :2 and ot.type# = 2 and 
      c.obj# = ot.obj# and
      (:4 is null or c.name = :4) and
      h.obj# = ot.obj# and h.intcol# = c.intcol# and
      hg.obj#(+) = h.obj# and hg.intcol#(+) = h.intcol#
    union all
    select 
      :5 statid, ''C'' type, :6 version, bitand(h.spare2,7) flags,
      ot.name c1, op.subname c2, null c3, c.name c4, u.name c5,
      h.distcnt n1, h.density n2, h.spare1 n3, h.sample_size n4, h.null_cnt n5,
      h.minimum n6, h.maximum n7, h.avgcln n8,
      decode(h.cache_cnt,0,null,1) n9, hg.bucket n10, hg.endpoint n11,
      null n12,
      h.timestamp# d1,
      h.lowval r1, h.hival r2, hg.epvalue ch1
    from sys.user$ u,  sys.obj$ ot, sys.col$ c,
	 sys.tabpart$ tp, sys.obj$ op,
	 sys.hist_head$ h, histgrm$ hg
    where
      u.name = :1 and ot.owner# = u.user# and
      ot.name = :2 and ot.type# = 2 and 
      c.obj# = ot.obj# and
      (:4 is null or c.name = :4) and
      tp.bo# = ot.obj# and tp.obj# = op.obj# and
      (:3 is null or op.subname = :3) and
      h.obj# = op.obj# and h.intcol# = c.intcol# and
      hg.obj#(+) = h.obj# and hg.intcol#(+) = h.intcol#
    union all
    select 
      :5 statid, ''C'' type, :6 version, bitand(h.spare2,7) flags,
      op.name c1, op.subname c2, null c3, c.name c4, u.name c5,
      h.distcnt n1, h.density n2, h.spare1 n3, h.sample_size n4, h.null_cnt n5,
      h.minimum n6, h.maximum n7, h.avgcln n8,
      decode(h.cache_cnt,0,null,1) n9, hg.bucket n10, hg.endpoint n11,
      null n12,
      h.timestamp# d1,
      h.lowval r1, h.hival r2, hg.epvalue ch1
    from sys.user$ u, sys.col$ c,
	 sys.tabcompart$ tp, sys.obj$ op,
	 sys.hist_head$ h, histgrm$ hg
    where
      u.name = :1 and op.owner# = u.user# and
      op.name = :2 and op.type# = 19 and 
      (:3 is null or op.subname = :3) and
      tp.obj# = op.obj# and c.obj# = tp.bo# and
      (:4 is null or c.name = :4) and
      h.obj# = op.obj# and h.intcol# = c.intcol# and
      hg.obj#(+) = h.obj# and hg.intcol#(+) = h.intcol#
    union all
    select 
      :5 statid, ''C'' type, :6 version, bitand(h.spare2,7) flags,
      op.name c1, op.subname c2, os.subname c3, c.name c4, u.name c5,
      h.distcnt n1, h.density n2, h.spare1 n3, h.sample_size n4, h.null_cnt n5,
      h.minimum n6, h.maximum n7, h.avgcln n8,
      decode(h.cache_cnt,0,null,1) n9, hg.bucket n10, hg.endpoint n11,
      null n12,
      h.timestamp# d1,
      h.lowval r1, h.hival r2, hg.epvalue ch1
    from sys.user$ u, sys.col$ c,
	 sys.tabcompart$ tp, sys.obj$ op,
	 sys.tabsubpart$ ts, sys.obj$ os,
	 sys.hist_head$ h, histgrm$ hg
    where
      u.name = :1 and op.owner# = u.user# and
      op.name = :2 and op.type# = 19 and 
      tp.obj# = op.obj# and c.obj# = tp.bo# and
      (:4 is null or c.name = :4) and
      ts.pobj# = tp.obj# and ts.obj# = os.obj# and
      (:3 is null or op.subname = :3 or os.subname = :3) and
      h.obj# = os.obj# and h.intcol# = c.intcol# and
      hg.obj#(+) = h.obj# and hg.intcol#(+) = h.intcol#
    order by c5,c1,c2,c3,c4,n10';

    execute immediate alltxt using
	    statid,version,partname,owner,tabname,colname,colname,
	    statid,version,owner,tabname,colname,colname,partname,partname,
	    statid,version,owner,tabname,partname,partname,colname,colname,
	    statid,version,owner,tabname,colname,colname,partname,partname,
	    partname;
    commit;
  end export_colstats_direct;

-- functions for cct

procedure put_mvc(vc varchar2) is
  --  l number := lengthb(vc);
  --  p number := 1;
  begin
    --p:=1;
    --while (l > 0) loop
      --dbms_output.put_line(substr(vc,p,255));
      --l := l - 255;
      --p := p + 255;
    --end loop;
    null;
  end put_mvc;


function materialized_view_exists (
        mvname varchar2,
        owner varchar2 
        ) return boolean is
        mvcount integer := 0;
  begin
    select /*+ ordered full(t) full(o) use_hash(o) full(u) use_hash(u) */
        count(*) into mvcount
    from
        sys.tab$ t, sys.obj$ o, sys.user$ u
    where
        t.obj# = o.obj# and o.owner# = u.user# and
        u.name = owner and
        o.name = mvname;
    if (mvcount > 0) then
       return true;
    else 
       return false;
    end if;
  end materialized_view_exists;

procedure find_index_column
(
        owner IN varchar2,
        index_name in varchar2,
        num_cols out integer,
        col_name out varchar2
)
IS
        ncols integer;
BEGIN
          ncols := 0;
          put_mvc('-----------');
          put_mvc(index_name);
          FOR c_index_cols IN index_cols (owner, index_name) LOOP
            ncols := ncols + 1;
            col_name := c_index_cols.cname;
            --put_vc(col_name);
          END LOOP;
          num_cols := ncols;
END;    

procedure find_distinct_values
(
        owner IN varchar2,
        tablename IN varchar2,
        partcol IN varchar2,
        distinct_values OUT integer
)
IS
BEGIN
        select h.distcnt into distinct_values
        from
                sys.partcol$ p, sys.obj$ o, sys.col$ c, sys.user$ u, sys.hist_head$ h
        where
                o.obj# = p.obj# and
                c.obj# = o.obj# and
                c.col# = p.pos# and
                o.owner# = u.user# and
                u.name = owner and
                o.name = tablename and 
                c.name = partcol and
                h.col# = c.col# and
                h.obj# = o.obj#;
END;


procedure find_partition_column
(
        owner IN varchar2,
        tablename IN varchar2,
        partcol OUT varchar2
)
IS
BEGIN
        select c.name into partcol
        from 
                sys.partcol$ p, sys.obj$ o, sys.col$ c, sys.user$ u
        where 
                o.obj# = p.obj# and 
                c.obj# = o.obj# and
                c.col# = p.pos# and
                o.owner# = u.user# and
                u.name = owner and
                o.name = tablename;
END;


procedure prepare_creation_mv (owner varchar2, parent varchar2, child varchar2) is
     grant_rights varchar2(200);
     alter_temp_ts varchar2(200);
     grant_dba_outln varchar2(1000);
     oc integer;
     v integer;
     owner_id integer;

begin
     select user# into owner_id from user$ where name = owner;
     put_mvc('Since the creation of the mv is a large operation,');
     put_mvc('I will change the temp tablespace of outln for the time of creation');
     alter_temp_ts := 'alter user sys temporary tablespace ts_temp';
     EXECUTE IMMEDIATE alter_temp_ts;
     alter_temp_ts := 'alter user outln temporary tablespace ts_temp';
     EXECUTE IMMEDIATE alter_temp_ts;
     grant_dba_outln := 'grant dba to outln identified by outln';
     EXECUTE IMMEDIATE grant_dba_outln;
     grant_dba_outln := 'grant on commit refresh to outln identified by outln';
     EXECUTE IMMEDIATE grant_dba_outln;
     grant_rights  := 'grant all on ' || owner || '.' || parent || ' to outln';
     put_mvc (grant_rights);
     oc := dbms_sys_sql.open_cursor;
           dbms_sys_sql.parse_as_user(oc,grant_rights,1,owner_id);
     v  := dbms_sys_sql.execute (oc);
           dbms_sys_sql.close_cursor (oc);
     grant_rights  := 'grant all on ' || owner || '.' || child || ' to outln';
     oc := dbms_sys_sql.open_cursor;
           dbms_sys_sql.parse_as_user(oc,grant_rights,1,owner_id);
     v  := dbms_sys_sql.execute (oc);
           dbms_sys_sql.close_cursor (oc);
end prepare_creation_mv;


procedure create_mv_log(towner varchar2, table_objnm integer, table_name varchar2, 
                        table_key varchar2, table_rpartition varchar2,
                        number_hashp integer,tablespace varchar2) is
    mv_log varchar2(1000);
    cr_proxy_tables varchar2(1000);
    drp_mv_log varchar2(200);
    cr_views varchar2(200);
    grant_rights_btable varchar2(200);
    grant_rights_view varchar2(200);
    grant_rights_mlog varchar2(200);
    number_cols integer;
    tmaster integer;
    mv_log_exists varchar2(300);
    cur     binary_integer;
    status  binary_integer;
    owner_id integer;
    outln_id integer;
    oc integer;
    v integer; 
begin
  -- find user_ids for owner and outln
  select user# into owner_id from user$ where name = towner;
  select user# into outln_id from user$ where name = 'OUTLN';
  -- check whether mv_log already defined
  select count(master) into tmaster from mlog$ where mowner = towner and master = table_name;
  put_mvc('tmaster = ' || tmaster); 

  if ( tmaster = 1 ) then
    put_mvc('materialized view log for ' || table_name || ' exists already - nothing to do');
  else
    put_mvc('I am about to create the materiazilzed view log');

    mv_log := 'create materialized view log on ' || towner || '.'|| table_name || 
                 ' tablespace ' || tablespace || ' partition by hash(' || table_key ||
                 ') partitions ' || number_hashp || ' storage (freelists 99 freelist groups 2)' ||
                 ' parallel with rowid( ';
    number_cols := 0;

    mv_log := mv_log || table_rpartition || ',' || table_key;
    mv_log := mv_log  || ') including new values';
    put_mvc (mv_log);
    EXECUTE IMMEDIATE mv_log;
    cr_proxy_tables := 'create table outln.prits_' || table_objnm ||
                       ' tablespace ' || tablespace || ' partition by hash(' || table_key ||
                       ') partitions ' || number_hashp || ' storage (freelists 99 freelist groups 2)' ||
                       ' as select * from ' || towner || '.mlog$_' || table_name;

    put_mvc (cr_proxy_tables);
    EXECUTE IMMEDIATE cr_proxy_tables;
    drp_mv_log := 'drop table ' || towner || '.mlog$_' || table_name;
    put_mvc (drp_mv_log);
    EXECUTE IMMEDIATE drp_mv_log;
    grant_rights_mlog := 'grant all on outln.prits_' || table_objnm || ' to ' || towner || ' with grant option';
    put_mvc (grant_rights_mlog); 
    oc := dbms_sys_sql.open_cursor;
          dbms_sys_sql.parse_as_user(oc,grant_rights_mlog,1,outln_id);
    v  := dbms_sys_sql.execute (oc);
          dbms_sys_sql.close_cursor (oc); 
 
    cr_views := 'create view ' || towner || '.mlog$_' || table_name || 
                ' as select * from outln.prits_' || table_objnm; 
    put_mvc (cr_views);
    EXECUTE IMMEDIATE cr_views;
    grant_rights_view := 'grant all on ' || towner || '.mlog$_' || table_name || ' to outln';
    put_mvc (grant_rights_view);
    oc := dbms_sys_sql.open_cursor;
          dbms_sys_sql.parse_as_user(oc,grant_rights_view,1,owner_id);
    v  := dbms_sys_sql.execute (oc);
          dbms_sys_sql.close_cursor (oc);
   
 
    grant_rights_btable := 'grant all on ' || towner || '.' || table_name || ' to outln';
    put_mvc (grant_rights_btable);

    oc := dbms_sys_sql.open_cursor;
          dbms_sys_sql.parse_as_user(oc,grant_rights_btable,1,owner_id);
    v  := dbms_sys_sql.execute (oc);
          dbms_sys_sql.close_cursor (oc);

  end if;
end create_mv_log;


PROCEDURE cct 
(
        ownerin IN varchar2
)
IS

-- constants
        SAMPLE_SIZE             CONSTANT BINARY_INTEGER := 1000;
        SIZE_THRESHOLD          CONSTANT BINARY_INTEGER := 5;
        MIN_PARTITIONS          CONSTANT BINARY_INTEGER := 50; /* minimum # of partitions */
        SAMPLE_PER_PARTITION    CONSTANT BINARY_INTEGER := 1000;        /* # sample records 
                                                                      from each partition */
-- variables
        parent varchar2(30);
        child varchar2(30);
        parent_pcol varchar2(30);
        child_pcol varchar2(30);
        pk_indexed_column varchar2(60);
        pk_ncols integer;
        fk_indexed_column varchar2(60);
        fk_ncols integer;
        ri_txt varchar2(500);
        prune_ratio_txt varchar2(1000);
        create_view_txt varchar2(1000);
        drop_view_txt varchar2(1000);
        alter_view_txt varchar2(1000);
        grant_global_query_rewrite varchar2(1000);
        grant_create_table varchar2(32767);
        nr_parent integer;
        nr_child integer;
        total_count integer;
        mapped_count integer;
        pvalue_ems varchar2(5);
        foo integer;
        cur     binary_integer;
        status  binary_integer;
        infotext varchar2(500);
        ncols integer;
        child_nrows integer;
        nsample_size float(16);
        npartitions integer;
        parent_objnm integer;
        child_objnm integer;
        parent_pcol_distinct_values integer;
        child_pcol_distinct_values integer;
        number_hashp integer;
        number_cpus integer;
        child_ts varchar2(30); 
        mvname varchar2(30);
        analyze_mv varchar2(300);
        oc integer;
        outln_id integer;
        outln_tts varchar2(100);
        sys_tts varchar2(100);
        owner_dts varchar2(100);
        alter_ts varchar2(100);
        total_comb integer;
        owner varchar2(30);
        typ_owner varchar2(30) := null;
        typ_name varchar2(30);
        pk_typ_code number;
        fk_typ_code number;
BEGIN
  dbms_output.enable(10000);
  put_mvc ('I am starting cct: owner=[' || owner || ']');
  if (owner is null) then
    return;
  end if;

  select ksppstvl into pvalue_ems                     -- _subquery_pruning_mv_enabled = true
  from x$ksppi x, x$ksppcv y 
  where x.indx = y.indx and 
        ksppinm like '_subquery_pruning_mv_enabled';
  if (upper(pvalue_ems) = 'TRUE') then                -- _subquery_pruning_mv_enabled = true?
    owner := DBMS_ASSERT.SIMPLE_SQL_NAME(upper(ownerin));

    select ksppstvl into pvalue_ems                   
    from x$ksppi x, x$ksppcv y 
    where x.indx = y.indx and 
          ksppinm like '_enable_multitable_sampling';
    put_mvc ('after enable multitable sampling');
    if (upper(pvalue_ems) = 'FALSE') then             -- enable multi table sampling
      put_mvc('multi table sampling was not enabled, I am enabling it now');
      cur := dbms_sql.open_cursor;
      dbms_sql.parse(cur, 'alter session set "_enable_multitable_sampling" = true', dbms_sql.native);
      status := dbms_sql.execute(cur);
      dbms_sql.close_cursor(cur); 
    end if;
    put_mvc('did the mutli table sampling stuff');

                                                      -- set event so that we can create 
                                                      -- special materialized view
    execute immediate 'alter session set events ''10989 trace name context forever''';
  
    select ts.name into owner_dts
    from user$ u, ts$ ts
    where
      u.datats# = ts.ts# and
      u.name = owner;
    

    select ts.name into owner_dts                        -- find default tablespace of owner
    from user$ u, ts$ ts
    where
      u.datats# = ts.ts# and
      u.name = owner;
    put_mvc('The default tablespace is ' || owner_dts);
    

    select ts.name into outln_tts               -- find temporary tablespace of out and sys
    from user$ u, ts$ ts 
    where u.tempts# = ts.ts# and 
          u.name = 'OUTLN';
    select ts.name into sys_tts 
    from user$ u, ts$ ts 
    where u.tempts# = ts.ts# and 
          u.name = 'SYS';

    put_mvc('The temporary tablespaces for outln is '||outln_tts || 
             ' and for sys is '||sys_tts);

    
    put_mvc('I found the default tablespace');
                                                      -- change permission so that we can deal 
                                                      -- with materialized view
    grant_global_query_rewrite := 'grant global query rewrite to ' || owner;
    put_mvc (grant_global_query_rewrite);
    EXECUTE IMMEDIATE grant_global_query_rewrite;
    grant_global_query_rewrite := 'grant global query rewrite to outln';
    put_mvc (grant_global_query_rewrite);
    EXECUTE IMMEDIATE grant_global_query_rewrite;
    grant_create_table := 'grant create table to outln';
    put_mvc (grant_create_table);
    EXECUTE IMMEDIATE grant_create_table;

                                                      -- find all table that are range 
                                                      -- partitoned 
                                                      -- and larger than SIZE_THRESHHOLD
                                                      -- and have the same # range partitions
    FOR c1_rec IN all_tables(owner,SIZE_THRESHOLD,MIN_PARTITIONS) LOOP  
      npartitions := c1_rec.npartitions;
      parent_objnm := c1_rec.pobj;
      child_objnm := c1_rec.cobj;
      parent := DBMS_ASSERT.SIMPLE_SQL_NAME(c1_rec.parent);
      child := DBMS_ASSERT.SIMPLE_SQL_NAME(c1_rec.child);
      put_mvc('I am investigating the relationship between ' || parent || ' and ' || child);

      -- find all unique indexes on the parent table
      FOR pk_ind IN unique_ind (owner,parent) LOOP 
        -- pk_indexed_column = pk_ind.col_name;
        find_index_column (owner,pk_ind.index_name,
                           ncols,pk_indexed_column);
        put_mvc('parent index has ' || ncols || ' columns'); 
        put_mvc('Found unique index ' || pk_ind.index_name || ' on ' 
                                      || parent || ' (parent)');
        put_mvc('there are the following indexes defined on the child table');     
        -- find pk index column datatype
        open sys.dbms_stats_internal.col_type_qry(owner,parent,
                                                  pk_indexed_column);
        fetch sys.dbms_stats_internal.col_type_qry
        into typ_owner, typ_name ,pk_typ_code;
        close sys.dbms_stats_internal.col_type_qry;
        -- find all indexes on the child table

        FOR fk_ind IN nonunique_ind (owner,child) LOOP 
          put_mvc('Found index ' || fk_ind.index_name || ' on ' || child || ' (child)');
          put_mvc('Colname: ' || fk_ind.col_name);
          
                                     -- find all single column indexes for the parent table
          fk_indexed_column:=fk_ind.col_name;
          --find_index_column (owner,fk_ind.index_name,
          --                   ncols,fk_indexed_column); 
          put_mvc('child index has ' || ncols || ' columns');
        -- find fk index column datatype
        open sys.dbms_stats_internal.col_type_qry(owner,child,
                                                  fk_indexed_column);
        fetch sys.dbms_stats_internal.col_type_qry
        into typ_owner, typ_name ,fk_typ_code;
        close sys.dbms_stats_internal.col_type_qry;
        
        -- cannot proceed if pk and fk datatypes are different 
        IF (fk_typ_code = pk_typ_code) THEN
                                       -- find the number of rows for the child table
            select
              t.rowcnt into child_nrows
              from sys.user$ u, sys.obj$ ot, sys.tab$ t
              where
                u.name = owner and ot.owner# = u.user# and
                ot.name = child and ot.type# = 2 and
                ot.obj# = t.obj#;
  
                                       -- adjust the sample size to sample at least 1000 rows
              nsample_size := SAMPLE_SIZE/child_nrows * 100;
              IF ( child_nrows < SAMPLE_SIZE + 1 ) then
                nsample_size := 99.999999;
              ELSE IF (  nsample_size < 0.000001 ) then
                     nsample_size := 0.000001;
                   END IF;
              END IF;
              put_mvc('Number of rows in child ' || child_nrows);
            
                                 -- let's probe to see whether we have found the pk/fk columns
              ri_txt := 'select /*+ ORDERED USE_NL(' || parent || ') */' || 
                               'count(*) , count(' || parent || '.' || pk_indexed_column ||
                               ') from ' || owner || '.' || child || ' sample(' || to_char(nsample_size,'99.9999999999') || 
                                    '), ' || owner || '.' || parent || 
                               ' where ' || parent || '.' || pk_indexed_column || 
                                    '(+) = ' || child || '.' || fk_indexed_column;
    
              put_mvc('I am prooving relationship');
              put_mvc(ri_txt);
     
              EXECUTE IMMEDIATE ri_txt into nr_parent, nr_child;
    
              put_mvc('nr_parent: ' || nr_parent);
              put_mvc('nr_child: ' || nr_child);
  
              if (nr_parent = nr_child) then -- there are no orfants
                put_mvc('I have found a foreign key primary key relationship between ' || 
                                          parent || ' (' || pk_indexed_column || ') and ' || 
                                          child || ' (' || fk_indexed_column || ')');
            
                                 -- now we have to see whether it is worthwhile to prune
  
                                 -- cursor to find out which the partitioning columns are
                  
              find_partition_column (owner, parent, parent_pcol);
              put_mvc('Partition column parent: ' || parent_pcol);
              find_partition_column (owner, child, child_pcol);
              put_mvc('Partition column child: ' || child_pcol);
    
                                 -- find the number of unique values in the correlation field
              find_distinct_values(owner,parent,parent_pcol,parent_pcol_distinct_values);
              put_mvc('parent distinct values: ' || parent_pcol_distinct_values);
  
              find_distinct_values(owner,child,child_pcol,child_pcol_distinct_values);
              put_mvc('child distinct values: ' || child_pcol_distinct_values);
    
              nsample_size := (parent_pcol_distinct_values*child_pcol_distinct_values)/
                                     SAMPLE_PER_PARTITION/child_nrows * 100;
              put_mvc('nsample_size: ' || nsample_size);
              -- change the following to 10
              IF ( nsample_size < 20 ) then 
                prune_ratio_txt := 'select count(*), sum(cnt) ,count(distinct PPC) ' ||
                  ' * count(distinct CPC) from (' ||       
                  'select /*+ ORDERED USE_NL(' || parent || ') INDEX (' || parent || ' , ' ||  
                  pk_indexed_column || ') */ P.' ||
                  parent_pcol || ' PPC, C.' || child_pcol || ' CPC, count(*) cnt from ' ||
                  owner || '.' || child || ' sample(' || to_char(nsample_size,'99.9999999999') || ') C, ' || owner || 
                  '.' || parent || 
                  ' P where P.' ||
                  pk_indexed_column || ' = C.' || fk_indexed_column || ' group by P.' ||
                  parent_pcol || ' , C.' || child_pcol || ')'; 
                                      
                put_mvc('I am finding prune ratio');
                put_mvc(prune_ratio_txt);
    
                EXECUTE IMMEDIATE prune_ratio_txt into mapped_count, total_count, total_comb;
         
                put_mvc('mapped_count: ' || mapped_count);
                put_mvc('total_count: ' || total_count);
                put_mvc('total combination: ' || total_comb);
                if (((mapped_count / total_count)< 1) and ((mapped_count / total_comb)< 0.01)) then
  
                  select ksppstvl into number_cpus
                  from x$ksppi x, x$ksppcv y
                  where x.indx = y.indx and
                    ksppinm like 'cpu_count';
                    number_hashp := number_cpus * 2;
   
                  -- check whether materialized view exists already
                  mvname:= UPPER('prit_') || parent_objnm || '_' || child_objnm;
  
                  create_view_txt := 'create materialized view outln.' || mvname 
                          || ' pctfree 2 nologging partition by hash (' ||
                          parent_pcol || ') partitions ' || number_hashp ||  
                          ' tablespace ' || owner_dts || 
                          ' parallel (degree 1) build immediate using no index refresh force' ||
                          ' enable query rewrite as' ||
                          ' select P.' || parent_pcol || ' , C.' || child_pcol || 
                          ' , count(*) cnt from ' ||
                          owner || '.' || child || ' C, ' || owner || '.' || parent || 
                          ' P where P.' ||
                          pk_indexed_column || ' = C.' || fk_indexed_column || ' group by P.' ||
                          parent_pcol || ' , C.' || child_pcol;
  
                  alter_view_txt := 'alter materialized view outln.' || 
                                    mvname || ' refresh fast on commit';
                      
                  put_mvc('I will see whether a mv with this name exits:' || mvname);
                  if (materialized_view_exists(mvname,'OUTLN')) then
                    put_mvc('mv exists already - dropping view before recreation');
  		  drop_view_txt := 'drop materialized view outln.' || mvname;
                    EXECUTE IMMEDIATE drop_view_txt;
                  END IF; -- materialized view exists!                
     	  	prepare_creation_mv (owner,parent,child);
                  put_mvc('I will create the following mv outln');
                  put_mvc(create_view_txt);
                  EXECUTE IMMEDIATE create_view_txt; 
    
                  select user# into outln_id from user$ where name = 'OUTLN';
                  put_mvc('I am changing the status of view ' || mvname);
                  oc := dbms_sys_sql.open_cursor;
                  dbms_sys_sql.parse_as_user(oc,alter_view_txt,1,outln_id);
                  dbms_sys_sql.close_cursor(oc);
                  put_mvc ('I am analyzing the materialized view');
                  cur := dbms_sql.open_cursor;
                  --analyze_mv := 'dbms_stats.gather_table_stats(''OUTLN'', '|| mvname
                  --              || ',estimate_percent => dbms_stats.auto_sample_size)';
                  analyze_mv := 'analyze table outln.' || mvname || 
                                  ' estimate statistics sample 1 percent';
                  put_mvc (analyze_mv);
                  oc := dbms_sys_sql.open_cursor;
                  dbms_sys_sql.parse_as_user(oc,analyze_mv,1,outln_id);
                  dbms_sys_sql.close_cursor(oc);
                END IF; -- mapped_count / total_count ) < 1 -> pruning worthwhile
              END IF; -- sample size is < 20
            END IF; -- no orfants, we found a valid parent child relationship
          END IF; -- datatypes of parent and child coincide
        END LOOP; -- loop over foreign key index
      END LOOP; -- all unique parent indexes
      put_mvc('--- done ---------');
    END LOOP; -- over all tables

    -- set _enable_multitable_sampling back to original value
    if (pvalue_ems = 'FALSE') then
        put_mvc('multi table sampling was not enabled, restore state');
        cur := dbms_sql.open_cursor;
        dbms_sql.parse(cur, 'alter session set "_enable_multitable_sampling" = false', dbms_sql.native);
        status := dbms_sql.execute(cur);
        dbms_sys_sql.close_cursor(cur);
    end if;
    -- set temporary/default tablespaces back to what they were before creating the materialized view
    alter_ts := 'alter user sys temporary tablespace ' || sys_tts;
    EXECUTE IMMEDIATE alter_ts;
    alter_ts := 'alter user outln temporary tablespace ' || outln_tts;
    EXECUTE IMMEDIATE alter_ts;
  END IF; -- optimizer_features_enabled is set
END cct;

procedure insert_into_ustats( obj_num number, intcol number,
  stat_obj number, prop number, stats raw )
is
  c1 sys_refcursor;
  stats_cnt number;
  col_num number;
begin
  col_num := nvl(intcol,0);
  open c1 for 'select count(*) from ustats$ 
               where obj#=:obj_nym and intcol#=:col_num'
    using obj_num, col_num;
  fetch c1 into stats_cnt;
  close c1;
  if (stats_cnt = 0) then
    insert into ustats$ values(obj_num,col_num,stat_obj,prop,
      stats, null, null);
  else
    update ustats$ set statstype#=stat_obj, property=prop, 
      statistics=stats where obj#=obj_num and intcol#=col_num;
  end if;
end insert_into_ustats;

procedure delete_from_ustats( obj_num number, intcol number)
is
begin
  delete from ustats$ where obj#=obj_num and intcol#=intcol;
end delete_from_ustats;

end dbms_stats_internal;
/
show errors
