# $Header: alert.tcl 30-aug-02.16:41:37 mkiran Exp $
#
# copyright (c) 1999 by the Oracle Corporation
#
# NAME:
#
#  alert.tcl : Event tcl file that checks if errors exist in the alert file
#
# ARGUMENTS:
#           None    
#
# RETURN:
#           $CLEAR_EVENT or
#           $NO_EVENT or
#           $WARNING_EVENT or
#           $ALERT_EVENT
#
# OUTPUT:
#           The error messages from the alert files.
# $Log:  $
#
# Revision 1.6  2002/07/08  mkiran
# bug#1473558 : Forcing script failure when alert log file is not accessible
#
# Revision 1.5  1998/02/10  groyal
# missing alert file should not be a script failure
#
# Revision 1.4  1996/01/09  23:59:49  yliu
# use oramsg(orahome)
#
# Revision 1.3  1995/10/19 00:17:50  yliu
# moved the init code into EvalEvent
#
# Revision 1.2  1995/09/28  20:28:58  yliu
# use message file
#
# Revision 1.1  1995/09/07  17:19:27  yliu
# Initial revision
#


# Event definition
oradefine event /oracle/rdbms/fault/alert description=VOC-01011 \
report=VOC-01012
oraresults output
oradefine end


# Initialize global variables
#comment out to fix bug#606739
#set last_report $NO_EVENT
set output ""
set initial 1
set last_alert_size 0
oraeventpersist last_err_msg {}
oraeventpersist last_alert_size -1


# The main event checking functions
proc EvalEvent {} {


    # Declare globals we will use
    global last_report output last_alert_size 
    global SCRIPT_FAIL CLEAR_EVENT NO_EVENT WARNING_EVENT ALERT_EVENT
    global oramsg env
    global last_err_msg


    # initialize the return code and output
    set ret_code $NO_EVENT
    set output ""
    set alertfile ""
    set err ""

    # set related variables
    set alert_errors {ORA-006 ORA-01578 ORA-00060 ORA-000060}
    # Note: The reason we have both ORA-00060 ORA-000060 for Deadlock is
    # when the person fixed bug 616548, the 80 version has ORA-00060,
    # and the main (later 8i, 9i) version has ORA-000060.
    set warning_errors {ORA-}

    if {[catch {set alertfile [ALERTFILE]} err]} {
        lappend output $err
        return [ FailureCheck last_report last_err_msg $output ]
    }


    # during initialization, set the alert file size
    if {$last_alert_size == -1} {
        if { [file exists $alertfile] } {
            if {[catch {set last_alert_size [file size $alertfile]} err]} {
                lappend output [format [msgtxt [NETWORK] nms 1005] alert.tcl] 
                lappend output $err
                return [ FailureCheck last_report last_err_msg $output ]
            }
        } else {
            set last_alert_size 0
        }
        return $NO_EVENT
    }

    # check for the existence of the alert file
    if { ![file exists $alertfile] } {
        ORATCL_DEBUG "alert : $oramsg(oraobject) : [oratime] : $alertfile does not exist"
	lappend output "$alertfile does not exist"
	return [ FailureCheck last_report last_err_msg $output ]
    }

    # get the new alert file size
    if {[catch {set current_alert_size [file size $alertfile]} err]} {
        lappend output [format [msgtxt [NETWORK] nms 1005] alert.tcl] 
        lappend output $err
        return [ FailureCheck last_report last_err_msg $output ]
    }

    # print out event trace info 
    ORATCL_DEBUG "alert : $oramsg(oraobject) : [oratime] : $alertfile, $current_alert_size"


    # must be a new alert log
    if {$last_alert_size > $current_alert_size} {
        ORATCL_DEBUG "alert : $oramsg(oraobject) : [oratime] : $alertfile looks new : $last_alert_size, $current_alert_size"
        set last_alert_size 0
        set alert_file_recycle 1
    } else {
        set alert_file_recycle 0
    }
    
    set alerts ""

    # check for new errors
    if {$last_alert_size != $current_alert_size} {
        set number_of_errors 0
        if {[catch {set fd [open $alertfile r]} err]} {
            lappend output [format [msgtxt [NETWORK] nms 1005] alert.tcl] 
            lappend output $err
            return [ FailureCheck last_report last_err_msg $output ]
        } else {                        
            if {[catch {seek $fd $last_alert_size start} err]} {
                lappend output [format [msgtxt [NETWORK] nms 1005] alert.tcl] 
                lappend output $err
                catch {close $fd} err
                return [ FailureCheck last_report last_err_msg $output ]
            }
        }

        while {[gets $fd line] >= 0} {
            set alert_found 0
            foreach error $alert_errors {
                if {[string first $error $line] >= 0} {
                    set alert_found 1
                    set ret_code $ALERT_EVENT
                    lappend alerts [convertout $oramsg(db_characterset) $line]
                    incr number_of_errors 
                }
            }
            if {$alert_found == 0} {
                foreach error $warning_errors {
                    if {[string first $error $line] >= 0} {
                        if {$ret_code == $NO_EVENT} {
                            set ret_code $WARNING_EVENT
                        }
                        lappend alerts [convertout $oramsg(db_characterset) $line]
                        incr number_of_errors 
                    }
                }
            }
            if {$number_of_errors == 20} {
                lappend alerts ...
                break
            }
        }
        catch {close $fd} err
    }

    set last_alert_size $current_alert_size

    # clear the event if script failed last time
    if { $ret_code == $NO_EVENT && $last_report == $SCRIPT_FAIL } {
        set last_report $CLEAR_EVENT
        return $CLEAR_EVENT
    }

    # clear the event if no errors in the new alert log file AND previous state is not CLEAR_EVENT
    if { $ret_code == $NO_EVENT && $alert_file_recycle == 1 && $last_report != $CLEAR_EVENT } {
        set last_report $CLEAR_EVENT
        return $CLEAR_EVENT
    }

    # return
    # this event triggers even if it's already triggered if new errors
    # are present in the alert file
    if { $ret_code != $NO_EVENT } {
        append output $alerts
        ORATCL_DEBUG "alert : $oramsg(oraobject) : [oratime] : MESSAGE - $ret_code, $output"
    }

    set last_report $ret_code
    return $ret_code

}

