# $Header: chunksml.tcl 24-jan-2006.00:22:09 bkesavan Exp $
#
# Copyright (c) 1999, 2006, Oracle. All rights reserved.  
#
# NAME:
#
#  chunksml.tcl : Event tcl file that checks if the largest chunk in a
#                 tablespace is large enough for additional extents.
#
# ARGUMENTS:
#           argv(0) == connect string
#           argv(1) == filter of the tablespace names to be monitored, or *
#                      for all tablespaces. Examples of filters are
#                      "= 'SYSTEM'" or "in ('SYSTEM', 'TOOL')"
#           argv(2) == filter of the segment names to be monitored, or *
#                      for all segment names.
#           argv(3) == filter of the segment types to be monitored, or *
#                      for all segment types. Segment types 'CACHE',
#                      'DEFERRED ROLLBACK' and 'SPACE HEADER' are excluded.
#           argv(4) == threshold value for alert
#           argv(5) == threshold value for warning
#
# RETURN:
#           $SCRIPT_FAIL or
#           $CLEAR_EVENT or
#           $NO_EVENT or
#           $WARNING_EVENT or
#           $ALERT_EVENT
#
# OUTPUT:
#           A list of the tablespace name, segment names, segment types,
#           next extent sizes and largest chunk in the tablespace.
#
# $Log:  $
#
# Revision 1.12  2003/09/01  23:56:33  mkiran
# fixed bug 3115544
#
# Revision 1.11  1996/06/24  23:56:33  yliu
# fixed bug 370148
#
# Revision 1.10  1996/02/07  01:57:31  yliu
# use character set conversion
#
# Revision 1.9  1996/01/10 02:13:50  yliu
# use message file
#
# Revision 1.8  1995/11/11 01:30:21  yliu
# set alert and warning limit to 20
#
# Revision 1.7  1995/11/08  21:29:12  yliu
# fully qualify dba tables
#
# Revision 1.6  1995/10/31  21:25:11  yliu
# outer join dba_free_space in the sql statement
#

# Event definition
oradefine event /oracle/rdbms/space/chunksml description=VOC-01051 \
report=VOC-01052 frequency=600
oraarguments connect_str tablespace_name segment_name segment_type alert_threshold warning_threshold
oravardesc connect_str oracle_signon
oravardesc tablespace_name string default=* message=VOC-01053
oravardesc segment_name string default=* message=VOC-01054
oravardesc segment_type string default=* message=VOC-01055
oravardesc alert_threshold unsigned default=1 message=VOC-01056
oravardesc warning_threshold unsigned default=2 message=VOC-01057
oraresults segments tablespaces largest_chunk
oravardesc segments string
oravardesc tablespaces string
oravardesc largest_chunk unsigned
orafixit /oracle/rdbms/space/chgnext
oradefine end

# Initialize global variables
#comment out to fix bug#606739
#set last_report $CLEAR_EVENT
set output ""
oraeventpersist last_alert_tablespaces {}
oraeventpersist last_alert_segments {}
oraeventpersist last_warn_tablespaces {}
oraeventpersist last_warn_segments {}
oraeventpersist last_err_msg {}


#
# CheckReturn: Check event evaluation
#
proc CheckReturn {lda cda curr_report output} {
  global oramsg last_report

  if { $cda != -1 } {
     catch {oraclose $cda}
  }
  if { $lda != -1 } {
     catch {oralogoff $lda}
  }
  if { $output != "" } {
     ORATCL_DEBUG "chunksml: [oratime] : $oramsg(oraobject) : $output"
  }
  if { $last_report == $curr_report } {
     return $NO_EVENT
  } else  {
    set last_report $curr_report
    return $curr_report
  }
}

#
# next_extent_size: Return string with extent size to work with
#
proc next_extent_size { extsize blksize extpct num_segs } {
  set res ""
  for {set i 1} {$i <= $num_segs} {incr i} {
      set res "$res + $blksize * CEIL($extsize * POWER((100 + $extpct) / 100, $i))"
  }
  return [string trim $res "+"]
}


# The main event checking functions
proc EvalEvent {} {

    # Declare globals we will use
    global argv last_report output
    global SCRIPT_FAIL CLEAR_EVENT NO_EVENT WARNING_EVENT ALERT_EVENT
    global oramsg
    global last_alert_tablespaces last_alert_segments
    global last_warn_tablespaces last_warn_segments
    global last_err_msg

    ORATCL_DEBUG "chunksml: [oratime] : $oramsg(oraobject) : Start"

    # initialize the return code and output
    set ret_code $CLEAR_EVENT
    set output ""
    set err ""
    # only send first 20 to oms, but compare all so don't miss diffs
    set max_segments 20
    set db_version ""
    set alert_event_segments {}
    set alert_event_tablespaces {}
    set warn_event_segments {}
    set warn_event_tablespaces {}

    # connect to the target database
    set connect [format "%s@%s" [convertin $oramsg(agent_characterset) [lindex $argv 0]] $oramsg(oraobject)]
    if { [catch {oralogon $connect} lda] } {
        lappend output [msgtxt [RDBMS] ora $oramsg(rc)] ""
        return [CheckReturn -1 -1 $SCRIPT_FAIL $output]
    }

    if { [catch {set cda [oraopen $lda]} err] } {
        lappend output [convertout $oramsg(agent_characterset) $err]
        return [CheckReturn $lda -1 $SCRIPT_FAIL $output]
    }

    # get the database version
    if { [catch {set db_version [DB_VERSION]} err] } {
        lappend output [convertout $oramsg(agent_characterset) $err]
        return [CheckReturn $lda $cda $SCRIPT_FAIL $output]
    }

    set tablespace_name ""
    if { [lindex $argv 1] != "*" } {
        set tablespace_name " and ts.name "
        set tbspname [convertin $oramsg(agent_characterset) [lindex $argv 1]]
        append tablespace_name " $tbspname"
    }

    #
    # Setup the queries to get the tablespace basic info
    #
    switch $db_version {
        73 {
            set tbsp_query " SELECT t.name, MAX(t.ts#), v.value, MAX(s.maxbytes)
                             FROM   (SELECT /*+ ORDERED */ ts.name, MAX(ts.ts#) ts#, f.file_id
                                     FROM   sys.ts\$ ts, sys.dba_data_files f
                                     WHERE  ts.online\$ = 1 $tablespace_name
                                       AND  f.tablespace_name = ts.name
                                     GROUP BY ts.name, f.file_id
                                    ) t,
                                    (SELECT /*+ ORDERED */ ts.name, fi.file# file_id,
                                            MAX(f.length * ts.blocksize) maxbytes
                                     FROM   sys.ts\$ ts, sys.file\$ fi, sys.fet\$ f
                                     WHERE  ts.online\$ = 1 $tablespace_name
                                       AND  ts.ts# = f.ts#
                                       AND  f.ts#  = fi.ts#
                                     GROUP BY ts.name, fi.file#, f.file#
                                    ) s, v\$parameter v
                             WHERE  t.name = s.name (+)
                               AND  t.file_id = s.file_id (+)
                               AND  v.name = 'db_block_size'
                             GROUP BY t.name, v.value ";
        }
        default {
            set tbsp_query " SELECT ts.name, MAX(ts.ts#) ts#, MAX(ts.blocksize) blk,
			        to_char(MAX(DECODE(df.maxbytes, 0, fs.bytes, df.maxbytes- df.bytes))) free_space
			     FROM sys.ts\$ ts, sys.dba_data_files df, sys.dba_free_space fs
			     WHERE ts.online\$ = 1
			       AND ts.contents\$ = 0 $tablespace_name 
			       AND df.tablespace_name = ts.name 
			       AND df.tablespace_name = fs.tablespace_name(+)
                             GROUP BY ts.name ";
        }
    }

    set alert_segs [lindex $argv 5]
    set warning_segs [lindex $argv 4]
    set num_segs $warning_segs
    if { $alert_segs > $warning_segs } {
        set num_segs $alert_segs
    }

    set seg_name_part ""
    if { [lindex $argv 2] != "*" } {
        set seg_name_part " AND o.name "
        set seg_name [convertin $oramsg(agent_characterset) [lindex $argv 2]]
        append seg_name_part " $seg_name"
    }
    set seg_type_part ""
    if { [lindex $argv 3] != "*" } {
        set seg_type_part " AND so.object_type "
        set seg_type [convertin $oramsg(agent_characterset) [lindex $argv 3]]
        append seg_type_part " $seg_type"
    }

    if { [catch {orasql $cda $tbsp_query} err] } {
        set output [msgtxt [RDBMS] ora $oramsg(rc)]
        lappend output [convertout $oramsg(agent_characterset) $err]
        lappend output [convertout $oramsg(agent_characterset) $oramsg(errortxt)] $tbsp_query
        ORATCL_DEBUG "chunksml: [oratime] : $oramsg(oraobject) : orasql failed"
        return [CheckReturn $lda $cda $SCRIPT_FAIL $output]
    }

    if { [catch {set row [orafetch $cda]} err] } {
        set output [msgtxt [RDBMS] ora $oramsg(rc)]
        lappend output [convertout $oramsg(agent_characterset) $err]
        lappend output [convertout $oramsg(agent_characterset) $oramsg(errortxt)] $tbsp_query
        ORATCL_DEBUG "chunksml: [oratime] : $oramsg(oraobject) : orafetch failed"
        return [CheckReturn $lda $cda $SCRIPT_FAIL $output]
    }

    set list ""
    while { $oramsg(rc) == 0 } {
        lappend list $row
        if { [catch {set row [orafetch $cda]} err] } {
            set output [msgtxt [RDBMS] ora $oramsg(rc)]
            lappend output [convertout $oramsg(agent_characterset) $err]
            lappend output [convertout $oramsg(agent_characterset) $oramsg(errortxt)] $tbsp_query
            ORATCL_DEBUG "chunksml: [oratime] : $oramsg(oraobject) : orafetch failed"
            return [CheckReturn $lda $cda $SCRIPT_FAIL $output]
        }
    }

    if { [catch {oraclose $cda} err] } {
        set output [msgtxt [RDBMS] ora $oramsg(rc)]
        lappend output [convertout $oramsg(agent_characterset) $err]
        ORATCL_DEBUG "chunksml: [oratime] : $oramsg(oraobject) : oraclose failed"
        return [CheckReturn $lda -1 $SCRIPT_FAIL $output]
    }

    #
    # Now see which segments are in violation
    #
    foreach row $list {
        set tbsp_name    [lindex $row 0]
        set tbsp_num     [lindex $row 1]
        set tbsp_blksize [lindex $row 2]
        set tbsp_maxsize [lindex $row 3]

        switch $db_version {
          73 {
            set seg_query " SELECT u.name || '.' || o.name, so.object_type,
                                   SIGN($tbsp_maxsize - ([next_extent_size s.extsize $tbsp_blksize s.extpct $alert_segs]))
                            FROM   sys.user\$ u, sys.obj\$ o, sys.sys_objects so,
                                   ( SELECT * FROM sys.seg\$ WHERE ts# = $tbsp_num ) s
                            WHERE  s.file#  = so.header_file
                              AND  s.block# = so.header_block
                              AND  s.type   = so.segment_type_id $seg_type_part $seg_name_part
                              AND  o.owner# = u.user#
                              AND  o.obj#   = so.object_id
                              AND  o.type   = so.object_type_id
                              AND  [next_extent_size s.extsize $tbsp_blksize s.extpct $num_segs] > $tbsp_maxsize
                            ORDER BY s.extsize DESC "
          }
          default {
            set seg_query " SELECT u.name || '.' || o.name || DECODE(o.subname, NULL, '', '.' || o.subname), so.object_type,
                                   SIGN($tbsp_maxsize - ([next_extent_size s.extsize $tbsp_blksize s.extpct $alert_segs]))
                            FROM   sys.user\$ u, sys.obj\$ o, sys.seg\$ s,
                                   ( SELECT * FROM sys.sys_objects WHERE ts_number = $tbsp_num ) so
                            WHERE  s.file#  = so.header_file
                              AND  s.block# = so.header_block
                              AND  s.type#  = so.segment_type_id $seg_type_part $seg_name_part
                              AND  s.ts#    = so.ts_number
                              AND  o.owner# = u.user#
                              AND  o.obj#   = so.object_id
                              AND  o.type#  = so.object_type_id
                              AND  [next_extent_size s.extsize $tbsp_blksize s.extpct $num_segs] > $tbsp_maxsize
                            ORDER BY s.extsize DESC "
          }
        }

        if { [catch {set cda [oraopen $lda]} err] } {
            set output [msgtxt [RDBMS] ora $oramsg(rc)]
            lappend output [convertout $oramsg(agent_characterset) $err]
            ORATCL_DEBUG "chunksml: [oratime] : $oramsg(oraobject) : oraopen failed"
            return [CheckReturn $lda -1 $SCRIPT_FAIL $output]
        }

        if { [catch {orasql $cda $seg_query} err] } {
            set output [msgtxt [RDBMS] ora $oramsg(rc)]
            lappend output [convertout $oramsg(agent_characterset) $err]
            lappend output [convertout $oramsg(agent_characterset) $oramsg(errortxt)] $seg_query
            ORATCL_DEBUG "chunksml: [oratime] : $oramsg(oraobject) : orasql failed"
            return [CheckReturn $lda $cda $SCRIPT_FAIL $output]
        }

        if { [catch {set row [orafetch $cda]} err] } {
            set output [msgtxt [RDBMS] ora $oramsg(rc)]
            lappend output [convertout $oramsg(agent_characterset) $err]
            lappend output [convertout $oramsg(agent_characterset) $oramsg(errortxt)] $seg_query
            ORATCL_DEBUG "chunksml: [oratime] : $oramsg(oraobject) : orafetch failed"
            return [CheckReturn $lda $cda $SCRIPT_FAIL $output]
        }

        set alert_segments ""
        set warning_segments ""
        while { $oramsg(rc) == 0 } {
            set seg_name      [lindex $row 0]
            set seg_type      [lindex $row 1]
            set alert_warning [lindex $row 2]

            if { $alert_warning < 0 } {
                set ret_code $ALERT_EVENT
                lappend alert_event_tablespaces $tbsp_name
                lappend alert_event_segments [lindex $row 0]
                lappend alert_event_chunk $tbsp_maxsize
            } else {
                if {$ret_code == $CLEAR_EVENT} {
                    set ret_code $WARNING_EVENT
                }
                lappend warn_event_tablespaces $tbsp_name
                lappend warn_event_segments [lindex $row 0]
                lappend warn_event_chunk $tbsp_maxsize
            }

            if { [catch {set row [orafetch $cda]} err] } {
                set output [msgtxt [RDBMS] ora $oramsg(rc)]
                lappend output [convertout $oramsg(agent_characterset) $err]
                lappend output [convertout $oramsg(agent_characterset) $oramsg(errortxt)] $seg_query
                ORATCL_DEBUG "chunksml: [oratime] : $oramsg(oraobject) : orafetch failed"
                return [CheckReturn $lda $cda $SCRIPT_FAIL $output]
            }
        }

        if { [catch {oraclose $cda} output] } {
            set output [msgtxt [RDBMS] ora $oramsg(rc)]
            lappend output [convertout $oramsg(agent_characterset) $err]
            ORATCL_DEBUG "chunksml: [oratime] : $oramsg(oraobject) : oraclose failed"
            return [CheckReturn $lda -1 $SCRIPT_FAIL $output]
        }
    }

    #
    # Logoff from database
    #
    if { [catch {oralogon $connect} lda] } {
        set output [msgtxt [RDBMS] ora $oramsg(rc)]
        return [CheckReturn -1 -1 $SCRIPT_FAIL $output]
    }

    #
    # Report errors...
    #
    set alerts [llength $alert_event_segments]
    set warnings [llength $warn_event_segments]
    ORATCL_DEBUG "chunksml: [oratime] : $oramsg(oraobject) : # Alerts = $alerts, # Warnings = $warnings"

    # return
    if { $ret_code == $last_report &&
         [string compare $last_alert_segments $alert_event_segments] == 0 &&
         [string compare $last_alert_tablespaces $alert_event_tablespaces] == 0 &&
         [string compare $last_warn_segments $warn_event_segments] == 0 &&
         [string compare $last_warn_tablespaces $warn_event_tablespaces] == 0 } {
        return $NO_EVENT
    } else {
        if { $ret_code != $CLEAR_EVENT } {
            set number_of_segments 0

            for {set i 0} {$i < $alerts} {incr i} {
                lappend event_segments [lindex $alert_event_segments $i]
                lappend event_tablespaces [lindex $alert_event_tablespaces $i]
                lappend event_chunk [lindex $alert_event_chunk $i]
                incr number_of_segments
                if { $number_of_segments == $max_segments } {
                    break;
                }
            }

            if { $number_of_segments < $max_segments } {
                for {set i 0} {$i < $warnings} {incr i} {
                    lappend event_segments [lindex $warn_event_segments $i]
                    lappend event_tablespaces [lindex $warn_event_tablespaces $i]
                    lappend event_chunk [lindex $warn_event_chunk $i]
                    incr number_of_segments
                    if { $number_of_segments == $max_segments } {
                        break;
                    }
                }
            }

            if { $number_of_segments == $max_segments } {
                lappend event_segments ...
                lappend event_tablespaces ...
                lappend event_chunk ...
            }

            lappend output $event_segments $event_tablespaces $event_chunk
        }
        ORATCL_DEBUG "chunksml: [oratime] : $oramsg(oraobject) : MESSAGE - $ret_code, $output"
        set last_report $ret_code
        set last_alert_segments $alert_event_segments
        set last_alert_tablespaces $alert_event_tablespaces
        set last_warn_segments $warn_event_segments
        set last_warn_tablespaces $warn_event_tablespaces
        return $ret_code
    }
}
